__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

from qt.core import (
    QPushButton, QIcon, QMessageBox, QDialog, QVBoxLayout, QLabel, QHBoxLayout, QWidget, QPixmap, Qt, QDialogButtonBox
)
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.dialogs.palette import PaletteConfig as QuickPaletteCustomize
from calibre.gui2 import Application
from calibre.customize.zipplugin import get_icons, get_resources
from calibre.gui2 import open_url
from . import QuickPaletteCustomize as QuickPaletteCustomizePlugin

load_translations()

def _install_palette_config_apply_patch():
    if getattr(QuickPaletteCustomize, '_palette_config_apply_patch_installed', False):
        return True

    original_setup_ui = QuickPaletteCustomize.setup_ui
    original_init = QuickPaletteCustomize.__init__

    def patched_init(self, parent=None):
        # Set flag if we're in plugin context
        if getattr(QuickPaletteCustomize, '_creating_from_plugin', False):
            self._from_plugin_action = True
        return original_init(self, parent)

    def patched_setup_ui(self):
        ans = original_setup_ui(self)
        try:
            h = self.l.itemAt(self.l.count() - 1).layout()
            if h is not None and not hasattr(self, '_apply_button_added'):
                # Only add buttons if opened from plugin action
                if hasattr(self, '_from_plugin_action'):
                    # Modify the QDialogButtonBox to replace OK with Apply
                    from qt.core import QDialogButtonBox
                    
                    # Remove OK button from standard buttons
                    current_buttons = self.bb.standardButtons()
                    self.bb.setStandardButtons(current_buttons & ~QDialogButtonBox.StandardButton.Ok)
                    
                    # Add Apply button with Apply role (does not close dialog)
                    apply_button = self.bb.addButton(_('Apply'), QDialogButtonBox.ButtonRole.ApplyRole)
                    apply_button.setToolTip(_(
                        'Apply the color changes without closing the dialog.\n\n'
                        'Note: This only works when "User interface style" is set to "calibre" in Preferences → Interface → Look & Feel.\n'
                        'If using system style, changes will not apply.'
                    ))
                    # Try to set the standard OK icon
                    try:
                        from qt.core import QStyle
                        ok_icon = self.style().standardIcon(QStyle.StandardPixmap.SP_DialogOkButton)
                        if not ok_icon.isNull():
                            apply_button.setIcon(ok_icon)
                    except:
                        pass  # Fall back to no icon if not available
                    apply_button.clicked.connect(self._apply_settings_with_check)
                    
                    # Add About button
                    about_button = self.bb.addButton(_('About'), QDialogButtonBox.ButtonRole.HelpRole)
                    about_button.setToolTip(_('Show information about this plugin'))
                    try:
                        from qt.core import QIcon
                        about_button.setIcon(QIcon.ic('help.png'))
                    except Exception:
                        pass
                    about_button.clicked.connect(self.show_about_dialog)
                    # Pre-build the About dialog in the background so first click is instant
                    from qt.core import QTimer
                    QTimer.singleShot(300, self._prebuild_about_dialog)
                self._apply_button_added = True
            # Set current tab based on current theme
            try:
                pm = Application.instance().palette_manager
                if pm.use_dark_palette:
                    self.tabs.setCurrentIndex(1)  # Dark tab
                else:
                    self.tabs.setCurrentIndex(0)  # Light tab
            except Exception as e:
                pass
        except Exception as e:
            pass
        return ans

    QuickPaletteCustomize.__init__ = patched_init
    QuickPaletteCustomize.setup_ui = patched_setup_ui
    # Add the about method
    def _prebuild_about_dialog(self):
        if hasattr(self, '_about_dialog'):
            return
        dialog = QDialog(self)
        dialog.setWindowTitle(_('About Quick Palette Customize'))
        layout = QVBoxLayout(dialog)

        ver = ".".join(str(x) for x in QuickPaletteCustomizePlugin.version)

        # Choose emoji based on UI language
        try:
            from calibre.utils.localization import get_lang
            ui_lang = get_lang()
        except Exception:
            ui_lang = None
        emoji = ''
        sushi_langs = {'ja', 'zh', 'zh_CN', 'zh_TW', 'zh_HK', 'ko', 'vi', 'id', 'ms'}
        if ui_lang == 'pt':
            emoji = '&#129478;'  # falafel
        elif ui_lang == 'ar':
            emoji = '&#129478;'  # falafel
        elif ui_lang in ('tr', 'fa'):
            emoji = '&#127842;'  # oden/kebab
        elif ui_lang == 'ca':
            emoji = '&#129472;'  # cheese
        elif ui_lang == 'es':
            emoji = '&#127844;'  # shrimp
        elif ui_lang == 'gl':
            emoji = '&#129375;'  # dumpling/empanada
        elif ui_lang == 'fr':
            emoji = '&#127863;'  # wine
        elif ui_lang in sushi_langs:
            emoji = '&#127843;'  # sushi
        elif ui_lang == 'it':
            emoji = '&#127829;'  # pizza
        elif ui_lang in ('pt_BR', 'de'):
            emoji = '&#127866;'  # beer
        elif ui_lang in ('ru', 'uk', 'pl', 'cs', 'sk', 'sr', 'hr', 'sl', 'bg'):
            emoji = '&#129347;'  # tumbler glass
        else:
            emoji = '&#10084;'  # heart

        tip_text = _('Show appreciation')

        # Single rich-text label for all text content — one HTML parse, HTML entities for emoji
        html = (
            '<style>a { text-decoration: none; }</style>'
            '<div align="center">'
            '<h2>Quick Palette Customize</h2>'
            '<p>' + _('Quick access to Calibre\'s palette customization dialog') + '</p>'
            '<p style="font-size:9pt; color:gray;">' + (_('Version: %s') % ver) + '</p>'
            '<p style="font-size:11pt;"><a href="https://www.mobileread.com/forums/showthread.php?t=372047">'
            + _('MR help thread') + '</a></p>'
            '<p style="font-size:12pt;">' + emoji + '<a href="https://ko-fi.com/comfy_n">'
            + tip_text + '</a></p>'
            '</div>'
        )
        text_label = QLabel(html)
        text_label.setAlignment(Qt.AlignCenter)
        text_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
        text_label.setOpenExternalLinks(False)
        text_label.linkActivated.connect(open_url)
        layout.addWidget(text_label)

        # QR Code — use pre-cached image data from the action class
        self._qr_label = QLabel()
        self._qr_label.setAlignment(Qt.AlignCenter)
        self._qr_showing_qr = True
        layout.addWidget(self._qr_label)

        qr_data = getattr(self, '_cached_qr_data', None)
        if qr_data:
            qp = QPixmap()
            if qp.loadFromData(qr_data):
                self._qr_label.setPixmap(qp.scaledToWidth(150, Qt.SmoothTransformation))
                self._qr_label.setCursor(Qt.CursorShape.PointingHandCursor)
                self._qr_label.mousePressEvent = self._flip_qr_image
        else:
            self._qr_label.setVisible(False)

        self._about_dialog = dialog

    def show_about_dialog(self):
        if not hasattr(self, '_about_dialog'):
            self._prebuild_about_dialog()
        self._about_dialog.show()
        self._about_dialog.raise_()
        self._about_dialog.activateWindow()

    def _apply_settings_with_check(self):
        app = Application.instance()
        if not getattr(app, 'using_calibre_style', True):
            QMessageBox.information(
                self, _('Apply Settings'),
                _('Applying color changes is only available when using "calibre style" interface.\n\n'
                  'To use this feature:\n'
                  '1. Go to Preferences → Interface → Look & Feel\n'
                  '2. Change "User interface style" to "calibre"\n'
                  '3. Restart calibre\n\n'
                  'Note: Switching between user interface style types (system ↔ calibre) requires a restart '
                  'because it involves fundamental changes to how the application handles styling.')
            )
            return
        setattr(app.palette_manager, 'ignore_palette_changes', True)
        # Temporarily suppress accept() so apply_settings() cannot close the dialog
        original_accept = self.accept
        self.accept = lambda: None
        try:
            self.apply_settings()
        finally:
            self.accept = original_accept
            setattr(app.palette_manager, 'ignore_palette_changes', False)

    def _flip_qr_image(self, event):
        """Flip between QR code and easter egg image and open Ko-fi link"""
        try:
            if self._qr_showing_qr:
                pic_data = getattr(self, '_cached_pic_data', None)
                if pic_data:
                    pm = QPixmap()
                    if pm.loadFromData(pic_data):
                        self._qr_label.setPixmap(pm.scaledToWidth(150, Qt.TransformationMode.SmoothTransformation))
                        self._qr_label.setToolTip(_('You found the hidden easter egg! 🥚'))
                        self._qr_showing_qr = False
            else:
                qr_data = getattr(self, '_cached_qr_data', None)
                if qr_data:
                    pm = QPixmap()
                    if pm.loadFromData(qr_data):
                        self._qr_label.setPixmap(pm.scaledToWidth(150, Qt.TransformationMode.SmoothTransformation))
                        self._qr_label.setToolTip('')
                        self._qr_showing_qr = True

            from qt.core import QTimer
            QTimer.singleShot(0, lambda: open_url('https://ko-fi.com/comfy_n'))
        except Exception:
            pass

    QuickPaletteCustomize._prebuild_about_dialog = _prebuild_about_dialog
    QuickPaletteCustomize.show_about_dialog = show_about_dialog
    QuickPaletteCustomize._apply_settings_with_check = _apply_settings_with_check
    QuickPaletteCustomize._flip_qr_image = _flip_qr_image
    QuickPaletteCustomize._palette_config_apply_patch_installed = True
    return True

class QuickPaletteCustomizeAction(InterfaceAction):
    name = 'Quick Palette Customize'
    action_spec = ('Quick Palette Customize', None, 'Quick access to Calibre palette customization dialog', ())
    action_shortcut_name = 'Quick Palette Customize'
    dont_add_to = frozenset([
        'context-menu', 'context-menu-device', 
        'menubar-device', 'context-menu-cover-browser', 
        'context-menu-split'
    ])

    def genesis(self):
        # Install the patch to add Apply button
        _install_palette_config_apply_patch()
        # Load theme-aware icon
        self.rebuild_icon()
        self.qaction.triggered.connect(self.show_palette_config)

        # Listen for theme changes to update icon
        app = Application.instance()
        app.palette_changed.connect(self.rebuild_icon)

        # Pre-cache image bytes from zip so About dialog never needs disk I/O
        try:
            self._cached_qr_data = get_resources(self.plugin_path, 'images/qrcode.png')
        except Exception:
            self._cached_qr_data = None
        try:
            self._cached_pic_data = get_resources(self.plugin_path, 'images/pic.png')
        except Exception:
            self._cached_pic_data = None

    def rebuild_icon(self):
        pm = Application.instance().palette_manager
        icon_path = 'images/icon-dark.png' if pm.use_dark_palette else 'images/icon.png'
        icon = get_icons(self.plugin_path, icon_path, 'Quick Palette Customize')
        self.qaction.setIcon(icon)

    def show_palette_config(self):
        # Set global flag to indicate we're creating from plugin
        QuickPaletteCustomize._creating_from_plugin = True
        try:
            d = QuickPaletteCustomize(parent=self.gui)
            d.plugin_path = self.plugin_path  # Keep for about dialog
            # Pass pre-cached image data so About dialog skips zip I/O
            d._cached_qr_data = self._cached_qr_data
            d._cached_pic_data = self._cached_pic_data
            d.show()  # Modeless dialog
        finally:
            QuickPaletteCustomize._creating_from_plugin = False