__license__   = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import open_local_file
from calibre.customize.zipplugin import get_icons
from calibre.utils.config import config_dir

def _is_dark_theme():
    try:
        try:
            from qt.core import QApplication
        except Exception:
            from PyQt5.QtWidgets import QApplication  # type: ignore
        app = QApplication.instance()
        if not app:
            return False
        palette = app.palette()
        window_color = palette.color(palette.Window)
        text_color = palette.color(palette.WindowText)
        return text_color.lightness() > window_color.lightness()
    except Exception:
        return False


def _get_plugin_icon(icon_name, plugin_path=None, plugin_name=None):
    icon_name_backslash = icon_name.replace('/', '\\') if isinstance(icon_name, str) else icon_name
    icon_base = icon_name.replace('images/', '') if isinstance(icon_name, str) and icon_name.startswith('images/') else icon_name
    icon_base_backslash = icon_base.replace('/', '\\') if isinstance(icon_base, str) else icon_base

    candidates = []
    candidates.append((icon_name,))
    if icon_name_backslash != icon_name:
        candidates.append((icon_name_backslash,))
    if isinstance(icon_name, str) and icon_name.startswith('images/'):
        candidates.append((icon_base,))
        if icon_base_backslash != icon_base:
            candidates.append((icon_base_backslash,))

    if plugin_name:
        candidates.append((icon_name, plugin_name))
        if icon_name_backslash != icon_name:
            candidates.append((icon_name_backslash, plugin_name))
        if isinstance(icon_name, str) and icon_name.startswith('images/'):
            candidates.append((icon_base, plugin_name))
            if icon_base_backslash != icon_base:
                candidates.append((icon_base_backslash, plugin_name))

    if plugin_path:
        candidates.append((plugin_path, icon_name))
        if icon_name_backslash != icon_name:
            candidates.append((plugin_path, icon_name_backslash))
        if isinstance(icon_name, str) and icon_name.startswith('images/'):
            candidates.append((plugin_path, icon_base))
            if icon_base_backslash != icon_base:
                candidates.append((plugin_path, icon_base_backslash))

    for args in candidates:
        try:
            icon = get_icons(*args)
            if icon is not None and hasattr(icon, 'isNull') and not icon.isNull():
                return icon
        except TypeError:
            continue
        except Exception:
            continue
    return None

def _config_dir_path():
    try:
        return config_dir() if callable(config_dir) else config_dir
    except Exception:
        return config_dir


class OpenConfigFolderAction(InterfaceAction):
    name = 'Open Config Folder'
    action_spec = ('Open Config Folder', None, 'Open Calibre configuration folder', None)
    
    def genesis(self):
        # Load and set the icon
        self.rebuild_icon()
        
        # Connect the action
        self.qaction.triggered.connect(self.open_config_dir)
        
        # Register keyboard shortcut
        self.gui.keyboard.register_shortcut(
            'plugin:open_config_folder:open',
            'Open Config Folder: open config folder',
            description='Open the Calibre configuration folder.',
            action=self.qaction,
            group='Open Config Folder',
            persist_shortcut=True,
        )
    
    def rebuild_icon(self):
        icon = _get_plugin_icon('images/plugin.png', plugin_path=getattr(self, 'plugin_path', None), plugin_name=self.name)
        if icon is not None:
            self.qaction.setIcon(icon)
    
    def open_config_dir(self):
        open_local_file(_config_dir_path())