#!/usr/bin/env python3
# Definir _ como fallback si load_translations() falla
def _(text):
    return text  # Fallback que devuelve el texto original si no hay traducciones

try:
    load_translations()
except NameError:
    pass  # Ignorar si load_translations() no está disponible

from qt.core import Qt, QDialog, QVBoxLayout, QPushButton, QMessageBox, QLabel, QListWidget, QFileDialog, QPalette, QColor
import os
import ast
import json
import zipfile
import reprlib  # Para serializar diccionarios de manera segura

# Recursively ensure all dict keys are strings (Qt expects attribute names as strings)
def normalize_keys(obj):
    if isinstance(obj, dict):
        return {str(k): normalize_keys(v) for k, v in obj.items()}
    if isinstance(obj, list):
        return [normalize_keys(i) for i in obj]
    return obj


# Local replacement for Calibre's `set_palette_from_spec` helper using QPalette/QColor.
def set_palette_from_spec(spec):
    pal = QPalette()
    # Map of string role names to QPalette.ColorRole attributes
    role_map = {
        'Window': QPalette.Window,
        'WindowText': QPalette.WindowText,
        'Button': QPalette.Button,
        'ButtonText': QPalette.ButtonText,
        'Base': QPalette.Base,
        'AlternateBase': QPalette.AlternateBase,
        'ToolTipBase': getattr(QPalette, 'ToolTipBase', QPalette.Window),
        'ToolTipText': getattr(QPalette, 'ToolTipText', QPalette.WindowText),
        'Text': QPalette.Text,
        'BrightText': getattr(QPalette, 'BrightText', QPalette.Text),
        'Highlight': QPalette.Highlight,
        'HighlightedText': QPalette.HighlightedText,
        'Link': getattr(QPalette, 'Link', QPalette.Highlight),
        'LinkVisited': getattr(QPalette, 'LinkVisited', QPalette.Highlight),
        'PlaceholderText': getattr(QPalette, 'PlaceholderText', QPalette.Text),
    }

    # Helper to set a color for a given role and groups
    def apply_color(role_name, color_value, groups=None):
        if role_name not in role_map:
            return
        role = role_map[role_name]
        qcolor = QColor(color_value)
        if groups is None:
            # set for all groups
            pal.setColor(QPalette.Active, role, qcolor)
            pal.setColor(QPalette.Inactive, role, qcolor)
            pal.setColor(QPalette.Disabled, role, qcolor)
        else:
            for g in groups:
                pal.setColor(g, role, qcolor)

    for key, val in spec.items():
        if key.endswith('-disabled'):
            base_key = key[:-9]
            apply_color(base_key, val, groups=[QPalette.Disabled])
        else:
            apply_color(key, val)

    return pal

from calibre.utils.config import config_dir
from calibre_plugins.palette_switcher.config import prefs
from calibre.gui2 import info_dialog, error_dialog
from PyQt5.QtWidgets import QApplication

class Dialog(QDialog):

    def __init__(self, gui, icon, do_user_config):
        QDialog.__init__(self, gui)
        self.gui = gui
        self.do_user_config = do_user_config

        self.palettes_dir = os.path.join(config_dir, prefs['palettes_subdir'])
        print(f"Palettes directory: {self.palettes_dir}")  # Depuración
        if not os.path.exists(self.palettes_dir):
            os.makedirs(self.palettes_dir)
            print(f"Created palettes directory: {self.palettes_dir}")

        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.label = QLabel(_('Palette Switcher'))
        self.l.addWidget(self.label)

        self.list_widget = QListWidget(self)
        self.l.addWidget(self.list_widget)
        self.refresh_list()
        self.list_widget.itemDoubleClicked.connect(self.apply_theme)
        self.list_widget.itemSelectionChanged.connect(self.apply_theme_silent)

        self.apply_button = QPushButton(_('Apply Selected Palette'), self)
        self.apply_button.clicked.connect(self.apply_theme)
        self.l.addWidget(self.apply_button)

        self.export_button = QPushButton(_('Export Selected Palette'), self)
        self.export_button.clicked.connect(self.export_palette)
        self.l.addWidget(self.export_button)

        self.import_button = QPushButton(_('Import Palette'), self)
        self.import_button.clicked.connect(self.import_palette)
        self.l.addWidget(self.import_button)

        self.conf_button = QPushButton(_('Configure this plugin'), self)
        self.conf_button.clicked.connect(self.config)
        self.l.addWidget(self.conf_button)

        # --- nuevo botón para restaurar configuración ---
        self.reset_button = QPushButton(_('Restore default settings'), self)
        self.reset_button.clicked.connect(self.reset_to_defaults)
        self.l.addWidget(self.reset_button)

        self.delete_button = QPushButton(_('Delete Selected Palette'), self)
        self.delete_button.clicked.connect(self.delete_palette)
        self.l.addWidget(self.delete_button)

        self.about_button = QPushButton(_('About'), self)
        self.about_button.clicked.connect(self.about)
        self.l.addWidget(self.about_button)

        
        self.setWindowTitle(_('Palette Switcher'))
        self.setWindowIcon(icon)

        self.resize(self.sizeHint())

    def refresh_list(self):
        self.list_widget.clear()
        print(f"Refreshing list from: {self.palettes_dir}")  # Depuración
        try:
            for file in os.listdir(self.palettes_dir):
                if file.lower().endswith('.calibre-palette'):
                    print(f"Found file: {file}")  # Depuración
                    self.list_widget.addItem(file)
            if not self.list_widget.count():
                print("No .calibre-palette files found in the directory.")
                self.list_widget.addItem("No palettes found")
        except Exception as e:
            print(f"Error reading directory: {e}")
            self.list_widget.addItem(f"Error: {str(e)}")

    def about(self):
        text = get_resources('about.txt')
        QMessageBox.about(self, _('About Palette Switcher'),
                          text.decode('utf-8') if text else _('No about text available'))

    def apply_theme(self):
        selected = self.list_widget.currentItem()
        if not selected:
            return error_dialog(self, _('No palette selected'), _('Please select a palette first.'), show=True)
        
        palette_path = os.path.join(self.palettes_dir, selected.text())
        try:
            with open(palette_path, 'r', encoding='utf-8') as f:
                content = f.read()
                print(f"Raw content of {selected.text()}: {content}")  # Depuración
                try:
                    palette_data = json.loads(content)
                except Exception:
                    palette_data = ast.literal_eval(content)
            
            from calibre.utils.config_base import prefs as base_prefs
            # Handle different ConfigProxy.get() signatures across Calibre versions.
            # Some implementations accept a default, others only accept (self, key).
            try:
                current_theme = base_prefs.get('ui_theme', 'light')
            except TypeError:
                try:
                    current_theme = base_prefs.get('ui_theme')
                except Exception:
                    current_theme = None

            if not current_theme:
                current_theme = 'light'
            if current_theme not in ['dark', 'light']:
                current_theme = 'light'  # Fallback
            
            if isinstance(palette_data, dict) and 'dark' in palette_data and 'light' in palette_data:
                if current_theme in palette_data and 'palette' in palette_data[current_theme]:
                    palette = palette_data[current_theme]['palette']
                else:
                    raise ValueError(_(f"Invalid theme '{current_theme}' in {selected.text()}. Missing 'palette' key."))
            else:
                palette = palette_data  # Asumir paleta simple
            
            if not isinstance(palette, dict):
                raise ValueError(_(f"Invalid palette format in {selected.text()}. Must be a dictionary."))

            normalized_palette = normalize_keys(palette)

            # Diagnostic: find any non-string keys (should be none)
            def find_nonstring_keys(obj, path=""):
                problems = []
                if isinstance(obj, dict):
                    for k, v in obj.items():
                        if not isinstance(k, str):
                            problems.append((path + repr(k), type(k)))
                        problems.extend(find_nonstring_keys(v, path + f"[{k}]"))
                elif isinstance(obj, list):
                    for idx, item in enumerate(obj):
                        problems.extend(find_nonstring_keys(item, path + f"[{idx}]"))
                return problems

            problems = find_nonstring_keys(normalized_palette)
            if problems:
                print(f"Warning: non-string keys found after normalization: {problems}")

            print(f"Extracted palette for {current_theme}: {normalized_palette}")  # Depuración adicional
            # Log top-level key types for debugging
            print('Palette top-level key types:')
            for k, v in normalized_palette.items():
                print('  key:', repr(k), 'type:', type(k), 'value type:', type(v))

            # Try to apply via calibre's gprefs/palette_manager so it will be applied immediately and persist across restarts.
            try:
                from calibre.gui2 import gprefs, Application
                # If the palette file includes both light & dark specs, persist both and do NOT change the active palette mode
                if isinstance(palette_data, dict) and 'dark' in palette_data and 'light' in palette_data:
                    dark_spec = normalize_keys(palette_data['dark'].get('palette', {}))
                    light_spec = normalize_keys(palette_data['light'].get('palette', {}))
                    with gprefs:
                        dv = gprefs.get('dark_palettes', {})
                        dv['__current__'] = dark_spec
                        gprefs['dark_palettes'] = dv
                        gprefs['dark_palette_name'] = '__current__'

                        lv = gprefs.get('light_palettes', {})
                        lv['__current__'] = light_spec
                        gprefs['light_palettes'] = lv
                        gprefs['light_palette_name'] = '__current__'
                else:
                    # Single palette: persist to the current theme only
                    value = normalized_palette
                    with gprefs:
                        v = gprefs.get(f'{current_theme}_palettes', {})
                        v['__current__'] = value
                        gprefs[f'{current_theme}_palettes'] = v
                        gprefs[f'{current_theme}_palette_name'] = '__current__'
                # Tell calibre to refresh palettes immediately
                try:
                    Application.instance().palette_manager.refresh_palette()
                except Exception:
                    # fallback: also set on QApplication directly if palette_manager not available
                    pal = set_palette_from_spec(normalized_palette)
                    QApplication.instance().setPalette(pal)
            except Exception as e:
                print('Failed to apply via gprefs/palette_manager:', e)
                # Fallback to direct application
                try:
                    pal = set_palette_from_spec(normalized_palette)
                    QApplication.instance().setPalette(pal)
                except Exception as e2:
                    import traceback
                    tb = traceback.format_exc()
                    details = repr(normalized_palette)[:2000]
                    msg = f"Applying palette failed: {e2}; palette (truncated): {details}\n\nTraceback:\n{tb}"
                    print(msg)
                    raise RuntimeError(msg) from e2

            # Persist the palette in the plugin prefs for convenience
            try:
                prefs['last_palette'] = normalized_palette
                prefs['last_palette_name'] = selected.text()
            except Exception as e:
                print('Failed to persist last_palette in plugin prefs:', e)

            print(f"Palette applied successfully")  # Depuración
            info_dialog(self, _('Palette Applied'), _('The palette has been applied.'), show=True)
        except SyntaxError as e:
            error_dialog(self, _('Error Applying Palette'), _(f"Invalid syntax in {selected.text()} on line {e.lineno}: {str(e)}. Please check the file format."), show=True)
        except ValueError as e:
            error_dialog(self, _('Error Applying Palette'), str(e), show=True)
        except Exception as e:
            import traceback
            tb = traceback.format_exc()
            # Show more context in the error dialog to help debugging
            error_dialog(self, _('Error Applying Palette'), f"{e}\n\nTraceback:\n{tb}", show=True)

    def apply_theme_silent(self):
        selected = self.list_widget.currentItem()
        if not selected or selected.text() == "No palettes found" or selected.text().startswith("Error:"):
            return
        
        palette_path = os.path.join(self.palettes_dir, selected.text())
        try:
            with open(palette_path, 'r', encoding='utf-8') as f:
                content = f.read()
                print(f"Raw content of {selected.text()}: {content}")  # Depuración
                try:
                    palette_data = json.loads(content)
                except Exception:
                    palette_data = ast.literal_eval(content)
            
            from calibre.utils.config_base import prefs as base_prefs
            # Handle different ConfigProxy.get() signatures across Calibre versions.
            # Some implementations accept a default, others only accept (self, key).
            try:
                current_theme = base_prefs.get('ui_theme', 'light')
            except TypeError:
                try:
                    current_theme = base_prefs.get('ui_theme')
                except Exception:
                    current_theme = None

            if not current_theme:
                current_theme = 'light'
            if current_theme not in ['dark', 'light']:
                current_theme = 'light'  # Fallback
            
            if isinstance(palette_data, dict) and 'dark' in palette_data and 'light' in palette_data:
                if current_theme in palette_data and 'palette' in palette_data[current_theme]:
                    palette = palette_data[current_theme]['palette']
                else:
                    raise ValueError(_(f"Invalid palette '{current_theme}' in {selected.text()}. Missing 'palette' key."))
            else:
                palette = palette_data  # Asumir paleta simple
            
            if not isinstance(palette, dict):
                raise ValueError(_(f"Invalid palette format in {selected.text()}. Must be a dictionary."))

            normalized_palette = normalize_keys(palette)

            # Diagnostic: find any non-string keys (should be none)
            def find_nonstring_keys(obj, path=""):
                problems = []
                if isinstance(obj, dict):
                    for k, v in obj.items():
                        if not isinstance(k, str):
                            problems.append((path + repr(k), type(k)))
                        problems.extend(find_nonstring_keys(v, path + f"[{k}]"))
                elif isinstance(obj, list):
                    for idx, item in enumerate(obj):
                        problems.extend(find_nonstring_keys(item, path + f"[{idx}]"))
                return problems

            problems = find_nonstring_keys(normalized_palette)
            if problems:
                print(f"Warning: non-string keys found after normalization: {problems}")

            print(f"Extracted palette for {current_theme}: {normalized_palette}")  # Depuración adicional
            # Log top-level key types for debugging
            print('Palette top-level key types:')
            for k, v in normalized_palette.items():
                print('  key:', repr(k), 'type:', type(k), 'value type:', type(v))

            # Try to apply via calibre's gprefs/palette_manager so it will be applied immediately and persist across restarts.
            try:
                from calibre.gui2 import gprefs, Application
                # If the palette file includes both light & dark specs, persist both and do NOT change the active palette mode
                if isinstance(palette_data, dict) and 'dark' in palette_data and 'light' in palette_data:
                    dark_spec = normalize_keys(palette_data['dark'].get('palette', {}))
                    light_spec = normalize_keys(palette_data['light'].get('palette', {}))
                    with gprefs:
                        dv = gprefs.get('dark_palettes', {})
                        dv['__current__'] = dark_spec
                        gprefs['dark_palettes'] = dv
                        gprefs['dark_palette_name'] = '__current__'

                        lv = gprefs.get('light_palettes', {})
                        lv['__current__'] = light_spec
                        gprefs['light_palettes'] = lv
                        gprefs['light_palette_name'] = '__current__'
                else:
                    # Single palette: persist to the current theme only
                    value = normalized_palette
                    with gprefs:
                        v = gprefs.get(f'{current_theme}_palettes', {})
                        v['__current__'] = value
                        gprefs[f'{current_theme}_palettes'] = v
                        gprefs[f'{current_theme}_palette_name'] = '__current__'
                # Tell calibre to refresh palettes immediately
                try:
                    Application.instance().palette_manager.refresh_palette()
                except Exception:
                    # fallback: also set on QApplication directly if palette_manager not available
                    pal = set_palette_from_spec(normalized_palette)
                    QApplication.instance().setPalette(pal)
            except Exception as e:
                print('Failed to apply via gprefs/palette_manager:', e)
                # Fallback to direct application
                try:
                    pal = set_palette_from_spec(normalized_palette)
                    QApplication.instance().setPalette(pal)
                except Exception as e2:
                    import traceback
                    tb = traceback.format_exc()
                    details = repr(normalized_palette)[:2000]
                    msg = f"Applying palette failed: {e2}; palette (truncated): {details}\n\nTraceback:\n{tb}"
                    print(msg)
                    raise RuntimeError(msg) from e2

            # Persist the palette in the plugin prefs for convenience
            try:
                prefs['last_palette'] = normalized_palette
                prefs['last_palette_name'] = selected.text()
            except Exception as e:
                print('Failed to persist last_palette in plugin prefs:', e)

            print(f"Palette applied successfully (silent)")  # Depuración
        except Exception as e:
            print(f"Silent apply failed for {selected.text()}: {e}")  # Just print, no dialog

    def export_palette(self):
        selected = self.list_widget.currentItem()
        if not selected:
            return error_dialog(self, _('No palette selected'), _('Please select a palette to export.'), show=True)
        
        palette_path = os.path.join(self.palettes_dir, selected.text())
        try:
            with open(palette_path, 'r', encoding='utf-8') as f:
                content = f.read()
                try:
                    palette_data = json.loads(content)
                except Exception:
                    palette_data = ast.literal_eval(content)
            
            export_path, _filter = QFileDialog.getSaveFileName(self, _('Export Palette'), '', _('Calibre Palette (*.calibre-palette)'))
            if export_path:
                with open(export_path, 'w', encoding='utf-8') as f:
                    f.write(repr(palette_data))
                info_dialog(self, _('Palette Exported'), _('The palette has been exported successfully.'), show=True)
        except Exception as e:
            error_dialog(self, _('Error Exporting Palette'), str(e), show=True)

    def extract_palettes_from_zip(self, zip_path, dest_dir, depth=0, max_depth=5):
        if depth > max_depth:
            return 0, 0
        imported = 0
        skipped = 0
        try:
            with zipfile.ZipFile(zip_path, 'r') as zf:
                for name in zf.namelist():
                    if name.lower().endswith('.calibre-palette'):
                        base_name = os.path.basename(name)
                        # Sanitize filename: replace newlines and carriage returns with spaces, strip
                        base_name = base_name.replace('\n', ' ').replace('\r', ' ').strip()
                        if not base_name:
                            continue  # Skip if empty after sanitizing
                        dest_path = os.path.join(dest_dir, base_name)
                        if os.path.exists(dest_path):
                            skipped += 1
                            continue
                        with zf.open(name) as src, open(dest_path, 'wb') as dest:
                            dest.write(src.read())
                        imported += 1
                    elif name.lower().endswith('.zip') and depth < max_depth:
                        # Extract nested zip to temp and recurse
                        temp_base = os.path.basename(name).replace('\n', ' ').replace('\r', ' ').strip()
                        if not temp_base:
                            continue
                        temp_zip_path = os.path.join(dest_dir, temp_base)
                        with zf.open(name) as src, open(temp_zip_path, 'wb') as dest:
                            dest.write(src.read())
                        try:
                            sub_imported, sub_skipped = self.extract_palettes_from_zip(temp_zip_path, dest_dir, depth + 1, max_depth)
                            imported += sub_imported
                            skipped += sub_skipped
                        except zipfile.BadZipFile:
                            pass  # Skip invalid nested zip
                        os.remove(temp_zip_path)
        except zipfile.BadZipFile:
            pass  # If the zip itself is bad, return 0,0
        return imported, skipped

    def import_palette(self):
        file_path, _filter = QFileDialog.getOpenFileName(self, _('Select Palette File or Zip'), '', _('Calibre Palette or Zip (*.calibre-palette *.zip)'))
        if file_path:
            if file_path.lower().endswith('.zip'):
                try:
                    imported_count, skipped_count = self.extract_palettes_from_zip(file_path, self.palettes_dir)
                    self.refresh_list()
                    if imported_count > 0:
                        msg = _('{} palettes imported successfully.').format(imported_count)
                        if skipped_count > 0:
                            msg += _(' {} duplicates skipped.').format(skipped_count)
                        info_dialog(self, _('Palettes Imported'), msg, show=True)
                    else:
                        if skipped_count > 0:
                            info_dialog(self, _('All Palettes Skipped'), _('All palettes in the zip were duplicates and skipped.'), show=True)
                        else:
                            info_dialog(self, _('No Palettes Found'), _('No .calibre-palette files were found in the zip archive.'), show=True)
                except zipfile.BadZipFile:
                    error_dialog(self, _('Invalid Zip File'), _('The selected file is not a valid zip archive or is corrupted.'), show=True)
                except Exception as e:
                    error_dialog(self, _('Error Importing Zip'), str(e), show=True)
            else:
                base_name = os.path.basename(file_path).replace('\n', ' ').replace('\r', ' ').strip()
                if not base_name:
                    error_dialog(self, _('Invalid File Name'), _('The selected file has an invalid name.'), show=True)
                    return
                dest_path = os.path.join(self.palettes_dir, base_name)
                if os.path.exists(dest_path):
                    info_dialog(self, _('Palette Skipped'), _('A palette with this name already exists.'), show=True)
                else:
                    with open(file_path, 'rb') as src, open(dest_path, 'wb') as dest:
                        dest.write(src.read())
                    self.refresh_list()
                    info_dialog(self, _('Palette Imported'), _('The palette has been imported successfully.'), show=True)

    def delete_palette(self):
        selected = self.list_widget.currentItem()
        if not selected or selected.text() in ["No palettes found"] or selected.text().startswith("Error:"):
            return error_dialog(self, _('No palette selected'), _('Please select a palette to delete.'), show=True)

        palette_path = os.path.join(self.palettes_dir, selected.text())
        if not os.path.exists(palette_path):
            return error_dialog(self, _('File Not Found'), _('The selected palette file does not exist.'), show=True)

    # Confirmación antes de borrar
        reply = QMessageBox.question(
            self,
            _('Confirm Deletion'),
            _('Are you sure you want to delete "{}"?').format(selected.text()),
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No
    )

        if reply == QMessageBox.Yes:
            try:
                os.remove(palette_path)
                self.refresh_list()
                info_dialog(self, _('Palette Deleted'), _('The palette has been deleted successfully.'), show=True)
            except Exception as e:
                error_dialog(self, _('Error Deleting Palette'), str(e), show=True)

    def config(self):
        self.do_user_config(parent=self)
        
    def reset_to_defaults(self): 
        app = QApplication.instance()
        if app is None:
            return 
        
        from calibre.gui2 import gprefs, Application 
        # Borrar preferencias de tema y paletas 
        gprefs['ui_theme'] = None 
        gprefs['dark_mode'] = None 
        gprefs['light_palettes'] = {} 
        gprefs['dark_palettes'] = {} 
        gprefs['light_palette_name'] = None 
        gprefs['dark_palette_name'] = None 
        
        # Refrescar paleta global 
        try: 
            Application.instance().palette_manager.refresh_palette()
        except Exception:
            app.setPalette(app.style().standardPalette())
            
        info_dialog(self, _('Configuration restored'),
                    _('calibre has been reverted to its default settings.'),
                     show=True)
