__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

try:
    from qt.core import Qt, QEvent, QObject, QTableWidgetItem
except Exception:
    from PyQt5.Qt import Qt, QEvent, QObject
    from PyQt5.QtWidgets import QTableWidgetItem


class SortKeyTableWidgetItem(QTableWidgetItem):
    def __init__(self, display_text, sort_key=0):
        super().__init__(display_text)
        try:
            self.sort_key = int(sort_key or 0)
        except Exception:
            self.sort_key = 0

    def __lt__(self, other):
        if isinstance(other, SortKeyTableWidgetItem):
            return self.sort_key < other.sort_key
        return super().__lt__(other)


class CtrlMiddleAutoscroll(QObject):
    """Event filter: Ctrl + middle-click drag to scroll a scrollable widget."""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.active = False
        self.last_pos = None
        self.target = None
        self.orig_cursor = None

    def eventFilter(self, obj, event):
        try:
            from qt.core import Qt as _Qt
        except Exception:
            from PyQt5.Qt import Qt as _Qt

        try:
            etype = event.type()
        except Exception:
            return False

        # Mouse press: start autoscroll when middle button + Ctrl
        try:
            if etype == QEvent.Type.MouseButtonPress and event.button() == _Qt.MouseButton.MiddleButton and (
                int(event.modifiers()) & int(_Qt.KeyboardModifier.ControlModifier)
            ):
                self.active = True
                self.target = obj
                self.last_pos = event.globalPos()
                try:
                    self.orig_cursor = obj.cursor()
                    obj.setCursor(_Qt.ClosedHandCursor)
                except Exception:
                    self.orig_cursor = None
                return True
        except Exception:
            pass

        # Mouse move: scroll by delta
        try:
            if self.active and etype == QEvent.Type.MouseMove and self.target is obj:
                gp = event.globalPos()
                dy = gp.y() - self.last_pos.y()
                dx = gp.x() - self.last_pos.x()
                self.last_pos = gp
                try:
                    vs = obj.verticalScrollBar()
                    hs = obj.horizontalScrollBar()
                    if vs is not None:
                        vs.setValue(vs.value() - dy)
                    if hs is not None:
                        hs.setValue(hs.value() - dx)
                except Exception:
                    pass
                return True
        except Exception:
            pass

        # Mouse release: stop autoscroll
        try:
            if self.active and etype == QEvent.Type.MouseButtonRelease and event.button() == _Qt.MouseButton.MiddleButton:
                self.active = False
                try:
                    if self.target is not None:
                        self.target.unsetCursor()
                except Exception:
                    pass
                self.target = None
                self.orig_cursor = None
                return True
        except Exception:
            pass

        return False
