__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

# General utility functions for RSS Reader
import datetime
import xml.etree.ElementTree as ET

def iso_to_ts(iso_text):
    try:
        if not iso_text:
            return 0
        if isinstance(iso_text, (int, float)):
            return int(iso_text)
        if isinstance(iso_text, str):
            try:
                return int(iso_text)
            except Exception:
                pass
        dt = datetime.datetime.fromisoformat(str(iso_text))
        return int(dt.timestamp())
    except Exception:
        return 0

def format_published_display(ts, iso_text):
    try:
        ts_i = int(ts or 0)
    except Exception:
        ts_i = 0
    if not ts_i:
        return str(iso_text or '')

    # Prefer Qt formatting so the user's setting (yyyy-MM-dd, etc.) is applied consistently.
    try:
        from calibre_plugins.rss_reader.config import plugin_prefs
        qt_fmt = str(plugin_prefs.get('published_timestamp_format', 'yyyy-MM-dd HH:mm') or 'yyyy-MM-dd HH:mm')
    except Exception:
        qt_fmt = 'yyyy-MM-dd HH:mm'

    try:
        try:
            from qt.core import QDateTime
        except Exception:
            from PyQt5.Qt import QDateTime
        qdt = QDateTime.fromSecsSinceEpoch(int(ts_i))
        try:
            qdt = qdt.toLocalTime()
        except Exception:
            pass
        s = qdt.toString(qt_fmt)
        if s:
            return str(s)
    except Exception:
        pass

    # Fallback: Python local-time formatting.
    try:
        dt = datetime.datetime.fromtimestamp(ts_i)
        return str(dt.strftime('%Y-%m-%d %H:%M') or '')
    except Exception:
        return str(iso_text or '')


def format_ts_display(ts, fmt='%Y-%m-%d %H:%M'):
    try:
        ts = int(ts or 0)
    except Exception:
        ts = 0
    if not ts:
        return ''
    try:
        dt = datetime.datetime.fromtimestamp(ts)
        return str(dt.strftime(fmt) or '')
    except Exception:
        return ''

def safe_folder_path(folder_path, default_folder=''):
    try:
        p = str(folder_path or '').strip().strip('/')
    except Exception:
        p = ''
    if not p and default_folder:
        try:
            p = str(default_folder or '').strip().strip('/')
        except Exception:
            p = ''
    return p

def parse_opml_file(path, default_folder=''):
    tree = ET.parse(path)
    root = tree.getroot()
    body = root.find('body')
    if body is None:
        body = root.find('.//body')
    start = body if body is not None else root
    out = []
    def walk(parent, folder_path):
        for o in list(parent.findall('outline')):
            try:
                attrib = dict(o.attrib or {})
            except Exception:
                attrib = {}
            url = attrib.get('xmlUrl') or attrib.get('xmlurl') or attrib.get('url')
            title = (attrib.get('title') or attrib.get('text') or '').strip()
            folder_attr = (attrib.get('category') or attrib.get('folder') or '').strip().strip('/')
            child_outlines = list(o.findall('outline') or [])
            next_folder = folder_path
            if child_outlines and not url:
                seg = folder_attr or title
                if seg:
                    next_folder = (folder_path.rstrip('/') + '/' + seg.strip().strip('/')).strip('/') if folder_path else seg.strip().strip('/')
            elif folder_attr:
                next_folder = folder_attr
            if url:
                url = str(url or '').strip()
                if url:
                    folder_effective = folder_attr or folder_path
                    if folder_attr and folder_path:
                        try:
                            fp = str(folder_path or '').strip().strip('/')
                            fa = str(folder_attr or '').strip().strip('/')
                        except Exception:
                            fp, fa = folder_path, folder_attr
                        # If we're inside a parent folder outline (e.g. roots), keep that nesting.
                        # Only avoid double-prefixing when the feed folder already includes it.
                        if fp and fa and not fa.startswith(fp + '/') and fa != fp:
                            folder_effective = (fp.rstrip('/') + '/' + fa.lstrip('/')).strip('/')
                    out.append({
                        'title': title or url,
                        'url': url,
                        'folder': safe_folder_path(folder_effective, default_folder=default_folder),
                    })
            if child_outlines:
                walk(o, next_folder)
    walk(start, '')
    seen_urls = set()
    uniq = []
    for entry in out:
        url = entry.get('url')
        if url and url not in seen_urls:
            uniq.append(entry)
            seen_urls.add(url)
    return uniq
