__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

from calibre_plugins.rss_reader.dialogs import _AddFeedDialog, SendFeedsViaEmailDialog, TagEditorDialog
from calibre_plugins.rss_reader.tagging_utils import auto_tags_for_item as _auto_tags_for_item_util
from calibre_plugins.rss_reader.article_fetch import (
    _fetch_article_content,
    _fetch_article_content_calibre_recipe_like,
    _fetch_article_content_calibre_readability,
)
#!/usr/bin/env python

# from calibre_plugins.rss_reader.debug import _debug, DEBUG_RSS_READER
from calibre_plugins.rss_reader.debug import _debug, DEBUG_RSS_READER
from calibre_plugins.rss_reader.utils import format_published_display

# CRITICAL: calibre injects a module-specific load_translations() into each
# plugin module via functools.partial, bound to THAT module's __dict__.
# ``from ui_part1 import *`` would overwrite our load_translations with
# ui_part1's version (bound to ui_part1's __dict__), so calling it would
# install _() into ui_part1 instead of here.  Save our own copy first.
try:
    _own_load_translations = load_translations
except NameError:
    _own_load_translations = None

# Split from ui_impl.py (part 2)
from calibre_plugins.rss_reader.ui_part1 import *  # noqa: F401,F403

# Now call OUR saved load_translations to install _() into THIS module.
if _own_load_translations is not None:
    try:
        _own_load_translations()
    except Exception:
        pass

# Explicit imports required at module import time.
# Do not rely on ui_part1's star-import to provide these, since ui_part1 is
# intentionally a partial split and may not import all mixins.
from calibre_plugins.rss_reader.profiles_mixin import ProfilesMixin
from calibre_plugins.rss_reader.ai_panel_mixin import AIPanelMixin
from calibre_plugins.rss_reader.widgets import SortKeyTableWidgetItem, CtrlMiddleAutoscroll
from calibre_plugins.rss_reader.utils import iso_to_ts, format_published_display, safe_folder_path, parse_opml_file
from calibre_plugins.rss_reader.preview_browser import (
    _sanitize_url_for_fetch,
    _normalize_images_for_preview,
)

class _RSSReaderDialogPartB:
    def add_feed(self):
        try:
            dlg = _AddFeedDialog(self)
            if dlg.exec() != QDialog.DialogCode.Accepted:
                return
            url, title, download_images, use_recipe_engine, oldest_days, max_articles = dlg.get_values()
            if not url:
                return

            folder = self.selected_folder_path() or str(plugin_prefs.get('default_folder', '') or '')

            feeds = list(rss_db.get_feeds() or [])
            is_recipe = False
            recipe_urn = ''
            try:
                from calibre_plugins.rss_reader.recipe_utils import is_recipe_urn, get_recipe_feeds_from_urn
                if is_recipe_urn(url):
                    # Only allow RSS-feed-based recipes; reject scrape-only recipes
                    subfeeds = get_recipe_feeds_from_urn(str(url or ''))
                    if not subfeeds:
                        raise ValueError(_('Recipe provides no RSS feeds'))
                    is_recipe = True
                    recipe_urn = str(url or '')
            except Exception as e:
                try:
                    # If it looks like a recipe URN but can't be used, cancel import
                    from calibre_plugins.rss_reader.recipe_utils import is_recipe_urn
                    if is_recipe_urn(url):
                        error_dialog(self, _('Cannot import recipe'), _('This recipe cannot be imported because it does not declare RSS feeds that the plugin can update.'), show=True, det_msg=str(e))
                        return
                except Exception:
                    pass
            feeds.append({'id': str(uuid.uuid4()), 'title': title or url, 'url': url, 'enabled': True, 'folder': folder, 'download_images': bool(download_images), 'always_notify': False, 'feed_starred': False, 'is_recipe': is_recipe, 'recipe_urn': recipe_urn, 'use_recipe_engine': bool(use_recipe_engine), 'oldest_article_days': int(oldest_days or 0), 'max_articles': int(max_articles or 0)})
            # Remember the newly added feed id so we can update only it (avoid updating entire DB)
            try:
                new_feed_id = str(feeds[-1].get('id') or '')
            except Exception:
                new_feed_id = ''
            rss_db.save_feeds(feeds)

            # Ensure folder exists
            try:
                folders = list(rss_db.get_folders() or [])
                fp = str(folder or '').strip().strip('/')
                if fp and fp not in folders:
                    folders.append(fp)
                    rss_db.save_folders(folders)
            except Exception:
                pass

            self.refresh()

            # Kick off an update for the newly added feed so the title gets auto-detected
            try:
                if new_feed_id:
                    self.update_all(feed_ids=[new_feed_id])
                else:
                    self.update_all()
            except Exception:
                try:
                    self.update_all()
                except Exception:
                    pass
        except Exception as e:
            error_dialog(self, _('RSS Reader Error'), _('Failed to add feed: %s') % str(e),
                        show=True, det_msg=traceback.format_exc())

    def add_multiple_feeds(self):
        try:
            from calibre_plugins.rss_reader.add_multiple_feeds_dialog import AddMultipleFeedsDialog
        except Exception:
            from add_multiple_feeds_dialog import AddMultipleFeedsDialog
        dlg = AddMultipleFeedsDialog(self)
        if dlg.exec() != QDialog.DialogCode.Accepted:
            return
        urls, folder, oldest_days, max_articles = dlg.get_values()
        if not urls:
            return
        # Optionally create folder
        folder_path = folder.strip()
        if folder_path:
            folder_path = self.add_folder(parent_path='') if not folder_path in (rss_db.get_folders() or []) else folder_path
            if folder_path:
                folder = folder_path
        else:
            folder = self.selected_folder_path() or str(plugin_prefs.get('default_folder', '') or '')
        feeds = list(rss_db.get_feeds() or [])
        for url in urls:
            feeds.append({'id': str(uuid.uuid4()), 'title': url, 'url': url, 'enabled': True, 'folder': folder, 'download_images': True, 'always_notify': False, 'feed_starred': False, 'is_recipe': False, 'recipe_urn': '', 'use_recipe_engine': False, 'oldest_article_days': int(oldest_days or 0), 'max_articles': int(max_articles or 0)})
        rss_db.save_feeds(feeds)
        # Ensure folder exists
        if folder:
            folders = list(rss_db.get_folders() or [])
            fp = str(folder or '').strip().strip('/')
            if fp and fp not in folders:
                folders.append(fp)
                rss_db.save_folders(folders)
        self.refresh()

    def add_sample_feeds(self):
        """Show bundled featured feeds dialog and add selected feeds."""
        try:
            from calibre_plugins.rss_reader.bundled_feeds_dialog import BundledFeedsDialog
        except Exception:
            try:
                from bundled_feeds_dialog import BundledFeedsDialog
            except Exception:
                try:
                    QMessageBox.warning(self, _('Error'), _('Could not load featured feeds dialog.'))
                except Exception:
                    pass
                return

        dlg = BundledFeedsDialog(self)
        try:
            res = dlg.exec()
        except Exception:
            try:
                res = dlg.exec_()
            except Exception:
                res = 0

        if res != QDialog.Accepted:
            return

        urls = getattr(dlg, 'selected_urls', []) or []
        entries = getattr(dlg, 'selected_entries', []) or []
        if not urls and not entries:
            return

        # Add selected feeds to the database (shared logic + de-dup)
        try:
            from calibre_plugins.rss_reader.sample_feeds import add_sample_feeds
            payload = entries if entries else urls
            try:
                target_folder = self.selected_folder_path() or ''
            except Exception:
                target_folder = ''
            # If no folder is selected, keep featured feeds tidy in a dedicated folder.
            if not str(target_folder or '').strip():
                target_folder = 'Featured'
            added, skipped = add_sample_feeds(payload, db_path=rss_db.db_path(), folder=target_folder)
            self.refresh()
            if skipped:
                self.status.setText(_('Added %d featured feed(s) (%d already existed).') % (added, skipped))
            else:
                self.status.setText(_('Added %d featured feed(s).') % added)
        except Exception as e:
            error_dialog(self, _('RSS Reader Error'), _('Failed to add featured feeds: %s') % str(e),
                        show=True, det_msg=traceback.format_exc())

    def remove_selected_feed(self):
        feed_ids = self.selected_feed_ids()
        # Determine if user selected folders (which imply removing multiple feeds)
        try:
            indexes = self.feeds_tree.selectionModel().selectedIndexes()
            items = [self.feeds_tree.model().itemFromIndex(idx) for idx in indexes]
        except Exception:
            items = []
        selected_folders = []
        for it in items:
            try:
                data = it.data(ROLE_USER)
            except Exception:
                data = None
            if isinstance(data, dict) and data.get('type') == 'folder':
                selected_folders.append(str(data.get('path') or ''))

        if not feed_ids and not selected_folders:
            return

        # Build context-aware confirmation message
        if selected_folders:
            if len(selected_folders) == 1:
                title = _('Remove folder')
                folder_name = selected_folders[0]
                # Count feeds inside this folder
                num = len(list(self._folder_to_feed_ids.get(folder_name, []) or []))
                msg = _('Remove folder "%s" and its %d feed(s)?') % (folder_name, num)
            else:
                title = _('Remove folders')
                total_feeds = 0
                for p in selected_folders:
                    total_feeds += len(list(self._folder_to_feed_ids.get(p, []) or []))
                msg = _('Remove %d selected folders and their %d feed(s)?') % (len(selected_folders), total_feeds)
        else:
            # No folders selected, simple feed removal
            title = _('Remove feed')
            if len(feed_ids) == 1:
                msg = _('Remove selected feed?')
            else:
                msg = _('Remove %d selected feeds?') % len(feed_ids)

        yes = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'Yes', getattr(QMessageBox, 'Yes', None))
        no = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'No', getattr(QMessageBox, 'No', None))
        ans = QMessageBox.question(self, title, msg, yes | no, no)
        if ans != yes:
            return

        try:
            rss_db.delete_feeds(list(feed_ids))
        except Exception:
            # Fallback: best-effort rebuild
            feeds = [f for f in (rss_db.get_feeds() or []) if str(f.get('id') or '') not in set(feed_ids)]
            try:
                rss_db.save_feeds(feeds)
            except Exception:
                pass


        for fid in list(feed_ids):
            self._items_by_feed_id.pop(fid, None)

        # Remove any selected folders that are now empty
        try:
            folders = list(rss_db.get_folders() or [])
            feeds = list(rss_db.get_feeds() or [])
            folders_to_remove = []
            for folder in selected_folders:
                # If no feeds remain in this folder, remove it
                if not any((f.get('folder') or '').strip() == folder.strip() for f in feeds):
                    folders_to_remove.append(folder.strip())
            if folders_to_remove:
                folders = [f for f in folders if f.strip() not in folders_to_remove]
                rss_db.save_folders(folders)
        except Exception:
            pass

        try:
            rss_db.purge_orphans(clear_folders_if_no_feeds=True)
        except Exception:
            pass

        self.refresh()

        # If user removed everything, offer to shrink the DB on disk.
        try:
            if not (rss_db.get_feeds() or []):
                self._prompt_vacuum_if_empty_db()
        except Exception:
            pass

    def _prompt_vacuum_if_empty_db(self):
        try:
            total = 0
            try:
                total = int(rss_db.total_size_bytes() or 0)
            except Exception:
                total = 0

            # Only bother the user if the on-disk footprint is non-trivial.
            if total < 200 * 1024:
                return

            yes = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'Yes', getattr(QMessageBox, 'Yes', None))
            no = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'No', getattr(QMessageBox, 'No', None))
            msg = _(
                'All feeds have been removed.\n\n'
                'SQLite does not automatically shrink the database file after deletions.\n'
                'Do you want to compact (VACUUM) the database now to reclaim disk space?'
            )
            ans = QMessageBox.question(self, _('Compact database?'), msg, yes | no, yes)
            if ans != yes:
                return

            # Run vacuum in a background thread to avoid freezing the UI.
            try:
                self.status.setText(_('Compacting database...'))
            except Exception:
                pass
            try:
                self.set_busy(True, _('Compacting database...'))
            except Exception:
                pass

            try:
                import threading
            except Exception:
                threading = None

            before = total

            def _done(err=None):
                try:
                    self.set_busy(False)
                except Exception:
                    pass
                try:
                    after = int(rss_db.total_size_bytes() or 0)
                except Exception:
                    after = 0
                if err:
                    try:
                        self.status.setText(_('Database compact failed.'))
                    except Exception:
                        pass
                    return
                try:
                    if after and before:
                        self.status.setText(_('Database compacted: %.2f MB → %.2f MB') % (before / 1024.0 / 1024.0, after / 1024.0 / 1024.0))
                    else:
                        self.status.setText(_('Database compacted.'))
                except Exception:
                    pass

            # Use a queue + QTimer polling so the UI always clears even if
            # Dispatcher delivery is delayed.
            try:
                import queue as _queue
                result_q = _queue.Queue(maxsize=1)
            except Exception:
                result_q = None

            def _worker():
                err = None
                try:
                    try:
                        rss_db.purge_orphans(clear_folders_if_no_feeds=True)
                    except Exception:
                        pass
                    rss_db.vacuum()
                except Exception as e:
                    err = e
                if result_q is not None:
                    try:
                        result_q.put_nowait(err)
                    except Exception:
                        pass
                else:
                    # No queue available; best-effort direct dispatch.
                    try:
                        Dispatcher(_done)(err)
                    except Exception:
                        pass

            def _start_polling():
                try:
                    # Avoid multiple timers
                    try:
                        if getattr(self, '_vacuum_timer', None) is not None:
                            self._vacuum_timer.stop()
                    except Exception:
                        pass

                    t = QTimer(self)
                    t.setInterval(250)

                    def _poll():
                        if result_q is None:
                            try:
                                t.stop()
                            except Exception:
                                pass
                            return
                        try:
                            err = result_q.get_nowait()
                        except Exception:
                            return
                        try:
                            t.stop()
                        except Exception:
                            pass
                        _done(err)

                    t.timeout.connect(_poll)
                    self._vacuum_timer = t
                    t.start()
                except Exception:
                    pass

            if threading is None:
                # Fallback: do it inline
                _worker()
                try:
                    if result_q is not None:
                        try:
                            err = result_q.get_nowait()
                        except Exception:
                            err = None
                        _done(err)
                except Exception:
                    pass
            else:
                t = threading.Thread(target=_worker, name='rss_reader_vacuum', daemon=True)
                t.start()
                _start_polling()
        except Exception:
            pass

    def update_selected(self):
        fids = self.selected_feed_ids()
        if not fids:
            return
        if self.action is not None:
            self.status.setText(_('Updating %d feed(s)...') % len(fids))
            self.set_busy(True, _('Updating %d feed(s)...') % len(fids))
            self.action.update_all_feeds(feed_ids=list(fids))

    def update_all(self, feed_ids=None):
        if self.action is not None:
            try:
                if feed_ids:
                    # Update only specific feeds
                    update_count = len(feed_ids)
                else:
                    # Update all enabled feeds
                    feeds = list(rss_db.get_feeds() or [])
                    update_count = sum(1 for f in feeds if f.get('enabled', True))
            except Exception:
                update_count = 0
            self.status.setText(_('Updating %d feeds...') % update_count)
            self.set_busy(True, _('Updating %d feeds...') % update_count)
            self.action.update_all_feeds(feed_ids=feed_ids)

    def on_update_results(self, feeds_results):
        _debug('[FOLDER] on_update_results ENTER')
        # Update finished
        self.set_busy(False)

        # Preserve current selection while rebuilding the tree, otherwise an update can
        # clear the selection and wipe the item list/preview mid-reading.
        try:
            prev_selected_feed_ids = list(self.selected_feed_ids() or [])
            prev_selected_folder = str(self.selected_folder_path() or '')
        except Exception:
            prev_selected_feed_ids = []
            prev_selected_folder = ''

        self._feeds_results = feeds_results or {}

        # Cache per-feed NEW item ids for QuiteRSS-style "Show New" filtering.
        # This is intentionally ephemeral: it reflects only the most recent update run.
        try:
            new_map = {}
            for fid, r in (feeds_results or {}).items():
                if not isinstance(r, dict):
                    continue
                new_items = list(r.get('new_items') or [])
                if not new_items:
                    continue
                ids = set()
                for it in new_items:
                    if not isinstance(it, dict):
                        continue
                    iid = str(it.get('id') or it.get('link') or it.get('title') or '').strip()
                    if iid:
                        ids.add(iid)
                if ids:
                    new_map[str(fid)] = ids
            self._new_item_ids_by_feed = new_map
        except Exception:
            try:
                self._new_item_ids_by_feed = {}
            except Exception:
                pass

        # Auto-tag caching: force recompute after update results so failure tags show immediately.
        try:
            self._auto_feed_tags_cache = {}
        except Exception:
            pass
        for feed_id, r in (feeds_results or {}).items():
            if isinstance(r, dict) and r.get('ok'):
                self._items_by_feed_id[str(feed_id)] = list(r.get('items') or [])

        # Set flag to prevent accidental persist during tree rebuild
        _debug('[FOLDER] on_update_results: setting _refreshing_feeds_tree=True before load_feeds')
        try:
            self._refreshing_feeds_tree = True
        except Exception:
            pass
        try:
            self.load_feeds()
        finally:
            try:
                self._refreshing_feeds_tree = False
                _debug('[FOLDER] on_update_results: _refreshing_feeds_tree=False after load_feeds')
            except Exception:
                pass
        # Restore selection after tree rebuild
        try:
            try:
                self.feeds_tree.blockSignals(True)
            except Exception:
                pass
            try:
                if prev_selected_folder:
                    it = self._find_tree_item_by_folder(prev_selected_folder)
                    if it is not None:
                        it.setSelected(True)
                if not self.selected_feed_ids() and prev_selected_feed_ids:
                    for fid in prev_selected_feed_ids:
                        it = self._find_tree_item_by_feed(fid)
                        if it is not None:
                            it.setSelected(True)
                if not self.selected_feed_ids():
                    self._restore_feed_selection()
            finally:
                try:
                    self.feeds_tree.blockSignals(False)
                except Exception:
                    pass
        except Exception:
            try:
                self._restore_feed_selection()
            except Exception:
                pass

        self.load_items_for_selected_feed()

        errors = [r for r in (feeds_results or {}).values() if isinstance(r, dict) and not r.get('ok')]
        # If everything failed with DNS/timeout, treat as offline and avoid noisy dialogs.
        try:
            ok_count = sum(1 for r in (feeds_results or {}).values() if isinstance(r, dict) and r.get('ok'))
        except Exception:
            ok_count = 0
        offline_batch = False
        if errors and ok_count == 0:
            try:
                offline_batch = all((str((r or {}).get('error_kind') or '').strip() in ('dns-fail', 'timeout')) for r in errors)
            except Exception:
                offline_batch = False

        if offline_batch:
            try:
                self.status.setText(_('Offline (network unavailable). Will retry later.'))
            except Exception:
                pass
            return

        if errors:
            self.status.setText(_('Updated with errors (%d). Hover a feed for details.') % len(errors))
            # If popup notifications are enabled, avoid disruptive modal dialogs.
            # Users can inspect failures via feed tooltips and the Error log tab.
            try:
                if bool(plugin_prefs.get('notify_in_app_popup', False)):
                    return
            except Exception:
                pass

            # Otherwise show a concise summary and detailed errors in a dialog so the user
            # can inspect what failed and why.
            try:
                try:
                    feeds = list(rss_db.get_feeds() or [])
                except Exception:
                    feeds = []
                feeds_by_id = {str(f.get('id') or ''): f for f in (feeds or []) if str(f.get('id') or '').strip()}

                details = []
                for fid, r in (feeds_results or {}).items():
                    try:
                        if isinstance(r, dict) and not r.get('ok'):
                            f = feeds_by_id.get(str(fid) or '') or {}
                            title = r.get('title') or f.get('title') or f.get('url') or str(fid)
                            url = r.get('url') or f.get('url') or ''
                            msg = r.get('message') or r.get('error') or str(r)
                            if url:
                                details.append('%s\n  %s: %s\n  %s: %s' % (title, _('URL'), url, _('Error'), msg))
                            else:
                                details.append('%s\n  %s: %s' % (title, _('Error'), msg))
                    except Exception:
                        pass
                if details:
                    # Avoid modal QMessageBox.exec() here; it has caused rare calibre crashes
                    # (likely due to widget lifetime/re-entrancy). Use calibre's error_dialog
                    # which is safer in embed contexts.
                    try:
                        if hasattr(self, 'isVisible') and not bool(self.isVisible()):
                            return
                    except Exception:
                        pass
                    try:
                        error_dialog(
                            self,
                            _('Update completed with errors'),
                            _('Some feeds failed to update (%d). See details.') % len(details),
                            show=True,
                            det_msg='\n\n'.join(details),
                        )
                    except Exception:
                        pass
            except Exception:
                pass
        else:
            self.status.setText(_('Updated.'))

    def on_feed_selected(self):
        # During refresh/tree rebuilds, selectionChanged can fire; avoid triggering
        # an expensive item reload twice.
        try:
            if bool(getattr(self, '_refreshing_feeds_tree', False)):
                try:
                    gprefs['rss_reader_last_selected_feed_ids'] = list(self.selected_feed_ids() or [])
                    gprefs['rss_reader_last_selected_folder'] = str(self.selected_folder_path() or '')
                except Exception:
                    pass
                return
        except Exception:
            pass
        try:
            gprefs['rss_reader_last_selected_feed_ids'] = list(self.selected_feed_ids() or [])
            gprefs['rss_reader_last_selected_folder'] = str(self.selected_folder_path() or '')
        except Exception:
            pass
        self.load_items_for_selected_feed()

        # Surface feed tags when selecting a single feed.
        try:
            fids = list(self.selected_feed_ids() or [])
        except Exception:
            fids = []
        if len(fids) == 1:
            fid = str(fids[0] or '').strip()
            if fid:
                try:
                    manual = list(rss_db.get_feed_tags(fid) or [])
                except Exception:
                    manual = []
                try:
                    auto = list(self._auto_tags_for_feed(fid) or [])
                except Exception:
                    auto = []
                out = []
                seen = set()
                for t in list(manual or []) + list(auto or []):
                    tt = self._normalize_tag(t)
                    if not tt or tt in seen:
                        continue
                    seen.add(tt)
                    out.append(tt)
                try:
                    if out:
                        self.status.setText(_('Feed tags: %s') % ', '.join(out))
                    else:
                        # Avoid misleading/stale tag info when switching to a feed with no tags.
                        cur = str(self.status.text() or '')
                        if cur.startswith(_('Feed tags:')):
                            self.status.setText('')
                except Exception:
                    pass

    def _on_feed_tree_item_pressed(self, index):
        # Only treat explicit mouse/touch selection as a trigger.
        # This prevents auto-clearing badges during refresh/tree rebuilds.
        try:
            self._feed_selection_user_initiated = True
        except Exception:
            pass

    def _auto_mark_selected_feed_seen_if_user_clicked(self):
        # NOTE: Auto-marking a feed as read on selection is no longer acceptable UX
        # (it makes it too easy to lose unread state). Keep this method as a
        # harmless no-op for backward compatibility with older code paths.
        try:
            self._feed_selection_user_initiated = False
        except Exception:
            pass
        return

        if bool(getattr(self, '_auto_mark_seen_guard', False)):
            return

        # Do not auto-mark when selecting folders.
        try:
            for idx in self.feeds_tree.selectionModel().selectedIndexes():
                item = self.feeds_tree.model().itemFromIndex(idx)
                d = item.data(ROLE_USER)
                if isinstance(d, dict) and d.get('type') == 'folder':
                    return
        except Exception:
            pass

        fids = list(self.selected_feed_ids() or [])
        if not fids:
            return

        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}

        changed = False
        changed_feed_ids = []
        for fid in fids:
            # Prefer in-memory items (fast), fallback to DB cache.
            items = list(self._items_by_feed_id.get(fid, []) or [])
            if not items:
                try:
                    c = rss_db.get_feed_cache(fid) or {}
                    items = list((c or {}).get('items', []) or [])
                except Exception:
                    items = []

            ids = [str(it.get('id') or '') for it in (items or []) if str(it.get('id') or '').strip()]
            if not ids:
                continue

            seen_set = set(str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip())
            if all(i in seen_set for i in ids):
                continue

            merged = list(seen_set.union(ids))
            try:
                rss_db.set_seen_item_ids(fid, merged)
                seen_map[fid] = merged
                changed = True
                changed_feed_ids.append(fid)
            except Exception:
                pass

        if not changed:
            return

        # Update badges for the affected feeds only, without rebuilding the entire tree.
        # This prevents the visible "jump" and selection loss during reading.
        self._auto_mark_seen_guard = True
        try:
            self._update_feed_badges(changed_feed_ids)
        finally:
            self._auto_mark_seen_guard = False

    def _update_feed_badges(self, feed_ids):
        """Update badges for specific feeds without rebuilding the tree."""
        if not feed_ids:
            return

        try:
            # Get unread counts for the affected feeds.
            # Derive unread from seen_item_ids + cached items (not feeds_results.new_count).
            try:
                seen_map = dict(rss_db.get_seen_item_ids_map() or {})
            except Exception:
                seen_map = {}

            unread_by_feed = {}
            for fid in feed_ids:
                try:
                    seen_set = set(str(x) for x in (seen_map.get(str(fid), []) or []) if str(x).strip())
                except Exception:
                    seen_set = set()

                try:
                    items = list(self._items_by_feed_id.get(str(fid), []) or [])
                except Exception:
                    items = []
                if not items:
                    try:
                        c = rss_db.get_feed_cache(str(fid)) or {}
                        items = list((c or {}).get('items', []) or [])
                    except Exception:
                        items = []

                try:
                    if items:
                        unread = sum(
                            1
                            for it in (items or [])
                            if str(it.get('id') or '').strip() and (str(it.get('id') or '').strip() not in seen_set)
                        )
                    else:
                        unread = 0
                except Exception:
                    unread = 0
                unread_by_feed[str(fid)] = int(unread or 0)

            # Update each feed item's text to show/hide unread count
            try:
                feeds = list(rss_db.get_feeds() or [])
            except Exception:
                feeds = []
            feeds_by_id = {str(f.get('id') or ''): f for f in feeds if str(f.get('id') or '').strip()}

            for fid in feed_ids:
                item = self._find_tree_item_by_feed(fid)
                if item is None:
                    continue

                feed = feeds_by_id.get(fid, {})
                title = feed.get('title', '') or feed.get('url', '') or fid
                unread = unread_by_feed.get(fid, 0)

                # Maintain cached unread totals for fast folder badge updates.
                try:
                    old_unread = int(getattr(self, '_feed_unread_by_id', {}).get(fid, 0))
                except Exception:
                    old_unread = None
                try:
                    if getattr(self, '_feed_unread_by_id', None) is not None:
                        self._feed_unread_by_id[fid] = int(unread or 0)
                except Exception:
                    pass
                if old_unread is not None:
                    try:
                        delta = int(unread or 0) - int(old_unread or 0)
                    except Exception:
                        delta = 0
                    if delta:
                        try:
                            folder_path = str(getattr(self, '_folder_by_feed', {}).get(fid, '') or '').strip().strip('/')
                        except Exception:
                            folder_path = ''
                        # Update root and all prefixes for this feed's folder.
                        try:
                            m = getattr(self, '_folder_total_unread', None)
                            if isinstance(m, dict):
                                m[''] = int(m.get('', 0)) + int(delta)
                                if folder_path:
                                    parts = [p for p in folder_path.split('/') if p]
                                    for i in range(1, len(parts) + 1):
                                        prefix = '/'.join(parts[:i])
                                        m[prefix] = int(m.get(prefix, 0)) + int(delta)
                        except Exception:
                            pass

                if unread > 0:
                    item.setText(f'{title} ({unread})')
                else:
                    item.setText(title)

                # Update parent folder badges
                parent = item.parent()
                if parent is not None:
                    self._update_folder_badge(parent)
        except Exception:
            pass

    def _update_folder_badge(self, folder_item):
        """Recursively update folder badge by summing unread counts of contained feeds."""
        try:
            data = folder_item.data(ROLE_USER)
            if not isinstance(data, dict) or data.get('type') != 'folder':
                return

            folder_path = str(data.get('path') or '')
            # Fast path: use cached totals computed during tree build.
            try:
                total_unread = int(getattr(self, '_folder_total_unread', {}).get(folder_path, 0))
            except Exception:
                total_unread = 0
                for fid in self._folder_to_feed_ids.get(folder_path, []):
                    try:
                        # Prefer the cached unread-by-feed map (kept in sync by load_feeds/_update_feed_badges)
                        m = getattr(self, '_feed_unread_by_id', {})
                        if isinstance(m, dict):
                            total_unread += int(m.get(str(fid), 0) or 0)
                        else:
                            r = self._feeds_results.get(fid) or {}
                            total_unread += int((r or {}).get('new_count') or 0)
                    except Exception:
                        pass

            name = folder_path.rpartition('/')[2] or folder_path
            if total_unread > 0:
                folder_item.setText(f'{name} ({total_unread})')
                try:
                    from qt.core import QBrush, QColor, QApplication
                except ImportError:
                    from PyQt5.Qt import QBrush, QColor, QApplication
                palette = QApplication.instance().palette()
                is_dark = palette.color(palette.Window).value() < 128
                if is_dark:
                    color = QColor('#00E6E6')
                else:
                    color = palette.color(palette.Link)
                folder_item.setForeground(QBrush(color))
            else:
                folder_item.setText(name)
                folder_item.setForeground(QBrush())

            # Update parent folder if this folder is nested
            parent = folder_item.parent()
            if parent is not None:
                self._update_folder_badge(parent)
        except Exception:
            pass

    def load_items_for_selected_feed(self):
        _t0 = time.perf_counter()
        fids = list(self.selected_feed_ids() or [])
        if not fids:
            try:
                self.items_table.setRowCount(0)
                self.preview.setHtml('')
            except Exception:
                pass
            return

        _debug('load_items_for_selected_feed: %d feeds' % len(fids))
        feeds = list(rss_db.get_feeds() or [])
        feeds_by_id = {str(f.get('id') or ''): f for f in feeds}

        # combine items across selected feeds
        items = []
        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}

        # Load starred items for selected feeds
        try:
            starred_map = rss_db.get_starred_items_map()
        except Exception:
            starred_map = {}

        # Manual tags for selected feeds (keyed by item_id). Auto-tags are computed on the fly.
        tags_map_by_feed = {}
        try:
            for fid in fids:
                try:
                    tags_map_by_feed[str(fid)] = dict(rss_db.get_item_tags_map(fid) or {})
                except Exception:
                    tags_map_by_feed[str(fid)] = {}
        except Exception:
            tags_map_by_feed = {}

        for fid in fids:
            starred_set = starred_map.get(str(fid), set())
            for it in (self._items_by_feed_id.get(fid, []) or []):
                d = dict(it)
                d['_feed_id'] = fid
                d['_feed_title'] = (feeds_by_id.get(fid) or {}).get('title') or (feeds_by_id.get(fid) or {}).get('url') or ''
                d['_feed_url'] = (feeds_by_id.get(fid) or {}).get('url') or ''
                # Check if starred
                iid = str(d.get('id') or d.get('link') or d.get('title') or '')
                d['_starred'] = (iid in starred_set)
                try:
                    nm = getattr(self, '_new_item_ids_by_feed', {}) or {}
                    d['_new'] = bool(iid and (iid in set(nm.get(str(fid), set()) or set())))
                except Exception:
                    d['_new'] = False
                items.append(d)

        # Ensure items are newest-first in the backing list as a stable default.
        try:
            items.sort(key=lambda x: int(x.get('published_ts') or iso_to_ts(x.get('published') or '') or 0), reverse=True)
        except Exception:
            pass

        # When aggregating many feeds (e.g. selecting the root "Feeds"
        # folder), rendering *all* items can be very expensive in Qt.
        # Apply a soft cap so the UI stays responsive.
        try:
            max_items = int(plugin_prefs.get('max_items_per_selection', 2000) or 2000)
        except Exception:
            max_items = 2000
        try:
            total_items = len(items)
        except Exception:
            total_items = 0
        if total_items > max_items:
            try:
                items = items[:max_items]
            except Exception:
                pass
            try:
                self.status.setText(_('Showing newest %d of %d items. Refine your selection to see more.') % (max_items, total_items))
            except Exception:
                pass

        # Critical: disable sorting while populating, otherwise QTableWidget will reorder rows
        # while we're inserting, causing mixed/duplicated rows.
        was_sorting = False
        try:
            was_sorting = bool(self.items_table.isSortingEnabled())
            self.items_table.setSortingEnabled(False)
        except Exception:
            pass

        try:
            self.items_table.setRowCount(len(items))
            # Pre-import html once (not per-row)
            try:
                import html as _html
            except Exception:
                _html = None

            # Check if auto-tagging is enabled (avoid heavy computation if disabled)
            auto_tagging_enabled = bool(plugin_prefs.get('auto_tagging_enabled', True))

            for row, it in enumerate(items):
                try:
                    title = (_html.unescape(it.get('title') or '') if _html else (it.get('title') or ''))
                except Exception:
                    title = it.get('title') or ''
                if len(fids) > 1:
                    ft = it.get('_feed_title') or ''
                    if ft:
                        title = f'[{ft}] {title}'
                published = it.get('published') or ''
                ts = int(it.get('published_ts') or iso_to_ts(published) or 0)
                published_disp = format_published_display(ts, published)

                try:
                    author = (_html.unescape(it.get('author') or '') if _html else (it.get('author') or ''))
                except Exception:
                    author = it.get('author') or ''
                try:
                    author = str(author or '').strip()
                except Exception:
                    author = ''

                # Star column
                is_starred = bool(it.get('_starred', False))
                t_star = QTableWidgetItem('★' if is_starred else '')
                try:
                    t_star.setTextAlignment(Qt.AlignmentFlag.AlignCenter)
                except Exception:
                    try:
                        t_star.setTextAlignment(Qt.AlignCenter)
                    except Exception:
                        pass
                # Add tooltip to indicate the star column is clickable
                try:
                    t_star.setToolTip('Click to star/unstar this item')
                except Exception:
                    pass
                if is_starred:
                    try:
                        t_star.setForeground(QColor(255, 180, 0))  # Gold color for star
                    except Exception:
                        pass

                t0 = QTableWidgetItem(title)
                t_author = QTableWidgetItem(author)
                t1 = SortKeyTableWidgetItem(published_disp, ts)
                # Tags (manual + auto)
                tags_text = ''
                try:
                    tags = self._tags_for_item(it, tags_map_by_feed=tags_map_by_feed)
                    it['_tags'] = list(tags)
                    tags_text = ', '.join(list(tags) or [])
                except Exception:
                    tags_text = ''
                t2 = QTableWidgetItem(tags_text)
                # Tooltip: show manual vs auto tags separately
                try:
                    fid0 = str(it.get('_feed_id') or '').strip()
                except Exception:
                    fid0 = ''
                try:
                    iid0 = str(it.get('id') or '').strip()
                except Exception:
                    iid0 = ''

                manual_tags = []
                try:
                    if fid0 and iid0 and isinstance(tags_map_by_feed, dict):
                        m = (tags_map_by_feed.get(fid0) or {}).get(iid0)
                        if isinstance(m, list):
                            manual_tags = m
                except Exception:
                    manual_tags = []

                auto_tags = []
                try:
                    auto_tags = list(self._auto_tags_for_item(it) or [])
                except Exception:
                    auto_tags = []

                try:
                    lines = []
                    if manual_tags:
                        mt = [self._normalize_tag(x) for x in (manual_tags or [])]
                        mt = [x for x in mt if x]
                        if mt:
                            lines.append(_('Manual: %s') % ', '.join(mt))
                    if auto_tags:
                        at = [self._normalize_tag(x) for x in (auto_tags or [])]
                        at = [x for x in at if x]
                        if at:
                            lines.append(_('Auto: %s') % ', '.join(at))
                    if lines:
                        t2.setToolTip('\n'.join(lines))
                except Exception:
                    pass
                try:
                    if published:
                        t1.setToolTip(str(published))
                except Exception:
                    pass
                t_star.setData(ROLE_USER, it)
                t0.setData(ROLE_USER, it)
                t_author.setData(ROLE_USER, it)
                t1.setData(ROLE_USER, it)
                t2.setData(ROLE_USER, it)

                # Bold unread items (QuiteRSS-style)
                try:
                    fid1 = str(it.get('_feed_id') or '')
                    iid1 = str(it.get('id') or '')
                    seen_set = set(seen_map.get(fid1, []) or [])
                    is_unread = bool(iid1) and (iid1 not in seen_set)
                    try:
                        it['_unread'] = bool(is_unread)
                    except Exception:
                        pass
                    try:
                        if is_unread:
                            ic = self._unread_dot_icon()
                            if ic is not None:
                                t0.setIcon(ic)
                        else:
                            # Use transparent placeholder to maintain spacing
                            ic = self._transparent_placeholder_icon()
                            if ic is not None:
                                t0.setIcon(ic)
                    except Exception:
                        pass
                    if is_unread:
                        try:
                            f0 = QFont(t0.font())
                            f0.setBold(True)
                            t0.setFont(f0)
                        except Exception:
                            pass
                        try:
                            fa = QFont(t_author.font())
                            fa.setBold(True)
                            t_author.setFont(fa)
                        except Exception:
                            pass
                        try:
                            fp = QFont(t1.font())
                            fp.setBold(True)
                            t1.setFont(fp)
                        except Exception:
                            pass
                        try:
                            ft = QFont(t2.font())
                            ft.setBold(True)
                            t2.setFont(ft)
                        except Exception:
                            pass
                except Exception:
                    pass
                self.items_table.setItem(row, 0, t_star)
                self.items_table.setItem(row, 1, t0)
                self.items_table.setItem(row, 2, t_author)
                self.items_table.setItem(row, 3, t1)
                self.items_table.setItem(row, 4, t2)
        finally:
            try:
                self.items_table.setSortingEnabled(was_sorting)
            except Exception:
                pass

        # Ensure star column stays narrow (fix cases where autosize makes it huge)
        try:
            self.items_table.setColumnWidth(0, 30)
        except Exception:
            pass

        # Apply default/persisted sort (newest first)
        try:
            sc = gprefs.get('rss_reader_sort_column', None)
            so = gprefs.get('rss_reader_sort_order', None)
            if sc is None or so is None:
                sc, so = 3, int(Qt.SortOrder.DescendingOrder)

            # Migration: older versions had 4 columns and Published was index 2.
            try:
                prev_cols = gprefs.get('rss_reader_items_table_colcount', None)
            except Exception:
                prev_cols = None
            try:
                if int(prev_cols or 0) == 4 and int(sc) == 2:
                    sc = 3
            except Exception:
                pass
            try:
                gprefs['rss_reader_items_table_colcount'] = int(self.items_table.columnCount())
            except Exception:
                pass
            self.items_table.sortItems(int(sc), int(so))
        except Exception:
            pass

        # Respect saved widths; if none present, auto-size and persist
        try:
            iw = gprefs.get('rss_reader_items_table_widths', {}) or {}
        except Exception:
            iw = {}
        if not iw:
            self.items_table.resizeColumnsToContents()
            # Star column: small fixed width
            try:
                self.items_table.setColumnWidth(0, 30)
            except Exception:
                pass
            # Title column: wider
            try:
                self.items_table.setColumnWidth(1, max(350, self.items_table.columnWidth(1)))
            except Exception:
                pass
            try:
                # Tags column: reasonable width
                self.items_table.setColumnWidth(4, max(120, self.items_table.columnWidth(4)))
            except Exception:
                pass
            # persist the widths now
            try:
                w = {}
                for col in range(self.items_table.columnCount()):
                    w[str(col)] = self.items_table.columnWidth(col)
                gprefs['rss_reader_items_table_widths'] = w
            except Exception:
                pass
        else:
            # Apply saved widths while forcing Interactive mode briefly so widths stick.
            try:
                header = self.items_table.horizontalHeader()
                modes = {}
                try:
                    cnt = header.count()
                except Exception:
                    try:
                        cnt = self.items_table.columnCount()
                    except Exception:
                        cnt = 0
                for c in range(cnt):
                    try:
                        modes[c] = header.sectionResizeMode(c)
                    except Exception:
                        modes[c] = None
                for c in range(cnt):
                    try:
                        header.setSectionResizeMode(c, header.ResizeMode.Interactive)
                    except Exception:
                        try:
                            header.setSectionResizeMode(c, header.Interactive)
                        except Exception:
                            pass
                for col, width in iw.items():
                    try:
                        self.items_table.setColumnWidth(int(col), int(width))
                    except Exception:
                        pass
                for c, m in modes.items():
                    if m is None:
                        continue
                    try:
                        header.setSectionResizeMode(c, m)
                    except Exception:
                        try:
                            if m == getattr(header, 'Fixed', None):
                                header.setSectionResizeMode(c, getattr(header, 'Fixed', None))
                            elif m == getattr(header, 'Stretch', None):
                                header.setSectionResizeMode(c, getattr(header, 'Stretch', None))
                            elif m == getattr(header, 'ResizeToContents', None):
                                header.setSectionResizeMode(c, getattr(header, 'ResizeToContents', None))
                            elif m == getattr(header, 'Interactive', None):
                                header.setSectionResizeMode(c, getattr(header, 'Interactive', None))
                        except Exception:
                            pass
            except Exception:
                # fallback: naive apply
                try:
                    for col, width in iw.items():
                        try:
                            self.items_table.setColumnWidth(int(col), int(width))
                        except Exception:
                            pass
                except Exception:
                    pass

        # Restore item selection by stable id ONLY if feed context matches exactly
        try:
            if self.items_table.rowCount() and not self.items_table.selectionModel().selectedRows():
                want_item_id = str(gprefs.get('rss_reader_last_selected_item_id', '') or '')
                want_feed_id = str(gprefs.get('rss_reader_last_selected_item_feed_id', '') or '')
                prev_fids = list(gprefs.get('rss_reader_last_selected_feed_ids', []) or [])
                restored = False
                if want_item_id and want_feed_id:
                    # Only restore if both item and feed id match exactly
                    for r in range(self.items_table.rowCount()):
                        it0 = self.items_table.item(r, 0)
                        data = it0.data(ROLE_USER) if it0 is not None else None
                        if not isinstance(data, dict):
                            continue
                        fid = str(data.get('_feed_id') or '')
                        iid = str(data.get('id') or data.get('link') or data.get('title') or '')
                        if iid == want_item_id and fid == want_feed_id:
                            self.items_table.selectRow(r)
                            restored = True
                            break
                # Do NOT restore by id if feed context is ambiguous or missing
                # (prevents wrong-article selection under root/folder views)
        except Exception:
            pass

        # Select first item by default (or restore last selected)
        try:
            if self.items_table.rowCount() and not self.items_table.selectionModel().selectedRows():
                # Prefer the newest unread item (sorted desc), else last-saved, else first row
                unread_row = None
                for r in range(self.items_table.rowCount()):
                    it0 = self.items_table.item(r, 0)
                    data = it0.data(ROLE_USER) if it0 is not None else None
                    if isinstance(data, dict) and data.get('id'):
                        fid = str(data.get('_feed_id') or '')
                        seen_set = set(seen_map.get(fid, []) or [])
                        if data.get('id') not in seen_set:
                            unread_row = r
                            break
                if unread_row is not None:
                    self.items_table.selectRow(unread_row)
                else:
                    sel = gprefs.get('rss_reader_last_selected_item_row', None)
                    if sel is not None and 0 <= int(sel) < self.items_table.rowCount():
                        self.items_table.selectRow(int(sel))
                    else:
                        self.items_table.selectRow(0)
        except Exception:
            pass

        # Preview current selection
        if self.items_table.rowCount():
            try:
                self.on_item_selected()
            except Exception:
                pass
        else:
            self.preview.setHtml('')
        _debug('load_items_for_selected_feed done in %.3fs' % (time.perf_counter() - _t0))

    def selected_item(self):
        rows = self.items_table.selectionModel().selectedRows()
        if not rows:
            return None
        row = rows[0].row()
        it0 = self.items_table.item(row, 0)
        if it0 is None:
            return None
        data = it0.data(ROLE_USER)
        if isinstance(data, dict):
            # Remember last selected item for this feed
            try:
                gprefs['rss_reader_last_selected_item_row'] = row
            except Exception:
                pass
            # Also remember a stable identifier so we can restore selection after refresh
            try:
                gprefs['rss_reader_last_selected_item_id'] = str(data.get('id') or data.get('link') or data.get('title') or '')
                gprefs['rss_reader_last_selected_item_feed_id'] = str(data.get('_feed_id') or '')
            except Exception:
                pass
            return data
        return None

    def focus_item(self, feed_id, item_id):
        try:
            fid = str(feed_id or '').strip()
        except Exception:
            fid = ''
        try:
            iid = str(item_id or '').strip()
        except Exception:
            iid = ''
        if not fid or not iid:
            return

        # Hint the existing selection-restore logic so a feed change will land on the target.
        try:
            gprefs['rss_reader_last_selected_item_id'] = iid
            gprefs['rss_reader_last_selected_item_feed_id'] = fid
            gprefs['rss_reader_last_selected_feed_ids'] = [fid]
            gprefs['rss_reader_last_selected_folder'] = ''
        except Exception:
            pass

        # Select the feed in the tree (this normally triggers load_items_for_selected_feed via selectionChanged).
        try:
            tree_item = self._find_tree_item_by_feed(fid)
        except Exception:
            tree_item = None
        if tree_item is not None:
            try:
                model = self.feeds_tree.model()
                idx = model.indexFromItem(tree_item)
                sel_model = self.feeds_tree.selectionModel()
                try:
                    sel_model.clearSelection()
                except Exception:
                    pass
                try:
                    sel_model.select(idx, sel_model.SelectionFlag.ClearAndSelect)
                except Exception:
                    try:
                        sel_model.select(idx, sel_model.SelectionFlag.Select)
                    except Exception:
                        pass
                try:
                    self.feeds_tree.scrollTo(idx)
                except Exception:
                    pass
            except Exception:
                pass

        # Ensure items are loaded (in case signals are suppressed during refresh).
        try:
            self.load_items_for_selected_feed()
        except Exception:
            pass

        # Select the specific item row.
        try:
            table = self.items_table
            for r in range(table.rowCount()):
                it0 = table.item(r, 0)
                data = it0.data(ROLE_USER) if it0 is not None else None
                if not isinstance(data, dict):
                    continue
                if str(data.get('_feed_id') or '') != fid:
                    continue
                if str(data.get('id') or '') != iid:
                    continue
                try:
                    table.selectRow(r)
                except Exception:
                    pass
                try:
                    table.scrollToItem(table.item(r, 1) or it0)
                except Exception:
                    pass
                try:
                    self.on_item_selected()
                except Exception:
                    pass
                return
        except Exception:
            pass
        try:
            if getattr(self, 'items_table', None) is not None:
                self.items_table.setFocus(Qt.FocusReason.MouseFocusReason)
        except Exception:
            try:
                if getattr(self, 'items_table', None) is not None:
                    self.items_table.setFocus()
            except Exception:
                pass

    def on_item_selected(self):
        it = self.selected_item() or {}
        self._current_item = it
        try:
            import html as _html
        except Exception:
            _html = None

        try:
            title = (_html.unescape(it.get('title') or '') if _html else (it.get('title') or ''))
        except Exception:
            title = it.get('title') or ''
        link = it.get('link') or ''

        # Update compact current-article label
        try:
            self._update_current_article_label(it)
        except Exception:
            pass

        # Update clickable URL in preview footer
        try:
            if getattr(self, 'preview_url_label', None) is not None:
                if link:
                    try:
                        import html as _htmlmod
                        href = _htmlmod.escape(str(link), quote=True)
                        disp = str(link)
                        max_len = 90
                        if len(disp) > max_len:
                            disp = disp[:55] + '…' + disp[-(max_len - 56):]
                        disp = _htmlmod.escape(disp)
                        self.preview_url_label.setText('<a href="%s">%s</a>' % (href, disp))
                    except Exception:
                        self.preview_url_label.setText(str(link))
                else:
                    self.preview_url_label.setText('')
        except Exception:
            pass
        try:
            # Prefer full HTML content when feeds provide it (e.g. content:encoded).
            _content = it.get('content') or ''
            _summary = it.get('summary') or ''
            chosen = _content if _content else _summary
            summary = (_html.unescape(chosen or '') if _html else (chosen or ''))
        except Exception:
            summary = (it.get('content') or it.get('summary') or '')
        enclosures = it.get('enclosures') or []

        # Keep preview whitespace tight and consistent (Qt's default HTML margins can be large)
        show_images = True
        try:
            show_images = bool(plugin_prefs.get('load_images_in_preview', True))
        except Exception:
            show_images = True
        try:
            autofit = bool(plugin_prefs.get('autofit_images', True))
        except Exception:
            autofit = True

        # Decide image CSS based on both flags: show_images controls visibility/download,
        # autofit controls whether images are scaled down to fit the preview width.
        # Use calibre's proven approach: document-level default stylesheet for
        # aspect-ratio-preserving containment (max-width + height:auto + display:block).
        if not show_images:
            img_css = 'img{display:none !important;}'
        else:
            img_css = 'img{max-width:100% !important;height:auto !important;width:auto !important;display:block !important;}' if autofit else ''

        # Apply document-level default stylesheet (calibre-style) for reliable
        # aspect-ratio-preserving image containment in QTextBrowser.
        try:
            if autofit and show_images:
                self.preview.document().setDefaultStyleSheet(
                    'img { max-width: 100%; height: auto; display: block; }'
                )
            else:
                self.preview.document().setDefaultStyleSheet('')
        except Exception:
            pass

        html = (
            '<html><head><meta charset="utf-8">'
            '<style>'
            'body{margin:0;padding:6px;} '
            'h1,h2,h3,h4{margin:0 0 .4em 0;} '
            'p{margin:.35em 0;}'
            + img_css +
            '</style>'
            '</head><body>'
        )
        html += '<h3>%s</h3>' % title
        if link:
            html += '<p><a href="%s">%s</a></p>' % (link, link)

        # (Removed) full-article fetch engine banner

        # If images are globally disabled, show a small notice so users know why
        # images are not displayed or downloaded.
        try:
            if not show_images:
                html += '<p style="color:gray;font-style:italic">%s</p>' % _('Images are disabled in settings (Preview will not download images).')
        except Exception:
            pass

        # Basic audio support: collect podcast-style enclosures for native player.
        try:
            audio_urls = []
            for e in (enclosures or []):
                if not isinstance(e, dict):
                    continue
                u = (e.get('url') or '').strip()
                if not u:
                    continue
                t = (e.get('type') or '').strip().lower()
                u_l = u.lower()
                if t.startswith('audio/') or u_l.endswith(('.mp3', '.m4a', '.aac', '.ogg', '.opus', '.wav')):
                    audio_urls.append(u)
            # De-dup while preserving order
            seen = set()
            audio_urls = [u for u in audio_urls if not (u in seen or seen.add(u))]
        except Exception:
            audio_urls = []

        # --- Normalize summary HTML before inserting into preview ---
        norm_summary = normalize_summary_to_html(summary) if summary else ''
        fid = str(it.get('_feed_id') or '')
        try:
            feed_info = self._all_feeds.get(fid, {})
            allow_images = bool(feed_info.get('download_images', True))
        except Exception:
            allow_images = True
        if allow_images:
            try:
                norm_summary = _normalize_images_for_preview(norm_summary, base_url=link or '')
            except Exception:
                pass
        else:
            # Remove all <img> tags
            try:
                import re
                norm_summary = re.sub(r'<img[^>]*>', '', norm_summary, flags=re.IGNORECASE)
            except Exception:
                pass
        if norm_summary:
            html += norm_summary

        # Image enclosures (common pattern: RSS <enclosure type="image/..."> without <img> in description)
        try:
            if show_images and allow_images:
                from urllib.parse import urljoin as _urljoin
                img_urls = []
                for e in (enclosures or []):
                    if not isinstance(e, dict):
                        continue
                    u = (e.get('url') or '').strip()
                    if not u:
                        continue
                    t = (e.get('type') or '').strip().lower()
                    u_l = u.lower()
                    if t.startswith('image/') or u_l.endswith(('.jpg', '.jpeg', '.png', '.gif', '.webp', '.bmp', '.svg', '.avif')):
                        try:
                            u = _sanitize_url_for_fetch(_urljoin(link or '', u))
                        except Exception:
                            try:
                                u = _sanitize_url_for_fetch(u)
                            except Exception:
                                pass
                        img_urls.append(u)
                # De-dup while preserving order
                seen = set()
                img_urls = [u for u in img_urls if u and not (u in seen or seen.add(u))]
                if img_urls:
                    # Avoid duplicates if the URL is already in the summary HTML
                    summary_html = norm_summary or ''
                    for u in img_urls[:8]:
                        if u in summary_html:
                            continue
                        html += '<div style="margin-top:.5em"><img src="%s"/></div>' % u
        except Exception:
            pass

        # Document enclosures (EPUB, PDF, etc.) - commonly used by eBook feeds like Standard Ebooks
        try:
            doc_links = []
            for e in (enclosures or []):
                if not isinstance(e, dict):
                    continue
                u = (e.get('url') or '').strip()
                if not u:
                    continue
                t = (e.get('type') or '').strip().lower()
                u_l = u.lower()
                try:
                    u_path_l = u.split('?', 1)[0].lower()
                except Exception:
                    u_path_l = u_l
                # Include common document/eBook MIME types and file extensions
                doc_types = {
                    'application/epub+zip', 'application/pdf', 'application/x-pdf',
                    'application/vnd.amazon.ebook', 'application/vnd.kobo.ebook',
                    'application/x-mobipocket-ebook', 'application/x-mobi',
                    'application/kepub+zip'
                }
                doc_exts = ('.epub', '.pdf', '.mobi', '.azw', '.azw3', '.ibooks')
                if t in doc_types or t.startswith('application/x-ebook') or u_path_l.endswith(doc_exts):
                    try:
                        u = _sanitize_url_for_fetch(u)
                    except Exception:
                        pass
                    # Extract filename from URL or generate one
                    filename = u.split('/')[-1].split('?')[0]
                    if not filename:
                        filename = 'ebook'
                    # Get file size if available
                    length = (e.get('length') or '').strip()
                    size_str = ''
                    if length:
                        try:
                            size_bytes = int(length)
                            if size_bytes > 1024 * 1024:
                                size_str = ' (%.1f MB)' % (size_bytes / (1024 * 1024))
                            elif size_bytes > 1024:
                                size_str = ' (%.1f KB)' % (size_bytes / 1024)
                            else:
                                size_str = ' (%d B)' % size_bytes
                        except Exception:
                            pass
                    doc_links.append((filename, u, size_str))
            # De-dup while preserving order
            seen = set()
            doc_links = [(f, u, s) for f, u, s in doc_links if u and not (u in seen or seen.add(u))]
            if doc_links:
                html += '<p><b>%s</b></p>' % _('Downloads')
                for filename, url, size_str in doc_links[:8]:
                    html += '<p><a href="%s">%s%s</a></p>' % (url, filename, size_str)
        except Exception:
            pass

        # Optional: attempt to fetch full article HTML for richer preview (backgrounded)
        # Save base HTML (without article fetch) so background updater can append results later
        try:
            self._preview_base_html = html
        except Exception:
            self._preview_base_html = html

        # Determine if background article fetching is enabled
        try:
            fetch_enabled = bool(plugin_prefs.get('preview_fetch_article_content', False))
        except Exception:
            fetch_enabled = False

        if fetch_enabled and link:
            try:
                # Avoid redundant network fetches when the preview is re-rendered
                # multiple times for the same selected item (e.g. UI refreshes).
                try:
                    inflight = bool(getattr(self, '_preview_article_fetch_inflight', False))
                    inflight_link = str(getattr(self, '_preview_article_fetch_inflight_link', '') or '')
                except Exception:
                    inflight = False
                    inflight_link = ''

                if inflight and inflight_link and str(link) == inflight_link:
                    try:
                        _debug('[QUEUE ARTICLE FETCH] already in-flight; queuing duplicate fetch for link=%s feed_id=%s' % (str(link), str(fid)))
                    except Exception:
                        pass

                try:
                    self._preview_article_fetch_inflight = True
                    self._preview_article_fetch_inflight_link = str(link)
                except Exception:
                    pass

                # Bump generation token to allow cancelling/stale-result detection
                try:
                    self._preview_generation = int(getattr(self, '_preview_generation', 0) or 0) + 1
                except Exception:
                    self._preview_generation = 1
                gen = int(self._preview_generation)
                timeout = int(plugin_prefs.get('timeout_seconds', 25) or 25)

                try:
                    # Pass feed id so the worker can decide whether to use recipe engine
                    _debug('[QUEUE ARTICLE FETCH] link=%s gen=%s timeout=%s feed_id=%s' % (str(link), str(gen), str(timeout), str(fid)))
                    self._queue_article_fetch(link, gen, timeout, feed_id=fid)
                except Exception:
                    _debug('[QUEUE ARTICLE FETCH] exception: %s' % traceback.format_exc()[:500])
                    try:
                        if str(getattr(self, '_preview_article_fetch_inflight_link', '') or '') == str(link):
                            self._preview_article_fetch_inflight = False
                            self._preview_article_fetch_inflight_link = ''
                    except Exception:
                        pass
            except Exception:
                pass
        html += '</body></html>'
        try:
            # Base URL helps resolve relative <img src> etc.
            if link:
                self.preview.document().setBaseUrl(QUrl(link))
        except Exception:
            pass
        # Cancel/ignore any late image downloads from the previously selected item.
        try:
            if hasattr(self.preview, 'start_new_page'):
                self.preview.start_new_page()
        except Exception:
            pass

        # Make feed URL available on preview for 'Copy feed URL' context action
        try:
            try:
                self.preview._current_feed_url = str(it.get('_feed_url') or '')
            except Exception:
                self.preview._current_feed_url = ''
        except Exception:
            pass

        # Make page base URL available to PreviewBrowser (used as Referer for hotlink-protected images)
        try:
            self.preview._page_base_url = str(link or '')
        except Exception:
            pass

        # Extract image URLs from the generated HTML so we can offer reload/copy actions.
        try:
            import re, urllib.parse

            def _first_url_from_srcset(ss):
                try:
                    ss = (ss or '').strip()
                    if not ss:
                        return ''
                    first = ss.split(',')[0].strip()
                    if not first:
                        return ''
                    return first.split()[0].strip()
                except Exception:
                    return ''

            imgs = []
            try:
                # src + common lazy-load attributes
                for m in re.finditer(r'<img[^>]+(?:src|data-src|data-original|data-lazy-src)=["\']([^"\']+)["\']', html, re.IGNORECASE):
                    src = m.group(1)
                    if src:
                        imgs.append(src)
                # srcset on img/source
                for m in re.finditer(r'(?:srcset|data-srcset)=["\']([^"\']+)["\']', html, re.IGNORECASE):
                    ss = m.group(1)
                    first = _first_url_from_srcset(ss)
                    if first:
                        imgs.append(first)
            except Exception:
                imgs = []

            resolved = []
            base = link or ''
            for s in imgs:
                try:
                    r = urllib.parse.urljoin(base, s)
                    r = _sanitize_url_for_fetch(r)
                    resolved.append(r)
                except Exception:
                    try:
                        resolved.append(_sanitize_url_for_fetch(str(s)))
                    except Exception:
                        pass

            try:
                self.preview._current_img_urls = list(dict.fromkeys(resolved))
            except Exception:
                try:
                    self.preview._current_img_urls = list(resolved)
                except Exception:
                    self.preview._current_img_urls = []
        except Exception:
            try:
                self.preview._current_img_urls = []
            except Exception:
                pass

        # Compute and show preview stats (chars, words, images)
        try:
            # Use the actual preview HTML so stats reflect what the user sees.
            # (Some feeds have empty/short summaries while Preview shows full content.)
            try:
                source_html = html
            except Exception:
                source_html = ''
            if not source_html:
                source_html = normalize_summary_to_html(summary) if summary else ''
            text_only = self._strip_html_to_text(source_html)
            # Safety cutoff for extremely large content
            try:
                if len(text_only or '') > 1024 * 1024:
                    chars = len((text_only or '')[:1024 * 1024])
                    words = len([w for w in (text_only or '')[:1024 * 1024].split() if w])
                    large = True
                else:
                    chars = len(text_only or '')
                    words = len([w for w in (text_only or '').split() if w])
                    large = False
            except Exception:
                chars = len(text_only or '')
                words = len([w for w in (text_only or '').split() if w])
                large = False
            images = len(getattr(self.preview, '_current_img_urls', []) or [])
            stats_str = _('%(chars)d chars, %(words)d words, %(images)d images') % {'chars': chars, 'words': words, 'images': images}
            if large:
                stats_str += ' (' + _('truncated') + ')'
            try:
                if getattr(self, 'preview_stats', None) is not None:
                    self.preview_stats.setText(stats_str)
                    try:
                        self.preview_stats.setToolTip(stats_str)
                    except Exception:
                        pass
            except Exception:
                pass
        except Exception:
            try:
                if getattr(self, 'preview_stats', None) is not None:
                    self.preview_stats.setText('')
                    try:
                        self.preview_stats.setToolTip('')
                    except Exception:
                        pass
            except Exception:
                pass

        # Render base preview immediately (article content may be appended later)
        try:
            self.preview.setHtml(html)
        except Exception:
            try:
                self.preview.setHtml(html)
            except Exception:
                pass
        try:
            self._apply_zoom_absolute()
        except Exception:
            pass

        # Show/hide and configure native audio player for enclosures
        try:
            if getattr(self, 'preview_audio_player', None) is not None:
                if audio_urls:
                    try:
                        self.preview_audio_player.set_sources(audio_urls)
                        try:
                            self.preview_audio_player.show()
                        except Exception:
                            pass
                    except Exception:
                        pass
                else:
                    try:
                        self.preview_audio_player.stop()
                    except Exception:
                        pass
                    try:
                        self.preview_audio_player.hide()
                    except Exception:
                        pass
        except Exception:
            pass

    def _on_feed_tag_clicked(self, tag, checked):
        # Legacy wrapper (old footer used toggle buttons). Keep behavior but map to new dropdown.
        try:
            if not checked and tag is not None:
                return

            if tag is None:
                self._clear_feed_tag_filters()
                return

            t = str(tag)
            if t == 'star':
                data = '__feed_starred__'
            elif t == 'img':
                data = '__img__'
            elif t == 'long':
                data = '__long__'
            else:
                data = self._normalize_tag(t)

            cb = getattr(self, 'feed_tag_filter_combo', None)
            if cb is not None:
                try:
                    i = cb.findData(data)
                    if i >= 0:
                        cb.setCurrentIndex(i)
                        return
                except Exception:
                    pass

            self._active_feed_tag = data
            self._apply_feeds_tag_filter(data)
        except Exception:
            pass

    def _on_feed_tag_combo_changed(self, idx):
        # Unified dropdown selection.
        try:
            _debug('Feed tag combo changed to index %d' % idx)
            cb = getattr(self, 'feed_tag_filter_combo', None)
            if cb is None:
                _debug('  -> combo is None!')
                return

            try:
                data = cb.currentData()
            except Exception as e:
                _debug('  -> error getting currentData: %s' % e)
                data = None

            _debug('  -> data: %s' % repr(data))

            # Index 0 is "All" (no filter)
            if idx <= 0 or not data:
                _debug('  -> clearing filter')
                self._active_feed_tag = None
                self._apply_feeds_tag_filter(None)
                return

            _debug('  -> applying filter: %s' % data)
            self._active_feed_tag = data
            self._apply_feeds_tag_filter(data)
        except Exception as e:
            _debug('ERROR in _on_feed_tag_combo_changed: %s' % e)
            import traceback
            _debug(traceback.format_exc())

    def _clear_feed_tag_filters(self):
        try:
            try:
                cb = getattr(self, 'feed_tag_filter_combo', None)
                if cb is not None:
                    try:
                        cb.blockSignals(True)
                        cb.setCurrentIndex(0)
                    finally:
                        cb.blockSignals(False)
            except Exception:
                pass
            try:
                self._active_feed_tag = None
            except Exception:
                pass
            self._apply_feeds_tag_filter(None)
        except Exception:
            pass

    def _refresh_feed_tag_dropdown(self):
        _debug('Entering _refresh_feed_tag_dropdown')
        try:
            cb = getattr(self, 'feed_tag_filter_combo', None)
            if cb is None:
                _debug('  -> feed_tag_filter_combo is None!')
                return

            current = getattr(self, '_active_feed_tag', None)

            tags = set()
            try:
                feed_tags_map = dict(rss_db.get_feed_tags_map() or {})
            except Exception as e:
                feed_tags_map = {}
                _debug('  -> error getting feed_tags_map:', e)
            for _k, lst in (feed_tags_map or {}).items():
                if not isinstance(lst, list):
                    continue
                for t in lst:
                    tt = self._normalize_tag(t)
                    if tt:
                        tags.add(tt)

            # Failure auto-tags (persisted in feed_status)
            try:
                if bool(plugin_prefs.get('auto_feed_failure_tagging_enabled', True)):
                    try:
                        status_map = dict(rss_db.get_feed_status_map() or {})
                    except Exception:
                        status_map = {}
                    for _fid, st in (status_map or {}).items():
                        if not isinstance(st, dict):
                            continue
                        if st.get('ok') is not False:
                            continue
                        for t in (st.get('tags') or []):
                            tt = self._normalize_tag(t)
                            if tt:
                                tags.add(tt)
                        try:
                            hs = st.get('http_status')
                            if hs is not None:
                                tags.add('http-%d' % int(hs))
                        except Exception:
                            pass
            except Exception as e:
                _debug('  -> error getting failure auto-tags:', e)

            try:
                if bool(plugin_prefs.get('auto_feed_tagging_enabled', False)):
                    try:
                        _t = str(plugin_prefs.get('auto_feed_updates_tag_name', 'updates-frequently') or 'updates-frequently')
                    except Exception:
                        _t = 'updates-frequently'
                    try:
                        _t = '-'.join(_t.replace('_', '-').split())
                    except Exception:
                        pass
                    at = self._normalize_tag(_t)
                    if at:
                        tags.add(at)
            except Exception as e:
                _debug('  -> error getting auto_feed_tag:', e)

            try:
                cb.blockSignals(True)
                cb.clear()
                cb.addItem(_('All feeds'), None)
                cb.addItem(_('★ Starred feeds'), '__feed_starred__')
                cb.addItem(_('img (has images)'), '__img__')
                cb.addItem(_('audio (has audio)'), '__audio__')
                cb.addItem(_('long (long items)'), '__long__')
                if tags:
                    cb.insertSeparator(cb.count())
                for t in sorted(tags):
                    cb.addItem(t, t)
                _debug('Dropdown populated with %d tags (+ 4 built-in), total items: %d' % (len(tags), cb.count()))
                if cb.count() == 0:
                    _debug('  -> WARNING: Dropdown is empty after population!')

                if current:
                    try:
                        i = cb.findData(current)
                        cb.setCurrentIndex(i if i >= 0 else 0)
                        _debug('Set current index to %d for filter: %s' % (i if i >= 0 else 0, current))
                    except Exception as e:
                        _debug('  -> error setting current index:', e)
                        cb.setCurrentIndex(0)
                else:
                    cb.setCurrentIndex(0)
            finally:
                try:
                    cb.blockSignals(False)
                except Exception:
                    pass
        except Exception as e:
            _debug('ERROR in _refresh_feed_tag_dropdown:', e)
        _debug('Leaving _refresh_feed_tag_dropdown')

    def _on_feed_tag_text_changed(self, text):
        # No longer used (footer is dropdown-based).
        return

    def _apply_feed_tag_text_filter_now(self):
        # No longer used (footer is dropdown-based).
        return

    def _apply_feeds_tag_filter(self, active_tag):
        # Walk the feeds tree and hide nodes that don't match the current text/tag filters.
        try:
            import urllib
        except Exception:
            pass

        # Advanced filter box (for feeds tree)
        try:
            adv_q = (getattr(self, 'feed_advanced_filter_input', None) or None)
            adv_q = adv_q.text().strip().lower() if adv_q else ''
        except Exception:
            adv_q = ''

        # Old item filter box (for items table)
        try:
            q = (self.feeds_filter_input.text() or '').strip().lower()
        except Exception:
            q = ''
        # Advanced filter logic for feeds (AND logic for all terms)
        def feed_matches_advanced(feed_id):
            try:
                if not adv_q:
                    return True
                fid = str(feed_id or '')
                if not fid:
                    return True

                def _spec_for(fid0):
                    try:
                        return (feed_cache_map or {}).get(str(fid0), {}) or {}
                    except Exception:
                        return {}

                def _spec_match(key, term_value):
                    try:
                        meta = _spec_for(fid)
                        val = str((meta or {}).get(key) or '').strip().casefold()
                        want = str(term_value or '').strip().casefold()
                        if not want:
                            return True
                        # substring match so users can type "atom" or "utf" etc.
                        return want in val
                    except Exception:
                        return False

                # Allow users to type updates-frequently/updates_frequently as shorthand for the configured tag name
                try:
                    updates_tag = str(plugin_prefs.get('auto_feed_updates_tag_name', 'updates-frequently') or 'updates-frequently')
                except Exception:
                    updates_tag = 'updates-frequently'
                # Ensure our built-in auto-tag uses hyphens even if older prefs used spaces/underscores.
                try:
                    updates_tag = '-'.join(str(updates_tag).replace('_', '-').split())
                except Exception:
                    pass
                updates_tag_norm = self._normalize_tag(updates_tag)

                def norm_term_tag(raw):
                    raw = str(raw or '')
                    raw = raw.replace('_', ' ')
                    return self._normalize_tag(raw)

                pending_not = False
                for raw_term in (adv_q.split() or []):
                    term = str(raw_term or '').strip()
                    if not term:
                        continue

                    term_lower = term.lower()
                    if term_lower == 'not':
                        pending_not = True
                        continue

                    is_not = pending_not
                    pending_not = False

                    if term_lower.startswith('not:'):
                        is_not = True
                        term = term[4:].strip()
                        term_lower = term.lower()
                    elif term_lower.startswith('not '):
                        is_not = True
                        term = term[3:].strip()
                        term_lower = term.lower()

                    # Spec filters: type:/encoding:/version:
                    # Examples:
                    #   type:atom
                    #   encoding:utf-16
                    #   version:2.0
                    #   not:encoding:utf-8
                    #   not type:atom
                    tl = term_lower
                    if ':' in tl:
                        k, _, v = tl.partition(':')
                        k = (k or '').strip()
                        v = (v or '').strip()
                        spec_key = None
                        if k in ('type', 'format', 'feedtype', 'feed_type'):
                            spec_key = 'feed_type'
                        elif k in ('enc', 'encoding', 'charset', 'feed_encoding'):
                            spec_key = 'feed_encoding'
                        elif k in ('ver', 'version', 'feed_version'):
                            spec_key = 'feed_version'
                        if spec_key is not None:
                            ok = _spec_match(spec_key, v)
                            if is_not:
                                if ok:
                                    return False
                            else:
                                if not ok:
                                    return False
                            continue

                    if term.startswith('tag:'):
                        term = term[4:]

                    term_norm = norm_term_tag(term)
                    if not term_norm:
                        continue

                    # Built-ins / aliases
                    lookup = term_norm
                    if lookup in ('starred', 'feed starred', 'feed_starred', 'feedstarred'):
                        lookup = '__feed_starred__'
                    elif lookup == 'img':
                        lookup = '__img__'
                    elif lookup == 'audio':
                        lookup = '__audio__'
                    elif lookup == 'long':
                        lookup = '__long__'
                    elif lookup in ('updates frequently', 'updates_frequently', 'updates-frequently'):
                        lookup = updates_tag_norm or 'updates-frequently'

                    has_it = True if not lookup else feed_has_tag(fid, lookup)
                    if is_not:
                        if has_it:
                            return False
                    else:
                        if not has_it:
                            return False
                return True
            except Exception:
                return True

        try:
            model = self.feeds_tree.model()
            root = model.invisibleRootItem()
        except Exception:
            return

        # Pre-fetch starred items map for efficiency
        try:
            starred_map = rss_db.get_starred_items_map()
        except Exception:
            starred_map = {}

        # Pre-fetch manual feed tags (auto-tags computed on the fly only when needed)
        try:
            feed_tags_map = dict(rss_db.get_feed_tags_map() or {})
        except Exception:
            feed_tags_map = {}

        # Pre-fetch failure status tags for fast filtering
        try:
            feed_status_map = dict(rss_db.get_feed_status_map() or {})
        except Exception:
            feed_status_map = {}

        # Pre-fetch feed-level starred flags
        try:
            feeds = list(rss_db.get_feeds() or [])
            feed_starred_map = {str(f.get('id') or ''): bool(f.get('feed_starred', False)) for f in (feeds or [])}
        except Exception:
            feed_starred_map = {}

        # Pre-fetch feed cache for encoding/type search
        try:
            feed_cache_map = dict(rss_db.get_feed_cache_map() or {})
        except Exception:
            feed_cache_map = {}

        def feed_matches_text(data):
            try:
                if not isinstance(data, dict):
                    return False
                title = str(data.get('title') or '')
                url = str(data.get('url') or '')
                # Include encoding and type from cache for searchability
                fid = str(data.get('id') or '')
                cache_entry = feed_cache_map.get(fid, {}) if fid else {}
                enc = str(cache_entry.get('feed_encoding') or '').lower()
                ftype = str(cache_entry.get('feed_type') or '').lower()
                fver = str(cache_entry.get('feed_version') or '').lower()
                hay = (title + ' ' + url + ' ' + enc + ' ' + ftype + ' ' + fver).lower()
                return (q in hay) if q else True
            except Exception:
                return True

        def folder_matches_text(data):
            try:
                if not q:
                    return True
                if not isinstance(data, dict):
                    return False
                if data.get('type') != 'folder':
                    return False
                p = str(data.get('path') or '')
                return (q in p.lower()) if p else False
            except Exception:
                return False

        def feed_has_tag(feed_id, tag):
            try:
                if not feed_id or not tag:
                    return True

                tag = str(tag)

                # Feed-level star (independent of starred articles)
                if tag == '__feed_starred__':
                    return bool(feed_starred_map.get(str(feed_id), False))

                # Item heuristics
                if tag == '__img__':
                    tag = 'img'
                elif tag == '__audio__':
                    tag = 'audio'
                elif tag == '__long__':
                    tag = 'long'

                tag = self._normalize_tag(tag)
                if not tag:
                    return True

                # 'img'/'audio'/'long' filters: check auto-tags for cached items (no DB scans)
                if tag in ('img', 'audio', 'long'):
                    items = list(self._items_by_feed_id.get(str(feed_id), []) or [])
                    for it in items:
                        try:
                            d = dict(it)
                            d['_feed_id'] = str(feed_id)
                            if tag in (self._auto_tags_for_item(d) or []):
                                return True
                        except Exception:
                            continue
                    return False

                # Manual feed tags (fast, from pre-fetched map)
                try:
                    v = feed_tags_map.get(str(feed_id))
                    if isinstance(v, list) and v:
                        for t in v:
                            if self._normalize_tag(t) == tag:
                                return True
                except Exception:
                    pass

                # Failure auto-tags (fast, from pre-fetched feed_status_map)
                try:
                    if bool(plugin_prefs.get('auto_feed_failure_tagging_enabled', True)):
                        st = feed_status_map.get(str(feed_id))
                        if isinstance(st, dict) and st and (st.get('ok') is False):
                            for t in (st.get('tags') or []):
                                if self._normalize_tag(t) == tag:
                                    return True
                            try:
                                hs = st.get('http_status')
                                if hs is not None and ('http-%d' % int(hs)) == tag:
                                    return True
                            except Exception:
                                pass
                except Exception:
                    pass

                # Auto feed tag (only compute when filtering by that tag)
                try:
                    if bool(plugin_prefs.get('auto_feed_tagging_enabled', False)):
                        try:
                            _t = str(plugin_prefs.get('auto_feed_updates_tag_name', 'updates-frequently') or 'updates-frequently')
                        except Exception:
                            _t = 'updates-frequently'
                        try:
                            _t = '-'.join(_t.replace('_', '-').split())
                        except Exception:
                            pass
                        at = self._normalize_tag(_t)
                        if at and tag == at:
                            return tag in [self._normalize_tag(x) for x in (self._auto_tags_for_feed(feed_id) or [])]
                except Exception:
                    pass
            except Exception:
                return False
            return False

        def walk(item, parent_index):
            try:
                data = item.data(ROLE_USER)
            except Exception:
                data = None
            node_type = data.get('type') if isinstance(data, dict) else None
            if node_type == 'root':
                # Root container should always remain visible; only its children are filtered.
                try:
                    index = model.indexFromItem(item)
                    self.feeds_tree.setRowHidden(index.row(), parent_index, False)
                except Exception:
                    pass
                any_child = False
                for i in range(item.rowCount()):
                    if walk(item.child(i), model.indexFromItem(item)):
                        any_child = True
                try:
                    self.feeds_tree.expand(model.indexFromItem(item))
                except Exception:
                    pass
                return any_child or True
            if node_type == 'feed':
                fid = str(data.get('id') or '')
                ok_text = feed_matches_text(data)
                ok_tag = True if not active_tag else feed_has_tag(fid, active_tag)
                ok_adv = feed_matches_advanced(fid)
                hidden = not (ok_text and ok_tag and ok_adv)
                try:
                    index = model.indexFromItem(item)
                    self.feeds_tree.setRowHidden(index.row(), parent_index, hidden)
                except Exception:
                    pass
                return not hidden

            # Folder node: recurse children
            any_child = False
            for i in range(item.rowCount()):
                if walk(item.child(i), model.indexFromItem(item)):
                    any_child = True

            try:
                folder_ok = folder_matches_text(data)
            except Exception:
                folder_ok = False

            if q:
                visible = bool(folder_ok or any_child)
            else:
                # If no text filter is active, show empty folders only when no
                # tag/advanced filter is active. This makes newly-created folders
                # immediately visible, but still keeps tag filtering compact.
                if active_tag or adv_q:
                    visible = bool(any_child)
                else:
                    visible = True

            if q and visible and any_child:
                try:
                    self.feeds_tree.expand(model.indexFromItem(item))
                except Exception:
                    pass

            hidden = not visible
            try:
                index = model.indexFromItem(item)
                self.feeds_tree.setRowHidden(index.row(), parent_index, hidden)
            except Exception:
                pass
            return visible

        try:
            for i in range(root.rowCount()):
                try:
                    walk(root.child(i), QModelIndex())
                except Exception:
                    pass
        except Exception:
            pass

    def on_items_context_menu(self, pos):
        rows = self.items_table.selectionModel().selectedRows()
        if not rows:
            return

        # Check if first selected item is starred
        is_starred = False
        try:
            row = rows[0].row()
            it0 = self.items_table.item(row, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            if isinstance(data, dict):
                fid = str(data.get('_feed_id') or '')
                iid = str(data.get('id') or data.get('link') or data.get('title') or '')
                if fid and iid:
                    is_starred = rss_db.is_item_starred(fid, iid)
        except Exception:
            is_starred = False

        menu = QMenu(self)
        # Star/unstar action
        if is_starred:
            star_act = menu.addAction(_('★ Unstar'))
        else:
            star_act = menu.addAction(_('☆ Star'))

        # Read/unread actions (per-article)
        mark_read_act = menu.addAction(_('Mark as read'))
        mark_unread_act = menu.addAction(_('Mark as unread'))
        menu.addSeparator()
        open_act = menu.addAction(_('Open link'))
        copy_act = menu.addAction(_('Copy link'))
        copy_title_act = menu.addAction(_('Copy title'))
        copy_feed_url_act = menu.addAction(_('Copy feed URL'))
        menu.addSeparator()
        export_ebook_act = menu.addAction(_('Export'))
        export_listed_act = menu.addAction(_('Export listed items'))
        menu.addSeparator()
        tags_act = menu.addAction(_('Edit tags…'))
        copy_tags_act = menu.addAction(_('Copy tags'))
        clear_tags_act = menu.addAction(_('Clear tags'))
        menu.addSeparator()
        del_act = menu.addAction(_('Delete cached item(s)'))

        # Share submenu (same actions as toolbar) - keep it last in the top-level menu
        try:
            share_menu = QMenu(_('Share'), menu)
            try:
                prev = getattr(self, '_share_menu', None)
                self._share_menu = share_menu
                self._rebuild_share_menu()
                self._share_menu = prev
            except Exception:
                pass
            # Always add Share as the last menu item
            menu.addSeparator()
            menu.addMenu(share_menu)
        except Exception:
            pass

        # Clarify intent via hover text (context menus don't reliably show tooltips)
        try:
            prev = str(self.status.text() or '')
            def _wire(a, tip):
                try:
                    a.setStatusTip(str(tip))
                except Exception:
                    pass
                try:
                    a.hovered.connect(lambda _t=str(tip): self.status.setText(_t))
                except Exception:
                    pass
            _wire(star_act, _('Toggle star/favorite status for the selected item(s)'))
            _wire(mark_read_act, _('Mark the selected item(s) as read (not bold)'))
            _wire(mark_unread_act, _('Mark the selected item(s) as unread (bold)'))
            _wire(open_act, _('Open the selected item link in your browser'))
            _wire(copy_act, _('Copy the selected item link to the clipboard'))
            _wire(copy_title_act, _('Copy the selected item title to the clipboard'))
            _wire(copy_feed_url_act, _('Copy the feed URL for the selected item'))
            _wire(export_ebook_act, _('Export the selected item(s) to a file (using your preferred output format)'))
            _wire(export_listed_act, _('Export all currently listed items (respects the search filter)'))
            _wire(tags_act, _('Edit tags for the selected item(s)'))
            _wire(copy_tags_act, _('Copy the tags for the selected item'))
            _wire(clear_tags_act, _('Remove all manual tags from the selected item(s)'))
            _wire(del_act, _('Remove selected items from the local cache (does not affect the website)'))
            try:
                menu.aboutToHide.connect(lambda _p=prev: self.status.setText(_p))
            except Exception:
                pass
        except Exception:
            pass
        act = menu.exec(self.items_table.viewport().mapToGlobal(pos))
        if act == star_act:
            # Explicitly star or unstar based on menu label for deterministic behavior
            # Use the is_starred variable already computed at the top of this function
            if is_starred:
                self.set_star_selected_items(False)
            else:
                self.set_star_selected_items(True)
        elif act == mark_read_act:
            self.mark_selected_items_read(True)
        elif act == mark_unread_act:
            self.mark_selected_items_read(False)
        elif act == open_act:
            row = rows[0].row()
            it0 = self.items_table.item(row, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            if data and isinstance(data, dict) and data.get('link'):
                try:
                    self._open_url(data.get('link'))
                except Exception:
                    pass
        elif act == copy_act:
            row = rows[0].row()
            it0 = self.items_table.item(row, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            if data and isinstance(data, dict) and data.get('link'):
                try:
                    QApplication.clipboard().setText(str(data.get('link')))
                    self.status.setText(_('Link copied to clipboard'))
                except Exception:
                    pass
        elif act == copy_title_act:
            row = rows[0].row()
            it0 = self.items_table.item(row, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            if data and isinstance(data, dict) and data.get('title'):
                try:
                    QApplication.clipboard().setText(str(data.get('title')))
                    self.status.setText(_('Title copied to clipboard'))
                except Exception:
                    pass
        elif act == copy_feed_url_act:
            row = rows[0].row()
            it0 = self.items_table.item(row, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            fedurl = ''
            if data and isinstance(data, dict):
                try:
                    fedurl = str(data.get('_feed_url') or '')
                except Exception:
                    fedurl = ''
                if not fedurl:
                    try:
                        fid = str(data.get('_feed_id') or '')
                        if fid:
                            feeds = list(rss_db.get_feeds() or [])
                            for f in feeds:
                                try:
                                    if str(f.get('id') or '') == fid:
                                        fedurl = str(f.get('url') or '')
                                        break
                                except Exception:
                                    continue
                    except Exception:
                        fedurl = ''
            if fedurl:
                try:
                    QApplication.clipboard().setText(fedurl)
                    self.status.setText(_('Feed URL copied to clipboard'))
                except Exception:
                    pass
        elif act == del_act:
            self.delete_selected_items()
        elif act == export_ebook_act:
            self.export_selected_items()
        elif act == export_listed_act:
            try:
                self.export_listed_items()
            except Exception:
                pass
        elif act == tags_act:
            self.edit_tags_for_selected_items()
        elif act == copy_tags_act:
            try:
                row = rows[0].row()
                it0 = self.items_table.item(row, 0)
                data = it0.data(ROLE_USER) if it0 is not None else None
                if isinstance(data, dict):
                    tags = self._tags_for_item(data)
                    try:
                        QApplication.clipboard().setText(', '.join(list(tags or []) or []))
                        self.status.setText(_('Tags copied to clipboard'))
                    except Exception:
                        pass
            except Exception:
                pass
        elif act == clear_tags_act:
            self.clear_tags_for_selected_items()

    def mark_selected_items_read(self, read=True):
        """Mark selected article rows as read/unread.

        Read/unread is tracked via `seen_item_ids` per feed.
        """
        try:
            rows = list(self.items_table.selectionModel().selectedRows() or [])
        except Exception:
            rows = []
        if not rows:
            return

        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}

        by_feed = {}
        for mi in rows:
            try:
                r = int(mi.row())
            except Exception:
                continue
            it0 = self.items_table.item(r, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            if not isinstance(data, dict):
                continue
            fid = str(data.get('_feed_id') or '')
            iid = str(data.get('id') or data.get('link') or data.get('title') or '')
            if not fid or not iid:
                continue
            by_feed.setdefault(fid, set()).add(iid)

        changed_feeds = set()
        for fid, ids in by_feed.items():
            cur = [str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip()]
            s = set(cur)
            if read:
                new_set = s.union(set(ids))
            else:
                new_set = s.difference(set(ids))
            if new_set != s:
                try:
                    new_list = list(new_set)
                    rss_db.set_seen_item_ids(fid, new_list)
                    seen_map[fid] = new_list
                    changed_feeds.add(fid)
                except Exception:
                    pass

        # Update row fonts (bold unread like QuiteRSS)
        try:
            from qt.core import QFont
        except Exception:
            try:
                from PyQt5.Qt import QFont
            except Exception:
                QFont = None

        for mi in rows:
            try:
                r = int(mi.row())
            except Exception:
                continue
            it0 = self.items_table.item(r, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            if not isinstance(data, dict):
                continue
            fid = str(data.get('_feed_id') or '')
            iid = str(data.get('id') or '')
            if not fid or not iid:
                continue
            seen_set = set(str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip())
            is_unread = iid not in seen_set
            try:
                # Keep the row's dict in sync for items_search quick filters.
                data['_unread'] = bool(is_unread)
            except Exception:
                pass
            try:
                for col in (1, 2, 3, 4):
                    cell = self.items_table.item(r, col)
                    if cell is None or QFont is None:
                        continue
                    f = QFont(cell.font())
                    f.setBold(bool(is_unread))
                    cell.setFont(f)
            except Exception:
                pass
            try:
                title_cell = self.items_table.item(r, 1)
                if title_cell is not None:
                    if is_unread:
                        ic = self._unread_dot_icon()
                        if ic is not None:
                            title_cell.setIcon(ic)
                    else:
                        ic = self._transparent_placeholder_icon()
                        if ic is not None:
                            title_cell.setIcon(ic)
            except Exception:
                pass

        # Update feed badges for affected feeds by recomputing unread from cache/seen
        try:
            for fid in changed_feeds:
                try:
                    items = list(self._items_by_feed_id.get(fid, []) or [])
                except Exception:
                    items = []
                seen_set = set(str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip())
                try:
                    unread = sum(1 for it in (items or []) if str(it.get('id') or '').strip() and str(it.get('id') or '').strip() not in seen_set)
                except Exception:
                    unread = 0
                try:
                    if not isinstance(getattr(self, '_feeds_results', None), dict):
                        self._feeds_results = {}
                    r = self._feeds_results.get(fid)
                    if not isinstance(r, dict):
                        r = {}
                        self._feeds_results[fid] = r
                    r['new_count'] = int(unread)
                except Exception:
                    pass
            self._update_feed_badges(list(changed_feeds))
        except Exception:
            pass

    def _on_items_table_item_clicked(self, qitem):
        """Auto mark read when the user clicks an item row (QuiteRSS style)."""
        try:
            row = int(qitem.row())
        except Exception:
            return
        try:
            col = int(qitem.column())
        except Exception:
            col = -1

        try:
            it0 = self.items_table.item(row, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
        except Exception:
            data = None

        if not isinstance(data, dict):
            return

        fid = str(data.get('_feed_id') or '')
        iid = str(data.get('id') or data.get('link') or data.get('title') or '')
        if not fid or not iid:
            return

        # Clicking the ★ column toggles star without affecting read state.
        if col == 0:
            try:
                # Query database for current star state to avoid stale in-memory cache
                cur_star = rss_db.is_item_starred(fid, iid)
            except Exception:
                cur_star = False
            try:
                new_star = not cur_star
                rss_db.set_item_starred(fid, iid, starred=new_star)
                data['_starred'] = new_star
            except Exception:
                return
            try:
                star_cell = self.items_table.item(row, 0)
                if star_cell is not None:
                    is_starred = bool(data.get('_starred', False))
                    star_cell.setText('★' if is_starred else '')
                    if is_starred:
                        try:
                            star_cell.setForeground(QColor(255, 180, 0))
                        except Exception:
                            pass
                    else:
                        try:
                            from qt.core import QBrush
                        except Exception:
                            try:
                                from PyQt5.Qt import QBrush
                            except Exception:
                                QBrush = None
                        try:
                            if QBrush is not None:
                                star_cell.setForeground(QBrush())
                        except Exception:
                            pass
            except Exception:
                pass
            return

        # Only apply auto-mark-read for non-star clicks.
        try:
            enabled = bool(plugin_prefs.get('auto_mark_read_on_item_click', True))
        except Exception:
            enabled = True
        if not enabled:
            return

        # Only apply auto-mark-read for non-star clicks.
        try:
            enabled = bool(plugin_prefs.get('auto_mark_read_on_item_click', True))
        except Exception:
            enabled = True
        if not enabled:
            return

        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}

        try:
            cur = [str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip()]

            # Header tooltip for the star column
            try:
                hi = self.items_table.horizontalHeaderItem(0)
                if hi is not None:
                    hi.setToolTip(_('Click a cell to star/unstar the item'))
            except Exception:
                pass

        except Exception:
            cur = []
        if iid in set(cur):
            return

        try:
            new_list = list(set(cur).union({iid}))
            rss_db.set_seen_item_ids(fid, new_list)
        except Exception:
            return

        try:
            data['_unread'] = False
        except Exception:
            pass

        # Update row fonts immediately (bold = unread; non-bold = read)
        try:
            from qt.core import QFont
        except Exception:
            try:
                from PyQt5.Qt import QFont
            except Exception:
                QFont = None

        if QFont is not None:
            try:
                for col in (1, 2, 3, 4):
                    cell = self.items_table.item(row, col)
                    if cell is None:
                        continue
                    f = QFont(cell.font())
                    f.setBold(False)
                    cell.setFont(f)
            except Exception:
                pass

        # Clear the unread dot icon
        try:
            title_cell = self.items_table.item(row, 1)
            if title_cell is not None:
                ic = self._transparent_placeholder_icon()
                if ic is not None:
                    title_cell.setIcon(ic)
        except Exception:
            pass

        # Update feed + folder badges for this feed without rebuilding the whole tree
        try:
            self._update_feed_badges([fid])
        except Exception:
            pass

    def _set_items_quick_filter(self, label, query):
        try:
            self._items_quick_filter_query = str(query or '').strip()
        except Exception:
            self._items_quick_filter_query = ''
        try:
            if getattr(self, 'items_quick_filter_btn', None) is not None:
                self.items_quick_filter_btn.setText(str(label or _('Show All')))
        except Exception:
            pass
        # Update checkmarks in filter menu to show current selection
        try:
            filter_actions = getattr(self, '_filter_actions', None)
            if filter_actions:
                q = str(query or '').strip()
                for action_query, action in filter_actions.items():
                    action.setChecked(action_query == q)
        except Exception:
            pass
        try:
            # Re-apply filter using current search box text.
            txt = ''
            try:
                txt = str(self.filter_input.currentText() if hasattr(self.filter_input, 'currentText') else self.filter_input.text())
            except Exception:
                txt = ''
            self._schedule_filter_items(txt)
        except Exception:
            pass

    def _clear_items_quick_filter(self):
        self._set_items_quick_filter(_('Show All'), '')

    def _normalize_tag(self, s):
        try:
            s = str(s or '').strip()
        except Exception:
            s = ''
        if not s:
            return ''
        s = s.replace('\u00A0', ' ')
        s = ' '.join(s.split())
        return s.casefold()

    def _parse_tags_text(self, text):
        raw = str(text or '').strip()
        if not raw:
            return []
        parts = []
        for chunk in raw.replace(';', ',').split(','):
            t = self._normalize_tag(chunk)
            if t:
                parts.append(t)
        # de-dup preserve order
        out = []
        seen = set()
        for t in parts:
            if t in seen:
                continue
            seen.add(t)
            out.append(t)
        return out

    def _strip_html_to_text(self, html):
        try:
            import re
            s = str(html or '')
            s = re.sub(r'(?is)<\s*br\s*/?>', '\n', s)
            s = re.sub(r'(?is)</\s*p\s*>', '\n\n', s)
            s = re.sub(r'(?is)<[^>]+>', ' ', s)
            return ' '.join(s.split())
        except Exception:
            try:
                return ' '.join(str(html or '').split())
            except Exception:
                return ''

    def _sanitize_folder_name(self, s):
        # Only sanitize illegal chars, allow '/'
        s = str(s or '').strip()
        for ch in '<>:"\\|?*#':
            s = s.replace(ch, '_')
        return s


    def _auto_tags_for_item(self, item_dict):
        try:
            tags = set(_auto_tags_for_item_util(item_dict) or [])
        except Exception:
            tags = set()
        # Propagate feed-level auto-tags (e.g. updates-frequently) to items
        # so users can filter items with tag:updates-frequently in the search box.
        try:
            fid = str(item_dict.get('_feed_id') or '').strip()
            if fid:
                feed_tags = self._auto_tags_for_feed(fid)
                if feed_tags:
                    tags.update(feed_tags)
        except Exception:
            pass
        return tags

    def _auto_tags_for_feed(self, feed_id):
        """Compute auto-tags for a feed based on recent item timestamps.

        Currently implements a user-configurable "updates-frequently" rule.
        """
        tags = set()

        # Failure-based auto-tags (from persisted status)
        try:
            if bool(plugin_prefs.get('auto_feed_failure_tagging_enabled', True)):
                fid0 = str(feed_id or '').strip()
                st_map = getattr(self, '_feed_status_map', None)
                if isinstance(st_map, dict):
                    st = st_map.get(fid0) or {}
                else:
                    st = {}
                if isinstance(st, dict) and st and (st.get('ok') is False):
                    for t in (st.get('tags') or []):
                        try:
                            tags.add(self._normalize_tag(t))
                        except Exception:
                            pass
                    try:
                        hs = st.get('http_status')
                        if hs is not None:
                            tags.add('http-%d' % int(hs))
                    except Exception:
                        pass
        except Exception:
            pass

        # Frequency-based auto-tag
        try:
            do_freq = bool(plugin_prefs.get('auto_feed_tagging_enabled', False))
        except Exception:
            do_freq = False

        if not do_freq:
            return tags

        try:
            raw_tag_name = str(plugin_prefs.get('auto_feed_updates_tag_name', 'updates-frequently') or 'updates-frequently')
        except Exception:
            raw_tag_name = 'updates-frequently'
        # Ensure the auto-tag itself is always hyphenated.
        try:
            raw_tag_name = '-'.join(raw_tag_name.replace('_', '-').split())
        except Exception:
            pass
        try:
            tag_name = self._normalize_tag(raw_tag_name)
        except Exception:
            tag_name = 'updates-frequently'
        if not tag_name:
            return tags

        try:
            window_days = int(plugin_prefs.get('auto_feed_updates_window_days', 5) or 5)
        except Exception:
            window_days = 5
        try:
            min_dates = int(plugin_prefs.get('auto_feed_updates_min_distinct_dates', 4) or 4)
        except Exception:
            min_dates = 4
        try:
            min_avg = float(plugin_prefs.get('auto_feed_updates_min_avg_per_day', 3.0) or 3.0)
        except Exception:
            min_avg = 3.0

        if window_days <= 0:
            return tags
        if min_dates <= 0:
            min_dates = 1
        if min_avg < 0:
            min_avg = 0.0

        fid = str(feed_id or '').strip()
        if not fid:
            return tags

        # Lightweight cache to keep UI snappy (tooltips, selection changes, filtering)
        try:
            cache = getattr(self, '_auto_feed_tags_cache', None)
            if not isinstance(cache, dict):
                cache = {}
                self._auto_feed_tags_cache = cache
            hit = cache.get(fid)
            if isinstance(hit, tuple) and len(hit) == 2:
                ts, val = hit
                if (time.time() - float(ts)) < 30.0:
                    return set(val or [])
        except Exception:
            pass

        try:
            import datetime
            now = datetime.datetime.now(datetime.timezone.utc)
            cutoff = now - datetime.timedelta(days=int(window_days))
        except Exception:
            cutoff = None

        items = list(self._items_by_feed_id.get(fid, []) or [])
        if not items:
            try:
                self._auto_feed_tags_cache[fid] = (time.time(), list(tags))
            except Exception:
                pass
            return tags

        count = 0
        dates = set()
        for it in (items or []):
            try:
                published = it.get('published') or ''
                ts = int(it.get('published_ts') or iso_to_ts(published) or 0)
            except Exception:
                ts = 0
            if ts <= 0:
                continue
            try:
                import datetime
                dt = datetime.datetime.fromtimestamp(ts, datetime.timezone.utc)
            except Exception:
                dt = None
            if dt is None:
                continue
            if cutoff is not None and dt < cutoff:
                continue
            count += 1
            try:
                dates.add(dt.date())
            except Exception:
                pass

        try:
            distinct_dates = len(dates)
        except Exception:
            distinct_dates = 0

        if distinct_dates < int(min_dates):
            return tags

        try:
            avg = float(count) / float(window_days)
        except Exception:
            avg = 0.0

        if avg >= float(min_avg):
            try:
                tags.add(tag_name)
            except Exception:
                pass

        try:
            self._auto_feed_tags_cache[fid] = (time.time(), list(tags))
        except Exception:
            pass
        return tags

    def _manual_tags_for_feed(self, feed_id):
        try:
            return list(rss_db.get_feed_tags(feed_id) or [])
        except Exception:
            return []

    def edit_tags_for_selected_feeds(self):
        try:
            fids = list(self.selected_feed_ids() or [])
        except Exception:
            fids = []
        if not fids:
            return

        initial = []
        if len(fids) == 1:
            try:
                initial = list(self._manual_tags_for_feed(fids[0]) or [])
            except Exception:
                initial = []

        readonly = []
        if len(fids) == 1:
            try:
                readonly = list(self._auto_tags_for_feed(fids[0]) or [])
            except Exception:
                readonly = []

        try:
            title = _('Edit feed tags')
            subtitle = _('Add/remove tags for the selected feed(s).')
            d = TagEditorDialog(self, title=title, subtitle=subtitle, initial_tags=initial, readonly_tags=readonly)
            if not d.exec():
                return
            tags = list(d.tags() or [])
        except Exception:
            return
        for fid in fids:
            try:
                rss_db.set_feed_tags(fid, tags)
            except Exception:
                pass

        try:
            self._refresh_feed_tag_dropdown()
        except Exception:
            pass

        try:
            self.refresh()
        except Exception:
            try:
                self.load_feeds()
            except Exception:
                pass

    def clear_tags_for_selected_feeds(self):
        try:
            fids = list(self.selected_feed_ids() or [])
        except Exception:
            fids = []
        if not fids:
            return

        for fid in fids:
            try:
                rss_db.delete_feed_tags(fid)
            except Exception:
                pass

        try:
            self._refresh_feed_tag_dropdown()
        except Exception:
            pass

        try:
            self.refresh()
        except Exception:
            try:
                self.load_feeds()
            except Exception:
                pass

    def _tags_for_item(self, item_dict, tags_map_by_feed=None):
        try:
            fid = str(item_dict.get('_feed_id') or '').strip()
        except Exception:
            fid = ''
        try:
            iid = str(item_dict.get('id') or '').strip()
        except Exception:
            iid = ''

        manual = []
        if fid and iid:
            try:
                if isinstance(tags_map_by_feed, dict):
                    lst = (tags_map_by_feed.get(fid) or {}).get(iid)
                    if isinstance(lst, list):
                        manual = lst
            except Exception:
                manual = []
            if not manual:
                try:
                    manual = (rss_db.get_item_tags_map(fid) or {}).get(iid) or []
                except Exception:
                    manual = []

        out = []
        seen = set()
        for t in list(manual or []) + list(self._auto_tags_for_item(item_dict) or []):
            tt = self._normalize_tag(t)
            if not tt or tt in seen:
                continue
            seen.add(tt)
            out.append(tt)
        return out

    def edit_tags_for_selected_items(self):
        try:
            rows = list(self.items_table.selectionModel().selectedRows() or [])
        except Exception:
            rows = []
        if not rows:
            return

        first = None
        try:
            it0 = self.items_table.item(rows[0].row(), 0)
            first = it0.data(ROLE_USER) if it0 is not None else None
        except Exception:
            first = None

        initial = []
        if isinstance(first, dict):
            try:
                fid0 = str(first.get('_feed_id') or '').strip()
                iid0 = str(first.get('id') or '').strip()
                if fid0 and iid0:
                    initial = list((rss_db.get_item_tags_map(fid0) or {}).get(iid0) or [])
            except Exception:
                initial = []

        readonly = []
        if isinstance(first, dict):
            try:
                readonly = list(self._auto_tags_for_item(first) or [])
            except Exception:
                readonly = []

        try:
            d = TagEditorDialog(self, title=_('Edit tags'), subtitle=_('Add/remove manual tags for the selected item(s).'), initial_tags=initial, readonly_tags=readonly)
            if not d.exec():
                return
            tags = list(d.tags() or [])
        except Exception:
            return

        for r in rows:
            try:
                it0 = self.items_table.item(r.row(), 0)
                data = it0.data(ROLE_USER) if it0 is not None else None
                if not isinstance(data, dict):
                    continue
                fid = str(data.get('_feed_id') or '').strip()
                iid = str(data.get('id') or '').strip()
                if not fid or not iid:
                    continue
                rss_db.set_item_tags(fid, iid, tags)
            except Exception:
                pass

        try:
            self.load_items_for_selected_feed()
        except Exception:
            pass

    def _apply_quick_tag_to_selected_items(self, tag):
        try:
            tag_n = self._normalize_tag(tag)
        except Exception:
            tag_n = str(tag or '').strip()
        if not tag_n:
            return

        try:
            rows = list(self.items_table.selectionModel().selectedRows() or [])
        except Exception:
            rows = []
        if not rows:
            return

        for r in rows:
            try:
                it0 = self.items_table.item(r.row(), 0)
                data = it0.data(ROLE_USER) if it0 is not None else None
                if not isinstance(data, dict):
                    continue
                fid = str(data.get('_feed_id') or '').strip()
                iid = str(data.get('id') or '').strip()
                if not fid or not iid:
                    continue
                try:
                    cur = list((rss_db.get_item_tags_map(fid) or {}).get(iid) or [])
                except Exception:
                    cur = []
                cur_n = []
                seen = set()
                for x in (cur or []):
                    xx = self._normalize_tag(x)
                    if not xx or xx in seen:
                        continue
                    seen.add(xx)
                    cur_n.append(xx)
                if tag_n not in seen:
                    cur_n.append(tag_n)
                rss_db.set_item_tags(fid, iid, cur_n)
            except Exception:
                pass

        try:
            self.load_items_for_selected_feed()
        except Exception:
            pass

    def clear_tags_for_selected_items(self):
        try:
            rows = list(self.items_table.selectionModel().selectedRows() or [])
        except Exception:
            rows = []
        if not rows:
            return

        for r in rows:
            try:
                it0 = self.items_table.item(r.row(), 0)
                data = it0.data(ROLE_USER) if it0 is not None else None
                if not isinstance(data, dict):
                    continue
                fid = str(data.get('_feed_id') or '').strip()
                iid = str(data.get('id') or '').strip()
                if not fid or not iid:
                    continue
                rss_db.delete_item_tags(fid, iid)
            except Exception:
                pass

        try:
            self.load_items_for_selected_feed()
        except Exception:
            pass

    # Removed toggle_star_selected_items; replaced by set_star_selected_items
    def set_star_selected_items(self, star: bool):
        """Set star/favorite status for selected items to True (star) or False (unstar)."""
        try:
            rows = list(self.items_table.selectionModel().selectedRows() or [])
        except Exception:
            rows = []
        if not rows:
            return

        count = 0
        for r in rows:
            try:
                row_idx = r.row()
                star_cell = self.items_table.item(row_idx, 0)
                data = star_cell.data(ROLE_USER) if star_cell is not None else None
                if not isinstance(data, dict):
                    continue
                fid = str(data.get('_feed_id') or '').strip()
                iid = str(data.get('id') or data.get('link') or data.get('title') or '').strip()
                if not fid or not iid:
                    continue
                rss_db.set_item_starred(fid, iid, star)
                # Update in-memory data
                data['_starred'] = star
                # Update the star cell display
                star_cell.setText('★' if star else '')
                if star:
                    try:
                        star_cell.setForeground(QColor(255, 180, 0))  # Gold
                    except Exception:
                        pass
                else:
                    try:
                        star_cell.setForeground(QColor(0, 0, 0))  # Reset to black
                    except Exception:
                        pass
                count += 1
            except Exception:
                pass

        if count > 0:
            try:
                if star:
                    self.status.setText(_('Starred %d item(s)') % count)
                else:
                    self.status.setText(_('Unstarred %d item(s)') % count)
            except Exception:
                pass

    def delete_selected_items(self):
        try:
            rows = list(self.items_table.selectionModel().selectedRows() or [])
        except Exception:
            rows = []
        if not rows:
            return

        # Confirm deletion
        try:
            n = len({r.row() for r in rows})
        except Exception:
            n = len(rows)
        try:
            title = _('Delete item') if n == 1 else _('Delete items')
            msg = _('Delete the selected item from the cache?') if n == 1 else _('Delete %d selected items from the cache?') % n
            ret = QMessageBox.question(self, title, msg, QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
            if ret != QMessageBox.StandardButton.Yes:
                return
        except Exception:
            pass

        # Preserve current feed selection
        try:
            prev_selected_feed_ids = list(self.selected_feed_ids() or [])
            prev_selected_folder = str(self.selected_folder_path() or '')
        except Exception:
            prev_selected_feed_ids = []
            prev_selected_folder = ''

        cache = dict(rss_db.get_feed_cache_map() or {})
        seen = dict(rss_db.get_seen_item_ids_map() or {})

        # Gather deletions by feed
        to_delete = {}
        for r in rows:
            row = r.row()
            it0 = self.items_table.item(row, 0)
            data = it0.data(ROLE_USER) if it0 is not None else None
            if not isinstance(data, dict):
                continue
            fid = str(data.get('_feed_id') or '')
            iid = str(data.get('id') or '')
            if fid and iid:
                to_delete.setdefault(fid, set()).add(iid)

        if not to_delete:
            return

        # Apply deletions
        for fid, ids in to_delete.items():
            c = dict(cache.get(fid, {}) or {})
            items = list(c.get('items', []) or [])
            items = [it for it in items if str(it.get('id') or '') not in ids]
            c['items'] = items
            cache[fid] = c

            try:
                s = list(seen.get(fid, []) or [])
                s = [x for x in s if str(x or '') not in ids]
                seen[fid] = s
            except Exception:
                pass

            # Keep in-memory cache in sync
            try:
                self._items_by_feed_id[fid] = [it for it in (self._items_by_feed_id.get(fid, []) or []) if str(it.get('id') or '') not in ids]
            except Exception:
                pass

        for fid in to_delete.keys():
            try:
                rss_db.set_feed_cache(fid, cache.get(fid, {}))
            except Exception:
                pass
            try:
                rss_db.set_seen_item_ids(fid, list(seen.get(fid, []) or []))
            except Exception:
                pass

        # Refresh UI + restore selection
        self.load_feeds()
        try:
            try:
                self.feeds_tree.blockSignals(True)
            except Exception:
                pass
            try:
                if prev_selected_folder:
                    it = self._find_tree_item_by_folder(prev_selected_folder)
                    if it is not None:
                        it.setSelected(True)
                if not self.selected_feed_ids() and prev_selected_feed_ids:
                    for fid in prev_selected_feed_ids:
                        it = self._find_tree_item_by_feed(fid)
                        if it is not None:
                            it.setSelected(True)
                if not self.selected_feed_ids():
                    self._restore_feed_selection()
            finally:
                try:
                    self.feeds_tree.blockSignals(False)
                except Exception:
                    pass
        except Exception:
            pass
        self.load_items_for_selected_feed()

    def mark_selected_feed_read(self):
        try:
            fids = list(self.selected_feed_ids() or [])
        except Exception:
            fids = []
        if not fids:
            return

        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}

        changed = []
        for fid in (fids or []):
            fid = str(fid or '').strip()
            if not fid:
                continue

            # Prefer in-memory items (most accurate), fallback to DB cache.
            try:
                items = list(self._items_by_feed_id.get(fid, []) or [])
            except Exception:
                items = []
            if not items:
                try:
                    c = rss_db.get_feed_cache(fid) or {}
                    items = list((c or {}).get('items', []) or [])
                except Exception:
                    items = []

            ids = [str(it.get('id') or '').strip() for it in (items or []) if str(it.get('id') or '').strip()]
            if not ids:
                # Nothing to mark; still clear any transient count in the UI.
                try:
                    if not isinstance(getattr(self, '_feeds_results', None), dict):
                        self._feeds_results = {}
                    r = self._feeds_results.get(fid)
                    if not isinstance(r, dict):
                        r = {}
                        self._feeds_results[fid] = r
                    r['new_count'] = 0
                except Exception:
                    pass
                changed.append(fid)
                continue

            cur = [str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip()]
            merged = list(set(cur).union(set(ids)))
            try:
                rss_db.set_seen_item_ids(fid, merged)
                seen_map[fid] = merged
                changed.append(fid)
            except Exception:
                pass

            # Ensure badges clear even if unread was coming from a stale new_count.
            try:
                if not isinstance(getattr(self, '_feeds_results', None), dict):
                    self._feeds_results = {}
                r = self._feeds_results.get(fid)
                if not isinstance(r, dict):
                    r = {}
                    self._feeds_results[fid] = r
                r['new_count'] = 0
            except Exception:
                pass

        try:
            self._update_feed_badges(list(dict.fromkeys(changed)))
        except Exception:
            pass
        try:
            self.load_items_for_selected_feed()
        except Exception:
            pass

    def mark_all_feeds_read(self):
        try:
            feeds = list(rss_db.get_feeds() or [])
        except Exception:
            feeds = []
        feed_ids = [str(f.get('id') or '').strip() for f in (feeds or []) if str(f.get('id') or '').strip()]
        if not feed_ids:
            return

        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}

        try:
            cache_map = dict(rss_db.get_feed_cache_map() or {})
        except Exception:
            cache_map = {}

        changed = []
        for fid in (feed_ids or []):
            # Prefer in-memory items, then cache map.
            try:
                items = list(self._items_by_feed_id.get(fid, []) or [])
            except Exception:
                items = []
            if not items:
                try:
                    items = list((cache_map.get(fid, {}) or {}).get('items', []) or [])
                except Exception:
                    items = []

            ids = [str(it.get('id') or '').strip() for it in (items or []) if str(it.get('id') or '').strip()]
            cur = [str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip()]
            if ids:
                merged = list(set(cur).union(set(ids)))
            else:
                merged = cur
            try:
                rss_db.set_seen_item_ids(fid, merged)
                seen_map[fid] = merged
                changed.append(fid)
            except Exception:
                pass

            try:
                if not isinstance(getattr(self, '_feeds_results', None), dict):
                    self._feeds_results = {}
                r = self._feeds_results.get(fid)
                if not isinstance(r, dict):
                    r = {}
                    self._feeds_results[fid] = r
                r['new_count'] = 0
            except Exception:
                pass

        try:
            self._update_feed_badges(list(dict.fromkeys(changed)))
        except Exception:
            pass
        try:
            self.load_items_for_selected_feed()
        except Exception:
            pass

    def import_opml(self):
        from calibre_plugins.rss_reader import opml_logic
        return opml_logic.import_opml(self)

    def import_opml_advanced(self):
        from calibre_plugins.rss_reader import opml_logic
        return opml_logic.import_opml_advanced(self)

    def _write_opml(self, fname, feeds):
        from calibre_plugins.rss_reader import opml_logic
        return opml_logic.write_opml(fname, feeds)

    def export_opml(self):
        from calibre_plugins.rss_reader import opml_logic
        return opml_logic.export_opml(self)

    def export_opml_advanced(self):
        from calibre_plugins.rss_reader import opml_logic
        return opml_logic.export_opml_advanced(self)

    def export_selected_feeds(self):
        from calibre_plugins.rss_reader import export_logic
        return export_logic.export_selected_feeds(self)

    def _on_export_worker_finished(self, result, progress, thread, td, out_path, add_to_library, news_title, final_out_path, convert_ext, final_ext, markdown_via_txt):
        from calibre_plugins.rss_reader import export_logic
        return export_logic.on_export_worker_finished(
            self, result, progress, thread, td, out_path, add_to_library, news_title,
            final_out_path, convert_ext, final_ext, markdown_via_txt,
        )

    def export_selected_items(self):
        from calibre_plugins.rss_reader import export_logic
        return export_logic.export_selected_items(self)

    def export_listed_items(self):
        from calibre_plugins.rss_reader import export_logic
        return export_logic.export_listed_items(self)

    def _export_job_done(self, job):
        from calibre_plugins.rss_reader import export_logic
        return export_logic.export_job_done(self, job)

    def _schedule_filter_items(self, text):
        from calibre_plugins.rss_reader import items_search
        return items_search.schedule_filter_items(self, text)

    def _apply_debounced_filter(self):
        from calibre_plugins.rss_reader import items_search
        return items_search.apply_debounced_filter(self)

    def filter_items(self, text):
        from calibre_plugins.rss_reader import items_search
        return items_search.filter_items(self, text)

    def _collect_items_for_feeds(self, feed_ids, max_total=200000):
        from calibre_plugins.rss_reader import items_search
        return items_search.collect_items_for_feeds(feed_ids, max_total=max_total)

    def _render_items_table(self, items, fids, seen_map, tags_map_by_feed):
        from calibre_plugins.rss_reader import items_search
        return items_search.render_items_table(self, items, fids, seen_map, tags_map_by_feed)

    def open_selected_item(self, *args):
        it = self.selected_item() or {}
        link = it.get('link') or ''
        if not link:
            return
        self._open_url(link)

    def _open_url(self, url):
        # Respect calibre's openers_by_scheme tweak
        try:
            from calibre.gui2 import open_url
            open_url(url)
            return
        except Exception:
            pass

    def _queue_article_fetch(self, link, generation, timeout, feed_id=None):
        try:
            import threading
            import queue as _queue
            result_q = _queue.Queue(maxsize=1)
        except Exception:
            result_q = None

        def _worker():
            # If the feed requested the recipe-engine option and the feed
            # is a recipe, try to resolve recipe-declared subfeeds and
            # prefer content coming from the recipe feed item. Fall back
            # to the conservative extractor on any error.
            _debug('[ARTICLE_FETCH_WORKER] start: link=%s feed_id=%s' % (str(link), str(feed_id)))
            article_html, article_imgs, engine = None, [], ''
            engine_label = ''
            try:
                use_recipe_path = False
                use_recipe_engine = False
                feed = None
                recipe_urn = ''
                try:
                    if feed_id:
                        feeds = list(rss_db.get_feeds() or [])
                        feed = next((f for f in feeds if str(f.get('id') or '') == str(feed_id)), None)
                        try:
                            global_recipe = bool(plugin_prefs.get('preview_use_recipe_engine', False))
                        except Exception:
                            global_recipe = False
                        use_recipe_engine = bool(global_recipe) or bool(feed and feed.get('use_recipe_engine'))
                        use_recipe_path = bool(use_recipe_engine and feed and feed.get('is_recipe'))

                        # Even for plain RSS feeds, try to find a matching calibre recipe
                        # so recipe mode can apply preprocess_raw_html etc.
                        try:
                            recipe_urn = str((feed or {}).get('recipe_urn') or '').strip()
                        except Exception:
                            recipe_urn = ''
                        if use_recipe_engine and not recipe_urn:
                            try:
                                from calibre_plugins.rss_reader.recipe_utils import find_recipe_urn_for_feed_url
                                recipe_urn = find_recipe_urn_for_feed_url((feed or {}).get('url') or '') or ''
                            except Exception:
                                recipe_urn = ''
                except Exception:
                    feed = None
                    use_recipe_path = False
                    use_recipe_engine = False
                    recipe_urn = ''

                _debug('[ARTICLE_FETCH_WORKER] recipe flags: use_recipe_engine=%s use_recipe_path=%s recipe_urn=%s' % (str(use_recipe_engine), str(use_recipe_path), str(recipe_urn)))
                # Determine banner engine label up-front for debugging visibility
                try:
                    engine_label = 'calibre recipe' if use_recipe_engine else 'basic'
                except Exception:
                    engine_label = 'basic' if not use_recipe_engine else 'calibre recipe'

                # Skip recipe-subfeeds logic for builtin recipes (they don't declare subfeeds);
                # fall through directly to _fetch_article_content_calibre_recipe_like instead.
                is_builtin_recipe = bool(recipe_urn and recipe_urn.startswith('builtin:'))
                if use_recipe_path and not is_builtin_recipe:
                    try:
                        from calibre_plugins.rss_reader.recipe_utils import get_recipe_feeds_from_urn
                        from calibre_plugins.rss_reader.rss import fetch_url, parse_feed
                        # Resolve declared feeds from the recipe URN
                        urn = feed.get('recipe_urn') or feed.get('url') or ''
                        _debug('[ARTICLE_FETCH_WORKER] trying recipe subfeeds: urn=%s' % str(urn))
                        subfeeds = []
                        try:
                            subfeeds = get_recipe_feeds_from_urn(urn) or []
                        except Exception:
                            subfeeds = []

                        # Try each declared subfeed to find a matching item by link
                        for _title, sub_url in (subfeeds or []):
                            if not sub_url:
                                continue
                            try:
                                raw, final = fetch_url(sub_url, timeout_seconds=timeout)
                                parsed = parse_feed(raw, base_url=final)
                                for it in (parsed.get('items') or []):
                                    try:
                                        if str(it.get('link') or '').strip() == str(link).strip():
                                            # Prefer item content/summary when available
                                            article_html = it.get('content') or it.get('summary') or None
                                            # Attempt to extract images from the HTML
                                            if article_html:
                                                import re, urllib.parse
                                                imgs = []
                                                for m in re.finditer(r'<img[^>]+(?:src|data-src|data-original|data-lazy-src)=["\']([^"\']+)["\']', str(article_html), re.IGNORECASE):
                                                    imgs.append(m.group(1))
                                                for m in re.finditer(r'(?:srcset|data-srcset)=["\']([^"\']+)["\']', str(article_html), re.IGNORECASE):
                                                    ss = m.group(1)
                                                    first = ss.split(',')[0].strip().split()[0].strip() if ss else ''
                                                    if first:
                                                        imgs.append(first)
                                                resolved = []
                                                for s in imgs:
                                                    try:
                                                        r = urllib.parse.urljoin(final or '', s)
                                                        r = _sanitize_url_for_fetch(r)
                                                        resolved.append(r)
                                                    except Exception:
                                                        try:
                                                            resolved.append(_sanitize_url_for_fetch(str(s)))
                                                        except Exception:
                                                            pass
                                                article_imgs = list(dict.fromkeys(resolved))
                                                # If we found meaningful content via the recipe path, stop here
                                                if article_html:
                                                    engine = 'calibre recipe'
                                                    break
                                    except Exception:
                                        continue
                                if article_html:
                                    break
                            except Exception:
                                continue
                    except Exception:
                        _debug('[ARTICLE_FETCH_WORKER] recipe path exception: %s' % traceback.format_exc()[:500])
                        pass

                # Recipe-engine mode for full-article fetch (works for non-recipe feeds too)
                if not article_html and use_recipe_engine:
                    try:
                        _debug('[ARTICLE_FETCH_WORKER] calling _fetch_article_content_calibre_recipe_like')
                        article_html, article_imgs = _fetch_article_content_calibre_recipe_like(link, timeout=timeout, recipe_urn=recipe_urn)
                        _debug('[ARTICLE_FETCH_WORKER] recipe-like result len=%s imgs=%s' % (len(article_html or ''), len(article_imgs or [])))
                        if article_html:
                            engine = 'calibre recipe'
                    except Exception:
                        _debug('[ARTICLE_FETCH_WORKER] recipe-like exception: %s' % traceback.format_exc()[:500])
                        pass

                # If enabled for this feed, try calibre readability extraction for richer content.
                if not article_html and use_recipe_engine:
                    try:
                        _debug('[ARTICLE_FETCH_WORKER] calling _fetch_article_content_calibre_readability')
                        article_html, article_imgs = _fetch_article_content_calibre_readability(link, timeout=timeout)
                        _debug('[ARTICLE_FETCH_WORKER] readability result len=%s imgs=%s' % (len(article_html or ''), len(article_imgs or [])))
                        if article_html:
                            engine = 'calibre recipe'
                    except Exception:
                        _debug('[ARTICLE_FETCH_WORKER] readability exception: %s' % traceback.format_exc()[:500])
                        pass

                # If we didn't get article_html from recipe path, fall back
                if not article_html:
                    try:
                        _debug('[ARTICLE_FETCH_WORKER] calling _fetch_article_content')
                        article_html, article_imgs = _fetch_article_content(link, timeout=timeout)
                        _debug('[ARTICLE_FETCH_WORKER] fallback result len=%s imgs=%s' % (len(article_html or ''), len(article_imgs or [])))
                        if article_html and not engine:
                            engine = 'basic'
                    except Exception:
                        _debug('[ARTICLE_FETCH_WORKER] fallback exception: %s' % traceback.format_exc()[:500])
                        article_html, article_imgs = None, []
                # Ensure engine is always set for banner visibility
                try:
                    if not engine:
                        engine = engine_label or ('basic' if not use_recipe_engine else 'calibre recipe')
                except Exception:
                    if not engine:
                        engine = 'basic'
            except Exception:
                try:
                    article_html, article_imgs = _fetch_article_content(link, timeout=timeout)
                    if article_html and not engine:
                        engine = 'basic'
                except Exception:
                    article_html, article_imgs = None, []
            if result_q is not None:
                try:
                    result_q.put_nowait((article_html, article_imgs, engine))
                except Exception:
                    _debug('[ARTICLE_FETCH_WORKER] put_nowait failed: %s' % traceback.format_exc()[:500])
                    pass
            else:
                try:
                    _debug('[ARTICLE_FETCH_WORKER] dispatching result: engine=%s html_len=%s imgs=%s' % (str(engine), len(article_html or ''), len(article_imgs or [])))
                    Dispatcher(self._on_article_fetch_done)(generation, link, article_html, article_imgs, engine)
                except Exception:
                    _debug('[ARTICLE_FETCH_WORKER] dispatch exception: %s' % traceback.format_exc()[:500])

        try:
            t = threading.Thread(target=_worker, name='rss_reader_article_fetch', daemon=True)
            t.start()
        except Exception:
            # Fallback: run inline (best-effort)
            try:
                article_html, article_imgs = _fetch_article_content(link, timeout=timeout)
                Dispatcher(self._on_article_fetch_done)(generation, link, article_html, article_imgs, 'basic' if article_html else '')
            except Exception:
                pass

        # If we have a queue, poll it via QTimer so we can dispatch back on the main thread
        if result_q is not None:
            try:
                tmr = QTimer(self)
                tmr.setInterval(150)

                def _poll():
                    try:
                        item = result_q.get_nowait()
                    except Exception:
                        return
                    try:
                        tmr.stop()
                    except Exception:
                        pass
                    try:
                        article_html, article_imgs, engine = item
                        _debug('[ARTICLE_FETCH_POLL] got item engine=%s html_len=%s imgs=%s' % (str(engine), len(article_html or ''), len(article_imgs or [])))
                        Dispatcher(self._on_article_fetch_done)(generation, link, article_html, article_imgs, engine)
                    except Exception:
                        _debug('[ARTICLE_FETCH_POLL] dispatch exception: %s' % traceback.format_exc()[:500])
                        pass

                tmr.timeout.connect(_poll)
                tmr.start()
            except Exception:
                pass

    def _on_article_fetch_done(self, generation, link, article_html, article_imgs, engine=''):
        _debug("[ON_ARTICLE_FETCH_DONE] called:", {
            'generation': generation,
            'link': link,
            'engine': engine,
            'article_html_len': len(article_html or '') if article_html else 0,
            'imgs': len(article_imgs or []),
        })
        try:
            # Only apply results for the most recent preview generation
            if int(getattr(self, '_preview_generation', 0) or 0) != int(generation or 0):
                return
        except Exception:
            pass

        try:
            cur = getattr(self, '_current_item', None) or {}
            cur_link = str(cur.get('link') or '')
            if link and cur_link and str(link) != cur_link:
                return
        except Exception:
            pass

        # Mark fetch as no longer in-flight for this link.
        try:
            if str(getattr(self, '_preview_article_fetch_inflight_link', '') or '') == str(link):
                self._preview_article_fetch_inflight = False
                self._preview_article_fetch_inflight_link = ''
        except Exception:
            pass

        try:
            base = getattr(self, '_preview_base_html', '') or ''

            # (Removed) fetch-engine banner updates

            # Append fetched article with a separator — keep banner at the top.
            try:
                safe_article_html = _sanitize_article_html_for_embed(article_html)
                _debug('[SANITIZE] input len=%s output len=%s has_tags=%s' % (len(article_html or ''), len(safe_article_html or ''), '<' in safe_article_html))
            except Exception:
                safe_article_html = article_html or ''
                _debug('[SANITIZE] exception: %s' % traceback.format_exc()[:300])
            content_html = ('<div>%s</div>' % safe_article_html) if safe_article_html else ''
            insert_html = '<hr/>' + content_html
            try:
                low = (base or '').lower()
                b_idx = low.rfind('</body>')
                if b_idx != -1:
                    new_html = base[:b_idx] + insert_html + base[b_idx:]
                else:
                    h_idx = low.rfind('</html>')
                    if h_idx != -1:
                        new_html = base[:h_idx] + insert_html + base[h_idx:]
                    else:
                        if (base or '').strip().endswith('</body></html>'):
                            new_html = base[:-len('</body></html>')] + insert_html + '</body></html>'
                        else:
                            new_html = base + insert_html + '</body></html>'
            except Exception:
                new_html = base + insert_html + '</body></html>'

            try:
                _debug("[FT PREVIEW HTML] final (first 500 chars): %r" % new_html[:500])
                self.preview.setHtml(new_html)
            except Exception:
                try:
                    self.preview.setHtml(new_html)
                except Exception:
                    pass
            try:
                self._extract_and_set_preview_images(new_html, getattr(self.preview, '_page_base_url', ''))
            except Exception:
                pass
            # Update preview footer stats for the final HTML shown to the user
            try:
                text_only = self._strip_html_to_text(new_html)
                try:
                    if len(text_only or '') > 1024 * 1024:
                        chars = len((text_only or '')[:1024 * 1024])
                        words = len([w for w in (text_only or '')[:1024 * 1024].split() if w])
                        large = True
                    else:
                        chars = len(text_only or '')
                        words = len([w for w in (text_only or '').split() if w])
                        large = False
                except Exception:
                    chars = len(text_only or '')
                    words = len([w for w in (text_only or '').split() if w])
                    large = False
                images = len(getattr(self.preview, '_current_img_urls', []) or [])
                stats_str = _('%(chars)d chars, %(words)d words, %(images)d images') % {'chars': chars, 'words': words, 'images': images}
                if large:
                    stats_str += ' (' + _('truncated') + ')'
                try:
                    if getattr(self, 'preview_stats', None) is not None:
                        self.preview_stats.setText(stats_str)
                        try:
                            self.preview_stats.setToolTip(stats_str)
                        except Exception:
                            pass
                except Exception:
                    pass

            except Exception:
                pass
        except Exception:
            pass

    def _extract_and_set_preview_images(self, html, base):
        try:
            import re, urllib.parse

            def _first_url_from_srcset(ss):
                try:
                    ss = (ss or '').strip()
                    if not ss:
                        return ''
                    first = ss.split(',')[0].strip()
                    if not first:
                        return ''
                    return first.split()[0].strip()
                except Exception:
                    return ''

            imgs = []
            try:
                for m in re.finditer(r'<img[^>]+(?:src|data-src|data-original|data-lazy-src)=["\']([^"\']+)["\']', html, re.IGNORECASE):
                    src = m.group(1)
                    if src:
                        imgs.append(src)
                for m in re.finditer(r'(?:srcset|data-srcset)=["\']([^"\']+)["\']', html, re.IGNORECASE):
                    ss = m.group(1)
                    first = _first_url_from_srcset(ss)
                    if first:
                        imgs.append(first)
            except Exception:
                imgs = []

            resolved = []
            for s in imgs:
                try:
                    r = urllib.parse.urljoin(base or '', s)
                    r = _sanitize_url_for_fetch(r)
                    resolved.append(r)
                except Exception:
                    try:
                        resolved.append(_sanitize_url_for_fetch(str(s)))
                    except Exception:
                        pass

            try:
                self.preview._current_img_urls = list(dict.fromkeys(resolved))
            except Exception:
                try:
                    self.preview._current_img_urls = list(resolved)
                except Exception:
                    self.preview._current_img_urls = []
        except Exception:
            try:
                self.preview._current_img_urls = []
            except Exception:
                pass

        # Fallbacks
        try:
            __import__('calibre').gui2.open_url(url)
            return
        except Exception:
            pass
        try:
            from qt.core import QDesktopServices, QUrl
        except Exception:
            from PyQt5.Qt import QDesktopServices, QUrl
        try:
            QDesktopServices.openUrl(QUrl(str(url)))
        except Exception:
            pass

    def on_feeds_context_menu(self, pos):
        try:
            index = self.feeds_tree.indexAt(pos)
            item = None
            if index.isValid():
                item = self.feeds_tree.model().itemFromIndex(index)

            # Tree-wide actions should work even when right-clicking empty space.
            def _persist_expanded_folders(expand_all=False):
                try:
                    if expand_all:
                        folders = list(rss_db.get_folders() or [])
                        gprefs['rss_reader_expanded_folders'] = [str(x).strip().strip('/') for x in folders if str(x).strip()]
                    else:
                        gprefs['rss_reader_expanded_folders'] = []
                except Exception:
                    pass

            if item is None:
                menu = QMenu(self)
                expand_all_act = menu.addAction(_('Expand all folders'))
                collapse_all_act = menu.addAction(_('Collapse all folders'))
                menu.addSeparator()
                mark_all_read_act = menu.addAction(_('Mark all feeds read'))
                menu.addSeparator()
                add_root_folder_act = menu.addAction(_('Add folder'))
                act = menu.exec(self.feeds_tree.viewport().mapToGlobal(pos))
                if act == expand_all_act:
                    try:
                        self.feeds_tree.expandAll()
                    except Exception:
                        pass
                    _persist_expanded_folders(expand_all=True)
                elif act == collapse_all_act:
                    try:
                        self.feeds_tree.collapseAll()
                    except Exception:
                        pass
                    _persist_expanded_folders(expand_all=False)
                elif act == mark_all_read_act:
                    self.mark_all_feeds_read()
                elif act == add_root_folder_act:
                    # No selection / empty space: create a new top-level folder
                    self.add_folder('')
                return

            # Right-click context menus should not auto-mark items as seen.
            try:
                self._feed_selection_user_initiated = False
            except Exception:
                pass

            # Only change selection if right-clicked item is not already selected
            try:
                index = self.feeds_tree.indexAt(pos)
                if index.isValid():
                    index = index.sibling(index.row(), 0)  # Ensure column 0
                sel = self.feeds_tree.selectionModel()
                already_selected = False
                if sel is not None and index.isValid():
                    selected_indexes = sel.selectedRows(0)
                    for sidx in selected_indexes:
                        if sidx == index:
                            already_selected = True
                            break
                    if not already_selected:
                        try:
                            sel.select(index, sel.SelectionFlag.ClearAndSelect | sel.SelectionFlag.Rows)
                        except Exception:
                            try:
                                from PyQt5.Qt import QItemSelectionModel
                                sel.select(index, QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows)
                            except Exception:
                                try:
                                    sel.select(index, sel.SelectionFlag.ClearAndSelect)
                                except Exception:
                                    pass
                        try:
                            self.feeds_tree.setCurrentIndex(index)
                        except Exception:
                            pass
                try:
                    self.feeds_tree.setFocus()
                except Exception:
                    pass
            except Exception:
                pass

            data = item.data(ROLE_USER)
            feeds = list(rss_db.get_feeds() or [])
            feeds_by_id = {str(f.get('id') or ''): f for f in feeds}

            # Determine whether the current selection includes folders.
            try:
                indexes = self.feeds_tree.selectionModel().selectedIndexes()
                sel_items = [self.feeds_tree.model().itemFromIndex(idx) for idx in indexes]
            except Exception:
                sel_items = []
            has_folder_sel = False
            try:
                for it in sel_items:
                    d = it.data(ROLE_USER)
                    if isinstance(d, dict) and d.get('type') == 'folder':
                        has_folder_sel = True
                        break
            except Exception:
                has_folder_sel = False

            menu = QMenu(self)

            update_act = menu.addAction(_('Update'))
            export_act = menu.addAction(_('Export selected feeds to ebook'))
            send_email_act = menu.addAction(_('Send selected feeds via email…'))
            menu.addSeparator()

            add_subfolder_act = add_sibling_folder_act = add_root_folder_act = None
            rename_folder_act = delete_folder_act = move_folder_act = move_folder_new_act = None
            use_recipe_engine_folder_act = None
            clicked_folder_path = ''
            if isinstance(data, dict) and data.get('type') == 'root':
                add_root_folder_act = menu.addAction(_('Add folder'))
                menu.addSeparator()
            elif isinstance(data, dict) and data.get('type') == 'folder':
                clicked_folder_path = str(data.get('path') or '').strip().strip('/')
                add_subfolder_act = menu.addAction(_('Add subfolder'))
                add_sibling_folder_act = menu.addAction(_('Add sibling folder'))
                add_root_folder_act = menu.addAction(_('Add folder at root'))
                rename_folder_act = menu.addAction(_('Rename folder'))
                delete_folder_act = menu.addAction(_('Delete folder'))
                move_folder_act = menu.addAction(_('Move folder'))
                move_folder_new_act = menu.addAction(_('Move to new folder...'))
                menu.addSeparator()
                use_recipe_engine_folder_act = menu.addAction(_('Use calibre recipe engine for all feeds in this folder'))

            # Only enable Edit for feeds, not folders
            edit_act = None
            if isinstance(data, dict) and data.get('type') == 'feed':
                edit_act = menu.addAction(_('Edit'))

            # Feed tags (manual/user-defined)
            edit_feed_tags_act = clear_feed_tags_act = None
            try:
                if list(self.selected_feed_ids() or []):
                    edit_feed_tags_act = menu.addAction(_('Edit feed tags…'))
                    clear_feed_tags_act = menu.addAction(_('Clear feed tags'))
            except Exception:
                edit_feed_tags_act = clear_feed_tags_act = None

            # Suspend fetching (enable/disable) per-feed
            suspend_fetching_act = None
            try:
                sel_fids_for_state = list(self.selected_feed_ids() or [])
            except Exception:
                sel_fids_for_state = []
            if sel_fids_for_state:
                try:
                    feeds = list(rss_db.get_feeds() or [])
                    feeds_map = {str(f.get('id') or ''): f for f in feeds}
                    enabled_vals = []
                    for fid in sel_fids_for_state:
                        f = feeds_map.get(str(fid) or '')
                        if f is None:
                            continue
                        enabled_vals.append(bool(f.get('enabled', True)))
                    if enabled_vals:
                        all_enabled = all(enabled_vals)
                        all_disabled = not any(enabled_vals)
                        if all_enabled:
                            label = _('Suspend fetching')
                        elif all_disabled:
                            # Show current state inline (Qt menu checkmarks are not visible in this environment)
                            label = '🚫 ' + _('Enable fetching')
                        else:
                            label = _('Toggle fetching')
                        suspend_fetching_act = menu.addAction(label)
                except Exception:
                    suspend_fetching_act = None

            # Always notify per-feed (menu checkmarks are not visible here, so we use 🔔 in the label)
            always_notify_act = None
            try:
                sel_fids_for_state = list(self.selected_feed_ids() or [])
            except Exception:
                sel_fids_for_state = []
            if sel_fids_for_state:
                try:
                    feeds_map = {str(f.get('id') or ''): f for f in (feeds or [])}
                    notify_vals = []
                    for fid in sel_fids_for_state:
                        f = feeds_map.get(str(fid) or '')
                        if f is None:
                            continue
                        notify_vals.append(bool(f.get('always_notify', False)))
                    if notify_vals:
                        all_true = all(notify_vals)
                        all_false = not any(notify_vals)
                        if all_true:
                            label = '🔔 ' + _('Always notify')
                        elif all_false:
                            label = _('Always notify')
                        else:
                            label = _('Always notify (mixed)')
                        always_notify_act = menu.addAction(label)
                except Exception:
                    always_notify_act = None

            # Feed-level star (independent of starred items)
            feed_star_act = None
            try:
                sel_fids_for_state = list(self.selected_feed_ids() or [])
            except Exception:
                sel_fids_for_state = []
            if sel_fids_for_state:
                try:
                    feeds_map = {str(f.get('id') or ''): f for f in (feeds or [])}
                    star_vals = []
                    for fid in sel_fids_for_state:
                        f = feeds_map.get(str(fid) or '')
                        if f is None:
                            continue
                        star_vals.append(bool(f.get('feed_starred', False)))
                    if star_vals:
                        all_true = all(star_vals)
                        all_false = not any(star_vals)
                        if all_true:
                            label = '★ ' + _('Star feed')
                        elif all_false:
                            label = _('Star feed')
                        else:
                            label = _('Star feed (mixed)')
                        feed_star_act = menu.addAction(label)
                except Exception:
                    feed_star_act = None

            # Per-feed cleanup: prune old items for the selected feed(s).
            cleanup_feed_act = None
            try:
                if list(self.selected_feed_ids() or []):
                    cleanup_feed_act = menu.addAction(_('Clean up items for selected feed(s)…'))
            except Exception:
                cleanup_feed_act = None

            mark_read_act = menu.addAction(_('Mark read'))
            remove_act = menu.addAction(_('Remove folder(s) and feeds') if has_folder_sel else _('Remove feed(s)'))
            open_act = menu.addAction(_('Open feed home'))
            # Add a 'Fetch recipe now' action if any selected feed is a recipe
            fetch_recipe_act = None
            try:
                sel_feed_ids = []
                for it in sel_items:
                    try:
                        d = it.data(ROLE_USER)
                    except Exception:
                        d = None
                    if isinstance(d, dict) and d.get('type') == 'feed':
                        sel_feed_ids.append(str(d.get('id') or ''))
                def _is_recipe_feed(feed):
                    if not feed:
                        return False
                    if feed.get('is_recipe'):
                        return True
                    url = feed.get('url') or ''
                    return url.startswith('builtin:')

                any_recipe = any(_is_recipe_feed(feeds_by_id.get(fid)) for fid in sel_feed_ids)
                if any_recipe:
                    fetch_recipe_act = menu.addAction(_('Fetch recipe now'))
                    copy_recipe_urls_act = menu.addAction(_('Copy all recipe feed URLs'))
            except Exception:
                fetch_recipe_act = None
                copy_recipe_urls_act = None

            # Clarify intent via hover text
            try:
                prev = str(self.status.text() or '')
                def _wire(a, tip):
                    try:
                        a.setStatusTip(str(tip))
                    except Exception:
                        pass
                    try:
                        a.hovered.connect(lambda _t=str(tip): self.status.setText(_t))
                    except Exception:
                        pass
                _wire(update_act, _('Fetch updates for the selected feed(s)'))
                _wire(export_act, _('Export selected feed cache to an ebook'))
                if add_subfolder_act is not None:
                    _wire(add_subfolder_act, _('Create a new folder inside the selected folder'))
                if add_sibling_folder_act is not None:
                    _wire(add_sibling_folder_act, _('Create a new folder alongside the selected folder'))
                if add_root_folder_act is not None:
                    _wire(add_root_folder_act, _('Create a new top-level folder'))
                if rename_folder_act is not None:
                    _wire(rename_folder_act, _('Rename the selected folder'))
                if delete_folder_act is not None:
                    _wire(delete_folder_act, _('Delete the folder (feeds are moved to parent folder)'))
                if move_folder_act is not None:
                    _wire(move_folder_act, _('Move the folder to another location'))
                if move_folder_new_act is not None:
                    _wire(move_folder_new_act, _('Create a new folder at the top level, then move the selected folder(s) into it'))
                if use_recipe_engine_folder_act is not None:
                    _wire(use_recipe_engine_folder_act, _('Enable calibre recipe engine for all feeds in this folder and subfolders'))
                _wire(edit_act, _('Edit feed URL/title'))
                if edit_feed_tags_act is not None:
                    _wire(edit_feed_tags_act, _('Set tags for the selected feed(s) (manual tags only)'))
                if clear_feed_tags_act is not None:
                    _wire(clear_feed_tags_act, _('Remove all manual tags from the selected feed(s)'))
                if feed_star_act is not None:
                    _wire(feed_star_act, _('Toggle a feed-level star (independent of starred articles)'))
                _wire(mark_read_act, _('Mark the selected feed(s) as read (clears unread badge)'))
                if has_folder_sel:
                    _wire(remove_act, _('Remove selected folder(s) and contained feeds from RSS Reader (does not affect the website)'))
                else:
                    _wire(remove_act, _('Remove selected feed(s) from RSS Reader (does not affect the website)'))
                _wire(open_act, _('Open the feed URL in your browser'))
                try:
                    menu.aboutToHide.connect(lambda _p=prev: self.status.setText(_p))
                except Exception:
                    pass
            except Exception:
                pass

            move_action_to_folder = {}
            if isinstance(data, dict) and data.get('type') == 'feed':
                menu.addSeparator()
                move_to_folder_menu = menu.addMenu(_('Move to folder'))
                root_act = move_to_folder_menu.addAction(_('(Top level)'))
                move_action_to_folder[root_act] = ''
                folders = list(rss_db.get_folders() or [])
                for fp in sorted({str(x).strip().strip('/') for x in folders if str(x).strip()}):
                    a = move_to_folder_menu.addAction(fp)
                    move_action_to_folder[a] = fp
                move_to_folder_menu.addSeparator()
                newf_act = move_to_folder_menu.addAction(_('New folder...'))
                move_action_to_folder[newf_act] = '__new__'
                # Capture selected feed IDs for move-to-new-folder
                selected_fids_for_move = list(self.selected_feed_ids() or [])

            act = menu.exec(self.feeds_tree.viewport().mapToGlobal(pos))
            if act is None:
                return

            if 'copy_recipe_urls_act' in locals() and act == copy_recipe_urls_act:
                # Copy all URLs of selected recipe feeds to clipboard
                try:
                    from calibre_plugins.rss_reader.recipe_utils import get_recipe_feeds_from_urn
                except Exception:
                    get_recipe_feeds_from_urn = None
                try:
                    recipe_urls = []
                    seen = set()
                    for fid in sel_feed_ids:
                        feed = feeds_by_id.get(fid)
                        if not feed or not _is_recipe_feed(feed):
                            continue
                        url = feed.get('url') or ''
                        # If it's a recipe URN and we can resolve subfeeds, use those
                        subfeeds = []
                        if get_recipe_feeds_from_urn and url.startswith('builtin:'):
                            try:
                                subfeeds = get_recipe_feeds_from_urn(url) or []
                            except Exception:
                                subfeeds = []
                        if subfeeds:
                            for _t, _u in subfeeds:
                                if _u and _u not in seen:
                                    recipe_urls.append(_u)
                                    seen.add(_u)
                        elif url and url not in seen:
                            recipe_urls.append(url)
                            seen.add(url)
                    if recipe_urls:
                        cb = QApplication.instance().clipboard()
                        cb.setText('\n'.join(recipe_urls))
                        QMessageBox.information(self, _('Copy Recipe Feed URLs'), _('Copied %d recipe feed URL(s) to clipboard.') % len(recipe_urls))
                    else:
                        QMessageBox.information(self, _('Copy Recipe Feed URLs'), _('No recipe feed URLs found in selection.'))
                except Exception as e:
                    error_dialog(self, _('RSS Reader Error'), _('Failed to copy recipe feed URLs: %s') % str(e), show=True, det_msg=traceback.format_exc())
                return
            if act == update_act:
                self.update_selected(); return
            if act == fetch_recipe_act:
                # Fetch recipes for selected feeds
                try:
                    sel = [str(it.data(ROLE_USER).get('id') or '') for it in sel_items if isinstance(it.data(ROLE_USER), dict) and it.data(ROLE_USER).get('type') == 'feed']
                except Exception:
                    sel = []
                if sel:
                    self.fetch_recipes_now(sel)
                return
            if act == export_act:
                self.export_selected_feeds(); return
            if act == send_email_act:
                try:
                    dlg = SendFeedsViaEmailDialog(self)
                    if dlg.exec() != QDialog.DialogCode.Accepted:
                        return
                    to_email, subject, message = dlg.get_values()
                    from calibre_plugins.rss_reader import export_logic
                    export_logic.email_selected_feeds(self, to_email, subject=subject, message=message)
                except Exception as e:
                    error_dialog(self, _('RSS Reader Error'), _('Failed to send feeds via email: %s') % str(e), show=True, det_msg=traceback.format_exc())
                return

            if add_subfolder_act is not None and act == add_subfolder_act:
                # Use the folder that was right-clicked, not inferred selection
                self.add_folder(clicked_folder_path); return
            if add_sibling_folder_act is not None and act == add_sibling_folder_act:
                parent_path = str(clicked_folder_path).rpartition('/')[0]
                self.add_folder(parent_path); return
            if add_root_folder_act is not None and act == add_root_folder_act:
                self.add_folder(''); return
            if rename_folder_act is not None and act == rename_folder_act:
                self.rename_folder(clicked_folder_path); return
            if delete_folder_act is not None and act == delete_folder_act:
                self.delete_folder(clicked_folder_path); return
            if move_folder_act is not None and act == move_folder_act:
                try:
                    sel_folders = list(self.selected_folder_paths() or [])
                except Exception:
                    sel_folders = []
                # If selection doesn't include any folders, fall back to the clicked folder.
                if not sel_folders:
                    sel_folders = [str(clicked_folder_path or '').strip().strip('/')]
                # If multiple folders are selected, move them all.
                try:
                    sel_folders = [str(x or '').strip().strip('/') for x in sel_folders if str(x or '').strip()]
                except Exception:
                    sel_folders = []
                if len(sel_folders) > 1:
                    self.move_folders(sel_folders)
                else:
                    self.move_folder(sel_folders[0] if sel_folders else clicked_folder_path)
                return

            if move_folder_new_act is not None and act == move_folder_new_act:
                try:
                    sel_folders = list(self.selected_folder_paths() or [])
                except Exception:
                    sel_folders = []
                if not sel_folders:
                    sel_folders = [str(clicked_folder_path or '').strip().strip('/')]
                try:
                    sel_folders = [str(x or '').strip().strip('/') for x in sel_folders if str(x or '').strip()]
                except Exception:
                    sel_folders = []
                if not sel_folders:
                    return
                target = self.add_folder('')
                if not target:
                    return
                try:
                    self.move_folders(sel_folders, target=target)
                except Exception:
                    # Fallback: move single folder
                    try:
                        if len(sel_folders) == 1:
                            # Move the folder under the newly created folder
                            self.move_folders(sel_folders, target=target)
                    except Exception:
                        pass
                return
            if use_recipe_engine_folder_act is not None and act == use_recipe_engine_folder_act:
                # Enable calibre recipe engine for all feeds in this folder and subfolders
                try:
                    feeds = list(rss_db.get_feeds() or [])
                    prefix = str(clicked_folder_path or '').strip().strip('/') + '/'
                    if not prefix or prefix == '/':
                        prefix = ''
                    changed = False
                    for f in feeds:
                        folder = str(f.get('folder') or '').strip().strip('/')
                        # Match this folder or subfolders
                        if not prefix:
                            # Root folder: all feeds
                            if not bool(f.get('use_recipe_engine', False)):
                                f['use_recipe_engine'] = True
                                changed = True
                        elif folder == prefix.rstrip('/') or folder.startswith(prefix):
                            # Exact match or subfolder
                            if not bool(f.get('use_recipe_engine', False)):
                                f['use_recipe_engine'] = True
                                changed = True
                    if changed:
                        try:
                            rss_db.save_feeds(feeds)
                        except Exception:
                            pass
                        self.refresh()
                        QMessageBox.information(self, _('Recipe Engine'), _('Calibre recipe engine enabled for feeds in this folder and subfolders.'))
                    else:
                        QMessageBox.information(self, _('Recipe Engine'), _('All feeds in this folder already use the calibre recipe engine.'))
                except Exception as e:
                    error_dialog(self, _('RSS Reader Error'), _('Failed to enable recipe engine: %s') % str(e), show=True, det_msg=traceback.format_exc())
                return

            if act in move_action_to_folder:
                target = str(move_action_to_folder.get(act) or '')
                if target == '__new__':
                    # Use captured selection for new folder move
                    target = self.add_folder('')
                    if target and selected_fids_for_move:
                        # Move the originally selected feeds to the new folder
                        feeds = list(rss_db.get_feeds() or [])
                        fid_set = set(selected_fids_for_move)
                        for f in feeds:
                            if str(f.get('id') or '') in fid_set:
                                f['folder'] = target
                        try:
                            rss_db.save_feeds(feeds)
                        except Exception:
                            pass
                        folders = list(rss_db.get_folders() or [])
                        if target not in folders:
                            folders.append(target)
                            try:
                                rss_db.save_folders(folders)
                            except Exception:
                                pass
                        self.refresh()
                    return

                # Allow moving feeds to top level (empty folder).
                self.move_selected_feeds_to_folder(target)
                return

            if act == edit_act:
                self.edit_selected_feed(); return
            if edit_feed_tags_act is not None and act == edit_feed_tags_act:
                self.edit_tags_for_selected_feeds(); return
            if clear_feed_tags_act is not None and act == clear_feed_tags_act:
                self.clear_tags_for_selected_feeds(); return
            if suspend_fetching_act is not None and act == suspend_fetching_act:
                # Toggle enabled state for selected feeds
                sel_fids = self.selected_feed_ids()
                if sel_fids:
                    feeds = list(rss_db.get_feeds() or [])
                    fid_set = set(sel_fids)
                    changed = False
                    for i, f in enumerate(feeds):
                        if str(f.get('id') or '') in fid_set:
                            feeds[i]['enabled'] = not bool(f.get('enabled', True))
                            changed = True
                    if changed:
                        try:
                            rss_db.save_feeds(feeds)
                        except Exception:
                            pass
                        self.refresh()
                return

            if always_notify_act is not None and act == always_notify_act:
                # Toggle always_notify for selected feeds.
                # If selection is mixed/false -> set True; if all true -> set False.
                try:
                    sel_fids = list(self.selected_feed_ids() or [])
                except Exception:
                    sel_fids = []
                if sel_fids:
                    feeds = list(rss_db.get_feeds() or [])
                    feeds_map = {str(f.get('id') or ''): f for f in feeds}
                    cur_vals = []
                    for fid in sel_fids:
                        f = feeds_map.get(str(fid) or '')
                        if f is None:
                            continue
                        cur_vals.append(bool(f.get('always_notify', False)))
                    target = False if (cur_vals and all(cur_vals)) else True

                    fid_set = set(str(x) for x in sel_fids)
                    changed = False
                    for i, f in enumerate(feeds):
                        if str(f.get('id') or '') in fid_set:
                            feeds[i]['always_notify'] = bool(target)
                            changed = True
                    if changed:
                        try:
                            rss_db.save_feeds(feeds)
                        except Exception:
                            pass
                        self.refresh()
                return

            if feed_star_act is not None and act == feed_star_act:
                # Toggle feed_starred for selected feeds.
                # If selection is mixed/false -> set True; if all true -> set False.
                try:
                    sel_fids = list(self.selected_feed_ids() or [])
                except Exception:
                    sel_fids = []
                if sel_fids:
                    feeds = list(rss_db.get_feeds() or [])
                    feeds_map = {str(f.get('id') or ''): f for f in feeds}
                    cur_vals = []
                    for fid in sel_fids:
                        f = feeds_map.get(str(fid) or '')
                        if f is None:
                            continue
                        cur_vals.append(bool(f.get('feed_starred', False)))
                    target = False if (cur_vals and all(cur_vals)) else True

                    fid_set = set(str(x) for x in sel_fids)
                    changed = False
                    for i, f in enumerate(feeds):
                        if str(f.get('id') or '') in fid_set:
                            feeds[i]['feed_starred'] = bool(target)
                            changed = True
                    if changed:
                        try:
                            rss_db.save_feeds(feeds)
                        except Exception:
                            pass
                        self.refresh()
                return
            if cleanup_feed_act is not None and act == cleanup_feed_act:
                try:
                    sel_fids = list(self.selected_feed_ids() or [])
                except Exception:
                    sel_fids = []
                if sel_fids:
                    self._show_cleanup_dialog(preselected_feed_ids=sel_fids)
                return
            if act == mark_read_act:
                self.mark_selected_feed_read(); return
            if act == remove_act:
                self.remove_selected_feed(); return
            if act == open_act:
                fid = self.selected_feed_id()
                feed = feeds_by_id.get(fid)
                if feed is not None:
                    link = feed.get('url') or ''
                    try:
                        self._open_url(link)
                    except Exception:
                        pass
                return
        except Exception as e:
            error_dialog(self, _('RSS Reader Error'), _('Feed menu error: %s') % str(e), show=True, det_msg=traceback.format_exc())

    def fetch_recipes_now(self, feed_ids):
        """Fetch recipe(s) now using Calibre's recipe fetch pipeline and add results to library."""
        try:
            if not feed_ids:
                return
            try:
                from calibre.gui2.tools import fetch_scheduled_recipe
            except Exception:
                fetch_scheduled_recipe = None
            for fid in feed_ids:
                try:
                    feeds = list(rss_db.get_feeds() or [])
                    feed = next((f for f in feeds if str(f.get('id') or '') == str(fid)), None)
                    if not feed or not feed.get('is_recipe') or not fetch_scheduled_recipe:
                        continue
                    arg = {'title': feed.get('title') or '', 'urn': feed.get('recipe_urn') or feed.get('url') or '', 'username': None, 'password': None}
                    func, args, desc, fmt, temp_files = fetch_scheduled_recipe(arg)
                    job = self.gui.job_manager.run_job(Dispatcher(self._recipe_fetched), func, args=args, description=desc)
                    self._recipe_jobs[job] = (temp_files, fmt, arg, fid)
                    try:
                        self.gui.status_bar.show_message(_('Fetching recipe: %s') % (feed.get('title') or ''), 4000)
                    except Exception:
                        pass
                except Exception:
                    pass
        except Exception as e:
            error_dialog(self, _('Fetch recipes failed'), _('Failed to fetch recipe(s): %s') % str(e), show=True, det_msg=traceback.format_exc())

    def _recipe_fetched(self, job):
        try:
            v = self._recipe_jobs.pop(job, None)
            if not v:
                return
            temp_files, fmt, arg, fid = v
            fname = temp_files[0].name if temp_files else None
            if job.failed:
                return self.gui.job_exception(job)
            # Use the same behavior as Calibre's Fetch News: add the resulting news ebook to library
            try:
                id = self.gui.library_view.model().add_news(fname, arg)
                try:
                    self.gui.status_bar.show_message(_('%s fetched and added to library') % (arg.get('title') or ''), 5000)
                except Exception:
                    pass
            except Exception:
                # Just show the job exception if available
                try:
                    self.gui.job_exception(job)
                except Exception:
                    pass
        except Exception:
            pass

    def on_items_column_resized(self, index, old, new):
        try:
            widths = gprefs.get('rss_reader_items_table_widths', {}) or {}
            widths[str(index)] = int(new or 0)
            gprefs['rss_reader_items_table_widths'] = widths
        except Exception:
            pass

    def edit_selected_feed(self):
        feed_id = self.selected_feed_id()
        if not feed_id:
            return
        feeds = list(rss_db.get_feeds() or [])
        feed = next((f for f in feeds if str(f.get('id') or '') == feed_id), None)
        if feed is None:
            return
        # reuse Add dialog but prefill
        dlg = _AddFeedDialog(self)
        dlg.setWindowTitle(_('Edit feed'))
        dlg.url_edit.setText(feed.get('url') or '')
        dlg.title_edit.setText(feed.get('title') or '')
        # Add enabled checkbox and Test button
        try:
            from qt.core import QCheckBox, QSpinBox
        except Exception:
            from PyQt5.Qt import QCheckBox, QSpinBox
        enabled_cb = QCheckBox(_('Enabled'), dlg)
        enabled_cb.setChecked(bool(feed.get('enabled', True)))
        dlg.layout().insertWidget(2, enabled_cb)
        always_notify_cb = QCheckBox(_('Always notify'), dlg)
        always_notify_cb.setChecked(bool(feed.get('always_notify', False)))
        dlg.layout().insertWidget(3, always_notify_cb)
        # Per-feed cleanup retention in days (0 = use global cleanup default)
        retention_lbl = QLabel(_('Cleanup retention (days, 0 = use default):'), dlg)
        try:
            rd_val = int(feed.get('retention_days') or 0)
        except Exception:
            rd_val = 0
        retention_spin = QSpinBox(dlg)
        retention_spin.setRange(0, 9999)
        retention_spin.setValue(max(0, rd_val))
        dlg.layout().insertWidget(4, retention_lbl)
        dlg.layout().insertWidget(5, retention_spin)
        # Prefill oldest/max (use Calibre-like defaults if missing)
        try:
            default_oldest = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
        except Exception:
            default_oldest = 7
        try:
            default_max = int(plugin_prefs.get('default_max_articles', 100) or 100)
        except Exception:
            default_max = 100
        try:
            val = feed.get('oldest_article_days')
            od = int(val) if val is not None else int(default_oldest)
        except (ValueError, TypeError):
            od = int(default_oldest)
        except Exception:
            od = int(default_oldest)
        try:
            val = feed.get('max_articles')
            ma = int(val) if val is not None else int(default_max)
        except (ValueError, TypeError):
            ma = int(default_max)
        except Exception:
            ma = int(default_max)
        try:
            if getattr(dlg, 'oldest_days_spin', None) is not None:
                dlg.oldest_days_spin.setValue(max(0, od))
        except Exception:
            pass
        try:
            if getattr(dlg, 'max_articles_spin', None) is not None:
                dlg.max_articles_spin.setValue(max(0, ma))
        except Exception:
            pass
        # Remove duplicate download_images checkbox: just use the one from the Add dialog
        dlg.download_images_cb.setText(_('Download images for this feed'))
        dlg.download_images_cb.setChecked(bool(feed.get('download_images', True)))
        # Prefill per-feed recipe-engine opt-in if present
        try:
            if getattr(dlg, 'use_recipe_engine_cb', None) is not None:
                dlg.use_recipe_engine_cb.setChecked(bool(feed.get('use_recipe_engine', False)))
        except Exception:
            pass
        # Show recipe indicator / allow clearing it
        try:
            if feed.get('is_recipe'):
                recipe_info_lbl = QLabel(_('Recipe: %s') % str(feed.get('recipe_urn') or ''), dlg)
                dlg.layout().insertWidget(6, recipe_info_lbl)
                # Allow clearing recipe association
                clear_recipe_btn = QPushButton(_('Clear recipe association'), dlg)
                def _clear_recipe():
                    try:
                        dlg.url_edit.setText('')
                    except Exception:
                        pass
                clear_recipe_btn.clicked.connect(_clear_recipe)
                dlg.layout().insertWidget(7, clear_recipe_btn)
        except Exception:
            pass
        test_btn = QPushButton(_('Test'), dlg)
        def on_test():
            url = dlg.url_edit.text().strip()
            if not url:
                error_dialog(dlg, _('Test failed'), _('No URL provided'), show=True)
                return
            try:
                from calibre_plugins.rss_reader.rss import fetch_url, parse_feed
                from calibre_plugins.rss_reader.recipe_utils import is_recipe_urn, get_recipe_feeds_from_urn

                timeout = int(plugin_prefs.get('timeout_seconds', 25) or 25)

                if is_recipe_urn(url):
                    feeds = get_recipe_feeds_from_urn(url)
                    if not feeds:
                        raise ValueError('Recipe provides no RSS feeds')
                    # Test the first feed URL
                    sub_title, sub_url = feeds[0]
                    raw, final = fetch_url(sub_url, timeout_seconds=timeout)
                    parsed = parse_feed(raw, base_url=final)
                    msg = _('Recipe provides %(n)d feed(s). First feed parsed: %(t)s') % {
                        'n': len(feeds),
                        't': (parsed.get('title') or sub_title or sub_url or url),
                    }
                    QMessageBox.information(dlg, _('Test successful'), msg)
                    # Keep the dialog title as-is unless empty
                    if not dlg.title_edit.text().strip():
                        dlg.title_edit.setText(str(feed.get('title') or url))
                else:
                    raw, final = fetch_url(url, timeout_seconds=timeout)
                    parsed = parse_feed(raw, base_url=final)
                    title = parsed.get('title') or ''
                    if title:
                        dlg.title_edit.setText(title)
                        QMessageBox.information(dlg, _('Test successful'), _('Feed parsed successfully: %s') % title)
                    else:
                        QMessageBox.information(dlg, _('Test successful'), _('Feed parsed but title not found'))
            except Exception as e:
                error_dialog(dlg, _('Test failed'), _('Could not fetch/parse feed: %s') % str(e), show=True)
        test_btn.clicked.connect(on_test)
        dlg.layout().insertWidget(3, test_btn)

        # The base Add dialog is sized for a compact UI. Once we inject extra
        # controls (Enabled/Always notify/Test/etc), ensure the dialog grows so
        # those widgets aren't clipped below the bottom edge.
        try:
            dlg.adjustSize()
            try:
                dlg.resize(max(520, dlg.width()), max(260, dlg.height()))
            except Exception:
                pass
        except Exception:
            pass

        # Change OK button text to 'Save'
        try:
            if getattr(dlg, 'ok_btn', None) is not None:
                dlg.ok_btn.setText(_('Save'))
        except Exception:
            pass
        # Back-compat: if we can't find the stored OK button, fall back to scanning.
        if getattr(dlg, 'ok_btn', None) is None:
            for w in dlg.findChildren(QPushButton):
                if w.text() == _('Add') or w.text() == 'Add':
                    w.setText(_('Save'))

        # Swap button order in this dialog: Save then Cancel
        try:
            bl = getattr(dlg, '_buttons_layout', None)
            okb = getattr(dlg, 'ok_btn', None)
            ccb = getattr(dlg, 'cancel_btn', None)
            if bl is not None and okb is not None and ccb is not None:
                try:
                    bl.removeWidget(okb)
                except Exception:
                    pass
                try:
                    bl.removeWidget(ccb)
                except Exception:
                    pass
                bl.addWidget(okb)
                bl.addWidget(ccb)
        except Exception:
            pass
        if dlg.exec() != QDialog.DialogCode.Accepted:
            return
        url, title, download_images, use_recipe_engine, oldest_days, max_articles = dlg.get_values()
        if not url:
            return
        enabled = enabled_cb.isChecked()
        always_notify = always_notify_cb.isChecked()
        # Update feed entry
        for i, f in enumerate(feeds):
            if str(f.get('id') or '') == feed_id:
                feeds[i]['url'] = url
                feeds[i]['title'] = title or url
                feeds[i]['enabled'] = enabled
                feeds[i]['always_notify'] = always_notify
                feeds[i]['download_images'] = bool(dlg.download_images_cb.isChecked())
                # Persist per-feed cleanup retention (days). 0 means
                # "inherit" the global cleanup default.
                try:
                    feeds[i]['retention_days'] = int(retention_spin.value() or 0)
                except Exception:
                    feeds[i]['retention_days'] = 0
                try:
                    feeds[i]['use_recipe_engine'] = bool(getattr(dlg, 'use_recipe_engine_cb', None) and dlg.use_recipe_engine_cb.isChecked())
                except Exception:
                    pass

                # Persist per-feed limits (defaults are already in the dialog).
                try:
                    if getattr(dlg, 'oldest_days_spin', None) is not None:
                        feeds[i]['oldest_article_days'] = int(dlg.oldest_days_spin.value())
                    else:
                        feeds[i]['oldest_article_days'] = int(oldest_days)
                except Exception:
                    feeds[i]['oldest_article_days'] = int(default_oldest)
                try:
                    if getattr(dlg, 'max_articles_spin', None) is not None:
                        feeds[i]['max_articles'] = int(dlg.max_articles_spin.value())
                    else:
                        feeds[i]['max_articles'] = int(max_articles)
                except Exception:
                    feeds[i]['max_articles'] = int(default_max)
                # clear recipe association if URL isn't a recipe urn
                try:
                    from calibre_plugins.rss_reader.recipe_utils import is_recipe_urn
                    if is_recipe_urn(url):
                        feeds[i]['is_recipe'] = True
                        feeds[i]['recipe_urn'] = str(url or '')
                    else:
                        feeds[i]['is_recipe'] = False
                        feeds[i]['recipe_urn'] = ''
                except Exception:
                    # conservative default
                    feeds[i]['is_recipe'] = False
                    feeds[i]['recipe_urn'] = ''
                break
        try:
            rss_db.save_feeds(feeds)
        except Exception:
            pass
        self.refresh()

    def edit_selected_feed_or_folder(self):
        from calibre_plugins.rss_reader import folders_logic
        return folders_logic.edit_selected_feed_or_folder(self)

    def update_toolbar_button_states(self):
        """Enable/disable toolbar buttons based on current selection."""
        try:
            # Get selected items types
            indexes = self.feeds_tree.selectionModel().selectedIndexes()
            sel_items = [self.feeds_tree.model().itemFromIndex(idx) for idx in indexes]
            types = []
            for it in sel_items:
                d = it.data(ROLE_USER)
                if isinstance(d, dict):
                    types.append(d.get('type'))

            has_single_feed = types == ['feed']
            has_single_folder = types == ['folder']
            has_single_root = types == ['root']

            # Edit button: enabled for single feed or single folder (not root or multiple)
            if getattr(self, 'edit_btn', None) is not None:
                self.edit_btn.setEnabled(has_single_feed or has_single_folder)

            # Other buttons...
            has_feed_selected = bool(self.selected_feed_id())
            has_folder_selected = bool(self.selected_folder_path())
            has_any_selected = has_feed_selected or has_folder_selected
            has_multiple_feeds = len(self.selected_feed_ids()) > 1

            # Mark read button: enabled if feeds are selected
            if getattr(self, 'mark_read_btn', None) is not None:
                self.mark_read_btn.setEnabled(has_feed_selected)

            # Update button: enabled if feeds are selected
            if getattr(self, 'update_btn', None) is not None:
                self.update_btn.setEnabled(has_feed_selected)

            # Update all button: always enabled (updates all enabled feeds)
            if getattr(self, 'update_all_btn', None) is not None:
                self.update_all_btn.setEnabled(True)

            # Remove button: enabled if any feeds or folders are selected
            if getattr(self, 'remove_btn', None) is not None:
                self.remove_btn.setEnabled(has_any_selected)

            # Export button: enabled if feeds are selected
            if getattr(self, 'export_ebook_btn', None) is not None:
                self.export_ebook_btn.setEnabled(has_feed_selected)

            # Save article button: enabled if an item is selected
            has_item_selected = bool(self.selected_item())
            if getattr(self, 'save_article_btn', None) is not None:
                self.save_article_btn.setEnabled(has_item_selected)

            # Share button: enabled if an item is selected
            if getattr(self, 'share_btn', None) is not None:
                self.share_btn.setEnabled(has_item_selected)

        except Exception:
            # Disable all if error
            for btn_name in ['edit_btn', 'mark_read_btn', 'update_btn', 'update_all_btn', 'remove_btn', 'export_ebook_btn', 'save_article_btn', 'share_btn']:
                btn = getattr(self, btn_name, None)
                if btn is not None:
                    try:
                        btn.setEnabled(False)
                    except Exception:
                        pass

    def add_folder(self, parent_path=''):
        from calibre_plugins.rss_reader import folders_logic
        return folders_logic.add_folder(self, parent_path=parent_path)

    def rename_folder(self, folder_path):
        from calibre_plugins.rss_reader import folders_logic
        return folders_logic.rename_folder(self, folder_path)

    def delete_folder(self, folder_path):
        from calibre_plugins.rss_reader import folders_logic
        return folders_logic.delete_folder(self, folder_path)

    def move_selected_feeds_to_folder(self, folder_path):
        from calibre_plugins.rss_reader import folders_logic
        return folders_logic.move_selected_feeds_to_folder(self, folder_path)

    def move_folder(self, folder_path):
        from calibre_plugins.rss_reader import folders_logic
        return folders_logic.move_folder(self, folder_path)

    def move_folders(self, folder_paths, target=None):
        from calibre_plugins.rss_reader import folders_logic
        return folders_logic.move_folders(self, folder_paths, target=target)

    def purge_database(self):
        from calibre_plugins.rss_reader import cleanup_logic
        return cleanup_logic.purge_database(self)

    def _show_cleanup_dialog(self, preselected_feed_ids=None):
        from calibre_plugins.rss_reader import cleanup_logic
        return cleanup_logic.show_cleanup_dialog(self, preselected_feed_ids=preselected_feed_ids)

    def open_cleanup_dialog(self):
        """Entry point for toolbar button: clean up across all feeds."""
        from calibre_plugins.rss_reader import cleanup_logic
        return cleanup_logic.open_cleanup_dialog(self)

    def _run_cleanup_for_feeds(self, feed_ids, max_days=None, max_items=None,
                               never_unread=True, never_starred=True,
                               never_tagged=True, purge_after=True):
        from calibre_plugins.rss_reader import cleanup_logic
        return cleanup_logic.run_cleanup_for_feeds(self, feed_ids, max_days=max_days, max_items=max_items,
                                                   never_unread=never_unread, never_starred=never_starred,
                                                   never_tagged=never_tagged, purge_after=purge_after)

# Resolve `_RSSReaderDialogPartA` at import time to avoid NameError when
# circular imports cause it not to be defined yet. Fall back to importing
# the module and using a safe `object` base if necessary.
try:
    _base_A = _RSSReaderDialogPartA
except NameError:
    try:
        from calibre_plugins.rss_reader import ui_part1 as _ui_part1
        _base_A = getattr(_ui_part1, '_RSSReaderDialogPartA', object)
    except Exception:
        _base_A = object

class RSSReaderDialog(_base_A, _RSSReaderDialogPartB, ProfilesMixin, AIPanelMixin, QMainWindow):
    pass
