from __future__ import absolute_import

import uuid
import traceback
import base64
import tempfile
import os
import shutil
import threading
import queue
import hashlib
import time
import re
from calibre_plugins.rss_reader.debug import _debug, DEBUG_RSS_READER
from calibre_plugins.rss_reader.utils import iso_to_ts, format_published_display

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import (
        Qt,
        QAction,
        QActionGroup,
        QEvent,
        QObject,
        QAbstractItemView,
        QComboBox,
        QDialog,
        QDialogButtonBox,
        QDateTime,
        QGridLayout,
        QHBoxLayout,
        QInputDialog,
        QIcon,
        QLabel,
        QLineEdit,
        QMessageBox,
        QPushButton,
        QSizePolicy,
        QToolButton,
        QSize,
        QSplitter,
        QTableWidget,
        QTableWidgetItem,
        QTextBrowser,
        QTextDocument,
        QTimer,
        QUrl,
        QImage,
        QVBoxLayout,
        QWidget,
        QMenu,
        QColor,
        QPalette,
        QApplication,
        QFileDialog,
        QProgressDialog,
        QTreeWidget,
        QTreeWidgetItem,
        QTreeView,
        QStandardItemModel,
        QStandardItem,
        QModelIndex,
        QItemSelectionModel,
        QMainWindow,
        QDockWidget,
        QScrollArea,
        QListWidget,
        QListWidgetItem,
        pyqtSignal,
        QThread,
    )
except ImportError:
    from PyQt5.Qt import (
        Qt,
        QAction,
        QActionGroup,
        QAbstractItemView,
        QComboBox,
        QDialog,
        QDialogButtonBox,
        QDateTime,
        QGridLayout,
        QHBoxLayout,
        QInputDialog,
        QIcon,
        QLabel,
        QLineEdit,
        QMessageBox,
        QPushButton,
        QSizePolicy,
        QToolButton,
        QSize,
        QSplitter,
        QTableWidget,
        QTableWidgetItem,
        QTextBrowser,
        QTextDocument,
        QTimer,
        QUrl,
        QImage,
        QVBoxLayout,
        QWidget,
        QMenu,
        QColor,
        QPalette,
        QApplication,
        QFileDialog,
        QProgressDialog,
        QTreeWidget,
        QTreeWidgetItem,
        QTreeView,
        QStandardItemModel,
        QStandardItem,
        QModelIndex,
        QItemSelectionModel,
        QMainWindow,
        QDockWidget,
        QScrollArea,
        QListWidget,
        QListWidgetItem,
    )
from PyQt5.QtCore import QObject, pyqtSignal, QThread
from calibre.gui2 import Dispatcher, error_dialog, gprefs
from calibre.gui2.notify import get_notifier
from calibre.gui2.qt_file_dialogs import choose_save_file
from calibre_plugins.rss_reader.common_icons import get_icon

from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader import rss_db
from calibre_plugins.rss_reader.rss import normalize_summary_to_html
from calibre_plugins.rss_reader.tagging_utils import auto_tags_for_item as _auto_tags_for_item_util
from calibre_plugins.rss_reader.history_dropdown import HistoryDropdown
from calibre_plugins.rss_reader.dialogs import _AddEditProfileDialog, ShareViaEmailDialog
load_translations()

# Robust Qt UserRole constant for item data
try:
    ROLE_USER = getattr(getattr(Qt, 'ItemDataRole', None), 'UserRole', getattr(Qt, 'UserRole', 32))
except Exception:
    ROLE_USER = 32

# Preview browser and helpers
from calibre_plugins.rss_reader.preview_browser import (
    PreviewBrowser,
    AudioPlayer,
    _sanitize_url_for_fetch,
    _normalize_images_for_preview,
    _process_images_for_export,
)


class _RSSReaderDialogPartA:
    def _init_tray_icon(self):
        """Set up QSystemTrayIcon for minimize-to-tray (Windows only)."""
        import platform
        if not platform.system().lower().startswith('win'):
            self._tray_icon = None
            return
        try:
            from PyQt5.Qt import QSystemTrayIcon, QMenu, QAction, QIcon
        except ImportError:
            try:
                from qt.core import QSystemTrayIcon, QMenu, QAction, QIcon
            except Exception:
                self._tray_icon = None
                return
        # Use the plugin's own icon, NOT calibre's window icon
        try:
            icon = get_icon('images/iconplugin')
            if icon is None or icon.isNull():
                icon = get_icon('images/iconplugin_light.png')
        except Exception:
            icon = None
        if icon is None or (hasattr(icon, 'isNull') and icon.isNull()):
            try:
                icon = self.windowIcon()
            except Exception:
                icon = QIcon()
        self._tray_icon = QSystemTrayIcon(icon, self)
        menu = QMenu()
        restore_action = QAction(_('Restore'), self)
        restore_action.triggered.connect(self._restore_from_tray)
        menu.addAction(restore_action)
        menu.addSeparator()
        replay_action = QAction(_('Show last / test notification'), self)
        replay_action.setToolTip(_('Replay the last popup notification, or show a test sample if none available.'))
        replay_action.triggered.connect(self._replay_last_notification_from_tray)
        menu.addAction(replay_action)
        menu.addSeparator()
        about_action = QAction(_('About'), self)
        about_action.triggered.connect(self._show_tray_about_dialog)
        menu.addAction(about_action)
        menu.addSeparator()
        exit_action = QAction(_('Exit'), self)
        exit_action.triggered.connect(self._exit_from_tray)
        menu.addAction(exit_action)
        self._tray_icon.setContextMenu(menu)
        self._tray_icon.activated.connect(self._on_tray_activated)
        self._tray_icon.setToolTip(_('RSS Reader'))
        self._tray_icon.hide()

    def _show_tray_about_dialog(self):
        """Show an About dialog from the tray icon context menu."""
        try:
            from calibre.customize.zipplugin import get_resources
            from calibre.gui2 import open_url
        except Exception:
            return
        try:
            from calibre_plugins.rss_reader import RSSReaderPlugin
            version_str = '.'.join(str(x) for x in RSSReaderPlugin.version)
        except Exception:
            version_str = '?'

        dialog = QDialog(self)
        dialog.setWindowTitle(_('About RSS Reader'))
        layout = QVBoxLayout(dialog)

        title = QLabel('<h2>RSS Reader</h2>')
        title.setAlignment(Qt.AlignmentFlag.AlignCenter if hasattr(Qt, 'AlignmentFlag') else Qt.AlignCenter)
        layout.addWidget(title)

        desc = QLabel(_('A feature-rich RSS/Atom feed reader for calibre'))
        desc.setAlignment(Qt.AlignmentFlag.AlignCenter if hasattr(Qt, 'AlignmentFlag') else Qt.AlignCenter)
        layout.addWidget(desc)

        version_lbl = QLabel(_('Version: %s') % version_str)
        version_lbl.setAlignment(Qt.AlignmentFlag.AlignCenter if hasattr(Qt, 'AlignmentFlag') else Qt.AlignCenter)
        version_lbl.setStyleSheet(
            'QLabel { font-size: 9pt; color: palette(mid); background: transparent;'
            ' padding: 2px 6px; border: 1px solid rgba(0,0,0,0.06); border-radius: 4px; }'
        )
        layout.addWidget(version_lbl)

        # Key features
        features_lbl = QLabel(_(
            '<b>Key features:</b><br><br>'
            '• Save individual articles as standalone files<br>'
            '• Add single articles to your calibre library<br>'
            '• Email articles and feeds (supports Kindle device addresses)<br>'
            '• Custom tags and flexible auto-tagging<br>'                
            '• AI integration (optional) for article summaries and chat<br>'
            '• Transliterated UI for CJK and Cyrillic languages<br>'
            '• OPML selective import/export'
        ))
        features_lbl.setWordWrap(True)
        features_lbl.setStyleSheet('font-size: 9pt; margin-top: 6px;')
        layout.addWidget(features_lbl)

        # Forum link
        forum_container = QWidget()
        forum_layout_h = QHBoxLayout(forum_container)
        forum_link = QLabel(
            '\U0001F517<a href="https://www.mobileread.com/forums/showthread.php?t=371755" style="text-decoration:none;">'
            + _('MR help thread') + '</a>'
        )
        try:
            forum_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        except Exception:
            forum_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
        forum_link.setStyleSheet('QLabel, a { font-size: 11pt; text-decoration: none !important; color: palette(link); }')
        forum_link.linkActivated.connect(open_url)
        forum_layout_h.addWidget(forum_link)
        forum_layout_h.addStretch()
        layout.addWidget(forum_container)

        # Donate link with humorous per-language emoji + message
        # Use plugin UI language if set, otherwise fall back to Calibre's UI language
        _tray_ui_lang = str(plugin_prefs.get('plugin_ui_language', '') or '').strip()
        if not _tray_ui_lang:
            try:
                from calibre.utils.localization import get_lang
                _tray_ui_lang = str(get_lang() or '').strip()
            except Exception:
                _tray_ui_lang = ''
        _tray_emoji = '\u2764\uFE0F'
        _tray_sushi_langs = {'ja', 'zh', 'zh_CN', 'zh_TW', 'zh_HK', 'ko', 'vi', 'id', 'ms'}
        if _tray_ui_lang == 'pt':
            _tray_emoji = '\U0001F9C6'  # 🧆
        elif _tray_ui_lang == 'ar':
            _tray_emoji = '\U0001F9C6'  # 🧆
        elif _tray_ui_lang in ('tr', 'fa'):
            _tray_emoji = '\U0001F362'  # 🍢
        elif _tray_ui_lang == 'ca':
            _tray_emoji = '\U0001F9C0'  # 🧀
        elif _tray_ui_lang == 'es':
            _tray_emoji = '\U0001F364'  # 🍤
        elif _tray_ui_lang == 'gl':
            _tray_emoji = '\U0001F95F'  # 🥟
        elif _tray_ui_lang == 'fr':
            _tray_emoji = '\U0001F377'  # 🍷
        elif _tray_ui_lang in _tray_sushi_langs:
            _tray_emoji = '\U0001F363'  # 🍣
        elif _tray_ui_lang == 'it':
            _tray_emoji = '\U0001F355'  # 🍕
        elif _tray_ui_lang in ('pt_BR', 'de'):
            _tray_emoji = '\U0001F37A'  # 🍺
        elif _tray_ui_lang in ('ru', 'uk', 'pl', 'cs', 'sk', 'sr', 'hr', 'sl', 'bg'):
            _tray_emoji = '\U0001F943'  # 🥃
        donate_container = QWidget()
        donate_layout_h = QHBoxLayout(donate_container)
        donate_link = QLabel(
            f'{_tray_emoji}<a href="https://ko-fi.com/comfy_n" style="text-decoration:none;">' + _('Show appreciation') + '</a><br>'
        )
        try:
            donate_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        except Exception:
            donate_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
        donate_link.setStyleSheet('QLabel, a { font-size: 12pt; text-decoration: none !important; color: palette(link); }')
        donate_link.linkActivated.connect(open_url)
        donate_layout_h.addWidget(donate_link)
        donate_layout_h.addStretch()
        layout.addWidget(donate_container)

        # QR code (deferred load)
        try:
            from qt.core import QPixmap
        except ImportError:
            from PyQt5.Qt import QPixmap
        qr_label = QLabel()
        qr_label.setVisible(False)
        qr_container = QWidget()
        qr_container.setVisible(False)
        qr_layout_h = QHBoxLayout(qr_container)
        qr_layout_h.addStretch()
        qr_layout_h.addWidget(qr_label)
        qr_layout_h.addStretch()
        layout.addWidget(qr_container)

        plugin_path = getattr(getattr(self, 'action', None), 'plugin_path', None)

        def _load_qr():
            try:
                if plugin_path is None:
                    return
                data = get_resources(plugin_path, 'images/qrcode.png')
                if not data:
                    return
                qp = QPixmap()
                qp.loadFromData(data)
                if qp.isNull():
                    return
                smooth = getattr(Qt, 'SmoothTransformation', None) or getattr(Qt, 'TransformationMode', Qt).SmoothTransformation
                qr_label.setPixmap(qp.scaledToWidth(150, smooth))
                try:
                    qr_label.setCursor(Qt.CursorShape.PointingHandCursor)
                except Exception:
                    qr_label.setCursor(Qt.PointingHandCursor)
                qr_label.mousePressEvent = lambda ev: _flip_qr(ev)
                qr_label.setVisible(True)
                qr_container.setVisible(True)
            except Exception:
                pass

        def _flip_qr(event):
            try:
                if plugin_path is None:
                    return
                cur = qr_label.pixmap()
                if cur is None:
                    return
                smooth = getattr(Qt, 'SmoothTransformation', None) or getattr(Qt, 'TransformationMode', Qt).SmoothTransformation
                # Load both images
                qr_data = get_resources(plugin_path, 'images/qrcode.png')
                ee_data = get_resources(plugin_path, 'images/pic.png')
                if not qr_data or not ee_data:
                    return
                qr_px = QPixmap()
                qr_px.loadFromData(qr_data)
                ee_px = QPixmap()
                ee_px.loadFromData(ee_data)
                if qr_px.isNull() or ee_px.isNull():
                    return
                # Compare current to QR to decide which to show
                qr_scaled = qr_px.scaledToWidth(150, smooth)
                if cur.toImage() == qr_scaled.toImage():
                    qr_label.setPixmap(ee_px.scaledToWidth(150, smooth))
                    qr_label.setToolTip(_('You found the hidden easter egg! \U0001F95A'))
                else:
                    qr_label.setPixmap(qr_scaled)
                    qr_label.setToolTip('')
                QTimer.singleShot(0, lambda: open_url('https://ko-fi.com/comfy_n'))
            except Exception:
                pass

        try:
            QTimer.singleShot(0, _load_qr)
        except Exception:
            _load_qr()

        # QuiteRSS inspiration credit
        try:
            credit_lbl = QLabel(
                '<span style="font-size: 8pt; color: palette(mid);">'
                + _('UI design inspired by') +
                ' <a href="https://quiterss.org" style="text-decoration:none; color: palette(mid);">QuiteRSS</a></span>'
            )
            try:
                credit_lbl.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
            except Exception:
                credit_lbl.setTextInteractionFlags(Qt.TextBrowserInteraction)
            credit_lbl.linkActivated.connect(open_url)
            credit_lbl.setAlignment(Qt.AlignmentFlag.AlignCenter if hasattr(Qt, 'AlignmentFlag') else Qt.AlignCenter)
            layout.addWidget(credit_lbl)
        except Exception:
            pass

        dialog.exec()

    def _update_tray_icon(self):
        """Refresh the tray icon after a theme/palette change."""
        try:
            tray = getattr(self, '_tray_icon', None)
            if tray is None:
                return
            icon = get_icon('images/iconplugin')
            if icon is None or icon.isNull():
                icon = get_icon('images/iconplugin_light.png')
            if icon is not None and not icon.isNull():
                tray.setIcon(icon)
        except Exception:
            pass

    def _restore_from_tray(self):
        self.showNormal()
        self.raise_()
        if getattr(self, '_tray_icon', None):
            self._tray_icon.hide()

    def _exit_from_tray(self):
        self._tray_icon.hide()
        self.close()

    def _on_tray_activated(self, reason):
        # Handle Qt enum/ActivationReason safely across bindings.
        try:
            if isinstance(reason, int):
                r = reason
            else:
                # Some Qt wrappers expose `.value`.
                r = getattr(reason, 'value', None)
                if r is None:
                    # Fall back to string name matching.
                    r = str(reason)
        except Exception:
            r = None

        # Prefer enum comparison when available.
        try:
            try:
                from qt.core import QSystemTrayIcon
            except Exception:
                from PyQt5.Qt import QSystemTrayIcon

            try:
                trigger = QSystemTrayIcon.ActivationReason.Trigger
                dbl = QSystemTrayIcon.ActivationReason.DoubleClick
            except Exception:
                trigger = getattr(QSystemTrayIcon, 'Trigger', None)
                dbl = getattr(QSystemTrayIcon, 'DoubleClick', None)

            if dbl is not None and reason == dbl:
                self._restore_from_tray()
                return
            if trigger is not None and reason == trigger:
                # Single-click: if window is visible, replay last notification
                # (QuiteRSS-inspired); if hidden, restore window.
                if self.isVisible() and not self.isMinimized():
                    self._replay_last_notification_from_tray()
                else:
                    self._restore_from_tray()
                return
        except Exception:
            pass

        # Numeric fallback (common values: Trigger=3, DoubleClick=2)
        try:
            if r == 2:
                self._restore_from_tray()
                return
            if r == 3:
                if self.isVisible() and not self.isMinimized():
                    self._replay_last_notification_from_tray()
                else:
                    self._restore_from_tray()
                return
        except Exception:
            pass

        # String-name fallback
        try:
            if isinstance(r, str) and 'DoubleClick' in r:
                self._restore_from_tray()
            elif isinstance(r, str) and 'Trigger' in r:
                if self.isVisible() and not self.isMinimized():
                    self._replay_last_notification_from_tray()
                else:
                    self._restore_from_tray()
        except Exception:
            pass

    def _replay_last_notification_from_tray(self):
        """Replay the most recent popup notification (QuiteRSS-style tray single-click)."""
        try:
            action = getattr(self, 'action', None)
            if action is not None and hasattr(action, 'preview_in_app_popup'):
                action.preview_in_app_popup()
                return
        except Exception:
            pass
        # If no action object or no notification, just restore
        try:
            self._restore_from_tray()
        except Exception:
            pass

    def _update_feeds_label(self):
        try:
            model = self.feeds_tree.model()
            count = 0
            if model is not None:
                def count_feeds(item):
                    c = 0
                    for i in range(item.rowCount()):
                        child = item.child(i)
                        if child is not None:
                            if child.hasChildren():
                                c += count_feeds(child)
                            else:
                                c += 1
                    return c
                root = model.invisibleRootItem()
                count = count_feeds(root)
            self.feeds_label.setText(_('Feeds ({})').format(count))
            self.feeds_label.setToolTip(_('Total feeds in this database: {}').format(count))
        except Exception:
            self.feeds_label.setText(_('Feeds'))

    def _update_items_label(self):
        try:
            table = self.items_table
            # Show a count only when the main search box contains a query.
            q = ''
            try:
                if getattr(self, 'filter_input', None) is not None:
                    fi = self.filter_input
                    if hasattr(fi, 'currentText'):
                        q = str(fi.currentText() or '').strip()
                    elif hasattr(fi, 'text'):
                        q = str(fi.text() or '').strip()
            except Exception:
                q = ''

            if q:
                # Count visible rows (not hidden by current in-widget filter)
                try:
                    visible_count = 0
                    for r in range(table.rowCount() if table is not None else 0):
                        try:
                            if not table.isRowHidden(r):
                                visible_count += 1
                        except Exception:
                            visible_count = table.rowCount() if table is not None else 0
                            break
                except Exception:
                    visible_count = table.rowCount() if table is not None else 0
                self.items_label.setText(_('Items ({})').format(visible_count))
                try:
                    self.items_label.setToolTip(_('Items matching current filter: {}').format(visible_count))
                except Exception:
                    pass
            else:
                self.items_label.setText(_('Items'))
        except Exception:
            self.items_label.setText(_('Items'))


    def _update_current_feed_label(self):
        try:
            # Get visible (filtered) items count from items table
            visible_items = 0
            total_items = 0
            try:
                table = getattr(self, 'items_table', None)
                if table is not None:
                    total_items = table.rowCount()
                    for r in range(total_items):
                        try:
                            if not table.isRowHidden(r):
                                visible_items += 1
                        except Exception:
                            visible_items = total_items
                            break
            except Exception:
                visible_items = total_items

            sel = self.feeds_tree.selectionModel()
            model = self.feeds_tree.model()
            if sel is not None and model is not None:
                indexes = sel.selectedIndexes()
                if indexes:
                    idx = indexes[0]
                    item = model.itemFromIndex(idx)
                    if item is not None:
                        title = str(item.text() or '')
                        # Determine if this is a folder (has children) or a feed (leaf)
                        if item.hasChildren():
                            # Folder: count feeds under this node
                            def count_feeds(node):
                                feed_count = 0
                                for i in range(node.rowCount()):
                                    child = node.child(i)
                                    if child is not None:
                                        if child.hasChildren():
                                            feed_count += count_feeds(child)
                                        else:
                                            feed_count += 1
                                return feed_count
                            feed_count = count_feeds(item)
                            if len(title) > 48:
                                shown = title[:45] + '...'
                            else:
                                shown = title
                            # Show visible items from the items table (respects current filter)
                            self.current_feed_label.setText(_('Current folder: {} - {} feeds, {} items').format(shown, feed_count, visible_items))
                            self.current_feed_label.setToolTip(title)
                            return
                        else:
                            # Feed: show feed title and visible items count
                            if len(title) > 48:
                                shown = title[:45] + '...'
                            else:
                                shown = title
                            self.current_feed_label.setText(_('Current feed: {} ({} items)').format(shown, visible_items))
                            self.current_feed_label.setToolTip(title)
                            return
            self.current_feed_label.setText(_('Current feed: (none)'))
            self.current_feed_label.setToolTip(_('No feed selected'))
        except Exception:
            self.current_feed_label.setText(_('Current feed'))

    def __init__(self, gui, action=None, parent=None):
        # Qt requires the base class initializer to run before any QWidget/QObject
        # methods are used, otherwise you'll get:
        # RuntimeError: super-class __init__() of type RSSReaderDialog was never called
        try:
            QMainWindow.__init__(self, parent)
        except Exception:
            try:
                super().__init__(parent)
            except Exception:
                pass

        _t0 = time.perf_counter()
        _debug('RSSReaderDialog.__init__ start')
        self.gui = gui
        self.action = action
        self.notifier = get_notifier()

        # Minimize to tray support (Windows only)
        self._init_tray_icon()


        # Ensure SQLite storage exists.
        try:
            rss_db.ensure_ready()
        except Exception:
            pass
        try:
            # Remember the built-in DB path separately (useful for diagnostics)
            try:
                self._db_config_path = str(getattr(rss_db, 'suggested_default_db_path', lambda: '')() or '').strip()
            except Exception:
                self._db_config_path = None

            # Determine configured "Default DB" (can be outside config dir)
            try:
                dpath = str(plugin_prefs.get('db_default_path') or '').strip()
            except Exception:
                dpath = ''
            try:
                d_ro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
            except Exception:
                d_ro = False
            # If not configured, default DB is the calibre-config DB
            self._db_orig_path = dpath or self._db_config_path
            self._db_default_readonly = bool(d_ro)

            self._db_prev_path = None
            self._db_prev_readonly = False
            try:
                self._db_current_path = rss_db.db_path()
            except Exception:
                self._db_current_path = None
        except Exception:
            self._db_config_path = None
            self._db_orig_path = None
            self._db_default_readonly = False
            self._db_prev_path = None
            self._db_prev_readonly = False
            self._db_current_path = None

        # Restore active profile on startup if set in preferences
        try:
            active_id = str(plugin_prefs.get('db_profiles_active') or '')
        except Exception:
            active_id = ''
        if active_id:
            try:
                for p in (plugin_prefs.get('db_profiles') or []):
                    try:
                        if str(p.get('id') or '') != active_id:
                            continue
                        # remember previous (default) DB so "Switch back" can restore it
                        try:
                            self._db_prev_path = self._db_orig_path
                        except Exception:
                            self._db_prev_path = None
                        try:
                            self._db_prev_readonly = False
                        except Exception:
                            pass

                        p_path = str(p.get('path') or '')
                        readonly = bool(p.get('readonly', False) or p.get('mirror', False))
                        try:
                            rss_db.set_db_path(p_path, readonly=readonly)
                        except Exception:
                            # Disallowed or invalid DB path. Deactivate the profile.
                            try:
                                plugin_prefs['db_profiles_active'] = ''
                            except Exception:
                                pass
                            continue
                        self._db_current_path = rss_db.db_path()
                        try:
                            if getattr(self, 'action', None) is not None:
                                try:
                                    self.action._refresh_toolbar_label()
                                except Exception:
                                    pass
                        except Exception:
                            pass
                        if not readonly:
                            rss_db.ensure_ready()
                        break
                    except Exception:
                        continue
            except Exception:
                pass
        else:
            # No active profile: apply configured default DB location (if any)
            try:
                default_path = str(getattr(self, '_db_orig_path', '') or '').strip()
            except Exception:
                default_path = ''
            try:
                default_ro = bool(getattr(self, '_db_default_readonly', False))
            except Exception:
                default_ro = False
            try:
                if default_path and default_path != str(rss_db.db_path() or '').strip():
                    try:
                        rss_db.set_db_path(default_path, readonly=default_ro)
                    except Exception:
                        default_path = ''
                    self._db_current_path = rss_db.db_path()
                    try:
                        if getattr(self, 'action', None) is not None:
                            try:
                                self.action._refresh_toolbar_label()
                            except Exception:
                                pass
                    except Exception:
                        pass
                    if not default_ro:
                        rss_db.ensure_ready()
            except Exception:
                pass

        _debug('  rss_db.ensure_ready done in %.3fs' % (time.perf_counter() - _t0))

        self._items_by_feed_id = {}
        self._feeds_results = {}
        self._all_feeds = {}

        # Used by legacy code paths that previously auto-marked feeds as read.
        # Feed selection no longer changes read/unread state.
        self._feed_selection_user_initiated = False
        self._auto_mark_seen_guard = False

        self._progress = None
        self._export_jobs = {}
        self._ai_export_jobs = {}
        self._recipe_jobs = {}
        # Keep strong refs to background export threads/workers.
        # Otherwise Python GC can destroy a running QThread.
        self._active_export_threads = []  # list[(QThread, ExportWorker)]
        self._folder_to_feed_ids = {}
        # Update progress diagnostics
        self._update_progress_history = []  # (ts, message)
        self._last_update_progress_ts = 0
        self._progress_stall_timer = None
        self._stall_threshold_seconds = int(plugin_prefs.get('progress_stall_threshold', 4) or 4)

        self._current_item = None
        self._ai_call_id = 0
        self._ai_thread = None
        self._ai_acc = None
        self._ai_sys_prompt = ''
        self._ai_user_prompt = ''
        self._ai_last_answer = ''
        self._ai_export_jobs = {}
        self._ai_dispatch = Dispatcher(self._on_ai_response)

        self.setWindowTitle(_('RSS Reader'))
        try:
            self.setWindowIcon(get_icon('images/iconplugin'))
        except Exception:
            try:
                self.setWindowIcon(get_icon('images/icon'))
            except Exception:
                pass

        # Configure dock widget options for movable/resizable panels
        try:
            self.setDockOptions(
                QMainWindow.DockOption.AnimatedDocks |
                QMainWindow.DockOption.AllowTabbedDocks |
                QMainWindow.DockOption.AllowNestedDocks
            )
        except Exception:
            try:
                self.setDockOptions(
                    QMainWindow.AnimatedDocks |
                    QMainWindow.AllowTabbedDocks |
                    QMainWindow.AllowNestedDocks
                )
            except Exception:
                pass

        # Make this window modeless (do not block the main calibre UI)
        try:
            self.setModal(False)
        except Exception:
            pass
        try:
            self.setWindowModality(Qt.WindowModality.NonModal)
        except Exception:
            try:
                self.setWindowModality(Qt.NonModal)
            except Exception:
                pass

        # Ensure standard window controls (minimize/maximize) are available
        try:
            flags = self.windowFlags()
            try:
                flags |= Qt.WindowType.WindowMinimizeButtonHint
                flags |= Qt.WindowType.WindowMaximizeButtonHint
                flags |= Qt.WindowType.WindowSystemMenuHint
            except Exception:
                flags |= Qt.WindowMinimizeButtonHint
                flags |= Qt.WindowMaximizeButtonHint
                flags |= Qt.WindowSystemMenuHint
            self.setWindowFlags(flags)
        except Exception:
            pass
        _debug('  QMainWindow.__init__ + window setup done in %.3fs' % (time.perf_counter() - _t0))

        # Setup the UI
        try:
            self.setup_ui()
        except Exception:
            _debug('  setup_ui failed: %s' % traceback.format_exc())
            raise

        # Update toolbar button states initially
        try:
            self.update_toolbar_button_states()
        except Exception:
            pass

        # Status label context menu (Stop fetching / Suspend fetching)
        try:
            lbl = getattr(self, 'status', None)
            if lbl is not None:
                try:
                    lbl.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
                except Exception:
                    try:
                        lbl.setContextMenuPolicy(Qt.CustomContextMenu)
                    except Exception:
                        lbl = None
                if lbl is not None:
                    try:
                        lbl.customContextMenuRequested.connect(self._on_status_context_menu)
                    except Exception:
                        pass
        except Exception:
            pass

        # Old QAction-wrapper helper removed; toolbar actions are created
        # explicitly after the toolbar mapping below to avoid bloat.

        # Register dock widgets with the main window
        try:
            self.addDockWidget(Qt.DockWidgetArea.RightDockWidgetArea, self.ai_dock)
        except Exception:
            _debug('  addDockWidget failed: %s' % traceback.format_exc())

        # Set up window constraints (minimum size)
        try:
            self.setup_window_constraints()
        except Exception:
            _debug('  setup_window_constraints failed: %s' % traceback.format_exc())

        # Restore saved window state (geometry and dock layout)
        try:
            self.restore_state()
        except Exception:
            _debug('  restore_state failed: %s' % traceback.format_exc())

        # Populate from cache immediately, then restore UI and optionally auto-update.
        _t1 = time.perf_counter()

        # --- Patch: Save current feed/folder selection before refresh ---
        try:
            prev_feed_ids = self.selected_feed_ids()
            prev_folder = self.selected_folder_path()
        except Exception:
            prev_feed_ids = []
            prev_folder = ''

        try:
            self.load_from_cache()
        except Exception:
            pass
        _debug('  load_from_cache done in %.3fs' % (time.perf_counter() - _t1))
        _t1 = time.perf_counter()
        try:
            self.refresh()
        except Exception:
            pass
        _debug('  refresh done in %.3fs' % (time.perf_counter() - _t1))

        # --- Patch: Restore previous feed selection after refresh ---
        try:
            # Try to restore previous feed selection
            if prev_feed_ids:
                self.feeds_tree.clearSelection()
                for fid in prev_feed_ids:
                    item = self._find_tree_item_by_feed(fid)
                    if item is not None:
                        item.setSelected(True)
                        self.feeds_tree.scrollToItem(item)
            elif prev_folder:
                item = self._find_tree_item_by_folder(prev_folder)
                if item is not None:
                    item.setSelected(True)
                    self.feeds_tree.scrollToItem(item)
        except Exception:
            pass

        # Restore UI state (splitter sizes, column widths, last sort)
        _t1 = time.perf_counter()
        try:
            self.restore_state()
        except Exception:
            pass
        _debug('  restore_state done in %.3fs' % (time.perf_counter() - _t1))

        # Save state when closing (QMainWindow has no finished() signal)
        try:
            self.destroyed.connect(lambda *args: self.save_state())
        except Exception:
            pass



        _debug('RSSReaderDialog.__init__ total: %.3fs' % (time.perf_counter() - _t0))

        if self.action is not None and bool(plugin_prefs.get('auto_update_on_open', True)):
            try:
                # run with notifications/badge
                self.action.update_all_feeds(silent=False)
            except Exception:
                pass

    def _icon_with_badge(self, icon, text):
        try:
            from qt.core import QIcon, QPixmap, QPainter, QPainterPath, QRectF, QFont, QColor
        except Exception:
            from PyQt5.Qt import QIcon, QPixmap, QPainter, QPainterPath, QRectF, QFont, QColor

        try:
            base = icon.pixmap(32, 32)
            if base.isNull():
                return icon
            pm = QPixmap(base.size())
            pm.fill(Qt.GlobalColor.transparent)
            p = QPainter(pm)
            p.setRenderHint(QPainter.RenderHint.Antialiasing, True)
            p.drawPixmap(0, 0, base)

            rect = QRectF(pm.width() - 18, pm.height() - 14, 16, 12)
            path = QPainterPath()
            path.addRoundedRect(rect, 3, 3)
            p.fillPath(path, QColor(0, 0, 0, 170))
            p.setPen(QColor(255, 255, 255))
            f = QFont(p.font())
            f.setBold(True)
            f.setPointSize(max(7, f.pointSize() - 2))
            p.setFont(f)
            p.drawText(rect, Qt.AlignmentFlag.AlignCenter, text)
            p.end()
            return QIcon(pm)
        except Exception:
            return icon

    def _unread_dot_icon(self):
        try:
            cached = getattr(self, '_unread_dot_icon_cache', None)
        except Exception:
            cached = None
        if cached is not None:
            return cached

        try:
            from qt.core import QIcon, QPixmap, QPainter, QColor
        except Exception:
            from PyQt5.Qt import QIcon, QPixmap, QPainter, QColor

        try:
            size = 10
            pm = QPixmap(size, size)
            pm.fill(Qt.GlobalColor.transparent)
            p = QPainter(pm)
            try:
                p.setRenderHint(QPainter.RenderHint.Antialiasing, True)
            except Exception:
                pass
            # QuiteRSS-like green dot
            try:
                p.setPen(QColor(204, 102, 0))
            except Exception:
                pass
            try:
                p.setBrush(QColor(255, 165, 0))
            except Exception:
                pass
            try:
                p.drawEllipse(1, 1, size - 2, size - 2)
            except Exception:
                pass
            try:
                p.end()
            except Exception:
                pass
            ic = QIcon(pm)
        except Exception:
            ic = None

        try:
            self._unread_dot_icon_cache = ic
        except Exception:
            pass
        return ic

    def _transparent_placeholder_icon(self):
        """Return a transparent placeholder icon (10x10) to maintain spacing in table items."""
        try:
            cached = getattr(self, '_transparent_placeholder_icon_cache', None)
        except Exception:
            cached = None
        if cached is not None:
            return cached

        try:
            from qt.core import QIcon, QPixmap
        except Exception:
            from PyQt5.Qt import QIcon, QPixmap

        try:
            size = 10
            pm = QPixmap(size, size)
            pm.fill(Qt.GlobalColor.transparent)
            ic = QIcon(pm)
        except Exception:
            try:
                ic = QIcon()
            except Exception:
                ic = None

        try:
            self._transparent_placeholder_icon_cache = ic
        except Exception:
            pass
        return ic

    def set_busy(self, busy, message=''):
        # Disable actions while background update is running to prevent re-entrancy
        try:
            for w in (
                getattr(self, 'add_btn', None),
                getattr(self, 'import_btn', None),
                getattr(self, 'export_btn', None),
                getattr(self, 'export_ebook_btn', None),
                getattr(self, 'remove_btn', None),
                getattr(self, 'edit_btn', None),
                getattr(self, 'mark_read_btn', None),
                getattr(self, 'settings_btn', None),
                getattr(self, 'update_btn', None),
                getattr(self, 'update_all_btn', None),
            ):
                try:
                    if w is None:
                        continue
                    w.setEnabled(not busy)
                except Exception:
                    pass
        except Exception:
            pass
        if busy:
            try:
                if self._progress is None:
                    self._progress = QProgressDialog(message or _('Updating feeds...'), _('Cancel'), 0, 0, self)
                    self._progress.setWindowTitle(_('RSS Reader'))
                    try:
                        self._progress.setWindowModality(Qt.WindowModality.NonModal)
                    except Exception:
                        self._progress.setWindowModality(Qt.NonModal)
                    self._progress.setAutoClose(False)
                    self._progress.setAutoReset(False)
                    try:
                        self._progress.canceled.connect(self._cancel_update)
                    except Exception:
                        pass
                else:
                    try:
                        self._progress.setLabelText(message or _('Updating feeds...'))
                    except Exception:
                        pass
                self._progress.show()
                # Track the base label so we can augment it with live diagnostics
                try:
                    self._progress_label_base = str(message or _('Updating feeds...'))
                except Exception:
                    self._progress_label_base = _('Updating feeds...')
                # Reset progress diagnostics
                try:
                    self._update_progress_history = []
                    self._last_update_progress_ts = 0
                except Exception:
                    pass
                # Start a stall-check timer to show "stalled" hints
                try:
                    if getattr(self, '_progress_stall_timer', None) is None:
                        t = QTimer(self)
                        t.setInterval(1000)
                        t.timeout.connect(self._progress_stall_tick)
                        self._progress_stall_timer = t
                    try:
                        self._progress_stall_timer.start()
                    except Exception:
                        pass
                except Exception:
                    pass
            except Exception:
                self._progress = None
        else:
            try:
                if self._progress is not None:
                    self._progress.hide()
                    self._progress.deleteLater()
            except Exception:
                pass
            try:
                if getattr(self, '_progress_stall_timer', None) is not None:
                    try:
                        self._progress_stall_timer.stop()
                    except Exception:
                        pass
                    try:
                        self._progress_stall_timer.deleteLater()
                    except Exception:
                        pass
                    self._progress_stall_timer = None
            except Exception:
                pass
            self._progress = None

    def _cancel_update(self):
        try:
            if self.action is not None and hasattr(self.action, 'cancel_update'):
                self.action.cancel_update()
        except Exception:
            pass
        # keep UI disabled until the callback arrives
        return

    def on_update_progress(self, message, fraction=None):
        try:
            if getattr(self, 'status', None) is not None and message:
                self.status.setText(str(message))
        except Exception:
            pass

        # Update progress dialog label/value if present
        try:
            if self._progress is not None and message:
                # Record timestamped history for diagnostics
                try:
                    import time as _time
                    ts = float(_time.time())
                    self._last_update_progress_ts = ts
                    try:
                        self._update_progress_history.append((ts, str(message)))
                        # Keep history bounded
                        if len(self._update_progress_history) > 200:
                            self._update_progress_history = self._update_progress_history[-200:]
                    except Exception:
                        pass
                except Exception:
                    pass

                # Show the immediate message; the stall timer will add a "Last:" line
                try:
                    self._progress.setLabelText(str(message))
                except Exception:
                    pass
                if fraction is not None:
                    try:
                        v = int(max(0, min(100, round(float(fraction) * 100))))
                        self._progress.setRange(0, 100)
                        self._progress.setValue(v)
                    except Exception:
                        pass
        except Exception:
            pass

    def _progress_stall_tick(self):
        try:
            if getattr(self, '_progress', None) is None:
                return
            try:
                import time as _time
                now = float(_time.time())
            except Exception:
                now = None
            last_ts = float(getattr(self, '_last_update_progress_ts', 0) or 0)
            age = None
            try:
                if now is not None and last_ts:
                    age = int(max(0, round(now - last_ts)))
            except Exception:
                age = None

            try:
                base = str(getattr(self, '_progress_label_base', '') or '')
            except Exception:
                base = ''

            try:
                last_msg = ''
                if getattr(self, '_update_progress_history', None):
                    last_msg = str(self._update_progress_history[-1][1])
                label = base
                if last_msg:
                    if age is None:
                        label = '%s\nLast: %s' % (base, last_msg)
                    else:
                        label = '%s\nLast: %s (%ds ago)' % (base, last_msg, age)
                        if age >= int(self._stall_threshold_seconds or 10):
                            label = label + '  (stalled?)'
                try:
                    self._progress.setLabelText(label)
                except Exception:
                    pass
            except Exception:
                pass
        except Exception:
            pass

    # --- Items table column management (visibility, order, widths) ---
    def setup_items_table_columns(self):
        try:
            self._items_table_columns = [
                {'id': 0, 'label': _('★'), 'default_visible': True, 'default_width': 30},
                {'id': 1, 'label': _('Title'), 'default_visible': True, 'default_width': 400},
                {'id': 2, 'label': _('Author'), 'default_visible': True, 'default_width': 180},
                {'id': 3, 'label': _('Published'), 'default_visible': True, 'default_width': 140},
                {'id': 4, 'label': _('Tags'), 'default_visible': True, 'default_width': 200},
            ]
            header = self.items_table.horizontalHeader()
            try:
                header.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
                header.customContextMenuRequested.connect(self.show_items_header_context_menu)
            except Exception:
                pass
            try:
                header.setSectionsMovable(True)
            except Exception:
                pass

            hidden = gprefs.get('rss_reader_items_table_hidden_columns', None)
            if isinstance(hidden, (list, tuple)):
                for i in range(self.items_table.columnCount()):
                    try:
                        self.items_table.setColumnHidden(i, i in hidden)
                    except Exception:
                        pass

            widths = gprefs.get('rss_reader_items_table_widths', {}) or {}
            for col, w in widths.items():
                try:
                    self.items_table.setColumnWidth(int(col), int(w))
                except Exception:
                    pass

            try:
                # Use Interactive for Title so user can drag to resize; keep small icon column fixed
                header.setSectionResizeMode(0, header.ResizeMode.Fixed)
                header.setSectionResizeMode(1, header.ResizeMode.Interactive)
                # Use Interactive for Author/Published so resize handles are visible
                header.setSectionResizeMode(2, header.ResizeMode.Interactive)
                header.setSectionResizeMode(3, header.ResizeMode.Interactive)
                header.setSectionResizeMode(4, header.ResizeMode.Interactive)
                try:
                    header.setSectionsClickable(True)
                except Exception:
                    pass
                try:
                    header.setStretchLastSection(False)
                except Exception:
                    pass
            except Exception:
                try:
                    header.setSectionResizeMode(0, header.Fixed)
                    header.setSectionResizeMode(1, header.Interactive)
                    # Use Interactive fallback to ensure resize handles show up
                    header.setSectionResizeMode(2, header.Interactive)
                    header.setSectionResizeMode(3, header.Interactive)
                    header.setSectionResizeMode(4, header.Interactive)
                    try:
                        header.setSectionsClickable(True)
                    except Exception:
                        pass
                except Exception:
                    pass

            try:
                header.sectionMoved.connect(self.on_items_columns_moved)
            except Exception:
                pass
            try:
                header.sectionResized.connect(self.on_items_column_resized)
            except Exception:
                pass
        except Exception:
            pass

    def show_items_header_context_menu(self, pos):
        try:
            header = self.items_table.horizontalHeader()
            menu = QMenu(self)
            for i in range(self.items_table.columnCount()):
                try:
                    text = self.items_table.horizontalHeaderItem(i).text() if self.items_table.horizontalHeaderItem(i) else str(i)
                    act = QAction(text, menu)
                    act.setCheckable(True)
                    visible = not self.items_table.isColumnHidden(i)
                    act.setChecked(visible)
                    act.triggered.connect(lambda checked, idx=i: self.toggle_items_column_visibility(idx, checked))
                    menu.addAction(act)
                except Exception:
                    pass
            # Auto-size actions for the clicked column / all columns
            try:
                # Determine which logical column was clicked
                try:
                    clicked_logical = header.logicalIndexAt(pos)
                except Exception:
                    try:
                        clicked_logical = header.logicalIndexAt(int(pos.x()))
                    except Exception:
                        clicked_logical = None
                if clicked_logical is None or clicked_logical < 0:
                    clicked_logical = 0
                # Provide full AdjustColumnSize dialog for accessibility
                try:
                    from calibre.gui2.library.views import AdjustColumnSize
                    try:
                        col_name = self.items_table.horizontalHeaderItem(int(clicked_logical)).text() if self.items_table.horizontalHeaderItem(int(clicked_logical)) else str(clicked_logical)
                    except Exception:
                        col_name = str(clicked_logical)
                    adjust_act = QAction(_('Adjust column width...'), menu)
                    adjust_act.triggered.connect(lambda _checked=False, idx=clicked_logical, name=col_name: AdjustColumnSize(self.items_table, int(idx), name).exec_())
                    menu.addAction(adjust_act)
                except Exception:
                    pass
                auto_act = QAction(_('Auto-size column'), menu)
                auto_act.triggered.connect(lambda _checked=False, idx=clicked_logical: self.items_table.resizeColumnToContents(int(idx)))
                menu.addAction(auto_act)
                auto_all = QAction(_('Auto-size all columns'), menu)
                auto_all.triggered.connect(lambda _checked=False: self.items_table.resizeColumnsToContents())
                menu.addAction(auto_all)
            except Exception:
                pass
            menu.addSeparator()
            reset_act = QAction(_('Restore default columns'), menu)
            reset_act.triggered.connect(self.restore_default_items_columns)
            menu.addAction(reset_act)
            menu.exec(header.mapToGlobal(pos))
        except Exception:
            pass

    def toggle_items_column_visibility(self, logical_index, visible):
        try:
            self.items_table.setColumnHidden(logical_index, not bool(visible))
            hidden = [i for i in range(self.items_table.columnCount()) if self.items_table.isColumnHidden(i)]
            gprefs['rss_reader_items_table_hidden_columns'] = hidden
        except Exception:
            pass

    def on_items_columns_moved(self, logicalIndex, oldVisualIndex, newVisualIndex):
        try:
            header = self.items_table.horizontalHeader()
            order = []
            for visual in range(header.count()):
                try:
                    order.append(header.logicalIndex(visual))
                except Exception:
                    order.append(visual)
            gprefs['rss_reader_items_table_column_order'] = order
        except Exception:
            pass

    def on_items_column_resized(self, logical_index, old_size, new_size):
        try:
            widths = gprefs.get('rss_reader_items_table_widths', {}) or {}
            widths[str(logical_index)] = int(new_size or 0)
            gprefs['rss_reader_items_table_widths'] = widths
        except Exception:
            pass

    def restore_default_items_columns(self):
        try:
            for col_def in getattr(self, '_items_table_columns', []) or []:
                idx = int(col_def.get('id') or 0)
                try:
                    self.items_table.setColumnHidden(idx, not bool(col_def.get('default_visible', True)))
                except Exception:
                    pass
                try:
                    w = int(col_def.get('default_width') or 0)
                    if w:
                        self.items_table.setColumnWidth(idx, w)
                except Exception:
                    pass
            try:
                if 'rss_reader_items_table_hidden_columns' in gprefs:
                    del gprefs['rss_reader_items_table_hidden_columns']
            except Exception:
                pass
            try:
                if 'rss_reader_items_table_widths' in gprefs:
                    del gprefs['rss_reader_items_table_widths']
            except Exception:
                pass
        except Exception:
            pass

    def show_column_manager_dialog(self):
        try:
            dlg = QDialog(self)
            dlg.setWindowTitle(_('Manage Columns'))
            layout = QVBoxLayout(dlg)
            listw = QListWidget(dlg)
            try:
                listw.setDragDropMode(QAbstractItemView.DragDropMode.InternalMove)
            except Exception:
                try:
                    listw.setDragDropMode(QAbstractItemView.InternalMove)
                except Exception:
                    pass
            header = self.items_table.horizontalHeader()
            count = self.items_table.columnCount()
            for visual in range(count):
                try:
                    logical = header.logicalIndex(visual)
                    text = self.items_table.horizontalHeaderItem(logical).text() if self.items_table.horizontalHeaderItem(logical) else str(logical)
                    item = QListWidgetItem(text)
                    item.setData(Qt.ItemDataRole.UserRole, int(logical))
                    item.setFlags(item.flags() | Qt.ItemFlag.ItemIsUserCheckable | Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsSelectable | Qt.ItemFlag.ItemIsDragEnabled)
                    checked = not self.items_table.isColumnHidden(int(logical))
                    item.setCheckState(Qt.CheckState.Checked if checked else Qt.CheckState.Unchecked)
                    listw.addItem(item)
                except Exception:
                    pass
            layout.addWidget(listw)
            btn_row = QHBoxLayout()
            btn_row.addStretch(1)
            ok_btn = QPushButton(_('OK'), dlg)
            cancel_btn = QPushButton(_('Cancel'), dlg)
            reset_btn = QPushButton(_('Restore Defaults'), dlg)
            btn_row.addWidget(reset_btn)
            btn_row.addWidget(cancel_btn)
            btn_row.addWidget(ok_btn)
            layout.addLayout(btn_row)

            def do_reset():
                try:
                    listw.clear()
                    cols = getattr(self, '_items_table_columns', []) or []
                    for col_def in cols:
                        try:
                            idx = int(col_def.get('id') or 0)
                            text = str(col_def.get('label') or '')
                            item = QListWidgetItem(text)
                            item.setData(Qt.ItemDataRole.UserRole, idx)
                            item.setFlags(item.flags() | Qt.ItemFlag.ItemIsUserCheckable | Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsSelectable | Qt.ItemFlag.ItemIsDragEnabled)
                            item.setCheckState(Qt.CheckState.Checked if col_def.get('default_visible', True) else Qt.CheckState.Unchecked)
                            listw.addItem(item)
                        except Exception:
                            pass
                except Exception:
                    pass

            def do_accept():
                try:
                    new_order = []
                    hidden = []
                    for i in range(listw.count()):
                        try:
                            it = listw.item(i)
                            logical = int(it.data(Qt.ItemDataRole.UserRole) or 0)
                            new_order.append(logical)
                            if it.checkState() != Qt.CheckState.Checked:
                                hidden.append(logical)
                        except Exception:
                            pass
                    gprefs['rss_reader_items_table_column_order'] = new_order
                    gprefs['rss_reader_items_table_hidden_columns'] = hidden
                    hdr = self.items_table.horizontalHeader()
                    for target_visual, logical in enumerate(new_order):
                        try:
                            cur_vis = hdr.visualIndex(int(logical))
                            if cur_vis != target_visual:
                                try:
                                    hdr.moveSection(cur_vis, target_visual)
                                except Exception:
                                    pass
                        except Exception:
                            pass
                    for c in range(self.items_table.columnCount()):
                        try:
                            self.items_table.setColumnHidden(c, c in hidden)
                        except Exception:
                            pass
                except Exception:
                    pass
                dlg.accept()

            ok_btn.clicked.connect(do_accept)
            cancel_btn.clicked.connect(dlg.reject)
            reset_btn.clicked.connect(do_reset)

            dlg.resize(360, 420)
            dlg.exec()
        except Exception:
            pass

    def _on_import_clicked(self):
        try:
            mode = str(plugin_prefs.get('opml_import_single_click', 'advanced') or 'advanced')
        except Exception:
            mode = 'advanced'
        if mode == 'recipes':
            try:
                self.import_from_recipes()
            except Exception:
                pass
        elif mode == 'advanced':
            try:
                self.import_opml_advanced()
            except Exception:
                try:
                    self.import_opml()
                except Exception:
                    pass
        else:
            try:
                self.import_opml()
            except Exception:
                pass

    def _on_export_clicked(self):
        try:
            mode = str(plugin_prefs.get('opml_export_single_click', 'advanced') or 'advanced')
        except Exception:
            mode = 'advanced'
        if mode == 'advanced':
            try:
                self.export_opml_advanced()
            except Exception:
                try:
                    self.export_opml()
                except Exception:
                    pass
        else:
            try:
                self.export_opml()
            except Exception:
                pass

    # --- Database switching helpers (minimal implementation for testing) ---
    def _on_db_open(self):
        try:
            from qt.core import QFileDialog, QMessageBox
        except Exception:
            from PyQt5.Qt import QFileDialog, QMessageBox
        try:
            self.status.setText(_('Opening database file picker...'))
            try:
                from calibre.gui2.qt_file_dialogs import choose_open_file
                fname = choose_open_file(self, _('Open database (one-off)'), '', _('SQLite files (*.sqlite *.db);;All files (*)'))
            except Exception:
                fname, _dummy = QFileDialog.getOpenFileName(self, _('Open database (one-off)'), '', _('SQLite files (*.sqlite *.db);;All files (*)'))
            if not fname:
                self.status.setText('Cancelled.')
                return
            try:
                rss_db.assert_db_path_allowed(fname)
            except Exception as e:
                try:
                    QMessageBox.warning(self, _('Invalid DB location'), str(e))
                except Exception:
                    pass
                self.status.setText(_('Cancelled.'))
                return
            # One-off switch: do not automatically create a profile
            try:
                plugin_prefs['db_profiles_active'] = ''
            except Exception:
                pass
            # Explicit user action: allow this switch without the fallback prompt
            try:
                setattr(self, '_allow_config_db_switch', True)
            except Exception:
                pass
            try:
                try:
                    self._switch_database(fname, readonly=False)
                finally:
                    try:
                        setattr(self, '_allow_config_db_switch', False)
                    except Exception:
                        pass
            except Exception:
                pass
        except Exception as e:
            self.status.setText('Error: ' + str(e))
            try:
                QMessageBox.warning(self, _('Error'), str(e))
            except Exception:
                pass

    def _show_current_db_path(self):
        try:
            from qt.core import QMessageBox
        except Exception:
            from PyQt5.Qt import QMessageBox
        try:
            path = str(rss_db.db_path() or 'Unknown')
            self.status.setText('DB: ' + path)
            QMessageBox.information(self, _('DB Path'), path)
        except Exception as e:
            self.status.setText('Error reading DB path: ' + str(e))
            try:
                QMessageBox.information(self, _('DB Path'), _('Unknown'))
            except Exception:
                pass

    def _switch_to_config_database(self):
        try:
            p = str(getattr(self, '_db_config_path', '') or '').strip()
        except Exception:
            p = ''
        if not p:
            return

        # Switching to the suggested default is a runtime switch only.
        # Users can explicitly adopt it as the configured Default DB via
        # "Manage profiles…" -> "Set Default".
        try:
            plugin_prefs['db_profiles_active'] = ''
        except Exception:
            pass
        # This is an explicit user action: allow switching to the suggested
        # Config DB without triggering the automatic-fallback confirmation.
        try:
            setattr(self, '_allow_config_db_switch', True)
        except Exception:
            pass
        try:
            try:
                self._switch_database(p, readonly=False)
            finally:
                try:
                    setattr(self, '_allow_config_db_switch', False)
                except Exception:
                    pass
        except Exception:
            pass
        try:
            self._update_profile_label()
        except Exception:
            pass

    def _save_current_db_as_profile(self):
        try:
            from qt.core import QMessageBox
        except Exception:
            from PyQt5.Qt import QMessageBox
        try:
            cur_path = ''
            try:
                cur_path = str(rss_db.db_path() or '').strip()
            except Exception:
                cur_path = ''
            if not cur_path:
                try:
                    QMessageBox.information(self, _('Info'), _('No current database path.'))
                except Exception:
                    pass
                return

            try:
                base = os.path.splitext(os.path.basename(cur_path))[0]
            except Exception:
                base = ''
            try:
                cur_ro = bool(getattr(rss_db, 'DB_READONLY', False))
            except Exception:
                cur_ro = False

            dlg = _AddEditProfileDialog(self, initial={'name': base, 'path': cur_path, 'readonly': cur_ro, 'mirror': False}, mode='save')
            # Saving an existing DB as a profile should not offer creating a
            # new blank DB file.
            try:
                if getattr(dlg, 'create_blank_cb', None) is not None:
                    dlg.create_blank_cb.setChecked(False)
                    dlg.create_blank_cb.setEnabled(False)
            except Exception:
                pass
            if dlg.exec() != QDialog.DialogCode.Accepted:
                return
            v = dlg.get_values() or {}
            if not v.get('path'):
                try:
                    QMessageBox.warning(self, _('Error'), _('Please choose a database file path.'))
                except Exception:
                    pass
                return

            try:
                rss_db.assert_db_path_allowed(v.get('path'))
            except Exception as e:
                try:
                    QMessageBox.warning(self, _('Invalid DB location'), str(e))
                except Exception:
                    pass
                return

            try:
                target_path = os.path.normpath(str(v.get('path') or '').strip())
            except Exception:
                target_path = str(v.get('path') or '').strip()

            try:
                profiles = plugin_prefs.get('db_profiles') or []
            except Exception:
                profiles = []

            entered_name = str(v.get('name') or '').strip()
            try:
                # Name collision check (ignore same-path updates)
                for p in profiles:
                    try:
                        p_path = os.path.normpath(str(p.get('path') or '').strip())
                    except Exception:
                        p_path = str(p.get('path') or '').strip()
                    if p_path == target_path:
                        continue
                    if str(p.get('name') or '').strip().lower() == entered_name.lower() and entered_name:
                        QMessageBox.warning(self, _('Duplicate Profile Name'), _('A profile with name "%s" already exists. Please choose a unique name.') % entered_name)
                        return
            except Exception:
                pass

            # If this DB path is already saved as a profile, update it instead
            # of adding a duplicate entry.
            existing_id = ''
            try:
                for p in profiles:
                    try:
                        p_path = os.path.normpath(str(p.get('path') or '').strip())
                    except Exception:
                        p_path = str(p.get('path') or '').strip()
                    if p_path and target_path and p_path == target_path:
                        existing_id = str(p.get('id') or '')
                        # Update fields
                        try:
                            if entered_name:
                                p['name'] = entered_name
                        except Exception:
                            pass
                        try:
                            p['emoji'] = v.get('emoji') or ''
                        except Exception:
                            pass
                        try:
                            p['readonly'] = bool(v.get('readonly', False))
                        except Exception:
                            pass
                        try:
                            p['mirror'] = bool(v.get('mirror', False))
                        except Exception:
                            pass
                        break
            except Exception:
                existing_id = ''

            if existing_id:
                pid = existing_id
            else:
                pid = str(uuid.uuid4())
                prof = {
                    'id': pid,
                    'name': entered_name or pid,
                    'emoji': v.get('emoji') or '',
                    'path': target_path or (v.get('path') or ''),
                    'readonly': v.get('readonly', False),
                    'mirror': v.get('mirror', False),
                }
                profiles.append(prof)

            try:
                plugin_prefs['db_profiles'] = profiles
            except Exception:
                pass

            # Activate the newly created profile
            try:
                self._set_active_profile(pid, persist=True)
            except Exception:
                try:
                    plugin_prefs['db_profiles_active'] = str(pid)
                except Exception:
                    pass
            try:
                self._rebuild_profiles_menu()
            except Exception:
                pass
            try:
                self._update_profile_label()
            except Exception:
                pass
        except Exception:
            _debug('ERROR in _save_current_db_as_profile: %s' % traceback.format_exc())

    def _add_existing_db_as_profile(self):
        """Add an existing database file from disk as a named profile.

        This is a convenience wrapper around the same profile-creation
        logic used by "Save current DB as profile…", but starts with an
        empty path so the user can pick any existing DB file.
        """
        try:
            from qt.core import QMessageBox
        except Exception:
            from PyQt5.Qt import QMessageBox
        try:
            # Start with an empty path so the dialog prompts the user to
            # choose a DB file. Defaults: writable, not mirror.
            initial = {
                'name': '',
                'path': '',
                'readonly': False,
                'mirror': False,
                'emoji': '',
            }
            dlg = _AddEditProfileDialog(self, initial=initial, mode='add')
            # Hide create-blank in Add flow
            try:
                if getattr(dlg, 'create_blank_cb', None) is not None:
                    dlg.create_blank_cb.setChecked(False)
                    dlg.create_blank_cb.setEnabled(False)
                    dlg.create_blank_cb.setVisible(False)
            except Exception:
                pass
            if dlg.exec() != QDialog.DialogCode.Accepted:
                return
            v = dlg.get_values() or {}
            path = str(v.get('path') or '').strip()
            if not path:
                try:
                    QMessageBox.warning(self, _('Error'), _('Please choose a database file path.'))
                except Exception:
                    pass
                return

            # Require the file to exist: this action is explicitly for
            # adding an existing DB from a folder.
            try:
                if not os.path.exists(path):
                    QMessageBox.warning(self, _('File not found'), _('Database file not found: %s') % path)
                    return
            except Exception:
                pass

            try:
                rss_db.assert_db_path_allowed(path)
            except Exception as e:
                try:
                    QMessageBox.warning(self, _('Invalid DB location'), str(e))
                except Exception:
                    pass
                return

            pid = str(uuid.uuid4())
            prof = {
                'id': pid,
                'name': v.get('name') or pid,
                'emoji': v.get('emoji') or '',
                'path': path,
                'readonly': v.get('readonly', False),
                'mirror': v.get('mirror', False),
            }

            try:
                profiles = plugin_prefs.get('db_profiles') or []
            except Exception:
                profiles = []
            profiles.append(prof)
            try:
                plugin_prefs['db_profiles'] = profiles
            except Exception:
                pass

            # Do not automatically switch; offer the user a choice.
            try:
                msg = _('Profile created: %s\n\nSwitch to it now?') % (prof.get('name') or prof.get('path') or '')
            except Exception:
                msg = 'Switch to it now?'
            try:
                if QMessageBox.question(self, _('Switch now?'), msg) == QMessageBox.StandardButton.Yes:
                    try:
                        self._set_active_profile(pid, persist=True)
                    except Exception:
                        pass
            except Exception:
                pass
            try:
                self._rebuild_profiles_menu()
            except Exception:
                pass
            try:
                self._update_profile_label()
            except Exception:
                pass
        except Exception:
            _debug('ERROR in _add_existing_db_as_profile: %s' % traceback.format_exc())

    def _create_blank_database(self):
        try:
            from qt.core import QFileDialog, QMessageBox
        except Exception:
            from PyQt5.Qt import QFileDialog, QMessageBox
        try:
            self.status.setText(_('Creating blank DB...'))
            default = 'rss_reader_blank.sqlite'
            fname, _dummy = QFileDialog.getSaveFileName(self, _('Create blank database'), default, _('SQLite files (*.sqlite *.db);;All files (*)'))
            if not fname:
                self.status.setText(_('Cancelled.'))
                return
            try:
                rss_db.assert_db_path_allowed(fname)
            except Exception as e:
                try:
                    QMessageBox.warning(self, _('Invalid DB location'), str(e))
                except Exception:
                    pass
                self.status.setText(_('Cancelled.'))
                return
            # If exists, confirm overwrite
            try:
                if os.path.exists(fname):
                    if QMessageBox.question(self, _('Overwrite?'), _('File exists. Overwrite?')) != QMessageBox.StandardButton.Yes:
                        self.status.setText(_('Cancelled.'))
                        return
                    try:
                        os.remove(fname)
                    except Exception:
                        pass
            except Exception:
                pass
            # Cancel any running update before switching DB so we don't continue fetching the old DB
            try:
                if getattr(self, 'action', None) is not None:
                    try:
                        self.action.cancel_update()
                    except Exception:
                        pass
            except Exception:
                pass

            # Switch DB path and initialize schema
            try:
                try:
                    self._db_prev_path = rss_db.db_path()
                    try:
                        self._db_prev_readonly = bool(getattr(rss_db, 'DB_READONLY', False))
                    except Exception:
                        self._db_prev_readonly = False
                except Exception:
                    self._db_prev_path = None
                try:
                    plugin_prefs['db_profiles_active'] = ''
                except Exception:
                    pass
                try:
                    rss_db.set_db_path(str(fname), readonly=False)
                except Exception:
                    raise
                self._db_current_path = str(fname)
                try:
                    if getattr(self, 'action', None) is not None:
                        try:
                            self.action._refresh_toolbar_label()
                        except Exception:
                            pass
                except Exception:
                    pass
                rss_db.init_db()
            except Exception as e:
                self.status.setText(_('Create blank DB failed: %s') % str(e))
                try:
                    QMessageBox.warning(self, _('Error'), str(e))
                except Exception:
                    pass
                return

            # Persist as a profile so it shows up in Manage DB Profiles.
            try:
                try:
                    new_path = os.path.normpath(str(fname))
                except Exception:
                    new_path = str(fname)

                try:
                    profiles = list(plugin_prefs.get('db_profiles') or [])
                except Exception:
                    profiles = []

                existing_id = ''
                try:
                    for p in profiles:
                        try:
                            p_path = str(p.get('path') or '').strip()
                            if not p_path:
                                continue
                            try:
                                p_path = os.path.normpath(p_path)
                            except Exception:
                                pass
                            if p_path == new_path:
                                existing_id = str(p.get('id') or '')
                                break
                        except Exception:
                            continue
                except Exception:
                    existing_id = ''

                if existing_id:
                    pid = existing_id
                else:
                    pid = str(uuid.uuid4())
                    try:
                        base = os.path.splitext(os.path.basename(new_path))[0]
                    except Exception:
                        base = ''
                    prof = {
                        'id': pid,
                        'name': base or pid,
                        'emoji': '',
                        'path': new_path,
                        'readonly': False,
                        'mirror': False,
                    }
                    profiles.append(prof)
                    try:
                        plugin_prefs['db_profiles'] = profiles
                    except Exception:
                        pass

                try:
                    plugin_prefs['db_profiles_active'] = str(pid)
                except Exception:
                    pass
                try:
                    self._rebuild_profiles_menu()
                except Exception:
                    pass
                try:
                    self._update_profile_label()
                except Exception:
                    pass
            except Exception:
                pass
            # Reload UI
            try:
                try:
                    self._feeds_results = {}
                except Exception:
                    pass
                self.load_from_cache()
            except Exception:
                pass
            try:
                self.refresh()
            except Exception:
                pass
            try:
                if hasattr(self, '_update_feeds_label'):
                    self._update_feeds_label()
                if hasattr(self, '_update_items_label'):
                    self._update_items_label()
                if hasattr(self, '_update_current_feed_label'):
                    self._update_current_feed_label()
            except Exception:
                pass
            self.status.setText(_('Created and switched to blank DB: %s') % str(fname))
            try:
                QMessageBox.information(self, _('Blank DB created'), _('Created and switched to: %s') % str(fname))
            except Exception:
                pass
        except Exception as e:
            self.status.setText(_('Create blank DB failed: %s') % str(e))
            try:
                QMessageBox.warning(self, _('Error'), str(e))
            except Exception:
                pass

    def _switch_database(self, path, readonly=False):
        # Runtime switch: set rss_db path (+ readonly) and reload cache.
        try:
            # Cancel any running update before changing DB path so ongoing fetches stop
            try:
                if getattr(self, 'action', None) is not None:
                    try:
                        self.action.cancel_update()
                    except Exception:
                        pass
            except Exception:
                pass
            old = None
            try:
                old = rss_db.db_path()
            except Exception:
                old = None
            try:
                # store previous only if different
                if old and old != str(path):
                    self._db_prev_path = old
                    try:
                        self._db_prev_readonly = bool(getattr(rss_db, 'DB_READONLY', False))
                    except Exception:
                        self._db_prev_readonly = False
            except Exception:
                pass
            try:
                rss_db.set_db_path(str(path), readonly=bool(readonly))
                self._db_current_path = str(path)
                try:
                    if getattr(self, 'action', None) is not None:
                        try:
                            self.action._refresh_toolbar_label()
                        except Exception:
                            pass
                except Exception:
                    pass
            except Exception:
                raise

            # Only create/migrate schema if writable
            try:
                if not bool(readonly):
                    rss_db.ensure_ready()
                else:
                    # Validate the DB is readable (best-effort)
                    try:
                        rss_db.get_feeds()
                    except Exception:
                        pass
            except Exception:
                pass
            # Reload UI data
            try:
                try:
                    self._feeds_results = {}
                except Exception:
                    pass
                self.load_from_cache()
            except Exception:
                pass
            try:
                self.refresh()
            except Exception:
                pass
            try:
                if hasattr(self, '_update_feeds_label'):
                    self._update_feeds_label()
                if hasattr(self, '_update_items_label'):
                    self._update_items_label()
                if hasattr(self, '_update_current_feed_label'):
                    self._update_current_feed_label()
            except Exception:
                pass
            try:
                msg = 'Switched DB%s: %s' % (' (read-only)' if bool(readonly) else '', str(path))
                self.status.setText(msg)
            except Exception:
                pass
        except Exception as e:
            self.status.setText('Switch failed: ' + str(e))
            try:
                from qt.core import QMessageBox
            except Exception:
                try:
                    from PyQt5.Qt import QMessageBox
                except Exception:
                    QMessageBox = None
            if QMessageBox is not None:
                try:
                    QMessageBox.warning(self, _('Switch DB failed'), str(e))
                except Exception:
                    pass

    def _install_toolbar_context_menu(self):
        try:
            w = getattr(self, '_toolbar_widget', None)
            if w is None:
                return
            try:
                w.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            except Exception:
                try:
                    w.setContextMenuPolicy(Qt.CustomContextMenu)
                except Exception:
                    return
            try:
                w.customContextMenuRequested.connect(self._on_toolbar_context_menu)
            except Exception:
                pass
        except Exception:
            pass

    def _on_toolbar_context_menu(self, pos):
        try:
            w = getattr(self, '_toolbar_widget', None)
            if w is None:
                return
            menu = QMenu(w)

            act = QAction(_('Cu&stomize toolbar…'), menu)
            act.triggered.connect(self._show_toolbar_customizer)
            menu.addAction(act)

            menu.addSeparator()

            compact = False
            try:
                compact = bool(plugin_prefs.get('toolbar_compact', False))
            except Exception:
                compact = False
            act_comp = QAction(_('Compact toolbar (icons only)'), menu)
            try:
                act_comp.setCheckable(True)
                act_comp.setChecked(compact)
            except Exception:
                pass
            act_comp.triggered.connect(lambda checked=False: self._set_toolbar_compact(bool(checked)))
            menu.addAction(act_comp)

            try:
                menu.exec(w.mapToGlobal(pos))
            except Exception:
                try:
                    menu.exec_(w.mapToGlobal(pos))
                except Exception:
                    pass
        except Exception:
            pass

    def _install_mini_toolbar_context_menu(self):
        try:
            w = getattr(self, '_mini_toolbar_widget', None)
            if w is None:
                return
            try:
                w.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            except Exception:
                try:
                    w.setContextMenuPolicy(Qt.CustomContextMenu)
                except Exception:
                    return
            try:
                w.customContextMenuRequested.connect(self._on_mini_toolbar_context_menu)
            except Exception:
                pass
        except Exception:
            pass

    def _on_mini_toolbar_context_menu(self, pos):
        try:
            w = getattr(self, '_mini_toolbar_widget', None)
            if w is None:
                return
            menu = QMenu(w)

            act = QAction(_('Customize mini toolbar…'), menu)
            act.triggered.connect(self._show_mini_toolbar_customizer)
            menu.addAction(act)

            try:
                menu.exec(w.mapToGlobal(pos))
            except Exception:
                try:
                    menu.exec_(w.mapToGlobal(pos))
                except Exception:
                    pass
        except Exception:
            pass

    def _set_customize_toolbar_button_visible(self, visible):
        try:
            plugin_prefs['show_customize_toolbar_button'] = bool(visible)
        except Exception:
            pass
        try:
            btn = getattr(self, 'customize_toolbar_btn', None)
            if btn is not None:
                btn.setVisible(bool(visible))
        except Exception:
            pass

    def _set_toolbar_compact(self, compact):
        try:
            plugin_prefs['toolbar_compact'] = bool(compact)
        except Exception:
            pass

        # Apply style to toolbuttons that have text.
        try:
            style = Qt.ToolButtonStyle.ToolButtonIconOnly if compact else Qt.ToolButtonStyle.ToolButtonTextBesideIcon
        except Exception:
            style = None

        try:
            for wid in (getattr(self, '_toolbar_items', {}) or {}).values():
                try:
                    if wid is None:
                        continue
                    if isinstance(wid, QToolButton):
                        try:
                            if style is not None:
                                wid.setToolButtonStyle(style)
                        except Exception:
                            try:
                                wid.setToolButtonStyle(Qt.ToolButtonIconOnly if compact else Qt.ToolButtonTextBesideIcon)
                            except Exception:
                                pass
                except Exception:
                    continue
        except Exception:
            pass

    def _apply_toolbar_customization(self):
        # Read prefs and rebuild the toolbar layout.
        try:
            default_order = list(getattr(self, '_toolbar_item_ids', []) or [])
            if not default_order:
                return
            saved_order = []
            try:
                saved_order = list(plugin_prefs.get('toolbar_order', []) or [])
            except Exception:
                saved_order = []

            hidden = set()
            try:
                hidden = set(plugin_prefs.get('toolbar_hidden', []) or [])
            except Exception:
                hidden = set()

            # Hide the optional stop button by default unless user explicitly added it
            try:
                if 'stop_update' in default_order and 'stop_update' not in hidden and 'stop_update' not in saved_order:
                    hidden.add('stop_update')
                    try:
                        plugin_prefs['toolbar_hidden'] = list(hidden)
                    except Exception:
                        pass
            except Exception:
                pass

            # New optional toolbar items should default to hidden unless the
            # user explicitly adds them via the customizer.
            try:
                # Stop button: keep hidden until explicitly enabled.
                if 'stop_update' in default_order and 'stop_update' not in hidden and 'stop_update' not in saved_order:
                    hidden.add('stop_update')
                # Cleanup button: also start hidden so the initial toolbar
                # remains compact. Users can enable it via Customize Toolbar.
                if 'cleanup' in default_order and 'cleanup' not in hidden and 'cleanup' not in saved_order:
                    hidden.add('cleanup')
                try:
                    plugin_prefs['toolbar_hidden'] = list(hidden)
                except Exception:
                    pass
            except Exception:
                pass

            # Normalize order: keep only known ids, keep at most one stretch.
            order = []
            saw_stretch = False
            for i in saved_order:
                try:
                    i = str(i)
                except Exception:
                    continue
                if i == '__stretch__':
                    if saw_stretch:
                        continue
                    saw_stretch = True
                    order.append(i)
                    continue
                if i in default_order and i not in order:
                    order.append(i)

            # Add missing defaults.
            for i in default_order:
                if i == '__stretch__':
                    continue
                if i not in order:
                    order.append(i)

            # Ensure a stretch exists.
            if '__stretch__' not in order:
                try:
                    idx = order.index('layout') + 1
                except Exception:
                    idx = 7
                order.insert(max(0, min(len(order), idx)), '__stretch__')

            self._rebuild_toolbar_layout(order, hidden)

            # Apply compact mode after rebuild.
            try:
                self._set_toolbar_compact(bool(plugin_prefs.get('toolbar_compact', False)))
            except Exception:
                pass
        except Exception:
            pass

    def _rebuild_toolbar_layout(self, order, hidden_ids):
        try:
            w = getattr(self, '_toolbar_widget', None)
            lay = getattr(self, '_toolbar_layout', None)
            items = getattr(self, '_toolbar_items', {}) or {}
            if w is None or lay is None:
                return

            # Clear current layout (keep widgets alive).
            try:
                while lay.count():
                    it = lay.takeAt(0)
                    try:
                        ww = it.widget()
                    except Exception:
                        ww = None
                    if ww is not None:
                        try:
                            ww.setParent(w)
                        except Exception:
                            pass
            except Exception:
                pass

            hidden = set(hidden_ids or [])

            for tid in (order or []):
                if tid == '__stretch__':
                    try:
                        lay.addStretch(1)
                    except Exception:
                        pass
                    continue
                wid = items.get(tid)
                if wid is None:
                    continue
                try:
                    wid.setVisible(tid not in hidden)
                except Exception:
                    pass
                if tid in hidden:
                    continue
                try:
                    lay.addWidget(wid)
                except Exception:
                    pass

            try:
                w.updateGeometry()
            except Exception:
                pass
        except Exception:
            pass

    def _apply_mini_toolbar_customization(self):
        try:
            default_order = list(getattr(self, '_mini_toolbar_item_ids', []) or [])
            if not default_order:
                return
            saved_order = []
            try:
                saved_order = list(plugin_prefs.get('mini_toolbar_order', []) or [])
            except Exception:
                saved_order = []

            hidden = set()
            try:
                hidden = set(plugin_prefs.get('mini_toolbar_hidden', []) or [])
            except Exception:
                hidden = set()

            order = []
            saw_stretch = False
            for i in saved_order:
                try:
                    i = str(i)
                except Exception:
                    continue
                if i == '__stretch__':
                    if saw_stretch:
                        continue
                    saw_stretch = True
                    order.append(i)
                    continue
                if i in default_order and i not in order:
                    order.append(i)

            for i in default_order:
                if i == '__stretch__':
                    continue
                if i not in order:
                    order.append(i)

            if '__stretch__' not in order:
                order.append('__stretch__')

            self._rebuild_mini_toolbar_layout(order, hidden)
        except Exception:
            pass

    def _rebuild_mini_toolbar_layout(self, order, hidden_ids):
        try:
            w = getattr(self, '_mini_toolbar_widget', None)
            lay = getattr(self, '_mini_toolbar_layout', None)
            items = getattr(self, '_mini_toolbar_items', {}) or {}
            if w is None or lay is None:
                return

            try:
                while lay.count():
                    it = lay.takeAt(0)
                    try:
                        ww = it.widget()
                    except Exception:
                        ww = None
                    if ww is not None:
                        try:
                            ww.setParent(w)
                        except Exception:
                            pass
            except Exception:
                pass

            hidden = set(hidden_ids or [])

            for tid in (order or []):
                if tid == '__stretch__':
                    try:
                        lay.addStretch(1)
                    except Exception:
                        pass
                    continue
                wid = items.get(tid)
                if wid is None:
                    continue
                try:
                    wid.setVisible(tid not in hidden)
                except Exception:
                    pass
                if tid in hidden:
                    continue
                try:
                    lay.addWidget(wid)
                except Exception:
                    pass

            try:
                w.updateGeometry()
            except Exception:
                pass
        except Exception:
            pass

    def _show_toolbar_customizer(self):
        try:
            # Build label mapping for user-friendly names
            labels = {
                'add': _('Add'),
                'import': _('Import'),
                'export': _('Export'),
                'remove': _('Remove'),
                'edit': _('Edit'),
                'layout': _('Layout'),
                'customize_toolbar': _('Customize Toolbar'),
                'settings': _('Settings'),
                'profiles': _('Profiles'),
                'db': _('DB'),
                'zoom_out': _('Zoom out'),
                'zoom_label': _('Zoom %'),
                'zoom_in': _('Zoom in'),
                'zoom_reset': _('Reset zoom'),
                'cleanup': _('Clean up'),
                'update': _('Update'),
                'update_all': _('Update all'),
                'stop_update': _('Stop fetching now'),
                'auto_update': _('Auto-update interval'),
            }

            default_order = list(getattr(self, '_toolbar_item_ids', []) or [])
            items = getattr(self, '_toolbar_items', {}) or {}
            if not default_order or not items:
                return

            saved_order = []
            try:
                saved_order = list(plugin_prefs.get('toolbar_order', []) or [])
            except Exception:
                saved_order = []
            hidden = set()
            try:
                hidden = set(plugin_prefs.get('toolbar_hidden', []) or [])
            except Exception:
                hidden = set()

            # Start with effective order used by the toolbar.
            eff = []
            saw_stretch = False
            for tid in saved_order:
                tid = str(tid)
                if tid == '__stretch__':
                    if saw_stretch:
                        continue
                    saw_stretch = True
                    eff.append(tid)
                    continue
                if tid in default_order and tid not in eff:
                    eff.append(tid)
            for tid in default_order:
                if tid != '__stretch__' and tid not in eff:
                    eff.append(tid)
            if '__stretch__' not in eff:
                try:
                    eff.insert(eff.index('layout') + 1, '__stretch__')
                except Exception:
                    eff.insert(7, '__stretch__')

            d = QDialog(self)
            try:
                d.setWindowTitle(_('Customize toolbar'))
            except Exception:
                pass
            try:
                d.setMinimumHeight(420)
            except Exception:
                pass
            layout = QVBoxLayout(d)
            layout.setContentsMargins(10, 10, 10, 10)

            hint = QLabel(_('Drag to reorder. Uncheck to hide.'), d)
            try:
                hint.setStyleSheet('color: gray;')
            except Exception:
                pass
            layout.addWidget(hint)

            lw = QListWidget(d)
            try:
                lw.setDragDropMode(QAbstractItemView.DragDropMode.InternalMove)
            except Exception:
                try:
                    lw.setDragDropMode(QAbstractItemView.InternalMove)
                except Exception:
                    pass
            layout.addWidget(lw)

            for tid in eff:
                if tid in ('__stretch__', '__separator__'):
                    it = QListWidgetItem(_('Separator'), lw)
                    try:
                        it.setFlags(Qt.ItemFlag.ItemIsEnabled)
                    except Exception:
                        try:
                            it.setFlags(Qt.ItemIsEnabled)
                        except Exception:
                            pass
                    try:
                        it.setData(Qt.ItemDataRole.UserRole, '__stretch__')
                    except Exception:
                        try:
                            it.setData(Qt.UserRole, '__stretch__')
                        except Exception:
                            pass
                    continue

                if items.get(tid) is None:
                    continue
                it = QListWidgetItem(labels.get(tid, tid), lw)
                try:
                    it.setFlags(it.flags() | Qt.ItemFlag.ItemIsUserCheckable | Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsSelectable | Qt.ItemFlag.ItemIsDragEnabled)
                except Exception:
                    try:
                        it.setFlags(it.flags() | Qt.ItemIsUserCheckable | Qt.ItemIsEnabled | Qt.ItemIsSelectable | Qt.ItemIsDragEnabled)
                    except Exception:
                        pass
                try:
                    it.setCheckState(Qt.CheckState.Unchecked if tid in hidden else Qt.CheckState.Checked)
                except Exception:
                    try:
                        it.setCheckState(Qt.Unchecked if tid in hidden else Qt.Checked)
                    except Exception:
                        pass
                try:
                    it.setData(Qt.ItemDataRole.UserRole, tid)
                except Exception:
                    try:
                        it.setData(Qt.UserRole, tid)
                    except Exception:
                        pass

            buttons = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, d)
            try:
                reset_btn = buttons.addButton(_('Reset'), QDialogButtonBox.ButtonRole.ResetRole)
                reset_btn.clicked.connect(lambda: self._reset_toolbar_customization_and_close(d))
            except Exception:
                reset_btn = None
            buttons.accepted.connect(d.accept)
            buttons.rejected.connect(d.reject)
            layout.addWidget(buttons)

            # Restore saved geometry for the customize-toolbar dialog if available.
            try:
                import base64
                geom_b64 = str(plugin_prefs.get('toolbar_customizer_geometry') or '')
                if geom_b64:
                    try:
                        geom_bytes = base64.b64decode(geom_b64)
                        d.restoreGeometry(geom_bytes)
                    except Exception:
                        pass
            except Exception:
                pass

            try:
                res = d.exec()
            except Exception:
                try:
                    res = d.exec_()
                except Exception:
                    res = 0

            # Save dialog geometry so next time we restore its position/size.
            try:
                import base64
                try:
                    g = d.saveGeometry()
                    if g:
                        try:
                            plugin_prefs['toolbar_customizer_geometry'] = base64.b64encode(bytes(g)).decode('ascii')
                        except Exception:
                            try:
                                plugin_prefs['toolbar_customizer_geometry'] = str(g)
                            except Exception:
                                pass
                except Exception:
                    pass
            except Exception:
                pass
            try:
                accepted = int(getattr(QDialog, 'Accepted', 1))
            except Exception:
                accepted = 1
            try:
                if int(res) != accepted:
                    return
            except Exception:
                return

            new_order = []
            new_hidden = []
            stretch_seen = False
            for i in range(lw.count()):
                it = lw.item(i)
                try:
                    tid = it.data(Qt.ItemDataRole.UserRole)
                except Exception:
                    tid = it.data(Qt.UserRole)
                tid = str(tid)
                if tid == '__stretch__':
                    if stretch_seen:
                        continue
                    stretch_seen = True
                    new_order.append(tid)
                    continue
                new_order.append(tid)
                try:
                    cs = it.checkState()
                    unchecked = (cs == Qt.CheckState.Unchecked) if hasattr(Qt, 'CheckState') else (cs == Qt.Unchecked)
                except Exception:
                    unchecked = False
                if unchecked:
                    new_hidden.append(tid)

            try:
                plugin_prefs['toolbar_order'] = new_order
                plugin_prefs['toolbar_hidden'] = new_hidden
            except Exception:
                pass

            try:
                self._apply_toolbar_customization()
            except Exception:
                pass
        except Exception:
            pass

    def _show_mini_toolbar_customizer(self):
        try:
            labels = {
                'items_quick_filter': _('Quick filter'),
                'items_quick_filter_clear': _('Clear quick filter'),
                'items_quick_tags': _('Tags'),
                'mark_read': _('Mark feed read'),
                'export_ebook': _('Export'),
                'save_article': _('Save article'),
                'images': _('Images'),
                'autofit_images': _('Autofit images'),
                'share': _('Share'),
                'email': _('Email'),
            }

            default_order = list(getattr(self, '_mini_toolbar_item_ids', []) or [])
            items = getattr(self, '_mini_toolbar_items', {}) or {}
            if not default_order or not items:
                return

            saved_order = []
            try:
                saved_order = list(plugin_prefs.get('mini_toolbar_order', []) or [])
            except Exception:
                saved_order = []
            hidden = set()
            try:
                hidden = set(plugin_prefs.get('mini_toolbar_hidden', []) or [])
            except Exception:
                hidden = set()

            eff = []
            saw_stretch = False
            for tid in saved_order:
                tid = str(tid)
                if tid == '__stretch__':
                    if saw_stretch:
                        continue
                    saw_stretch = True
                    eff.append(tid)
                    continue
                if tid in default_order and tid not in eff:
                    eff.append(tid)
            for tid in default_order:
                if tid != '__stretch__' and tid not in eff:
                    eff.append(tid)
            if '__stretch__' not in eff:
                eff.append('__stretch__')

            d = QDialog(self)
            try:
                d.setWindowTitle(_('Customize mini toolbar'))
            except Exception:
                pass
            try:
                d.setMinimumHeight(380)
            except Exception:
                pass
            layout = QVBoxLayout(d)
            layout.setContentsMargins(10, 10, 10, 10)

            hint = QLabel(_('Drag to reorder. Uncheck to hide.'), d)
            try:
                hint.setStyleSheet('color: gray;')
            except Exception:
                pass
            layout.addWidget(hint)

            lw = QListWidget(d)
            try:
                lw.setDragDropMode(QAbstractItemView.DragDropMode.InternalMove)
            except Exception:
                try:
                    lw.setDragDropMode(QAbstractItemView.InternalMove)
                except Exception:
                    pass
            layout.addWidget(lw)

            for tid in eff:
                if tid in ('__stretch__', '__separator__'):
                    it = QListWidgetItem(_('Separator'), lw)
                    try:
                        it.setFlags(Qt.ItemFlag.ItemIsEnabled)
                    except Exception:
                        try:
                            it.setFlags(Qt.ItemIsEnabled)
                        except Exception:
                            pass
                    try:
                        it.setData(Qt.ItemDataRole.UserRole, '__stretch__')
                    except Exception:
                        try:
                            it.setData(Qt.UserRole, '__stretch__')
                        except Exception:
                            pass
                    continue

                if items.get(tid) is None:
                    continue
                it = QListWidgetItem(labels.get(tid, tid), lw)
                try:
                    it.setFlags(it.flags() | Qt.ItemFlag.ItemIsUserCheckable | Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsSelectable | Qt.ItemFlag.ItemIsDragEnabled)
                except Exception:
                    try:
                        it.setFlags(it.flags() | Qt.ItemIsUserCheckable | Qt.ItemIsEnabled | Qt.ItemIsSelectable | Qt.ItemIsDragEnabled)
                    except Exception:
                        pass
                try:
                    it.setCheckState(Qt.CheckState.Unchecked if tid in hidden else Qt.CheckState.Checked)
                except Exception:
                    try:
                        it.setCheckState(Qt.Unchecked if tid in hidden else Qt.Checked)
                    except Exception:
                        pass
                try:
                    it.setData(Qt.ItemDataRole.UserRole, tid)
                except Exception:
                    try:
                        it.setData(Qt.UserRole, tid)
                    except Exception:
                        pass

            buttons = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, d)
            try:
                reset_btn = buttons.addButton(_('Reset'), QDialogButtonBox.ButtonRole.ResetRole)
                reset_btn.clicked.connect(lambda: self._reset_mini_toolbar_customization_and_close(d))
            except Exception:
                reset_btn = None
            buttons.accepted.connect(d.accept)
            buttons.rejected.connect(d.reject)
            layout.addWidget(buttons)

            try:
                import base64
                geom_b64 = str(plugin_prefs.get('mini_toolbar_customizer_geometry') or '')
                if geom_b64:
                    try:
                        geom_bytes = base64.b64decode(geom_b64)
                        d.restoreGeometry(geom_bytes)
                    except Exception:
                        pass
            except Exception:
                pass

            try:
                res = d.exec()
            except Exception:
                try:
                    res = d.exec_()
                except Exception:
                    res = 0

            try:
                import base64
                try:
                    g = d.saveGeometry()
                    if g:
                        try:
                            plugin_prefs['mini_toolbar_customizer_geometry'] = base64.b64encode(bytes(g)).decode('ascii')
                        except Exception:
                            try:
                                plugin_prefs['mini_toolbar_customizer_geometry'] = str(g)
                            except Exception:
                                pass
                except Exception:
                    pass
            except Exception:
                pass
            try:
                accepted = int(getattr(QDialog, 'Accepted', 1))
            except Exception:
                accepted = 1
            try:
                if int(res) != accepted:
                    return
            except Exception:
                return

            new_order = []
            new_hidden = []
            stretch_seen = False
            for i in range(lw.count()):
                it = lw.item(i)
                try:
                    tid = it.data(Qt.ItemDataRole.UserRole)
                except Exception:
                    tid = it.data(Qt.UserRole)
                tid = str(tid)
                if tid == '__stretch__':
                    if stretch_seen:
                        continue
                    stretch_seen = True
                    new_order.append(tid)
                    continue
                new_order.append(tid)
                try:
                    cs = it.checkState()
                    unchecked = (cs == Qt.CheckState.Unchecked) if hasattr(Qt, 'CheckState') else (cs == Qt.Unchecked)
                except Exception:
                    unchecked = False
                if unchecked:
                    new_hidden.append(tid)

            try:
                plugin_prefs['mini_toolbar_order'] = new_order
                plugin_prefs['mini_toolbar_hidden'] = new_hidden
            except Exception:
                pass

            try:
                self._apply_mini_toolbar_customization()
            except Exception:
                pass
        except Exception:
            pass

    def _reset_mini_toolbar_customization_and_close(self, dialog):
        try:
            try:
                plugin_prefs['mini_toolbar_order'] = []
                plugin_prefs['mini_toolbar_hidden'] = []
            except Exception:
                pass
            try:
                self._apply_mini_toolbar_customization()
            except Exception:
                pass
            try:
                dialog.accept()
            except Exception:
                pass
        except Exception:
            pass

    def _reset_toolbar_customization_and_close(self, dialog):
        try:
            try:
                plugin_prefs['toolbar_order'] = []
                plugin_prefs['toolbar_hidden'] = []
                plugin_prefs['toolbar_compact'] = False
            except Exception:
                pass
            try:
                self._apply_toolbar_customization()
            except Exception:
                pass
            try:
                dialog.accept()
            except Exception:
                pass
        except Exception:
            pass

    def setup_ui(self):
        _t0 = time.perf_counter()
        _debug('setup_ui start')

        # Used by AIPanelMixin._install_zoom_support for timing/debug.
        try:
            self._setup_ui_t0 = _t0
        except Exception:
            pass

        # Active feed-tag filter (dropdown). Initialize early so signal handlers are safe.
        if not hasattr(self, '_active_feed_tag'):
            self._active_feed_tag = None

        # Create central widget with toolbar and main content
        central_widget = QWidget(self)
        central_layout = QVBoxLayout(central_widget)
        central_layout.setContentsMargins(0, 0, 0, 0)

        self._zoom_steps = 0

        # Toolbar (horizontally scrollable so narrow windows remain usable)
        toolbar_scroll = QScrollArea(central_widget)
        try:
            toolbar_scroll.setWidgetResizable(True)
        except Exception:
            pass
        try:
            toolbar_scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAsNeeded)
            toolbar_scroll.setVerticalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        except Exception:
            try:
                toolbar_scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
                toolbar_scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            except Exception:
                pass
        try:
            toolbar_scroll.setFrameShape(0)
        except Exception:
            try:
                toolbar_scroll.setFrameStyle(0)
            except Exception:
                pass
        try:
            toolbar_scroll.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
        except Exception:
            try:
                toolbar_scroll.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
            except Exception:
                pass

        toolbar_widget = QWidget(toolbar_scroll)
        bar = QHBoxLayout(toolbar_widget)
        try:
            self._toolbar_widget = toolbar_widget
            self._toolbar_layout = bar
        except Exception:
            pass
        try:
            bar.setContentsMargins(0, 0, 0, 0)
            bar.setSpacing(0)
        except Exception:
            pass

        self.add_btn = QToolButton(central_widget)
        self.add_btn.setText(_('Add'))
        try:
            ic = QIcon.ic('plus.png')
            if ic is not None and not ic.isNull():
                self.add_btn.setIcon(ic)
        except Exception:
            pass
        self.add_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        self.add_btn.setPopupMode(QToolButton.ToolButtonPopupMode.MenuButtonPopup)
        add_menu = QMenu(self.add_btn)
        act_add_single = QAction(_('Add single feed…'), add_menu)
        act_add_single.triggered.connect(self.add_feed)
        add_menu.addAction(act_add_single)
        act_add_multiple = QAction(_('Add multiple feeds…'), add_menu)
        act_add_multiple.triggered.connect(self.add_multiple_feeds)
        add_menu.addAction(act_add_multiple)
        add_menu.addSeparator()
        act_add_sample = QAction(_('Add featured feeds…'), add_menu)
        act_add_sample.triggered.connect(self.add_sample_feeds)
        add_menu.addAction(act_add_sample)
        self.add_btn.setMenu(add_menu)
        try:
            self.add_btn.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            self.add_btn.customContextMenuRequested.connect(lambda p: (self.add_btn.menu() or QMenu(self.add_btn)).exec(self.add_btn.mapToGlobal(p)))
            self.add_btn.clicked.connect(lambda: add_menu.actions()[0].trigger() if add_menu.actions() else None)
        except Exception:
            pass
        bar.addWidget(self.add_btn)

        # OPML Export dropdown
        self.export_btn = QToolButton(central_widget)
        try:
            self.export_btn.setText(_('Export'))
        except Exception:
            pass
        try:
            self.export_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.export_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        try:
            self.export_btn.setPopupMode(QToolButton.ToolButtonPopupMode.MenuButtonPopup)
        except Exception:
            try:
                self.export_btn.setPopupMode(QToolButton.MenuButtonPopup)
            except Exception:
                pass
        # OPML Import dropdown
        self.import_btn = QToolButton(central_widget)
        try:
            self.import_btn.setText(_('Import'))
        except Exception:
            pass
        try:
            self.import_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.import_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        try:
            self.import_btn.setPopupMode(QToolButton.ToolButtonPopupMode.MenuButtonPopup)
        except Exception:
            try:
                self.import_btn.setPopupMode(QToolButton.MenuButtonPopup)
            except Exception:
                pass
        import_menu = QMenu(self.import_btn)
        act_import_recipe = QAction(_('Import feeds from recipes…'), import_menu)
        act_import_recipe.triggered.connect(self.import_from_recipes)
        import_menu.addAction(act_import_recipe)
        act_import_simple = QAction(_('Import OPML'), import_menu)
        act_import_simple.triggered.connect(self.import_opml)
        act_import_adv = QAction(_('Advanced import…'), import_menu)
        act_import_adv.triggered.connect(self.import_opml_advanced)
        import_menu.addAction(act_import_simple)
        import_menu.addAction(act_import_adv)
        try:
            self.import_btn.setMenu(import_menu)
        except Exception:
            self.import_btn.setMenu(import_menu)
        try:
            ic = QIcon.ic('mimetypes/opml.png')
            if ic is not None and not ic.isNull():
                self.import_btn.setIcon(ic)
        except Exception:
            pass
        self.import_btn.setToolTip(_('Import: recipes, OPML, and advanced OPML import.'))
        try:
            self.import_btn.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            self.import_btn.customContextMenuRequested.connect(lambda p: (self.import_btn.menu() or QMenu(self.import_btn)).exec(self.import_btn.mapToGlobal(p)))
            self.import_btn.clicked.connect(self._on_import_clicked)
        except Exception:
            pass
        bar.addWidget(self.import_btn)

        export_menu = QMenu(self.export_btn)
        act_export_simple = QAction(_('Export OPML'), export_menu)
        act_export_simple.triggered.connect(self.export_opml)
        act_export_adv = QAction(_('Advanced export…'), export_menu)
        act_export_adv.triggered.connect(self.export_opml_advanced)
        export_menu.addAction(act_export_simple)
        export_menu.addAction(act_export_adv)
        try:
            self.export_btn.setMenu(export_menu)
            self.export_btn.clicked.connect(self._on_export_clicked)
        except Exception:
            self.export_btn.clicked.connect(self._on_export_clicked)
            self.export_btn.setMenu(export_menu)
        try:
            ic = QIcon.ic('mimetypes/opml.png')
            if ic is not None and not ic.isNull():
                self.export_btn.setIcon(ic)
        except Exception:
            pass
        self.export_btn.setToolTip(_('Export your configured feeds as an OPML file. Use the arrow for Advanced export.'))
        try:
            self.export_btn.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            self.export_btn.customContextMenuRequested.connect(lambda p: (self.export_btn.menu() or QMenu(self.export_btn)).exec(self.export_btn.mapToGlobal(p)))
        except Exception:
            pass
        bar.addWidget(self.export_btn)

        self.remove_btn = QToolButton(toolbar_widget)
        self.remove_btn.setText(_('Remove'))
        try:
            ic = QIcon.ic('minus.png')
            if ic is not None and not ic.isNull():
                self.remove_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.remove_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.remove_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.remove_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        self.remove_btn.setToolTip(_('Remove the selected feed or folder'))
        self.remove_btn.clicked.connect(self.remove_selected_feed)
        bar.addWidget(self.remove_btn)

        self.edit_btn = QToolButton(toolbar_widget)
        self.edit_btn.setText(_('Edit'))
        try:
            ic = QIcon.ic('edit_input.png')
            if ic is not None and not ic.isNull():
                self.edit_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.edit_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.edit_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.edit_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        self.edit_btn.setToolTip(_('Edit the selected feed configuration'))
        self.edit_btn.clicked.connect(self.edit_selected_feed_or_folder)
        bar.addWidget(self.edit_btn)

        self.mark_read_btn = QToolButton(toolbar_widget)
        self.mark_read_btn.setText(_('Mark feed read'))
        self.mark_read_btn.setToolTip(_('Mark all items in the selected feed as read'))
        try:
            ic = QIcon.ic('ok.png')
            if ic is not None and not ic.isNull():
                self.mark_read_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.mark_read_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.mark_read_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.mark_read_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        self.mark_read_btn.clicked.connect(self.mark_selected_feed_read)

        # Layout switcher button
        self.layout_btn = QToolButton(central_widget)
        try:
            self.layout_btn.setText(_('Layout'))
        except Exception:
            pass
        try:
            ic = QIcon.ic('layout.png')
            if ic is not None and not ic.isNull():
                self.layout_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.layout_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.layout_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        layout_menu = QMenu(self.layout_btn)

        act_layout_articles_top = QAction(_('Articles on top (default)'), layout_menu)
        act_layout_articles_top.triggered.connect(lambda: self.switch_layout('articles_top'))
        layout_menu.addAction(act_layout_articles_top)

        act_layout_preview_top = QAction(_('Preview on top'), layout_menu)
        act_layout_preview_top.triggered.connect(lambda: self.switch_layout('preview_top'))
        layout_menu.addAction(act_layout_preview_top)

        act_layout_articles_left = QAction(_('Articles on left'), layout_menu)
        act_layout_articles_left.triggered.connect(lambda: self.switch_layout('articles_left'))
        layout_menu.addAction(act_layout_articles_left)

        act_layout_preview_left = QAction(_('Preview on left'), layout_menu)
        act_layout_preview_left.triggered.connect(lambda: self.switch_layout('preview_left'))
        layout_menu.addAction(act_layout_preview_left)

        layout_menu.addSeparator()

        act_ai_show = QAction(_('Show AI Panel (Docked)'), layout_menu)
        act_ai_show.triggered.connect(lambda: self.switch_ai_panel_mode('docked_right'))
        layout_menu.addAction(act_ai_show)

        act_ai_beside = QAction(_('Show AI Panel (Next to Preview)'), layout_menu)
        act_ai_beside.triggered.connect(lambda: self.switch_ai_panel_mode('beside_preview'))
        layout_menu.addAction(act_ai_beside)

        act_ai_hidden = QAction(_('Hide AI Panel'), layout_menu)
        act_ai_hidden.triggered.connect(lambda: self.switch_ai_panel_mode('hidden'))
        layout_menu.addAction(act_ai_hidden)

        try:
            self.layout_btn.setMenu(layout_menu)
            self.layout_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
        except Exception:
            try:
                self.layout_btn.setPopupMode(QToolButton.InstantPopup)
            except Exception:
                pass
            self.layout_btn.setMenu(layout_menu)

        self.layout_btn.setToolTip(_('Switch between article/preview layouts'))
        bar.addWidget(self.layout_btn)
        # Customize toolbar is available via right-click context menu on the toolbar.
        # Do not show a dedicated button by default.
        self.customize_toolbar_btn = None

        self.settings_btn = QToolButton(toolbar_widget)
        self.settings_btn.setText(_('Config'))
        try:
            ic = QIcon.ic('config.png')
            if ic is not None and not ic.isNull():
                self.settings_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.settings_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.settings_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.settings_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        self.settings_btn.clicked.connect(self.open_settings)
        bar.addWidget(self.settings_btn)

        # Cleanup button: opens the cleanup dialog. This is wired into the
        # customizable toolbar but defaults to hidden so it doesn't bloat the
        # initial toolbar layout.
        try:
            self.cleanup_btn = QToolButton(toolbar_widget)
            try:
                self.cleanup_btn.setText(_('Clean up'))
            except Exception:
                pass
            try:
                self.cleanup_btn.setAutoRaise(True)
            except Exception:
                pass
            try:
                self.cleanup_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.cleanup_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            try:
                ic = QIcon.ic('edit-clear.png')
                if ic is not None and not ic.isNull():
                    self.cleanup_btn.setIcon(ic)
            except Exception:
                pass
            try:
                self.cleanup_btn.setToolTip(_('Clean up older items for selected feeds or all feeds'))
            except Exception:
                pass
            try:
                self.cleanup_btn.clicked.connect(self.open_cleanup_dialog)
            except Exception:
                pass
            bar.addWidget(self.cleanup_btn)
        except Exception:
            self.cleanup_btn = None

        # Dedicated Profiles button: quick switch between DB profiles
        try:
            self.profiles_btn = QToolButton(central_widget)
            try:
                self.profiles_btn.setText('👤 ' + _('Profiles'))
            except Exception:
                self.profiles_btn.setText(_('Profiles'))
            try:
                self.profiles_btn.setAutoRaise(True)
            except Exception:
                pass
            try:
                self.profiles_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.profiles_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            try:
                ic = QIcon.ic('user.png')
                if ic is not None and not ic.isNull():
                    self.profiles_btn.setIcon(ic)
            except Exception:
                pass
            self.profiles_btn.setToolTip(_('Switch DB profile'))
            try:
                self._profiles_switch_menu = QMenu(self.profiles_btn)
                self._profiles_switch_menu.aboutToShow.connect(lambda: self._rebuild_profiles_menu(self._profiles_switch_menu))
                self.profiles_btn.setMenu(self._profiles_switch_menu)
                # Profiles toolbar menu is populated dynamically by _rebuild_profiles_menu
                try:
                    self.profiles_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
                except Exception:
                    try:
                        self.profiles_btn.setPopupMode(QToolButton.InstantPopup)
                    except Exception:
                        pass
            except Exception:
                self._profiles_switch_menu = None
            bar.addWidget(self.profiles_btn)
        except Exception:
            self.profiles_btn = None
            self._profiles_switch_menu = None

        # Small DB button: allow quick switching of the plugin SQLite database
        try:
            self.db_btn = QToolButton(central_widget)
            try:
                self.db_btn.setText('🗄️ ' + _('DB'))
            except Exception:
                pass
            try:
                self.db_btn.setAutoRaise(True)
            except Exception:
                pass
            try:
                self.db_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.db_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            self.db_btn.setToolTip(_('Switch/Open plugin database'))
            try:
                # Create persistent menu
                self._db_menu = QMenu(self.db_btn)

                # Profiles quick-switch moved to the dedicated Profiles toolbar button


                # (Clear current DB is intentionally not present in DB menu to avoid duplicates)

                # One-off open (does not auto-save as a profile)
                act_open = QAction(_('Open database (one-off)…'), self._db_menu)
                act_open.triggered.connect(self._on_db_open)
                try:
                    act_open.setToolTip(_('Open a database temporarily (one-off). It will not be saved as a profile. Use "Save current DB as profile…" to persist it or "Set Default" in Manage profiles to change the default.'))
                except Exception:
                    pass
                self._db_menu.addAction(act_open)

                # Add existing database as a named profile (common workflow
                # for users that already have a DB file in a folder).
                act_add_existing = QAction(_('Add existing database as profile…'), self._db_menu)
                act_add_existing.triggered.connect(self._add_existing_db_as_profile)
                try:
                    act_add_existing.setToolTip(_('Pick an existing database file from disk and save it as a named profile (without switching to it immediately).'))
                except Exception:
                    pass
                self._db_menu.addAction(act_add_existing)

                act_save_prof = QAction(_('Save current DB as profile…'), self._db_menu)
                act_save_prof.triggered.connect(self._save_current_db_as_profile)
                try:
                    act_save_prof.setToolTip(_('Save the current database path as a named profile. Shortcut is configurable in calibre Preferences → Keyboard shortcuts (group: RSS Reader).'))
                except Exception:
                    pass
                self._db_menu.addAction(act_save_prof)

                act_new = QAction(_('Create blank database…'), self._db_menu)
                act_new.triggered.connect(self._create_blank_database)
                try:
                    act_new.setToolTip(_('Create a new empty database file and switch to it. Shortcut is configurable in calibre Preferences → Keyboard shortcuts (group: RSS Reader).'))
                except Exception:
                    pass
                self._db_menu.addAction(act_new)

                self._db_menu.addSeparator()

                act_back = QAction(_('Switch back'), self._db_menu)
                act_back.triggered.connect(self._switch_back_database)
                try:
                    act_back.setToolTip(_('Switch back to the previous database. Shortcut is configurable in calibre Preferences → Keyboard shortcuts (group: RSS Reader).'))
                except Exception:
                    pass
                self._db_menu.addAction(act_back)

                # Add 'Clear current DB' action (single instance)
                act_clear_db = QAction(_('Clear current DB…'), self._db_menu)
                def _on_clear_current_db():
                    try:
                        try:
                            from qt.core import QMessageBox
                        except Exception:
                            from PyQt5.Qt import QMessageBox
                        try:
                            db_path = str(rss_db.db_path() or '').strip()
                        except Exception:
                            db_path = ''
                        if not db_path or not os.path.exists(db_path):
                            QMessageBox.warning(self, _('File not found'), _('Database file not found: %s') % db_path)
                            return
                        ro = bool(getattr(rss_db, 'DB_READONLY', False))
                        if ro:
                            QMessageBox.warning(self, _('Read-only'), _('This database/profile is configured as read-only or mirror. Switch to writable mode before clearing.'))
                            return
                        msg = (
                            _('This will DELETE ALL feeds, cached items, tags, stars, and history\nfrom the current database.\n\nContinue?')
                            + '\n\n' + _('Database: %s') % db_path
                        )
                        res = QMessageBox.question(self, _('Clear current DB'), msg, QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No, QMessageBox.StandardButton.No)
                        if res != QMessageBox.StandardButton.Yes:
                            return
                        try:
                            rss_db.clear_all_data_at_path(db_path, vacuum_after=True)
                            QMessageBox.information(self, _('Clear current DB'), _('Database cleared.'))
                        except Exception as e:
                            QMessageBox.warning(self, _('Error'), _('Failed to clear database: %s') % str(e))
                            return
                        # Refresh feeds/items if this is the active DB
                        try:
                            if hasattr(self, 'load_feeds'):
                                self.load_feeds()
                        except Exception:
                            pass
                        try:
                            if hasattr(self, 'refresh'):
                                self.refresh()
                        except Exception:
                            pass
                    except Exception:
                        pass
                act_clear_db.triggered.connect(_on_clear_current_db)
                act_clear_db.setToolTip(_('Delete ALL data from the current database (feeds, cache, tags, history). Keeps the file/path.'))
                self._db_menu.addAction(act_clear_db)


                act_show = QAction(_('Show current DB path'), self._db_menu)
                act_show.triggered.connect(self._show_current_db_path)
                try:
                    act_show.setToolTip(_('Show the full path of the currently active database. Shortcut is configurable in calibre Preferences → Keyboard shortcuts (group: RSS Reader).'))
                except Exception:
                    pass
                self._db_menu.addAction(act_show)

                self.db_btn.setMenu(self._db_menu)
                self.db_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
            except Exception:
                try:
                    self.db_btn.setPopupMode(QToolButton.InstantPopup)
                except Exception:
                    pass
            bar.addWidget(self.db_btn)
        except Exception:
            pass

        # Auto-update interval dropdown (⏳) left-aligned next to the DB button for consistency.
        try:
            self.auto_update_btn = QToolButton(toolbar_widget)
            try:
                self.auto_update_btn.setAutoRaise(True)
            except Exception:
                pass
            try:
                self.auto_update_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.auto_update_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            try:
                self.auto_update_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
            except Exception:
                try:
                    self.auto_update_btn.setPopupMode(QToolButton.InstantPopup)
                except Exception:
                    pass
            try:
                self.auto_update_btn.setToolTip(_('Auto-update interval (background updates). 0 disables automatic updates.'))
            except Exception:
                pass

            def _format_minutes_label(m):
                try:
                    m = int(m or 0)
                except Exception:
                    m = 0
                if m <= 0:
                    return _('Off')
                if m == 1440:
                    return _('Daily')
                return _('%d min') % m

            def _set_auto_update_minutes(m):
                try:
                    m = int(m or 0)
                except Exception:
                    m = 0
                try:
                    plugin_prefs['auto_update_minutes'] = int(m)
                except Exception:
                    pass
                try:
                    if getattr(self, 'action', None) is not None and hasattr(self.action, 'apply_settings'):
                        self.action.apply_settings()
                except Exception:
                    pass
                try:
                    if getattr(self, 'action', None) is not None and hasattr(self.action, '_refresh_toolbar_label'):
                        self.action._refresh_toolbar_label()
                except Exception:
                    pass
                try:
                    self.auto_update_btn.setText('⏳ ' + _format_minutes_label(m))
                except Exception:
                    try:
                        self.auto_update_btn.setText(_format_minutes_label(m))
                    except Exception:
                        pass

            def _rebuild_auto_update_menu():
                try:
                    current = int(plugin_prefs.get('auto_update_minutes', 0) or 0)
                except Exception:
                    current = 0

                m = QMenu(self.auto_update_btn)

                options = [0, 1, 5, 10, 15, 30, 60, 120, 240, 1440]
                if int(current or 0) > 0 and int(current or 0) not in options:
                    options = list(options) + [int(current)]

                for minutes in options:
                    label = _format_minutes_label(minutes)
                    if minutes not in (0, 1440) and int(minutes) > 0 and int(minutes) not in (1, 5, 10, 15, 30, 60, 120, 240):
                        try:
                            label = _('Custom (%d min)') % int(minutes)
                        except Exception:
                            label = 'Custom (%d min)' % int(minutes)
                    a = QAction(label, m)
                    try:
                        a.setCheckable(True)
                        a.setChecked(int(minutes or 0) == int(current or 0))
                    except Exception:
                        pass
                    try:
                        a.setData(int(minutes or 0))
                    except Exception:
                        pass

                    def _mk_triggered(mm):
                        return lambda _checked=False: _set_auto_update_minutes(mm)
                    try:
                        a.triggered.connect(_mk_triggered(int(minutes or 0)))
                    except Exception:
                        try:
                            a.triggered.connect(lambda _checked=False, mm=int(minutes or 0): _set_auto_update_minutes(mm))
                        except Exception:
                            pass
                    m.addAction(a)

                self.auto_update_btn.setMenu(m)
                try:
                    self.auto_update_btn.setText('⏳ ' + _format_minutes_label(current))
                except Exception:
                    try:
                        self.auto_update_btn.setText(_format_minutes_label(current))
                    except Exception:
                        pass

            try:
                _rebuild_auto_update_menu()
                try:
                    self.auto_update_btn.menu().aboutToShow.connect(_rebuild_auto_update_menu)
                except Exception:
                    pass
            except Exception:
                pass

            bar.addWidget(self.auto_update_btn)
        except Exception:
            self.auto_update_btn = None

        bar.addStretch(1)

        # Replace image toggle button with a QCheckBox for clarity
        from PyQt5.QtWidgets import QCheckBox
        self.images_checkbox = QCheckBox('🖼️', central_widget)
        self.images_checkbox.setChecked(True)  # Default to checked; sync below
        self.images_checkbox.setToolTip(_('Show images in Preview (does not affect export; export always uses the config dialog setting).'))
        self.images_checkbox.stateChanged.connect(lambda state: self.toggle_images_in_preview(bool(state)))
        try:
            self._sync_images_toggle = lambda: self.images_checkbox.setChecked(self._get_images_enabled_for_preview())
            self._sync_images_toggle()
        except Exception:
            pass

        # Autofit images checkbox (separate from show/hide images)
        self.autofit_images_checkbox = QCheckBox('⤢', central_widget)
        try:
            self.autofit_images_checkbox.setChecked(bool(plugin_prefs.get('autofit_images', True)))
        except Exception:
            self.autofit_images_checkbox.setChecked(True)
        self.autofit_images_checkbox.setToolTip(_('Autofit images to preview width'))
        self.autofit_images_checkbox.stateChanged.connect(lambda state: self.toggle_autofit_in_preview(bool(state)))

        # Zoom buttons

        self.zoom_out_btn = QToolButton(central_widget)
        try:
            ic = get_icon('images/zoom_out')
            if ic is None or ic.isNull():
                ic = QIcon.ic('zoom-out.png')
            if ic is not None and not ic.isNull():
                self.zoom_out_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.zoom_out_btn.setAutoRaise(True)
        except Exception:
            pass
        self.zoom_out_btn.setToolTip(_('Zoom out (Ctrl+-)'))
        self.zoom_out_btn.clicked.connect(self.zoom_out)

        # Add zoom out button
        bar.addWidget(self.zoom_out_btn)

        # Zoom percentage indicator (e.g. 95%)
        try:
            self.zoom_label = QLabel('', central_widget)
            try:
                self.zoom_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            except Exception:
                try:
                    self.zoom_label.setAlignment(Qt.AlignCenter)
                except Exception:
                    pass
            try:
                self.zoom_label.setFixedWidth(36)  # just enough for '100%'
            except Exception:
                pass
            try:
                self.zoom_label.setStyleSheet('color: gray; font-size: 9pt; margin:0px; padding:0px;')
            except Exception:
                pass
            self.zoom_label.setToolTip(_('Current zoom percent'))
            bar.addWidget(self.zoom_label)
        except Exception:
            self.zoom_label = None

        self.zoom_in_btn = QToolButton(central_widget)
        try:
            ic = get_icon('images/zoom_in')
            if ic is None or ic.isNull():
                ic = QIcon.ic('zoom-in.png')
            if ic is not None and not ic.isNull():
                self.zoom_in_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.zoom_in_btn.setAutoRaise(True)
        except Exception:
            pass
        self.zoom_in_btn.setToolTip(_('Zoom in (Ctrl++)'))
        self.zoom_in_btn.clicked.connect(self.zoom_in)
        bar.addWidget(self.zoom_in_btn)

        self.zoom_reset_btn = QToolButton(central_widget)
        try:
            try:
                ic = get_icon('images/zoom-reset')
            except Exception:
                ic = None
            if not ic or ic.isNull():
                try:
                    base = os.path.dirname(__file__)
                    local_path = os.path.join(base, 'images', 'zoom-reset.png')
                    if os.path.exists(local_path):
                        ic = QIcon(local_path)
                except Exception:
                    ic = None
            try:
                self.zoom_out_btn.setStyleSheet('margin:0px;padding:0px;')
            except Exception:
                pass
            if not ic or ic.isNull():
                try:
                    ic = QIcon.ic('restart.png')
                except Exception:
                    ic = None
            if ic is not None and not ic.isNull():
                self.zoom_reset_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.zoom_reset_btn.setAutoRaise(True)
        except Exception:
            pass
        self.zoom_reset_btn.setToolTip(_('Reset zoom (Ctrl+0)'))
        self.zoom_reset_btn.clicked.connect(self.zoom_reset)
        bar.addWidget(self.zoom_reset_btn)

        # Normalize zoom button widths so Zoom Out / Zoom In / Reset look consistent.
        try:
            w = int(getattr(self, 'zoom_in_btn', None).sizeHint().width() or 0)
            if not w or w <= 0:
                w = 24
            try:
                self.zoom_in_btn.setFixedWidth(w)
            except Exception:
                pass
            try:
                self.zoom_out_btn.setFixedWidth(w)
            except Exception:
                pass
            try:
                self.zoom_reset_btn.setFixedWidth(w)
            except Exception:
                pass
        except Exception:
            pass

        # Ensure zoom label is initialized to current value (if mixin applied later,
        # ai_panel_mixin._set_zoom_steps will call _update_zoom_label()).
        try:
            def _init_zoom_label():
                try:
                    if getattr(self, 'zoom_label', None) is None:
                        return
                    try:
                        steps = int(getattr(self, '_zoom_steps', 0) or 0)
                    except Exception:
                        steps = 0
                    # Derive percent from preview base font size when available
                    base = None
                    try:
                        base = (getattr(self, '_zoom_base_points', {}) or {}).get('preview')
                    except Exception:
                        base = None
                    if base is None:
                        try:
                            base = int(getattr(self, 'preview', None).font().pointSize() or 10)
                        except Exception:
                            base = 10
                    cur = max(6, min(40, int(base + steps)))
                    pct = int(round(float(cur) / float(base) * 100.0)) if base else 100
                    try:
                        self.zoom_label.setText('%d%%' % int(pct))
                    except Exception:
                        pass
                except Exception:
                    pass

            try:
                _init_zoom_label()
            except Exception:
                pass
        except Exception:
            pass

        self.update_btn = QToolButton(toolbar_widget)
        self.update_btn.setText(_('Update'))
        try:
            ic = QIcon.ic('view-refresh.png')
            if ic is not None and not ic.isNull():
                self.update_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.update_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.update_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.update_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        self.update_btn.setToolTip(_('Fetch updates for the currently selected feed(s)'))
        self.update_btn.clicked.connect(self.update_selected)
        bar.addWidget(self.update_btn)

        self.update_all_btn = QToolButton(toolbar_widget)
        self.update_all_btn.setText(_('Update all'))
        try:
            ic = QIcon.ic('auto-reload.png')
            if ic is not None and not ic.isNull():
                self.update_all_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.update_all_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.update_all_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.update_all_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        self.update_all_btn.setToolTip(_('Fetch updates for all enabled feeds'))
        self.update_all_btn.clicked.connect(self.update_all)
        bar.addWidget(self.update_all_btn)

        # Optional toolbar button to abort in-progress feed updates.
        self.stop_update_btn = QToolButton(toolbar_widget)
        try:
            _stop_text = _('Stop')
            self.stop_update_btn.setText(_stop_text)
        except Exception:
            pass
        try:
            ic = None
            try:
                ic = QIcon.ic('process-stop.png')
            except Exception:
                ic = None

            # Calibre does not ship a generic "process-stop.png" icon on all
            # versions; fall back to a themed close icon.
            if ic is None or ic.isNull():
                try:
                    from calibre_plugins.rss_reader.common_icons import is_dark_theme
                    close_name = 'close-for-dark-theme.png' if is_dark_theme() else 'close-for-light-theme.png'
                    ic = QIcon.ic(close_name)
                except Exception:
                    ic = None

            if ic is not None and not ic.isNull():
                self.stop_update_btn.setIcon(ic)
            else:
                # Final fallback: add an emoji prefix so the action remains
                # visually distinct even with missing icon theme resources.
                try:
                    self.stop_update_btn.setText('⏹ ' + (_stop_text if '_stop_text' in locals() else _('Stop')))
                except Exception:
                    pass
        except Exception:
            pass
        try:
            self.stop_update_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.stop_update_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.stop_update_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        try:
            self.stop_update_btn.setToolTip(_('Abort any in-progress feed update.'))
        except Exception:
            pass

        def _toolbar_stop_update():
            try:
                if getattr(self, 'action', None) is not None:
                    try:
                        self.action.cancel_update()
                    except Exception:
                        pass
                try:
                    self.status.setText(_('Stopping fetch…'))
                except Exception:
                    pass
            except Exception:
                pass

        try:
            self.stop_update_btn.clicked.connect(_toolbar_stop_update)
        except Exception:
            try:
                self.stop_update_btn.clicked.connect(lambda _checked=False: _toolbar_stop_update())
            except Exception:
                pass
        bar.addWidget(self.stop_update_btn)

        self.export_ebook_btn = QToolButton(toolbar_widget)
        self.export_ebook_btn.setText(_('Export'))
        try:
            ic = QIcon.ic('convert.png')
            if ic is not None and not ic.isNull():
                self.export_ebook_btn.setIcon(ic)
        except Exception:
            pass
        try:
            self.export_ebook_btn.setAutoRaise(True)
        except Exception:
            pass
        try:
            self.export_ebook_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
        except Exception:
            try:
                self.export_ebook_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            except Exception:
                pass
        self.export_ebook_btn.setToolTip(_('Export selected feeds/items to your preferred output format'))
        self.export_ebook_btn.clicked.connect(self.export_selected_feeds)
        # Update export button tooltip dynamically when feed selection changes
        try:
            self._update_export_btn_tooltip()
        except Exception:
            pass

        # Share button (toolbar) similar to QuiteRSS share menu
        try:
            self.share_btn = QToolButton(central_widget)
            try:
                self.share_btn.setText(_('Share'))
            except Exception:
                pass
            try:
                self.share_btn.setAutoRaise(True)
            except Exception:
                pass
            try:
                self.share_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.share_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            try:
                ic = QIcon.ic('share.png')
                if ic is None or ic.isNull():
                    ic = QIcon.ic('mail.png')
                if ic is None or ic.isNull():
                    ic = QIcon.ic('insert-link.png')
                if ic is not None and not ic.isNull():
                    self.share_btn.setIcon(ic)
            except Exception:
                pass
            try:
                self.share_btn.setToolTip(_('Send the selected article as an ebook by email (including Kindle addresses), like calibre.'))
            except Exception:
                pass
            try:
                self._share_menu = QMenu(self.share_btn)
                self._share_menu.aboutToShow.connect(self._rebuild_share_menu)
                self.share_btn.setMenu(self._share_menu)
                try:
                    self.share_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
                except Exception:
                    try:
                        self.share_btn.setPopupMode(QToolButton.InstantPopup)
                    except Exception:
                        pass
            except Exception:
                self._share_menu = None
        except Exception:
            self.share_btn = None
            self._share_menu = None

        # Dedicated email button (quick access to Share via email without opening Share menu)
        try:
            self.email_btn = QToolButton(central_widget)
            self.email_btn.setText(_('Email'))
            try:
                self.email_btn.setAutoRaise(True)
            except Exception:
                pass
            try:
                self.email_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.email_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            try:
                ic = QIcon.ic('gmail_logo.png')
                if ic is None or (hasattr(ic, 'isNull') and ic.isNull()):
                    ic = get_icon('images/gmail_logo.png')
                if ic is None or (hasattr(ic, 'isNull') and ic.isNull()):
                    ic = QIcon.ic('mail.png')
                if ic is not None and not (hasattr(ic, 'isNull') and ic.isNull()):
                    self.email_btn.setIcon(ic)
            except Exception:
                pass
            self.email_btn.setToolTip(_('Send the selected article by email (including Kindle addresses)'))
            self.email_btn.clicked.connect(self._email_selected_article)
        except Exception:
            self.email_btn = None

        # Save article button (single-article export)
        try:
            self.save_article_btn = QToolButton(central_widget)
            self.save_article_btn.setText(_('Save'))
            try:
                self.save_article_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.save_article_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            try:
                ic = QIcon.ic('save.png')
                if ic is None or ic.isNull():
                    ic = QIcon.ic('document-save.png')
                if ic is not None and not ic.isNull():
                    self.save_article_btn.setIcon(ic)
            except Exception:
                pass
            try:
                self.save_article_btn.setAutoRaise(True)
            except Exception:
                pass
            self.save_article_btn.setToolTip(_('Export the selected article'))
            self.save_article_btn.clicked.connect(self.export_selected_items)
        except Exception:
            self.save_article_btn = None

        # Purge is available in Settings now (to keep the toolbar compact)
        self.purge_btn = None

        # Toolbar customization (persisted in preferences)
        try:
            self._toolbar_item_ids = [
                'add', 'import', 'export', 'remove', 'edit', 'layout', 'profiles', 'db',
                '__stretch__',
                'cleanup', 'settings', 'zoom_out', 'zoom_label', 'zoom_in', 'zoom_reset',
                'update', 'update_all', 'stop_update', 'auto_update',
            ]
            self._toolbar_items = {
                'add': getattr(self, 'add_btn', None),
                'import': getattr(self, 'import_btn', None),
                'export': getattr(self, 'export_btn', None),
                'remove': getattr(self, 'remove_btn', None),
                'edit': getattr(self, 'edit_btn', None),
                'layout': getattr(self, 'layout_btn', None),
                'settings': getattr(self, 'settings_btn', None),
                'cleanup': getattr(self, 'cleanup_btn', None),
                'profiles': getattr(self, 'profiles_btn', None),
                'db': getattr(self, 'db_btn', None),
                'zoom_out': getattr(self, 'zoom_out_btn', None),
                'zoom_label': getattr(self, 'zoom_label', None),
                'zoom_in': getattr(self, 'zoom_in_btn', None),
                'zoom_reset': getattr(self, 'zoom_reset_btn', None),
                'update': getattr(self, 'update_btn', None),
                'update_all': getattr(self, 'update_all_btn', None),
                'stop_update': getattr(self, 'stop_update_btn', None),
                'auto_update': getattr(self, 'auto_update_btn', None),
            }
        except Exception:
            self._toolbar_item_ids = []
            self._toolbar_items = {}

        # No explicit QAction wrappers here; keep toolbar widgets as canonical objects.

        try:
            self._install_toolbar_context_menu()
        except Exception:
            pass
        try:
            self._apply_toolbar_customization()
        except Exception:
            pass

        try:
            toolbar_widget.setStyleSheet('QToolButton { padding: 1px 4px; margin: 0px; }')
        except Exception:
            pass
        toolbar_scroll.setWidget(toolbar_widget)
        try:
            toolbar_scroll.setMaximumHeight(48)
        except Exception:
            pass
        central_layout.addWidget(toolbar_scroll)

        # Main content: create splitter for feeds list and items/preview
        splitter = QSplitter(central_widget)
        splitter.setOrientation(Qt.Orientation.Horizontal)
        # Make the feeds pane resizable down to a small width without collapsing.
        try:
            splitter.setChildrenCollapsible(False)
        except Exception:
            pass

        # Left panel: Feeds list (use modular builder)
        from .ui_feeds_panel import build_feeds_panel
        left_panel = build_feeds_panel(self, central_widget)
        splitter.addWidget(left_panel)
        # Ensure feeds are loaded into the tree after panel is built
        try:
            self.load_feeds()
        except Exception as e:
            _debug(f'Error calling load_feeds after building feeds panel: {e}')

        # Right panel: Items table and preview
        right_panel = QWidget(central_widget)
        right_layout = QVBoxLayout(right_panel)
        right_layout.setContentsMargins(0, 0, 0, 0)
        # Items label with count

        self.items_label = QLabel('', right_panel)
        try:
            self.items_label.setStyleSheet('padding-left: 10px;')
        except Exception:
            pass
        right_layout.addWidget(self.items_label)
        self._update_items_label()

        self.filter_input = HistoryDropdown(
            right_panel,
            history_key='items_search_history',
            # NOTE: advanced operators (tag:, img:, words:, not, etc.) are
            # English-only keywords understood by the search engine and must
            # NOT be translated, so the placeholder mixes translated prose
            # with literal operator examples.
            placeholder=_('Search') + ': tag:img  words:>300  img:true  not tag:long ...',
            max_items=30,
            min_entry_length=1,
        )
        try:
            self.filter_input.setStyleSheet('padding-left: 10px;')
        except Exception:
            pass
        try:
            self.filter_input.setClearButtonEnabled(True)
        except Exception:
            pass
        # Debounce heavy filtering so typing doesn't trigger expensive updates
        # immediately. Schedule the actual filter to run after a short delay.
        self.filter_input.editTextChanged.connect(self._schedule_filter_items)
        self.filter_input.editTextChanged.connect(self._update_items_label)
        right_layout.addWidget(self.filter_input)

        # Quick filters (QuiteRSS-like) below the search bar
        try:
            qf_row = QWidget(right_panel)
            qf_lay = QHBoxLayout(qf_row)
            qf_lay.setContentsMargins(8, 0, 8, 0)
            try:
                self._mini_toolbar_widget = qf_row
                self._mini_toolbar_layout = qf_lay
            except Exception:
                pass
            try:
                self._install_mini_toolbar_context_menu()
            except Exception:
                pass
            try:
                qf_lay.setSpacing(6)
            except Exception:
                pass

            self.items_quick_filter_btn = QToolButton(qf_row)
            self.items_quick_filter_btn.setObjectName('items_quick_filter_btn')
            self.items_quick_filter_btn.setText(_('Show All'))
            try:
                self.items_quick_filter_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
            except Exception:
                try:
                    self.items_quick_filter_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                except Exception:
                    pass
            try:
                self.items_quick_filter_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
            except Exception:
                try:
                    self.items_quick_filter_btn.setPopupMode(QToolButton.InstantPopup)
                except Exception:
                    pass

            try:
                from qt.core import QMenu as _QMenu
            except Exception:
                from PyQt5.Qt import QMenu as _QMenu
            m = _QMenu(self.items_quick_filter_btn)

            act_all = m.addAction(_('Show All'))
            act_new = m.addAction(_('Show New'))
            act_unread = m.addAction(_('Show Unread'))
            act_star = m.addAction(_('Show Starred'))
            act_unread_or_star = m.addAction(_('Show Unread or Starred'))
            m.addSeparator()
            act_last_day = m.addAction(_('Show Last Day'))
            act_last_7 = m.addAction(_('Show Last 7 Days'))

            # Make actions checkable and store references for updating the checkmark
            self._filter_actions = {
                '': act_all,
                'new:true': act_new,
                'unread:true': act_unread,
                'star:true': act_star,
                'unread:true or star:true': act_unread_or_star,
                'days:<1': act_last_day,
                'days:<7': act_last_7,
            }
            for act in self._filter_actions.values():
                act.setCheckable(True)
            act_all.setChecked(True)  # Default

            # Try to give the button a filter icon
            filter_icon = None
            try:
                candidate = QIcon.ic('filter.png')
                if not candidate.isNull():
                    filter_icon = candidate
            except Exception:
                filter_icon = None
            if filter_icon is None or filter_icon.isNull():
                try:
                    candidate = QIcon.fromTheme('view-filter')
                    if candidate and not candidate.isNull():
                        filter_icon = candidate
                except Exception:
                    pass
            if filter_icon and not filter_icon.isNull():
                try:
                    self.items_quick_filter_btn.setIcon(filter_icon)
                except Exception:
                    pass

            act_all.triggered.connect(lambda: self._set_items_quick_filter(_('Show All'), ''))
            act_new.triggered.connect(lambda: self._set_items_quick_filter(_('Show New'), 'new:true'))
            act_unread.triggered.connect(lambda: self._set_items_quick_filter(_('Show Unread'), 'unread:true'))
            act_star.triggered.connect(lambda: self._set_items_quick_filter(_('Show Starred'), 'star:true'))
            act_unread_or_star.triggered.connect(lambda: self._set_items_quick_filter(_('Show Unread or Starred'), 'unread:true or star:true'))
            act_last_day.triggered.connect(lambda: self._set_items_quick_filter(_('Show Last Day'), 'days:<1'))
            act_last_7.triggered.connect(lambda: self._set_items_quick_filter(_('Show Last 7 Days'), 'days:<7'))

            self.items_quick_filter_btn.setMenu(m)
            qf_lay.addWidget(self.items_quick_filter_btn)

            self.items_quick_filter_clear_btn = QToolButton(qf_row)
            self.items_quick_filter_clear_btn.setText('×')
            self.items_quick_filter_clear_btn.setToolTip(_('Clear quick filter'))
            self.items_quick_filter_clear_btn.clicked.connect(self._clear_items_quick_filter)
            qf_lay.addWidget(self.items_quick_filter_clear_btn)

            # Quick tag menu (QuiteRSS-like)
            try:
                self.items_quick_tags_btn = QToolButton(qf_row)
                self.items_quick_tags_btn.setObjectName('items_quick_tags_btn')
                self.items_quick_tags_btn.setText(_('Tags'))
                try:
                    self.items_quick_tags_btn.setToolButtonStyle(Qt.ToolButtonStyle.ToolButtonTextBesideIcon)
                except Exception:
                    try:
                        self.items_quick_tags_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
                    except Exception:
                        pass
                try:
                    tags_icon = QIcon.ic('tags.png')
                    if not tags_icon.isNull():
                        self.items_quick_tags_btn.setIcon(tags_icon)
                except Exception:
                    pass
                try:
                    self.items_quick_tags_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
                except Exception:
                    try:
                        self.items_quick_tags_btn.setPopupMode(QToolButton.InstantPopup)
                    except Exception:
                        pass
                try:
                    from qt.core import QMenu as _QMenu
                except Exception:
                    from PyQt5.Qt import QMenu as _QMenu
                tm = _QMenu(self.items_quick_tags_btn)
                # Defaults inspired by QuiteRSS labels; user can override via prefs.
                try:
                    tags = list(plugin_prefs.get('quick_item_tags', []) or [])
                except Exception:
                    tags = []
                if not tags:
                    tags = [_('Important'), _('Work'), _('Personal'), _('To Do'), _('Later'), _('Amusingly')]
                for t in tags[:20]:
                    tt = str(t or '').strip()
                    if not tt:
                        continue
                    tm.addAction(tt).triggered.connect(lambda _=None, _t=tt: self._apply_quick_tag_to_selected_items(_t))
                tm.addSeparator()
                tm.addAction(_('Edit tags…')).triggered.connect(self.edit_tags_for_selected_items)
                tm.addAction(_('Clear tags')).triggered.connect(self.clear_tags_for_selected_items)
                self.items_quick_tags_btn.setMenu(tm)
                qf_lay.addWidget(self.items_quick_tags_btn)
            except Exception:
                self.items_quick_tags_btn = None

            try:
                qf_lay.addWidget(self.mark_read_btn)
                qf_lay.addWidget(self.export_ebook_btn)
                qf_lay.addWidget(self.save_article_btn)
                qf_lay.addWidget(self.images_checkbox)
                qf_lay.addWidget(self.autofit_images_checkbox)
                qf_lay.addWidget(self.share_btn)
                if getattr(self, 'email_btn', None) is not None:
                    qf_lay.addWidget(self.email_btn)
            except Exception:
                pass

            try:
                self._mini_toolbar_item_ids = [
                    'items_quick_filter', 'items_quick_filter_clear', 'items_quick_tags',
                    'mark_read', 'export_ebook', 'save_article', 'images', 'autofit_images', 'share', 'email',
                    '__stretch__',
                ]
                self._mini_toolbar_items = {
                    'items_quick_filter': getattr(self, 'items_quick_filter_btn', None),
                    'items_quick_filter_clear': getattr(self, 'items_quick_filter_clear_btn', None),
                    'items_quick_tags': getattr(self, 'items_quick_tags_btn', None),
                    'mark_read': getattr(self, 'mark_read_btn', None),
                    'export_ebook': getattr(self, 'export_ebook_btn', None),
                    'save_article': getattr(self, 'save_article_btn', None),
                    'images': getattr(self, 'images_checkbox', None),
                    'autofit_images': getattr(self, 'autofit_images_checkbox', None),
                    'share': getattr(self, 'share_btn', None),
                    'email': getattr(self, 'email_btn', None),
                }
            except Exception:
                self._mini_toolbar_item_ids = []
                self._mini_toolbar_items = {}

            try:
                self._apply_mini_toolbar_customization()
            except Exception:
                pass
            right_layout.addWidget(qf_row)
        except Exception:
            self.items_quick_filter_btn = None
            self.items_quick_filter_clear_btn = None

        # Manage Columns button removed to save screen space; use header context menu instead.

        self.items_table = QTableWidget(right_panel)
        self.items_table.setColumnCount(5)
        self.items_table.setHorizontalHeaderLabels([_('★'), _('Title'), _('Author'), _('Published'), _('Tags')])
        self.items_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.items_table.setSelectionMode(QAbstractItemView.SelectionMode.ExtendedSelection)
        self.items_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)
        # Allow narrowing the items pane (especially in horizontal layouts) without
        # the header's content-based sizing forcing a large minimum width.
        try:
            self.items_table.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAsNeeded)
        except Exception:
            try:
                self.items_table.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
            except Exception:
                pass
        try:
            self.items_table.setSortingEnabled(True)
            self.items_table.horizontalHeader().sortIndicatorChanged.connect(self.on_sort_changed)
            ith = self.items_table.horizontalHeader()
            try:
                ith.setSectionResizeMode(0, ith.ResizeMode.Fixed)
                # Use Interactive resize mode for title/author/published so persisted widths apply
                ith.setSectionResizeMode(1, ith.ResizeMode.Interactive)
                ith.setSectionResizeMode(2, ith.ResizeMode.Interactive)
                ith.setSectionResizeMode(3, ith.ResizeMode.Interactive)
                # Tags can get very wide; keep it interactive so the pane can shrink.
                ith.setSectionResizeMode(4, ith.ResizeMode.Interactive)
            except Exception:
                try:
                    ith.setSectionResizeMode(0, ith.Fixed)
                    ith.setSectionResizeMode(1, ith.Stretch)
                    ith.setSectionResizeMode(2, ith.ResizeToContents)
                    ith.setSectionResizeMode(3, ith.ResizeToContents)
                    ith.setSectionResizeMode(4, ith.Interactive)
                except Exception:
                    pass
            try:
                self.items_table.setColumnWidth(0, 30)
            except Exception:
                pass
            try:
                ith.sectionResized.connect(self.on_items_column_resized)
            except Exception:
                pass
            try:
                # Allow users to reorder columns and manage visibility via header menu
                ith.setSectionsMovable(True)
                try:
                    ith.sectionMoved.connect(self.on_items_columns_moved)
                except Exception:
                    pass
                try:
                    ith.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
                    ith.customContextMenuRequested.connect(self.show_items_header_context_menu)
                except Exception:
                    pass
            except Exception:
                pass
        except Exception:
            pass

        try:
            # Apply any saved column visibility/order state
            try:
                self.setup_items_table_columns()
            except Exception:
                pass
        except Exception:
            pass

        # Make it possible to shrink the items pane aggressively in horizontal layouts.
        try:
            self.items_table.setMinimumWidth(0)
            self.items_table.setMinimumHeight(0)
        except Exception:
            pass
        try:
            self.items_table.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        except Exception:
            try:
                self.items_table.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
            except Exception:
                pass
        self.items_table.itemSelectionChanged.connect(self.on_item_selected)
        self.items_table.itemSelectionChanged.connect(self._update_items_label)
        self.items_table.itemSelectionChanged.connect(self.update_toolbar_button_states)
        try:
            # QuiteRSS-like behavior: clicking an item marks it as read.
            self.items_table.itemClicked.connect(self._on_items_table_item_clicked)
        except Exception:
            pass
        self.items_table.itemDoubleClicked.connect(self.open_selected_item)
        self.items_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.items_table.customContextMenuRequested.connect(self.on_items_context_menu)
        try:
            if not hasattr(self, '_autoscroll_filter'):
                self._autoscroll_filter = CtrlMiddleAutoscroll(self)
            self.items_table.installEventFilter(self._autoscroll_filter)
        except Exception:
            pass

        try:
            self._delete_items_action = QAction(_('Delete item(s)'), self.items_table)
            try:
                self._delete_items_action.setShortcut('Del')
            except Exception:
                pass
            try:
                self._delete_items_action.setShortcutContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
            except Exception:
                try:
                    self._delete_items_action.setShortcutContext(Qt.WidgetWithChildrenShortcut)
                except Exception:
                    pass
            self._delete_items_action.triggered.connect(self.delete_selected_items)
            self.items_table.addAction(self._delete_items_action)
        except Exception:
            self._delete_items_action = None

        # Vertical splitter: items on top, preview on bottom
        right_splitter = QSplitter(Qt.Orientation.Vertical, right_panel)
        right_splitter.addWidget(self.items_table)

        preview_container = QWidget(right_panel)
        try:
            preview_container.setMinimumWidth(0)
            preview_container.setMinimumHeight(0)
        except Exception:
            pass
        try:
            preview_container.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        except Exception:
            try:
                preview_container.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
            except Exception:
                pass
        preview_layout = QVBoxLayout(preview_container)
        preview_layout.setContentsMargins(0, 0, 0, 0)
        # Current feed context label above preview (center-aligned)
        self.current_feed_label = QLabel('', preview_container)
        try:
            self.current_feed_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        except Exception:
            try:
                self.current_feed_label.setAlignment(Qt.AlignCenter)
            except Exception:
                pass
        try:
            self.current_feed_label.setMinimumWidth(0)
        except Exception:
            pass
        try:
            self.current_feed_label.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Fixed)
        except Exception:
            try:
                self.current_feed_label.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Fixed)
            except Exception:
                pass
        preview_layout.addWidget(self.current_feed_label)

        # Compact current-article info row (title/author/published/received)
        try:
            self.current_article_label = QLabel('', preview_container)
            try:
                self.current_article_label.setStyleSheet('color: gray; font-size: 8pt;')
            except Exception:
                pass
            try:
                self.current_article_label.setWordWrap(False)
            except Exception:
                pass
            try:
                self.current_article_label.setMinimumWidth(0)
            except Exception:
                pass
            try:
                self.current_article_label.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Fixed)
            except Exception:
                try:
                    self.current_article_label.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Fixed)
                except Exception:
                    pass
            try:
                self.current_article_label.setText('')
            except Exception:
                pass
            preview_layout.addWidget(self.current_article_label)
        except Exception:
            self.current_article_label = None
        self._update_current_feed_label()
        preview_layout.addWidget(QLabel(_('Preview'), preview_container))

        # Horizontal splitter: preview and AI panel
        preview_row = QSplitter(Qt.Orientation.Horizontal, preview_container)
        try:
            preview_row.setChildrenCollapsible(True)
        except Exception:
            pass

        self.preview = PreviewBrowser(preview_container)
        try:
            self.preview.setMinimumWidth(0)
            self.preview.setMinimumHeight(0)
        except Exception:
            pass
        try:
            self.preview.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        except Exception:
            try:
                self.preview.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
            except Exception:
                pass
        try:
            self.preview.setOpenExternalLinks(False)
        except Exception:
            pass
        try:
            self.preview.setOpenLinks(False)
        except Exception:
            pass
        try:
            self.preview.anchorClicked.connect(self._on_preview_anchor_clicked)
        except Exception:
            pass
        try:
            self.preview.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            self.preview.customContextMenuRequested.connect(self.on_preview_context_menu)
        except Exception:
            pass
        preview_row.addWidget(self.preview)

        # AI Panel (extracted)
        self.setup_ai_panel()

        # Encourage the preview/AI splitter to be fully draggable.
        try:
            if getattr(self, 'preview_splitter', None) is not None:
                self.preview_splitter.setChildrenCollapsible(True)
        except Exception:
            pass
        try:
            if getattr(self, 'ai_panel', None) is not None:
                preview_row.setStretchFactor(preview_row.indexOf(self.preview), 3)
                preview_row.setStretchFactor(preview_row.indexOf(self.ai_panel), 1)
        except Exception:
            pass

        preview_layout.addWidget(preview_row, 1)

        # Audio player for enclosures (hidden by default)
        try:
            self.preview_audio_player = AudioPlayer(preview_container)
            try:
                self.preview_audio_player.hide()
            except Exception:
                pass
            try:
                preview_layout.addWidget(self.preview_audio_player)
            except Exception:
                pass
        except Exception:
            self.preview_audio_player = None

        # Preview footer: stats
        try:
            stats_row = QHBoxLayout()
            stats_row.setContentsMargins(6, 4, 6, 4)
            self.preview_url_label = QLabel('', preview_container)
            try:
                self.preview_url_label.setStyleSheet('color: gray; font-size: 9pt;')
            except Exception:
                pass
            try:
                self.preview_url_label.setWordWrap(False)
            except Exception:
                pass
            try:
                self.preview_url_label.setMinimumWidth(0)
            except Exception:
                pass
            try:
                self.preview_url_label.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Fixed)
            except Exception:
                try:
                    self.preview_url_label.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Fixed)
                except Exception:
                    pass
            try:
                self.preview_url_label.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
            except Exception:
                try:
                    self.preview_url_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
                except Exception:
                    pass
            try:
                self.preview_url_label.setOpenExternalLinks(False)
            except Exception:
                pass
            try:
                self.preview_url_label.linkActivated.connect(lambda u: self._open_url(u))
            except Exception:
                pass
            stats_row.addWidget(self.preview_url_label, 1)
            self.preview_stats = QLabel('', preview_container)
            try:
                self.preview_stats.setStyleSheet('color: gray; font-size: 9pt;')
                try:
                    self.preview_stats.setAlignment(Qt.AlignmentFlag.AlignRight)
                except Exception:
                    try:
                        self.preview_stats.setAlignment(Qt.AlignRight)
                    except Exception:
                        pass
            except Exception:
                pass
            try:
                self.preview_stats.setMinimumWidth(80)
            except Exception:
                pass
            try:
                self.preview_stats.setSizePolicy(QSizePolicy.Policy.Preferred, QSizePolicy.Policy.Fixed)
            except Exception:
                try:
                    self.preview_stats.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
                except Exception:
                    pass
            try:
                stats_row.addWidget(self.preview_stats, 0)
                preview_layout.addLayout(stats_row)
            except Exception:
                preview_layout.addWidget(self.preview_stats)
        except Exception:
            self.preview_stats = None
            self.preview_url_label = None

        self.preview_splitter = preview_row

        # If there is no persisted preview splitter sizes, ensure a minimal AI panel width
        try:
            s = gprefs.get('rss_reader_preview_splitter', None)
            if (not s or not isinstance(s, (list, tuple))) and getattr(self, 'preview_splitter', None) is not None:
                try:
                    ai_min = int(plugin_prefs.get('preview_ai_min_width', 120) or 120)
                except Exception:
                    ai_min = 120
                try:
                    total_w = int(self.preview_splitter.size().width() or self.width() or 1250)
                except Exception:
                    total_w = 1250
                try:
                    preview_w = max(460, int(total_w) - int(ai_min))
                    ai_w = max(int(ai_min), int(total_w) - int(preview_w))
                    self.preview_splitter.setSizes([preview_w, ai_w])
                    gprefs['rss_reader_preview_splitter'] = [preview_w, ai_w]
                except Exception:
                    pass
        except Exception:
            pass

        right_splitter.addWidget(preview_container)

        # Allow extreme resizing/collapsing in both orientations.
        try:
            right_splitter.setChildrenCollapsible(True)
            right_splitter.setCollapsible(0, True)
            right_splitter.setCollapsible(1, True)
        except Exception:
            pass
        # Ensure the splitter always takes the remaining vertical space.
        # Without a stretch factor, horizontal orientation can cause the layout
        # to allocate only the splitter's (small) sizeHint height, leaving a large
        # empty area above the items/preview panes.
        right_layout.addWidget(right_splitter, 1)

        self.main_splitter = splitter
        self.right_splitter = right_splitter
        # Persist splitter sizes as the user drags them (so manual adjustments are remembered).
        try:
            def _persist_main_splitter(_pos=None, _idx=None):
                try:
                    if getattr(self, 'main_splitter', None) is not None:
                        gprefs['rss_reader_main_splitter'] = list(self.main_splitter.sizes() or [])
                except Exception:
                    pass

            def _persist_right_splitter(_pos=None, _idx=None):
                try:
                    rs = getattr(self, 'right_splitter', None)
                    if rs is None:
                        return
                    try:
                        layout_type = str(gprefs.get('rss_reader_layout_type', 'articles_top') or 'articles_top')
                    except Exception:
                        layout_type = 'articles_top'
                    sizes = list(rs.sizes() or [])
                    if sizes and len(sizes) >= 2:
                        gprefs['rss_reader_right_splitter'] = sizes[:2]  # legacy
                        gprefs['rss_reader_right_splitter_%s' % layout_type] = sizes[:2]
                except Exception:
                    pass

            def _persist_preview_splitter(_pos=None, _idx=None):
                try:
                    ps = getattr(self, 'preview_splitter', None)
                    if ps is None:
                        return
                    # Only persist when AI panel is actually beside preview.
                    try:
                        if getattr(self, 'ai_panel', None) is None:
                            return
                        if ps.indexOf(self.ai_panel) < 0:
                            return
                    except Exception:
                        return
                    sizes = list(ps.sizes() or [])
                    if sizes and len(sizes) >= 2:
                        gprefs['rss_reader_preview_splitter'] = sizes[:2]
                except Exception:
                    pass

            try:
                self.main_splitter.splitterMoved.connect(_persist_main_splitter)
            except Exception:
                pass
            try:
                self.right_splitter.splitterMoved.connect(_persist_right_splitter)
            except Exception:
                pass
            try:
                self.preview_splitter.splitterMoved.connect(_persist_preview_splitter)
            except Exception:
                pass
        except Exception:
            pass

        splitter.addWidget(right_panel)
        splitter.setStretchFactor(1, 2)
        try:
            splitter.setSizes([280, 900])
        except Exception:
            pass

        central_layout.addWidget(splitter, 1)



        # Footer: status line
        footer = QHBoxLayout()
        # Profile indicator (profile_label) - leftmost
        try:
            self.profile_label = QLabel('', central_widget)
            try:
                self.profile_label.setStyleSheet('QLabel { font-size: 9pt; padding: 4px; }')
            except Exception:
                pass
            footer.addWidget(self.profile_label, 0, Qt.AlignmentFlag.AlignLeft)
        except Exception:
            self.profile_label = None

        # Status label (next to profile label)
        self.status = QLabel('', central_widget)
        try:
            self.status.setStyleSheet('QLabel { font-size: 9pt; padding: 4px; }')
        except Exception:
            pass
        footer.addWidget(self.status, 0, Qt.AlignmentFlag.AlignLeft)
        try:
            self._install_status_context_menu()
        except Exception:
            pass

        # Add flexible space so the version label is right-aligned.
        try:
            footer.addStretch(1)
        except Exception:
            pass

        # Version label (right side)
        try:
            version_text = 'RSS Reader v1.0.0'
            calibre_version = ''
            try:
                from calibre_plugins.rss_reader import RSSReaderPlugin
                v = getattr(RSSReaderPlugin, 'version', None)
                if isinstance(v, (tuple, list)) and v:
                    version_text = 'RSS Reader v' + '.'.join(str(x) for x in v)
            except Exception:
                pass
            try:
                from calibre.constants import numeric_version
                if isinstance(numeric_version, (tuple, list)):
                    calibre_version = 'calibre ' + '.'.join(str(x) for x in numeric_version)
                else:
                    calibre_version = 'calibre ' + str(numeric_version)
            except Exception:
                calibre_version = ''
            label_text = version_text
            if calibre_version:
                label_text += '   |   ' + calibre_version
            self.version_label = QLabel(label_text, central_widget)
            try:
                self.version_label.setStyleSheet('QLabel { font-size: 9pt; padding: 4px; }')
            except Exception:
                pass
            footer.addWidget(self.version_label, 0, Qt.AlignmentFlag.AlignRight)
        except Exception:
            pass

        central_layout.addLayout(footer)

        try:
            self._apply_footer_styles()
        except Exception:
            pass

        self.setCentralWidget(central_widget)
        self.resize(1210, 700)

        # Apply scrollbar styling
        try:
            self.setStyleSheet("""
                QScrollBar::handle {
                   border: 1px solid #5B6985;
                }
                QScrollBar:vertical {
                    background: transparent;
                    width: 12px;
                    margin: 0px 0px 0px 0px;
                    padding: 6px 0px 6px 0px;
                }
                QScrollBar::handle:vertical {
                    background: rgba(140, 172, 204, 0.25);
                    min-height: 22px;
                    border-radius: 4px;
                    margin: 4px 0px;
                }
                QScrollBar::handle:vertical:hover {
                    background: rgba(140, 172, 204, 0.45);
                }
                QScrollBar:horizontal {
                    background: transparent;
                    height: 12px;
                    margin: 0px 0px 0px 0px;
                    padding: 0px 6px 0px 6px;
                }
                QScrollBar::handle:horizontal {
                    background: rgba(140, 172, 204, 0.25);
                    min-width: 22px;
                    border-radius: 4px;
                    margin: 0px 4px;
                }
                QScrollBar::handle:horizontal:hover {
                    background: rgba(140, 172, 204, 0.45);
                }
                QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,
                QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {
                    background: none;
                }
            """)
        except Exception:
            pass

        try:
            self.update_ai_ui_state()
        except Exception:
            pass

        try:
            self._install_zoom_support()
        except Exception:
            pass

        # Populate dynamic profile UI elements
        try:
            self._rebuild_profiles_menu()
        except Exception:
            pass
        try:
            self._update_profile_label()
        except Exception:
            pass

        # DO NOT call self.resize() here - it will override restored geometry in restore_state()
        # Geometry is restored separately in restore_state() and during __init__

    def sizeHint(self):
        return QSize(1250, 650)

    def _update_zoom_label(self):
        try:
            zl = getattr(self, 'zoom_label', None)
            if zl is None:
                return
            try:
                steps = int(getattr(self, '_zoom_steps', 0) or 0)
            except Exception:
                steps = 0
            base = None
            try:
                base = (getattr(self, '_zoom_base_points', {}) or {}).get('preview')
            except Exception:
                base = None
            if base is None:
                try:
                    base = int(getattr(self, 'preview', None).font().pointSize() or 10)
                except Exception:
                    base = 10
            cur = max(6, min(40, int(base + steps)))
            try:
                pct = int(round(float(cur) / float(base) * 100.0)) if base else 100
            except Exception:
                pct = 100
            try:
                zl.setText('%d%%' % int(pct))
            except Exception:
                pass
        except Exception:
            pass

    def _update_current_article_label(self, it=None):
        try:
            lab = getattr(self, 'current_article_label', None)
            if lab is None:
                return
            it = it or {}
            if not isinstance(it, dict) or not it:
                try:
                    lab.setText('')
                except Exception:
                    pass
                return

            try:
                import html as _html
            except Exception:
                _html = None

            try:
                title = (_html.unescape(it.get('title') or '') if _html else (it.get('title') or ''))
            except Exception:
                title = it.get('title') or ''
            try:
                author = str(it.get('author') or it.get('creator') or it.get('dc_creator') or '').strip()
            except Exception:
                author = ''
            try:
                pub_ts = int(it.get('published_ts') or iso_to_ts(it.get('published') or '') or 0)
            except Exception:
                pub_ts = 0
            try:
                recv_ts = int(it.get('received_ts') or it.get('fetched_ts') or it.get('downloaded_ts') or 0)
            except Exception:
                recv_ts = 0

            pub_s = ''
            try:
                pub_s = format_published_display(pub_ts, it.get('published') or '') if pub_ts else ''
            except Exception:
                pub_s = ''

            recv_s = ''
            try:
                recv_s = format_published_display(recv_ts, '') if recv_ts else ''
            except Exception:
                try:
                    recv_s = format_published_display(recv_ts, '') if recv_ts else ''
                except Exception:
                    recv_s = ''

            parts = []
            if title:
                parts.append(str(title))
            if author:
                parts.append(str(author))
            if pub_s:
                parts.append(_('Pub: %s') % str(pub_s))
            if recv_s:
                parts.append(_('Recv: %s') % str(recv_s))

            try:
                lab.setText(' | '.join([p for p in parts if p]))
            except Exception:
                pass
        except Exception:
            pass

    def _copy_to_clipboard(self, text, status_msg=None):
        try:
            QApplication.clipboard().setText(str(text or ''))
            if status_msg:
                try:
                    self.status.setText(str(status_msg))
                except Exception:
                    pass
        except Exception:
            pass

    def _email_selected_article(self):
        """Open the Share via email dialog for the currently selected article."""
        try:
            it = self.selected_item() or {}
            if not isinstance(it, dict) or not it:
                try:
                    self.status.setText(_('No article selected'))
                except Exception:
                    pass
                return
            ShareViaEmailDialog(self, it).exec()
        except Exception:
            pass

    def _rebuild_share_menu(self):
        try:
            m = getattr(self, '_share_menu', None)
            if m is None:
                return
            m.clear()
            it = self.selected_item() or {}
            if not isinstance(it, dict) or not it:
                a = QAction(_('No article selected'), m)
                a.setEnabled(False)
                m.addAction(a)
                return

            title = str(it.get('title') or '').strip()
            link = str(it.get('link') or '').strip()

            open_act = QAction(_('Open link'), m)
            copy_link_act = QAction(_('Copy link'), m)
            copy_title_link_act = QAction(_('Copy title + link'), m)
            copy_md_act = QAction(_('Copy as Markdown'), m)

            open_act.triggered.connect(lambda: self._open_url(link) if link else None)
            copy_link_act.triggered.connect(lambda: self._copy_to_clipboard(link, _('Link copied to clipboard')))
            copy_title_link_act.triggered.connect(
                lambda: self._copy_to_clipboard(
                    ('%s\n%s' % (title, link)).strip(),
                    _('Share text copied to clipboard')
                )
            )
            copy_md_act.triggered.connect(
                lambda: self._copy_to_clipboard(
                    ('[%s](%s)' % (title or link, link)).strip() if link else (title or ''),
                    _('Markdown copied to clipboard')
                )
            )

            if not link:
                try:
                    open_act.setEnabled(False)
                    copy_link_act.setEnabled(False)
                    copy_title_link_act.setEnabled(bool(title))
                    copy_md_act.setEnabled(bool(title))
                except Exception:
                    pass

            # Move Open Link to last position in Share menu
            m.addSeparator()
            m.addAction(copy_link_act)
            m.addAction(copy_title_link_act)
            m.addAction(copy_md_act)
            m.addSeparator()
            m.addAction(open_act)

            # Web share targets
            try:
                import urllib.parse as _up
                q_title = _up.quote(title or '')
                q_link = _up.quote(link or '')
            except Exception:
                q_title, q_link = '', ''

            if link:
                m.addSeparator()
                email_act = QAction(_('Share via email'), m)
                x_act = QAction(_('Share to X/Twitter'), m)
                li_act = QAction(_('Share to LinkedIn'), m)
                rd_act = QAction(_('Share to Reddit'), m)

                x_url = 'https://twitter.com/intent/tweet?text=%s&url=%s' % (q_title, q_link)
                li_url = 'https://www.linkedin.com/sharing/share-offsite/?url=%s' % (q_link)
                rd_url = 'https://www.reddit.com/submit?title=%s&url=%s' % (q_title, q_link)

                email_act.triggered.connect(lambda: ShareViaEmailDialog(self, it).exec())
                x_act.triggered.connect(lambda: self._open_url(x_url))
                li_act.triggered.connect(lambda: self._open_url(li_url))
                rd_act.triggered.connect(lambda: self._open_url(rd_url))

                m.addAction(email_act)
                m.addAction(x_act)
                m.addAction(li_act)
                m.addAction(rd_act)
        except Exception:
            pass

    def on_preview_context_menu(self, pos):
        try:
            menu = QMenu(self)

            # Build context menu: Share last, Open Link last, Copy as label
            it = self.selected_item() or {}
            link = ''
            try:
                if isinstance(it, dict):
                    link = str(it.get('link') or '').strip()
            except Exception:
                link = ''

            # Let the preview add image-specific actions (copy/save/block/etc.)
            try:
                try:
                    self.preview.add_rss_reader_context_actions(menu, pos)
                except Exception:
                    try:
                        vp_pos = self.preview.viewport().mapFromGlobal(self.preview.viewport().mapToGlobal(pos))
                        self.preview.add_rss_reader_context_actions(menu, vp_pos)
                    except Exception:
                        pass
            except Exception:
                pass

            # Only add plain Copy if there's a selection in the preview
            try:
                try:
                    has_selection = bool(self.preview.textCursor().hasSelection())
                except Exception:
                    has_selection = bool(self.preview.textCursor().selectedText())
            except Exception:
                has_selection = False

            copy_sel_act = None
            if has_selection:
                copy_sel_act = menu.addAction(_('Copy'))

            # Always offer Select All and Copy HTML source
            select_all_act = menu.addAction(_('Select All'))
            copy_html_act = menu.addAction(_('Copy HTML source'))

            open_link_act = menu.addAction(_('Open link'))
            try:
                if not link:
                    open_link_act.setEnabled(False)
            except Exception:
                pass

            # Share menu last
            share_menu = QMenu(_('Share'), menu)
            try:
                prev = getattr(self, '_share_menu', None)
                self._share_menu = share_menu
                self._rebuild_share_menu()
                self._share_menu = prev
            except Exception:
                pass
            menu.addMenu(share_menu)

            act = menu.exec(self.preview.viewport().mapToGlobal(pos))
            try:
                if act == copy_sel_act and copy_sel_act is not None:
                    try:
                        self.preview.copy()
                    except Exception:
                        pass
                elif act == select_all_act:
                    try:
                        self.preview.selectAll()
                    except Exception:
                        pass
                elif act == copy_html_act:
                    try:
                        html = self.preview.toHtml() or ''
                        self._copy_to_clipboard(html, _('HTML copied to clipboard'))
                    except Exception:
                        pass
                elif act == open_link_act:
                    if link:
                        self._open_url(link)
            except Exception:
                pass
        except Exception:
            pass

    def setup_window_constraints(self):
        """Set up minimum window size and constraints."""
        try:
            self.setMinimumSize(QSize(1270, 680))
        except Exception:
            pass

    def keyPressEvent(self, event):
        """Allow Escape key to close the plugin window like before."""
        try:
            k = event.key()
            try:
                esc = Qt.Key_Escape
            except Exception:
                from PyQt5.Qt import Qt as _Qt
                esc = _Qt.Key_Escape
            if k == esc:
                try:
                    self.close()
                except Exception:
                    pass
                return
        except Exception:
            pass
        try:
            return super().keyPressEvent(event)
        except Exception:
            try:
                return QMainWindow.keyPressEvent(self, event)
            except Exception:
                return

    def closeEvent(self, ev):
        """Save window state on close. Hide tray icon if present."""
        try:
            self.save_state()
        except Exception:
            pass
        try:
            if getattr(self, '_tray_icon', None):
                self._tray_icon.hide()
        except Exception:
            pass
        return QMainWindow.closeEvent(self, ev)

    def changeEvent(self, ev):
        """Handle minimize-to-tray (Windows only) when enabled."""
        try:
            import platform

            if platform.system().lower().startswith('win') and getattr(self, '_tray_icon', None) is not None:
                try:
                    if bool(plugin_prefs.get('minimize_to_tray', False)):
                        try:
                            wsc = getattr(QEvent, 'WindowStateChange', None)
                            is_wsc = (ev.type() == wsc) if wsc is not None else False
                        except Exception:
                            is_wsc = False

                        if is_wsc and self.isMinimized():
                            try:
                                self.hide()
                            except Exception:
                                pass
                            try:
                                self._tray_icon.show()
                            except Exception:
                                pass
                except Exception:
                    pass
        except Exception:
            pass

        try:
            return QMainWindow.changeEvent(self, ev)
        except Exception:
            try:
                return super().changeEvent(ev)
            except Exception:
                return None

    def save_state(self):
        try:
            # --- Window geometry + dock/mainwindow layout ---
            # Save full geometry (preferred) and also a tuple fallback.
            try:
                gprefs['rss_reader_window_geometry'] = bytearray(self.saveGeometry())
            except Exception:
                pass
            try:
                g = self.geometry()
                gprefs['rss_reader_geometry'] = (int(g.x()), int(g.y()), int(g.width()), int(g.height()))
            except Exception:
                pass
            try:
                gprefs['rss_reader_window_state'] = bytearray(self.saveState(1))
            except Exception:
                pass

            # Persist AI panel visibility/mode separately so we can reapply after restoreState().
            try:
                if hasattr(self, 'ai_dock'):
                    gprefs['rss_reader_ai_panel_visible'] = bool(self.ai_dock.isVisible())
            except Exception:
                pass
            try:
                gprefs['rss_reader_ai_panel_mode'] = gprefs.get('rss_reader_ai_panel_mode', 'docked_right')
            except Exception:
                pass

            if getattr(self, 'main_splitter', None) is not None:
                gprefs['rss_reader_main_splitter'] = self.main_splitter.sizes()
            if getattr(self, 'right_splitter', None) is not None:
                rs_sizes = list(self.right_splitter.sizes() or [])
                if rs_sizes and len(rs_sizes) >= 2:
                    rs_sizes = rs_sizes[:2]
                    gprefs['rss_reader_right_splitter'] = rs_sizes  # legacy
                    # Per-layout persistence
                    try:
                        lt = str(gprefs.get('rss_reader_layout_type', 'articles_top') or 'articles_top')
                    except Exception:
                        lt = 'articles_top'
                    try:
                        gprefs['rss_reader_right_splitter_%s' % lt] = rs_sizes
                    except Exception:
                        pass
                # Persist orientation for completeness/debug
                try:
                    gprefs['rss_reader_right_splitter_orientation'] = int(self.right_splitter.orientation())
                except Exception:
                    pass
            if getattr(self, 'preview_splitter', None) is not None:
                try:
                    # Only meaningful when AI panel is beside preview.
                    if getattr(self, 'ai_panel', None) is not None and self.preview_splitter.indexOf(self.ai_panel) >= 0:
                        ps_sizes = list(self.preview_splitter.sizes() or [])
                        if ps_sizes and len(ps_sizes) >= 2:
                            gprefs['rss_reader_preview_splitter'] = ps_sizes[:2]
                except Exception:
                    pass
            # Folder tree expanded state + selection
            try:
                expanded = []
                def walk(item):
                    for i in range(item.childCount()):
                        ch = item.child(i)
                        data = ch.data(ROLE_USER)
                        if isinstance(data, dict) and data.get('type') == 'folder':
                            if ch.isExpanded():
                                expanded.append(str(data.get('path') or ''))
                        walk(ch)
                root = self.feeds_tree.invisibleRootItem()
                walk(root)
                gprefs['rss_reader_expanded_folders'] = [p for p in expanded if p]
            except Exception:
                pass
            try:
                gprefs['rss_reader_last_selected_feed_ids'] = list(self.selected_feed_ids() or [])
                gprefs['rss_reader_last_selected_folder'] = str(self.selected_folder_path() or '')
            except Exception:
                pass
            items_widths = {}
            for col in range(self.items_table.columnCount()):
                items_widths[str(col)] = self.items_table.columnWidth(col)
            gprefs['rss_reader_items_table_widths'] = items_widths
            # Sort
            try:
                header = self.items_table.horizontalHeader()
                gprefs['rss_reader_sort_column'] = header.sortIndicatorSection()
                gprefs['rss_reader_sort_order'] = int(header.sortIndicatorOrder())
            except Exception:
                pass
        except Exception:
            pass

    def restore_state(self):
        """Restore window state from preferences."""
        try:
            # Restore window geometry
            try:
                geom_bytes = gprefs.get('rss_reader_window_geometry')
                if geom_bytes:
                    try:
                        self.restoreGeometry(geom_bytes)
                    except Exception:
                        pass
                else:
                    geom = gprefs.get('rss_reader_geometry')
                    if geom and isinstance(geom, (list, tuple)) and len(geom) == 4:
                        x, y, w, h = geom
                        self.setGeometry(int(x), int(y), max(1200, int(w)), max(600, int(h)))
                    else:
                        # No saved geometry, use default size
                        self.resize(1210, 700)
                try:
                    QApplication.instance().ensure_window_on_screen(self)
                except Exception:
                    pass
            except Exception as e:
                _debug('Failed to restore geometry: %s' % str(e))
                try:
                    QApplication.instance().ensure_window_on_screen(self)
                except Exception:
                    pass

            # Set AI panel visibility BEFORE restoring dock state
            # This way the dock state restoration won't overwrite our visibility setting
            try:
                if hasattr(self, 'ai_dock'):
                    is_visible = gprefs.get('rss_reader_ai_panel_visible', True)
                    # Don't call setVisible yet - wait for dock state to be restored
                    self._ai_visibility_target = is_visible
                    _debug('AI panel visibility target: %s' % is_visible)
            except Exception:
                self._ai_visibility_target = True

            # Restore dock/main window state
            try:
                state_data = gprefs.get('rss_reader_window_state')
                if state_data:
                    self.restoreState(state_data, 1)  # version 1
                    _debug('Restored dock state')
            except Exception as e:
                _debug('Failed to restore dock state: %s' % str(e))
                pass

            # Now apply the AI panel visibility after dock state is restored
            try:
                if hasattr(self, 'ai_dock') and hasattr(self, '_ai_visibility_target'):
                    self.ai_dock.setVisible(self._ai_visibility_target)
                    _debug('AI panel visibility set to: %s' % self._ai_visibility_target)
            except Exception as e:
                _debug('Failed to set AI panel visibility: %s' % str(e))

            # Restore AI panel mode (docked vs beside preview)
            try:
                if hasattr(self, 'ai_dock') and hasattr(self, 'preview_splitter'):
                    mode = gprefs.get('rss_reader_ai_panel_mode', 'beside_preview')
                    if mode in ('docked_right', 'beside_preview', 'hidden'):
                        # Use switch_ai_panel_mode to properly restore the layout
                        self.switch_ai_panel_mode(mode)
                        _debug('Restored AI panel mode: %s' % mode)
            except Exception as e:
                _debug('Failed to restore AI panel mode: %s' % str(e))

            # Restore layout type (includes orientation + per-layout right_splitter sizes)
            try:
                layout_type = gprefs.get('rss_reader_layout_type', 'articles_top')
                if hasattr(self, 'switch_layout'):
                    self.switch_layout(layout_type)
            except Exception:
                pass

            try:
                if getattr(self, 'main_splitter', None) is not None:
                    s = gprefs.get('rss_reader_main_splitter', None)
                    if s:
                        try:
                            self.main_splitter.setSizes(list(s))
                        except Exception:
                            pass
                if getattr(self, 'preview_splitter', None) is not None:
                    s = gprefs.get('rss_reader_preview_splitter', None)
                    if s:
                        try:
                            self.preview_splitter.setSizes(list(s))
                        except Exception:
                            pass
            except Exception:
                pass

            # Restore items table column widths. Temporarily set header
            # sections to Interactive so setColumnWidth takes effect reliably,
            # then restore original modes.
            try:
                iw = gprefs.get('rss_reader_items_table_widths', {}) or {}
            except Exception:
                iw = {}
            if iw:
                try:
                    header = self.items_table.horizontalHeader()
                    modes = {}
                    try:
                        cnt = header.count()
                    except Exception:
                        try:
                            cnt = self.items_table.columnCount()
                        except Exception:
                            cnt = 0
                    for c in range(cnt):
                        try:
                            modes[c] = header.sectionResizeMode(c)
                        except Exception:
                            modes[c] = None
                    # Force Interactive while applying widths
                    for c in range(cnt):
                        try:
                            header.setSectionResizeMode(c, header.ResizeMode.Interactive)
                        except Exception:
                            try:
                                header.setSectionResizeMode(c, header.Interactive)
                            except Exception:
                                pass
                    for col, w in iw.items():
                        try:
                            self.items_table.setColumnWidth(int(col), int(w))
                        except Exception:
                            pass
                    # Restore previous modes
                    for c, m in modes.items():
                        if m is None:
                            continue
                        try:
                            header.setSectionResizeMode(c, m)
                        except Exception:
                            try:
                                # common enum fallbacks
                                if m == getattr(header, 'Fixed', None):
                                    header.setSectionResizeMode(c, getattr(header, 'Fixed', None))
                                elif m == getattr(header, 'Stretch', None):
                                    header.setSectionResizeMode(c, getattr(header, 'Stretch', None))
                                elif m == getattr(header, 'ResizeToContents', None):
                                    header.setSectionResizeMode(c, getattr(header, 'ResizeToContents', None))
                                elif m == getattr(header, 'Interactive', None):
                                    header.setSectionResizeMode(c, getattr(header, 'Interactive', None))
                            except Exception:
                                pass
                except Exception:
                    # Best-effort: fallback to naive application
                    try:
                        for col, w in iw.items():
                            try:
                                self.items_table.setColumnWidth(int(col), int(w))
                            except Exception:
                                pass
                    except Exception:
                        pass

            # Restore column order if persisted
            try:
                order = gprefs.get('rss_reader_items_table_column_order', None)
                if isinstance(order, (list, tuple)) and order:
                    header = self.items_table.horizontalHeader()
                    # header.logicalIndex(visual) gives logical index at visual position
                    # We want to move sections so that visual positions reflect saved order.
                    # Iterate desired visual indices and move current section to target position.
                    for target_visual, logical in enumerate(order):
                        try:
                            # find current visual index of this logical
                            current_visual = header.visualIndex(int(logical))
                            if current_visual != target_visual:
                                try:
                                    header.moveSection(current_visual, target_visual)
                                except Exception:
                                    try:
                                        header.moveSection(int(current_visual), int(target_visual))
                                    except Exception:
                                        pass
                        except Exception:
                            pass

                # Re-apply saved column widths after column order/visibility restoration
                try:
                    iw = gprefs.get('rss_reader_items_table_widths', {}) or {}
                    for col, w in iw.items():
                        try:
                            self.items_table.setColumnWidth(int(col), int(w))
                        except Exception:
                            pass
                except Exception:
                    pass
            except Exception:
                pass

            # Restore sorting
            sc = gprefs.get('rss_reader_sort_column', None)
            so = gprefs.get('rss_reader_sort_order', None)
            if sc is None or so is None:
                # Default: newest first (Published column)
                try:
                    sc, so = 3, int(Qt.SortOrder.DescendingOrder)
                except Exception:
                    sc, so = 3, 1
            # Migration: older versions had 4 columns and Published was index 2.
            try:
                prev_cols = gprefs.get('rss_reader_items_table_colcount', None)
                if int(prev_cols or 0) == 4 and int(sc) == 2:
                    sc = 3
            except Exception:
                pass
            try:
                self.items_table.sortItems(int(sc), int(so))
            except Exception:
                pass
        except Exception:
            pass

    def restore_default_layout(self):
        # Reset splitters/columns to sane defaults and persist them.
        try:
            w = int(self.width() or 0)
            h = int(self.height() or 0)
        except Exception:
            w, h = 0, 0

        # Set default layout: articles on top
        try:
            gprefs['rss_reader_layout_type'] = 'articles_top'
        except Exception:
            pass

        # Set default AI panel mode: beside preview
        try:
            gprefs['rss_reader_ai_panel_mode'] = 'beside_preview'
        except Exception:
            pass

        # Defaults tuned for ~1366x720 but applied proportionally.
        # main_splitter: Feeds pane vs right pane
        try:
            if getattr(self, 'main_splitter', None) is not None:
                ww = w if w > 0 else int(self.main_splitter.size().width() or 0)
                if ww <= 0:
                    ww = 1210
                left = max(220, int(ww * 0.24))
                right = max(360, ww - left)
                self.main_splitter.setSizes([left, right])
                gprefs['rss_reader_main_splitter'] = [left, right]
        except Exception:
            pass

        # right_splitter: Items list vs Preview area (60% items, 40% preview with articles on top)
        try:
            if getattr(self, 'right_splitter', None) is not None:
                try:
                    self.right_splitter.setOrientation(Qt.Orientation.Vertical)
                except Exception:
                    pass
                rh = int(self.right_splitter.size().height() or 0)
                if rh <= 0:
                    rh = h if h > 0 else 650
                items = max(220, int(rh * 0.60))
                preview = max(200, rh - items)
                self.right_splitter.setSizes([items, preview])
                gprefs['rss_reader_right_splitter'] = [items, preview]
                try:
                    gprefs['rss_reader_right_splitter_articles_top'] = [items, preview]
                except Exception:
                    pass
        except Exception:
            pass

        # preview_splitter: Preview vs AI panel (70% preview, 30% AI beside preview)
        try:
            if getattr(self, 'preview_splitter', None) is not None:
                pw = int(self.preview_splitter.size().width() or 0)
                if pw <= 0:
                    pw = max(700, int((w or 1250) * 0.66))
                preview_w = max(460, int(pw * 0.70))
                ai_w = max(220, pw - preview_w)
                self.preview_splitter.setSizes([preview_w, ai_w])
                gprefs['rss_reader_preview_splitter'] = [preview_w, ai_w]
        except Exception:
            pass

        # Reset any user-resized column widths
        try:
            if 'rss_reader_items_table_widths' in gprefs:
                del gprefs['rss_reader_items_table_widths']
        except Exception:
            pass
        try:
            if getattr(self, 'items_table', None) is not None:
                # Keep Published reasonably tight
                self.items_table.setColumnWidth(1, 160)
                try:
                    self.items_table.setColumnWidth(2, 140)
                except Exception:
                    pass
        except Exception:
            pass

    def on_sort_changed(self, logical_index, order):
        try:
            gprefs['rss_reader_sort_column'] = logical_index
            gprefs['rss_reader_sort_order'] = int(order)
        except Exception:
            pass

    def refresh(self):
        # Rebuilding the feeds tree triggers selectionChanged, which calls
        # on_feed_selected() and may load items. Calling load_items here as well
        # can cause a full reload twice (very expensive for big folders).
        try:
            self._refreshing_feeds_tree = True
        except Exception:
            pass
        try:
            self.load_feeds()
            self._restore_feed_selection()
        finally:
            try:
                self._refreshing_feeds_tree = False
            except Exception:
                pass

        try:
            self._update_export_btn_tooltip()
        except Exception:
            pass

        self.load_items_for_selected_feed()

    def _sync_images_toggle(self):
        try:
            on = bool(plugin_prefs.get('load_images_in_preview', True))
        except Exception:
            on = True
        if getattr(self, 'images_toggle', None) is None:
            return
        try:
            self.images_toggle.blockSignals(True)
        except Exception:
            pass
        try:
            self.images_toggle.setChecked(on)
        finally:
            try:
                self.images_toggle.blockSignals(False)
            except Exception:
                pass

    def _sync_autofit_toggle(self):
        try:
            on = bool(plugin_prefs.get('autofit_images', True))
        except Exception:
            on = True
        if getattr(self, 'autofit_images_checkbox', None) is None:
            return
        try:
            self.autofit_images_checkbox.blockSignals(True)
        except Exception:
            pass
        try:
            self.autofit_images_checkbox.setChecked(on)
        finally:
            try:
                self.autofit_images_checkbox.blockSignals(False)
            except Exception:
                pass

    def toggle_images_in_preview(self, checked):
        try:
            plugin_prefs['load_images_in_preview'] = bool(checked)
        except Exception:
            pass

        try:
            if hasattr(self, 'preview') and hasattr(self.preview, '_image_cache'):
                self.preview._image_cache.clear()
        except Exception:
            pass

        # If images were disabled, remove any disk cache entries for the current
        # preview so we don't accidentally use cached files later.
        try:
            if not bool(plugin_prefs.get('load_images_in_preview', True)) and hasattr(self, 'preview'):
                try:
                    urls = getattr(self.preview, '_current_img_urls', None) or []
                    for u in urls:
                        try:
                            p = self.preview._img_cache_path_for_url(u)
                            if p and os.path.exists(p):
                                try:
                                    os.remove(p)
                                except Exception:
                                    pass
                        except Exception:
                            pass
                except Exception:
                    pass
        except Exception:
            pass

        try:
            self.on_item_selected()
        except Exception:
            pass

    def toggle_autofit_in_preview(self, checked):
        try:
            plugin_prefs['autofit_images'] = bool(checked)
        except Exception:
            pass
        try:
            # Re-render preview to apply CSS change
            if hasattr(self, 'on_item_selected'):
                self.on_item_selected()
        except Exception:
            pass

    def filter_feeds_tree(self, text):
        try:
            # Delegate to the unified filter that combines text + tag filters.
            try:
                active = getattr(self, '_active_feed_tag', None)
            except Exception:
                active = None
            self._apply_feeds_tag_filter(active)
        except Exception:
            # Filtering should never break the UI.
            return

    def switch_layout(self, layout_type):
        """Switch between article/preview layouts.

        Persists per-layout splitter sizes so user manual adjustments are remembered.
        Layouts:
          - articles_top / preview_top (vertical)
          - articles_left / preview_left (horizontal)
        """
        try:
            rs = getattr(self, 'right_splitter', None)
            if rs is None:
                try:
                    gprefs['rss_reader_layout_type'] = layout_type
                except Exception:
                    pass
                return

            # Save current layout sizes before switching away
            try:
                prev_lt = str(gprefs.get('rss_reader_layout_type', 'articles_top') or 'articles_top')
            except Exception:
                prev_lt = 'articles_top'
            if str(layout_type) != str(prev_lt):
                try:
                    prev_sizes = list(rs.sizes() or [])
                    if prev_sizes and len(prev_sizes) >= 2:
                        prev_sizes = prev_sizes[:2]
                        gprefs['rss_reader_right_splitter'] = prev_sizes  # legacy
                        gprefs['rss_reader_right_splitter_%s' % prev_lt] = prev_sizes
                except Exception:
                    pass

            # Save chosen layout type
            try:
                gprefs['rss_reader_layout_type'] = layout_type
            except Exception:
                pass

            is_horizontal = layout_type in ('articles_left', 'preview_left')
            items_first = layout_type in ('articles_top', 'articles_left')

            # Set orientation
            try:
                rs.setOrientation(Qt.Orientation.Horizontal if is_horizontal else Qt.Orientation.Vertical)
            except Exception:
                pass

            # Ensure widget order: widget(0) is items or preview depending on layout
            try:
                w0 = rs.widget(0)
                w1 = rs.widget(1)
                items_w = getattr(self, 'items_table', None)
                if items_w is None:
                    items_w = w0
                preview_w = w1 if w0 == items_w else w0

                if items_first:
                    if rs.widget(0) != items_w:
                        rs.insertWidget(0, items_w)
                else:
                    if rs.widget(0) == items_w:
                        rs.insertWidget(0, preview_w)
            except Exception:
                pass

            # Restore sizes for the target layout (per-layout key, fallback to legacy, then defaults)
            sizes_to_apply = None
            try:
                k = 'rss_reader_right_splitter_%s' % str(layout_type)
                v = gprefs.get(k, None)
                if v and isinstance(v, (list, tuple)) and len(v) >= 2:
                    sizes_to_apply = list(v)[:2]
            except Exception:
                sizes_to_apply = None
            if sizes_to_apply is None:
                try:
                    v = gprefs.get('rss_reader_right_splitter', None)
                    if v and isinstance(v, (list, tuple)) and len(v) >= 2:
                        sizes_to_apply = list(v)[:2]
                except Exception:
                    sizes_to_apply = None

            if sizes_to_apply is not None:
                try:
                    rs.setSizes(list(sizes_to_apply))
                    return
                except Exception:
                    pass

            # Defaults if nothing stored yet
            try:
                if is_horizontal:
                    w = int(rs.width() or 0) or int(self.width() or 1250)
                    rs.setSizes([int(w * 0.45), int(w * 0.55)])
                else:
                    h = int(rs.height() or 0) or int(self.height() or 680)
                    rs.setSizes([int(h * 0.6), int(h * 0.4)])
            except Exception:
                pass
        except Exception:
            pass

    def switch_ai_panel_mode(self, mode):
        """Switch AI panel between docked, beside preview, or hidden."""
        try:
            gprefs['rss_reader_ai_panel_mode'] = mode
            _debug('Switching AI panel mode to: %s' % mode)

            if not hasattr(self, 'ai_dock') or not hasattr(self, 'preview_splitter'):
                _debug('Missing ai_dock or preview_splitter')
                return

            if mode == 'docked_right':
                # Move AI panel back to dock (if it's in splitter, remove it first)
                try:
                    if self.preview_splitter.indexOf(self.ai_panel) >= 0:
                        self.ai_panel.setParent(None)
                except Exception:
                    pass

                # Ensure dock widget has the panel
                if self.ai_dock.widget() != self.ai_panel:
                    self.ai_dock.setWidget(self.ai_panel)

                # Show the dock widget
                self.ai_dock.setVisible(True)
                self.ai_panel.show()
                try:
                    self.ai_dock.raise_()
                except Exception:
                    pass
                gprefs['rss_reader_ai_panel_visible'] = True
                _debug('AI panel: docked on right')

            elif mode == 'beside_preview':
                # Move AI panel from dock to horizontal splitter beside preview
                # First hide the dock
                self.ai_dock.setVisible(False)

                # Remove from dock and add to splitter
                if self.ai_dock.widget() == self.ai_panel:
                    self.ai_dock.setWidget(None)
                    _debug('Removed AI panel from dock')

                # Add to the preview_row splitter (horizontal)
                if self.preview_splitter.indexOf(self.ai_panel) < 0:
                    self.preview_splitter.addWidget(self.ai_panel)
                    _debug('Added AI panel to splitter')

                # Ensure panel is visible
                self.ai_panel.show()
                try:
                    self.preview_splitter.updateGeometry()
                    self.preview_splitter.repaint()
                except Exception:
                    pass

                # Set reasonable split sizes. Prefer persisted preview_splitter if present,
                # otherwise allocate a minimal width to the AI panel so it starts unobtrusive.
                try:
                    s = gprefs.get('rss_reader_preview_splitter', None)
                    if s and isinstance(s, (list, tuple)):
                        try:
                            self.preview_splitter.setSizes(list(s))
                            _debug('AI panel sizes restored from prefs: %s' % str(s))
                        except Exception:
                            pass
                    else:
                        try:
                            ai_min = int(plugin_prefs.get('preview_ai_min_width', 120) or 120)
                        except Exception:
                            ai_min = 120
                        try:
                            w = self.preview_splitter.width() or int(self.width() or 1250)
                        except Exception:
                            w = 1250
                        try:
                            preview_w = max(460, int(w) - int(ai_min))
                            ai_w = max(int(ai_min), int(w) - int(preview_w))
                            self.preview_splitter.setSizes([preview_w, ai_w])
                            _debug('AI panel sizes set: preview=%d, ai=%d' % (int(preview_w), int(ai_w)))
                        except Exception as e:
                            _debug('Failed to set sizes: %s' % str(e))
                except Exception:
                    pass

                gprefs['rss_reader_ai_panel_visible'] = True
                _debug('AI panel: beside preview')

            elif mode == 'hidden':
                # Hide the AI panel completely
                # First try dock
                if self.ai_dock.widget() == self.ai_panel:
                    self.ai_dock.setVisible(False)

                # Also remove from splitter if present
                try:
                    if self.preview_splitter.indexOf(self.ai_panel) >= 0:
                        self.ai_panel.setParent(None)
                except Exception:
                    pass

                # Hide the panel regardless
                self.ai_panel.hide()

                gprefs['rss_reader_ai_panel_visible'] = False
                _debug('AI panel: hidden')

        except Exception as e:
            _debug('Error switching AI panel mode: %s' % str(e))

        # Force a relayout so the change is visible immediately.
        try:
            self.updateGeometry()
            self.repaint()
        except Exception:
            pass

    def open_settings(self):
        try:
            from calibre_plugins.rss_reader.config import ConfigDialog
            dlg = ConfigDialog(self.gui, self)
            if dlg.exec() == QDialog.DialogCode.Accepted:
                if self.action is not None:
                    self.action.apply_settings()
                try:
                    	self._sync_images_toggle()
                    	try:
                    		self._sync_autofit_toggle()
                    	except Exception:
                    		pass
                except Exception:
                    pass
                # Refresh auto-update button text to reflect any interval change
                try:
                    btn = getattr(self, 'auto_update_btn', None)
                    if btn is not None:
                        m_val = int(plugin_prefs.get('auto_update_minutes', 0) or 0)
                        def _fmt_min(mm):
                            mm = int(mm or 0)
                            if mm <= 0:
                                return _('Off')
                            if mm == 1440:
                                return _('Daily')
                            return _('%d min') % mm
                        btn.setText('⏳ ' + _fmt_min(m_val))
                except Exception:
                    pass
                # Apply published-column formatting changes immediately
                try:
                    self.load_items_for_selected_feed()
                except Exception:
                    pass
        except Exception as e:
            error_dialog(self, _('RSS Reader Error'), _('Failed to open settings: %s') % str(e),
                        show=True, det_msg=traceback.format_exc())

    def import_from_recipes(self):
        # Safe recipe picker (no Scheduler instantiation).
        # Only shows recipes that declare RSS feeds (importable by the plugin).
        try:
            try:
                from qt.core import (
                    Qt, QDialog, QVBoxLayout, QHBoxLayout, QLineEdit, QLabel, QPushButton,
                    QTableWidget, QTableWidgetItem, QAbstractItemView, QCheckBox, QMessageBox
                )
            except Exception:
                from PyQt5.Qt import (
                    Qt, QDialog, QVBoxLayout, QHBoxLayout, QLineEdit, QLabel, QPushButton,
                    QTableWidget, QTableWidgetItem, QAbstractItemView, QCheckBox, QMessageBox
                )

            try:
                shown = bool(plugin_prefs.get('import_recipes_first_run_notice_shown', False))
            except Exception:
                shown = True
            if not shown:
                try:
                    QMessageBox.question(
                        self,
                        _('Import feeds from recipes…'),
                        _('The first time you open this dialog, calibre may take a moment to load the recipe collection.\n\nSubsequent opens should be instant.')
                    )
                except Exception:
                    pass
                try:
                    plugin_prefs['import_recipes_first_run_notice_shown'] = True
                except Exception:
                    pass

            from calibre.web.feeds.recipes.collection import get_builtin_recipe_collection
            from calibre_plugins.rss_reader.recipe_utils import get_recipe_feeds_from_urn

            dlg = QDialog(self)
            dlg.setWindowTitle(_('Import feeds from recipes…'))
            v = QVBoxLayout(dlg)
            v.addWidget(QLabel(_('Filter recipes:'), dlg))

            q = QLineEdit(dlg)
            q.setPlaceholderText(_('Type to filter…'))
            try:
                q.setClearButtonEnabled(True)
            except Exception:
                pass
            v.addWidget(q)

            table = QTableWidget(dlg)
            table.setColumnCount(5)
            table.setHorizontalHeaderLabels([_('Import'), _('Title'), _('Language'), 'URN', _('Feeds')])
            table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
            table.setSelectionMode(QAbstractItemView.SelectionMode.MultiSelection)
            table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)
            v.addWidget(table)

            btn_row = QHBoxLayout()
            btn_row.addStretch(1)
            import_btn = QPushButton(_('Import selected'), dlg)
            export_btn = QPushButton(_('Export selected to OPML'), dlg)
            select_all_btn = QPushButton(_('Select All'), dlg)
            cancel_btn = QPushButton(_('Cancel'), dlg)
            btn_row.addWidget(import_btn)
            btn_row.addWidget(export_btn)
            btn_row.addWidget(select_all_btn)
            btn_row.addWidget(cancel_btn)
            v.addLayout(btn_row)
            def _select_all():
                for r in range(table.rowCount()):
                    chk = table.cellWidget(r, 0)
                    if chk:
                        chk.setChecked(True)
            def _do_export():
                selected_urns = []
                for r in range(table.rowCount()):
                    chk = table.cellWidget(r, 0)
                    if chk and chk.isChecked():
                        urn = table.item(r, 3).text() if table.item(r, 3) else ''
                        if urn:
                            selected_urns.append(str(urn))

                if not selected_urns:
                    QMessageBox.information(dlg, _('Export OPML'), _('No recipes selected.'))
                    return

                # Gather feeds from selected recipes, including language
                from calibre_plugins.rss_reader.recipe_utils import get_recipe_feeds_from_urn
                feeds = []
                coll = get_builtin_recipe_collection() or []
                for urn in selected_urns:
                    title = urn
                    lang = ''
                    for rec in coll:
                        if str(rec.get('id') or '').strip() == urn:
                            title = str(rec.get('title') or urn)
                            lang = str(rec.get('language') or '')
                            break
                    recipe_feeds = get_recipe_feeds_from_urn(urn) or []
                    for t, u in recipe_feeds:
                        feeds.append({'title': t or u, 'url': u, 'folder': '', 'is_recipe': True, 'recipe_urn': urn, 'language': lang})

                if not feeds:
                    QMessageBox.information(dlg, _('Export OPML'), _('No feeds found for selected recipes.'))
                    return

                # Ask user for file location
                fname = choose_save_file(
                    self,
                    'rss-reader-export-opml',
                    _('Export OPML'),
                    filters=[(_('OPML files'), ['opml']), (_('XML files'), ['xml'])],
                    all_files=True,
                    initial_filename='recipes_export.opml',
                )
                if not fname:
                    return

                # Write OPML file
                try:
                    self._write_opml(fname, feeds)
                    QMessageBox.information(self, _('Export complete'), _('OPML exported to %s') % fname)
                except Exception as e:
                    error_dialog(self, _('Export error'), _('Failed to export OPML: %s') % str(e), show=True, det_msg=traceback.format_exc())

            # Get importable recipes
            importable = []
            cache_file = os.path.join(rss_db.plugin_cache_dir(), 'recipe_import_cache.pkl')
            legacy_cache_file = ''
            try:
                legacy_cache_file = os.path.join(os.path.dirname(rss_db.legacy_config_db_path()), 'recipe_import_cache.pkl')
            except Exception:
                legacy_cache_file = ''
            try:
                if os.path.exists(cache_file) or (legacy_cache_file and os.path.exists(legacy_cache_file)):
                    import pickle
                    src = cache_file if os.path.exists(cache_file) else legacy_cache_file
                    with open(src, 'rb') as f:
                        cached = pickle.load(f)
                        # Backward/forward compatible cache format:
                        # - v1: list[dict]
                        # - v2: {'version': 2, 'rows': list[dict]}
                        if isinstance(cached, dict):
                            if int(cached.get('version') or 0) == 2 and isinstance(cached.get('rows'), list):
                                importable = list(cached.get('rows') or [])
                            else:
                                importable = []
                        elif isinstance(cached, list):
                            # Old cache: invalidate if it doesn't have language info
                            if cached and isinstance(cached[0], dict) and 'language' in cached[0]:
                                importable = list(cached)
                            else:
                                importable = []
                        else:
                            importable = []
            except Exception:
                importable = []

            if not importable:
                try:
                    coll = get_builtin_recipe_collection() or []
                except Exception:
                    coll = []

                if not hasattr(self, '_recipe_feeds_cache'):
                    self._recipe_feeds_cache = {}

                for rec in coll:
                    try:
                        urn = str(rec.get('id') or '').strip()
                        if not urn:
                            continue
                        try:
                            lang = str(rec.get('language') or '').strip()
                        except Exception:
                            lang = ''
                        lang_disp = lang
                        if lang:
                            try:
                                # Match the language grouping shown in calibre's recipe dialog
                                from calibre.web.feeds.recipes.model import parse_lang_code
                                lang_disp = parse_lang_code(lang.replace('-', '_'))
                            except Exception:
                                lang_disp = lang
                        if urn not in self._recipe_feeds_cache:
                            self._recipe_feeds_cache[urn] = get_recipe_feeds_from_urn(urn)
                        feeds = self._recipe_feeds_cache[urn]
                        if feeds:
                            title = str(rec.get('title') or urn)
                            feeds_str = '; '.join(f"{t}: {u}" for t, u in feeds[:3])  # limit to 3
                            importable.append({'urn': urn, 'title': title, 'language': lang_disp, 'feeds': feeds_str})
                    except Exception:
                        pass

                # Save cache
                try:
                    import pickle
                    os.makedirs(os.path.dirname(cache_file), exist_ok=True)
                    with open(cache_file, 'wb') as f:
                        pickle.dump({'version': 2, 'rows': importable}, f)
                except Exception:
                    pass

            # Populate table
            table.setRowCount(len(importable))
            for r, rec in enumerate(importable):
                chk = QCheckBox(table)
                chk.setChecked(False)
                table.setCellWidget(r, 0, chk)
                table.setItem(r, 1, QTableWidgetItem(rec['title']))
                table.setItem(r, 2, QTableWidgetItem(str(rec.get('language') or '')))
                table.setItem(r, 3, QTableWidgetItem(rec['urn']))
                table.setItem(r, 4, QTableWidgetItem(rec['feeds']))

            # Enable sorting (useful for Language)
            try:
                table.setSortingEnabled(True)
            except Exception:
                pass

            table.resizeColumnsToContents()
            table.setColumnWidth(0, 60)
            table.setColumnWidth(1, 200)
            table.setColumnWidth(2, 140)
            table.setColumnWidth(3, 150)
            table.setColumnWidth(4, 300)

            # Add context menu for feeds column
            def _on_table_context_menu(pos):
                item = table.itemAt(pos)
                if item and item.column() == 4:  # Feeds column
                    feeds_text = item.text()
                    if feeds_text:
                        menu = QMenu(dlg)
                        copy_act = menu.addAction(_('Copy feeds'))
                        act = menu.exec(table.viewport().mapToGlobal(pos))
                        if act == copy_act:
                            try:
                                QApplication.clipboard().setText(feeds_text)
                            except Exception:
                                pass

            table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            table.customContextMenuRequested.connect(_on_table_context_menu)

            def _do_import():
                selected_urns = []
                for r in range(table.rowCount()):
                    chk = table.cellWidget(r, 0)
                    if chk and chk.isChecked():
                        urn = table.item(r, 3).text() if table.item(r, 3) else ''
                        if urn:
                            selected_urns.append(str(urn))

                if not selected_urns:
                    QMessageBox.information(dlg, _('Import recipe'), _('No recipes selected.'))
                    return

                try:
                    feeds = list(rss_db.get_feeds() or [])
                except Exception:
                    feeds = []

                imported = 0
                skipped = 0
                coll = get_builtin_recipe_collection() or []
                try:
                    default_oldest = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
                except Exception:
                    default_oldest = 7
                try:
                    default_max = int(plugin_prefs.get('default_max_articles', 100) or 100)
                except Exception:
                    default_max = 100
                for urn in selected_urns:
                    title = urn
                    for rec in coll:
                        if str(rec.get('id') or '').strip() == urn:
                            title = str(rec.get('title') or urn)
                            break

                    folder = self.selected_folder_path() or str(plugin_prefs.get('default_folder', '') or '')

                    feeds.append({
                        'id': str(uuid.uuid4()),
                        'title': title,
                        'url': str(urn),
                        'enabled': True,
                        'folder': folder,
                        'download_images': True,
                        'always_notify': False,
                        'feed_starred': False,
                        'is_recipe': True,
                        'recipe_urn': str(urn),
                        'use_recipe_engine': True,
                        'oldest_article_days': max(0, int(default_oldest)),
                        'max_articles': max(0, int(default_max)),
                    })
                    imported += 1

                if imported > 0:
                    rss_db.save_feeds(feeds)
                    try:
                        self.refresh()
                    except Exception:
                        pass

                msg = _('Imported %(imported)d recipe(s)') % {'imported': imported}
                QMessageBox.information(self, _('Import complete'), msg)
                dlg.accept()

            def _apply_filter(text):
                q = (text or '').strip().lower()
                for r in range(table.rowCount()):
                    title = table.item(r, 1).text() if table.item(r, 1) else ''
                    lang = table.item(r, 2).text() if table.item(r, 2) else ''
                    urn = table.item(r, 3).text() if table.item(r, 3) else ''
                    feeds = table.item(r, 4).text() if table.item(r, 4) else ''
                    hay = (title + ' ' + lang + ' ' + urn + ' ' + feeds).lower()
                    show = (q in hay) if q else True
                    table.setRowHidden(r, not show)


            q.textChanged.connect(_apply_filter)
            import_btn.clicked.connect(_do_import)
            export_btn.clicked.connect(_do_export)
            select_all_btn.clicked.connect(_select_all)
            cancel_btn.clicked.connect(dlg.reject)
            table.doubleClicked.connect(lambda idx: _do_import() if idx.column() != 0 else None)

            dlg.resize(800, 600)
            dlg.exec()
        except Exception as e:
            error_dialog(self, _('Import failed'), _('Failed to import recipe: %s') % str(e), show=True, det_msg=traceback.format_exc())

    def _schedule_persist_feeds_tree(self):
        # Debounce: the model can emit multiple rowsMoved signals for a single user action.
        _debug('[FOLDER] _schedule_persist_feeds_tree called')
        try:
            if bool(getattr(self, '_persist_tree_scheduled', False)):
                _debug('[FOLDER] _schedule_persist_feeds_tree: already scheduled, skipping')
                return
            self._persist_tree_scheduled = True
        except Exception:
            return
        try:
            QTimer.singleShot(0, self._persist_feeds_tree_structure)
        except Exception:
            try:
                QTimer.singleShot(0, lambda: self._persist_feeds_tree_structure())
            except Exception:
                pass

    def _persist_feeds_tree_structure(self):
        _debug('[FOLDER] _persist_feeds_tree_structure ENTER')
        try:
            self._persist_tree_scheduled = False
        except Exception:
            pass

        # Avoid persisting while we are programmatically rebuilding the tree.
        try:
            if bool(getattr(self, '_refreshing_feeds_tree', False)):
                _debug('[FOLDER] _persist_feeds_tree_structure: _refreshing_feeds_tree=True, aborting')
                return
        except Exception:
            pass

        try:
            model = self.feeds_tree.model()
        except Exception:
            model = None
        if model is None:
            return

        # Helper to strip unread badges from folder display text.
        def _folder_display_name(item):
            try:
                d = item.data(ROLE_USER)
                if isinstance(d, dict) and d.get('type') == 'folder' and d.get('name'):
                    return str(d.get('name') or '')
            except Exception:
                pass
            try:
                t = str(item.text() or '')
            except Exception:
                t = ''
            try:
                import re
                t2 = re.sub(r'\s*\(\d+\)\s*$', '', t).strip()
                return t2
            except Exception:
                return t.strip()

        folder_paths_in_order = []
        feed_ids_in_order = []
        feed_folder_by_id = {}

        try:
            root = model.invisibleRootItem()
        except Exception:
            return

        def walk(item, parent_path=''):
            try:
                data = item.data(ROLE_USER)
            except Exception:
                data = None
            item_type = data.get('type') if isinstance(data, dict) else None
            item_text = ''
            try:
                item_text = str(item.text() or '')[:30]
            except Exception:
                pass
            _debug('[FOLDER] walk: type=%s text=%r parent_path=%r' % (item_type, item_text, parent_path))

            if isinstance(data, dict) and data.get('type') == 'folder':
                name = _folder_display_name(item)
                if parent_path:
                    cur_path = (parent_path.rstrip('/') + '/' + name).strip().strip('/')
                else:
                    cur_path = name.strip().strip('/')
                _debug('[FOLDER] walk: folder name=%r -> cur_path=%r' % (name, cur_path))
                if cur_path:
                    folder_paths_in_order.append(cur_path)
                try:
                    data2 = dict(data)
                    data2['path'] = cur_path
                    data2['name'] = name
                    item.setData(data2, ROLE_USER)
                except Exception:
                    pass
                for i in range(item.rowCount()):
                    walk(item.child(i), cur_path)
                return

            if isinstance(data, dict) and data.get('type') == 'feed':
                try:
                    fid = str(data.get('id') or '').strip()
                except Exception:
                    fid = ''
                if fid:
                    feed_ids_in_order.append(fid)
                    feed_folder_by_id[fid] = parent_path
                    _debug('[FOLDER] walk: feed id=%s -> folder=%r' % (fid[:8], parent_path))
                return

            # For root container or unknown types, recurse into children
            _debug('[FOLDER] walk: recursing into %d children' % item.rowCount())
            for i in range(item.rowCount()):
                walk(item.child(i), parent_path)

        for i in range(root.rowCount()):
            walk(root.child(i), '')

        try:
            rss_db.ensure_ready()
        except Exception:
            pass
        try:
            feeds = list(rss_db.get_feeds() or [])
        except Exception:
            feeds = []
        feeds_by_id = {str((f or {}).get('id') or '').strip(): dict(f or {}) for f in (feeds or [])}

        new_feeds = []
        for fid in feed_ids_in_order:
            f = feeds_by_id.get(fid)
            if not f:
                continue
            try:
                f['folder'] = str(feed_folder_by_id.get(fid, '') or '').strip().strip('/')
            except Exception:
                pass
            new_feeds.append(f)

        try:
            remaining = [f for fid, f in feeds_by_id.items() if fid and fid not in set(feed_ids_in_order)]
            if remaining:
                new_feeds.extend(remaining)
        except Exception:
            pass

        _debug('[FOLDER] _persist_feeds_tree_structure: saving %d feeds, folders=%s' % (len(new_feeds), folder_paths_in_order))
        for f in new_feeds:
            _debug('[FOLDER]   feed %s -> folder=%r' % (f.get('title', '')[:30], f.get('folder', '')))

        try:
            rss_db.save_feeds(new_feeds)
            _debug('[FOLDER] _persist_feeds_tree_structure: save_feeds() succeeded')
            # Verify the save by reading back
            verify_feeds = list(rss_db.get_feeds() or [])
            _debug('[FOLDER] VERIFY after save:')
            for vf in verify_feeds[:5]:
                _debug('[FOLDER]   %s -> folder=%r' % (vf.get('title', '')[:25], vf.get('folder', '')))
        except Exception as e:
            _debug('[FOLDER] _persist_feeds_tree_structure: save_feeds() FAILED: %s' % e)
            import traceback
            _debug('[FOLDER] traceback: %s' % traceback.format_exc())
            return

        try:
            seen = set()
            uniq_folders = []
            for p in folder_paths_in_order:
                pp = str(p or '').strip().strip('/')
                if not pp or pp in seen:
                    continue
                seen.add(pp)
                uniq_folders.append(pp)
            rss_db.save_folders(uniq_folders)
            _debug('[FOLDER] _persist_feeds_tree_structure: save_folders(%s) succeeded' % uniq_folders)
        except Exception as e:
            _debug('[FOLDER] _persist_feeds_tree_structure: save_folders() FAILED: %s' % e)

        # Do NOT call refresh() here - it can trigger a feedback loop.
        # The tree is already up-to-date since we just walked it.
        _debug('[FOLDER] _persist_feeds_tree_structure EXIT')

    def load_from_cache(self):
        try:
            self._items_by_feed_id = {}
        except Exception:
            pass
        try:
            cache = rss_db.get_feed_cache_map() or {}
        except Exception:
            cache = {}
        if isinstance(cache, dict):
            for feed_id, data in cache.items():
                if isinstance(data, dict) and data.get('items'):
                    self._items_by_feed_id[str(feed_id)] = list(data.get('items') or [])

    def load_feeds(self):
        _t0 = time.perf_counter()
        _debug('load_feeds start')
        feeds = list(rss_db.get_feeds() or [])
        try:
            self._all_feeds = {str(f.get('id') or ''): f for f in feeds if str(f.get('id') or '').strip()}
        except Exception:
            self._all_feeds = {}
        _debug('  get_feeds returned %d feeds in %.3fs' % (len(feeds), time.perf_counter() - _t0))
        # Ensure folder metadata exists
        try:
            default_folder = str(plugin_prefs.get('default_folder', '') or '')
        except Exception:
            default_folder = ''
        # Normalize folder paths consistently (tree nodes use stripped paths).
        try:
            default_folder = default_folder.strip().strip('/')
        except Exception:
            pass

        try:
            raw_folders = list(rss_db.get_folders() or [])
        except Exception:
            raw_folders = []

        folders = []
        try:
            seen_fp = set()
            for x in (raw_folders or []):
                fp = str(x or '').strip().strip('/')
                if fp in seen_fp:
                    continue
                seen_fp.add(fp)
                # Keep empty folder as a real bucket if used.
                folders.append(fp)
        except Exception:
            folders = []

        if default_folder and default_folder not in folders:
            folders.append(default_folder)
            try:
                rss_db.save_folders(list(folders))
            except Exception:
                pass

        # One-time migration: ensure every feed has a folder field, and
        # migrate legacy 'main' folder to root (empty string). Also ensure
        # Calibre-like default limits exist for older DBs that predate the fields.
        try:
            changed = False
            try:
                default_oldest = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
            except Exception:
                default_oldest = 7
            try:
                default_max = int(plugin_prefs.get('default_max_articles', 100) or 100)
            except Exception:
                default_max = 100
            for f in feeds:
                # Check if folder key is missing entirely (not just empty)
                folder_key_exists = 'folder' in f
                cur_folder = str(f.get('folder') or '').strip().strip('/')

                # Migrate legacy 'main' folder to root
                if cur_folder.lower() == 'main':
                    _debug('[FOLDER] load_feeds migration: %s folder=%r -> ""' % (f.get('title', '')[:20], cur_folder))
                    f['folder'] = ''
                    changed = True
                elif not folder_key_exists:
                    # Folder key missing entirely - assign default
                    _debug('[FOLDER] load_feeds migration: %s has no folder KEY, setting to %r' % (f.get('title', '')[:20], default_folder))
                    f['folder'] = default_folder
                    changed = True
                else:
                    # Normalize stored folder paths to match UI tree paths.
                    # Empty string '' is valid and means root level - don't change it!
                    f['folder'] = cur_folder

                # Add defaults only when keys are missing; respect explicit 0.
                try:
                    if 'oldest_article_days' not in f:
                        f['oldest_article_days'] = max(0, int(default_oldest))
                        changed = True
                except Exception:
                    pass
                try:
                    if 'max_articles' not in f:
                        f['max_articles'] = max(0, int(default_max))
                        changed = True
                except Exception:
                    pass
            if changed:
                _debug('[FOLDER] load_feeds: MIGRATION save_feeds() called!')
                try:
                    rss_db.save_feeds(feeds)
                except Exception:
                    pass
            # Also remove 'main' from folders table if present
            try:
                if 'main' in folders:
                    _debug('[FOLDER] load_feeds: removing "main" from folders table')
                    folders = [fp for fp in folders if fp.lower() != 'main']
                    rss_db.save_folders(folders)
            except Exception:
                pass
        except Exception:
            pass

        # Log current folder assignments from DB
        _debug('[FOLDER] load_feeds: feeds from DB:')
        for f in feeds[:5]:  # First 5 only
            _debug('[FOLDER]   %s -> folder=%r' % (f.get('title', '')[:25], f.get('folder', '')))

        # unread counts per feed
        unread_by_feed = {}
        feed_title_by_id = {}
        folder_by_feed = {}
        icon_bytes_by_feed = {}
        feed_url_by_id = {}
        failed_tip_by_feed = {}

        _t1 = time.perf_counter()
        cache = dict(rss_db.get_feed_cache_map() or {})
        _debug('  get_feed_cache_map done in %.3fs' % (time.perf_counter() - _t1))
        _t1 = time.perf_counter()
        seen = dict(rss_db.get_seen_item_ids_map() or {})
        _debug('  get_seen_item_ids_map done in %.3fs' % (time.perf_counter() - _t1))

        for f in feeds:
            feed_id = str(f.get('id') or '')
            if not feed_id:
                continue
            name = f.get('title') or f.get('url') or ''
            url = f.get('url') or ''
            feed_title_by_id[feed_id] = name
            feed_url_by_id[feed_id] = url

            # Read the folder from DB - respect explicit empty string as "root level".
            # Do NOT apply default_folder here; migration logic already handled that.
            raw_folder = f.get('folder')
            if raw_folder is None:
                # Field missing entirely - use default
                folder = str(default_folder or '').strip().strip('/')
            else:
                folder = str(raw_folder).strip().strip('/')
            _debug('[FOLDER] load_feeds: feed %s raw folder=%r -> computed=%r' % (name[:20], raw_folder, folder))
            folder_by_feed[feed_id] = folder
            if folder and folder not in folders:
                folders.append(folder)

            r = self._feeds_results.get(feed_id) if isinstance(self._feeds_results, dict) else None
            unread = 0
            try:
                # IMPORTANT: Read/unread state is derived from `seen_item_ids`.
                # `feeds_results[...]['new_count']` is a transient update result and
                # can become stale after the user marks items as read.
                try:
                    items = list(self._items_by_feed_id.get(feed_id, []) or [])
                except Exception:
                    items = []

                if not items:
                    try:
                        items = list((cache.get(feed_id, {}) or {}).get('items', []) or [])
                    except Exception:
                        items = []

                seen_set = set(str(x) for x in (seen.get(feed_id, []) or []) if str(x).strip())
                if items:
                    unread = sum(
                        1
                        for it in (items or [])
                        if str(it.get('id') or '').strip() and (str(it.get('id') or '').strip() not in seen_set)
                    )
                elif isinstance(r, dict) and 'new_count' in r:
                    # Fallback when we have no cached items yet.
                    unread = int(r.get('new_count') or 0)
                else:
                    unread = 0
            except Exception:
                unread = 0
            unread_by_feed[feed_id] = unread

            if isinstance(r, dict) and not r.get('ok') and r.get('error'):
                failed_tip_by_feed[feed_id] = _('Update failed: %s') % (r.get('error') or '')

            # favicon
            try:
                c = cache.get(feed_id, {})
                icon_val = c.get('icon')
                icon_bytes = None
                if isinstance(icon_val, str) and icon_val.startswith('b64:'):
                    try:
                        icon_bytes = base64.b64decode(icon_val[4:].encode('ascii'))
                    except Exception:
                        icon_bytes = None
                elif isinstance(icon_val, (bytes, bytearray)):
                    icon_bytes = bytes(icon_val)
                if icon_bytes:
                    icon_bytes_by_feed[feed_id] = icon_bytes
            except Exception:
                pass

        # Build folder -> feed ids mapping (including nested folders)
        folder_to_direct = {}
        for fid, folder in folder_by_feed.items():
            folder_to_direct.setdefault(folder, []).append(fid)

        _debug('[FOLDER] folder_to_direct: %s' % {k: len(v) for k, v in folder_to_direct.items()})

        # Build folder -> feed ids mapping (including nested folders) in a single pass.
        # Also compute per-folder totals so we can badge folders without repeatedly
        # summing over large lists (important for many subfolders).
        try:
            ordered_fids = [str((f or {}).get('id') or '') for f in (feeds or []) if str((f or {}).get('id') or '').strip()]
        except Exception:
            ordered_fids = list(feed_title_by_id.keys())

        # Collect all folder paths (including parent prefixes) so empty folders still exist.
        all_folder_keys = set([''])
        try:
            for p in (folders or []):
                fp = str(p or '').strip().strip('/')
                all_folder_keys.add(fp)
                parts = [x for x in fp.split('/') if x]
                for i in range(1, len(parts) + 1):
                    all_folder_keys.add('/'.join(parts[:i]))
        except Exception:
            pass
        try:
            for p in (folder_to_direct or {}).keys():
                fp = str(p or '').strip().strip('/')
                all_folder_keys.add(fp)
                parts = [x for x in fp.split('/') if x]
                for i in range(1, len(parts) + 1):
                    all_folder_keys.add('/'.join(parts[:i]))
        except Exception:
            pass
        _debug('[FOLDER] all_folder_keys: %d' % len(all_folder_keys))

        folder_to_all = {k: [] for k in all_folder_keys}
        folder_total_unread = {k: 0 for k in all_folder_keys}
        folder_total_count = {k: 0 for k in all_folder_keys}
        folder_direct_count = {k: 0 for k in all_folder_keys}

        # Direct counts
        try:
            for p, fids in (folder_to_direct or {}).items():
                pp = str(p or '').strip().strip('/')
                folder_direct_count[pp] = len(fids or [])
        except Exception:
            pass

        # Propagate each feed id to its folder and parent prefixes, preserving DB order.
        for fid in (ordered_fids or []):
            try:
                fpath = str(folder_by_feed.get(fid, default_folder) or '').strip().strip('/')
            except Exception:
                fpath = ''
            try:
                unread = int(unread_by_feed.get(fid) or 0)
            except Exception:
                unread = 0

            # Root bucket (all feeds)
            try:
                folder_to_all.setdefault('', []).append(fid)
                folder_total_unread[''] = int(folder_total_unread.get('', 0)) + unread
                folder_total_count[''] = int(folder_total_count.get('', 0)) + 1
            except Exception:
                pass

            if not fpath:
                continue

            parts = [p for p in fpath.split('/') if p]
            for i in range(1, len(parts) + 1):
                prefix = '/'.join(parts[:i])
                try:
                    folder_to_all.setdefault(prefix, []).append(fid)
                    folder_total_unread[prefix] = int(folder_total_unread.get(prefix, 0)) + unread
                    folder_total_count[prefix] = int(folder_total_count.get(prefix, 0)) + 1
                except Exception:
                    pass

        self._folder_to_feed_ids = folder_to_all
        self._folder_total_unread = folder_total_unread
        self._folder_total_count = folder_total_count
        self._folder_direct_count = folder_direct_count
        self._folder_by_feed = dict(folder_by_feed)
        self._feed_unread_by_id = dict(unread_by_feed)

        # Rebuild tree - block BOTH view and model signals to prevent
        # accidental rowsMoved/persist during programmatic tree construction.
        try:
            model = self.feeds_tree.model()
            try:
                model.blockSignals(True)
            except Exception:
                pass
            model.clear()
            self.feeds_tree.blockSignals(True)
            try:
                self.feeds_tree.setUpdatesEnabled(False)
            except Exception:
                pass
        except Exception:
            pass
        try:
            # View-level perf hints
            try:
                self.feeds_tree.setUniformRowHeights(True)
            except Exception:
                pass

            # Helper to create folder nodes
            folder_items = {}

            # Use a visible root node (QuiteRSS-style) to contain folders/feeds.
            root_container = None
            try:
                root_label = str(plugin_prefs.get('feed_tree_root_label', '') or '').strip()
            except Exception:
                root_label = ''
            if not root_label:
                root_label = _('Feeds')
            try:
                root_text = _('%(name)s (%(count)d)') % {'name': root_label, 'count': int(len(feeds) or 0)}
            except Exception:
                root_text = '%s (%d)' % (root_label, int(len(feeds) or 0))
            try:
                root_container = QStandardItem(root_text)
                root_container.setData({'type': 'root', 'name': root_label}, ROLE_USER)
                try:
                    flags = root_container.flags()
                    try:
                        flags |= Qt.ItemFlag.ItemIsDropEnabled
                        flags &= ~Qt.ItemFlag.ItemIsDragEnabled
                    except Exception:
                        flags |= Qt.ItemIsDropEnabled
                        flags &= ~Qt.ItemIsDragEnabled
                    root_container.setFlags(flags)
                except Exception:
                    pass
                root_container.setToolTip(_('All folders and feeds'))
                model.appendRow(root_container)
            except Exception:
                root_container = None

            def ensure_folder(path):
                path = str(path or '').strip().strip('/')
                if not path:
                    return None
                if path in folder_items:
                    return folder_items[path]
                parent_path = path.rpartition('/')[0]
                parent_item = ensure_folder(parent_path) if parent_path else None
                name = path.rpartition('/')[2] or path
                item = QStandardItem(name)
                item.setData({'type': 'folder', 'path': path, 'name': name}, ROLE_USER)
                # Allow folders to be both dragged and used as drop targets.
                try:
                    flags = item.flags()
                    flags |= Qt.ItemFlag.ItemIsDragEnabled
                    flags |= Qt.ItemFlag.ItemIsDropEnabled
                    item.setFlags(flags)
                except Exception:
                    try:
                        flags = item.flags()
                        flags |= Qt.ItemIsDragEnabled
                        flags |= Qt.ItemIsDropEnabled
                        item.setFlags(flags)
                    except Exception:
                        pass
                if parent_item is None:
                    try:
                        (root_container or model.invisibleRootItem()).appendRow(item)
                    except Exception:
                        model.appendRow(item)
                else:
                    parent_item.appendRow(item)
                folder_items[path] = item
                return item

            # Create all folders (preserve DB order; ensure_folder creates parents as needed)
            try:
                seen_fp = set()
                ordered_folders = []
                for x in (folders or []):
                    fp = str(x or '').strip().strip('/')
                    if not fp or fp in seen_fp:
                        continue
                    seen_fp.add(fp)
                    ordered_folders.append(fp)
            except Exception:
                ordered_folders = []
            for fp in (ordered_folders or []):
                ensure_folder(fp)

            # Add feeds as leaf nodes
            try:
                from qt.core import QIcon, QPixmap
            except Exception:
                try:
                    from PyQt5.Qt import QIcon, QPixmap
                except Exception:
                    QIcon = None
                    QPixmap = None
            try:
                feed_tags_map = dict(rss_db.get_feed_tags_map() or {})
            except Exception:
                feed_tags_map = {}

            try:
                self._feed_status_map = dict(rss_db.get_feed_status_map() or {})
            except Exception:
                self._feed_status_map = {}

            # Preserve feed order as stored in DB (supports drag-to-reorder).
            try:
                ordered_fids = [str((f or {}).get('id') or '') for f in (feeds or []) if str((f or {}).get('id') or '').strip()]
            except Exception:
                ordered_fids = list(feed_title_by_id.keys())
            # Feed flags are already available in the `feeds` list. Avoid per-feed DB calls.
            try:
                feed_obj_by_id = {str((x or {}).get('id') or ''): (x or {}) for x in (feeds or [])}
            except Exception:
                feed_obj_by_id = {}

            for fid in (ordered_fids or []):
                folder_path = folder_by_feed.get(fid, default_folder)
                parent = ensure_folder(folder_path) or root_container or model.invisibleRootItem()
                title = feed_title_by_id.get(fid, '')
                unread = int(unread_by_feed.get(fid) or 0)
                # Emoji indicators: 🚫 for suspended (enabled=False), 🔔 for always_notify, ★ for feed-level starred
                try:
                    feed_obj = feed_obj_by_id.get(str(fid), {})
                except Exception:
                    feed_obj = {}
                notify_flag = bool(feed_obj.get('always_notify', False))
                suspended_flag = not bool(feed_obj.get('enabled', True))
                feed_starred_flag = bool(feed_obj.get('feed_starred', False))
                # Use both emoji and short ASCII tags so state is visible even when
                # emoji fonts do not render in some Qt/platform setups.
                emoji_prefix = ''
                if suspended_flag:
                    emoji_prefix += '🚫 '
                if notify_flag:
                    emoji_prefix += '🔔 '
                if feed_starred_flag:
                    emoji_prefix += '★ '
                if unread > 0:
                    text = f'{emoji_prefix}{title} ({unread})'
                else:
                    text = f'{emoji_prefix}{title}'
                leaf = QStandardItem(text)
                leaf.setData({'type': 'feed', 'id': fid, 'title': title, 'url': feed_url_by_id.get(fid, '')}, ROLE_USER)
                # Feeds can be dragged but should not accept drops (prevents nesting feeds under feeds).
                try:
                    flags = leaf.flags()
                    flags |= Qt.ItemFlag.ItemIsDragEnabled
                    flags &= ~Qt.ItemFlag.ItemIsDropEnabled
                    leaf.setFlags(flags)
                except Exception:
                    try:
                        flags = leaf.flags()
                        flags |= Qt.ItemIsDragEnabled
                        flags &= ~Qt.ItemIsDropEnabled
                        leaf.setFlags(flags)
                    except Exception:
                        pass
                # Color the whole item if unread > 0
                if unread > 0:
                    try:
                        from qt.core import QBrush, QColor, QApplication
                    except ImportError:
                        from PyQt5.Qt import QBrush, QColor, QApplication
                    palette = QApplication.instance().palette()
                    # Detect dark theme by comparing window text and base colors
                    is_dark = palette.color(palette.Window).value() < 128
                    if is_dark:
                        color = QColor('#00E6E6')  # Aqua for dark
                    else:
                        color = palette.color(palette.Link)  # Use palette link color for light
                    try:
                        leaf.setForeground(QBrush(color))
                    except Exception:
                        pass
                tip = feed_url_by_id.get(fid, '')

                # RSS Guard-style feed metadata: Encoding + Type.
                try:
                    cmeta = cache.get(str(fid), {}) if isinstance(cache, dict) else {}
                except Exception:
                    cmeta = {}
                try:
                    enc = str((cmeta or {}).get('feed_encoding') or '').strip()
                except Exception:
                    enc = ''
                try:
                    ftype = str((cmeta or {}).get('feed_type') or '').strip()
                except Exception:
                    ftype = ''
                if enc:
                    tip = (tip + '\n' if tip else '') + (_('Encoding: %s') % enc)
                if ftype:
                    tip = (tip + '\n' if tip else '') + (_('Type: %s') % ftype)

                if fid in failed_tip_by_feed:
                    tip = (tip + '\n' if tip else '') + failed_tip_by_feed[fid]

                # Append tags (manual + auto) to tooltip for discoverability
                try:
                    manual = []
                    try:
                        v = feed_tags_map.get(str(fid))
                        if isinstance(v, list):
                            manual = v
                    except Exception:
                        manual = []

                    auto = []
                    try:
                        auto = list(self._auto_tags_for_feed(fid) or [])
                    except Exception:
                        auto = []

                    t_out = []
                    seen = set()
                    for t in list(manual or []) + list(auto or []):
                        tt = self._normalize_tag(t)
                        if not tt or tt in seen:
                            continue
                        seen.add(tt)
                        t_out.append(tt)
                    if t_out:
                        tip = (tip + '\n' if tip else '') + (_('Tags: ') + ', '.join(t_out))
                except Exception:
                    pass
                if feed_starred_flag:
                    try:
                        tip = (tip + '\n' if tip else '') + _('Feed is starred')
                    except Exception:
                        pass
                if tip:
                    try:
                        leaf.setToolTip(tip)
                    except Exception:
                        pass
                if fid in icon_bytes_by_feed:
                    try:
                        if QPixmap is not None and QIcon is not None:
                            pix = QPixmap(); pix.loadFromData(icon_bytes_by_feed[fid])
                            leaf.setIcon(QIcon(pix))
                    except Exception:
                        pass
                try:
                    parent.appendRow(leaf)
                except Exception:
                    model.appendRow(leaf)

            # Add visual separators after each folder's last feed (at top level only)
            # This creates a visual gap between folder sections
            try:
                root = root_container or model.invisibleRootItem()
                # Track which folders have direct child feeds
                folder_last_child = {}
                for i in range(root.rowCount()):
                    folder_item = root.child(i)
                    data = folder_item.data(ROLE_USER)
                    if isinstance(data, dict) and data.get('type') == 'folder':
                        # Check if this folder has child feeds
                        if folder_item.rowCount() > 0:
                            folder_last_child[i] = folder_item
                # Add a spacing/separator item after each folder that has feeds
                if folder_last_child:
                    # Add bottom margin to folders via size hint
                    for folder_item in folder_last_child.values():
                        try:
                            # Set a custom font size hint to add visual padding
                            from qt.core import QFont
                            font = folder_item.font()
                            # Make folder text slightly bolder for distinction
                            font.setBold(True)
                            folder_item.setFont(font)
                        except Exception:
                            pass
            except Exception:
                pass

            # Apply folder badges (sum unread of contained feeds)
            def apply_folder_badges(item, full_path):
                data = item.data(ROLE_USER)
                if isinstance(data, dict) and data.get('type') == 'folder':
                    p = str(data.get('path') or full_path or '')
                    # Tooltip: show feed counts for this folder.
                    try:
                        direct_n = int(getattr(self, '_folder_direct_count', {}).get(p, 0))
                    except Exception:
                        direct_n = 0
                    try:
                        total_n = int(getattr(self, '_folder_total_count', {}).get(p, len(self._folder_to_feed_ids.get(p, []) or [])))
                    except Exception:
                        total_n = 0
                    try:
                        tip_lines = [p] if p else []
                        tip_lines.append(_('Feeds in this folder: %d') % int(direct_n))
                        if int(total_n) != int(direct_n):
                            tip_lines.append(_('Feeds including subfolders: %d') % int(total_n))
                        item.setToolTip('\n'.join([x for x in tip_lines if x]))
                    except Exception:
                        pass
                    try:
                        total = int(getattr(self, '_folder_total_unread', {}).get(p, 0))
                    except Exception:
                        total = 0
                    name = p.rpartition('/')[2] or p
                    if total > 0:
                        text = f'{name} ({total})'
                    else:
                        text = name
                    item.setText(text)
                    if total > 0:
                        try:
                            from qt.core import QBrush, QColor, QApplication
                        except ImportError:
                            from PyQt5.Qt import QBrush, QColor, QApplication
                        palette = QApplication.instance().palette()
                        is_dark = palette.color(palette.Window).value() < 128
                        if is_dark:
                            color = QColor('#00E6E6')
                        else:
                            color = palette.color(palette.Link)
                        try:
                            item.setForeground(QBrush(color))
                        except Exception:
                            pass
                    else:
                        try:
                            item.setForeground(QBrush())
                        except Exception:
                            pass
                for i in range(item.rowCount()):
                    apply_folder_badges(item.child(i), full_path)

            root = root_container or model.invisibleRootItem()
            for i in range(root.rowCount()):
                apply_folder_badges(root.child(i), '')

            # Restore expanded folders
            expanded = set(gprefs.get('rss_reader_expanded_folders', []) or [])
            if expanded:
                def expand_if(item):
                    data = item.data(ROLE_USER)
                    if isinstance(data, dict) and data.get('type') == 'folder':
                        p = str(data.get('path') or '')
                        if p in expanded:
                            self.feeds_tree.setExpanded(model.indexFromItem(item), True)
                    for j in range(item.rowCount()):
                        expand_if(item.child(j))
                for i in range(root.rowCount()):
                    expand_if(root.child(i))
            else:
                # Expand the default folder to feel like QuiteRSS
                if default_folder in folder_items:
                    self.feeds_tree.setExpanded(model.indexFromItem(folder_items[default_folder]), True)

            # Keep the root container expanded.
            try:
                if root_container is not None:
                    self.feeds_tree.setExpanded(model.indexFromItem(root_container), True)
            except Exception:
                pass

        finally:
            try:
                self.feeds_tree.blockSignals(False)
            except Exception:
                pass
            try:
                model = self.feeds_tree.model()
                if model is not None:
                    model.blockSignals(False)
            except Exception:
                pass
            try:
                self.feeds_tree.setUpdatesEnabled(True)
            except Exception:
                pass

        # If action has a badge/count, sync the dialog status text
        try:
            total_new = 0
            for r in self._feeds_results.values():
                if isinstance(r, dict) and r.get('ok'):
                    total_new += int(r.get('new_count') or 0)
            if total_new > 0:
                self.status.setText(_('%d new items available') % total_new)
        except Exception:
            pass

        _debug('load_feeds total: %.3fs' % (time.perf_counter() - _t0))

        # Update feeds/folders stats label
        try:
            if getattr(self, 'feeds_stats_label', None) is not None:
                num_feeds = len(feeds)
                num_folders = len([f for f in folders if f])
                self.feeds_stats_label.setText(_('%(feeds)d feeds, %(folders)d folders') % {'feeds': num_feeds, 'folders': num_folders})
        except Exception:
            pass

        try:
            self._refresh_feed_tag_dropdown()
        except Exception:
            pass

        # Re-apply current feed filter after rebuilding the tree
        try:
            self.filter_feeds_tree(getattr(self, 'feeds_filter_input', None).text() if getattr(self, 'feeds_filter_input', None) is not None else '')
        except Exception:
            pass

    def _update_export_btn_tooltip(self):
        """Update the export button tooltip to describe what will be exported."""
        try:
            btn = getattr(self, 'export_ebook_btn', None)
            if btn is None:
                return
            selection_model = self.feeds_tree.selectionModel()
            if selection_model is None or not selection_model.selectedIndexes():
                btn.setToolTip(_('Export selected feeds to your preferred output format (nothing selected)'))
                return

            folder_count = 0
            feed_count = 0
            folder_names = []
            feed_names = []
            model = self.feeds_tree.model()
            for index in selection_model.selectedIndexes():
                item = model.itemFromIndex(index)
                data = item.data(ROLE_USER)
                if isinstance(data, dict):
                    if data.get('type') == 'folder':
                        folder_count += 1
                        folder_names.append(str(data.get('path', '') or '').split('/')[-1] or data.get('path', ''))
                    elif data.get('type') == 'feed':
                        feed_count += 1
                        feed_names.append(str(data.get('title', '') or '')[:30])

            parts = []
            if folder_count > 0:
                if folder_count == 1:
                    parts.append(_('folder "%s"') % folder_names[0])
                else:
                    parts.append(_('%d folders') % folder_count)
            if feed_count > 0:
                if feed_count == 1:
                    parts.append(_('feed "%s"') % feed_names[0])
                elif feed_count <= 3:
                    parts.append(_('feeds: %s') % ', '.join(feed_names))
                else:
                    parts.append(_('%d feeds') % feed_count)

            if parts:
                desc = ', '.join(parts)
                btn.setToolTip(_('Export %s to ebook') % desc)
            else:
                btn.setToolTip(_('Export selected feeds to your preferred output format'))
        except Exception:
            pass

    def _on_batch_test_recipes(self):
        import traceback, csv, os, concurrent.futures
        try:
            from calibre.web.feeds.recipes.collection import get_builtin_recipe_collection
            from calibre_plugins.rss_reader.recipe_utils import get_recipe_feeds_from_urn
        except Exception as e:
            error_dialog(self, _('Batch Test RSS Feeds'), _('Required Calibre APIs not available: %s') % str(e), show=True)
            return

        results = []
        try:
            coll = get_builtin_recipe_collection() or []
        except Exception:
            coll = []

        # Collect all feed tasks
        tasks = []
        for rec in coll:
            try:
                urn = str(rec.get('id') or '').strip()
                if not urn:
                    continue
                try:
                    feeds = get_recipe_feeds_from_urn(urn)
                    for title, url in feeds:
                        tasks.append((urn, url))
                except Exception:
                    pass
            except Exception:
                continue

        def check_feed(task):
            urn, url = task
            try:
                import urllib.request
                import xml.etree.ElementTree as ET
                import re
                from calibre.ebooks.chardet import detect as calibre_detect
                try:
                    from lxml import etree as lxml_etree
                except ImportError:
                    lxml_etree = None
                with urllib.request.urlopen(url, timeout=10) as response:
                    raw_bytes = response.read()

                # Try to detect encoding using calibre's chardet
                enc = None
                try:
                    enc = calibre_detect(raw_bytes).get('encoding')
                except Exception:
                    pass
                encodings_to_try = [enc] if enc else []
                encodings_to_try += ['utf-8', 'windows-1251', 'koi8-r', 'koi8-u', 'iso-8859-5', 'ibm866', 'latin1']

                content = None
                for encoding in encodings_to_try:
                    if not encoding:
                        continue
                    try:
                        content = raw_bytes.decode(encoding, errors='replace')
                        break
                    except Exception:
                        continue
                if content is None:
                    # Fallback: decode as utf-8 with replacement
                    content = raw_bytes.decode('utf-8', errors='replace')

                # Remove illegal XML characters
                def sanitize_xml(text):
                    return re.sub(r'[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]', '', text)
                sanitized = sanitize_xml(content)

                # Try ElementTree first
                try:
                    root = ET.fromstring(sanitized)
                except Exception:
                    # Fallback: try lxml if available
                    if lxml_etree is not None:
                        try:
                            root = lxml_etree.fromstring(sanitized.encode('utf-8'))
                        except Exception as e2:
                            valid = False
                            error = f'Not valid XML (lxml fallback failed): {e2}'
                            return {'urn': urn, 'feed_url': url, 'valid': valid, 'error': error}
                    else:
                        valid = False
                        error = 'Not valid XML (no lxml fallback)'
                        return {'urn': urn, 'feed_url': url, 'valid': valid, 'error': error}

                tag = root.tag.lower()
                if tag.endswith('rss') or tag.endswith('feed'):
                    valid = True
                    error = ''
                else:
                    valid = False
                    error = f'Root tag is not rss or atom feed: {tag}'
            except Exception as e:
                valid = False
                error = str(e)
            return {'urn': urn, 'feed_url': url, 'valid': valid, 'error': error}

        out_path = os.path.join(os.path.expanduser('~'), 'rss_feed_integrity_results.csv')
        try:
            with open(out_path, 'w', encoding='utf-8', newline='') as f:
                writer = csv.DictWriter(f, fieldnames=['urn', 'feed_url', 'valid', 'error'])
                writer.writeheader()
                f.flush()
                with concurrent.futures.ThreadPoolExecutor(max_workers=8) as executor:
                    futures = [executor.submit(check_feed, task) for task in tasks]
                    for future in concurrent.futures.as_completed(futures):
                        result = future.result()
                        writer.writerow(result)
                        f.flush()
            msg = _('Batch test complete. Results written to: %s') % out_path
        except Exception as e:
            msg = _('Batch test failed: %s') % str(e)
        error_dialog(self, _('Batch Test RSS Feeds'), msg, show=True)

    def selected_feed_id(self):
        fids = self.selected_feed_ids()
        return fids[0] if fids else ''

    def selected_feed_ids(self):
        out = []
        try:
            selection_model = self.feeds_tree.selectionModel()
            if selection_model is None:
                return []
            indexes = selection_model.selectedIndexes()
            model = self.feeds_tree.model()
            for index in indexes:
                item = model.itemFromIndex(index)
                data = item.data(ROLE_USER)
                if isinstance(data, dict) and data.get('type') == 'feed':
                    fid = str(data.get('id') or '')
                    if fid:
                        out.append(fid)
                elif isinstance(data, dict) and data.get('type') == 'folder':
                    p = str(data.get('path') or '').strip().strip('/')
                    out.extend(list(self._folder_to_feed_ids.get(p, []) or []))
                elif isinstance(data, dict) and data.get('type') == 'root':
                    # Selecting the root container should behave like "All feeds".
                    try:
                        out.extend([str(f.get('id') or '') for f in (rss_db.get_feeds() or []) if str(f.get('id') or '').strip()])
                    except Exception:
                        pass
        except Exception:
            pass
        # de-dup preserve order
        seen_ids = set()
        uniq = []
        for fid in out:
            if fid not in seen_ids:
                uniq.append(fid); seen_ids.add(fid)
        return uniq

    def selected_folder_path(self):
        try:
            selection_model = self.feeds_tree.selectionModel()
            if selection_model is None:
                return ''
            indexes = selection_model.selectedIndexes()
            model = self.feeds_tree.model()
            for index in indexes:
                item = model.itemFromIndex(index)
                data = item.data(ROLE_USER)
                if isinstance(data, dict) and data.get('type') == 'folder':
                    return str(data.get('path') or '')
        except Exception:
            pass
        # If only a feed is selected, infer its folder
        fid = self.selected_feed_id()
        if fid:
            feeds = list(rss_db.get_feeds() or [])
            f = next((x for x in feeds if str(x.get('id') or '') == fid), None)
            if f is not None:
                # Return the folder as stored - empty string '' means root level
                folder_val = f.get('folder')
                if folder_val is None:
                    return str(plugin_prefs.get('default_folder', '') or '')
                return str(folder_val)
        return ''

    def selected_folder_paths(self):
        """Return selected folder paths (normalized, de-duped).

        If both a parent folder and its subfolder(s) are selected, only the
        parent folder is returned (moving the parent implicitly moves children).
        """
        paths = []
        try:
            selection_model = self.feeds_tree.selectionModel()
            if selection_model is None:
                return []
            indexes = selection_model.selectedIndexes()
            model = self.feeds_tree.model()
            for index in indexes:
                try:
                    item = model.itemFromIndex(index)
                except Exception:
                    item = None
                if item is None:
                    continue
                try:
                    data = item.data(ROLE_USER)
                except Exception:
                    data = None
                if isinstance(data, dict) and data.get('type') == 'folder':
                    p = str(data.get('path') or '').strip().strip('/')
                    if p:
                        paths.append(p)
        except Exception:
            paths = []

        # de-dup preserve order
        seen = set()
        uniq = []
        for p in paths:
            if p not in seen:
                uniq.append(p)
                seen.add(p)

        # Remove subfolders when their parent is selected
        try:
            uniq_sorted = sorted(uniq, key=lambda x: (len(x), x))
        except Exception:
            uniq_sorted = list(uniq)
        kept = []
        for p in uniq_sorted:
            is_child = False
            for parent in kept:
                if p == parent or p.startswith(parent + '/'):
                    is_child = True
                    break
            if not is_child:
                kept.append(p)

        # Preserve original selection order as much as possible
        kept_set = set(kept)
        return [p for p in uniq if p in kept_set]

    def _restore_feed_selection(self):
        try:
            model = self.feeds_tree.model()
            if model.rowCount() == 0:
                return
            selection_model = self.feeds_tree.selectionModel()
            if selection_model is not None and selection_model.selectedIndexes():
                return

            want_fids = list(gprefs.get('rss_reader_last_selected_feed_ids', []) or [])
            want_folder = str(gprefs.get('rss_reader_last_selected_folder', '') or '')

            # Prefer selecting the previously selected feeds first
            for fid in want_fids:
                item = self._find_tree_item_by_feed(fid)
                if item is not None:
                    index = model.indexFromItem(item)
                    selection_model.select(index, selection_model.SelectionFlag.Select)
                    self.feeds_tree.scrollTo(index)
                    return

            # Else select the previously selected folder
            if want_folder:
                item = self._find_tree_item_by_folder(want_folder)
                if item is not None:
                    index = model.indexFromItem(item)
                    selection_model.select(index, selection_model.SelectionFlag.Select)
                    self.feeds_tree.scrollTo(index)
                    return

                # If the folder was deleted, try selecting the closest existing
                # ancestor instead of falling back to the first top-level item.
                cur = str(want_folder or '').strip().strip('/')
                while '/' in cur:
                    cur = cur.rpartition('/')[0]
                    if not cur:
                        break
                    item = self._find_tree_item_by_folder(cur)
                    if item is not None:
                        index = model.indexFromItem(item)
                        selection_model.select(index, selection_model.SelectionFlag.Select)
                        self.feeds_tree.scrollTo(index)
                        return

            # Fallback: select first top-level item
            if model.rowCount() > 0:
                first_item = model.item(0)
                if first_item is not None:
                    index = model.indexFromItem(first_item)
                    selection_model.select(index, selection_model.SelectionFlag.Select)
        except Exception:
            pass

    def _find_tree_item_by_feed(self, feed_id):
        feed_id = str(feed_id or '')
        if not feed_id:
            return None
        model = self.feeds_tree.model()
        root = model.invisibleRootItem()
        stack = [root]
        while stack:
            cur = stack.pop()
            for i in range(cur.rowCount()):
                ch = cur.child(i)
                data = ch.data(ROLE_USER)
                if isinstance(data, dict) and data.get('type') == 'feed' and str(data.get('id') or '') == feed_id:
                    return ch
                stack.append(ch)
        return None

    def _find_tree_item_by_folder(self, folder_path):
        folder_path = str(folder_path or '').strip().strip('/')
        if not folder_path:
            return None
        model = self.feeds_tree.model()
        root = model.invisibleRootItem()
        stack = [root]
        while stack:
            cur = stack.pop()
            for i in range(cur.rowCount()):
                ch = cur.child(i)
                data = ch.data(ROLE_USER)
                if isinstance(data, dict) and data.get('type') == 'folder' and str(data.get('path') or '') == folder_path:
                    return ch
                stack.append(ch)
        return None

