from __future__ import absolute_import

import uuid
from contextlib import contextmanager

try:
    load_translations()
except NameError:
    pass

from calibre_plugins.rss_reader.debug import _debug
from calibre_plugins.rss_reader import rss_db
from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader.utils import safe_folder_path


SAMPLE_FEEDS_FOLDER = 'Featured'


def default_sample_feeds():
    """Return the built-in sample feeds list.

    Each entry is a dict with at least: name, url.
    Optional: lang (BCP-47-ish), tags (list[str]).
    """
    return get_bundled_feeds_merged()


def get_bundled_feeds_merged():
    """Return the bundled feeds list as builtin defaults + any user additions.

    Some users may have an older/smaller stored `plugin_prefs['bundled_feeds']`
    which would otherwise hide newly added bundled feeds.
    """
    try:
        builtin = list((getattr(plugin_prefs, 'defaults', {}) or {}).get('bundled_feeds', []) or [])
    except Exception:
        builtin = []
    try:
        user_list = list(plugin_prefs.get('bundled_feeds', []) or [])
    except Exception:
        user_list = []

    def _key(entry):
        try:
            return str((entry or {}).get('url') or '').strip()
        except Exception:
            return ''

    out = []
    seen = set()

    for src in (builtin or []):
        k = _key(src)
        if not k or k in seen:
            continue
        seen.add(k)
        out.append(dict(src))

    # Include user-defined extras that aren't part of builtins
    for src in (user_list or []):
        k = _key(src)
        if not k or k in seen:
            continue
        seen.add(k)
        out.append(dict(src))

    return out


def _normalize_entries(urls_or_entries):
    out = []
    for it in (urls_or_entries or []):
        if isinstance(it, dict):
            try:
                name = str(it.get('name', '') or '').strip()
            except Exception:
                name = ''
            try:
                url = str(it.get('url', '') or '').strip()
            except Exception:
                url = ''
            if not url:
                continue
            entry = dict(it)
            entry['name'] = name or url
            entry['url'] = url
            out.append(entry)
        else:
            try:
                url = str(it or '').strip()
            except Exception:
                url = ''
            if not url:
                continue
            out.append({'name': url, 'url': url})
    return out


@contextmanager
def _with_db_path(db_path, readonly=False):
    """Temporarily switch the global rss_db path for this thread."""
    try:
        old_path = getattr(rss_db, 'DB_PATH', '')
        old_ro = bool(getattr(rss_db, 'DB_READONLY', False))
    except Exception:
        old_path, old_ro = '', False

    try:
        if db_path:
            rss_db.set_db_path(str(db_path), readonly=bool(readonly))
        yield
    finally:
        try:
            if old_path:
                rss_db.set_db_path(old_path, readonly=old_ro)
        except Exception:
            pass


def add_sample_feeds(urls_or_entries, db_path=None, folder=None, readonly=False):
    """Add sample feeds to the database.

    Returns (added_count, skipped_count).
    Skips duplicates by URL.
    """
    entries = _normalize_entries(urls_or_entries)
    if not entries:
        return 0, 0

    # If caller doesn't specify a folder, put bundled/sample feeds in a dedicated folder.
    # This keeps onboarding additions tidy and avoids accidental nesting under a legacy
    # implicit folder name like "main".
    try:
        folder_value = str(folder or '').strip()
    except Exception:
        folder_value = ''
    if not folder_value:
        folder_value = SAMPLE_FEEDS_FOLDER

    # Normalize and avoid forcing any implicit fallback like 'main'.
    try:
        default_folder = str(plugin_prefs.get('default_folder', '') or '').strip()
    except Exception:
        default_folder = ''
    folder_value = safe_folder_path(folder_value, default_folder=default_folder)

    with _with_db_path(db_path, readonly=readonly):
        try:
            rss_db.ensure_ready()
        except Exception:
            # ensure_ready calls init_db; keep going even if it fails, save_feeds will surface errors.
            pass

        try:
            existing = list(rss_db.get_feeds() or [])
        except Exception:
            existing = []

        existing_urls = set()
        for f in existing:
            try:
                u = str((f or {}).get('url') or '').strip()
            except Exception:
                u = ''
            if u:
                existing_urls.add(u)

        added = 0
        skipped = 0

        try:
            default_oldest = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
        except Exception:
            default_oldest = 7
        try:
            default_max = int(plugin_prefs.get('default_max_articles', 100) or 100)
        except Exception:
            default_max = 100

        for e in entries:
            try:
                url = str(e.get('url') or '').strip()
            except Exception:
                url = ''
            if not url:
                continue
            if url in existing_urls:
                skipped += 1
                continue

            try:
                title = str(e.get('name') or url)
            except Exception:
                title = url

            # Per-entry overrides: if the bundled entry provides explicit
            # settings, honour them. Otherwise default to showing the
            # fullest content for bundled feeds (no age limit, no max).
            try:
                if 'download_images' in e:
                    download_images_val = bool(e.get('download_images'))
                else:
                    download_images_val = True
            except Exception:
                download_images_val = True

            try:
                v = e.get('oldest_article_days')
                if v is None:
                    oldest_val = 0
                else:
                    oldest_val = max(0, int(v))
            except Exception:
                oldest_val = 0

            try:
                v2 = e.get('max_articles')
                if v2 is None:
                    max_val = 0
                else:
                    max_val = max(0, int(v2))
            except Exception:
                max_val = 0

            existing.append({
                'id': str(uuid.uuid4()),
                'title': title or url,
                'url': url,
                'enabled': True,
                'folder': folder_value,
                'download_images': download_images_val,
                'always_notify': False,
                'feed_starred': False,
                'is_recipe': False,
                'recipe_urn': '',
                'use_recipe_engine': False,
                'oldest_article_days': oldest_val,
                'max_articles': max_val,
            })
            existing_urls.add(url)
            added += 1

        if added:
            rss_db.save_feeds(existing)

            # Ensure the folder exists in the folders table so it shows up in a stable order.
            try:
                fp = str(folder_value or '').strip().strip('/')
            except Exception:
                fp = ''
            if fp:
                try:
                    folders = list(rss_db.get_folders() or [])
                except Exception:
                    folders = []
                if fp not in folders:
                    folders.append(fp)
                    try:
                        rss_db.save_folders(folders)
                    except Exception:
                        pass

        return int(added), int(skipped)
