from __future__ import absolute_import

import os
import json
import sqlite3
import zlib
import tempfile
from contextlib import contextmanager

try:
    load_translations()
except NameError:
    pass

try:
    # calibre's cache_dir() points at LocalAppData (Windows) / per-user cache dirs.
    from calibre.constants import cache_dir as _calibre_cache_dir, config_dir as _calibre_config_dir
except Exception:
    _calibre_cache_dir = None
    try:
        from calibre.utils.config import config_dir as _calibre_config_dir  # type: ignore
    except Exception:
        _calibre_config_dir = ''


DB_FILENAME = 'rss_reader.sqlite'

PLUGIN_NAME = 'rss_reader'

# Debug flag for verbose logging in rss_db module
DEBUG_RSS_READER = False

def _calibre_cache_base():
    try:
        if callable(_calibre_cache_dir):
            return _calibre_cache_dir()
        return str(_calibre_cache_dir or '')
    except Exception:
        return ''


def _calibre_config_base():
    try:
        return str(_calibre_config_dir or '')
    except Exception:
        return ''


def suggested_data_dir():
    """Return a durable default folder for the RSS Reader DB.

    - Portable installs: prefer a sibling folder next to "Calibre Settings".
    - Normal installs: prefer a folder under the user's home directory.

    This is intentionally *not* calibre's cache directory.
    """
    cfg = _calibre_config_base()
    try:
        tail = os.path.basename(cfg.rstrip('\\/')) if cfg else ''
    except Exception:
        tail = ''

    # Heuristic: calibre portable uses a "Calibre Settings" folder.
    try:
        if tail.lower() == 'calibre settings' and cfg:
            root = os.path.dirname(cfg.rstrip('\\/'))
            return os.path.join(root, 'rss_reader_data')
    except Exception:
        pass

    # Non-portable: store under user home (durable, user-managed).
    try:
        home = os.path.expanduser('~')
    except Exception:
        home = ''
    if home:
        return os.path.join(home, 'rss_reader_data')
    # Last resort: temp (still not calibre cache/config). Should be rare.
    try:
        return os.path.join(tempfile.gettempdir(), 'rss_reader_data')
    except Exception:
        return ''


def suggested_default_db_path():
    d = suggested_data_dir()
    if not d:
        return ''
    return os.path.join(d, DB_FILENAME)


def _fallback_os_cache_base():
    """Best-effort cache base that is never calibre's config dir.

    Used only when calibre.constants.cache_dir is unavailable or fails.
    """
    try:
        if os.name == 'nt':
            base = os.environ.get('LOCALAPPDATA') or os.environ.get('TEMP') or tempfile.gettempdir()
            return os.path.join(base, 'calibre')
        if os.path.exists('/Users') and os.path.isdir('/Users'):
            # macOS-like
            return os.path.expanduser('~/Library/Caches/calibre')
        # Linux/BSD
        base = os.environ.get('XDG_CACHE_HOME')
        if base:
            return os.path.join(os.path.expanduser(base), 'calibre')
        return os.path.expanduser('~/.cache/calibre')
    except Exception:
        try:
            return os.path.join(tempfile.gettempdir(), 'calibre')
        except Exception:
            return ''


def plugin_cache_dir():
    base = _calibre_cache_base()
    if not base:
        # Fallback: avoid calibre config dir even in edge cases.
        base = _fallback_os_cache_base()
    return os.path.join(base, 'plugins', PLUGIN_NAME)


def _norm_path(p):
    try:
        return os.path.normcase(os.path.realpath(str(p)))
    except Exception:
        try:
            return os.path.normcase(os.path.abspath(str(p)))
        except Exception:
            return str(p)


def _is_under_dir(path, parent_dir):
    try:
        p = _norm_path(path)
        d = _norm_path(parent_dir)
        if not p or not d:
            return False
        if p == d:
            return True
        return p.startswith(d.rstrip('\\/') + os.sep)
    except Exception:
        return False


def is_disallowed_db_path(path):
    """Return True if `path` is inside calibre's config or cache directory."""
    try:
        p = str(path or '').strip()
    except Exception:
        p = ''
    if not p:
        return False
    # Never allow sqlite DBs under calibre's config dir.
    if _is_under_dir(p, _calibre_config_dir):
        return True
    # Also avoid calibre cache dir for DB durability.
    try:
        cb = _calibre_cache_base()
    except Exception:
        cb = ''
    if cb and _is_under_dir(p, cb):
        return True
    return False


def assert_db_path_allowed(path):
    if is_disallowed_db_path(path):
        raise ValueError(
            'SQLite database files are not allowed inside calibre\'s configuration or cache folders. '
            'Please choose a durable location outside calibre\'s folders (portable installs: a sibling folder next to "Calibre Settings" is recommended).\n\nPath: %s' % (path,)
        )

# Default DB location is durable (outside calibre config/cache).
DB_PATH = suggested_default_db_path() or os.path.join(_fallback_os_cache_base(), 'plugins', PLUGIN_NAME, DB_FILENAME)
# If True, open DB in read-only mode (uses URI mode). Set via set_db_path().
DB_READONLY = False


def legacy_config_db_path():
    # e.g. %APPDATA%\calibre\plugins\rss_reader\rss_reader.sqlite
    return os.path.join(str(_calibre_config_dir or ''), 'plugins', PLUGIN_NAME, DB_FILENAME)


def _move_file_best_effort(src, dst):
    if not src or not dst:
        return False
    try:
        if not os.path.exists(src):
            return False
    except Exception:
        return False
    try:
        os.makedirs(os.path.dirname(dst), exist_ok=True)
    except Exception:
        pass
    try:
        os.replace(src, dst)
        return True
    except Exception:
        pass
    try:
        import shutil
        shutil.copy2(src, dst)
        try:
            os.remove(src)
        except Exception:
            pass
        return True
    except Exception:
        return False


def migrate_legacy_db_from_config_dir():
    """Move the legacy DB out of calibre config dir into the cache dir.

    This is best-effort and runs before opening the DB, so we don't leave
    sqlite databases in calibre's config directory.
    """
    src = legacy_config_db_path()
    try:
        if not src or not os.path.exists(src):
            return
    except Exception:
        return

    # If we're already not in config dir (portable setups), do nothing.
    if not is_disallowed_db_path(src):
        return

    dst = os.path.join(plugin_cache_dir(), DB_FILENAME)
    try:
        if os.path.exists(dst):
            # Avoid clobbering an existing cache DB.
            import time
            base, ext = os.path.splitext(DB_FILENAME)
            dst = os.path.join(plugin_cache_dir(), f'{base}.from_config_{int(time.time())}{ext}')
    except Exception:
        pass

    for suffix in ('', '-wal', '-shm'):
        _move_file_best_effort(src + suffix, dst + suffix)


def set_db_path(path, readonly=False):
    """Set the database path and optional read-only flag.

    When readonly=True, subsequent connections will open the DB in
    SQLite URI mode with mode=ro, preventing any writes from this process.
    """
    global DB_PATH, DB_READONLY
    # Disallow placing sqlite DBs under calibre config dir.
    assert_db_path_allowed(path)
    try:
        DB_PATH = str(path or '')
    except Exception:
        DB_PATH = path
    try:
        DB_READONLY = bool(readonly)
    except Exception:
        DB_READONLY = False


@contextmanager
def temporary_db_path(path, readonly=False):
    """Temporarily switch DB_PATH/DB_READONLY for helper operations.

    Intended for non-UI helpers that need to initialize or inspect a
    different DB file *without* permanently switching the live
    session's database. Always restores the original DB state.
    """
    try:
        old_path = DB_PATH
        old_ro = DB_READONLY
    except Exception:
        old_path, old_ro = None, False
    try:
        set_db_path(path, readonly=readonly)
        yield
    finally:
        try:
            if old_path is not None:
                set_db_path(old_path, readonly=old_ro)
        except Exception:
            pass


SCHEMA_VERSION = 11

SCHEMA = '''
CREATE TABLE IF NOT EXISTS meta (
    k TEXT PRIMARY KEY,
    v TEXT
);

CREATE TABLE IF NOT EXISTS feeds (
    id TEXT PRIMARY KEY,
    title TEXT,
    url TEXT,
    folder TEXT,
    enabled INTEGER DEFAULT 1,
    download_images INTEGER DEFAULT 1,
    always_notify INTEGER DEFAULT 0,
    use_recipe_engine INTEGER DEFAULT 0,
    -- Per-feed update limits. 0 means "no limit".
    oldest_article_days INTEGER,
    max_articles INTEGER,
    feed_starred INTEGER DEFAULT 0,
    is_recipe INTEGER DEFAULT 0,
    recipe_urn TEXT,
    -- Optional per-feed retention in days for cleanup. If NULL/0, uses
    -- the global/default retention from the cleanup dialog.
    retention_days INTEGER
);

CREATE TABLE IF NOT EXISTS folders (
    path TEXT PRIMARY KEY
);

-- Store per-feed lists as compressed JSON.
-- JSONConfig wrote a dict: feed_id -> [item_id, ...] (newest-first).
CREATE TABLE IF NOT EXISTS seen_item_ids (
    feed_id TEXT PRIMARY KEY,
    data BLOB
);

-- Store per-feed lists as compressed JSON.
-- These are item IDs that have been fetched before (used to detect what is *new*).
CREATE TABLE IF NOT EXISTS known_item_ids (
    feed_id TEXT PRIMARY KEY,
    data BLOB
);

-- Store per-feed cache as compressed JSON.
-- JSONConfig wrote a dict: feed_id -> {title, home_link, updated, items:[...]}
CREATE TABLE IF NOT EXISTS feed_cache (
    feed_id TEXT PRIMARY KEY,
    data BLOB
);

-- Per-feed status (last failure classification, for auto-tagging / sanitization)
CREATE TABLE IF NOT EXISTS feed_status (
    feed_id TEXT PRIMARY KEY,
    data BLOB
);

-- Feed failure history (append-only; periodically purged by retention policy)
CREATE TABLE IF NOT EXISTS feed_fail_history (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    ts INTEGER,
    feed_id TEXT,
    title TEXT,
    url TEXT,
    error TEXT,
    http_status INTEGER,
    kind TEXT,
    tags TEXT
);

CREATE INDEX IF NOT EXISTS feed_fail_history_ts_idx ON feed_fail_history(ts);
CREATE INDEX IF NOT EXISTS feed_fail_history_feed_id_idx ON feed_fail_history(feed_id);

-- Optional per-feed tags (manual/user-defined). Auto-tags are computed on the fly.
CREATE TABLE IF NOT EXISTS feed_tags (
    feed_id TEXT PRIMARY KEY,
    data BLOB
);

-- Optional per-item tags (manual/user-defined). Auto-tags are computed on the fly.
CREATE TABLE IF NOT EXISTS item_tags (
    feed_id TEXT,
    item_id TEXT,
    data BLOB,
    PRIMARY KEY(feed_id, item_id)
);

-- Starred/favorite items (bookmarks)
CREATE TABLE IF NOT EXISTS starred_items (
    feed_id TEXT,
    item_id TEXT,
    starred_at INTEGER,
    PRIMARY KEY(feed_id, item_id)
);

-- Normalized items table to support DB-backed search (QuiteRSS-style).
-- This mirrors the per-feed JSON cache (feed_cache.items) but in a queryable form.
CREATE TABLE IF NOT EXISTS items (
    feed_id TEXT,
    item_id TEXT,
    published_ts INTEGER,
    title TEXT,
    author TEXT,
    link TEXT,
    summary TEXT,
    data BLOB,
    PRIMARY KEY(feed_id, item_id)
);

CREATE INDEX IF NOT EXISTS items_feed_published_idx ON items(feed_id, published_ts DESC);
CREATE INDEX IF NOT EXISTS items_published_idx ON items(published_ts DESC);
'''


def _schema_version(conn):
    try:
        cur = conn.execute("SELECT v FROM meta WHERE k='schema_version'")
        row = cur.fetchone()
        if not row:
            return 0
        try:
            return int(row[0] or 0)
        except Exception:
            return 0
    except Exception:
        return 0


def _sync_items_for_feed(conn, feed_id, entry_dict):
    """Replace items rows for a single feed based on feed_cache entry."""
    fid = str(feed_id or '').strip()
    if not fid:
        return
    try:
        items = list((entry_dict or {}).get('items') or [])
    except Exception:
        items = []

    rows = []
    for it in (items or []):
        if not isinstance(it, dict):
            continue
        try:
            iid = str(it.get('id') or it.get('link') or it.get('title') or '').strip()
        except Exception:
            iid = ''
        if not iid:
            continue
        try:
            ts = int(it.get('published_ts') or 0)
        except Exception:
            ts = 0
        try:
            title = str(it.get('title') or '')
        except Exception:
            title = ''
        try:
            author = str(it.get('author') or '')
        except Exception:
            author = ''
        try:
            link = str(it.get('link') or '')
        except Exception:
            link = ''
        try:
            summary = str(it.get('summary') or '')
        except Exception:
            summary = ''
        try:
            blob = _dumps(dict(it))
        except Exception:
            blob = _dumps({'id': iid, 'title': title, 'author': author, 'link': link, 'summary': summary, 'published_ts': ts})
        rows.append((fid, iid, ts, title, author, link, summary, blob))

    try:
        conn.execute('DELETE FROM items WHERE feed_id=?', (fid,))
    except Exception:
        pass

    if rows:
        try:
            conn.executemany(
                'INSERT OR REPLACE INTO items(feed_id, item_id, published_ts, title, author, link, summary, data) '
                'VALUES(?, ?, ?, ?, ?, ?, ?, ?)',
                rows,
            )
        except Exception:
            # Best-effort: if anything goes wrong, leave items table empty for this feed.
            try:
                conn.execute('DELETE FROM items WHERE feed_id=?', (fid,))
            except Exception:
                pass


def _backfill_items_from_feed_cache(conn):
    """Populate items table from existing feed_cache rows (best-effort)."""
    try:
        cur = conn.execute('SELECT feed_id, data FROM feed_cache')
        rows = cur.fetchall() or []
    except Exception:
        rows = []

    if not rows:
        return

    try:
        conn.execute('BEGIN')
    except Exception:
        pass

    try:
        # Start fresh to avoid duplicate work.
        try:
            conn.execute('DELETE FROM items')
        except Exception:
            pass

        for fid, blob in rows:
            try:
                entry = _loads(blob, default={})
            except Exception:
                entry = {}
            if not isinstance(entry, dict):
                entry = {}
            try:
                _sync_items_for_feed(conn, fid, entry)
            except Exception:
                continue

        try:
            conn.commit()
        except Exception:
            pass
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass


def search_items(feed_ids=None, query='', limit=2000, offset=0):
    """Search items in SQLite (DB-backed search).

    Returns a list of item dicts annotated with:
      - _feed_id, _feed_title, _feed_url
      - _starred (bool)
    """
    try:
        if not DB_READONLY:
            ensure_ready()
    except Exception:
        pass

    q = str(query or '').strip()
    try:
        limit = int(limit or 0)
    except Exception:
        limit = 2000
    try:
        offset = int(offset or 0)
    except Exception:
        offset = 0
    if limit <= 0:
        limit = 2000
    if offset < 0:
        offset = 0

    # Tokenize query on whitespace (QuiteRSS-style "find text").
    try:
        terms = [t for t in q.split() if t]
    except Exception:
        terms = []

    where = []
    params = []
    if feed_ids:
        fids = [str(x) for x in (feed_ids or []) if str(x or '').strip()]
        if fids:
            where.append('i.feed_id IN (%s)' % (','.join(['?'] * len(fids))))
            params.extend(fids)

    # Each term must match one of the searchable fields.
    for t in terms:
        like = '%%%s%%' % t
        where.append('(' + ' OR '.join([
            'i.title LIKE ? ESCAPE "\\"',
            'i.author LIKE ? ESCAPE "\\"',
            'i.summary LIKE ? ESCAPE "\\"',
            'i.link LIKE ? ESCAPE "\\"',
            'f.title LIKE ? ESCAPE "\\"',
            'f.url LIKE ? ESCAPE "\\"',
        ]) + ')')
        params.extend([like, like, like, like, like, like])

    sql = (
        'SELECT i.feed_id, i.item_id, i.published_ts, i.data, '
        'f.title as feed_title, f.url as feed_url, '
        'EXISTS(SELECT 1 FROM starred_items s WHERE s.feed_id=i.feed_id AND s.item_id=i.item_id) as is_starred '
        'FROM items i LEFT JOIN feeds f ON f.id=i.feed_id '
    )
    if where:
        sql += ' WHERE ' + ' AND '.join(where)
    sql += ' ORDER BY i.published_ts DESC LIMIT ? OFFSET ?'
    params.extend([int(limit), int(offset)])

    conn = _connect()
    try:
        try:
            cur = conn.execute(sql, tuple(params))
            rows = cur.fetchall() or []
        except Exception:
            rows = []

        out = []
        for feed_id, item_id, published_ts, blob, feed_title, feed_url, is_starred in (rows or []):
            it = None
            try:
                it = _loads(blob, default=None)
            except Exception:
                it = None
            if not isinstance(it, dict):
                it = {
                    'id': str(item_id or ''),
                    'published_ts': int(published_ts or 0),
                }
            try:
                it['_feed_id'] = str(feed_id or '')
                it['_feed_title'] = str(feed_title or '') if feed_title else ''
                it['_feed_url'] = str(feed_url or '') if feed_url else ''
                it['_starred'] = bool(is_starred)
            except Exception:
                pass
            out.append(it)
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def _ensure_dir():
    try:
        os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
    except Exception:
        pass


def _connect():
    _ensure_dir()
    # Respect read-only flag by using SQLite URI mode (file:... ?mode=ro)
    try:
        if DB_READONLY:
            # Use forward slashes for URI compatibility
            try:
                from pathlib import Path
                p = Path(DB_PATH).absolute().as_posix()
            except Exception:
                p = str(DB_PATH)
            uri = 'file:%s?mode=ro' % p
            conn = sqlite3.connect(uri, uri=True, timeout=30)
        else:
            conn = sqlite3.connect(DB_PATH, timeout=30)
    except Exception:
        conn = sqlite3.connect(DB_PATH, timeout=30)

    try:
        conn.execute('PRAGMA journal_mode=WAL')
    except Exception:
        pass
    try:
        conn.execute('PRAGMA synchronous=NORMAL')
    except Exception:
        pass
    try:
        conn.execute('PRAGMA temp_store=MEMORY')
    except Exception:
        pass
    try:
        conn.execute('PRAGMA busy_timeout=5000')
    except Exception:
        pass
    return conn


def _dumps(obj):
    try:
        raw = json.dumps(obj, ensure_ascii=False, separators=(',', ':')).encode('utf-8')
    except Exception:
        raw = json.dumps(obj).encode('utf-8')
    try:
        return sqlite3.Binary(zlib.compress(raw, 6))
    except Exception:
        return sqlite3.Binary(raw)


def _loads(blob, default=None):
    if blob is None:
        return default
    try:
        b = bytes(blob)
    except Exception:
        b = blob
    try:
        try:
            b = zlib.decompress(b)
        except Exception:
            pass
        return json.loads(b.decode('utf-8'))
    except Exception:
        return default


def db_path():
    return DB_PATH


def init_db():
    conn = _connect()
    try:
        conn.executescript(SCHEMA)
        # For older DBs, ensure new columns exist (best-effort)
        try:
            cur = conn.execute("PRAGMA table_info(feeds)")
            cols = [r[1] for r in (cur.fetchall() or [])]
            if 'oldest_article_days' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN oldest_article_days INTEGER')
                except Exception:
                    pass
            if 'max_articles' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN max_articles INTEGER')
                except Exception:
                    pass
            if 'download_images' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN download_images INTEGER DEFAULT 1')
                except Exception:
                    pass
            if 'always_notify' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN always_notify INTEGER DEFAULT 0')
                except Exception:
                    pass
            if 'feed_starred' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN feed_starred INTEGER DEFAULT 0')
                except Exception:
                    pass
            if 'is_recipe' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN is_recipe INTEGER DEFAULT 0')
                except Exception:
                    pass
            if 'recipe_urn' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN recipe_urn TEXT')
                except Exception:
                    pass
            if 'use_recipe_engine' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN use_recipe_engine INTEGER DEFAULT 0')
                except Exception:
                    pass
            if 'retention_days' not in cols:
                try:
                    conn.execute('ALTER TABLE feeds ADD COLUMN retention_days INTEGER')
                except Exception:
                    pass
        except Exception:
            pass
        # Migrations
        try:
            cur_ver = _schema_version(conn)
        except Exception:
            cur_ver = 0
        if cur_ver < 10:
            try:
                _backfill_items_from_feed_cache(conn)
            except Exception:
                pass

        conn.execute('INSERT OR REPLACE INTO meta(k, v) VALUES(?, ?)', ('schema_version', str(SCHEMA_VERSION)))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def _table_has_rows(conn, table_name):
    try:
        cur = conn.execute('SELECT 1 FROM %s LIMIT 1' % table_name)
        return cur.fetchone() is not None
    except Exception:
        return False


def ensure_ready():
    """Ensure the SQLite database exists and has the right schema.

    This plugin is unreleased (no existing user JSON databases), so we do not
    support any JSON→SQLite migration here.
    """
    init_db()


def get_feeds():
    conn = _connect()
    try:
        # Inspect table columns so we can cope with older DBs lacking newer columns
        try:
            cur = conn.execute("PRAGMA table_info(feeds)")
            cols = [r[1] for r in (cur.fetchall() or [])]
        except Exception:
            cols = []

        out = []

        # Build a projection list based on available columns
        proj = ['id', 'title', 'url', 'folder']
        if 'enabled' in cols:
            proj.append('enabled')
        if 'oldest_article_days' in cols:
            proj.append('oldest_article_days')
        if 'max_articles' in cols:
            proj.append('max_articles')
        if 'download_images' in cols:
            proj.append('download_images')
        if 'always_notify' in cols:
            proj.append('always_notify')
        if 'use_recipe_engine' in cols:
            proj.append('use_recipe_engine')
        if 'feed_starred' in cols:
            proj.append('feed_starred')
        if 'is_recipe' in cols:
            proj.append('is_recipe')
        if 'recipe_urn' in cols:
            proj.append('recipe_urn')
        if 'retention_days' in cols:
            proj.append('retention_days')

        # Preserve insertion order so drag-to-reorder can be persisted.
        sql = 'SELECT %s FROM feeds ORDER BY rowid' % ','.join(proj)
        try:
            cur = conn.execute(sql)
            rows = cur.fetchall() or []
        except sqlite3.OperationalError:
            # If the feeds table is missing expected columns or malformed, fall back
            try:
                # Try minimal select
                cur = conn.execute('SELECT id, title, url, folder FROM feeds')
                rows = cur.fetchall() or []
                proj = ['id', 'title', 'url', 'folder']
            except Exception:
                # Give up gracefully and return empty
                return []

        for row in rows:
            # Map row to fields according to proj
            vals = dict(zip(proj, list(row)))
            fid = str(vals.get('id') or '')
            title = vals.get('title') or ''
            url = vals.get('url') or ''
            folder = vals.get('folder') or ''
            enabled = bool(int(vals.get('enabled') or 0)) if 'enabled' in vals else True
            oldest_article_days = None
            max_articles = None
            try:
                if 'oldest_article_days' in vals and vals.get('oldest_article_days') is not None:
                    oldest_article_days = int(vals.get('oldest_article_days'))
            except Exception:
                oldest_article_days = None
            try:
                if 'max_articles' in vals and vals.get('max_articles') is not None:
                    max_articles = int(vals.get('max_articles'))
            except Exception:
                max_articles = None
            download_images = bool(int(vals.get('download_images') if vals.get('download_images') is not None else 1)) if 'download_images' in vals else True
            always_notify = bool(int(vals.get('always_notify') or 0)) if 'always_notify' in vals else False
            use_recipe_engine = bool(int(vals.get('use_recipe_engine') or 0)) if 'use_recipe_engine' in vals else False
            feed_starred = bool(int(vals.get('feed_starred') or 0)) if 'feed_starred' in vals else False
            is_recipe = bool(int(vals.get('is_recipe') or 0)) if 'is_recipe' in vals else False
            recipe_urn = vals.get('recipe_urn') or ''
            try:
                retention_days = int(vals.get('retention_days') or 0) if 'retention_days' in vals else 0
            except Exception:
                retention_days = 0

            ff = {
                'id': fid,
                'title': title,
                'url': url,
                'folder': folder,
                'enabled': enabled,
                'download_images': download_images,
                'always_notify': always_notify,
                'use_recipe_engine': use_recipe_engine,
                'feed_starred': feed_starred,
                'is_recipe': is_recipe,
                'recipe_urn': recipe_urn,
                'retention_days': retention_days,
            }
            # Only include keys if the DB column is present and value is known.
            if oldest_article_days is not None:
                ff['oldest_article_days'] = oldest_article_days
            if max_articles is not None:
                ff['max_articles'] = max_articles
            out.append(ff)
        if DEBUG_RSS_READER:
            print('[DB] get_feeds returning %d feeds' % len(out))
            for ff in out[:5]:
                print('[DB]   %s -> folder=%r' % (ff.get('title', '')[:25], ff.get('folder', '')))
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def save_feeds(feeds):
    if DEBUG_RSS_READER:
        print('[DB] save_feeds called with %d feeds' % len(feeds or []))
        for f in (feeds or [])[:5]:
            print('[DB]   %s -> folder=%r' % ((f or {}).get('title', '')[:25], (f or {}).get('folder', '')))
    conn = _connect()
    try:
        # Determine available columns and construct appropriate INSERT
        try:
            cur = conn.execute("PRAGMA table_info(feeds)")
            cols = [r[1] for r in (cur.fetchall() or [])]
        except Exception:
            cols = []

        # Which columns we'll write (must include id)
        write_cols = ['id', 'title', 'url', 'folder']
        if 'enabled' in cols:
            write_cols.append('enabled')
        if 'oldest_article_days' in cols:
            write_cols.append('oldest_article_days')
        if 'max_articles' in cols:
            write_cols.append('max_articles')
        if 'download_images' in cols:
            write_cols.append('download_images')
        if 'always_notify' in cols:
            write_cols.append('always_notify')
        if 'use_recipe_engine' in cols:
            write_cols.append('use_recipe_engine')
        if 'feed_starred' in cols:
            write_cols.append('feed_starred')
        if 'is_recipe' in cols:
            write_cols.append('is_recipe')
        if 'recipe_urn' in cols:
            write_cols.append('recipe_urn')
        if 'retention_days' in cols:
            write_cols.append('retention_days')

        placeholders = ','.join(['?'] * len(write_cols))
        sql = 'INSERT OR REPLACE INTO feeds(%s) VALUES(%s)' % (', '.join(write_cols), placeholders)

        conn.execute('BEGIN')
        conn.execute('DELETE FROM feeds')
        for f in (feeds or []):
            try:
                fid = str((f or {}).get('id') or '').strip()
                if not fid:
                    continue
                vals = []
                for c in write_cols:
                    if c == 'id':
                        vals.append(fid)
                    elif c == 'title':
                        vals.append((f or {}).get('title') or '')
                    elif c == 'url':
                        vals.append((f or {}).get('url') or '')
                    elif c == 'folder':
                        vals.append((f or {}).get('folder') or '')
                    elif c == 'enabled':
                        vals.append(1 if bool((f or {}).get('enabled', True)) else 0)
                    elif c == 'oldest_article_days':
                        try:
                            v = (f or {}).get('oldest_article_days')
                            vals.append(int(v) if v is not None else None)
                        except Exception:
                            vals.append(None)
                    elif c == 'max_articles':
                        try:
                            v = (f or {}).get('max_articles')
                            vals.append(int(v) if v is not None else None)
                        except Exception:
                            vals.append(None)
                    elif c == 'download_images':
                        vals.append(1 if bool((f or {}).get('download_images', True)) else 0)
                    elif c == 'always_notify':
                        vals.append(1 if bool((f or {}).get('always_notify', False)) else 0)
                    elif c == 'use_recipe_engine':
                        vals.append(1 if bool((f or {}).get('use_recipe_engine', False)) else 0)
                    elif c == 'feed_starred':
                        vals.append(1 if bool((f or {}).get('feed_starred', False)) else 0)
                    elif c == 'is_recipe':
                        vals.append(1 if bool((f or {}).get('is_recipe', False)) else 0)
                    elif c == 'recipe_urn':
                        vals.append((f or {}).get('recipe_urn') or '')
                    elif c == 'retention_days':
                        try:
                            vals.append(int((f or {}).get('retention_days') or 0))
                        except Exception:
                            vals.append(0)
                    else:
                        vals.append((f or {}).get(c) or '')
                conn.execute(sql, tuple(vals))
            except Exception:
                continue
        conn.commit()
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass
        raise
    finally:
        try:
            conn.close()
        except Exception:
            pass


def update_feed_titles(feed_id_to_title):
    """Update feed titles without rewriting the entire feeds table.

    This is intentionally a narrow UPDATE to avoid clobbering user-driven
    folder/reorder changes when background updates run.
    """
    if not isinstance(feed_id_to_title, dict) or not feed_id_to_title:
        return
    conn = _connect()
    try:
        conn.execute('BEGIN')
        for fid, title in (feed_id_to_title or {}).items():
            try:
                fid = str(fid or '').strip()
                if not fid:
                    continue
                conn.execute('UPDATE feeds SET title=? WHERE id=?', (str(title or ''), fid))
            except Exception:
                continue
        conn.commit()
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass
        raise
    finally:
        try:
            conn.close()
        except Exception:
            pass


def delete_feeds(feed_ids):
    ids = [str(x or '').strip() for x in (feed_ids or []) if str(x or '').strip()]
    if not ids:
        return
    conn = _connect()
    try:
        conn.execute('BEGIN')
        for fid in ids:
            conn.execute('DELETE FROM feeds WHERE id=?', (fid,))
            conn.execute('DELETE FROM seen_item_ids WHERE feed_id=?', (fid,))
            conn.execute('DELETE FROM known_item_ids WHERE feed_id=?', (fid,))
            conn.execute('DELETE FROM feed_cache WHERE feed_id=?', (fid,))
            conn.execute('DELETE FROM feed_tags WHERE feed_id=?', (fid,))
            conn.execute('DELETE FROM item_tags WHERE feed_id=?', (fid,))
            conn.execute('DELETE FROM starred_items WHERE feed_id=?', (fid,))
        conn.commit()
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass
        raise
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_folders():
    conn = _connect()
    try:
        # Preserve insertion order so drag-to-reorder can be persisted.
        cur = conn.execute('SELECT path FROM folders ORDER BY rowid')
        return [str(r[0]) for r in (cur.fetchall() or []) if str(r[0] or '').strip()]
    finally:
        try:
            conn.close()
        except Exception:
            pass


def save_folders(paths):
    conn = _connect()
    try:
        conn.execute('BEGIN')
        conn.execute('DELETE FROM folders')
        for p in (paths or []):
            p = str(p or '').strip().strip('/')
            if not p:
                continue
            conn.execute('INSERT OR IGNORE INTO folders(path) VALUES(?)', (p,))
        conn.commit()
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass
        raise
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_seen_item_ids_map():
    conn = _connect()
    try:
        cur = conn.execute('SELECT feed_id, data FROM seen_item_ids')
        out = {}
        for fid, blob in (cur.fetchall() or []):
            out[str(fid)] = list(_loads(blob, default=[]) or [])
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def set_seen_item_ids(feed_id, ids_list):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    conn = _connect()
    try:
        conn.execute('INSERT OR REPLACE INTO seen_item_ids(feed_id, data) VALUES(?, ?)', (fid, _dumps(list(ids_list or []))))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def delete_seen_item_ids(feed_id):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    conn = _connect()
    try:
        conn.execute('DELETE FROM seen_item_ids WHERE feed_id=?', (fid,))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_feed_cache(feed_id):
    fid = str(feed_id or '').strip()
    if not fid:
        return None
    conn = _connect()
    try:
        cur = conn.execute('SELECT data FROM feed_cache WHERE feed_id=?', (fid,))
        row = cur.fetchone()
        if not row:
            return None
        return _loads(row[0], default=None)
    finally:
        try:
            conn.close()
        except Exception:
            pass


def set_feed_cache(feed_id, entry_dict):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    if entry_dict is None:
        delete_feed_cache(fid)
        return
    # Ensure schema exists so the items mirror table can be updated.
    try:
        if not DB_READONLY:
            ensure_ready()
    except Exception:
        pass
    conn = _connect()
    try:
        try:
            conn.execute('BEGIN')
        except Exception:
            pass
        conn.execute('INSERT OR REPLACE INTO feed_cache(feed_id, data) VALUES(?, ?)', (fid, _dumps(dict(entry_dict))))
        try:
            _sync_items_for_feed(conn, fid, entry_dict)
        except Exception:
            pass
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_known_item_ids_map():
    conn = _connect()
    try:
        cur = conn.execute('SELECT feed_id, data FROM known_item_ids')
        out = {}
        for fid, blob in (cur.fetchall() or []):
            out[str(fid)] = list(_loads(blob, default=[]) or [])
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def set_known_item_ids(feed_id, ids_list):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    conn = _connect()
    try:
        conn.execute('INSERT OR REPLACE INTO known_item_ids(feed_id, data) VALUES(?, ?)', (fid, _dumps(list(ids_list or []))))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def delete_known_item_ids(feed_id):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    conn = _connect()
    try:
        conn.execute('DELETE FROM known_item_ids WHERE feed_id=?', (fid,))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def delete_feed_cache(feed_id):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    conn = _connect()
    try:
        try:
            conn.execute('BEGIN')
        except Exception:
            pass
        conn.execute('DELETE FROM feed_cache WHERE feed_id=?', (fid,))
        try:
            conn.execute('DELETE FROM items WHERE feed_id=?', (fid,))
        except Exception:
            pass
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_feed_status(feed_id):
    """Return status dict for a feed (used for auto-tags), or {}."""
    fid = str(feed_id or '').strip()
    if not fid:
        return {}
    conn = _connect()
    try:
        cur = conn.execute('SELECT data FROM feed_status WHERE feed_id=?', (fid,))
        row = cur.fetchone()
        if not row:
            return {}
        v = _loads(row[0], default={})
        return dict(v) if isinstance(v, dict) else {}
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_feed_status_map():
    """Return {feed_id: status_dict} for all feeds with status rows."""
    conn = _connect()
    try:
        cur = conn.execute('SELECT feed_id, data FROM feed_status')
        out = {}
        for fid, blob in (cur.fetchall() or []):
            try:
                v = _loads(blob, default={})
                out[str(fid)] = dict(v) if isinstance(v, dict) else {}
            except Exception:
                out[str(fid)] = {}
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def set_feed_status(feed_id, status_dict):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    try:
        d = dict(status_dict or {})
    except Exception:
        d = {}
    conn = _connect()
    try:
        conn.execute(
            'INSERT OR REPLACE INTO feed_status(feed_id, data) VALUES(?, ?)',
            (fid, _dumps(d)),
        )
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def delete_feed_status(feed_id):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    conn = _connect()
    try:
        conn.execute('DELETE FROM feed_status WHERE feed_id=?', (fid,))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def purge_feed_fail_history(retention_days=60):
    """Delete failure-history rows older than `retention_days` (best-effort)."""
    try:
        days = int(retention_days or 0)
    except Exception:
        days = 60
    if days <= 0:
        return
    try:
        import time
        cutoff = int(time.time()) - (days * 24 * 60 * 60)
    except Exception:
        cutoff = 0
    if cutoff <= 0:
        return
    conn = _connect()
    try:
        conn.execute('DELETE FROM feed_fail_history WHERE ts IS NOT NULL AND ts < ?', (int(cutoff),))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def clear_feed_fail_history():
    conn = _connect()
    try:
        conn.execute('DELETE FROM feed_fail_history')
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def add_feed_fail_history(feed_id, title='', url='', error='', http_status=None, kind='', tags=None, ts=None):
    # Ensure schema exists so inserts don't silently fail on older DBs.
    try:
        if not DB_READONLY:
            ensure_ready()
    except Exception:
        pass
    fid = str(feed_id or '').strip()
    try:
        title = str(title or '')
    except Exception:
        title = ''
    try:
        url = str(url or '')
    except Exception:
        url = ''
    try:
        error = str(error or '')
    except Exception:
        error = ''
    try:
        kind = str(kind or '')
    except Exception:
        kind = ''
    try:
        if ts is None:
            import time
            ts = int(time.time())
        else:
            ts = int(ts)
    except Exception:
        ts = None
    try:
        if tags is None:
            tags_s = ''
        else:
            tags_s = json.dumps(list(tags or []), ensure_ascii=False)
    except Exception:
        tags_s = ''
    try:
        hs = int(http_status) if http_status is not None else None
    except Exception:
        hs = None

    conn = _connect()
    try:
        conn.execute(
            'INSERT INTO feed_fail_history(ts, feed_id, title, url, error, http_status, kind, tags) VALUES(?, ?, ?, ?, ?, ?, ?, ?)',
            (ts, fid, title, url, error, hs, kind, tags_s),
        )
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def count_feed_fail_history():
    conn = _connect()
    try:
        cur = conn.execute('SELECT COUNT(*) FROM feed_fail_history')
        return int((cur.fetchone() or [0])[0] or 0)
    except Exception:
        return 0
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_feed_fail_history(limit=500, offset=0):
    try:
        if not DB_READONLY:
            ensure_ready()
    except Exception:
        pass
    try:
        limit = int(limit)
    except Exception:
        limit = 500
    try:
        offset = int(offset)
    except Exception:
        offset = 0
    limit = max(1, min(20000, limit))
    offset = max(0, offset)

    conn = _connect()
    try:
        cur = conn.execute(
            'SELECT ts, feed_id, title, url, error, http_status, kind, tags FROM feed_fail_history ORDER BY ts DESC, id DESC LIMIT ? OFFSET ?',
            (limit, offset),
        )
        out = []
        for ts, fid, title, url, error, http_status, kind, tags_s in (cur.fetchall() or []):
            try:
                tags = json.loads(tags_s) if tags_s else []
            except Exception:
                tags = []
            out.append({
                'ts': int(ts) if ts is not None else 0,
                'feed_id': str(fid or ''),
                'title': str(title or ''),
                'url': str(url or ''),
                'error': str(error or ''),
                'http_status': http_status,
                'kind': str(kind or ''),
                'tags': tags,
            })
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def _list_tables(conn):
    try:
        cur = conn.execute("SELECT name FROM sqlite_master WHERE type='table'")
        return [str(r[0]) for r in (cur.fetchall() or []) if r and r[0]]
    except Exception:
        return []


def _table_info(conn, table_name):
    try:
        cur = conn.execute('PRAGMA table_info(%s)' % table_name)
        # rows: cid, name, type, notnull, dflt_value, pk
        return [str(r[1]) for r in (cur.fetchall() or []) if r and len(r) > 1]
    except Exception:
        return []


def _pick_first(cols, preferred):
    s = set([c.lower() for c in (cols or [])])
    for p in preferred:
        if p.lower() in s:
            # return original case
            for c in cols:
                if c.lower() == p.lower():
                    return c
    return ''


def _find_legacy_error_table(conn):
    """Best-effort discovery of a legacy error log table.

    Older plugin versions (or forks) may store failures in a table like
    `error`, `errors`, `error_log`, etc.
    """
    try:
        tables = _list_tables(conn)
    except Exception:
        tables = []
    if not tables:
        return ''

    # Prefer exact/common names first
    preferred = (
        'error', 'errors', 'error_log', 'errors_log',
        'error_history', 'errors_history',
        'feed_errors', 'feed_error',
        'fail_history', 'failure_history', 'feed_failures',
    )
    lower_to_real = {t.lower(): t for t in tables}
    for nm in preferred:
        if nm in lower_to_real and nm != 'feed_fail_history':
            return lower_to_real[nm]

    # Heuristic: any table containing 'error' and having at least one of url/message columns
    for t in tables:
        tl = t.lower()
        if 'error' not in tl:
            continue
        if tl in ('feed_fail_history', 'feeds', 'folders', 'feed_cache', 'feed_status'):
            continue
        cols = [c.lower() for c in (_table_info(conn, t) or [])]
        if not cols:
            continue
        if any(x in cols for x in ('error', 'message', 'details')) and any(x in cols for x in ('url', 'feed_url', 'link')):
            return t

    return ''


def count_error_log_history():
    """Count rows in whichever error-history table is present."""
    # Prefer the new canonical table
    try:
        n = count_feed_fail_history()
        if n:
            return int(n)
    except Exception:
        pass

    conn = _connect()
    try:
        legacy = _find_legacy_error_table(conn)
        if not legacy:
            return 0
        cur = conn.execute('SELECT COUNT(*) FROM %s' % legacy)
        return int((cur.fetchone() or [0])[0] or 0)
    except Exception:
        return 0
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_error_log_history(limit=500, offset=0):
    """Return error log rows, regardless of whether the DB uses the new or legacy table.

    Output rows are dicts with keys: ts, title, url, error.
    """
    try:
        limit = int(limit)
    except Exception:
        limit = 500
    try:
        offset = int(offset)
    except Exception:
        offset = 0
    limit = max(1, min(20000, limit))
    offset = max(0, offset)

    # Prefer canonical history (newer code)
    try:
        rows = get_feed_fail_history(limit=limit, offset=offset) or []
    except Exception:
        rows = []
    if rows:
        out = []
        for r in rows:
            out.append({
                'ts': int(r.get('ts') or 0),
                'title': str(r.get('title') or ''),
                'url': str(r.get('url') or ''),
                'error': str(r.get('error') or ''),
            })
        return out

    # Legacy fallback
    conn = _connect()
    try:
        legacy = _find_legacy_error_table(conn)
        if not legacy:
            return []
        cols = _table_info(conn, legacy)
        if not cols:
            return []

        ts_col = _pick_first(cols, ('ts', 'timestamp', 'time', 'created', 'created_ts', 'date'))
        title_col = _pick_first(cols, ('title', 'feed_title', 'name'))
        url_col = _pick_first(cols, ('url', 'feed_url', 'link'))
        err_col = _pick_first(cols, ('error', 'message', 'details', 'traceback', 'exception'))

        select_cols = []
        for c in (ts_col, title_col, url_col, err_col):
            if c:
                select_cols.append(c)
        if not select_cols:
            return []

        order = ''
        if ts_col:
            order = ' ORDER BY %s DESC' % ts_col
        else:
            order = ' ORDER BY rowid DESC'

        q = 'SELECT %s FROM %s%s LIMIT ? OFFSET ?' % (', '.join(select_cols), legacy, order)
        cur = conn.execute(q, (limit, offset))
        fetched = cur.fetchall() or []

        out = []
        for row in fetched:
            d = {}
            try:
                for i, c in enumerate(select_cols):
                    d[c] = row[i] if i < len(row) else None
            except Exception:
                d = {}

            ts_v = 0
            try:
                if ts_col and d.get(ts_col) is not None:
                    ts_v = int(d.get(ts_col) or 0)
            except Exception:
                ts_v = 0
            out.append({
                'ts': ts_v,
                'title': str(d.get(title_col) or '') if title_col else '',
                'url': str(d.get(url_col) or '') if url_col else '',
                'error': str(d.get(err_col) or '') if err_col else '',
            })

        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def clear_error_log_history():
    """Clear whichever error-history table exists (best-effort)."""
    conn = _connect()
    try:
        # Prefer canonical table
        try:
            conn.execute('DELETE FROM feed_fail_history')
            conn.commit()
            return
        except Exception:
            pass

        legacy = _find_legacy_error_table(conn)
        if legacy:
            try:
                conn.execute('DELETE FROM %s' % legacy)
                conn.commit()
            except Exception:
                pass
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_feed_cache_map():
    conn = _connect()
    try:
        cur = conn.execute('SELECT feed_id, data FROM feed_cache')
        out = {}
        for fid, blob in (cur.fetchall() or []):
            val = _loads(blob, default=None)
            if isinstance(val, dict):
                out[str(fid)] = val
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def purge_orphans(clear_folders_if_no_feeds=True):
    """Remove seen/cache rows whose feed_id no longer exists."""
    conn = _connect()
    try:
        conn.execute('BEGIN')
        conn.execute('DELETE FROM seen_item_ids WHERE feed_id NOT IN (SELECT id FROM feeds)')
        conn.execute('DELETE FROM known_item_ids WHERE feed_id NOT IN (SELECT id FROM feeds)')
        conn.execute('DELETE FROM feed_cache WHERE feed_id NOT IN (SELECT id FROM feeds)')
        conn.execute('DELETE FROM feed_tags WHERE feed_id NOT IN (SELECT id FROM feeds)')
        conn.execute('DELETE FROM item_tags WHERE feed_id NOT IN (SELECT id FROM feeds)')
        if clear_folders_if_no_feeds:
            cur = conn.execute('SELECT COUNT(*) FROM feeds')
            n = int((cur.fetchone() or [0])[0] or 0)
            if n == 0:
                conn.execute('DELETE FROM folders')
        conn.commit()
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass
        raise
    finally:
        try:
            conn.close()
        except Exception:
            pass


def file_size_bytes():
    try:
        if os.path.exists(DB_PATH):
            return int(os.path.getsize(DB_PATH))
    except Exception:
        pass
    return 0


def total_size_bytes():
    """Best-effort on-disk footprint for SQLite main db + wal/shm."""
    total = 0
    for p in (DB_PATH, DB_PATH + '-wal', DB_PATH + '-shm'):
        try:
            if os.path.exists(p):
                total += int(os.path.getsize(p))
        except Exception:
            pass
    return int(total)


def vacuum():
    """Reclaim unused pages and truncate WAL.

    SQLite does not shrink the main database file after DELETE operations;
    a VACUUM is required to return space to the OS.
    """
    conn = _connect()
    try:
        # Ensure WAL content is checkpointed/truncated first (best effort).
        try:
            conn.execute('PRAGMA wal_checkpoint(TRUNCATE)')
        except Exception:
            pass
        try:
            conn.execute('VACUUM')
        except Exception:
            # VACUUM can fail if another connection is open; don't crash the plugin.
            pass
        try:
            conn.execute('PRAGMA optimize')
        except Exception:
            pass
        try:
            conn.commit()
        except Exception:
            pass
    finally:
        try:
            conn.close()
        except Exception:
            pass


def clear_all_data(vacuum_after=False):
    """Delete all user data from the current database.

    This keeps the DB file/path intact, but removes feeds, cached items,
    read/known markers, tags, stars, and failure history.

    If vacuum_after=True, will attempt VACUUM to reclaim disk space.
    """
    conn = _connect()
    try:
        conn.execute('BEGIN')
        for t in (
            'feeds',
            'folders',
            'seen_item_ids',
            'known_item_ids',
            'feed_cache',
            'items',
            'feed_status',
            'feed_fail_history',
            'feed_tags',
            'item_tags',
            'starred_items',
        ):
            try:
                conn.execute('DELETE FROM %s' % t)
            except Exception:
                # Best-effort: continue even if some tables don't exist yet.
                pass
        conn.commit()
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass
        raise
    finally:
        try:
            conn.close()
        except Exception:
            pass

    if vacuum_after:
        try:
            vacuum()
        except Exception:
            # Vacuum can fail due to other open connections; ignore.
            pass


def clear_all_data_at_path(db_path, vacuum_after=False):
    """Delete all RSS Reader data from a specific database file.

    This is intended for DB profile management UIs where the user wants to
    wipe a database without switching to it first.
    """
    p = str(db_path or '').strip()
    if not p:
        raise ValueError('No database path provided')
    if not os.path.exists(p):
        raise FileNotFoundError(p)

    conn = sqlite3.connect(p, timeout=30)
    try:
        conn.execute('BEGIN')
        for t in (
            'feeds',
            'folders',
            'seen_item_ids',
            'known_item_ids',
            'feed_cache',
            'items',
            'feed_status',
            'feed_fail_history',
            'feed_tags',
            'item_tags',
            'starred_items',
        ):
            try:
                conn.execute('DELETE FROM %s' % t)
            except Exception:
                pass
        conn.commit()
    except Exception:
        try:
            conn.rollback()
        except Exception:
            pass
        raise
    finally:
        try:
            conn.close()
        except Exception:
            pass

    if vacuum_after:
        try:
            conn = sqlite3.connect(p, timeout=30)
            try:
                conn.execute('PRAGMA wal_checkpoint(TRUNCATE)')
            except Exception:
                pass
            try:
                conn.execute('VACUUM')
            except Exception:
                pass
            try:
                conn.commit()
            except Exception:
                pass
        finally:
            try:
                conn.close()
            except Exception:
                pass


# ============================================================
# Feed Tags
# ============================================================

def get_feed_tags(feed_id):
    """Return the manual/user-defined tags for a feed as a list."""
    fid = str(feed_id or '').strip()
    if not fid:
        return []
    conn = _connect()
    try:
        cur = conn.execute('SELECT data FROM feed_tags WHERE feed_id=?', (fid,))
        row = cur.fetchone()
        if not row:
            return []
        return list(_loads(row[0], default=[]) or [])
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_feed_tags_map():
    """Return {feed_id: [tag, ...]} for all feeds."""
    conn = _connect()
    try:
        cur = conn.execute('SELECT feed_id, data FROM feed_tags')
        out = {}
        for fid, blob in (cur.fetchall() or []):
            out[str(fid)] = list(_loads(blob, default=[]) or [])
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def set_feed_tags(feed_id, tags_list):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    tags = list(tags_list or [])
    conn = _connect()
    try:
        conn.execute(
            'INSERT OR REPLACE INTO feed_tags(feed_id, data) VALUES(?, ?)',
            (fid, _dumps(tags)),
        )
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def delete_feed_tags(feed_id):
    fid = str(feed_id or '').strip()
    if not fid:
        return
    conn = _connect()
    try:
        conn.execute('DELETE FROM feed_tags WHERE feed_id=?', (fid,))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_item_tags_map(feed_id=None):
    """Return a mapping of item tags.

    If feed_id is provided, returns {item_id: [tag, ...]} for that feed.
    If feed_id is None, returns {(feed_id, item_id): [tag, ...]} for all feeds.
    """
    conn = _connect()
    try:
        if feed_id is None:
            cur = conn.execute('SELECT feed_id, item_id, data FROM item_tags')
            out = {}
            for fid, iid, blob in (cur.fetchall() or []):
                out[(str(fid), str(iid))] = list(_loads(blob, default=[]) or [])
            return out

        fid = str(feed_id or '').strip()
        if not fid:
            return {}
        cur = conn.execute('SELECT item_id, data FROM item_tags WHERE feed_id=?', (fid,))
        out = {}
        for iid, blob in (cur.fetchall() or []):
            out[str(iid)] = list(_loads(blob, default=[]) or [])
        return out
    finally:
        try:
            conn.close()
        except Exception:
            pass


def set_item_tags(feed_id, item_id, tags_list):
    fid = str(feed_id or '').strip()
    iid = str(item_id or '').strip()
    if not fid or not iid:
        return
    tags = list(tags_list or [])
    conn = _connect()
    try:
        conn.execute(
            'INSERT OR REPLACE INTO item_tags(feed_id, item_id, data) VALUES(?, ?, ?)',
            (fid, iid, _dumps(tags)),
        )
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def delete_item_tags(feed_id, item_id):
    fid = str(feed_id or '').strip()
    iid = str(item_id or '').strip()
    if not fid or not iid:
        return
    conn = _connect()
    try:
        conn.execute('DELETE FROM item_tags WHERE feed_id=? AND item_id=?', (fid, iid))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


# ============================================================
# Starred/Favorite Items
# ============================================================

def get_starred_items_map(feed_id=None):
    """Return starred items.

    If feed_id is provided, returns set of item_ids that are starred for that feed.
    If feed_id is None, returns {feed_id: set(item_id, ...)} for all feeds.
    """
    conn = _connect()
    try:
        if feed_id is None:
            cur = conn.execute('SELECT feed_id, item_id FROM starred_items')
            out = {}
            for fid, iid in (cur.fetchall() or []):
                fid = str(fid)
                if fid not in out:
                    out[fid] = set()
                out[fid].add(str(iid))
            return out

        fid = str(feed_id or '').strip()
        if not fid:
            return set()
        cur = conn.execute('SELECT item_id FROM starred_items WHERE feed_id=?', (fid,))
        return set(str(row[0]) for row in (cur.fetchall() or []))
    finally:
        try:
            conn.close()
        except Exception:
            pass


def is_item_starred(feed_id, item_id):
    fid = str(feed_id or '').strip()
    iid = str(item_id or '').strip()
    if not fid or not iid:
        return False
    conn = _connect()
    try:
        cur = conn.execute('SELECT 1 FROM starred_items WHERE feed_id=? AND item_id=?', (fid, iid))
        return cur.fetchone() is not None
    finally:
        try:
            conn.close()
        except Exception:
            pass


def set_item_starred(feed_id, item_id, starred=True):
    fid = str(feed_id or '').strip()
    iid = str(item_id or '').strip()
    if not fid or not iid:
        return
    conn = _connect()
    try:
        if starred:
            import time
            conn.execute(
                'INSERT OR REPLACE INTO starred_items(feed_id, item_id, starred_at) VALUES(?, ?, ?)',
                (fid, iid, int(time.time())),
            )
        else:
            conn.execute('DELETE FROM starred_items WHERE feed_id=? AND item_id=?', (fid, iid))
        conn.commit()
    finally:
        try:
            conn.close()
        except Exception:
            pass


def get_all_starred_item_ids():
    """Return a set of (feed_id, item_id) tuples for all starred items."""
    conn = _connect()
    try:
        cur = conn.execute('SELECT feed_id, item_id FROM starred_items')
        return set((str(fid), str(iid)) for fid, iid in (cur.fetchall() or []))
    finally:
        try:
            conn.close()
        except Exception:
            pass
