from __future__ import absolute_import

import ast


_FEED_URL_TO_RECIPE_URN = {}


def find_recipe_urn_for_feed_url(feed_url, include_custom=False, max_scan=1200):
    """Best-effort: find a calibre recipe URN whose declared `feeds` includes `feed_url`.

    This lets the plugin apply recipe-specific scraping/cleanup even when the
    user added the feed as a plain RSS URL.

    Returns a URN like `builtin:financial_times` or '' if not found.
    """
    try:
        u = str(feed_url or '').strip()
    except Exception:
        u = ''
    if not u:
        return ''

    # Cache hit
    try:
        hit = _FEED_URL_TO_RECIPE_URN.get(u)
        if hit is not None:
            return hit
    except Exception:
        pass

    # Fast-path: FT feeds are common and known
    try:
        from urllib.parse import urlparse
        host = (urlparse(u).netloc or '').lower()
    except Exception:
        host = ''
    if host.endswith('ft.com'):
        try:
            _FEED_URL_TO_RECIPE_URN[u] = 'builtin:financial_times'
        except Exception:
            pass
        return 'builtin:financial_times'

    # Scan builtin recipe collection for a recipe that declares this feed URL
    try:
        from calibre.web.feeds.recipes.collection import get_builtin_recipe_collection
        root = get_builtin_recipe_collection()
        n = 0
        # Check if root is iterable; avoid truth-testing XML elements (FutureWarning)
        try:
            recipes = list(root) if hasattr(root, '__iter__') and root is not None else []
        except Exception:
            recipes = []
        for r in recipes:
            try:
                urn = str(r.get('id') or '').strip()
            except Exception:
                urn = ''
            if not urn:
                continue
            n += 1
            if max_scan and n > int(max_scan):
                break
            try:
                src = get_recipe_source_from_urn(urn)
            except Exception:
                continue
            feeds = extract_feeds_from_recipe_source(src) or []
            for _t, fu in feeds:
                try:
                    if str(fu or '').strip() == u:
                        _FEED_URL_TO_RECIPE_URN[u] = urn
                        return urn
                except Exception:
                    continue
    except Exception:
        pass

    # Optionally scan custom recipes too (disabled by default to avoid disk I/O)
    if include_custom:
        try:
            from calibre.web.feeds.recipes.collection import get_custom_recipe_collection
            root = get_custom_recipe_collection()
            # Check if root is iterable; avoid truth-testing XML elements (FutureWarning)
            try:
                recipes = list(root) if hasattr(root, '__iter__') and root is not None else []
            except Exception:
                recipes = []
            for r in recipes:
                try:
                    urn = str(r.get('id') or '').strip()
                except Exception:
                    urn = ''
                if not urn:
                    continue
                try:
                    src = get_recipe_source_from_urn(urn)
                except Exception:
                    continue
                feeds = extract_feeds_from_recipe_source(src) or []
                for _t, fu in feeds:
                    try:
                        if str(fu or '').strip() == u:
                            _FEED_URL_TO_RECIPE_URN[u] = urn
                            return urn
                    except Exception:
                        continue
        except Exception:
            pass

    try:
        _FEED_URL_TO_RECIPE_URN[u] = ''
    except Exception:
        pass
    return ''


def is_recipe_urn(url):
    try:
        u = str(url or '').strip()
    except Exception:
        u = ''
    return u.startswith('builtin:') or u.startswith('custom:')


def get_recipe_source_from_urn(urn, download_latest=False, log=None):
    """Return recipe python source (unicode string) for a calibre recipe URN.

    Supports builtin:... and custom:...
    """
    try:
        u = str(urn or '').strip()
    except Exception:
        u = ''
    if not u:
        raise ValueError('Empty recipe URN')

    from calibre.web.feeds.recipes.collection import get_builtin_recipe_by_id, get_custom_recipe

    if u.startswith('builtin:'):
        return get_builtin_recipe_by_id(u, log=log, download_recipe=bool(download_latest))
    if u.startswith('custom:'):
        rid = u.split(':', 1)[1]
        return get_custom_recipe(rid)

    raise ValueError('Unsupported recipe URN: %s' % u)


def get_recipe_class_from_urn(urn, download_latest=False, log=None):
    src = get_recipe_source_from_urn(urn, download_latest=download_latest, log=log)
    if not src:
        raise ValueError('No recipe source for: %s' % (urn,))

    from calibre.web.feeds.recipes import compile_recipe

    recipe_class = compile_recipe(src)
    if recipe_class is None:
        raise ValueError('Failed to compile recipe: %s' % (urn,))
    return recipe_class


def _normalize_feeds_list(feeds_value):
    """Normalize a feeds declaration into [(title, url), ...]."""
    out = []
    if not feeds_value:
        return out
    if not isinstance(feeds_value, (list, tuple)):
        return out

    for f in feeds_value:
        try:
            # Common: (title, url)
            if isinstance(f, (list, tuple)):
                title = f[0] if len(f) > 0 else ''
                url = f[1] if len(f) > 1 else ''
            else:
                title = ''
                url = f
            title = str(title or '').strip()
            url = str(url or '').strip()
            if url:
                out.append((title, url))
        except Exception:
            continue

    # De-dup while preserving order
    seen = set()
    deduped = []
    for t, u in out:
        if u in seen:
            continue
        seen.add(u)
        deduped.append((t, u))
    return deduped


def extract_feeds_from_recipe_source(src):
    """Try to statically extract a class attribute named `feeds` from recipe source.

    This works for many recipes (including BasicUserRecipe) without executing the recipe,
    and avoids needing to instantiate BasicNewsRecipe (which requires options/log).
    """
    try:
        if not isinstance(src, str):
            src = str(src or '')
    except Exception:
        src = ''

    if not src.strip():
        return []

    try:
        tree = ast.parse(src)
    except Exception:
        return []

    feeds_nodes = []
    for node in tree.body:
        if isinstance(node, ast.ClassDef):
            for stmt in node.body:
                if isinstance(stmt, ast.Assign):
                    for tgt in stmt.targets:
                        if isinstance(tgt, ast.Name) and tgt.id == 'feeds':
                            feeds_nodes.append(stmt.value)
                elif isinstance(stmt, ast.AnnAssign):
                    if isinstance(stmt.target, ast.Name) and stmt.target.id == 'feeds':
                        feeds_nodes.append(stmt.value)

    out = []
    for v in feeds_nodes:
        try:
            lit = ast.literal_eval(v)
        except Exception:
            continue
        out.extend(_normalize_feeds_list(lit))

    return _normalize_feeds_list(out)


def get_recipe_feeds_from_urn(urn, download_latest=False, log=None):
    """Return a list of (title, url) RSS feeds declared by the recipe.

    Many recipes are *scrapers* and do not declare RSS feeds. In that case we raise
    a ValueError with a helpful message.
    """

    # First: try static extraction from source (safe, no execution).
    src = get_recipe_source_from_urn(urn, download_latest=download_latest, log=log)
    out = extract_feeds_from_recipe_source(src)
    if out:
        return out

    # Fallback: compile and inspect the class attribute (still avoids instantiating).
    # We do not instantiate the recipe: calibre 8+ requires options/log/progress_reporter.
    recipe_class = get_recipe_class_from_urn(urn, download_latest=download_latest, log=log)
    try:
        feeds_attr = getattr(recipe_class, 'feeds', None)
    except Exception:
        feeds_attr = None
    out = _normalize_feeds_list(feeds_attr)
    if out:
        return out

    raise ValueError('Recipe does not declare RSS feeds (it likely scrapes web pages). Use “Fetch recipe now” instead: %s' % (urn,))
