__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import QAction
except Exception:
    from PyQt5.Qt import QAction

import os
from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader import rss_db
from calibre_plugins.rss_reader.dialogs import ProfilesDialog


class ProfilesMixin:
    def _get_db_feed_count(self, db_path):
        """Return the number of feeds in the given DB path, or None on error.

        Uses a small in-memory cache keyed by (db_path, mtime) so opening the
        Profiles menu is fast even with many profiles.
        """
        import os
        try:
            p = str(db_path or '').strip()
        except Exception:
            p = ''
        if not p:
            return None
        # Critical safety: never create a new SQLite DB just by opening menus.
        try:
            if not os.path.isfile(p):
                return None
        except Exception:
            return None

        try:
            mtime = os.path.getmtime(p)
        except Exception:
            mtime = None

        try:
            cache = getattr(self, '_db_feed_count_cache', None)
            if not isinstance(cache, dict):
                cache = {}
                setattr(self, '_db_feed_count_cache', cache)
        except Exception:
            cache = {}

        try:
            if mtime is not None:
                cached = cache.get(p)
                if cached and cached[0] == mtime:
                    return cached[1]
        except Exception:
            pass

        import sqlite3
        try:
            # Prefer read-only URI mode so this can't create/modify the DB.
            try:
                uri = 'file:%s?mode=ro' % (p.replace('\\', '/'),)
                conn = sqlite3.connect(uri, uri=True, timeout=2)
            except Exception:
                # Fallback: regular connect (should still be safe due to isfile check above)
                conn = sqlite3.connect(p, timeout=2)
            try:
                cur = conn.execute('SELECT COUNT(*) FROM feeds')
                row = cur.fetchone()
                count = int(row[0]) if row else 0
            finally:
                conn.close()
        except Exception:
            return None

        try:
            if mtime is not None:
                cache[p] = (mtime, count)
        except Exception:
            pass
        return count

    def _get_db_mtime(self, db_path):
        """Return the last modified time (as a string) for the DB file, or None on error."""
        import os, time
        try:
            ts = os.path.getmtime(db_path)
            # Format as YYYY-MM-DD HH:MM
            return time.strftime('%Y-%m-%d %H:%M', time.localtime(ts))
        except Exception:
            return None
    # ----- Profile helpers -----
    def _profiles_load(self):
        try:
            return list(plugin_prefs.get('db_profiles', []) or [])
        except Exception:
            return []

    def _profiles_save(self, profiles):
        try:
            plugin_prefs['db_profiles'] = list(profiles or [])
        except Exception:
            pass

    def _get_profile(self, pid):
        try:
            for p in (self._profiles_load() or []):
                if str(p.get('id') or '') == str(pid or ''):
                    return p
        except Exception:
            pass
        return None

    def _set_active_profile(self, pid, persist=True):
        try:
            p = self._get_profile(pid)
            if p is None:
                return
            try:
                if persist:
                    plugin_prefs['db_profiles_active'] = str(pid)
            except Exception:
                pass
            try:
                self._db_prev_path = rss_db.db_path()
                try:
                    self._db_prev_readonly = bool(getattr(rss_db, 'DB_READONLY', False))
                except Exception:
                    self._db_prev_readonly = False
            except Exception:
                self._db_prev_path = None
            try:
                readonly = bool(p.get('readonly', False) or p.get('mirror', False))
                # Mark this as an explicit user-initiated switch so we don't
                # treat it as an automatic fallback to the Config DB.
                try:
                    setattr(self, '_allow_config_db_switch', True)
                except Exception:
                    pass
                try:
                    self._switch_database(str(p.get('path') or ''), readonly=readonly)
                finally:
                    try:
                        setattr(self, '_allow_config_db_switch', False)
                    except Exception:
                        pass
            except Exception:
                pass
            try:
                self._update_profile_label()
            except Exception:
                pass
        except Exception:
            pass

    def _rebuild_profiles_menu(self, menu=None):
        try:
            m = menu if menu is not None else getattr(self, '_profiles_menu', None)
            if m is None:
                return
            m.clear()
            try:
                active_id = str(plugin_prefs.get('db_profiles_active') or '')
            except Exception:
                active_id = ''
            try:
                cur_path = str(rss_db.db_path() or '').strip()
            except Exception:
                cur_path = ''
            # Dedicated entry: configured Default DB (aka "Current DB")
            try:
                try:
                    default_path = str(plugin_prefs.get('db_default_path') or '').strip()
                except Exception:
                    default_path = ''
                if default_path:
                    try:
                        d_name = str(plugin_prefs.get('db_default_name') or '').strip()
                    except Exception:
                        d_name = ''
                    try:
                        d_emoji = str(plugin_prefs.get('db_default_emoji') or '').strip()
                    except Exception:
                        d_emoji = ''
                    try:
                        d_ro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
                    except Exception:
                        d_ro = False
                    label = (d_emoji + ' ' if d_emoji else '') + (d_name or _('Current DB'))
                    try:
                        fc = self._get_db_feed_count(default_path) if default_path else None
                    except Exception:
                        fc = None
                    if fc is not None:
                        label += f'  [{fc} feeds]'
                    a_def = QAction(label, m)
                    try:
                        tip = str(default_path)
                        if fc is not None:
                            tip += f'\nFeeds: {fc}'
                        try:
                            if d_ro:
                                tip = tip + '\n' + _('Mode: Read-only')
                            else:
                                tip = tip + '\n' + _('Mode: Default')
                        except Exception:
                            pass
                        a_def.setToolTip(tip)
                    except Exception:
                        pass
                    try:
                        a_def.setCheckable(True)
                        # When no active profile, consider "default" selected if current path matches.
                        a_def.setChecked((not bool(active_id)) and (cur_path and cur_path == default_path))
                    except Exception:
                        pass
                    a_def.triggered.connect(self._restore_default_database)
                    m.addAction(a_def)
            except Exception:
                pass
            # Dedicated entry: built-in (plugin) DB
            try:
                cfg_path = str(getattr(self, '_db_config_path', '') or '').strip()
            except Exception:
                cfg_path = ''
            try:
                if cfg_path:
                    # Only show when it differs from the configured Default DB path.
                    try:
                        default_path = str(getattr(self, '_db_orig_path', '') or '').strip()
                    except Exception:
                        default_path = ''
                    if default_path and str(cfg_path).strip() == default_path:
                        raise Exception('redundant')
                    # Do not show the suggested-default entry in the Profiles submenu.
                    # It remains visible/usable in the Manage Profiles dialog.
            except Exception:
                pass

            profiles = self._profiles_load() or []
            try:
                default_emoji = str(plugin_prefs.get('db_profile_default_emoji') or '📰').strip()
            except Exception:
                default_emoji = '📰'
            if not profiles:
                a = QAction(_('No profiles'), m)
                a.setEnabled(False)
                m.addAction(a)
            else:
                for p in profiles:
                    try:
                        emoji = str(p.get('emoji') or '').strip()
                    except Exception:
                        emoji = ''
                    if not emoji:
                        emoji = default_emoji
                    name = (emoji + ' ' if emoji else '') + (p.get('name') or p.get('path') or '')
                    db_path = p.get('path') or ''
                    feed_count = self._get_db_feed_count(db_path) if db_path else None
                    # Compose label: Name [N feeds] (no mtime in toolbar)
                    label = name
                    if feed_count is not None:
                        label += f'  [{feed_count} feeds]'
                    act = QAction(label, m)
                    act.setData(str(p.get('id') or ''))
                    try:
                        tip = str(db_path)
                        if feed_count is not None:
                            tip += f'\nFeeds: {feed_count}'
                        if bool(p.get('mirror', False)):
                            tip = tip + '\n' + _('Mode: Mirror (read-only)')
                        elif bool(p.get('readonly', False)):
                            tip = tip + '\n' + _('Mode: Read-only')
                        else:
                            tip = tip + '\n' + _('Mode: Writable')
                        act.setToolTip(tip)
                    except Exception:
                        pass
                    try:
                        act.setCheckable(True)
                        act.setChecked(str(p.get('id') or '') == str(active_id or ''))
                    except Exception:
                        pass
                    act.triggered.connect(lambda checked, aid=str(p.get('id') or ''): self._set_active_profile(aid))
                    m.addAction(act)
            # After listing quick-switch profile actions, add extra management actions
            try:
                m.addSeparator()
            except Exception:
                pass
            try:
                a_rename = QAction(_('Rename current profile…'), m)
                def _on_rename_current_profile():
                    try:
                        try:
                            from qt.core import QInputDialog, QMessageBox
                        except Exception:
                            from PyQt5.Qt import QInputDialog, QMessageBox

                        # Determine current DB path
                        try:
                            cur_path = str(rss_db.db_path() or '').strip()
                        except Exception:
                            cur_path = ''

                        # 1) Prefer explicitly active profile
                        try:
                            active_id = str(plugin_prefs.get('db_profiles_active') or '')
                        except Exception:
                            active_id = ''
                        p = self._get_profile(active_id) if active_id else None

                        profiles_all = self._profiles_load() or []

                        # 2) Fallback: match current DB path to a saved profile
                        if p is None and cur_path:
                            try:
                                for _p in profiles_all:
                                    try:
                                        if str(_p.get('path') or '').strip() == cur_path:
                                            p = _p
                                            break
                                    except Exception:
                                        continue
                            except Exception:
                                p = None

                        is_default_db = False
                        cur_name = ''

                        if p is not None:
                            # Renaming a saved profile
                            cur_name = str(p.get('name') or p.get('path') or '')
                        else:
                            # 3) If current DB is the configured Default DB, rename its label
                            try:
                                default_path = str(plugin_prefs.get('db_default_path') or '').strip()
                            except Exception:
                                default_path = ''
                            if not default_path:
                                try:
                                    default_path = str(getattr(self, '_db_orig_path', '') or '').strip()
                                except Exception:
                                    default_path = ''
                            if default_path and cur_path and cur_path == default_path:
                                is_default_db = True
                                try:
                                    cur_name = str(plugin_prefs.get('db_default_name') or '').strip()
                                except Exception:
                                    cur_name = ''
                                if not cur_name:
                                    cur_name = _('Current DB')
                            else:
                                QMessageBox.information(self, _('Rename profile'), _('No active profile to rename.'))
                                return

                        new_name, ok = QInputDialog.getText(self, _('Rename profile'), _('Enter new profile name:'), text=cur_name)
                        if not ok:
                            return
                        new_name = str(new_name or '').strip()
                        if not new_name:
                            QMessageBox.information(self, _('Rename profile'), _('Profile name cannot be empty.'))
                            return

                        # Persist change
                        if is_default_db:
                            try:
                                plugin_prefs['db_default_name'] = new_name
                            except Exception:
                                pass
                        else:
                            try:
                                for i, _p in enumerate(profiles_all):
                                    try:
                                        if str(_p.get('id') or '') == str(p.get('id') or ''):
                                            profiles_all[i] = dict(_p)
                                            profiles_all[i]['name'] = new_name
                                            break
                                    except Exception:
                                        continue
                                self._profiles_save(profiles_all)
                            except Exception as e:
                                QMessageBox.warning(self, _('Rename profile'), _('Failed to rename profile: %s') % str(e))
                                return

                        try:
                            self._rebuild_profiles_menu()
                        except Exception:
                            pass
                        try:
                            self._update_profile_label()
                        except Exception:
                            pass
                    except Exception:
                        pass
                a_rename.triggered.connect(_on_rename_current_profile)
                a_rename.setToolTip(_('Rename the currently active profile.'))
                m.addAction(a_rename)
            except Exception:
                pass
            # Add Manage profiles action so it's present in the Profiles toolbar menu
            try:
                a_manage = QAction(_('Manage profiles…'), m)
                try:
                    a_manage.setToolTip(_('Open the DB Profiles manager. Shortcut is configurable in calibre Preferences → Keyboard shortcuts (group: RSS Reader).'))
                except Exception:
                    pass
                a_manage.triggered.connect(self._show_profiles_manager)
                m.addAction(a_manage)
            except Exception:
                pass
            # 'Clear current DB' intentionally omitted from Profiles quick menu
        except Exception:
            pass

    def _update_profile_label(self):
        try:
            active_id = str(plugin_prefs.get('db_profiles_active') or '')
        except Exception:
            active_id = ''
        try:
            default_emoji = str(plugin_prefs.get('db_profile_default_emoji') or '📰').strip()
        except Exception:
            default_emoji = '📰'
        try:
            p = self._get_profile(active_id) if active_id else None
            cur_path = ''
            try:
                cur_path = str(rss_db.db_path() or '').strip()
            except Exception:
                cur_path = ''

            if p is not None:
                try:
                    emoji = str(p.get('emoji') or '').strip()
                except Exception:
                    emoji = ''
                if not emoji:
                    emoji = default_emoji
                name = str(p.get('name') or p.get('path') or '').strip()
                mode = ''
                try:
                    if bool(p.get('mirror', False)):
                        mode = _('mirror')
                    elif bool(p.get('readonly', False)):
                        mode = _('read-only')
                    else:
                        mode = _('writable')
                except Exception:
                    mode = ''
                text = (emoji + ' ' if emoji else '') + (name or _('(profile)'))
                tip = (cur_path or '')
                if mode:
                    tip = (tip + '\n' if tip else '') + _('Mode: %s') % mode
            else:
                # If there's no active profile, try to display a saved profile
                # that matches the current DB path (so emoji/name appear).
                try:
                    profiles = self._profiles_load() or []
                except Exception:
                    profiles = []
                match = None
                try:
                    for _p in profiles:
                        try:
                            if (
                                str(_p.get('path') or '').strip()
                                and cur_path
                                and str(_p.get('path') or '').strip() == cur_path
                            ):
                                match = _p
                                break
                        except Exception:
                            continue
                except Exception:
                    match = None

                if match is not None:
                    try:
                        emoji = str(match.get('emoji') or '').strip()
                    except Exception:
                        emoji = ''
                    if not emoji:
                        emoji = default_emoji
                    name = str(match.get('name') or match.get('path') or '').strip()
                    mode = ''
                    try:
                        if bool(match.get('mirror', False)):
                            mode = _('Mirror (read-only)')
                        elif bool(match.get('readonly', False)):
                            mode = _('Read-only')
                        else:
                            mode = _('Default')
                    except Exception:
                        mode = ''
                    text = (emoji + ' ' if emoji else '') + (name or _('(profile)'))
                    try:
                        prof_line = (emoji + ' ' if emoji else '') + (name or _('(profile)'))
                    except Exception:
                        prof_line = name or ''
                    tip_lines = []
                    tip_lines.append(_('Current DB: %s') % (cur_path or ''))
                    tip_lines.append(_('Current profile: %s') % (prof_line or _('None')))
                    if mode:
                        tip_lines.append(_('Mode: %s') % mode)
                    tip = '\n'.join([x for x in tip_lines if x])
                else:
                    try:
                        default_path = str(getattr(self, '_db_orig_path', '') or '').strip()
                    except Exception:
                        default_path = ''
                    try:
                        cfg_path = str(getattr(self, '_db_config_path', '') or '').strip()
                    except Exception:
                        cfg_path = ''
                    # Treat the configured Default DB as primary; only show a
                    # separate "Config DB" label when it really differs from
                    # the saved default path.
                    if default_path and cur_path and cur_path == default_path:
                        try:
                            d_name = str(plugin_prefs.get('db_default_name') or '').strip()
                        except Exception:
                            d_name = ''
                        try:
                            d_emoji = str(plugin_prefs.get('db_default_emoji') or '').strip()
                        except Exception:
                            d_emoji = ''
                        disp_name = d_name or _('Current DB')
                        text = (d_emoji + ' ' if d_emoji else '') + disp_name
                        prof_line = (d_emoji + ' ' if d_emoji else '') + disp_name
                        tip_lines = [
                            _('Current DB: %s') % (cur_path or ''),
                            _('Current profile: %s') % (prof_line or _('None')),
                        ]
                        try:
                            if bool(getattr(self, '_db_default_readonly', False)):
                                tip_lines.append(_('Mode: %s') % _('Read-only'))
                            else:
                                tip_lines.append(_('Mode: %s') % _('Default'))
                        except Exception:
                            pass
                        tip = '\n'.join([x for x in tip_lines if x])
                    elif cfg_path and cur_path and cur_path == cfg_path and (not default_path or cfg_path != default_path):
                        text = _('Config DB')
                        tip = '\n'.join([
                            _('Current DB: %s') % (cur_path or ''),
                            _('Current profile: %s') % _('None'),
                            _('Mode: %s') % _('Default'),
                        ])
                    else:
                        text = _('Current DB')
                        tip = '\n'.join([
                            _('Current DB: %s') % (cur_path or ''),
                            _('Current profile: %s') % _('None'),
                        ])

            # This label should only appear when the builtin/suggested DB is
            # active instead of the configured Default DB. At this point the
            # codebase no longer performs any implicit fallback, so we only
            # keep a lightweight diagnostic hook here.
            if text == _('Config DB'):
                try:
                    if getattr(self, '_allow_config_db_switch', False):
                        try:
                            setattr(self, '_allow_config_db_switch', False)
                        except Exception:
                            pass
                    else:
                        import traceback
                        # Log the unexpected label so issues can be diagnosed,
                        # but do not block or modify the UI.
                        print('WARNING: Config DB label shown without explicit switch!\n' + ''.join(traceback.format_stack()))
                except Exception:
                    pass
            try:
                # Keep this single-line so footer doesn't get jumbled
                self.profile_label.setText(text)
            except Exception:
                pass
            try:
                self.profile_label.setToolTip(tip or '')
            except Exception:
                pass
        except Exception:
            pass

    def _show_profiles_manager(self):
        try:
            dlg = ProfilesDialog(self)
            dlg.exec()
            try:
                self._rebuild_profiles_menu()
            except Exception:
                pass
            try:
                self._update_profile_label()
            except Exception:
                pass
        except Exception:
            pass

    def _switch_back_database(self):
        try:
            from qt.core import QMessageBox
        except Exception:
            from PyQt5.Qt import QMessageBox
        try:
            if not getattr(self, '_db_prev_path', None):
                msg = 'No previous DB to switch back to.'
                self.status.setText(msg)
                QMessageBox.information(self, _('Info'), msg)
                return
            try:
                plugin_prefs['db_profiles_active'] = ''
            except Exception:
                pass
            self._switch_database(self._db_prev_path, readonly=bool(getattr(self, '_db_prev_readonly', False)))
        except Exception as e:
            self.status.setText('Switch back failed: ' + str(e))

    def _restore_default_database(self):
        try:
            from qt.core import QMessageBox
        except Exception:
            from PyQt5.Qt import QMessageBox
        try:
            # Default DB may be configured outside calibre config dir
            try:
                dpath = str(plugin_prefs.get('db_default_path') or '').strip()
            except Exception:
                dpath = ''
            try:
                dro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
            except Exception:
                dro = False
            if not dpath:
                dpath = getattr(self, '_db_orig_path', None)
            if not dpath:
                msg = 'No original DB path recorded.'
                self.status.setText(msg)
                QMessageBox.information(self, _('Info'), msg)
                return
            try:
                plugin_prefs['db_profiles_active'] = ''
            except Exception:
                pass
            # Explicit user action: allow switching to configured default DB
            try:
                setattr(self, '_allow_config_db_switch', True)
            except Exception:
                pass
            try:
                try:
                    self._switch_database(str(dpath), readonly=bool(dro))
                finally:
                    try:
                        setattr(self, '_allow_config_db_switch', False)
                    except Exception:
                        pass
            except Exception:
                pass
            try:
                self._update_profile_label()
            except Exception:
                pass
        except Exception as e:
            self.status.setText('Restore failed: ' + str(e))
