from __future__ import absolute_import, division, print_function, unicode_literals

import hashlib
import os
import queue
import shutil
import threading

try:
    from calibre.constants import cache_dir
except Exception:
    cache_dir = None
from calibre_plugins.rss_reader.config import plugin_prefs

try:
    from qt.core import (
        Qt,
        QAction,
        QApplication,
        QFileDialog,
        QImage,
        QMenu,
        QMessageBox,
        QTextBrowser,
        QTextDocument,
        QTimer,
        QSlider,
        QWidget,
        QUrl,
    )
except Exception:
    from PyQt5.Qt import (
        Qt,
        QAction,
        QApplication,
        QFileDialog,
        QImage,
        QMenu,
        QMessageBox,
        QTextBrowser,
        QTextDocument,
        QTimer,
        QSlider,
        QWidget,
        QUrl,
    )

try:
    load_translations()
except NameError:
    pass

from urllib.parse import urljoin as _urljoin, urlparse as _urlparse
from calibre_plugins.rss_reader.rss import fetch_url
from calibre.utils.imghdr import what as _imghdr_what


def _audio_debug_enabled():
    try:
        if bool(plugin_prefs.get('debug_audio_player', False)):
            return True
    except Exception:
        pass
    try:
        import os as _os
        v = str(_os.environ.get('RSS_READER_AUDIO_DEBUG', '') or '').strip().lower()
        return v in ('1', 'true', 'yes', 'on')
    except Exception:
        return False


def _sanitize_url_for_fetch(u):
    """Best-effort URL cleanup before fetch_url()."""
    try:
        if hasattr(u, 'toString'):
            s = str(u.toString())
        else:
            s = str(u)
    except Exception:
        return u

    s = (s or '').strip()
    if not s:
        return ''

    # Handle protocol-relative URLs (e.g. //addons.mozilla.org/...) by
    # defaulting to HTTPS so the Preview can fetch them as http(s) resources.
    try:
        if s.startswith('//'):
            s = 'https:' + s
    except Exception:
        pass

    # Common HTML entity in feeds
    try:
        s = s.replace('&amp;', '&')
    except Exception:
        pass

    if s.lower().startswith('data:'):
        return s

    # IMPORTANT: do not rewrite/query-encode signed URLs.
    # Many CDNs (e.g. GitHub private-user-images / S3 signed URLs) include
    # signatures/JWTs in the query string where *any* encoding changes can
    # invalidate the URL.
    try:
        from urllib.parse import urlsplit, urlunsplit, quote

        parts = urlsplit(s)

        # Only normalize the path; keep query/fragment intact.
        path = quote(parts.path or '', safe='/%:@-._~')

        query = parts.query or ''
        if ' ' in query:
            query = query.replace(' ', '%20')

        fragment = parts.fragment or ''
        if ' ' in fragment:
            fragment = fragment.replace(' ', '%20')

        return urlunsplit((parts.scheme, parts.netloc, path, query, fragment))
    except Exception:
        # Fallback for the most common breakage: spaces.
        try:
            return s.replace(' ', '%20')
        except Exception:
            return s


def _normalize_images_for_preview(html_str, base_url='', preserve_local=False):
    """Normalize <img> tags for preview.

    - Moves common lazy-load attributes into src
    - Resolves relative URLs against base_url
    - Optionally preserves already-local paths (exported HTML)
    """

    try:
        from lxml import html as _lhtml
    except Exception:
        return html_str or ''

    try:
        frags = _lhtml.fragments_fromstring(html_str or '')
    except Exception:
        try:
            root = _lhtml.fragment_fromstring(html_str or '', create_parent='div')
            frags = [root]
        except Exception:
            return html_str or ''

    def _first_url_from_srcset(srcset_text):
        try:
            s = (srcset_text or '').strip()
            if not s:
                return ''
            first = s.split(',', 1)[0].strip()
            return first.split()[0].strip()
        except Exception:
            return ''

    import re as _re

    for node in frags:
        imgs = []
        try:
            if getattr(node, 'tag', None) == 'img':
                imgs.append(node)
        except Exception:
            pass
        try:
            imgs.extend(list(node.xpath('.//img') or []))
        except Exception:
            pass

        # Deduplicate images within the fragment.
        # Some sites emit the same <img> multiple times or include WordPress size variants.
        # Keep the best candidate for a canonical URL.
        # Value: (score, img_element)
        seen = {}


        for img in imgs:
            try:
                src = (img.get('src') or '').strip()
            except Exception:
                src = ''

            if not src:
                for k in ('data-src', 'data-original', 'data-lazy-src', 'data-lazy', 'data-url'):
                    try:
                        v = (img.get(k) or '').strip()
                    except Exception:
                        v = ''
                    if v:
                        src = v
                        try:
                            img.set('src', v)
                        except Exception:
                            pass
                        break

            if not src:
                try:
                    srcset = img.get('srcset')
                except Exception:
                    srcset = None
                if srcset:
                    src = _first_url_from_srcset(srcset)
                    if src:
                        try:
                            img.set('src', src)
                        except Exception:
                            pass

            if not src:
                continue

            is_local = False
            if preserve_local:
                try:
                    s_l = src.lower()
                    is_local = not (s_l.startswith('http://') or s_l.startswith('https://') or s_l.startswith('data:'))
                except Exception:
                    is_local = False

            try:
                abs_url = src if is_local else (_urljoin(base_url or '', src) if base_url else src)
            except Exception:
                abs_url = src

            try:
                abs_url = _sanitize_url_for_fetch(abs_url)
            except Exception:
                pass

            # Canonicalize for dedup (strip query/fragment and WordPress -WxH suffix).
            try:
                p = _urlparse(abs_url)
                abs_no_q = p._replace(query='', fragment='').geturl()
            except Exception:
                abs_no_q = abs_url
            try:
                canonical = _re.sub(r'(-\d+x\d+)(?=\.[a-z0-9]+(?:$|\?))', '', abs_no_q, count=1, flags=_re.IGNORECASE)
            except Exception:
                canonical = abs_no_q

            # Score: prefer non-suffixed URL; then prefer larger declared area.
            try:
                has_size_suffix = bool(_re.search(r'-\d+x\d+(?=\.[a-z0-9]+(?:[?#]|$))', abs_no_q, flags=_re.IGNORECASE))
            except Exception:
                has_size_suffix = False
            try:
                w = int(img.get('width') or 0)
            except Exception:
                w = 0
            try:
                h = int(img.get('height') or 0)
            except Exception:
                h = 0
            area = 0
            try:
                if w > 0 and h > 0:
                    area = w * h
            except Exception:
                area = 0
            score = (0 if has_size_suffix else 1, area)

            prev = seen.get(canonical) if canonical else None
            if prev is not None:
                prev_score, prev_img = prev
                if score > prev_score:
                    try:
                        prev_img.getparent().remove(prev_img)
                    except Exception:
                        pass
                    seen[canonical] = (score, img)
                else:
                    try:
                        img.getparent().remove(img)
                    except Exception:
                        pass
                    continue
            elif canonical:
                seen[canonical] = (score, img)

            if not is_local:
                try:
                    img.set('src', abs_url)
                except Exception:
                    pass

            # srcset often triggers remote loads; keep it simple
            try:
                img.attrib.pop('srcset', None)
            except Exception:
                pass

            # Inject Calibre-style CSS for image fit
            try:
                style = img.get('style') or ''
                if 'max-width' not in style:
                    style = style.rstrip(';') + ';max-width:100%;height:auto;width:auto;display:block;'
                    img.set('style', style)
                # Strip explicit width/height HTML attributes so QTextBrowser
                # does not skew images (it honours the attributes and ignores
                # the CSS aspect-ratio).
                for attr in ('width', 'height'):
                    try:
                        img.attrib.pop(attr, None)
                    except Exception:
                        pass
            except Exception:
                pass

        # Replace YouTube iframes with clickable placeholders
        try:
            iframes = node.xpath('.//iframe')
        except Exception:
            iframes = []
        for iframe in iframes:
            try:
                src = (iframe.get('src') or '').strip()
                if 'youtube.com' in src or 'youtu.be' in src:
                    a = _lhtml.fromstring(
                        '<div class="youtube-placeholder"><a href="%s" target="_blank">▶ YouTube video: %s</a></div>'
                        % (src, src)
                    )
                    iframe.addprevious(a)
                    iframe.getparent().remove(iframe)
            except Exception:
                pass

        try:
            links = node.xpath('.//a')
        except Exception:
            links = []
        for a in links:
            try:
                href = (a.get('href') or '').strip()
                if ('youtube.com' in href or 'youtu.be' in href) and not a.text_content().strip():
                    a.text = '▶ YouTube video: %s' % href
            except Exception:
                pass

    try:
        return ''.join(_lhtml.tostring(x, encoding='unicode') for x in frags)
    except Exception:
        return html_str or ''


def _process_images_for_export(html_str, base_url, td, do_download=True, images_subdir=None):
    try:
        from calibre_plugins.rss_reader.export_images import process_images_for_export
    except Exception:
        return html_str or ''

    return process_images_for_export(
        html_str=html_str,
        base_url=base_url,
        td=td,
        do_download=do_download,
        images_subdir=images_subdir,
        sanitize_url_for_fetch=_sanitize_url_for_fetch,
        plugin_prefs=plugin_prefs,
        calibre_cache_dir=(cache_dir() if callable(cache_dir) else ''),
    )


class PreviewBrowser(QTextBrowser):
    def __init__(self, parent=None):
        super().__init__(parent)
        self._image_cache = {}
        self._zoom_callback = None

        # Set custom context menu policy to ensure our contextMenuEvent is called
        try:
            self.setContextMenuPolicy(Qt.ContextMenuPolicy.DefaultContextMenu)
        except Exception:
            try:
                self.setContextMenuPolicy(Qt.DefaultContextMenu)
            except Exception:
                pass

        # Async image loading (avoid blocking the UI thread on network fetches)
        base = ''
        try:
            base = cache_dir() if callable(cache_dir) else ''
        except Exception:
            base = ''
        self._img_cache_dir = os.path.join(base, 'plugins', 'rss_reader', 'img_cache') if base else ''
        try:
            if self._img_cache_dir:
                os.makedirs(self._img_cache_dir, exist_ok=True)
        except Exception:
            pass
        self._img_lock = threading.RLock()
        self._img_gen = 0
        self._img_inflight = set()
        self._img_pending = []
        self._img_active = 0
        self._img_max_concurrency = 4
        # Increase max image bytes to allow larger AVIFs to download (12MB)
        self._img_max_bytes = 12 * 1024 * 1024
        self._img_results = queue.Queue()
        # Cache whether Pillow is available to avoid repeated import attempts
        self._pil_available = None
        self._img_timer = QTimer(self)
        self._img_timer.setInterval(100)
        # Defer lookup of _drain_image_results until the timer fires so
        # the method need not exist at this exact point during class
        # construction/import in all runtime contexts.
        try:
            self._img_timer.timeout.connect(lambda: getattr(self, '_drain_image_results', lambda: None)())
        except Exception:
            # Fallback for bindings that don't accept lambda connects
            try:
                self._img_timer.timeout.connect(self._drain_image_results)
            except Exception:
                pass
        try:
            self._img_timer.start()
        except Exception:
            pass

    def _is_image_blocked(self, url):
        # AdBlock support has been removed; always allow images.
        return False

    def _block_image(self, url):
        # No-op: AdBlock removed.
        return False

    def _block_domain(self, url):
        # No-op: AdBlock removed.
        return False

    def _unblock_image(self, url):
        # No-op: AdBlock removed.
        return False

    def start_new_page(self):
        # Bump generation to ignore any late image downloads from previous pages.
        with self._img_lock:
            self._img_gen += 1
            self._img_pending = []
            self._img_inflight = set()
            self._img_active = 0
            # Clear current page image list
            try:
                self._current_img_urls = []
            except Exception:
                pass

    def _image_fetch_headers(self):
        headers = {
            # Accept WebP explicitly so CDNs like imgproxy serve the format they have.
            # Qt's QImage or Pillow (fallback) can decode WebP on most setups.
            'Accept': 'image/webp,image/jpeg,image/png,image/gif;q=0.9,*/*;q=0.1',
            # Avoid brotli/gzip for images: some hosts send br unexpectedly and our
            # low-level fetcher only guarantees gzip handling.
            'Accept-Encoding': 'identity',
        }
        try:
            ref = str(getattr(self, '_page_base_url', '') or '')
            if ref:
                headers['Referer'] = ref
        except Exception:
            pass
        return headers

    def _img_cache_path_for_url(self, url):
        try:
            h = hashlib.sha1(str(url).encode('utf-8', 'ignore')).hexdigest()
        except Exception:
            h = str(abs(hash(url)))
        if not self._img_cache_dir:
            return ''
        return os.path.join(self._img_cache_dir, h + '.img')

    def _pump_image_downloads(self):
        # Start background downloads up to concurrency limit.
        while True:
            with self._img_lock:
                if self._img_active >= int(self._img_max_concurrency or 1):
                    return
                if not self._img_pending:
                    return
                url, gen = self._img_pending.pop(0)
                if gen != self._img_gen:
                    continue
                if url in self._img_inflight:
                    continue
                self._img_inflight.add(url)
                self._img_active += 1

            def _worker(u=url, g=gen):
                raw = None
                err = None
                try:
                    from calibre_plugins.rss_reader.rss import fetch_url
                    u_fetch = _sanitize_url_for_fetch(u)
                    raw, _hdr = fetch_url(u_fetch, timeout_seconds=12, headers=self._image_fetch_headers())
                    if raw and self._img_max_bytes and len(raw) > int(self._img_max_bytes):
                        raw = None
                except Exception as e:
                    err = e
                    raw = None

                try:
                    if raw:
                        p = self._img_cache_path_for_url(u)
                        try:
                            with open(p, 'wb') as f:
                                f.write(raw)
                        except Exception:
                            pass
                except Exception:
                    pass

                try:
                    self._img_results.put((g, u, raw, err))
                except Exception:
                    pass

                with self._img_lock:
                    try:
                        self._img_inflight.discard(u)
                    except Exception:
                        pass
                    try:
                        self._img_active = max(0, int(self._img_active) - 1)
                    except Exception:
                        self._img_active = 0

                # Do not touch Qt objects/timers from this worker thread.
                # The main-thread drain loop will kick off more downloads.

            t = threading.Thread(target=_worker, name='rss_reader_img', daemon=True)
            t.start()

    def _drain_image_results(self):
        # Main-thread: apply downloaded images to the current document.
        try:
            processed_any = False
            while True:
                try:
                    g, url, raw, _err = self._img_results.get_nowait()
                except Exception:
                    break

                if g != self._img_gen:
                    continue

                processed_any = True

                if not raw:
                    # Try disk cache fallback if present.
                    try:
                        p = self._img_cache_path_for_url(url)
                        if os.path.exists(p):
                            with open(p, 'rb') as f:
                                raw = f.read()
                    except Exception:
                        raw = None

                if not raw:
                    continue

                img = QImage()
                try:
                    img.loadFromData(raw)
                except Exception:
                    img = QImage()

                # If QImage couldn't decode (common for SVG/AVIF/WebP), try fallbacks.
                if img.isNull():
                    # Try Pillow -> PNG conversion if available.
                    try:
                        from io import BytesIO
                        if self._pil_available is None:
                            try:
                                import PIL  # noqa: F401
                                self._pil_available = True
                            except Exception:
                                self._pil_available = False
                        if self._pil_available:
                            try:
                                from PIL import Image as PILImage
                                bio = BytesIO(raw)
                                pil = PILImage.open(bio)
                                pil = pil.convert('RGBA')
                                out = BytesIO()
                                pil.save(out, format='PNG')
                                png = out.getvalue()
                                tmp = QImage()
                                tmp.loadFromData(png)
                                if not tmp.isNull():
                                    img = tmp
                            except Exception:
                                pass
                    except Exception:
                        pass

                if img.isNull():
                    # Try rendering SVG via QSvgRenderer if available.
                    try:
                        try:
                            from qt.core import QSvgRenderer, QByteArray, QPainter
                        except Exception:
                            from PyQt5.QtSvg import QSvgRenderer
                            from PyQt5.QtCore import QByteArray
                            from PyQt5.QtGui import QPainter

                        qb = QByteArray(raw)
                        renderer = QSvgRenderer(qb)
                        dsize = renderer.defaultSize()
                        if not dsize or dsize.isEmpty():
                            w, h = 800, 600
                        else:
                            w, h = int(dsize.width() or 800), int(dsize.height() or 600)
                        w = min(w, 1600)
                        h = min(h, 1600)
                        tmp = QImage(w, h, QImage.Format_ARGB32)
                        try:
                            tmp.fill(0)
                        except Exception:
                            pass
                        p = QPainter(tmp)
                        try:
                            renderer.render(p)
                        finally:
                            try:
                                p.end()
                            except Exception:
                                pass
                        if not tmp.isNull():
                            img = tmp
                    except Exception:
                        pass

                if img.isNull():
                    # If still not decodable, optionally show a neutral placeholder for AVIF/WebP.
                    try:
                        lowu = (str(url) or '').lower()
                    except Exception:
                        lowu = ''

                    is_avif = False
                    is_webp = False
                    try:
                        if raw and b'ftypavif' in raw[:64]:
                            is_avif = True
                    except Exception:
                        pass
                    try:
                        if raw and len(raw) >= 12 and raw[:4] == b'RIFF' and raw[8:12] == b'WEBP':
                            is_webp = True
                    except Exception:
                        pass

                    def _placeholder(msg):
                        try:
                            w, h = 360, 140
                            ph = QImage(w, h, QImage.Format_ARGB32)
                            try:
                                from qt.core import QColor, QRect, QPainter, Qt as _Qt
                            except Exception:
                                from PyQt5.QtGui import QColor, QPainter
                                from PyQt5.QtCore import QRect, Qt as _Qt
                            try:
                                ph.fill(QColor(240, 240, 240))
                            except Exception:
                                pass
                            p = QPainter(ph)
                            try:
                                try:
                                    p.setPen(QColor(80, 80, 80))
                                except Exception:
                                    pass
                                rect = QRect(0, 0, w, h)
                                try:
                                    p.drawText(rect, int(_Qt.AlignCenter | _Qt.TextWordWrap), msg)
                                except Exception:
                                    try:
                                        p.drawText(rect, msg)
                                    except Exception:
                                        pass
                            finally:
                                try:
                                    p.end()
                                except Exception:
                                    pass
                            return ph
                        except Exception:
                            return QImage()

                    if ('avif' in lowu) or is_avif:
                        img = _placeholder('Image not displayed: AVIF format not supported')
                    elif ('webp' in lowu) or is_webp:
                        img = _placeholder('Image not displayed: WebP format not supported')

                    if img.isNull():
                        continue

                # Cache in memory (best-effort cap).
                try:
                    self._image_cache[url] = img
                    if len(self._image_cache) > 128:
                        try:
                            self._image_cache.clear()
                        except Exception:
                            pass
                except Exception:
                    pass

                try:
                    self.document().addResource(QTextDocument.ResourceType.ImageResource, QUrl(url), img)
                    try:
                        self.document().markContentsDirty(0, max(1, int(self.document().characterCount() or 1)))
                    except Exception:
                        pass
                    try:
                        self.viewport().update()
                    except Exception:
                        pass
                except Exception:
                    pass

            # If we processed results, kick the pump to start more downloads.
            try:
                if processed_any:
                    self._pump_image_downloads()
            except Exception:
                pass
        except Exception:
            pass

    def set_zoom_callback(self, callback):
        self._zoom_callback = callback

    def wheelEvent(self, event):
        try:
            mods = event.modifiers() if hasattr(event, 'modifiers') else None
            ctrl = bool(mods & Qt.KeyboardModifier.ControlModifier) if mods is not None else False
        except Exception:
            ctrl = False

        if ctrl and callable(getattr(self, '_zoom_callback', None)):
            dy = 0
            try:
                dy = int(event.angleDelta().y())
            except Exception:
                try:
                    dy = int(event.delta())
                except Exception:
                    dy = 0
            if dy:
                try:
                    self._zoom_callback(1 if dy > 0 else -1)
                except Exception:
                    pass
                try:
                    event.accept()
                except Exception:
                    pass
                return

        return super().wheelEvent(event)

    def loadResource(self, res_type, name):
        try:
            if res_type == QTextDocument.ResourceType.ImageResource:
                try:
                    if not bool(plugin_prefs.get('load_images_in_preview', True)):
                        return QImage()
                except Exception:
                    pass
                try:
                    url = name.toString() if hasattr(name, 'toString') else str(name)
                except Exception:
                    url = str(name)

                if not url:
                    return super().loadResource(res_type, name)

                # Check AdBlock
                if self._is_image_blocked(url):
                    try:
                        w, h = 200, 100
                        blocked_img = QImage(w, h, QImage.Format_ARGB32)
                        try:
                            from qt.core import QColor, QRect, QPainter, Qt as _Qt
                        except Exception:
                            from PyQt5.QtGui import QColor, QPainter
                            from PyQt5.QtCore import QRect, Qt as _Qt

                        try:
                            blocked_img.fill(QColor(255, 220, 220))
                        except Exception:
                            pass

                        p = QPainter(blocked_img)
                        try:
                            try:
                                p.setPen(QColor(150, 0, 0))
                            except Exception:
                                pass
                            rect = QRect(0, 0, w, h)
                            msg = '🚫 Image blocked by AdBlock'
                            try:
                                p.drawText(rect, int(_Qt.AlignCenter), msg)
                            except Exception:
                                try:
                                    p.drawText(rect, msg)
                                except Exception:
                                    pass
                        finally:
                            try:
                                p.end()
                            except Exception:
                                pass

                        return blocked_img
                    except Exception:
                        return QImage()

                cached = self._image_cache.get(url)
                if cached is not None:
                    return cached

                # Disk cache
                try:
                    p = self._img_cache_path_for_url(url)
                    if os.path.exists(p):
                        with open(p, 'rb') as f:
                            raw = f.read()
                        img = QImage()
                        img.loadFromData(raw)
                        if not img.isNull():
                            self._image_cache[url] = img
                            return img
                except Exception:
                    pass

                # Only allow http(s) images
                if url.lower().startswith('http://') or url.lower().startswith('https://'):
                    # Fetch asynchronously; return a placeholder now.
                    try:
                        with self._img_lock:
                            g = self._img_gen
                            if url not in self._img_inflight:
                                self._img_pending.append((url, g))
                        self._pump_image_downloads()
                    except Exception:
                        pass
                    return QImage()

                return super().loadResource(res_type, name)
        except Exception:
            return super().loadResource(res_type, name)

    def _image_url_at_pos(self, pos):
        candidates = []
        try:
            candidates.append(pos)
        except Exception:
            pass
        try:
            vp = self.viewport() if hasattr(self, 'viewport') else None
            if vp is not None:
                try:
                    candidates.append(vp.mapFrom(self, pos))
                except Exception:
                    pass
        except Exception:
            pass

        for p in candidates:
            try:
                cursor = self.cursorForPosition(p)
                fmt = cursor.charFormat()
                if hasattr(fmt, 'isImageFormat') and fmt.isImageFormat():
                    try:
                        img_fmt = fmt.toImageFormat()
                        return str(img_fmt.name() or '')
                    except Exception:
                        return ''
            except Exception:
                continue

        return ''

    def add_rss_reader_context_actions(self, menu, pos):
        added_any = False

        try:
            img_url = self._image_url_at_pos(pos)
        except Exception:
            img_url = ''

        def _copy_text(text):
            try:
                QApplication.clipboard().setText(text or '')
            except Exception:
                pass

        def _open_in_browser(u):
            try:
                if not u:
                    return
                try:
                    from qt.core import QDesktopServices, QUrl
                except Exception:
                    from PyQt5.Qt import QDesktopServices, QUrl
                QDesktopServices.openUrl(QUrl(str(u)))
            except Exception:
                pass

        def _get_image_for_url(u):
            try:
                img = self._image_cache.get(u)
            except Exception:
                img = None
            if img is not None:
                return img
            try:
                p = self._img_cache_path_for_url(u)
                if p and os.path.exists(p):
                    with open(p, 'rb') as f:
                        raw = f.read()
                    tmp = QImage()
                    tmp.loadFromData(raw)
                    if not tmp.isNull():
                        return tmp
            except Exception:
                pass
            return None

        def _save_image_from_url(u):
            try:
                import re
                url_ext = ''
                try:
                    m = re.search(r'\.([a-zA-Z0-9]+)(?:[?#].*)?$', u)
                    if m:
                        url_ext = m.group(1).lower()
                        if url_ext not in ('jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'svg', 'avif'):
                            url_ext = ''
                except Exception:
                    url_ext = ''

                p = None
                try:
                    p = self._img_cache_path_for_url(u)
                    if p and os.path.exists(p):
                        base = os.path.splitext(os.path.basename(p))[0]
                    else:
                        p = None
                        base = os.path.splitext(os.path.basename(u))[0]
                except Exception:
                    p = None
                    base = os.path.splitext(os.path.basename(u))[0]

                if url_ext:
                    default = f"{base}.{url_ext}"
                else:
                    default = os.path.basename(p) if p else os.path.basename(u)

                try:
                    last_dir = plugin_prefs.get('last_image_save_dir', None)
                except Exception:
                    last_dir = None
                if last_dir and os.path.isdir(last_dir):
                    default_path = os.path.join(last_dir, default)
                else:
                    default_path = default

                fname, _dummy = QFileDialog.getSaveFileName(self, _('Save image as'), default_path)
                if not fname:
                    return

                try:
                    plugin_prefs['last_image_save_dir'] = os.path.dirname(fname)
                except Exception:
                    pass

                if p and os.path.exists(p):
                    shutil.copyfile(p, fname)
                    return

                try:
                    raw, _hdr = fetch_url(u, timeout_seconds=12, headers=self._image_fetch_headers())
                    if raw:
                        with open(fname, 'wb') as f:
                            f.write(raw)
                        return
                except Exception:
                    pass

                try:
                    img_obj = _get_image_for_url(u)
                    if img_obj is not None and not img_obj.isNull():
                        img_obj.save(fname)
                except Exception:
                    pass
            except Exception:
                pass

        try:
            feed_url = str(getattr(self, '_current_feed_url', '') or '')
        except Exception:
            feed_url = ''
        if feed_url:
            try:
                menu.addSeparator()
            except Exception:
                pass
            act_copy_feed = QAction(_('Copy feed URL'), self)
            act_copy_feed.triggered.connect(lambda: _copy_text(feed_url))
            menu.addAction(act_copy_feed)
            added_any = True

        if img_url:
            try:
                menu.addSeparator()
            except Exception:
                pass

            act_open = QAction(_('Open image in browser'), self)
            act_copy_img = QAction(_('Copy image'), self)
            act_copy_url = QAction(_('Copy image URL'), self)
            act_save_img = QAction(_('Save image as...'), self)

            def _copy_img():
                try:
                    img = _get_image_for_url(img_url)
                    if img is not None:
                        QApplication.clipboard().setImage(img)
                except Exception:
                    pass

            act_open.triggered.connect(lambda _checked=False, u=img_url: _open_in_browser(str(u)))
            act_copy_img.triggered.connect(_copy_img)
            act_copy_url.triggered.connect(lambda: _copy_text(img_url))
            act_save_img.triggered.connect(lambda: _save_image_from_url(img_url))

            menu.addAction(act_open)
            menu.addAction(act_copy_img)
            menu.addAction(act_copy_url)
            menu.addAction(act_save_img)
            return True

        try:
            urls = getattr(self, '_current_img_urls', None) or []
        except Exception:
            urls = []

        if urls:
            try:
                menu.addSeparator()
            except Exception:
                pass
            images_menu = menu.addMenu(_('Images'))
            for u in list(urls)[:20]:
                try:
                    label = str(u)
                    if len(label) > 72:
                        label = '...%s' % label[-69:]
                except Exception:
                    label = _('Image')
                sub = images_menu.addMenu(label)
                a_open = QAction(_('Open image in browser'), self)
                a_copy_url = QAction(_('Copy image URL'), self)
                a_save = QAction(_('Save image as...'), self)
                a_open.triggered.connect(lambda _checked=False, uu=u: _open_in_browser(str(uu)))
                a_copy_url.triggered.connect(lambda _checked=False, uu=u: _copy_text(str(uu)))
                a_save.triggered.connect(lambda _checked=False, uu=u: _save_image_from_url(str(uu)))
                sub.addAction(a_open)
                sub.addAction(a_copy_url)
                sub.addAction(a_save)
            added_any = True

        return added_any


class ClickableSlider(QSlider):
    """QSlider that seeks to the clicked position on mouse press.

    This makes clicking on the slider groove jump to that position like
    typical media players.
    """
    def mousePressEvent(self, event):
        try:
            if self.orientation() == Qt.Horizontal:
                x = event.pos().x()
                w = max(1, self.width())
                pct = float(x) / float(w)
                mn = int(self.minimum() or 0)
                mx = int(self.maximum() or 0)
                val = int(mn + pct * (mx - mn))
                # Update the slider and emit moved/changed signals so
                # connected handlers perform seeking.
                try:
                    self.setValue(val)
                except Exception:
                    pass
                try:
                    self.sliderMoved.emit(int(val))
                except Exception:
                    try:
                        self.valueChanged.emit(int(val))
                    except Exception:
                        pass
                return
        except Exception:
            pass
        return super().mousePressEvent(event)


class AudioPlayer(QWidget):
    """Minimal audio playback widget using Qt multimedia APIs.

    Provides basic play/pause/stop and a position slider for a single URL.
    """
    def __init__(self, parent=None):
        try:
            from qt.core import QMediaPlayer, QAudioOutput, QPushButton, QHBoxLayout, QLabel, QSlider, QSizePolicy, QUrl
        except Exception:
            try:
                from PyQt5.QtWidgets import QPushButton, QHBoxLayout, QLabel, QSlider
                from PyQt5.QtMultimedia import QMediaPlayer
                from PyQt5.QtCore import QUrl
            except Exception:
                QPushButton = None
                QHBoxLayout = None
                QLabel = None
                QSlider = None
                QMediaPlayer = None
                QUrl = None

        super().__init__(parent)

        self._sources = []
        self._current = 0

        self._duration = 0
        self._position = 0

        self._dbg_enabled = _audio_debug_enabled()

        def _dbg(msg):
            if not self._dbg_enabled:
                return
            try:
                print('RSS_Reader AudioPlayer:', msg)
            except Exception:
                pass

        self._dbg = _dbg

        # UI
        try:
            self.play_btn = QPushButton('▶', self)
            self.stop_btn = QPushButton('■', self)
            self.label = QLabel('', self)
            self.time_label = QLabel('00:00 / 00:00', self)
            self.slider = ClickableSlider(self)
            try:
                self.slider.setOrientation(Qt.Horizontal)
            except Exception:
                try:
                    self.slider.setOrientation(1)
                except Exception:
                    pass
            lay = QHBoxLayout(self)
            lay.setContentsMargins(4, 2, 4, 2)
            lay.addWidget(self.play_btn)
            lay.addWidget(self.stop_btn)
            lay.addWidget(self.label)
            lay.addWidget(self.slider)
            lay.addWidget(self.time_label)
            # Ensure the slider remains visible even when the label text is long:
            try:
                if QSizePolicy is not None:
                    # Label should take minimal space; slider expands to fill remaining area.
                    self.label.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
                    self.slider.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
                    self.time_label.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
            except Exception:
                pass
        except Exception:
            # Minimal fallback
            from PyQt5.QtWidgets import QPushButton, QHBoxLayout, QLabel
            self.play_btn = QPushButton('▶', self)
            self.stop_btn = QPushButton('■', self)
            self.label = QLabel('', self)
            self.time_label = QLabel('00:00 / 00:00', self)
            from PyQt5.QtWidgets import QSlider as _QSlider
            self.slider = _QSlider(self)
            lay = QHBoxLayout(self)
            lay.addWidget(self.play_btn)
            lay.addWidget(self.stop_btn)
            lay.addWidget(self.label)
            lay.addWidget(self.slider)
            lay.addWidget(self.time_label)

        # Multimedia backend
        try:
            from qt.core import QMediaPlayer, QAudioOutput
            self._player = QMediaPlayer(self)
            try:
                self._audio_out = QAudioOutput(self)
                try:
                    self._player.setAudioOutput(self._audio_out)
                except Exception:
                    pass
            except Exception:
                self._audio_out = None
        except Exception:
            try:
                from PyQt5.QtMultimedia import QMediaPlayer
                self._player = QMediaPlayer(self)
                self._audio_out = None
            except Exception:
                self._player = None
                self._audio_out = None

        # Connect signals
        try:
            if self._player is not None:
                try:
                    self._player.positionChanged.connect(self._on_position)
                except Exception:
                    pass
                try:
                    self._player.durationChanged.connect(self._on_duration)
                except Exception:
                    pass
                # Optional debug hooks for Qt5/Qt6
                try:
                    if hasattr(self._player, 'errorOccurred'):
                        self._player.errorOccurred.connect(lambda e=None: self._dbg('errorOccurred: %r' % (e,)))
                except Exception:
                    pass
                try:
                    if hasattr(self._player, 'error'):
                        self._player.error.connect(lambda e=None: self._dbg('error: %r' % (e,)))
                except Exception:
                    pass
                try:
                    if hasattr(self._player, 'mediaStatusChanged'):
                        self._player.mediaStatusChanged.connect(lambda s=None: self._dbg('mediaStatus: %r' % (s,)))
                except Exception:
                    pass
                try:
                    if hasattr(self._player, 'stateChanged'):
                        self._player.stateChanged.connect(lambda s=None: self._dbg('stateChanged: %r' % (s,)))
                except Exception:
                    pass
                try:
                    if hasattr(self._player, 'playbackStateChanged'):
                        self._player.playbackStateChanged.connect(lambda s=None: self._dbg('playbackStateChanged: %r' % (s,)))
                except Exception:
                    pass
        except Exception:
            pass

        try:
            self.play_btn.clicked.connect(self._toggle_play)
        except Exception:
            pass
        try:
            self.stop_btn.clicked.connect(self.stop)
        except Exception:
            pass
        try:
            self.slider.sliderMoved.connect(self._on_slider_moved)
        except Exception:
            try:
                self.slider.valueChanged.connect(self._on_slider_moved)
            except Exception:
                pass

        self._update_time_label()

    def _toggle_play(self):
        try:
            if self._player is None:
                try:
                    self._dbg('toggle_play: no player')
                except Exception:
                    pass
                return

            # Determine whether currently playing (Qt5/Qt6).
            playing = False
            try:
                if callable(getattr(self._player, 'playbackState', None)):
                    st = self._player.playbackState()
                    try:
                        playing = 'playing' in str(st).lower()
                    except Exception:
                        playing = False
                else:
                    # Qt5: QMediaPlayer.state() + QMediaPlayer.PlayingState
                    try:
                        from PyQt5.QtMultimedia import QMediaPlayer as _QMP
                    except Exception:
                        _QMP = None
                    try:
                        st = self._player.state()
                    except Exception:
                        st = None
                    if _QMP is not None:
                        try:
                            playing = st == _QMP.PlayingState
                        except Exception:
                            playing = False
                    else:
                        try:
                            playing = 'playing' in str(st).lower() or int(st) == 1
                        except Exception:
                            playing = False
            except Exception:
                playing = False

            try:
                self._dbg('toggle_play: playing=%r' % (playing,))
            except Exception:
                pass
            if playing:
                try:
                    self._player.pause()
                except Exception:
                    try:
                        self._player.pause()
                    except Exception:
                        pass
                try:
                    self.play_btn.setText('▶')
                except Exception:
                    pass
            else:
                try:
                    self._player.play()
                except Exception:
                    pass
                try:
                    self.play_btn.setText('⏸')
                except Exception:
                    pass
        except Exception:
            pass

    def stop(self):
        try:
            if self._player is not None:
                try:
                    self._player.stop()
                except Exception:
                    pass
            try:
                self.play_btn.setText('▶')
            except Exception:
                pass
        except Exception:
            pass
        try:
            self.slider.setValue(0)
        except Exception:
            pass
        self._position = 0
        self._update_time_label()

    def set_sources(self, urls):
        try:
            self._sources = list(urls or [])
            self._current = 0
            if not self._sources:
                self.stop()
                self.label.setText('')
                return
            self._load_current()
        except Exception:
            pass

    def _load_current(self):
        try:
            if not self._sources or self._player is None:
                return
            url = self._sources[int(self._current) % len(self._sources)]
            try:
                from qt.core import QUrl
            except Exception:
                from PyQt5.QtCore import QUrl
            # On Qt5 (calibre 5.44), QMediaPlayer frequently cannot play https
            # streams depending on codecs/backends. Prefer downloading to a
            # local cache file for reliable playback.
            played_url = url
            try:
                is_qt6 = bool(hasattr(self._player, 'setSource'))
            except Exception:
                is_qt6 = False

            if (not is_qt6) and str(url or '').lower().startswith(('http://', 'https://')):
                try:
                    max_mb = int(plugin_prefs.get('audio_cache_max_mb', 25) or 25)
                except Exception:
                    max_mb = 25
                max_bytes = max(1, max_mb) * 1024 * 1024
                try:
                    # Choose cache folder
                    base_cache = ''
                    try:
                        if callable(cache_dir):
                            base_cache = cache_dir()
                        else:
                            base_cache = str(cache_dir or '')
                    except Exception:
                        base_cache = ''
                    if not base_cache:
                        base_cache = os.path.expanduser('~')
                    cdir = os.path.join(base_cache, 'rss_reader_audio_cache')
                    os.makedirs(cdir, exist_ok=True)

                    # Stable filename per URL
                    try:
                        import hashlib as _hashlib
                        h = _hashlib.sha1(str(url).encode('utf-8', 'ignore')).hexdigest()
                    except Exception:
                        h = str(abs(hash(url)))

                    # Extension hint
                    ext = ''
                    try:
                        from urllib.parse import urlparse as _urlparse
                        p = _urlparse(str(url))
                        ext = os.path.splitext(p.path or '')[1]
                    except Exception:
                        ext = ''
                    if not ext:
                        ext = '.mp3'
                    local_path = os.path.join(cdir, h + ext)

                    if not os.path.exists(local_path):
                        self._dbg('downloading audio (qt5): %s' % url)
                        raw, _hdr = fetch_url(_sanitize_url_for_fetch(url), timeout_seconds=45)
                        if raw and len(raw) <= max_bytes:
                            with open(local_path, 'wb') as f:
                                f.write(raw)
                        else:
                            self._dbg('download skipped/too large: %s bytes (cap %s)' % (len(raw) if raw else 0, max_bytes))
                            local_path = ''
                    else:
                        self._dbg('using cached audio: %s' % local_path)

                    if local_path and os.path.exists(local_path):
                        played_url = QUrl.fromLocalFile(local_path)
                except Exception as e:
                    try:
                        self._dbg('download failed: %s' % str(e))
                    except Exception:
                        pass

            try:
                # Qt6 API
                if hasattr(self._player, 'setSource'):
                    self._player.setSource(QUrl(url))
                else:
                    # PyQt5 QMediaPlayer
                    try:
                        from PyQt5.QtMultimedia import QMediaContent
                    except Exception:
                        QMediaContent = None
                    if QMediaContent is not None:
                        try:
                            self._player.setMedia(QMediaContent(played_url if hasattr(played_url, 'toString') else QUrl(url)))
                        except Exception:
                            try:
                                self._player.setMedia(QMediaContent(QUrl(url)))
                            except Exception:
                                pass
                    else:
                        # last resort: try setMedia with QUrl directly
                        try:
                            self._player.setMedia(played_url if hasattr(played_url, 'toString') else QUrl(url))
                        except Exception:
                            try:
                                self._player.setMedia(QUrl(url))
                            except Exception:
                                pass
                try:
                    self._dbg('media loaded: %s' % (str(url),))
                except Exception:
                    pass
            except Exception as e:
                try:
                    self._dbg('set media failed: %s' % str(e))
                except Exception:
                    pass
            try:
                self.label.setText(url)
            except Exception:
                pass
            try:
                # Elide long URLs for the label while keeping full URL in tooltip.
                try:
                    from qt.core import QFontMetrics
                except Exception:
                    from PyQt5.QtGui import QFontMetrics
                try:
                    fm = QFontMetrics(self.label.font())
                    elided = fm.elidedText(url, Qt.ElideMiddle, max(120, int(self.width() * 0.4)))
                    self.label.setText(elided)
                    self.label.setToolTip(url)
                except Exception:
                    # Fallback to full URL if eliding fails
                    try:
                        self.label.setText(url)
                        self.label.setToolTip(url)
                    except Exception:
                        pass
            except Exception:
                pass
            # Do not auto-play when loading a new source; leave in paused
            # state so the user can start playback explicitly.
            try:
                # Ensure play button shows play glyph
                self.play_btn.setText('▶')
            except Exception:
                pass
        except Exception:
            pass

    def _on_position(self, pos):
        try:
            new_pos = int(pos or 0)
        except Exception:
            try:
                new_pos = int(getattr(self, '_position', 0))
            except Exception:
                new_pos = 0
        self._position = new_pos
        try:
            self.slider.blockSignals(True)
            self.slider.setValue(self._position)
            self.slider.blockSignals(False)
        except Exception:
            pass
        self._update_time_label(self._position)

        try:
            if self._dbg_enabled and (self._position == 0 or (self._position % 5000) < 250):
                # avoid spamming: roughly every ~5s
                self._dbg('position=%s' % self._position)
        except Exception:
            pass

    def _on_duration(self, dur):
        try:
            self._duration = int(dur or 0)
        except Exception:
            self._duration = 0
        try:
            self.slider.setRange(0, self._duration)
        except Exception:
            pass
        self._update_time_label()

        try:
            if self._dbg_enabled:
                self._dbg('duration=%s' % self._duration)
        except Exception:
            pass

    def _on_slider_moved(self, val):
        try:
            self._position = int(val or 0)
        except Exception:
            self._position = getattr(self, '_position', 0)
        try:
            if self._player is None:
                return
            # Qt6: setPosition; PyQt5: setPosition
            if hasattr(self._player, 'setPosition'):
                self._player.setPosition(int(self._position))
        except Exception:
            pass
        try:
            if self._dbg_enabled:
                self._dbg('seek -> %s' % self._position)
        except Exception:
            pass
        self._update_time_label(self._position)

    def _format_timestamp(self, millis):
        try:
            total_seconds = max(0, int(int(millis or 0) / 1000))
        except Exception:
            total_seconds = 0
        hours, remainder = divmod(total_seconds, 3600)
        minutes, seconds = divmod(remainder, 60)
        if hours:
            return f"{hours:d}:{minutes:02d}:{seconds:02d}"
        return f"{minutes:d}:{seconds:02d}"

    def _update_time_label(self, pos=None):
        try:
            current = self._position if pos is None else int(pos or 0)
        except Exception:
            current = getattr(self, '_position', 0)
        try:
            total = getattr(self, '_duration', 0)
        except Exception:
            total = 0
        try:
            label_text = '%s / %s' % (
                self._format_timestamp(current),
                self._format_timestamp(total),
            )
            self.time_label.setText(label_text)
        except Exception:
            pass

    # click-to-seek handled by ClickableSlider.mousePressEvent which emits
    # sliderMoved/valueChanged and triggers _on_slider_moved.

    def _is_image_blocked(self, url):
        # AdBlock support has been removed; always allow images.
        return False

    def _block_image(self, url):
        # No-op: AdBlock removed.
        return False

    def _block_domain(self, url):
        # No-op: AdBlock removed.
        return False

    def _unblock_image(self, url):
        # No-op: AdBlock removed.
        return False

    def start_new_page(self):
        # Bump generation to ignore any late image downloads from previous pages.
        with self._img_lock:
            self._img_gen += 1
            self._img_pending = []
            self._img_inflight = set()
            self._img_active = 0
            # Clear current page image list
            try:
                self._current_img_urls = []
            except Exception:
                pass

    def _image_fetch_headers(self):
        headers = {
            # Accept WebP explicitly so CDNs like imgproxy serve the format they have.
            # Qt's QImage or Pillow (fallback) can decode WebP on most setups.
            'Accept': 'image/webp,image/jpeg,image/png,image/gif;q=0.9,*/*;q=0.1',
            # Avoid brotli/gzip for images: some hosts send br unexpectedly and our
            # low-level fetcher only guarantees gzip handling.
            'Accept-Encoding': 'identity',
        }
        try:
            ref = str(getattr(self, '_page_base_url', '') or '')
            if ref:
                headers['Referer'] = ref
        except Exception:
            pass
        return headers

    def _img_cache_path_for_url(self, url):
        try:
            h = hashlib.sha1(str(url).encode('utf-8', 'ignore')).hexdigest()
        except Exception:
            h = str(abs(hash(url)))
        if not self._img_cache_dir:
            return ''
        return os.path.join(self._img_cache_dir, h + '.img')

    def _pump_image_downloads(self):
        # Start background downloads up to concurrency limit.
        while True:
            with self._img_lock:
                if self._img_active >= int(self._img_max_concurrency or 1):
                    return
                if not self._img_pending:
                    return
                url, gen = self._img_pending.pop(0)
                if gen != self._img_gen:
                    continue
                if url in self._img_inflight:
                    continue
                self._img_inflight.add(url)
                self._img_active += 1

            def _worker(u=url, g=gen):
                raw = None
                err = None
                try:
                    from calibre_plugins.rss_reader.rss import fetch_url
                    u_fetch = _sanitize_url_for_fetch(u)
                    raw, _hdr = fetch_url(u_fetch, timeout_seconds=12, headers=self._image_fetch_headers())
                    if raw and self._img_max_bytes and len(raw) > int(self._img_max_bytes):
                        raw = None
                except Exception as e:
                    err = e
                    raw = None

                try:
                    if raw:
                        p = self._img_cache_path_for_url(u)
                        try:
                            with open(p, 'wb') as f:
                                f.write(raw)
                        except Exception:
                            pass
                except Exception:
                    pass

                try:
                    self._img_results.put((g, u, raw, err))
                except Exception:
                    pass

                with self._img_lock:
                    try:
                        self._img_inflight.discard(u)
                    except Exception:
                        pass
                    try:
                        self._img_active = max(0, int(self._img_active) - 1)
                    except Exception:
                        self._img_active = 0

                # Do not touch Qt objects/timers from this worker thread.
                # The main-thread drain loop will kick off more downloads.

            t = threading.Thread(target=_worker, name='rss_reader_img', daemon=True)
            t.start()

    def _drain_image_results(self):
        # Main-thread: apply downloaded images to the current document.
        try:
            processed_any = False
            while True:
                try:
                    g, url, raw, _err = self._img_results.get_nowait()
                except Exception:
                    break

                if g != self._img_gen:
                    continue

                # A worker finished (success or failure). Treat this as progress so we can
                # kick the pump and allow pending downloads to proceed.
                processed_any = True

                if not raw:
                    continue

                img = QImage()
                try:
                    img.loadFromData(raw)
                except Exception:
                    img = QImage()

                # If QImage couldn't decode (common for SVG/AVIF), try fallbacks.

                if img.isNull():
                    # Try rendering SVG via QSvgRenderer if available.
                    try:
                        try:
                            from qt.core import QSvgRenderer, QByteArray, QPainter, QSize
                        except Exception:
                            from PyQt5.QtSvg import QSvgRenderer
                            from PyQt5.QtCore import QByteArray, QSize
                            from PyQt5.QtGui import QPainter

                        qb = QByteArray(raw)
                        renderer = QSvgRenderer(qb)
                        dsize = renderer.defaultSize()
                        if not dsize or dsize.isEmpty():
                            w, h = 800, 600
                        else:
                            w, h = int(dsize.width() or 800), int(dsize.height() or 600)
                        # Cap to reasonable size to avoid huge allocations
                        w = min(w, 1600)
                        h = min(h, 1600)
                        tmp = QImage(w, h, QImage.Format_ARGB32)
                        try:
                            tmp.fill(0)
                        except Exception:
                            pass
                        p = QPainter(tmp)
                        try:
                            renderer.render(p)
                        finally:
                            try:
                                p.end()
                            except Exception:
                                pass
                        if not tmp.isNull():
                            img = tmp
                    except Exception:
                        pass

                if img.isNull():
                    # If the image failed to decode and the URL looks like an AVIF,
                    # generate a neutral placeholder image explaining the format
                    # isn't supported, so users see why an image is missing.
                    try:
                        lowu = (str(url) or '').lower()
                    except Exception:
                        lowu = ''

                    # Detect formats by signature as well (some CDNs return AVIF/WebP
                    # even when the URL ends with .jpg).
                    is_avif = False
                    is_webp = False
                    try:
                        if raw and b'ftypavif' in raw[:64]:
                            is_avif = True
                    except Exception:
                        pass
                    try:
                        if raw and len(raw) >= 12 and raw[:4] == b'RIFF' and raw[8:12] == b'WEBP':
                            is_webp = True
                    except Exception:
                        pass

                    if ('avif' in lowu) or is_avif:
                        try:
                            w, h = 360, 140
                            placeholder = QImage(w, h, QImage.Format_ARGB32)
                            try:
                                from qt.core import QColor, QRect, QPainter, Qt as _Qt
                            except Exception:
                                from PyQt5.QtGui import QColor, QPainter
                                from PyQt5.QtCore import QRect, Qt as _Qt

                            try:
                                placeholder.fill(QColor(240, 240, 240))
                            except Exception:
                                pass

                            p = QPainter(placeholder)
                            try:
                                try:
                                    p.setPen(QColor(80, 80, 80))
                                except Exception:
                                    pass
                                rect = QRect(0, 0, w, h)
                                msg = 'Image not displayed: AVIF format not supported'
                                try:
                                    p.drawText(rect, int(_Qt.AlignCenter | _Qt.TextWordWrap), msg)
                                except Exception:
                                    try:
                                        p.drawText(rect, msg)
                                    except Exception:
                                        pass
                            finally:
                                try:
                                    p.end()
                                except Exception:
                                    pass

                            img = placeholder
                        except Exception:
                            pass

                    if img.isNull() and (('webp' in lowu) or is_webp):
                        try:
                            w, h = 360, 140
                            placeholder = QImage(w, h, QImage.Format_ARGB32)
                            try:
                                from qt.core import QColor, QRect, QPainter, Qt as _Qt
                            except Exception:
                                from PyQt5.QtGui import QColor, QPainter
                                from PyQt5.QtCore import QRect, Qt as _Qt

                            try:
                                placeholder.fill(QColor(240, 240, 240))
                            except Exception:
                                pass

                            p = QPainter(placeholder)
                            try:
                                try:
                                    p.setPen(QColor(80, 80, 80))
                                except Exception:
                                    pass
                                rect = QRect(0, 0, w, h)
                                msg = 'Image not displayed: WebP format not supported'
                                try:
                                    p.drawText(rect, int(_Qt.AlignCenter | _Qt.TextWordWrap), msg)
                                except Exception:
                                    try:
                                        p.drawText(rect, msg)
                                    except Exception:
                                        pass
                            finally:
                                try:
                                    p.end()
                                except Exception:
                                    pass

                            img = placeholder
                        except Exception:
                            pass

                    if img.isNull():
                        continue

                # Cache in memory (best-effort cap).
                try:
                    self._image_cache[url] = img
                    if len(self._image_cache) > 128:
                        try:
                            self._image_cache.clear()
                        except Exception:
                            pass
                except Exception:
                    pass

                try:
                    self.document().addResource(QTextDocument.ResourceType.ImageResource, QUrl(url), img)
                    try:
                        self.document().markContentsDirty(0, max(1, int(self.document().characterCount() or 1)))
                    except Exception:
                        pass
                    try:
                        self.viewport().update()
                    except Exception:
                        pass
                except Exception:
                    pass

            # If we processed results, kick the pump to start more downloads.
            try:
                if processed_any:
                    self._pump_image_downloads()
            except Exception:
                pass
        except Exception:
            pass

    def set_zoom_callback(self, callback):
        self._zoom_callback = callback

    def wheelEvent(self, event):
        try:
            mods = event.modifiers() if hasattr(event, 'modifiers') else None
            ctrl = bool(mods & Qt.KeyboardModifier.ControlModifier) if mods is not None else False
        except Exception:
            ctrl = False

        if ctrl and callable(getattr(self, '_zoom_callback', None)):
            dy = 0
            try:
                dy = int(event.angleDelta().y())
            except Exception:
                try:
                    dy = int(event.delta())
                except Exception:
                    dy = 0
            if dy:
                try:
                    self._zoom_callback(1 if dy > 0 else -1)
                except Exception:
                    pass
                try:
                    event.accept()
                except Exception:
                    pass
                return

        return super().wheelEvent(event)

    def loadResource(self, res_type, name):
        try:
            if res_type == QTextDocument.ResourceType.ImageResource:
                try:
                    if not bool(plugin_prefs.get('load_images_in_preview', True)):
                        return QImage()
                except Exception:
                    pass
                try:
                    url = name.toString() if hasattr(name, 'toString') else str(name)
                except Exception:
                    url = str(name)

                if not url:
                    return super().loadResource(res_type, name)

                # Check AdBlock
                if self._is_image_blocked(url):
                    # Return a small placeholder image indicating the image is blocked
                    try:
                        w, h = 200, 100
                        blocked_img = QImage(w, h, QImage.Format_ARGB32)
                        try:
                            from qt.core import QColor, QRect, QPainter, Qt as _Qt
                        except Exception:
                            from PyQt5.QtGui import QColor, QPainter
                            from PyQt5.QtCore import QRect, Qt as _Qt

                        try:
                            blocked_img.fill(QColor(255, 220, 220))
                        except Exception:
                            pass

                        p = QPainter(blocked_img)
                        try:
                            try:
                                p.setPen(QColor(150, 0, 0))
                            except Exception:
                                pass
                            rect = QRect(0, 0, w, h)
                            msg = '🚫 Image blocked by AdBlock'
                            try:
                                p.drawText(rect, int(_Qt.AlignCenter), msg)
                            except Exception:
                                try:
                                    p.drawText(rect, msg)
                                except Exception:
                                    pass
                        finally:
                            try:
                                p.end()
                            except Exception:
                                pass

                        return blocked_img
                    except Exception:
                        return QImage()

                cached = self._image_cache.get(url)
                if cached is not None:
                    return cached

                # Disk cache
                try:
                    p = self._img_cache_path_for_url(url)
                    if os.path.exists(p):
                        with open(p, 'rb') as f:
                            raw = f.read()
                        img = QImage()
                        img.loadFromData(raw)
                        if not img.isNull():
                            self._image_cache[url] = img
                            return img
                except Exception:
                    pass

                # Only allow http(s) images
                if url.lower().startswith('http://') or url.lower().startswith('https://'):
                    # Fetch asynchronously; return a placeholder now.
                    try:
                        with self._img_lock:
                            g = self._img_gen
                            if url not in self._img_inflight:
                                self._img_pending.append((url, g))
                        self._pump_image_downloads()
                    except Exception:
                        pass
                    return QImage()

                # Fallback to default behavior
                return super().loadResource(res_type, name)
        except Exception:
            return super().loadResource(res_type, name)

    def _image_url_at_pos(self, pos):
        # QTextBrowser (QAbstractScrollArea) can report customContextMenuRequested
        # coordinates in either widget or viewport space depending on bindings.
        # Try both to reliably detect the image under the cursor.
        candidates = []
        try:
            candidates.append(pos)
        except Exception:
            pass
        try:
            vp = self.viewport() if hasattr(self, 'viewport') else None
            if vp is not None:
                try:
                    candidates.append(vp.mapFrom(self, pos))
                except Exception:
                    pass
        except Exception:
            pass

        for p in candidates:
            try:
                cursor = self.cursorForPosition(p)
                fmt = cursor.charFormat()
                if hasattr(fmt, 'isImageFormat') and fmt.isImageFormat():
                    try:
                        img_fmt = fmt.toImageFormat()
                        return str(img_fmt.name() or '')
                    except Exception:
                        return ''
            except Exception:
                continue

        return ''

    def add_rss_reader_context_actions(self, menu, pos):
        """Augment an existing QMenu with RSS Reader-specific actions.

        Note: RSS Reader's main UI uses CustomContextMenu on the Preview widget,
        so the menu is created in ui.py via createStandardContextMenu(). This
        method lets that code path still show image actions.
        """

        added_any = False

        try:
            img_url = self._image_url_at_pos(pos)
        except Exception:
            img_url = ''

        def _copy_text(text):
            try:
                QApplication.clipboard().setText(text or '')
            except Exception:
                pass

        def _open_in_browser(u):
            try:
                if not u:
                    return
                try:
                    from qt.core import QDesktopServices, QUrl
                except Exception:
                    from PyQt5.Qt import QDesktopServices, QUrl
                QDesktopServices.openUrl(QUrl(str(u)))
            except Exception:
                pass

        def _get_image_for_url(u):
            try:
                img = self._image_cache.get(u)
            except Exception:
                img = None
            if img is not None:
                return img
            try:
                p = self._img_cache_path_for_url(u)
                if p and os.path.exists(p):
                    with open(p, 'rb') as f:
                        raw = f.read()
                    tmp = QImage()
                    tmp.loadFromData(raw)
                    if not tmp.isNull():
                        return tmp
            except Exception:
                pass
            return None

        def _save_image_from_url(u):
            try:
                import os, re
                url_ext = ''
                try:
                    m = re.search(r'\.([a-zA-Z0-9]+)(?:[?#].*)?$', u)
                    if m:
                        url_ext = m.group(1).lower()
                        if url_ext not in ('jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'svg', 'avif'):
                            url_ext = ''
                except Exception:
                    url_ext = ''

                p = None
                try:
                    p = self._img_cache_path_for_url(u)
                    if p and os.path.exists(p):
                        base = os.path.splitext(os.path.basename(p))[0]
                    else:
                        p = None
                        base = os.path.splitext(os.path.basename(u))[0]
                except Exception:
                    p = None
                    base = os.path.splitext(os.path.basename(u))[0]

                if url_ext:
                    default = f"{base}.{url_ext}"
                else:
                    default = os.path.basename(p) if p else os.path.basename(u)

                # Remember last-used directory
                try:
                    from calibre_plugins.rss_reader.config import plugin_prefs
                    last_dir = plugin_prefs.get('last_image_save_dir', None)
                except Exception:
                    last_dir = None
                if last_dir and os.path.isdir(last_dir):
                    default_path = os.path.join(last_dir, default)
                else:
                    default_path = default

                fname, _dummy = QFileDialog.getSaveFileName(self, _('Save image as'), default_path)
                if not fname:
                    return

                try:
                    from calibre_plugins.rss_reader.config import plugin_prefs
                    plugin_prefs['last_image_save_dir'] = os.path.dirname(fname)
                except Exception:
                    pass

                if p and os.path.exists(p):
                    shutil.copyfile(p, fname)
                    return

                try:
                    from calibre_plugins.rss_reader.rss import fetch_url
                    raw, _hdr = fetch_url(u, timeout_seconds=12, headers=self._image_fetch_headers())
                    if raw:
                        with open(fname, 'wb') as f:
                            f.write(raw)
                        return
                except Exception:
                    pass

                try:
                    img_obj = _get_image_for_url(u)
                    if img_obj is not None and not img_obj.isNull():
                        img_obj.save(fname)
                except Exception:
                    pass
            except Exception:
                pass

        # Provide the current feed URL regardless of whether an image is under the cursor.
        try:
            feed_url = str(getattr(self, '_current_feed_url', '') or '')
        except Exception:
            feed_url = ''
        if feed_url:
            try:
                menu.addSeparator()
            except Exception:
                pass
            act_copy_feed = QAction(_('Copy feed URL'), self)
            act_copy_feed.triggered.connect(lambda: _copy_text(feed_url))
            menu.addAction(act_copy_feed)
            added_any = True

        if img_url:
            try:
                menu.addSeparator()
            except Exception:
                pass

            act_open = QAction(_('Open image in browser'), self)
            act_copy_img = QAction(_('Copy image'), self)
            act_copy_url = QAction(_('Copy image URL'), self)
            act_save_img = QAction(_('Save image as...'), self)

            def _copy_img():
                try:
                    img = _get_image_for_url(img_url)
                    if img is not None:
                        QApplication.clipboard().setImage(img)
                except Exception:
                    pass

            act_open.triggered.connect(lambda _checked=False, u=img_url: _open_in_browser(str(u)))
            act_copy_img.triggered.connect(_copy_img)
            act_copy_url.triggered.connect(lambda: _copy_text(img_url))
            act_save_img.triggered.connect(lambda: _save_image_from_url(img_url))

            menu.addAction(act_open)
            menu.addAction(act_copy_img)
            menu.addAction(act_copy_url)
            menu.addAction(act_save_img)
            return True

        # Not directly on an image: add a lightweight Images submenu if we have any
        try:
            urls = getattr(self, '_current_img_urls', None) or []
        except Exception:
            urls = []

        if urls:
            try:
                menu.addSeparator()
            except Exception:
                pass
            images_menu = menu.addMenu(_('Images'))
            for u in list(urls)[:20]:
                try:
                    label = str(u)
                    if len(label) > 72:
                        label = '...%s' % label[-69:]
                except Exception:
                    label = _('Image')
                sub = images_menu.addMenu(label)
                a_open = QAction(_('Open image in browser'), self)
                a_copy_url = QAction(_('Copy image URL'), self)
                a_save = QAction(_('Save image as...'), self)
                a_open.triggered.connect(lambda _checked=False, uu=u: _open_in_browser(str(uu)))
                a_copy_url.triggered.connect(lambda _checked=False, uu=u: _copy_text(str(uu)))
                a_save.triggered.connect(lambda _checked=False, uu=u: _save_image_from_url(str(uu)))
                sub.addAction(a_open)
                sub.addAction(a_copy_url)
                sub.addAction(a_save)
            added_any = True

        return added_any


