from __future__ import absolute_import

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import (
        Qt,
        QEvent,
        QTimer,
        QSize,
        QIcon,
        QPixmap,
        QWidget,
        QLabel,
        QToolButton,
        QStackedWidget,
        QVBoxLayout,
        QHBoxLayout,
        QSizePolicy,
        QFont,
        QPalette,
        QColor,
        pyqtSignal,
        QApplication,
    )
except Exception:
    from PyQt5.Qt import (
        Qt,
        QEvent,
        QTimer,
        QSize,
        QIcon,
        QPixmap,
        QWidget,
        QLabel,
        QToolButton,
        QStackedWidget,
        QVBoxLayout,
        QHBoxLayout,
        QSizePolicy,
        QFont,
        QPalette,
        QColor,
        pyqtSignal,
        QApplication,
    )

from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader.debug import _debug, DEBUG_RSS_READER

try:
    from calibre_plugins.rss_reader.common_icons import get_icon as get_plugin_icon
except Exception:
    try:
        from common_icons import get_icon as get_plugin_icon
    except Exception:
        get_plugin_icon = None


def _icon_from_bytes(icon_bytes, fallback_icon=None, size=16):
    try:
        if icon_bytes:
            pm = QPixmap()
            try:
                pm.loadFromData(icon_bytes)
            except Exception:
                pm.loadFromData(bytes(icon_bytes))
            if not pm.isNull():
                return QIcon(pm)
    except Exception:
        pass
    return fallback_icon


def _is_dark_palette(palette):
    try:
        return palette.color(palette.Window).value() < 128
    except Exception:
        try:
            return palette.color(QPalette.Window).value() < 128
        except Exception:
            return False


def _popup_theme_name():
    try:
        t = str(plugin_prefs.get('popup_theme', 'dark') or 'dark').strip().lower()
    except Exception:
        t = 'dark'

    # User-facing options are only: light|dark.
    # Back-compat: treat dark_high_contrast as dark, and auto as palette-follow.
    if t in ('dark_high_contrast', 'dark-hc', 'high_contrast'):
        return 'dark'
    if t in ('light', 'dark'):
        return t

    # auto (legacy)
    try:
        pal = QApplication.palette()
    except Exception:
        pal = None
    if pal is not None and _is_dark_palette(pal):
        return 'dark'
    return 'light'


def _popup_colors(theme_name=None):
    tn = theme_name or _popup_theme_name()

    # Dark now uses the Dark High Contrast palette/styling.
    if str(tn).strip().lower() == 'dark':
        key = 'popup_theme_colors_dark_high_contrast'
    else:
        key = 'popup_theme_colors_%s' % str(tn).strip().lower()
    try:
        colors = dict(plugin_prefs.get(key, {}) or {})
    except Exception:
        colors = {}

    # Back-compat: if dark_high_contrast isn't set yet, fall back to old dark.
    if not colors and str(tn).strip().lower() == 'dark':
        try:
            colors = dict(plugin_prefs.get('popup_theme_colors_dark', {}) or {})
        except Exception:
            colors = {}

    # Defensive fallbacks
    if not colors:
        try:
            colors = dict(plugin_prefs.get('popup_theme_colors_dark', {}) or {})
        except Exception:
            colors = {}
    return colors


class PopupNewsItem(QWidget):
    open_in_app = pyqtSignal(str, str)      # feed_id, item_id
    open_external = pyqtSignal(str)         # url
    mark_read = pyqtSignal(str, str)        # feed_id, item_id (unused: row action removed)

    def __init__(self, entry, show_feed_icon=True, colors=None, item_min_height_px=None, item_padding_v_px=None, parent=None):
        QWidget.__init__(self, parent)
        self._entry = dict(entry or {})
        try:
            self._colors = dict(colors or {}) if isinstance(colors, dict) else {}
        except Exception:
            self._colors = {}
        self.setObjectName('popup_news_item')

        try:
            mh = int(item_min_height_px) if item_min_height_px is not None else 0
        except Exception:
            mh = 0
        if mh and mh > 0:
            try:
                self.setMinimumHeight(max(0, mh))
            except Exception:
                pass

        try:
            pad_v = int(item_padding_v_px) if item_padding_v_px is not None else 0
        except Exception:
            pad_v = 0
        pad_v = max(0, min(pad_v, 80))

        try:
            self.setAttribute(Qt.WidgetAttribute.WA_TranslucentBackground)
        except Exception:
            pass

        self._icon = QLabel(self)
        self._icon.setFixedSize(QSize(18, 18))

        self._title = QLabel(self)
        self._title.setObjectName('popup_item_title')
        # On Qt6/PyQt6, be explicit: we only need link clicks, not text selection.
        # Some wrapper combinations behave oddly with TextBrowserInteraction.
        try:
            self._title.setTextInteractionFlags(Qt.TextInteractionFlag.LinksAccessibleByMouse)
        except Exception:
            self._title.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        self._title.setOpenExternalLinks(False)
        self._title.linkActivated.connect(self._on_title_link)
        try:
            # Force rich text so <a href> reliably becomes clickable.
            self._title.setTextFormat(Qt.TextFormat.RichText)
        except Exception:
            pass
        try:
            self._title.setCursor(Qt.CursorShape.PointingHandCursor)
        except Exception:
            pass
        try:
            # Debug-only: log whether the title label receives mouse events.
            self._title.installEventFilter(self)
        except Exception:
            pass
        # Allow the popup to honor narrow widths by not letting QLabel size hints
        # force the parent widget wider.
        self._title.setWordWrap(False)
        try:
            self._title.setMinimumWidth(0)
        except Exception:
            pass
        self._title.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Preferred)

        self._btn_open_ext = QToolButton(self)
        self._btn_open_ext.setObjectName('popup_btn_open_ext')
        self._btn_open_ext.setToolTip(_('Open in browser'))
        self._btn_open_ext.setCursor(Qt.CursorShape.PointingHandCursor)
        # Some environments lose MouseRelease; prefer `pressed` (and avoid double-fire).
        try:
            self._btn_open_ext.pressed.connect(self._on_open_external)
        except Exception:
            try:
                self._btn_open_ext.clicked.connect(self._on_open_external)
            except Exception:
                pass

        # Per-item "mark as read" button removed (keep popup less cluttered).
        self._btn_mark_read = None

        lay = QHBoxLayout()
        lay.setContentsMargins(6, pad_v, 6, pad_v)
        lay.setSpacing(6)
        if show_feed_icon:
            lay.addWidget(self._icon)
        lay.addWidget(self._title, 1)
        lay.addWidget(self._btn_open_ext)
        self.setLayout(lay)

        self._apply(entry)

    def mousePressEvent(self, e):
        # Trigger in-app open on mouse press (not release) because on some
        # Qt6 builds the popup can lose the release event when the main window
        # activates beneath it.
        try:
            if DEBUG_RSS_READER:
                _debug('PopupNewsItem.mousePressEvent at', e.pos())
        except Exception:
            pass

        # Ignore presses on the explicit toolbuttons / title label (title is handled
        # via eventFilter so we don't double-fire).
        try:
            child = self.childAt(e.pos())
        except Exception:
            child = None
        try:
            if isinstance(child, QToolButton):
                return QWidget.mousePressEvent(self, e)
        except Exception:
            pass
        try:
            if child is getattr(self, '_title', None):
                return QWidget.mousePressEvent(self, e)
        except Exception:
            pass

        try:
            if DEBUG_RSS_READER:
                _debug('Popup row mouse-press -> open_in_app', self._entry.get('feed_id'), self._entry.get('item_id'))
        except Exception:
            pass
        try:
            self._on_title_link('open')
        except Exception:
            pass
        try:
            e.accept()
        except Exception:
            pass
        try:
            return QWidget.mousePressEvent(self, e)
        except Exception:
            return

    def mouseReleaseEvent(self, e):
        try:
            if DEBUG_RSS_READER:
                _debug('PopupNewsItem.mouseReleaseEvent at', e.pos(), 'feed_id=', self._entry.get('feed_id'), 'item_id=', self._entry.get('item_id'))
        except Exception:
            pass
        # Make the entire row behave as a click target to open in-app,
        # except when clicking on the explicit buttons.
        try:
            child = self.childAt(e.pos())
        except Exception:
            child = None
        try:
            from PyQt5.Qt import QToolButton as _QTBtn
        except Exception:
            _QTBtn = QToolButton
        try:
            if isinstance(child, (_QTBtn, QToolButton)):
                if DEBUG_RSS_READER:
                    _debug('  (skipping: clicked on button)')
                return QWidget.mouseReleaseEvent(self, e)
        except Exception:
            pass
        try:
            if DEBUG_RSS_READER:
                _debug('Popup row clicked -> open_in_app', self._entry.get('feed_id'), self._entry.get('item_id'))
        except Exception:
            pass
        try:
            self._on_title_link('open')
        except Exception:
            pass
        try:
            e.accept()
        except Exception:
            pass
        try:
            return QWidget.mouseReleaseEvent(self, e)
        except Exception:
            return

    def eventFilter(self, obj, event):
        # Only used to diagnose click delivery on Qt6.
        try:
            if obj is getattr(self, '_title', None):
                et = event.type()
                if et == QEvent.Type.MouseButtonPress:
                    if DEBUG_RSS_READER:
                        _debug('PopupNewsItem.title MouseButtonPress')
                    # Fire on press as well; some environments never deliver
                    # MouseButtonRelease/linkActivated.
                    try:
                        self._on_title_link('open')
                    except Exception:
                        pass
                    try:
                        event.accept()
                    except Exception:
                        pass
                    return True
                elif et == QEvent.Type.MouseButtonRelease:
                    if DEBUG_RSS_READER:
                        _debug('PopupNewsItem.title MouseButtonRelease')
        except Exception:
            pass
        try:
            return QWidget.eventFilter(self, obj, event)
        except Exception:
            return False

    def _apply(self, entry):
        entry = dict(entry or {})
        title = str(entry.get('item_title') or '').strip() or _('(untitled)')
        # clickable label (in-app open)
        safe = title.replace('&', '&amp;').replace('<', '&lt;').replace('>', '&gt;')
        # Use the popup's effective link color (preview overrides included).
        link = ''
        try:
            link = str((self._colors or {}).get('clickable_url') or (self._colors or {}).get('link') or '').strip()
        except Exception:
            link = ''
        # Force anchor color via inline style (most reliable across Qt/palette overrides).
        try:
            self._title.setTextFormat(Qt.TextFormat.RichText)
        except Exception:
            pass
        try:
            # Keep widget-level stylesheet minimal so it doesn't fight the inline anchor style.
            self._title.setStyleSheet('QLabel#popup_item_title{padding:0px;}')
        except Exception:
            pass

        if link:
            self._title.setText(
                '<a href="open" style="color:%s; text-decoration:none;">%s</a>' % (link, safe)
            )
        else:
            self._title.setText('<a href="open">%s</a>' % safe)
        self._title.setToolTip(str(entry.get('feed_title') or ''))

        # icon
        try:
            icon = entry.get('feed_icon')
            if isinstance(icon, QIcon):
                pm = icon.pixmap(16, 16)
            else:
                pm = QPixmap()
            if not pm.isNull():
                self._icon.setPixmap(pm)
        except Exception:
            pass

        # buttons
        try:
            # Prefer icons (avoids font-dependent glyph differences across calibre).
            open_ic = None
            ok_ic = None
            try:
                open_ic = QIcon.ic('external-link')
            except Exception:
                open_ic = None
            try:
                ok_ic = QIcon.ic('ok')
            except Exception:
                ok_ic = None

            try:
                self._btn_open_ext.setIconSize(QSize(14, 14))
            except Exception:
                pass

            if open_ic is not None and not open_ic.isNull():
                try:
                    self._btn_open_ext.setIcon(open_ic)
                    self._btn_open_ext.setText('')
                except Exception:
                    self._btn_open_ext.setText('↗')
            else:
                self._btn_open_ext.setText('↗')

            # Style is applied at popup level; keep local styling minimal.
        except Exception:
            pass

    def _on_title_link(self, _href):
        try:
            if DEBUG_RSS_READER:
                _debug('Popup click: open_in_app', self._entry.get('feed_id'), self._entry.get('item_id'))
        except Exception:
            pass
        try:
            self.open_in_app.emit(str(self._entry.get('feed_id') or ''), str(self._entry.get('item_id') or ''))
        except Exception:
            pass

    def _on_open_external(self):
        try:
            if DEBUG_RSS_READER:
                _debug('Popup click: open_external', self._entry.get('link'))
        except Exception:
            pass
        try:
            url = str(self._entry.get('link') or '').strip()
            if url:
                self.open_external.emit(url)
        except Exception:
            pass

    def _on_mark_read(self):
        try:
            if DEBUG_RSS_READER:
                _debug('Popup click: mark_read', self._entry.get('feed_id'), self._entry.get('item_id'))
        except Exception:
            pass
        try:
            self.mark_read.emit(str(self._entry.get('feed_id') or ''), str(self._entry.get('item_id') or ''))
        except Exception:
            pass


class PopupNotification(QWidget):
    closed = pyqtSignal()
    # Expose popup-level signals so action.py can connect once and still work
    # even if pages/items are rebuilt (theme refresh, etc.).
    open_in_app = pyqtSignal(str, str)
    open_external = pyqtSignal(str)
    mark_read = pyqtSignal(str, str)

    def __init__(
        self,
        entries,
        total_new=0,
        title_text=None,
        max_items_per_page=10,
        timeout_ms=15000,
        theme_name=None,
        colors_override=None,
        position=None,
        margin_px=None,
        width_px=None,
        item_min_height_px=None,
        item_padding_v_px=None,
        parent=None,
    ):
        QWidget.__init__(self, parent)

        # Guard against palette/theme-change re-entrancy. Calibre theme toggles
        # can emit multiple change events in a tight sequence; rebuilding pages
        # inside changeEvent can cause C++-side crashes in Qt.
        self._closing = False
        self._theme_refresh_scheduled = False

        self._entries = list(entries or [])
        self._total_new = int(total_new or 0)
        # If caller didn't override, pull from prefs.
        try:
            if max_items_per_page is None:
                max_items_per_page = plugin_prefs.get('popup_max_items_per_page', 10)
        except Exception:
            pass
        try:
            if timeout_ms is None:
                timeout_ms = plugin_prefs.get('popup_timeout_ms', 15000)
        except Exception:
            pass
        self._max_items_per_page = max(3, int(max_items_per_page or 10))
        self._timeout_ms = max(0, int(timeout_ms or 0))

        # Optional runtime overrides (used by settings preview)
        try:
            self._theme_override = str(theme_name or '').strip().lower() or None
        except Exception:
            self._theme_override = None
        try:
            self._colors_override = dict(colors_override or {}) if isinstance(colors_override, dict) else None
        except Exception:
            self._colors_override = None
        try:
            self._position_override = str(position or '').strip().lower() or None
        except Exception:
            self._position_override = None
        try:
            self._margin_override = int(margin_px) if margin_px is not None else None
        except Exception:
            self._margin_override = None
        try:
            self._width_override = int(width_px) if width_px is not None else None
        except Exception:
            self._width_override = None

        try:
            self._item_min_height_override = int(item_min_height_px) if item_min_height_px is not None else None
        except Exception:
            self._item_min_height_override = None
        try:
            self._item_padding_v_override = int(item_padding_v_px) if item_padding_v_px is not None else None
        except Exception:
            self._item_padding_v_override = None

        self._page_index = 0
        self._pages = []

        # Window style: frameless, top-most tool window
        try:
            flags = Qt.WindowType.FramelessWindowHint | Qt.WindowType.Tool | Qt.WindowType.WindowStaysOnTopHint
        except Exception:
            flags = Qt.FramelessWindowHint | Qt.Tool | Qt.WindowStaysOnTopHint
        self.setWindowFlags(flags)
        # Important on Windows: avoid stealing focus when showing the popup.
        # This prevents other plugin dialogs (e.g. CCR) from being minimized.
        try:
            self.setAttribute(Qt.WidgetAttribute.WA_ShowWithoutActivating, True)
        except Exception:
            try:
                self.setAttribute(Qt.WA_ShowWithoutActivating, True)
            except Exception:
                pass
        try:
            # Qt5/Qt6 compat: if available, explicitly opt out of focus.
            self.setWindowFlag(Qt.WindowType.WindowDoesNotAcceptFocus, True)
        except Exception:
            try:
                self.setWindowFlag(Qt.WindowDoesNotAcceptFocus, True)
            except Exception:
                pass
        try:
            self.setAttribute(Qt.WidgetAttribute.WA_TranslucentBackground)
        except Exception:
            pass
        try:
            self.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        except Exception:
            pass

        # Title bar
        self._icon = QLabel(self)
        self._icon.setFixedSize(QSize(16, 16))

        self._title = QLabel(self)
        self._title.setText(title_text or (_('Incoming News: %d') % self._total_new))
        try:
            self._title.setMinimumWidth(0)
            self._title.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Preferred)
        except Exception:
            pass
        try:
            f = QFont(self._title.font())
            f.setBold(True)
            self._title.setFont(f)
        except Exception:
            pass

        self._btn_close = QToolButton(self)
        self._btn_close.setObjectName('popup_btn_close')
        self._btn_close.setToolTip(_('Close'))
        self._btn_close.setText('×')
        self._btn_close.setCursor(Qt.CursorShape.PointingHandCursor)
        self._btn_close.clicked.connect(self.close)

        title_lay = QHBoxLayout()
        title_lay.setContentsMargins(8, 6, 8, 6)
        title_lay.setSpacing(6)
        title_lay.addWidget(self._icon)
        title_lay.addWidget(self._title, 1)
        title_lay.addWidget(self._btn_close)

        self._title_bar = QWidget(self)
        self._title_bar.setLayout(title_lay)

        # Pages
        self._stack = QStackedWidget(self)

        # Bottom bar
        self._lbl_page = QLabel(self)
        self._lbl_page.setObjectName('popup_page_label')
        self._btn_prev = QToolButton(self)
        self._btn_prev.setObjectName('popup_btn_prev')
        self._btn_prev.setText('')
        self._btn_prev.setToolTip(_('Previous page'))
        self._btn_prev.setCursor(Qt.CursorShape.PointingHandCursor)
        self._btn_prev.clicked.connect(lambda: self._goto_page(self._page_index - 1))

        self._btn_next = QToolButton(self)
        self._btn_next.setObjectName('popup_btn_next')
        self._btn_next.setText('')
        self._btn_next.setToolTip(_('Next page'))
        self._btn_next.setCursor(Qt.CursorShape.PointingHandCursor)
        self._btn_next.clicked.connect(lambda: self._goto_page(self._page_index + 1))

        # Prefer plugin-provided navigation icons (consistent across fonts/themes).
        try:
            self._btn_prev.setIconSize(QSize(14, 14))
            self._btn_next.setIconSize(QSize(14, 14))
        except Exception:
            pass
        prev_ic = None
        next_ic = None
        try:
            prev_ic = get_plugin_icon('images/custom_back.png') if get_plugin_icon else None
        except Exception:
            prev_ic = None
        try:
            next_ic = get_plugin_icon('images/custom_forward.png') if get_plugin_icon else None
        except Exception:
            next_ic = None
        if prev_ic is not None and not prev_ic.isNull():
            try:
                self._btn_prev.setIcon(prev_ic)
            except Exception:
                pass
        else:
            try:
                self._btn_prev.setText('‹')
            except Exception:
                pass
        if next_ic is not None and not next_ic.isNull():
            try:
                self._btn_next.setIcon(next_ic)
            except Exception:
                pass
        else:
            try:
                self._btn_next.setText('›')
            except Exception:
                pass

        bottom_lay = QHBoxLayout()
        bottom_lay.setContentsMargins(8, 4, 8, 6)
        bottom_lay.setSpacing(6)
        bottom_lay.addWidget(self._lbl_page)
        bottom_lay.addStretch(1)
        bottom_lay.addWidget(self._btn_prev)
        bottom_lay.addWidget(self._btn_next)

        self._bottom_bar = QWidget(self)
        self._bottom_bar.setLayout(bottom_lay)

        # Outer
        outer = QVBoxLayout()
        outer.setContentsMargins(1, 1, 1, 1)
        outer.setSpacing(0)
        outer.addWidget(self._title_bar)
        outer.addWidget(self._stack)
        outer.addWidget(self._bottom_bar)
        self.setLayout(outer)

        # Apply theme-aware styling
        self._apply_theme()

        # Make a frame widget to apply rounded background
        self._frame = QWidget(self)
        self._frame.setObjectName('popup_frame')
        self._frame.setLayout(outer)
        frame_outer = QVBoxLayout()
        frame_outer.setContentsMargins(0, 0, 0, 0)
        frame_outer.addWidget(self._frame)
        QWidget.setLayout(self, frame_outer)

        # Timer (auto-close; paused while hovered)
        self._timer = QTimer(self)
        self._timer.setSingleShot(True)
        if self._timeout_ms > 0:
            self._timer.timeout.connect(self.close)

        self._build_pages()
        self._goto_page(0)

    def _apply_theme(self):
        try:
            colors = dict(_popup_colors(self._theme_override) or {})
        except Exception:
            colors = dict(_popup_colors() or {})
        try:
            if isinstance(getattr(self, '_colors_override', None), dict) and self._colors_override:
                colors.update(self._colors_override)
        except Exception:
            pass
        bg = str(colors.get('bg') or '#0D1117')
        fg = str(colors.get('fg') or '#F0F6FC')
        muted_fg = str(colors.get('muted_fg') or fg)
        border = str(colors.get('border') or '#30363D')
        header_bg = str(colors.get('header_bg') or bg)
        header_fg = str(colors.get('header_fg') or fg)
        feed_header_bg = str(colors.get('feed_header_bg') or header_bg)
        feed_header_fg = str(colors.get('feed_header_fg') or header_fg)
        item_hover_bg = str(colors.get('item_hover_bg') or 'rgba(255,255,255,0.08)')
        link = str(colors.get('clickable_url') or colors.get('link') or '#58A6FF')
        btn_bg = str(colors.get('btn_bg') or header_bg)
        btn_fg = str(colors.get('btn_fg') or fg)
        btn_border = str(colors.get('btn_border') or border)
        btn_hover_bg = str(colors.get('btn_hover_bg') or border)
        btn_primary_bg = str(colors.get('btn_primary_bg') or link)
        btn_primary_fg = str(colors.get('btn_primary_fg') or '#000000')

        # Ensure links render using our palette even when HTML anchors are used.
        try:
            pal = self.palette()
            pal.setColor(QPalette.WindowText, QColor(fg))
            pal.setColor(QPalette.Text, QColor(fg))
            pal.setColor(QPalette.ButtonText, QColor(btn_fg))
            pal.setColor(QPalette.Link, QColor(link))
            pal.setColor(QPalette.LinkVisited, QColor(link))
            self.setPalette(pal)
        except Exception:
            pass

        # Expose the effective colors to child items (so preview overrides apply)
        try:
            self._effective_colors = dict(colors)
            # Ensure key exists for downstream lookup
            if 'clickable_url' not in self._effective_colors and link:
                self._effective_colors['clickable_url'] = link
        except Exception:
            self._effective_colors = {'clickable_url': link, 'link': link}

        # Frame + controls
        self.setStyleSheet(
            '#popup_frame{'
            f'background:{bg};'
            f'border:1px solid {border};'
            'border-radius:6px;'
            '}'
            'QLabel{'
            f'color:{fg};'
            '}'
            # Ensure rich-text anchors always use our intended link color.
            'QLabel a{'
            f'color:{link};'
            'text-decoration:none;'
            '}'
            'QLabel a:hover{'
            f'color:{link};'
            'text-decoration:underline;'
            '}'
            '#popup_page_label{'
            f'color:{muted_fg};'
            '}'
            'QToolButton{'
            f'color:{btn_fg};'
            f'background:{btn_bg};'
            f'border:1px solid {btn_border};'
            'border-radius:4px;'
            'padding:2px 6px;'
            'min-width:18px;'
            '}'
            'QToolButton:hover{'
            f'background:{btn_hover_bg};'
            '}'
            '#popup_btn_close{'
            f'background:{btn_bg};'
            f'border:1px solid {btn_border};'
            'padding:0px 8px;'
            'min-width:0px;'
            '}'
        )

        # Title bar and bottom bar backgrounds
        try:
            self._title_bar.setStyleSheet('background:%s; color:%s;' % (header_bg, header_fg))
        except Exception:
            pass
        try:
            self._bottom_bar.setStyleSheet('background:%s; color:%s;' % (header_bg, header_fg))
        except Exception:
            pass
        try:
            # Keep feed header palette in instance for _build_pages
            self._feed_header_bg = feed_header_bg
            self._feed_header_fg = feed_header_fg
        except Exception:
            self._feed_header_bg = feed_header_bg
            self._feed_header_fg = feed_header_fg

    def set_header_icon(self, icon):
        try:
            if isinstance(icon, QIcon):
                pm = icon.pixmap(16, 16)
            elif isinstance(icon, QPixmap):
                pm = icon
            else:
                pm = QPixmap()
            if not pm.isNull():
                self._icon.setPixmap(pm)
        except Exception:
            pass



    def _build_pages(self):
        # entries are expected already in display order
        self._pages = []
        try:
            while self._stack.count():
                w = self._stack.widget(0)
                self._stack.removeWidget(w)
                try:
                    w.deleteLater()
                except Exception:
                    pass
        except Exception:
            pass

        chunks = []
        cur = []
        for e in self._entries:
            cur.append(e)
            if len(cur) >= self._max_items_per_page:
                chunks.append(cur)
                cur = []
        if cur:
            chunks.append(cur)

        for chunk in chunks or [[]]:
            page = QWidget(self)
            v = QVBoxLayout()
            v.setContentsMargins(0, 0, 0, 0)
            v.setSpacing(0)

            # Per-item layout prefs (with runtime overrides for preview)
            try:
                if getattr(self, '_item_min_height_override', None) is not None:
                    item_min_h = int(self._item_min_height_override)
                else:
                    item_min_h = int(plugin_prefs.get('popup_item_min_height_px', 0) or 0)
            except Exception:
                item_min_h = 0
            try:
                item_pad_v = int(getattr(self, '_item_padding_v_override', None) or 0)
            except Exception:
                item_pad_v = 0
            if getattr(self, '_item_padding_v_override', None) is None:
                try:
                    item_pad_v = int(plugin_prefs.get('popup_item_padding_v_px', 2) or 2)
                except Exception:
                    item_pad_v = 2
            item_min_h = max(0, min(int(item_min_h or 0), 300))
            item_pad_v = max(0, min(int(item_pad_v or 0), 80))

            for e in chunk:
                # optional feed heading rows
                if e.get('_is_feed_header'):
                    hdr = QLabel(page)
                    txt = str(e.get('feed_title') or '')
                    safe = txt.replace('&', '&amp;').replace('<', '&lt;').replace('>', '&gt;')
                    hdr.setText(safe)
                    try:
                        hdr.setMinimumWidth(0)
                        hdr.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Preferred)
                    except Exception:
                        pass
                    try:
                        f = QFont(hdr.font())
                        f.setBold(True)
                        hdr.setFont(f)
                    except Exception:
                        pass
                    try:
                        bg = getattr(self, '_feed_header_bg', 'rgba(0,0,0,18)')
                        fg = getattr(self, '_feed_header_fg', '')
                        css = 'padding:4px 8px;'
                        css += 'background:%s;' % bg
                        if fg:
                            css += 'color:%s;' % fg
                        hdr.setStyleSheet(css)
                    except Exception:
                        hdr.setStyleSheet('padding:4px 8px;')
                    v.addWidget(hdr)
                    continue

                try:
                    eff = dict(getattr(self, '_effective_colors', {}) or {})
                except Exception:
                    eff = {}
                item = PopupNewsItem(
                    e,
                    show_feed_icon=True,
                    colors=eff,
                    item_min_height_px=item_min_h,
                    item_padding_v_px=item_pad_v,
                    parent=page,
                )
                # Relay item-level signals up to the popup. Use `.emit` instead of
                # connecting signal-to-signal, which can be unreliable across
                # Per-item mark_read removed from UI; keep only popup-level actions.
                try:
                    item.open_in_app.connect(self.open_in_app.emit)
                except Exception:
                    pass
                try:
                    item.open_external.connect(self.open_external.emit)
                except Exception:
                    pass
                try:
                    item.mark_read.connect(self.mark_read.emit)
                except Exception:
                    pass
                v.addWidget(item)

            v.addStretch(1)
            page.setLayout(v)
            self._stack.addWidget(page)
            self._pages.append(page)

        try:
            self._btn_prev.setEnabled(len(self._pages) > 1)
            self._btn_next.setEnabled(len(self._pages) > 1)
        except Exception:
            pass

    def iter_news_items(self):
        # yield PopupNewsItem widgets from all pages
        for i in range(self._stack.count()):
            w = self._stack.widget(i)
            if w is None:
                continue
            lay = w.layout()
            if lay is None:
                continue
            for j in range(lay.count()):
                it = lay.itemAt(j)
                try:
                    ww = it.widget()
                except Exception:
                    ww = None
                if isinstance(ww, PopupNewsItem):
                    yield ww

    def showEvent(self, e):
        try:
            self._position_on_screen()
        except Exception:
            pass
        try:
            if self._timeout_ms > 0:
                self._timer.start(self._timeout_ms)
        except Exception:
            pass
        try:
            return QWidget.showEvent(self, e)
        except Exception:
            return

    def changeEvent(self, e):
        # React to calibre palette/theme changes (useful when popup_theme is Auto).
        try:
            et = e.type()
        except Exception:
            et = None
        try:
            # Avoid StyleChange here: on some platforms/style engines this can fire
            # repeatedly while the widget is visible, which is expensive because our
            # refresh rebuilds pages.
            if et in (QEvent.Type.PaletteChange, QEvent.Type.ApplicationPaletteChange):
                try:
                    self._schedule_theme_refresh()
                except Exception:
                    pass
        except Exception:
            pass
        try:
            return QWidget.changeEvent(self, e)
        except Exception:
            return

    def closeEvent(self, e):
        try:
            self._closing = True
        except Exception:
            pass
        try:
            self._timer.stop()
        except Exception:
            pass
        try:
            self.closed.emit()
        except Exception:
            pass
        try:
            return QWidget.closeEvent(self, e)
        except Exception:
            return

    def _is_deleted(self):
        try:
            import sip
            return bool(sip.isdeleted(self))
        except Exception:
            try:
                from PyQt5 import sip  # type: ignore
                return bool(sip.isdeleted(self))
            except Exception:
                return False

    def _schedule_theme_refresh(self):
        try:
            if getattr(self, '_closing', False):
                return
        except Exception:
            return
        try:
            if self._is_deleted():
                return
        except Exception:
            pass

        try:
            if getattr(self, '_theme_refresh_scheduled', False):
                return
            self._theme_refresh_scheduled = True
        except Exception:
            pass

        # Throttle to avoid tight refresh loops.
        try:
            QTimer.singleShot(150, self._do_theme_refresh)
        except Exception:
            try:
                QTimer.singleShot(150, lambda: self._do_theme_refresh())
            except Exception:
                pass

    def _do_theme_refresh(self):
        try:
            self._theme_refresh_scheduled = False
        except Exception:
            pass

        try:
            if getattr(self, '_closing', False):
                return
        except Exception:
            return
        try:
            if self._is_deleted():
                return
        except Exception:
            pass
        try:
            if not self.isVisible():
                return
        except Exception:
            pass

        try:
            cur = int(getattr(self, '_page_index', 0) or 0)
        except Exception:
            cur = 0
        try:
            self._apply_theme()
        except Exception:
            pass
        try:
            self._build_pages()
        except Exception:
            pass
        try:
            self._goto_page(cur)
        except Exception:
            pass
        try:
            self._position_on_screen()
        except Exception:
            pass

    def enterEvent(self, e):
        try:
            if DEBUG_RSS_READER:
                _debug('PopupNotification.enterEvent')
        except Exception:
            pass
        try:
            self._timer.stop()
        except Exception:
            pass
        try:
            return QWidget.enterEvent(self, e)
        except Exception:
            return

    def leaveEvent(self, e):
        try:
            if self._timeout_ms > 0:
                self._timer.start(self._timeout_ms)
        except Exception:
            pass
        try:
            return QWidget.leaveEvent(self, e)
        except Exception:
            return

    def eventFilter(self, obj, event):
        return QWidget.eventFilter(self, obj, event)

    def _goto_page(self, idx):
        try:
            idx = int(idx)
        except Exception:
            idx = 0
        if idx < 0:
            idx = 0
        if idx >= len(self._pages):
            idx = max(0, len(self._pages) - 1)
        self._page_index = idx
        try:
            self._stack.setCurrentIndex(idx)
        except Exception:
            pass
        try:
            if len(self._pages) <= 1:
                self._lbl_page.setText('')
            else:
                self._lbl_page.setText(_('Page %d / %d') % (idx + 1, len(self._pages)))
        except Exception:
            pass

    def _position_on_screen(self):
        try:
            app = QApplication.instance()
            screen = None
            try:
                screen = app.primaryScreen() if app else None
            except Exception:
                screen = None
            geo = None
            try:
                if screen is not None:
                    geo = screen.availableGeometry()
            except Exception:
                geo = None

            if geo is None:
                try:
                    desktop = QApplication.desktop()
                    geo = desktop.availableGeometry(self)
                except Exception:
                    geo = None
            if geo is None:
                return

            try:
                pos = str(getattr(self, '_position_override', None) or '').strip().lower()
            except Exception:
                pos = ''
            if not pos:
                try:
                    pos = str(plugin_prefs.get('popup_position', 'bottom_right') or 'bottom_right').strip().lower()
                except Exception:
                    pos = 'bottom_right'
            allowed = ('bottom_right', 'bottom_left', 'top_right', 'top_left', 'bottom_center', 'top_center', 'right_middle', 'left_middle')
            if pos not in allowed:
                pos = 'bottom_right'

            try:
                if getattr(self, '_margin_override', None) is not None:
                    margin = int(self._margin_override)
                else:
                    margin = int(plugin_prefs.get('popup_margin_px', 10) or 10)
            except Exception:
                margin = 10
            margin = max(0, margin)

            # Sizing: keep close to QuiteRSS proportions
            min_w = 120
            w = min(520, max(260, int(geo.width() * 0.33)))
            h = min(360, max(220, int(geo.height() * 0.30)))

            # Optional user override: if explicitly provided (even 0), do not fall back to prefs.
            try:
                if getattr(self, '_width_override', None) is not None:
                    pref_w = int(self._width_override)
                else:
                    pref_w = int(plugin_prefs.get('popup_width_px', 0) or 0)
            except Exception:
                pref_w = 0
            if pref_w and pref_w > 0:
                try:
                    max_w = max(min_w, int(geo.width() - (2 * margin)))
                    w = max(min_w, min(int(pref_w), max_w))
                except Exception:
                    pass
            try:
                if pref_w and pref_w > 0:
                    self.setFixedWidth(int(w))
                else:
                    self.setMinimumWidth(int(min_w))
                    self.setMaximumWidth(16777215)
            except Exception:
                pass
            self.resize(w, h)

            # Horizontal
            if pos in ('left_middle',):
                x = geo.x() + margin
            elif pos.endswith('left'):
                x = geo.x() + margin
            elif pos in ('right_middle',):
                x = geo.x() + geo.width() - self.width() - margin
            elif pos.endswith('right'):
                x = geo.x() + geo.width() - self.width() - margin
            else:
                x = geo.x() + int((geo.width() - self.width()) / 2)

            # Vertical
            if pos.endswith('middle'):
                y = geo.y() + int((geo.height() - self.height()) / 2)
            elif pos.startswith('top'):
                y = geo.y() + margin
            else:
                y = geo.y() + geo.height() - self.height() - margin

            # Clamp into available geometry
            try:
                x = max(geo.x(), min(x, geo.x() + geo.width() - self.width()))
                y = max(geo.y(), min(y, geo.y() + geo.height() - self.height()))
            except Exception:
                x = max(0, x)
                y = max(0, y)

            self.move(int(x), int(y))
        except Exception:
            pass

    def _position_bottom_right(self):
        # Backwards compatible alias
        try:
            self._position_on_screen()
        except Exception:
            pass
