"""OPML import/export helpers for RSS Reader.

These functions mirror methods on RSSReaderDialog but live outside `ui.py`
so the main UI module stays smaller. They expect a `self` argument that is an
instance of RSSReaderDialog (or compatible).
"""

from __future__ import absolute_import

import traceback
import uuid
import xml.etree.ElementTree as ET

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import QFileDialog, QMessageBox, QDialog
except Exception:  # pragma: no cover
    from PyQt5.Qt import QFileDialog, QMessageBox, QDialog

try:
    from calibre.gui2 import error_dialog
except Exception:  # pragma: no cover
    error_dialog = None

try:
    from calibre.gui2.qt_file_dialogs import choose_save_file
except Exception:  # pragma: no cover
    choose_save_file = None

from calibre_plugins.rss_reader import rss_db
from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader.dialogs import _OpmlFeedPickerDialog
from calibre_plugins.rss_reader.utils import safe_folder_path, parse_opml_file


def write_opml(fname, feeds):
    opml = ET.Element('opml', version='1.0')
    head = ET.SubElement(opml, 'head')
    ET.SubElement(head, 'title').text = _('RSS Reader export')
    body = ET.SubElement(opml, 'body')
    for f in (feeds or []):
        outline = ET.SubElement(body, 'outline')
        outline.set('text', f.get('title') or '')
        outline.set('title', f.get('title') or '')
        outline.set('xmlUrl', f.get('url') or '')
        lang = f.get('language') or ''
        if lang:
            outline.set('language', lang)
        try:
            folder = str(f.get('folder') or '').strip().strip('/')
            if folder:
                outline.set('category', folder)
        except Exception:
            pass
    tree = ET.ElementTree(opml)
    tree.write(fname, encoding='utf-8', xml_declaration=True)


def import_opml(self):
    try:
        res = QFileDialog.getOpenFileName(
            self,
            _('Import OPML'),
            '',
            'OPML files (*.opml);;XML files (*.xml);;All files (*)',
        )
        if isinstance(res, (tuple, list)):
            fname = res[0]
        else:
            fname = res
        if not fname:
            return

        default_folder = self.selected_folder_path() or str(plugin_prefs.get('default_folder', '') or '')
        candidates = list(parse_opml_file(fname, default_folder=default_folder) or [])

        feeds = list(rss_db.get_feeds() or [])
        urls = {str(f.get('url') or '') for f in feeds}

        added = 0
        folders_added = set()
        new_feed_ids = []

        try:
            default_oldest = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
        except Exception:
            default_oldest = 7
        try:
            default_max = int(plugin_prefs.get('default_max_articles', 100) or 100)
        except Exception:
            default_max = 100

        for e in candidates:
            url = str(e.get('url') or '').strip()
            if not url or url in urls:
                continue
            title = str(e.get('title') or url)
            folder = safe_folder_path(e.get('folder'), default_folder=default_folder)
            new_id = str(uuid.uuid4())
            feeds.append({
                'id': new_id,
                'title': title or url,
                'url': url,
                'enabled': True,
                'folder': folder,
                'always_notify': False,
                'feed_starred': False,
                'download_images': True,
                'is_recipe': False,
                'recipe_urn': '',
                'use_recipe_engine': False,
                'oldest_article_days': max(0, int(default_oldest)),
                'max_articles': max(0, int(default_max)),
            })
            urls.add(url)
            folders_added.add(folder)
            added += 1
            new_feed_ids.append(new_id)

        if added > 0:
            rss_db.save_feeds(feeds)
            try:
                folders = list(rss_db.get_folders() or [])
                for fp in sorted({str(x).strip().strip('/') for x in folders_added if str(x).strip()}):
                    if fp and fp not in folders:
                        folders.append(fp)
                rss_db.save_folders(folders)
            except Exception:
                pass

            QMessageBox.information(self, _('Import complete'), _('%d feeds added.') % added)
            try:
                self.refresh()
            except Exception:
                try:
                    self.load_feeds()
                except Exception:
                    pass

            if getattr(self, 'action', None) is not None and new_feed_ids:
                try:
                    reply = QMessageBox.question(
                        self,
                        _('Update imported feeds?'),
                        _('Update imported feeds now?'),
                        QMessageBox.Yes | QMessageBox.No,
                        QMessageBox.Yes,
                    )
                except Exception:
                    reply = QMessageBox.Yes

                if reply == QMessageBox.Yes:
                    try:
                        self.status.setText(_('Updating imported feeds...'))
                    except Exception:
                        pass
                    try:
                        self.set_busy(True, _('Updating imported feeds...'))
                    except Exception:
                        pass
                    try:
                        self.action.update_all_feeds(feed_ids=new_feed_ids)
                    except Exception:
                        pass
        else:
            QMessageBox.information(self, _('Import complete'), _('No new feeds found.'))
    except Exception as e:
        if error_dialog is not None:
            error_dialog(
                self,
                _('Import error'),
                _('Failed to import OPML: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )


def import_opml_advanced(self):
    try:
        res = QFileDialog.getOpenFileName(
            self,
            _('Advanced import OPML'),
            '',
            'OPML files (*.opml);;XML files (*.xml);;All files (*)',
        )
        if isinstance(res, (tuple, list)):
            fname = res[0]
        else:
            fname = res
        if not fname:
            return

        default_folder = self.selected_folder_path() or str(plugin_prefs.get('default_folder', '') or '')
        candidates = list(parse_opml_file(fname, default_folder=default_folder) or [])
        if not candidates:
            QMessageBox.information(self, _('Advanced import'), _('No feeds found in this OPML file.'))
            return

        existing = list(rss_db.get_feeds() or [])
        existing_urls = {str(f.get('url') or '').strip() for f in existing}

        dlg = _OpmlFeedPickerDialog(
            self,
            _('Advanced OPML import'),
            candidates,
            prechecked_urls=[
                e.get('url')
                for e in candidates
                if str(e.get('url') or '').strip() not in existing_urls
            ],
            disabled_urls=existing_urls,
            help_text=_('Choose which feeds to import from this OPML file.'),
        )
        try:
            dlg.table.setHorizontalHeaderLabels([_('Import'), _('Title'), _('Folder'), _('URL')])
        except Exception:
            pass

        if dlg.exec() != QDialog.DialogCode.Accepted:
            return

        chosen = list(dlg.selected_entries() or [])
        if not chosen:
            QMessageBox.information(self, _('Advanced import'), _('No feeds selected.'))
            return

        feeds = list(rss_db.get_feeds() or [])
        urls = {str(f.get('url') or '').strip() for f in feeds}
        folders_added = set()
        added = 0
        skipped = 0
        new_feed_ids = []

        try:
            default_oldest = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
        except Exception:
            default_oldest = 7
        try:
            default_max = int(plugin_prefs.get('default_max_articles', 100) or 100)
        except Exception:
            default_max = 100

        for e in chosen:
            url = str(e.get('url') or '').strip()
            if not url:
                continue
            if url in urls:
                skipped += 1
                continue
            title = str(e.get('title') or url)
            folder = safe_folder_path(e.get('folder'), default_folder=default_folder)
            new_id = str(uuid.uuid4())
            feeds.append({
                'id': new_id,
                'title': title or url,
                'url': url,
                'enabled': True,
                'folder': folder,
                'always_notify': False,
                'feed_starred': False,
                'download_images': True,
                'is_recipe': False,
                'recipe_urn': '',
                'use_recipe_engine': False,
                'oldest_article_days': max(0, int(default_oldest)),
                'max_articles': max(0, int(default_max)),
            })
            urls.add(url)
            folders_added.add(folder)
            added += 1
            new_feed_ids.append(new_id)

        if added <= 0:
            QMessageBox.information(self, _('Advanced import'), _('Nothing to import (all selected feeds already exist).'))
            return

        rss_db.save_feeds(feeds)
        try:
            folders = list(rss_db.get_folders() or [])
            for fp in sorted({str(x).strip().strip('/') for x in folders_added if str(x).strip()}):
                if fp and fp not in folders:
                    folders.append(fp)
            rss_db.save_folders(folders)
        except Exception:
            pass

        msg = _('%d feeds added.') % added
        if skipped:
            msg = msg + '\n' + (_('%d skipped (already existed).') % skipped)
        QMessageBox.information(self, _('Import complete'), msg)

        try:
            self.refresh()
        except Exception:
            try:
                self.load_feeds()
            except Exception:
                pass

        if getattr(self, 'action', None) is not None and new_feed_ids:
            try:
                reply = QMessageBox.question(
                    self,
                    _('Update imported feeds?'),
                    _('Update imported feeds now?'),
                    QMessageBox.Yes | QMessageBox.No,
                    QMessageBox.Yes,
                )
            except Exception:
                reply = QMessageBox.Yes

            if reply == QMessageBox.Yes:
                try:
                    self.status.setText(_('Updating imported feeds...'))
                except Exception:
                    pass
                try:
                    self.set_busy(True, _('Updating imported feeds...'))
                except Exception:
                    pass
                try:
                    self.action.update_all_feeds(feed_ids=new_feed_ids)
                except Exception:
                    pass
    except Exception as e:
        if error_dialog is not None:
            error_dialog(
                self,
                _('Import error'),
                _('Failed to import OPML: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )


def export_opml(self):
    try:
        if choose_save_file is None:
            return

        fname = choose_save_file(
            self,
            'rss-reader-export-opml',
            _('Export OPML'),
            filters=[(_('OPML files'), ['opml']), (_('XML files'), ['xml'])],
            all_files=True,
            initial_filename='feeds.opml',
        )
        if not fname:
            return

        write_opml(fname, list(rss_db.get_feeds() or []))
        QMessageBox.information(self, _('Export complete'), _('OPML exported to %s') % fname)
    except Exception as e:
        if error_dialog is not None:
            error_dialog(
                self,
                _('Export error'),
                _('Failed to export OPML: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )


def export_opml_advanced(self):
    try:
        feeds = list(rss_db.get_feeds() or [])
        if not feeds:
            QMessageBox.information(self, _('Advanced export'), _('No feeds configured.'))
            return

        try:
            selected_ids = set(str(x) for x in (self.selected_feed_ids() or []) if str(x).strip())
        except Exception:
            selected_ids = set()

        entries = []
        prechecked_urls = []
        for f in feeds:
            url = str(f.get('url') or '').strip()
            if not url:
                continue
            title = str(f.get('title') or url)
            folder = str(f.get('folder') or '')
            entries.append({'title': title, 'folder': folder, 'url': url})
            if selected_ids:
                if str(f.get('id') or '') in selected_ids:
                    prechecked_urls.append(url)
            else:
                prechecked_urls.append(url)

        dlg = _OpmlFeedPickerDialog(
            self,
            _('Advanced OPML export'),
            entries,
            prechecked_urls=prechecked_urls,
            disabled_urls=None,
            help_text=_('Choose which feeds to export to OPML.'),
        )
        try:
            dlg.table.setHorizontalHeaderLabels([_('Export'), _('Title'), _('Folder'), _('URL')])
        except Exception:
            pass

        if dlg.exec() != QDialog.DialogCode.Accepted:
            return

        chosen = list(dlg.selected_entries() or [])
        if not chosen:
            QMessageBox.information(self, _('Advanced export'), _('No feeds selected.'))
            return

        chosen_urls = {str(e.get('url') or '').strip() for e in chosen if str(e.get('url') or '').strip()}
        export_feeds = [f for f in feeds if str(f.get('url') or '').strip() in chosen_urls]

        if choose_save_file is None:
            return

        fname = choose_save_file(
            self,
            'rss-reader-export-opml',
            _('Export OPML'),
            filters=[(_('OPML files'), ['opml']), (_('XML files'), ['xml'])],
            all_files=True,
            initial_filename='feeds.opml',
        )
        if not fname:
            return

        write_opml(fname, export_feeds)
        QMessageBox.information(self, _('Export complete'), _('OPML exported to %s') % fname)
    except Exception as e:
        if error_dialog is not None:
            error_dialog(
                self,
                _('Export error'),
                _('Failed to export OPML: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )
