# Helper functions for RSS Reader items filtering and search.
# These operate on a dialog-like object (`self`) that provides
# the same attributes/methods as RSSReaderDialog in ui.py.

from __future__ import absolute_import

import re
import time

try:
    from qt.core import Qt
except Exception:  # pragma: no cover - fallback for older Qt bindings
    from PyQt5.Qt import Qt

from calibre_plugins.rss_reader import rss_db
from calibre_plugins.rss_reader.rss import normalize_summary_to_html
from calibre_plugins.rss_reader.config import plugin_prefs

# NOTE: `gprefs` is calibre's global GUI prefs store (dict-like). We import it
# directly from calibre to avoid coupling to the plugin's config module.
try:
    from calibre.gui2 import gprefs
except Exception:  # pragma: no cover
    gprefs = {}

try:
    load_translations()
except NameError:
    pass

# Define ROLE_USER constant for table items
# This mirrors the definition in ui.py to avoid circular imports
ROLE_USER = None


def _ensure_role_user():
    """Ensure ROLE_USER is available."""
    global ROLE_USER
    if ROLE_USER is None:
        try:
            # Try to get it from ui module if already loaded
            from calibre_plugins.rss_reader.ui import ROLE_USER as UI_ROLE_USER
            ROLE_USER = UI_ROLE_USER
        except Exception:
            # Fallback: compute it directly from Qt
            try:
                ROLE_USER = getattr(Qt, 'UserRole', 32)  # Qt.UserRole = 32
            except Exception:
                ROLE_USER = 32  # Hard default
    return ROLE_USER


def schedule_filter_items(self, text):
    try:
        try:
            from qt.core import QTimer
        except Exception:
            from PyQt5.Qt import QTimer
        self._pending_filter_text = str(text or '')
        if getattr(self, '_filter_items_timer', None) is None:
            self._filter_items_timer = QTimer(self)
            self._filter_items_timer.setSingleShot(True)
            self._filter_items_timer.timeout.connect(self._apply_debounced_filter)
        try:
            interval = int(plugin_prefs.get('items_filter_debounce_ms', 350) or 350)
        except Exception:
            interval = 350
        self._filter_items_timer.start(max(50, int(interval)))
    except Exception:
        try:
            filter_items(self, text)
        except Exception:
            pass


def apply_debounced_filter(self):
    try:
        txt = getattr(self, '_pending_filter_text', '') or ''
        filter_items(self, txt)
    except Exception:
        pass


def filter_items(self, text):
    query = str(text or '').strip()
    try:
        qf = str(getattr(self, '_items_quick_filter_query', '') or '').strip()
    except Exception:
        qf = ''
    if qf:
        query = ('%s %s' % (query, qf)).strip()

    def apply_visible_set(visible_rows):
        try:
            visible_rows_local = set(int(x) for x in (visible_rows or set()))
        except Exception:
            visible_rows_local = set()
        for r in range(self.items_table.rowCount()):
            try:
                self.items_table.setRowHidden(r, r not in visible_rows_local)
            except Exception:
                pass

        # If selection is hidden, move to the first visible row.
        try:
            sel = list(self.items_table.selectionModel().selectedRows() or [])
            if sel:
                row = int(sel[0].row())
                if row not in visible_rows_local:
                    for r in range(self.items_table.rowCount()):
                        if r in visible_rows_local:
                            self.items_table.selectRow(r)
                            break
        except Exception:
            pass

        # Update the current feed label to show filtered item count
        try:
            if hasattr(self, '_update_current_feed_label'):
                self._update_current_feed_label()
        except Exception:
            pass

    if not query:
        apply_visible_set(set(range(self.items_table.rowCount())))
        return

    # Build item list in current row order
    items = []
    role = _ensure_role_user()
    for r in range(self.items_table.rowCount()):
        it0 = self.items_table.item(r, 0)
        data = it0.data(role) if it0 is not None else None  # ROLE_USER for storing item data
        if isinstance(data, dict):
            items.append(data)
        else:
            # Minimal fallback so the search doesn't crash
            it1 = self.items_table.item(r, 1)
            it2 = self.items_table.item(r, 2)
            items.append({
                'title': (it0.text() if it0 is not None else ''),
                'summary': '',
                'link': '',
                '_feed_title': '',
                'published': (it1.text() if it1 is not None else ''),
                '_tags': self._parse_tags_text(it2.text() if it2 is not None else ''),
            })

    # Try to use calibre's search language.
    try:
        from calibre.utils.search_query_parser import SearchQueryParser, ParseException
    except Exception:
        SearchQueryParser = None
        ParseException = Exception

    if SearchQueryParser is None:
        # Plain substring fallback (title/body/link/tags)
        q = query.casefold()
        visible = set()
        for r, it in enumerate(items):
            try:
                hay = ' '.join([
                    str(it.get('title') or ''),
                    str(it.get('_feed_title') or ''),
                    str(it.get('link') or ''),
                    self._strip_html_to_text(normalize_summary_to_html(it.get('summary') or '') if it.get('summary') else ''),
                    ' '.join(list(it.get('_tags') or [])),
                ])
                if q in hay.casefold():
                    visible.add(r)
            except Exception:
                pass
        apply_visible_set(visible)
        return

    if not hasattr(self, '_items_search_parse_cache'):
        try:
            self._items_search_parse_cache = {}
        except Exception:
            self._items_search_parse_cache = None

    outer = self

    class _RSSItemsSearch(SearchQueryParser):
        # Add 'c' and 'content' as aliases for 'body'/'summary'
        LOCATIONS = ['all', 'title', 'body', 'summary', 'c', 'content', 'link', 'feed', 'tag', 'tags', 'published', 'date', 'id', 'words', 'chars', 'img', 'images', 'fav', 'star', 'starred', 'unread', 'read', 'seen', 'days', 'age', 'new']

        def __init__(self, items_list):
            SearchQueryParser.__init__(self, locations=self.LOCATIONS, optimize=True, parse_cache=getattr(outer, '_items_search_parse_cache', None))
            self.items = items_list

        def universal_set(self):
            return set(range(len(self.items)))

        def _parse_comparison(self, q):
            """Parse a query like '>300', '>=100', '<50', 'true', 'false', '1'.
            Returns (op, value) or None if not a comparison.
            op is one of '>', '>=', '<', '<=', '=', 'bool'
            """
            q = str(q or '').strip()
            if not q:
                return None
            # boolean
            if q.lower() in ('true', 'yes', '1'):
                return ('bool', True)
            if q.lower() in ('false', 'no', '0'):
                return ('bool', False)
            # >=, <=
            m = re.match(r'^(>=|<=|>|<|=)\s*(\d+)$', q)
            if m:
                return (m.group(1), int(m.group(2)))
            # bare number means equality
            if q.isdigit():
                return ('=', int(q))
            return None

        def _get_word_count(self, it):
            try:
                s = it.get('summary') or ''
                s = normalize_summary_to_html(s) if s else ''
                text = outer._strip_html_to_text(s) or ''
                # Limit to 1MB for safety
                if len(text) > 1024 * 1024:
                    text = text[:1024 * 1024]
                return len([w for w in text.split() if w])
            except Exception:
                return 0

        def _get_char_count(self, it):
            try:
                s = it.get('summary') or ''
                s = normalize_summary_to_html(s) if s else ''
                text = outer._strip_html_to_text(s) or ''
                if len(text) > 1024 * 1024:
                    text = text[:1024 * 1024]
                return len(text)
            except Exception:
                return 0

        def _get_image_count(self, it):
            try:
                s = it.get('summary') or ''
                s = normalize_summary_to_html(s) if s else ''
                # Count <img tags
                return len(re.findall(r'<img\s', s, re.IGNORECASE))
            except Exception:
                return 0

        def _eval_comparison(self, value, cmp_op, cmp_val):
            """Evaluate value against comparison operator and operand."""
            if cmp_op == 'bool':
                # For boolean: true means value > 0, false means value == 0
                if cmp_val:
                    return value > 0
                else:
                    return value == 0
            if cmp_op == '>':
                return value > cmp_val
            if cmp_op == '>=':
                return value >= cmp_val
            if cmp_op == '<':
                return value < cmp_val
            if cmp_op == '<=':
                return value <= cmp_val
            if cmp_op == '=':
                return value == cmp_val
            return False

        def _parse_date_query(self, q):
            """Parse a calibre-style date query like '>6daysago' or '<=2026-01-01'.

            Returns (op, ts) where ts is a unix timestamp (seconds).
            Default op is '>' if no explicit operator is present.
            """
            q = str(q or '').strip()
            if not q:
                return None

            m = re.match(r'^(>=|<=|>|<|=)\s*(.+?)\s*$', q)
            if m:
                op, rhs = m.group(1), (m.group(2) or '').strip()
            else:
                op, rhs = '>', q

            rhs_l = rhs.casefold().replace(' ', '')

            # Relative: 6daysago / 2weeksago / 3monthsago / 1yearsago
            m2 = re.match(r'^(\d+)(day|days|week|weeks|month|months|year|years)ago$', rhs_l)
            if m2:
                n = int(m2.group(1))
                unit = m2.group(2)
                mult = 86400
                if unit.startswith('week'):
                    mult = 7 * 86400
                elif unit.startswith('month'):
                    mult = 30 * 86400
                elif unit.startswith('year'):
                    mult = 365 * 86400
                try:
                    ts = int(time.time()) - int(n * mult)
                except Exception:
                    ts = 0
                return (op, ts)

            # Try ISO date (YYYY-MM-DD or YYYY/MM/DD)
            m3 = re.match(r'^(\d{4})[-/](\d{1,2})[-/](\d{1,2})$', rhs_l)
            if m3:
                try:
                    import datetime
                    y, mo, d = int(m3.group(1)), int(m3.group(2)), int(m3.group(3))
                    ts = int(datetime.datetime(y, mo, d, 0, 0, 0).timestamp())
                except Exception:
                    ts = 0
                return (op, ts)

            return None

        def get_matches(self, location, q, candidates=None):
            loc = str(location or 'all').strip().lower()
            q = str(q or '')
            q_norm = q.casefold().strip()

            if not q_norm:
                return set(candidates) if candidates is not None else self.universal_set()

            def iter_idxs():
                if candidates is None:
                    return range(len(self.items))
                try:
                    return list(candidates)
                except Exception:
                    return range(len(self.items))

            # Check if this is a numeric/comparison location
            cmp = self._parse_comparison(q)

            out = set()
            for idx in iter_idxs():
                try:
                    it = self.items[int(idx)]
                except Exception:
                    continue
                if not isinstance(it, dict):
                    continue

                try:
                    if loc in ('fav', 'star', 'starred'):
                        is_starred = bool(it.get('_starred', False))
                        if cmp:
                            if cmp[0] == 'bool':
                                ok = is_starred == cmp[1]
                            else:
                                ok = self._eval_comparison(1 if is_starred else 0, cmp[0], cmp[1])
                        else:
                            if q_norm in ('true', 'yes', '1'):
                                ok = is_starred
                            elif q_norm in ('false', 'no', '0'):
                                ok = not is_starred
                            else:
                                ok = is_starred
                    elif loc in ('unread', 'read', 'seen'):
                        # Prefer explicit flag if present; fallback to seen_item_ids table.
                        try:
                            is_unread = bool(it.get('_unread', None))
                            if it.get('_unread', None) is None:
                                fid = str(it.get('_feed_id') or '').strip()
                                iid = str(it.get('id') or it.get('link') or it.get('title') or '').strip()
                                if fid and iid:
                                    try:
                                        seen_map = dict(rss_db.get_seen_item_ids_map() or {})
                                    except Exception:
                                        seen_map = {}
                                    seen_set = set(str(x) for x in (seen_map.get(fid, []) or []) if str(x).strip())
                                    is_unread = iid not in seen_set
                                else:
                                    is_unread = False
                        except Exception:
                            is_unread = False

                        want_unread = (loc == 'unread')
                        if loc in ('read', 'seen'):
                            want_unread = False

                        if cmp and cmp[0] == 'bool':
                            # unread:true/false, read:true/false
                            if loc == 'unread':
                                ok = (is_unread == bool(cmp[1]))
                            else:
                                ok = ((not is_unread) == bool(cmp[1]))
                        else:
                            # default: unread = true, read/seen = true
                            ok = (is_unread if want_unread else (not is_unread))

                    elif loc == 'new':
                        try:
                            is_new = bool(it.get('_new', False))
                        except Exception:
                            is_new = False
                        if cmp and cmp[0] == 'bool':
                            ok = (is_new == bool(cmp[1]))
                        else:
                            if q_norm in ('true', 'yes', '1'):
                                ok = is_new
                            elif q_norm in ('false', 'no', '0'):
                                ok = not is_new
                            else:
                                ok = is_new

                    elif loc in ('days', 'age'):
                        try:
                            published_ts = int(it.get('published_ts') or 0)
                        except Exception:
                            published_ts = 0
                        try:
                            age_days = 999999
                            if published_ts > 0:
                                age_days = (int(time.time()) - int(published_ts)) / 86400.0
                                if age_days < 0:
                                    age_days = 0
                        except Exception:
                            age_days = 999999

                        if cmp:
                            ok = self._eval_comparison(age_days, cmp[0], cmp[1])
                        else:
                            ok = (q_norm in str(int(age_days)))
                    elif loc == 'words':
                        if cmp:
                            val = self._get_word_count(it)
                            ok = self._eval_comparison(val, cmp[0], cmp[1])
                        else:
                            ok = (q_norm in str(self._get_word_count(it)))
                    elif loc == 'chars':
                        if cmp:
                            val = self._get_char_count(it)
                            ok = self._eval_comparison(val, cmp[0], cmp[1])
                        else:
                            ok = (q_norm in str(self._get_char_count(it)))
                    elif loc in ('img', 'images'):
                        if cmp:
                            val = self._get_image_count(it)
                            ok = self._eval_comparison(val, cmp[0], cmp[1])
                        else:
                            ok = (q_norm in str(self._get_image_count(it)))
                    elif loc in ('tag', 'tags'):
                        tags = it.get('_tags')
                        if not isinstance(tags, (list, tuple, set)) or tags is None:
                            try:
                                tags = outer._tags_for_item(it)
                            except Exception:
                                tags = []
                        tags_text = ' '.join([outer._normalize_tag(x) for x in (tags or []) if outer._normalize_tag(x)])
                        ok = (q_norm in tags_text.casefold())
                    elif loc == 'title':
                        ok = (q_norm in str(it.get('title') or '').casefold())
                    elif loc in ('body', 'summary', 'c', 'content'):
                        try:
                            s = it.get('summary') or ''
                            s = normalize_summary_to_html(s) if s else ''
                        except Exception:
                            s = it.get('summary') or ''
                        ok = (q_norm in outer._strip_html_to_text(s).casefold())
                    elif loc == 'link':
                        ok = (q_norm in str(it.get('link') or '').casefold())
                    elif loc == 'feed':
                        ok = (q_norm in str(it.get('_feed_title') or '').casefold())
                    elif loc == 'published':
                        ok = (q_norm in str(it.get('published') or '').casefold())
                    elif loc == 'date':
                        dq = self._parse_date_query(q)
                        if not dq:
                            ok = False
                        else:
                            op, ts_cutoff = dq
                            try:
                                published_ts = int(it.get('published_ts') or 0)
                            except Exception:
                                published_ts = 0
                            if published_ts <= 0 or ts_cutoff <= 0:
                                ok = False
                            else:
                                if op == '>':
                                    ok = published_ts > ts_cutoff
                                elif op == '>=':
                                    ok = published_ts >= ts_cutoff
                                elif op == '<':
                                    ok = published_ts < ts_cutoff
                                elif op == '<=':
                                    ok = published_ts <= ts_cutoff
                                elif op == '=':
                                    # Treat '=' as same-day match (UTC-local ambiguity is acceptable here)
                                    ok = abs(int(published_ts) - int(ts_cutoff)) < 86400
                                else:
                                    ok = False
                    elif loc == 'id':
                        ok = (q_norm in str(it.get('id') or '').casefold())
                    else:
                        tags = it.get('_tags')
                        if not isinstance(tags, (list, tuple, set)) or tags is None:
                            tags = []
                        try:
                            s = it.get('summary') or ''
                            s = normalize_summary_to_html(s) if s else ''
                        except Exception:
                            s = it.get('summary') or ''
                        hay = ' '.join([
                            str(it.get('title') or ''),
                            str(it.get('_feed_title') or ''),
                            str(it.get('link') or ''),
                            str(it.get('published') or ''),
                            outer._strip_html_to_text(s),
                            ' '.join([outer._normalize_tag(x) for x in (tags or []) if outer._normalize_tag(x)]),
                        ])
                        ok = (q_norm in hay.casefold())
                except Exception:
                    ok = False

                if ok:
                    try:
                        out.add(int(idx))
                    except Exception:
                        pass
            return out

    parser = _RSSItemsSearch(items)
    try:
        visible = parser.parse(query)
        apply_visible_set(visible)
        try:
            if getattr(self, 'status', None) is not None:
                self.status.setText('')
        except Exception:
            pass
    except ParseException as e:  # type: ignore[name-defined]
        # While typing, the query is often temporarily invalid (e.g. unmatched quotes).
        # Fall back to simple substring search so the UI remains usable.
        try:
            if getattr(self, 'status', None) is not None:
                self.status.setText(_('Invalid search: %s') % str(getattr(e, 'msg', '') or e))
        except Exception:
            pass
        q = query.casefold()
        visible = set()
        for r, it in enumerate(items):
            try:
                hay = ' '.join([
                    str(it.get('title') or ''),
                    str(it.get('_feed_title') or ''),
                    str(it.get('link') or ''),
                    self._strip_html_to_text(normalize_summary_to_html(it.get('summary') or '') if it.get('summary') else ''),
                    ' '.join(list(it.get('_tags') or [])),
                ])
                if q in hay.casefold():
                    visible.add(r)
            except Exception:
                pass
        apply_visible_set(visible)


def collect_items_for_feeds(feed_ids, max_total=200000):
    """Collect matching items from the DB for the given feed_ids.

    Uses `rss_db.search_items` in paged mode to avoid loading enormous
    result sets into memory in one call. Returns a list of item dicts
    sorted newest-first.
    """
    out = []
    try:
        step = 2000
        offset = 0
        # page through results
        while True:
            try:
                chunk = list(rss_db.search_items(feed_ids=feed_ids, query='', limit=step, offset=offset) or [])
            except Exception:
                chunk = []
            if not chunk:
                break
            out.extend(chunk)
            if len(out) >= int(max_total or 0):
                break
            if len(chunk) < step:
                break
            offset += step
    except Exception:
        pass
    return out


def render_items_table(self, items, fids, seen_map, tags_map_by_feed):
    """Populate `self.items_table` from `items` list.

    This encapsulates the existing population logic so both the normal
    loader and the DB-backed search can reuse it. `items` is expected
    to be newest-first.
    """
    from calibre_plugins.rss_reader.config import plugin_prefs as _plugin_prefs
    try:
        from calibre.gui2 import gprefs as _gprefs
    except Exception:  # pragma: no cover
        _gprefs = {}
    from calibre_plugins.rss_reader.ui import ROLE_USER as _ROLE_USER  # circular-safe usage
    from calibre_plugins.rss_reader.rss import iso_to_ts, format_published_display

    try:
        from qt.core import QTableWidgetItem, QColor, QFont
    except Exception:
        from PyQt5.Qt import QTableWidgetItem, QColor, QFont

    try:
        from calibre_plugins.rss_reader.ui import SortKeyTableWidgetItem
    except Exception:
        SortKeyTableWidgetItem = None

    try:
        max_items = int(_plugin_prefs.get('max_items_per_selection', 2000) or 2000)
    except Exception:
        max_items = 2000

    # Preserve the old behavior of applying the UI cap elsewhere; caller
    # can choose to slice `items` before passing here. We only update
    # the status message if caller included the cap.
    was_sorting = False
    try:
        was_sorting = bool(self.items_table.isSortingEnabled())
        self.items_table.setSortingEnabled(False)
    except Exception:
        pass

    try:
        self.items_table.setRowCount(len(items))
        try:
            import html as _html
        except Exception:
            _html = None

        auto_tagging_enabled = bool(_plugin_prefs.get('auto_tagging_enabled', True))

        for row, it in enumerate(items):
            try:
                title = (_html.unescape(it.get('title') or '') if _html else (it.get('title') or ''))
            except Exception:
                title = it.get('title') or ''
            if len(fids) > 1:
                ft = it.get('_feed_title') or ''
                if ft:
                    title = f'[{ft}] {title}'
            published = it.get('published') or ''
            ts = int(it.get('published_ts') or iso_to_ts(published) or 0)
            published_disp = format_published_display(ts, published)

            try:
                author = (_html.unescape(it.get('author') or '') if _html else (it.get('author') or ''))
            except Exception:
                author = it.get('author') or ''
            try:
                author = str(author or '').strip()
            except Exception:
                author = ''

            is_starred = bool(it.get('_starred', False))
            t_star = QTableWidgetItem('★' if is_starred else '')
            try:
                t_star.setTextAlignment(Qt.AlignmentFlag.AlignCenter)
            except Exception:
                try:
                    t_star.setTextAlignment(Qt.AlignCenter)
                except Exception:
                    pass
            if is_starred:
                try:
                    t_star.setForeground(QColor(255, 180, 0))
                except Exception:
                    pass

            t0 = QTableWidgetItem(title)
            t_author = QTableWidgetItem(author)
            if SortKeyTableWidgetItem is not None:
                t1 = SortKeyTableWidgetItem(published_disp, ts)
            else:
                t1 = QTableWidgetItem(published_disp)

            tags_text = ''
            try:
                tags = self._tags_for_item(it, tags_map_by_feed=tags_map_by_feed)
                it['_tags'] = list(tags)
                tags_text = ', '.join(list(tags) or [])
            except Exception:
                tags_text = ''
            t2 = QTableWidgetItem(tags_text)

            try:
                fid0 = str(it.get('_feed_id') or '').strip()
            except Exception:
                fid0 = ''
            try:
                iid0 = str(it.get('id') or '').strip()
            except Exception:
                iid0 = ''

            manual_tags = []
            try:
                if fid0 and iid0 and isinstance(tags_map_by_feed, dict):
                    m = (tags_map_by_feed.get(fid0) or {}).get(iid0)
                    if isinstance(m, list):
                        manual_tags = m
            except Exception:
                manual_tags = []

            auto_tags = []
            try:
                auto_tags = list(self._auto_tags_for_item(it) or [])
            except Exception:
                auto_tags = []

            try:
                lines = []
                if manual_tags:
                    mt = [self._normalize_tag(x) for x in (manual_tags or [])]
                    mt = [x for x in mt if x]
                    if mt:
                        lines.append(_('Manual: %s') % ', '.join(mt))
                if auto_tags:
                    at = [self._normalize_tag(x) for x in (auto_tags or [])]
                    at = [x for x in at if x]
                    if at:
                        lines.append(_('Auto: %s') % ', '.join(at))
                if lines:
                    t2.setToolTip('\n'.join(lines))
            except Exception:
                pass
            try:
                if published:
                    t1.setToolTip(str(published))
            except Exception:
                pass
            t_star.setData(_ROLE_USER, it)
            t0.setData(_ROLE_USER, it)
            t_author.setData(_ROLE_USER, it)
            t1.setData(_ROLE_USER, it)
            t2.setData(_ROLE_USER, it)

            try:
                fid1 = str(it.get('_feed_id') or '')
                iid1 = str(it.get('id') or '')
                seen_set = set(seen_map.get(fid1, []) or [])
                is_unread = bool(iid1) and (iid1 not in seen_set)
                if is_unread:
                    try:
                        f0 = QFont(t0.font())
                        f0.setBold(True)
                        t0.setFont(f0)
                    except Exception:
                        pass
                    try:
                        fa = QFont(t_author.font())
                        fa.setBold(True)
                        t_author.setFont(fa)
                    except Exception:
                        pass
                    try:
                        fp = QFont(t1.font())
                        fp.setBold(True)
                        t1.setFont(fp)
                    except Exception:
                        pass
                    try:
                        ft = QFont(t2.font())
                        ft.setBold(True)
                        t2.setFont(ft)
                    except Exception:
                        pass
            except Exception:
                pass
            self.items_table.setItem(row, 0, t_star)
            self.items_table.setItem(row, 1, t0)
            self.items_table.setItem(row, 2, t_author)
            self.items_table.setItem(row, 3, t1)
            self.items_table.setItem(row, 4, t2)
    finally:
        try:
            self.items_table.setSortingEnabled(was_sorting)
        except Exception:
            pass

    try:
        self.items_table.setColumnWidth(0, 30)
    except Exception:
        pass

    try:
        sc = _gprefs.get('rss_reader_sort_column', None)
        so = _gprefs.get('rss_reader_sort_order', None)
        if sc is None or so is None:
            sc, so = 3, int(Qt.SortOrder.DescendingOrder)
        try:
            _prev_cols = _gprefs.get('rss_reader_items_table_colcount', None)  # noqa: F841
        except Exception:
            _prev_cols = None  # noqa: F841
        try:
            self.items_table.sortItems(int(sc), int(so))
        except Exception:
            try:
                self.items_table.sortItems(3, int(Qt.SortOrder.DescendingOrder))
            except Exception:
                pass
    except Exception:
        pass

