from __future__ import annotations

import io
import sys
import gettext

# ---------------------------------------------------------------------------
# Language → (pref_key, pref_default, native_locale, latin_locale)
# Each entry maps a calibre UI language prefix to the config key that stores
# the user's script preference and the .mo locale names for native / Latin.
# ---------------------------------------------------------------------------
_SCRIPT_LANGS = {
    'sr': ('serbian_graphia', 'latin',  'sr',    'sr_Latn'),
    'ru': ('russian_graphia', 'native', 'ru',    'ru_Latn'),
    'ja': ('japanese_graphia', 'native', 'ja',   'ja_Latn'),
    'ko': ('korean_graphia', 'native', 'ko', 'ko_Latn'),
    'zh_cn': ('chinese_graphia', 'native', 'zh_CN', 'zh_CN_Latn'),
    'zh_tw': ('chinese_graphia', 'native', 'zh_TW', 'zh_TW_Latn'),
    'zh_hk': ('chinese_graphia', 'native', 'zh_HK', 'zh_HK_Latn'),
    'zh':    ('chinese_graphia', 'native', 'zh_CN', 'zh_CN_Latn'),  # bare zh → zh_CN
    'hi': ('hindi_graphia', 'native', 'hi', 'hi_Latn'),
    'ar': ('arabic_graphia', 'native', 'ar', 'ar_Latn'),
    'he': ('hebrew_graphia', 'native', 'he', 'he_Latn'),
    'el': ('greek_graphia', 'native', 'el', 'el_Latn'),
}


def _get_calibre_lang() -> str:
    try:
        from calibre.utils.localization import get_lang  # type: ignore

        return str(get_lang() or '').strip()
    except Exception:
        return ''


def _get_pref(key: str, default: str) -> str:
    """Read a plugin preference by key, with fallback. Returns lowercase."""
    try:
        from calibre_plugins.rss_reader.config import plugin_prefs  # type: ignore

        val = str(plugin_prefs.get(key, default) or default).strip().lower()
        return val
    except Exception:
        pass

    try:
        from calibre.utils.config import JSONConfig  # type: ignore

        prefs = JSONConfig('plugins/rss_reader/rss_reader')
        val = str(prefs.get(key, default) or default).strip().lower()
        return val
    except Exception:
        return default


def _get_pref_raw(key: str, default: str) -> str:
    """Read a plugin preference by key, preserving case (for locale codes like zh_CN)."""
    try:
        from calibre_plugins.rss_reader.config import plugin_prefs  # type: ignore

        val = str(plugin_prefs.get(key, default) or default).strip()
        return val
    except Exception:
        pass

    try:
        from calibre.utils.config import JSONConfig  # type: ignore

        prefs = JSONConfig('plugins/rss_reader/rss_reader')
        val = str(prefs.get(key, default) or default).strip()
        return val
    except Exception:
        return default


def _resolve_target_locale() -> str:
    """Determine which .mo locale to load based on user preference or calibre language + script prefs.

    Returns the locale string (e.g. 'sr_Latn', 'ru', 'ja_Latn', …) or ''
    if no override is needed.
    """
    # Check if user set an explicit plugin UI language preference
    plugin_ui_lang = _get_pref_raw('plugin_ui_language', '').strip()
    if plugin_ui_lang:
        # Explicit English means: reset to source strings (NullTranslations).
        # Do NOT load en.mo — its msgstr fields are empty and will blank the UI.
        try:
            if plugin_ui_lang.lower().startswith('en'):
                return '__en_reset__'
        except Exception:
            pass
        return plugin_ui_lang
    
    lang = _get_calibre_lang().lower()
    if not lang:
        return ''

    # Try most-specific match first (e.g. zh_cn before zh)
    match = None
    for prefix in sorted(_SCRIPT_LANGS.keys(), key=len, reverse=True):
        if lang.startswith(prefix):
            match = _SCRIPT_LANGS[prefix]
            break

    if match is None:
        return ''

    pref_key, pref_default, native_locale, latin_locale = match
    pref_val = _get_pref(pref_key, pref_default)

    if pref_val == 'latin':
        return latin_locale
    elif pref_key == 'serbian_graphia' and pref_val == 'cyrillic':
        return native_locale
    else:
        # 'native' — use calibre's default .mo (no override needed)
        return ''


def _patch_to_auto() -> bool:
    """Patch the translation cache to match the current calibre language (Auto mode).

    Calls _resolve_target_locale() first (handles script-lang overrides like
    ru_Latn). If that returns '' it falls through to loading calibre's own .mo
    for the plugin (e.g. translations/fr.mo when calibre is set to French).
    Returns True if the cache was updated.
    """
    target = _resolve_target_locale()
    if target == '__en_reset__':
        return _reset_to_source_language()
    if target:
        return _patch_translations_cache(target)

    # No script-lang override — load calibre's own UI-language .mo from the zip.
    lang = _get_calibre_lang()
    if not lang:
        return False
    for candidate in [lang, lang.split('_')[0], lang.lower(), lang.lower().split('_')[0]]:
        if _patch_translations_cache(candidate):
            return True
    return False


def _find_plugin_zfp() -> str:
    """Find the zip file path for this plugin from any loaded module's
    ``load_translations`` partial.
    """
    for name, mod in sys.modules.items():
        if not name or not name.startswith('calibre_plugins.rss_reader'):
            continue
        d = getattr(mod, '__dict__', None)
        if not isinstance(d, dict):
            continue
        lt = d.get('load_translations')
        if lt is None:
            continue
        try:
            args = getattr(lt, 'args', None)
            if args and len(args) >= 2:
                candidate = args[-1]
                if candidate and isinstance(candidate, str) and candidate.endswith('.zip'):
                    return candidate
        except Exception:
            pass
    return ''


def _load_mo_bytes(locale: str) -> bytes:
    """Load .mo data from the plugin zip via calibre's injected ``get_resources()``."""
    path = f'translations/{locale}.mo'

    for name, mod in sys.modules.items():
        if not name or not name.startswith('calibre_plugins.rss_reader'):
            continue
        d = getattr(mod, '__dict__', None)
        if not isinstance(d, dict):
            continue
        gr = d.get('get_resources')
        if gr is None:
            continue
        try:
            data = gr(path)
            if data:
                return bytes(data)
        except Exception:
            continue

    return b''


def _patch_translations_cache(locale: str) -> bool:
    """Replace the GNUTranslations in calibre's global cache for this plugin."""
    mo_data = _load_mo_bytes(locale)
    if not mo_data:
        return False

    zfp = _find_plugin_zfp()
    if not zfp:
        return False

    try:
        trans = gettext.GNUTranslations(io.BytesIO(mo_data))
    except Exception:
        return False

    try:
        import calibre.customize.zipplugin as _zp  # type: ignore

        _zp._translations_cache[zfp] = trans
    except Exception:
        return False

    return True


def _reset_to_source_language() -> bool:
    """Install a NullTranslations so _() returns msgid strings (i.e. English source).

    This is the correct way to 'select English' — do NOT load en.mo because
    en.po has empty msgstr fields which would blank-out the UI.
    """
    zfp = _find_plugin_zfp()
    if not zfp:
        return False
    try:
        import calibre.customize.zipplugin as _zp  # type: ignore

        _zp._translations_cache[zfp] = gettext.NullTranslations()
        return True
    except Exception:
        return False


def apply_language_overrides(module_globals: dict) -> None:
    """Patch calibre's translation cache and set _() / ngettext() in the
    caller's namespace if a script override is active.

    Supports: Serbian, Russian, Japanese, Chinese (all variants).
    """
    target = _resolve_target_locale()
    if not target:
        return

    _patch_translations_cache(target)

    try:
        import calibre.customize.zipplugin as _zp  # type: ignore

        zfp = _find_plugin_zfp()
        if zfp:
            trans = _zp._translations_cache.get(zfp)
            if trans is not None:
                module_globals['_'] = trans.gettext
                module_globals['ngettext'] = trans.ngettext
    except Exception:
        pass


def refresh_all_plugin_modules() -> None:
    """Re-inject whatever translation is currently in the cache into every
    already-imported plugin module.  Must be called *after* the cache has
    been updated (e.g. via _patch_translations_cache or _patch_to_auto)."""
    try:
        import calibre.customize.zipplugin as _zp  # type: ignore

        zfp = _find_plugin_zfp()
        if not zfp:
            return
        trans = _zp._translations_cache.get(zfp)
        if trans is None:
            return
    except Exception:
        return

    for name, mod in list(sys.modules.items()):
        try:
            if not name or not name.startswith('calibre_plugins.rss_reader'):
                continue
            d = getattr(mod, '__dict__', None)
            if isinstance(d, dict):
                d['_'] = trans.gettext
                d['ngettext'] = trans.ngettext
        except Exception:
            continue
