from __future__ import absolute_import

from calibre_plugins.rss_reader.config import plugin_prefs

try:
    from qt.core import QComboBox, QIcon, QSize, Qt
except Exception:
    from PyQt5.Qt import QComboBox, QIcon, Qt
    try:
        from PyQt5.QtCore import QSize
    except Exception:
        QSize = None

try:
    from qt.core import QFont
except Exception:
    try:
        from PyQt5.Qt import QFont
    except Exception:
        QFont = None


_CLEAR_SENTINEL = '__clear_history__'


ROLE_FONT = getattr(getattr(Qt, 'ItemDataRole', None), 'FontRole', getattr(Qt, 'FontRole', 6))


class HistoryDropdown(QComboBox):
    def __init__(
        self,
        parent=None,
        history_key='',
        placeholder='',
        max_items=30,
        min_entry_length=1,
        clear_label=None,
    ):
        QComboBox.__init__(self, parent)
        self._history_key = str(history_key or '').strip()
        self._max_items = int(max_items or 30)
        self._min_entry_length = int(min_entry_length or 1)
        self._clear_label = clear_label or _('Clear history')

        try:
            self.setEditable(True)
        except Exception:
            pass

        try:
            self.setInsertPolicy(QComboBox.InsertPolicy.NoInsert)
        except Exception:
            try:
                self.setInsertPolicy(QComboBox.NoInsert)
            except Exception:
                pass

        try:
            le = self.lineEdit()
            if le is not None:
                try:
                    le.setPlaceholderText(str(placeholder or ''))
                except Exception:
                    pass
        except Exception:
            pass

        # Prefer programmatic control of icon size so both the list item and
        # the combo's current-item icon (left side) are small.
        try:
            if QSize is not None:
                try:
                    self.setIconSize(QSize(10, 10))
                except Exception:
                    pass
        except Exception:
            pass

        # Keep a QSS fallback in case style doesn't respect setIconSize.
        try:
            self.setStyleSheet("QComboBox { icon-size: 10px; }")
        except Exception:
            pass

        self._rebuild_items(keep_text=True)

        try:
            self.activated[int].connect(self._on_activated)
        except Exception:
            try:
                self.activated.connect(self._on_activated)
            except Exception:
                pass

        # Save entries on Enter or when focus leaves
        try:
            le = self.lineEdit()
            if le is not None:
                try:
                    le.returnPressed.connect(self.save_current_to_history)
                except Exception:
                    pass
                try:
                    le.editingFinished.connect(self.save_current_to_history)
                except Exception:
                    pass
        except Exception:
            pass

    # --- QLineEdit-ish convenience wrappers (minimize ui.py churn) ---

    def text(self):
        try:
            return str(self.currentText() or '')
        except Exception:
            return ''

    def setText(self, text):
        try:
            self.setEditText(str(text or ''))
        except Exception:
            pass

    def setPlaceholderText(self, text):
        try:
            le = self.lineEdit()
            if le is not None:
                le.setPlaceholderText(str(text or ''))
        except Exception:
            pass

    def setClearButtonEnabled(self, enabled):
        try:
            le = self.lineEdit()
            if le is not None and hasattr(le, 'setClearButtonEnabled'):
                le.setClearButtonEnabled(bool(enabled))
        except Exception:
            pass

    # --- History management ---

    def _load_history(self):
        if not self._history_key:
            return []
        try:
            vals = plugin_prefs.get(self._history_key, [])
            if not isinstance(vals, list):
                return []
            out = []
            for v in vals:
                s = str(v or '').strip()
                if s:
                    out.append(s)
            return out
        except Exception:
            return []

    def _save_history(self, items):
        if not self._history_key:
            return
        try:
            plugin_prefs[self._history_key] = list(items or [])
        except Exception:
            pass

    def clear_history(self):
        self._save_history([])
        self._rebuild_items(keep_text=True)

    def save_current_to_history(self):
        try:
            cur = str(self.currentText() or '').strip()
        except Exception:
            cur = ''

        if len(cur) < self._min_entry_length:
            return

        hist = self._load_history()
        try:
            hist.remove(cur)
        except Exception:
            pass
        hist.insert(0, cur)

        if self._max_items is not None and self._max_items > 0:
            del hist[self._max_items :]

        self._save_history(hist)
        self._rebuild_items(keep_text=True)

    def _rebuild_items(self, keep_text=False):
        try:
            cur = self.text() if keep_text else ''
        except Exception:
            cur = ''

        hist = self._load_history()

        try:
            self.blockSignals(True)
        except Exception:
            pass

        try:
            self.clear()
        except Exception:
            pass

        try:
            for h in hist:
                self.addItem(h, h)

            # Add in-dropdown clear entry
            if hist:
                try:
                    self.insertSeparator(self.count())
                except Exception:
                    pass

            try:
                self.addItem(self._clear_label, _CLEAR_SENTINEL)
                # Italicize the clear history entry (no icon).
                try:
                    idx = int(self.count() - 1)
                    if idx >= 0 and QFont is not None:
                        f = QFont()
                        f.setItalic(True)
                        self.setItemData(idx, f, ROLE_FONT)
                except Exception:
                    pass
            except Exception:
                # Worst case: no clear entry
                pass
        finally:
            try:
                self.blockSignals(False)
            except Exception:
                pass

        if keep_text:
            try:
                self.setEditText(cur)
            except Exception:
                pass

    def _on_activated(self, index):
        try:
            data = self.itemData(int(index))
        except Exception:
            data = None

        if data == _CLEAR_SENTINEL:
            self.clear_history()
            try:
                self.setEditText('')
            except Exception:
                pass
            return

        # Selecting a past entry should act like typing it (and put it at top)
        try:
            txt = str(self.itemText(int(index)) or '').strip()
        except Exception:
            txt = ''
        if txt:
            self.setEditText(txt)
            self.save_current_to_history()


try:
    load_translations()
except NameError:
    pass
