# Folder/tree manipulation helpers for RSS Reader.
# This module exists to keep `ui.py` smaller; functions expect `self` to be
# an instance of RSSReaderDialog (or compatible).

from __future__ import absolute_import

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import QInputDialog, QMessageBox
except Exception:  # pragma: no cover
    from PyQt5.Qt import QInputDialog, QMessageBox

from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader import rss_db

# Define ROLE_USER for compatibility
try:
    from qt.core import Qt
    ROLE_USER = getattr(getattr(Qt, 'ItemDataRole', None), 'UserRole', getattr(Qt, 'UserRole', 32))
except Exception:
    try:
        from PyQt5.QtCore import Qt
        ROLE_USER = getattr(Qt, 'UserRole', 32)
    except Exception:
        ROLE_USER = 32


def edit_selected_feed_or_folder(self):
    # Toolbar "Edit": if a folder is selected, rename the folder.
    # If a single feed is selected, edit that feed.
    try:
        indexes = self.feeds_tree.selectionModel().selectedIndexes()
        sel_items = [self.feeds_tree.model().itemFromIndex(idx) for idx in indexes]
    except Exception:
        sel_items = []

    folder_paths = []
    feed_ids = []
    for it in sel_items:
        try:
            d = it.data(ROLE_USER)
        except Exception:
            d = None
        if isinstance(d, dict) and d.get('type') == 'folder':
            p = str(d.get('path') or '').strip().strip('/')
            if p:
                folder_paths.append(p)
        elif isinstance(d, dict) and d.get('type') == 'feed':
            fid = str(d.get('id') or '').strip()
            if fid:
                feed_ids.append(fid)

    # If folder(s) are selected, prioritize folder rename.
    if folder_paths:
        if len(folder_paths) == 1:
            return rename_folder(self, folder_paths[0])
        QMessageBox.information(self, _('Edit'), _('Please select a single folder to rename.'))
        return

    # Otherwise, only edit if exactly one feed node is selected.
    if len(feed_ids) == 1:
        return self.edit_selected_feed()
    if len(feed_ids) > 1:
        QMessageBox.information(self, _('Edit'), _('Please select a single feed to edit.'))
        return


def add_folder(self, parent_path=''):
    parent_path = str(parent_path or '').strip().strip('/')
    name, ok = QInputDialog.getText(self, _('Add folder'), _('Folder name (use / for subfolders):'))
    if not ok:
        return ''
    name = str(name or '').strip().strip('/')
    # Allow nested paths, sanitize only illegal chars except '/'
    for ch in '<>:"\\|?*#':
        name = name.replace(ch, '_')
    if not name:
        return ''
    # If name is a nested path, join with parent_path if given
    full = f'{parent_path}/{name}' if parent_path else name
    full = '/'.join([seg for seg in full.split('/') if seg])
    folders = list(rss_db.get_folders() or [])
    if full not in folders:
        folders.append(full)
        try:
            rss_db.save_folders(folders)
        except Exception:
            pass
    self.load_feeds()
    return full


def rename_folder(self, folder_path):
    folder_path = str(folder_path or '').strip().strip('/')
    if not folder_path:
        return
    new_name, ok = QInputDialog.getText(self, _('Rename folder'), _('New folder name:'), text=folder_path.rpartition('/')[2])
    if not ok:
        return
    new_name = str(new_name or '').strip().strip('/')
    new_name = self._sanitize_folder_name(new_name)
    if not new_name:
        return
    parent = folder_path.rpartition('/')[0]
    new_path = f'{parent}/{new_name}' if parent else new_name
    if new_path == folder_path:
        return

    feeds = list(rss_db.get_feeds() or [])
    for f in feeds:
        fp = str(f.get('folder') or '')
        if fp == folder_path or fp.startswith(folder_path + '/'):
            f['folder'] = new_path + fp[len(folder_path):]
    try:
        rss_db.save_feeds(feeds)
    except Exception:
        pass

    folders = [str(x).strip().strip('/') for x in (rss_db.get_folders() or []) if str(x).strip()]
    new_folders = []
    for fp in folders:
        if fp == folder_path or fp.startswith(folder_path + '/'):
            new_folders.append(new_path + fp[len(folder_path):])
        else:
            new_folders.append(fp)
    try:
        rss_db.save_folders(sorted(set(new_folders)))
    except Exception:
        pass
    self.refresh()


def delete_folder(self, folder_path):
    folder_path = str(folder_path or '').strip().strip('/')
    if not folder_path:
        return
    yes = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'Yes', getattr(QMessageBox, 'Yes', None))
    no = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'No', getattr(QMessageBox, 'No', None))
    ans = QMessageBox.question(self, _('Delete folder'), _('Delete folder "%s"? Feeds will be moved to the parent folder.') % folder_path, yes | no, no)
    if ans != yes:
        return
    parent = folder_path.rpartition('/')[0] or str(plugin_prefs.get('default_folder', '') or '')

    # Always reload feeds from DB to avoid stale state
    feeds = list(rss_db.get_feeds() or [])
    feeds_to_move = []
    for f in feeds:
        fp = str(f.get('folder') or '')
        if fp == folder_path or fp.startswith(folder_path + '/'):
            suffix = fp[len(folder_path):].lstrip('/')
            f['folder'] = f'{parent}/{suffix}' if suffix else parent
            feeds_to_move.append(f.get('id'))
    if feeds_to_move:
        try:
            rss_db.save_feeds(feeds)
        except Exception:
            pass

    # Always reload folders from DB to avoid stale state
    folders = [str(x).strip().strip('/') for x in (rss_db.get_folders() or []) if str(x).strip()]
    folders = [fp for fp in folders if not (fp == folder_path or fp.startswith(folder_path + '/'))]
    if parent and parent not in folders:
        folders.append(parent)
    try:
        rss_db.save_folders(sorted(set(folders)))
    except Exception:
        pass

    # If the user is deleting the currently-selected folder, prefer selecting its parent after refresh.
    try:
        from calibre_plugins.rss_reader.ui import gprefs
        if str(self.selected_folder_path() or '').strip().strip('/') == folder_path:
            gprefs['rss_reader_last_selected_folder'] = str(parent or '')
            gprefs['rss_reader_last_selected_feed_ids'] = []
    except Exception:
        pass
    self.refresh()


def move_selected_feeds_to_folder(self, folder_path):
    try:
        from calibre_plugins.rss_reader.debug import _debug
    except Exception:
        _debug = None

    folder_path = str(folder_path or '').strip().strip('/')
    # Support nested path syntax (e.g., news/tech). Allow empty => top level.
    folder_path = '/'.join([seg for seg in str(folder_path or '').split('/') if seg])

    if _debug:
        _debug('move_selected_feeds_to_folder called with:', folder_path)
    fids = list(self.selected_feed_ids() or [])
    if _debug:
        _debug('Selected feed IDs:', fids)
    if not fids:
        if _debug:
            _debug('No feeds selected, aborting move.')
        return

    feeds = list(rss_db.get_feeds() or [])
    fid_set = set(fids)
    moved = []
    for f in feeds:
        if str(f.get('id') or '') in fid_set:
            if _debug:
                _debug('Moving feed', f.get('id'), 'from folder', f.get('folder'), 'to', folder_path)
            f['folder'] = folder_path
            moved.append(f.get('id'))

    try:
        rss_db.save_feeds(feeds)
        if _debug:
            _debug('Feeds saved after move:', moved)
    except Exception as e:
        if _debug:
            _debug('Exception during rss_db.save_feeds:', e)

    folders = list(rss_db.get_folders() or [])
    if folder_path and folder_path not in folders:
        folders.append(folder_path)
        try:
            rss_db.save_folders(folders)
            if _debug:
                _debug('New folder added to DB:', folder_path)
        except Exception as e:
            if _debug:
                _debug('Exception during rss_db.save_folders:', e)

    self.refresh()
    if _debug:
        _debug('move_selected_feeds_to_folder complete.')


def move_folder(self, folder_path):
    folder_path = str(folder_path or '').strip().strip('/')
    if not folder_path:
        return

    folders = sorted(set(str(x).strip().strip('/') for x in (rss_db.get_folders() or []) if str(x).strip()))
    # The user chooses the *parent* folder. Explicitly show the top-level option.
    top_level_label = _('(Top level)')
    available_folders = [f for f in folders if not (f == folder_path or f.startswith(folder_path + '/'))]
    display_folders = [top_level_label] + available_folders
    target, ok = QInputDialog.getItem(self, _('Move folder'), _('Move to:'), display_folders, 0, False)
    if not ok:
        return
    target = str(target or '').strip()
    if target == top_level_label:
        target = ''
    target = target.strip().strip('/')

    basename = folder_path.rpartition('/')[2]
    new_base = f'{target}/{basename}' if target else basename
    if new_base == folder_path:
        return

    feeds = list(rss_db.get_feeds() or [])
    for f in feeds:
        fp = str(f.get('folder') or '')
        if fp == folder_path or fp.startswith(folder_path + '/'):
            suffix = fp[len(folder_path):].lstrip('/')
            f['folder'] = f'{new_base}/{suffix}' if suffix else new_base
    try:
        rss_db.save_feeds(feeds)
    except Exception:
        pass

    new_folders = []
    for fp in folders:
        if fp == folder_path or fp.startswith(folder_path + '/'):
            suffix = fp[len(folder_path):].lstrip('/')
            new_folders.append(f'{new_base}/{suffix}' if suffix else new_base)
        else:
            new_folders.append(fp)
    try:
        rss_db.save_folders(sorted(set(new_folders)))
    except Exception:
        pass

    self.refresh()


def move_folders(self, folder_paths, target=None):
    folder_paths = [str(x or '').strip().strip('/') for x in (folder_paths or []) if str(x or '').strip()]
    if not folder_paths:
        return

    # Normalize + drop nested selections (moving parent moves children)
    try:
        folder_paths_sorted = sorted(set(folder_paths), key=lambda x: (len(x), x))
    except Exception:
        folder_paths_sorted = list(dict.fromkeys(folder_paths))
    to_move = []
    for p in folder_paths_sorted:
        is_child = False
        for parent in to_move:
            if p == parent or p.startswith(parent + '/'):
                is_child = True
                break
        if not is_child:
            to_move.append(p)
    if not to_move:
        return

    folders = sorted(set(str(x).strip().strip('/') for x in (rss_db.get_folders() or []) if str(x).strip()))
    top_level_label = _('(Top level)')

    # Exclude moving a folder into itself/its own subtree
    def _is_descendant(path, ancestor):
        try:
            return path == ancestor or path.startswith(ancestor + '/')
        except Exception:
            return False

    available_folders = []
    for f in folders:
        skip = False
        for src in to_move:
            if _is_descendant(f, src):
                skip = True
                break
        if not skip:
            available_folders.append(f)

    if target is None:
        display_folders = [top_level_label] + available_folders
        target, ok = QInputDialog.getItem(self, _('Move folders'), _('Move to:'), display_folders, 0, False)
        if not ok:
            return
        target = str(target or '').strip()
        if target == top_level_label:
            target = ''
    target = str(target or '').strip().strip('/')

    # Preflight: detect basename collisions
    collisions = []
    planned = {}
    for src in to_move:
        base = src.rpartition('/')[2]
        new_base = f'{target}/{base}' if target else base
        planned[src] = new_base
    dest_bases = list(planned.values())
    if len(set(dest_bases)) != len(dest_bases):
        collisions.append(_('Two selected folders would map to the same destination.'))

    # If any destination already exists (and isn't just a moved folder), abort
    existing = set(folders)
    moving_all_old = set()
    for src in to_move:
        for f in folders:
            if _is_descendant(f, src):
                moving_all_old.add(f)
    for _src, dst in planned.items():
        if dst in existing and dst not in moving_all_old:
            collisions.append(_('Destination already exists: %s') % dst)

    if collisions:
        try:
            QMessageBox.warning(self, _('Move folders'), '\n'.join(collisions))
        except Exception:
            pass
        return

    feeds = list(rss_db.get_feeds() or [])
    # Update feeds in one pass
    for f in feeds:
        fp = str(f.get('folder') or '').strip().strip('/')
        if not fp:
            continue
        for src in to_move:
            if fp == src or fp.startswith(src + '/'):
                suffix = fp[len(src):].lstrip('/')
                nb = planned.get(src) or src
                f['folder'] = f'{nb}/{suffix}' if suffix else nb
                break
    try:
        rss_db.save_feeds(feeds)
    except Exception:
        pass

    # Update folders
    new_folders = []
    for fp in folders:
        replaced = False
        for src in to_move:
            if fp == src or fp.startswith(src + '/'):
                suffix = fp[len(src):].lstrip('/')
                nb = planned.get(src) or src
                new_folders.append(f'{nb}/{suffix}' if suffix else nb)
                replaced = True
                break
        if not replaced:
            new_folders.append(fp)
    try:
        rss_db.save_folders(sorted(set(new_folders)))
    except Exception:
        pass

    self.refresh()
