"""Export-related helpers for RSS Reader.

These functions mirror methods on RSSReaderDialog but live outside `ui.py`
to keep the main UI module smaller. They expect a `self` argument that is an
instance of RSSReaderDialog (or compatible).
"""

from __future__ import absolute_import

import os
import shutil
import traceback

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import Qt, QMessageBox, QProgressDialog, QThread
except Exception:  # pragma: no cover
    from PyQt5.Qt import Qt, QMessageBox, QProgressDialog, QThread

from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader import rss_db


def email_selected_feeds(self, to_email, subject='', message=''):
    import tempfile
    import datetime

    try:
        from calibre.gui2 import error_dialog
    except Exception:
        error_dialog = None

    try:
        from calibre.utils.smtp import config as smtp_config
    except Exception:
        smtp_config = None

    try:
        article_fetch_mod = __import__('calibre_plugins.rss_reader.article_fetch', fromlist=['ExportWorker'])
    except Exception:
        article_fetch_mod = None

    if not str(to_email or '').strip():
        try:
            QMessageBox.information(self, _('Email'), _('No email address selected.'))
        except Exception:
            pass
        return

    try:
        if smtp_config is not None:
            opts = smtp_config().parse()
            if not getattr(opts, 'from_', None):
                if error_dialog is not None:
                    error_dialog(
                        self,
                        _('Email not configured'),
                        _('Please configure your email settings in calibre (Preferences > Email).'),
                        show=True,
                    )
                return
    except Exception:
        # If we cannot validate, allow send_mails to surface errors.
        pass

    try:
        feed_ids = list(self.selected_feed_ids() or [])
        if not feed_ids:
            QMessageBox.information(self, _('Email'), _('No feeds selected.'))
            return

        cache = dict(rss_db.get_feed_cache_map() or {})
        feeds = list(rss_db.get_feeds() or [])
        feeds_by_id = {str(f.get('id') or ''): f for f in feeds}

        def _safe_filename(s):
            s = (s or '').strip()
            if not s:
                return 'news'
            for ch in '<>:"/\\|?*':
                s = s.replace(ch, '_')
            s = ' '.join(s.split())
            return s[:140].strip() or 'news'

        now = datetime.datetime.now()
        date_in_title = now.strftime('%a, %d %b %Y')
        ts_suffix = now.strftime('%Y-%m-%d_%H-%M-%S')

        if len(feed_ids) == 1:
            fid0 = feed_ids[0]
            entry0 = cache.get(fid0, {})
            base_title = entry0.get('title') or (feeds_by_id.get(fid0) or {}).get('title') or _('RSS Reader')
        else:
            base_title = _('RSS Reader (%d feeds)') % len(feed_ids)

        news_title = _('News - %s [%s]') % (str(base_title), str(date_in_title))

        try:
            from calibre_plugins.rss_reader.convert_utils import (
                get_effective_output_format,
                get_available_output_formats,
                resolve_conversion_output,
            )
            desired_fmt = get_effective_output_format(plugin_prefs)
            available_fmts = get_available_output_formats()
            convert_ext, final_ext, markdown_via_txt = resolve_conversion_output(
                desired_fmt, available_output_formats=available_fmts
            )
        except Exception:
            convert_ext, final_ext, markdown_via_txt = 'epub', 'epub', False

        td = tempfile.mkdtemp(prefix='rss_email_export_')
        base_name = f"{_safe_filename(str(news_title))} - {ts_suffix}"
        out_path = os.path.join(td, base_name + '.' + convert_ext)
        final_out_path = os.path.join(td, base_name + '.' + final_ext)
        attachment_name = _safe_filename(str(news_title)) + '.' + final_ext

        if not str(subject or '').strip():
            subject = str(news_title)
        if not str(message or '').strip():
            message = _('See attached ebook.')

        progress = QProgressDialog(_('Preparing export...'), _('Cancel'), 0, 0, self)
        progress.setWindowTitle(_('RSS Reader Export'))
        progress.setWindowModality(Qt.WindowModality.NonModal)
        progress.setAutoClose(False)
        progress.setAutoReset(False)
        progress.show()

        export_data = {
            'feed_ids': feed_ids,
            'cache': cache,
            'feeds_by_id': feeds_by_id,
            'news_title': news_title,
            'td': td,
            'ts_suffix': ts_suffix,
            'out_path': out_path,
            'final_out_path': final_out_path,
        }

        if article_fetch_mod is None or not hasattr(article_fetch_mod, 'ExportWorker'):
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
            QMessageBox.information(self, _('Export'), _('Export worker unavailable.'))
            return

        worker = article_fetch_mod.ExportWorker(export_data)
        thread = QThread(self)
        worker.moveToThread(thread)

        try:
            self._active_export_threads.append((thread, worker))
        except Exception:
            pass

        def _release_export_thread_refs(_=None, _thread=thread, _worker=worker):
            try:
                self._active_export_threads = [
                    (t, w) for (t, w) in (self._active_export_threads or [])
                    if t is not _thread and w is not _worker
                ]
            except Exception:
                pass

        worker.start.connect(worker.run)
        worker.progress.connect(progress.setLabelText)
        try:
            progress.canceled.connect(worker.cancel)
        except Exception:
            pass

        worker.finished.connect(
            lambda result: on_email_export_worker_finished(
                self,
                result,
                progress,
                thread,
                td,
                out_path,
                final_out_path,
                convert_ext,
                final_ext,
                markdown_via_txt,
                to_email,
                subject,
                message,
                attachment_name,
            )
        )
        worker.finished.connect(thread.quit)
        worker.finished.connect(worker.deleteLater)
        thread.finished.connect(_release_export_thread_refs)
        thread.finished.connect(thread.deleteLater)

        thread.start()
        worker.start.emit()

    except Exception as e:
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        if error_dialog is not None:
            error_dialog(self, _('Email export error'), _('Failed to export: %s') % str(e), show=True, det_msg=traceback.format_exc())


def on_email_export_worker_finished(
    self,
    result,
    progress,
    thread,
    td,
    out_path,
    final_out_path,
    convert_ext,
    final_ext,
    markdown_via_txt,
    to_email,
    subject,
    message,
    attachment_name,
):
    try:
        from calibre.gui2 import error_dialog
    except Exception:
        error_dialog = None

    try:
        progress.close()
        progress.deleteLater()
    except Exception:
        pass

    try:
        if thread is not None and thread.isRunning():
            thread.quit()
    except Exception:
        pass

    try:
        if isinstance(result, dict) and result.get('error'):
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
            if error_dialog is not None:
                error_dialog(self, _('Export error'), str(result.get('error') or _('Export failed.')), show=True)
            return
    except Exception:
        pass

    html_path = None
    try:
        if isinstance(result, dict):
            html_path = result.get('html_path')
            markdown_via_txt = bool(result.get('markdown_via_txt', markdown_via_txt))
        elif isinstance(result, (str, bytes, os.PathLike)):
            html_path = result
    except Exception:
        html_path = None

    if not html_path:
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        if error_dialog is not None:
            error_dialog(self, _('Export error'), _('Failed to prepare export (no HTML was generated).'), show=True)
        return

    try:
        ui_mod = __import__('calibre_plugins.rss_reader.ui', fromlist=['Dispatcher'])
    except Exception:
        ui_mod = None

    try:
        if not getattr(self.gui, 'job_manager', None):
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
            if error_dialog is not None:
                error_dialog(self, _('Export error'), _('Cannot start conversion job (job manager unavailable).'), show=True)
            return

        func = 'gui_convert'
        try:
            from calibre.customize.conversion import OptionRecommendation
            recs = [
                ('level1_toc', '//*[@id][starts-with(@id, "item-")]', OptionRecommendation.HIGH),
            ]
        except Exception:
            recs = []
        try:
            if markdown_via_txt and isinstance(recs, list):
                try:
                    from calibre.customize.conversion import OptionRecommendation
                    recs.append(('txt_output_formatting', 'markdown', OptionRecommendation.HIGH))
                except Exception:
                    recs.append(('txt_output_formatting', 'markdown', None))
        except Exception:
            pass

        args = [html_path, out_path, recs]
        desc = _('Convert RSS items for email')
        if ui_mod is None or not hasattr(ui_mod, 'Dispatcher'):
            raise RuntimeError('Dispatcher unavailable')

        try:
            jobs = getattr(self, '_email_export_jobs', None)
            if jobs is None:
                self._email_export_jobs = {}
        except Exception:
            self._email_export_jobs = {}

        cb = ui_mod.Dispatcher(lambda job: email_export_job_done(self, job))
        job = self.gui.job_manager.run_job(cb, func, args=args, description=desc)
        self._email_export_jobs[job] = (td, out_path, final_out_path, to_email, subject, message, attachment_name)
        try:
            self.gui.jobs_pointer.start()
        except Exception:
            pass
        try:
            self.gui.status_bar.show_message(_('Email export queued'), 5000)
        except Exception:
            pass
    except Exception as e:
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        if error_dialog is not None:
            error_dialog(self, _('Email export error'), _('Failed to start conversion job: %s') % str(e), show=True, det_msg=traceback.format_exc())


def email_export_job_done(self, job):
    try:
        from calibre.gui2 import error_dialog
    except Exception:
        error_dialog = None

    try:
        from calibre.gui2 import Dispatcher
    except Exception:
        Dispatcher = None

    job_data = None
    try:
        job_data = getattr(self, '_email_export_jobs', {}).pop(job, None)
    except Exception:
        job_data = None

    if not job_data or len(job_data) != 7:
        return

    td, out_path, final_out_path, to_email, subject, message, attachment_name = job_data

    try:
        if getattr(job, 'failed', False):
            try:
                self.gui.job_exception(job)
            except Exception:
                if error_dialog is not None:
                    error_dialog(self, _('Email export error'), _('Export job failed.'), show=True)
            return

        attach_path = out_path
        try:
            if final_out_path and out_path and os.path.abspath(final_out_path) != os.path.abspath(out_path):
                if os.path.exists(out_path) and not os.path.exists(final_out_path):
                    try:
                        os.replace(out_path, final_out_path)
                    except Exception:
                        try:
                            shutil.copy2(out_path, final_out_path)
                            os.remove(out_path)
                        except Exception:
                            pass
                if os.path.exists(final_out_path):
                    attach_path = final_out_path
        except Exception:
            attach_path = out_path

        if not attach_path or not os.path.exists(attach_path):
            if error_dialog is not None:
                error_dialog(self, _('Email export error'), _('Converted file not found.'), show=True)
            return

        try:
            from calibre.gui2 import email as email_mod
        except Exception as e:
            if error_dialog is not None:
                error_dialog(self, _('Email error'), _('Failed to load calibre email module: %s') % str(e), show=True)
            return

        def _email_done(_job):
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass

        try:
            email_mod.send_mails(
                jobnames=[subject or _('RSS Reader export')],
                callback=Dispatcher(_email_done) if Dispatcher is not None else (lambda j: _email_done(j)),
                attachments=[attach_path],
                to_s=[to_email],
                subjects=[subject],
                texts=[message],
                attachment_names=[attachment_name],
                job_manager=self.gui.job_manager,
            )
            try:
                self.gui.status_bar.show_message(_('Email queued'), 3000)
            except Exception:
                pass
        except Exception as e:
            if error_dialog is not None:
                error_dialog(self, _('Email error'), _('Failed to queue email: %s') % str(e), show=True, det_msg=traceback.format_exc())
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
            return
    finally:
        # If sending could not be queued, ensure td is not leaked.
        # If it *was* queued, td will be cleaned up in _email_done.
        try:
            if td and os.path.isdir(td):
                # Heuristic: if attach_path still exists, keep td for the send job.
                # Otherwise, clean up now.
                if not (out_path and os.path.exists(out_path)) and not (final_out_path and os.path.exists(final_out_path)):
                    shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass


def export_selected_feeds(self):
    import tempfile
    import datetime

    try:
        from calibre.gui2 import error_dialog
    except Exception:
        error_dialog = None

    try:
        article_fetch_mod = __import__('calibre_plugins.rss_reader.article_fetch', fromlist=['ExportWorker'])
    except Exception:
        article_fetch_mod = None


    try:
        feed_ids = list(self.selected_feed_ids() or [])
        if not feed_ids:
            QMessageBox.information(self, _('Export'), _('No feeds selected.'))
            return

        cache = dict(rss_db.get_feed_cache_map() or {})
        feeds = list(rss_db.get_feeds() or [])
        feeds_by_id = {str(f.get('id') or ''): f for f in feeds}

        def _safe_filename(s):
            s = (s or '').strip()
            if not s:
                return 'news'
            for ch in '<>:"/\\|?*':
                s = s.replace(ch, '_')
            s = ' '.join(s.split())
            return s[:140].strip() or 'news'

        now = datetime.datetime.now()
        date_in_title = now.strftime('%a, %d %b %Y')
        ts_suffix = now.strftime('%Y-%m-%d_%H-%M-%S')

        if len(feed_ids) == 1:
            fid0 = feed_ids[0]
            entry0 = cache.get(fid0, {})
            base_title = entry0.get('title') or (feeds_by_id.get(fid0) or {}).get('title') or _('RSS Reader')
        else:
            base_title = _('RSS Reader (%d feeds)') % len(feed_ids)

        news_title = _('News - %s [%s]') % (str(base_title), str(date_in_title))

        td = tempfile.mkdtemp(prefix='rss_export_')

        try:
            from calibre_plugins.rss_reader.convert_utils import (
                get_effective_output_format,
                get_available_output_formats,
                resolve_conversion_output,
            )
            desired_fmt = get_effective_output_format(plugin_prefs)
            available_fmts = get_available_output_formats()
            convert_ext, final_ext, markdown_via_txt = resolve_conversion_output(
                desired_fmt, available_output_formats=available_fmts
            )
        except Exception:
            convert_ext, final_ext, markdown_via_txt = 'epub', 'epub', False

        add_to_library = bool(plugin_prefs.get('export_add_to_library', False))

        final_out_path = None
        if add_to_library:
            out_path = os.path.join(
                tempfile.gettempdir(),
                f"{_safe_filename(str(news_title))} - {ts_suffix}.{convert_ext}",
            )
        else:
            try:
                from calibre_plugins.rss_reader.ui import choose_save_file
            except Exception:
                QMessageBox.information(self, _('Export'), _('File picker unavailable.'))
                return

            default_name = f"{_safe_filename(str(news_title))} - {ts_suffix}.{final_ext}"
            chosen = choose_save_file(
                self,
                'rss-reader-export-ebook',
                _('Convert'),
                filters=[(_('%s files') % (final_ext.upper() if final_ext != 'md' else 'MD'), [final_ext])],
                all_files=True,
                initial_filename=default_name,
            )
            if not chosen:
                return

            base, _ext = os.path.splitext(str(chosen))
            if not base:
                return
            final_out_path = base + '.' + final_ext
            out_path = base + '.' + convert_ext

        progress = QProgressDialog(_('Preparing export...'), _('Cancel'), 0, 0, self)
        progress.setWindowTitle(_('RSS Reader Export'))
        progress.setWindowModality(Qt.WindowModality.NonModal)
        progress.setAutoClose(False)
        progress.setAutoReset(False)
        progress.show()

        export_data = {
            'feed_ids': feed_ids,
            'cache': cache,
            'feeds_by_id': feeds_by_id,
            'news_title': news_title,
            'td': td,
            'ts_suffix': ts_suffix,
            'out_path': out_path,
            'final_out_path': final_out_path,
        }

        if article_fetch_mod is None or not hasattr(article_fetch_mod, 'ExportWorker'):
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
            QMessageBox.information(self, _('Export'), _('Export worker unavailable.'))
            return

        worker = article_fetch_mod.ExportWorker(export_data)
        thread = QThread(self)
        worker.moveToThread(thread)

        try:
            self._active_export_threads.append((thread, worker))
        except Exception:
            pass

        def _release_export_thread_refs(_=None, _thread=thread, _worker=worker):
            try:
                self._active_export_threads = [
                    (t, w) for (t, w) in (self._active_export_threads or [])
                    if t is not _thread and w is not _worker
                ]
            except Exception:
                pass

        worker.start.connect(worker.run)
        worker.progress.connect(progress.setLabelText)
        try:
            progress.canceled.connect(worker.cancel)
        except Exception:
            pass
        worker.finished.connect(
            lambda result: self._on_export_worker_finished(
                result, progress, thread, td, out_path, add_to_library, news_title,
                final_out_path, convert_ext, final_ext, markdown_via_txt
            )
        )
        worker.finished.connect(thread.quit)
        worker.finished.connect(worker.deleteLater)
        thread.finished.connect(_release_export_thread_refs)
        thread.finished.connect(thread.deleteLater)

        thread.start()
        worker.start.emit()
        try:
            if ui_mod is not None and hasattr(ui_mod, '_debug'):
                ui_mod._debug('Export worker thread started')
        except Exception:
            pass

    except Exception as e:
        if error_dialog is not None:
            error_dialog(self, _('Export error'), _('Failed to export: %s') % str(e), show=True, det_msg=traceback.format_exc())


def export_listed_items(self):
    """Export all currently listed (visible) item rows to an ebook.

    This is intended to work with the search/filter box (e.g. `tag:img`).
    """
    try:
        ui_mod = __import__('calibre_plugins.rss_reader.ui', fromlist=['ROLE_USER'])
    except Exception:
        ui_mod = None

    role = getattr(ui_mod, 'ROLE_USER', None)
    if role is None:
        try:
            from qt.core import Qt
            role = getattr(Qt, 'UserRole', 32)
        except Exception:
            role = 32

    items = []
    try:
        for r in range(self.items_table.rowCount()):
            try:
                if self.items_table.isRowHidden(r):
                    continue
            except Exception:
                # If we can't determine hidden state, include the row.
                pass
            it0 = self.items_table.item(r, 0)
            data = it0.data(role) if it0 is not None else None
            if isinstance(data, dict):
                items.append(data)
    except Exception:
        items = []

    if not items:
        try:
            QMessageBox.information(self, _('Export'), _('No listed items to export.'))
        except Exception:
            pass
        return

    # Reuse the existing selected-items export implementation by temporarily selecting.
    # To avoid messing with user selection, we call a small internal copy of the logic.
    # (Keep behavior consistent with export_selected_items.)
    try:
        from calibre_plugins.rss_reader.export_logic import export_selected_items as _export_sel
    except Exception:
        _export_sel = None

    if _export_sel is None:
        return

    # Reuse export_selected_items by temporarily selecting all visible rows.
    # Restore previous selection best-effort (without try/finally syntax to
    # avoid edge-case SyntaxError reports from older packaged copies).
    try:
        orig = list(self.items_table.selectionModel().selectedRows() or [])
    except Exception:
        orig = None

    try:
        self.items_table.clearSelection()
    except Exception:
        pass

    try:
        for r in range(self.items_table.rowCount()):
            try:
                if self.items_table.isRowHidden(r):
                    continue
            except Exception:
                pass
            try:
                self.items_table.selectRow(r)
            except Exception:
                pass
    except Exception:
        pass

    result = None
    try:
        result = _export_sel(self)
    except Exception:
        # Let the underlying export path show error dialogs as usual.
        raise
    finally:
        try:
            if orig is not None:
                self.items_table.clearSelection()
                for mi in orig:
                    try:
                        self.items_table.selectRow(int(mi.row()))
                    except Exception:
                        pass
        except Exception:
            pass

    return result


def on_export_worker_finished(self, result, progress, thread, td, out_path, add_to_library, news_title,
                             final_out_path, convert_ext, final_ext, markdown_via_txt):
    try:
        from calibre.gui2 import error_dialog
    except Exception:
        error_dialog = None

    progress.close()
    try:
        progress.deleteLater()
    except Exception:
        pass

    try:
        if thread is not None and thread.isRunning():
            thread.quit()
    except Exception:
        pass

    try:
        if isinstance(result, dict) and result.get('error'):
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
            if error_dialog is not None:
                error_dialog(self, _('Export error'), str(result.get('error') or _('Export failed.')), show=True)
            return
    except Exception:
        pass

    html_path = None
    comments_html = ''
    try:
        if isinstance(result, dict):
            html_path = result.get('html_path')
            comments_html = result.get('comments_html') or ''
            markdown_via_txt = bool(result.get('markdown_via_txt', markdown_via_txt))
        elif isinstance(result, (str, bytes, os.PathLike)):
            html_path = result
    except Exception:
        html_path = None

    if not html_path:
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        if error_dialog is not None:
            error_dialog(self, _('Export error'), _('Failed to prepare export (no HTML was generated).'), show=True)
        return

    try:
        ui_mod = __import__('calibre_plugins.rss_reader.ui', fromlist=['Dispatcher'])
    except Exception:
        ui_mod = None

    try:
        func = 'gui_convert'
        try:
            from calibre.customize.conversion import OptionRecommendation
            recs = [
                ('level1_toc', '//*[@id][starts-with(@id, "item-")]', OptionRecommendation.HIGH),
            ]
        except Exception:
            recs = []
        try:
            if markdown_via_txt and isinstance(recs, list):
                try:
                    from calibre.customize.conversion import OptionRecommendation
                    recs.append(('txt_output_formatting', 'markdown', OptionRecommendation.HIGH))
                except Exception:
                    recs.append(('txt_output_formatting', 'markdown', None))
        except Exception:
            pass
        args = [html_path, out_path, recs]
        desc = _('Convert RSS items')
        if ui_mod is None or not hasattr(ui_mod, 'Dispatcher'):
            raise RuntimeError('Dispatcher unavailable')
        job = self.gui.job_manager.run_job(ui_mod.Dispatcher(self._export_job_done), func, args=args, description=desc)
        self._export_jobs[job] = (td, out_path, add_to_library, news_title, comments_html, final_out_path)
        try:
            self.gui.jobs_pointer.start()
        except Exception:
            pass
        try:
            if add_to_library:
                self.gui.status_bar.show_message(_('Converting and adding to library: %s') % news_title, 5000)
            else:
                self.gui.status_bar.show_message(_('Export queued: %s') % out_path, 5000)
        except Exception:
            pass
    except Exception as e:
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        if error_dialog is not None:
            error_dialog(self, _('Export error'), _('Failed to start conversion/export job: %s') % str(e), show=True, det_msg=traceback.format_exc())


def export_selected_items(self):
    """Export selected item(s) from the items table to an ebook."""
    import tempfile
    import datetime

    try:
        from calibre.gui2 import error_dialog
    except Exception:
        error_dialog = None

    try:
        ui_mod = __import__('calibre_plugins.rss_reader.ui', fromlist=['ROLE_USER'])
    except Exception:
        ui_mod = None

    # Image export helpers live in preview_browser after the code split.
    try:
        from calibre_plugins.rss_reader.preview_browser import _process_images_for_export as _process_images_for_export
        from calibre_plugins.rss_reader.preview_browser import _sanitize_url_for_fetch as _sanitize_url_for_fetch
        from calibre_plugins.rss_reader.preview_browser import _normalize_images_for_preview as _normalize_images_for_preview
    except Exception:
        _process_images_for_export = None
        _sanitize_url_for_fetch = None
        def _normalize_images_for_preview(html, base_url='', preserve_local=False):
            return html

    try:
        rows = list(self.items_table.selectionModel().selectedRows() or [])
        if not rows:
            QMessageBox.information(self, _('Export'), _('No items selected.'))
            return

        items = []
        for idx in rows:
            it0 = self.items_table.item(idx.row(), 0)
            data = it0.data(ui_mod.ROLE_USER) if (it0 is not None and ui_mod is not None) else None
            if isinstance(data, dict):
                items.append(data)

        if not items:
            QMessageBox.information(self, _('Export'), _('No items to export.'))
            return

        feeds = list(rss_db.get_feeds() or [])
        feeds_by_id = {str(f.get('id') or ''): f for f in feeds}

        def _safe_filename(s):
            s = (s or '').strip()
            if not s:
                return 'article'
            for ch in '<>:"/\\|?*':
                s = s.replace(ch, '_')
            s = ' '.join(s.split())
            return s[:140].strip() or 'article'

        now = datetime.datetime.now()
        date_in_title = now.strftime('%a, %d %b %Y')
        ts_suffix = now.strftime('%Y-%m-%d_%H-%M-%S')

        if len(items) == 1:
            base_title = items[0].get('title') or _('Article')
        else:
            base_title = _('%d articles') % len(items)

        news_title = '%s [%s]' % (str(base_title), str(date_in_title))

        if ui_mod is None:
            QMessageBox.information(self, _('Export'), _('Export helpers unavailable.'))
            return

        td = tempfile.mkdtemp(prefix='rss_export_')

        try:
            from calibre_plugins.rss_reader.export_oeb import build_oeb_periodical, append_enclosure_images
        except Exception:
            build_oeb_periodical = None
            append_enclosure_images = None

        if build_oeb_periodical is None:
            QMessageBox.information(self, _('Export'), _('Export helpers unavailable.'))
            return

        comments_items = []

        export_feeds = [
            {
                'title': news_title,
                'items': [],
            }
        ]

        item_idx = 0
        for it in items:
            item_idx += 1
            title = it.get('title') or _('(untitled)')
            link = it.get('link') or ''
            try:
                fid_for_base = str(it.get('_feed_id') or '')
            except Exception:
                fid_for_base = ''
            try:
                feed_url = str((feeds_by_id.get(fid_for_base) or {}).get('url') or '')
            except Exception:
                feed_url = ''
            try:
                base_url_for_images = str(link or it.get('_feed_url') or feed_url or '')
            except Exception:
                base_url_for_images = str(link or feed_url or '')
            published = it.get('published') or ''
            try:
                import html as _html
                _raw_summary = _html.unescape(it.get('summary') or '')
            except Exception:
                _raw_summary = it.get('summary') or ''
            summary = ui_mod.normalize_summary_to_html(_raw_summary)

            # Export should try to embed images regardless of the per-feed preview
            # setting. Network fetching is controlled by `export_download_uncached_images`.
            do_download = True

            if do_download:
                try:
                    from urllib.parse import urljoin as _urljoin, urlparse as _urlparse
                    enc_urls = []
                    for enc in (it.get('enclosures') or []):
                        try:
                            if not isinstance(enc, dict):
                                continue
                            eurl = (enc.get('url') or '').strip()
                            etype = (enc.get('type') or '').strip().lower()
                            if not eurl:
                                continue
                            is_img = False
                            if etype and etype.startswith('image/'):
                                is_img = True
                            else:
                                p = _urlparse(eurl).path or ''
                                ext = os.path.splitext(p)[1].lower()
                                if ext in ('.jpg', '.jpeg', '.png', '.gif', '.webp', '.avif', '.bmp', '.svg'):
                                    is_img = True
                            if not is_img:
                                continue
                            try:
                                if base_url_for_images:
                                    eurl = _urljoin(base_url_for_images, eurl)
                            except Exception:
                                pass
                            try:
                                if _sanitize_url_for_fetch is not None:
                                    eurl = _sanitize_url_for_fetch(eurl)
                                elif ui_mod is not None and hasattr(ui_mod, '_sanitize_url_for_fetch'):
                                    eurl = ui_mod._sanitize_url_for_fetch(eurl)
                            except Exception:
                                pass
                            enc_urls.append(eurl)
                        except Exception:
                            continue
                    seen = set()
                    enc_urls = [u for u in enc_urls if u and not (u in seen or seen.add(u))]
                    if enc_urls and append_enclosure_images is not None:
                        summary = append_enclosure_images(
                            summary,
                            enc_urls[:8],
                            base_url=base_url_for_images,
                            sanitize_url=_sanitize_url_for_fetch,
                        )
                except Exception:
                    pass

            try:
                images_subdir = f"feed_0/article_{item_idx - 1}/images"
                if _process_images_for_export is not None:
                    summary = _process_images_for_export(
                        summary, base_url=base_url_for_images, td=td, do_download=do_download, images_subdir=images_subdir
                    )
            except Exception:
                pass

            try:
                summary = _normalize_images_for_preview(summary, base_url=base_url_for_images or '', preserve_local=True)
            except Exception:
                pass

            comments_items.append((title, link))
            export_feeds[0]['items'].append(
                {
                    'title': title,
                    'link': link,
                    'published': published,
                    'body_html': summary,
                }
            )

        try:
            from calibre_plugins.rss_reader.convert_utils import (
                get_effective_output_format,
                get_available_output_formats,
                resolve_conversion_output,
            )
            desired_fmt = get_effective_output_format(plugin_prefs)
            available_fmts = get_available_output_formats()
            convert_ext, final_ext, markdown_via_txt = resolve_conversion_output(
                desired_fmt, available_output_formats=available_fmts
            )
        except Exception:
            convert_ext, final_ext, markdown_via_txt = 'epub', 'epub', False

        add_to_library = bool(plugin_prefs.get('export_add_to_library', False))
        final_out_path = None

        if add_to_library:
            out_path = os.path.join(
                tempfile.gettempdir(),
                f"{_safe_filename(str(news_title))} - {ts_suffix}.{convert_ext}",
            )
        else:
            default_name = f"{_safe_filename(str(news_title))} - {ts_suffix}.{final_ext}"
            chosen = ui_mod.choose_save_file(
                self,
                'rss-reader-export-ebook',
                _('Convert'),
                filters=[(_('%s files') % (final_ext.upper() if final_ext != 'md' else 'MD'), [final_ext])],
                all_files=True,
                initial_filename=default_name,
            )
            if not chosen:
                return

            base, _ext = os.path.splitext(str(chosen))
            if not base:
                return
            final_out_path = base + '.' + final_ext
            out_path = base + '.' + convert_ext

        try:
            import html as _htmlmod
            comments_html = '<ul>' + ''.join([
                ('<li>%s</li>' % (_htmlmod.escape(t) if t else _('(untitled)')))
                if not l else '<li><a href="%s">%s</a></li>' % (_htmlmod.escape(l), _htmlmod.escape(t or _('(untitled)')))
                for (t, l) in (comments_items or [])
            ]) + '</ul>'
        except Exception:
            comments_html = ''

        html_path = build_oeb_periodical(td, news_title, export_feeds)

        if not getattr(self.gui, 'job_manager', None):
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
            if error_dialog is not None:
                error_dialog(self, _('Export error'), _('Cannot start conversion job (job manager unavailable).'), show=True)
            return

        func = 'gui_convert'
        try:
            from calibre.customize.conversion import OptionRecommendation
            recs = [
                ('level1_toc', '//*[@id][starts-with(@id, "item-")]', OptionRecommendation.HIGH),
            ]
        except Exception:
            recs = []
        try:
            if markdown_via_txt and isinstance(recs, list):
                try:
                    from calibre.customize.conversion import OptionRecommendation
                    recs.append(('txt_output_formatting', 'markdown', OptionRecommendation.HIGH))
                except Exception:
                    recs.append(('txt_output_formatting', 'markdown', None))
        except Exception:
            pass
        args = [html_path, out_path, recs]
        desc = _('Convert RSS items')
        job = self.gui.job_manager.run_job(ui_mod.Dispatcher(self._export_job_done), func, args=args, description=desc)
        self._export_jobs[job] = (td, out_path, add_to_library, news_title, comments_html, final_out_path)
        try:
            self.gui.jobs_pointer.start()
        except Exception:
            pass
        try:
            if add_to_library:
                self.gui.status_bar.show_message(_('Converting and adding to library: %s') % news_title, 5000)
            else:
                self.gui.status_bar.show_message(_('Export queued: %s') % out_path, 5000)
        except Exception:
            pass

    except Exception as e:
        if error_dialog is not None:
            error_dialog(self, _('Export error'), _('Failed to export items: %s') % str(e), show=True, det_msg=traceback.format_exc())


def export_job_done(self, job):
    """Handler for calibre conversion job completion."""
    from calibre.gui2 import error_dialog

    job_data = self._export_jobs.pop(job, (None, None, False, ''))
    comments_html = ''
    final_out_path = None
    if len(job_data) == 2:
        td, out_path = job_data[0], job_data[1]
        add_to_library, book_title = False, ''
    elif len(job_data) == 4:
        td, out_path, add_to_library, book_title = job_data
    elif len(job_data) == 5:
        td, out_path, add_to_library, book_title, comments_html = job_data
    else:
        td, out_path, add_to_library, book_title, comments_html, final_out_path = job_data

    try:
        if getattr(job, 'failed', False):
            try:
                self.gui.job_exception(job)
            except Exception:
                error_dialog(self, _('Export error'), _('Export job failed.'), show=True)
            return

        if add_to_library:
            try:
                from calibre.ebooks.metadata.meta import get_metadata
                from calibre.ebooks.metadata import MetaInformation

                try:
                    with open(out_path, 'rb') as f:
                        mi = get_metadata(f, os.path.splitext(out_path)[1][1:].lower())
                except Exception:
                    mi = MetaInformation(book_title or _('RSS Feed Export'), [_('RSS Reader')])

                if book_title:
                    mi.title = book_title
                mi.authors = [_('RSS Reader Plugin')]
                mi.tags = [_('RSS Reader Plugin'), _('News'), _('Feed Export')]
                if comments_html:
                    try:
                        mi.comments = comments_html
                    except Exception:
                        pass

                try:
                    book_id = self.gui.library_view.model().db.add_books(
                        [out_path],
                        [os.path.splitext(out_path)[1][1:].lower()],
                        [mi],
                        add_duplicates=False
                    )[0]

                    try:
                        self.gui.library_view.model().books_added(1)
                        self.gui.library_view.model().refresh_ids([book_id])
                    except Exception:
                        pass

                    try:
                        self.gui.status_bar.show_message(_('RSS export added to library: %s') % (book_title or _('RSS Feed')), 5000)
                    except Exception:
                        pass

                    QMessageBox.information(
                        self,
                        _('Export complete'),
                        _('RSS feed export added to library as:\n%s') % (book_title or _('RSS Feed')),
                    )
                except Exception as e:
                    error_dialog(
                        self,
                        _('Add to library failed'),
                        _('Conversion succeeded but failed to add to library: %s') % str(e),
                        show=True,
                        det_msg=traceback.format_exc(),
                    )
            except Exception as e:
                error_dialog(
                    self,
                    _('Export error'),
                    _('Failed to add ebook to library: %s') % str(e),
                    show=True,
                    det_msg=traceback.format_exc(),
                )
        else:
            try:
                if final_out_path and out_path and os.path.abspath(final_out_path) != os.path.abspath(out_path):
                    if os.path.exists(out_path) and not os.path.exists(final_out_path):
                        try:
                            os.replace(out_path, final_out_path)
                        except Exception:
                            try:
                                shutil.copy2(out_path, final_out_path)
                                os.remove(out_path)
                            except Exception:
                                pass
                        out_path = final_out_path
            except Exception:
                pass

            try:
                self.gui.status_bar.show_message(_('Export complete: %s') % out_path, 5000)
            except Exception:
                pass
            QMessageBox.information(self, _('Export complete'), _('Exported to %s') % out_path)
    finally:
        if td:
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass
        if add_to_library and out_path and os.path.exists(out_path):
            try:
                os.remove(out_path)
            except Exception:
                pass
