from __future__ import absolute_import, division, print_function, unicode_literals

import os


def _portable_cache_dir_from_config():
    try:
        from calibre.constants import config_dir as _config_dir  # type: ignore
    except Exception:
        return ''
    try:
        cfg = _config_dir() if callable(_config_dir) else str(_config_dir or '')
    except Exception:
        cfg = ''
    if not cfg:
        return ''
    cfg_norm = cfg.rstrip('\\/')
    if not cfg_norm:
        return ''
    tail = os.path.basename(cfg_norm)
    if tail.lower() == 'calibre settings':
        base = os.path.dirname(cfg_norm)
        if base:
            return os.path.join(base, 'caches')
    return ''


def process_images_for_export(
    html_str,
    base_url,
    td,
    do_download=True,
    images_subdir=None,
    sanitize_url_for_fetch=None,
    plugin_prefs=None,
    calibre_cache_dir=None,
):
    import logging

    if sanitize_url_for_fetch is None:
        def sanitize_url_for_fetch(u):
            return u

    if plugin_prefs is None:
        plugin_prefs = {}

    debug_images = False
    try:
        debug_images = bool(plugin_prefs.get('debug_export_images', False))
    except Exception:
        debug_images = False

    if calibre_cache_dir is None:
        cache_base = ''
        try:
            from calibre.constants import cache_dir as _cache_dir  # type: ignore
            cache_base = _cache_dir() if callable(_cache_dir) else ''
        except Exception:
            cache_base = ''
        portable_cache = _portable_cache_dir_from_config()
        calibre_cache_dir = (portable_cache or cache_base) or ''
    # Avoid always-on logging here; it is hot-path code during preview/export.

    """Download/relativize inline <img> tags for export.

    - Cache-first: if the preview cache (sha1(url).img) contains the bytes, use it.
    - If cache miss and export_download_uncached_images is enabled, download the image and
      also seed the preview cache.
    - Rewrites <img src> to relative local paths under images_subdir.
    """

    try:
        from lxml import html as _lhtml
        from urllib.parse import urljoin as _urljoin, urlparse as _urlparse
        from calibre_plugins.rss_reader.rss import fetch_url, _mechanize_fetch
        from calibre.utils.imghdr import what as _imghdr_what
        import hashlib as _hashlib, re as _re
        _have_lxml = True
    except Exception as _imp_e:
        # Diagnostic: capture import failure but try a safe fallback rather than
        # bailing out entirely. Preview often runs in a different environment
        # (Qt) so exports executed under calibre may have missing imports.
        try:
            import logging as _logging
            _logging.warning('export_images: lxml/import failure: %r', _imp_e)
        except Exception:
            pass
        _have_lxml = False
        try:
            # still try to import helpers we can use in a regex-based fallback
            from urllib.parse import urljoin as _urljoin, urlparse as _urlparse
            from calibre_plugins.rss_reader.rss import fetch_url, _mechanize_fetch
            from calibre.utils.imghdr import what as _imghdr_what
            import hashlib as _hashlib, re as _re
        except Exception:
            # If even these helpers aren't available, give up gracefully.
            return html_str or ''

    # Optional debug: log the HTML before processing (can be huge)
    try:
        if debug_images:
            logging.warning('PRE-IMG html_str: %r', html_str)
    except Exception:
        pass

    if images_subdir:
        images_dir = os.path.join(td, images_subdir)
        rel_img_prefix = images_subdir.replace('\\', '/').replace('\\', '/')
    else:
        images_dir = os.path.join(td, 'images')
        rel_img_prefix = 'images'

    try:
        os.makedirs(images_dir, exist_ok=True)
    except Exception:
        pass

    # Track seen images within this article by canonical URL.
    # Value: (score, img_element)
    seen_images = {}

    if _have_lxml:
        try:
            frags = _lhtml.fragments_fromstring(html_str or '')
        except Exception:
            try:
                root = _lhtml.fragment_fromstring(html_str or '', create_parent='div')
                frags = [root]
            except Exception:
                return html_str or ''
    else:
        # Simple fallback: if lxml not available, we'll process <img> tags
        # via regex. This won't handle full HTML parsing, but will attempt
        # to download/embed images so exports don't end up with remote URLs
        # that calibre later replaces with [image].
        try:
            # Build a very small compatibility wrapper that mimics the
            # behaviour of process_images_for_export for whole-document inputs.
            raw = html_str or ''
        except Exception:
            return html_str or ''

    def _first_url_from_srcset(srcset_text):
        try:
            s = (srcset_text or '').strip()
            if not s:
                return ''
            first = s.split(',')[0].strip()
            if not first:
                return ''
            return first.split()[0].strip()
        except Exception:
            return ''
    def _looks_like_html_not_svg(raw_bytes):
        try:
            if not raw_bytes:
                return False
            b = bytes(raw_bytes)
            head = b.lstrip()[:512].lower()
            if b'<svg' in head:
                return False
            if head.startswith(b'<!doctype') or head.startswith(b'<html'):
                return True
            if head.startswith(b'<?xml'):
                return True
        except Exception:
            return False
        return False

    def _safe_ext_from_type(raw, url):
        try:
            ext = None
            try:
                ext = _imghdr_what(None, raw)
            except Exception:
                ext = None
            if not ext:
                try:
                    path = (_urlparse(url).path or '').lower()
                    m = _re.search(r'\.(png|jpe?g|gif|webp|bmp|svg|avif)(?:$|\?)', path)
                    if m:
                        ext = m.group(1)
                except Exception:
                    ext = None
            if ext == 'jpeg':
                ext = 'jpg'
            return ext or 'img'
        except Exception:
            return 'img'

    def _candidate_variants(url_text):
        variants = []
        try:
            if url_text and url_text not in variants:
                variants.append(url_text)
        except Exception:
            pass
        try:
            s_url = sanitize_url_for_fetch(url_text)
            if s_url and s_url not in variants:
                variants.append(s_url)
        except Exception:
            s_url = url_text
        try:
            cand = _re.sub(r'(-\d+x\d+)(?=\.[a-z0-9]+(?:[?#]|$))', '', s_url or '', count=1, flags=_re.IGNORECASE)
            if cand and cand not in variants:
                variants.append(cand)
        except Exception:
            pass
        try:
            qless = (s_url or '').split('?', 1)[0].split('#', 1)[0]
            if qless and qless not in variants:
                variants.append(qless)
        except Exception:
            pass
        try:
            if (s_url or '').startswith('https://'):
                alt = 'http://' + s_url[len('https://'):]
                if alt not in variants:
                    variants.append(alt)
            elif (s_url or '').startswith('http://'):
                alt = 'https://' + s_url[len('http://'):]
                if alt not in variants:
                    variants.append(alt)
        except Exception:
            pass
        return variants or [url_text]

    def _download_one(img_url):
        try:
            from calibre_plugins.rss_reader.debug import _debug as _dbg
        except Exception:
            def _dbg(msg):
                try:
                    logging.warning('%s', msg)
                except Exception:
                    pass

        if debug_images:
            _dbg('export img url: %r' % (img_url,))
            logging.warning('Attempting export image (cache-first): %r', img_url)

        # data: URLs
        try:
            u0 = str(img_url or '').strip()
        except Exception:
            u0 = ''
        if u0.lower().startswith('data:'):
            try:
                import base64
                header, data = u0.split(',', 1)
                is_b64 = ';base64' in header.lower()
                raw = base64.b64decode(data) if is_b64 else data.encode('utf-8', 'ignore')
                final = u0
            except Exception:
                return None
        else:
            raw = None
            final = None

        img_url_s = sanitize_url_for_fetch(img_url)

        cache_dir = os.path.join(calibre_cache_dir, 'plugins', 'rss_reader', 'img_cache') if calibre_cache_dir else ''

        candidates = _candidate_variants(img_url_s)
        if debug_images:
            try:
                _dbg('  resolved cache_dir=%r' % (cache_dir,))
                _dbg('  candidate variants=%r' % (candidates,))
            except Exception:
                pass

        # cache lookup
        if raw is None and cache_dir:
            for c in candidates:
                try:
                    h = _hashlib.sha1(str(c).encode('utf-8', 'ignore')).hexdigest()
                    p = os.path.join(cache_dir, h + '.img')
                    if debug_images:
                        _dbg('  try cache: %s' % p)
                    if os.path.exists(p):
                        with open(p, 'rb') as f:
                            raw = f.read()
                        final = c
                        if debug_images:
                            _dbg('    HIT cache for %r -> %s' % (c, p))
                        break
                except Exception:
                    if debug_images:
                        try:
                            _dbg('    cache lookup exception for %r' % (c,))
                        except Exception:
                            pass
                    continue

        # cache miss -> optional net fetch
        if raw is None:
            allow_net = True
            try:
                allow_net = bool(plugin_prefs.get('export_download_uncached_images', True))
            except Exception:
                allow_net = True
            if not allow_net:
                if debug_images:
                    _dbg('    MISS: network disabled')
                return None

            headers = {'Accept': 'image/webp,image/jpeg,image/png,image/gif;q=0.9,*/*;q=0.1'}
            try:
                if base_url:
                    headers['Referer'] = str(base_url)
            except Exception:
                pass
            try:
                from calibre_plugins.rss_reader.rss import _stable_user_agent
                try:
                    ua = _stable_user_agent()
                except Exception:
                    ua = str(_stable_user_agent or '')
                ua = (ua or '').strip()
                if ua:
                    headers['User-Agent'] = ua
            except Exception:
                pass

            for c in candidates:
                try:
                    u_fetch = sanitize_url_for_fetch(c)
                    if debug_images:
                        _dbg('    net fetch candidate: %r (sanitized: %r)' % (c, u_fetch))
                    raw2, _hdr2 = fetch_url(u_fetch, timeout_seconds=15, headers=headers)
                    if not raw2:
                        if debug_images:
                            _dbg('    fetch returned empty for %r' % (u_fetch,))
                        continue
                    if debug_images:
                        try:
                            _dbg('    fetch bytes=%d for %r' % (len(raw2) if raw2 else 0, u_fetch))
                        except Exception:
                            pass
                    if _looks_like_html_not_svg(raw2):
                        if debug_images:
                            try:
                                _dbg('    net fetch returned HTML (not image): %r' % (u_fetch,))
                            except Exception:
                                pass
                        # Try mechanize/browser fallback for this candidate
                        try:
                            if debug_images:
                                try:
                                    _dbg('    attempting mechanize fallback: %r' % (u_fetch,))
                                except Exception:
                                    pass
                            mraw, mfinal = _mechanize_fetch(u_fetch, timeout_seconds=18, headers=headers, user_agent=headers.get('User-Agent'))
                            if mraw and not _looks_like_html_not_svg(mraw):
                                raw2 = mraw
                                if debug_images:
                                    try:
                                        _dbg('    mechanize fallback succeeded: %r' % (u_fetch,))
                                    except Exception:
                                        pass
                            else:
                                if debug_images:
                                    try:
                                        _dbg('    mechanize fallback returned non-image for: %r' % (u_fetch,))
                                    except Exception:
                                        pass
                        except Exception as mech_e:
                            if debug_images:
                                try:
                                    _dbg('    mechanize exception: %r' % (mech_e,))
                                except Exception:
                                    pass
                        if not raw2:
                            continue
                    raw = raw2
                    final = c
                    # seed cache
                    if cache_dir:
                        try:
                            os.makedirs(cache_dir, exist_ok=True)
                        except Exception:
                            pass
                        try:
                            h = _hashlib.sha1(str(c).encode('utf-8', 'ignore')).hexdigest()
                            p = os.path.join(cache_dir, h + '.img')
                            with open(p, 'wb') as f:
                                f.write(raw)
                            if debug_images:
                                try:
                                    _dbg('    seeded cache %s for %r' % (p, c))
                                except Exception:
                                    pass
                        except Exception as e:
                            if debug_images:
                                try:
                                    _dbg('    failed to seed cache for %r: %r' % (c, e))
                                except Exception:
                                    pass
                    break
                except Exception as e:
                    if debug_images:
                        try:
                            _dbg('    net fetch error: %r (%s)' % (c, str(e)))
                        except Exception:
                            pass
                    try:
                        if debug_images:
                            try:
                                _dbg('    attempting mechanize fallback after error: %r' % (c,))
                            except Exception:
                                pass
                        mraw, mfinal = _mechanize_fetch(sanitize_url_for_fetch(c), timeout_seconds=18, headers=headers, user_agent=headers.get('User-Agent'))
                        if mraw and not _looks_like_html_not_svg(mraw):
                            raw = mraw
                            final = c
                            if cache_dir:
                                try:
                                    os.makedirs(cache_dir, exist_ok=True)
                                except Exception:
                                    pass
                                try:
                                    h = _hashlib.sha1(str(c).encode('utf-8', 'ignore')).hexdigest()
                                    p = os.path.join(cache_dir, h + '.img')
                                    with open(p, 'wb') as f:
                                        f.write(raw)
                                except Exception:
                                    pass
                            if debug_images:
                                try:
                                    _dbg('    mechanize fallback succeeded after error: %r' % (c,))
                                except Exception:
                                    pass
                            break
                    except Exception as e2:
                        if debug_images:
                            try:
                                _dbg('    mechanize fallback after error raised: %r' % (e2,))
                            except Exception:
                                pass
                    continue

        if raw is None:
            if debug_images:
                try:
                    _dbg('    MISS: failed (candidates=%r)' % (candidates,))
                except Exception:
                    _dbg('    MISS: failed')
            return None

        def _qt_convert_to_png(raw_bytes):
            try:
                # Prefer calibre's Qt wrapper when available
                from qt.core import QImage, QBuffer, QByteArray, QIODevice
            except Exception:
                try:
                    from PyQt5.Qt import QImage, QBuffer, QByteArray, QIODevice
                except Exception:
                    return None
            try:
                img = QImage()
                if not img.loadFromData(bytes(raw_bytes)):
                    return None
                ba = QByteArray()
                buf = QBuffer(ba)
                try:
                    buf.open(QIODevice.OpenModeFlag.WriteOnly)
                except Exception:
                    buf.open(QIODevice.WriteOnly)
                ok = img.save(buf, 'PNG')
                try:
                    buf.close()
                except Exception:
                    pass
                if not ok:
                    return None
                try:
                    return bytes(ba)
                except Exception:
                    try:
                        return ba.data()
                    except Exception:
                        return None
            except Exception:
                return None

        # Detect AVIF/WebP and convert to PNG if possible
        is_avif = False
        is_webp = False
        try:
            if raw and b'ftypavif' in raw[:64]:
                is_avif = True
        except Exception:
            pass
        try:
            if raw and len(raw) >= 12 and raw[:4] == b'RIFF' and raw[8:12] == b'WEBP':
                is_webp = True
        except Exception:
            pass

        png_bytes = None
        if is_avif or is_webp:
            try:
                from io import BytesIO
                from PIL import Image as PILImage
                bio = BytesIO(raw)
                im = PILImage.open(bio)
                im = im.convert('RGBA')
                out = BytesIO()
                im.save(out, format='PNG')
                png_bytes = out.getvalue()
            except Exception:
                png_bytes = None

            # Pillow may not have AVIF/WebP support in some calibre builds.
            # Fall back to Qt image plugins when available.
            if png_bytes is None:
                try:
                    png_bytes = _qt_convert_to_png(raw)
                except Exception:
                    png_bytes = None

        out_raw = png_bytes if png_bytes is not None else raw

        # Optional: downscale/recompress very large images to avoid massive EPUBs.
        # This is especially common for large PNG screenshots.
        ext_override = None
        try:
            optimize_images = bool(plugin_prefs.get('export_optimize_images', True))
        except Exception:
            optimize_images = True
        try:
            optimize_min_bytes = int(plugin_prefs.get('export_optimize_images_min_bytes', 2 * 1024 * 1024) or 2 * 1024 * 1024)
        except Exception:
            optimize_min_bytes = 2 * 1024 * 1024
        try:
            optimize_force_jpeg_bytes = int(plugin_prefs.get('export_optimize_force_jpeg_bytes', 8 * 1024 * 1024) or 8 * 1024 * 1024)
        except Exception:
            optimize_force_jpeg_bytes = 8 * 1024 * 1024
        try:
            max_dim = int(plugin_prefs.get('export_image_max_dim', 2000) or 2000)
        except Exception:
            max_dim = 2000
        try:
            jpeg_quality = int(plugin_prefs.get('export_image_jpeg_quality', 85) or 85)
        except Exception:
            jpeg_quality = 85

        try:
            if optimize_images and out_raw and optimize_min_bytes and len(out_raw) >= optimize_min_bytes:
                from io import BytesIO
                from PIL import Image as PILImage
                try:
                    from PIL import ImageOps as PILImageOps
                except Exception:
                    PILImageOps = None

                im = PILImage.open(BytesIO(out_raw))
                try:
                    if PILImageOps is not None:
                        im = PILImageOps.exif_transpose(im)
                except Exception:
                    pass

                try:
                    w, h = im.size
                except Exception:
                    w, h = (0, 0)

                if max_dim and w and h and max(w, h) > max_dim:
                    try:
                        resample = getattr(getattr(PILImage, 'Resampling', PILImage), 'LANCZOS', getattr(PILImage, 'LANCZOS', 1))
                        im.thumbnail((max_dim, max_dim), resample=resample)
                    except Exception:
                        try:
                            im.thumbnail((max_dim, max_dim))
                        except Exception:
                            pass

                # Prefer JPEG for huge images (better size), even if the source is PNG.
                force_jpeg = bool(len(out_raw) >= optimize_force_jpeg_bytes)
                has_alpha = False
                try:
                    has_alpha = ('A' in (im.getbands() or ()))
                except Exception:
                    has_alpha = False

                if force_jpeg or not has_alpha:
                    try:
                        if im.mode not in ('RGB', 'L'):
                            im = im.convert('RGBA' if has_alpha else 'RGB')
                    except Exception:
                        pass
                    try:
                        if has_alpha and im.mode == 'RGBA':
                            bg = PILImage.new('RGB', im.size, (255, 255, 255))
                            bg.paste(im, mask=im.split()[-1])
                            im_out = bg
                        else:
                            im_out = im.convert('RGB') if im.mode != 'RGB' else im

                        buf = BytesIO()
                        im_out.save(buf, format='JPEG', quality=jpeg_quality, optimize=True, progressive=True)
                        cand = buf.getvalue()
                        if cand and len(cand) < len(out_raw):
                            out_raw = cand
                            ext_override = 'jpg'
                    except Exception:
                        pass
        except Exception:
            pass

        # Enforce max bytes after any optimization/conversion.
        try:
            max_bytes = int(plugin_prefs.get('export_image_max_bytes', 12 * 1024 * 1024) or 12 * 1024 * 1024)
        except Exception:
            max_bytes = 12 * 1024 * 1024
        try:
            if max_bytes and len(out_raw) > max_bytes:
                if debug_images:
                    _dbg('    SKIP: image too large (%d bytes)' % len(out_raw))
                return None
        except Exception:
            pass

        ext = (ext_override or ('png' if png_bytes is not None else None) or _safe_ext_from_type(out_raw, final or img_url_s))
        try:
            digest = _hashlib.sha1(out_raw).hexdigest()
        except Exception:
            digest = str(abs(hash(out_raw)))

        fname = digest + '.' + ext
        fpath = os.path.join(images_dir, fname)
        try:
            if not os.path.exists(fpath):
                os.makedirs(images_dir, exist_ok=True)
                with open(fpath, 'wb') as f:
                    f.write(out_raw)
        except Exception:
            return None

        return rel_img_prefix.rstrip('/') + '/' + fname

    # Regex fallback processor when lxml isn't present: find <img ... src="...">
    if not _have_lxml:
        try:
            # Find img tags and their src attribute (simple, permissive regex)
            pattern = _re.compile(r'(<img\b[^>]*?\bsrc=["\'])([^"\']+)(["\'][^>]*>)', _re.IGNORECASE)
            def _repl(m):
                before, srcval, after = m.group(1), m.group(2), m.group(3)
                try:
                    abs_url = _urljoin(base_url or '', srcval)
                except Exception:
                    abs_url = srcval
                try:
                    rel = _download_one(abs_url)
                except Exception:
                    rel = None
                if rel:
                    return before + rel + after
                else:
                    # mark missing so caller can inspect
                    return before + srcval + after.replace('>', ' data-export-missing="1">')
            out = pattern.sub(_repl, raw)
            return out
        except Exception:
            return html_str or ''

    for node in frags:
        try:
            imgs = node.xpath('.//img')
        except Exception:
            imgs = []

        # lxml's .//img selects descendants only and does not include the
        # fragment root itself. Many feeds (e.g. xkcd) have summaries that are
        # a single <img> tag, so include the node when it is an <img>.
        try:
            tag = getattr(node, 'tag', None)
            if isinstance(tag, str) and tag.lower() == 'img':
                if node not in imgs:
                    imgs = [node] + (imgs or [])
        except Exception:
            pass

        for img in imgs:
            # Many sites lazy-load images with non-standard attributes.
            # Prefer an explicit src, otherwise try common fallbacks.
            try:
                src0 = img.get('src')
            except Exception:
                src0 = None

            if not src0:
                fallback_src = None
                for attr in (
                    'data-src',
                    'data-original',
                    'data-lazy-src',
                    'data-actualsrc',
                    'data-url',
                ):
                    try:
                        v = img.get(attr)
                    except Exception:
                        v = None
                    if v:
                        fallback_src = v
                        break
                if fallback_src:
                    try:
                        img.set('src', fallback_src)
                    except Exception:
                        pass

            # Some use srcset-like attributes without src.
            try:
                src_now = img.get('src')
            except Exception:
                src_now = None
            if not src_now:
                srcset_like = None
                for attr in ('srcset', 'data-srcset', 'data-lazy-srcset'):
                    try:
                        v = img.get(attr)
                    except Exception:
                        v = None
                    if v:
                        srcset_like = v
                        break
                if srcset_like:
                    try:
                        cand = _first_url_from_srcset(srcset_like)
                    except Exception:
                        cand = ''
                    if cand:
                        try:
                            img.set('src', cand)
                        except Exception:
                            pass

            try:
                src = img.get('src')
            except Exception:
                src = None

            try:
                srcset = img.get('srcset')
            except Exception:
                srcset = None

            if not src and srcset:
                src = _first_url_from_srcset(srcset)

            if not src:
                continue

            try:
                abs_url = _urljoin(base_url or '', src)
            except Exception:
                abs_url = src

            # Deduplicate *before* downloading to avoid wasted work and to
            # ensure we prefer original images over WordPress size variants.
            try:
                # Canonicalize: strip query/fragment and WordPress -WxH suffix.
                parsed = _urlparse(abs_url)
                abs_no_q = parsed._replace(query='', fragment='').geturl()
            except Exception:
                abs_no_q = abs_url
            try:
                canonical_url = _re.sub(
                    r'(-\d+x\d+)(?=\.[a-z0-9]+(?:[?#]|$))',
                    '',
                    abs_no_q,
                    count=1,
                    flags=_re.IGNORECASE,
                )
            except Exception:
                canonical_url = abs_no_q

            # Score: prefer non-suffixed URL; then prefer larger declared area.
            try:
                has_size_suffix = bool(_re.search(r'-\d+x\d+(?=\.[a-z0-9]+(?:[?#]|$))', abs_no_q, flags=_re.IGNORECASE))
            except Exception:
                has_size_suffix = False
            try:
                w = int(img.get('width') or 0)
            except Exception:
                w = 0
            try:
                h = int(img.get('height') or 0)
            except Exception:
                h = 0
            area = 0
            try:
                if w > 0 and h > 0:
                    area = w * h
            except Exception:
                area = 0
            score = (0 if has_size_suffix else 1, area)

            prev = seen_images.get(canonical_url)
            if prev is not None:
                prev_score, prev_img = prev
                # Keep the better one; drop the other.
                if score > prev_score:
                    try:
                        prev_img.getparent().remove(prev_img)
                    except Exception:
                        pass
                    seen_images[canonical_url] = (score, img)
                else:
                    try:
                        img.getparent().remove(img)
                    except Exception:
                        pass
                    continue
            else:
                seen_images[canonical_url] = (score, img)

            if not do_download:
                try:
                    alt = img.get('alt') or ''
                except Exception:
                    alt = ''
                try:
                    if alt:
                        img.addprevious(_lhtml.fromstring('<span>%s</span>' % _lhtml.escape(alt)))
                except Exception:
                    pass
                try:
                    img.getparent().remove(img)
                except Exception:
                    pass
                continue

            try:
                rel = _download_one(abs_url)
            except Exception as e:
                if debug_images:
                    try:
                        logging.warning('EXC in _download_one %r: %s', abs_url, str(e))
                    except Exception:
                        pass
                rel = None
            if rel:
                try:
                    img.set('src', rel)
                    img.set('data-export-local', '1')
                except Exception:
                    pass
                try:
                    if img.get('srcset') is not None:
                        img.attrib.pop('srcset', None)
                except Exception:
                    pass
                # Inject Calibre-style CSS for image fit
                try:
                    style = img.get('style') or ''
                    if 'max-width' not in style:
                        style = style.rstrip(';') + ';max-width:95%;height:auto;width:auto;'
                        img.set('style', style)
                except Exception:
                    pass
            else:
                # If we cannot embed the image (cache miss + network disabled, or
                # network fetch failed), keep the original <img> so the export
                # stays faithful. (Some readers may still load remote images.)
                try:
                    img.set('data-export-missing', '1')
                except Exception:
                    pass

        # Replace YouTube iframes/links with placeholders
        try:
            iframes = node.xpath('.//iframe')
        except Exception:
            iframes = []
        for iframe in iframes:
            try:
                src = iframe.get('src') or ''
                if 'youtube.com' in src or 'youtu.be' in src:
                    a = _lhtml.fromstring('<div class="youtube-placeholder"><a href="%s" target="_blank">▶ YouTube video: %s</a></div>' % (src, src))
                    iframe.addprevious(a)
                    iframe.getparent().remove(iframe)
            except Exception:
                pass

        try:
            links = node.xpath('.//a')
        except Exception:
            links = []
        for a in links:
            try:
                href = a.get('href') or ''
                if ('youtube.com' in href or 'youtu.be' in href) and not a.text_content().strip():
                    a.text = '▶ YouTube video: %s' % href
            except Exception:
                pass

    try:
        out_html = ''.join(_lhtml.tostring(x, encoding='unicode') for x in frags)
        # Add Calibre-style export footer if this is a full article export
        if '<body' in out_html:
            footer = '<div style="margin-top:2em;text-align:left;font-size:90%;color:#888;">This article was downloaded by RSS Reader Plugin for calibre</div>'
            out_html = out_html.replace('</body>', f'{footer}</body>')
        return out_html
    except Exception:
        return html_str or ''
