# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import absolute_import, division, print_function, unicode_literals

import os
import traceback
import uuid
import shutil
import tempfile
import datetime


from calibre_plugins.rss_reader import rss_db
from calibre_plugins.rss_reader.config import plugin_prefs

# `load_translations` may not be available in all calibre embed contexts;
# translations are handled at plugin entry-points. Omit importing/calling here.

try:
    from qt.core import (
        QAction,
        QAbstractItemView,
        QBrush,
        QButtonGroup,
        QCheckBox,
        QColor,
        QDialog,
        QDialogButtonBox,
        QFont,
        QHBoxLayout,
        QLabel,
        QLineEdit,
        QListWidget,
        QListWidgetItem,
        QMenu,
        QMessageBox,
        QPlainTextEdit,
        QPushButton,
        QRadioButton,
        QTableWidget,
        QTableWidgetItem,
        QToolButton,
        Qt,
        QVBoxLayout,
        QWidget,
    )
except Exception:
    from PyQt5.Qt import (
        QAction,
        QAbstractItemView,
        QBrush,
        QButtonGroup,
        QCheckBox,
        QColor,
        QDialog,
        QDialogButtonBox,
        QFont,
        QHBoxLayout,
        QLabel,
        QLineEdit,
        QListWidget,
        QListWidgetItem,
        QMenu,
        QMessageBox,
        QPlainTextEdit,
        QPushButton,
        QRadioButton,
        QTableWidget,
        QTableWidgetItem,
        QToolButton,
        Qt,
        QVBoxLayout,
        QWidget,
    )

from calibre.gui2 import error_dialog
from calibre.utils.smtp import config as smtp_config, compose_mail, sendmail
from polyglot.binary import from_hex_unicode
from calibre.gui2.threaded_jobs import ThreadedJob
from calibre.gui2 import Dispatcher

try:
    from qt.core import QComboBox
except Exception:
    from PyQt5.QtWidgets import QComboBox


def _safe_filename(s, fallback='article'):
    s = (s or '').strip()
    if not s:
        return fallback
    for ch in '<>:"/\\|?*':
        s = s.replace(ch, '_')
    s = ' '.join(s.split())
    return s[:140].strip() or fallback


def _export_single_article_to_ebook_via_job(gui, article, plugin_prefs, done_callback, parent=None):
    """Export a single RSS article dict to an ebook file using calibre's conversion job.

    Calls done_callback(success, out_path, err) on completion.
    """
    try:
        ui_mod = __import__('calibre_plugins.rss_reader.ui', fromlist=['normalize_summary_to_html', 'choose_save_file', 'ROLE_USER'])
    except Exception as e:
        return done_callback(False, None, e)

    try:
        from calibre_plugins.rss_reader.export_oeb import build_oeb_periodical, append_enclosure_images
    except Exception as e:
        return done_callback(False, None, e)

    # Image export helpers live in preview_browser
    try:
        from calibre_plugins.rss_reader.preview_browser import _process_images_for_export as _process_images_for_export
        from calibre_plugins.rss_reader.preview_browser import _sanitize_url_for_fetch as _sanitize_url_for_fetch
        from calibre_plugins.rss_reader.preview_browser import _normalize_images_for_preview as _normalize_images_for_preview
    except Exception:
        _process_images_for_export = None
        _sanitize_url_for_fetch = None
        def _normalize_images_for_preview(html, base_url='', preserve_local=False):
            return html

    try:
        from calibre_plugins.rss_reader.convert_utils import (
            get_effective_output_format,
            get_available_output_formats,
            resolve_conversion_output,
        )
        desired_fmt = get_effective_output_format(plugin_prefs)
        available_fmts = get_available_output_formats()
        convert_ext, final_ext, markdown_via_txt = resolve_conversion_output(desired_fmt, available_output_formats=available_fmts)
    except Exception:
        convert_ext, final_ext, markdown_via_txt = 'epub', 'epub', False

    title = str(article.get('title') or _('Article'))
    link = str(article.get('link') or '')
    published = article.get('published') or ''
    try:
        import html as _html
        _raw_summary = _html.unescape(article.get('summary') or '')
    except Exception:
        _raw_summary = article.get('summary') or ''
    summary = ui_mod.normalize_summary_to_html(_raw_summary)

    # Embed enclosure images similarly to export_selected_items
    try:
        from urllib.parse import urljoin as _urljoin, urlparse as _urlparse
        base_url_for_images = str(link or article.get('_feed_url') or '')
        enc_urls = []
        for enc in (article.get('enclosures') or []):
            try:
                if not isinstance(enc, dict):
                    continue
                eurl = (enc.get('url') or '').strip()
                etype = (enc.get('type') or '').strip().lower()
                if not eurl:
                    continue
                is_img = False
                if etype and etype.startswith('image/'):
                    is_img = True
                else:
                    p = _urlparse(eurl).path or ''
                    ext = os.path.splitext(p)[1].lower()
                    if ext in ('.jpg', '.jpeg', '.png', '.gif', '.webp', '.avif', '.bmp', '.svg'):
                        is_img = True
                if not is_img:
                    continue
                try:
                    if base_url_for_images:
                        eurl = _urljoin(base_url_for_images, eurl)
                except Exception:
                    pass
                try:
                    if _sanitize_url_for_fetch is not None:
                        eurl = _sanitize_url_for_fetch(eurl)
                except Exception:
                    pass
                enc_urls.append(eurl)
            except Exception:
                continue
        seen = set()
        enc_urls = [u for u in enc_urls if u and not (u in seen or seen.add(u))]
        if enc_urls and append_enclosure_images is not None:
            summary = append_enclosure_images(summary, enc_urls[:8], base_url=base_url_for_images, sanitize_url=_sanitize_url_for_fetch)
    except Exception:
        base_url_for_images = str(link or article.get('_feed_url') or '')

    try:
        if _process_images_for_export is not None:
            td = tempfile.mkdtemp(prefix='rss_share_export_')
            images_subdir = "feed_0/article_0/images"
            summary = _process_images_for_export(summary, base_url=base_url_for_images, td=td, do_download=True, images_subdir=images_subdir)
        else:
            td = tempfile.mkdtemp(prefix='rss_share_export_')
    except Exception:
        td = tempfile.mkdtemp(prefix='rss_share_export_')

    try:
        summary = _normalize_images_for_preview(summary, base_url=base_url_for_images or '', preserve_local=True)
    except Exception:
        pass

    now = datetime.datetime.now()
    date_in_title = now.strftime('%a, %d %b %Y')
    ts_suffix = now.strftime('%Y-%m-%d_%H-%M-%S')
    news_title = '%s [%s]' % (title, date_in_title)

    export_feeds = [{'title': news_title, 'items': [{'title': title, 'link': link, 'published': published, 'body_html': summary}]}]
    try:
        html_path = build_oeb_periodical(td, news_title, export_feeds)
    except Exception as e:
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        return done_callback(False, None, e)

    base_name = f"{_safe_filename(title)} - {ts_suffix}"
    out_path = os.path.join(tempfile.gettempdir(), f"{base_name}.{convert_ext}")
    final_out_path = os.path.join(tempfile.gettempdir(), f"{base_name}.{final_ext}") if final_ext and final_ext != convert_ext else None

    # Start calibre conversion job
    if not getattr(gui, 'job_manager', None):
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        return done_callback(False, None, Exception(_('Cannot start conversion job (job manager unavailable).')))

    func = 'gui_convert'
    try:
        from calibre.customize.conversion import OptionRecommendation
        recs = [('level1_toc', '//*[@id][starts-with(@id, "item-")]', OptionRecommendation.HIGH)]
    except Exception:
        recs = []
    try:
        if markdown_via_txt and isinstance(recs, list):
            try:
                from calibre.customize.conversion import OptionRecommendation
                recs.append(('txt_output_formatting', 'markdown', OptionRecommendation.HIGH))
            except Exception:
                recs.append(('txt_output_formatting', 'markdown', None))
    except Exception:
        pass

    args = [html_path, out_path, recs]
    desc = _('Convert RSS article for email')

    def _on_convert_done(job):
        try:
            if getattr(job, 'failed', False):
                try:
                    gui.job_exception(job)
                except Exception:
                    pass
                return done_callback(False, None, Exception(_('Conversion job failed.')))

            produced = out_path
            try:
                if final_out_path and produced and os.path.abspath(final_out_path) != os.path.abspath(produced):
                    if os.path.exists(produced) and not os.path.exists(final_out_path):
                        try:
                            os.replace(produced, final_out_path)
                        except Exception:
                            try:
                                shutil.copy2(produced, final_out_path)
                                os.remove(produced)
                            except Exception:
                                pass
                    if os.path.exists(final_out_path):
                        produced = final_out_path
            except Exception:
                pass

            return done_callback(True, produced, None)
        finally:
            try:
                shutil.rmtree(td, ignore_errors=True)
            except Exception:
                pass

    try:
        job = gui.job_manager.run_job(Dispatcher(_on_convert_done), func, args=args, description=desc)
        try:
            gui.jobs_pointer.start()
        except Exception:
            pass
        try:
            if hasattr(gui, 'status_bar'):
                gui.status_bar.show_message(_('Converting article for email…'), 4000)
        except Exception:
            pass
        return job
    except Exception as e:
        try:
            shutil.rmtree(td, ignore_errors=True)
        except Exception:
            pass
        return done_callback(False, None, e)

try:
    load_translations()
except NameError:
    pass

# Role constant for storing item user-data (Qt version compatibility)
try:
    ROLE_USER = getattr(getattr(Qt, 'ItemDataRole', None), 'UserRole', getattr(Qt, 'UserRole', 32))
except Exception:
    ROLE_USER = 32


# Profile emoji choices (user-visible list)
PROFILE_EMOJI_CHOICES = [
    '📁', '📚', '📖', '📕', '📗', '📘', '📙', '🔖', '📌', '🎧', '🌟', '🛠️', '🏠', '🎯', '🔒', '🧪', '☁️', '🗃️', '🧾', '🏝️', '🖼️', '🖌️', '❤️', '📦', '🗠', '🏔️'
]

ROLE_USER = getattr(getattr(Qt, 'ItemDataRole', None), 'UserRole', getattr(Qt, 'UserRole', 32))


class TagEditorDialog(QDialog):
    """Simple tag editor: shows manual tags, + to add, per-tag - to remove.

    Optionally shows read-only tags (e.g. auto-tags) for context.
    """

    def __init__(self, parent=None, title=None, subtitle=None, initial_tags=None, readonly_tags=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle(str(title or _('Edit tags')))

        self._tags = []
        self._seen = set()

        layout = QVBoxLayout(self)
        if subtitle:
            layout.addWidget(QLabel(str(subtitle), self))

        self.readonly_label = None
        try:
            ro = [str(x).strip() for x in (readonly_tags or []) if str(x).strip()]
        except Exception:
            ro = []
        if ro:
            try:
                self.readonly_label = QLabel(self)
                self.readonly_label.setWordWrap(True)
                self.readonly_label.setStyleSheet('color: gray;')
                self.readonly_label.setText(_('Auto tags: %s') % ', '.join(ro))
                layout.addWidget(self.readonly_label)
            except Exception:
                self.readonly_label = None

        self.list = QListWidget(self)
        try:
            self.list.setSelectionMode(QAbstractItemView.SelectionMode.NoSelection)
        except Exception:
            try:
                self.list.setSelectionMode(QAbstractItemView.NoSelection)
            except Exception:
                pass
        layout.addWidget(self.list)

        row = QHBoxLayout()
        self.add_edit = QLineEdit(self)
        self.add_edit.setPlaceholderText(_('Add tag…'))
        row.addWidget(self.add_edit)

        self.add_btn = QToolButton(self)
        self.add_btn.setText('+')
        self.add_btn.setToolTip(_('Add tag'))
        row.addWidget(self.add_btn)
        layout.addLayout(row)

        btns = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, self)
        btns.accepted.connect(self.accept)
        btns.rejected.connect(self.reject)
        layout.addWidget(btns)

        try:
            self.add_btn.clicked.connect(self._on_add_clicked)
            self.add_edit.returnPressed.connect(self._on_add_clicked)
        except Exception:
            pass

        for t in (initial_tags or []):
            self._add_tag(t)
        self._rebuild()

        try:
            self.resize(420, 320)
        except Exception:
            pass

    def _normalize(self, s):
        try:
            s = str(s or '').strip()
        except Exception:
            s = ''
        if not s:
            return ''
        s = s.replace('\u00A0', ' ')
        s = ' '.join(s.split())
        return s.casefold()

    def _add_tag(self, raw):
        for chunk in str(raw or '').replace(';', ',').split(','):
            t = self._normalize(chunk)
            if not t or t in self._seen:
                continue
            self._seen.add(t)
            self._tags.append(t)

    def _remove_tag(self, tag):
        t = self._normalize(tag)
        if not t:
            return
        try:
            self._tags = [x for x in (self._tags or []) if x != t]
            self._seen = set(self._tags or [])
        except Exception:
            pass

    def _rebuild(self):
        try:
            self.list.clear()
        except Exception:
            pass

        for t in (self._tags or []):
            try:
                it = QListWidgetItem(self.list)
                w = QWidget(self.list)
                l = QHBoxLayout(w)
                l.setContentsMargins(6, 2, 6, 2)
                lab = QLabel(str(t), w)
                l.addWidget(lab)
                l.addStretch(1)
                b = QToolButton(w)
                b.setText('-')
                b.setToolTip(_('Remove tag'))
                try:
                    b.clicked.connect(lambda _=False, _t=t: (self._remove_tag(_t), self._rebuild()))
                except Exception:
                    pass
                l.addWidget(b)
                try:
                    it.setSizeHint(w.sizeHint())
                except Exception:
                    pass
                try:
                    self.list.setItemWidget(it, w)
                except Exception:
                    pass
            except Exception:
                pass

    def _on_add_clicked(self):
        try:
            s = str(self.add_edit.text() or '')
        except Exception:
            s = ''
        if not s.strip():
            return
        try:
            self._add_tag(s)
        except Exception:
            pass
        try:
            self.add_edit.setText('')
        except Exception:
            pass
        self._rebuild()

    def tags(self):
        return list(self._tags or [])


class _AddFeedDialog(QDialog):
    def __init__(self, parent=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle(_('Add feed'))

        layout = QVBoxLayout(self)

        layout.addWidget(QLabel(_('Feed URL:'), self))
        self.url_edit = QLineEdit(self)
        self.url_edit.setPlaceholderText('https://example.com/feed.xml')
        layout.addWidget(self.url_edit)

        layout.addWidget(QLabel(_('Display name (optional):'), self))
        self.title_edit = QLineEdit(self)
        self.title_edit.setPlaceholderText(_('Leave blank to auto-detect'))
        layout.addWidget(self.title_edit)

        # Download images checkbox (default: enabled)
        try:
            from qt.core import QCheckBox
        except Exception:
            from PyQt5.Qt import QCheckBox
        self.download_images_cb = QCheckBox(_('Include images in export and downloads for this feed'), self)
        try:
            self.download_images_cb.setChecked(True)
            self.download_images_cb.setToolTip(_('If checked, images will be downloaded and included in exports and persistent storage. This does not affect the preview toggle.'))
        except Exception:
            pass
        layout.addWidget(self.download_images_cb)

        # Per-feed recipe engine opt-in (off by default)
        self.use_recipe_engine_cb = QCheckBox(_('Use calibre recipe engine for full-article fetch (Experimental)'), self)
        try:
            try:
                default_on = bool(plugin_prefs.get('preview_use_recipe_engine', False))
            except Exception:
                default_on = False
            self.use_recipe_engine_cb.setChecked(bool(default_on))
            self.use_recipe_engine_cb.setToolTip(_('If enabled, the plugin will invoke calibre\'s recipe engine for this feed when fetching full article content. Use only when needed; it may be slower and use more memory.'))
        except Exception:
            pass
        layout.addWidget(self.use_recipe_engine_cb)

        # Oldest article (days) and Max articles per feed
        try:
            from qt.core import QSpinBox
        except Exception:
            from PyQt5.Qt import QSpinBox
        self.oldest_days_lbl = QLabel(_('Oldest article (days, 0 = no limit):'), self)
        self.oldest_days_spin = QSpinBox(self)
        try:
            self.oldest_days_spin.setRange(0, 36500)
            try:
                d = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
            except Exception:
                d = 7
            self.oldest_days_spin.setValue(max(0, int(d)))
        except Exception:
            pass
        layout.addWidget(self.oldest_days_lbl)
        layout.addWidget(self.oldest_days_spin)

        self.max_articles_lbl = QLabel(_('Max. number of articles (0 = no limit):'), self)
        self.max_articles_spin = QSpinBox(self)
        try:
            self.max_articles_spin.setRange(0, 100000)
            try:
                m = int(plugin_prefs.get('default_max_articles', 100) or 100)
            except Exception:
                m = 100
            self.max_articles_spin.setValue(max(0, int(m)))
        except Exception:
            pass
        layout.addWidget(self.max_articles_lbl)
        layout.addWidget(self.max_articles_spin)

        btns = QHBoxLayout()
        btns.addStretch(1)
        self.cancel_btn = QPushButton(_('Cancel'), self)
        self.ok_btn = QPushButton(_('Add'), self)
        try:
            self.cancel_btn.clicked.connect(self.reject)
            self.ok_btn.clicked.connect(self.accept)
        except Exception:
            pass
        # Place primary action (Add) before Cancel for a primary-first layout
        btns.addWidget(self.ok_btn)
        btns.addWidget(self.cancel_btn)
        self._buttons_layout = btns
        layout.addLayout(btns)

        try:
            self.resize(520, 160)
        except Exception:
            pass

    def get_values(self):
        try:
            url = self.url_edit.text().strip()
        except Exception:
            url = ''
        try:
            title = self.title_edit.text().strip()
        except Exception:
            title = ''
        try:
            download_images = bool(self.download_images_cb.isChecked())
        except Exception:
            download_images = True
        try:
            use_recipe_engine = bool(self.use_recipe_engine_cb.isChecked())
        except Exception:
            use_recipe_engine = False
        try:
            oldest_days = int(getattr(self, 'oldest_days_spin', None).value() or 0)
        except Exception:
            oldest_days = 0
        try:
            max_articles = int(getattr(self, 'max_articles_spin', None).value() or 0)
        except Exception:
            max_articles = 0
        return (url, title, download_images, use_recipe_engine, oldest_days, max_articles)


class _AddEditProfileDialog(QDialog):
    def __init__(self, parent=None, profile=None, initial=None, mode=None):
        QDialog.__init__(self, parent)
        self._mode = mode or ''
        # Context-aware window title
        if mode == 'add':
            self.setWindowTitle(_('Add existing database as profile'))
        elif mode == 'new':
            self.setWindowTitle(_('Create new database profile'))
        elif mode == 'save':
            self.setWindowTitle(_('Save current DB as profile'))
        elif profile:
            self.setWindowTitle(_('Edit profile'))
        else:
            self.setWindowTitle(_('Profile'))
        self._parent = parent
        self._initial = dict(initial or {})

        layout = QVBoxLayout(self)

        layout.addWidget(QLabel(_('Profile name:'), self))
        self.name_edit = QLineEdit(self)
        layout.addWidget(self.name_edit)

        layout.addWidget(QLabel(_('Emoji (optional):'), self))
        try:
            from qt.core import QComboBox, QFileDialog
        except Exception:
            from PyQt5.Qt import QComboBox, QFileDialog
        self.emoji_cb = QComboBox(self)
        try:
            # Keep the legacy behavior: first item is a blank/empty choice.
            # (User can choose a real emoji from the list below.)
            self.emoji_cb.addItem('')
            for e in PROFILE_EMOJI_CHOICES:
                self.emoji_cb.addItem(e)
        except Exception:
            pass
        layout.addWidget(self.emoji_cb)

        layout.addWidget(QLabel(_('Database file path:'), self))
        row = QHBoxLayout()
        self.path_edit = QLineEdit(self)
        row.addWidget(self.path_edit)
        browse = QPushButton(_('Browse…'), self)

        def _b():
            try:
                def _sanitize_filename(x):
                    try:
                        s = str(x or '').strip()
                    except Exception:
                        s = ''
                    if not s:
                        s = 'RSS_Reader'
                    # Windows + general filesystem sanitation
                    for ch in '<>:"/\\|?*':
                        s = s.replace(ch, '_')
                    try:
                        s = s.strip().strip('.')
                    except Exception:
                        pass
                    return s or 'RSS_Reader'

                def _ensure_ext(path):
                    try:
                        p = str(path or '').strip()
                    except Exception:
                        return path
                    if not p:
                        return p
                    try:
                        root, ext = os.path.splitext(p)
                        if not ext:
                            return root + '.sqlite'
                    except Exception:
                        pass
                    return p

                def _suggest_save_path():
                    try:
                        cur = str(self.path_edit.text() or '').strip()
                    except Exception:
                        cur = ''
                    try:
                        name = _sanitize_filename(self.name_edit.text() or '')
                    except Exception:
                        name = 'RSS_Reader'
                    default_dir = ''
                    default_name = name + '.sqlite'

                    if cur:
                        try:
                            if os.path.isdir(cur):
                                default_dir = cur
                            else:
                                bn = os.path.basename(cur)
                                # If a file is already typed, preserve it (but ensure extension)
                                if bn and bn.lower().endswith(('.sqlite', '.db')):
                                    return cur
                                if bn and os.path.splitext(bn)[0]:
                                    default_name = os.path.splitext(bn)[0] + '.sqlite'
                                default_dir = os.path.dirname(cur) or ''
                        except Exception:
                            default_dir = ''

                    if not default_dir:
                        try:
                            default_dir = os.path.expanduser('~')
                        except Exception:
                            default_dir = ''

                    try:
                        return os.path.join(default_dir, default_name) if default_dir else default_name
                    except Exception:
                        return default_name

                # Determine desired flow
                try:
                    readonly_or_mirror = bool(self.mode_readonly_rb.isChecked() or self.mode_mirror_rb.isChecked())
                except Exception:
                    readonly_or_mirror = False
                try:
                    create_blank = bool(getattr(self, 'create_blank_cb', None) is not None and self.create_blank_cb.isChecked())
                except Exception:
                    create_blank = False
                try:
                    allow_create_new = bool(getattr(self, 'create_blank_cb', None) is not None and self.create_blank_cb.isEnabled())
                except Exception:
                    allow_create_new = False

                # If this dialog is in a "create new blank DB" flow, go straight to Save.
                if (not readonly_or_mirror) and create_blank:
                    default = _suggest_save_path()
                    try:
                        from calibre.gui2.qt_file_dialogs import choose_save_file
                        fname = choose_save_file(self, _('Database file'), default, _('SQLite files (*.sqlite *.db);;All files (*)'))
                    except Exception:
                        fname, _dummy = QFileDialog.getSaveFileName(self, _('Database file'), default, _('SQLite files (*.sqlite *.db);;All files (*)'))
                    fname = _ensure_ext(fname)
                else:
                    # Otherwise, Open existing DB.
                    if readonly_or_mirror:
                        title = _('Select database file')
                    else:
                        title = _('Select database file (or Cancel to create new)') if allow_create_new else _('Select database file')
                    try:
                        from calibre.gui2.qt_file_dialogs import choose_open_file
                        fname = choose_open_file(self, title, self.path_edit.text() or '', _('SQLite files (*.sqlite *.db);;All files (*)'))
                    except Exception:
                        fname, _dummy = QFileDialog.getOpenFileName(self, title, self.path_edit.text() or '', _('SQLite files (*.sqlite *.db);;All files (*)'))

                    # If user cancelled Open and this dialog allows creating a new DB, fall back to Save.
                    if (not fname) and (not readonly_or_mirror) and allow_create_new:
                        default = _suggest_save_path()
                        try:
                            from calibre.gui2.qt_file_dialogs import choose_save_file
                            fname = choose_save_file(self, _('Database file'), default, _('SQLite files (*.sqlite *.db);;All files (*)'))
                        except Exception:
                            fname, _dummy = QFileDialog.getSaveFileName(self, _('Database file'), default, _('SQLite files (*.sqlite *.db);;All files (*)'))
                        fname = _ensure_ext(fname)
                        # If user picked a new file location, reflect that intent.
                        try:
                            if getattr(self, 'create_blank_cb', None) is not None and self.create_blank_cb.isEnabled():
                                self.create_blank_cb.setChecked(True)
                        except Exception:
                            pass
                if fname:
                    try:
                        rss_db.assert_db_path_allowed(str(fname))
                    except Exception as e:
                        try:
                            QMessageBox.warning(self, _('Invalid DB location'), str(e))
                        except Exception:
                            pass
                        return
                    self.path_edit.setText(str(fname))
            except Exception:
                pass

        browse.clicked.connect(_b)
        row.addWidget(browse)
        layout.addLayout(row)

        try:
            from qt.core import QCheckBox
        except Exception:
            from PyQt5.Qt import QCheckBox
        self.create_blank_cb = QCheckBox(_('Create blank DB file at this path'), self)
        layout.addWidget(self.create_blank_cb)

        # Access mode
        self.mode_group = QButtonGroup(self)
        self.mode_writable_rb = QRadioButton(_('Writable (default)'), self)
        self.mode_readonly_rb = QRadioButton(_('Read-only (safe for sharing)'), self)
        self.mode_mirror_rb = QRadioButton(_('Mirror (read-only, disables all editing)'), self)
        try:
            self.mode_group.addButton(self.mode_writable_rb)
            self.mode_group.addButton(self.mode_readonly_rb)
            self.mode_group.addButton(self.mode_mirror_rb)
        except Exception:
            pass
        layout.addWidget(self.mode_writable_rb)
        layout.addWidget(self.mode_readonly_rb)
        layout.addWidget(self.mode_mirror_rb)
        try:
            self.mode_writable_rb.setChecked(True)
        except Exception:
            pass

        def _sync_mode():
            try:
                writable = bool(self.mode_writable_rb.isChecked())
            except Exception:
                writable = True
            try:
                self.create_blank_cb.setEnabled(writable)
            except Exception:
                pass

        try:
            self.mode_writable_rb.toggled.connect(_sync_mode)
            self.mode_readonly_rb.toggled.connect(_sync_mode)
            self.mode_mirror_rb.toggled.connect(_sync_mode)
        except Exception:
            pass

        btns = QHBoxLayout()
        btns.addStretch(1)
        cancel = QPushButton(_('Cancel'), self)
        # Context-aware button label
        if self._mode == 'add':
            ok = QPushButton(_('Add'), self)
        elif self._mode == 'new':
            ok = QPushButton(_('Create'), self)
        else:
            ok = QPushButton(_('Save'), self)
        cancel.clicked.connect(self.reject)
        ok.clicked.connect(self.accept)
        btns.addWidget(cancel)
        btns.addWidget(ok)
        layout.addLayout(btns)

        if profile:
            try:
                self.name_edit.setText(str(profile.get('name') or ''))
            except Exception:
                pass
            try:
                emoji = str(profile.get('emoji') or '')
                if emoji and emoji in PROFILE_EMOJI_CHOICES:
                    i = PROFILE_EMOJI_CHOICES.index(emoji) + 1
                    try:
                        self.emoji_cb.setCurrentIndex(i)
                    except Exception:
                        pass
            except Exception:
                pass
            try:
                self.path_edit.setText(str(profile.get('path') or ''))
            except Exception:
                pass
            try:
                if bool(profile.get('mirror', False)):
                    self.mode_mirror_rb.setChecked(True)
                elif bool(profile.get('readonly', False)):
                    self.mode_readonly_rb.setChecked(True)
                else:
                    self.mode_writable_rb.setChecked(True)
            except Exception:
                pass
        else:
            # Initial values (used when saving current DB as a profile)
            try:
                if self._initial.get('name'):
                    self.name_edit.setText(str(self._initial.get('name') or ''))
            except Exception:
                pass
            try:
                if self._initial.get('path'):
                    self.path_edit.setText(str(self._initial.get('path') or ''))
            except Exception:
                pass
            try:
                if bool(self._initial.get('mirror', False)):
                    self.mode_mirror_rb.setChecked(True)
                elif bool(self._initial.get('readonly', False)):
                    self.mode_readonly_rb.setChecked(True)
                else:
                    self.mode_writable_rb.setChecked(True)
            except Exception:
                pass

        try:
            _sync_mode()
        except Exception:
            pass

        try:
            # Make dialog wider so the DB path field has room
            try:
                self.setMinimumWidth(640)
            except Exception:
                try:
                    self.resize(640, 240)
                except Exception:
                    pass
        except Exception:
            pass

    def get_values(self):
        try:
            readonly = bool(self.mode_readonly_rb.isChecked())
        except Exception:
            readonly = False
        try:
            mirror = bool(self.mode_mirror_rb.isChecked())
        except Exception:
            mirror = False
        return {
            'name': str(self.name_edit.text() or '').strip(),
            'emoji': (self.emoji_cb.currentText() or '').strip(),
            'path': str(self.path_edit.text() or '').strip(),
            'create_blank': bool(getattr(self, 'create_blank_cb', None) and self.create_blank_cb.isChecked()),
            'readonly': readonly,
            'mirror': mirror,
        }


class ProfilesDialog(QDialog):
    def __init__(self, parent=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle(_('Manage profiles'))
        self.parent = parent
        # Restore geometry if available
        try:
            import base64
            geom_b64 = str(plugin_prefs.get('profiles_dialog_geometry') or '')
            if geom_b64:
                try:
                    geom_bytes = base64.b64decode(geom_b64)
                    self.restoreGeometry(geom_bytes)
                except Exception:
                    pass
        except Exception:
            pass

        try:
            from qt.core import QListWidget, QListWidgetItem
        except Exception:
            from PyQt5.Qt import QListWidget, QListWidgetItem

        layout = QVBoxLayout(self)

        self.list = QListWidget(self)
        # Theme-aware readability: let Qt use palette Base/AlternateBase
        try:
            self.list.setAlternatingRowColors(True)
        except Exception:
            pass
        try:
            self.list.setWordWrap(True)
        except Exception:
            pass
        try:
            self.list.setSpacing(2)
        except Exception:
            pass
        try:
            # Only spacing/padding; avoid hard-coded colors (theme aware)
            self.list.setStyleSheet('QListWidget::item{padding:6px 6px;}')
        except Exception:
            pass
        layout.addWidget(self.list)

        btn_row = QHBoxLayout()
        self.new_btn = QPushButton(_('New'), self)
        self.add_btn = QPushButton(_('Add'), self)
        self.edit_btn = QPushButton(_('Edit'), self)
        self.remove_btn = QPushButton(_('Remove'), self)
        self.reset_btn = QPushButton(_('Reset'), self)
        try:
            self.reset_btn.setToolTip(_('Reset profiles (Ctrl+R). Clears all saved profiles and the active profile. Does not delete any database files.'))
        except Exception:
            pass
        self.clear_btn = QPushButton(_('Clear DB…'), self)
        self.set_default_btn = QPushButton(_('Set Default'), self)
        self.switch_btn = QPushButton(_('Switch'), self)

        # Make Enter/Return activate Edit (not New)
        try:
            for b in (self.new_btn, self.add_btn, self.remove_btn, self.reset_btn, self.clear_btn, self.set_default_btn, self.switch_btn):
                try:
                    b.setAutoDefault(False)
                except Exception:
                    pass
                try:
                    b.setDefault(False)
                except Exception:
                    pass
        except Exception:
            pass
        try:
            self.edit_btn.setAutoDefault(True)
        except Exception:
            pass
        try:
            self.edit_btn.setDefault(True)
        except Exception:
            pass

        # Button tooltips with usage hints + shortcuts
        try:
            self.new_btn.setToolTip(_('Create a new blank database profile (Ctrl+N).'))
        except Exception:
            pass
        try:
            self.add_btn.setToolTip(_('Add an existing database profile (Ctrl+Shift+A). Requires the file to already exist.'))
        except Exception:
            pass
        try:
            self.edit_btn.setToolTip(_('Edit the selected entry (Enter/F2/Ctrl+E). For the "(default profile)" row, this opens Current DB settings (path, emoji, mode).'))
        except Exception:
            pass
        try:
            self.remove_btn.setToolTip(_('Remove the selected saved profile (Del). This only removes the profile entry; it does not delete any database files.'))
        except Exception:
            pass
        try:
            self.clear_btn.setToolTip(_('Delete ALL data from the selected database (feeds, cache, tags, history). Keeps the file/path.'))
        except Exception:
            pass
        try:
            self.switch_btn.setToolTip(_('Switch to the selected database/profile (Ctrl+S).'))
        except Exception:
            pass
        try:
            # Initial tooltip; updated dynamically in _update_buttons to note default behavior
            self.set_default_btn.setToolTip(_('Set the selected row as the startup default — the database that opens automatically when RSS Reader launches. (Ctrl+D)'))
        except Exception:
            pass
        btn_row.addWidget(self.new_btn)
        btn_row.addWidget(self.add_btn)
        btn_row.addWidget(self.edit_btn)
        btn_row.addWidget(self.remove_btn)
        btn_row.addWidget(self.reset_btn)
        btn_row.addWidget(self.clear_btn)
        btn_row.addWidget(self.set_default_btn)
        btn_row.addWidget(self.switch_btn)
        layout.addLayout(btn_row)

        # Help footer (shortcuts + quick tips)
        try:
            help_text = (
                _('Shortcuts: ') +
                _('New') + ' Ctrl+N, ' +
                _('Add') + ' Ctrl+Shift+A, ' +
                _('Edit') + ' Enter/F2/Ctrl+E, ' +
                _('Remove') + ' Del, ' +
                _('Switch') + ' Ctrl+S, ' +
                _('Set Default') + ' Ctrl+D, ' +
                _('Reset') + ' Ctrl+R.\n' +
                _('Tip: Double-click a row to edit. Right-click a row to open its folder in the file manager.')
            )
            self._help_label = QLabel(help_text, self)
            try:
                self._help_label.setWordWrap(True)
            except Exception:
                pass
            try:
                self._help_label.setStyleSheet('color: gray; font-size: 9pt;')
            except Exception:
                pass
            layout.addWidget(self._help_label)
        except Exception:
            self._help_label = None

        close_row = QHBoxLayout()
        close_row.addStretch(1)
        close_btn = QPushButton(_('Close'), self)
        close_btn.clicked.connect(self.reject)
        close_row.addWidget(close_btn)
        layout.addLayout(close_row)

        self.new_btn.clicked.connect(self.new_profile)
        self.add_btn.clicked.connect(self.add_profile)
        self.edit_btn.clicked.connect(self.edit_profile)
        self.remove_btn.clicked.connect(self.remove_profile)
        self.reset_btn.clicked.connect(self.reset_profiles)
        self.clear_btn.clicked.connect(self.clear_database)
        self.set_default_btn.clicked.connect(self.set_default_profile)
        self.switch_btn.clicked.connect(self.switch_profile)

        # Enable right-click context menu on list
        try:
            self.list.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            self.list.customContextMenuRequested.connect(self._on_list_context_menu)
        except Exception:
            try:
                self.list.setContextMenuPolicy(Qt.CustomContextMenu)
                self.list.customContextMenuRequested.connect(self._on_list_context_menu)
            except Exception:
                pass

        # Update the action buttons whenever the selection changes
        try:
            self.list.itemSelectionChanged.connect(self._update_buttons)
        except Exception:
            pass

        # Only resize if geometry wasn't restored
        try:
            if not str(plugin_prefs.get('profiles_dialog_geometry') or '').strip():
                self.resize(700, 360)
        except Exception:
            self.resize(700, 360)
        self._populate()

        # ---- Wiring that must happen on construction (not on close) ----
        try:
            # itemActivated fires on Enter/Return and also double-click.
            # Do not also connect itemDoubleClicked or we will edit twice.
            try:
                self.list.itemActivated.connect(lambda _it=None: self.edit_profile())
            except Exception:
                pass
        except Exception:
            pass

        # Add keyboard shortcuts for common actions
        try:
            from qt.core import QShortcut, QKeySequence
        except Exception:
            try:
                from PyQt5.Qt import QShortcut, QKeySequence
            except Exception:
                QShortcut = None
                QKeySequence = None
        try:
            if QShortcut is not None and QKeySequence is not None:
                def _ctx(sc):
                    try:
                        sc.setContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
                    except Exception:
                        try:
                            sc.setContext(Qt.WidgetWithChildrenShortcut)
                        except Exception:
                            pass
                    return sc

                self._sc_new = _ctx(QShortcut(QKeySequence('Ctrl+N'), self, self.new_profile))
                self._sc_add = _ctx(QShortcut(QKeySequence('Ctrl+Shift+A'), self, self.add_profile))
                self._sc_edit = _ctx(QShortcut(QKeySequence('Ctrl+E'), self, self.edit_profile))
                # Enter/Return/F2 also trigger Edit
                try:
                    self._sc_enter = _ctx(QShortcut(QKeySequence('Return'), self, self.edit_profile))
                except Exception:
                    pass
                try:
                    self._sc_enter2 = _ctx(QShortcut(QKeySequence('Enter'), self, self.edit_profile))
                except Exception:
                    pass
                try:
                    self._sc_f2 = _ctx(QShortcut(QKeySequence('F2'), self, self.edit_profile))
                except Exception:
                    pass
                try:
                    self._sc_remove = _ctx(QShortcut(QKeySequence('Del'), self, self.remove_profile))
                except Exception:
                    pass
                try:
                    self._sc_switch = _ctx(QShortcut(QKeySequence('Ctrl+S'), self, self.switch_profile))
                except Exception:
                    pass
                try:
                    self._sc_default = _ctx(QShortcut(QKeySequence('Ctrl+D'), self, self.set_default_profile))
                except Exception:
                    pass
                try:
                    self._sc_reset = _ctx(QShortcut(QKeySequence('Ctrl+R'), self, self.reset_profiles))
                except Exception:
                    pass
        except Exception:
            pass

        # Make the list take initial focus for keyboard-driven workflows
        try:
            self.list.setFocus()
        except Exception:
            pass
        try:
            self._update_buttons()
        except Exception:
            pass

    def accept(self):
        self._save_geometry()
        super().accept()

    def reject(self):
        self._save_geometry()
        super().reject()

    def closeEvent(self, event):
        self._save_geometry()
        super().closeEvent(event)

    def _save_geometry(self):
        try:
            import base64
            g = self.saveGeometry()
            if g:
                try:
                    plugin_prefs['profiles_dialog_geometry'] = base64.b64encode(bytes(g)).decode('ascii')
                except Exception:
                    pass
        except Exception:
            pass

    def _on_list_context_menu(self, pos):
        """Show context menu for DB entries (right-click to open in file explorer)."""
        try:
            it = self.list.itemAt(pos)
            if it is None:
                return

            pid = str(it.data(ROLE_USER) or '')

            # Get the path from the item
            db_path = ''
            if pid == '__default__':
                try:
                    db_path = str(getattr(self.parent, '_db_orig_path', None) or '')
                    if not db_path:
                        db_path = rss_db.db_path() or ''
                except Exception:
                    pass
            elif pid == '__config__':
                try:
                    db_path = str(getattr(self.parent, '_db_config_path', '') or '').strip()
                except Exception:
                    pass
            elif pid == '__current__':
                try:
                    db_path = str(rss_db.db_path() or '').strip()
                except Exception:
                    pass
            else:
                # Named profile
                try:
                    profiles = plugin_prefs.get('db_profiles') or []
                    for p in profiles:
                        if str(p.get('id') or '') == pid:
                            db_path = str(p.get('path') or '').strip()
                            break
                except Exception:
                    pass

            if not db_path:
                return

            # Ensure path is normalized
            try:
                db_path = os.path.normpath(db_path)
            except Exception:
                pass

            # Create context menu
            try:
                m = QMenu(self)
                act = QAction(_('Open location in file manager'), m)
                try:
                    act.setToolTip(_('Open the folder containing this database. If this is a temporarily opened "one-off" database (not saved as a profile), use "Add" to save it or "Set Default" to make it the default.'))
                except Exception:
                    pass

                def _open_location():
                    self._open_db_location(db_path)

                act.triggered.connect(_open_location)
                m.addAction(act)

                try:
                    m.exec(self.list.mapToGlobal(pos))
                except Exception:
                    try:
                        m.exec_(self.list.mapToGlobal(pos))
                    except Exception:
                        pass
            except Exception:
                pass
        except Exception:
            pass

    def _open_db_location(self, db_path):
        """Open the database file location in the system file manager (cross-platform)."""
        if not db_path or not os.path.exists(db_path):
            try:
                QMessageBox.warning(self, _('File not found'), _('Database file not found: %s') % db_path)
            except Exception:
                pass
            return

        try:
            import platform
            import subprocess

            system = platform.system()

            if system == 'Darwin':  # macOS
                # Use 'open -R' to reveal the file in Finder
                subprocess.Popen(['open', '-R', db_path])
            elif system == 'Windows':
                # Use 'explorer /select' to select the file
                try:
                    # Windows: use explorer with /select to highlight the file
                    subprocess.Popen(['explorer', '/select,', os.path.normpath(db_path)])
                except Exception:
                    # Fallback: just open the folder
                    folder = os.path.dirname(db_path)
                    subprocess.Popen(['explorer', os.path.normpath(folder)])
            else:  # Linux and other Unix-like systems
                # Try common file managers
                folder = os.path.dirname(db_path)
                file_managers = ['nautilus', 'thunar', 'dolphin', 'nemo', 'caja', 'pcmanfm']
                opened = False

                for fm in file_managers:
                    try:
                        subprocess.Popen([fm, folder])
                        opened = True
                        break
                    except (OSError, subprocess.CalledProcessError):
                        continue

                if not opened:
                    # Last resort: try xdg-open
                    try:
                        subprocess.Popen(['xdg-open', folder])
                    except Exception:
                        try:
                            QMessageBox.warning(self, _('Error'), _('Could not open file manager for: %s') % folder)
                        except Exception:
                            pass
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Error'), _('Failed to open file manager: %s') % str(e))
            except Exception:
                pass

    def reset_profiles(self):
        try:
            res = QMessageBox.question(
                self,
                _('Reset profiles'),
                _('This will remove all saved profiles and clear the active profile.\n\nIt will NOT delete any database files. Continue?'),
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
            )
        except Exception:
            return
        if res != QMessageBox.StandardButton.Yes:
            return

        try:
            plugin_prefs['db_profiles'] = []
        except Exception:
            pass
        try:
            plugin_prefs['db_profiles_active'] = ''
        except Exception:
            pass

        # Switch back to Default DB (best-effort)
        try:
            if getattr(self, 'parent', None) is not None and hasattr(self.parent, '_restore_default_database'):
                self.parent._restore_default_database()
        except Exception:
            pass

        try:
            self._populate()
        except Exception:
            pass
        try:
            if getattr(self, 'parent', None) is not None and hasattr(self.parent, '_rebuild_profiles_menu'):
                self.parent._rebuild_profiles_menu()
        except Exception:
            pass
        try:
            if getattr(self, 'parent', None) is not None and hasattr(self.parent, '_update_profile_label'):
                self.parent._update_profile_label()
        except Exception:
            pass

    def _populate(self):
        self.list.clear()
        try:
            try:
                from qt.core import QListWidgetItem
            except Exception:
                from PyQt5.Qt import QListWidgetItem
        except Exception:
            QListWidgetItem = None
        # Use the module-level plugin_prefs imported at module top to avoid import/circular issues
        try:
            def _norm_display_path(p):
                try:
                    s = str(p or '').strip()
                except Exception:
                    s = ''
                if not s:
                    return ''
                try:
                    return os.path.normpath(s)
                except Exception:
                    return s

            def _db_feed_count_and_mtime(db_path):
                """Return (count_or_None, mtime_str_or_None) for a sqlite DB file."""
                try:
                    db_path = str(db_path or '').strip()
                except Exception:
                    db_path = ''
                if not db_path:
                    return None, None
                try:
                    if not os.path.exists(db_path):
                        return None, None
                except Exception:
                    return None, None

                feed_count = None
                mtime_str = None
                try:
                    import sqlite3
                    conn = sqlite3.connect(db_path, timeout=2)
                    try:
                        cur = conn.execute('SELECT COUNT(*) FROM feeds')
                        row = cur.fetchone()
                        feed_count = int(row[0]) if row else 0
                    finally:
                        conn.close()
                except Exception:
                    feed_count = None
                try:
                    import time
                    ts = os.path.getmtime(db_path)
                    mtime_str = time.strftime('%Y-%m-%d %H:%M', time.localtime(ts))
                except Exception:
                    mtime_str = None
                return feed_count, mtime_str

            default_path = ''
            try:
                default_path = getattr(self.parent, '_db_orig_path', None) or ''
                if not default_path:
                    default_path = rss_db.db_path() or ''
            except Exception:
                default_path = ''
            default_path_n = _norm_display_path(default_path)

            try:
                d_ro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
            except Exception:
                d_ro = False
            try:
                d_mirror = bool(plugin_prefs.get('db_default_mirror', False))
            except Exception:
                d_mirror = False
            try:
                d_emoji = str(plugin_prefs.get('db_default_emoji') or '').strip()
            except Exception:
                d_emoji = ''
            try:
                d_name = str(plugin_prefs.get('db_default_name') or '').strip()
            except Exception:
                d_name = ''
            suffix = ''
            try:
                if d_ro:
                    suffix = '  (' + _('read-only') + ')'
            except Exception:
                suffix = ''
            d_count, d_mtime = _db_feed_count_and_mtime(default_path_n)
            default_line1 = (d_emoji + ' ' if d_emoji else '') + (d_name or _('Current DB')) + suffix + '  ' + _('(default profile)')
            default_line2_parts = [default_path_n or str(default_path or '')]
            if d_count is not None:
                default_line2_parts.append(f'{d_count} feeds')
            if d_mtime:
                default_line2_parts.append(f'Modified: {d_mtime}')
            default_text = default_line1 + '\n' + ' | '.join(default_line2_parts)
            if QListWidgetItem is not None:
                it = QListWidgetItem(default_text, self.list)
                it.setData(ROLE_USER, '__default__')
                try:
                    it.setToolTip(default_path_n or str(default_path or ''))
                except Exception:
                    pass
            else:
                # Fallback: add plain text item
                try:
                    self.list.addItem(default_text)
                except Exception:
                    pass

            # Dedicated entry for the built-in (plugin) DB
            try:
                config_path = str(getattr(self.parent, '_db_config_path', '') or '').strip()
            except Exception:
                config_path = ''
            config_path_n = _norm_display_path(config_path)
            try:
                if config_path:
                    # Only show when it differs from Default DB, otherwise it's redundant.
                    if str(config_path_n).strip() and str(config_path_n).strip() != str(default_path_n).strip():
                        c_count, c_mtime = _db_feed_count_and_mtime(config_path_n)
                        cfg_line1 = _('Suggested default DB (auto)')
                        cfg_line2_parts = [config_path_n]
                        if c_count is not None:
                            cfg_line2_parts.append(f'{c_count} feeds')
                        if c_mtime:
                            cfg_line2_parts.append(f'Modified: {c_mtime}')
                        cfg_text = cfg_line1 + '\n' + ' | '.join(cfg_line2_parts)
                    else:
                        cfg_text = ''
                    if QListWidgetItem is not None:
                        if cfg_text:
                            it = QListWidgetItem(cfg_text, self.list)
                            it.setData(ROLE_USER, '__config__')
                            # Visual distinction: dim + italic since it's an auto suggestion.
                            try:
                                f = QFont(it.font())
                                f.setItalic(True)
                                it.setFont(f)
                            except Exception:
                                pass
                            try:
                                it.setToolTip(_('An automatic suggested default location. You can Switch to it or Set Default.'))
                            except Exception:
                                pass
                            # Add a visual separator after the suggested default row
                            try:
                                sep = QListWidgetItem('────────', self.list)
                                try:
                                    sep.setFlags(Qt.ItemFlag.ItemIsEnabled)
                                except Exception:
                                    try:
                                        sep.setFlags(Qt.ItemIsEnabled)
                                    except Exception:
                                        pass
                                try:
                                    sep.setData(ROLE_USER, '__separator__')
                                except Exception:
                                    try:
                                        sep.setData(Qt.UserRole, '__separator__')
                                    except Exception:
                                        pass
                            except Exception:
                                pass
                    else:
                        try:
                            if cfg_text:
                                self.list.addItem(cfg_text)
                        except Exception:
                            pass
            except Exception:
                pass

            profiles = []
            try:
                profiles = plugin_prefs.get('db_profiles') or []
            except Exception as e:
                # Surface the error so we can see why profiles failed to load
                try:
                    error_dialog(self, _('RSS Reader Error'), _('Failed to read saved profiles: %s') % str(e), show=True)
                except Exception:
                    pass
                profiles = []

            # If user switched DB one-off (not saved as a profile), show it explicitly
            try:
                current_path = str(rss_db.db_path() or '').strip()
            except Exception:
                current_path = ''
            current_path_n = _norm_display_path(current_path)
            try:
                prof_paths = set(_norm_display_path((x or {}).get('path') or '') for x in profiles)
            except Exception:
                prof_paths = set()
            try:
                # Only show one-off current DB if it isn't already represented by:
                # - configured Current DB, or
                # - the auto Suggested default DB row, or
                # - a saved profile.
                if (
                    current_path
                    and current_path_n != str(default_path_n or '').strip()
                    and (not config_path_n or current_path_n != str(config_path_n or '').strip())
                    and current_path_n not in prof_paths
                ):
                    cur_count, cur_mtime = _db_feed_count_and_mtime(current_path_n)
                    cur_line1 = _('Current DB (not saved)')
                    cur_line2_parts = [current_path_n]
                    if cur_count is not None:
                        cur_line2_parts.append(f'{cur_count} feeds')
                    if cur_mtime:
                        cur_line2_parts.append(f'Modified: {cur_mtime}')
                    cur_text = cur_line1 + '\n' + ' | '.join(cur_line2_parts)
                    if QListWidgetItem is not None:
                        it = QListWidgetItem(cur_text, self.list)
                        it.setData(ROLE_USER, '__current__')
                        try:
                            f = QFont(it.font())
                            f.setItalic(True)
                            it.setFont(f)
                        except Exception:
                            pass
                        try:
                            it.setToolTip(_('This database was opened temporarily (not saved as a profile). Use "Add" to save it as a profile or "Set Default" to make it the default.'))
                        except Exception:
                            pass
                    else:
                        try:
                            self.list.addItem(cur_text)
                        except Exception:
                            pass
            except Exception:
                pass

            # Determine default "mode" so we can hide redundant entries that match it.
            try:
                default_mode = 'mirror' if d_mirror else ('read-only' if d_ro else 'writable')
            except Exception:
                default_mode = 'read-only' if d_ro else 'writable'

            try:
                default_emoji = str(plugin_prefs.get('db_profile_default_emoji') or '📰').strip()
            except Exception:
                default_emoji = '📰'

            for p in profiles:
                try:
                    mode = ''
                    try:
                        if bool(p.get('mirror', False)):
                            mode = '  (' + _('mirror') + ')'
                        elif bool(p.get('readonly', False)):
                            mode = '  (' + _('read-only') + ')'
                    except Exception:
                        mode = ''
                    db_path = str(p.get('path') or '')
                    db_path_n = _norm_display_path(db_path)

                    # Hide redundant saved profile entries that are literally the Default DB + same mode
                    try:
                        p_mode = 'mirror' if bool(p.get('mirror', False)) else ('read-only' if bool(p.get('readonly', False)) else 'writable')
                    except Exception:
                        p_mode = 'read-only' if bool(p.get('readonly', False)) else 'writable'
                    try:
                        if db_path_n and default_path_n and db_path_n == default_path_n and p_mode == default_mode:
                            continue
                    except Exception:
                        pass

                    feed_count, mtime_str = _db_feed_count_and_mtime(db_path_n)

                    try:
                        emoji = str(p.get('emoji') or '').strip()
                    except Exception:
                        emoji = ''
                    if not emoji:
                        emoji = default_emoji

                    line1 = (emoji + ' ' if emoji else '') + (p.get('name') or p.get('path') or '') + mode
                    line2_parts = [db_path_n or db_path]
                    if feed_count is not None:
                        line2_parts.append(f'{feed_count} feeds')
                    if mtime_str:
                        line2_parts.append(f'Modified: {mtime_str}')
                    line2 = ' | '.join(line2_parts)
                    text = line1 + '\n' + line2
                except Exception:
                    try:
                        text = str(p.get('path') or '')
                    except Exception:
                        text = _('(invalid profile)')
                try:
                    it = QListWidgetItem(text, self.list)
                    it.setData(ROLE_USER, str(p.get('id') or ''))
                    try:
                        it.setToolTip(db_path_n or db_path)
                    except Exception:
                        pass
                except Exception:
                    # If creating the list item fails, keep going
                    continue
        except Exception as e:
            # Global failure: show a dialog so user and developer can see the problem
            try:
                error_dialog(self, _('RSS Reader Error'), _('Failed to populate profiles list: %s') % str(e), show=True, det_msg=traceback.format_exc())
            except Exception:
                pass
            return
        finally:
            try:
                self._update_buttons()
            except Exception:
                pass

    def _update_buttons(self):
        try:
            it = self.list.currentItem()
            if it is None:
                self.edit_btn.setEnabled(False)
                self.remove_btn.setEnabled(False)
                self.switch_btn.setEnabled(False)
                self.set_default_btn.setEnabled(False)
                try:
                    self.clear_btn.setEnabled(False)
                except Exception:
                    pass
                return
            pid = str(it.data(ROLE_USER) or '')

            # Non-actionable separator row
            if pid == '__separator__':
                try:
                    self.edit_btn.setEnabled(False)
                    self.remove_btn.setEnabled(False)
                    self.switch_btn.setEnabled(False)
                    self.set_default_btn.setEnabled(False)
                    try:
                        self.clear_btn.setEnabled(False)
                    except Exception:
                        pass
                except Exception:
                    pass
                return

            # Edit: allow editing the configured Current DB row so users can change
            # its path/emoji without forcing them to create a profile first.
            self.edit_btn.setEnabled(pid == '__default__' or pid not in ('__default__', '__current__', '__config__'))

            # Remove: only for real profiles
            self.remove_btn.setEnabled(pid not in ('__default__', '__current__', '__config__'))

            # Clear: allowed when we can resolve a real, writable DB file
            can_clear = False
            try:
                db_path = str(self._db_path_for_pid(pid) or '').strip()
            except Exception:
                db_path = ''
            if db_path and os.path.exists(db_path):
                # Disallow if the selected entry is read-only/mirror.
                ro = False
                try:
                    if pid == '__default__':
                        ro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
                    elif pid == '__current__':
                        ro = bool(getattr(rss_db, 'DB_READONLY', False))
                    elif pid == '__config__':
                        ro = False
                    else:
                        profiles = plugin_prefs.get('db_profiles') or []
                        for p in profiles:
                            if str(p.get('id') or '') == pid:
                                ro = bool(p.get('readonly', False) or p.get('mirror', False))
                                break
                except Exception:
                    ro = False
                can_clear = (not ro)
            try:
                self.clear_btn.setEnabled(bool(can_clear))
                if not can_clear:
                    self.clear_btn.setToolTip(_('Not available for read-only/mirror or missing DB files.'))
                else:
                    self.clear_btn.setToolTip(_('Delete ALL data from the selected database (feeds, cache, tags, history). Keeps the file/path.'))
            except Exception:
                pass

            # Switch: allowed for default + suggested + real profiles
            self.switch_btn.setEnabled(pid != '__current__')

            # Set Default: allowed for everything except the Current DB row itself
            self.set_default_btn.setEnabled(pid != '__default__')
            try:
                self.set_default_btn.setToolTip(_('Set the selected row as the startup default — the database that opens automatically when RSS Reader launches. (Ctrl+D)'))
            except Exception:
                pass
        except Exception:
            pass

    def _db_path_for_pid(self, pid):
        pid = str(pid or '')
        db_path = ''
        if pid == '__default__':
            try:
                db_path = str(getattr(self.parent, '_db_orig_path', None) or '')
                if not db_path:
                    db_path = str(rss_db.db_path() or '')
            except Exception:
                db_path = ''
        elif pid == '__config__':
            try:
                db_path = str(getattr(self.parent, '_db_config_path', '') or '').strip()
            except Exception:
                db_path = ''
        elif pid == '__current__':
            try:
                db_path = str(rss_db.db_path() or '').strip()
            except Exception:
                db_path = ''
        else:
            try:
                profiles = plugin_prefs.get('db_profiles') or []
            except Exception:
                profiles = []
            for p in profiles:
                try:
                    if str(p.get('id') or '') == pid:
                        db_path = str(p.get('path') or '').strip()
                        break
                except Exception:
                    continue
        try:
            return os.path.normpath(db_path) if db_path else ''
        except Exception:
            return db_path

    def clear_database(self):
        try:
            it = self.list.currentItem()
            if it is None:
                return
            pid = str(it.data(ROLE_USER) or '')
        except Exception:
            return

        try:
            db_path = str(self._db_path_for_pid(pid) or '').strip()
        except Exception:
            db_path = ''
        if not db_path or not os.path.exists(db_path):
            try:
                QMessageBox.warning(self, _('File not found'), _('Database file not found: %s') % db_path)
            except Exception:
                pass
            return

        # Safety: do not allow clearing read-only/mirror entries
        try:
            ro = False
            if pid == '__default__':
                ro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
            elif pid == '__current__':
                ro = bool(getattr(rss_db, 'DB_READONLY', False))
            elif pid not in ('__config__', '__default__', '__current__'):
                profiles = plugin_prefs.get('db_profiles') or []
                for p in profiles:
                    if str(p.get('id') or '') == pid:
                        ro = bool(p.get('readonly', False) or p.get('mirror', False))
                        break
            if ro:
                QMessageBox.warning(self, _('Read-only'), _('This database/profile is configured as read-only or mirror. Switch to writable mode before clearing.'))
                return
        except Exception:
            pass

        msg = (
            _('This will DELETE ALL feeds, cached items, tags, stars, and history\nfrom the selected database.\n\nContinue?')
            + '\n\n' + _('Database: %s') % db_path
        )
        try:
            res = QMessageBox.question(self, _('Clear database'), msg, QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No, QMessageBox.StandardButton.No)
        except Exception:
            return
        if res != QMessageBox.StandardButton.Yes:
            return

        try:
            rss_db.clear_all_data_at_path(db_path, vacuum_after=True)
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Error'), _('Failed to clear database: %s') % str(e))
            except Exception:
                pass
            return

        try:
            QMessageBox.information(self, _('Clear database'), _('Database cleared.'))
        except Exception:
            pass

        # If the cleared DB is currently active, reload UI caches and notify parent/main UI.
        try:
            if str(rss_db.db_path() or '').strip() == str(db_path or '').strip():
                try:
                    if hasattr(self.parent, 'load_feeds'):
                        self.parent.load_feeds()
                except Exception:
                    pass
                try:
                    if hasattr(self.parent, 'refresh'):
                        self.parent.refresh()
                except Exception:
                    pass
                # Also emit a signal or call a method to notify main UI if needed
                try:
                    if hasattr(self.parent, 'on_db_cleared'):
                        self.parent.on_db_cleared()
                except Exception:
                    pass
        except Exception:
            pass

    def add_profile(self):
        try:
            # Add existing DB profile: do NOT prefill a path.
            dlg = _AddEditProfileDialog(self, initial={'path': '', 'name': 'RSS_Reader_added', 'readonly': False, 'mirror': False}, mode='add')
            # Hide create-blank in Add flow (user is adding an existing file)
            try:
                if getattr(dlg, 'create_blank_cb', None) is not None:
                    dlg.create_blank_cb.setChecked(False)
                    dlg.create_blank_cb.setEnabled(False)
                    dlg.create_blank_cb.setVisible(False)
            except Exception:
                pass
            if dlg.exec() != QDialog.DialogCode.Accepted:
                return
            v = dlg.get_values()
            if not v.get('path'):
                QMessageBox.warning(self, _('Error'), _('Please choose a database file path.'))
                return
            try:
                rss_db.assert_db_path_allowed(v.get('path'))
            except Exception as e:
                QMessageBox.warning(self, _('Invalid DB location'), str(e))
                return
            # Require existing file for Add
            try:
                if not os.path.exists(v.get('path')):
                    QMessageBox.warning(self, _('File not found'), _('Database file does not exist at the selected path. Use "New" to create a blank DB.'))
                    return
            except Exception:
                pass
            profiles = plugin_prefs.get('db_profiles') or []
            entered_name = v.get('name') or ''
            if any((p.get('name') or '').strip().lower() == entered_name.strip().lower() for p in profiles):
                QMessageBox.warning(self, _('Duplicate Profile Name'), _('A profile with name "%s" already exists. Please choose a unique name.') % entered_name)
                return
            pid = str(uuid.uuid4())
            prof = {
                'id': pid,
                'name': entered_name or pid,
                'emoji': v.get('emoji') or '',
                'path': v.get('path'),
                'readonly': v.get('readonly', False),
                'mirror': v.get('mirror', False),
            }
            profiles.append(prof)
            plugin_prefs['db_profiles'] = profiles

            # Offer to switch immediately to reduce clicks.
            try:
                msg = _('Profile created: %s\n\nSwitch to it now?') % (prof.get('name') or prof.get('path') or '')
            except Exception:
                msg = 'Switch to it now?'
            try:
                if QMessageBox.question(self, _('Switch now?'), msg) == QMessageBox.StandardButton.Yes:
                    try:
                        self.parent._set_active_profile(pid, persist=True)
                    except Exception:
                        pass
            except Exception:
                pass
            self._populate()
            try:
                self.parent._rebuild_profiles_menu()
            except Exception:
                pass
        except Exception:
            pass

    def new_profile(self):
        try:
            # Prefill name from current DB base; suggest a path for the new DB
            try:
                current_path = str(rss_db.db_path() or '').strip()
            except Exception:
                current_path = ''
            try:
                base = 'RSS_Reader_new'
            except Exception:
                base = 'RSS_Reader_new'

            # Prefill path: use last used dir, or default/suggested
            suggested_path = ''
            try:
                import os
                # Try last used DB dir
                last_path = current_path or ''
                last_dir = os.path.dirname(last_path) if last_path else ''
                if last_dir and os.path.isdir(last_dir):
                    suggested_path = os.path.join(last_dir, base + '.sqlite')
                else:
                    # Fallback: use user's home or a standard location
                    import getpass
                    home = os.path.expanduser('~')
                    suggested_path = os.path.join(home, base + '.sqlite')
            except Exception:
                suggested_path = ''
            # Normalize path separators for the current OS (fixes mixed slashes on Windows)
            try:
                if suggested_path:
                    suggested_path = os.path.normpath(suggested_path)
            except Exception:
                pass

            dlg = _AddEditProfileDialog(self, initial={'path': suggested_path, 'name': base, 'readonly': False, 'mirror': False}, mode='new')
            # Force create-blank checked/disabled for New flow
            try:
                if getattr(dlg, 'create_blank_cb', None) is not None:
                    dlg.create_blank_cb.setChecked(True)
                    dlg.create_blank_cb.setEnabled(False)
            except Exception:
                pass
            if dlg.exec() != QDialog.DialogCode.Accepted:
                return
            v = dlg.get_values()
            if not v.get('path'):
                QMessageBox.warning(self, _('Error'), _('Please choose a database file path.'))
                return
            try:
                rss_db.assert_db_path_allowed(v.get('path'))
            except Exception as e:
                QMessageBox.warning(self, _('Invalid DB location'), str(e))
                return
            profiles = plugin_prefs.get('db_profiles') or []
            entered_name = v.get('name') or ''
            if any((p.get('name') or '').strip().lower() == entered_name.strip().lower() for p in profiles):
                QMessageBox.warning(self, _('Duplicate Profile Name'), _('A profile with name "%s" already exists. Please choose a unique name.') % entered_name)
                return
            pid = str(uuid.uuid4())
            prof = {
                'id': pid,
                'name': entered_name or pid,
                'emoji': v.get('emoji') or '',
                'path': v.get('path'),
                'readonly': False,
                'mirror': False,
            }
            profiles.append(prof)
            plugin_prefs['db_profiles'] = profiles

            # Create the blank DB file now, without permanently switching
            # the live session's DB.
            try:
                pdir = os.path.dirname(prof['path'])
                os.makedirs(pdir, exist_ok=True)
                try:
                    from contextlib import contextmanager as _cm_dummy  # noqa: F401
                except Exception:
                    pass
                try:
                    with rss_db.temporary_db_path(prof['path'], readonly=False):
                        rss_db.init_db()
                except Exception:
                    pass
            except Exception:
                pass

            # Offer to switch immediately
            try:
                msg = _('Profile created: %s\n\nSwitch to it now?') % (prof.get('name') or prof.get('path') or '')
            except Exception:
                msg = 'Switch to it now?'
            try:
                if QMessageBox.question(self, _('Switch now?'), msg) == QMessageBox.StandardButton.Yes:
                    try:
                        self.parent._set_active_profile(pid, persist=True)
                    except Exception:
                        pass
            except Exception:
                pass
            self._populate()
            try:
                self.parent._rebuild_profiles_menu()
            except Exception:
                pass
        except Exception:
            pass

    def edit_profile(self):
        try:
            it = self.list.currentItem()
            if it is None:
                return
            pid = str(it.data(ROLE_USER) or '')

            # Special case: edit configured Current DB settings (path + emoji)
            if pid == '__default__':
                try:
                    self._edit_current_db_settings()
                except Exception:
                    pass
                return

            if pid in ('__current__', '__config__'):
                return

            profiles = plugin_prefs.get('db_profiles') or []
            idx = next((i for i, x in enumerate(profiles) if str(x.get('id') or '') == pid), None)
            if idx is None:
                return
            p = profiles[idx]
            dlg = _AddEditProfileDialog(self, profile=p)
            if dlg.exec() != QDialog.DialogCode.Accepted:
                return
            v = dlg.get_values()
            p['name'] = v.get('name') or p.get('name') or ''
            p['emoji'] = v.get('emoji') or ''
            newpath = v.get('path') or p.get('path')
            try:
                rss_db.assert_db_path_allowed(newpath)
            except Exception as e:
                QMessageBox.warning(self, _('Invalid DB location'), str(e))
                return
            p['path'] = newpath
            p['readonly'] = v.get('readonly', False)
            p['mirror'] = v.get('mirror', False)
            plugin_prefs['db_profiles'] = profiles
            # Editing no longer creates blank DBs here; use New to create.
            self._populate()
            try:
                self.parent._rebuild_profiles_menu()
            except Exception:
                pass
        except Exception:
            pass

    def _edit_current_db_settings(self):
        """Edit the configured 'Current DB' (default path + emoji) in-place."""
        try:
            from qt.core import QDialog
        except Exception:
            from PyQt5.Qt import QDialog

        try:
            cur_path = str(getattr(self.parent, '_db_orig_path', '') or '').strip()
        except Exception:
            cur_path = ''
        if not cur_path:
            try:
                cur_path = str(rss_db.db_path() or '').strip()
            except Exception:
                cur_path = ''

        try:
            cur_ro = bool(plugin_prefs.get('db_default_readonly', False))
        except Exception:
            cur_ro = False
        try:
            cur_m = bool(plugin_prefs.get('db_default_mirror', False))
        except Exception:
            cur_m = False
        try:
            cur_emoji = str(plugin_prefs.get('db_default_emoji') or '').strip()
        except Exception:
            cur_emoji = ''
        try:
            cur_name = str(plugin_prefs.get('db_default_name') or '').strip()
        except Exception:
            cur_name = ''

        pseudo = {
            'name': cur_name or _('Current DB'),
            'emoji': cur_emoji,
            'path': cur_path,
            'readonly': cur_ro,
            'mirror': cur_m,
        }
        dlg = _AddEditProfileDialog(self, profile=pseudo)
        dlg.setWindowTitle(_('Current DB settings'))
        if dlg.exec() != QDialog.DialogCode.Accepted:
            return

        v = dlg.get_values() or {}
        newpath = str(v.get('path') or '').strip()
        if not newpath:
            try:
                QMessageBox.warning(self, _('Error'), _('Please choose a database file path.'))
            except Exception:
                pass
            return
        try:
            rss_db.assert_db_path_allowed(newpath)
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Invalid DB location'), str(e))
            except Exception:
                pass
            return

        # Creating blank DBs is handled via New; Current DB edit does not create files.

        try:
            plugin_prefs['db_default_path'] = str(newpath)
            plugin_prefs['db_default_readonly'] = bool(v.get('readonly', False))
            plugin_prefs['db_default_mirror'] = bool(v.get('mirror', False))
            plugin_prefs['db_default_emoji'] = str(v.get('emoji') or '').strip()
            plugin_prefs['db_default_name'] = str(v.get('name') or '').strip()
            plugin_prefs['db_onboarded'] = True
        except Exception:
            pass

        # Update parent cached default and, if currently on default, switch immediately
        try:
            self.parent._db_orig_path = str(newpath)
            self.parent._db_default_readonly = bool(v.get('readonly', False) or v.get('mirror', False))
        except Exception:
            pass
        try:
            active_id = str(plugin_prefs.get('db_profiles_active') or '')
        except Exception:
            active_id = ''
        try:
            if not active_id:
                self.parent._switch_database(str(newpath), readonly=bool(v.get('readonly', False) or v.get('mirror', False)))
        except Exception:
            pass
        try:
            self.parent._rebuild_profiles_menu()
        except Exception:
            pass
        try:
            self.parent._update_profile_label()
        except Exception:
            pass
        try:
            self._populate()
        except Exception:
            pass

    def remove_profile(self):
        try:
            it = self.list.currentItem()
            if it is None:
                return
            pid = str(it.data(ROLE_USER) or '')
            profiles = plugin_prefs.get('db_profiles') or []
            idx = next((i for i, x in enumerate(profiles) if str(x.get('id') or '') == pid), None)
            if idx is None:
                return
            p = profiles[idx]
            if QMessageBox.question(self, _('Confirm remove'), _('Remove profile "%s"?') % (p.get('name') or p.get('path') or '')) != QMessageBox.StandardButton.Yes:
                return
            profiles = [x for x in profiles if str(x.get('id') or '') != pid]
            plugin_prefs['db_profiles'] = profiles
            self._populate()
            try:
                self.parent._rebuild_profiles_menu()
            except Exception:
                pass
        except Exception:
            pass

    def set_default_profile(self):
        try:
            it = self.list.currentItem()
            if it is None:
                return
            pid = str(it.data(ROLE_USER) or '')
            # Set Default defines what "Default DB" means (path + readonly flags)
            if pid == '__default__':
                return

            path = ''
            readonly = False
            mirror = False
            name = ''
            emoji = ''

            if pid == '__current__':
                try:
                    path = str(rss_db.db_path() or '').strip()
                except Exception:
                    path = ''
                try:
                    readonly = bool(getattr(rss_db, 'DB_READONLY', False))
                except Exception:
                    readonly = False
                try:
                    mirror = False
                except Exception:
                    mirror = False
                # Derive a sensible label from the filename.
                try:
                    base = os.path.splitext(os.path.basename(path))[0] if path else ''
                except Exception:
                    base = ''
                name = str(base or '').strip()
            elif pid == '__config__':
                try:
                    path = str(getattr(self.parent, '_db_config_path', '') or '').strip()
                except Exception:
                    path = ''
                readonly = False
                mirror = False
                try:
                    base = os.path.splitext(os.path.basename(path))[0] if path else ''
                except Exception:
                    base = ''
                name = str(base or '').strip()
            else:
                # Selected a named profile: load its values
                try:
                    profiles = plugin_prefs.get('db_profiles') or []
                except Exception:
                    profiles = []
                try:
                    idx = next((i for i, x in enumerate(profiles) if str(x.get('id') or '') == pid), None)
                except Exception:
                    idx = None
                if idx is None:
                    return
                try:
                    p = profiles[idx]
                except Exception:
                    return
                try:
                    path = str(p.get('path') or '').strip()
                except Exception:
                    path = ''
                try:
                    readonly = bool(p.get('readonly', False))
                except Exception:
                    readonly = False
                try:
                    mirror = bool(p.get('mirror', False))
                except Exception:
                    mirror = False
                try:
                    name = str(p.get('name') or '').strip()
                except Exception:
                    name = ''
                try:
                    emoji = str(p.get('emoji') or '').strip()
                except Exception:
                    emoji = ''

            if not path:
                return

            try:
                rss_db.assert_db_path_allowed(path)
            except Exception as e:
                try:
                    QMessageBox.warning(self, _('Invalid DB location'), str(e))
                except Exception:
                    pass
                return

            try:
                plugin_prefs['db_default_path'] = str(path)
                plugin_prefs['db_default_readonly'] = bool(readonly)
                plugin_prefs['db_default_mirror'] = bool(mirror)
                if name:
                    plugin_prefs['db_default_name'] = str(name)
                if emoji:
                    plugin_prefs['db_default_emoji'] = str(emoji)
            except Exception:
                pass

            # Update parent cached default and, if currently on "default", switch immediately
            try:
                self.parent._db_orig_path = str(path)
                self.parent._db_default_readonly = bool(readonly)
            except Exception:
                pass
            try:
                active_id = str(plugin_prefs.get('db_profiles_active') or '')
            except Exception:
                active_id = ''
            try:
                if not active_id:
                    self.parent._switch_database(str(path), readonly=bool(readonly))
            except Exception:
                pass
            try:
                self.parent._rebuild_profiles_menu()
            except Exception:
                pass
            try:
                self.parent._update_profile_label()
            except Exception:
                pass

            self._populate()
        except Exception:
            pass

    def switch_profile(self):
        try:
            it = self.list.currentItem()
            if it is None:
                return
            pid = str(it.data(ROLE_USER) or '')
            if pid == '__separator__':
                return
            if pid == '__default__':
                try:
                    self.parent._restore_default_database()
                except Exception:
                    pass
            elif pid == '__config__':
                try:
                    self.parent._switch_to_config_database()
                except Exception:
                    pass
            elif pid == '__current__':
                pass
            else:
                self.parent._set_active_profile(pid)
            self.accept()
        except Exception:
            pass


class _OpmlFeedPickerDialog(QDialog):
    """Simple feed list picker with checkboxes."""

    def __init__(self, parent, title, entries, prechecked_urls=None, disabled_urls=None, help_text=''):
        QDialog.__init__(self, parent)
        self.setWindowTitle(str(title or _('OPML')))
        self._entries = list(entries or [])
        self._prechecked = set(str(x or '').strip() for x in (prechecked_urls or []) if str(x or '').strip())
        self._disabled = set(str(x or '').strip() for x in (disabled_urls or []) if str(x or '').strip())

        layout = QVBoxLayout(self)
        if help_text:
            layout.addWidget(QLabel(str(help_text), self))

        # Filter
        self.filter_input = QLineEdit(self)
        self.filter_input.setPlaceholderText(_('Filter (title / folder / url)...'))
        try:
            self.filter_input.setClearButtonEnabled(True)
        except Exception:
            pass
        layout.addWidget(self.filter_input)

        self.table = QTableWidget(self)
        self.table.setColumnCount(4)
        self.table.setHorizontalHeaderLabels([_('Import'), _('Title'), _('Folder'), _('URL')])
        self.table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.table.setSelectionMode(QAbstractItemView.SelectionMode.ExtendedSelection)
        self.table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)
        try:
            self.table.setSortingEnabled(True)
        except Exception:
            pass
        layout.addWidget(self.table, 1)

        # Buttons
        btn_row = QHBoxLayout()
        self._btn_row = btn_row
        self.btn_all = QPushButton(_('Select all'), self)
        self.btn_none = QPushButton(_('Select none'), self)
        btn_row.addWidget(self.btn_all)
        btn_row.addWidget(self.btn_none)
        btn_row.addStretch(1)
        layout.addLayout(btn_row)

        self.btn_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, self)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        layout.addWidget(self.btn_box)

        self._populate()

        try:
            self.filter_input.textChanged.connect(self._apply_filter)
        except Exception:
            pass
        self.btn_all.clicked.connect(lambda: self._set_all_checked(True))
        self.btn_none.clicked.connect(lambda: self._set_all_checked(False))

        try:
            self.resize(980, 560)
        except Exception:
            pass

    def _populate(self):
        self.table.setRowCount(len(self._entries))
        for r, e in enumerate(self._entries):
            title = str(e.get('title') or '')
            folder = str(e.get('folder') or '')
            url = str(e.get('url') or '')

            chk = QTableWidgetItem('')
            try:
                flags = chk.flags()
                try:
                    chk.setFlags(flags | Qt.ItemFlag.ItemIsUserCheckable)
                except Exception:
                    chk.setFlags(flags | Qt.ItemIsUserCheckable)
            except Exception:
                pass

            disabled = url in self._disabled
            checked = (url in self._prechecked) if self._prechecked else (not disabled)
            try:
                chk.setCheckState(Qt.CheckState.Checked if checked and not disabled else Qt.CheckState.Unchecked)
            except Exception:
                try:
                    chk.setCheckState(Qt.Checked if checked and not disabled else Qt.Unchecked)
                except Exception:
                    pass

            if disabled:
                try:
                    chk.setFlags(chk.flags() & ~Qt.ItemFlag.ItemIsEnabled)
                except Exception:
                    try:
                        chk.setFlags(chk.flags() & ~Qt.ItemIsEnabled)
                    except Exception:
                        pass
                try:
                    chk.setToolTip(_('Already in your feed list'))
                except Exception:
                    pass

            t_title = QTableWidgetItem(title)
            t_folder = QTableWidgetItem(folder)
            t_url = QTableWidgetItem(url)

            # Store entry on all cells so selection works even when sorting
            try:
                for cell in (chk, t_title, t_folder, t_url):
                    cell.setData(ROLE_USER, dict(e))
            except Exception:
                pass

            self.table.setItem(r, 0, chk)
            self.table.setItem(r, 1, t_title)
            self.table.setItem(r, 2, t_folder)
            self.table.setItem(r, 3, t_url)

        try:
            self.table.resizeColumnsToContents()
            self.table.setColumnWidth(0, 72)
            self.table.setColumnWidth(1, max(240, self.table.columnWidth(1)))
            self.table.setColumnWidth(2, max(150, self.table.columnWidth(2)))
            self.table.setColumnWidth(3, max(320, self.table.columnWidth(3)))
        except Exception:
            pass

        self._apply_filter(self.filter_input.text() if getattr(self, 'filter_input', None) is not None else '')

    def _apply_filter(self, text):
        q = (text or '').strip().casefold()
        for r in range(self.table.rowCount()):
            try:
                t_title = self.table.item(r, 1).text() if self.table.item(r, 1) is not None else ''
                t_folder = self.table.item(r, 2).text() if self.table.item(r, 2) is not None else ''
                t_url = self.table.item(r, 3).text() if self.table.item(r, 3) is not None else ''
            except Exception:
                t_title = t_folder = t_url = ''
            hay = (t_title + ' ' + t_folder + ' ' + t_url).casefold()
            show = True if not q else (q in hay)
            try:
                self.table.setRowHidden(r, not show)
            except Exception:
                pass

    def _set_all_checked(self, checked):
        for r in range(self.table.rowCount()):
            try:
                if self.table.isRowHidden(r):
                    continue
                it = self.table.item(r, 0)
                if it is None:
                    continue
                # Skip disabled rows
                try:
                    if not bool(it.flags() & Qt.ItemFlag.ItemIsEnabled):
                        continue
                except Exception:
                    try:
                        if not bool(it.flags() & Qt.ItemIsEnabled):
                            continue
                    except Exception:
                        pass
                try:
                    it.setCheckState(Qt.CheckState.Checked if checked else Qt.CheckState.Unchecked)
                except Exception:
                    it.setCheckState(Qt.Checked if checked else Qt.Unchecked)
            except Exception:
                pass

    def selected_entries(self):
        selected = []
        for r in range(self.table.rowCount()):
            try:
                chk = self.table.item(r, 0)
                if chk is None:
                    continue
                try:
                    is_checked = chk.checkState() == Qt.CheckState.Checked
                except Exception:
                    is_checked = chk.checkState() == Qt.Checked
                if not is_checked:
                    continue
                data = chk.data(ROLE_USER)
                if isinstance(data, dict) and data.get('url'):
                    selected.append(dict(data))
            except Exception:
                pass
        # De-dup again (sorting can duplicate selection data reads in weird Qt edge cases)
        seen = set()
        out = []
        for e in selected:
            u = str(e.get('url') or '').strip()
            if not u or u in seen:
                continue
            seen.add(u)
            out.append(e)
        return out


class ShareViaEmailDialog(QDialog):
    def __init__(self, parent=None, article=None):
        super().__init__(parent)
        self.article = article if isinstance(article, dict) else {}
        self.title = str(self.article.get('title') or '')
        self.link = str(self.article.get('link') or '')
        self.setWindowTitle(_('Share via email'))
        self.setModal(True)
        
        layout = QVBoxLayout(self)
        
        # Email address selection
        email_layout = QHBoxLayout()
        email_layout.addWidget(QLabel(_('Send to:')))
        self.email_combo = QComboBox(self)
        self.load_email_addresses()
        email_layout.addWidget(self.email_combo)
        layout.addLayout(email_layout)
        
        # Subject
        subject_layout = QHBoxLayout()
        subject_layout.addWidget(QLabel(_('Subject:')))
        self.subject_edit = QLineEdit(self)
        self.subject_edit.setText(self.title or _('Shared article'))
        subject_layout.addWidget(self.subject_edit)
        layout.addLayout(subject_layout)
        
        # Message
        layout.addWidget(QLabel(_('Message:')))
        self.message_edit = QPlainTextEdit(self)
        footer_note = _('Delivered via RSS Reader plugin for calibre.')
        footer_block = f"\n\n--\n{footer_note}" if footer_note else ''

        message_text = self.link or ''
        if self.title:
            message_text = f"{self.title}\n\n{self.link}" if self.link else self.title
        if footer_block:
            message_text = (message_text or '') + footer_block
        self.message_edit.setPlainText(message_text)
        layout.addWidget(self.message_edit)

        # Attachment option (default ON)
        self.attach_checkbox = QCheckBox(_('Attach exported file (recommended)'))
        try:
            self.attach_checkbox.setChecked(True)
        except Exception:
            pass
        layout.addWidget(self.attach_checkbox)
        
        # Job queue option
        self.job_checkbox = QCheckBox(_('Show in job queue'))
        layout.addWidget(self.job_checkbox)

        try:
            self.attach_checkbox.toggled.connect(self._sync_options_visibility)
        except Exception:
            pass
        self._sync_options_visibility()
        
        # Buttons
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel)
        button_box.accepted.connect(self.send_email)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.resize(500, 200)

    def _sync_options_visibility(self, *_args):
        try:
            attaching = bool(self.attach_checkbox.isChecked())
        except Exception:
            attaching = True
        try:
            self.job_checkbox.setVisible(not attaching)
        except Exception:
            pass
        try:
            self.job_checkbox.setEnabled(not attaching)
        except Exception:
            pass
    
    def load_email_addresses(self):
        try:
            opts = smtp_config().parse()
            accounts = opts.accounts or {}
            for email in accounts:
                self.email_combo.addItem(email, email)
            if self.email_combo.count() == 0:
                self.email_combo.addItem(_('No email addresses configured'), '')
        except Exception:
            self.email_combo.addItem(_('Error loading email addresses'), '')
    
    def send_email(self):
        to_email = self.email_combo.currentData()
        if not to_email:
            error_dialog(self, _('No email address'), 
                        _('Please configure email addresses in Calibre preferences first.'), show=True)
            return
        
        subject = self.subject_edit.text().strip()
        message = self.message_edit.toPlainText().strip()
        
        if not subject:
            subject = _('Shared article')
        if not message:
            footer_note = _('Delivered via RSS Reader plugin for calibre.')
            if self.title and self.link:
                message = f"{self.title}\n\n{self.link}"
            else:
                message = self.link or ''
            if footer_note:
                message = (message or '') + f"\n\n--\n{footer_note}"
        
        try:
            opts = smtp_config().parse()
            if not opts.from_:
                error_dialog(self, _('Email not configured'), 
                            _('Please configure your email settings in Calibre preferences (Preferences > Email).'), show=True)
                return

            gui = getattr(self.parent(), 'gui', None)
            if gui is None:
                gui = getattr(self.parent(), 'parent', None)

            # If attaching, always go through calibre jobs: convert -> send_mails
            if self.attach_checkbox.isChecked():
                if not self.article:
                    error_dialog(self, _('No article selected'), _('No article data available to export.'), show=True)
                    return
                if gui is None:
                    error_dialog(self, _('Email error'), _('GUI context unavailable.'), show=True)
                    return

                try:
                    from calibre_plugins.rss_reader.config import plugin_prefs as _plugin_prefs
                except Exception:
                    _plugin_prefs = None

                def _after_export(success, out_path, err):
                    if not success:
                        try:
                            error_dialog(self.parent(), _('Export error'), _('Failed to export article for email: %s') % str(err), show=True)
                        except Exception:
                            pass
                        return

                    try:
                        from calibre.gui2 import email as email_mod
                    except Exception as e:
                        try:
                            error_dialog(self.parent(), _('Email error'), _('Failed to load calibre email module: %s') % str(e), show=True)
                        except Exception:
                            pass
                        return

                    attachment = out_path
                    ext = os.path.splitext(attachment)[1].lstrip('.').lower() or 'epub'
                    attachment_name = _safe_filename(self.title or _('Article')) + '.' + ext

                    def _email_done(job):
                        # Clean up attachment after sending attempt
                        try:
                            if attachment and os.path.exists(attachment):
                                os.remove(attachment)
                        except Exception:
                            pass

                    try:
                        email_mod.send_mails(
                            jobnames=[self.title or _('RSS article')],
                            callback=Dispatcher(_email_done),
                            attachments=[attachment],
                            to_s=[to_email],
                            subjects=[subject],
                            texts=[message],
                            attachment_names=[attachment_name],
                            job_manager=gui.job_manager,
                        )
                        try:
                            if hasattr(gui, 'status_bar'):
                                gui.status_bar.show_message(_('Email queued'), 3000)
                        except Exception:
                            pass
                    except Exception as e:
                        try:
                            error_dialog(self.parent(), _('Email error'), _('Failed to queue email: %s') % str(e), show=True)
                        except Exception:
                            pass

                # Start conversion job and close dialog immediately
                _export_single_article_to_ebook_via_job(gui, self.article, _plugin_prefs, _after_export, parent=self.parent())
                self.accept()
                return

            # Link-only mode (optionally show in job queue)
            if self.job_checkbox.isChecked():
                try:
                    job_manager = self.parent().gui.job_manager
                    description = _('Send email: {}').format(subject)

                    def _send_link_only(from_, to, subject, message, opts, log=None, abort=None, notifications=None):
                        msg = compose_mail(from_, to, message, subject=subject)
                        return sendmail(
                            msg, from_=from_, to=[to],
                            relay=opts.relay_host,
                            username=opts.relay_username,
                            password=from_hex_unicode(opts.relay_password),
                            encryption=opts.encryption,
                            port=opts.relay_port,
                        )

                    job = ThreadedJob('email', description, _send_link_only,
                                      (opts.from_, to_email, subject, message, opts), {}, lambda job: None)
                    job_manager.run_threaded_job(job)
                    self.accept()
                except Exception as e:
                    error_dialog(self, _('Failed to queue email'),
                                _('Could not add email to job queue: {}').format(str(e)), show=True)
            else:
                msg = compose_mail(opts.from_, to_email, message, subject=subject)
                sendmail(msg, from_=opts.from_, to=[to_email],
                        relay=opts.relay_host,
                        username=opts.relay_username,
                        password=from_hex_unicode(opts.relay_password),
                        encryption=opts.encryption,
                        port=opts.relay_port)
                try:
                    gui2 = getattr(self.parent(), 'gui', None)
                    if gui2 is not None and hasattr(gui2, 'status_bar'):
                        gui2.status_bar.show_message(_('Email sent'), 2500)
                except Exception:
                    pass
                self.accept()
        except Exception as e:
            error_dialog(self, _('Failed to send email'), 
                        _('An error occurred while sending the email: {}').format(str(e)), show=True)


class SendFeedsViaEmailDialog(QDialog):
    def __init__(self, parent=None, default_subject='', default_message=''):
        super().__init__(parent)
        self.setWindowTitle(_('Send selected feeds via email'))
        self.setModal(True)

        layout = QVBoxLayout(self)

        email_layout = QHBoxLayout()
        email_layout.addWidget(QLabel(_('Send to:')))
        self.email_combo = QComboBox(self)
        self._load_email_addresses()
        email_layout.addWidget(self.email_combo)
        layout.addLayout(email_layout)

        subject_layout = QHBoxLayout()
        subject_layout.addWidget(QLabel(_('Subject:')))
        self.subject_edit = QLineEdit(self)
        self.subject_edit.setText(str(default_subject or '').strip() or _('RSS Reader export'))
        subject_layout.addWidget(self.subject_edit)
        layout.addLayout(subject_layout)

        layout.addWidget(QLabel(_('Message:')))
        self.message_edit = QPlainTextEdit(self)
        self.message_edit.setPlainText(str(default_message or '').strip() or _('See attached ebook.'))
        layout.addWidget(self.message_edit)

        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        self.resize(520, 220)

    def _load_email_addresses(self):
        try:
            opts = smtp_config().parse()
            accounts = opts.accounts or {}
            for email in accounts:
                self.email_combo.addItem(email, email)
            if self.email_combo.count() == 0:
                self.email_combo.addItem(_('No email addresses configured'), '')
        except Exception:
            self.email_combo.addItem(_('Error loading email addresses'), '')

    def get_values(self):
        to_email = self.email_combo.currentData()
        subject = self.subject_edit.text().strip()
        message = self.message_edit.toPlainText().strip()
        return to_email, subject, message
