"""Centralized debug helper for the RSS Reader plugin.

Use `from calibre_plugins.rss_reader.debug import _debug, DEBUG_RSS_READER`
to log consistently to calibre's debug console (or stderr fallback).
"""
from __future__ import absolute_import

import os
import tempfile
import time

DEBUG_RSS_READER = False

def _debug(*args):
    """Print debug message if DEBUG_RSS_READER is enabled.

    This is intentionally minimal and self-contained to avoid importing
    other plugin modules and creating circular imports.
    """
    if not DEBUG_RSS_READER:
        return
    try:
        msg = ' '.join(str(a) for a in args)
    except Exception:
        try:
            msg = str(args)
        except Exception:
            msg = '<unprintable debug>'

    try:
        from calibre import prints
        try:
            prints('[RSS Reader] ' + msg)
        except Exception:
            prints('[RSS Reader]', msg)
    except Exception:
        try:
            import sys
            try:
                sys.stderr.write('[RSS Reader] ' + msg + '\n')
                sys.stderr.flush()
            except Exception:
                pass
            try:
                print('[RSS Reader] ' + msg)
            except Exception:
                pass
        except Exception:
            pass

    # Also write to a persistent temp log file for cases where calibre console
    # output isn't easily visible.
    try:
        log_path = os.path.join(tempfile.gettempdir(), 'rss_reader_debug.log')
        ts = time.strftime('%Y-%m-%d %H:%M:%S')
        with open(log_path, 'a', encoding='utf-8', errors='replace') as f:
            f.write('[%s] %s\n' % (ts, msg))
    except Exception:
        pass
