from __future__ import absolute_import

import json
import os


def safe_lower(s, default=''):
    try:
        return str(s or '').strip().lower()
    except Exception:
        return default


REQUESTED_OUTPUT_FORMATS = (
    ('epub', 'EPUB'),
    ('azw3', 'AZW3'),
    ('mobi', 'MOBI'),
    ('pdf', 'PDF'),
    ('docx', 'DOCX'),
    ('txt', 'TXT'),
    ('md', 'Markdown'),
)


def get_calibre_default_output_format():
    # calibre has multiple preference stores; the GUI "Behaviour" preferred output
    # format is not guaranteed to be in calibre.utils.config.prefs on all versions.
    # Try to honor the prefs object used by Output Format Cycler, then fallback to
    # reading the underlying JSON file and, finally, gprefs.
    candidates = []  # tuples of (store_name, value)
    debug_keys = []
    keys = ('output_format', 'preferred_output_format')

    def _append_debug(store_name, key, value):
        debug_keys.append((store_name, key, value))

    def _collect_from_getter(store_name, getter):
        for key in keys:
            try:
                try:
                    value = getter(key, None)
                except TypeError:
                    # calibre's ConfigProxy.get() takes only 1 positional arg
                    value = getter(key)
            except Exception as exc:
                _append_debug(store_name, key, repr(exc))
                continue
            _append_debug(store_name, key, value)
            if value:
                candidates.append((store_name, value))

    def _collect_from_json(store_name, data):
        if not isinstance(data, dict):
            return
        for key in keys:
            value = data.get(key)
            _append_debug(store_name, key, value)
            if value:
                candidates.append((store_name, value))

    try:
        from calibre.utils.config import prefs as calibre_prefs
    except Exception as exc:
        _append_debug('prefs', '__import__', repr(exc))
    else:
        _collect_from_getter('prefs', calibre_prefs.get)

    config_dir = None
    try:
        from calibre.constants import config_dir as calibre_config_dir
        config_dir = calibre_config_dir
    except Exception as exc:
        _append_debug('prefs_json', '__config_dir__', repr(exc))

    if config_dir:
        json_path = os.path.join(config_dir, 'global.py.json')
        try:
            with open(json_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
        except Exception as exc:
            _append_debug('prefs_json', '__read__', repr(exc))
        else:
            _collect_from_json('prefs_json', data)

    try:
        from calibre.gui2 import gprefs
    except Exception as exc:
        _append_debug('gprefs', '__import__', repr(exc))
    else:
        _collect_from_getter('gprefs', gprefs.get)

    # DEBUG: print which keys were found
    try:
        import sys
        sys.stderr.write('[RSS Reader] calibre output format keys: %r\n' % debug_keys)
    except Exception:
        pass

    for store_name in ('prefs', 'prefs_json', 'gprefs'):
        for source, value in candidates:
            if source != store_name:
                continue
            s = safe_lower(value, default='')
            if s:
                return s

    return 'epub'


def get_plugin_output_format_pref(plugin_prefs):
    fmt = safe_lower(getattr(plugin_prefs, 'get', lambda *_a, **_k: None)('export_output_format', 'calibre_default') or 'calibre_default')
    return fmt or 'calibre_default'


def get_effective_output_format(plugin_prefs):
    fmt = get_plugin_output_format_pref(plugin_prefs)
    if fmt in ('', 'calibre_default', 'default'):
        return get_calibre_default_output_format()
    # Guard against tainted prefs (e.g. translated text saved by older code)
    _valid = {'epub', 'azw3', 'mobi', 'pdf', 'docx', 'txt', 'md', 'htmlz', 'oeb', 'lit', 'lrf', 'pdb', 'rtf', 'snb', 'tcr', 'zip'}
    if fmt not in _valid:
        return get_calibre_default_output_format()
    return fmt


def get_available_output_formats():
    try:
        from calibre.customize.ui import available_output_formats
        fmts = available_output_formats() or set()
        try:
            return {safe_lower(x) for x in fmts if x}
        except Exception:
            return set()
    except Exception:
        return set()


def resolve_conversion_output(desired_ext, available_output_formats=None):
    """Return (convert_ext, final_ext, markdown_via_txt).

    - For most formats, convert_ext == final_ext == desired_ext
    - For Markdown, calibre typically has no 'md' output plugin, so we convert to 'txt'
      with txt_output_formatting=markdown and then rename .txt -> .md.
    """
    desired_ext = safe_lower(desired_ext)
    if not desired_ext:
        desired_ext = 'epub'

    if available_output_formats is None:
        available_output_formats = set()

    if desired_ext == 'md':
        if 'md' in available_output_formats:
            return 'md', 'md', False
        return 'txt', 'md', True

    return desired_ext, desired_ext, False


def qt_save_filters(requested_only=True, include_all_files=True):
    """Return (filter_string, filter_to_ext).

    filter_to_ext maps the exact filter string returned by Qt to an extension.
    """
    filters = []
    filter_to_ext = {}

    # Keep ordering stable and user-requested.
    for ext, label in REQUESTED_OUTPUT_FORMATS:
        if requested_only:
            # Always include md even if it maps to txt later.
            pass
        flt = '%s (*.%s)' % (label, ext)
        filters.append(flt)
        filter_to_ext[flt] = ext

    if include_all_files:
        filters.append('All files (*)')

    return ';;'.join(filters), filter_to_ext
