from __future__ import absolute_import

import base64

from calibre.utils.config import JSONConfig

try:
    load_translations()
except NameError:
    pass

# Store plugin preferences in a dedicated subfolder under calibre config dir
# e.g. %APPDATA%\calibre\plugins\rss_reader\rss_reader.json
plugin_prefs = JSONConfig('plugins/rss_reader/rss_reader')


def _get_available_languages():
    """Return list of (locale_code, display_name) tuples for available translations.
    
    Hardcoded because the plugin runs from a ZIP — os.listdir() cannot
    access the translations/ folder inside the archive.
    """
    # Map locale codes to display names (native script first)
    locale_names = {
        'ar': 'العربية  (Arabic)',
        'ar_Latn': 'العربية  (Arabic – Latin)',
        'bg': 'Български  (Bulgarian)',
        'bn_BD': 'বাংলা  (Bengali – Bangladesh)',
        'bn_IN': 'বাংলা  (Bengali – India)',
        'ca': 'Català  (Catalan)',
        'cs': 'Čeština  (Czech)',
        'da': 'Dansk  (Danish)',
        'de': 'Deutsch  (German)',
        'el': 'Ελληνικά  (Greek)',
        'el_Latn': 'Ελληνικά  (Greek – Latin)',
        'en': 'English',
        'eo': 'Esperanto',
        'es': 'Español  (Spanish)',
        'et': 'Eesti  (Estonian)',
        'eu': 'Euskara  (Basque)',
        'fa': 'فارسی  (Persian)',
        'fi': 'Suomi  (Finnish)',
        'fr': 'Français  (French)',
        'gl': 'Galego  (Galician)',
        'he': 'עברית  (Hebrew)',
        'he_Latn': 'עברית  (Hebrew – Latin)',
        'hi': 'हिन्दी  (Hindi)',
        'hi_Latn': 'हिन्दी  (Hindi – Latin)',
        'hr': 'Hrvatski  (Croatian)',
        'hu': 'Magyar  (Hungarian)',
        'id': 'Bahasa Indonesia  (Indonesian)',
        'is': 'Íslenska  (Icelandic)',
        'it': 'Italiano  (Italian)',
        'ja': '日本語  (Japanese)',
        'ja_Latn': '日本語  (Japanese – Latin)',
        'ka': 'ქართული  (Georgian)',
        'kn': 'ಕನ್ನಡ  (Kannada)',
        'ko': '한국어  (Korean)',
        'ko_Latn': '한국어  (Korean – Latin)',
        'lt': 'Lietuvių  (Lithuanian)',
        'lv': 'Latviešu  (Latvian)',
        'mr': 'मराठी  (Marathi)',
        'ms': 'Bahasa Melayu  (Malay)',
        'nb': 'Norsk Bokmål  (Norwegian)',
        'nl': 'Nederlands  (Dutch)',
        'pl': 'Polski  (Polish)',
        'pt': 'Português  (Portuguese)',
        'pt_BR': 'Português do Brasil  (Brazilian Portuguese)',
        'ro': 'Română  (Romanian)',
        'ru': 'Русский  (Russian)',
        'ru_Latn': 'Русский  (Russian – Latin)',
        'sk': 'Slovenčina  (Slovak)',
        'sl': 'Slovenščina  (Slovenian)',
        'sr': 'Српски  (Serbian)',
        'sr_Latn': 'Srpski  (Serbian – Latin)',
        'sv': 'Svenska  (Swedish)',
        'ta': 'தமிழ்  (Tamil)',
        'tr': 'Türkçe  (Turkish)',
        'uk': 'Українська  (Ukrainian)',
        'vi': 'Tiếng Việt  (Vietnamese)',
        'zh_CN': '简体中文  (Chinese Simplified)',
        'zh_CN_Latn': '简体中文  (Chinese Simplified – Latin)',
        'zh_HK': '繁體中文  (Chinese – Hong Kong)',
        'zh_HK_Latn': '繁體中文  (Chinese – Hong Kong – Latin)',
        'zh_TW': '繁體中文  (Chinese – Taiwan)',
        'zh_TW_Latn': '繁體中文  (Chinese – Taiwan – Latin)',
    }

    result = [('', 'Auto  (use Calibre language)')]
    for locale in sorted(locale_names.keys()):
        result.append((locale, locale_names[locale]))
    return result


# Defaults
plugin_prefs.defaults['notify_on_new'] = True
plugin_prefs.defaults['notify_on_first_fetch'] = False
plugin_prefs.defaults['max_notifications_per_update'] = 5
plugin_prefs.defaults['auto_update_on_open'] = False
plugin_prefs.defaults['auto_update_minutes'] = 10
plugin_prefs.defaults['max_seen_per_feed'] = 200
plugin_prefs.defaults['max_cached_items_per_feed'] = 200
plugin_prefs.defaults['max_items_per_selection'] = 2000
plugin_prefs.defaults['timeout_seconds'] = 12
plugin_prefs.defaults['max_parallel_fetches'] = 4
plugin_prefs.defaults['published_timestamp_format'] = 'yyyy-MM-dd HH:mm'
plugin_prefs.defaults['load_images_in_preview'] = True
plugin_prefs.defaults['autofit_images'] = True
plugin_prefs.defaults['suspend_fetching'] = False
plugin_prefs.defaults['export_add_to_library'] = False
plugin_prefs.defaults['article_export_add_to_library'] = False
plugin_prefs.defaults['export_output_format'] = 'calibre_default'

# AI panel
plugin_prefs.defaults['ai_export_append_timestamp'] = True
plugin_prefs.defaults['ai_export_append_timestamp'] = True
plugin_prefs.defaults['ai_error_log'] = []

# Export image processing
plugin_prefs.defaults['export_download_uncached_images'] = True
# Defaults aim to avoid huge output files while preserving readability.
# These values are intentionally conservative and can be adjusted by the user.
plugin_prefs.defaults['export_image_max_bytes'] = 3 * 1024 * 1024
plugin_prefs.defaults['export_optimize_images'] = True
plugin_prefs.defaults['export_optimize_images_min_bytes'] = 512 * 1024
plugin_prefs.defaults['export_optimize_force_jpeg_bytes'] = 1 * 1024 * 1024
plugin_prefs.defaults['export_image_max_dim'] = 1200
plugin_prefs.defaults['export_image_jpeg_quality'] = 75
plugin_prefs.defaults['debug_export_images'] = False

# Settings UI
plugin_prefs.defaults['show_advanced_settings'] = False
plugin_prefs.defaults['settings_dialog_geometry'] = ''
plugin_prefs.defaults['plugin_ui_language'] = ''  # '' = calibre default, else locale code (en, fr, ru, etc.)
plugin_prefs.defaults['serbian_graphia'] = 'latin'  # cyrillic|latin
plugin_prefs.defaults['russian_graphia'] = 'native'  # native|latin
plugin_prefs.defaults['japanese_graphia'] = 'native'  # native|latin
plugin_prefs.defaults['korean_graphia'] = 'native'  # native|latin
plugin_prefs.defaults['chinese_graphia'] = 'native'  # native|latin
plugin_prefs.defaults['hindi_graphia'] = 'native'  # native|latin
plugin_prefs.defaults['arabic_graphia'] = 'native'  # native|latin
plugin_prefs.defaults['hebrew_graphia'] = 'native'  # native|latin
plugin_prefs.defaults['greek_graphia'] = 'native'  # native|latin

# Minimize to tray (Windows only)
plugin_prefs.defaults['minimize_to_tray'] = True

# Suppress notifications when a fullscreen window is active
plugin_prefs.defaults['suppress_notifications_fullscreen'] = False

# Failed-feed history
plugin_prefs.defaults['failed_feeds_history_retention_days'] = 60
plugin_prefs.defaults['failed_feeds_history_page_size'] = 500

# Search/filter history (dropdown search boxes)
plugin_prefs.defaults['items_search_history'] = []
plugin_prefs.defaults['feeds_advanced_filter_history'] = []

# Database profiles / default DB location
# If set, this becomes the "Default DB" location (can be outside calibre config dir).
plugin_prefs.defaults['db_default_path'] = ''
# User-facing label for the configured default DB (shown in footer, tooltips, profile manager).
plugin_prefs.defaults['db_default_name'] = ''
# Optional emoji prefix for the configured default DB.
plugin_prefs.defaults['db_default_emoji'] = ''
plugin_prefs.defaults['db_default_readonly'] = False
plugin_prefs.defaults['db_default_mirror'] = False

# First-run onboarding for choosing a durable DB location
plugin_prefs.defaults['db_onboarded'] = False

# Saved DB profiles and currently active profile id
plugin_prefs.defaults['db_profiles'] = []
plugin_prefs.defaults['db_profiles_active'] = ''

# Tagging / auto-tagging
plugin_prefs.defaults['auto_tagging_enabled'] = True
plugin_prefs.defaults['auto_tag_img'] = True
plugin_prefs.defaults['auto_tag_audio'] = True
plugin_prefs.defaults['auto_tag_long'] = True
plugin_prefs.defaults['auto_tag_long_words'] = 300

# Feed auto-tagging (user-configurable)
plugin_prefs.defaults['auto_feed_tagging_enabled'] = True
plugin_prefs.defaults['auto_feed_updates_tag_name'] = 'updates-frequently'
plugin_prefs.defaults['auto_feed_updates_window_days'] = 5
plugin_prefs.defaults['auto_feed_updates_min_distinct_dates'] = 4
plugin_prefs.defaults['auto_feed_updates_min_avg_per_day'] = 3.0

# Failure auto-tagging (based on last update error classification)
plugin_prefs.defaults['auto_feed_failure_tagging_enabled'] = True

# AdBlock (image blocking)
# Preview image debug banner (visible in preview when true)
plugin_prefs.defaults['debug_preview_images'] = True
# Minimum AI panel width (pixels) when shown beside preview
plugin_prefs.defaults['preview_ai_min_width'] = 120

# Folder organizer
plugin_prefs.defaults['default_folder'] = ''
# OPML single-click behavior: 'simple' or 'advanced'
plugin_prefs.defaults['opml_import_single_click'] = 'advanced'
plugin_prefs.defaults['opml_export_single_click'] = 'advanced'

# Feed defaults (Calibre-like): applied unless a feed explicitly overrides.
# Per-feed values can be set to 0 for "no limit".
plugin_prefs.defaults['default_oldest_article_days'] = 7
plugin_prefs.defaults['default_max_articles'] = 100

# Preview article fetch: when enabled, RSS Reader will try to fetch
# the full article HTML for preview when the feed/item summary is missing
# or when explicitly requested. Enabled by default to show full content
# and fetch-engine banner; disable if extra network activity is a concern.
plugin_prefs.defaults['preview_fetch_article_content'] = True
# When fetching full article content for Preview, prefer calibre's recipe/readability
# extraction path (may be slower). Per-feed flags can still enable it selectively.
plugin_prefs.defaults['preview_use_recipe_engine'] = False
# Debug: show a small banner in the preview indicating which full-article
# fetching path was used (if any).
plugin_prefs.defaults['debug_preview_fetch_engine'] = True
# Toolbar customization
plugin_prefs.defaults['toolbar_hidden'] = ['mark_read']
plugin_prefs.defaults['show_customize_toolbar_button'] = False

# Footer label colors (theme-aware: light vs dark)
plugin_prefs.defaults['footer_status_color_dark'] = '#88BB88'
plugin_prefs.defaults['footer_status_color_light'] = '#2E7D32'
plugin_prefs.defaults['footer_profile_color_dark'] = '#E8C97A'
plugin_prefs.defaults['footer_profile_color_light'] = '#8A5A00'
plugin_prefs.defaults['footer_version_color_dark'] = '#AABBCC'
plugin_prefs.defaults['footer_version_color_light'] = '#3E4B5A'

# In-app popup notifications (theme-aware)
# Style preset selection: 'light' or 'dark'.
# Dark uses the GitHub Dark (High Contrast)-inspired palette and styling.
plugin_prefs.defaults['popup_theme'] = 'dark'

# Ensure in-app popup notifications are enabled by default
plugin_prefs.defaults['notify_in_app_popup'] = True
# Popup behavior defaults
plugin_prefs.defaults['popup_timeout_ms'] = 15000
plugin_prefs.defaults['popup_max_items_per_page'] = 10
plugin_prefs.defaults['popup_max_items_total'] = 40
plugin_prefs.defaults['popup_position'] = 'bottom_right'  # bottom_right|bottom_left|top_right|top_left|bottom_center|top_center|right_middle|left_middle
plugin_prefs.defaults['popup_margin_px'] = 10
plugin_prefs.defaults['popup_width_px'] = 0

# Popup item layout (per-article rows)
# 0 height means auto; padding is applied vertically inside each row.
plugin_prefs.defaults['popup_item_min_height_px'] = 0
plugin_prefs.defaults['popup_item_padding_v_px'] = 2

# Color palettes. These are intentionally not exposed in the config dialog.
# Defaults are inspired by GitHub Dark (High Contrast) UI colors.
plugin_prefs.defaults['popup_theme_colors_light'] = {
    'bg': '#FFFFFF',
    'fg': '#24292F',
    'muted_fg': '#57606A',
    'border': 'rgba(27,31,36,0.25)',
    'header_bg': '#F6F8FA',
    'header_fg': '#24292F',
    'feed_header_bg': '#EAECEF',
    'feed_header_fg': '#24292F',
    'item_hover_bg': '#F3F4F6',
    'link': '#0969DA',
    'btn_bg': '#F6F8FA',
    'btn_fg': '#24292F',
    'btn_border': 'rgba(27,31,36,0.25)',
    'btn_hover_bg': '#EAECEF',
    'btn_danger_bg': '#DA3633',
    'btn_danger_fg': '#FFFFFF',
    'btn_primary_bg': '#1F6FEB',
    'btn_primary_fg': '#FFFFFF',
}

plugin_prefs.defaults['popup_theme_colors_dark'] = {
    'bg': '#0D1117',
    'fg': '#F0F6FC',
    'muted_fg': '#7D8590',
    'border': '#30363D',
    'header_bg': '#161B22',
    'header_fg': '#F0F6FC',
    'feed_header_bg': '#21262D',
    'feed_header_fg': '#F0F6FC',
    'item_hover_bg': 'rgba(177,186,196,0.12)',
    'link': '#6CB4EE',
    'btn_bg': '#21262D',
    'btn_fg': '#F0F6FC',
    'btn_border': '#30363D',
    'btn_hover_bg': '#30363D',
    'btn_danger_bg': '#DA3633',
    'btn_danger_fg': '#FFFFFF',
    'btn_primary_bg': '#1F6FEB',
    'btn_primary_fg': '#FFFFFF',
}

plugin_prefs.defaults['popup_theme_colors_dark_high_contrast'] = {
    'bg': '#0A0E14',
    'fg': '#FFFFFF',
    'muted_fg': '#C9D1D9',
    'border': '#7A828E',
    'header_bg': '#0D1117',
    'header_fg': '#FFFFFF',
    'feed_header_bg': '#161B22',
    'feed_header_fg': '#FFFFFF',
    'item_hover_bg': 'rgba(255,255,255,0.10)',
    'link': '#6CB4EE',
    'btn_bg': '#0D1117',
    'btn_fg': '#FFFFFF',
    'btn_border': '#7A828E',
    'btn_hover_bg': '#161B22',
    'btn_danger_bg': '#FF4D4D',
    'btn_danger_fg': '#000000',
    # Mark-all / primary action needs strong contrast
    'btn_primary_bg': '#58A6FF',
    'btn_primary_fg': '#000000',
}

# Bundled sample feeds for quick-start onboarding
plugin_prefs.defaults['bundled_feeds'] = [
    {'name': 'Project Gutenberg Recently Posted or Updated EBooks', 'url': 'https://www.gutenberg.org/cache/epub/feeds/today.rss', 'lang': 'en', 'tags': ['books']},
    {'name': 'ManyBooks (direct links)', 'url': 'https://manybooks.net/rss.xml', 'lang': 'en', 'tags': ['books'], 'max_articles': 20},
    {'name': 'ManyBooks (summary)', 'url': 'https://manybooks.net/rss', 'lang': 'en', 'tags': ['books', 'fetch_webpage'], 'max_articles': 20},
    {'name': 'Standard Ebooks - New Releases', 'url': 'https://standardebooks.org/feeds/rss/new-releases', 'lang': 'en', 'tags': ['books']},
    {'name': 'Merriam-Webster Word of the Day', 'url': 'https://www.merriam-webster.com/wotd/feed/rss2', 'lang': 'en', 'tags': ['words']},
    {'name': 'MobileRead Ebooks', 'url': 'https://www.mobileread.com/feeds/130_rss20.xml', 'lang': 'en', 'tags': ['books']},
    {'name': 'TED Talks Daily (audio)', 'url': 'https://feeds.feedburner.com/TEDTalks_audio', 'lang': 'en', 'tags': ['audio', 'talks', 'podcast'], 'max_articles': 25},
    {'name': "LibriVox's New Releases", 'url': 'https://librivox.org/rss/latest_releases', 'lang': 'en', 'tags': ['audiobooks']},
]
plugin_prefs.defaults['bundled_feeds_onboarded'] = False
# Legacy migration removed: bundled feeds are defined explicitly above.

# Migration: clear legacy 'main' default_folder if it was set by older versions.
# The root folder is now called 'Feeds' (just a display label), and feeds at the
# root level should have folder='' (empty string), not 'main'.
try:
    _df = plugin_prefs.get('default_folder', '')
    if str(_df or '').strip().lower() == 'main':
        plugin_prefs['default_folder'] = ''
        try:
            commit = getattr(plugin_prefs, 'commit', None)
            if callable(commit):
                commit()
        except Exception:
            pass
except Exception:
    pass

try:
    from qt.core import (
    QColor,
    QColorDialog,
        QApplication,
    QTimer,
        QMenu,
        QComboBox,
        QCheckBox,
                QTabWidget,
                QWidget,
        QTextBrowser,
        QDialog,
        QDialogButtonBox,
        QDoubleSpinBox,
        QFormLayout,
        QGroupBox,
        QHBoxLayout,
        QLabel,
        QLineEdit,
        QMessageBox,
        QPushButton,
        QTableWidget,
        QTableWidgetItem,
        QAbstractItemView,
        QHeaderView,
        QSizePolicy,
        QSpinBox,
        QVBoxLayout,
        QDesktopServices,
        QUrl,
        Qt,
        QPalette,
        QListWidget,
        QListWidgetItem,
        QScrollArea,
    )
except ImportError:
    from PyQt5.Qt import (
        QColor,
        QColorDialog,
        QApplication,
        QTimer,
        QMenu,
        QComboBox,
        QCheckBox,
                QTabWidget,
                QWidget,
        QTextBrowser,
        QDialog,
        QDialogButtonBox,
        QDoubleSpinBox,
        QFormLayout,
        QGroupBox,
        QHBoxLayout,
        QLabel,
        QLineEdit,
        QMessageBox,
        QPushButton,
        QTableWidget,
        QTableWidgetItem,
        QAbstractItemView,
        QHeaderView,
        QSizePolicy,
        QSpinBox,
        QVBoxLayout,
        QDesktopServices,
        QUrl,
        Qt,
        QPalette,
    )


DEFAULT_PUBLISHED_TIMESTAMP_FORMAT = 'yyyy-MM-dd HH:mm'
AVAILABLE_PUBLISHED_TIMESTAMP_FORMATS = {
    _('24h European (31/12/2024 23:59)'): 'dd/MM/yyyy HH:mm',
    _('12h American (12/31/2024 11:59 PM)'): 'MM/dd/yyyy hh:mm AP',
    _('ISO (2024-12-31 23:59)'): 'yyyy-MM-dd HH:mm',
}


class ConfigDialog(QDialog):
    def __init__(self, gui, parent=None, embed=False):
        # If the user has selected a Latin/transliteration UI, refresh all
        # already-imported plugin modules so this dialog's _() binding matches.
        try:
            from calibre_plugins.rss_reader.i18n import refresh_all_plugin_modules

            refresh_all_plugin_modules()
        except Exception:
            pass

        QDialog.__init__(self, parent)
        self.gui = gui
        self._embed_mode = bool(embed)
        self.setWindowTitle(_('RSS Reader settings'))

        # Apply scrollbar styling directly so it works in both standalone
        # and calibre-embedded (context menu) modes.
        try:
            self.setStyleSheet(self.styleSheet() + '''
                QScrollBar::handle {
                   border: 1px solid #5B6985;
                }
                QScrollBar:vertical {
                    background: transparent;
                    width: 12px;
                    margin: 0px 0px 0px 0px;
                    padding: 6px 0px 6px 0px;
                }
                QScrollBar::handle:vertical {
                    background: rgba(140, 172, 204, 0.25);
                    min-height: 22px;
                    border-radius: 4px;
                    margin: 4px 0px;
                }
                QScrollBar::handle:vertical:hover {
                    background: rgba(140, 172, 204, 0.45);
                }
                QScrollBar:horizontal {
                    background: transparent;
                    height: 12px;
                    margin: 0px 0px 0px 0px;
                    padding: 0px 6px 0px 6px;
                }
                QScrollBar::handle:horizontal {
                    background: rgba(140, 172, 204, 0.25);
                    min-width: 22px;
                    border-radius: 4px;
                    margin: 0px 4px;
                }
                QScrollBar::handle:horizontal:hover {
                    background: rgba(140, 172, 204, 0.45);
                }
                QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,
                QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {
                    background: none;
                }
            ''')
        except Exception:
            pass

        # Remove minimum height constraint so dialog can shrink
        try:
            self.setMinimumHeight(0)
        except Exception:
            pass

        # calibre sometimes enforces a smaller minimum width for embedded
        # config widgets; enforce our preferred minimum.
        try:
            self.setMinimumWidth(770)
        except Exception:
            pass

        restored = False
        try:
            restored = self._restore_dialog_geometry()
        except Exception:
            restored = False
        if not restored:
            try:
                self.resize(990, 680)
            except Exception:
                pass

        try:
            QTimer.singleShot(0, self._enforce_min_width)
        except Exception:
            try:
                self._enforce_min_width()
            except Exception:
                pass

    def _enforce_min_width(self):
        min_w = 700
        try:
            if int(self.minimumWidth() or 0) < min_w:
                self.setMinimumWidth(min_w)
        except Exception:
            pass
        try:
            if int(self.width() or 0) < min_w:
                self.resize(min_w, int(self.height() or 0))
        except Exception:
            pass
        # In embed mode, calibre wraps this widget in another dialog.
        try:
            w = self.window()
        except Exception:
            w = None
        try:
            if w is not None and w is not self:
                try:
                    if int(w.minimumWidth() or 0) < min_w:
                        w.setMinimumWidth(min_w)
                except Exception:
                    pass
                try:
                    if int(w.width() or 0) < min_w:
                        w.resize(min_w, int(w.height() or 0))
                except Exception:
                    pass
        except Exception:
            pass

    def showEvent(self, ev):
        # Ensure _ (translation function) is available in this method's scope
        try:
            _ = globals().get('_', lambda x: x)
            if not callable(_):
                _ = lambda x: x
        except Exception:
            _ = lambda x: x
        
        try:
            QDialog.showEvent(self, ev)
        except Exception:
            pass
        try:
            QTimer.singleShot(0, self._enforce_min_width)
        except Exception:
            try:
                self._enforce_min_width()
            except Exception:
                pass

        layout = QVBoxLayout(self)

        # Use tabs: first tab is Config (all existing controls), second tab is About
        try:
            self._tabs = QTabWidget(self)
            # Apply OPF_Helper/CCR-style tab styling
            self._tabs.setStyleSheet('''
                QTabWidget::pane {
                    border-top: none;
                    margin-left: 8px;
                    margin-right: 8px;
                }
                QTabBar::tab {
                    min-width: 124px;
                    width: 3px;
                    padding-top: 6px;
                    padding-bottom: 6px;
                    font-size: 9pt;
                    background: transparent;
                    border-top: 1px solid palette(mid);
                    border-left: 1px solid palette(mid);
                    border-right: 1px solid palette(mid);
                    border-top-left-radius: 8px;
                    border-top-right-radius: 8px;
                }
                QTabBar::tab:selected {
                    font-weight: bold;
                    font-style: normal;
                    border-top: 2px solid palette(link);
                    border-left: 2px solid palette(link);
                    border-right: 2px solid palette(link);
                    color: palette(link);
                }
                QTabBar::tab:!selected {
                    color: palette(text);
                    margin-top: 2px;
                }
            ''')
        except Exception:
            self._tabs = None

        # Basic vs Advanced toggle (placed inside Config tab)
        try:
            self.show_advanced_cb = QCheckBox(_('Show advanced settings'), self)
            self.show_advanced_cb.setChecked(bool(plugin_prefs.get('show_advanced_settings', False)))
            try:
                self.show_advanced_cb.setToolTip(_('When disabled, only the most common options are shown.'))
            except Exception:
                pass
        except Exception:
            self.show_advanced_cb = None

        # Two-column layout for better density inside the Config tab.
        # Always keep the Config tab scrollable so advanced-mode overflow
        # shows a vertical scrollbar without forcing the dialog to grow.
        try:
            from qt.core import QScrollArea
        except Exception:
            from PyQt5.QtWidgets import QScrollArea

        cols = QHBoxLayout()
        left_col = QVBoxLayout()
        right_col = QVBoxLayout()
        cols.addLayout(left_col, 1)
        cols.addLayout(right_col, 1)

        config_inner = QWidget(self)
        config_inner_layout = QVBoxLayout(config_inner)
        config_inner_layout.setContentsMargins(0, 0, 0, 0)
        if self.show_advanced_cb is not None:
            config_inner_layout.addWidget(self.show_advanced_cb)
        config_inner_layout.addLayout(cols, 1)

        scroll = QScrollArea(self)
        scroll.setWidgetResizable(True)
        try:
            scroll.setFrameShape(QScrollArea.NoFrame)
        except Exception:
            pass
        scroll.setWidget(config_inner)

        config_page = QWidget(self)
        config_layout = QVBoxLayout(config_page)
        config_layout.setContentsMargins(0, 0, 0, 0)
        config_layout.addWidget(scroll, 1)

        # Notifications tab page
        try:
            notifications_page = QWidget(self)
            notifications_layout = QVBoxLayout(notifications_page)
            notifications_layout.setContentsMargins(0, 0, 0, 0)
        except Exception:
            notifications_page = None
            notifications_layout = None

        # About tab page
        try:
            about_page = QWidget(self)
            about_layout = QVBoxLayout(about_page)
            about_layout.setContentsMargins(6, 6, 6, 6)
        except Exception:
            about_page = None

        # Failed feeds history tab page (inserted before About)
        try:
            failed_page = QWidget(self)
            failed_layout = QVBoxLayout(failed_page)
            failed_layout.setContentsMargins(6, 6, 6, 6)
        except Exception:
            failed_page = None
            failed_layout = None

        # AI error log tab page (errors from calibre AI backends)
        try:
            ai_errors_page = QWidget(self)
            ai_errors_layout = QVBoxLayout(ai_errors_page)
            ai_errors_layout.setContentsMargins(6, 6, 6, 6)
        except Exception:
            ai_errors_page = None
            ai_errors_layout = None

        # Add tabs to main layout
        if getattr(self, '_tabs', None) is not None:
            try:
                if config_page is not None:
                    self._tabs.addTab(config_page, _('Config'))
            except Exception:
                pass
            try:
                if notifications_page is not None:
                    self._tabs.addTab(notifications_page, _('Notifications'))
                    try:
                        self._notifications_tab_index = int(self._tabs.indexOf(notifications_page))
                    except Exception:
                        self._notifications_tab_index = None
            except Exception:
                pass
            try:
                    if failed_page is not None:
                        self._tabs.addTab(failed_page, _('Error log'))
            except Exception:
                pass
            try:
                if ai_errors_page is not None:
                    self._tabs.addTab(ai_errors_page, _('AI errors'))
            except Exception:
                pass
            try:
                if about_page is not None:
                    self._tabs.addTab(about_page, _('About'))
                    try:
                        self._about_tab_index = int(self._tabs.indexOf(about_page))
                    except Exception:
                        self._about_tab_index = None
            except Exception:
                pass

            # Restore last active tab (annoying otherwise when reopening)
            try:
                idx = int(plugin_prefs.get('settings_last_tab', 0) or 0)
            except Exception:
                idx = 0
            try:
                if idx < 0 or idx >= self._tabs.count():
                    idx = 0
                self._tabs.setCurrentIndex(idx)
            except Exception:
                pass

            def _remember_tab(i):
                try:
                    plugin_prefs['settings_last_tab'] = int(i)
                except Exception:
                    pass
                try:
                    _update_maintenance_buttons_visibility(int(i))
                except Exception:
                    pass

            def _update_maintenance_buttons_visibility(i):
                try:
                    nti = getattr(self, '_notifications_tab_index', None)
                    abi = getattr(self, '_about_tab_index', None)
                    hide = (nti is not None and int(i) == int(nti)) or (abi is not None and int(i) == int(abi))
                except Exception:
                    hide = False
                for btn_name in ('purge_btn', 'clear_db_btn', 'restore_settings_btn', 'restore_layout_btn'):
                    try:
                        b = getattr(self, btn_name, None)
                        if b is not None:
                            b.setVisible(not hide)
                    except Exception:
                        pass

            try:
                self._tabs.currentChanged.connect(_remember_tab)
            except Exception:
                pass
            layout.addWidget(self._tabs, 1)
        else:
            # Fallback: behave like previous single-page layout
            layout.addWidget(config_page if config_page is not None else QWidget(self), 1)

        def _make_group(title, parent_widget=None):
            gb = QGroupBox(title, parent_widget if parent_widget is not None else self)
            try:
                gb.setFlat(False)
            except Exception:
                pass
            frm = QFormLayout(gb)
            try:
                frm.setFieldGrowthPolicy(QFormLayout.FieldGrowthPolicy.AllNonFixedFieldsGrow)
            except Exception:
                pass
            gb.setLayout(frm)
            return gb, frm

        updates_gb, updates_form = _make_group(_('Updates && notifications'))
        network_gb, network_form = _make_group(_('Network'))
        storage_gb, storage_form = _make_group(_('Storage'))
        display_gb, display_form = _make_group(_('Display'))
        preview_gb, preview_form = _make_group(_('Preview && export'))
        opml_gb, opml_form = _make_group(_('OPML'))
        tagging_gb, tagging_form = _make_group(_('Tagging'))

        # Build Notifications tab UI
        try:
            self._init_notifications_tab(notifications_page, notifications_layout, _make_group)
        except Exception:
            pass

        # Build Failed feeds tab UI now that we have widgets/layouts
        try:
            self._init_failed_feeds_tab(failed_layout)
        except Exception:
            pass

        # Build AI errors tab UI
        try:
            self._init_ai_errors_tab(ai_errors_layout)
        except Exception:
            pass

        # Store refs so we can toggle visibility for Basic/Advanced modes.
        self.network_gb = network_gb
        self.storage_gb = storage_gb
        self.opml_gb = opml_gb
        self._display_form = display_form
        self._preview_form = preview_form
        self._tagging_form = tagging_form

        self.notify_cb = QCheckBox(_('Show notifications for new items'), self)
        self.notify_cb.setChecked(bool(plugin_prefs.get('notify_on_new', True)))
        updates_form.addRow(self.notify_cb)

        self.notify_first_cb = QCheckBox(_('Notify on first fetch for a new feed'), self)
        self.notify_first_cb.setChecked(bool(plugin_prefs.get('notify_on_first_fetch', False)))
        updates_form.addRow(self.notify_first_cb)

        self.max_notify_spin = QSpinBox(self)
        self.max_notify_spin.setRange(1, 20)
        self.max_notify_spin.setValue(int(plugin_prefs.get('max_notifications_per_update', 3) or 3))
        self.max_notify_spin.setToolTip(_('Maximum number of notifications shown per update cycle (not per feed). If multiple feeds have new items, only this many notifications will be shown in total.'))
        updates_form.addRow(_('Max notifications per update:'), self.max_notify_spin)

        self.suppress_fullscreen_cb = QCheckBox(_('Do not show notification in fullscreen mode'), self)
        self.suppress_fullscreen_cb.setChecked(bool(plugin_prefs.get('suppress_notifications_fullscreen', False)))
        self.suppress_fullscreen_cb.setToolTip(_('When enabled, notifications (both OS and in-app popup) will be suppressed while any window is in fullscreen mode.'))
        updates_form.addRow(self.suppress_fullscreen_cb)

        self.auto_open_cb = QCheckBox(_('Auto-update when opening RSS Reader'), self)
        self.auto_open_cb.setChecked(bool(plugin_prefs.get('auto_update_on_open', True)))
        updates_form.addRow(self.auto_open_cb)

        self.auto_spin = QSpinBox(self)
        self.auto_spin.setRange(0, 24 * 60)
        self.auto_spin.setSuffix(_(' min'))
        self.auto_spin.setToolTip(_('0 disables automatic updates'))
        self.auto_spin.setValue(int(plugin_prefs.get('auto_update_minutes', 0) or 0))
        updates_form.addRow(_('Auto-update every:'), self.auto_spin)

        self.timeout_spin = QSpinBox(self)
        self.timeout_spin.setRange(5, 120)
        self.timeout_spin.setSuffix(_(' seconds'))
        self.timeout_spin.setValue(int(plugin_prefs.get('timeout_seconds', 25) or 25))
        network_form.addRow(_('Timeout:'), self.timeout_spin)

        self.parallel_spin = QSpinBox(self)
        self.parallel_spin.setRange(1, 16)
        self.parallel_spin.setValue(int(plugin_prefs.get('max_parallel_fetches', 4) or 4))
        self.parallel_spin.setToolTip(_('How many feeds to fetch in parallel. Higher is faster but uses more network/battery.'))
        network_form.addRow(_('Parallel fetches:'), self.parallel_spin)

        self.max_seen_spin = QSpinBox(self)
        self.max_seen_spin.setRange(50, 2000)
        self.max_seen_spin.setValue(int(plugin_prefs.get('max_seen_per_feed', 200) or 200))
        storage_form.addRow(_('Remember last item IDs per feed:'), self.max_seen_spin)

        self.max_cache_spin = QSpinBox(self)
        self.max_cache_spin.setRange(50, 2000)
        self.max_cache_spin.setValue(int(plugin_prefs.get('max_cached_items_per_feed', 200) or 200))
        storage_form.addRow(_('Cached items per feed:'), self.max_cache_spin)

        try:
            self.max_items_spin = QSpinBox(self)
            self.max_items_spin.setRange(50, 20000)
            self.max_items_spin.setValue(int(plugin_prefs.get('max_items_per_selection', 2000) or 2000))
            self.max_items_spin.setToolTip(_('Maximum number of rows shown for a selection when no search query is active (helps performance).'))
            storage_form.addRow(_('Max items per selection:'), self.max_items_spin)
        except Exception:
            self.max_items_spin = None

        try:
            self.max_cache_spin.setToolTip(_('Cache last items per feed (for offline viewing).'))
        except Exception:
            pass

        # --- Plugin UI Language Selector ---
        self.language_combo = None
        try:
            self.language_combo = QComboBox(self)
            available_langs = _get_available_languages()
            for locale, display_name in available_langs:
                self.language_combo.addItem(display_name, locale)
            
            # Set current selection
            current_lang = str(plugin_prefs.get('plugin_ui_language', '') or '').strip()
            idx = 0
            for i, (loc, dname) in enumerate(available_langs):
                if loc == current_lang:
                    idx = i
                    break
            self.language_combo.setCurrentIndex(idx)
            
            # Connect signal AFTER setting initial index to avoid unwanted triggers
            try:
                self.language_combo.currentIndexChanged.connect(self._on_language_changed)
            except Exception as e:
                import sys
                print(f'ERROR: Failed to connect language_combo signal: {e}', file=sys.stderr)
            
            try:
                self.language_combo.setToolTip(
                    _('Override the plugin UI language independently of Calibre.')
                    + '\n' + _('Re-open this dialog to see the updated translations.')
                    + '\n' + _('A full Calibre restart is required to update the toolbar button and its context menus.')
                )
            except Exception:
                pass
            
            display_form.addRow(_('Plugin UI language:'), self.language_combo)
        except Exception as e:
            import sys
            print(f'ERROR: Failed to create language_combo: {e}', file=sys.stderr)
            self.language_combo = None

        self.published_format_combo = QComboBox(self)
        for label in AVAILABLE_PUBLISHED_TIMESTAMP_FORMATS:
            self.published_format_combo.addItem(label)
        current_format = str(plugin_prefs.get('published_timestamp_format', DEFAULT_PUBLISHED_TIMESTAMP_FORMAT) or DEFAULT_PUBLISHED_TIMESTAMP_FORMAT)
        current_index = 0
        for i, fmt in enumerate(AVAILABLE_PUBLISHED_TIMESTAMP_FORMATS.values()):
            if str(fmt) == current_format:
                current_index = i
                break
        self.published_format_combo.setCurrentIndex(current_index)
        display_form.addRow(_('Published column format:'), self.published_format_combo)

        # Script selection dropdowns (only shown when calibre UI language matches)
        # Supported: Serbian, Russian, Japanese, Chinese (all variants)
        _script_tooltip_suffix = _('The calibre toolbar button and context menu labels require a calibre restart to update.')

        # Platform detection for Windows-only options
        try:
            import platform

            _is_windows = platform.system().lower().startswith('win')
        except Exception:
            try:
                import sys

                _is_windows = str(getattr(sys, 'platform', '') or '').lower().startswith('win')
            except Exception:
                _is_windows = False

        try:
            from calibre.utils.localization import get_lang

            _lang = str(get_lang() or '').strip().lower()
        except Exception:
            _lang = ''

        # --- Serbian ---
        self.serbian_graphia_combo = None
        try:
            if _lang.startswith('sr') and not _lang.startswith('sr_latn'):
                self.serbian_graphia_combo = QComboBox(self)
                self.serbian_graphia_combo.addItem(_('Cyrillic'), 'cyrillic')
                self.serbian_graphia_combo.addItem(_('Latin'), 'latin')
                cur = str(plugin_prefs.get('serbian_graphia', 'latin') or 'latin').strip().lower()
                try:
                    # Normalize: ensure we only have 'cyrillic' or 'latin' before setting index
                    if cur not in ('cyrillic', 'latin'):
                        cur = 'latin'
                    idx = 1 if cur == 'latin' else 0
                    self.serbian_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.serbian_graphia_combo.setToolTip(_('Controls whether Serbian UI text is shown in Cyrillic or Latin. Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Serbian script:'), self.serbian_graphia_combo)
        except Exception:
            self.serbian_graphia_combo = None

        # --- Russian ---
        self.russian_graphia_combo = None
        try:
            if _lang.startswith('ru'):
                self.russian_graphia_combo = QComboBox(self)
                self.russian_graphia_combo.addItem(_('Cyrillic'), 'native')
                self.russian_graphia_combo.addItem(_('Latin'), 'latin')
                cur = str(plugin_prefs.get('russian_graphia', 'native') or 'native').strip().lower()
                try:
                    # Normalize: ensure we only have 'native' or 'latin' before setting index
                    if cur not in ('native', 'latin'):
                        cur = 'native'
                    idx = 1 if cur == 'latin' else 0
                    self.russian_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.russian_graphia_combo.setToolTip(_('Controls whether Russian UI text is shown in Cyrillic or Latin transliteration. Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Russian script:'), self.russian_graphia_combo)
        except Exception:
            self.russian_graphia_combo = None

        # --- Japanese ---
        self.japanese_graphia_combo = None
        try:
            if _lang.startswith('ja'):
                self.japanese_graphia_combo = QComboBox(self)
                self.japanese_graphia_combo.addItem(_('Japanese'), 'native')
                self.japanese_graphia_combo.addItem(_('Romaji'), 'latin')
                cur = str(plugin_prefs.get('japanese_graphia', 'native') or 'native').strip().lower()
                try:
                    idx = 1 if cur == 'latin' else 0
                    self.japanese_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.japanese_graphia_combo.setToolTip(_('Controls whether Japanese UI text is shown in native script or Romaji (Latin). Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Japanese script:'), self.japanese_graphia_combo)
        except Exception:
            self.japanese_graphia_combo = None

        # --- Chinese (all variants: zh_CN, zh_TW, zh_HK) ---
        self.chinese_graphia_combo = None
        try:
            if _lang.startswith('zh'):
                self.chinese_graphia_combo = QComboBox(self)
                self.chinese_graphia_combo.addItem(_('Chinese'), 'native')
                self.chinese_graphia_combo.addItem(_('Pinyin'), 'latin')
                cur = str(plugin_prefs.get('chinese_graphia', 'native') or 'native').strip().lower()
                try:
                    idx = 1 if cur == 'latin' else 0
                    self.chinese_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.chinese_graphia_combo.setToolTip(_('Controls whether Chinese UI text is shown in native characters or Pinyin (Latin). Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Chinese script:'), self.chinese_graphia_combo)
        except Exception:
            self.chinese_graphia_combo = None

        # --- Korean ---
        self.korean_graphia_combo = None
        try:
            if _lang.startswith('ko'):
                self.korean_graphia_combo = QComboBox(self)
                self.korean_graphia_combo.addItem(_('Korean'), 'native')
                self.korean_graphia_combo.addItem(_('Romanized'), 'latin')
                cur = str(plugin_prefs.get('korean_graphia', 'native') or 'native').strip().lower()
                try:
                    idx = 1 if cur == 'latin' else 0
                    self.korean_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.korean_graphia_combo.setToolTip(_('Controls whether Korean UI text is shown in native Hangul or romanized Latin. Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Korean script:'), self.korean_graphia_combo)
        except Exception:
            self.korean_graphia_combo = None

        # --- Hindi ---
        self.hindi_graphia_combo = None
        try:
            if _lang.startswith('hi'):
                self.hindi_graphia_combo = QComboBox(self)
                self.hindi_graphia_combo.addItem(_('Hindi'), 'native')
                self.hindi_graphia_combo.addItem(_('Latin'), 'latin')
                cur = str(plugin_prefs.get('hindi_graphia', 'native') or 'native').strip().lower()
                try:
                    idx = 1 if cur == 'latin' else 0
                    self.hindi_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.hindi_graphia_combo.setToolTip(_('Controls whether Hindi UI text is shown in native Devanagari script or Latin IAST romanization. Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Hindi script:'), self.hindi_graphia_combo)
        except Exception:
            self.hindi_graphia_combo = None

        # --- Arabic ---
        self.arabic_graphia_combo = None
        try:
            if _lang.startswith('ar'):
                self.arabic_graphia_combo = QComboBox(self)
                self.arabic_graphia_combo.addItem(_('Arabic'), 'native')
                self.arabic_graphia_combo.addItem(_('Latin'), 'latin')
                cur = str(plugin_prefs.get('arabic_graphia', 'native') or 'native').strip().lower()
                try:
                    idx = 1 if cur == 'latin' else 0
                    self.arabic_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.arabic_graphia_combo.setToolTip(_('Controls whether Arabic UI text is shown in native script or Latin transliteration. Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Arabic script:'), self.arabic_graphia_combo)
        except Exception:
            self.arabic_graphia_combo = None

        # --- Hebrew ---
        self.hebrew_graphia_combo = None
        try:
            if _lang.startswith('he'):
                self.hebrew_graphia_combo = QComboBox(self)
                self.hebrew_graphia_combo.addItem(_('Hebrew'), 'native')
                self.hebrew_graphia_combo.addItem(_('Latin'), 'latin')
                cur = str(plugin_prefs.get('hebrew_graphia', 'native') or 'native').strip().lower()
                try:
                    idx = 1 if cur == 'latin' else 0
                    self.hebrew_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.hebrew_graphia_combo.setToolTip(_('Controls whether Hebrew UI text is shown in native script or Latin transliteration. Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Hebrew script:'), self.hebrew_graphia_combo)
        except Exception:
            self.hebrew_graphia_combo = None

        # --- Greek ---
        self.greek_graphia_combo = None
        try:
            if _lang.startswith('el'):
                self.greek_graphia_combo = QComboBox(self)
                self.greek_graphia_combo.addItem(_('Greek'), 'native')
                self.greek_graphia_combo.addItem(_('Latin'), 'latin')
                cur = str(plugin_prefs.get('greek_graphia', 'native') or 'native').strip().lower()
                try:
                    idx = 1 if cur == 'latin' else 0
                    self.greek_graphia_combo.setCurrentIndex(idx)
                except Exception:
                    pass
                try:
                    self.greek_graphia_combo.setToolTip(_('Controls whether Greek UI text is shown in native script or Latin transliteration. Takes effect after closing and re-opening RSS Reader.') + '\n' + _script_tooltip_suffix)
                except Exception:
                    pass
                display_form.addRow(_('Greek script:'), self.greek_graphia_combo)
        except Exception:
            self.greek_graphia_combo = None

        # --- Minimize to tray (Windows only) ---
        self.minimize_to_tray_cb = None
        try:
            if _is_windows:
                self.minimize_to_tray_cb = QCheckBox(_('Minimize to tray (Windows only)'), self)
                self.minimize_to_tray_cb.setChecked(bool(plugin_prefs.get('minimize_to_tray', False)))
                self.minimize_to_tray_cb.setToolTip(_('When enabled, minimizing the RSS Reader window will hide it to the Windows system tray. The tray icon allows restoring or exiting the app.'))
                display_form.addRow(self.minimize_to_tray_cb)
        except Exception:
            self.minimize_to_tray_cb = None

        # Export image processing (advanced)
        try:
            self.export_download_uncached_images_cb = QCheckBox(_('Download uncached images when exporting'), self)
            self.export_download_uncached_images_cb.setChecked(bool(plugin_prefs.get('export_download_uncached_images', True)))
            self.export_download_uncached_images_cb.setToolTip(_('When disabled, only cached images will be embedded; missing images will be left as-is.'))
            preview_form.addRow(self.export_download_uncached_images_cb)
        except Exception:
            self.export_download_uncached_images_cb = None

        try:
            self.export_optimize_images_cb = QCheckBox(_('Optimize images on export (downscale/recompress oversized images)'), self)
            self.export_optimize_images_cb.setChecked(bool(plugin_prefs.get('export_optimize_images', True)))
            self.export_optimize_images_cb.setToolTip(_('Helps avoid enormous EPUBs when a page contains huge images (e.g. screenshots).'))
            preview_form.addRow(self.export_optimize_images_cb)
        except Exception:
            self.export_optimize_images_cb = None

        def _bytes_to_mb(v, default_mb):
            try:
                # Preserve explicit zero (means "always"). Only use default when value is None or missing.
                if v is None:
                    return int(default_mb)
                v = int(v)
                return max(0, int(round(v / (1024.0 * 1024.0))))
            except Exception:
                return int(default_mb)

        try:
            self.export_image_max_bytes_mb = QSpinBox(self)
            self.export_image_max_bytes_mb.setRange(0, 1024)
            self.export_image_max_bytes_mb.setSuffix(_(' MB'))
            self.export_image_max_bytes_mb.setToolTip(_('Maximum embedded image size after processing. 0 disables the limit.'))
            self.export_image_max_bytes_mb.setValue(_bytes_to_mb(plugin_prefs.get('export_image_max_bytes', plugin_prefs.defaults['export_image_max_bytes']), int(round(plugin_prefs.defaults['export_image_max_bytes'] / (1024.0 * 1024.0)))))
            preview_form.addRow(_('Max image size:'), self.export_image_max_bytes_mb)
        except Exception:
            self.export_image_max_bytes_mb = None

        try:
            self.export_optimize_min_mb = QSpinBox(self)
            self.export_optimize_min_mb.setRange(0, 1024)
            self.export_optimize_min_mb.setSuffix(_(' MB'))
            self.export_optimize_min_mb.setToolTip(_('Only optimize images at or above this size. 0 means always.'))
            self.export_optimize_min_mb.setValue(_bytes_to_mb(plugin_prefs.get('export_optimize_images_min_bytes', plugin_prefs.defaults['export_optimize_images_min_bytes']), int(round(plugin_prefs.defaults['export_optimize_images_min_bytes'] / (1024.0 * 1024.0)))))
            preview_form.addRow(_('Optimize if ≥'), self.export_optimize_min_mb)
        except Exception:
            self.export_optimize_min_mb = None

        try:
            self.export_force_jpeg_mb = QSpinBox(self)
            self.export_force_jpeg_mb.setRange(0, 1024)
            self.export_force_jpeg_mb.setSuffix(_(' MB'))
            self.export_force_jpeg_mb.setToolTip(_('If an image is at or above this size, RSS Reader will prefer JPEG re-encode when possible to reduce size. 0 disables.'))
            self.export_force_jpeg_mb.setValue(_bytes_to_mb(plugin_prefs.get('export_optimize_force_jpeg_bytes', plugin_prefs.defaults['export_optimize_force_jpeg_bytes']), int(round(plugin_prefs.defaults['export_optimize_force_jpeg_bytes'] / (1024.0 * 1024.0)))))
            preview_form.addRow(_('Prefer JPEG if ≥'), self.export_force_jpeg_mb)
        except Exception:
            self.export_force_jpeg_mb = None

        try:
            self.export_image_max_dim_spin = QSpinBox(self)
            self.export_image_max_dim_spin.setRange(0, 20000)
            self.export_image_max_dim_spin.setSuffix(_(' px'))
            self.export_image_max_dim_spin.setToolTip(_('Maximum width/height for exported images. 0 disables downscaling.'))
            self.export_image_max_dim_spin.setValue(int(plugin_prefs.get('export_image_max_dim', plugin_prefs.defaults['export_image_max_dim']) or plugin_prefs.defaults['export_image_max_dim']))
            preview_form.addRow(_('Max image dimension:'), self.export_image_max_dim_spin)
        except Exception:
            self.export_image_max_dim_spin = None

        try:
            self.export_image_jpeg_quality_spin = QSpinBox(self)
            self.export_image_jpeg_quality_spin.setRange(30, 95)
            self.export_image_jpeg_quality_spin.setToolTip(_('JPEG quality used when recompressing images.'))
            self.export_image_jpeg_quality_spin.setValue(int(plugin_prefs.get('export_image_jpeg_quality', plugin_prefs.defaults['export_image_jpeg_quality']) or plugin_prefs.defaults['export_image_jpeg_quality']))
            preview_form.addRow(_('JPEG quality:'), self.export_image_jpeg_quality_spin)
        except Exception:
            self.export_image_jpeg_quality_spin = None

        try:
            self.debug_export_images_cb = QCheckBox(_('Debug export images (verbose logging)'), self)
            self.debug_export_images_cb.setChecked(bool(plugin_prefs.get('debug_export_images', False)))
            self.debug_export_images_cb.setToolTip(_('When enabled, logs image cache hits/misses and export decisions.'))
            preview_form.addRow(self.debug_export_images_cb)
        except Exception:
            self.debug_export_images_cb = None

        self.load_images_cb = QCheckBox(_('Load inline images in Preview'), self)
        try:
            self.load_images_cb.setToolTip(_('Load inline images in Preview (can slow loading on image-heavy feeds).'))
        except Exception:
            pass
        self.load_images_cb.setChecked(bool(plugin_prefs.get('load_images_in_preview', True)))
        preview_form.addRow(self.load_images_cb)

        try:
            self.autofit_images_cb = QCheckBox(_('Autofit images in Preview (scale down to fit)'), self)
            self.autofit_images_cb.setToolTip(_('When enabled, images will be scaled down to fit the preview width.'))
            self.autofit_images_cb.setChecked(bool(plugin_prefs.get('autofit_images', True)))
            preview_form.addRow(self.autofit_images_cb)
        except Exception:
            self.autofit_images_cb = None

        try:
            self.fetch_article_cb = QCheckBox(_('Fetch full article content for Preview'), self)
            self.fetch_article_cb.setChecked(bool(plugin_prefs.get('preview_fetch_article_content', False)))
            self.fetch_article_cb.setToolTip(_(
                'When enabled, clicking an item fetches the full article HTML from the web '
                'and appends it below the RSS summary (separated by a horizontal rule).\n\n'
                'When disabled, Preview shows only the content included in the RSS/Atom feed itself.\n\n'
                'This makes one extra HTTP request per article click.'
            ))
            preview_form.addRow(self.fetch_article_cb)
        except Exception:
            self.fetch_article_cb = None

        try:
            self.preview_use_recipe_engine_cb = QCheckBox(_('Use advanced extraction for full-article fetch'), self)
            self.preview_use_recipe_engine_cb.setChecked(bool(plugin_prefs.get('preview_use_recipe_engine', False)))
            self.preview_use_recipe_engine_cb.setToolTip(_(
                'When enabled, the full-article fetch uses calibre\'s Readability engine and '
                'site-specific extraction rules (e.g. Financial Times) for cleaner results.\n\n'
                'When disabled, a basic extractor is used (JSON-LD / <article> tag / first paragraphs).\n\n'
                'This option has no effect unless "Fetch full article content" above is also enabled.'
            ))
            preview_form.addRow(self.preview_use_recipe_engine_cb)
            # Grey out recipe engine checkbox when fetch is disabled
            try:
                self.preview_use_recipe_engine_cb.setEnabled(self.fetch_article_cb.isChecked())
                self.fetch_article_cb.toggled.connect(self.preview_use_recipe_engine_cb.setEnabled)
            except Exception:
                pass
        except Exception:
            self.preview_use_recipe_engine_cb = None

        # AdBlock removed: image blocking via context menu is not available in this release.

        self.opml_import_combo = QComboBox(self)
        self.opml_import_combo.addItem(_('Simple'))
        self.opml_import_combo.addItem(_('Advanced'))
        self.opml_import_combo.addItem(_('Recipes'))
        cur_imp = str(plugin_prefs.get('opml_import_single_click', 'advanced') or 'advanced')
        if cur_imp == 'recipes':
            self.opml_import_combo.setCurrentIndex(2)
        else:
            self.opml_import_combo.setCurrentIndex(1 if cur_imp == 'advanced' else 0)
        opml_form.addRow(_('Import single-click:'), self.opml_import_combo)

        self.opml_export_combo = QComboBox(self)
        self.opml_export_combo.addItem(_('Simple'))
        self.opml_export_combo.addItem(_('Advanced'))
        cur_exp = str(plugin_prefs.get('opml_export_single_click', 'advanced') or 'advanced')
        self.opml_export_combo.setCurrentIndex(1 if cur_exp == 'advanced' else 0)
        opml_form.addRow(_('Export single-click:'), self.opml_export_combo)

        self.export_add_cb = QCheckBox(_('Add exported feeds to library automatically'), self)
        self.export_add_cb.setChecked(bool(plugin_prefs.get('export_add_to_library', False)))
        self.export_add_cb.setToolTip(_('When enabled, exported RSS feeds will be added directly to your Calibre library instead of being saved to a file.'))
        preview_form.addRow(self.export_add_cb)

        self.article_export_add_cb = QCheckBox(_('Add exported articles to library automatically'), self)
        self.article_export_add_cb.setChecked(bool(plugin_prefs.get('article_export_add_to_library', False)))
        self.article_export_add_cb.setToolTip(_('When enabled, single-article exports will be added directly to your Calibre library instead of being saved to a file.'))
        preview_form.addRow(self.article_export_add_cb)

        # AI export naming
        # Always append timestamp to AI export filenames; config option removed
        self.ai_export_ts_cb = None

        self.export_format_combo = QComboBox(self)
        self.export_format_combo.addItem(_('Use calibre default'))
        self.export_format_combo.addItem('EPUB')
        self.export_format_combo.addItem('AZW3')
        self.export_format_combo.addItem('MOBI')
        self.export_format_combo.addItem('PDF')
        self.export_format_combo.addItem('DOCX')
        self.export_format_combo.addItem('TXT')
        self.export_format_combo.addItem('MD (Markdown)')
        cur_fmt = str(plugin_prefs.get('export_output_format', 'calibre_default') or 'calibre_default').strip().lower()
        idx = 0
        if cur_fmt in ('', 'calibre_default', 'default'):
            idx = 0
        else:
            m = {
                'epub': 1,
                'azw3': 2,
                'mobi': 3,
                'pdf': 4,
                'docx': 5,
                'txt': 6,
                'md': 7,
            }
            idx = int(m.get(cur_fmt, 0) or 0)
        try:
            self.export_format_combo.setCurrentIndex(idx)
        except Exception:
            pass
        try:
            self.export_format_combo.setToolTip(_('Choose the default output format when exporting feeds/items. “MD” uses calibre TXT output with Markdown formatting.'))
        except Exception:
            pass
        preview_form.addRow(_('Export output format:'), self.export_format_combo)

        # Tagging
        self.auto_tag_cb = QCheckBox(_('Auto-tag items (adds tags like img/long automatically)'), self)
        self.auto_tag_cb.setChecked(bool(plugin_prefs.get('auto_tagging_enabled', True)))
        tagging_form.addRow(self.auto_tag_cb)

        self.auto_tag_img_cb = QCheckBox(_('Auto-tag items containing images as "img"'), self)
        self.auto_tag_img_cb.setChecked(bool(plugin_prefs.get('auto_tag_img', True)))
        tagging_form.addRow(self.auto_tag_img_cb)

        self.auto_tag_audio_cb = QCheckBox(_('Auto-tag items containing audio enclosures as "audio"'), self)
        self.auto_tag_audio_cb.setChecked(bool(plugin_prefs.get('auto_tag_audio', True)))
        tagging_form.addRow(self.auto_tag_audio_cb)

        self.auto_tag_long_cb = QCheckBox(_('Auto-tag long items as "long"'), self)
        self.auto_tag_long_cb.setChecked(bool(plugin_prefs.get('auto_tag_long', True)))
        tagging_form.addRow(self.auto_tag_long_cb)

        self.auto_tag_words_spin = QSpinBox(self)
        self.auto_tag_words_spin.setRange(50, 20000)
        self.auto_tag_words_spin.setValue(int(plugin_prefs.get('auto_tag_long_words', 300) or 300))
        try:
            self.auto_tag_words_spin.setToolTip(
                _('This threshold is intended to distinguish feeds that provide actual text content in their articles, '
                  'as opposed to feeds that mostly provide images along with headlines.')
            )
        except Exception:
            pass
        tagging_form.addRow(_('"long" if words ≥'), self.auto_tag_words_spin)

        # Feed auto-tagging
        try:
            self.auto_feed_tag_cb = QCheckBox(_('Auto-tag feeds by posting frequency'), self)
            self.auto_feed_tag_cb.setChecked(bool(plugin_prefs.get('auto_feed_tagging_enabled', True)))
            self.auto_feed_tag_cb.setToolTip(_('When enabled, RSS Reader will analyze recent item dates per feed and add an auto tag when a feed meets your frequency thresholds.'))
            tagging_form.addRow(self.auto_feed_tag_cb)

            self.auto_feed_tag_name = QLineEdit(self)
            self.auto_feed_tag_name.setText(str(plugin_prefs.get('auto_feed_updates_tag_name', 'updates frequently') or 'updates frequently'))
            self.auto_feed_tag_name.setToolTip(_('Tag name to apply when a feed matches the frequency rule.'))
            tagging_form.addRow(_('Frequency tag name:'), self.auto_feed_tag_name)

            self.auto_feed_window_days = QSpinBox(self)
            self.auto_feed_window_days.setRange(1, 60)
            self.auto_feed_window_days.setSuffix(_(' days'))
            self.auto_feed_window_days.setValue(int(plugin_prefs.get('auto_feed_updates_window_days', 5) or 5))
            self.auto_feed_window_days.setToolTip(_('Lookback window for frequency analysis.'))
            tagging_form.addRow(_('Lookback window:'), self.auto_feed_window_days)

            self.auto_feed_min_dates = QSpinBox(self)
            self.auto_feed_min_dates.setRange(1, 60)
            self.auto_feed_min_dates.setValue(int(plugin_prefs.get('auto_feed_updates_min_distinct_dates', 4) or 4))
            self.auto_feed_min_dates.setToolTip(_('Minimum number of distinct published dates within the lookback window required before tagging.'))
            tagging_form.addRow(_('Min distinct dates:'), self.auto_feed_min_dates)

            self.auto_feed_min_avg = QDoubleSpinBox(self)
            try:
                self.auto_feed_min_avg.setDecimals(2)
            except Exception:
                pass
            self.auto_feed_min_avg.setRange(0.0, 1000.0)
            self.auto_feed_min_avg.setSingleStep(0.25)
            self.auto_feed_min_avg.setValue(float(plugin_prefs.get('auto_feed_updates_min_avg_per_day', 3.0) or 3.0))
            self.auto_feed_min_avg.setToolTip(_('Minimum average items/day over the lookback window required before tagging.'))
            tagging_form.addRow(_('Min avg items/day:'), self.auto_feed_min_avg)
        except Exception:
            self.auto_feed_tag_cb = None
            self.auto_feed_tag_name = None
            self.auto_feed_window_days = None
            self.auto_feed_min_dates = None
            self.auto_feed_min_avg = None

        # Place groups into columns
        left_col.addWidget(updates_gb)
        left_col.addWidget(network_gb)
        left_col.addWidget(storage_gb)
        left_col.addWidget(display_gb)
        left_col.addWidget(opml_gb)
        left_col.addStretch(1)

        right_col.addWidget(preview_gb)
        right_col.addWidget(tagging_gb)
        right_col.addStretch(1)

        # About tab (match OPF_Helper/CCR: two columns + wired clickable links)
        try:
            if about_page is not None:
                # Wrap the about tab content in a QScrollArea to prevent text cutoff
                about_scroll = QScrollArea(about_page)
                about_scroll.setWidgetResizable(True)
                try:
                    about_scroll.setFrameShape(QScrollArea.NoFrame)
                except Exception:
                    pass

                about_tab = QWidget()
                about_tab_layout = QHBoxLayout(about_tab)

                # Column 1: description and profiles highlight
                desc_column = QWidget(about_tab)
                desc_layout = QVBoxLayout(desc_column)

                desc_title = QLabel(_('<b>About RSS Reader</b>'))
                try:
                    desc_title.setAlignment(Qt.AlignmentFlag.AlignHCenter)
                except Exception:
                    try:
                        desc_title.setAlignment(Qt.AlignHCenter)
                    except Exception:
                        pass
                desc_title.setStyleSheet('font-size: 12pt;')
                desc_layout.addWidget(desc_title)

                desc_label = QLabel(_(
                    'Read and manage your favorite blogs and feeds right inside calibre — '
                    'full article view with images and audio, export to ebook formats, '
                    'send via email, add to your calibre library, AI-powered summaries, '
                    'and multiple database profiles for separate feed collections.'
                ))
                desc_label.setWordWrap(True)
                desc_label.setStyleSheet('font-size: 10pt;')
                desc_layout.addWidget(desc_label)

                profiles_title = QLabel(_('<b>Profiles</b>'))
                # Theme-safe: avoid hard-coded dark text colors
                profiles_title.setStyleSheet('font-size: 11pt; margin-top: 14px; color: palette(link);')
                desc_layout.addWidget(profiles_title)

                profiles_text = _(
                    'RSS Reader supports multiple database profiles. You can create, switch, '
                    'and manage separate feed collections for different use cases (work, personal, '
                    'testing, etc.).<br>'
                    '<b>Tip:</b> Use the profile manager to add, rename, or clear profiles. '
                    'The current profile is always shown in the footer.'
                )
                profiles_label = QLabel(profiles_text)
                profiles_label.setTextFormat(Qt.TextFormat.RichText)
                profiles_label.setWordWrap(True)
                profiles_label.setMinimumWidth(200)
                profiles_label.setStyleSheet(
                    'font-size: 9pt;'
                    'background: palette(alternate-base);'
                    'color: palette(text);'
                    'border: 1px solid palette(mid);'
                    'border-radius: 6px;'
                    'padding: 8px;'
                    'margin-bottom: 4px;'
                )
                desc_layout.addWidget(profiles_label)

                features_title = QLabel(_('<b>Key features:</b>'))
                features_title.setStyleSheet('font-size: 12pt; margin-top: 4px;')
                desc_layout.addWidget(features_title)

                features_text = _(
                    '• Save individual articles as standalone files<br>'
                    '• Add single articles to your calibre library<br>'
                    '• Email articles and feeds (supports Kindle device addresses)<br>'
                    '• Export to common ebook formats via calibre conversion<br>'
                    '• OPML selective import/export<br>'
                    '• AI integration (optional) for article summaries and chat<br>'
                    '• Transliterated UI for CJK and Cyrillic languages (optional)<br>'
                    '• Multiple database profiles for separate feed collections<br>'
                    '• Popup notifications for new articles with clickable links<br>'
                    '• Auto-tagging of articles (images, audio, long reads)'
                )
                features_label = QLabel(features_text)
                features_label.setTextFormat(Qt.TextFormat.RichText)
                features_label.setWordWrap(True)
                features_label.setMinimumWidth(200)
                features_label.setStyleSheet('font-size: 10pt;')
                desc_layout.addWidget(features_label)

                desc_layout.addStretch()
                about_tab_layout.addWidget(desc_column, 3)

                # Column 2: links/support
                links_column = QWidget(about_tab)
                links_layout = QVBoxLayout(links_column)

                links_title = QLabel(_('<b>Support & Resources</b>'))
                try:
                    links_title.setAlignment(Qt.AlignmentFlag.AlignHCenter)
                except Exception:
                    try:
                        links_title.setAlignment(Qt.AlignHCenter)
                    except Exception:
                        pass
                links_title.setStyleSheet('font-size: 13pt;')
                links_layout.addWidget(links_title)

                def _wire_label(label):
                    try:
                        label.setOpenExternalLinks(False)
                    except Exception:
                        pass
                    try:
                        label.linkActivated.connect(lambda u: __import__('calibre').gui2.open_url(u))
                    except Exception:
                        try:
                            label.linkActivated.connect(lambda u: QDesktopServices.openUrl(QUrl(u)))
                        except Exception:
                            pass

                # Forum link
                forum_link = QLabel('🔗 <a href="https://www.mobileread.com/forums/showthread.php?t=371755" style="text-decoration:none;">' + _('MR help thread') + '</a>')
                _wire_label(forum_link)
                try:
                    forum_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
                except Exception:
                    try:
                        forum_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
                    except Exception:
                        pass
                forum_link.setStyleSheet('QLabel, a { font-size: 11pt; text-decoration: none !important; color: palette(link); }')
                try:
                    forum_link.setAlignment(Qt.AlignmentFlag.AlignHCenter)
                except Exception:
                    try:
                        forum_link.setAlignment(Qt.AlignHCenter)
                    except Exception:
                        pass
                links_layout.addWidget(forum_link)

                # Add vertical spacing between forum and donate links
                try:
                    links_layout.addSpacing(12)
                except Exception:
                    pass

                # Donate link with humorous per-language emoji + message
                # Use plugin UI language if set, otherwise fall back to Calibre's UI language
                _cfg_ui_lang = str(plugin_prefs.get('plugin_ui_language', '') or '').strip()
                if not _cfg_ui_lang:
                    try:
                        from calibre.utils.localization import get_lang
                        _cfg_ui_lang = str(get_lang() or '').strip()
                    except Exception:
                        _cfg_ui_lang = ''
                _cfg_emoji = '❤️'
                _cfg_sushi_langs = {'ja', 'zh', 'zh_CN', 'zh_TW', 'zh_HK', 'ko', 'vi', 'id', 'ms'}
                if _cfg_ui_lang == 'pt':
                    _cfg_emoji = '🧆'
                elif _cfg_ui_lang == 'ar':
                    _cfg_emoji = '🧆'
                elif _cfg_ui_lang in ('tr', 'fa'):
                    _cfg_emoji = '🍢'
                elif _cfg_ui_lang == 'ca':
                    _cfg_emoji = '🧀'
                elif _cfg_ui_lang == 'es':
                    _cfg_emoji = '🍤'
                elif _cfg_ui_lang == 'gl':
                    _cfg_emoji = '🥟'
                elif _cfg_ui_lang == 'fr':
                    _cfg_emoji = '🍷'
                elif _cfg_ui_lang in _cfg_sushi_langs:
                    _cfg_emoji = '🍣'
                elif _cfg_ui_lang == 'it':
                    _cfg_emoji = '🍕'
                elif _cfg_ui_lang in ('pt_BR', 'de'):
                    _cfg_emoji = '🍺'
                elif _cfg_ui_lang in ('ru', 'uk', 'pl', 'cs', 'sk', 'sr', 'hr', 'sl', 'bg'):
                    _cfg_emoji = '🥃'
                donate_html = f'{_cfg_emoji} <a href="https://ko-fi.com/comfy_n" style="text-decoration:none;">' + _('Show appreciation') + '</a><br>'
                donate_link = QLabel(donate_html)
                _wire_label(donate_link)
                try:
                    donate_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
                except Exception:
                    try:
                        donate_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
                    except Exception:
                        pass
                donate_link.setStyleSheet('QLabel, a { font-size: 11pt; text-decoration: none !important; color: palette(link); }')
                try:
                    donate_link.setAlignment(Qt.AlignmentFlag.AlignHCenter)
                except Exception:
                    try:
                        donate_link.setAlignment(Qt.AlignHCenter)
                    except Exception:
                        pass
                links_layout.addWidget(donate_link)

                # QR code (images/qrcode.png) below support link — with easter egg flip
                try:
                    from calibre_plugins.rss_reader.common_icons import get_icon as get_plugin_icon
                except Exception:
                    try:
                        from common_icons import get_icon as get_plugin_icon
                    except Exception:
                        get_plugin_icon = None

                try:
                    if get_plugin_icon is not None:
                        ic = get_plugin_icon('images/qrcode.png')
                    else:
                        ic = None
                except Exception:
                    ic = None

                # Easter egg: also try to load images/pic.png
                _ee_ic = None
                try:
                    if get_plugin_icon is not None:
                        _ee_ic = get_plugin_icon('images/pic.png')
                        if _ee_ic is not None and _ee_ic.isNull():
                            _ee_ic = None
                except Exception:
                    _ee_ic = None

                try:
                    if ic is not None and not ic.isNull():
                        qr = QLabel('')
                        qr._showing_qr = True  # track which image is shown
                        try:
                            qr.setAlignment(Qt.AlignmentFlag.AlignHCenter)
                        except Exception:
                            try:
                                qr.setAlignment(Qt.AlignHCenter)
                            except Exception:
                                pass
                        try:
                            qr.setPixmap(ic.pixmap(170, 170))
                        except Exception:
                            try:
                                qr.setPixmap(ic.pixmap(170))
                            except Exception:
                                pass

                        # Easter egg: click QR code to flip between qrcode.png and pic.png
                        def _make_qr_flip(qr_lbl, qr_icon, ee_icon):
                            def _flip(event):
                                try:
                                    if qr_lbl._showing_qr and ee_icon is not None:
                                        try:
                                            qr_lbl.setPixmap(ee_icon.pixmap(170, 170))
                                        except Exception:
                                            qr_lbl.setPixmap(ee_icon.pixmap(170))
                                        qr_lbl._showing_qr = False
                                        qr_lbl.setToolTip(_('You found the hidden easter egg! \U0001F95A'))
                                    else:
                                        try:
                                            qr_lbl.setPixmap(qr_icon.pixmap(170, 170))
                                        except Exception:
                                            qr_lbl.setPixmap(qr_icon.pixmap(170))
                                        qr_lbl._showing_qr = True
                                        qr_lbl.setToolTip('')
                                    try:
                                        __import__('calibre').gui2.open_url('https://ko-fi.com/comfy_n')
                                    except Exception:
                                        pass
                                except Exception:
                                    pass
                            return _flip

                        try:
                            qr.setCursor(Qt.CursorShape.PointingHandCursor)
                        except Exception:
                            try:
                                qr.setCursor(Qt.PointingHandCursor)
                            except Exception:
                                pass
                        qr.mousePressEvent = _make_qr_flip(qr, ic, _ee_ic)

                        links_layout.addSpacing(8)
                        links_layout.addWidget(qr)
                except Exception:
                    pass

                # QuiteRSS inspiration credit
                try:
                    credit_label = QLabel(
                        '<span style="font-size: 8pt; color: palette(mid);">'
                        + _('UI design inspired by') +
                        ' <a href="https://quiterss.org" style="text-decoration:none; color: palette(mid);">QuiteRSS</a></span>'
                    )
                    _wire_label(credit_label)
                    try:
                        credit_label.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
                    except Exception:
                        try:
                            credit_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
                        except Exception:
                            pass
                    try:
                        credit_label.setAlignment(Qt.AlignmentFlag.AlignHCenter)
                    except Exception:
                        try:
                            credit_label.setAlignment(Qt.AlignHCenter)
                        except Exception:
                            pass
                    credit_label.setWordWrap(True)
                    links_layout.addSpacing(12)
                    links_layout.addWidget(credit_label)
                except Exception:
                    pass

                links_layout.addStretch()
                about_tab_layout.addWidget(links_column, 2)

                about_scroll.setWidget(about_tab)
                about_layout.addWidget(about_scroll)
        except Exception:
            pass

        # Apply basic/advanced visibility
        try:
            if self.show_advanced_cb is not None:
                self.show_advanced_cb.toggled.connect(self._apply_settings_visibility)
        except Exception:
            pass
        try:
            self._apply_settings_visibility(bool(self.show_advanced_cb.isChecked()) if self.show_advanced_cb is not None else False)
        except Exception:
            pass
            # ...existing code...

        try:
            # Gentle shrinking behavior if the dialog is made narrow.
            for w in (self.notify_cb, self.notify_first_cb, self.suppress_fullscreen_cb, self.auto_open_cb, self.load_images_cb, self.export_add_cb, self.auto_tag_cb, self.auto_tag_img_cb, self.auto_tag_audio_cb, self.auto_tag_long_cb):
                try:
                    w.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Fixed)
                except Exception:
                    try:
                        w.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Fixed)
                    except Exception:
                        pass
        except Exception:
            pass

        # Layout reset (affects the main RSS Reader window, not stored plugin prefs)
        try:
            row = QHBoxLayout()

            # Version label — bottom-left, mirrors main dialog footer
            try:
                from calibre_plugins.rss_reader import RSSReaderPlugin as _RSSPlugin
                _cfg_ver_text = 'RSS Reader v' + '.'.join(str(x) for x in _RSSPlugin.version)
            except Exception:
                _cfg_ver_text = 'RSS Reader'
            try:
                from calibre.constants import numeric_version as _cal_ver
                _cfg_ver_text += '   |   calibre ' + ('.'.join(str(x) for x in _cal_ver) if isinstance(_cal_ver, (tuple, list)) else str(_cal_ver))
            except Exception:
                pass
            self._cfg_version_label = QLabel(_cfg_ver_text, self)
            try:
                self._cfg_version_label.setStyleSheet('QLabel { font-size: 9pt; color: palette(mid); padding: 2px 4px; }')
            except Exception:
                pass
            row.addWidget(self._cfg_version_label)

            row.addStretch(1)

            # Maintenance
            self.purge_btn = QPushButton(_('Purge database…'), self)
            self.purge_btn.setToolTip(_('Purge database: remove orphaned cache entries and compact storage'))
            try:
                self.purge_btn.clicked.connect(self._purge_database)
            except Exception:
                pass
            row.addWidget(self.purge_btn)

            self.clear_db_btn = QPushButton(_('Clear database…'), self)
            self.clear_db_btn.setToolTip(_('Delete all feeds, cached items, and history from the current database (keeps the same DB path).'))
            try:
                self.clear_db_btn.clicked.connect(self._clear_database)
            except Exception:
                pass
            row.addWidget(self.clear_db_btn)

            self.restore_settings_btn = QPushButton(_('Restore default settings'), self)
            self.restore_settings_btn.setToolTip(_('Reset preferences to their default values (DB profiles are preserved).'))
            try:
                # Use a familiar refresh icon for reset
                from qt.core import QIcon
            except Exception:
                from PyQt5.Qt import QIcon
            try:
                ic = QIcon.ic('view-refresh.png')
                if ic is not None and not ic.isNull():
                    self.restore_settings_btn.setIcon(ic)
            except Exception:
                pass
            self.restore_settings_btn.clicked.connect(self._restore_default_settings)
            row.addWidget(self.restore_settings_btn)
            self.restore_layout_btn = QPushButton(_('Restore default layout'), self)
            self.restore_layout_btn.setToolTip(_('Reset splitters/columns to the default layout (useful if panes were resized badly).'))
            try:
                ic = QIcon.ic('view-refresh.png')
                if ic is not None and not ic.isNull():
                    self.restore_layout_btn.setIcon(ic)
            except Exception:
                pass
            self.restore_layout_btn.clicked.connect(self._restore_default_layout)
            row.addWidget(self.restore_layout_btn)
            layout.addLayout(row)
            try:
                layout.addSpacing(10)
            except Exception:
                pass

            # Hide maintenance buttons while on the Notifications tab (cleaner UX).
            try:
                if getattr(self, '_tabs', None) is not None:
                    _update_maintenance_buttons_visibility(int(self._tabs.currentIndex()))
            except Exception:
                pass
        except Exception:
            self.restore_settings_btn = None
            self.restore_layout_btn = None
            self.purge_btn = None
            self.clear_db_btn = None

        # When embedded as a calibre plugin configuration widget, calibre
        # provides its own OK/Cancel buttons. Adding another button box causes
        # duplicate OK/Cancel rows.
        if not self._embed_mode:
            buttons = QDialogButtonBox(
                QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel,
                self,
            )
            buttons.accepted.connect(self.accept)
            buttons.rejected.connect(self.reject)
            layout.addWidget(buttons)

    def _restore_default_settings(self):
        try:
            res = QMessageBox.question(self, _('Restore default settings'), _('This will reset all preferences to their default values. Continue?'), QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
        except Exception:
            try:
                QMessageBox.warning(self, _('Restore default settings'), _('This will reset all preferences to their default values. Continue?'))
            except Exception:
                pass
            return
        if res != QMessageBox.StandardButton.Yes:
            return
        try:
            # Do not reset DB profile manager settings from the global reset.
            preserve_keys = {
                'db_default_path',
                'db_default_readonly',
                'db_default_mirror',
                'db_profiles',
                'db_profiles_active',
            }
            # Apply defaults from plugin_prefs.defaults
            for key, value in list(plugin_prefs.defaults.items()):
                if key in preserve_keys:
                    continue
                try:
                    plugin_prefs[key] = value
                except Exception:
                    # best-effort: continue on failure
                    pass
            try:
                QMessageBox.information(self, _('Restore default settings'), _('Default settings have been restored. Please restart RSS Reader for changes to take effect.'))
            except Exception:
                pass
            self.reject()
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Error'), _('Failed to restore default settings: %s') % str(e))
            except Exception:
                pass

    def _init_failed_feeds_tab(self, failed_layout):
        if failed_layout is None:
            return
        try:
            info = QLabel(_(
                'Shows a history of failed feed fetch/parse attempts recorded during updates.\n'
                'Tip: double-click a row to re-check that URL now.'
            ), self)
            try:
                info.setWordWrap(True)
            except Exception:
                pass
            failed_layout.addWidget(info)
        except Exception:
            pass

        # Controls
        try:
            row = QHBoxLayout()
            self.failed_count_lbl = QLabel('', self)
            row.addWidget(self.failed_count_lbl)
            row.addStretch(1)

            self.failed_limit_spin = QSpinBox(self)
            self.failed_limit_spin.setRange(50, 20000)
            self.failed_limit_spin.setValue(int(plugin_prefs.get('failed_feeds_history_page_size', 500) or 500))
            try:
                self.failed_limit_spin.setToolTip(_('How many rows to load into the table (newest first).'))
            except Exception:
                pass
            row.addWidget(QLabel(_('Show:'), self))
            row.addWidget(self.failed_limit_spin)

            self.failed_refresh_btn = QPushButton(_('Refresh'), self)
            self.failed_clear_btn = QPushButton(_('Clear history'), self)
            try:
                self.failed_clear_btn.setToolTip(_('Deletes all failed-feed history entries.'))
            except Exception:
                pass
            row.addWidget(self.failed_refresh_btn)
            row.addWidget(self.failed_clear_btn)
            failed_layout.addLayout(row)
        except Exception:
            self.failed_count_lbl = None
            self.failed_limit_spin = None
            self.failed_refresh_btn = None
            self.failed_clear_btn = None

        # Filter
        try:
            frow = QHBoxLayout()
            frow.addWidget(QLabel(_('Filter:'), self))
            self.failed_filter_edit = QLineEdit(self)
            try:
                self.failed_filter_edit.setPlaceholderText(_('Search…'))
            except Exception:
                pass
            try:
                self.failed_filter_edit.setClearButtonEnabled(True)
            except Exception:
                pass
            frow.addWidget(self.failed_filter_edit, 1)
            failed_layout.addLayout(frow)
        except Exception:
            self.failed_filter_edit = None

        # Table
        try:
            self.failed_table = QTableWidget(self)
            self.failed_table.setColumnCount(4)
            self.failed_table.setHorizontalHeaderLabels([_('When'), _('Title'), _('URL'), _('Error')])
            self.failed_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
            self.failed_table.setSelectionMode(QAbstractItemView.SelectionMode.ExtendedSelection)
            self.failed_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)
            try:
                self.failed_table.setSortingEnabled(True)
            except Exception:
                pass
            try:
                self.failed_table.setAlternatingRowColors(True)
            except Exception:
                pass
            try:
                self.failed_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            except Exception:
                pass
            try:
                hh = self.failed_table.horizontalHeader()
                hh.setSectionResizeMode(0, QHeaderView.ResizeMode.ResizeToContents)
                hh.setSectionResizeMode(1, QHeaderView.ResizeMode.ResizeToContents)
                hh.setSectionResizeMode(2, QHeaderView.ResizeMode.Stretch)
                hh.setSectionResizeMode(3, QHeaderView.ResizeMode.Stretch)
            except Exception:
                try:
                    self.failed_table.horizontalHeader().setStretchLastSection(True)
                except Exception:
                    pass
            failed_layout.addWidget(self.failed_table, 1)
        except Exception:
            self.failed_table = None

        # Wiring
        try:
            if self.failed_refresh_btn is not None:
                self.failed_refresh_btn.clicked.connect(self._load_failed_feeds_history)
        except Exception:
            pass
        try:
            if self.failed_clear_btn is not None:
                self.failed_clear_btn.clicked.connect(self._on_clear_failed_feeds_history)
        except Exception:
            pass
        try:
            if self.failed_limit_spin is not None:
                self.failed_limit_spin.valueChanged.connect(lambda v: self._load_failed_feeds_history())
        except Exception:
            pass
        try:
            if self.failed_table is not None:
                self.failed_table.cellDoubleClicked.connect(self._on_failed_history_double_clicked)
        except Exception:
            pass
        try:
            if self.failed_filter_edit is not None:
                self.failed_filter_edit.textChanged.connect(self._apply_failed_feeds_filter)
        except Exception:
            pass
        try:
            if self.failed_table is not None:
                self.failed_table.customContextMenuRequested.connect(self._on_failed_feeds_context_menu)
        except Exception:
            pass

        # Initial load
        try:
            self._load_failed_feeds_history()
        except Exception:
            pass

    def _init_ai_errors_tab(self, ai_errors_layout):
        if ai_errors_layout is None:
            return

        try:
            info = QLabel(_(
                'Shows errors raised by calibre AI providers/backends when RSS Reader tries to talk to them.\n'
                'This is useful for diagnosing authentication problems, HTTP errors, token limits, etc.'
            ), self)
            try:
                info.setWordWrap(True)
            except Exception:
                pass
            ai_errors_layout.addWidget(info)
        except Exception:
            pass

        # Controls
        try:
            row = QHBoxLayout()
            self.ai_errors_count_lbl = QLabel('', self)
            row.addWidget(self.ai_errors_count_lbl)
            row.addStretch(1)
            self.ai_errors_refresh_btn = QPushButton(_('Refresh'), self)
            self.ai_errors_copy_btn = QPushButton(_('Copy'), self)
            self.ai_errors_clear_btn = QPushButton(_('Clear'), self)
            try:
                self.ai_errors_clear_btn.setToolTip(_('Deletes all logged AI errors.'))
            except Exception:
                pass
            row.addWidget(self.ai_errors_refresh_btn)
            row.addWidget(self.ai_errors_copy_btn)
            row.addWidget(self.ai_errors_clear_btn)
            ai_errors_layout.addLayout(row)
        except Exception:
            self.ai_errors_count_lbl = None
            self.ai_errors_refresh_btn = None
            self.ai_errors_copy_btn = None
            self.ai_errors_clear_btn = None

        try:
            self.ai_errors_view = QTextBrowser(self)
            try:
                self.ai_errors_view.setOpenExternalLinks(False)
            except Exception:
                pass
            ai_errors_layout.addWidget(self.ai_errors_view, 1)
        except Exception:
            self.ai_errors_view = None

        def _format_log():
            try:
                log = list(plugin_prefs.get('ai_error_log', []) or [])
            except Exception:
                log = []
            lines = []
            for e in log:
                try:
                    ts = str((e or {}).get('ts') or '')
                    where = str((e or {}).get('where') or '')
                    err = str((e or {}).get('err') or '')
                    details = str((e or {}).get('details') or '')
                except Exception:
                    continue
                block = []
                if ts or where:
                    block.append(f'[{ts}] {where}'.strip())
                if err:
                    block.append(err)
                if details:
                    block.append(details)
                if block:
                    lines.append('\n'.join(block))
            return '\n\n' + ('\n\n' + ('-' * 60) + '\n\n').join(lines) if lines else _('No AI errors logged.')

        def _refresh():
            text = ''
            try:
                log = list(plugin_prefs.get('ai_error_log', []) or [])
            except Exception:
                log = []
            try:
                if getattr(self, 'ai_errors_count_lbl', None) is not None:
                    self.ai_errors_count_lbl.setText(_('%d entries') % int(len(log)))
            except Exception:
                pass
            try:
                text = _format_log()
            except Exception:
                text = ''
            try:
                if getattr(self, 'ai_errors_view', None) is not None:
                    self.ai_errors_view.setPlainText(text)
            except Exception:
                pass
            return text

        def _copy():
            try:
                txt = _refresh()
            except Exception:
                txt = ''
            try:
                cb = QApplication.clipboard()
                cb.setText(txt or '')
            except Exception:
                pass

        def _clear():
            try:
                plugin_prefs['ai_error_log'] = []
                try:
                    commit = getattr(plugin_prefs, 'commit', None)
                    if callable(commit):
                        commit()
                except Exception:
                    pass
            except Exception:
                pass
            _refresh()

        try:
            if self.ai_errors_refresh_btn is not None:
                self.ai_errors_refresh_btn.clicked.connect(_refresh)
        except Exception:
            pass
        try:
            if self.ai_errors_copy_btn is not None:
                self.ai_errors_copy_btn.clicked.connect(_copy)
        except Exception:
            pass
        try:
            if self.ai_errors_clear_btn is not None:
                self.ai_errors_clear_btn.clicked.connect(_clear)
        except Exception:
            pass

        try:
            _refresh()
        except Exception:
            pass

    def _init_notifications_tab(self, notifications_page, notifications_layout, make_group_fn):
        if notifications_page is None or notifications_layout is None:
            return

        # Keep this tab scrollable to avoid forcing dialog growth
        try:
            from qt.core import QScrollArea
        except Exception:
            try:
                from PyQt5.QtWidgets import QScrollArea
            except Exception:
                QScrollArea = None
        if QScrollArea is None:
            return

        class _SwatchButton(QPushButton):
            def __init__(self, title, initial='#000000', parent=None):
                QPushButton.__init__(self, parent)
                self._title = title
                self._color = str(initial or '').strip() or '#000000'
                try:
                    self.setCursor(Qt.CursorShape.PointingHandCursor)
                except Exception:
                    pass
                self.clicked.connect(self._choose)
                self._refresh()

            def color(self):
                return str(self._color or '').strip()

            def set_color(self, c):
                c = str(c or '').strip()
                if c:
                    self._color = c
                    self._refresh()

            def _refresh(self):
                self.setText(self._color)
                try:
                    # Choose a readable text color based on background luminance.
                    fg = None
                    try:
                        qc = QColor(self._color)
                        if qc is not None and qc.isValid():
                            r = int(qc.red())
                            g = int(qc.green())
                            b = int(qc.blue())
                            lum = (0.2126 * r) + (0.7152 * g) + (0.0722 * b)
                            fg = '#000000' if lum >= 140 else '#FFFFFF'
                    except Exception:
                        fg = None
                    css = 'padding:4px 10px; text-align:left; border:1px solid palette(mid);'
                    css += ' background:%s;' % self._color
                    if fg:
                        css += ' color:%s;' % fg
                    self.setStyleSheet(css)
                except Exception:
                    pass

            def _choose(self):
                try:
                    initial = QColor(self._color)
                except Exception:
                    initial = QColor()
                c = None
                # Prefer the non-native dialog so labels remain legible under
                # theme/palette overrides (the native Windows dialog can end up
                # with low-contrast label text).
                try:
                    dlg = QColorDialog(self)
                    try:
                        dlg.setCurrentColor(initial)
                    except Exception:
                        pass
                    try:
                        dlg.setWindowTitle(self._title)
                    except Exception:
                        pass
                    try:
                        dlg.setOption(QColorDialog.ColorDialogOption.DontUseNativeDialog, True)
                    except Exception:
                        try:
                            dlg.setOption(QColorDialog.DontUseNativeDialog, True)
                        except Exception:
                            pass
                    try:
                        ok = bool(dlg.exec())
                    except Exception:
                        ok = bool(dlg.exec_())
                    if ok:
                        try:
                            c = dlg.currentColor()
                        except Exception:
                            c = None
                except Exception:
                    try:
                        c = QColorDialog.getColor(initial, self, self._title)
                    except Exception:
                        c = QColorDialog.getColor(initial)
                try:
                    if c is not None and c.isValid():
                        self._color = str(c.name())
                        self._refresh()
                        try:
                            cb = getattr(self.window(), '_on_popup_color_changed', None)
                            if cb is not None:
                                cb()
                        except Exception:
                            pass
                except Exception:
                    pass

        inner = QWidget(notifications_page)
        inner_layout = QVBoxLayout(inner)
        inner_layout.setContentsMargins(0, 0, 0, 0)

        scroll = QScrollArea(notifications_page)
        scroll.setWidgetResizable(True)
        try:
            scroll.setFrameShape(QScrollArea.NoFrame)
        except Exception:
            pass
        scroll.setWidget(inner)
        notifications_layout.addWidget(scroll, 1)

        popup_gb, popup_form = make_group_fn(_('Popup notifications'), parent_widget=inner)

        self.popup_enable_cb = QCheckBox(_('Enable in-app popup notifications'), inner)
        self.popup_enable_cb.setChecked(bool(plugin_prefs.get('notify_in_app_popup', False)))
        popup_form.addRow(self.popup_enable_cb)

        self.popup_theme_combo = QComboBox(inner)
        self.popup_theme_combo.addItem(_('Auto (follow calibre theme)'), 'auto')
        self.popup_theme_combo.addItem(_('Light'), 'light')
        self.popup_theme_combo.addItem(_('Dark'), 'dark')

        # Back-compat: treat dark_high_contrast as dark.
        cur_theme = str(plugin_prefs.get('popup_theme', 'dark') or 'dark').strip().lower()
        if cur_theme in ('dark_high_contrast', 'dark-hc', 'high_contrast'):
            cur_theme = 'dark'
        if cur_theme not in ('auto', 'light', 'dark'):
            cur_theme = 'dark'
        try:
            idx = self.popup_theme_combo.findData(cur_theme)
            if idx >= 0:
                self.popup_theme_combo.setCurrentIndex(idx)
        except Exception:
            pass
        popup_form.addRow(_('Popup theme:'), self.popup_theme_combo)

        self.popup_position_combo = QComboBox(inner)
        self.popup_position_combo.addItem(_('Bottom right'), 'bottom_right')
        self.popup_position_combo.addItem(_('Bottom left'), 'bottom_left')
        self.popup_position_combo.addItem(_('Top right'), 'top_right')
        self.popup_position_combo.addItem(_('Top left'), 'top_left')
        self.popup_position_combo.addItem(_('Bottom center'), 'bottom_center')
        self.popup_position_combo.addItem(_('Top center'), 'top_center')
        self.popup_position_combo.addItem(_('Right middle'), 'right_middle')
        self.popup_position_combo.addItem(_('Left middle'), 'left_middle')
        cur_pos = str(plugin_prefs.get('popup_position', 'bottom_right') or 'bottom_right').strip().lower()
        try:
            idx = self.popup_position_combo.findData(cur_pos)
            if idx >= 0:
                self.popup_position_combo.setCurrentIndex(idx)
        except Exception:
            pass
        popup_form.addRow(_('Popup placement:'), self.popup_position_combo)

        self.popup_margin_spin = QSpinBox(inner)
        self.popup_margin_spin.setRange(0, 200)
        self.popup_margin_spin.setSuffix(_(' px'))
        self.popup_margin_spin.setValue(int(plugin_prefs.get('popup_margin_px', 10) or 10))
        popup_form.addRow(_('Screen margin:'), self.popup_margin_spin)

        self.popup_width_spin = QSpinBox(inner)
        self.popup_width_spin.setRange(0, 4000)
        self.popup_width_spin.setSuffix(_(' px'))
        try:
            self.popup_width_spin.setToolTip(_('0 = automatic width based on screen size'))
        except Exception:
            pass
        self.popup_width_spin.setValue(int(plugin_prefs.get('popup_width_px', 0) or 0))
        popup_form.addRow(_('Popup width:'), self.popup_width_spin)

        self.popup_item_height_spin = QSpinBox(inner)
        self.popup_item_height_spin.setRange(0, 200)
        self.popup_item_height_spin.setSuffix(_(' px'))
        try:
            self.popup_item_height_spin.setToolTip(_('Minimum height for each article row in the popup (0 = automatic).'))
        except Exception:
            pass
        self.popup_item_height_spin.setValue(int(plugin_prefs.get('popup_item_min_height_px', 0) or 0))
        popup_form.addRow(_('Item height (min):'), self.popup_item_height_spin)

        self.popup_item_padding_spin = QSpinBox(inner)
        self.popup_item_padding_spin.setRange(0, 40)
        self.popup_item_padding_spin.setSuffix(_(' px'))
        try:
            self.popup_item_padding_spin.setToolTip(_('Vertical padding inside each article row (affects spacing between rows).'))
        except Exception:
            pass
        self.popup_item_padding_spin.setValue(int(plugin_prefs.get('popup_item_padding_v_px', 2) or 2))
        popup_form.addRow(_('Item padding (vertical):'), self.popup_item_padding_spin)

        self.popup_items_per_page_spin = QSpinBox(inner)
        self.popup_items_per_page_spin.setRange(3, 100)
        self.popup_items_per_page_spin.setValue(int(plugin_prefs.get('popup_max_items_per_page', 10) or 10))
        popup_form.addRow(_('Max items per page:'), self.popup_items_per_page_spin)

        self.popup_items_total_spin = QSpinBox(inner)
        self.popup_items_total_spin.setRange(1, 1000)
        self.popup_items_total_spin.setValue(int(plugin_prefs.get('popup_max_items_total', 40) or 40))
        popup_form.addRow(_('Max items total:'), self.popup_items_total_spin)

        self.popup_close_after_spin = QSpinBox(inner)
        self.popup_close_after_spin.setRange(0, 600)
        self.popup_close_after_spin.setSuffix(_(' seconds'))
        try:
            self.popup_close_after_spin.setToolTip(_('0 disables auto-close'))
        except Exception:
            pass
        try:
            secs = int(int(plugin_prefs.get('popup_timeout_ms', 15000) or 15000) / 1000)
        except Exception:
            secs = 15
        self.popup_close_after_spin.setValue(max(0, secs))
        popup_form.addRow(_('Close after:'), self.popup_close_after_spin)

        # Preview button (uses current values, does not save)
        try:
            self.popup_preview_btn = QPushButton(_('Preview popup'), inner)
            self.popup_preview_btn.setToolTip(_('Show a test notification popup using the current values on this tab (without saving).'))
        except Exception:
            self.popup_preview_btn = None

        def _preview_popup():
            import traceback as _tb
            try:
                from calibre.gui2 import error_dialog as _error_dialog
            except Exception:
                _error_dialog = None
            PopupNotification = None
            tb1 = ''
            tb2 = ''
            try:
                from calibre_plugins.rss_reader.popup_notifications import PopupNotification as _PN
                PopupNotification = _PN
            except Exception:
                tb1 = _tb.format_exc()
                try:
                    from popup_notifications import PopupNotification as _PN2
                    PopupNotification = _PN2
                except Exception:
                    tb2 = _tb.format_exc()
            if PopupNotification is None:
                try:
                    if _error_dialog is not None:
                        det = ''
                        try:
                            det = (tb1 or '').strip() + ('\n\n' if (tb1 and tb2) else '') + (tb2 or '').strip()
                        except Exception:
                            det = ''
                        if not det:
                            try:
                                det = 'PopupNotification import failed, but no traceback was captured.'
                            except Exception:
                                det = ''
                        _error_dialog(self, _('RSS Reader – Popup preview'), _('Could not import PopupNotification (see details).'),
                                     det_msg=det, show=True)
                except Exception:
                    pass
                return

            try:
                effective = str(getattr(self.popup_theme_combo, 'currentData', lambda: None)() or 'dark').strip().lower()
            except Exception:
                effective = 'dark'
            if effective not in ('auto', 'light', 'dark'):
                effective = 'dark'
            if effective == 'auto':
                try:
                    pal = QApplication.palette()
                except Exception:
                    pal = None
                try:
                    is_dark = bool(pal is not None and pal.color(pal.Window).value() < 128)
                except Exception:
                    is_dark = False
                effective = 'dark' if is_dark else 'light'

            # Use the matching palette overrides for the effective theme
            try:
                colors_override = dict((getattr(self, '_popup_color_overrides', {}) or {}).get(effective, {}) or {})
            except Exception:
                colors_override = {}

            # Prefer a dedicated key for link styling to avoid palette conflicts.
            try:
                if isinstance(colors_override, dict):
                    if 'clickable_url' not in colors_override and str(colors_override.get('link') or '').strip():
                        colors_override['clickable_url'] = str(colors_override.get('link') or '').strip()
            except Exception:
                pass

            try:
                pos = str(getattr(self.popup_position_combo, 'currentData', lambda: None)() or 'bottom_right').strip().lower()
            except Exception:
                pos = 'bottom_right'
            try:
                margin = int(self.popup_margin_spin.value())
            except Exception:
                margin = 10
            try:
                width_px = int(self.popup_width_spin.value())
            except Exception:
                width_px = 0
            try:
                item_min_h = int(self.popup_item_height_spin.value())
            except Exception:
                item_min_h = 0
            try:
                item_pad_v = int(self.popup_item_padding_spin.value())
            except Exception:
                item_pad_v = 2
            try:
                per_page = int(self.popup_items_per_page_spin.value())
            except Exception:
                per_page = 10
            try:
                secs = int(self.popup_close_after_spin.value())
            except Exception:
                secs = 15
            timeout_ms = max(0, secs) * 1000

            entries = [
                {'_is_feed_header': True, 'feed_title': _('Preview: Calibre News')},
                {
                    'feed_id': 'preview_feed_a',
                    'feed_title': _('Preview: Calibre News'),
                    'item_id': 'preview_item_a1',
                    'item_title': _('Themed popup notifications (preview)'),
                    'link': 'https://example.com/preview/a1',
                },
                {
                    'feed_id': 'preview_feed_a',
                    'feed_title': _('Preview: Calibre News'),
                    'item_id': 'preview_item_a2',
                    'item_title': _('Open / Mark read buttons styling'),
                    'link': 'https://example.com/preview/a2',
                },
                {'_is_feed_header': True, 'feed_title': _('Preview: Standard Ebooks')},
                {
                    'feed_id': 'preview_feed_b',
                    'feed_title': _('Preview: Standard Ebooks'),
                    'item_id': 'preview_item_b1',
                    'item_title': _('Second feed header + longer text overflow check'),
                    'link': 'https://example.com/preview/b1',
                },
            ]

            try:
                # Keep reference to avoid GC
                try:
                    if getattr(self, '_popup_preview_instance', None) is not None:
                        self._popup_preview_instance.close()
                except Exception:
                    pass
                self._popup_preview_instance = PopupNotification(
                    entries,
                    total_new=3,
                    max_items_per_page=per_page,
                    timeout_ms=timeout_ms,
                    theme_name=effective,
                    colors_override=colors_override,
                    position=pos,
                    margin_px=margin,
                    width_px=width_px,
                    item_min_height_px=item_min_h,
                    item_padding_v_px=item_pad_v,
                    parent=self,
                )
                self._popup_preview_instance.show()
                self._popup_preview_instance.raise_()
            except Exception:
                try:
                    if _error_dialog is not None:
                        _error_dialog(self, _('RSS Reader – Popup preview'), _('Failed to create the popup preview.'),
                                     det_msg=_tb.format_exc(), show=True)
                except Exception:
                    pass

        try:
            if self.popup_preview_btn is not None:
                self.popup_preview_btn.clicked.connect(_preview_popup)
                popup_form.addRow(self.popup_preview_btn)
        except Exception:
            pass

        colors_gb, colors_form = make_group_fn(_('Popup colors (basic)'), parent_widget=inner)

        self.popup_palette_combo = QComboBox(inner)
        self.popup_palette_combo.addItem(_('Light palette'), 'light')
        # Dark maps to the High Contrast palette internally.
        self.popup_palette_combo.addItem(_('Dark palette'), 'dark')
        colors_form.addRow(_('Edit colors for:'), self.popup_palette_combo)

        # Initial state: match the palette editor to the active popup theme.
        try:
            eff = str(getattr(self.popup_theme_combo, 'currentData', lambda: None)() or 'dark').strip().lower()
        except Exception:
            eff = 'dark'
        if eff not in ('auto', 'light', 'dark'):
            eff = 'dark'
        if eff == 'auto':
            try:
                pal = QApplication.palette()
            except Exception:
                pal = None
            try:
                is_dark = bool(pal is not None and pal.color(pal.Window).value() < 128)
            except Exception:
                is_dark = False
            eff = 'dark' if is_dark else 'light'
        try:
            idx = self.popup_palette_combo.findData(eff)
            if idx >= 0:
                self.popup_palette_combo.setCurrentIndex(idx)
        except Exception:
            pass

        # Keep per-palette edits in memory until OK is pressed
        self._popup_color_overrides = {}
        for tn in ('light', 'dark'):
            key = 'popup_theme_colors_%s' % ('dark_high_contrast' if tn == 'dark' else tn)
            try:
                d = dict(plugin_prefs.get(key, {}) or {})
            except Exception:
                d = {}
            if not d:
                try:
                    d = dict(plugin_prefs.defaults.get(key, {}) or {})
                except Exception:
                    d = {}
            self._popup_color_overrides[tn] = {
                'bg': str(d.get('bg') or '').strip() or '#000000',
                'link': str(d.get('clickable_url') or d.get('link') or '').strip() or '#0000FF',
            }

        self.popup_bg_swatch = _SwatchButton(_('Popup background color'), parent=inner)
        self.popup_link_swatch = _SwatchButton(_('Popup link color'), parent=inner)
        colors_form.addRow(_('Background:'), self.popup_bg_swatch)
        colors_form.addRow(_('Link:'), self.popup_link_swatch)

        self.popup_reset_colors_btn = QPushButton(_('Reset bg/link to defaults'), inner)

        def _reset_colors():
            try:
                tn = str(self.popup_palette_combo.currentData() or '').strip().lower()
            except Exception:
                tn = 'dark'
            if tn not in ('light', 'dark'):
                tn = 'dark'
            key = 'popup_theme_colors_%s' % ('dark_high_contrast' if tn == 'dark' else tn)
            try:
                d = dict(plugin_prefs.defaults.get(key, {}) or {})
            except Exception:
                d = {}
            bg = str(d.get('bg') or '').strip() or '#000000'
            link = str(d.get('clickable_url') or d.get('link') or '').strip() or '#0000FF'
            try:
                self._popup_color_overrides[tn] = {'bg': bg, 'link': link}
            except Exception:
                pass
            self._load_popup_color_swatches()

        try:
            self.popup_reset_colors_btn.clicked.connect(_reset_colors)
        except Exception:
            pass
        colors_form.addRow(self.popup_reset_colors_btn)

        def _load_popup_color_swatches():
            try:
                tn = str(self.popup_palette_combo.currentData() or '').strip().lower()
            except Exception:
                tn = 'dark'
            if tn not in ('light', 'dark'):
                tn = 'dark'
            vals = dict((self._popup_color_overrides or {}).get(tn, {}) or {})
            bg = str(vals.get('bg') or '').strip() or '#000000'
            link = str(vals.get('link') or '').strip() or '#0000FF'
            try:
                self.popup_bg_swatch.set_color(bg)
                self.popup_link_swatch.set_color(link)
            except Exception:
                pass

        self._load_popup_color_swatches = _load_popup_color_swatches

        def _on_popup_color_changed():
            try:
                tn = str(self.popup_palette_combo.currentData() or '').strip().lower()
            except Exception:
                tn = 'dark'
            if tn not in ('light', 'dark'):
                tn = 'dark'
            try:
                self._popup_color_overrides[tn] = {
                    'bg': str(self.popup_bg_swatch.color() or '').strip(),
                    'link': str(self.popup_link_swatch.color() or '').strip(),
                }
            except Exception:
                pass

        self._on_popup_color_changed = _on_popup_color_changed

        try:
            self.popup_palette_combo.currentIndexChanged.connect(lambda _i=None: self._load_popup_color_swatches())
        except Exception:
            pass
        self._load_popup_color_swatches()

        inner_layout.addWidget(popup_gb)
        inner_layout.addWidget(colors_gb)
        inner_layout.addStretch(1)
        try:
            if self.failed_clear_btn is not None:
                self.failed_clear_btn.clicked.connect(self._clear_failed_feeds_history)
        except Exception:
            pass
        try:
            if self.failed_limit_spin is not None:
                self.failed_limit_spin.valueChanged.connect(lambda v: self._load_failed_feeds_history())
        except Exception:
            pass
        try:
            if self.failed_table is not None:
                self.failed_table.cellDoubleClicked.connect(self._on_failed_history_double_clicked)
        except Exception:
            pass

        try:
            if self.failed_filter_edit is not None:
                self.failed_filter_edit.textChanged.connect(self._apply_failed_feeds_filter)
        except Exception:
            pass

        try:
            if self.failed_table is not None:
                self.failed_table.customContextMenuRequested.connect(self._on_failed_feeds_context_menu)
        except Exception:
            pass

        # Initial load
        try:
            self._load_failed_feeds_history()
        except Exception:
            pass

    def _apply_failed_feeds_filter(self, text):
        table = getattr(self, 'failed_table', None)
        if table is None:
            return
        try:
            q = (text or '').strip().lower()
        except Exception:
            q = ''
        try:
            for row in range(table.rowCount()):
                show = (q == '')
                if not show:
                    for col in range(table.columnCount()):
                        it = table.item(row, col)
                        if it is not None and q in str(it.text() or '').lower():
                            show = True
                            break
                table.setRowHidden(row, not show)
        except Exception:
            pass

    def _copy_failed_feeds_rows(self):
        table = getattr(self, 'failed_table', None)
        if table is None:
            return
        try:
            selected_rows = set()
            for item in table.selectedItems():
                try:
                    selected_rows.add(int(item.row()))
                except Exception:
                    pass
            if not selected_rows:
                return
            lines = []
            for r in sorted(selected_rows):
                parts = []
                for c in range(table.columnCount()):
                    it = table.item(r, c)
                    parts.append(str(it.text() if it is not None else ''))
                lines.append('\t'.join(parts))
            QApplication.clipboard().setText('\n'.join(lines))
        except Exception:
            pass

    def _on_failed_feeds_context_menu(self, pos):
        table = getattr(self, 'failed_table', None)
        if table is None:
            return
        try:
            menu = QMenu(table)
            act_copy = menu.addAction(_('Copy row data'))
            act_copy.triggered.connect(self._copy_failed_feeds_rows)
            menu.popup(table.viewport().mapToGlobal(pos))
        except Exception:
            pass

    def _load_failed_feeds_history(self):
        if getattr(self, 'failed_table', None) is None:
            return
        try:
            from calibre_plugins.rss_reader import rss_db
        except Exception:
            return

        # Ensure we are pointing at an appropriate DB without ever
        # silently "falling back" away from the DB currently in use.
        #
        # Rules:
        # - If this process is already pointing at a non-builtin DB, treat
        #   that as authoritative and do *not* override it from here.
        # - Otherwise, prefer the active profile path, then the configured
        #   default DB path, and only finally the builtin suggested path.
        try:
            desired_path = ''
            desired_ro = False

            try:
                cur_path = str(rss_db.db_path() or '').strip()
            except Exception:
                cur_path = ''

            try:
                builtin_path = str(getattr(rss_db, 'suggested_default_db_path', lambda: '')() or '').strip()
            except Exception:
                builtin_path = ''

            # If we are already using a non-builtin DB in this session, do not
            # change it from the settings dialog. This avoids "silent reset"
            # behaviour when, for example, viewing history while a one-off DB
            # is active.
            try:
                if cur_path and builtin_path and cur_path not in ('', builtin_path):
                    desired_path = cur_path
                    try:
                        desired_ro = bool(getattr(rss_db, 'DB_READONLY', False))
                    except Exception:
                        desired_ro = False
                else:
                    # No explicit DB in use yet (or we're still on the builtin
                    # path): follow the persisted configuration.
                    try:
                        active_id = str(plugin_prefs.get('db_profiles_active') or '').strip()
                    except Exception:
                        active_id = ''

                    if active_id:
                        try:
                            for p in (plugin_prefs.get('db_profiles') or []):
                                try:
                                    if str(p.get('id') or '') != active_id:
                                        continue
                                    desired_path = str(p.get('path') or '').strip()
                                    desired_ro = bool(p.get('readonly', False) or p.get('mirror', False))
                                    break
                                except Exception:
                                    continue
                        except Exception:
                            pass

                    if not desired_path:
                        try:
                            desired_path = str(plugin_prefs.get('db_default_path') or '').strip()
                        except Exception:
                            desired_path = ''
                        try:
                            desired_ro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
                        except Exception:
                            desired_ro = False

                    if not desired_path:
                        desired_path = builtin_path

            except Exception:
                pass

            # Only steer rss_db when we have a concrete target path that
            # differs from the current one, and only in the "safe" cases
            # above. When cur_path already points at a non-builtin DB, we
            # leave it unchanged.
            try:
                if desired_path and desired_path != cur_path:
                    try:
                        rss_db.set_db_path(desired_path, readonly=desired_ro)
                    except Exception:
                        pass
            except Exception:
                pass
        except Exception:
            pass

        # Ensure schema exists (older DBs may lack the history table).
        try:
            rss_db.ensure_ready()
        except Exception:
            pass

        # Purge old entries first (only applies to the canonical history table).
        try:
            keep_days = int(plugin_prefs.get('failed_feeds_history_retention_days', 60) or 60)
        except Exception:
            keep_days = 60
        try:
            # If the canonical table is empty and we're reading from a legacy table,
            # do not purge (we don't know its retention semantics).
            if int(rss_db.count_feed_fail_history() or 0) > 0:
                rss_db.purge_feed_fail_history(retention_days=keep_days)
        except Exception:
            pass

        try:
            limit = int(self.failed_limit_spin.value()) if getattr(self, 'failed_limit_spin', None) is not None else 500
        except Exception:
            limit = 500

        try:
            rows = rss_db.get_error_log_history(limit=limit, offset=0) or []
        except Exception:
            rows = []

        try:
            total = rss_db.count_error_log_history()
        except Exception:
            total = None

        try:
            import datetime
            self.failed_table.setRowCount(len(rows))
            for r, row in enumerate(rows):
                ts = int(row.get('ts') or 0)
                when = ''
                try:
                    if ts:
                        when = datetime.datetime.fromtimestamp(ts).strftime('%Y-%m-%d %H:%M')
                except Exception:
                    when = str(ts or '')

                title = str(row.get('title') or '')
                url = str(row.get('url') or '')
                err = str(row.get('error') or '')

                it_when = QTableWidgetItem(when)
                it_title = QTableWidgetItem(title)
                it_url = QTableWidgetItem(url)
                it_err = QTableWidgetItem(err)
                try:
                    it_url.setData(Qt.ItemDataRole.UserRole, url)
                except Exception:
                    try:
                        it_url.setData(Qt.UserRole, url)
                    except Exception:
                        pass

                self.failed_table.setItem(r, 0, it_when)
                self.failed_table.setItem(r, 1, it_title)
                self.failed_table.setItem(r, 2, it_url)
                self.failed_table.setItem(r, 3, it_err)

            try:
                self.failed_table.resizeRowsToContents()
            except Exception:
                pass
        except Exception:
            pass

        try:
            if getattr(self, 'failed_count_lbl', None) is not None:
                if total is None:
                    self.failed_count_lbl.setText(_('Loaded %(n)d row(s)') % {'n': len(rows)})
                else:
                    self.failed_count_lbl.setText(_('Showing %(n)d of %(t)d') % {'n': len(rows), 't': int(total)})
        except Exception:
            pass

        try:
            plugin_prefs['failed_feeds_history_page_size'] = int(limit)
        except Exception:
            pass

    def _clear_failed_feeds_history(self):
        try:
            res = QMessageBox.question(
                self,
                _('Clear failed feeds history'),
                _('This will delete all failed-feed history entries. Continue?'),
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
            )
        except Exception:
            return
        if res != QMessageBox.StandardButton.Yes:
            return

        try:
            from calibre_plugins.rss_reader import rss_db
            rss_db.clear_error_log_history()
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Error'), _('Failed to clear history: %s') % str(e))
            except Exception:
                pass
            return

        try:
            self._load_failed_feeds_history()
        except Exception:
            pass

    def _on_failed_history_double_clicked(self, row, col):
        try:
            it = self.failed_table.item(row, 2)
            if it is None:
                return
            try:
                url = it.data(Qt.ItemDataRole.UserRole)
            except Exception:
                url = it.data(Qt.UserRole)
            url = str(url or '').strip()
            if not url:
                return
            self._recheck_failed_url(url)
        except Exception:
            pass

    def _recheck_failed_url(self, url):
        try:
            from calibre_plugins.rss_reader.rss import fetch_url, parse_feed
            from calibre_plugins.rss_reader.recipe_utils import is_recipe_urn, get_recipe_feeds_from_urn
        except Exception:
            return

        try:
            timeout = int(plugin_prefs.get('timeout_seconds', 25) or 25)
        except Exception:
            timeout = 25

        try:
            if is_recipe_urn(url):
                feeds = get_recipe_feeds_from_urn(url)
                if not feeds:
                    raise ValueError('Recipe provides no RSS feeds')
                sub_title, sub_url = feeds[0]
                raw, final = fetch_url(sub_url, timeout_seconds=timeout)
                parsed = parse_feed(raw, base_url=final)
                msg = _('Recipe provides %(n)d feed(s). First feed parsed: %(t)s') % {
                    'n': len(feeds),
                    't': (parsed.get('title') or sub_title or sub_url or url),
                }
                QMessageBox.information(self, _('Re-check successful'), msg)
            else:
                raw, final = fetch_url(url, timeout_seconds=timeout)
                parsed = parse_feed(raw, base_url=final)
                title = parsed.get('title') or ''
                if title:
                    QMessageBox.information(self, _('Re-check successful'), _('Feed parsed successfully: %s') % title)
                else:
                    QMessageBox.information(self, _('Re-check successful'), _('Feed parsed but title not found'))
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Re-check failed'), _('Could not fetch/parse feed: %s') % str(e))
            except Exception:
                pass

    def _restore_default_layout(self):
        parent = self.parent()
        if parent is None or not hasattr(parent, 'restore_default_layout'):
            try:
                QMessageBox.information(self, _('Restore default layout'), _('Open RSS Reader first, then try again.'))
            except Exception:
                pass
            return
        try:
            parent.restore_default_layout()
            try:
                QMessageBox.information(self, _('Restore default layout'), _('Default layout restored.'))
            except Exception:
                pass
        except Exception:
            try:
                QMessageBox.warning(self, _('Restore default layout'), _('Failed to restore layout.'))
            except Exception:
                pass

    def _purge_database(self):
        parent = self.parent()
        if parent is None or not hasattr(parent, 'purge_database'):
            try:
                QMessageBox.information(self, _('Purge Database'), _('Open RSS Reader first, then try again.'))
            except Exception:
                pass
            return
        try:
            parent.purge_database()
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Error'), _('Failed to purge database: %s') % str(e))
            except Exception:
                pass
            return
        try:
            QMessageBox.information(self, _('Purge database'), _('Database purge completed.'))
        except Exception:
            pass

    def _clear_database(self):
        # Clears the current DB contents without changing DB path.
        try:
            from calibre_plugins.rss_reader import rss_db
        except Exception:
            try:
                import rss_db  # type: ignore
            except Exception:
                rss_db = None
        if rss_db is None:
            try:
                QMessageBox.warning(self, _('Error'), _('Could not import database module.'))
            except Exception:
                pass
            return

        try:
            db_path = str(getattr(rss_db, 'DB_PATH', '') or '')
        except Exception:
            db_path = ''

        msg = _('This will DELETE ALL feeds, cached items, tags, stars, and history\nfrom the current RSS Reader database.\n\nContinue?')
        if db_path:
            msg += '\n\n' + _('Database: %s') % db_path
        try:
            res = QMessageBox.question(
                self,
                _('Clear database'),
                msg,
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
                QMessageBox.StandardButton.No,
            )
        except Exception:
            return
        try:
            if res != QMessageBox.StandardButton.Yes:
                return
        except Exception:
            return

        try:
            # VACUUM is optional; best-effort.
            rss_db.clear_all_data(vacuum_after=True)
        except Exception as e:
            try:
                QMessageBox.warning(self, _('Error'), _('Failed to clear database: %s') % str(e))
            except Exception:
                pass
            return

        try:
            QMessageBox.information(self, _('Clear database'), _('Database cleared.'))
        except Exception:
            pass

    def save_settings(self):
        # Save plugin UI language preference
        try:
            if getattr(self, 'language_combo', None) is not None:
                lang = str(self.language_combo.currentData() or '').strip()
                plugin_prefs['plugin_ui_language'] = lang
        except Exception:
            pass
        
        # Minimize to tray
        try:
            if getattr(self, 'minimize_to_tray_cb', None) is not None:
                plugin_prefs['minimize_to_tray'] = bool(self.minimize_to_tray_cb.isChecked())
        except Exception:
            pass
        try:
            if getattr(self, 'show_advanced_cb', None) is not None:
                plugin_prefs['show_advanced_settings'] = bool(self.show_advanced_cb.isChecked())
        except Exception:
            pass
        plugin_prefs['notify_on_new'] = bool(self.notify_cb.isChecked())
        plugin_prefs['notify_on_first_fetch'] = bool(self.notify_first_cb.isChecked())
        plugin_prefs['max_notifications_per_update'] = int(self.max_notify_spin.value())
        try:
            if getattr(self, 'suppress_fullscreen_cb', None) is not None:
                plugin_prefs['suppress_notifications_fullscreen'] = bool(self.suppress_fullscreen_cb.isChecked())
        except Exception:
            pass
        plugin_prefs['auto_update_on_open'] = bool(self.auto_open_cb.isChecked())
        plugin_prefs['auto_update_minutes'] = int(self.auto_spin.value())
        plugin_prefs['timeout_seconds'] = int(self.timeout_spin.value())
        plugin_prefs['max_parallel_fetches'] = int(self.parallel_spin.value())
        plugin_prefs['max_seen_per_feed'] = int(self.max_seen_spin.value())
        plugin_prefs['max_cached_items_per_feed'] = int(self.max_cache_spin.value())
        try:
            if getattr(self, 'max_items_spin', None) is not None:
                plugin_prefs['max_items_per_selection'] = int(self.max_items_spin.value())
        except Exception:
            pass
        try:
            label = str(self.published_format_combo.currentText())
            plugin_prefs['published_timestamp_format'] = str(AVAILABLE_PUBLISHED_TIMESTAMP_FORMATS.get(label) or DEFAULT_PUBLISHED_TIMESTAMP_FORMAT)
        except Exception:
            plugin_prefs['published_timestamp_format'] = str(DEFAULT_PUBLISHED_TIMESTAMP_FORMAT)

        try:
            if getattr(self, 'serbian_graphia_combo', None) is not None:
                val = str(self.serbian_graphia_combo.currentData() or 'latin').strip().lower()
                # Explicitly validate: only 'latin' or 'cyrillic'
                if val == 'cyrillic':
                    val = 'cyrillic'
                else:
                    val = 'latin'
                plugin_prefs['serbian_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'russian_graphia_combo', None) is not None:
                val = str(self.russian_graphia_combo.currentData() or 'native').strip().lower()
                # Explicitly validate: only 'latin' or 'native'
                if val == 'latin':
                    val = 'latin'
                else:
                    val = 'native'
                plugin_prefs['russian_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'japanese_graphia_combo', None) is not None:
                val = str(self.japanese_graphia_combo.currentData() or 'native').strip().lower()
                if val not in ('native', 'latin'):
                    val = 'native'
                plugin_prefs['japanese_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'chinese_graphia_combo', None) is not None:
                val = str(self.chinese_graphia_combo.currentData() or 'native').strip().lower()
                if val not in ('native', 'latin'):
                    val = 'native'
                plugin_prefs['chinese_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'korean_graphia_combo', None) is not None:
                val = str(self.korean_graphia_combo.currentData() or 'native').strip().lower()
                if val not in ('native', 'latin'):
                    val = 'native'
                plugin_prefs['korean_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'hindi_graphia_combo', None) is not None:
                val = str(self.hindi_graphia_combo.currentData() or 'native').strip().lower()
                if val not in ('native', 'latin'):
                    val = 'native'
                plugin_prefs['hindi_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'arabic_graphia_combo', None) is not None:
                val = str(self.arabic_graphia_combo.currentData() or 'native').strip().lower()
                if val not in ('native', 'latin'):
                    val = 'native'
                plugin_prefs['arabic_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'hebrew_graphia_combo', None) is not None:
                val = str(self.hebrew_graphia_combo.currentData() or 'native').strip().lower()
                if val not in ('native', 'latin'):
                    val = 'native'
                plugin_prefs['hebrew_graphia'] = val
        except Exception:
            pass
        try:
            if getattr(self, 'greek_graphia_combo', None) is not None:
                val = str(self.greek_graphia_combo.currentData() or 'native').strip().lower()
                if val not in ('native', 'latin'):
                    val = 'native'
                plugin_prefs['greek_graphia'] = val
        except Exception:
            pass
        plugin_prefs['load_images_in_preview'] = bool(self.load_images_cb.isChecked())
        try:
            if getattr(self, 'autofit_images_cb', None) is not None:
                plugin_prefs['autofit_images'] = bool(self.autofit_images_cb.isChecked())
        except Exception:
            pass
        try:
            if getattr(self, 'fetch_article_cb', None) is not None:
                plugin_prefs['preview_fetch_article_content'] = bool(self.fetch_article_cb.isChecked())
        except Exception:
            pass
        try:
            if getattr(self, 'preview_use_recipe_engine_cb', None) is not None:
                plugin_prefs['preview_use_recipe_engine'] = bool(self.preview_use_recipe_engine_cb.isChecked())
        except Exception:
            pass
        plugin_prefs['export_add_to_library'] = bool(self.export_add_cb.isChecked())
        try:
            if getattr(self, 'article_export_add_cb', None) is not None:
                plugin_prefs['article_export_add_to_library'] = bool(self.article_export_add_cb.isChecked())
        except Exception:
            pass
        # No longer store ai_export_append_timestamp; always append

        # Export image processing
        try:
            if getattr(self, 'export_download_uncached_images_cb', None) is not None:
                plugin_prefs['export_download_uncached_images'] = bool(self.export_download_uncached_images_cb.isChecked())
        except Exception:
            pass
        try:
            if getattr(self, 'export_optimize_images_cb', None) is not None:
                plugin_prefs['export_optimize_images'] = bool(self.export_optimize_images_cb.isChecked())
        except Exception:
            pass
        try:
            if getattr(self, 'export_image_max_bytes_mb', None) is not None:
                mb = int(self.export_image_max_bytes_mb.value())
                plugin_prefs['export_image_max_bytes'] = int(mb) * 1024 * 1024
        except Exception:
            pass
        try:
            if getattr(self, 'export_optimize_min_mb', None) is not None:
                mb = int(self.export_optimize_min_mb.value())
                plugin_prefs['export_optimize_images_min_bytes'] = int(mb) * 1024 * 1024
        except Exception:
            pass
        try:
            if getattr(self, 'export_force_jpeg_mb', None) is not None:
                mb = int(self.export_force_jpeg_mb.value())
                plugin_prefs['export_optimize_force_jpeg_bytes'] = int(mb) * 1024 * 1024
        except Exception:
            pass
        try:
            if getattr(self, 'export_image_max_dim_spin', None) is not None:
                plugin_prefs['export_image_max_dim'] = int(self.export_image_max_dim_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'export_image_jpeg_quality_spin', None) is not None:
                plugin_prefs['export_image_jpeg_quality'] = int(self.export_image_jpeg_quality_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'debug_export_images_cb', None) is not None:
                plugin_prefs['debug_export_images'] = bool(self.debug_export_images_cb.isChecked())
        except Exception:
            pass
        try:
            _fmt_idx = self.export_format_combo.currentIndex()
            _fmt_map = {0: 'calibre_default', 1: 'epub', 2: 'azw3', 3: 'mobi', 4: 'pdf', 5: 'docx', 6: 'txt', 7: 'md'}
            plugin_prefs['export_output_format'] = _fmt_map.get(_fmt_idx, 'calibre_default')
        except Exception:
            plugin_prefs['export_output_format'] = 'calibre_default'
        plugin_prefs['auto_tagging_enabled'] = bool(self.auto_tag_cb.isChecked())
        plugin_prefs['auto_tag_img'] = bool(self.auto_tag_img_cb.isChecked())
        plugin_prefs['auto_tag_audio'] = bool(getattr(self, 'auto_tag_audio_cb', None).isChecked() if getattr(self, 'auto_tag_audio_cb', None) is not None else True)
        plugin_prefs['auto_tag_long'] = bool(self.auto_tag_long_cb.isChecked())
        plugin_prefs['auto_tag_long_words'] = int(self.auto_tag_words_spin.value())

        # Feed auto-tagging
        try:
            if getattr(self, 'auto_feed_tag_cb', None) is not None:
                plugin_prefs['auto_feed_tagging_enabled'] = bool(self.auto_feed_tag_cb.isChecked())
            if getattr(self, 'auto_feed_tag_name', None) is not None:
                plugin_prefs['auto_feed_updates_tag_name'] = str(self.auto_feed_tag_name.text() or '').strip() or 'updates frequently'
            if getattr(self, 'auto_feed_window_days', None) is not None:
                plugin_prefs['auto_feed_updates_window_days'] = int(self.auto_feed_window_days.value())
            if getattr(self, 'auto_feed_min_dates', None) is not None:
                plugin_prefs['auto_feed_updates_min_distinct_dates'] = int(self.auto_feed_min_dates.value())
            if getattr(self, 'auto_feed_min_avg', None) is not None:
                plugin_prefs['auto_feed_updates_min_avg_per_day'] = float(self.auto_feed_min_avg.value())
        except Exception:
            pass
        try:
            txt = str(self.opml_import_combo.currentText())
            if txt == _('Recipes'):
                plugin_prefs['opml_import_single_click'] = 'recipes'
            elif txt == _('Advanced'):
                plugin_prefs['opml_import_single_click'] = 'advanced'
            else:
                plugin_prefs['opml_import_single_click'] = 'simple'
        except Exception:
            plugin_prefs['opml_import_single_click'] = 'advanced'
        try:
            plugin_prefs['opml_export_single_click'] = 'advanced' if str(self.opml_export_combo.currentText()) == _('Advanced') else 'simple'
        except Exception:
            plugin_prefs['opml_export_single_click'] = 'advanced'

        # Popup notifications
        try:
            if getattr(self, 'popup_enable_cb', None) is not None:
                plugin_prefs['notify_in_app_popup'] = bool(self.popup_enable_cb.isChecked())
        except Exception:
            pass
        try:
            if getattr(self, 'popup_theme_combo', None) is not None:
                v = str(getattr(self.popup_theme_combo, 'currentData', lambda: None)() or 'dark').strip().lower()
                if v not in ('auto', 'light', 'dark'):
                    v = 'dark'
                plugin_prefs['popup_theme'] = v
        except Exception:
            pass
        try:
            if getattr(self, 'popup_position_combo', None) is not None:
                plugin_prefs['popup_position'] = str(getattr(self.popup_position_combo, 'currentData', lambda: None)() or 'bottom_right').strip()
        except Exception:
            pass
        try:
            if getattr(self, 'popup_margin_spin', None) is not None:
                plugin_prefs['popup_margin_px'] = int(self.popup_margin_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'popup_width_spin', None) is not None:
                plugin_prefs['popup_width_px'] = int(self.popup_width_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'popup_item_height_spin', None) is not None:
                plugin_prefs['popup_item_min_height_px'] = int(self.popup_item_height_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'popup_item_padding_spin', None) is not None:
                plugin_prefs['popup_item_padding_v_px'] = int(self.popup_item_padding_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'popup_items_per_page_spin', None) is not None:
                plugin_prefs['popup_max_items_per_page'] = int(self.popup_items_per_page_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'popup_items_total_spin', None) is not None:
                plugin_prefs['popup_max_items_total'] = int(self.popup_items_total_spin.value())
        except Exception:
            pass
        try:
            if getattr(self, 'popup_close_after_spin', None) is not None:
                secs = int(self.popup_close_after_spin.value())
                plugin_prefs['popup_timeout_ms'] = max(0, secs) * 1000
        except Exception:
            pass

        # Popup theme colors (basic customization: bg + link)
        try:
            overrides = getattr(self, '_popup_color_overrides', None)
        except Exception:
            overrides = None
        if isinstance(overrides, dict):
            for theme_name, vals in overrides.items():
                try:
                    tn = str(theme_name or '').strip().lower()
                except Exception:
                    tn = ''
                if tn not in ('light', 'dark'):
                    continue
                # Dark maps to dark_high_contrast internally.
                key = 'popup_theme_colors_%s' % ('dark_high_contrast' if tn == 'dark' else tn)
                try:
                    base = dict(plugin_prefs.get(key, {}) or {})
                except Exception:
                    base = {}
                if not base:
                    try:
                        base = dict(plugin_prefs.defaults.get(key, {}) or {})
                    except Exception:
                        base = {}
                try:
                    bg = str((vals or {}).get('bg') or '').strip()
                    if bg:
                        base['bg'] = bg
                except Exception:
                    pass
                try:
                    link = str((vals or {}).get('link') or '').strip()
                    if link:
                        base['link'] = link
                        base['clickable_url'] = link
                except Exception:
                    pass
                try:
                    plugin_prefs[key] = base
                except Exception:
                    pass

                # Back-compat: also update the legacy dark palette if user edited Dark.
                if tn == 'dark':
                    try:
                        legacy = dict(plugin_prefs.get('popup_theme_colors_dark', {}) or {})
                    except Exception:
                        legacy = {}
                    try:
                        if bg:
                            legacy['bg'] = bg
                    except Exception:
                        pass
                    try:
                        if link:
                            legacy['link'] = link
                            legacy['clickable_url'] = link
                    except Exception:
                        pass
                    try:
                        plugin_prefs['popup_theme_colors_dark'] = legacy
                    except Exception:
                        pass

    def _on_language_changed(self):
        """Handle plugin UI language selection change.
        
        Saves preference immediately so the next dialog open picks it up,
        then patches the translation cache so all plugin modules use the
        new language.
        """
        if self.language_combo is None:
            return
        
        try:
            new_locale = str(self.language_combo.currentData() or '').strip()
            
            # Save immediately so _resolve_target_locale() picks it up on next dialog open
            plugin_prefs['plugin_ui_language'] = new_locale
            
            # Load the .mo file for the new locale (or revert to calibre default)
            try:
                from calibre_plugins.rss_reader.i18n import (
                    _patch_translations_cache, _patch_to_auto, refresh_all_plugin_modules
                )
                if not new_locale:
                    # Auto: load calibre's current UI language for this plugin.
                    _patch_to_auto()
                elif str(new_locale).strip().lower().startswith('en'):
                    # English = install NullTranslations so _() returns source msgid strings.
                    # Never load en.mo — its msgstr entries are empty and would blank the UI.
                    from calibre_plugins.rss_reader.i18n import _reset_to_source_language
                    _reset_to_source_language()
                else:
                    _patch_translations_cache(new_locale)
                refresh_all_plugin_modules()
            except Exception as e:
                import sys
                print(f'[RSS Reader] ERROR: Failed to load language {new_locale!r}: {e}', file=sys.stderr)
        except Exception as e:
            import sys
            print(f'[RSS Reader] ERROR: Language change handler failed: {e}', file=sys.stderr)

    def accept(self):
        self.save_settings()
        return QDialog.accept(self)

    def done(self, r):
        try:
            self._persist_dialog_geometry()
        except Exception:
            pass
        return QDialog.done(self, r)

    def _restore_dialog_geometry(self):
        try:
            # Version migration: reset geometry if it was saved by version < 2.0
            # (old height was 560, now it's 680)
            last_geom_version = plugin_prefs.get('settings_dialog_geometry_version', '')
            current_version = '2.0.0'
            
            # Compare versions: if last saved version < 2.0, clear old geometry
            if last_geom_version and last_geom_version < current_version:
                # Old geometry is incompatible; clear it
                plugin_prefs.pop('settings_dialog_geometry', None)
                plugin_prefs['settings_dialog_geometry_version'] = current_version
                return False
            
            # Update version on successful restore
            if last_geom_version != current_version:
                plugin_prefs['settings_dialog_geometry_version'] = current_version
            
            geom_b64 = plugin_prefs.get('settings_dialog_geometry', '')
            if not geom_b64:
                return False
            raw = base64.b64decode(geom_b64)
            return bool(self.restoreGeometry(raw))
        except Exception:
            return False

    def _persist_dialog_geometry(self):
        try:
            raw = bytes(self.saveGeometry())
            plugin_prefs['settings_dialog_geometry'] = base64.b64encode(raw).decode('ascii')
        except Exception:
            pass

    def _apply_settings_visibility(self, show_advanced):
        show_advanced = bool(show_advanced)
        try:
            # Entire groups
            for gb_name in ('network_gb', 'storage_gb', 'opml_gb'):
                gb = getattr(self, gb_name, None)
                if gb is not None:
                    gb.setVisible(show_advanced)
        except Exception:
            pass

        # Tagging: keep the primary toggle in Basic, hide detailed knobs.
        try:
            tagging_form = getattr(self, '_tagging_form', None)
            for w in (
                getattr(self, 'auto_tag_img_cb', None),
                getattr(self, 'auto_tag_audio_cb', None),
                getattr(self, 'auto_tag_long_cb', None),
                getattr(self, 'auto_tag_words_spin', None),
                getattr(self, 'auto_feed_tag_cb', None),
                getattr(self, 'auto_feed_tag_name', None),
                getattr(self, 'auto_feed_window_days', None),
                getattr(self, 'auto_feed_min_dates', None),
                getattr(self, 'auto_feed_min_avg', None),
            ):
                if w is None:
                    continue
                w.setVisible(show_advanced)
                try:
                    if tagging_form is not None:
                        lbl = tagging_form.labelForField(w)
                        if lbl is not None:
                            lbl.setVisible(show_advanced)
                except Exception:
                    pass
        except Exception:
            pass

        # Display: always show the published-timestamp format control (move to Basic)
        try:
            display_form = getattr(self, '_display_form', None)
            w = getattr(self, 'published_format_combo', None)
            if w is not None:
                w.setVisible(True)
                try:
                    lbl = display_form.labelForField(w) if display_form is not None else None
                    if lbl is not None:
                        lbl.setVisible(True)
                except Exception:
                    pass
        except Exception:
            pass

        # Preview & export: hide advanced export-image controls in Basic.
        try:
            preview_form = getattr(self, '_preview_form', None)
            adv_preview_widgets = (
                getattr(self, 'export_download_uncached_images_cb', None),
                getattr(self, 'export_optimize_images_cb', None),
                getattr(self, 'export_image_max_bytes_mb', None),
                getattr(self, 'export_optimize_min_mb', None),
                getattr(self, 'export_force_jpeg_mb', None),
                getattr(self, 'export_image_max_dim_spin', None),
                getattr(self, 'export_image_jpeg_quality_spin', None),
                getattr(self, 'debug_export_images_cb', None),
            )
            for w in adv_preview_widgets:
                if w is None:
                    continue
                w.setVisible(show_advanced)
                try:
                    lbl = preview_form.labelForField(w) if preview_form is not None else None
                    if lbl is not None:
                        lbl.setVisible(show_advanced)
                except Exception:
                    pass
        except Exception:
            pass
