# Cleanup and purge helpers for RSS Reader.
# These mirror methods on RSSReaderDialog but live outside ui.py to
# keep the main UI module smaller. They all take `self` as the first
# argument (an instance of RSSReaderDialog).

from __future__ import absolute_import

import traceback

try:
    from qt.core import Qt, QDialog, QVBoxLayout, QLabel, QTreeWidget, QTreeWidgetItem, QDialogButtonBox, QWidget, QGridLayout, QCheckBox, QSpinBox, QMessageBox, QPushButton
except Exception:  # pragma: no cover - fallback for older Qt bindings
    from PyQt5.Qt import Qt
    from PyQt5.QtWidgets import QDialog, QVBoxLayout, QLabel, QTreeWidget, QTreeWidgetItem, QDialogButtonBox, QWidget, QGridLayout, QCheckBox, QSpinBox, QMessageBox, QPushButton

from calibre_plugins.rss_reader import rss_db
from calibre_plugins.rss_reader.config import plugin_prefs


def purge_database(self):
    """Clean up orphaned cache/seen entries and show SQLite database statistics."""
    try:
        import os

        feeds = list(rss_db.get_feeds() or [])
        valid_feed_ids = {str(f.get('id') or '') for f in feeds if str(f.get('id') or '').strip()}

        seen_item_ids = dict(rss_db.get_seen_item_ids_map() or {})
        feed_cache = dict(rss_db.get_feed_cache_map() or {})

        db_path = rss_db.db_path()
        file_size_before = 0
        total_size_before = 0
        try:
            if os.path.exists(db_path):
                file_size_before = os.path.getsize(db_path)
        except Exception:
            pass
        try:
            total_size_before = int(rss_db.total_size_bytes() or 0)
        except Exception:
            total_size_before = 0

        orphaned_seen = [fid for fid in seen_item_ids.keys() if fid not in valid_feed_ids]
        orphaned_cache = [fid for fid in feed_cache.keys() if fid not in valid_feed_ids]

        total_cached_items = sum(len(entry.get('items', [])) for entry in feed_cache.values() if isinstance(entry, dict))
        total_seen_ids = sum(len(ids) for ids in seen_item_ids.values() if isinstance(ids, list))

        # Build info message
        info_parts = []
        info_parts.append(_('Database Statistics:'))
        info_parts.append('')
        info_parts.append(_('SQLite file size: %.2f MB') % (file_size_before / 1024.0 / 1024.0))
        if total_size_before and total_size_before != file_size_before:
            info_parts.append(_('SQLite total on-disk (db+wal+shm): %.2f MB') % (total_size_before / 1024.0 / 1024.0))
        info_parts.append(_('Active feeds: %d') % len(valid_feed_ids))
        info_parts.append(_('Total cached items: %d') % total_cached_items)
        info_parts.append(_('Total tracked item IDs: %d') % total_seen_ids)
        info_parts.append('')
        info_parts.append(_('Orphaned entries (from deleted feeds):'))
        info_parts.append(_('  - Orphaned cache entries: %d') % len(orphaned_cache))
        info_parts.append(_('  - Orphaned seen_item_ids: %d') % len(orphaned_seen))

        if not orphaned_seen and not orphaned_cache:
            info_parts.append('')
            info_parts.append(_('✓ No orphaned entries found. Database is clean.'))
            QMessageBox.information(self, _('Purge Database'), '\n'.join(info_parts))
            return

        info_parts.append('')
        info_parts.append(_('Purge these orphaned entries?'))

        yes = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'Yes', getattr(QMessageBox, 'Yes', None))
        no = getattr(getattr(QMessageBox, 'StandardButton', QMessageBox), 'No', getattr(QMessageBox, 'No', None))
        ans = QMessageBox.question(self, _('Purge Database'), '\n'.join(info_parts), yes | no, no)
        if ans != yes:
            return

        # Purge
        try:
            rss_db.purge_orphans(clear_folders_if_no_feeds=True)
        except Exception:
            pass

        # Reclaim disk space (SQLite won't shrink the file without VACUUM)
        try:
            rss_db.vacuum()
        except Exception:
            pass
        for fid in orphaned_cache:
            self._items_by_feed_id.pop(str(fid), None)

        # Force write and calculate new size
        try:
            import time
            time.sleep(0.1)  # Give time for write to complete
        except Exception:
            pass

        file_size_after = 0
        total_size_after = 0
        try:
            if os.path.exists(db_path):
                file_size_after = os.path.getsize(db_path)
        except Exception:
            pass
        try:
            total_size_after = int(rss_db.total_size_bytes() or 0)
        except Exception:
            total_size_after = 0

        saved_mb = (file_size_before - file_size_after) / 1024.0 / 1024.0

        result_parts = []
        result_parts.append(_('Purge Complete!'))
        result_parts.append('')
        result_parts.append(_('Removed:'))
        result_parts.append(_('  - %d orphaned cache entries') % len(orphaned_cache))
        result_parts.append(_('  - %d orphaned seen_item_ids entries') % len(orphaned_seen))
        try:
            if not (rss_db.get_feeds() or []):
                result_parts.append(_('  - All folders (no feeds remain)'))
        except Exception:
            pass
        result_parts.append('')
        result_parts.append(_('File size: %.2f MB → %.2f MB') % (file_size_before / 1024.0 / 1024.0, file_size_after / 1024.0 / 1024.0))
        if total_size_before and total_size_after and (total_size_before != file_size_before or total_size_after != file_size_after):
            result_parts.append(_('Total on-disk: %.2f MB → %.2f MB') % (total_size_before / 1024.0 / 1024.0, total_size_after / 1024.0 / 1024.0))
        if saved_mb > 0:
            result_parts.append(_('Saved: %.2f MB (%.1f%%)') % (saved_mb, (saved_mb / (file_size_before / 1024.0 / 1024.0) * 100)))

        QMessageBox.information(self, _('Purge Complete'), '\n'.join(result_parts))
        self.refresh()

    except Exception as e:
        from calibre.gui2 import error_dialog
        error_dialog(self, _('RSS Reader Error'), _('Failed to purge database: %s') % str(e),
                    show=True, det_msg=traceback.format_exc())


def show_cleanup_dialog(self, preselected_feed_ids=None):
    """Interactive cleanup dialog similar in spirit to QuiteRSS' Clean Up.

    - Allows the user to choose which feeds to clean.
    - Lets the user specify age/count limits and safety options.
    - Persists options in plugin_prefs.
    """
    from calibre.gui2 import error_dialog

    try:
        if bool(getattr(rss_db, 'DB_READONLY', False)):
            QMessageBox.warning(self, _('Read-only database'), _('Current database/profile is read-only; cleanup is disabled.'))
            return

        feeds = list(rss_db.get_feeds() or [])
        if not feeds:
            QMessageBox.information(self, _('Clean up'), _('No feeds are configured.'))
            return

        dlg = QDialog(self)
        dlg.setWindowTitle(_('Clean up'))
        layout = QVBoxLayout(dlg)

        label = QLabel(_('Choose feeds to clean up:'))
        try:
            label.setWordWrap(True)
        except Exception:
            pass
        layout.addWidget(label)

        tree = QTreeWidget(dlg)
        tree.setHeaderHidden(True)
        top_text = _('All feeds') if not preselected_feed_ids else _('Selected feeds')
        top = QTreeWidgetItem([top_text])
        top.setCheckState(0, Qt.CheckState.Checked)
        top.setData(0, Qt.ItemDataRole.UserRole, '__all__')
        tree.addTopLevelItem(top)

        pre_ids = set(str(x) for x in (preselected_feed_ids or []) if str(x or '').strip())
        for f in feeds:
            fid = str(f.get('id') or '').strip()
            if not fid:
                continue
            if pre_ids and fid not in pre_ids:
                continue
            title = f.get('title') or f.get('url') or fid
            it = QTreeWidgetItem([title])
            it.setData(0, Qt.ItemDataRole.UserRole, fid)
            it.setCheckState(0, Qt.CheckState.Checked)
            top.addChild(it)

        tree.expandAll()

        # Keep parent/child checkboxes in sync.
        def _on_item_changed(item, column):
            try:
                if column != 0:
                    return
                state = item.checkState(0)
                if item is top:
                    for i in range(top.childCount()):
                        ch = top.child(i)
                        ch.setCheckState(0, state)
                else:
                    # Update top node to reflect partial selection.
                    any_checked = False
                    any_unchecked = False
                    for i in range(top.childCount()):
                        ch = top.child(i)
                        cs = ch.checkState(0)
                        if cs == Qt.CheckState.Checked:
                            any_checked = True
                        elif cs == Qt.CheckState.Unchecked:
                            any_unchecked = True
                    new_state = Qt.CheckState.Checked
                    if any_checked and any_unchecked:
                        new_state = Qt.CheckState.PartiallyChecked
                    elif any_unchecked and not any_checked:
                        new_state = Qt.CheckState.Unchecked
                    top.setCheckState(0, new_state)
            except Exception:
                pass

        try:
            tree.itemChanged.connect(_on_item_changed)
        except Exception:
            pass

        layout.addWidget(tree)

        # Options block
        opts = dict(plugin_prefs.get('cleanup_options', {}) or {})
        try:
            default_days = int(opts.get('max_days', 30) or 30)
        except Exception:
            default_days = 30
        try:
            default_max_items = int(opts.get('max_items', 200) or 200)
        except Exception:
            default_max_items = 200
        use_days = bool(opts.get('use_days', True))
        use_max_items = bool(opts.get('use_max_items', True))
        never_unread = bool(opts.get('never_unread', False))
        never_starred = bool(opts.get('never_starred', True))
        never_tagged = bool(opts.get('never_tagged', True))
        purge_after = bool(opts.get('purge_after', True))

        box = QWidget(dlg)
        box_layout = QGridLayout(box)

        cb_days = QCheckBox(_('Maximum age of items in days to keep:'))
        spin_days = QSpinBox(box)
        spin_days.setRange(1, 9999)
        spin_days.setValue(default_days)
        spin_days.setEnabled(use_days)
        cb_days.setChecked(use_days)
        cb_days.toggled.connect(spin_days.setEnabled)

        cb_max = QCheckBox(_('Maximum number of items per feed to keep:'))
        spin_max = QSpinBox(box)
        spin_max.setRange(1, 99999)
        spin_max.setValue(default_max_items)
        spin_max.setEnabled(use_max_items)
        cb_max.setChecked(use_max_items)
        cb_max.toggled.connect(spin_max.setEnabled)

        cb_never_unread = QCheckBox(_('Never delete unread items'))
        cb_never_unread.setChecked(never_unread)
        cb_never_starred = QCheckBox(_('Never delete starred items'))
        cb_never_starred.setChecked(never_starred)
        # "Tagged" here refers to manual/user-defined tags stored in the
        # database, not auto-tags that are computed on the fly.
        cb_never_tagged = QCheckBox(_('Never delete manually tagged items'))
        cb_never_tagged.setChecked(never_tagged)

        cb_purge = QCheckBox(_('Purge DB after cleanup (reclaim space)'))
        cb_purge.setChecked(purge_after)

        row = 0
        box_layout.addWidget(cb_days, row, 0, 1, 1)
        box_layout.addWidget(spin_days, row, 1, 1, 1)
        row += 1
        box_layout.addWidget(cb_max, row, 0, 1, 1)
        box_layout.addWidget(spin_max, row, 1, 1, 1)
        row += 1
        box_layout.addWidget(cb_never_unread, row, 0, 1, 2)
        row += 1
        box_layout.addWidget(cb_never_starred, row, 0, 1, 2)
        row += 1
        box_layout.addWidget(cb_never_tagged, row, 0, 1, 2)
        row += 1
        box_layout.addWidget(cb_purge, row, 0, 1, 2)

        layout.addWidget(box)

        # Dedicated button for clearing image cache
        clear_img_cache_btn = QPushButton(_('Clear Image Cache'))
        layout.addWidget(clear_img_cache_btn)

        def _clear_image_cache():
            try:
                from calibre.constants import cache_dir
                import os, shutil
                base = cache_dir() if callable(cache_dir) else ''
                img_cache_dir = os.path.join(base, 'plugins', 'rss_reader', 'img_cache') if base else ''
                if img_cache_dir and os.path.exists(img_cache_dir):
                    shutil.rmtree(img_cache_dir)
                    QMessageBox.information(dlg, _('Clear Image Cache'), _('Image cache cleared.'))
                    dlg.accept()  # Close the dialog after clearing
                else:
                    QMessageBox.information(dlg, _('Clear Image Cache'), _('No image cache found.'))
                    dlg.accept()  # Close the dialog even if no cache
            except Exception as e:
                QMessageBox.warning(dlg, _('Error'), _('Failed to clear image cache: %s') % str(e))

        clear_img_cache_btn.clicked.connect(_clear_image_cache)

        btns = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, parent=dlg)
        layout.addWidget(btns)
        btns.accepted.connect(dlg.accept)
        btns.rejected.connect(dlg.reject)

        if dlg.exec() != QDialog.DialogCode.Accepted:
            return

        # Persist options
        new_opts = {
            'use_days': bool(cb_days.isChecked()),
            'max_days': int(spin_days.value()),
            'use_max_items': bool(cb_max.isChecked()),
            'max_items': int(spin_max.value()),
            'never_unread': bool(cb_never_unread.isChecked()),
            'never_starred': bool(cb_never_starred.isChecked()),
            'never_tagged': bool(cb_never_tagged.isChecked()),
            'purge_after': bool(cb_purge.isChecked()),
        }
        try:
            plugin_prefs['cleanup_options'] = new_opts
        except Exception:
            pass

        selected_ids = []
        for i in range(top.childCount()):
            ch = top.child(i)
            try:
                if ch.checkState(0) != Qt.CheckState.Checked:
                    continue
                fid = str(ch.data(0, Qt.ItemDataRole.UserRole) or '').strip()
            except Exception:
                fid = ''
            if fid:
                selected_ids.append(fid)
        if not selected_ids:
            QMessageBox.information(self, _('Clean up'), _('No feeds selected for cleanup.'))
            return

        max_days = int(spin_days.value()) if cb_days.isChecked() else None
        max_items = int(spin_max.value()) if cb_max.isChecked() else None

        run_cleanup_for_feeds(
            self,
            selected_ids,
            max_days=max_days,
            max_items=max_items,
            never_unread=bool(cb_never_unread.isChecked()),
            never_starred=bool(cb_never_starred.isChecked()),
            never_tagged=bool(cb_never_tagged.isChecked()),
            purge_after=bool(cb_purge.isChecked()),
        )

    except Exception as e:  # pragma: no cover - keep behavior identical
        error_dialog(self, _('RSS Reader Error'), _('Failed to run cleanup: %s') % str(e),
                    show=True, det_msg=traceback.format_exc())


def open_cleanup_dialog(self):
    """Entry point for toolbar button: clean up across all feeds."""
    return show_cleanup_dialog(self, preselected_feed_ids=None)


def run_cleanup_for_feeds(self, feed_ids, max_days=None, max_items=None,
                          never_unread=True, never_starred=True,
                          never_tagged=True, purge_after=True):
    """Core cleanup implementation operating on feed_cache/items.

    This prunes old items from the JSON feed cache (and mirrored items
    table), while preserving starred/important content according to the
    options provided.
    """
    from calibre.gui2 import error_dialog

    try:
        ids = [str(x or '').strip() for x in (feed_ids or []) if str(x or '').strip()]
        if not ids:
            return

        try:
            all_cache = dict(rss_db.get_feed_cache_map() or {})
        except Exception:
            all_cache = {}
        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}
        try:
            known_map = dict(rss_db.get_known_item_ids_map() or {})
        except Exception:
            known_map = {}
        try:
            starred_map = dict(rss_db.get_starred_items_map() or {})
        except Exception:
            starred_map = {}
        try:
            # Item tags map contains only manual/user-defined tags; auto
            # tags are never stored in the DB.
            tags_map_all = dict(rss_db.get_item_tags_map() or {})
        except Exception:
            tags_map_all = {}

        # Per-feed retention override (in days). If a feed has a
        # retention_days > 0, that value is used instead of the global
        # max_days from the dialog.
        try:
            feeds_all = list(rss_db.get_feeds() or [])
            retention_by_feed = {}
            for f in feeds_all:
                try:
                    fid = str(f.get('id') or '').strip()
                except Exception:
                    fid = ''
                if not fid:
                    continue
                try:
                    rd = int(f.get('retention_days') or 0)
                except Exception:
                    rd = 0
                if rd > 0:
                    retention_by_feed[fid] = rd
        except Exception:
            retention_by_feed = {}

        # Build tagged-items map by feed for quick lookup.
        tagged_by_feed = {}
        try:
            for (fid, iid), tags in (tags_map_all or {}).items():
                if not tags:
                    continue
                fid_s = str(fid)
                if fid_s not in tagged_by_feed:
                    tagged_by_feed[fid_s] = set()
                tagged_by_feed[fid_s].add(str(iid))
        except Exception:
            tagged_by_feed = {}

        import time as _time
        now_ts = int(_time.time())

        total_deleted = 0
        per_feed_deleted = {}

        def _item_id(it):
            try:
                return str(it.get('id') or it.get('link') or it.get('title') or '').strip()
            except Exception:
                return ''

        for fid in ids:
            entry = all_cache.get(str(fid))
            if not isinstance(entry, dict):
                continue
            try:
                items = list(entry.get('items') or [])
            except Exception:
                items = []
            if not items:
                continue

            seen_ids = set(str(x) for x in (seen_map.get(str(fid)) or [])) if never_unread else set()
            starred_ids = set(str(x) for x in (starred_map.get(str(fid)) or [])) if never_starred else set()
            tagged_ids = set(str(x) for x in (tagged_by_feed.get(str(fid)) or [])) if never_tagged else set()

            triples = []
            for it in items:
                if not isinstance(it, dict):
                    continue
                iid = _item_id(it)
                if not iid:
                    continue
                try:
                    ts = int(it.get('published_ts') or 0)
                except Exception:
                    ts = 0
                triples.append((iid, ts, it))
            if not triples:
                continue

            # Effective retention (days) for this feed: prefer per-feed
            # override; fall back to the global dialog value.
            feed_cutoff_ts = None
            try:
                rd = int(retention_by_feed.get(str(fid), 0) or 0)
            except Exception:
                rd = 0
            if rd <= 0 and max_days is not None:
                try:
                    rd = int(max_days or 0)
                except Exception:
                    rd = 0
            if rd > 0:
                feed_cutoff_ts = now_ts - rd * 24 * 60 * 60

            # Candidates based on age
            to_delete = set()
            if feed_cutoff_ts is not None:
                for iid, ts, _it in triples:
                    if ts and ts < feed_cutoff_ts:
                        to_delete.add(iid)

            # Candidates based on max_items (keep newest by published_ts)
            if max_items is not None:
                try:
                    m = int(max_items or 0)
                except Exception:
                    m = 0
                if m > 0 and len(triples) > m:
                    sorted_triples = sorted(triples, key=lambda r: r[1] or 0, reverse=True)
                    keep_ids = set(iid for iid, _ts, _it in sorted_triples[:m])
                    for iid, _ts, _it in sorted_triples[m:]:
                        to_delete.add(iid)

            if not to_delete:
                continue

            # Apply safety guards
            final_delete = set()
            for iid, _ts, _it in triples:
                if iid not in to_delete:
                    continue
                if never_unread and iid not in seen_ids:
                    continue
                if never_starred and iid in starred_ids:
                    continue
                if never_tagged and iid in tagged_ids:
                    continue
                final_delete.add(iid)

            if not final_delete:
                continue

            new_items = [it for (iid, _ts, it) in triples if iid not in final_delete]
            deleted_count = len(triples) - len(new_items)
            if deleted_count <= 0:
                continue

            # Update feed_cache (this will also sync the items mirror table).
            try:
                new_entry = dict(entry)
            except Exception:
                new_entry = entry
            new_entry['items'] = new_items
            try:
                rss_db.set_feed_cache(fid, new_entry)
            except Exception:
                continue

            # Prune seen/known lists
            try:
                if str(fid) in seen_map:
                    cur = list(seen_map.get(str(fid)) or [])
                    cur = [i for i in cur if str(i) not in final_delete]
                    rss_db.set_seen_item_ids(fid, cur)
            except Exception:
                pass
            try:
                if str(fid) in known_map:
                    cur = list(known_map.get(str(fid)) or [])
                    cur = [i for i in cur if str(i) not in final_delete]
                    rss_db.set_known_item_ids(fid, cur)
            except Exception:
                pass

            # Remove per-item tags and stars for deleted items
            for iid in final_delete:
                try:
                    rss_db.delete_item_tags(fid, iid)
                except Exception:
                    pass
                try:
                    rss_db.set_item_starred(fid, iid, False)
                except Exception:
                    pass

            total_deleted += deleted_count
            per_feed_deleted[str(fid)] = deleted_count

        if purge_after and total_deleted > 0:
            try:
                rss_db.purge_orphans(clear_folders_if_no_feeds=False)
            except Exception:
                pass
            try:
                rss_db.vacuum()
            except Exception:
                pass

        if total_deleted > 0:
            try:
                self.refresh()
            except Exception:
                pass

        # Summary dialog
        lines = [
            _('Cleanup complete.'),
            '',
            _('Total items deleted: %d') % total_deleted,
        ]
        if per_feed_deleted:
            lines.append('')
            lines.append(_('Per-feed deletions:'))
            # Only show top few feeds to keep the dialog compact.
            try:
                from itertools import islice  # noqa: F401  # kept for back-compat, even if unused
                items = list(per_feed_deleted.items())
            except Exception:
                items = list(per_feed_deleted.items())
            max_rows = 15
            for fid, n in items[:max_rows]:
                title = ''
                try:
                    for f in (rss_db.get_feeds() or []):
                        if str(f.get('id') or '') == str(fid):
                            title = f.get('title') or f.get('url') or fid
                            break
                except Exception:
                    title = fid
                lines.append('  - %s: %d' % (title, n))
            if len(items) > max_rows:
                lines.append('  …')

        QMessageBox.information(self, _('Clean up'), '\n'.join(lines))

    except Exception as e:  # pragma: no cover - keep identical behavior
        error_dialog(self, _('RSS Reader Error'), _('Cleanup failed: %s') % str(e),
                    show=True, det_msg=traceback.format_exc())
