from __future__ import absolute_import

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import (
        QDialog,
        QVBoxLayout,
        QHBoxLayout,
        QLabel,
        QPushButton,
        QListWidget,
        QListWidgetItem,
        QScrollArea,
        QWidget,
        Qt,
    )
except ImportError:
    from PyQt5.Qt import (
        QDialog,
        QVBoxLayout,
        QHBoxLayout,
        QLabel,
        QPushButton,
        QListWidget,
        QListWidgetItem,
        QScrollArea,
        QWidget,
        Qt,
    )

from calibre_plugins.rss_reader.sample_feeds import get_bundled_feeds_merged


class BundledFeedsDialog(QDialog):
    """Dialog to select and add bundled featured feeds."""

    def __init__(self, parent=None, skip_label=False):
        QDialog.__init__(self, parent)
        self.setWindowTitle(_('Add Featured Feeds'))
        self.setMinimumWidth(500)
        self.setMinimumHeight(300)
        self._skip_label = bool(skip_label)

        layout = QVBoxLayout(self)

        # Intro label
        intro = QLabel(
            _('Select which featured feeds you would like to add to your database.\n'
              'You can always remove them later.')
        )
        intro.setWordWrap(True)
        layout.addWidget(intro)

        # List of bundled feeds with checkboxes
        self.feed_list = QListWidget(self)
        self.feed_entries = []

        bundled = get_bundled_feeds_merged() or []
        for feed in bundled:
            try:
                name = str(feed.get('name', '') or '').strip()
                url = str(feed.get('url', '') or '').strip()
                lang = str(feed.get('lang', '') or '').strip()
                if not name or not url:
                    continue

                entry = dict(feed)
                entry['name'] = name
                entry['url'] = url
                if lang:
                    entry['lang'] = lang
                self.feed_entries.append(entry)

                label = name
                if lang and lang.lower() not in ('en', 'en-us', 'en-gb'):
                    label = '%s (%s)' % (name, lang)

                item = QListWidgetItem(label)
                try:
                    item.setFlags(item.flags() | Qt.ItemFlag.ItemIsUserCheckable)
                except Exception:
                    try:
                        item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                    except Exception:
                        pass

                try:
                    item.setToolTip(url)
                except Exception:
                    pass

                # Pre-check the first bundled feed by default
                try:
                    if len(self.feed_entries) == 1:
                        item.setCheckState(Qt.CheckState.Checked)
                    else:
                        item.setCheckState(Qt.CheckState.Unchecked)
                except Exception:
                    try:
                        if len(self.feed_entries) == 1:
                            item.setCheckState(Qt.Checked)
                        else:
                            item.setCheckState(Qt.Unchecked)
                    except Exception:
                        pass

                # Store the entry index so duplicate names are safe.
                try:
                    item.setData(Qt.ItemDataRole.UserRole, len(self.feed_entries) - 1)
                except Exception:
                    try:
                        item.setData(Qt.UserRole, len(self.feed_entries) - 1)
                    except Exception:
                        pass

                self.feed_list.addItem(item)
            except Exception:
                pass

        layout.addWidget(self.feed_list, 1)

        # Buttons
        btn_layout = QHBoxLayout()
        btn_layout.addStretch(1)

        skip_btn = QPushButton(_('Skip') if self._skip_label else _('Cancel'), self)
        add_btn = QPushButton(_('Add Selected'), self)
        add_all_btn = QPushButton(_('Add All'), self)

        skip_btn.clicked.connect(self.reject)
        add_btn.clicked.connect(self._add_selected)
        add_all_btn.clicked.connect(self._add_all)

        btn_layout.addWidget(skip_btn)
        btn_layout.addWidget(add_btn)
        btn_layout.addWidget(add_all_btn)

        layout.addLayout(btn_layout)

        self.selected_urls = []
        self.selected_entries = []

    def _add_selected(self):
        """Get list of URLs for checked items."""
        self.selected_urls = []
        self.selected_entries = []
        for i in range(self.feed_list.count()):
            try:
                item = self.feed_list.item(i)
                if item is None:
                    continue
                try:
                    checked = (item.checkState() == Qt.CheckState.Checked)
                except Exception:
                    checked = (item.checkState() == Qt.Checked)
                if checked:
                    idx = None
                    try:
                        idx = item.data(Qt.ItemDataRole.UserRole)
                    except Exception:
                        try:
                            idx = item.data(Qt.UserRole)
                        except Exception:
                            idx = None
                    try:
                        idx = int(idx)
                    except Exception:
                        idx = None
                    if idx is not None and 0 <= idx < len(self.feed_entries):
                        entry = dict(self.feed_entries[idx])
                        self.selected_entries.append(entry)
                        try:
                            u = str(entry.get('url') or '').strip()
                        except Exception:
                            u = ''
                        if u:
                            self.selected_urls.append(u)
            except Exception:
                pass
        self.accept()

    def _add_all(self):
        """Get URLs for all items."""
        self.selected_entries = [dict(e) for e in (self.feed_entries or [])]
        self.selected_urls = []
        for e in (self.feed_entries or []):
            try:
                u = str((e or {}).get('url') or '').strip()
            except Exception:
                u = ''
            if u:
                self.selected_urls.append(u)
        self.accept()
