__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

import os
import shutil
import tempfile
import traceback
import time
import json

try:
    load_translations()
except NameError:
    pass

try:
    from qt.core import (
        Qt,
        QAction,
        QComboBox,
        QDialog,
        QDialogButtonBox,
        QDockWidget,
        QApplication,
        QFontMetrics,
        QHBoxLayout,
        QLabel,
        QLineEdit,
        QMenu,
        QMessageBox,
        QPlainTextEdit,
        QPushButton,
        QSizePolicy,
        QTextBrowser,
        QToolButton,
        QVBoxLayout,
        QWidget,
        QIcon,
    )
except Exception:
    from PyQt5.Qt import Qt, QAction, QDialog, QDialogButtonBox, QIcon
    try:
        from PyQt5.QtGui import QFontMetrics
    except Exception:
        QFontMetrics = None
    from PyQt5.QtWidgets import (
        QComboBox,
        QDockWidget,
        QApplication,
        QHBoxLayout,
        QLabel,
        QLineEdit,
        QMenu,
        QMessageBox,
        QPlainTextEdit,
        QPushButton,
        QSizePolicy,
        QTextBrowser,
        QToolButton,
        QVBoxLayout,
        QWidget,
    )

from calibre.gui2 import Dispatcher, error_dialog, gprefs
from calibre.gui2.qt_file_dialogs import choose_save_file

from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader.debug import _debug


class AIPanelMixin:
    def setup_ai_panel(self):
        """Create AI dock/panel widgets. Called from setup_ui."""

        # AI Panel as a proper dock widget
        self.ai_dock = QDockWidget(_('Ask AI'), self)
        self.ai_dock.setObjectName('ask-ai-dock')
        try:
            self.ai_dock.setMinimumWidth(0)
        except Exception:
            pass

        self.ai_panel = QWidget(self)
        try:
            self.ai_panel.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        except Exception:
            try:
                self.ai_panel.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
            except Exception:
                pass
        try:
            self.ai_panel.setMinimumWidth(0)
        except Exception:
            pass
        ai_layout = QVBoxLayout(self.ai_panel)
        ai_layout.setContentsMargins(0, 0, 0, 0)

        # Prompt input (multi-line) + history dropdown
        self.ai_input = QPlainTextEdit(self.ai_panel)
        try:
            self.ai_input.setPlaceholderText(_('Ask a question… (blank = use selection)'))
        except Exception:
            pass
        try:
            self.ai_input.setToolTip(_('Type a question/term. If left blank, the current selection in Preview will be used.'))
        except Exception:
            pass
        # Show a few visible lines by default
        try:
            self.ai_input.setMinimumHeight(56)
        except Exception:
            pass
        # Keep the prompt compact (roughly 5 lines tall)
        try:
            if QFontMetrics is not None:
                fm = QFontMetrics(self.ai_input.font())
                # A little extra for frame/margins
                max_h = int((fm.lineSpacing() * 5) + 18)
                if max_h > 0:
                    self.ai_input.setMaximumHeight(max_h)
        except Exception:
            pass
        try:
            self.ai_input.setMinimumWidth(0)
        except Exception:
            pass
        try:
            self.ai_input.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
        except Exception:
            try:
                self.ai_input.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
            except Exception:
                pass

        self.ai_history_btn = QToolButton(self.ai_panel)
        try:
            self.ai_history_btn.setText(_('History…'))
        except Exception:
            pass
        try:
            self.ai_history_btn.setToolTip(_('Choose a previous question.'))
        except Exception:
            pass
        self.ai_history_menu = QMenu(self.ai_history_btn)
        try:
            self.ai_history_btn.setMenu(self.ai_history_menu)
            try:
                self.ai_history_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
            except Exception:
                self.ai_history_btn.setPopupMode(QToolButton.InstantPopup)
        except Exception:
            pass

        # Populate history menu
        try:
            self._ai_refresh_history_menu()
        except Exception:
            pass

        # Quick actions (viewer-style)
        self.ai_quick_btn = QToolButton(self.ai_panel)
        try:
            self.ai_quick_btn.setText(_('Quick actions'))
        except Exception:
            pass
        try:
            self.ai_quick_btn.setToolTip(_('Common actions applied to the current selection in Preview.'))
        except Exception:
            pass
        self.ai_quick_menu = QMenu(self.ai_quick_btn)
        try:
            self.ai_quick_btn.setMenu(self.ai_quick_menu)
            try:
                self.ai_quick_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
            except Exception:
                self.ai_quick_btn.setPopupMode(QToolButton.InstantPopup)
        except Exception:
            pass
        try:
            self._ai_refresh_quick_actions_menu()
        except Exception:
            pass

        self.ai_use_sel_btn = QPushButton(_('Use selection'), self.ai_panel)
        self.ai_use_sel_btn.clicked.connect(self.ai_use_selection)
        self.ai_use_sel_btn.setToolTip(_('Copy the currently selected text from Preview into the question box.'))

        self.ai_ask_btn = QPushButton(_('Ask'), self.ai_panel)
        self.ai_ask_btn.clicked.connect(self.ask_ai_about_selection)
        self.ai_ask_btn.setToolTip(_('Send the question to the configured calibre AI provider and show the answer here.'))

        self.ai_summarize_btn = QPushButton(_('Summarize'), self.ai_panel)
        self.ai_summarize_btn.clicked.connect(self.ai_summarize_article)
        self.ai_summarize_btn.setToolTip(_('Summarize the entire article currently shown in Preview.'))

        self.ai_ask_article_btn = QPushButton(_('Ask about Article'), self.ai_panel)
        self.ai_ask_article_btn.clicked.connect(self.ask_ai_about_article)
        self.ai_ask_article_btn.setToolTip(
            _('Ask a custom question about the entire article currently shown in Preview.\n\nExperimental: may fail for long articles depending on your AI provider/token limits.\nTested with OpenRouter (reasoning low).')
        )

        self.ai_help_btn = QToolButton(self.ai_panel)
        try:
            self.ai_help_btn.setText('?')
        except Exception:
            pass
        try:
            self.ai_help_btn.setToolTip(_('AI panel help and limitations.'))
        except Exception:
            pass
        try:
            self.ai_help_btn.clicked.connect(self._show_ai_help)
        except Exception:
            pass

        try:
            ic = QIcon.ic('ai.png')
            if ic is not None and not ic.isNull():
                self.ai_ask_article_btn.setIcon(ic)
        except Exception:
            pass

        self.ai_actions_btn = QToolButton(self.ai_panel)
        try:
            self.ai_actions_btn.setText(_('Copy/Export…'))
        except Exception:
            pass

        try:
            self.ai_actions_btn.setToolTip(
                _('Copy the AI conversation to the clipboard or export it as PDF, Markdown, TXT, DOCX or EPUB.')
            )
        except Exception:
            pass

        try:
            self.ai_actions_menu = QMenu(self.ai_actions_btn)
            act_copy = QAction(_('Copy conversation'), self.ai_actions_menu)
            act_copy.triggered.connect(self.ai_copy_transcript)
            self.ai_actions_menu.addAction(act_copy)
            act_export = QAction(_('Export conversation…'), self.ai_actions_menu)
            act_export.triggered.connect(self.ai_export_transcript)
            self.ai_actions_menu.addAction(act_export)
            self.ai_actions_btn.setMenu(self.ai_actions_menu)
            try:
                self.ai_actions_btn.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
            except Exception:
                try:
                    self.ai_actions_btn.setPopupMode(QToolButton.InstantPopup)
                except Exception:
                    pass
        except Exception:
            self.ai_actions_menu = None

        try:
            # Keep buttons visible (avoid disappearing when dock/splitter gets narrow).
            for b in (self.ai_use_sel_btn, self.ai_ask_btn, self.ai_summarize_btn, self.ai_ask_article_btn):
                try:
                    b.setSizePolicy(QSizePolicy.Policy.Maximum, QSizePolicy.Policy.Fixed)
                except Exception:
                    try:
                        b.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Fixed)
                    except Exception:
                        pass
            try:
                self.ai_help_btn.setSizePolicy(QSizePolicy.Policy.Maximum, QSizePolicy.Policy.Fixed)
            except Exception:
                try:
                    self.ai_help_btn.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Fixed)
                except Exception:
                    pass
            try:
                self.ai_actions_btn.setSizePolicy(QSizePolicy.Policy.Maximum, QSizePolicy.Policy.Fixed)
            except Exception:
                try:
                    self.ai_actions_btn.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Fixed)
                except Exception:
                    pass
            try:
                if getattr(self, 'ai_history_btn', None) is not None:
                    self.ai_history_btn.setSizePolicy(QSizePolicy.Policy.Maximum, QSizePolicy.Policy.Fixed)
            except Exception:
                try:
                    if getattr(self, 'ai_history_btn', None) is not None:
                        self.ai_history_btn.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Fixed)
                except Exception:
                    pass
            try:
                if getattr(self, 'ai_quick_btn', None) is not None:
                    self.ai_quick_btn.setSizePolicy(QSizePolicy.Policy.Maximum, QSizePolicy.Policy.Fixed)
            except Exception:
                try:
                    if getattr(self, 'ai_quick_btn', None) is not None:
                        self.ai_quick_btn.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Fixed)
                except Exception:
                    pass
        except Exception:
            pass

        self.ai_settings_btn = QPushButton(_('Settings'), self.ai_panel)
        self.ai_settings_btn.clicked.connect(self.open_ai_settings)
        self.ai_settings_btn.setToolTip(_('Choose/configure which AI provider calibre should use.'))

        try:
            try:
                self.ai_settings_btn.setSizePolicy(QSizePolicy.Policy.Maximum, QSizePolicy.Policy.Fixed)
            except Exception:
                self.ai_settings_btn.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Fixed)
        except Exception:
            pass

        try:
            compact_push_css = 'QPushButton { font-size: 8pt; padding: 2px 6px; }'
            compact_tool_css = 'QToolButton { font-size: 8pt; padding: 2px 6px; }'
            for b in (self.ai_use_sel_btn, self.ai_ask_btn, self.ai_summarize_btn, self.ai_ask_article_btn, self.ai_settings_btn):
                try:
                    b.setStyleSheet(compact_push_css)
                except Exception:
                    pass
            try:
                self.ai_actions_btn.setStyleSheet(compact_tool_css)
            except Exception:
                pass
            try:
                self.ai_help_btn.setStyleSheet(compact_tool_css)
            except Exception:
                pass
            try:
                if getattr(self, 'ai_history_btn', None) is not None:
                    self.ai_history_btn.setStyleSheet(compact_tool_css)
            except Exception:
                pass
            try:
                if getattr(self, 'ai_quick_btn', None) is not None:
                    self.ai_quick_btn.setStyleSheet(compact_tool_css)
            except Exception:
                pass
        except Exception:
            pass

        # Button icons
        try:
            ic = QIcon.ic('edit-paste.png')
            if ic is not None and not ic.isNull():
                self.ai_use_sel_btn.setIcon(ic)
        except Exception:
            pass
        try:
            ic = QIcon.ic('ai.png')
            if ic is None or ic.isNull():
                ic = QIcon.ic('search.png')
            if ic is not None and not ic.isNull():
                self.ai_ask_btn.setIcon(ic)
        except Exception:
            pass
        try:
            ic = QIcon.ic('document-properties.png')
            if ic is None or ic.isNull():
                ic = QIcon.ic('document.png')
            if ic is None or ic.isNull():
                ic = QIcon.ic('news.png')
            if ic is not None and not ic.isNull():
                self.ai_summarize_btn.setIcon(ic)
        except Exception:
            pass
        try:
            ic = QIcon.ic('save.png')
            if ic is None or ic.isNull():
                ic = QIcon.ic('document_save.png')
            if ic is not None and not ic.isNull():
                self.ai_actions_btn.setIcon(ic)
        except Exception:
            pass
        try:
            ic = QIcon.ic('edit-copy.png')
            if ic is None or ic.isNull():
                ic = QIcon.ic('copy.png')
            if ic is not None and not ic.isNull():
                try:
                    if self.ai_actions_menu is not None:
                        a = self.ai_actions_menu.actions()[0]
                        a.setIcon(ic)
                except Exception:
                    pass
        except Exception:
            pass
        try:
            ic = QIcon.ic('config.png')
            if ic is not None and not ic.isNull():
                self.ai_settings_btn.setIcon(ic)
        except Exception:
            pass

        # Prompt row
        prompt_row = QHBoxLayout()
        prompt_row.setContentsMargins(0, 0, 0, 0)
        prompt_row.addWidget(self.ai_input, 1)
        try:
            side_btns = QVBoxLayout()
            side_btns.setContentsMargins(0, 0, 0, 0)
            try:
                side_btns.setSpacing(4)
            except Exception:
                pass
            side_btns.addWidget(self.ai_history_btn)
            side_btns.addWidget(self.ai_quick_btn)
            prompt_row.addLayout(side_btns)
        except Exception:
            # Fallback to side-by-side
            prompt_row.addWidget(self.ai_history_btn)
            prompt_row.addWidget(self.ai_quick_btn)
        ai_layout.addLayout(prompt_row)

        # Two-row compact action layout
        btn_row1 = QHBoxLayout()
        btn_row1.setContentsMargins(0, 0, 0, 0)
        btn_row1.addWidget(self.ai_use_sel_btn)
        btn_row1.addWidget(self.ai_ask_btn)
        btn_row1.addWidget(self.ai_summarize_btn)
        btn_row1.addStretch(1)
        ai_layout.addLayout(btn_row1)

        btn_row2 = QHBoxLayout()
        btn_row2.setContentsMargins(0, 0, 0, 0)
        btn_row2.addWidget(self.ai_ask_article_btn)
        btn_row2.addWidget(self.ai_actions_btn)
        btn_row2.addWidget(self.ai_settings_btn)
        btn_row2.addWidget(self.ai_help_btn)
        btn_row2.addStretch(1)
        ai_layout.addLayout(btn_row2)

        self.ai_status = QLabel('', self.ai_panel)
        try:
            self.ai_status.setStyleSheet('color: gray;')
        except Exception:
            pass
        try:
            self.ai_status.setMinimumWidth(0)
        except Exception:
            pass
        try:
            self.ai_status.setWordWrap(True)
        except Exception:
            pass
        try:
            self.ai_status.setSizePolicy(QSizePolicy.Policy.Ignored, QSizePolicy.Policy.Fixed)
        except Exception:
            try:
                self.ai_status.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Fixed)
            except Exception:
                pass
        self.ai_status.setToolTip(_('AI provider status. Configure via Settings… if needed.'))
        ai_layout.addWidget(self.ai_status)

        self.ai_output = QTextBrowser(self.ai_panel)
        try:
            self.ai_output.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        except Exception:
            try:
                self.ai_output.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
            except Exception:
                pass
        try:
            self.ai_output.setMinimumWidth(0)
        except Exception:
            pass
        try:
            self.ai_output.setOpenExternalLinks(False)
        except Exception:
            pass
        try:
            self.ai_output.setOpenLinks(False)
        except Exception:
            pass
        try:
            self.ai_output.anchorClicked.connect(self._on_ai_anchor_clicked)
        except Exception:
            pass
        try:
            self.ai_output.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
            self.ai_output.customContextMenuRequested.connect(self.on_ai_output_context_menu)
        except Exception:
            pass
        ai_layout.addWidget(self.ai_output, 1)

        self.ai_dock.setWidget(self.ai_panel)
        self.ai_dock.setAllowedAreas(
            Qt.DockWidgetArea.RightDockWidgetArea
            | Qt.DockWidgetArea.BottomDockWidgetArea
            | Qt.DockWidgetArea.LeftDockWidgetArea
        )
        try:
            self.ai_dock.setFeatures(
                QDockWidget.DockWidgetFeature.DockWidgetClosable
                | QDockWidget.DockWidgetFeature.DockWidgetMovable
                | QDockWidget.DockWidgetFeature.DockWidgetFloatable
            )
        except Exception:
            pass

    def ask_ai_about_article(self):
        try:
            plugin = self._get_ai_provider_plugin()
            if plugin is None or not getattr(plugin, 'is_ready_for_use', False):
                self.update_ai_ui_state()
                return

            question = self._ai_prompt_text()
            if not question:
                QMessageBox.information(self, _('Ask about Article'), _('Please enter a question to ask about the article.'))
                return

            self._ai_remember_prompt(question)

            article_text = self._get_current_article_text()
            if not article_text:
                QMessageBox.information(self, _('Ask about Article'), _('Could not retrieve the article text.'))
                return

            title = ''
            link = ''
            try:
                it = self._current_item or {}
                title = str(it.get('title') or '')
                link = str(it.get('link') or '')
            except Exception:
                pass

            from threading import Thread
            from calibre.ai import ChatMessage, ChatMessageType
            from calibre.ai.utils import StreamedResponseAccumulator

            sys_prompt = (
                'You are a helpful reading assistant. '
                'The user will ask a question about the following article. '
                'Answer the question as best you can, using only the information in the article. '
                'If the question is about the author, their views, or people mentioned, base your answer strictly on the article content.'
            )
            user_prompt = f"Question: {question}\n\nArticle:\n{article_text}\n\nTitle: {title}\nLink: {link}".strip()

            self._ai_sys_prompt = str(sys_prompt)
            self._ai_user_prompt = str(user_prompt)
            self._ai_last_answer = ''

            self._ai_call_id += 1
            call_id = int(self._ai_call_id)
            self._ai_acc = StreamedResponseAccumulator()
            try:
                self.ai_output.setHtml('')
            except Exception:
                pass
            try:
                self.ai_status.setText(_('Asking AI…'))
                self.ai_ask_article_btn.setEnabled(False)
            except Exception:
                pass

            messages = (
                ChatMessage(type=ChatMessageType.system, query=sys_prompt),
                ChatMessage(type=ChatMessageType.user, query=user_prompt),
            )

            def run():
                try:
                    for res in plugin.text_chat(messages, ''):
                        self._ai_dispatch(call_id, res)
                    self._ai_dispatch(call_id, None)
                except Exception as e:
                    from calibre.ai import ChatResponse
                    self._ai_dispatch(call_id, ChatResponse(exception=e, error_details=traceback.format_exc()))
                    self._ai_dispatch(call_id, None)

            self._ai_thread = Thread(name='rss_reader_ai_article', daemon=True, target=run)
            self._ai_thread.start()
        except Exception as e:
            try:
                self._ai_log_error('ask_ai_about_article', str(e), traceback.format_exc())
            except Exception:
                pass
            error_dialog(
                self,
                _('RSS Reader Error'),
                _('Failed to start AI article query: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )
            try:
                self.update_ai_ui_state()
            except Exception:
                pass

    def ai_summarize_article(self):
        try:
            plugin = self._get_ai_provider_plugin()
            if plugin is None or not getattr(plugin, 'is_ready_for_use', False):
                self.update_ai_ui_state()
                return

            article_text = self._get_current_article_text()
            if not article_text:
                QMessageBox.information(self, _('Summarize'), _('Could not retrieve the article text.'))
                return

            title = ''
            link = ''
            try:
                it = self._current_item or {}
                title = str(it.get('title') or '')
                link = str(it.get('link') or '')
            except Exception:
                pass

            from threading import Thread
            from calibre.ai import ChatMessage, ChatMessageType
            from calibre.ai.utils import StreamedResponseAccumulator

            sys_prompt = (
                'You are a helpful reading assistant. '
                'Summarize the following article in two short paragraphs. '
                'Focus on the main ideas and any key conclusions. '
                'Avoid introducing information that is not in the article.'
            )
            user_prompt = f"Article:\n{article_text}\n\nTitle: {title}\nLink: {link}".strip()

            self._ai_sys_prompt = str(sys_prompt)
            self._ai_user_prompt = str(user_prompt)
            self._ai_last_answer = ''

            self._ai_call_id += 1
            call_id = int(self._ai_call_id)
            self._ai_acc = StreamedResponseAccumulator()
            try:
                self.ai_output.setHtml('')
            except Exception:
                pass
            try:
                self.ai_status.setText(_('Summarizing…'))
                self.ai_summarize_btn.setEnabled(False)
            except Exception:
                pass

            messages = (
                ChatMessage(type=ChatMessageType.system, query=sys_prompt),
                ChatMessage(type=ChatMessageType.user, query=user_prompt),
            )

            def run():
                try:
                    for res in plugin.text_chat(messages, ''):
                        self._ai_dispatch(call_id, res)
                    self._ai_dispatch(call_id, None)
                except Exception as e:
                    from calibre.ai import ChatResponse

                    self._ai_dispatch(call_id, ChatResponse(exception=e, error_details=traceback.format_exc()))
                    self._ai_dispatch(call_id, None)

            self._ai_thread = Thread(name='rss_reader_ai_summarize', daemon=True, target=run)
            self._ai_thread.start()
        except Exception as e:
            try:
                self._ai_log_error('ai_summarize_article', str(e), traceback.format_exc())
            except Exception:
                pass
            error_dialog(
                self,
                _('RSS Reader Error'),
                _('Failed to start AI summarize: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )
            try:
                self.update_ai_ui_state()
            except Exception:
                pass
    def update_ai_ui_state(self):
        if not self._ai_supported_by_calibre():
            try:
                for w in (
                    self.ai_input,
                    getattr(self, 'ai_history_btn', None),
                    getattr(self, 'ai_quick_btn', None),
                    self.ai_use_sel_btn,
                    self.ai_ask_btn,
                    self.ai_summarize_btn,
                    self.ai_ask_article_btn,
                    self.ai_actions_btn,
                    self.ai_settings_btn,
                    self.ai_help_btn,
                    self.ai_output,
                ):
                    try:
                        if w is None:
                            continue
                        w.setEnabled(False)
                    except Exception:
                        pass
            except Exception:
                pass
            try:
                self.ai_status.setText(_('AI requires calibre 8.16 or newer.'))
            except Exception:
                pass
            return

        p = self._get_ai_provider_plugin()
        ready = bool(p is not None and getattr(p, 'is_ready_for_use', False))
        try:
            self.ai_input.setEnabled(True)
        except Exception:
            pass
        try:
            self.ai_output.setEnabled(True)
        except Exception:
            pass
        try:
            self.ai_ask_btn.setEnabled(ready)
        except Exception:
            pass
        try:
            self.ai_summarize_btn.setEnabled(ready)
        except Exception:
            pass
        try:
            self.ai_ask_article_btn.setEnabled(ready)
        except Exception:
            pass
        try:
            self.ai_use_sel_btn.setEnabled(True)
        except Exception:
            pass
        try:
            self.ai_actions_btn.setEnabled(True)
        except Exception:
            pass
        try:
            self.ai_settings_btn.setEnabled(True)
        except Exception:
            pass
        try:
            self.ai_help_btn.setEnabled(True)
        except Exception:
            pass
        try:
            if getattr(self, 'ai_history_btn', None) is not None:
                self.ai_history_btn.setEnabled(True)
        except Exception:
            pass
        try:
            if getattr(self, 'ai_quick_btn', None) is not None:
                self.ai_quick_btn.setEnabled(True)
        except Exception:
            pass
        if ready:
            try:
                self.ai_status.setText(_('Ready. Select text in Preview and click Ask, or use Summarize / Ask about Article.'))
            except Exception:
                pass
        else:
            try:
                self.ai_status.setText(_('Not configured. Click Settings… to choose an AI provider.'))
            except Exception:
                pass

    def _ai_prompt_text(self):
        try:
            return str(self.ai_input.toPlainText() or '').strip()
        except Exception:
            return ''

    def _ai_set_prompt_text(self, text):
        try:
            t = str(text or '')
        except Exception:
            t = ''
        try:
            self.ai_input.setPlainText(t)
        except Exception:
            pass

    def _ai_remember_prompt(self, text):
        try:
            t = str(text or '').strip()
        except Exception:
            t = ''
        if not t:
            return
        # Don't store huge prompts
        if len(t) > 500:
            t = t[:500].rstrip()
        try:
            hist = list(gprefs.get('rss_reader_ai_prompt_history', []) or [])
        except Exception:
            hist = []
        try:
            new_hist = []
            for s in hist:
                try:
                    s = str(s or '').strip()
                except Exception:
                    s = ''
                if s and s != t:
                    new_hist.append(s)
            new_hist.insert(0, t)
            new_hist = new_hist[:25]
            try:
                gprefs.set('rss_reader_ai_prompt_history', new_hist)
            except Exception:
                pass

            try:
                self._ai_refresh_history_menu()
            except Exception:
                pass
        except Exception:
            pass

    def _ai_refresh_history_menu(self):
        try:
            menu = getattr(self, 'ai_history_menu', None)
            if menu is None:
                return
            menu.clear()
        except Exception:
            return

        try:
            hist = list(gprefs.get('rss_reader_ai_prompt_history', []) or [])
        except Exception:
            hist = []

        cleaned = []
        seen = set()
        for s in (hist or []):
            try:
                s = str(s or '').strip()
            except Exception:
                s = ''
            if not s or s in seen:
                continue
            seen.add(s)
            cleaned.append(s)

        if not cleaned:
            try:
                a = QAction(_('No history yet'), menu)
                a.setEnabled(False)
                menu.addAction(a)
            except Exception:
                pass
            return

        for s in cleaned[:25]:
            try:
                a = QAction(s if len(s) <= 80 else (s[:77] + '…'), menu)
                a.setToolTip(s)
                a.triggered.connect(lambda _=False, ss=s: self._ai_set_prompt_text(ss))
                menu.addAction(a)
            except Exception:
                pass

        try:
            menu.addSeparator()
            clear_act = QAction(_('Clear history'), menu)
            clear_act.triggered.connect(self._ai_clear_history)
            menu.addAction(clear_act)
        except Exception:
            pass

    def _ai_clear_history(self):
        try:
            gprefs.set('rss_reader_ai_prompt_history', [])
        except Exception:
            try:
                gprefs['rss_reader_ai_prompt_history'] = []
            except Exception:
                pass
        try:
            self._ai_refresh_history_menu()
        except Exception:
            pass
        try:
            self._ai_refresh_quick_actions_menu()
        except Exception:
            pass

    def _ai_refresh_quick_actions_menu(self):
        menu = getattr(self, 'ai_quick_menu', None)
        if menu is None:
            return
        try:
            menu.clear()
        except Exception:
            return

        def _load_viewer_quick_actions_dict():
            # Prefer calibre viewer prefs API (reads viewer-webengine.json)
            try:
                from calibre.gui2.viewer.config import vprefs

                p = vprefs.get('llm_quick_actions') or {}
                if isinstance(p, dict):
                    return p
            except Exception:
                pass

            # Fallback: try reading viewer-webengine.json directly
            try:
                from calibre.utils.config import config_dir

                path = os.path.join(config_dir, 'viewer-webengine.json')
                with open(path, 'rb') as f:
                    raw = f.read()
                try:
                    data = json.loads(raw.decode('utf-8'))
                except Exception:
                    data = json.loads(raw.decode('utf-8-sig'))
                if isinstance(data, dict):
                    p = data.get('llm_quick_actions')
                    if isinstance(p, dict):
                        return p
            except Exception:
                pass
            return {}

        def _default_action_data():
            # Mirror calibre viewer defaults.
            try:
                from calibre.gui2.llm import ActionData

                return (
                    ActionData('explain', _('Explain'), 'Explain the following text in simple, easy to understand language. {selected}'),
                    ActionData('define', _('Define'), 'Identify and define any technical or complex terms in the following text. {selected}'),
                    ActionData('summarize', _('Summarize'), 'Provide a concise summary of the following text. {selected}'),
                    ActionData('points', _('Key points'), 'Extract the key points from the following text as a bulleted list. {selected}'),
                    ActionData('grammar', _('Fix grammar'), 'Correct any grammatical errors in the following text and provide the corrected version. {selected}'),
                    ActionData('translate', _('Translate'), 'Translate the following text into the language {language}. {selected}'),
                )
            except Exception:
                return ()

        viewer_actions = []
        try:
            from calibre.gui2.llm import ActionData

            p = _load_viewer_quick_actions_dict()
            defaults = _default_action_data()
            if defaults:
                viewer_actions = list(ActionData.unserialize(p, defaults, include_disabled=False))
        except Exception:
            viewer_actions = []

        if viewer_actions:
            for ac in viewer_actions:
                try:
                    label = getattr(ac, 'human_name', None) or getattr(ac, 'name', '')
                    prompt_template = getattr(ac, 'prompt_template', '')
                    a = QAction(str(label), menu)
                    a.setToolTip(str(prompt_template or ''))
                    a.triggered.connect(lambda _=False, pt=prompt_template: self._ai_run_quick_action_template(pt))
                    menu.addAction(a)
                except Exception:
                    pass
        else:
            # Fallback defaults
            actions = [
                (_('Define'), 'Define the selected text.'),
                (_('Explain'), 'Explain the selected text.'),
                (_('Fix grammar'), 'Fix grammar and improve clarity of the selected text without changing meaning.'),
                (_('Key points'), 'Extract key points from the selected text as bullet points.'),
                (_('Summarize'), 'Summarize the selected text in a short paragraph.'),
                (_('Translate'), 'Translate the selected text to English.'),
            ]
            for label, instr in actions:
                try:
                    a = QAction(label, menu)
                    a.setToolTip(instr)
                    a.triggered.connect(lambda _=False, ii=instr: self._ai_run_quick_action(ii))
                    menu.addAction(a)
                except Exception:
                    pass

        # User-included: allow running any recent prompt as an action
        try:
            hist = list(gprefs.get('rss_reader_ai_prompt_history', []) or [])
        except Exception:
            hist = []
        cleaned = []
        seen = set()
        for s in (hist or []):
            try:
                s = str(s or '').strip()
            except Exception:
                s = ''
            if not s or s in seen:
                continue
            seen.add(s)
            cleaned.append(s)

        try:
            if cleaned:
                menu.addSeparator()
                sub = menu.addMenu(_('From history'))
                for s in cleaned[:10]:
                    a = QAction(s if len(s) <= 60 else (s[:57] + '…'), sub)
                    a.setToolTip(s)
                    a.triggered.connect(lambda _=False, ii=s: self._ai_run_quick_action(ii))
                    sub.addAction(a)
        except Exception:
            pass

    def _ai_run_quick_action(self, instruction):
        sel = ''
        try:
            sel = self._selected_text_in_preview()
        except Exception:
            sel = ''
        sel = (sel or '').strip()
        if not sel:
            try:
                QMessageBox.information(self, _('Quick actions'), _('Please select some text in Preview first.'))
            except Exception:
                pass
            return
        try:
            self._ai_set_prompt_text(str(instruction or '').strip())
        except Exception:
            pass
        # Run Ask using typed prompt + selection behavior
        try:
            self.ask_ai_about_selection()
        except Exception:
            pass

    def _ai_run_quick_action_template(self, prompt_template):
        sel = ''
        try:
            sel = self._selected_text_in_preview()
        except Exception:
            sel = ''
        sel = (sel or '').strip()
        if not sel:
            try:
                QMessageBox.information(self, _('Quick actions'), _('Please select some text in Preview first.'))
            except Exception:
                pass
            return

        # Calibre viewer uses templates with optional {selected}/{language}. Our Ask code sends
        # the selection separately, so strip {selected} and keep only the instruction.
        pt = ''
        try:
            pt = str(prompt_template or '').strip()
        except Exception:
            pt = ''
        try:
            if '{selected}' in pt:
                pt = pt.replace('{selected}', '').strip()
        except Exception:
            pass
        try:
            from calibre.utils.localization import ui_language_as_english

            lang = ui_language_as_english()
        except Exception:
            lang = 'English'
        try:
            pt = pt.format(language=lang).strip()
        except Exception:
            pass

        try:
            self._ai_set_prompt_text(pt)
        except Exception:
            pass
        try:
            self.ask_ai_about_selection()
        except Exception:
            pass

    def _ai_log_error(self, where, err='', details=''):
        try:
            item = {
                'ts': time.strftime('%Y-%m-%d %H:%M:%S'),
                'where': str(where or '').strip(),
                'err': str(err or '').strip(),
                'details': str(details or '').strip(),
            }
        except Exception:
            return

        try:
            log = list(plugin_prefs.get('ai_error_log', []) or [])
        except Exception:
            log = []

        try:
            log.append(item)
            if len(log) > 200:
                log = log[-200:]
            plugin_prefs['ai_error_log'] = log
            try:
                commit = getattr(plugin_prefs, 'commit', None)
                if callable(commit):
                    commit()
            except Exception:
                pass
        except Exception:
            pass

    def _show_ai_help(self):
        try:
            txt = _(
                'AI panel tips:\n\n'
                '• Ask: uses your typed prompt, or the current selection if the prompt is blank.\n'
                '• Summarize: sends the entire article text currently shown in Preview.\n'
                '• Ask about Article (experimental): sends the entire article + your question, which may exceed provider limits for long pages.\n\n'
                'If you see errors like “Payload Too Large” (HTTP 413), try:\n'
                '• asking about a smaller selection\n'
                '• shortening the question\n'
                '• switching AI provider in Settings\n'
            )
            QMessageBox.information(self, _('AI Help'), txt)
        except Exception:
            pass

    def _get_current_article_text(self):
        # Fetch full article text from Preview or current item
        article_text = ''
        try:
            article_text = self.preview.toPlainText()
        except Exception:
            pass
        if not article_text:
            try:
                it = self._current_item or {}
                article_text = str(it.get('content') or it.get('summary') or '')
            except Exception:
                article_text = ''
        return (article_text or '').strip()

    def _ai_supported_by_calibre(self):
        try:
            from calibre.constants import numeric_version

            try:
                v = tuple(numeric_version)
            except Exception:
                v = (0, 0, 0)
            return v >= (8, 16, 0)
        except Exception:
            return True

    def _get_ai_provider_plugin(self):
        try:
            from calibre.ai import AICapabilities
            from calibre.ai.prefs import plugin_for_purpose

            return plugin_for_purpose(AICapabilities.text_to_text)
        except Exception:
            try:
                self._ai_log_error('_get_ai_provider_plugin', 'Failed to load AI provider plugin', traceback.format_exc())
            except Exception:
                pass
            return None

    def open_ai_settings(self):
        try:
            from calibre.ai import AICapabilities
            from calibre.ai.config import ConfigureAI

            d = QDialog(self)
            d.setWindowTitle(_('Configure AI'))
            l = QVBoxLayout(d)
            w = ConfigureAI(purpose=AICapabilities.text_to_text, parent=d)
            l.addWidget(w)
            bb = QDialogButtonBox(
                QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, d
            )
            l.addWidget(bb)

            def accept():
                if not w.commit():
                    return
                d.accept()

            bb.accepted.connect(accept)
            bb.rejected.connect(d.reject)
            try:
                d.resize(720, 520)
            except Exception:
                pass
            d.exec()
        except Exception as e:
            try:
                self._ai_log_error('open_ai_settings', str(e), traceback.format_exc())
            except Exception:
                pass
            error_dialog(
                self,
                _('RSS Reader Error'),
                _('Failed to open AI settings: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )
        finally:
            try:
                self.update_ai_ui_state()
            except Exception:
                pass

    def _selected_text_in_preview(self):
        try:
            t = self.preview.textCursor().selectedText() or ''
            # Qt uses U+2029 for paragraph breaks
            t = t.replace('\u2029', '\n').strip()
            return t
        except Exception:
            return ''

    def ai_use_selection(self):
        t = self._selected_text_in_preview()
        if not t:
            return
        try:
            self._ai_set_prompt_text(t)
        except Exception:
            pass

    def on_preview_context_menu(self, pos):
        try:
            menu = self.preview.createStandardContextMenu()
        except Exception:
            menu = None
        if menu is None:
            return
        try:
            # Add RSS Reader image actions (Copy/Save/AdBlock) when appropriate.
            try:
                if hasattr(self.preview, 'add_rss_reader_context_actions'):
                    self.preview.add_rss_reader_context_actions(menu, pos)
            except Exception:
                pass

            menu.addSeparator()

            z_in = QAction(_('Zoom In'), self)
            z_in.setShortcut('Ctrl++')
            z_in.triggered.connect(self.zoom_in)
            menu.addAction(z_in)

            z_out = QAction(_('Zoom Out'), self)
            z_out.setShortcut('Ctrl+-')
            z_out.triggered.connect(self.zoom_out)
            menu.addAction(z_out)

            z_reset = QAction(_('Reset Zoom'), self)
            z_reset.setShortcut('Ctrl+0')
            z_reset.triggered.connect(self.zoom_reset)
            menu.addAction(z_reset)

            menu.addSeparator()
            sel = self._selected_text_in_preview()
            act = QAction(_('Ask AI about selection'), self)
            act.setEnabled(bool(sel))
            act.triggered.connect(self.ask_ai_about_selection)
            menu.addAction(act)
            try:
                menu.exec_(self.preview.mapToGlobal(pos))
            except AttributeError:
                menu.exec(self.preview.mapToGlobal(pos))
        finally:
            try:
                menu.deleteLater()
            except Exception:
                pass

    def on_ai_output_context_menu(self, pos):
        try:
            menu = self.ai_output.createStandardContextMenu()
        except Exception:
            menu = None
        if menu is None:
            return
        try:
            menu.addSeparator()

            z_in = QAction(_('Zoom In'), self)
            z_in.setShortcut('Ctrl++')
            z_in.triggered.connect(self.zoom_in)
            menu.addAction(z_in)

            z_out = QAction(_('Zoom Out'), self)
            z_out.setShortcut('Ctrl+-')
            z_out.triggered.connect(self.zoom_out)
            menu.addAction(z_out)

            z_reset = QAction(_('Reset Zoom'), self)
            z_reset.setShortcut('Ctrl+0')
            z_reset.triggered.connect(self.zoom_reset)
            menu.addAction(z_reset)

            menu.exec(self.ai_output.mapToGlobal(pos))
        finally:
            try:
                menu.deleteLater()
            except Exception:
                pass

    def _install_zoom_support(self):
        try:
            self.preview.set_zoom_callback(self._zoom_by_delta)
        except Exception:
            pass

        try:
            def _ai_wheel(event):
                try:
                    mods = event.modifiers() if hasattr(event, 'modifiers') else None
                    shift = bool(mods & Qt.KeyboardModifier.ShiftModifier) if mods is not None else False
                    ctrl = bool(mods & Qt.KeyboardModifier.ControlModifier) if mods is not None else False
                except Exception:
                    shift = False
                    ctrl = False

                # Zoom behavior:
                # - Shift+wheel: AI panel only
                # - Ctrl+wheel: global zoom (same as other panes)
                if shift or ctrl:
                    dy = 0
                    try:
                        dy = int(event.angleDelta().y())
                    except Exception:
                        try:
                            dy = int(event.delta())
                        except Exception:
                            dy = 0
                    if dy:
                        try:
                            if shift:
                                self._ai_zoom_by_delta(1 if dy > 0 else -1)
                            else:
                                self._zoom_by_delta(1 if dy > 0 else -1)
                        except Exception:
                            pass
                        try:
                            event.accept()
                        except Exception:
                            pass
                        return

                try:
                    return QTextBrowser.wheelEvent(self.ai_output, event)
                except Exception:
                    return

            self.ai_output.wheelEvent = _ai_wheel
        except Exception:
            pass

        # Keyboard shortcuts within the dialog (not registered in calibre's global shortcut manager)
        try:
            self._act_zoom_in_1 = QAction(self)
            self._act_zoom_in_1.setShortcut('Ctrl++')
            self._act_zoom_in_1.setShortcutContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
            self._act_zoom_in_1.triggered.connect(self.zoom_in)
            self.addAction(self._act_zoom_in_1)

            self._act_zoom_in_2 = QAction(self)
            self._act_zoom_in_2.setShortcut('Ctrl+=')
            self._act_zoom_in_2.setShortcutContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
            self._act_zoom_in_2.triggered.connect(self.zoom_in)
            self.addAction(self._act_zoom_in_2)

            self._act_zoom_out = QAction(self)
            self._act_zoom_out.setShortcut('Ctrl+-')
            self._act_zoom_out.setShortcutContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
            self._act_zoom_out.triggered.connect(self.zoom_out)
            self.addAction(self._act_zoom_out)

            self._act_zoom_reset = QAction(self)
            self._act_zoom_reset.setShortcut('Ctrl+0')
            self._act_zoom_reset.setShortcutContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
            self._act_zoom_reset.triggered.connect(self.zoom_reset)
            self.addAction(self._act_zoom_reset)
        except Exception:
            pass

        # Restore persisted zoom
        try:
            steps = int(gprefs.get('rss_reader_zoom_steps', 0) or 0)
        except Exception:
            steps = 0

        # Restore persisted AI-only zoom offset (applied on top of global zoom)
        try:
            ai_steps = int(gprefs.get('rss_reader_ai_zoom_steps', 0) or 0)
        except Exception:
            ai_steps = 0

        # Capture base font sizes once so zoom can be applied absolutely.
        try:
            self._zoom_base_points = {}
            for k, w in (
                ('preview', getattr(self, 'preview', None)),
                ('ai_panel', getattr(self, 'ai_panel', None)),
                ('ai_output', getattr(self, 'ai_output', None)),
                ('feeds_tree', getattr(self, 'feeds_tree', None)),
                ('items_table', getattr(self, 'items_table', None)),
            ):
                if w is None:
                    continue
                try:
                    ps = int(w.font().pointSize() or 10)
                except Exception:
                    ps = 10
                self._zoom_base_points[k] = ps
        except Exception:
            self._zoom_base_points = {}

        # Apply AI zoom first so the initial global apply includes it.
        try:
            self._set_ai_zoom_steps(ai_steps, apply_now=False)
        except Exception:
            pass
        self._set_zoom_steps(steps)

        try:
            t0 = float(getattr(self, '_setup_ui_t0', 0) or 0)
        except Exception:
            t0 = 0
        if t0:
            _debug('setup_ui done in %.3fs' % (time.perf_counter() - t0))

    def _apply_zoom_absolute(self):
        """Apply the current zoom as an absolute font-size delta."""

        try:
            steps = int(getattr(self, '_zoom_steps', 0) or 0)
        except Exception:
            steps = 0

        try:
            ai_steps = int(getattr(self, '_ai_zoom_steps', 0) or 0)
        except Exception:
            ai_steps = 0

        def clamp(ps):
            try:
                ps = int(ps)
            except Exception:
                ps = 10
            return max(6, min(40, ps))

        def apply_widget(key, widget, apply_document=False):
            if widget is None:
                return
            base = None
            try:
                base = (getattr(self, '_zoom_base_points', {}) or {}).get(key)
            except Exception:
                base = None
            if base is None:
                try:
                    base = int(widget.font().pointSize() or 10)
                except Exception:
                    base = 10
                try:
                    if not hasattr(self, '_zoom_base_points') or self._zoom_base_points is None:
                        self._zoom_base_points = {}
                    self._zoom_base_points[key] = base
                except Exception:
                    pass
            new_ps = clamp(base + steps)
            try:
                f = widget.font()
                f.setPointSize(new_ps)
                widget.setFont(f)
                if apply_document and hasattr(widget, 'document'):
                    try:
                        widget.document().setDefaultFont(f)
                    except Exception:
                        pass
            except Exception:
                pass

        apply_widget('feeds_tree', getattr(self, 'feeds_tree', None))
        apply_widget('items_table', getattr(self, 'items_table', None))
        apply_widget('preview', getattr(self, 'preview', None), apply_document=True)
        # AI panel: apply global + AI-specific zoom so buttons/controls scale too.
        try:
            apply_widget('ai_output', getattr(self, 'ai_output', None), apply_document=True)
        except Exception:
            pass

        try:
            # If there's an AI-only offset, compute a target point-size and apply it
            # to the whole AI panel (including buttons) so the UI scales uniformly.
            if ai_steps:
                base = None
                try:
                    base = (getattr(self, '_zoom_base_points', {}) or {}).get('ai_panel')
                except Exception:
                    base = None
                if base is None:
                    try:
                        base = int(self.ai_panel.font().pointSize() or 10)
                    except Exception:
                        base = 10
                new_ps = clamp(int(base) + int(steps) + int(ai_steps))

                try:
                    # Apply on panel root so children inherit where possible
                    f = self.ai_panel.font()
                    f.setPointSize(new_ps)
                    self.ai_panel.setFont(f)
                except Exception:
                    pass

                # Ensure specific widgets get their document/default fonts updated
                try:
                    if getattr(self, 'ai_output', None) is not None:
                        try:
                            doc_f = self.ai_output.font()
                            doc_f.setPointSize(new_ps)
                            self.ai_output.setFont(doc_f)
                            try:
                                self.ai_output.document().setDefaultFont(doc_f)
                            except Exception:
                                pass
                        except Exception:
                            pass
                except Exception:
                    pass

                # Buttons and toolbuttons may have compact stylesheets that fix font-size.
                # Override with an explicit stylesheet containing the computed size so
                # buttons scale with the panel.
                btns = (
                    getattr(self, 'ai_use_sel_btn', None),
                    getattr(self, 'ai_ask_btn', None),
                    getattr(self, 'ai_summarize_btn', None),
                    getattr(self, 'ai_ask_article_btn', None),
                    getattr(self, 'ai_settings_btn', None),
                    getattr(self, 'ai_actions_btn', None),
                    getattr(self, 'ai_help_btn', None),
                    getattr(self, 'ai_history_btn', None),
                    getattr(self, 'ai_quick_btn', None),
                )
                for b in btns:
                    if b is None:
                        continue
                    try:
                        bf = b.font()
                        bf.setPointSize(new_ps)
                        b.setFont(bf)
                    except Exception:
                        pass
                    try:
                        # Preserve padding while enforcing font-size
                        b.setStyleSheet('font-size: %dpt; padding: 2px 6px;' % (new_ps,))
                    except Exception:
                        pass

                # AI input area should also get the updated font
                try:
                    if getattr(self, 'ai_input', None) is not None:
                        try:
                            inf = self.ai_input.font()
                            inf.setPointSize(new_ps)
                            self.ai_input.setFont(inf)
                        except Exception:
                            pass
                except Exception:
                    pass
        except Exception:
            pass

    def _set_ai_zoom_steps(self, steps, apply_now=True):
        try:
            steps = int(steps)
        except Exception:
            steps = 0
        # Keep it sane (smaller range than global; it's additive)
        if steps < -10:
            steps = -10
        if steps > 20:
            steps = 20
        self._ai_zoom_steps = steps
        try:
            gprefs['rss_reader_ai_zoom_steps'] = int(steps)
        except Exception:
            pass
        if apply_now:
            try:
                self._apply_zoom_absolute()
            except Exception:
                pass

    def _ai_zoom_by_delta(self, delta):
        try:
            cur = int(getattr(self, '_ai_zoom_steps', 0) or 0)
        except Exception:
            cur = 0
        try:
            self._set_ai_zoom_steps(cur + int(delta))
        except Exception:
            pass

    def _set_zoom_steps(self, steps):
        try:
            steps = int(steps)
        except Exception:
            steps = 0
        # Keep it sane
        if steps < -10:
            steps = -10
        if steps > 20:
            steps = 20

        self._zoom_steps = steps
        try:
            self._apply_zoom_absolute()
        except Exception:
            pass
        try:
            gprefs['rss_reader_zoom_steps'] = int(steps)
        except Exception:
            pass
        # Notify UI (if present) so any zoom indicator can update.
        try:
            if hasattr(self, '_update_zoom_label') and callable(getattr(self, '_update_zoom_label')):
                try:
                    self._update_zoom_label()
                except Exception:
                    pass
        except Exception:
            pass

    def _zoom_by_delta(self, delta):
        try:
            cur = int(getattr(self, '_zoom_steps', 0) or 0)
        except Exception:
            cur = 0
        try:
            self._set_zoom_steps(cur + int(delta))
        except Exception:
            pass

    def zoom_in(self):
        self._zoom_by_delta(1)

    def zoom_out(self):
        self._zoom_by_delta(-1)

    def zoom_reset(self):
        self._set_zoom_steps(0)

    def ask_ai_about_selection(self):
        try:
            plugin = self._get_ai_provider_plugin()
            if plugin is None or not getattr(plugin, 'is_ready_for_use', False):
                self.update_ai_ui_state()
                return

            typed = self._ai_prompt_text()
            sel = self._selected_text_in_preview()

            # Behavior:
            # - If typed prompt is blank -> ask about selection.
            # - If typed prompt is present and selection exists -> treat typed as instruction applied to selection.
            # - If typed prompt is present and no selection -> treat typed as the query.
            typed = (typed or '').strip()
            sel = (sel or '').strip()
            if not typed and not sel:
                return

            if typed:
                self._ai_remember_prompt(typed)

            title = ''
            link = ''
            try:
                it = self._current_item or {}
                title = str(it.get('title') or '')
                link = str(it.get('link') or '')
            except Exception:
                pass

            from threading import Thread
            from calibre.ai import ChatMessage, ChatMessageType
            from calibre.ai.utils import StreamedResponseAccumulator

            if typed and sel:
                sys_prompt = (
                    'You are a helpful reading assistant. '
                    'The user may provide an instruction and a selected excerpt. '
                    'Follow the instruction and apply it to the selected text. '
                    'If translation is requested, translate only the selected text.'
                )
                user_prompt = f"Instruction: {typed}\n\nSelected text:\n{sel}".strip()
                if title or link:
                    user_prompt = f"{user_prompt}\n\nContext:\nTitle: {title}\nLink: {link}".strip()
            else:
                term = typed or sel
                sys_prompt = (
                    'You are a helpful reading assistant. '
                    'When the user provides a long passage or article, write a concise summary in about two short paragraphs, '
                    'focusing only on the main ideas. '
                    'When the user provides a short term or phrase instead, explain it briefly and, if useful, add 2-4 related keywords.'
                )
                user_prompt = term
                if title or link:
                    user_prompt = f"Explain: {term}\n\nContext:\nTitle: {title}\nLink: {link}".strip()

            self._ai_sys_prompt = str(sys_prompt)
            self._ai_user_prompt = str(user_prompt)
            self._ai_last_answer = ''

            self._ai_call_id += 1
            call_id = int(self._ai_call_id)
            self._ai_acc = StreamedResponseAccumulator()
            try:
                self.ai_output.setHtml('')
            except Exception:
                pass
            try:
                self.ai_status.setText(_('Asking AI…'))
                self.ai_ask_btn.setEnabled(False)
            except Exception:
                pass

            messages = (
                ChatMessage(type=ChatMessageType.system, query=sys_prompt),
                ChatMessage(type=ChatMessageType.user, query=user_prompt),
            )

            def run():
                try:
                    for res in plugin.text_chat(messages, ''):
                        self._ai_dispatch(call_id, res)
                    self._ai_dispatch(call_id, None)
                except Exception as e:
                    from calibre.ai import ChatResponse

                    self._ai_dispatch(call_id, ChatResponse(exception=e, error_details=traceback.format_exc()))
                    self._ai_dispatch(call_id, None)

            self._ai_thread = Thread(name='rss_reader_ai', daemon=True, target=run)
            self._ai_thread.start()
        except Exception as e:
            try:
                self._ai_log_error('ask_ai_about_selection', str(e), traceback.format_exc())
            except Exception:
                pass
            error_dialog(
                self,
                _('RSS Reader Error'),
                _('Failed to start AI query: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )
            try:
                self.update_ai_ui_state()
            except Exception:
                pass

    def _on_ai_response(self, call_id, chat_response):
        try:
            if int(call_id) != int(self._ai_call_id):
                return
        except Exception:
            return

        try:
            from calibre.ai.utils import response_to_html
        except Exception:
            response_to_html = None

        if chat_response is None:
            try:
                if self._ai_acc is not None:
                    self._ai_acc.finalize()
                    if self._ai_acc.messages:
                        msg = self._ai_acc.messages[-1]
                        text = getattr(msg, 'query', '')
                        text = (text or '').strip()
                        if text:
                            self._ai_last_answer = text
                        html = response_to_html(text) if (response_to_html and text) else text
                        try:
                            self.ai_output.setHtml(html)
                        except Exception:
                            try:
                                self.ai_output.setPlainText(text)
                            except Exception:
                                pass
                        try:
                            self._apply_zoom_absolute()
                        except Exception:
                            pass
                self.ai_status.setText(_('Done.'))
            except Exception:
                pass
            try:
                self.update_ai_ui_state()
            except Exception:
                pass
            return

        try:
            if getattr(chat_response, 'exception', None) is not None:
                err = str(chat_response.exception)
                details = getattr(chat_response, 'error_details', '') or ''
                try:
                    self._ai_log_error('ai_backend', err, details)
                except Exception:
                    pass
                error_dialog(self, _('AI Error'), _('Talking to AI failed: %s') % err, show=True, det_msg=details)
                try:
                    self.ai_status.setText(_('AI error.'))
                except Exception:
                    pass
                try:
                    self.update_ai_ui_state()
                except Exception:
                    pass
                return

            if self._ai_acc is not None:
                self._ai_acc.accumulate(chat_response)
                text = (self._ai_acc.all_content or self._ai_acc.all_reasoning or '').strip()
                if text:
                    self._ai_last_answer = text
                    html = response_to_html(text) if response_to_html else text
                    try:
                        self.ai_output.setHtml(html)
                    except Exception:
                        try:
                            self.ai_output.setPlainText(text)
                        except Exception:
                            pass
                    try:
                        self._apply_zoom_absolute()
                    except Exception:
                        pass
        except Exception:
            pass

    def _build_ai_transcript(self, as_markdown=False):
        sys_prompt = (self._ai_sys_prompt or '').strip()
        user_prompt = (self._ai_user_prompt or '').strip()
        answer = (self._ai_last_answer or '').strip()

        if not user_prompt and not answer:
            return ''

        if as_markdown:
            parts = ['# RSS Reader – AI session']
            if user_prompt:
                parts.append('\n**User question / selection**\n')
                parts.append('\n')
                parts.append(user_prompt)
            if answer:
                parts.append('\n\n**AI answer**\n')
                parts.append('\n')
                parts.append(answer)
            if sys_prompt:
                parts.append('\n\n---\n')
                parts.append('\n_System prompt (English, internal to RSS Reader):_\n')
                parts.append('\n')
                parts.append(sys_prompt)
            return ''.join(parts).strip()

        # Plain text transcript
        parts = ['RSS Reader – AI session']
        if user_prompt:
            parts.append('\n\nUser question / selection:\n')
            parts.append(user_prompt)
        if answer:
            parts.append('\n\nAI answer:\n')
            parts.append(answer)
        if sys_prompt:
            parts.append('\n\nSystem prompt (English, internal to RSS Reader):\n')
            parts.append(sys_prompt)
        return ''.join(parts).strip()

    def ai_copy_transcript(self):
        try:
            text = self._build_ai_transcript(as_markdown=False)
            if not text:
                QMessageBox.information(self, _('Copy AI conversation'), _('There is no AI conversation to copy yet.'))
                return
            try:
                cb = QApplication.instance().clipboard()
            except Exception:
                cb = None
            if cb is None:
                QMessageBox.information(self, _('Copy AI conversation'), _('Unable to access the clipboard.'))
                return
            cb.setText(text)
            try:
                self.ai_status.setText(_('Conversation copied to clipboard.'))
            except Exception:
                pass
        except Exception as e:
            error_dialog(
                self,
                _('RSS Reader Error'),
                _('Failed to copy AI conversation: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )

    def ai_export_transcript(self):
        try:
            md = self._build_ai_transcript(as_markdown=True)
            plain = self._build_ai_transcript(as_markdown=False)
            if not md and not plain:
                QMessageBox.information(self, _('Export AI conversation'), _('There is no AI conversation to export yet.'))
                return

            from os.path import splitext
            import re
            import datetime


            def _timestamp_suffix():
                try:
                    dt = datetime.datetime.now()
                    return '_' + dt.strftime('%Y-%m-%d_%H-%M-%S')
                except Exception:
                    return ''

            ts_suffix = _timestamp_suffix()

            ts_suffix = _timestamp_suffix()

            initial_name = 'rss_reader_ai_session' + ts_suffix
            fname = choose_save_file(
                self,
                'rss-reader-export-ai',
                _('Export AI conversation'),
                filters=[
                    (_('Markdown files'), ['md']),
                    (_('Text files'), ['txt']),
                    (_('PDF files'), ['pdf']),
                    (_('Word DOCX files'), ['docx']),
                    (_('EPUB files'), ['epub']),
                ],
                all_files=True,
                initial_filename=initial_name + '.md',
            )
            if not fname:
                return

            base, ext = splitext(fname)
            # If enabled, ensure the chosen filename has a timestamp suffix.
            try:
                if ts_suffix:
                    if not re.search(r'_\d{4}-\d{2}-\d{2}_\d{2}-\d{2}-\d{2}$', base):
                        fname = base + ts_suffix + ext
                        base, ext = splitext(fname)
            except Exception:
                pass
            fmt = ext.lower().lstrip('.')
            if not fmt:
                fmt = 'md'

            if fmt in ('md', 'markdown'):
                with open(fname, 'w', encoding='utf-8') as f:
                    f.write(md or plain or '')
                try:
                    self.ai_status.setText(_('Exported to %s') % fname)
                except Exception:
                    pass
                return

            if fmt in ('txt', 'text'):
                with open(fname, 'w', encoding='utf-8') as f:
                    f.write(plain or md or '')
                try:
                    self.ai_status.setText(_('Exported to %s') % fname)
                except Exception:
                    pass
                return

            # For PDF, DOCX, EPUB: use calibre conversion job from a temporary HTML file
            html_parts = [
                '<html><head><meta charset="utf-8"><title>RSS Reader AI session</title></head><body>',
                '<h1>RSS Reader – AI session</h1>',
            ]
            if self._ai_user_prompt:
                html_parts.append('<h2>User question / selection</h2>')
                html_parts.append(
                    '<pre style="white-space:pre-wrap;">%s</pre>'
                    % (
                        self._ai_user_prompt.replace('&', '&amp;')
                        .replace('<', '&lt;')
                        .replace('>', '&gt;')
                    )
                )
            if self._ai_last_answer:
                html_parts.append('<h2>AI answer</h2>')
                html_parts.append(
                    '<div style="white-space:pre-wrap;">%s</div>'
                    % (
                        self._ai_last_answer.replace('&', '&amp;')
                        .replace('<', '&lt;')
                        .replace('>', '&gt;')
                    )
                )
            if self._ai_sys_prompt:
                html_parts.append('<hr><h3>System prompt (English, internal to RSS Reader)</h3>')
                html_parts.append(
                    '<pre style="white-space:pre-wrap; font-size:smaller;">%s</pre>'
                    % (
                        self._ai_sys_prompt.replace('&', '&amp;')
                        .replace('<', '&lt;')
                        .replace('>', '&gt;')
                    )
                )
            html_parts.append('</body></html>')
            html_content = '\n'.join(html_parts)

            if not getattr(self.gui, 'job_manager', None):
                QMessageBox.information(
                    self, _('Export AI conversation'), _('Cannot start conversion job (job manager unavailable).')
                )
                return

            td = tempfile.mkdtemp(prefix='rss_ai_export_')
            src = os.path.join(td, 'ai_session.html')
            with open(src, 'w', encoding='utf-8') as f:
                f.write(html_content)

            try:
                func = 'gui_convert'
                args = [src, fname, []]
                desc = _('Export AI conversation to %s') % fmt.upper()
                job = self.gui.job_manager.run_job(
                    Dispatcher(self._ai_export_job_done), func, args=args, description=desc
                )
                self._ai_export_jobs[job] = (td, fname)
                try:
                    self.gui.jobs_pointer.start()
                except Exception:
                    pass
                try:
                    self.gui.status_bar.show_message(_('AI export queued: %s') % fname, 5000)
                except Exception:
                    pass
            except Exception as e:
                try:
                    shutil.rmtree(td, ignore_errors=True)
                except Exception:
                    pass
                error_dialog(
                    self,
                    _('Export AI conversation'),
                    _('Failed to start AI export: %s') % str(e),
                    show=True,
                    det_msg=traceback.format_exc(),
                )
        except Exception as e:
            error_dialog(
                self,
                _('Export AI conversation'),
                _('Failed to export AI conversation: %s') % str(e),
                show=True,
                det_msg=traceback.format_exc(),
            )

    def _ai_export_job_done(self, job):
        td, out_path = self._ai_export_jobs.pop(job, (None, None))
        try:
            if getattr(job, 'failed', False):
                try:
                    self.gui.job_exception(job)
                except Exception:
                    error_dialog(self, _('Export AI conversation'), _('AI export job failed.'), show=True)
                return

            try:
                self.gui.status_bar.show_message(_('AI export complete: %s') % out_path, 5000)
            except Exception:
                pass
            try:
                self.ai_status.setText(_('Exported to %s') % out_path)
            except Exception:
                pass
        finally:
            if td:
                try:
                    shutil.rmtree(td, ignore_errors=True)
                except Exception:
                    pass

    def _on_preview_anchor_clicked(self, qurl):
        # Allow internal navigation (#anchors) while sending external URLs to calibre opener.
        try:
            s = str(qurl.toString())
        except Exception:
            try:
                s = str(qurl)
            except Exception:
                s = ''
        if not s:
            return
        try:
            if s.startswith('#'):
                try:
                    self.preview.scrollToAnchor(s[1:])
                except Exception:
                    pass
                return
        except Exception:
            pass
        try:
            if '://' not in s and not s.lower().startswith(('mailto:', 'tel:')):
                # Relative link: resolve against current item link if possible
                try:
                    base = (self.selected_item() or {}).get('link') or ''
                except Exception:
                    base = ''
                if base:
                    try:
                        import urllib.parse

                        self._open_url(urllib.parse.urljoin(base, s))
                    except Exception:
                        pass
                return
        except Exception:
            pass
        self._open_url(s)

    def _on_ai_anchor_clicked(self, qurl):
        try:
            s = str(qurl.toString())
        except Exception:
            try:
                s = str(qurl)
            except Exception:
                s = ''
        if s:
            self._open_url(s)
