from __future__ import absolute_import

from functools import partial
import traceback
import uuid
import datetime
import time
import urllib.parse
import queue
import threading
import re
import base64
import os

# Debug control – set True to enable debug logs to calibre console
from calibre_plugins.rss_reader.debug import _debug, DEBUG_RSS_READER

try:
    from qt.core import QAction, QMenu, QTimer, Qt, QApplication, QIcon, QPixmap
except ImportError:
    from PyQt5.Qt import QAction, QMenu, QTimer, Qt, QApplication, QIcon, QPixmap

try:
    from qt.core import QToolButton
except Exception:
    try:
        from PyQt5.Qt import QToolButton
    except Exception:
        QToolButton = None

from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import Dispatcher, error_dialog
from calibre.gui2.notify import get_notifier

from calibre_plugins.rss_reader.config import plugin_prefs
from calibre_plugins.rss_reader import rss_db
load_translations()

_DB_ONBOARDING_ACTIVE = False


# Emoji choices for the configured (default) DB. Kept local to avoid import
# coupling between onboarding and the profiles dialog.
_DB_EMOJI_CHOICES = [
    '📁', '📚', '📖', '📕', '📗', '📘', '📙', '🔖', '📌', '🎧', '🌟', '🛠️', '🏠', '🎯', '🔒', '🧪', '☁️', '🗃️', '🧾', '🏝️', '🖼️', '🖌️', '❤️', '📦', '🗠', '🏔️'
]


def _configured_db_path():
    try:
        p = str(plugin_prefs.get('db_default_path') or '').strip()
    except Exception:
        p = ''
    if not p:
        return ''
    # If the DB path is present and valid, treat it as configured even if
    # the onboarding flag was cleared/reset (avoid surprise onboarding popups).
    try:
        if not os.path.isfile(p):
            return ''
    except Exception:
        return ''
    try:
        rss_db.assert_db_path_allowed(p)
    except Exception:
        return ''
    try:
        if not bool(plugin_prefs.get('db_onboarded', False)):
            plugin_prefs['db_onboarded'] = True
    except Exception:
        pass
    return p


def _db_is_configured():
    return bool(_configured_db_path())


def _suggest_documents_db_path():
    try:
        home = os.path.expanduser('~')
    except Exception:
        home = ''
    docs = ''
    if home:
        docs = os.path.join(home, 'Documents')
    try:
        if docs and os.path.isdir(docs):
            return os.path.join(docs, 'rss_reader_data', rss_db.DB_FILENAME)
    except Exception:
        pass
    return ''


def _suggest_home_db_path():
    try:
        home = os.path.expanduser('~')
    except Exception:
        home = ''
    if not home:
        return ''
    return os.path.join(home, 'rss_reader_data', rss_db.DB_FILENAME)


def _normalize_db_filename(path):
    try:
        p = str(path or '').strip()
    except Exception:
        p = ''
    if not p:
        return ''
    # If user selected a directory, place the DB file inside it.
    # Important: on Windows, a directory name can contain dots (e.g. "rss_reader6.sqlite").
    # If the user entered something with a file extension, treat it as a file path.
    try:
        _base, ext = os.path.splitext(p)
    except Exception:
        ext = ''
    try:
        looks_like_dir = False
        try:
            if p.endswith(os.sep) or (os.altsep and p.endswith(os.altsep)):
                looks_like_dir = True
        except Exception:
            looks_like_dir = False

        # Only treat as directory if it has no extension.
        if not ext:
            try:
                if os.path.isdir(p):
                    looks_like_dir = True
            except Exception:
                pass

        if looks_like_dir:
            p = os.path.join(p.rstrip('\\/'), rss_db.DB_FILENAME)
    except Exception:
        pass

    # If no extension, add a conventional one.
    try:
        _base2, ext2 = os.path.splitext(p)
        if not ext2:
            p = p + '.sqlite'
    except Exception:
        pass
    return p


def _suggest_db_name_from_path(path):
    try:
        p = str(path or '').strip()
    except Exception:
        p = ''
    if not p:
        return ''
    try:
        base = os.path.splitext(os.path.basename(p))[0]
    except Exception:
        base = ''
    try:
        base = str(base or '').strip()
    except Exception:
        base = ''
    if not base:
        return ''
    # Avoid confusing "RSS Reader" everywhere; suggest a neutral profile name.
    if base.lower() in ('rss_reader', 'rss-reader', 'rssreader'):
        return 'Main'
    return base


def _apply_db_choice(parent, db_path, emoji='', name=''):
    """Validate, initialize, and persist the chosen DB path. Returns (ok, err_text)."""
    fname = _normalize_db_filename(db_path)
    if not fname:
        return False, _('Please choose a database file location.')

    # Validate + initialize first; only persist prefs on success.
    try:
        old_path = getattr(rss_db, 'DB_PATH', '')
        old_ro = getattr(rss_db, 'DB_READONLY', False)
    except Exception:
        old_path, old_ro = '', False

    try:
        rss_db.assert_db_path_allowed(fname)
        try:
            os.makedirs(os.path.dirname(fname), exist_ok=True)
        except Exception:
            pass

        rss_db.set_db_path(str(fname), readonly=False)
        rss_db.init_db()
    except Exception as e:
        # Revert global DB path so we don't leave the plugin half-switched.
        try:
            if old_path:
                rss_db.set_db_path(old_path, readonly=old_ro)
        except Exception:
            pass
        return False, str(e)

    try:
        plugin_prefs['db_default_path'] = str(fname)
        plugin_prefs['db_default_readonly'] = False
        plugin_prefs['db_default_mirror'] = False
        plugin_prefs['db_profiles_active'] = ''
        try:
            plugin_prefs['db_default_emoji'] = str(emoji or '').strip()
        except Exception:
            pass
        try:
            nm = str(name or '').strip()
        except Exception:
            nm = ''
        if not nm:
            nm = _suggest_db_name_from_path(fname)
        try:
            plugin_prefs['db_default_name'] = str(nm or '').strip()
        except Exception:
            pass
        plugin_prefs['db_onboarded'] = True
        try:
            commit = getattr(plugin_prefs, 'commit', None)
            if callable(commit):
                commit()
        except Exception:
            pass
    except Exception as e:
        return False, _('Failed to save settings: %s') % str(e)

    return True, ''


def _run_db_onboarding_wizard(parent=None, recommended_path=''):
    """Installer-style onboarding wizard.

    Returns (accepted: bool, chosen_path: str).
    """
    try:
        from qt.core import (
            QComboBox, QDialog, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit, QPushButton,
            QStackedWidget, QWidget, Qt
        )
    except Exception:
        from PyQt5.Qt import (
            QComboBox, QDialog, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit, QPushButton,
            QStackedWidget, QWidget, Qt
        )

    class _Wizard(QDialog):
        def __init__(self, parent=None, recommended_path=''):
            QDialog.__init__(self, parent)
            self.setWindowTitle(_('RSS Reader – Setup'))
            self.setMinimumWidth(490)
            self._recommended = str(recommended_path or '').strip()
            self._chosen_path = ''
            self._chosen_emoji = ''
            self._chosen_name = ''

            outer = QVBoxLayout(self)
            self._stack = QStackedWidget(self)
            outer.addWidget(self._stack)

            # Page 1: Intro
            p1 = QWidget(self)
            p1l = QVBoxLayout(p1)
            title = QLabel('<b>%s</b>' % _('Welcome to RSS Reader'), p1)
            title.setTextFormat(Qt.TextFormat.RichText)
            p1l.addWidget(title)

            intro = QLabel(
                _('RSS Reader stores its data in a SQLite database file.') + '\n\n' +
                _('For reliability, the database must be stored <b>outside</b> calibre\'s config and cache folders.') + '\n\n' +
                _('Click Next to choose where to store the database.'),
                p1,
            )
            try:
                intro.setWordWrap(True)
                intro.setTextFormat(Qt.TextFormat.RichText)
            except Exception:
                pass
            p1l.addWidget(intro)
            p1l.addStretch(1)
            self._stack.addWidget(p1)

            # Page 2: Choose location
            p2 = QWidget(self)
            p2l = QVBoxLayout(p2)
            title2 = QLabel('<b>%s</b>' % _('Choose database location'), p2)
            title2.setTextFormat(Qt.TextFormat.RichText)
            p2l.addWidget(title2)

            hint = QLabel(_('Choose a folder on a durable drive.'), p2)
            try:
                hint.setWordWrap(True)
            except Exception:
                pass
            p2l.addWidget(hint)

            row = QHBoxLayout()
            self._path_edit = QLineEdit(p2)
            try:
                self._path_edit.setPlaceholderText(_('e.g. %s') % (self._recommended or 'C:\\RSS_Reader\\rss_reader.sqlite'))
            except Exception:
                pass
            row.addWidget(self._path_edit, 1)
            self._browse_btn = QPushButton(_('Browse…'), p2)
            row.addWidget(self._browse_btn)
            p2l.addLayout(row)

            # Optional name for the configured DB (shown in UI). Defaults from filename.
            try:
                name_row = QHBoxLayout()
                name_row.addWidget(QLabel(_('Profile name:'), p2))
                self._name_edit = QLineEdit(p2)
                try:
                    cur_name = str(plugin_prefs.get('db_default_name') or '').strip()
                except Exception:
                    cur_name = ''
                try:
                    if cur_name:
                        self._name_edit.setText(cur_name)
                except Exception:
                    pass
                try:
                    self._name_edit.setPlaceholderText(_('e.g. %s') % (_suggest_db_name_from_path(self._recommended) or 'Main'))
                except Exception:
                    pass
                name_row.addWidget(self._name_edit, 1)
                p2l.addLayout(name_row)
            except Exception:
                self._name_edit = None

            # Optional emoji for the configured DB
            try:
                emoji_row = QHBoxLayout()
                emoji_row.addWidget(QLabel(_('Profile badge (optional):'), p2))
                self._emoji_cb = QComboBox(p2)
                try:
                    self._emoji_cb.setEditable(True)
                except Exception:
                    pass
                try:
                    # Keep legacy behavior: a blank/empty choice first.
                    self._emoji_cb.addItem('')
                    for e in _DB_EMOJI_CHOICES:
                        self._emoji_cb.addItem(e)
                except Exception:
                    pass
                try:
                    cur = str(plugin_prefs.get('db_default_emoji') or '').strip()
                except Exception:
                    cur = ''
                try:
                    if cur:
                        self._emoji_cb.setCurrentText(cur)
                except Exception:
                    pass
                emoji_row.addWidget(self._emoji_cb, 1)
                p2l.addLayout(emoji_row)
            except Exception:
                self._emoji_cb = None

            self._err = QLabel('', p2)
            try:
                self._err.setWordWrap(True)
            except Exception:
                pass
            try:
                self._err.setStyleSheet('color: #b00020;')
            except Exception:
                pass
            p2l.addWidget(self._err)
            p2l.addStretch(1)
            self._stack.addWidget(p2)

            # Buttons
            btns = QHBoxLayout()
            btns.addStretch(1)
            self._back = QPushButton(_('< Back'), self)
            self._next = QPushButton(_('Next >'), self)
            self._cancel = QPushButton(_('Cancel'), self)
            btns.addWidget(self._back)
            btns.addWidget(self._next)
            btns.addWidget(self._cancel)
            outer.addLayout(btns)

            self._back.clicked.connect(self._go_back)
            self._next.clicked.connect(self._go_next)
            self._cancel.clicked.connect(self.reject)
            self._browse_btn.clicked.connect(self._browse)

            self._update_buttons()

            # Prefill recommended path for convenience (user can edit).
            try:
                if self._recommended:
                    self._path_edit.setText(self._recommended)
                    self._path_edit.selectAll()
            except Exception:
                pass

            # If no saved name, prefill from recommended filename.
            try:
                if getattr(self, '_name_edit', None) is not None:
                    if not str(self._name_edit.text() or '').strip():
                        # Prefer blank over "RSS Reader" repetition; suggest a neutral label.
                        self._name_edit.setText(_suggest_db_name_from_path(self._recommended) or 'Main')
            except Exception:
                pass

        def _update_buttons(self):
            idx = int(self._stack.currentIndex())
            self._back.setEnabled(idx > 0)
            # This wizard is one step in a larger onboarding flow; keep
            # the call-to-action consistent.
            self._next.setText(_('Next >'))

        def _go_back(self):
            try:
                self._stack.setCurrentIndex(max(0, self._stack.currentIndex() - 1))
            except Exception:
                pass
            self._update_buttons()

        def _go_next(self):
            idx = int(self._stack.currentIndex())
            last = (idx >= (self._stack.count() - 1))
            if not last:
                try:
                    self._stack.setCurrentIndex(idx + 1)
                except Exception:
                    pass
                self._update_buttons()
                return

            # Finish: validate path.
            try:
                self._err.setText('')
            except Exception:
                pass

            try:
                p = str(self._path_edit.text() or '').strip()
            except Exception:
                p = ''
            if not p:
                try:
                    self._err.setText(_('Please choose a database file location.'))
                except Exception:
                    pass
                return
            self._chosen_path = p
            try:
                if getattr(self, '_emoji_cb', None) is not None:
                    self._chosen_emoji = str(self._emoji_cb.currentText() or '').strip()
            except Exception:
                self._chosen_emoji = ''
            try:
                if getattr(self, '_name_edit', None) is not None:
                    self._chosen_name = str(self._name_edit.text() or '').strip()
            except Exception:
                self._chosen_name = ''
            self.accept()

        def _browse(self):
            # Only open the file dialog when the user clicks Browse.
            fname = ''
            try:
                from calibre.gui2.qt_file_dialogs import choose_save_file
            except Exception:
                choose_save_file = None

            try:
                initial = ''
                try:
                    initial = str(self._path_edit.text() or '').strip()
                except Exception:
                    initial = ''
                if not initial:
                    initial = self._recommended or ''

                if choose_save_file is not None:
                    fname = choose_save_file(
                        self,
                        'rss-reader-onboarding-db',
                        _('Choose RSS Reader database location'),
                        filters=[(_('SQLite database files'), ['sqlite', 'db'])],
                        all_files=True,
                        initial_path=initial or None,
                    )
                else:
                    try:
                        from qt.core import QFileDialog
                    except Exception:
                        from PyQt5.Qt import QFileDialog
                    fname, _dummy = QFileDialog.getSaveFileName(
                        self,
                        _('Choose RSS Reader database location'),
                        initial,
                        _('SQLite files (*.sqlite *.db);;All files (*)'),
                    )
            except Exception:
                fname = ''

            try:
                if hasattr(fname, 'toLocalFile'):
                    fname = fname.toLocalFile()
            except Exception:
                pass
            try:
                fname = str(fname or '').strip()
            except Exception:
                fname = ''
            if not fname:
                return

            try:
                self._path_edit.setText(fname)
            except Exception:
                pass

    dlg = _Wizard(parent=parent, recommended_path=recommended_path)
    try:
        r = dlg.exec_()
    except Exception:
        try:
            r = dlg.exec()
        except Exception:
            r = 0
    try:
        accepted = bool(r)
    except Exception:
        accepted = False
    chosen = ''
    try:
        chosen = str(getattr(dlg, '_chosen_path', '') or '').strip()
    except Exception:
        chosen = ''
    chosen_emoji = ''
    try:
        chosen_emoji = str(getattr(dlg, '_chosen_emoji', '') or '').strip()
    except Exception:
        chosen_emoji = ''
    chosen_name = ''
    try:
        chosen_name = str(getattr(dlg, '_chosen_name', '') or '').strip()
    except Exception:
        chosen_name = ''
    return accepted, chosen, chosen_emoji, chosen_name


def _run_db_onboarding_if_needed(parent=None, action=None):
    """Interactive first-run flow for choosing a durable DB location.

    Runs only when `db_onboarded` is false.
    """
    global _DB_ONBOARDING_ACTIVE

    # Prevent re-entrancy (e.g. multiple show_dialog invocations).
    try:
        if _DB_ONBOARDING_ACTIVE:
            return
    except Exception:
        pass

    # Only consider onboarding complete if we have a configured default path.
    if _db_is_configured():
        return True

    _DB_ONBOARDING_ACTIVE = True
    try:
        try:
            suggested_auto = str(getattr(rss_db, 'suggested_default_db_path', lambda: '')() or '').strip()
        except Exception:
            suggested_auto = ''
        suggested_docs = _suggest_documents_db_path()
        suggested_home = _suggest_home_db_path()
        recommended = suggested_auto or suggested_docs or suggested_home

        accepted, chosen, chosen_emoji, chosen_name = _run_db_onboarding_wizard(parent=parent, recommended_path=recommended)
        if not accepted:
            return False

        ok, err = _apply_db_choice(parent, chosen, emoji=chosen_emoji, name=chosen_name)
        if not ok:
            try:
                error_dialog(parent, _('RSS Reader – Database location'), _('That database location cannot be used: %s') % str(err),
                            show=True, det_msg=traceback.format_exc())
            except Exception:
                pass
            return False

        # After DB is set up, offer to add bundled sample feeds
        try:
            if not bool(plugin_prefs.get('bundled_feeds_onboarded', False)):
                _run_bundled_feeds_onboarding(parent=parent, action=action)
        except Exception:
            pass

        return True
    finally:
        _DB_ONBOARDING_ACTIVE = False


def _run_bundled_feeds_onboarding(parent=None, action=None):
    """Show bundled feeds dialog during onboarding."""
    from calibre_plugins.rss_reader.debug import _debug
    try:
        from calibre_plugins.rss_reader.bundled_feeds_dialog import BundledFeedsDialog
    except Exception:
        try:
            from bundled_feeds_dialog import BundledFeedsDialog
        except Exception as e:
            _debug('Could not import BundledFeedsDialog:', e)
            return

    try:
        from qt.core import QDialog
    except Exception:
        try:
            from PyQt5.Qt import QDialog
        except Exception:
            QDialog = None

    dlg = BundledFeedsDialog(parent, skip_label=True)
    try:
        res = dlg.exec()
    except Exception:
        try:
            res = dlg.exec_()
        except Exception:
            res = 0

    accepted_code = None
    try:
        accepted_code = getattr(getattr(QDialog, 'DialogCode', QDialog), 'Accepted')
    except Exception:
        try:
            accepted_code = getattr(QDialog, 'Accepted')
        except Exception:
            accepted_code = 1

    if res == accepted_code:
        entries = getattr(dlg, 'selected_entries', None)
        urls = getattr(dlg, 'selected_urls', None)
        if entries is None:
            entries = []
        if urls is None:
            urls = []

        # Prefer entries (keeps names/lang metadata); fall back to URLs.
        payload = entries if entries else urls
        if payload:
            try:
                from calibre_plugins.rss_reader.sample_feeds import add_sample_feeds
                dbp = _configured_db_path() or rss_db.db_path()
                added, skipped = add_sample_feeds(payload, db_path=dbp, folder='Featured')
            except Exception as e:
                try:
                    error_dialog(parent, _('RSS Reader – Featured feeds'), _('Failed to add featured feeds: %s') % str(e),
                                show=True, det_msg=traceback.format_exc())
                except Exception:
                    pass
                return

            try:
                from qt.core import QMessageBox
            except Exception:
                try:
                    from PyQt5.Qt import QMessageBox
                except Exception:
                    QMessageBox = None

            if QMessageBox is not None:
                try:
                    if skipped:
                        QMessageBox.information(parent, _('RSS Reader – Featured feeds'),
                                                _('Added %d feed(s). (%d already existed)') % (added, skipped))
                    else:
                        QMessageBox.information(parent, _('RSS Reader – Featured feeds'),
                                                _('Added %d feed(s).') % added)
                except Exception:
                    pass

            # After adding feeds, offer to fetch them immediately.
            try:
                if QMessageBox is not None and int(added or 0) > 0:
                    try:
                        yn = QMessageBox.question(
                            parent,
                            _('RSS Reader'),
                            _('Fetch the newly added feeds now?'),
                            QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
                            QMessageBox.StandardButton.Yes,
                        )
                    except Exception:
                        yn = QMessageBox.Yes

                    try:
                        yes_btn = QMessageBox.StandardButton.Yes
                    except Exception:
                        yes_btn = getattr(QMessageBox, 'Yes', None)

                    if yn == yes_btn or yn == getattr(QMessageBox, 'Yes', None):
                        # Defer actual update until the main RSS Reader dialog opens
                        # so progress can be shown there.
                        try:
                            plugin_prefs['onboarding_fetch_after_add'] = True
                        except Exception:
                            pass
            except Exception:
                pass

            # If user asked to add feeds but none were added, keep onboarded False so we can retry.
            if added <= 0 and len(payload) > 0:
                return

    # User either added feeds successfully or explicitly skipped.
    try:
        plugin_prefs['bundled_feeds_onboarded'] = True
    except Exception:
        pass


def _add_bundled_feeds(urls, parent=None, show_errors=False):
    """Add a list of feed URLs to the database. Returns number added."""
    try:
        urls = [str(u or '').strip() for u in (urls or [])]
        urls = [u for u in urls if u]
        if not urls:
            return 0

        try:
            rss_db.ensure_ready()
        except Exception:
            pass

        try:
            existing = list(rss_db.get_feeds() or [])
        except Exception:
            existing = []

        existing_by_url = {}
        try:
            for f in existing:
                try:
                    u = str((f or {}).get('url') or '').strip()
                except Exception:
                    u = ''
                if u:
                    existing_by_url[u] = f
        except Exception:
            existing_by_url = {}

        added_count = 0
        for url in urls:
            if url in existing_by_url:
                continue
            try:
                feed = _ensure_feed_dict(url)
                existing.append(feed)
                existing_by_url[url] = feed
                added_count += 1
            except Exception:
                continue

        if added_count > 0:
            rss_db.save_feeds(existing)
        return int(added_count)
    except Exception as e:
        if bool(show_errors):
            try:
                error_dialog(parent, _('RSS Reader – Sample feeds'), _('Failed to add sample feeds: %s') % str(e),
                            show=True, det_msg=traceback.format_exc())
            except Exception:
                pass
        return 0


def _ensure_feed_dict(url, title=''):
    # Ensure new feeds always have Calibre-like default limits unless explicitly overridden.
    try:
        default_oldest = int(plugin_prefs.get('default_oldest_article_days', 7) or 7)
    except Exception:
        default_oldest = 7
    try:
        default_max = int(plugin_prefs.get('default_max_articles', 100) or 100)
    except Exception:
        default_max = 100

    return {
        'id': str(uuid.uuid4()),
        'title': title or url,
        'url': url,
        'enabled': True,
        # Empty folder == root under the Feeds container.
        'folder': str(plugin_prefs.get('default_folder', '') or ''),
        'download_images': True,
        'always_notify': False,
        'feed_starred': False,
        'is_recipe': False,
        'recipe_urn': '',
        'use_recipe_engine': False,
        'oldest_article_days': max(0, int(default_oldest)),
        'max_articles': max(0, int(default_max)),
    }


def _coerce_int(value, default):
    try:
        if value is None:
            return int(default)
        if isinstance(value, str) and not value.strip():
            return int(default)
        return int(value)
    except Exception:
        return int(default)


def _feed_int_setting(feed, key, default):
    # Use default only when key is missing; respect explicit 0.
    try:
        has_key = isinstance(feed, dict) and (key in feed)
    except Exception:
        has_key = False
    if not has_key:
        return _coerce_int(default, default)
    try:
        return _coerce_int(feed.get(key), default)
    except Exception:
        return _coerce_int(default, default)


def _iso_to_ts(iso_text):
    try:
        s = (iso_text or '').strip()
        if not s:
            return 0
        # rss.py normalizes to ISO 8601 with offset (usually +00:00)
        dt = datetime.datetime.fromisoformat(s)
        if dt.tzinfo is None:
            dt = dt.replace(tzinfo=datetime.timezone.utc)
        return int(dt.timestamp())
    except Exception:
        return 0


def _extract_icon_hrefs(html_text):
    # Very small, permissive parser for <link rel="...icon..." href="...">
    if not html_text:
        return []
    tags = re.findall(r'(?is)<link\b[^>]*>', html_text)
    out = []
    for tag in tags:
        try:
            m_href = re.search(r'href\s*=\s*["\']([^"\']+)["\']', tag, re.I)
            if not m_href:
                continue
            href = m_href.group(1).strip()
            m_rel = re.search(r'rel\s*=\s*["\']([^"\']+)["\']', tag, re.I)
            rel_l = (m_rel.group(1).lower() if m_rel else '')
            m_sizes = re.search(r'sizes\s*=\s*["\']\s*(\d+)\s*[xX]\s*(\d+)\s*["\']', tag, re.I)
            try:
                sizes = int(m_sizes.group(1)) * int(m_sizes.group(2)) if m_sizes else 0
            except Exception:
                sizes = 0
            out.append((href, sizes, rel_l))
        except Exception:
            continue
    return out


def _short_status_text(s, max_len=90):
    try:
        s = str(s or '').strip()
    except Exception:
        s = ''
    if not s:
        return ''
    if len(s) <= max_len:
        return s
    return s[:max(0, int(max_len) - 1)].rstrip() + '…'


def _update_worker(feeds, seen_item_ids, known_item_ids, timeout_seconds=12, max_known_per_feed=200, abort=None, log=None, notifications=None):
    # NOTE: This runs in calibre's main process (ThreadedJob), so plugin modules are importable.
    from calibre_plugins.rss_reader.rss import fetch_url, parse_feed
    from calibre_plugins.rss_reader.recipe_utils import is_recipe_urn, get_recipe_feeds_from_urn
    from calibre_plugins.rss_reader.error_tagging import classify_error

    results = {
        'feeds': {},
    }

    total = float(len(feeds) or 1)
    # Helper: run fetch_url in a short-lived thread and remain responsive to `abort`.
    def _fetch_with_abort(u, timeout_seconds_local):
        try:
            result = [None, None]
            exc = [None]

            def _fn():
                try:
                    r, f = fetch_url(u, timeout_seconds=timeout_seconds_local)
                    result[0], result[1] = r, f
                except Exception as e:
                    exc[0] = e

            t = threading.Thread(target=_fn, daemon=True)
            t.start()

            # Wait in small increments so we can abort quickly.
            waited = 0.0
            poll = 0.25
            maxwait = float(timeout_seconds_local or 0) * 1.5 + 5.0
            while t.is_alive():
                if abort is not None and abort.is_set():
                    return None, None
                t.join(poll)
                waited += poll
                if waited > maxwait:
                    return None, None

            if exc[0] is not None:
                raise exc[0]
            return result[0], result[1]
        except Exception:
            # Keep behaviour similar to fetch_url: propagate to caller which handles exceptions.
            raise

    # Filter to feeds that are actually eligible to fetch.
    feeds_in_scope = []
    for feed in (feeds or []):
        try:
            if abort is not None and abort.is_set():
                break
            if not feed.get('enabled', True):
                continue
            feed_id = str(feed.get('id') or '')
            url = feed.get('url') or ''
            if not feed_id or not url:
                continue
            feeds_in_scope.append(feed)
        except Exception:
            continue

    total_int = int(len(feeds_in_scope) or 1)
    total = float(total_int)

    def _process_one(feed):
        """Fetch + parse a single feed. Always returns (feed_id, result_dict, title_for_status)."""
        feed_id = ''
        url = ''
        parsed = None
        try:
            if abort is not None and abort.is_set():
                return '', None, ''
            if not feed.get('enabled', True):
                return '', None, ''

            feed_id = str(feed.get('id') or '')
            url = feed.get('url') or ''
            if not feed_id or not url:
                return '', None, ''

            items = []
            # Recipes use URNs like builtin:techcrunch; resolve to real RSS feed URLs.
            if bool(feed.get('is_recipe')) or is_recipe_urn(url):
                urn = str(feed.get('recipe_urn') or url)
                recipe_title = (feed.get('title') or urn)
                subfeeds = get_recipe_feeds_from_urn(urn)
                if not subfeeds:
                    raise ValueError('Recipe provides no feeds: %s' % urn)

                first_parsed = None
                combined = []
                subfeed_errors = []
                for sub_title, sub_url in subfeeds:
                    if abort is not None and abort.is_set():
                        break
                    if not sub_url:
                        continue

                    try:
                        raw, final_url = _fetch_with_abort(sub_url, timeout_seconds)
                        p = parse_feed(raw, base_url=final_url)
                        if first_parsed is None and isinstance(p, dict):
                            first_parsed = p
                        for it in (p.get('items') or []):
                            try:
                                it = dict(it)
                            except Exception:
                                continue
                            # Preserve source feed context (useful for debugging/UI later)
                            try:
                                it['_recipe_feed_title'] = str(sub_title or '')
                                it['_recipe_feed_url'] = str(sub_url or '')
                            except Exception:
                                pass
                            combined.append(it)
                    except Exception as e:
                        try:
                            subfeed_errors.append('%s: %s' % (sub_url, str(e)))
                        except Exception:
                            subfeed_errors.append(str(e))
                        continue

                items = combined
                if not items:
                    raise ValueError('All recipe feeds failed: %s' % ('; '.join(subfeed_errors[:4]) or urn))
                parsed = {
                    'title': recipe_title,
                    'link': (first_parsed or {}).get('link') if isinstance(first_parsed, dict) else '',
                    'image_url': (first_parsed or {}).get('image_url') if isinstance(first_parsed, dict) else '',
                }
                final_url = ''
            else:
                raw, final_url = _fetch_with_abort(url, timeout_seconds)
                parsed = parse_feed(raw, base_url=final_url)
                items = parsed.get('items') or []

            # Dedup + new detection
            # - known_item_ids: previously fetched IDs (to determine what's *new*)
            # - seen_item_ids: read IDs (used by UI unread counters)
            prev_known = list(known_item_ids.get(feed_id) or [])
            prev_known_set = set(prev_known)
            first_fetch = not bool(prev_known)

            normalized_items = []
            dedup_ids = set()

            is_recipe_feed = bool(feed.get('is_recipe')) or is_recipe_urn(url)

            def _canonicalize_link(u):
                try:
                    import urllib.parse
                    uu = str(u or '').strip()
                    if not uu:
                        return ''
                    p = urllib.parse.urlsplit(uu)
                    if not p.scheme:
                        return uu
                    # Drop fragments and common tracking parameters
                    try:
                        q = urllib.parse.parse_qsl(p.query, keep_blank_values=True)
                        drop = {'fbclid', 'gclid', 'igshid', 'mc_cid', 'mc_eid'}
                        q2 = [(k, v) for (k, v) in q if not (k.lower().startswith('utm_') or k.lower() in drop)]
                        query = urllib.parse.urlencode(q2, doseq=True)
                    except Exception:
                        query = p.query
                    path = p.path or ''
                    # Normalize trailing slash (but keep root '/')
                    if path.endswith('/') and path != '/':
                        path = path[:-1]
                    return urllib.parse.urlunsplit((p.scheme, p.netloc, path, query, ''))
                except Exception:
                    try:
                        return str(u or '').strip()
                    except Exception:
                        return ''

            for it in items:
                # Prefer stable, cross-feed IDs for recipe items to avoid duplicates across sections.
                try:
                    link0 = str(it.get('link') or '').strip()
                except Exception:
                    link0 = ''

                if is_recipe_feed and link0:
                    base_iid = 'url:' + _canonicalize_link(link0)
                else:
                    base_iid = str(it.get('id') or link0 or it.get('title') or '')

                iid = str(base_iid or '').strip()
                if not iid:
                    continue

                if iid in dedup_ids:
                    continue
                dedup_ids.add(iid)

                it = dict(it)
                it['id'] = iid
                it['published_ts'] = _iso_to_ts(it.get('published') or '')
                normalized_items.append(it)

            # Keep items newest-first for consistent UI + cache
            try:
                normalized_items.sort(key=lambda x: int(x.get('published_ts') or 0), reverse=True)
            except Exception:
                pass

            # Keep a copy for fallback behavior when oldest-days filter empties the feed.
            try:
                normalized_items_before_age = list(normalized_items)
            except Exception:
                normalized_items_before_age = []

            # Apply per-feed oldest/max restrictions if configured
            try:
                default_oldest = _coerce_int(plugin_prefs.get('default_oldest_article_days', 7), 7)
            except Exception:
                default_oldest = 7
            try:
                oldest_days = _feed_int_setting(feed, 'oldest_article_days', default_oldest)
            except Exception:
                oldest_days = int(default_oldest)
            if oldest_days and oldest_days > 0:
                try:
                    cutoff = int(time.time()) - int(oldest_days) * 86400
                    # Keep items with a published_ts newer than cutoff; keep items with no published_ts
                    normalized_items = [it for it in normalized_items if (int(it.get('published_ts') or 0) >= cutoff) or not int(it.get('published_ts') or 0)]
                except Exception:
                    pass

            # If a newly-added feed has no items within the oldest-days window,
            # show a small sample of the newest items so the feed doesn't appear empty.
            # This mirrors Calibre UX expectations for first-time fetches.
            try:
                if first_fetch and (oldest_days and oldest_days > 0) and not normalized_items and normalized_items_before_age:
                    normalized_items = list(normalized_items_before_age)[:3]
            except Exception:
                pass
            try:
                default_max = _coerce_int(plugin_prefs.get('default_max_articles', 100), 100)
            except Exception:
                default_max = 100
            try:
                max_articles = _feed_int_setting(feed, 'max_articles', default_max)
            except Exception:
                max_articles = int(default_max)
            if max_articles and max_articles > 0:
                try:
                    normalized_items = normalized_items[:int(max_articles)]
                except Exception:
                    pass

            # Recompute new IDs after applying limits
            try:
                new_ids = [it.get('id') for it in normalized_items if it.get('id') not in prev_known_set]
            except Exception:
                new_ids = []
            new_set = set(new_ids)
            new_items = [it for it in normalized_items if it.get('id') in new_set]

            # Try to find a favicon for this feed (heuristics)
            icon_bytes = None
            icon_headers = {
                'Accept': 'image/png,image/x-icon,image/vnd.microsoft.icon,image/jpeg,image/gif,image/*;q=0.9,*/*;q=0.8',
            }

            # Heartbeat: icon probing can be slow (multiple network attempts).
            try:
                if notifications is not None:
                    title0 = _short_status_text((parsed or {}).get('title') or feed.get('title') or url)
                    notifications.put((None, _('Looking up site icon: %s') % title0))
            except Exception:
                pass

            def _looks_like_image(raw):
                try:
                    if not raw or len(raw) < 16:
                        return False
                    b = bytes(raw)
                    if b.startswith(b'\x89PNG\r\n\x1a\n'):
                        return True
                    if b.startswith(b'\x00\x00\x01\x00') or b.startswith(b'\x00\x00\x02\x00'):
                        return True
                    if b.startswith(b'\xff\xd8\xff'):
                        return True
                    if b.startswith(b'GIF87a') or b.startswith(b'GIF89a'):
                        return True
                    if b.startswith(b'RIFF') and b[8:12] == b'WEBP':
                        return True
                    head = b.lstrip()[:32].lower()
                    if head.startswith(b'<!doctype') or head.startswith(b'<html') or head.startswith(b'<?xml'):
                        return False
                except Exception:
                    return False
                return False

            def _try_favicon_from(candidate_url):
                try:
                    pu = urllib.parse.urlparse(candidate_url or '')
                    if not (pu.scheme and pu.netloc):
                        return None
                    base = f"{pu.scheme}://{pu.netloc}/"

                    favicon_url = f"{pu.scheme}://{pu.netloc}/favicon.ico"
                    try:
                        fb, _ = fetch_url(favicon_url, timeout_seconds=12, headers=icon_headers)
                        if _looks_like_image(fb):
                            return fb
                    except Exception:
                        fb = None

                    try:
                        if pu.netloc.lower().endswith('.dreamwidth.org') and pu.netloc.lower() != 'www.dreamwidth.org':
                            fb_dw, _ = _fetch_with_abort('https://www.dreamwidth.org/favicon.ico', 6)
                            if _looks_like_image(fb_dw):
                                return fb_dw
                    except Exception:
                        pass

                    try:
                        domain = pu.netloc
                        if domain:
                            s2 = f"https://www.google.com/s2/favicons?domain={urllib.parse.quote(domain)}&sz=128"
                            fb2, _ = _fetch_with_abort(s2, 8)
                            if _looks_like_image(fb2):
                                return fb2
                    except Exception:
                        pass

                    try:
                        raw_html, final_home = _fetch_with_abort(base, 8)
                        try:
                            html = raw_html.decode('utf-8', 'ignore')
                        except Exception:
                            html = str(raw_html)
                        candidates = _extract_icon_hrefs(html)

                        def _rank(t):
                            href, sz, rel = t
                            href_l = href.lower()
                            ext_score = 0
                            if href_l.endswith('.ico'):
                                ext_score = 30
                            elif href_l.endswith('.png'):
                                ext_score = 20
                            elif href_l.endswith('.svg'):
                                ext_score = 10
                            rel_score = 5 if 'shortcut' in rel else 0
                            return (int(sz or 0), ext_score, rel_score)

                        candidates.sort(key=_rank, reverse=True)
                        for href, _sz, _rel in candidates[:6]:
                            icon_url = href
                            if icon_url.startswith('//'):
                                icon_url = f"{pu.scheme}:{icon_url}"
                            icon_url = urllib.parse.urljoin(final_home or base, icon_url)
                            try:
                                fb3, _ = _fetch_with_abort(icon_url, 8)
                                if _looks_like_image(fb3):
                                    return fb3
                            except Exception:
                                continue
                    except Exception:
                        pass

                    return None
                except Exception:
                    return None

            try:
                try:
                    img_url = str((parsed or {}).get('image_url') or '').strip()
                    if img_url:
                        fb0, _ = fetch_url(img_url, timeout_seconds=12, headers=icon_headers)
                        if _looks_like_image(fb0):
                            icon_bytes = fb0
                except Exception:
                    pass

                icon_bytes = icon_bytes or (
                    _try_favicon_from(final_url) or
                    _try_favicon_from((parsed or {}).get('link') or '') or
                    _try_favicon_from(url)
                )
            except Exception:
                icon_bytes = None

            try:
                if notifications is not None:
                    title0 = _short_status_text((parsed or {}).get('title') or feed.get('title') or url)
                    notifications.put((None, _('Finished icon lookup: %s') % title0))
            except Exception:
                pass

            merged_known = new_ids + prev_known
            if len(merged_known) > max_known_per_feed:
                merged_known = merged_known[:max_known_per_feed]

            prev_seen = list(seen_item_ids.get(feed_id) or [])
            if len(prev_seen) > max_known_per_feed:
                prev_seen = prev_seen[:max_known_per_feed]

            title_for_status = ((parsed or {}).get('title') or feed.get('title') or url)
            return feed_id, {
                'ok': True,
                'title': title_for_status,
                'home_link': (parsed or {}).get('link') or '',
                'feed_type': (parsed or {}).get('feed_type') or '',
                'feed_encoding': (parsed or {}).get('feed_encoding') or '',
                'items': normalized_items,
                'new_count': len(new_ids),
                'new_ids': new_ids,
                'new_items': new_items,
                'first_fetch': first_fetch,
                'seen_item_ids': prev_seen,
                'known_item_ids': merged_known,
                'icon': icon_bytes,
            }, title_for_status
        except Exception as e:
            if log is not None:
                try:
                    log('RSS Reader: feed failed:', url)
                    log.exception('Error:')
                except Exception:
                    pass
            title_for_status = (feed.get('title') or url)
            # Store a concise classification for auto-tagging/sanitization.
            try:
                tags, http_status, kind = classify_error(str(e), traceback.format_exc())
            except Exception:
                tags, http_status, kind = ['failed'], None, ''
            return feed_id, {
                'ok': False,
                'title': title_for_status,
                'url': url,
                'error': str(e),
                'auto_tags': list(tags or []),
                'http_status': http_status,
                'error_kind': kind,
                'traceback': traceback.format_exc(),
                'items': [],
                'new_count': 0,
                'new_ids': [],
                'seen_item_ids': list(seen_item_ids.get(feed_id) or []),
                'known_item_ids': list(known_item_ids.get(feed_id) or []),
            }, title_for_status

    # Concurrency: bounded parallel fetches (default is conservative).
    try:
        max_workers = int(plugin_prefs.get('max_parallel_fetches', 4) or 4)
    except Exception:
        max_workers = 4
    max_workers = max(1, min(16, max_workers))

    if max_workers <= 1 or len(feeds_in_scope) <= 1:
        for idx, feed in enumerate(feeds_in_scope):
            if abort is not None and abort.is_set():
                break
            if notifications is not None:
                try:
                    title0 = _short_status_text(feed.get('title') or (feed.get('url') or ''))
                    notifications.put(((idx + 0.05) / total, _('Fetching %d/%d: %s') % (idx + 1, total_int, title0)))
                except Exception:
                    pass

            feed_id, r, title_for_status = _process_one(feed)
            if feed_id and isinstance(r, dict):
                results['feeds'][feed_id] = r

            if notifications is not None:
                try:
                    title1 = _short_status_text(title_for_status or (feed.get('title') or feed.get('url') or ''))
                    notifications.put(((idx + 1.0) / total, _('Fetched %d/%d: %s') % (idx + 1, total_int, title1)))
                except Exception:
                    pass
        return results

    # Parallel mode
    try:
        from concurrent.futures import ThreadPoolExecutor, as_completed
    except Exception:
        # Fallback to sequential if concurrent.futures isn't available.
        for idx, feed in enumerate(feeds_in_scope):
            if abort is not None and abort.is_set():
                break
            feed_id, r, _title = _process_one(feed)
            if feed_id and isinstance(r, dict):
                results['feeds'][feed_id] = r
        return results

    if notifications is not None:
        try:
            notifications.put((0.0, _('Fetching %d feeds (%d workers)...') % (total_int, max_workers)))
        except Exception:
            pass

    done = 0
    with ThreadPoolExecutor(max_workers=max_workers, thread_name_prefix='rss_reader_fetch') as ex:
        futs = [ex.submit(_process_one, f) for f in feeds_in_scope]
        for fut in as_completed(futs):
            if abort is not None and abort.is_set():
                break
            try:
                feed_id, r, title_for_status = fut.result()
            except Exception:
                continue
            if feed_id and isinstance(r, dict):
                results['feeds'][feed_id] = r
            done += 1
            if notifications is not None:
                try:
                    title1 = _short_status_text(title_for_status or '')
                    notifications.put(((float(done) / total), _('Fetched %d/%d: %s') % (done, total_int, title1)))
                except Exception:
                    pass

    return results


class RSSReaderAction(InterfaceAction):
    name = 'RSS Reader'
    action_spec = (_('RSS Reader'), None, _('Read RSS/Atom feeds'), None)

    # Allow a menu on the toolbar button while keeping normal click behavior.
    try:
        if QToolButton is not None and hasattr(QToolButton, 'ToolButtonPopupMode'):
            popup_type = QToolButton.ToolButtonPopupMode.MenuButtonPopup
        elif QToolButton is not None and hasattr(QToolButton, 'MenuButtonPopup'):
            popup_type = QToolButton.MenuButtonPopup
    except Exception:
        pass

    def rebuild_icon(self, *args):
        try:
            from calibre_plugins.rss_reader.common_icons import get_icon, calibre_version
            # On older calibre, prefer a deterministic icon that we ship.
            if tuple(calibre_version) < (6, 2, 0):
                icon = get_icon('images/iconplugin_light.png')
            else:
                icon = get_icon('images/iconplugin')
            if icon and not icon.isNull():
                try:
                    self.qaction.setIcon(icon)
                except Exception:
                    pass
                try:
                    if self._dialog is not None:
                        self._dialog.setWindowIcon(icon)
                except Exception:
                    pass
                # Also update the system tray icon if present
                try:
                    if self._dialog is not None and hasattr(self._dialog, '_update_tray_icon'):
                        self._dialog._update_tray_icon()
                except Exception:
                    pass
        except Exception:
            pass

    def _send_notification(self, title, message, icon_bytes=None, url=None, timeout=6500):
        """Safe notification: only use Calibre's notifier (no external commands).

        Writes `icon_bytes` to a temporary PNG and passes the path to the notifier
        when possible, otherwise calls the notifier without an icon.
        """
        try:
            # Prefer Calibre notifier API
            if hasattr(self.notifier, 'notify'):
                if icon_bytes:
                    tmp = None
                    try:
                        import tempfile, os
                        fd, tmp = tempfile.mkstemp(suffix='.png')
                        with os.fdopen(fd, 'wb') as f:
                            f.write(icon_bytes)
                        try:
                            self.notifier.notify(title=title, msg=message, icon=tmp, url=url)
                        except Exception:
                            # fallback to no-icon notify
                            try:
                                self.notifier.notify(title=title, msg=message, url=url)
                            except Exception:
                                pass
                    finally:
                        try:
                            if tmp and os.path.exists(tmp):
                                os.remove(tmp)
                        except Exception:
                            pass
                else:
                    try:
                        self.notifier.notify(title=title, msg=message, url=url)
                    except Exception:
                        pass
            else:
                # Notifier may be callable
                try:
                    if callable(self.notifier):
                        try:
                            self.notifier(title, message, timeout=timeout)
                        except TypeError:
                            # older callable signature
                            self.notifier(title, message)
                except Exception:
                    pass
        except Exception:
            try:
                # last resort
                if callable(self.notifier):
                    self.notifier(title, message)
            except Exception:
                pass

    def genesis(self):
        self.notifier = get_notifier()
        self._popup_notification = None

        try:
            self._last_serbian_graphia = str(plugin_prefs.get('serbian_graphia', 'latin') or 'latin').strip().lower()
        except Exception:
            self._last_serbian_graphia = 'latin'
        try:
            self._last_russian_graphia = str(plugin_prefs.get('russian_graphia', 'native') or 'native').strip().lower()
        except Exception:
            self._last_russian_graphia = 'native'
        try:
            self._last_japanese_graphia = str(plugin_prefs.get('japanese_graphia', 'native') or 'native').strip().lower()
        except Exception:
            self._last_japanese_graphia = 'native'
        try:
            self._last_chinese_graphia = str(plugin_prefs.get('chinese_graphia', 'native') or 'native').strip().lower()
        except Exception:
            self._last_chinese_graphia = 'native'

        # Do NOT create any DB on plugin startup. Only initialize when a
        # configured db_default_path exists (or after onboarding completes).
        try:
            p = _configured_db_path()
            if p:
                try:
                    rss_db.set_db_path(p, readonly=False)
                except Exception:
                    pass
                try:
                    rss_db.ensure_ready()
                except Exception:
                    pass
        except Exception:
            pass

        # Theme-aware plugin/toolbar icon (dark/light variants).
        try:
            from calibre_plugins.rss_reader.common_icons import set_plugin_icon_resources, get_icon
            try:
                icon_resources = self.load_resources([
                    'images/iconplugin_dark.png',
                    'images/iconplugin_light.png',
                    # Provide toolbar icon resources so UI can use them via get_icon()
                    'images/zoom_in.png',
                    'images/zoom_out.png',
                    # Also include the reset icon so it is available when the plugin is installed as a zip
                    'images/zoom-reset.png',
                ])
            except Exception:
                icon_resources = {}
            set_plugin_icon_resources(self.name, icon_resources)
            self.rebuild_icon()
        except Exception:
            pass

        # Update icon instantly when the calibre palette/theme changes
        try:
            app = QApplication.instance()
            if app:
                try:
                    app.paletteChanged.connect(self.rebuild_icon)
                except Exception:
                    pass
        except Exception:
            pass

        # Single (solid) toolbar button: click opens the reader.
        # Configuration remains available via Preferences -> Plugins -> Customize,
        # and via the Settings button in the dialog.
        self.qaction.triggered.connect(self.show_dialog)

        # Toolbar menu (for quick actions and to suspend fetching)
        try:
            self.menu = QMenu(self.gui)
            act_open = QAction(_('Open RSS Reader'), self.gui)
            act_open.triggered.connect(self.show_dialog)
            self.menu.addAction(act_open)

            act_settings = QAction(_('Settings…'), self.gui)
            act_settings.triggered.connect(self.show_settings)
            self.menu.addAction(act_settings)
            self.menu.addSeparator()

            self._act_suspend_fetching = QAction(_('Suspend fetching'), self.gui)
            self._act_suspend_fetching.setCheckable(True)
            self._act_suspend_fetching.setToolTip(_('Pause all RSS network fetching (auto and manual) without uninstalling the plugin.'))
            self._act_suspend_fetching.setChecked(bool(plugin_prefs.get('suspend_fetching', False)))
            self._act_suspend_fetching.toggled.connect(self.set_fetching_suspended)
            self.menu.addAction(self._act_suspend_fetching)

            self.menu.addSeparator()

            # In-app popup notifications (QuiteRSS-style)
            try:
                self._act_in_app_popup = QAction(_('Use in-app popup notifications'), self.gui)
                self._act_in_app_popup.setCheckable(True)
                self._act_in_app_popup.setChecked(bool(plugin_prefs.get('notify_in_app_popup', False)))
                self._act_in_app_popup.toggled.connect(self._set_notify_in_app_popup)
                self.menu.addAction(self._act_in_app_popup)
            except Exception:
                self._act_in_app_popup = None

            # Quick preview/replay for testing popup theme colors
            try:
                self._act_preview_popup = QAction(_('Show last / test notification'), self.gui)
                self._act_preview_popup.setToolTip(_('Replay the last popup notification, or show a test sample if none available.'))
                self._act_preview_popup.triggered.connect(self.preview_in_app_popup)
                self.menu.addAction(self._act_preview_popup)
            except Exception:
                self._act_preview_popup = None

            try:
                self.menu.addSeparator()
            except Exception:
                pass
        except Exception:
            self.menu = None
            self._act_suspend_fetching = None
            self._act_in_app_popup = None

        # Register actions in Preferences -> Keyboard shortcuts.
        # Use persist_shortcut=True so user changes are remembered.
        # We avoid hard-coded default keybindings to reduce conflicts.
        self._act_update_all = QAction(_('RSS Reader: Update all feeds'), self.gui)
        self._act_update_all.triggered.connect(lambda: self.update_all_feeds(silent=False))

        self._act_update_silent = QAction(_('RSS Reader: Update all feeds (silent)'), self.gui)
        self._act_update_silent.triggered.connect(lambda: self.update_all_feeds(silent=True))

        try:
            if self.menu is not None:
                self.menu.addAction(self._act_update_all)
                self.menu.addAction(self._act_update_silent)
                self.qaction.setMenu(self.menu)
        except Exception:
            pass

        # Add Profiles action to the main toolbar button context menu
        try:
            if self.menu is not None:
                self.menu.addSeparator()
                # Add Profiles submenu (profile switcher)
                self._profiles_submenu = QMenu(_('Profiles'), self.menu)
                self._profiles_submenu.aboutToShow.connect(self._rebuild_profiles_submenu)
                self.menu.addMenu(self._profiles_submenu)
        except Exception:
            pass

        # Ensure these actions are attached to the main window so
        # keyboard shortcuts registered below can actually trigger.
        try:
            self._safe_add_action(self._act_update_all)
            self._safe_add_action(self._act_update_silent)
            if self._act_suspend_fetching is not None:
                self._safe_add_action(self._act_suspend_fetching)
        except Exception:
            pass

        try:
            self.gui.keyboard.register_shortcut(
                'plugin:rss_reader:open',
                'RSS Reader: open',
                description='Open the RSS Reader window.',
                action=self.qaction,
                group='RSS Reader',
                persist_shortcut=True,
            )
            self.gui.keyboard.register_shortcut(
                'plugin:rss_reader:update_all',
                'RSS Reader: update all feeds',
                description='Fetch updates for all enabled feeds.',
                action=self._act_update_all,
                group='RSS Reader',
                persist_shortcut=True,
            )
            self.gui.keyboard.register_shortcut(
                'plugin:rss_reader:update_all_silent',
                'RSS Reader: update all feeds (silent)',
                description='Fetch updates for all enabled feeds without pop-up notifications.',
                action=self._act_update_silent,
                group='RSS Reader',
                persist_shortcut=True,
            )
            if self._act_suspend_fetching is not None:
                self.gui.keyboard.register_shortcut(
                    'plugin:rss_reader:suspend_fetching',
                    'RSS Reader: suspend fetching',
                    description='Toggle suspending all RSS network fetching (auto and manual).',
                    action=self._act_suspend_fetching,
                    group='RSS Reader',
                    persist_shortcut=True,
                )

            # Register all main toolbar actions as shortcuts
            # These are created in RSSReaderDialog.setup_ui and attached to self._toolbar_items
            # We use the same group and persist_shortcut=True for all
            try:
                dlg = getattr(self, '_dialog', None)
                if dlg and hasattr(dlg, '_toolbar_items'):
                    for tid, btn in (getattr(dlg, '_toolbar_items', {}) or {}).items():
                        if btn is not None and hasattr(btn, 'text'):
                            label = str(btn.text() or tid).strip()
                            shortcut_id = f'plugin:rss_reader:toolbar:{tid}'
                            shortcut_name = f'RSS Reader toolbar: {label}'
                            self.gui.keyboard.register_shortcut(
                                shortcut_id,
                                shortcut_name,
                                description=f'Toolbar button: {label}',
                                action=btn,
                                group='RSS Reader',
                                persist_shortcut=True,
                            )
            except Exception:
                pass

            # DB menu shortcuts (appear in Preferences -> Keyboard shortcuts and work globally)
            try:
                self._db_menu_shortcut_actions = {}
                self._ai_menu_shortcut_actions = {}

                def _with_dialog(callable_on_dialog):
                    try:
                        self.show_dialog()
                    except Exception:
                        return
                    try:
                        dlg = getattr(self, '_dialog', None)
                    except Exception:
                        dlg = None
                    if dlg is None:
                        return
                    try:
                        callable_on_dialog(dlg)
                    except Exception:
                        pass

                def _make_shortcut(unique_name, display_name, default_keys, handler, description):
                    try:
                        act = QAction(display_name, self.gui)
                        try:
                            act.setToolTip(str(description or display_name))
                        except Exception:
                            pass
                        try:
                            act.triggered.connect(handler)
                        except Exception:
                            try:
                                act.triggered.connect(lambda _checked=False: handler())
                            except Exception:
                                pass
                        try:
                            # Scope these shortcuts to the RSS Reader window.
                            # We attach the actions to the dialog when it is shown.
                            try:
                                act.setShortcutContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
                            except Exception:
                                try:
                                    act.setShortcutContext(Qt.WidgetWithChildrenShortcut)
                                except Exception:
                                    pass
                        except Exception:
                            pass
                        try:
                            pass
                        except Exception:
                            pass
                        try:
                            self.gui.keyboard.register_shortcut(
                                unique_name,
                                display_name,
                                description=description,
                                action=act,
                                group='RSS Reader',
                                default_keys=tuple(default_keys or ()),
                                persist_shortcut=True,
                            )
                        except Exception:
                            pass
                        try:
                            self._db_menu_shortcut_actions[unique_name] = act
                        except Exception:
                            pass
                    except Exception:
                        pass

                def _make_ai_shortcut(unique_name, display_name, button_attr, description):
                    try:
                        act = QAction(display_name, self.gui)
                        try:
                            act.setToolTip(str(description or display_name))
                        except Exception:
                            pass

                        def _handler():
                            _with_dialog(lambda d: getattr(getattr(d, button_attr, None), 'click', lambda: None)())

                        try:
                            act.triggered.connect(_handler)
                        except Exception:
                            try:
                                act.triggered.connect(lambda _checked=False: _handler())
                            except Exception:
                                pass
                        try:
                            act.setShortcutContext(Qt.ShortcutContext.WidgetWithChildrenShortcut)
                        except Exception:
                            try:
                                act.setShortcutContext(Qt.WidgetWithChildrenShortcut)
                            except Exception:
                                pass

                        try:
                            self.gui.keyboard.register_shortcut(
                                unique_name,
                                display_name,
                                description=description,
                                action=act,
                                group='RSS Reader',
                                default_keys=tuple(),
                                persist_shortcut=True,
                            )
                        except Exception:
                            pass
                        try:
                            self._ai_menu_shortcut_actions[unique_name] = act
                        except Exception:
                            pass
                    except Exception:
                        pass

                _make_shortcut(
                    'plugin:rss_reader:db:manage_profiles',
                    'RSS Reader DB: manage profiles',
                    ('Ctrl+Shift+P',),
                    lambda: _with_dialog(lambda d: d._show_profiles_manager() if hasattr(d, '_show_profiles_manager') else None),
                    'Open the DB Profiles manager.',
                )
                _make_shortcut(
                    'plugin:rss_reader:db:open_oneoff',
                    'RSS Reader DB: open database (one-off)',
                    ('Ctrl+O',),
                    lambda: _with_dialog(lambda d: d._on_db_open() if hasattr(d, '_on_db_open') else None),
                    'Open a database temporarily (one-off).',
                )
                _make_shortcut(
                    'plugin:rss_reader:db:save_profile',
                    'RSS Reader DB: save current DB as profile',
                    ('Ctrl+Shift+S',),
                    lambda: _with_dialog(lambda d: d._save_current_db_as_profile() if hasattr(d, '_save_current_db_as_profile') else None),
                    'Save the current database as a named profile.',
                )
                _make_shortcut(
                    'plugin:rss_reader:db:create_blank',
                    'RSS Reader DB: create blank database',
                    ('Ctrl+Alt+Shift+N',),
                    lambda: _with_dialog(lambda d: d._create_blank_database() if hasattr(d, '_create_blank_database') else None),
                    'Create a new empty database and switch to it.',
                )
                _make_shortcut(
                    'plugin:rss_reader:db:switch_back',
                    'RSS Reader DB: switch back',
                    ('Alt+Left',),
                    lambda: _with_dialog(lambda d: d._switch_back_database() if hasattr(d, '_switch_back_database') else None),
                    'Switch back to the previous database.',
                )
                _make_shortcut(
                    'plugin:rss_reader:db:show_path',
                    'RSS Reader DB: show current DB path',
                    ('Ctrl+Shift+L',),
                    lambda: _with_dialog(lambda d: d._show_current_db_path() if hasattr(d, '_show_current_db_path') else None),
                    'Show the full path of the currently active database.',
                )

                # AI panel button shortcuts (listed in Preferences -> Keyboard shortcuts).
                # No defaults; user assigns keys.
                _make_ai_shortcut(
                    'plugin:rss_reader:ai:ai_ask_btn',
                    'RSS Reader AI: Ask',
                    'ai_ask_btn',
                    'AI panel button: Ask',
                )
                _make_ai_shortcut(
                    'plugin:rss_reader:ai:ai_use_sel_btn',
                    'RSS Reader AI: Use selection',
                    'ai_use_sel_btn',
                    'AI panel button: Use selection',
                )
                _make_ai_shortcut(
                    'plugin:rss_reader:ai:ai_summarize_btn',
                    'RSS Reader AI: Summarize',
                    'ai_summarize_btn',
                    'AI panel button: Summarize',
                )
                _make_ai_shortcut(
                    'plugin:rss_reader:ai:ai_ask_article_btn',
                    'RSS Reader AI: Ask article',
                    'ai_ask_article_btn',
                    'AI panel button: Ask article',
                )
                _make_ai_shortcut(
                    'plugin:rss_reader:ai:ai_settings_btn',
                    'RSS Reader AI: Settings',
                    'ai_settings_btn',
                    'AI panel button: Settings',
                )
                _make_ai_shortcut(
                    'plugin:rss_reader:ai:ai_actions_btn',
                    'RSS Reader AI: Actions',
                    'ai_actions_btn',
                    'AI panel button: Actions',
                )

                try:
                    self.gui.keyboard.finalize()
                except Exception:
                    pass
            except Exception:
                pass
        except Exception:
            pass

        self._dialog = None
        self._update_job = None  # legacy (no longer used)
        self._progress_queue = None
        self._progress_timer = None

        # Guard against UI updates during calibre shutdown. Some Qt objects
        # (Dispatcher/QTimer) can be deleted before worker callbacks fire.
        self._shutting_down = False

        self._update_thread = None
        self._update_abort = None
        self._update_silent = False

        self._dispatch_update_done = Dispatcher(self._update_done_from_thread)

        # Optional periodic updates
        self._timer = QTimer(self.gui)
        self._timer.timeout.connect(lambda: self.update_all_feeds(silent=False))
        self.apply_settings()

        self._pending_new_count = 0

        # Apply suspended state to toolbar label/tooltip on startup
        try:
            self._refresh_toolbar_label()
        except Exception:
            pass

    def show_settings(self):
        """Open the RSS Reader settings dialog."""
        try:
            from calibre_plugins.rss_reader.config import ConfigDialog
        except Exception:
            try:
                from config import ConfigDialog
            except Exception:
                ConfigDialog = None
        if ConfigDialog is None:
            return

        try:
            d = getattr(self, '_settings_dialog', None)
        except Exception:
            d = None

        # Reuse existing dialog instance if still alive
        try:
            if d is not None and hasattr(d, 'isVisible') and d.isVisible():
                try:
                    d.showNormal()
                    # Windows: toggle StaysOnTop to force to front
                    _cf = d.windowFlags()
                    if not (int(_cf) & int(Qt.WindowStaysOnTopHint)):
                        d.setWindowFlag(Qt.WindowStaysOnTopHint, True)
                        d.show()
                        d.setWindowFlag(Qt.WindowStaysOnTopHint, False)
                        d.show()
                    d.raise_()
                    d.activateWindow()
                except Exception:
                    pass
                return
        except Exception:
            pass

        try:
            d = ConfigDialog(self.gui, parent=self.gui)
        except Exception:
            try:
                d = ConfigDialog(self.gui)
            except Exception:
                return

        try:
            self._settings_dialog = d
        except Exception:
            pass

        try:
            def _clear():
                try:
                    if getattr(self, '_settings_dialog', None) is d:
                        self._settings_dialog = None
                except Exception:
                    pass
            d.finished.connect(_clear)
        except Exception:
            pass

        try:
            d.showNormal()
            d.raise_()
            d.activateWindow()
        except Exception:
            try:
                d.exec()
            except Exception:
                try:
                    d.exec_()
                except Exception:
                    pass

    def shutting_down(self):
        # Called by calibre during application shutdown. Ensure our modeless
        # window is closed and any background work is stopped so calibre can exit.
        try:
            self._shutting_down = True
        except Exception:
            pass

        def _qt_alive(obj):
            if obj is None:
                return False
            # sip.isdeleted() is the most reliable way to check PyQt wrapped objects.
            try:
                import sip
            except Exception:
                try:
                    from PyQt5 import sip  # type: ignore
                except Exception:
                    sip = None
            try:
                if sip is not None and hasattr(sip, 'isdeleted') and sip.isdeleted(obj):
                    return False
            except Exception:
                pass
            return True
        try:
            if getattr(self, '_timer', None) is not None:
                try:
                    if _qt_alive(self._timer):
                        self._timer.stop()
                except Exception:
                    pass
        except Exception:
            pass

        # Cancel any active update
        try:
            if getattr(self, '_update_abort', None) is not None:
                try:
                    self._update_abort.set()
                except Exception:
                    pass
        except Exception:
            pass

        try:
            if getattr(self, '_progress_timer', None) is not None:
                try:
                    if _qt_alive(self._progress_timer):
                        self._progress_timer.stop()
                except Exception:
                    pass
        except Exception:
            pass

        # Close the dialog window if it is open
        try:
            d = getattr(self, '_dialog', None)
            if d is not None:
                try:
                    d.close()
                except Exception:
                    pass
                try:
                    d.deleteLater()
                except Exception:
                    pass
        except Exception:
            pass

        try:
            self._dialog = None
        except Exception:
            pass

        # Close any lingering popup
        try:
            self._close_in_app_popup()
        except Exception:
            pass

        # Clear pending notifications to avoid COM deadlocks during shutdown.
        # Prefer the notifier API if available; otherwise fall back to calling
        # the WinToast extension directly (best-effort, safe during shutdown).
        try:
            cleared = False
            if getattr(self, 'notifier', None) is not None:
                clear = getattr(self.notifier, 'clear', None)
                if callable(clear):
                    try:
                        clear()
                        cleared = True
                    except Exception:
                        cleared = False

            if not cleared:
                # Best-effort fallback: call the wintoast extension directly to
                # clear any pending native toast notifications. This does not
                # change runtime notification behavior; it only attempts to
                # release COM resources on Windows during shutdown.
                try:
                    try:
                        from calibre_extensions.wintoast import clear_notifications
                        clear_notifications()
                        cleared = True
                    except Exception:
                        # Try alternate import path
                        try:
                            import calibre_extensions.wintoast as _w
                            if hasattr(_w, 'clear_notifications'):
                                _w.clear_notifications()
                                cleared = True
                        except Exception:
                            pass
                except Exception:
                    pass
        except Exception:
            pass

        try:
            # Unregister toolbar action wrappers and finalize keyboard manager
            try:
                self._unregister_toolbar_shortcuts()
            except Exception:
                pass
        except Exception:
            pass

    def _refresh_toolbar_label(self):
        base_text = _('RSS Reader')
        try:
            suspended = bool(plugin_prefs.get('suspend_fetching', False))
        except Exception:
            suspended = False
        try:
            n = int(getattr(self, '_pending_new_count', 0) or 0)
        except Exception:
            n = 0

        suffix = ''
        if suspended:
            suffix = _(' (suspended)')
        if n > 0:
            if suspended:
                label = _('%s (%d new, suspended)') % (base_text, n)
            else:
                label = _('%s (%d new)') % (base_text, n)
        else:
            label = base_text + suffix

        try:
            self.qaction.setText(label)
            # Append key settings to tooltip (notifications and auto-update)
            try:
                notify_on = bool(plugin_prefs.get('notify_on_new', True))
            except Exception:
                notify_on = True
            try:
                max_n = int(plugin_prefs.get('max_notifications_per_update', 3) or 3)
            except Exception:
                max_n = 3
            try:
                minutes = int(plugin_prefs.get('auto_update_minutes', 0) or 0)
            except Exception:
                minutes = 0
            tooltip_lines = [label, _('Notify for new items: %s') % (_('Yes') if notify_on else _('No')),
                             _('Max notifications per update: %d') % max_n]
            if minutes:
                tooltip_lines.append(_('Auto-update every: %d min') % minutes)
            else:
                tooltip_lines.append(_('Auto-update: disabled'))

            # Include current DB path and profile emoji (if available)
            try:
                current_db = None
                if getattr(self, '_dialog', None) is not None:
                    dlg = self._dialog
                    if hasattr(dlg, '_db_current_path') and getattr(dlg, '_db_current_path', None):
                        current_db = getattr(dlg, '_db_current_path', None)
                if not current_db:
                    current_db = rss_db.db_path()
            except Exception:
                current_db = None
            try:
                emoji = ''
                active_id = str(plugin_prefs.get('db_profiles_active') or '')
                profiles = list(plugin_prefs.get('db_profiles') or [])
                found = None
                if active_id:
                    for p in profiles:
                        try:
                            if str(p.get('id') or '') == active_id:
                                found = p
                                break
                        except Exception:
                            continue
                if not found and current_db:
                    for p in profiles:
                        try:
                            if str(p.get('path') or '') == str(current_db):
                                found = p
                                break
                        except Exception:
                            continue
                if found:
                    emoji = str(found.get('emoji') or '').strip()
                if not emoji and current_db:
                    try:
                        default_path = str(plugin_prefs.get('db_default_path') or '').strip()
                    except Exception:
                        default_path = ''
                    if default_path and str(current_db) == default_path:
                        try:
                            emoji = str(plugin_prefs.get('db_default_emoji') or '').strip()
                        except Exception:
                            emoji = ''
            except Exception:
                emoji = ''

            tooltip_lines.append(_('Database: %s') % (str(current_db or _('Unknown'))))
            tooltip_lines.append(_('Profile badge: %s') % (emoji or _('None')))
            tooltip = '\n'.join(tooltip_lines)
            self.qaction.setToolTip(tooltip)
        except Exception:
            pass

    def set_fetching_suspended(self, suspended=True):
        try:
            plugin_prefs['suspend_fetching'] = bool(suspended)
        except Exception:
            pass
        try:
            if self._act_suspend_fetching is not None:
                self._act_suspend_fetching.setChecked(bool(suspended))
        except Exception:
            pass
        # Stop/restart periodic timer based on the new state
        try:
            self.apply_settings()
        except Exception:
            pass
        try:
            self._refresh_toolbar_label()
        except Exception:
            pass
        try:
            msg = _('RSS Reader: fetching suspended') if suspended else _('RSS Reader: fetching resumed')
            self.gui.status_bar.show_message(msg, 2500, show_notification=False)
        except Exception:
            pass

    def apply_settings(self):
        # Apply Serbian graphia changes (Cyrillic/Latin) without requiring calibre restart.
        # Detect any script preference change (Serbian, Russian, Japanese, Chinese)
        _graphia_keys = [
            ('serbian_graphia', 'latin'),
            ('russian_graphia', 'native'),
            ('japanese_graphia', 'native'),
            ('chinese_graphia', 'native'),
        ]
        _graphia_changed = False
        for _gk, _gdef in _graphia_keys:
            try:
                _cur = str(plugin_prefs.get(_gk, _gdef) or _gdef).strip().lower()
            except Exception:
                _cur = _gdef
            try:
                _prev = str(getattr(self, '_last_' + _gk, _gdef) or _gdef).strip().lower()
            except Exception:
                _prev = _gdef
            if _cur != _prev:
                _graphia_changed = True
                try:
                    setattr(self, '_last_' + _gk, _cur)
                except Exception:
                    pass

        if _graphia_changed:
            try:
                from calibre_plugins.rss_reader.i18n import refresh_all_plugin_modules

                refresh_all_plugin_modules()
            except Exception:
                pass
            # Re-open dialog to apply freshly installed _() to newly created widgets.
            try:
                if getattr(self, '_dialog', None) is not None:
                    try:
                        self._dialog.close()
                    except Exception:
                        pass
                    self._dialog = None
            except Exception:
                pass
            try:
                self.gui.status_bar.show_message(_('RSS Reader: script changed — re-open RSS Reader'), 4000, show_notification=False)
            except Exception:
                pass

        # If DB isn't configured yet, never start periodic updates.
        try:
            if not _db_is_configured():
                try:
                    self._timer.stop()
                except Exception:
                    pass
                return
        except Exception:
            pass
        try:
            minutes = int(plugin_prefs.get('auto_update_minutes', 0) or 0)
        except Exception:
            minutes = 0

        # If fetching is suspended, never run periodic updates.
        try:
            if bool(plugin_prefs.get('suspend_fetching', False)):
                self._timer.stop()
                return
        except Exception:
            pass

        if minutes > 0:
            self._timer.start(minutes * 60 * 1000)
        else:
            self._timer.stop()

        # If the dialog is open, it will read cache immediately and can optionally auto-update.

    def show_dialog(self):
        try:
            # Modeless dialog (doesn't block the main calibre UI)
            if self._dialog is not None:
                try:
                    # Verify the dialog is still alive before trying to show it
                    if hasattr(self._dialog, 'isVisible'):
                        self._dialog.showNormal()
                        # Windows: toggle StaysOnTop to force to front
                        try:
                            _cf = self._dialog.windowFlags()
                            if not (int(_cf) & int(Qt.WindowStaysOnTopHint)):
                                self._dialog.setWindowFlag(Qt.WindowStaysOnTopHint, True)
                                self._dialog.show()
                                self._dialog.setWindowFlag(Qt.WindowStaysOnTopHint, False)
                                self._dialog.show()
                        except Exception:
                            pass
                        self._dialog.raise_()
                        self._dialog.activateWindow()
                        return
                except Exception:
                    pass
                # If we get here, the dialog is dead; clear the reference and continue
                # to create a new one
                self._dialog = None

            # First-run onboarding: choose a durable DB location.
            try:
                ok = _run_db_onboarding_if_needed(parent=self.gui, action=self)
            except Exception:
                ok = True
            if not ok:
                return

            from calibre_plugins.rss_reader.ui import RSSReaderDialog
            d = RSSReaderDialog(self.gui, action=self)
            self._dialog = d

            # Attach DB shortcut actions to the RSS Reader window so they only
            # trigger while the window (or its children) has focus.
            try:
                for act in (getattr(self, '_db_menu_shortcut_actions', {}) or {}).values():
                    try:
                        if act is None:
                            continue
                        d.addAction(act)
                    except Exception:
                        pass
            except Exception:
                pass

            # Attach AI shortcut actions so assigned keys trigger only while the
            # RSS Reader window (or its children) has focus.
            try:
                for act in (getattr(self, '_ai_menu_shortcut_actions', {}) or {}).values():
                    try:
                        if act is None:
                            continue
                        d.addAction(act)
                    except Exception:
                        pass
            except Exception:
                pass

            # Defensive: ensure it is non-modal
            try:
                d.setModal(False)
            except Exception:
                pass
            try:
                d.setWindowModality(Qt.WindowModality.NonModal)
            except Exception:
                try:
                    d.setWindowModality(Qt.NonModal)
                except Exception:
                    pass
            try:
                d.setAttribute(Qt.WidgetAttribute.WA_DeleteOnClose, True)
            except Exception:
                try:
                    d.setAttribute(Qt.WA_DeleteOnClose, True)
                except Exception:
                    pass

            def _clear_dialog_ref(*args):
                # Keep QAction wrappers registered so shortcuts remain visible
                # in Preferences. Cleanup is done on plugin shutdown.
                self._dialog = None

            try:
                d.destroyed.connect(_clear_dialog_ref)
            except Exception:
                pass
            try:
                d.finished.connect(_clear_dialog_ref)
            except Exception:
                pass

            d.showNormal()
            try:
                d.raise_()
                d.activateWindow()
            except Exception:
                pass

            # Onboarding: if user opted to fetch after adding feeds, do it now.
            try:
                if bool(plugin_prefs.get('onboarding_fetch_after_add', False)):
                    try:
                        plugin_prefs['onboarding_fetch_after_add'] = False
                    except Exception:
                        pass
                    try:
                        self.update_all_feeds(silent=False)
                    except Exception:
                        pass
            except Exception:
                pass
            try:
                # Create QAction wrappers for dialog toolbar buttons so
                # they become canonical actions that Calibre can register
                # keyboard shortcuts for.
                try:
                    self._register_toolbar_shortcuts(d)
                except Exception:
                    pass
            except Exception:
                pass
        except Exception as e:
            error_dialog(self.gui, _('RSS Reader Error'), _('Failed to open RSS Reader: %s') % str(e),
                        show=True, det_msg=traceback.format_exc())

    def _set_notify_in_app_popup(self, val):
        try:
            plugin_prefs['notify_in_app_popup'] = bool(val)
        except Exception:
            pass

    def _rebuild_profiles_submenu(self):
        """Rebuild the Profiles submenu in the main button menu."""
        try:
            m = getattr(self, '_profiles_submenu', None)
            if m is None:
                return
            try:
                active_id = str(plugin_prefs.get('db_profiles_active') or '')
            except Exception:
                active_id = ''
            try:
                cur_path = str(rss_db.db_path() or '').strip()
            except Exception:
                cur_path = ''

            # Cheap signature so we can avoid rebuilding the menu repeatedly.
            try:
                default_path = str(plugin_prefs.get('db_default_path') or '').strip()
            except Exception:
                default_path = ''
            try:
                profiles = list(plugin_prefs.get('db_profiles') or [])
            except Exception:
                profiles = []
            try:
                prof_sig = tuple(
                    (
                        str(p.get('id') or ''),
                        str(p.get('name') or ''),
                        str(p.get('emoji') or ''),
                        str(p.get('path') or ''),
                        bool(p.get('readonly', False)),
                        bool(p.get('mirror', False)),
                    )
                    for p in (profiles or [])
                    if isinstance(p, dict)
                )
            except Exception:
                prof_sig = ()
            try:
                sig = (active_id, cur_path, default_path, prof_sig)
            except Exception:
                sig = None
            try:
                if sig is not None and getattr(self, '_profiles_submenu_sig', None) == sig and (m.actions() or []):
                    return
            except Exception:
                pass
            try:
                self._profiles_submenu_sig = sig
            except Exception:
                pass

            m.clear()

            # Helper to switch via the main dialog (ensures UI-level switching is used)
            def _do_switch(aid):
                try:
                    # Ensure dialog exists and perform the switch there so all UI updates occur
                    self.show_dialog()
                except Exception:
                    pass
                try:
                    dlg = getattr(self, '_dialog', None)
                except Exception:
                    dlg = None
                if dlg is not None and hasattr(dlg, '_set_active_profile'):
                    try:
                        dlg._set_active_profile(aid)
                    except Exception:
                        pass
                else:
                    try:
                        # Fallback: persist active id; UI will pick it up when opened
                        plugin_prefs['db_profiles_active'] = str(aid)
                    except Exception:
                        pass

            # Dedicated entry: configured Default DB (aka "Current DB")
            try:
                if default_path:
                    try:
                        d_name = str(plugin_prefs.get('db_default_name') or '').strip()
                    except Exception:
                        d_name = ''
                    try:
                        d_emoji = str(plugin_prefs.get('db_default_emoji') or '').strip()
                    except Exception:
                        d_emoji = ''
                    try:
                        d_ro = bool(plugin_prefs.get('db_default_readonly', False) or plugin_prefs.get('db_default_mirror', False))
                    except Exception:
                        d_ro = False
                    label = (d_emoji + ' ' if d_emoji else '') + (d_name or _('Current DB'))
                    a_def = QAction(label, m)
                    try:
                        tip = str(default_path)
                        try:
                            if d_ro:
                                tip = tip + '\n' + _('Mode: Read-only')
                            else:
                                tip = tip + '\n' + _('Mode: Default')
                        except Exception:
                            pass
                        a_def.setToolTip(tip)
                    except Exception:
                        pass
                    try:
                        a_def.setCheckable(True)
                        a_def.setChecked((not bool(active_id)) and (cur_path and cur_path == default_path))
                    except Exception:
                        pass
                    # Selecting default will restore default DB via dialog
                    a_def.triggered.connect(lambda _checked=False: (self.show_dialog(), getattr(self, '_dialog', None) and getattr(self, '_dialog', None)._restore_default_database()))
                    m.addAction(a_def)
            except Exception:
                pass

            # Built-in/plugin DB entry (skip unless differs)
            try:
                cfg_path = str(getattr(self, '_db_config_path', '') or '').strip()
            except Exception:
                cfg_path = ''
            try:
                if cfg_path:
                    try:
                        default_path = str(getattr(self, '_db_orig_path', '') or '').strip()
                    except Exception:
                        default_path = ''
                    if default_path and str(cfg_path).strip() == default_path:
                        raise Exception('redundant')
            except Exception:
                pass

            try:
                default_emoji = str(plugin_prefs.get('db_profile_default_emoji') or '📰').strip()
            except Exception:
                default_emoji = '📰'

            if not profiles:
                a = QAction(_('No profiles'), m)
                a.setEnabled(False)
                m.addAction(a)
            else:
                for p in profiles:
                    try:
                        emoji = str(p.get('emoji') or '').strip()
                    except Exception:
                        emoji = ''
                    if not emoji:
                        emoji = default_emoji
                    try:
                        text = (emoji + ' ' if emoji else '') + (p.get('name') or p.get('path') or '')
                    except Exception:
                        text = str(p.get('name') or p.get('path') or '')
                    act = QAction(text, m)
                    aid = str(p.get('id') or '')
                    try:
                        tip = str(p.get('path') or '')
                        if bool(p.get('mirror', False)):
                            tip = tip + '\n' + _('Mode: Mirror (read-only)')
                        elif bool(p.get('readonly', False)):
                            tip = tip + '\n' + _('Mode: Read-only')
                        else:
                            tip = tip + '\n' + _('Mode: Writable')
                        act.setToolTip(tip)
                    except Exception:
                        pass
                    try:
                        act.setCheckable(True)
                        act.setChecked(aid == str(active_id or ''))
                    except Exception:
                        pass
                    # Connect to helper that ensures dialog exists then calls UI switch
                    act.triggered.connect(lambda _checked=False, _aid=aid: _do_switch(_aid))
                    m.addAction(act)
        except Exception:
            pass

    def show_profiles_manager(self):
        """Open the RSS Reader dialog and show the Profiles manager."""
        try:
            self.show_dialog()
        except Exception:
            pass
        try:
            dlg = getattr(self, '_dialog', None)
        except Exception:
            dlg = None
        if dlg is None:
            return
        try:
            if hasattr(dlg, '_show_profiles_manager'):
                dlg._show_profiles_manager()
        except Exception:
            pass

    def _open_url_external(self, url):
        try:
            if DEBUG_RSS_READER:
                _debug('Action: _open_url_external called with', url)
        except Exception:
            pass
        try:
            u = str(url or '').strip()
        except Exception:
            u = ''
        if not u:
            return
        try:
            __import__('calibre').gui2.open_url(u)
        except Exception:
            pass

    def open_item_in_app(self, feed_id, item_id):
        try:
            if DEBUG_RSS_READER:
                _debug('Action: open_item_in_app', feed_id, item_id)
        except Exception:
            pass
        try:
            self.show_dialog()
        except Exception:
            return
        try:
            if self._dialog is not None and hasattr(self._dialog, 'focus_item'):
                self._dialog.focus_item(feed_id, item_id)
        except Exception:
            pass

    def _mark_items_read(self, entries):
        try:
            if DEBUG_RSS_READER:
                _debug('Action: _mark_items_read', len(entries or []), 'entries')
        except Exception:
            pass
        # entries: iterable of dicts with feed_id/item_id
        by_feed = {}
        for e in (entries or []):
            try:
                if isinstance(e, dict) and e.get('_is_feed_header'):
                    continue
                fid = str((e or {}).get('feed_id') or '').strip()
                iid = str((e or {}).get('item_id') or '').strip()
                if not fid or not iid:
                    continue
                by_feed.setdefault(fid, set()).add(iid)
            except Exception:
                continue
        if not by_feed:
            return

        try:
            seen_map = dict(rss_db.get_seen_item_ids_map() or {})
        except Exception:
            seen_map = {}

        for fid, iids in by_feed.items():
            try:
                cur = set(seen_map.get(fid, []) or [])
                cur.update(set(iids or []))
                rss_db.set_seen_item_ids(fid, list(cur))
            except Exception:
                continue

        # Best-effort UI refresh if the dialog is open
        try:
            if self._dialog is not None and hasattr(self._dialog, 'load_items_for_selected_feed'):
                self._dialog.load_items_for_selected_feed()
        except Exception:
            pass

    def _close_in_app_popup(self):
        try:
            if self._popup_notification is not None:
                self._popup_notification.close()
        except Exception:
            pass
        self._popup_notification = None

    def _qicon_from_icon_bytes(self, icon_bytes):
        try:
            if not icon_bytes:
                return None
            pm = QPixmap()
            try:
                pm.loadFromData(icon_bytes)
            except Exception:
                pm.loadFromData(bytes(icon_bytes))
            if pm.isNull():
                return None
            return QIcon(pm)
        except Exception:
            return None

    def _show_in_app_popup(self, entries, total_new=0, header_icon=None, timeout_ms=None):
        try:
            from calibre_plugins.rss_reader.popup_notifications import PopupNotification
        except Exception:
            return

        # Store for preview/replay
        try:
            self._last_popup_payload = {
                'entries': list(entries or []),
                'total_new': int(total_new or 0),
                'header_icon': header_icon,
            }
        except Exception:
            self._last_popup_payload = None

        self._close_in_app_popup()
        try:
            try:
                per_page = plugin_prefs.get('popup_max_items_per_page', 10)
            except Exception:
                per_page = 10
            if timeout_ms is None:
                try:
                    timeout_ms = plugin_prefs.get('popup_timeout_ms', 15000)
                except Exception:
                    timeout_ms = 15000
            # Parentless popup avoids Windows focus/minimize quirks with other dialogs.
            popup = PopupNotification(entries, total_new=total_new, max_items_per_page=per_page, timeout_ms=timeout_ms, parent=None)
        except Exception:
            return

        try:
            if DEBUG_RSS_READER:
                _debug('PopupNotification created with', len(entries or []), 'entries; total_new=', total_new)
        except Exception:
            pass

        try:
            if header_icon is not None:
                popup.set_header_icon(header_icon)
        except Exception:
            pass

        # Wire popup-level actions once. Popup relays item signals up to these
        # signals every time pages are (re)built.
        try:
            popup.open_in_app.connect(self.open_item_in_app)
            popup.open_external.connect(self._open_url_external)
        except Exception:
            pass

        try:
            if DEBUG_RSS_READER:
                _debug('PopupNotification signals wired in action')
        except Exception:
            pass

        try:
            popup.closed.connect(lambda: setattr(self, '_popup_notification', None))
        except Exception:
            pass

        self._popup_notification = popup
        try:
            popup.show()
        except Exception:
            pass

    def preview_in_app_popup(self):
        """Show a test popup notification so theme colors can be verified."""
        payload = None
        try:
            payload = getattr(self, '_last_popup_payload', None)
        except Exception:
            payload = None

        try:
            if isinstance(payload, dict) and payload.get('entries'):
                entries = list(payload.get('entries') or [])
                total_new = int(payload.get('total_new') or 0)
                header_icon = payload.get('header_icon')
                self._show_in_app_popup(entries, total_new=total_new, header_icon=header_icon)
                return
        except Exception:
            pass

        # No previous payload; generate a deterministic synthetic sample.
        try:
            entries = [
                {'_is_feed_header': True, 'feed_title': _('[TEST] Sample Feed A')},
                {
                    'feed_id': 'preview_feed_a',
                    'feed_title': _('[TEST] Sample Feed A'),
                    'item_id': 'preview_item_a1',
                    'item_title': _('[Sample] Themed popup notifications'),
                    'link': 'https://example.com/preview/a1',
                },
                {
                    'feed_id': 'preview_feed_a',
                    'feed_title': _('[TEST] Sample Feed A'),
                    'item_id': 'preview_item_a2',
                    'item_title': _('[Sample] Mark as read / open in-app / browser'),
                    'link': 'https://example.com/preview/a2',
                },
                {'_is_feed_header': True, 'feed_title': _('[TEST] Sample Feed B')},
                {
                    'feed_id': 'preview_feed_b',
                    'feed_title': _('[TEST] Sample Feed B'),
                    'item_id': 'preview_item_b1',
                    'item_title': _('[Sample] Longer title text to test overflow handling'),
                    'link': 'https://example.com/preview/b1',
                },
            ]
        except Exception:
            entries = []
        self._show_in_app_popup(entries, total_new=len([e for e in entries if not (e or {}).get('_is_feed_header')]))

    def _encode_icon_for_prefs(self, icon_bytes):
        if not icon_bytes:
            return None
        # JSONConfig cannot reliably persist raw bytes; store base64.
        if isinstance(icon_bytes, (bytes, bytearray)):
            try:
                return 'b64:' + base64.b64encode(bytes(icon_bytes)).decode('ascii')
            except Exception:
                return None
        if isinstance(icon_bytes, str):
            # Already stored
            return icon_bytes
        return None

    def _register_toolbar_shortcuts(self, dlg):
        if dlg is None:
            return
        items = getattr(dlg, '_toolbar_items', None)
        if not items:
            return
        try:
            wrappers = getattr(self, '_toolbar_action_wrappers', None)
            if wrappers is None:
                self._toolbar_action_wrappers = {}
        except Exception:
            self._toolbar_action_wrappers = {}

        for tid, btn in (items or {}).items():
            try:
                if not btn or not hasattr(btn, 'text'):
                    continue
                label = str(btn.text() or tid).strip()

                # Build unique id/name used by calibre keyboard manager
                unique_name = f'plugin:rss_reader:toolbar:{tid}'
                display_name = f'RSS Reader toolbar: {label}'

                # Remove any stale wrapper for this tid so we don't leave dead connections
                try:
                    wrappers = getattr(self, '_toolbar_action_wrappers', {}) or {}
                    old = wrappers.get(tid)
                    if old is not None:
                        try:
                            self._safe_remove_action(old)
                        except Exception:
                            pass
                        try:
                            del self._toolbar_action_wrappers[tid]
                        except Exception:
                            pass
                except Exception:
                    pass

                # Create the wrapper QAction that invokes the toolbar action helper
                act = QAction(label, self.gui)
                try:
                    tip = str(btn.toolTip() or '').strip()
                    if tip:
                        act.setToolTip(tip)
                except Exception:
                    pass
                try:
                    act.triggered.connect(partial(self._invoke_toolbar_action, tid))
                except Exception:
                    try:
                        act.triggered.connect(lambda _checked=False, _tid=tid: self._invoke_toolbar_action(_tid))
                    except Exception:
                        pass

                # Ensure there is a registered shortcut entry. If none exists,
                # register a harmless placeholder action first so the keyboard
                # manager owns the unique name and user assignments persist.
                try:
                    kb = self.gui.keyboard
                except Exception:
                    kb = None

                try:
                    exists = kb and (unique_name in getattr(kb, 'shortcuts', {}))
                except Exception:
                    exists = False

                if not exists and kb is not None:
                    try:
                        placeholder = QAction(label, self.gui)
                        try:
                            self._safe_add_action(placeholder)
                        except Exception:
                            pass
                        kb.register_shortcut(
                            unique_name,
                            display_name,
                            description=f'Toolbar button: {label}',
                            action=placeholder,
                            group='RSS Reader',
                            persist_shortcut=True,
                        )
                        try:
                            kb.finalize()
                        except Exception:
                            pass
                    except Exception:
                        pass

                # Replace the action object for this unique_name with our wrapper
                try:
                    # Ensure action is added to the top-level GUI so shortcuts will trigger
                    try:
                        self._safe_add_action(act)
                    except Exception:
                        pass
                    try:
                        # Set application-wide shortcut context so it works regardless of focus
                        try:
                            self._set_application_shortcut_context(act)
                        except Exception:
                            # Best-effort fallback if helper fails
                            try:
                                act.setShortcutContext(Qt.ShortcutContext.ApplicationShortcut)
                            except Exception:
                                try:
                                    act.setShortcutContext(Qt.ApplicationShortcut)
                                except Exception:
                                    pass
                    except Exception:
                        pass

                    if kb is not None and unique_name in getattr(kb, 'shortcuts', {}):
                        try:
                            kb.replace_action(unique_name, act)
                        except Exception:
                            try:
                                kb.register_shortcut(
                                    unique_name,
                                    display_name,
                                    description=f'Toolbar button: {label}',
                                    action=act,
                                    group='RSS Reader',
                                    persist_shortcut=True,
                                )
                            except Exception:
                                pass
                        try:
                            kb.finalize()
                        except Exception:
                            pass
                    else:
                        # No keyboard manager available; nothing more to do
                        pass
                except Exception:
                    pass

                # Keep wrapper reference so it isn't GC'd
                try:
                    self._toolbar_action_wrappers[tid] = act
                except Exception:
                    pass
            except Exception:
                continue

    def _invoke_toolbar_action(self, tid):
        try:
            dlg = getattr(self, '_dialog', None)
            if dlg is None:
                return
            items = getattr(dlg, '_toolbar_items', {}) or {}
            btn = items.get(tid)
            if not btn:
                return
            try:
                btn.click()
            except Exception:
                try:
                    # Fallback: call activated/trigger if available
                    if hasattr(btn, 'trigger'):
                        btn.trigger()
                    elif hasattr(btn, 'activated'):
                        btn.activated()
                except Exception:
                    pass
        except Exception:
            pass

    def _unregister_toolbar_shortcuts(self):
        try:
            wrappers = getattr(self, '_toolbar_action_wrappers', None)
            if not wrappers:
                return
            for tid, act in list(wrappers.items()):
                try:
                    self._safe_remove_action(act)
                except Exception:
                    pass
                try:
                    del self._toolbar_action_wrappers[tid]
                except Exception:
                    pass
            try:
                self._toolbar_action_wrappers = {}
            except Exception:
                pass
        except Exception:
            pass

    def _set_application_shortcut_context(self, act):
        try:
            act.setShortcutContext(Qt.ShortcutContext.ApplicationShortcut)
        except Exception:
            try:
                act.setShortcutContext(Qt.ApplicationShortcut)
            except Exception:
                pass

    def _safe_add_action(self, act):
        try:
            if act is None:
                return
            if hasattr(self, 'gui') and getattr(self, 'gui', None) is not None and hasattr(self.gui, 'addAction'):
                try:
                    self.gui.addAction(act)
                except Exception:
                    pass
        except Exception:
            pass

    def _safe_remove_action(self, act):
        try:
            if act is None:
                return
            if hasattr(self, 'gui') and getattr(self, 'gui', None) is not None and hasattr(self.gui, 'removeAction'):
                try:
                    self.gui.removeAction(act)
                except Exception:
                    pass
        except Exception:
            pass

    def cancel_update(self):
        try:
            if self._update_abort is not None:
                self._update_abort.set()
        except Exception:
            pass

    def update_all_feeds(self, silent=False, feed_ids=None):
        # Never pop onboarding during feed updates. If DB isn't configured,
        # abort with a status message; user can configure via opening the UI.
        try:
            if not _db_is_configured():
                try:
                    self.gui.status_bar.show_message(
                        _('RSS Reader: database not configured (open RSS Reader to set it up).'),
                        5000,
                        show_notification=False,
                    )
                except Exception:
                    pass
                return
        except Exception:
            pass

        # During calibre shutdown, avoid starting new work or touching UI timers.
        try:
            if bool(getattr(self, '_shutting_down', False)):
                return
        except Exception:
            pass
        # Do not use calibre's JobManager for this operation.
        # In calibre core, the JobsButton calls QCoreApplication.instance().alert(...)
        # when all jobs finish (see CALIBRE_SOURCE_CURRENT/gui2/jobs.py), which flashes
        # the Windows taskbar even if tray notifications are disabled.
        if self._update_thread is not None and self._update_thread.is_alive():
            return

        # Global pause switch
        try:
            if bool(plugin_prefs.get('suspend_fetching', False)):
                try:
                    if self._dialog is not None:
                        self._dialog.on_update_progress(_('Fetching is suspended.'), None)
                except Exception:
                    pass
                try:
                    self.gui.status_bar.show_message(_('RSS Reader: fetching is suspended'), 2500, show_notification=False)
                except Exception:
                    pass
                return
        except Exception:
            pass

        feeds_all = list(rss_db.get_feeds() or [])
        if feed_ids:
            wanted = {str(x) for x in feed_ids}
            feeds = [f for f in feeds_all if str(f.get('id') or '') in wanted]
        else:
            feeds = feeds_all
        seen_item_ids = dict(rss_db.get_seen_item_ids_map() or {})
        known_item_ids = dict(rss_db.get_known_item_ids_map() or {})
        timeout_seconds = int(plugin_prefs.get('timeout_seconds', 25) or 25)
        max_seen_per_feed = int(plugin_prefs.get('max_seen_per_feed', 200) or 200)

        if not feeds:
            if self._dialog is not None:
                self._dialog.refresh()
            return

        # Progress updates from the worker
        self._progress_queue = queue.Queue()
        desc = _('Updating RSS feeds') if not feed_ids else _('Updating selected RSS feeds')
        self._update_silent = bool(silent)
        self._update_abort = threading.Event()
        self._update_started_at = time.time()
        try:
            self._update_total = int(len(feeds) or 0)
        except Exception:
            self._update_total = 0

        # Poll worker progress and update UI status text
        try:
            if self._progress_timer is None:
                self._progress_timer = QTimer(self.gui)
                self._progress_timer.setInterval(150)
                self._progress_timer.timeout.connect(self._poll_progress)
            self._progress_timer.start()
        except Exception:
            pass
        # Avoid calibre system-tray notifications for routine updates.
        try:
            self.gui.status_bar.show_message(desc, 2000, show_notification=False)
        except Exception:
            pass

        # Capture the DB path in use at launch so we can recognize results
        launch_db_path = None
        try:
            launch_db_path = rss_db.db_path()
        except Exception:
            launch_db_path = None

        try:
            self._update_source_db = launch_db_path
        except Exception:
            pass

        def _runner():
            try:
                res = _update_worker(
                    feeds,
                    seen_item_ids,
                    known_item_ids,
                    timeout_seconds=timeout_seconds,
                    max_known_per_feed=max_seen_per_feed,
                    abort=self._update_abort,
                    log=None,
                    notifications=self._progress_queue,
                )
                # Wrap results with the DB path used to run the worker so apply step can ignore
                wrapped = {'db_path': launch_db_path, 'feeds': (res or {}).get('feeds') if isinstance(res, dict) else {}}

                # If calibre is shutting down, skip any UI dispatch entirely.
                try:
                    if bool(getattr(self, '_shutting_down', False)):
                        return
                except Exception:
                    pass
                try:
                    self._dispatch_update_done(wrapped, None)
                except Exception:
                    # Best-effort fallback
                    self._update_done_from_thread(wrapped, None)
            except Exception:
                tb = traceback.format_exc()

                # If calibre is shutting down, skip any UI dispatch entirely.
                try:
                    if bool(getattr(self, '_shutting_down', False)):
                        return
                except Exception:
                    pass
                try:
                    self._dispatch_update_done(None, tb)
                except Exception:
                    self._update_done_from_thread(None, tb)

        self._update_thread = threading.Thread(target=_runner, name='rss_reader_update', daemon=True)
        self._update_thread.start()

    def _update_done_from_thread(self, results, tb):
        silent = bool(getattr(self, '_update_silent', False))
        self._update_silent = False

        # If calibre is shutting down, avoid touching any Qt objects that may
        # already be deleted. Only clear Python-side state.
        try:
            if bool(getattr(self, '_shutting_down', False)):
                self._progress_queue = None
                self._update_abort = None
                self._update_thread = None
                try:
                    self._update_source_db = None
                except Exception:
                    pass
                return
        except Exception:
            pass

        # Stop progress polling and clean up thread/process state
        try:
            if self._progress_timer is not None:
                try:
                    import sip
                except Exception:
                    try:
                        from PyQt5 import sip  # type: ignore
                    except Exception:
                        sip = None
                try:
                    if sip is None or not (hasattr(sip, 'isdeleted') and sip.isdeleted(self._progress_timer)):
                        self._progress_timer.stop()
                except Exception:
                    # If the wrapped object is already deleted, ignore.
                    pass
        except Exception as e:
            import traceback
            try:
                from calibre_plugins.rss_reader.debug import _debug
                _debug('RSSReaderAction: Failed to stop progress timer: %r %s' % (e, traceback.format_exc()[:800]))
            except Exception:
                pass
        self._progress_queue = None
        self._update_abort = None
        self._update_thread = None
        try:
            self._update_source_db = None
        except Exception:
            pass
        try:
            self._update_started_at = None
            self._update_total = 0
        except Exception as e:
            try:
                from calibre_plugins.rss_reader.debug import _debug
                _debug('RSSReaderAction: Failed to clear update state: %r' % (e,))
            except Exception:
                pass

        # Defensive error handling: show error and log traceback
        if tb:
            try:
                if self._dialog is not None:
                    self._dialog.set_busy(False)
            except Exception as e:
                try:
                    from calibre_plugins.rss_reader.debug import _debug
                    _debug('RSSReaderAction: Failed to set_busy(False) after error: %r' % (e,))
                except Exception:
                    pass
            if not silent:
                error_dialog(self.gui, _('RSS Reader Error'), _('Feed update failed.'), show=True, det_msg=tb)
            else:
                try:
                    self.gui.status_bar.show_message(_('RSS Reader: update failed'), 4000, show_notification=False)
                except Exception as e:
                    try:
                        from calibre_plugins.rss_reader.debug import _debug
                        _debug('RSSReaderAction: Failed to show status bar error: %r' % (e,))
                    except Exception:
                        pass
            # Extra logging for diagnostics
            try:
                from calibre_plugins.rss_reader.debug import _debug
                _debug('RSSReaderAction: Feed update failed. Traceback: %s' % (str(tb or '')[:4000],))
            except Exception:
                pass
            return

        # If results were wrapped with a source DB path, verify it matches current DB
        source_db = None
        feeds_results = {}
        try:
            if isinstance(results, dict) and 'db_path' in results:
                source_db = results.get('db_path')
                feeds_results = (results or {}).get('feeds') or {}
            else:
                feeds_results = (results or {}).get('feeds') or {}
        except Exception as e:
            try:
                from calibre_plugins.rss_reader.debug import _debug
                _debug('RSSReaderAction: Failed to extract feeds_results: %r' % (e,))
            except Exception:
                pass
            feeds_results = (results or {}).get('feeds') or {}

        # If the results originated from a different DB than the current one, ignore them
        try:
            current_db = rss_db.db_path()
        except Exception as e:
            try:
                from calibre_plugins.rss_reader.debug import _debug
                _debug('RSSReaderAction: Failed to get current DB path: %r' % (e,))
            except Exception:
                pass
            current_db = None

        if source_db is not None and source_db != current_db:
            # Skip applying results from another profile's DB to avoid cross-profile notifications
            try:
                if not silent:
                    self.gui.status_bar.show_message(_('RSS Reader: update completed for inactive profile (ignored)'), 3000, show_notification=False)
            except Exception as e:
                try:
                    from calibre_plugins.rss_reader.debug import _debug
                    _debug('RSSReaderAction: Failed to show inactive profile message: %r' % (e,))
                except Exception:
                    pass
            try:
                from calibre_plugins.rss_reader.debug import _debug
                _debug('RSSReaderAction: Ignored update results from inactive profile DB.')
            except Exception:
                pass
            return

        # Defensive: wrap finalization in try/except and log any errors
        try:
            self._apply_update_results(feeds_results, silent=silent)
        except Exception as e:
            import traceback
            tb2 = traceback.format_exc()
            try:
                from calibre_plugins.rss_reader.debug import _debug
                _debug('RSSReaderAction: Exception during _apply_update_results: %r %s' % (e, str(tb2 or '')[:2000]))
            except Exception:
                pass
            if not silent:
                error_dialog(self.gui, _('RSS Reader Error'), _('Failed to finalize feed update.'), show=True, det_msg=tb2)
            else:
                try:
                    self.gui.status_bar.show_message(_('RSS Reader: finalize failed'), 4000, show_notification=False)
                except Exception:
                    pass

    def _poll_progress(self):
        q = self._progress_queue
        if q is None:
            return

        # If the DB changed since this update started, suppress stale progress
        # messages so switching profiles doesn't show fetching from the old DB.
        try:
            source_db = getattr(self, '_update_source_db', None)
            if source_db is not None:
                try:
                    current_db = rss_db.db_path()
                except Exception:
                    current_db = None
                if current_db is not None and source_db != current_db:
                    return
        except Exception:
            pass
        # Drain the queue quickly
        last = None
        try:
            while True:
                last = q.get_nowait()
        except Exception:
            pass

        if not last:
            return

        try:
            frac, msg = last
        except Exception:
            frac, msg = None, None

        if not msg:
            return

        # Add a light-weight ETA based on observed progress.
        try:
            f = None
            try:
                f = float(frac)
            except Exception:
                f = None

            if f is not None and f > 0:
                started = getattr(self, '_update_started_at', None)
                if started:
                    elapsed = max(0.0, time.time() - float(started))
                    # Avoid noisy ETA when starting.
                    if elapsed >= 2.0 and f >= 0.03:
                        remaining = max(0.0, (elapsed * (1.0 - f) / f))
                        # Cap absurd ETAs from very slow first feeds.
                        remaining = min(remaining, 60.0 * 60.0 * 6.0)

                        def _fmt(sec):
                            try:
                                sec = int(round(float(sec)))
                            except Exception:
                                sec = 0
                            m, s = divmod(max(0, sec), 60)
                            h, m = divmod(m, 60)
                            if h:
                                return '%d:%02d:%02d' % (h, m, s)
                            return '%d:%02d' % (m, s)

                        pct = int(max(0, min(100, round(f * 100.0))))
                        msg = '%s   (%d%%, ETA %s)' % (str(msg), pct, _fmt(remaining))
        except Exception:
            pass

        try:
            if self._dialog is not None:
                self._dialog.on_update_progress(msg, frac)
            else:
                # Background updates: just update calibre's status bar
                try:
                    self.gui.status_bar.show_message(msg, 1200, show_notification=False)
                except Exception:
                    pass
        except Exception:
            pass

    def _apply_update_results(self, feeds_results, silent=False):
        # Persist seen ids + cached items + update titles if we detected better ones
        feeds = list(rss_db.get_feeds() or [])
        feeds_by_id = {str(f.get('id')): f for f in feeds}
        titles_to_update = {}
        max_cached = int(plugin_prefs.get('max_cached_items_per_feed', 200) or 200)

        # Auto-purge failed-feed history (default retention: 60 days)
        try:
            keep_days = int(plugin_prefs.get('failed_feeds_history_retention_days', 60) or 60)
        except Exception:
            keep_days = 60
        try:
            rss_db.purge_feed_fail_history(retention_days=keep_days)
        except Exception:
            pass

        total_new = 0
        total_failed = 0
        notify_first = bool(plugin_prefs.get('notify_on_first_fetch', False))

        # Detect "offline" runs: everything failed due to DNS/timeout and nothing succeeded.
        try:
            ok_count = sum(1 for r in (feeds_results or {}).values() if isinstance(r, dict) and r.get('ok'))
        except Exception:
            ok_count = 0
        try:
            err_kinds = [str((r or {}).get('error_kind') or '').strip() for r in (feeds_results or {}).values() if isinstance(r, dict) and not r.get('ok')]
        except Exception:
            err_kinds = []
        offline_batch = bool(err_kinds) and ok_count == 0 and all(k in ('dns-fail', 'timeout') for k in err_kinds)

        for feed_id, r in feeds_results.items():
            # Only process feeds that exist in the current DB (profile), but still
            # record failures in history for diagnostics.
            exists_in_db = feed_id in feeds_by_id
            if not isinstance(r, dict):
                continue
            if r.get('ok'):
                if not exists_in_db:
                    continue
                # Clear prior failure status so failure auto-tags drop off when the feed recovers.
                try:
                    rss_db.delete_feed_status(feed_id)
                except Exception:
                    pass
                try:
                    rss_db.set_known_item_ids(feed_id, list(r.get('known_item_ids') or []))
                except Exception:
                    pass

                is_first = bool(r.get('first_fetch'))
                new_count = int(r.get('new_count') or 0)
                if (not is_first) or notify_first:
                    total_new += new_count

                # Update feed title only if the user has not chosen
                # a custom name. We treat titles that are empty or that
                # still equal the URL as "auto" titles; anything else
                # is considered user-edited and should be preserved.
                f = feeds_by_id.get(feed_id)
                if f is not None and r.get('title'):
                    try:
                        cur_title = (f.get('title') or '').strip()
                        url_title = (f.get('url') or '').strip()
                        new_title = str(r.get('title') or '').strip()
                    except Exception:
                        cur_title = f.get('title') or ''
                        url_title = f.get('url') or ''
                        new_title = r.get('title') or ''

                    # Auto-update only when there is no meaningful
                    # existing title, or when it already matches the
                    # fetched title (so we are not overwriting a rename).
                    if (not cur_title) or (cur_title == url_title) or (cur_title == new_title):
                        f['title'] = new_title
                        try:
                            titles_to_update[str(feed_id)] = new_title
                        except Exception:
                            pass

                # Cache items so the dialog has content even before the next refresh
                try:
                    all_items = list(r.get('items') or [])
                    new_items = list(r.get('new_items') or [])
                    # Ensure new items appear first in cache so notifications are findable
                    new_ids = {it.get('id') for it in new_items}
                    other_items = [it for it in all_items if it.get('id') not in new_ids]
                    cached_items = new_items + other_items
                    if len(cached_items) > max_cached:
                        cached_items = cached_items[:max_cached]
                    entry = {}
                    try:
                        existing = rss_db.get_feed_cache(feed_id)
                        if isinstance(existing, dict):
                            entry = dict(existing)
                    except Exception:
                        entry = {}

                    # Preserve per-item received timestamps across updates.
                    try:
                        now_ts = int(time.time())
                    except Exception:
                        now_ts = 0
                    try:
                        prev_items = list((entry.get('items') or []) if isinstance(entry, dict) else [])
                    except Exception:
                        prev_items = []
                    prev_received_by_id = {}
                    try:
                        for pit in prev_items:
                            if not isinstance(pit, dict):
                                continue
                            pid = pit.get('id')
                            if not pid:
                                continue
                            rt = pit.get('received_ts') or pit.get('fetched_ts') or pit.get('downloaded_ts')
                            if rt:
                                prev_received_by_id[str(pid)] = int(rt)
                    except Exception:
                        prev_received_by_id = {}

                    try:
                        for it0 in cached_items:
                            if not isinstance(it0, dict):
                                continue
                            iid = str(it0.get('id') or '')
                            if not iid:
                                continue
                            if not it0.get('received_ts'):
                                if iid in prev_received_by_id:
                                    it0['received_ts'] = prev_received_by_id.get(iid)
                                elif iid in new_ids and now_ts:
                                    it0['received_ts'] = now_ts
                    except Exception:
                        pass

                    # Preserve existing icon unless we have a new one
                    icon_bytes = r.get('icon') or entry.get('icon')
                    icon_store = self._encode_icon_for_prefs(icon_bytes)
                    # Prefer the (possibly user-edited) feed title in
                    # prefs over the parsed one when they differ.
                    cache_title = ''
                    try:
                        if f is not None and (f.get('title') or '').strip():
                            cache_title = f.get('title')
                        else:
                            cache_title = r.get('title') or ''
                    except Exception:
                        cache_title = r.get('title') or (f.get('title') if f else '')
                    entry.update({
                        'title': cache_title,
                        'home_link': r.get('home_link') or '',
                        'feed_type': r.get('feed_type') or entry.get('feed_type') or '',
                        'feed_encoding': r.get('feed_encoding') or entry.get('feed_encoding') or '',
                        'updated': datetime.datetime.now(datetime.timezone.utc).isoformat(),
                        'items': cached_items,
                    })
                    if icon_store:
                        entry['icon'] = icon_store
                    try:
                        rss_db.set_feed_cache(feed_id, entry)
                    except Exception:
                        pass
                except Exception:
                    pass
            else:
                total_failed += 1
                if not exists_in_db:
                    # Feed isn't in current list (race / stale result), but still
                    # log it so the user can see what failed.
                    try:
                        kind0 = str(r.get('error_kind') or '').strip()
                    except Exception:
                        kind0 = ''
                    try:
                        if offline_batch and kind0 in ('dns-fail', 'timeout'):
                            raise Exception('offline batch')
                        rss_db.add_feed_fail_history(
                            str(feed_id),
                            title=str(r.get('title') or ''),
                            url=str(r.get('url') or ''),
                            error=str(r.get('error') or ''),
                            http_status=r.get('http_status'),
                            kind=kind0,
                            tags=list(r.get('auto_tags') or ['failed']),
                            ts=int(time.time()),
                        )
                    except Exception:
                        pass
                    continue
                # Persist last failure classification for UI auto-tags and DB sanitization.
                try:
                    prev = rss_db.get_feed_status(feed_id) or {}
                except Exception:
                    prev = {}
                # For transient offline failures, do not bump fail_count.
                try:
                    prev_fc = int(prev.get('fail_count') or 0)
                except Exception:
                    prev_fc = 0
                try:
                    kind0 = str(r.get('error_kind') or '').strip()
                except Exception:
                    kind0 = ''
                if kind0 in ('dns-fail', 'timeout') and offline_batch:
                    fail_count = prev_fc
                else:
                    fail_count = prev_fc + 1
                try:
                    status = {
                        'ok': False,
                        'ts': int(time.time()),
                        'url': r.get('url') or (feeds_by_id.get(feed_id) or {}).get('url') or '',
                        'error': r.get('error') or '',
                        'http_status': r.get('http_status'),
                        'kind': r.get('error_kind') or '',
                        'tags': list(r.get('auto_tags') or ['failed']),
                        'fail_count': fail_count,
                    }
                    if offline_batch and kind0 in ('dns-fail', 'timeout'):
                        try:
                            status['tags'] = sorted(set(list(status.get('tags') or []) + ['offline']))
                        except Exception:
                            pass
                    rss_db.set_feed_status(feed_id, status)
                except Exception:
                    pass

                # Append to failed-feed history (for diagnostics)
                try:
                    # When the machine is offline/DNS is down, don't spam the history log.
                    if offline_batch and kind0 in ('dns-fail', 'timeout'):
                        raise Exception('offline batch')
                    try:
                        title = (feeds_by_id.get(feed_id) or {}).get('title') or r.get('title') or ''
                    except Exception:
                        title = r.get('title') or ''
                    try:
                        rss_db.add_feed_fail_history(
                            feed_id,
                            title=title,
                            url=status.get('url') or '',
                            error=status.get('error') or '',
                            http_status=status.get('http_status'),
                            kind=status.get('kind') or '',
                            tags=status.get('tags') or [],
                            ts=status.get('ts'),
                        )
                    except Exception:
                        pass
                except Exception:
                    pass

        # IMPORTANT: do not rewrite the entire feeds table here.
        # Background updates can race with user drag/drop folder moves.
        try:
            if titles_to_update:
                rss_db.update_feed_titles(titles_to_update)
        except Exception:
            pass
        try:
            rss_db.purge_orphans(clear_folders_if_no_feeds=True)
        except Exception:
            pass

        # Update toolbar badge / tooltip
        try:
            self._pending_new_count = total_new
            self._refresh_toolbar_label()
        except Exception:
            pass

        use_popup = False
        try:
            use_popup = (not silent) and bool(plugin_prefs.get('notify_on_new', True)) and bool(plugin_prefs.get('notify_in_app_popup', False))
        except Exception:
            use_popup = False

        # Suppress all notifications while a fullscreen window is active (QuiteRSS-style)
        _suppress_fs = False
        try:
            if bool(plugin_prefs.get('suppress_notifications_fullscreen', False)):
                # First try Win32 API to detect ANY foreground fullscreen app (VLC, browser, etc.)
                import platform
                if platform.system().lower().startswith('win'):
                    try:
                        import ctypes
                        import ctypes.wintypes
                        user32 = ctypes.windll.user32
                        hwnd_fg = user32.GetForegroundWindow()
                        if hwnd_fg:
                            rect = ctypes.wintypes.RECT()
                            user32.GetWindowRect(hwnd_fg, ctypes.byref(rect))
                            # Get the monitor info for the monitor containing the foreground window
                            MONITOR_DEFAULTTONEAREST = 2
                            hmon = user32.MonitorFromWindow(hwnd_fg, MONITOR_DEFAULTTONEAREST)
                            if hmon:
                                class MONITORINFO(ctypes.Structure):
                                    _fields_ = [('cbSize', ctypes.wintypes.DWORD),
                                                 ('rcMonitor', ctypes.wintypes.RECT),
                                                 ('rcWork', ctypes.wintypes.RECT),
                                                 ('dwFlags', ctypes.wintypes.DWORD)]
                                mi = MONITORINFO()
                                mi.cbSize = ctypes.sizeof(MONITORINFO)
                                user32.GetMonitorInfoW(hmon, ctypes.byref(mi))
                                mr = mi.rcMonitor
                                # Window covers the entire monitor → fullscreen
                                if (rect.left <= mr.left and rect.top <= mr.top
                                        and rect.right >= mr.right and rect.bottom >= mr.bottom):
                                    _suppress_fs = True
                    except Exception:
                        pass
                # Fallback: check Qt windows (same-process only)
                if not _suppress_fs:
                    app = QApplication.instance()
                    if app is not None:
                        aw = app.activeWindow()
                        if aw is not None and hasattr(aw, 'isFullScreen') and aw.isFullScreen():
                            _suppress_fs = True
        except Exception:
            pass
        if _suppress_fs:
            use_popup = False

        if use_popup and total_new:
            try:
                entries = []
                header_icon = None
                max_items_total = int(plugin_prefs.get('popup_max_items_total', 40) or 40)
                max_items_total = max(10, min(200, max_items_total))

                added = 0
                for feed_id, r in (feeds_results or {}).items():
                    if not isinstance(r, dict) or not r.get('ok'):
                        continue
                    if bool(r.get('first_fetch')) and not notify_first:
                        continue
                    if added >= max_items_total:
                        break
                    new_items = list(r.get('new_items') or [])
                    if not new_items:
                        continue
                    f = feeds_by_id.get(feed_id) or {}
                    feed_title = (f.get('title') or r.get('title') or _('(unknown feed)'))
                    try:
                        icon = self._qicon_from_icon_bytes(r.get('icon'))
                        if header_icon is None and icon is not None:
                            header_icon = icon
                    except Exception:
                        icon = None
                    new_items_sorted = sorted(
                        new_items,
                        key=lambda it: int((it or {}).get('published_ts') or 0),
                        reverse=True,
                    )

                    remaining = max(0, max_items_total - added)
                    if remaining <= 0:
                        break
                    items_to_add = []
                    for it in new_items_sorted:
                        try:
                            iid = str((it or {}).get('id') or '').strip()
                            if not iid:
                                continue
                            items_to_add.append((iid, it))
                            if len(items_to_add) >= remaining:
                                break
                        except Exception:
                            continue

                    if not items_to_add:
                        continue

                    entries.append({'_is_feed_header': True, 'feed_title': feed_title})
                    for iid, it in items_to_add:
                        try:
                            entries.append({
                                'feed_id': str(feed_id),
                                'feed_title': feed_title,
                                'feed_icon': icon,
                                'item_id': iid,
                                'item_title': (it or {}).get('title') or '',
                                'link': (it or {}).get('link') or '',
                            })
                            added += 1
                        except Exception:
                            continue
                if entries:
                    # If any feeds failed in this update, keep the popup visible
                    # (no auto-dismiss) so the user can act on it.
                    try:
                        sticky_timeout = 0 if int(total_failed or 0) > 0 else None
                    except Exception:
                        sticky_timeout = None
                    self._show_in_app_popup(entries, total_new=total_new, header_icon=header_icon, timeout_ms=sticky_timeout)
            except Exception:
                pass

        # Notify (Comfy Reminders style) if enabled (suppressed when popup is enabled)
        if (not _suppress_fs) and (not use_popup) and (not silent) and bool(plugin_prefs.get('notify_on_new', True)) and self.notifier:
            can_notify = hasattr(self.notifier, 'ok') and self.notifier.ok or not hasattr(self.notifier, 'ok')
            if can_notify:
                try:
                    max_n = int(plugin_prefs.get('max_notifications_per_update', 3) or 3)
                    sent = 0
                    sent_ids = set()

                    # First: honor any per-feed override `always_notify` by sending
                    # notifications for those feeds regardless of the global cap.
                    try:
                        for feed_id, r in feeds_results.items():
                            if not isinstance(r, dict) or not r.get('ok'):
                                continue
                            # Skip first-fetch notifications unless configured
                            if bool(r.get('first_fetch')) and not notify_first:
                                continue
                            f = feeds_by_id.get(feed_id) or {}
                            if bool(f.get('always_notify', False)):
                                # Prefer the user-edited feed title from stored feeds over
                                # the parsed title from the feed update so toasts reflect
                                # the name the user sees in the UI.
                                feed_title = (f.get('title') or r.get('title') or _('(unknown feed)'))
                                new_items_sorted = sorted(
                                    list(r.get('new_items') or []),
                                    key=lambda it: int(it.get('published_ts') or 0),
                                    reverse=True,
                                )
                                for it in new_items_sorted:
                                    iid = str(it.get('id') or '')
                                    if not iid or iid in sent_ids:
                                        continue
                                    item_title = (it.get('title') or '').strip() or _('(untitled)')
                                    try:
                                        self._send_notification(feed_title, item_title, icon_bytes=r.get('icon'))
                                    except Exception:
                                        pass
                                    sent_ids.add(iid)
                                    # Do not increment `sent` so these do not count against the global cap
                    except Exception:
                        pass

                    # Then: Send per-item notifications for the first few remaining new items
                    for feed_id, r in feeds_results.items():
                        if sent >= max_n:
                            break
                        if not isinstance(r, dict) or not r.get('ok'):
                            continue
                        if bool(r.get('first_fetch')) and not notify_first:
                            continue

                        # Prefer the stored/user-edited feed title so notifications use
                        # the name the user set in the UI rather than the fetched title.
                        feed_title = (feeds_by_id.get(feed_id) or {}).get('title') or r.get('title') or _('(unknown feed)')
                        new_items_sorted = sorted(
                            list(r.get('new_items') or []),
                            key=lambda it: int(it.get('published_ts') or 0),
                            reverse=True,
                        )
                        for it in new_items_sorted:
                            if sent >= max_n:
                                break
                            iid = str(it.get('id') or '')
                            if not iid or iid in sent_ids:
                                continue
                            item_title = (it.get('title') or '').strip() or _('(untitled)')
                            # Put feed title in the notification title (more visible on Windows)
                            try:
                                self._send_notification(feed_title, item_title, icon_bytes=r.get('icon'))
                            except Exception:
                                pass
                            sent_ids.add(iid)
                            sent += 1

                    # If we didn't send per-item (or there are many), fall back to a summary
                    if sent == 0 and total_new:
                        try:
                            self._send_notification(_('%d new RSS items') % total_new, _('RSS Reader'))
                        except Exception:
                            pass
                except Exception:
                    pass

        if self._dialog is not None:
            self._dialog.on_update_results(feeds_results)
            # Clear badge once the dialog has ingested results (user can see counts inside)
            try:
                self._pending_new_count = 0
                # Refresh the toolbar label & tooltip so that counts are cleared
                # and the tooltip continues to show current settings
                try:
                    self._refresh_toolbar_label()
                except Exception:
                    # Fallback to plain label if refresh fails
                    try:
                        self.qaction.setText(_('RSS Reader'))
                        self.qaction.setToolTip(_('RSS Reader'))
                    except Exception:
                        pass
            except Exception:
                pass

        if total_failed and self._dialog is None:
            # If updated in the background (timer), don't spam dialogs.
            try:
                if offline_batch:
                    self.gui.status_bar.show_message(_('RSS Reader: offline (will retry later)'), 3000, show_notification=False)
                else:
                    self.gui.status_bar.show_message(_('RSS Reader: %d feeds failed to update') % total_failed, 4000, show_notification=False)
            except Exception:
                pass

