# -*- coding: utf-8 -*-
# Icon source: http://icon-library.com/recreation-001-512.html, icon #172376
# Keywords: Font, Black-and-white, Line art, Clip art, Playing sports, Graphics, Gesture

__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

from calibre.customize import InterfaceActionBase
try:
    # menu_action_unique_name is not available on very old calibre versions
    from calibre.gui2.actions import InterfaceAction, menu_action_unique_name
except Exception:
    from calibre.gui2.actions import InterfaceAction
    menu_action_unique_name = None
from calibre.constants import numeric_version as calibre_version
try:
    from qt.core import QLabel, QSizePolicy, QTimer, QToolButton
except Exception:
    try:
        from PyQt4.QtGui import QLabel, QSizePolicy, QToolButton
        try:
            from PyQt4.QtCore import QTimer
        except Exception:
            QTimer = None
    except Exception:
        try:
            from PyQt5.QtWidgets import QLabel, QSizePolicy, QToolButton
            try:
                from PyQt5.QtCore import QTimer
            except Exception:
                QTimer = None
        except Exception:
            raise
from calibre.utils.config import JSONConfig
try:
    from calibre.gui2 import Application
except Exception:
    Application = None

try:
    from qt.core import QIcon, QPixmap, QApplication
except Exception:
    try:
        from PyQt4.QtGui import QIcon, QPixmap, QApplication
    except Exception:
        try:
            from PyQt5.QtGui import QIcon, QPixmap
            from PyQt5.QtWidgets import QApplication
        except Exception:
            QIcon = None
            QPixmap = None
            QApplication = None

import os
from calibre.utils.logging import default_log as prints

DEBUG = False

try:
    text_type = unicode  # noqa: F821 (py2)
except Exception:
    text_type = str


def _debug(msg):
    if not DEBUG:
        return
    try:
        prints(msg)
    except Exception:
        try:
            import sys
            if isinstance(msg, text_type):
                try:
                    sys.stderr.write(msg + '\n')
                except Exception:
                    sys.stderr.write((msg.encode('utf-8') if hasattr(msg, 'encode') else str(msg)) + '\n')
            else:
                sys.stderr.write(str(msg) + '\n')
        except Exception:
            pass

KEY_SHOW_CONFIG_TITLE = 'show_config_title'
KEY_SHOW_CONFIG_STATUS = 'show_config_status'
KEY_SHOW_LIBRARY_STATUS = 'show_library_status'
KEY_SHOW_LIBRARY_TITLE = 'show_library_title'

def _get_prefs():
    prefs = JSONConfig('plugins/current_paths')
    prefs.defaults[KEY_SHOW_CONFIG_TITLE] = True
    prefs.defaults[KEY_SHOW_CONFIG_STATUS] = True
    prefs.defaults[KEY_SHOW_LIBRARY_STATUS] = False
    prefs.defaults[KEY_SHOW_LIBRARY_TITLE] = False
    return prefs

def _get_config_path():
    from calibre.utils.config import config_dir
    try:
        return config_dir() if callable(config_dir) else config_dir
    except Exception:
        return config_dir

def _get_library_path(gui):
    """Get the current library path in a backwards-compatible way."""
    try:
        if hasattr(gui, 'library_path'):
            return gui.library_path
    except Exception:
        pass
    try:
        if hasattr(gui, 'current_db') and hasattr(gui.current_db, 'path'):
            return gui.current_db.path
    except Exception:
        pass
    try:
        if hasattr(gui, 'library_view') and hasattr(gui.library_view, 'model'):
            model = gui.library_view.model()
            if hasattr(model, 'db') and hasattr(model.db, 'path'):
                return model.db.path
    except Exception:
        pass
    return None

def _is_dark_theme():
    try:
        if QApplication is None:
            return False
        app = QApplication.instance()
        if not app:
            return False
        palette = app.palette()
        window_color = palette.color(palette.Window)
        text_color = palette.color(palette.WindowText)
        return text_color.lightness() > window_color.lightness()
    except Exception:
        return False

def _call_get_icons(icon_name, plugin_path=None, plugin_name=None):
    gi = globals().get('get_icons', None)
    if not callable(gi):
        try:
            from calibre.customize.zipplugin import get_icons as gi  # noqa
        except Exception:
            gi = None
    if not callable(gi):
        return None

    candidates = []

    # Calibre plugins conventionally store images under images/ in the zip.
    # Some zip builders on Windows can end up with backslashes in member names.
    # Try both path separators to be safe (notably for Calibre 5.44).
    icon_name_backslash = icon_name.replace('/', '\\') if isinstance(icon_name, str) else icon_name
    icon_base = icon_name.replace('images/', '') if icon_name.startswith('images/') else icon_name
    icon_base_backslash = icon_base.replace('/', '\\') if isinstance(icon_base, str) else icon_base

    # Most common (injected) get_icons signatures
    candidates.append((icon_name,))
    if icon_name_backslash != icon_name:
        candidates.append((icon_name_backslash,))
    if icon_name.startswith('images/'):
        candidates.append((icon_base,))
        if icon_base_backslash != icon_base:
            candidates.append((icon_base_backslash,))

    # Some environments expose a get_icons that wants a plugin name
    if plugin_name:
        candidates.append((icon_name, plugin_name))
        if icon_name_backslash != icon_name:
            candidates.append((icon_name_backslash, plugin_name))
        if icon_name.startswith('images/'):
            candidates.append((icon_base, plugin_name))
            if icon_base_backslash != icon_base:
                candidates.append((icon_base_backslash, plugin_name))

    # Raw zipplugin.get_icons wants (zfp, name_or_list)
    if plugin_path:
        candidates.append((plugin_path, icon_name))
        if icon_name_backslash != icon_name:
            candidates.append((plugin_path, icon_name_backslash))
        if icon_name.startswith('images/'):
            candidates.append((plugin_path, icon_base))
            if icon_base_backslash != icon_base:
                candidates.append((plugin_path, icon_base_backslash))

    for args in candidates:
        try:
            icon = gi(*args)
            if icon is not None and hasattr(icon, 'isNull') and not icon.isNull():
                _debug("DEBUG: get_icons succeeded with args: {}".format(args))
                return icon
        except TypeError:
            continue
        except Exception:
            continue
    return None

def _get_plugin_icon(icon_name, plugin_path=None, plugin_name=None):
    if QIcon is None:
        return None

    _debug("DEBUG: _get_plugin_icon called with icon_name='{}', plugin_path='{}', plugin_name='{}'".format(icon_name, plugin_path, plugin_name))

    # Prefer plugin-provided get_icons (injected by plugin loader on older Calibre)
    icon = _call_get_icons(icon_name, plugin_path=plugin_path, plugin_name=plugin_name)
    if icon is not None:
        _debug("DEBUG: _call_get_icons succeeded for '{}'".format(icon_name))
        return icon

    # Calibre 6+ supports QIcon.ic (theme + cache)
    try:
        ic = getattr(QIcon, 'ic', None)
        if callable(ic):
            icon = ic(icon_name)
            if icon is not None and not icon.isNull():
                _debug("DEBUG: QIcon.ic succeeded for '{}'".format(icon_name))
                return icon
    except Exception:
        pass

    # Try Calibre resource lookup (I is usually injected by the loader)
    try:
        I_func = globals().get('I', None)
        if callable(I_func):
            icon = QIcon(I_func(icon_name))
            if icon is not None and not icon.isNull():
                _debug("DEBUG: I() succeeded for '{}'".format(icon_name))
                return icon
    except Exception:
        pass

    # Development/unzipped fallback: load from local images folder
    try:
        if QPixmap is not None:
            local_path = os.path.join(os.path.dirname(__file__), icon_name.replace('images/', 'images' + os.sep))
            if os.path.exists(local_path):
                pix = QPixmap()
                if pix.load(local_path):
                    icon = QIcon(pix)
                    if icon is not None and not icon.isNull():
                        _debug("DEBUG: Local load succeeded for '{}'".format(local_path))
                        return icon
    except Exception:
        pass

    # Last resort: try to load directly from the plugin zip file (for installed plugins)
    try:
        if plugin_path and os.path.exists(plugin_path):
            import zipfile
            with zipfile.ZipFile(plugin_path, 'r') as zf:
                # Try the icon name as is, and without 'images/' prefix
                possible_names = [icon_name]
                if icon_name.startswith('images/'):
                    possible_names.append(icon_name[len('images/'):])
                for name in possible_names:
                    try:
                        data = zf.read(name)
                        if data:
                            pix = QPixmap()
                            if pix.loadFromData(data):
                                icon = QIcon(pix)
                                if icon is not None and not icon.isNull():
                                    _debug("DEBUG: Direct zip read succeeded for '{}' in zip".format(name))
                                    return icon
                    except KeyError:
                        continue
        _debug("DEBUG: Direct zip read failed for '{}'".format(icon_name))
    except Exception:
        _debug("DEBUG: Direct zip read exception for '{}'".format(icon_name))

    _debug("DEBUG: All icon loading methods failed for '{}'".format(icon_name))
    return QIcon()

def _is_virtual_library_active(gui):
    try:
        if hasattr(gui, 'library_view') and hasattr(gui.library_view, 'model'):
            model = gui.library_view.model()
            if hasattr(model, 'db') and hasattr(model.db, 'data'):
                return bool(model.db.data.search_restriction_applied())
    except Exception:
        pass
    return False

def _get_virtual_library_name(gui):
    try:
        if not _is_virtual_library_active(gui):
            return None
        if hasattr(gui, 'library_view') and hasattr(gui.library_view, 'model'):
            model = gui.library_view.model()
            if hasattr(model, 'db') and hasattr(model.db, 'data'):
                name = model.db.data.get_base_restriction_name()
                if name:
                    name = name.strip()
                return name or None
    except Exception:
        pass
    return None

def _get_library_path(gui):
    """Get the current library path in a backwards-compatible way."""
    try:
        # Modern Calibre: use gui.library_path
        if hasattr(gui, 'library_path'):
            return gui.library_path
    except Exception:
        pass
    try:
        # Fallback: use current_db.path
        if hasattr(gui, 'current_db') and hasattr(gui.current_db, 'path'):
            return gui.current_db.path
    except Exception:
        pass
    try:
        # Another fallback: library_view.model().db.path
        if hasattr(gui, 'library_view') and hasattr(gui.library_view, 'model'):
            model = gui.library_view.model()
            if hasattr(model, 'db') and hasattr(model.db, 'path'):
                return model.db.path
    except Exception:
        pass
    return None

class CurrentPathsAction(InterfaceAction):
    
    def __init__(self, gui, site_customization):
        InterfaceAction.__init__(self, gui, site_customization)
    def _register_vl_listener(self):
        _debug("DEBUG: _register_vl_listener called")
        connected = {'any': False, 'model': False, 'menu': False, 'strong': False}

        # Try to get a queued connection type across Qt bindings/Calibre versions.
        queued = None
        try:
            from qt.core import Qt as _Qt
            queued = getattr(getattr(_Qt, 'ConnectionType', _Qt), 'QueuedConnection', None)
        except Exception:
            try:
                from PyQt5.QtCore import Qt as _Qt
                queued = getattr(getattr(_Qt, 'ConnectionType', _Qt), 'QueuedConnection', None)
            except Exception:
                try:
                    from PyQt4.QtCore import Qt as _Qt
                    queued = getattr(getattr(_Qt, 'ConnectionType', _Qt), 'QueuedConnection', None)
                except Exception:
                    queued = None

        def _try_connect(signal, slot):
            if signal is None:
                return False
            try:
                if queued is not None:
                    try:
                        signal.connect(slot, type=queued)
                        connected['any'] = True
                        return True
                    except Exception:
                        pass
                signal.connect(slot)
                connected['any'] = True
                return True
            except Exception:
                return False

        try:
            # Model signals: usually the most reliable way to learn about VL/search restriction changes.
            lv = getattr(self.gui, 'library_view', None)
            if lv and hasattr(lv, 'model'):
                model = lv.model()
                if _try_connect(getattr(model, 'database_changed', None), self._on_database_changed):
                    connected['model'] = True
                if _try_connect(getattr(model, 'count_changed_signal', None), self._on_count_changed):
                    connected['model'] = True
                if getattr(model, 'database_changed', None) is not None or getattr(model, 'count_changed_signal', None) is not None:
                    # Only count as strong if the connections actually succeeded (best-effort)
                    connected['strong'] = connected['any']

            # Virtual library menu: when present, it fires when user is about to change VL.
            vl_menu = getattr(self.gui, 'virtual_library_menu', None)
            if not vl_menu:
                # Some Calibre versions expose the action rather than a direct menu attribute
                try:
                    ia = getattr(self.gui, 'iactions', None)
                    if ia and isinstance(ia, dict):
                        vla = ia.get('Virtual library') or ia.get('Virtual Library')
                        if vla is not None:
                            vl_menu = getattr(vla, 'menu', None)
                            if vl_menu is None and hasattr(vla, 'qaction'):
                                try:
                                    vl_menu = vla.qaction.menu()
                                except Exception:
                                    vl_menu = None
                except Exception:
                    vl_menu = None
            if vl_menu:
                if _try_connect(getattr(vl_menu, 'aboutToShow', None), self._on_vl_menu_show):
                    connected['menu'] = True
                # Menu events are strong enough to avoid polling
                if getattr(vl_menu, 'aboutToShow', None) is not None and connected['any']:
                    connected['strong'] = True

            # Window title changes: reapply after Calibre overwrites it (e.g. library switch).
            try:
                _try_connect(getattr(self.gui, 'windowTitleChanged', None), self._on_window_title_changed)
            except Exception:
                pass
        except Exception:
            pass

        # Decide whether we can avoid polling.
        # On Calibre < 4.0, model signals typically do not fire for VL changes, but the VL menu does.
        if calibre_version < (4, 0, 0):
            strong = bool(connected['menu'])
        else:
            strong = bool(connected['strong'])
        _debug("DEBUG: _register_vl_listener connected model={}, menu={}, strong={}, final={}".format(
            bool(connected['model']), bool(connected['menu']), bool(connected['strong']), strong
        ))
        return strong

    def library_changed(self, db):
        # Called by Calibre on library switches; schedule update after Calibre finishes its own title changes.
        try:
            self._last_vl_state = (_is_virtual_library_active(self.gui), _get_virtual_library_name(self.gui))
            if DEBUG:
                prints("DEBUG: library_changed called, new VL state: {}".format(self._last_vl_state))
        except Exception:
            pass
        # Reconnect signals to the new model
        try:
            new_strong = self._register_vl_listener()
            if DEBUG:
                prints("DEBUG: Reconnected listeners in library_changed, strong: {}".format(new_strong))
            # If now strong, stop polling; if not, ensure polling is running
            if new_strong and hasattr(self, '_vl_poll_timer') and self._vl_poll_timer is not None:
                self._vl_poll_timer.stop()
                if DEBUG:
                    prints("DEBUG: Stopped polling as strong listeners connected")
            elif not new_strong and (not hasattr(self, '_vl_poll_timer') or self._vl_poll_timer is None or not self._vl_poll_timer.isActive()):
                if QTimer is not None:
                    self._vl_poll_timer = QTimer(self)
                    self._vl_poll_timer.timeout.connect(self._check_vl_state)
                    self._vl_poll_timer.start(2000)
                    if DEBUG:
                        prints("DEBUG: Started polling in library_changed")
        except Exception:
            if DEBUG:
                prints("DEBUG: Exception in library_changed reconnection")
        try:
            QTimer.singleShot(0, self._update_ui)
            QTimer.singleShot(250, self._update_ui)
        except Exception:
            try:
                self._update_ui()
            except Exception:
                pass

    def _on_database_changed(self):
        # Queue an update to run after Calibre finishes its own title updates
        try:
            QTimer.singleShot(0, self._update_ui)
        except Exception:
            self._update_ui()

    def _on_count_changed(self):
        try:
            QTimer.singleShot(0, self._update_ui)
        except Exception:
            self._update_ui()

    def _on_vl_menu_show(self):
        try:
            QTimer.singleShot(0, self._update_ui)
        except Exception:
            self._update_ui()

    def _on_window_title_changed(self, new_title):
        # Reapply our formatting after title changes
        try:
            QTimer.singleShot(0, self._update_ui)
        except Exception:
            self._update_ui()

    def _check_vl_state(self):
        """Poll for VL state changes as fallback."""
        try:
            current_vl_state = (_is_virtual_library_active(self.gui), _get_virtual_library_name(self.gui))
            if current_vl_state != getattr(self, '_last_vl_state', None):
                _debug("DEBUG: VL state changed from {} to {}".format(self._last_vl_state, current_vl_state))
                self._last_vl_state = current_vl_state
                try:
                    QTimer.singleShot(0, self._update_ui)
                except Exception:
                    self._update_ui()
        except Exception:
            _debug("DEBUG: Exception in _check_vl_state")
    def _register_shortcut_actions(self):
        try:
            from qt.core import QMenu
        except Exception:
            try:
                from PyQt4.QtGui import QMenu
            except Exception:
                from PyQt5.QtWidgets import QMenu
        menu = QMenu(self.gui)
        self.qaction.setMenu(menu)

        kb = getattr(self.gui, 'keyboard', None)

        # Prefer the newer create_menu_action API when available (modern Calibre).
        # Fallback to create_action for legacy Calibre versions.
        used_modern = hasattr(self, 'create_menu_action')

        if used_modern:
            try:
                prefs = _get_prefs()
                self.qaction_toggle_config_title = self.create_menu_action(
                    menu, 'toggle_config_title', 'Config Path in Title',
                    icon=None, shortcut=None, description='Show or hide the config path in the window title',
                    triggered=self.do_toggle_config_title
                )
                self.qaction_toggle_config_title.setCheckable(True)
                self.qaction_toggle_config_title.setChecked(bool(prefs.get(KEY_SHOW_CONFIG_TITLE, True)))

                self.qaction_toggle_library_title = self.create_menu_action(
                    menu, 'toggle_library_title', 'Library Path in Title',
                    icon=None, shortcut=None, description='Show or hide the library path in the window title',
                    triggered=self.do_toggle_library_title
                )
                self.qaction_toggle_library_title.setCheckable(True)
                self.qaction_toggle_library_title.setChecked(bool(prefs.get(KEY_SHOW_LIBRARY_TITLE, False)))

                self.qaction_toggle_config_status = self.create_menu_action(
                    menu, 'toggle_config_status', 'Config Path in Status Bar',
                    icon=None, shortcut=None, description='Show or hide the config path in the status bar',
                    triggered=self.do_toggle_config_status
                )
                self.qaction_toggle_config_status.setCheckable(True)
                self.qaction_toggle_config_status.setChecked(bool(prefs.get(KEY_SHOW_CONFIG_STATUS, True)))

                self.qaction_toggle_library_status = self.create_menu_action(
                    menu, 'toggle_library_status', 'Library Path in Status Bar',
                    icon=None, shortcut=None, description='Show or hide the library path in the status bar',
                    triggered=self.do_toggle_library_status
                )
                self.qaction_toggle_library_status.setCheckable(True)
                self.qaction_toggle_library_status.setChecked(bool(prefs.get(KEY_SHOW_LIBRARY_STATUS, False)))

            except Exception:
                used_modern = False

        if not used_modern:
            # Legacy path: create QAction via create_action and add to menu.
            def _make_legacy(menu_text, tooltip, slot, checked_default):
                ac = self.create_action(spec=(menu_text, None, tooltip, ()), attr=menu_text)
                ac.setCheckable(True)
                ac.setChecked(checked_default)
                if slot is not None:
                    ac.triggered.connect(slot)
                menu.addAction(ac)
                # Ensure Calibre's GUI knows about the action for shortcut listing
                try:
                    self.gui.addAction(ac)
                except Exception:
                    pass
                try:
                    self.qaction.addAction(ac)
                except Exception:
                    pass
                return ac

            prefs = _get_prefs()
            self.qaction_toggle_config_title = _make_legacy('Config Path in Title',
                                                    'Show or hide the config path in the window title',
                                                    self.do_toggle_config_title,
                                                    bool(prefs.get(KEY_SHOW_CONFIG_TITLE, True)))
            self.qaction_toggle_library_title = _make_legacy('Library Path in Title',
                                                    'Show or hide the library path in the window title',
                                                    self.do_toggle_library_title,
                                                    bool(prefs.get(KEY_SHOW_LIBRARY_TITLE, False)))
            self.qaction_toggle_config_status = _make_legacy('Config Path in Status Bar',
                                                         'Show or hide the config path in the status bar',
                                                         self.do_toggle_config_status,
                                                         bool(prefs.get(KEY_SHOW_CONFIG_STATUS, True)))
            self.qaction_toggle_library_status = _make_legacy('Library Path in Status Bar',
                                                       'Show or hide the library path in the status bar',
                                                       self.do_toggle_library_status,
                                                       bool(prefs.get(KEY_SHOW_LIBRARY_STATUS, False)))

        # Finalize keyboard manager if present
        if kb is not None:
            try:
                kb.finalize()
            except Exception:
                pass
    # Action names for Calibre shortcut registration
    TOGGLE_CONFIG_TITLE_ACTION = 'Config Path in Title'
    TOGGLE_LIBRARY_TITLE_ACTION = 'Library Path in Title'
    TOGGLE_CONFIG_STATUS_ACTION = 'Config Path in Status Bar'
    TOGGLE_LIBRARY_STATUS_ACTION = 'Library Path in Status Bar'

    def do_toggle_config_title(self):
        prefs = _get_prefs()
        new_state = not bool(prefs.get(KEY_SHOW_CONFIG_TITLE, True))
        prefs[KEY_SHOW_CONFIG_TITLE] = new_state
        if hasattr(self, 'qaction_toggle_config_title'):
            self.qaction_toggle_config_title.setChecked(new_state)
        self._update_ui()

    def do_toggle_library_title(self):
        prefs = _get_prefs()
        new_state = not bool(prefs.get(KEY_SHOW_LIBRARY_TITLE, False))
        prefs[KEY_SHOW_LIBRARY_TITLE] = new_state
        if hasattr(self, 'qaction_toggle_library_title'):
            self.qaction_toggle_library_title.setChecked(new_state)
        self._update_ui()

    def do_toggle_config_status(self):
        prefs = _get_prefs()
        new_state = not bool(prefs.get(KEY_SHOW_CONFIG_STATUS, True))
        prefs[KEY_SHOW_CONFIG_STATUS] = new_state
        if hasattr(self, 'qaction_toggle_config_status'):
            self.qaction_toggle_config_status.setChecked(new_state)
        self._update_ui()

    def do_toggle_library_status(self):
        prefs = _get_prefs()
        new_state = not bool(prefs.get(KEY_SHOW_LIBRARY_STATUS, False))
        prefs[KEY_SHOW_LIBRARY_STATUS] = new_state
        if hasattr(self, 'qaction_toggle_library_status'):
            self.qaction_toggle_library_status.setChecked(new_state)
        self._update_ui()

    def rebuild_icon(self):
        icon_path = 'images/iconplugin_dark.png' if _is_dark_theme() else 'images/iconplugin_light.png'
        icon = _get_plugin_icon(icon_path, plugin_path=getattr(self, 'plugin_path', None), plugin_name=self.name)
        if icon is not None:
            self.qaction.setIcon(icon)

    name = 'Current Paths'
    action_spec = (
        u'Current Paths',
        None,
        u'Show or hide current config and library directory paths in the Calibre window title and status bar.',
        None
    )
    action_type = 'current'
    
    def genesis(self):
        _debug("DEBUG: genesis called")
        try:
            # Run multiple times to reliably catch GUI components becoming available.
            try:
                self._update_ui()
            except Exception as e:
                _debug("DEBUG: Exception in initial _update_ui: {}".format(e))
            if QTimer is not None:
                try:
                    QTimer.singleShot(0, lambda: self._update_ui())
                    QTimer.singleShot(250, lambda: self._update_ui())
                    QTimer.singleShot(1500, lambda: self._update_ui())
                except Exception as e:
                    _debug("DEBUG: Exception in QTimer.singleShot: {}".format(e))

            # Register actions and ensure main plugin action is in GUI for legacy shortcut discovery
            try:
                self._register_shortcut_actions()
                _debug("DEBUG: _register_shortcut_actions completed")
            except Exception as e:
                _debug("DEBUG: Exception in _register_shortcut_actions: {}".format(e))
            try:
                self.gui.addAction(self.qaction)
            except Exception as e:
                _debug("DEBUG: Exception in addAction: {}".format(e))

            # Set up the theme-aware icon
            try:
                self.rebuild_icon()
                _debug("DEBUG: rebuild_icon completed")
            except Exception as e:
                _debug("DEBUG: Exception in rebuild_icon: {}".format(e))

            # Listen for theme changes to update the icon instantly
            try:
                if Application is not None:
                    app = Application.instance()
                    if app:
                        app.palette_changed.connect(self.rebuild_icon)
            except Exception as e:
                _debug("DEBUG: Exception connecting palette_changed: {}".format(e))

            # Register listener for VL/search restriction changes
            vl_listeners_connected = False
            try:
                vl_listeners_connected = bool(self._register_vl_listener())
                _debug("DEBUG: vl_listeners_connected: {}".format(vl_listeners_connected))
            except Exception as e:
                _debug("DEBUG: Exception in _register_vl_listener: {}".format(e))
                vl_listeners_connected = False

            # Cache VL state for polling fallback comparisons
            try:
                self._last_vl_state = (_is_virtual_library_active(self.gui), _get_virtual_library_name(self.gui))
                _debug("DEBUG: Initial VL state: {}".format(self._last_vl_state))
            except Exception as e:
                _debug("DEBUG: Exception in VL state caching: {}".format(e))
                self._last_vl_state = None

            # Start VL polling only if we failed to hook any meaningful signals
            if (not vl_listeners_connected) and QTimer is not None:
                try:
                    self._vl_poll_timer = QTimer(self)
                    self._vl_poll_timer.timeout.connect(self._check_vl_state)
                    # Keep polling low-frequency to avoid overhead; menu hooks (when available) avoid polling entirely.
                    poll_interval = 2000 if calibre_version < (4, 0, 0) else 3000
                    self._vl_poll_timer.start(poll_interval)  # Faster polling on older Calibre
                    _debug("DEBUG: Started VL polling timer with interval {}".format(poll_interval))
                except Exception as e:
                    _debug("DEBUG: Exception starting VL polling timer: {}".format(e))
        except Exception as e:
            _debug("DEBUG: Outer exception in genesis: {}".format(e))
            try:
                import traceback
                _debug(traceback.format_exc())
            except Exception:
                pass

    def initialization_complete(self):
        self._update_ui()
        if QTimer is not None:
            try:
                QTimer.singleShot(0, self._update_ui)
                QTimer.singleShot(250, self._update_ui)
            except Exception:
                pass
        # Ensure VL state is cached after initialization
        try:
            self._last_vl_state = (_is_virtual_library_active(self.gui), _get_virtual_library_name(self.gui))
            _debug("DEBUG: VL state updated in initialization_complete: {}".format(self._last_vl_state))
        except Exception:
            self._last_vl_state = None

    def _update_ui(self):
        try:
            config_path = _get_config_path()
            library_path = _get_library_path(self.gui)
            main = self.gui
            prefs = _get_prefs()
            show_config_title = bool(prefs.get(KEY_SHOW_CONFIG_TITLE, True))
            show_library_title = bool(prefs.get(KEY_SHOW_LIBRARY_TITLE, False))
            show_config_status = bool(prefs.get(KEY_SHOW_CONFIG_STATUS, True))
            show_library_status = bool(prefs.get(KEY_SHOW_LIBRARY_STATUS, False))

            vl_active = _is_virtual_library_active(main)
            vl_name = _get_virtual_library_name(main) if vl_active else None

            # Update window title: always show info, add VL name if active
            if hasattr(main, 'setWindowTitle') and hasattr(main, 'windowTitle'):
                current_title = main.windowTitle() or ''
                import re
                match = re.search(r' \u2014  (Config Path|Library Path):', current_title)
                if match:
                    title_up_to = current_title[:match.start()]
                else:
                    title_up_to = current_title

                # If Calibre already has a '|| name ||' segment, preserve it rather than replacing
                m_lib = re.search(r'\|\|\s*([^|]+?)\s*\|\|', title_up_to)
                detected_lib_name = None
                if m_lib:
                    lib_with_vl = m_lib.group(1).strip()
                    # Strip VL segment if present (e.g., "library :: vl_name" -> "library")
                    detected_lib_name = lib_with_vl.split(' :: ')[0].strip() if ' :: ' in lib_with_vl else lib_with_vl
                    base_title = title_up_to[:m_lib.start()]
                else:
                    base_title = title_up_to
                    # Trim trailing whitespace and normalize trailing em-dash spacing
                    base_title = base_title.rstrip()
                    try:
                        base_title = re.sub(r'\s*\u2014\s*$', ' \u2014', base_title)
                    except Exception:
                        base_title = re.sub(r'\s*-\s*$', ' -', base_title)

                suffix_parts = []
                if show_config_title:
                    suffix_parts.append("Config Path: {}".format(config_path))
                if show_library_title and library_path:
                    suffix_parts.append("Library Path: {}".format(library_path))
                vl_segment = ''
                if vl_active and vl_name:
                    vl_segment = ' :: {}'.format(vl_name)
                # Compose new title
                try:
                    # Title uses double pipes and no em-dash separator
                    if suffix_parts:
                        suffix = '  ' + '  ||  '.join(suffix_parts)
                    else:
                        suffix = ''
                except Exception:
                    suffix = (' ' + ' | '.join(suffix_parts)) if suffix_parts else ''
                import os
                # Prefer existing detected library title if present, otherwise use basename of library_path
                lib_name = detected_lib_name
                if not lib_name:
                    try:
                        if library_path:
                            lib_name = os.path.basename(os.path.normpath(library_path))
                    except Exception:
                        lib_name = None

                # Build the new title keeping Calibre's original library name if present
                if lib_name:
                    new_title = base_title + ' || {}'.format(lib_name)
                    if vl_segment:
                        new_title += vl_segment
                    new_title += ' ||' + suffix
                elif vl_segment:
                    # No library name, but VL present: show VL name alone
                    new_title = base_title + ' ||{} ||'.format(vl_segment) + suffix
                else:
                    # No lib name and no VL: just append suffix (if any) without adding a library segment
                    new_title = base_title + (' ||' + suffix if suffix else '')

                # Collapse any excessive whitespace before '||' to avoid visual gaps
                new_title = re.sub(r'\s+\|\|', ' ||', new_title)
                try:
                    main.setWindowTitle(new_title)
                except Exception as e:
                    pass

            # Update status bar (idempotent + reversible)
            if hasattr(main, 'status_bar'):
                status_bar = main.status_bar
                if hasattr(main, '_current_paths_widget') and main._current_paths_widget:
                    status_bar.removeWidget(main._current_paths_widget)
                    main._current_paths_widget = None

                defmsg = getattr(status_bar, 'defmsg', None)
                if defmsg:
                    if hasattr(status_bar, 'set_label'):
                        status_bar.set_label()

                if show_config_status or (show_library_status and library_path):
                    # Prefer a real em dash on modern Python/Qt. On Py2/Qt4 try
                    # to produce a proper QString from UTF-8 to avoid mojibake
                    # (e.g. "â" appearing). If that fails, fall back to ASCII.
                    import sys
                    try:
                        sep = u' \u2014'
                        if sys.version_info[0] < 3:
                            try:
                                # PyQt4: construct a QString from UTF-8 bytes
                                from PyQt4.QtCore import QString
                                sep = QString.fromUtf8(sep)
                            except Exception:
                                # If QString unavailable, use safe ASCII fallback
                                sep = ' -'
                        # Build the pieces using Unicode (Py3) or QString (Py2)
                        text_parts = []
                        if show_config_status:
                            text_parts.append(u"Config Path: {}".format(config_path))
                        if show_library_status and library_path:
                            text_parts.append(u"Library Path: {}".format(library_path))
                        text = sep + '  ' + '  ||  '.join(text_parts)
                        config_label = QLabel(text)
                    except Exception:
                        # Final fallback: ASCII dash and str() values
                        sep = ' -'
                        text_parts = []
                        if show_config_status:
                            text_parts.append('Config Path: ' + str(config_path))
                        if show_library_status and library_path:
                            text_parts.append('Library Path: ' + str(library_path))
                        text = sep + '  ' + '  |  '.join(text_parts)
                        config_label = QLabel(text)
                    try:
                        sp_max = QSizePolicy.Policy.Maximum
                        sp_pref = QSizePolicy.Policy.Preferred
                    except Exception:
                        sp_max = QSizePolicy.Maximum
                        sp_pref = QSizePolicy.Preferred
                    config_label.setSizePolicy(sp_max, sp_pref)
                    widgets = [status_bar.layout().itemAt(i).widget() for i in range(status_bar.layout().count())]
                    idx = widgets.index(defmsg) if defmsg in widgets else -1
                    if idx >= 0:
                        status_bar.insertWidget(idx+1, config_label)
                        main._current_paths_widget = config_label
                    else:
                        status_bar.addWidget(config_label)
                        main._current_paths_widget = config_label
                else:
                    pass
            else:
                pass
        except Exception as e:
            import traceback
            pass

        # Update menu action checked states if they exist
        try:
            if hasattr(self, 'qaction_toggle_config_title'):
                self.qaction_toggle_config_title.setChecked(show_config_title)
            if hasattr(self, 'qaction_toggle_library_title'):
                self.qaction_toggle_library_title.setChecked(show_library_title)
            if hasattr(self, 'qaction_toggle_config_status'):
                self.qaction_toggle_config_status.setChecked(show_config_status)
            if hasattr(self, 'qaction_toggle_library_status'):
                self.qaction_toggle_library_status.setChecked(show_library_status)
        except Exception as e:
            pass

class CurrentPathsPlugin(InterfaceActionBase):
    name = 'Current Paths'
    description = 'Show current config and library directory paths in the Calibre window title and status bar.'
    supported_platforms = ['windows', 'osx', 'linux']
    author = 'Comfy.n'
    version = (2, 0, 2)
    minimum_calibre_version = (3, 48, 0)

    def is_customizable(self):
        # Disable Calibre's built-in plugin-config dialog; plugin provides its own UI via toolbar/menu
        return False

    def config_widget(self):
        # Config dialog is disabled; return None to be safe if called unexpectedly
        return None

    def save_settings(self, config_widget):
        # No-op: settings are managed via plugin UI checkboxes
        try:
            if config_widget is not None and hasattr(config_widget, 'save_settings'):
                config_widget.save_settings()
        except Exception:
            pass

    def load_actual_plugin(self, gui):
        return CurrentPathsAction(gui, self.site_customization)
