# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from PyQt5.QtWidgets import QLabel
from PyQt5.QtCore import Qt

class ElidedLabel(QLabel):
    """Label that elides text in the middle when too long"""
    def __init__(self, parent=None):
        super().__init__(parent)
        self.full_text = ""
    
    def setText(self, text):
        self.full_text = text
        self.setToolTip(text)  # Show full text on hover
        # Elide in the middle
        metrics = self.fontMetrics()
        if metrics.horizontalAdvance(text) > self.width():
            # Split text to show start and end
            available_width = self.width() - metrics.horizontalAdvance("...")
            left_width = available_width // 2
            right_width = available_width - left_width
            
            left_text = metrics.elidedText(text, Qt.ElideRight, left_width)
            right_text = metrics.elidedText(text[::-1], Qt.ElideLeft, right_width)[::-1]
            
            display_text = left_text[:-3] + "..." + right_text[3:]
            super().setText(display_text)
        else:
            super().setText(text)
    
    def resizeEvent(self, event):
        super().resizeEvent(event)
        if self.full_text:
            self.setText(self.full_text)