# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from PyQt5.QtWidgets import (QWidget, QLabel, QVBoxLayout, QGroupBox, 
                           QGridLayout, QTreeWidget, QTreeWidgetItem)
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QColor

from calibre_plugins.opf_helper.constants import DARK_MODE_COLORS, LIGHT_MODE_COLORS
from calibre_plugins.opf_helper.utils import is_dark_theme

class StatsPanel(QWidget):
    """Panel showing statistics and metadata from OPF file"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.is_dark = is_dark_theme()
        self.metadata_colors = DARK_MODE_COLORS if self.is_dark else LIGHT_MODE_COLORS
        self.setup_ui()
        
    def setup_ui(self):
        layout = QVBoxLayout()
        self.setLayout(layout)
        
        # Add stats sections
        general_group = QGroupBox("General Statistics")
        general_layout = QGridLayout()
        general_group.setLayout(general_layout)
        layout.addWidget(general_group)
        
        self.total_elements = QLabel("Total Elements: 0")
        self.max_depth = QLabel("Maximum Depth: 0")
        self.namespaces = QLabel("Namespaces: 0")
        
        general_layout.addWidget(self.total_elements, 0, 0)
        general_layout.addWidget(self.max_depth, 1, 0)
        general_layout.addWidget(self.namespaces, 2, 0)
        
        # Metadata section
        metadata_group = QGroupBox("Metadata Fields")
        metadata_layout = QVBoxLayout()
        metadata_group.setLayout(metadata_layout)
        layout.addWidget(metadata_group)
        
        self.metadata_tree = QTreeWidget()
        self.metadata_tree.setHeaderLabels(["Field", "Value"])
        self.metadata_tree.setAlternatingRowColors(True)
        
        # Set up header style
        header = self.metadata_tree.header()
        header.setDefaultAlignment(Qt.AlignLeft)
        header.setStretchLastSection(True)
        
        metadata_layout.addWidget(self.metadata_tree)
        
        # Referenced files section  
        files_group = QGroupBox("Referenced Files")
        files_layout = QVBoxLayout()
        files_group.setLayout(files_layout)
        layout.addWidget(files_group)
        
        self.files_list = QTreeWidget()
        self.files_list.setHeaderLabels(["Type", "Path"])
        self.files_list.setAlternatingRowColors(True)
        files_layout.addWidget(self.files_list)
        
        # Add stretcher at bottom
        layout.addStretch()
        
    def update_stats(self, root):
        """Update statistics display for given XML root element"""
        # Reset all stats
        self.metadata_tree.clear()
        self.files_list.clear()
        
        if root is None:
            return
            
        # Calculate general stats
        total_elements = 0
        max_depth = 0
        namespaces = set()
        
        def count_stats(element, depth=0):
            nonlocal total_elements, max_depth
            total_elements += 1
            max_depth = max(max_depth, depth)
            
            # Track namespaces
            if element.tag.startswith("{"):
                ns = element.tag[1:].split("}")[0]
                namespaces.add(ns)
                
            # Recurse through children
            for child in element:
                count_stats(child, depth + 1)
        
        count_stats(root)
        
        # Update labels
        self.total_elements.setText(f"Total Elements: {total_elements}")
        self.max_depth.setText(f"Maximum Depth: {max_depth}")
        self.namespaces.setText(f"Namespaces: {len(namespaces)}")
        
        # Extract and display metadata
        metadata = root.find(".//{http://www.idpf.org/2007/opf}metadata") or root.find(".//metadata")
        if metadata is not None:
            dc_ns = "{http://purl.org/dc/elements/1.1/}"
            # Common DC metadata fields to look for
            dc_fields = ["title", "creator", "publisher", "date", "identifier", "language", "subject"]
            
            for field in dc_fields:
                # Try with DC namespace first
                elements = metadata.findall(f".//{dc_ns}{field}") or metadata.findall(f".//{field}")
                if elements:
                    for elem in elements:
                        item = QTreeWidgetItem(self.metadata_tree)
                        # Remove namespace from tag if present
                        tag = elem.tag
                        if "}" in tag:
                            tag = tag.split("}")[1]
                        item.setText(0, tag)
                        item.setText(1, elem.text if elem.text else "")
                        
                        # Apply color only to the value column if it's a known metadata field
                        if tag.lower() in self.metadata_colors:
                            color = QColor(self.metadata_colors[tag.lower()])
                            item.setForeground(1, color)  # Only color the value column
                        
                        # Add any attributes as child items
                        if elem.attrib:
                            for key, value in elem.attrib.items():
                                attr_item = QTreeWidgetItem(item)
                                if key.startswith("{"):
                                    # Handle namespaced attributes
                                    _, key = key[1:].split("}")
                                attr_item.setText(0, f"@{key}")
                                attr_item.setText(1, value)
                        
                        # Expand metadata items by default
                        item.setExpanded(True)
        
        # Find referenced files
        manifest = root.find(".//{http://www.idpf.org/2007/opf}manifest") or root.find(".//manifest")
        if manifest is not None:
            media_map = {}  # Group files by media-type
            for item in manifest.findall(".//item"):
                media_type = item.get("media-type", "unknown")
                href = item.get("href", "")
                if media_type not in media_map:
                    media_map[media_type] = []
                media_map[media_type].append(href)
            
            # Add to tree grouped by type
            for media_type, hrefs in media_map.items():
                type_item = QTreeWidgetItem(self.files_list)
                type_item.setText(0, media_type)
                type_item.setText(1, f"{len(hrefs)} file(s)")
                
                for href in sorted(hrefs):
                    file_item = QTreeWidgetItem(type_item)
                    file_item.setText(1, href)
        
        # Expand both trees
        self.metadata_tree.expandAll()
        self.files_list.expandAll()