# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

"""
OPF Comparison Dialog - Shows current OPF vs. standards-corrected version
"""

import re
import difflib
from xml.etree import ElementTree as ET
from xml.dom import minidom

try:
    # Calibre 8+ uses qt.core namespace
    from qt.core import (Qt, QTimer, QUrl, QRect, QPoint, QRegularExpression,
                        QDialog, QVBoxLayout, QHBoxLayout, QTextEdit,
                        QPushButton, QLabel, QSplitter, QGroupBox,
                        QDialogButtonBox, QFrame, QScrollArea, QLineEdit, QWidget, QComboBox,
                        QMenu, QAction, QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
                        QTextCursor, QTextDocument, QPainter, QPen, QBrush, QIcon)
except ImportError:
    try:
        # Calibre 6 uses calibre.gui2.qt namespace
        from calibre.gui2.qt.core import (Qt, QTimer, QUrl, QRect, QPoint, QRegularExpression)
        from calibre.gui2.qt.widgets import (QDialog, QVBoxLayout, QHBoxLayout, QTextEdit,
                                           QPushButton, QLabel, QSplitter, QGroupBox,
                                           QDialogButtonBox, QFrame, QScrollArea, QLineEdit, QWidget, QComboBox,
                                           QMenu, QAction)
        from calibre.gui2.qt.gui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
                                       QTextCursor, QTextDocument, QPainter, QPen, QBrush, QIcon)
    except ImportError:
        # Fall back to PyQt5 if running outside Calibre
        from PyQt5.QtWidgets import (QDialog, QVBoxLayout, QHBoxLayout, QTextEdit,
                                   QPushButton, QLabel, QSplitter, QGroupBox,
                                   QDialogButtonBox, QFrame, QScrollArea, QLineEdit, QWidget, QComboBox,
                                   QMenu, QAction)
        from PyQt5.QtGui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
                               QTextCursor, QTextDocument, QIcon, QPainter, QPen, QBrush)
        from PyQt5.QtCore import Qt, QTimer, QUrl, QRect, QPoint, QRegularExpression

from calibre.constants import DEBUG
from calibre.utils.logging import default_log as debug_print
try:
    from calibre.utils.localization import _
except Exception:
    try:
        _ = __builtins__['_']
    except Exception:
        _ = lambda x: x
from calibre.gui2 import gprefs


def patience_get_opcodes(a, b):
    """Return difflib-style opcodes using the patience diff strategy.

    This is a pure-Python implementation meant to produce more targeted blocks
    on texts with repeated similar lines (common in XML/OPF).
    """

    def unique_positions(seq):
        counts = {}
        pos = {}
        for i, s in enumerate(seq):
            counts[s] = counts.get(s, 0) + 1
            if counts[s] == 1:
                pos[s] = i
            else:
                # no longer unique
                pos.pop(s, None)
        return pos

    def lis_by_second(pairs):
        # pairs: list[(i, j)] sorted by i
        import bisect
        tails = []          # list of j values
        tails_idx = []      # index in pairs for each tail
        prev = [-1] * len(pairs)
        for idx, (_i, j) in enumerate(pairs):
            k = bisect.bisect_left(tails, j)
            if k == len(tails):
                tails.append(j)
                tails_idx.append(idx)
            else:
                tails[k] = j
                tails_idx[k] = idx
            prev[idx] = tails_idx[k - 1] if k > 0 else -1
        if not tails_idx:
            return []
        # reconstruct
        res = []
        idx = tails_idx[-1]
        while idx != -1:
            res.append(pairs[idx])
            idx = prev[idx]
        res.reverse()
        return res

    def recurse(alo, ahi, blo, bhi):
        # Base cases
        if alo >= ahi and blo >= bhi:
            return []
        if alo >= ahi:
            return [('insert', alo, alo, blo, bhi)]
        if blo >= bhi:
            return [('delete', alo, ahi, blo, blo)]

        a_slice = a[alo:ahi]
        b_slice = b[blo:bhi]

        a_uni = unique_positions(a_slice)
        b_uni = unique_positions(b_slice)
        common = []
        for s, ai in a_uni.items():
            bj = b_uni.get(s)
            if bj is not None:
                common.append((alo + ai, blo + bj))
        common.sort(key=lambda x: x[0])

        anchors = lis_by_second(common)
        if not anchors:
            sm = difflib.SequenceMatcher(None, a_slice, b_slice, autojunk=False)
            out = []
            for tag, i1, i2, j1, j2 in sm.get_opcodes():
                out.append((tag, alo + i1, alo + i2, blo + j1, blo + j2))
            return out

        out = []
        ai = alo
        bj = blo
        for (a_idx, b_idx) in anchors:
            out.extend(recurse(ai, a_idx, bj, b_idx))
            out.append(('equal', a_idx, a_idx + 1, b_idx, b_idx + 1))
            ai = a_idx + 1
            bj = b_idx + 1
        out.extend(recurse(ai, ahi, bj, bhi))
        return out

    # Merge adjacent opcodes of same tag (keeps output tidy)
    raw = recurse(0, len(a), 0, len(b))
    merged = []
    for op in raw:
        if not merged:
            merged.append(op)
            continue
        tag, i1, i2, j1, j2 = op
        ptag, pi1, pi2, pj1, pj2 = merged[-1]
        if tag == ptag and i1 == pi2 and j1 == pj2:
            merged[-1] = (ptag, pi1, i2, pj1, j2)
        else:
            merged.append(op)
    return merged


def histogram_get_opcodes(a, b):
    """Return difflib-style opcodes using a lightweight histogram-like strategy.

    This is *not* Git's xdiff histogram implementation, but a pure-Python approximation
    intended to be more targeted than the default SequenceMatcher on XML/OPF where some
    lines repeat.
    """

    def lis_by_second(pairs):
        # pairs: list[(i, j)] sorted by i
        import bisect
        tails = []
        tails_idx = []
        prev = [-1] * len(pairs)
        for idx, (_i, j) in enumerate(pairs):
            k = bisect.bisect_left(tails, j)
            if k == len(tails):
                tails.append(j)
                tails_idx.append(idx)
            else:
                tails[k] = j
                tails_idx[k] = idx
            prev[idx] = tails_idx[k - 1] if k > 0 else -1
        if not tails_idx:
            return []
        res = []
        idx = tails_idx[-1]
        while idx != -1:
            res.append(pairs[idx])
            idx = prev[idx]
        res.reverse()
        return res

    def positions_map(seq):
        d = {}
        for i, s in enumerate(seq):
            d.setdefault(s, []).append(i)
        return d

    def anchors_for(a_slice, b_slice, alo, blo):
        pos_a = positions_map(a_slice)
        pos_b = positions_map(b_slice)
        pairs = []

        # Anchor using "rare" lines; uniqueness relaxed to allow (1,2) or (2,1)
        for s, a_pos in pos_a.items():
            b_pos = pos_b.get(s)
            if not b_pos:
                continue
            la = len(a_pos)
            lb = len(b_pos)
            if la == 1 and lb == 1:
                pairs.append((alo + a_pos[0], blo + b_pos[0]))
            elif la == 1 and 1 < lb <= 2:
                for j in b_pos:
                    pairs.append((alo + a_pos[0], blo + j))
            elif lb == 1 and 1 < la <= 2:
                for i in a_pos:
                    pairs.append((alo + i, blo + b_pos[0]))

        pairs.sort(key=lambda x: x[0])
        return lis_by_second(pairs)

    def recurse(alo, ahi, blo, bhi):
        if alo >= ahi and blo >= bhi:
            return []
        if alo >= ahi:
            return [('insert', alo, alo, blo, bhi)]
        if blo >= bhi:
            return [('delete', alo, ahi, blo, blo)]

        a_slice = a[alo:ahi]
        b_slice = b[blo:bhi]

        anchors = anchors_for(a_slice, b_slice, alo, blo)
        if not anchors:
            sm = difflib.SequenceMatcher(None, a_slice, b_slice, autojunk=False)
            out = []
            for tag, i1, i2, j1, j2 in sm.get_opcodes():
                out.append((tag, alo + i1, alo + i2, blo + j1, blo + j2))
            return out

        out = []
        ai = alo
        bj = blo
        for (a_idx, b_idx) in anchors:
            out.extend(recurse(ai, a_idx, bj, b_idx))
            out.append(('equal', a_idx, a_idx + 1, b_idx, b_idx + 1))
            ai = a_idx + 1
            bj = b_idx + 1
        out.extend(recurse(ai, ahi, bj, bhi))
        return out

    raw = recurse(0, len(a), 0, len(b))
    merged = []
    for op in raw:
        if not merged:
            merged.append(op)
            continue
        tag, i1, i2, j1, j2 = op
        ptag, pi1, pi2, pj1, pj2 = merged[-1]
        if tag == ptag and i1 == pi2 and j1 == pj2:
            merged[-1] = (ptag, pi1, i2, pj1, j2)
        else:
            merged.append(op)
    return merged

class OPFComparisonDialog(QDialog):
    """Dialog for comparing current OPF with standards-corrected version"""

    def __init__(self, parent, current_opf_content, book_title="Unknown Book"):
        super().__init__(parent)
        self.current_opf = current_opf_content or ''
        self.book_title = book_title
        # Use schema-based validation/correction for the standards version
        corrected = self.generate_schema_corrected_opf(self.current_opf)
        if not corrected:
            corrected = self.current_opf
        self.corrected_opf = corrected

        self.setWindowTitle(f"OPF Standards Comparison (Experimental) - {book_title}")
        self.setMinimumWidth(1200)
        self.setMinimumHeight(700)

        self.setup_ui()
        self.generate_comparison()

        # Restore geometry
        geom = gprefs.get('opf_comparison_dialog_geometry', None)
        if geom:
            self.restoreGeometry(geom)
        else:
            self.resize(1200, 700)
    def generate_schema_corrected_opf(self, opf_content):
        """Generate a standards-corrected version of the OPF content using schema-based logic."""
        try:
            from calibre_plugins.opf_helper.schema_utils import load_schema
            from lxml import etree
            # Parse the XML
            parser = etree.XMLParser(recover=True, remove_blank_text=True)
            doc = etree.fromstring(opf_content.encode('utf-8'), parser)
            version = doc.get('version', '2.0')
            schema = load_schema(version)
            # If schema is available, validate and auto-correct order if possible
            if schema is not None:
                try:
                    schema.assertValid(doc)
                    # If valid, pretty-print and return
                    corrected_xml = etree.tostring(doc, pretty_print=True, encoding='unicode')
                    return corrected_xml
                except etree.DocumentInvalid as e:
                    # If not valid, attempt to reorder elements using the same logic as validator
                    # (For now, just pretty-print the original with errors noted)
                    corrected_xml = etree.tostring(doc, pretty_print=True, encoding='unicode')
                    # Optionally, append schema errors as comments
                    error_msgs = [str(err.message) for err in e.error_log]
                    if error_msgs:
                        corrected_xml += '\n<!-- Schema validation errors:\n' + '\n'.join(error_msgs) + '\n-->'
                    return corrected_xml
            else:
                # Fallback: pretty-print original
                corrected_xml = etree.tostring(doc, pretty_print=True, encoding='unicode')
                return corrected_xml
        except Exception as e:
            debug_print(f"OPFHelper: Error generating schema-corrected OPF: {str(e)}")
            return opf_content  # Return original if correction fails

        self.setWindowTitle(f"OPF Standards Comparison (Experimental) - {book_title}")
        self.setMinimumWidth(1200)
        self.setMinimumHeight(700)

        self.setup_ui()
        self.generate_comparison()

        # Restore geometry
        geom = gprefs.get('opf_comparison_dialog_geometry', None)
        if geom:
            self.restoreGeometry(geom)
        else:
            self.resize(1200, 700)

    def setup_ui(self):
        """Set up the user interface"""
        layout = QVBoxLayout(self)
        layout.setSpacing(6)
        layout.setContentsMargins(10, 10, 10, 10)

        # Guard against search signals firing during UI construction
        self._search_ready = False
        self._search_signals_wired = False
        self._search_line_edit = None

        # Apply scrollbar styling consistent with the main OPF Helper dialog
        self.setStyleSheet(self._scrollbar_stylesheet())

        # Header with instructions only
        desc_label = QLabel("This shows your current OPF file alongside a suggested version "
                           "that follows OPF standards more closely. This is for reference only "
                           "and does not modify your original file.")
        desc_label.setWordWrap(True)
        desc_label.setToolTip("The suggested OPF reorganizes elements according to OPF/Dublin Core standards.\n"
                              "Green highlights indicate additions in the suggested version.\n"
                              "Red highlights indicate content from your original that differs.")
        layout.addWidget(desc_label)

        # Diff algorithm selector
        algo_row = QHBoxLayout()
        algo_row.setSpacing(8)
        algo_label = QLabel('Diff algorithm:')
        algo_row.addWidget(algo_label)

        self.diff_algo_combo = QComboBox()
        # Values are stored in itemData for stable persistence
        self.diff_algo_combo.addItem('Default', 'default')
        self.diff_algo_combo.addItem('Patience', 'patience')
        self.diff_algo_combo.addItem('Histogram', 'histogram')
        self.diff_algo_combo.setToolTip('Patience/Histogram often produce more targeted blocks on repeated lines')
        algo_row.addWidget(self.diff_algo_combo)
        algo_row.addStretch()
        layout.addLayout(algo_row)

        # Restore selection from prefs (plugin prefs, not gprefs)
        try:
            from calibre_plugins.opf_helper.config import prefs
            saved = prefs.get('comparison_diff_algorithm', 'default')
            for i in range(self.diff_algo_combo.count()):
                try:
                    data = self.diff_algo_combo.itemData(i)
                except Exception:
                    data = None
                try:
                    text = (self.diff_algo_combo.itemText(i) or '').strip().lower()
                except Exception:
                    text = ''
                if (data == saved) or (saved in text):
                    self.diff_algo_combo.setCurrentIndex(i)
                    break
        except Exception:
            pass

        def _on_algo_changed(_idx):
            try:
                from calibre_plugins.opf_helper.config import prefs
                prefs['comparison_diff_algorithm'] = self.get_selected_diff_algorithm()
            except Exception:
                pass
            # Defer the expensive re-render so the combobox updates immediately
            try:
                QTimer.singleShot(0, self._refresh_after_diff_algo_change)
            except Exception:
                self._refresh_after_diff_algo_change()

        self.diff_algo_combo.currentIndexChanged.connect(_on_algo_changed)

        # Search row (Find) for comparison panes
        search_row = QHBoxLayout()
        search_row.setSpacing(6)
        search_label = QLabel('Find:')
        search_row.addWidget(search_label)

        # Use QComboBox for search history (like main OPF Helper)
        self.search_box = QComboBox()
        self.search_box.setEditable(True)
        self.search_box.setInsertPolicy(QComboBox.InsertAtTop)
        self.search_box.setMaxVisibleItems(12)
        try:
            le = self.search_box.lineEdit()
            if le is not None:
                le.setPlaceholderText('Find in comparison panes…')
                le.setClearButtonEnabled(True)
                # Wire signals after current_text/corrected_text exist
                self._search_line_edit = le
        except Exception:
            pass
        self._init_search_history_dropdown()
        search_row.addWidget(self.search_box, 1)

        # Search options
        # Restore from prefs so the user doesn't need to re-select Regex every time
        try:
            from calibre_plugins.opf_helper.config import prefs
            _cmp_case = bool(prefs.get('comparison_search_case_sensitive', False))
            _cmp_whole = bool(prefs.get('comparison_search_whole_words', False))
            _cmp_regex = bool(prefs.get('comparison_search_regex', False))
        except Exception:
            _cmp_case = False
            _cmp_whole = False
            _cmp_regex = False

        self.case_sensitive = QAction('Case sensitive', self)
        self.case_sensitive.setCheckable(True)
        self.case_sensitive.setChecked(_cmp_case)
        self.case_sensitive.triggered.connect(self._on_search_option_toggled)

        self.whole_words = QAction('Whole words', self)
        self.whole_words.setCheckable(True)
        self.whole_words.setChecked(_cmp_whole)
        self.whole_words.triggered.connect(self._on_search_option_toggled)

        self.regex_search = QAction('Regex search', self)
        self.regex_search.setCheckable(True)
        self.regex_search.setChecked(_cmp_regex)
        self.regex_search.triggered.connect(self._on_search_option_toggled)

        options_button = QPushButton()
        try:
            options_button.setIcon(QIcon.ic('config.png'))
        except Exception:
            pass
        options_button.setMaximumWidth(30)
        options_button.setToolTip('Search options')
        options_menu = QMenu(self)
        options_menu.addAction(self.case_sensitive)
        options_menu.addAction(self.whole_words)
        options_menu.addAction(self.regex_search)
        options_button.setMenu(options_menu)
        search_row.addWidget(options_button)

        # Visible mode indicator so the user doesn't need to open the options menu
        self.search_mode_label = QLabel('')
        self.search_mode_label.setAlignment(Qt.AlignCenter)
        self.search_mode_label.setMinimumWidth(70)
        search_row.addWidget(self.search_mode_label)
        try:
            self._update_search_mode_indicator()
        except Exception:
            pass

        self.match_label = QLabel('')
        self.match_label.setAlignment(Qt.AlignCenter)
        self.match_label.setMinimumWidth(80)
        search_row.addWidget(self.match_label)

        self.prev_button = QPushButton()
        try:
            self.prev_button.setIcon(QIcon.ic('arrow-up.png'))
        except Exception:
            pass
        self.prev_button.setMaximumWidth(30)
        self.prev_button.setToolTip('Find previous match')
        self.prev_button.clicked.connect(self.find_previous)
        self.prev_button.setEnabled(False)
        search_row.addWidget(self.prev_button)

        self.next_button = QPushButton()
        try:
            self.next_button.setIcon(QIcon.ic('arrow-down.png'))
        except Exception:
            pass
        self.next_button.setMaximumWidth(30)
        self.next_button.setToolTip('Find next match')
        self.next_button.clicked.connect(self.find_next)
        self.next_button.setEnabled(False)
        search_row.addWidget(self.next_button)

        layout.addLayout(search_row)

        # Main vertical splitter (top compare panes + bottom summary)
        v_splitter = QSplitter(Qt.Vertical)
        v_splitter.setChildrenCollapsible(False)
        layout.addWidget(v_splitter, 1)

        # Side-by-side compare splitter
        splitter = QSplitter(Qt.Horizontal)
        splitter.setChildrenCollapsible(False)
        v_splitter.addWidget(splitter)

        # WinMerge-style location pane (overview/minimap)
        self.location_pane = DiffLocationPane()
        splitter.addWidget(self.location_pane)

        # Left side - Current OPF
        current_group = QGroupBox("Current OPF")
        current_group.setToolTip("Your original OPF file content as stored in the EPUB.")
        current_layout = QVBoxLayout(current_group)

        self.current_text = QTextEdit()
        self.current_text.setReadOnly(True)
        self.current_text.setFont(QFont("Courier New", 10))
        try:
            self.current_text.setLineWrapMode(QTextEdit.WidgetWidth)
        except Exception:
            pass
        current_layout.addWidget(self.current_text)
        splitter.addWidget(current_group)

        # Right side - Suggested OPF
        corrected_group = QGroupBox("Suggested OPF")
        corrected_group.setToolTip(
            "OPF reorganized according to EPUB/Dublin Core standards.\n"
            "This is a suggestion - your original file is not modified."
        )
        corrected_layout = QVBoxLayout(corrected_group)

        self.corrected_text = QTextEdit()
        self.corrected_text.setReadOnly(True)
        self.corrected_text.setFont(QFont("Courier New", 10))
        try:
            self.corrected_text.setLineWrapMode(QTextEdit.WidgetWidth)
        except Exception:
            pass
        corrected_layout.addWidget(self.corrected_text)
        splitter.addWidget(corrected_group)

        # Now that the panes exist, wire search signals safely
        self._wire_search_signals()
        self._search_ready = True

        # Set splitter proportions (location pane + 2 equal panes)
        try:
            splitter.setSizes([24, 588, 588])
        except Exception:
            pass

        # Bottom section with summary
        summary_group = QGroupBox("Summary of Changes")
        summary_layout = QVBoxLayout(summary_group)
        summary_layout.setContentsMargins(4, 2, 4, 2)
        summary_layout.setSpacing(2)

        self.summary_text = QTextEdit()
        self.summary_text.setReadOnly(True)
        self.summary_text.setMinimumHeight(90)
        self.summary_text.setFont(QFont("Courier New", 9))
        try:
            self.summary_text.setLineWrapMode(QTextEdit.WidgetWidth)
        except Exception:
            pass
        self.summary_text.setStyleSheet("QTextEdit { padding: 2px; }")
        summary_layout.addWidget(self.summary_text)
        v_splitter.addWidget(summary_group)

        # Initial vertical split: favor the compare panes
        try:
            v_splitter.setSizes([520, 180])
            v_splitter.setStretchFactor(0, 3)
            v_splitter.setStretchFactor(1, 1)
        except Exception:
            pass

        # Buttons
        button_layout = QHBoxLayout()
        button_layout.setSpacing(6)
        button_layout.setContentsMargins(0, 4, 0, 0)

        copy_current_btn = QPushButton("&Copy Current OPF")
        copy_current_btn.setToolTip("Copy your original OPF content to clipboard")
        copy_current_btn.clicked.connect(lambda: self.copy_to_clipboard(self.current_opf))
        button_layout.addWidget(copy_current_btn)

        copy_corrected_btn = QPushButton("Copy &Suggested OPF")
        copy_corrected_btn.setToolTip("Copy the suggested OPF content to clipboard")
        copy_corrected_btn.clicked.connect(lambda: self.copy_to_clipboard(self.corrected_opf))
        button_layout.addWidget(copy_corrected_btn)

        colors_btn = QPushButton("&Highlight Colors…")
        colors_btn.setToolTip("Configure added/removed/unchanged highlight colors")
        colors_btn.clicked.connect(self.configure_highlight_colors)
        button_layout.addWidget(colors_btn)

        button_layout.addStretch()

        close_btn = QPushButton("C&lose")
        close_btn.clicked.connect(self.reject)
        button_layout.addWidget(close_btn)

        layout.addLayout(button_layout)

        # Apply syntax highlighting (will be overridden by HTML highlighting)
        self.current_highlighter = OPFXMLHighlighter(self.current_text.document())
        self.corrected_highlighter = OPFXMLHighlighter(self.corrected_text.document())

        # Search state
        self.search_positions = []
        self.current_match = -1
        self._search_target = None

    def _wire_search_signals(self):
        if getattr(self, '_search_signals_wired', False):
            return
        self._search_signals_wired = True

        try:
            le = getattr(self, '_search_line_edit', None)
            if le is None:
                le = self.search_box.lineEdit()
        except Exception:
            le = None

        try:
            if le is not None:
                le.textChanged.connect(self.on_search_text_changed)
                le.returnPressed.connect(self.find_next)
                le.editingFinished.connect(self._commit_search_history)
        except Exception:
            pass

        try:
            self.search_box.activated.connect(self._on_search_history_activated)
        except Exception:
            pass

    def _save_search_options(self):
        try:
            from calibre_plugins.opf_helper.config import prefs
            prefs['comparison_search_case_sensitive'] = bool(self.case_sensitive.isChecked())
            prefs['comparison_search_whole_words'] = bool(self.whole_words.isChecked())
            prefs['comparison_search_regex'] = bool(self.regex_search.isChecked())
        except Exception:
            pass

    def _update_search_mode_indicator(self):
        try:
            lbl = getattr(self, 'search_mode_label', None)
            if lbl is None:
                return
            is_rx = bool(self.regex_search.isChecked())
            is_case = bool(self.case_sensitive.isChecked())
            is_whole = bool(self.whole_words.isChecked())
            mode = 'Regex' if is_rx else 'Text'
            flags = []
            if is_case:
                flags.append('Aa')
            if is_whole:
                flags.append('W')
            text = mode if not flags else (mode + ' ' + ' '.join(flags))
            lbl.setText(text)
            lbl.setToolTip('Mode: ' + mode + '\n' +
                           ('Case sensitive: yes' if is_case else 'Case sensitive: no') + '\n' +
                           ('Whole words: yes' if is_whole else 'Whole words: no'))
        except Exception:
            pass

    def _on_search_option_toggled(self):
        try:
            self._save_search_options()
        except Exception:
            pass
        try:
            self._update_search_mode_indicator()
        except Exception:
            pass
        try:
            self.reset_search()
        except Exception:
            pass

    def _get_search_target(self):
        try:
            if hasattr(self, 'current_text') and self.current_text is not None and self.current_text.hasFocus():
                return self.current_text
            if hasattr(self, 'corrected_text') and self.corrected_text is not None and self.corrected_text.hasFocus():
                return self.corrected_text
        except Exception:
            pass
        if hasattr(self, 'current_text') and self.current_text is not None:
            return self.current_text
        if hasattr(self, 'corrected_text') and self.corrected_text is not None:
            return self.corrected_text
        return None

    def _read_search_history(self):
        try:
            from calibre_plugins.opf_helper.config import prefs
            return list(prefs.get('comparison_search_history', []) or [])
        except Exception:
            return []

    def _write_search_history(self, hist):
        try:
            from calibre_plugins.opf_helper.config import prefs
            max_n = int(prefs.get('search_history_max', 12) or 12)
        except Exception:
            max_n = 12
        try:
            from calibre_plugins.opf_helper.config import prefs
            prefs['comparison_search_history'] = list(hist or [])[:max_n]
        except Exception:
            pass

    def _init_search_history_dropdown(self):
        try:
            self._clear_search_history_user_data = '__opf_comparison_clear_search_history__'
            self._clear_search_history_label = _('Clear history')
        except Exception:
            self._clear_search_history_user_data = '__opf_comparison_clear_search_history__'
            self._clear_search_history_label = 'Clear history'
        self._reload_search_history_dropdown()

    def _reload_search_history_dropdown(self):
        try:
            current = self.search_box.currentText()
        except Exception:
            current = ''
        try:
            self.search_box.blockSignals(True)
        except Exception:
            pass
        try:
            self.search_box.clear()
            hist = self._read_search_history()
            for entry in hist:
                self.search_box.addItem(entry)
            if self.search_box.count() > 0:
                try:
                    self.search_box.insertSeparator(self.search_box.count())
                except Exception:
                    pass
            self.search_box.addItem(self._clear_search_history_label, self._clear_search_history_user_data)
            try:
                m = self.search_box.model()
                last_idx = self.search_box.count() - 1
                itm = None
                try:
                    itm = m.item(last_idx)
                except Exception:
                    itm = None
                if itm is not None:
                    f = itm.font()
                    f.setItalic(True)
                    itm.setFont(f)
            except Exception:
                pass
            try:
                self.search_box.setEditText(current)
            except Exception:
                pass
        finally:
            try:
                self.search_box.blockSignals(False)
            except Exception:
                pass

    def _commit_search_history(self):
        try:
            term = (self.search_box.currentText() or '').strip()
            if not term:
                return
            hist = self._read_search_history()
            hist = [h for h in hist if h != term]
            hist.insert(0, term)
            self._write_search_history(hist)
            self._reload_search_history_dropdown()
        except Exception:
            pass

    def _on_search_history_activated(self, index):
        try:
            ud = self.search_box.itemData(index)
            if ud == getattr(self, '_clear_search_history_user_data', None):
                self._write_search_history([])
                self._reload_search_history_dropdown()
                try:
                    self.search_box.setEditText('')
                except Exception:
                    pass
                self.on_search_text_changed('')
                return
        except Exception:
            pass
        try:
            self.on_search_text_changed(self.search_box.currentText())
        except Exception:
            pass

    def find_matches(self, search_text):
        """Find occurrences in the active comparison pane using QTextDocument.find."""
        if not search_text:
            return []
        target = self._get_search_target()
        if target is None:
            return []
        self._search_target = target
        doc = target.document()

        # Regex search
        if self.regex_search.isChecked():
            try:
                rx = QRegularExpression(search_text)
                flags = QRegularExpression.PatternOption.NoPatternOption
                if not self.case_sensitive.isChecked():
                    flags |= QRegularExpression.PatternOption.CaseInsensitiveOption
                rx.setPatternOptions(flags)
                if not rx.isValid():
                    return []
                positions = []
                cursor = QTextCursor(doc)
                cursor.movePosition(QTextCursor.MoveOperation.Start)
                cursor = doc.find(rx, cursor)
                while cursor and not cursor.isNull():
                    start = cursor.selectionStart()
                    length = cursor.selectionEnd() - start
                    if length > 0:
                        positions.append((start, length))
                    cursor = doc.find(rx, cursor)
                return positions
            except Exception:
                return []

        # Plain text search
        positions = []
        flags = QTextDocument.FindFlag(0)
        if self.case_sensitive.isChecked():
            flags |= QTextDocument.FindFlag.FindCaseSensitively
        if self.whole_words.isChecked():
            flags |= QTextDocument.FindFlag.FindWholeWords
        cursor = QTextCursor(doc)
        cursor.movePosition(QTextCursor.MoveOperation.Start)
        cursor = doc.find(search_text, cursor, flags)
        while cursor and not cursor.isNull():
            positions.append((cursor.selectionStart(), cursor.selectionEnd() - cursor.selectionStart()))
            cursor = doc.find(search_text, cursor, flags)
        return positions

    def on_search_text_changed(self, *_args):
        if not getattr(self, '_search_ready', False):
            return
        try:
            search_text = self.search_box.currentText()
        except Exception:
            search_text = ''
        if not search_text:
            self.search_positions = []
            self.current_match = -1
            try:
                self.prev_button.setEnabled(False)
                self.next_button.setEnabled(False)
            except Exception:
                pass
            try:
                self.match_label.setText('')
            except Exception:
                pass
            return

        self._current_search = search_text
        self.search_positions = self.find_matches(search_text)
        self.current_match = -1
        has_matches = len(self.search_positions) > 0
        try:
            self.prev_button.setEnabled(has_matches)
            self.next_button.setEnabled(has_matches)
        except Exception:
            pass
        if has_matches:
            try:
                self.match_label.setText(f"1 of {len(self.search_positions)}")
            except Exception:
                pass
            self.find_next()
        else:
            try:
                self.match_label.setText('No matches')
            except Exception:
                pass

    def _highlight_match(self, position_info):
        try:
            if self._search_target is None:
                self._search_target = self._get_search_target()
            target = self._search_target
            position, length = position_info
            cursor = QTextCursor(target.document())
            cursor.setPosition(position)
            cursor.movePosition(QTextCursor.MoveOperation.NextCharacter, QTextCursor.MoveMode.KeepAnchor, length)
            target.setTextCursor(cursor)
            target.ensureCursorVisible()
            return True
        except Exception:
            return False

    def find_next(self):
        if not self.search_positions:
            return
        self.current_match = (self.current_match + 1) % len(self.search_positions)
        self._highlight_match(self.search_positions[self.current_match])
        try:
            self.match_label.setText(f"{self.current_match + 1} of {len(self.search_positions)}")
        except Exception:
            pass

    def find_previous(self):
        if not self.search_positions:
            return
        self.current_match = (self.current_match - 1) % len(self.search_positions)
        self._highlight_match(self.search_positions[self.current_match])
        try:
            self.match_label.setText(f"{self.current_match + 1} of {len(self.search_positions)}")
        except Exception:
            pass

    def reset_search(self):
        try:
            txt = self.search_box.currentText()
        except Exception:
            txt = ''
        if txt:
            self.on_search_text_changed()

    def _refresh_after_diff_algo_change(self):
        # Force a full re-render so changes are visible immediately.
        # Preserve scroll positions so the view doesn't jump.
        try:
            lsb = self.current_text.verticalScrollBar()
            rsb = self.corrected_text.verticalScrollBar()
            lpos = lsb.value()
            rpos = rsb.value()
        except Exception:
            lsb = rsb = None
            lpos = rpos = None

        try:
            self.generate_comparison()
        except Exception:
            # Fall back to the cheaper path
            try:
                self.highlight_differences()
            finally:
                try:
                    self.generate_change_summary()
                except Exception:
                    pass

        try:
            if lsb is not None and lpos is not None:
                lsb.setValue(lpos)
            if rsb is not None and rpos is not None:
                rsb.setValue(rpos)
        except Exception:
            pass

    def _scrollbar_stylesheet(self):
        # Matches the styling used in the main OPF Helper dialog (ShowOPFPlugin tab widget).
        return (
            "QScrollBar::handle {"
            "   border: 1px solid #5B6985;"
            "}"
            "QScrollBar:vertical {"
            "   background: transparent;"
            "   width: 12px;"
            "   margin: 0px 0px 0px 0px;"
            "   padding: 6px 0px 6px 0px;"
            "}"
            "QScrollBar::handle:vertical {"
            "   background: rgba(140, 172, 204, 0.25);"
            "   min-height: 22px;"
            "   border-radius: 4px;"
            "   margin: 4px 0px;"
            "}"
            "QScrollBar::handle:vertical:hover {"
            "   background: rgba(140, 172, 204, 0.45);"
            "}"
            "QScrollBar:horizontal {"
            "   background: transparent;"
            "   height: 12px;"
            "   margin: 0px 0px 0px 0px;"
            "   padding: 0px 6px 0px 6px;"
            "}"
            "QScrollBar::handle:horizontal {"
            "   background: rgba(140, 172, 204, 0.25);"
            "   min-width: 22px;"
            "   border-radius: 4px;"
            "   margin: 0px 4px;"
            "}"
            "QScrollBar::handle:horizontal:hover {"
            "   background: rgba(140, 172, 204, 0.45);"
            "}"
            "QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,"
            "QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {"
            "   background: none;"
            "}"
        )

    def configure_highlight_colors(self):
        d = HighlightColorsDialog(self)
        exec_fn = getattr(d, 'exec', None) or getattr(d, 'exec_', None)
        if exec_fn is not None and exec_fn() == QDialog.Accepted:
            # Re-render highlights using updated colors
            self.highlight_differences()

    def generate_corrected_opf(self, opf_content):
        """Generate a standards-corrected version of the OPF content"""
        try:
            # Parse the XML
            root = ET.fromstring(opf_content)

            # Apply standard corrections
            self.apply_opf_corrections(root)

            # Convert back to string with proper formatting
            rough_string = ET.tostring(root, encoding='unicode')

            # Pretty print the XML
            reparsed = minidom.parseString(rough_string)
            corrected_xml = reparsed.toprettyxml(indent="  ", encoding=None)

            # Clean up the pretty-printed XML (remove extra newlines)
            lines = corrected_xml.split('\n')
            cleaned_lines = []
            prev_empty = False

            for line in lines:
                is_empty = line.strip() == ''
                if not (is_empty and prev_empty):
                    cleaned_lines.append(line)
                prev_empty = is_empty

            return '\n'.join(cleaned_lines)

        except Exception as e:
            debug_print(f"OPFHelper: Error generating corrected OPF: {str(e)}")
            return opf_content  # Return original if correction fails

    def apply_opf_corrections(self, root):
        """Apply OPF standard corrections to the XML tree"""
        # Get namespace
        ns = {'opf': 'http://www.idpf.org/2007/opf',
              'dc': 'http://purl.org/dc/elements/1.1/'}

        # 1. Ensure proper element ordering in package
        self.reorder_package_elements(root, ns)

        # 2. Ensure proper metadata ordering
        metadata = root.find('.//{http://www.idpf.org/2007/opf}metadata')
        if metadata is not None:
            self.reorder_metadata_elements(metadata, ns)

        # 3. Ensure proper manifest ordering
        manifest = root.find('.//{http://www.idpf.org/2007/opf}manifest')
        if manifest is not None:
            self.reorder_manifest_items(manifest, ns)

        # 4. Ensure proper spine ordering
        spine = root.find('.//{http://www.idpf.org/2007/opf}spine')
        if spine is not None:
            self.reorder_spine_items(spine, ns)

        # 5. Add missing required attributes
        self.add_required_attributes(root, ns)

        # 6. Normalize whitespace and formatting
        self.normalize_formatting(root)

    def reorder_package_elements(self, root, ns):
        """Reorder top-level elements in package according to OPF standard"""
        # Standard order: metadata, manifest, spine, guide (optional), tours (optional)
        desired_order = ['metadata', 'manifest', 'spine', 'guide', 'tours']

        # Remove all children temporarily
        children = []
        for child in list(root):
            root.remove(child)
            children.append((child.tag.split('}')[-1], child))  # Store tag name without namespace

        # Re-add in correct order
        for desired_tag in desired_order:
            for tag_name, child in children:
                if tag_name == desired_tag:
                    root.append(child)
                    children.remove((tag_name, child))
                    break

        # Add any remaining elements at the end
        for _, child in children:
            root.append(child)

    def reorder_metadata_elements(self, metadata, ns):
        """Reorder metadata elements according to Dublin Core and OPF standards"""
        # Dublin Core elements first, then OPF-specific metadata
        dc_order = ['title', 'creator', 'subject', 'description', 'publisher',
                   'contributor', 'date', 'type', 'format', 'identifier',
                   'source', 'language', 'relation', 'coverage', 'rights']

        # Remove all children temporarily
        children = []
        for child in list(metadata):
            tag_name = child.tag.split('}')[-1]  # Remove namespace
            metadata.remove(child)
            children.append((tag_name, child))

        # Re-add Dublin Core elements first in standard order
        for dc_tag in dc_order:
            for tag_name, child in children[:]:  # Copy list to avoid modification issues
                if tag_name == dc_tag:
                    metadata.append(child)
                    children.remove((tag_name, child))
                    break

        # Add OPF-specific metadata elements
        opf_specific = ['meta']
        for opf_tag in opf_specific:
            for tag_name, child in children[:]:
                if tag_name == opf_tag:
                    metadata.append(child)
                    children.remove((tag_name, child))

        # Add any remaining elements
        for _, child in children:
            metadata.append(child)

    def reorder_manifest_items(self, manifest, ns):
        """Reorder manifest items alphabetically by href"""
        if len(manifest) <= 1:
            return

        # Get all item elements
        items = []
        for child in list(manifest):
            if child.tag.endswith('item'):
                manifest.remove(child)
                items.append(child)

        # Sort by href attribute
        items.sort(key=lambda x: x.get('href', '').lower())

        # Re-add in sorted order
        for item in items:
            manifest.append(item)

    def reorder_spine_items(self, spine, ns):
        """Ensure spine itemsref elements are in proper order"""
        # For now, just ensure they're grouped together
        # More complex ordering would require reading the manifest
        pass

    def add_required_attributes(self, root, ns):
        """Add any missing required attributes"""
        # Ensure package has version attribute
        if 'version' not in root.attrib:
            root.set('version', '2.0')  # Default to 2.0 if not specified

        # Ensure package has unique-identifier attribute
        if 'unique-identifier' not in root.attrib:
            # Try to find an identifier in metadata
            metadata = root.find('.//{http://www.idpf.org/2007/opf}metadata')
            if metadata is not None:
                identifiers = metadata.findall('.//{http://purl.org/dc/elements/1.1/}identifier')
                if identifiers:
                    # Use the id of the first identifier
                    ident_id = identifiers[0].get('{http://www.idpf.org/2007/opf}id')
                    if ident_id:
                        root.set('unique-identifier', ident_id)

    def normalize_formatting(self, root):
        """Normalize whitespace and formatting"""
        # Remove excessive whitespace from text content
        for elem in root.iter():
            if elem.text:
                elem.text = elem.text.strip()
            if elem.tail:
                elem.tail = elem.tail.strip()

    def generate_comparison(self):
        """Generate the comparison view with highlighting"""
        # Set the text content
        self.current_text.setPlainText(self.current_opf)
        self.corrected_text.setPlainText(self.corrected_opf)

        # Apply difference highlighting
        self.highlight_differences()

        # Generate summary of changes
        self.generate_change_summary()

    def get_selected_diff_algorithm(self):
        # Be defensive: calibre's Qt wrappers vary (Qt5/Qt6, shim APIs).
        try:
            idx = self.diff_algo_combo.currentIndex()
        except Exception:
            idx = -1

        # Prefer itemData at the current index
        v = None
        try:
            if idx is not None and idx >= 0:
                v = self.diff_algo_combo.itemData(idx)
        except Exception:
            pass

        # Fall back to currentData (some builds only expose this)
        if not v:
            try:
                v = self.diff_algo_combo.currentData()
            except Exception:
                v = None

        # Last resort: infer from visible text
        if not v:
            try:
                txt = (self.diff_algo_combo.currentText() or '').strip().lower()
                if 'patience' in txt:
                    v = 'patience'
                elif 'histogram' in txt:
                    v = 'histogram'
                elif 'default' in txt:
                    v = 'default'
            except Exception:
                v = None

        return v or 'default'

    def highlight_differences(self):
        """Highlight differences between current and corrected OPF"""
        current_lines = self.current_opf.split('\n')
        corrected_lines = self.corrected_opf.split('\n')

        # Detach syntax highlighters; we apply our own block-level formatting.
        try:
            if getattr(self, 'current_highlighter', None) is not None:
                self.current_highlighter.setDocument(None)
            if getattr(self, 'corrected_highlighter', None) is not None:
                self.corrected_highlighter.setDocument(None)
        except Exception:
            pass

        # Build aligned, line-by-line views using opcodes
        algo = self.get_selected_diff_algorithm()
        if algo == 'patience':
            opcodes = patience_get_opcodes(current_lines, corrected_lines)
        elif algo == 'histogram':
            opcodes = histogram_get_opcodes(current_lines, corrected_lines)
        else:
            opcodes = difflib.SequenceMatcher(None, current_lines, corrected_lines, autojunk=False).get_opcodes()

        left_lines, right_lines = [], []
        left_kinds, right_kinds = [], []
        left_peers, right_peers = [], []
        for tag, i1, i2, j1, j2 in opcodes:
            if tag == 'equal':
                for k in range(i2 - i1):
                    left_lines.append(current_lines[i1 + k])
                    right_lines.append(corrected_lines[j1 + k])
                    left_kinds.append('unchanged')
                    right_kinds.append('unchanged')
                    left_peers.append(None)
                    right_peers.append(None)
            elif tag == 'delete':
                for k in range(i2 - i1):
                    left_lines.append(current_lines[i1 + k])
                    right_lines.append('')
                    left_kinds.append('removed')
                    right_kinds.append('blank')
                    left_peers.append(None)
                    right_peers.append(None)
            elif tag == 'insert':
                for k in range(j2 - j1):
                    left_lines.append('')
                    right_lines.append(corrected_lines[j1 + k])
                    left_kinds.append('blank')
                    right_kinds.append('added')
                    left_peers.append(None)
                    right_peers.append(None)
            elif tag == 'replace':
                n = max(i2 - i1, j2 - j1)
                for k in range(n):
                    l = current_lines[i1 + k] if (i1 + k) < i2 else ''
                    r = corrected_lines[j1 + k] if (j1 + k) < j2 else ''
                    left_lines.append(l)
                    right_lines.append(r)
                    left_kinds.append('removed' if l else 'blank')
                    right_kinds.append('added' if r else 'blank')
                    # Only do targeted intra-line highlighting when both sides have content
                    if l and r:
                        left_peers.append(r)
                        right_peers.append(l)
                    else:
                        left_peers.append(None)
                        right_peers.append(None)

        colors = self._get_diff_colors()
        self._set_text_with_diff_formatting(self.current_text, left_lines, left_kinds, colors, side='left', peer_lines=left_peers)
        self._set_text_with_diff_formatting(self.corrected_text, right_lines, right_kinds, colors, side='right', peer_lines=right_peers)

        # Update location pane overview
        try:
            self.location_pane.set_data(left_kinds, right_kinds)
            self.location_pane.attach(self.current_text, self.corrected_text)
        except Exception:
            pass

        # Keep scrollbars in sync if the user scrolls after render
        self._ensure_scroll_sync_connections()

    def _get_diff_colors(self):
        from calibre_plugins.opf_helper.config import prefs
        try:
            from calibre.gui2 import is_dark_theme
            dark_mode = is_dark_theme()
        except Exception:
            dark_mode = False

        def qc(val, fallback):
            v = prefs.get(val, fallback)
            try:
                c = QColor(v)
                if c.isValid():
                    return c, v
            except Exception:
                pass
            c = QColor(fallback)
            return c, fallback

        if dark_mode:
            added_bg, _ = qc('comparison_dark_added_bg', '#005A00')
            added_text, added_text_raw = qc('comparison_dark_added_text', '#FFFFFF')
            removed_bg, _ = qc('comparison_dark_removed_bg', '#5A0000')
            removed_text, removed_text_raw = qc('comparison_dark_removed_text', '#FFFFFF')
            unchanged_text, _ = qc('comparison_dark_unchanged_text', '#D0D0D0')
        else:
            added_bg, _ = qc('comparison_light_added_bg', '#D4EDDA')
            added_text, added_text_raw = qc('comparison_light_added_text', '#155724')
            removed_bg, _ = qc('comparison_light_removed_bg', '#F8D7DA')
            removed_text, removed_text_raw = qc('comparison_light_removed_text', '#721C24')
            unchanged_text, _ = qc('comparison_light_unchanged_text', '#333333')

        return {
            'added_bg': added_bg,
            'added_text': added_text,
            'added_text_raw': added_text_raw,
            'removed_bg': removed_bg,
            'removed_text': removed_text,
            'removed_text_raw': removed_text_raw,
            'unchanged_text': unchanged_text,
        }

    def _set_text_with_diff_formatting(self, text_edit, lines, kinds, colors, side, peer_lines=None):
        # Add a subtle marker column like WinMerge, without showing '+'/'-'
        marker = '▌'
        prefix = marker + ' '
        display_lines = []
        for line, kind in zip(lines, kinds):
            if kind in ('added', 'removed'):
                display_lines.append(prefix + line)
            else:
                display_lines.append('  ' + line)

        text_edit.setPlainText('\n'.join(display_lines))

        doc = text_edit.document()
        if peer_lines is None:
            peer_lines = [None] * len(kinds)

        for idx, kind in enumerate(kinds):
            block = doc.findBlockByNumber(idx)
            if not block.isValid():
                continue

            # Qt5/Qt6 compatibility: avoid QTextCursor.BlockUnderCursor
            move_op = getattr(QTextCursor, 'MoveOperation', None)
            move_mode = getattr(QTextCursor, 'MoveMode', None)
            end_of_block = getattr(QTextCursor, 'EndOfBlock', None) or (getattr(move_op, 'EndOfBlock', None) if move_op else None)
            keep_anchor = getattr(QTextCursor, 'KeepAnchor', None) or (getattr(move_mode, 'KeepAnchor', None) if move_mode else None)

            cursor = QTextCursor(doc)
            cursor.setPosition(block.position())
            if end_of_block is not None and keep_anchor is not None:
                cursor.movePosition(end_of_block, keep_anchor)
            else:
                # Fallback: select up to the end of the block text
                cursor.setPosition(block.position() + max(0, block.length() - 1), keep_anchor if keep_anchor is not None else QTextCursor.KeepAnchor)

            fmt = QTextCharFormat()
            peer = peer_lines[idx] if idx < len(peer_lines) else None
            targeted = (kind in ('added', 'removed')) and bool(peer)

            if kind == 'added':
                if not targeted:
                    fmt.setBackground(colors['added_bg'])
                    fmt.setForeground(colors['added_text'])
                else:
                    # Targeted intra-line highlight: keep line mostly normal, highlight only changed spans.
                    fmt.setForeground(colors['unchanged_text'])
            elif kind == 'removed':
                if not targeted:
                    fmt.setBackground(colors['removed_bg'])
                    fmt.setForeground(colors['removed_text'])
                else:
                    fmt.setForeground(colors['unchanged_text'])
            elif kind == 'unchanged':
                fmt.setForeground(colors['unchanged_text'])
            else:
                # blank placeholder
                continue
            cursor.setCharFormat(fmt)

            # Color just the marker character for added/removed lines
            if kind in ('added', 'removed'):
                cursor2 = QTextCursor(block)
                try:
                    cursor2.setPosition(block.position())
                    # Avoid Qt5/Qt6 enum differences (QTextCursor.Right may not exist)
                    cursor2.setPosition(block.position() + 1, getattr(QTextCursor, 'KeepAnchor', None) or (getattr(getattr(QTextCursor, 'MoveMode', None), 'KeepAnchor', None)))
                except Exception:
                    pass
                mfmt = QTextCharFormat()
                mfmt.setForeground(colors['added_text'] if kind == 'added' else colors['removed_text'])
                cursor2.setCharFormat(mfmt)

            # Intra-line targeted highlighting for replace blocks.
            # Only apply character-level highlighting if lines are sufficiently similar.
            # This prevents nonsensical highlights when comparing structurally different lines.
            if targeted:
                line_text = lines[idx] or ''
                other_text = peer or ''

                # Check similarity ratio - only do intra-line diff if lines are >50% similar
                sm = difflib.SequenceMatcher(None, line_text, other_text, autojunk=False)
                similarity = sm.ratio()

                # Skip intra-line highlighting for very different lines
                if similarity < 0.4:
                    # Fall back to full-line highlighting for dissimilar lines
                    cursor_full = QTextCursor(doc)
                    cursor_full.setPosition(block.position())
                    if end_of_block is not None and keep_anchor is not None:
                        cursor_full.movePosition(end_of_block, keep_anchor)
                    else:
                        cursor_full.setPosition(block.position() + max(0, block.length() - 1), keep_anchor if keep_anchor is not None else QTextCursor.KeepAnchor)
                    full_fmt = QTextCharFormat()
                    if kind == 'added':
                        full_fmt.setBackground(colors['added_bg'])
                        full_fmt.setForeground(colors['added_text'])
                    else:
                        full_fmt.setBackground(colors['removed_bg'])
                        full_fmt.setForeground(colors['removed_text'])
                    cursor_full.setCharFormat(full_fmt)
                    continue

                opcs = sm.get_opcodes()

                # Display includes two leading chars (marker+space or two spaces)
                content_start = block.position() + 2

                for t, a1, a2, b1, b2 in opcs:
                    if side == 'right':
                        # 'added' side: highlight inserts and replacements in *this* line
                        if t in ('insert', 'replace') and b2 > b1:
                            start = content_start + b1
                            end = content_start + b2
                            c = QTextCursor(doc)
                            c.setPosition(start)
                            if keep_anchor is not None:
                                c.setPosition(end, keep_anchor)
                            else:
                                c.setPosition(end, QTextCursor.KeepAnchor)
                            f = QTextCharFormat()
                            f.setBackground(colors['added_bg'])
                            f.setForeground(colors['added_text'])
                            c.setCharFormat(f)
                    else:
                        # 'removed' side: highlight deletes and replacements in *this* line
                        if t in ('delete', 'replace') and a2 > a1:
                            start = content_start + a1
                            end = content_start + a2
                            c = QTextCursor(doc)
                            c.setPosition(start)
                            if keep_anchor is not None:
                                c.setPosition(end, keep_anchor)
                            else:
                                c.setPosition(end, QTextCursor.KeepAnchor)
                            f = QTextCharFormat()
                            f.setBackground(colors['removed_bg'])
                            f.setForeground(colors['removed_text'])
                            c.setCharFormat(f)

    def _ensure_scroll_sync_connections(self):
        if getattr(self, '_scroll_sync_wired', False):
            return

        self._scroll_sync_wired = True
        self._in_scroll_sync = False

        def sync_from(src, dst):
            def _do(_value):
                if self._in_scroll_sync:
                    return
                self._in_scroll_sync = True
                try:
                    s = src.verticalScrollBar()
                    d = dst.verticalScrollBar()
                    if s.maximum() <= 0:
                        d.setValue(0)
                    else:
                        # Proportional sync; line counts should match due to placeholders.
                        ratio = s.value() / float(s.maximum())
                        d.setValue(int(ratio * d.maximum()))
                finally:
                    self._in_scroll_sync = False
            return _do

        try:
            self.current_text.verticalScrollBar().valueChanged.connect(sync_from(self.current_text, self.corrected_text))
            self.corrected_text.verticalScrollBar().valueChanged.connect(sync_from(self.corrected_text, self.current_text))
        except Exception:
            pass

    def escape_html(self, text):
        """Escape HTML special characters"""
        return (text.replace('&', '&amp;')
                .replace('<', '&lt;')
                .replace('>', '&gt;')
                .replace('"', '&quot;')
                .replace("'", '&#39;'))

    def generate_change_summary(self):
        """Generate a summary of the changes made"""
        current_lines = self.current_opf.split('\n')
        corrected_lines = self.corrected_opf.split('\n')

        diff = list(difflib.unified_diff(
            current_lines,
            corrected_lines,
            fromfile='Current OPF',
            tofile='Suggested OPF',
            lineterm='',
            n=3
        ))

        if not diff:
            summary = "No changes suggested - your OPF already follows standards closely!"
        else:
            # Skip file headers when counting
            additions = sum(1 for line in diff if line.startswith('+') and not line.startswith('+++'))
            deletions = sum(1 for line in diff if line.startswith('-') and not line.startswith('---'))

            summary = "Suggested changes to better conform to OPF standards:\n"
            summary += f"• Lines in suggested version: +{additions}\n"
            summary += f"• Lines from original: -{deletions}\n\n"
            summary += "Detailed changes:\n" + '\n'.join(diff[:50])

            if len(diff) > 50:
                summary += f"\n... and {len(diff) - 50} more changes"

        self.summary_text.setPlainText(summary)


    def copy_to_clipboard(self, text):
        """Copy text to clipboard"""
        from calibre.gui2 import QApplication
        clipboard = QApplication.clipboard()
        clipboard.setText(text)

        try:
            from calibre.gui2 import info_dialog
            info_dialog(self, 'Copied', 'Content copied to clipboard', show=True)
        except Exception:
            pass

    def closeEvent(self, e):
        """Save dialog geometry on close"""
        gprefs['opf_comparison_dialog_geometry'] = bytearray(self.saveGeometry())
        super().closeEvent(e)


class DiffLocationPane(QWidget):
    """WinMerge-like overview pane (minimap) showing where changes are."""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setMinimumWidth(26)
        self.setMaximumWidth(32)
        self._left_kinds = []
        self._right_kinds = []
        self._left_edit = None
        self._right_edit = None
        self._wired = False
        self._dragging = False
        self.setMouseTracking(True)
        self.setCursor(Qt.PointingHandCursor)
        self.setToolTip("Click or drag to navigate.\nGreen = added in suggested version\nRed = differs from original")

    def set_data(self, left_kinds, right_kinds):
        self._left_kinds = list(left_kinds or [])
        self._right_kinds = list(right_kinds or [])
        self.update()

    def attach(self, left_edit, right_edit):
        self._left_edit = left_edit
        self._right_edit = right_edit
        if self._wired:
            return
        self._wired = True
        try:
            left_edit.verticalScrollBar().valueChanged.connect(lambda _v: self.update())
            right_edit.verticalScrollBar().valueChanged.connect(lambda _v: self.update())
        except Exception:
            pass

    def _scroll_to_y(self, y):
        """Scroll to position based on y coordinate in the minimap."""
        if self._left_edit is None or self._right_edit is None:
            return
        ratio = min(1.0, max(0.0, y / float(max(1, self.height()))))
        try:
            sb = self._left_edit.verticalScrollBar()
            sb.setValue(int(ratio * sb.maximum()))
        except Exception:
            pass

    def mousePressEvent(self, e):
        self._dragging = True
        self._scroll_to_y(e.pos().y())

    def mouseMoveEvent(self, e):
        if self._dragging:
            self._scroll_to_y(e.pos().y())

    def mouseReleaseEvent(self, e):
        self._dragging = False

    def paintEvent(self, _e):
        p = QPainter(self)
        try:
            p.fillRect(self.rect(), self.palette().base())

            total = max(len(self._left_kinds), len(self._right_kinds), 1)
            h = max(1, self.height())
            w = max(1, self.width())

            # Two thin columns like WinMerge (left file, right file)
            col_gap = 2
            col_w = max(4, (w - col_gap - 4) // 2)
            left_x = 2
            right_x = left_x + col_w + col_gap

            def kind_color(kind):
                if kind == 'added':
                    return QColor(0, 160, 0, 170)
                if kind == 'removed':
                    return QColor(200, 0, 0, 170)
                if kind == 'unchanged':
                    return QColor(120, 120, 120, 60)
                return None

            def draw_column(x, kinds):
                for i, k in enumerate(kinds):
                    c = kind_color(k)
                    if c is None:
                        continue
                    y1 = int((i / float(total)) * h)
                    y2 = int(((i + 1) / float(total)) * h)
                    if y2 <= y1:
                        y2 = y1 + 1
                    p.fillRect(QRect(x, y1, col_w, y2 - y1), c)

            draw_column(left_x, self._left_kinds)
            draw_column(right_x, self._right_kinds)

            # Viewport marker (based on left scrollbar)
            if self._left_edit is not None:
                sb = self._left_edit.verticalScrollBar()
                maxv = sb.maximum()
                if maxv > 0:
                    top = sb.value() / float(maxv)
                    # approximate viewport size
                    page = sb.pageStep()
                    denom = float(maxv + page) if (maxv + page) > 0 else float(maxv)
                    frac_h = page / denom if denom > 0 else 0.1
                    y_top = int(top * h)
                    y_h = max(8, int(frac_h * h))
                    pen = QPen(QColor(80, 140, 220, 220))
                    pen.setWidth(1)
                    p.setPen(pen)
                    p.setBrush(QBrush(QColor(80, 140, 220, 40)))
                    p.drawRect(QRect(1, y_top, w - 2, min(h - y_top - 1, y_h)))
        finally:
            p.end()


class HighlightColorsDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        from calibre_plugins.opf_helper.config import prefs

        self.setWindowTitle('OPF Comparison Highlight Colors')
        self.setMinimumWidth(520)

        layout = QVBoxLayout(self)

        info = QLabel('Enter colors as #RRGGBB (or RRGGBB) or a CSS/Qt named color (e.g. rebeccapurple). Changes apply immediately after closing this dialog.')
        info.setWordWrap(True)
        layout.addWidget(info)

        dark_group = QGroupBox('Dark theme')
        dark_layout = QVBoxLayout(dark_group)
        layout.addWidget(dark_group)

        def row(label, widget1, label2=None, widget2=None):
            r = QHBoxLayout()
            r.addWidget(QLabel(label))
            r.addWidget(widget1)
            if label2 is not None and widget2 is not None:
                r.addWidget(QLabel(label2))
                r.addWidget(widget2)
            r.addStretch()
            return r

        def make_swatch():
            sw = QFrame()
            sw.setFixedSize(18, 18)
            sw.setFrameShape(QFrame.StyledPanel)
            sw.setFrameShadow(QFrame.Plain)
            return sw

        def normalize_for_css(v):
            v = (v or '').strip()
            if re.match(r'^[0-9a-fA-F]{6}$', v):
                return '#' + v
            return v

        def is_valid_color(v):
            v = (v or '').strip()
            if not v:
                return False
            if re.match(r'^#[0-9a-fA-F]{6}$', v) or re.match(r'^[0-9a-fA-F]{6}$', v):
                return True
            try:
                return QColor(v).isValid()
            except Exception:
                return False

        def update_swatch(line_edit, swatch):
            txt = (line_edit.text() or '').strip()
            if is_valid_color(txt):
                css = normalize_for_css(txt)
                swatch.setStyleSheet(f"QFrame {{ background-color: {css}; border: 1px solid #5B6985; }}")
            else:
                swatch.setStyleSheet("QFrame { background-color: transparent; border: 1px solid #CC3333; }")

        def make_color_input(initial):
            le = QLineEdit(initial)
            le.setMaximumWidth(140)
            sw = make_swatch()
            update_swatch(le, sw)
            le.textChanged.connect(lambda _t, le=le, sw=sw: update_swatch(le, sw))
            return le, sw

        self.dark_added_bg, self.dark_added_bg_sw = make_color_input(prefs.get('comparison_dark_added_bg', '#005A00'))
        self.dark_added_text, self.dark_added_text_sw = make_color_input(prefs.get('comparison_dark_added_text', '#FFFFFF'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Added: bg'))
        r.addWidget(self.dark_added_bg)
        r.addWidget(self.dark_added_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.dark_added_text)
        r.addWidget(self.dark_added_text_sw)
        r.addStretch()
        dark_layout.addLayout(r)

        self.dark_removed_bg, self.dark_removed_bg_sw = make_color_input(prefs.get('comparison_dark_removed_bg', '#5A0000'))
        self.dark_removed_text, self.dark_removed_text_sw = make_color_input(prefs.get('comparison_dark_removed_text', '#FFFFFF'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Removed: bg'))
        r.addWidget(self.dark_removed_bg)
        r.addWidget(self.dark_removed_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.dark_removed_text)
        r.addWidget(self.dark_removed_text_sw)
        r.addStretch()
        dark_layout.addLayout(r)

        self.dark_unchanged_text, self.dark_unchanged_text_sw = make_color_input(prefs.get('comparison_dark_unchanged_text', '#D0D0D0'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Unchanged: text'))
        r.addWidget(self.dark_unchanged_text)
        r.addWidget(self.dark_unchanged_text_sw)
        r.addStretch()
        dark_layout.addLayout(r)

        light_group = QGroupBox('Light theme')
        light_layout = QVBoxLayout(light_group)
        layout.addWidget(light_group)

        self.light_added_bg, self.light_added_bg_sw = make_color_input(prefs.get('comparison_light_added_bg', '#D4EDDA'))
        self.light_added_text, self.light_added_text_sw = make_color_input(prefs.get('comparison_light_added_text', '#155724'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Added: bg'))
        r.addWidget(self.light_added_bg)
        r.addWidget(self.light_added_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.light_added_text)
        r.addWidget(self.light_added_text_sw)
        r.addStretch()
        light_layout.addLayout(r)

        self.light_removed_bg, self.light_removed_bg_sw = make_color_input(prefs.get('comparison_light_removed_bg', '#F8D7DA'))
        self.light_removed_text, self.light_removed_text_sw = make_color_input(prefs.get('comparison_light_removed_text', '#721C24'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Removed: bg'))
        r.addWidget(self.light_removed_bg)
        r.addWidget(self.light_removed_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.light_removed_text)
        r.addWidget(self.light_removed_text_sw)
        r.addStretch()
        light_layout.addLayout(r)

        self.light_unchanged_text, self.light_unchanged_text_sw = make_color_input(prefs.get('comparison_light_unchanged_text', '#333333'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Unchanged: text'))
        r.addWidget(self.light_unchanged_text)
        r.addWidget(self.light_unchanged_text_sw)
        r.addStretch()
        light_layout.addLayout(r)

        reset_row = QHBoxLayout()
        reset_btn = QPushButton('Reset dark defaults')
        reset_btn.clicked.connect(self._reset_dark_defaults)
        reset_row.addWidget(reset_btn)
        reset_btn2 = QPushButton('Reset light defaults')
        reset_btn2.clicked.connect(self._reset_light_defaults)
        reset_row.addWidget(reset_btn2)
        reset_row.addStretch()
        layout.addLayout(reset_row)

        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)

    def _reset_dark_defaults(self):
        self.dark_added_bg.setText('#005A00')
        self.dark_added_text.setText('#FFFFFF')
        self.dark_removed_bg.setText('#5A0000')
        self.dark_removed_text.setText('#FFFFFF')
        self.dark_unchanged_text.setText('#D0D0D0')

    def _reset_light_defaults(self):
        self.light_added_bg.setText('#D4EDDA')
        self.light_added_text.setText('#155724')
        self.light_removed_bg.setText('#F8D7DA')
        self.light_removed_text.setText('#721C24')
        self.light_unchanged_text.setText('#333333')

    def accept(self):
        from calibre_plugins.opf_helper.config import prefs

        def norm(label, value):
            v = (value or '').strip()
            if not v:
                raise ValueError(f"{label} cannot be empty")

            # Accept hex in either form
            if re.match(r'^#[0-9a-fA-F]{6}$', v):
                return v.upper()
            if re.match(r'^[0-9a-fA-F]{6}$', v):
                return ('#' + v).upper()

            # Accept named colors (and other QColor-parseable CSS strings)
            try:
                if QColor(v).isValid():
                    return v
            except Exception:
                pass

            raise ValueError(f"{label} must be #RRGGBB, RRGGBB, or a named color")

        try:
            prefs['comparison_dark_added_bg'] = norm('Dark added background', self.dark_added_bg.text())
            prefs['comparison_dark_added_text'] = norm('Dark added text', self.dark_added_text.text())
            prefs['comparison_dark_removed_bg'] = norm('Dark removed background', self.dark_removed_bg.text())
            prefs['comparison_dark_removed_text'] = norm('Dark removed text', self.dark_removed_text.text())
            prefs['comparison_dark_unchanged_text'] = norm('Dark unchanged text', self.dark_unchanged_text.text())

            prefs['comparison_light_added_bg'] = norm('Light added background', self.light_added_bg.text())
            prefs['comparison_light_added_text'] = norm('Light added text', self.light_added_text.text())
            prefs['comparison_light_removed_bg'] = norm('Light removed background', self.light_removed_bg.text())
            prefs['comparison_light_removed_text'] = norm('Light removed text', self.light_removed_text.text())
            prefs['comparison_light_unchanged_text'] = norm('Light unchanged text', self.light_unchanged_text.text())
        except Exception as e:
            try:
                from calibre.gui2 import error_dialog
                error_dialog(self, 'Invalid color', str(e), show=True)
            except Exception:
                # If error_dialog isn't available for some reason, just keep the dialog open
                pass
            return

        super().accept()


class OPFXMLHighlighter(QSyntaxHighlighter):
    """XML syntax highlighter for OPF content"""

    def __init__(self, parent=None):
        super().__init__(parent)

        # Check if we're in dark mode
        try:
            from calibre.gui2 import is_dark_theme
            self.is_dark = is_dark_theme()
        except:
            self.is_dark = False

        # Define colors based on theme
        if self.is_dark:
            tag_color = "#88CCFF"     # Light blue
            attr_color = "#FFB366"    # Light orange
            value_color = "#90EE90"   # Light green
            comment_color = "#999999"  # Gray
        else:
            tag_color = "#000080"     # Navy blue
            attr_color = "#A0522D"    # Brown
            value_color = "#006400"   # Dark green
            comment_color = "#808080"  # Gray

        # XML element format
        tag_format = QTextCharFormat()
        tag_format.setForeground(QColor(tag_color))
        self.highlighting_rules = [(r'<[!?]?[a-zA-Z0-9_:-]+|/?>', tag_format)]

        # XML attribute format
        attribute_format = QTextCharFormat()
        attribute_format.setForeground(QColor(attr_color))
        self.highlighting_rules.append((r'\s[a-zA-Z0-9_:-]+(?=\s*=)', attribute_format))

        # XML value format
        value_format = QTextCharFormat()
        value_format.setForeground(QColor(value_color))
        self.highlighting_rules.append((r'"[^"]*"', value_format))

        # Comment format
        comment_format = QTextCharFormat()
        comment_format.setForeground(QColor(comment_color))
        self.highlighting_rules.append((r'<!--[\s\S]*?-->', comment_format))

        # Compile regex patterns for better performance
        import re
        self.rules = [(re.compile(pattern), fmt) for pattern, fmt in self.highlighting_rules]

    def highlightBlock(self, text):
        """Apply syntax highlighting to the given block of text."""
        for pattern, format in self.rules:
            for match in pattern.finditer(text):
                start, length = match.start(), match.end() - match.start()
                self.setFormat(start, length, format)