# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

"""MOBI OPF extraction for OPF-Helper using KindleUnpack.

This uses the same KindleUnpack core flow as the KindleUnpack plugin:
`Sectionizer` -> `MobiHeader` -> unpack -> generate OPF on disk.

We then read back the generated OPF (preferring KF8/mobi8 if present).
"""

import os
import shutil
import tempfile
import traceback
import uuid

from .kindleunpackcore import kindleunpack


def _make_workdir(prefix: str) -> str:
    # Prefer Calibre's cache directory (persists and is safe for plugins),
    # and avoid calibre.ptempfile here since it may import GUI/Qt modules
    # depending on Calibre version.
    try:
        from calibre.constants import cache_dir

        base = cache_dir()
        workdir = os.path.join(base, prefix + '_' + uuid.uuid4().hex)
        os.makedirs(workdir, exist_ok=True)
        return workdir
    except Exception:
        return tempfile.mkdtemp(prefix=prefix + '_')


def _best_opf_path(outdir: str):
    # KindleUnpack's unpack_structure.py writes these locations.
    k8 = os.path.join(outdir, 'mobi8', 'OEBPS', 'content.opf')
    m7 = os.path.join(outdir, 'mobi7', 'content.opf')
    if os.path.isfile(k8):
        return k8
    if os.path.isfile(m7):
        return m7

    # Fallback: search for any OPF.
    for root, _dirs, files in os.walk(outdir):
        for name in files:
            if name.lower().endswith('.opf'):
                return os.path.join(root, name)
    return None

def extract_mobi_opf(mobi_path, epubver='A', use_hd=False):
    """
    Extract OPF/XML from a MOBI file using KindleUnpack's OPFProcessor.
    Returns (opf_xml_string, error_message). If extraction fails, opf_xml_string is None.
    """
    workdir = None
    outdir = None
    try:
        workdir = _make_workdir('_opf_helper_mobi')
        outdir = os.path.join(workdir, 'kindleunpack')
        os.makedirs(outdir, exist_ok=True)

        kindleunpack.unpackBook(
            mobi_path,
            outdir,
            epubver=epubver,
            use_hd=use_hd,
            dodump=False,
            dowriteraw=False,
            dosplitcombos=False,
        )

        opf_path = _best_opf_path(outdir)
        if not opf_path:
            return None, 'KindleUnpack did not produce an OPF file.'

        with open(opf_path, 'rb') as f:
            data = f.read()
        try:
            return data.decode('utf-8'), None
        except Exception:
            return data.decode('utf-8', errors='replace'), None
    except Exception as e:
        tb = traceback.format_exc()
        return None, f'KindleUnpack extraction failed: {e}\n{tb}'
    finally:
        if workdir and os.path.isdir(workdir):
            shutil.rmtree(workdir, ignore_errors=True)
