# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import os
import sys
from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QTabWidget, QLabel,
    QScrollArea, QFrame, QSizePolicy, QPushButton,
    QLineEdit, QComboBox, QTableWidget, QTableWidgetItem, QHeaderView,
    QDialog, QDialogButtonBox, QTextEdit, QHBoxLayout, QMenu, QAction
)
from PyQt5.QtCore import Qt, QUrl
from PyQt5.QtGui import QDesktopServices

from calibre.utils.config import config_dir
from calibre_plugins.opf_helper.config import prefs
import json
from calibre_plugins.opf_helper import debug_print

class EducationPanel(QWidget):
    """Panel for displaying educational content about OPF and EPUB formats"""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent = parent
        self.setup_ui()

    def setup_ui(self):
        """Set up the user interface"""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)

        # Create tab widget for different educational content
        self.tabs = QTabWidget()

        # Apply minimal distinctive sub-tab styling to differentiate from main tabs
        self.tabs.setStyleSheet("""
            QTabWidget::pane {
                border: 1px solid palette(dark);
                background-color: palette(base);
            }
            QTabBar::tab {
                background-color: palette(window);
                border: 1px solid palette(dark);
                border-bottom: none;
                border-top-left-radius: 3px;
                border-top-right-radius: 3px;
                padding: 4px 8px;
                margin-right: 0px;
                font-size: 9pt;
                font-weight: normal;
                /* Slightly wider so longer tab titles are not truncated */
                min-width: 160px;
            }
            QTabBar::tab:selected {
                background-color: palette(window);
                color: palette(text);
                font-weight: bold;
                /* Emphasise the active sub-tab with a thin orange outline for
                   visibility across light and dark themes. Keep the look similar
                   to the main tabs but with a more visible highlight. */
                border: 2px solid #FFA500;
                border-bottom: none;
            }
            QTabBar::tab:!selected {
                color: palette(text);
                background-color: palette(window);
            }
        """)

        layout.addWidget(self.tabs)

        # Add historical context tab (text content with scrolling)
        self.history_widget = self._create_scrollable_text_widget(self._get_history_content())
        self.tabs.addTab(self.history_widget, "Historical Context")

        # Add specifications tab (links to official docs)
        self.specs_widget = self._create_scrollable_text_widget(self._get_specs_content())
        self.tabs.addTab(self.specs_widget, "Specifications")

        # Add OPF evolution tab with text content and PDF button
        evolution_widget = self._create_evolution_tab()
        self.tabs.addTab(evolution_widget, "OPF Evolution")

        # Add Common OPF warnings tab (placed last for minimal disruption)
        self.common_warnings_widget = self._create_common_warnings_tab()
        self.tabs.addTab(self.common_warnings_widget, "Common OPF warnings")

        # Restore previously-used sub-tab for the Resources (Education) panel
        try:
            last = prefs.get('resources_last_subtab', None)
            if isinstance(last, str):
                for i in range(self.tabs.count()):
                    try:
                        if self.tabs.tabText(i) == last:
                            self.tabs.setCurrentIndex(i)
                            break
                    except Exception:
                        continue
            elif isinstance(last, int):
                if 0 <= last < self.tabs.count():
                    self.tabs.setCurrentIndex(int(last))
        except Exception:
            pass

        # Persist sub-tab selection so dialog reopens with the same sub-tab
        try:
            self.tabs.currentChanged.connect(lambda idx: prefs.__setitem__('resources_last_subtab', self.tabs.tabText(idx) if idx >= 0 else None))
        except Exception:
            pass

    # Add missing load_diagram method
    def load_diagram(self):
        """This method is called when a book is loaded, but we don't need to do anything"""
        # This is a stub method to prevent the error
        debug_print("OPF Helper: Education panel diagram loading skipped (using PDF instead)")
        return True

    def _create_evolution_tab(self):
        """Create the OPF Evolution tab with text and PDF button"""
        container = QWidget()
        layout = QVBoxLayout(container)

        # Add introduction text
        intro = QLabel("""<h2>OPF and EPUB Format Evolution</h2>
        <p>This tab provides information about the evolution of the Open Package Format (OPF)
        and its role in the EPUB standard.</p>

        <p>The Open eBook Publication Structure (OEBPS) began in 1999 and evolved into the EPUB
        format we know today. A key milestone was the creation of EPUB 2.0 in 2007, which
        introduced the "triumvirate of specifications":</p>
        <ul>
            <li><b>Open Package Format (OPF)</b> - Package document format</li>
            <li><b>Open Publication Structure (OPS)</b> - Content document format</li>
            <li><b>OEBPS Container Format (OCF)</b> - Container file format</li>
        </ul>""")
        intro.setWordWrap(True)
        intro.setTextFormat(Qt.RichText)
        layout.addWidget(intro)

        # Add button to open the PDF
        open_pdf_button = QPushButton("View Evolution Diagram (PDF)")
        open_pdf_button.setMinimumHeight(40)
        # Use common_utils version instead
        from calibre_plugins.opf_helper.common_utils import open_documentation
        open_pdf_button.clicked.connect(lambda: open_documentation(self))
        layout.addWidget(open_pdf_button)

        # Additional explanation text
        more_info = QLabel("""<p>The detailed diagram shows the complete evolution from OEBPS 1.0
        through to modern EPUB versions, highlighting key milestones and relationships between
        the specifications.</p>""")
        more_info.setWordWrap(True)
        more_info.setTextFormat(Qt.RichText)
        layout.addWidget(more_info)

        layout.addStretch()
        return container

    def _create_common_warnings_tab(self):
        """Create the Common OPF warnings tab with filter and table"""
        container = QWidget()
        layout = QVBoxLayout(container)

        info = QLabel(
            "This tab shows a generic catalog of common OPF validation problems, based on a large test dataset. "
            "It does not scan your current library."
        )
        info.setWordWrap(True)
        layout.addWidget(info)

        # Filter input: use a small composite widget (line edit + clear + history)
        filter_widget = QWidget()
        fh_layout = QHBoxLayout(filter_widget)
        fh_layout.setContentsMargins(0, 0, 0, 0)

        le = QLineEdit()
        le.setMinimumWidth(240)
        le.setPlaceholderText('Filter warnings by text…')
        # Minimal, discrete fixed outline for the filter box (visible in
        # both light and dark themes). Use a subtle semi-transparent orange.
        try:
            from PyQt5.QtGui import QPalette
            pal = le.palette()
            col = pal.color(QPalette.Shadow)
            # Use a subtle semi-transparent outline based on theme shadow color
            rgba = f'rgba({col.red()},{col.green()},{col.blue()},0.12)'
            le.setStyleSheet(f'border: 1px solid {rgba}; border-radius: 3px; padding: 4px;')
        except Exception:
            try:
                le.setStyleSheet('border: 1px solid rgba(128,128,128,0.12); border-radius: 3px; padding: 4px;')
            except Exception:
                pass

        btn_clear = QPushButton('✕')
        btn_clear.setToolTip('Clear filter')
        btn_clear.setFixedWidth(28)

        btn_hist = QPushButton('▾')
        btn_hist.setToolTip('Show filter history')
        btn_hist.setFixedWidth(28)

        fh_layout.addWidget(le)
        fh_layout.addWidget(btn_clear)
        fh_layout.addWidget(btn_hist)

        # History persistence
        self._clear_history_user_data = '__opf_helper_clear_filter_history__'
        self._clear_history_label = 'Clear history'

        def _load_history():
            try:
                return prefs.get('warnings_filter_history', []) or []
            except Exception:
                return []

        def _save_history(hist):
            try:
                max_len = prefs.get('warnings_filter_history_max', 12)
                prefs['warnings_filter_history'] = hist[0:max_len]
            except Exception:
                pass

        history_menu = QMenu(self)

        def _rebuild_history_menu():
            try:
                history_menu.clear()
                hist = _load_history()
                for h in hist:
                    a = QAction(h, history_menu)
                    history_menu.addAction(a)
                if hist:
                    history_menu.addSeparator()
                clear_act = QAction(self._clear_history_label, history_menu)
                font = clear_act.font()
                font.setItalic(True)
                clear_act.setFont(font)
                history_menu.addAction(clear_act)
            except Exception:
                pass

        def _commit_to_history(q):
            try:
                q = (q or '').strip()
                if not q:
                    return
                hist = _load_history()
                if q in hist:
                    try:
                        hist.remove(q)
                    except Exception:
                        pass
                hist.insert(0, q)
                _save_history(hist)
                _rebuild_history_menu()
            except Exception:
                pass

        # Signal handlers
        last_non_empty = {'v': ''}

        def _on_text_changed(t):
            try:
                t = (t or '').strip()
                apply_filter(t)
                if t:
                    last_non_empty['v'] = t
            except Exception:
                pass

        def _on_editing_finished():
            try:
                q = (le.text() or '').strip()
                if q:
                    _commit_to_history(q)
            except Exception:
                pass

        def _on_return_pressed():
            try:
                q = (le.text() or '').strip()
                if q:
                    _commit_to_history(q)
                apply_filter(q)
            except Exception:
                pass

        def _on_clear_clicked():
            try:
                prev = last_non_empty.get('v') or ''
                if prev:
                    _commit_to_history(prev)
                le.setText('')
                apply_filter('')
                le.setFocus()
                last_non_empty['v'] = ''
            except Exception:
                pass

        def _on_history_requested():
            try:
                _rebuild_history_menu()
                history_menu.exec_(btn_hist.mapToGlobal(btn_hist.rect().bottomLeft()))
            except Exception:
                pass

        def _on_history_selected(action):
            try:
                if not action:
                    return
                txt = action.text() or ''
                if txt == self._clear_history_label:
                    try:
                        _save_history([])
                        _rebuild_history_menu()
                        apply_filter('')
                        le.setText('')
                        le.setFocus()
                    except Exception:
                        pass
                    return
                le.setText(txt)
                _commit_to_history(txt)
                apply_filter(txt)
                le.setFocus()
            except Exception:
                pass

        # Connect signals
        try:
            le.textChanged.connect(_on_text_changed)
            le.editingFinished.connect(_on_editing_finished)
            le.returnPressed.connect(_on_return_pressed)
        except Exception:
            pass
        try:
            btn_clear.clicked.connect(_on_clear_clicked)
            btn_hist.clicked.connect(_on_history_requested)
            history_menu.triggered.connect(_on_history_selected)
        except Exception:
            pass

        # Initial menu build
        _rebuild_history_menu()

        layout.addWidget(filter_widget)

        # Table setup: Error, Explanation, Suggestion
        # Custom subclass to handle Enter key for opening detail dialog
        class _TableWithEnter(QTableWidget):
            def keyPressEvent(self, event):
                if event.key() in (Qt.Key_Return, Qt.Key_Enter):
                    open_detail()
                    event.accept()
                    return
                super(_TableWithEnter, self).keyPressEvent(event)

        table = _TableWithEnter(0, 3)
        table.setHorizontalHeaderLabels(["Error", "Explanation", "Suggestion"])
        header = table.horizontalHeader()
        # Qt5/Qt6 compatibility for section resize mode
        def _set_stretch(col):
            try:
                header.setSectionResizeMode(col, QHeaderView.ResizeMode.Stretch)
            except Exception:
                try:
                    header.setSectionResizeMode(col, QHeaderView.Stretch)
                except Exception:
                    if col == table.columnCount() - 1:
                        header.setStretchLastSection(True)
        _set_stretch(0)
        _set_stretch(1)
        _set_stretch(2)
        table.setSelectionBehavior(table.SelectRows)
        table.setSelectionMode(table.SingleSelection)
        table.setEditTriggers(table.NoEditTriggers)
        # Allow the table to expand and take available vertical space so there's no empty gap
        try:
            table.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        except Exception:
            pass
        layout.addWidget(table, 1)

        # Always load the generic curated warnings list (no user/library data)
        warnings_data = self._load_curated_warnings_only()

        # Keep original list for filtering
        container._warnings_full = warnings_data
        container._table = table

        def populate(rows):
            table.setRowCount(0)
            for item in rows:
                r = table.rowCount()
                table.insertRow(r)
                title = item.get("title") or item.get("key") or ""
                table.setItem(r, 0, QTableWidgetItem(title))
                table.setItem(r, 1, QTableWidgetItem(item.get("explanation", "")))
                table.setItem(r, 2, QTableWidgetItem(item.get("suggestion", "")))
                # Store full object for detail dialog (safe under sorting)
                try:
                    table.item(r, 0).setData(Qt.UserRole, item)
                except Exception:
                    pass
                table.setRowHeight(r, 22)

        def apply_filter(text):
            t = (text or "").strip().lower()
            if not t:
                populate(container._warnings_full)
                return
            rows = []
            for it in container._warnings_full:
                hay = " ".join([
                    it.get("title", ""),
                    it.get("key", ""),
                    it.get("explanation", ""),
                    it.get("suggestion", ""),
                    " ".join(it.get("tags", [])),
                ]).lower()
                if all(tok in hay for tok in t.split()):
                    rows.append(it)
            populate(rows)

        # Initial population
        populate(warnings_data)
        # Use line edit's textChanged signal (QComboBox has no textChanged)
        try:
            le = filter_edit.lineEdit()
            le.textChanged.connect(lambda t: apply_filter(t))
        except Exception:
            pass

        # Double click -> detail dialog
        def open_detail():
            r = table.currentRow()
            if r < 0 or r >= table.rowCount():
                return
            title = table.item(r, 0).text() if table.item(r, 0) else "Detail"
            obj = {}
            try:
                obj = table.item(r, 0).data(Qt.UserRole) or {}
            except Exception:
                pass
            self._show_warning_detail_dialog(title, obj)

        table.doubleClicked.connect(lambda idx: open_detail())
        table.returnPressed = open_detail  # convenience if table gets focus

        layout.addStretch()
        return container

    def _load_curated_warnings_only(self):
        """Load the full curated warnings JSON from all possible plugin/config paths."""
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        # Use the single canonical configuration path (what we actually use):
        #   config_dir/plugins/OPF Helper/common_opf_warnings.json
        config_root_dir = os.path.join(config_dir, 'plugins', 'OPF Helper')
        p = os.path.join(config_root_dir, 'common_opf_warnings.json')
        debug_print(f"OPF Helper: Trying curated warnings JSON at {p}")
        try:
            if os.path.exists(p):
                with open(p, 'r', encoding='utf-8') as f:
                    txt = f.read()
                # Try to parse JSON, or recover from concatenated/malformed JSON blocks
                data = None
                try:
                    data = json.loads(txt)
                except Exception:
                    # Attempt to recover from concatenated or malformed JSON by
                    # extracting balanced JSON object blocks and parsing them.
                    debug_print(f"OPF Helper: Malformed curated JSON at {p}; attempting recovery")
                    objs = []
                    stack = 0
                    start = None
                    for i, ch in enumerate(txt):
                        if ch == '{':
                            if stack == 0:
                                start = i
                            stack += 1
                        elif ch == '}':
                            stack -= 1
                            if stack == 0 and start is not None:
                                block = txt[start:i+1]
                                try:
                                    objs.append(json.loads(block))
                                except Exception:
                                    # skip invalid blocks
                                    pass
                                start = None
                    if objs:
                        # Reconstruct as list
                        data = objs
                    else:
                        # Let outer exception handler catch this
                        raise

                # If we've parsed data (either directly or via recovery), normalise and return
                if isinstance(data, list):
                    items = data
                elif isinstance(data, dict) and 'items' in data:
                    items = data.get('items') or []
                else:
                    items = []

                def _normalize_item(it):
                    if not isinstance(it, dict):
                        return None
                    key = (it.get('key') or it.get('id') or '').strip()
                    title = (it.get('title') or '').strip()
                    explanation = (it.get('explanation') or '').strip()
                    suggestion = (it.get('suggestion') or '').strip()
                    tags = it.get('tags') or []
                    if not isinstance(tags, list):
                        try:
                            tags = list(tags)
                        except Exception:
                            tags = []
                    if not title and key:
                        if '|' in key:
                            parts = [p.strip() for p in key.split('|') if p.strip()]
                            title = parts[-1] if parts else key
                        else:
                            title = key if len(key) <= 80 else key[:77] + '...'
                    if not key and title:
                        key = title.lower().replace(' ', '-')[0:120]
                    return {
                        'key': key,
                        'title': title or key,
                        'explanation': explanation,
                        'suggestion': suggestion,
                        'tags': tags,
                    }
                norm = []
                seen = set()
                for raw in items:
                    try:
                        ni = _normalize_item(raw)
                        if ni is None:
                            continue
                        k = ni.get('key') or ni.get('title')
                        if not k:
                            continue
                        if k in seen:
                            continue
                        seen.add(k)
                        norm.append(ni)
                    except Exception:
                        continue
                if norm:
                    debug_print(f"OPF Helper: Loaded {len(norm)} curated warnings from {p}")
                    return norm
        except Exception as e:
            debug_print(f"OPF Helper: Failed to read curated warnings at {p}: {e}")
        debug_print("OPF Helper: No curated warnings JSON found; using built-in minimal set")
        # Minimal built-in fallback
        return [
            {
                "key": "attribute-not-allowed|item|properties",
                "title": "OPF item 'properties' attribute not allowed",
                "explanation": "In OPF 2.0, the 'properties' attribute on manifest 'item' is not permitted.",
                "suggestion": "Remove 'properties' for EPUB 2, or upgrade to EPUB 3 where it's valid.",
                "tags": ["opf2", "manifest", "compatibility"],
            },
            {
                "key": "missing-child|tours|tour",
                "title": "Missing required child <tour> under <tours>",
                "explanation": "The 'tours' element expects at least one 'tour' child per schema.",
                "suggestion": "Add a valid 'tour' child or remove the 'tours' element.",
                "tags": ["schema", "structure"],
            },
            {
                "key": "no-global-declaration|package",
                "title": "No matching global declaration for 'package'",
                "explanation": "The OPF root element did not validate against the schema in use.",
                "suggestion": "Verify XML namespace/version and that the correct OPF schema applies.",
                "tags": ["schema", "root"],
            },
        ]

    def _show_warning_detail_dialog(self, title, obj):
        """Show a simple dialog with details for a selected warning"""
        dlg = QDialog(self)
        dlg.setWindowTitle(title or "Common OPF warning")
        v = QVBoxLayout(dlg)

        txt = QTextEdit()
        txt.setReadOnly(True)
        txt.setAcceptRichText(True)

        key = obj.get('key') or ''
        explanation = obj.get('explanation') or 'N/A'
        suggestion = obj.get('suggestion') or 'N/A'
        tags = ", ".join(obj.get('tags') or [])

        # Build styled HTML body with bold labels
        try:
            from PyQt5.QtGui import QColor, QPalette
            palette = self.palette()
            # Try to use a slightly darker/different color for labels (theme-aware)
            try:
                label_color = palette.color(QPalette.Link).name()
            except Exception:
                label_color = '#0066CC'  # fallback blue
        except Exception:
            label_color = '#0066CC'

        body = (
            f"<b>{obj.get('title') or title or ''}</b><br><br>"
            f"<b style='color: {label_color}'>Key:</b> {key}<br><br>"
            f"<b style='color: {label_color}'>Explanation:</b><br>{explanation}<br><br>"
            f"<b style='color: {label_color}'>Suggestion:</b><br>{suggestion}<br><br>"
            f"<b style='color: {label_color}'>Tags:</b> {tags}<br>"
        )
        txt.setHtml(body)
        v.addWidget(txt)

        btns = QDialogButtonBox(QDialogButtonBox.Ok)
        copy_btn = btns.addButton('Copy', QDialogButtonBox.ActionRole)

        def do_copy():
            try:
                from PyQt5.QtWidgets import QApplication
                # Copy plain text version
                plain_body = (
                    f"{obj.get('title') or title or ''}\n\n"
                    f"Key: {key}\n\n"
                    f"Explanation:\n{explanation}\n\n"
                    f"Suggestion:\n{suggestion}\n\n"
                    f"Tags: {tags}\n"
                )
                QApplication.clipboard().setText(plain_body)
            except Exception:
                pass

        copy_btn.clicked.connect(do_copy)
        btns.accepted.connect(dlg.accept)
        v.addWidget(btns)

        dlg.resize(640, 360)
        dlg.exec_()

    def _create_scrollable_text_widget(self, html_content):
        """Create a scrollable widget containing html formatted text"""
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameStyle(QFrame.NoFrame)

        content = QLabel(html_content)
        content.setWordWrap(True)
        content.setTextFormat(Qt.RichText)
        content.setOpenExternalLinks(True)
        content.setTextInteractionFlags(Qt.TextBrowserInteraction)

        scroll.setWidget(content)
        return scroll

    def _get_history_content(self):
        """Return HTML formatted text about OPF/EPUB history"""
        return """
        <h2>EPUB and OPF Format History</h2>

        <h3>OEBPS Origins (1999-2002)</h3>
        <p>The Open eBook Publication Structure (OEBPS) was first approved in 1999 by the Open eBook Forum,
        which later became the International Digital Publishing Forum (IDPF). This was the foundational
        specification for what would eventually become EPUB.</p>

        <p>Key milestones in this period:</p>
        <ul>
            <li><b>OEBPS 1.0</b> (1999): Initial specification</li>
            <li><b>OEBPS 1.1</b> (2001): Revision with improvements</li>
            <li><b>OEBPS 1.2</b> (2002): Further refinements</li>
        </ul>

        <h3>EPUB 2.0 Development (2005-2007)</h3>
        <p>In late 2005, work began on a container format for OEBPS, which was approved as the
        OEBPS Container Format (OCF) in 2006. Parallel work began on OEBPS 2.0, which was eventually
        renamed to EPUB.</p>

        <p><b>EPUB 2.0</b> (October 2007) consisted of three core specifications:</p>
        <ol>
            <li><b>Open Package Format (OPF)</b>: Defines the package document that ties together all
            components of an EPUB publication</li>
            <li><b>Open Publication Structure (OPS)</b>: Defines the content format</li>
            <li><b>OEBPS Container Format (OCF)</b>: Defines how all components are packaged together</li>
        </ol>

        <h3>EPUB 2.0.1 and Later</h3>
        <p><b>EPUB 2.0.1</b> (September 2010) was a maintenance update clarifying and correcting
        issues in the 2.0 specification.</p>

        <p>The OPF file (content.opf) is the backbone of an EPUB publication, containing:</p>
        <ul>
            <li>Metadata about the publication (title, author, etc.)</li>
            <li>Manifest listing all publication resources</li>
            <li>Spine defining the reading order</li>
            <li>Guide (optional) with references to key content</li>
        </ul>

        <h3>Role of OPF in EPUB</h3>
        <p>The OPF (Open Package Format) document is the central component in an EPUB file. It:</p>
        <ul>
            <li>Provides a way to express metadata about the publication</li>
            <li>Creates an inventory of all resources included in the publication</li>
            <li>Defines a standard reading order</li>
            <li>Enables navigation and reference to key structural components</li>
        </ul>

        <p>The OPF specification continues to be central to EPUB, with refinements and
        extensions in EPUB 3.0 and later versions.</p>
        """

    def _get_specs_content(self):
        """Return HTML formatted text with specification links"""
        return """
        <h2>EPUB and OPF Specifications</h2>

        <h3>EPUB 2.0 Specifications</h3>
        <ul>
            <li><a href="http://idpf.org/epub/20/spec/OPF_2.0.1_draft.htm">Open Package Format 2.0.1</a></li>
            <li><a href="http://idpf.org/epub/20/spec/OPS_2.0.1_draft.htm">Open Publication Structure 2.0.1</a></li>
            <li><a href="http://idpf.org/epub/20/spec/OCF_2.0.1_draft.htm">Open Container Format 2.0.1</a></li>
        </ul>

        <h3>EPUB 3.0 Specifications</h3>
        <ul>
            <li><a href="https://www.w3.org/TR/epub-33/#sec-package-doc">EPUB 3.3 Package Document</a> (latest W3C version)</li>
            <li><a href="https://www.w3.org/TR/epub-33/#sec-container">EPUB 3.3 Container</a></li>
            <li><a href="https://www.w3.org/TR/epub-33/#sec-nav">EPUB 3.3 Navigation</a></li>
        </ul>

        <h3>Related Specifications</h3>
        <ul>
            <li><a href="https://www.dublincore.org/specifications/dublin-core/dces/">Dublin Core Metadata Element Set</a>: The basis for EPUB metadata</li>
            <li><a href="https://www.w3.org/TR/xml/">XML 1.0</a>: The foundation format for OPF documents</li>
        </ul>

        <h3>Tutorials and Resources</h3>
        <ul>
            <li><a href="https://wiki.mobileread.com/wiki/OPF">MobileRead Wiki: OPF</a>: Community documentation</li>
            <li><a href="https://idpf.github.io/epub-cmt/30/spec/epub30-contentdocs.html">EPUB 3 Content Documents</a>: Specification for content documents</li>
        </ul>

        <h3>Historical Specifications</h3>
        <ul>
            <li><a href="http://web.archive.org/web/20101124201142/http://www.openebook.org/oebps/oebps1.2/index.htm">OEBPS 1.2</a> (Web Archive)</li>
            <li><a href="http://web.archive.org/web/20101124112439/http://www.openebook.org/oebps/oebps1.0/index.htm">OEBPS 1.0</a> (Web Archive)</li>
        </ul>
        """

    def open_pdf_diagram(self):
        """Open the PDF diagram in the system's default PDF viewer"""
        try:
            # Lazily ensure the PDF exists. This is intentionally *not* done on
            # plugin init, to avoid touching the PDF on every Calibre start.
            try:
                from calibre_plugins.opf_helper.common_utils import extract_documentation_from_zip
                extract_documentation_from_zip()
            except Exception:
                pass

            # Use config_dir/plugins/OPF Helper root for PDF resource (no nested folder)
            config_root_dir = os.path.join(config_dir, 'plugins', 'OPF Helper')
            pdf_path = os.path.join(config_root_dir, 'opf_evolution_diagram.pdf')

            if os.path.exists(pdf_path):
                debug_print(f"Opening PDF diagram: {pdf_path}")
                QDesktopServices.openUrl(QUrl.fromLocalFile(pdf_path))
                return True
            else:
                debug_print(f"PDF diagram not found at {pdf_path}")
                from calibre.gui2 import error_dialog
                error_dialog(self.parent, 'Diagram Not Found',
                           'The diagram PDF file was not found. Try reinstalling this plugin.',
                           show=True)
                return False
        except Exception as e:
            debug_print(f"Error opening PDF diagram: {str(e)}")
            import traceback
            traceback.print_exc()
            from calibre.gui2 import error_dialog
            error_dialog(self.parent, 'Error Opening Diagram',
                       f'Failed to open the diagram: {str(e)}',
                       show=True)
            return False