# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from qt.core import (QWidget, QVBoxLayout, QHBoxLayout, QPushButton,
                    QLabel, QDialogButtonBox, QDialog, QDesktopServices,
                    QUrl, Qt, QListWidget, QSpinBox, QListWidgetItem,
                    QGroupBox, QCheckBox, QComboBox)

from calibre.gui2 import error_dialog, gprefs
from calibre.gui2.keyboard import ShortcutConfig
from calibre.utils.config import JSONConfig
from calibre_plugins.opf_helper.common_icons import set_plugin_icon_resources, get_icon
from calibre.gui2 import open_url

HELP_URL = 'https://www.mobileread.com/forums/showthread.php?t=371086'

def show_help():
    open_url(QUrl(HELP_URL))

# This is where all preferences for this plugin will be stored
prefs = JSONConfig('plugins/OPF Helper/opf_helper')

# Set defaults
prefs.defaults['tab_order'] = ['OPF Tree', 'Statistics', 'Validation', 'XML', 'Resources']
prefs.defaults['font_size'] = 12
prefs.defaults['font_size_increment'] = 2
prefs.defaults['auto_validate'] = True
prefs.defaults.pop('default_tab', None)
prefs.defaults['show_cover'] = True
prefs.defaults['show_book_id'] = True
prefs.defaults['epub_version_scope'] = 'Library'  # 'Library' or 'Selection'


# Bulk validator scope (independent from other scanners): 'Library' or 'Selection'
prefs.defaults['bulk_validate_opfs_scope'] = 'Library'

# Bulk validator: default max workers for parallel validation
prefs.defaults['bulk_max_workers'] = 32

# Search options + history (Find in OPF)
prefs.defaults['search_case_sensitive'] = False
prefs.defaults['search_whole_words'] = False
prefs.defaults['search_regex'] = False
prefs.defaults['search_history'] = []
prefs.defaults['search_history_max'] = 12

# Warnings filter history (used by the 'Common OPF warnings' tab)
prefs.defaults['warnings_filter_history'] = []
prefs.defaults['warnings_filter_history_max'] = 12

# Window reuse behavior
# When True, repeated invocations reuse a single modeless OPF Helper window and
# update its selection. When False, each invocation opens a new window.
prefs.defaults['reuse_single_opf_helper_window'] = True

prefs.defaults.pop('opf_edit_mode', None)

# OPF Comparison Dialog highlighting colors
# Dark theme colors
prefs.defaults['comparison_dark_added_bg'] = '#005A00'
prefs.defaults['comparison_dark_added_text'] = '#FFFFFF'
prefs.defaults['comparison_dark_removed_bg'] = '#5A0000'
prefs.defaults['comparison_dark_removed_text'] = '#FFFFFF'
prefs.defaults['comparison_dark_unchanged_text'] = '#D0D0D0'
# Light theme colors
prefs.defaults['comparison_light_added_bg'] = '#D4EDDA'
prefs.defaults['comparison_light_added_text'] = '#155724'
prefs.defaults['comparison_light_removed_bg'] = '#F8D7DA'
prefs.defaults['comparison_light_removed_text'] = '#721C24'
prefs.defaults['comparison_light_unchanged_text'] = '#333333'

# Diff algorithm used in the OPF comparison dialog
# Options: 'default', 'patience', 'histogram'
prefs.defaults['comparison_diff_algorithm'] = 'patience'

class ConfigWidget(QWidget):

    def __init__(self, plugin_action=None):
        QWidget.__init__(self)

        # Create a basic layout
        layout = QVBoxLayout()
        self.setLayout(layout)

        # Add title and help button
        title_layout = QHBoxLayout()
        title_layout.addWidget(QLabel('<h2>OPF Helper Configuration</h2>'))
        # Help button removed — keep header compact
        title_layout.addStretch()
        layout.addLayout(title_layout)

        # Display Settings Group
        display_group = QGroupBox('Display Settings')
        display_layout = QVBoxLayout()
        display_group.setLayout(display_layout)

        # Create font size section
        font_layout = QHBoxLayout()
        font_layout.addWidget(QLabel('Default Font Size:'))
        self.font_size_spin = QSpinBox()
        self.font_size_spin.setRange(8, 24)
        self.font_size_spin.setValue(prefs['font_size'])
        font_layout.addWidget(self.font_size_spin)

        font_layout.addWidget(QLabel('Font Size Increment:'))
        self.font_increment_spin = QSpinBox()
        self.font_increment_spin.setRange(1, 5)
        self.font_increment_spin.setValue(prefs['font_size_increment'])
        font_layout.addWidget(self.font_increment_spin)
        font_layout.addStretch()
        display_layout.addLayout(font_layout)

        # Cover display option
        cover_layout = QHBoxLayout()
        self.show_cover_checkbox = QCheckBox('Display cover image in OPF Helper dialog')
        self.show_cover_checkbox.setChecked(prefs.get('show_cover', True))
        cover_layout.addWidget(self.show_cover_checkbox)
        cover_layout.addStretch()
        display_layout.addLayout(cover_layout)

        # Book ID display option
        book_id_layout = QHBoxLayout()
        self.show_book_id_checkbox = QCheckBox('Display Calibre book ID in dialog heading')
        self.show_book_id_checkbox.setChecked(prefs.get('show_book_id', True))
        book_id_layout.addWidget(self.show_book_id_checkbox)
        book_id_layout.addStretch()
        display_layout.addLayout(book_id_layout)

        layout.addWidget(display_group)

        # Behavior Settings Group
        behavior_group = QGroupBox('Behavior Settings')
        behavior_layout = QVBoxLayout()
        behavior_group.setLayout(behavior_layout)

        # Default tab option removed; tab persistence is handled by the plugin

        # Auto Validate Option
        validate_layout = QHBoxLayout()
        self.auto_validate_checkbox = QCheckBox('Automatically validate OPF when opening')
        self.auto_validate_checkbox.setChecked(prefs.get('auto_validate', True))
        validate_layout.addWidget(self.auto_validate_checkbox)
        validate_layout.addStretch()
        behavior_layout.addLayout(validate_layout)

        # Window reuse option (modeless)
        reuse_layout = QHBoxLayout()
        self.reuse_single_window_checkbox = QCheckBox('Reuse a single OPF Helper window (modeless)')
        self.reuse_single_window_checkbox.setToolTip(
            'When enabled, running OPF Helper again will bring the existing window to the front and update its selection.\n'
            'Disable to allow multiple OPF Helper windows to remain open at once.'
        )
        self.reuse_single_window_checkbox.setChecked(prefs.get('reuse_single_opf_helper_window', True))
        reuse_layout.addWidget(self.reuse_single_window_checkbox)
        reuse_layout.addStretch()
        behavior_layout.addLayout(reuse_layout)

        # EPUB Version Finder Scope
        scope_layout = QHBoxLayout()
        scope_layout.addWidget(QLabel('EPUB Version Finder Scope:'))
        self.epub_scope_combo = QComboBox()
        self.epub_scope_combo.addItems(['Library', 'Selection'])
        self.epub_scope_combo.setCurrentText(prefs.get('epub_version_scope', 'Library'))
        scope_layout.addWidget(self.epub_scope_combo)
        scope_layout.addStretch()
        behavior_layout.addLayout(scope_layout)

        layout.addWidget(behavior_group)

        # OPF Editing: external editor mode removed (unreliable across environments)

        # Tab Order Group
        tab_order_group = QGroupBox('Tab Order')
        tab_order_layout = QVBoxLayout()
        tab_order_group.setLayout(tab_order_layout)

        tab_order_layout.addWidget(QLabel('Drag and drop to change tab order:'))

        self.tab_list = QListWidget()
        self.tab_list.setDragDropMode(QListWidget.DragDropMode.InternalMove)

        # Add all configurable tabs to the list (exclude About tab)
        configurable_tabs = [tab for tab in prefs['tab_order'] if tab != 'About']
        for tab in configurable_tabs:
            QListWidgetItem(tab, self.tab_list)

        tab_order_layout.addWidget(self.tab_list)

        layout.addWidget(tab_order_group)

        # Add some spacing
        layout.addStretch()

    def save_settings(self):
        # Save font settings
        prefs['font_size'] = self.font_size_spin.value()
        prefs['font_size_increment'] = self.font_increment_spin.value()

        # Save cover display preference
        prefs['show_cover'] = self.show_cover_checkbox.isChecked()

        # Save book ID display preference
        prefs['show_book_id'] = self.show_book_id_checkbox.isChecked()

        # Default tab option removed; nothing to save here

        # Save auto-validate preference
        prefs['auto_validate'] = self.auto_validate_checkbox.isChecked()

        # Save window reuse preference
        prefs['reuse_single_opf_helper_window'] = self.reuse_single_window_checkbox.isChecked()

        # Save EPUB version finder scope
        prefs['epub_version_scope'] = self.epub_scope_combo.currentText()

        # External OPF edit mode removed; do not persist stale prefs
        try:
            prefs.pop('opf_edit_mode', None)
        except Exception:
            pass

        # Save tab order
        tab_order = []
        for i in range(self.tab_list.count()):
            tab_order.append(self.tab_list.item(i).text())
        prefs['tab_order'] = tab_order

class KeyboardConfigDialog(QDialog):
    """Dialog for editing keyboard shortcuts"""
    def __init__(self, gui, group_name):
        QDialog.__init__(self, gui)
        self.gui = gui
        self.group_name = group_name

        self.setWindowTitle('Customize Keyboard Shortcuts')
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        # Create and initialize the keyboard shortcut widget
        self.keyboard_widget = ShortcutConfig(self)
        layout.addWidget(self.keyboard_widget)

        # Add standard dialog buttons
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        # Restore previous dialog geometry if available
        geom = gprefs.get('opf_helper_shortcuts_dialog_geometry', None)
        if geom is not None:
            self.restoreGeometry(geom)
        else:
            # Default size if no geometry saved
            self.resize(800, 600)

        # Initialize the keyboard widget
        self.keyboard_widget.initialize(self.gui.keyboard)
        self.keyboard_widget.highlight_group(self.group_name)

    def accept(self):
        """Save changes and close"""
        self.keyboard_widget.commit()
        self.save_geometry()
        super().accept()

    def reject(self):
        """Cancel without saving changes"""
        self.save_geometry()
        super().reject()

    def save_geometry(self):
        """Save dialog size and position"""
        gprefs['opf_helper_shortcuts_dialog_geometry'] = bytearray(self.saveGeometry())
