# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

try:
    from qt.core import QUrl, QDesktopServices
except ImportError:
    try:
        from calibre.gui2.qt.core import QUrl
        from calibre.gui2.qt.gui import QDesktopServices
    except ImportError:
        from PyQt5.QtCore import QUrl
        from PyQt5.QtGui import QDesktopServices
import os
import json
from calibre.utils.config import config_dir
from calibre.constants import iswindows
from calibre.gui2 import error_dialog
try:
    from calibre_plugins.opf_helper import DEBUG_OPF_HELPER, debug_print
except Exception:
    DEBUG_OPF_HELPER = False
    def debug_print(*args, **kwargs):
        pass
import zipfile

# Use the common icon handling from common_icons
from calibre_plugins.opf_helper.common_icons import get_icon

plugin_name = "OPF Helper"

def get_local_resources_dir(subfolder=None):
    '''Returns path to local resources directory, optionally with subfolder'''
    resources_dir = os.path.join(config_dir, 'plugins', plugin_name)
    if subfolder:
        resources_dir = os.path.join(resources_dir, subfolder)
    if iswindows:
        resources_dir = os.path.normpath(resources_dir)
    return resources_dir

def extract_documentation_from_zip():
    '''Extract documentation resources (PDF + curated JSON) to config dir (Portable-safe).'''
    try:
        # Setup destination paths (must live under config_dir/plugins/OPF_Helper)
        plugin_config_dir = os.path.join(config_dir, 'plugins', plugin_name)
        # Store resources directly under config_dir/plugins/OPF_Helper (no nested opf_helper_docs)
        docs_dir = plugin_config_dir
        pdf_path = os.path.join(docs_dir, 'opf_evolution_diagram.pdf')
        curated_json_path = os.path.join(docs_dir, 'common_opf_warnings.json')

        debug_print(f"OPF Helper: Config dir is: {config_dir}")
        debug_print(f"OPF Helper: Plugin config dir will be: {plugin_config_dir}")
        debug_print(f"OPF Helper: Docs dir will be: {docs_dir}")

        # If resources already exist, avoid overwriting to preserve timestamps.
        # However, if the curated JSON exists but is invalid/corrupted, repair it.
        pdf_exists = os.path.exists(pdf_path)
        curated_exists = os.path.exists(curated_json_path)
        if curated_exists:
            try:
                with open(curated_json_path, 'r', encoding='utf-8') as f:
                    json.load(f)
            except Exception:
                debug_print(f"OPF Helper: Curated JSON at {curated_json_path} is invalid; will repair")
                curated_exists = False
        if pdf_exists and curated_exists:
            debug_print(f"OPF Helper: Resources already present at {docs_dir}, skipping extraction")
            return True

        # Create plugin config directory only if we need to write resources
        os.makedirs(plugin_config_dir, exist_ok=True)
        debug_print(f"OPF Helper: Created/verified plugin config directory: {plugin_config_dir}")

        # Get plugin path
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        debug_print(f"OPF Helper: Plugin dir (__file__ parent) is: {plugin_dir}")

        # Try direct copy from plugin installation directory first (for unzipped dev mode)
        # Sources inside the plugin package still live under opf_helper_docs
        pdf_source = os.path.join(plugin_dir, 'opf_helper_docs', 'opf_evolution_diagram.pdf')
        curated_source = os.path.join(plugin_dir, 'opf_helper_docs', 'common_opf_warnings.json')
        debug_print(f"OPF Helper: Checking for PDF at: {pdf_source}")
        copied_any = False
        if (not pdf_exists) and os.path.exists(pdf_source):
            debug_print(f"OPF Helper: Found PDF in dev tree at {pdf_source}")
            import shutil
            shutil.copy2(pdf_source, pdf_path)
            debug_print(f"OPF Helper: Copied PDF to {pdf_path}")
            copied_any = True
        def _clean_and_write_json(src_bytes, dst_path):
            try:
                txt = src_bytes.decode('utf-8') if isinstance(src_bytes, (bytes, bytearray)) else str(src_bytes)
            except Exception:
                txt = str(src_bytes)
            items = None
            try:
                data = json.loads(txt)
                if isinstance(data, list):
                    items = data
                elif isinstance(data, dict) and 'items' in data:
                    items = data.get('items') or []
                elif isinstance(data, dict):
                    # Single dict -> wrap
                    items = [data]
            except Exception:
                # Attempt to recover concatenated or malformed JSON by extracting balanced objects
                objs = []
                stack = 0
                start = None
                for i, ch in enumerate(txt):
                    if ch == '{':
                        if stack == 0:
                            start = i
                        stack += 1
                    elif ch == '}':
                        stack -= 1
                        if stack == 0 and start is not None:
                            block = txt[start:i+1]
                            try:
                                objs.append(json.loads(block))
                            except Exception:
                                pass
                            start = None
                if objs:
                    items = objs
            if not items:
                debug_print(f"OPF Helper: Could not parse curated JSON for cleanup; skipping write to {dst_path}")
                return False

            # Normalize items
            norm = []
            seen = set()
            for it in items:
                try:
                    if not isinstance(it, dict):
                        continue
                    key = (it.get('key') or it.get('id') or '').strip()
                    title = (it.get('title') or '').strip()
                    explanation = (it.get('explanation') or '').strip()
                    suggestion = (it.get('suggestion') or '').strip()
                    tags = it.get('tags') or []
                    if not isinstance(tags, list):
                        try:
                            tags = list(tags)
                        except Exception:
                            tags = []
                    if not title and key:
                        if '|' in key:
                            parts = [p.strip() for p in key.split('|') if p.strip()]
                            title = parts[-1] if parts else key
                        else:
                            title = key if len(key) <= 80 else key[:77] + '...'
                    if not key and title:
                        key = title.lower().replace(' ', '-')[0:120]
                    k = key or title
                    if not k or k in seen:
                        continue
                    seen.add(k)
                    norm.append({'key': key, 'title': title or key, 'explanation': explanation, 'suggestion': suggestion, 'tags': tags})
                except Exception:
                    continue

            try:
                with open(dst_path, 'w', encoding='utf-8') as out_f:
                    json.dump(norm, out_f, ensure_ascii=False, indent=2)
                debug_print(f"OPF Helper: Wrote cleaned curated JSON to {dst_path} ({len(norm)} items)")
                return True
            except Exception as e:
                debug_print(f"OPF Helper: Failed writing cleaned JSON to {dst_path}: {e}")
                return False

        if (not curated_exists) and os.path.exists(curated_source):
            debug_print(f"OPF Helper: Found curated JSON in dev tree at {curated_source}")
            try:
                with open(curated_source, 'rb') as srcf:
                    src_bytes = srcf.read()
                if _clean_and_write_json(src_bytes, curated_json_path):
                    copied_any = True
            except Exception:
                debug_print(f"OPF Helper: Failed to read/clean curated JSON at {curated_source}")
        if copied_any:
            return True

        # Try extracting from plugin ZIP (normal/portable installs)
        # In Portable mode, the plugins are in <CalibrarPortable>/CalibrePortable/Data/settings/plugins/
        # The zip is typically named 'OPF Helper.zip' in the plugins folder.
        zip_candidates = [
            os.path.join(config_dir, 'OPF Helper.zip'),
            os.path.join(os.path.dirname(plugin_dir), 'OPF Helper.zip'),
        ]
        debug_print(f"OPF Helper: Will try zips: {zip_candidates}")
        for zip_path in zip_candidates:
            debug_print(f"OPF Helper: Checking for plugin ZIP at: {zip_path}")
            if os.path.exists(zip_path):
                debug_print(f"OPF Helper: Found plugin ZIP at: {zip_path}")
                try:
                    with zipfile.ZipFile(zip_path, 'r') as zf:
                        # List all entries for debugging
                        all_names = zf.namelist()
                        debug_print(f"OPF Helper: ZIP contains {len(all_names)} files")
                        pdf_in_zip = 'opf_helper_docs/opf_evolution_diagram.pdf'
                        curated_in_zip = 'opf_helper_docs/common_opf_warnings.json'
                        extracted_any = False
                        if (not pdf_exists) and pdf_in_zip in all_names:
                            debug_print(f"OPF Helper: Found {pdf_in_zip} in ZIP")
                            with zf.open(pdf_in_zip) as src:
                                with open(pdf_path, 'wb') as dst:
                                    dst.write(src.read())
                            debug_print(f"OPF Helper: Extracted PDF from ZIP to {pdf_path}")
                            extracted_any = True
                        if (not curated_exists) and curated_in_zip in all_names:
                            debug_print(f"OPF Helper: Found {curated_in_zip} in ZIP")
                            with zf.open(curated_in_zip) as src:
                                src_bytes = src.read()
                            if _clean_and_write_json(src_bytes, curated_json_path):
                                debug_print(f"OPF Helper: Extracted and cleaned curated JSON to {curated_json_path}")
                                extracted_any = True
                        if extracted_any:
                            if os.path.exists(pdf_path) or os.path.exists(curated_json_path):
                                debug_print(f"OPF Helper: Verified extracted resources in {docs_dir}")
                                return True
                        else:
                            debug_print(f"OPF Helper: Resources not found in ZIP. Sample entries: {[n for n in all_names[:10]]}")
                except Exception as e:
                    debug_print(f"OPF Helper: Failed to extract PDF from ZIP {zip_path}: {str(e)}")
                    import traceback
                    debug_print(traceback.format_exc())

        debug_print("OPF Helper: Documentation resources not found in any location")
        # It's not a fatal error if the PDF is missing
        return False

    except Exception as e:
        debug_print(f"OPF Helper ERROR: PDF extraction failed - {str(e)}")
        import traceback
        debug_print(traceback.format_exc())
        return False

def ensure_resources():
    '''Ensure documentation resources are in place, but make it optional'''
    try:
        # Extract documentation
        if not extract_documentation_from_zip():
            debug_print("OPF Helper: Failed to copy PDF to config dir")
            return False

        debug_print("OPF Helper: Resources installed successfully")
        return True
    except Exception as e:
        debug_print(f"OPF Helper: Error installing resources - {str(e)}")
        import traceback
        from calibre_plugins.opf_helper import DEBUG_OPF_HELPER
        if DEBUG_OPF_HELPER:
            traceback.print_exc()
        return False

def open_documentation(parent=None):
    '''Open the PDF documentation in system viewer'''
    try:
        # Check all possible paths where the PDF might be located
        pdf_paths = []
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        plugin_config_dir = os.path.join(config_dir, 'plugins', plugin_name)

        # Try common locations
        pdf_paths = [
            # Main plugin directory (legacy inside package)
            os.path.join(plugin_dir, 'opf_helper_docs', 'opf_evolution_diagram.pdf'),
            # Config directory path (resources stored directly under plugins/OPF_Helper)
            os.path.join(plugin_config_dir, 'opf_evolution_diagram.pdf'),
        ]

        # Find the first path that exists
        pdf_path = None
        for path in pdf_paths:
            debug_print(f"OPF Helper: Checking PDF at {path}")
            if os.path.exists(path):
                debug_print(f"OPF Helper: Found PDF at {path}")
                pdf_path = path
                break

        if pdf_path:
            QDesktopServices.openUrl(QUrl.fromLocalFile(pdf_path))
            return True
        else:
            if parent:
                # If the PDF isn't available, offer a web alternative
                from calibre.gui2 import question_dialog
                if question_dialog(parent, 'Documentation Not Found',
                           'The PDF documentation was not found. Would you like to view the online OPF documentation instead?'):
                    QDesktopServices.openUrl(QUrl('https://wiki.mobileread.com/wiki/OPF'))
                    return True
            return False
    except Exception as e:
        debug_print(f"Error opening documentation: {str(e)}")
        import traceback
        traceback.print_exc()
        if parent:
            error_dialog(parent, 'Error Opening Documentation',
                       f'Failed to open documentation: {str(e)}',
                       show=True)
        return False

def create_menu_action_unique(ia, parent_menu, menu_text, image=None, tooltip=None,
                            shortcut=None, triggered=None, is_checked=None):
    '''Create a menu action with unique name for shortcut purposes'''
    if menu_text is not None:
        ac = ia.create_menu_action(parent_menu, menu_text, menu_text,
                                 icon=None, shortcut=shortcut,
                                 description=tooltip, triggered=triggered)
    if image:
        ac.setIcon(get_icon(image))
    if is_checked is not None:
        ac.setCheckable(True)
        if is_checked:
            ac.setChecked(True)
    return ac