# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import os
import sys
import traceback
import shutil
from calibre.customize import InterfaceActionBase
from calibre.constants import config_dir

DEBUG_OPF_HELPER = False

def debug_print(*args, **kwargs):
    """Print debug messages using Calibre's logging system (CCR-style)."""
    if not DEBUG_OPF_HELPER:
        return
    try:
        from calibre.utils.logging import prints
        msg = ' '.join(str(a) for a in args)
        prints(f"[OPF-Helper] {msg}")
    except Exception:
        try:
            from calibre.utils.logging import default_log
            default_log('[OPF-Helper]', *args, **kwargs)
        except Exception:
            pass

# Expose ValidationPanel for importing
from .ValidationPanel import ValidationPanel
__all__ = ['ValidationPanel', 'OPFHelperPlugin']

PLUGIN_NAME = 'OPF Helper'
PLUGIN_VERSION = (2, 0, 1)
PLUGIN_DESCRIPTION = 'A toolkit for inspecting, validating, and managing OPF files inside your files, with advanced search and export features'

class OPFHelperPlugin(InterfaceActionBase):
    name                = PLUGIN_NAME
    description         = PLUGIN_DESCRIPTION
    supported_platforms = ['windows', 'osx', 'linux']
    author             = 'Comfy.n'
    version            = PLUGIN_VERSION
    minimum_calibre_version = (5, 44, 0)
    actual_plugin       = 'calibre_plugins.opf_helper.ShowOPFPlugin:ShowOPFAction'

    def initialize(self):
        '''Initialize the plugin'''


        try:
            # Let parent initialize first
            super().initialize()

            # Get plugin path
            plugin_path = os.path.dirname(os.path.abspath(__file__))

            # Set kindleunpackcore debug flag to match plugin debug
            try:
                import calibre_plugins.opf_helper.kindleunpackcore as kindleunpackcore
                kindleunpackcore.set_debug(DEBUG_OPF_HELPER)
            except Exception:
                pass

            # Import required modules
            from . import schema_utils


            # Install schemas first - required
            if not schema_utils.verify_schemas():
                debug_print('OPFHelper: Installing schemas...')
                if not schema_utils.install_schemas():
                    debug_print('OPFHelper ERROR: Schema installation failed')
                    raise Exception("Failed to install schema files")
                debug_print('OPFHelper: Schema installation successful')
            else:
                debug_print('OPFHelper: Schemas verified successfully')

            # Extract documentation - Optional but restore at init for reliability
            try:
                from .common_utils import ensure_resources
                debug_print('OPF Helper: Starting documentation resource installation')
                if not ensure_resources():
                    debug_print('OPF Helper WARNING: Documentation resource installation failed')
                else:
                    debug_print('OPF Helper: Documentation resources installed successfully')
            except Exception as e:
                debug_print(f'OPF Helper WARNING: Documentation error - {str(e)}')
                import traceback
                debug_print(traceback.format_exc())

            # Register plugin icon resources so Calibre can extract/display them when packaged
            try:
                from calibre.utils.zipfile import ZipFile
                from calibre_plugins.opf_helper.common_icons import set_plugin_icon_resources
                resources = {}
                icon_paths = [
                    'images/qrcode.png',
                    'images/icon-for-dark-theme.png',
                    'images/icon-for-light-theme.png',
                    'images/bulk_report-validator.png',
                    'images/xml_error.png'
                ]
                try:
                    with ZipFile(self.plugin_path, 'r') as zf:
                        for ip in icon_paths:
                            try:
                                with zf.open(ip, 'r') as f:
                                    resources[ip] = f.read()
                            except Exception:
                                # missing icon in zip is non-fatal
                                pass
                except Exception:
                    # If plugin is not a zip (development folder), try reading from filesystem
                    base = os.path.dirname(os.path.abspath(__file__))
                    for ip in icon_paths:
                        p = os.path.join(base, ip.replace('/', os.sep))
                        try:
                            with open(p, 'rb') as f:
                                resources[ip] = f.read()
                        except Exception:
                            pass
                # Register whatever resources we managed to read
                try:
                    set_plugin_icon_resources(PLUGIN_NAME, resources)
                except Exception:
                    pass
            except Exception:
                # Non-fatal - continue even if icons can't be registered
                pass


        except Exception as e:
            debug_print('OPFHelper ERROR: Initialization failed -', str(e))
            debug_print('OPFHelper ERROR stack trace:')
            debug_print(traceback.format_exc())
            raise

    def is_customizable(self):
        return True

    def config_widget(self):
        # Import here to avoid circular import
        from calibre_plugins.opf_helper.config import ConfigWidget
        return ConfigWidget()

    def save_settings(self, config_widget):
        config_widget.save_settings()
