# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from PyQt5.QtWidgets import (QWidget, QVBoxLayout, QTextEdit, QPushButton, QLabel,
                          QHBoxLayout, QSpacerItem, QSizePolicy, QApplication)
from PyQt5.QtGui import QFont, QColor, QTextCursor, QTextCharFormat
from PyQt5.QtCore import Qt, QTimer
from calibre_plugins.opf_helper.common_icons import get_icon
from calibre.utils.config import JSONConfig
from concurrent.futures import ThreadPoolExecutor, as_completed
import threading

class ValidationPanel(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self._validate_callback = None
        self._executor = ThreadPoolExecutor(max_workers=2)
        self._validation_lock = threading.Lock()

        # Load preferences
        self.prefs = JSONConfig('plugins/OPF_Helper')

        # Check if we're in dark mode
        try:
            from calibre.gui2 import is_dark_theme
            self.is_dark = is_dark_theme()
        except:
            self.is_dark = False

        # Define colors based on theme
        if self.is_dark:
            self.success_color = QColor("#90EE90")  # Light green
            self.error_color = QColor("#FF9D88")    # Light red
            self.warning_color = QColor("#FFB366")  # Light orange
            self.info_color = QColor("#88CCFF")     # Light blue
        else:
            self.success_color = QColor("#006400")  # Dark green
            self.error_color = QColor("#8B0000")    # Dark red
            self.warning_color = QColor("#A0522D")  # Brown
            self.info_color = QColor("#000080")     # Navy blue

        self.setup_ui()

    def setup_ui(self):
        layout = QVBoxLayout()
        self.setLayout(layout)

        # Add info label at the top
        info_label = QLabel("OPF validation results will appear here." +
                          (" (Auto-validation enabled)" if self.prefs.get('auto_validate', True) else ""))
        info_label.setWordWrap(True)
        layout.addWidget(info_label)

        # Results text area
        self.results = QTextEdit()
        self.results.setReadOnly(True)
        font = QFont("Courier New", 10)
        self.results.setFont(font)
        layout.addWidget(self.results)

        # Bottom toolbar
        toolbar = QHBoxLayout()

        # Validation button
        self.validate_button = QPushButton("Validate Now")
        self.validate_button.setIcon(get_icon('ok.png'))
        self.validate_button.clicked.connect(self._on_validate_clicked)
        toolbar.addWidget(self.validate_button)

        # Add spacer
        toolbar.addItem(QSpacerItem(20, 10, QSizePolicy.Expanding, QSizePolicy.Minimum))

        # Copy button
        self.copy_button = QPushButton("Copy Results")
        self.copy_button.setIcon(get_icon('edit-copy.png'))
        self.copy_button.clicked.connect(self.copy_to_clipboard)
        toolbar.addWidget(self.copy_button)

        layout.addLayout(toolbar)

    def set_validate_callback(self, callback):
        """Set the validation callback function"""
        self._validate_callback = callback
        # Note: Auto-validation is now handled in load_current_book, not here

    def _on_validate_clicked(self):
        """Internal handler for validate button click"""
        if self._validate_callback:
            self._validate_callback()

    def set_results(self, text):
        """Update validation results with color highlighting"""
        self.results.clear()
        cursor = self.results.textCursor()

        for line in text.split('\n'):
            fmt = QTextCharFormat()

            # Color based on line content
            if line.startswith('✓'):
                fmt.setForeground(self.success_color)
            elif line.startswith('❌'):
                fmt.setForeground(self.error_color)
            elif line.startswith('⚠️'):
                fmt.setForeground(self.warning_color)
            elif line.strip().endswith(':'):
                fmt.setForeground(self.info_color)

            # Add text with format
            cursor.insertText(line, fmt)
            cursor.insertText('\n')

        # Move cursor to start
        cursor.setPosition(0)
        self.results.setTextCursor(cursor)

    def copy_to_clipboard(self):
        """Copy validation results to clipboard"""
        QApplication.clipboard().setText(self.results.toPlainText())

    def validate_batch(self, opf_list):
        """Validate multiple OPF files in parallel.
        
        Args:
            opf_list: List of (book_id, opf_content) tuples to validate
        
        This method uses ThreadPoolExecutor for concurrent validation,
        significantly speeding up bulk operations.
        """
        if not opf_list:
            return {}
        
        results_dict = {}
        
        try:
            # Submit validation tasks for all OPFs
            futures_map = {}
            for book_id, opf_content in opf_list:
                if self._validate_callback:
                    # Submit validation task
                    future = self._executor.submit(
                        self._validate_opf_content,
                        book_id,
                        opf_content
                    )
                    futures_map[future] = book_id
            
            # Collect results as they complete
            for future in as_completed(futures_map):
                book_id = futures_map[future]
                try:
                    validation_result = future.result(timeout=5)
                    results_dict[book_id] = validation_result
                except Exception as e:
                    results_dict[book_id] = "Validation error: {0}".format(str(e))
        
        except Exception as e:
            import traceback
            from calibre_plugins.opf_helper import DEBUG_OPF_HELPER
            if DEBUG_OPF_HELPER:
                traceback.print_exc()
        
        return results_dict

    def _validate_opf_content(self, book_id, opf_content):
        """Internal thread-safe OPF validation worker.
        
        This is called by validate_batch in a thread pool.
        """
        with self._validation_lock:
            if self._validate_callback:
                try:
                    # Call the validation callback and return result
                    return self._validate_callback()
                except Exception as e:
                    return "Error: {0}".format(str(e))
        return None

    def __del__(self):
        """Cleanup executor on widget destruction"""
        try:
            if hasattr(self, '_executor'):
                self._executor.shutdown(wait=False)
        except Exception:
            pass