# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import datetime
import tempfile

import shutil
import os
import weakref


def _opfhelper_db_formats(db, book_id):
	"""Get list of formats for a book (lowercase)."""
	try:
		fmts = db.formats(book_id, index_is_id=True)
	except TypeError:
		fmts = db.formats(book_id)
	if not fmts:
		return []
	if isinstance(fmts, (tuple, list)):
		return [x.lower().strip() for x in fmts if x and isinstance(x, str)]
	return [x.lower().strip() for x in str(fmts).split(',') if x.strip()]


def _opfhelper_preferred_fmt(fmts):
	"""Prefer EPUB, then KEPUB, else first available."""
	for c in ('epub', 'kepub'):
		if c in fmts:
			return c
	return fmts[0] if fmts else None

# NOTE: This module defines some Qt-based dialogs very early in the file.
# Ensure required Qt/Calibre symbols exist before defining editor dialogs.
try:
	from calibre.gui2 import error_dialog, info_dialog
	from calibre.ptempfile import PersistentTemporaryDirectory, PersistentTemporaryFile
	from calibre import as_unicode
except Exception:
	# These will exist inside Calibre; keep import-time failures non-fatal
	# for tooling/py_compile.
	PersistentTemporaryDirectory = None
	PersistentTemporaryFile = None
	def as_unicode(x):
		return str(x)

try:
	from qt.core import Qt, QTextCursor, QTextDocument
	from qt.core import (
		QDialog, QVBoxLayout, QTextEdit, QPushButton, QLabel, QHBoxLayout,
		QPlainTextEdit, QComboBox, QMenu, QAction,
		QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
		QFontMetrics, QIcon, QPalette
	)
except ImportError:
	try:
		from calibre.gui2.qt.widgets import (
			QDialog, QVBoxLayout, QTextEdit, QPushButton, QLabel, QHBoxLayout,
			QPlainTextEdit, QComboBox, QMenu, QAction
		)
		from calibre.gui2.qt.gui import (
			QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
			QFontMetrics, QTextCursor, QTextDocument, QIcon, QPalette
		)
		from calibre.gui2.qt.core import Qt
	except ImportError:
		from PyQt5.QtWidgets import (
			QDialog, QVBoxLayout, QTextEdit, QPushButton, QLabel, QHBoxLayout,
			QPlainTextEdit, QComboBox, QMenu, QAction
		)
		from PyQt5.QtGui import (
			QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
			QFontMetrics, QTextCursor, QTextDocument, QIcon, QPalette
		)
		from PyQt5.QtCore import Qt

try:
	from qt.core import QApplication
except Exception:
	try:
		from PyQt5.QtWidgets import QApplication
	except Exception:
		QApplication = None

class OPFSimpleEditorDialog(QDialog):
	"""OPF editor dialog using Calibre's proper tweak infrastructure.

	This uses the same explode/rebuild mechanism as Calibre's native Unpack Book
	tool to ensure changes are reliably saved back into the library.

	External editor mode intentionally removed (too environment-dependent).
	"""
	def __init__(self, gui, db, book_id, fmt):
		# Make this a true top-level window for taskbar icon separation
		QDialog.__init__(self, None)
		self.gui = gui
		# Store db as weakref like Unpack Book does
		self.db_ref = weakref.ref(db)
		self.book_id = book_id
		self.fmt = fmt.upper()
		self._exploded = None
		self._opf_path = None
		self._cleanup_dirs = []
		self._cleanup_files = []
		self._original_content = ''
		# Preserve the OPF content as loaded when the dialog opened so "Restore Original"
		# always returns to the initially loaded OPF (even after rebuilding).
		self._initial_content = None
		self._cleaned_up = False

		# Ensure apply_btn exists even if _setup_ui fails early
		self.apply_btn = None
		# Track focus transitions for reliable focus toggling
		self._prev_focus_widget = None
		self._last_focus_widget = None

		debug_print(f"[Edit OPF] Initializing editor for book_id={book_id}, fmt={self.fmt}")

		# Get book title for window
		try:
			title = db.title(book_id, index_is_id=True)
			debug_print(f"[Edit OPF] Book title: {title}")
		except Exception as e:
			title = str(book_id)
			debug_print(f"[Edit OPF] Failed to get title: {e}")
		self.setWindowTitle(_('Edit OPF') + f' - {title} ({self.fmt})')
		self.resize(950, 700)
		# Use a distinct icon for the editor window and set as a true window
		try:
			from calibre_plugins.opf_helper.common_icons import get_icon
			self.setWindowIcon(get_icon('images/edit_icon.png'))
		except Exception:
			pass
		try:
			self.setWindowFlag(Qt.Window, True)
		except Exception:
			# Fallback for older Qt: setWindowFlags
			try:
				self.setWindowFlags(self.windowFlags() | Qt.Window)
			except Exception:
				pass
		# Create basic search option actions early to avoid AttributeError
		# if signals fire before _setup_ui completes. They will be reconfigured
		# inside _setup_ui() where prefs and menus are available.
		try:
			self.case_sensitive = QAction('Case sensitive', self)
			self.case_sensitive.setCheckable(True)
			self.case_sensitive.setChecked(False)
			self.case_sensitive.triggered.connect(self._reset_search)
		except Exception:
			self.case_sensitive = None
		try:
			self.whole_words = QAction('Whole words', self)
			self.whole_words.setCheckable(True)
			self.whole_words.setChecked(False)
			self.whole_words.triggered.connect(self._reset_search)
		except Exception:
			self.whole_words = None
		try:
			self.regex_search = QAction('Regex search', self)
			self.regex_search.setCheckable(True)
			self.regex_search.setChecked(False)
			self.regex_search.triggered.connect(self._reset_search)
		except Exception:
			self.regex_search = None

		self._setup_ui()
		self._do_explode()

		self.text.setReadOnly(False)

		# Initialize search state
		self.current_search = ''
		self.search_matches = []  # list of (start, length)
		self.current_match = -1

	@property
	def db(self):
		return self.db_ref()

	def _setup_ui(self):
		layout = QVBoxLayout(self)
		self.setLayout(layout)

		# Header
		self.help = QLabel(_(
			'Edit the OPF below.'
			'<ul style="margin:8px 0 4px 22px; padding:0;">'
			'<li><b>Rebuild &amp; Save</b>: Updates the OPF file inside your book (EPUB, KEPUB, etc). <b>Does not</b> update Calibre’s library metadata (title, author, tags, etc).</li>'
			'<li><b>Set metadata from OPF</b>: Updates Calibre’s library metadata (title, authors, series, identifiers, publisher, tags, language, date, comments) without rebuilding the book file.</li>'
			'</ul>'
		))
		self.help.setWordWrap(True)
		layout.addWidget(self.help)

		# --- Text editor with monospace font and syntax highlighting (must exist before search bar signals) ---
		self.text = QPlainTextEdit(self)
		try:
			font = QFont('Consolas', 11)
			if not font.exactMatch():
				font = QFont('Courier New', 11)
			if not font.exactMatch():
				font = QFont('monospace', 11)
			self.text.setFont(font)
			metrics = QFontMetrics(font)
			try:
				tab_width = 4 * metrics.horizontalAdvance(' ')
			except AttributeError:
				tab_width = 4 * metrics.width(' ')
			self.text.setTabStopDistance(tab_width)
		except Exception:
			pass
		layout.addWidget(self.text, 1)

		# Apply shared scrollbar styling for consistent plugin look
		try:
			self.text.setStyleSheet(_common_scrollbar_stylesheet())
		except Exception:
			pass

		# --- Robust search bar (main dialog style) ---
		search_layout = QHBoxLayout()
		search_layout.setSpacing(5)

		self.search_box = QComboBox()
		self.search_box.setEditable(True)
		self.search_box.setInsertPolicy(QComboBox.InsertAtTop)
		self.search_box.setMaxVisibleItems(12)
		try:
			self.search_box.setToolTip('Find in OPF (XML tab only). Use Options for case/whole-word/regex.')
		except Exception:
			pass
		try:
			le = self.search_box.lineEdit()
			if le is not None:
				le.setPlaceholderText('Find in OPF...')
				le.setClearButtonEnabled(True)
				# Connect after actions are created to avoid early signal issues
				le.textChanged.connect(self.on_search_text_changed)
				le.returnPressed.connect(self.find_next)
				le.editingFinished.connect(self._commit_search_history)
		except Exception:
			pass
		try:
			self.search_box.activated.connect(self._on_search_history_activated)
		except Exception:
			pass
		search_layout.addWidget(self.search_box, 1)

		# Visible mode indicator so the user doesn't need to open the options menu
		self.search_mode_label = QLabel('')
		self.search_mode_label.setAlignment(Qt.AlignCenter)
		self.search_mode_label.setMinimumWidth(70)
		search_layout.addWidget(self.search_mode_label)
		# Create match counter label early so search callbacks can safely update it
		self.match_label = QLabel("")
		self.match_label.setAlignment(Qt.AlignCenter)
		self.match_label.setMinimumWidth(80)
		search_layout.addWidget(self.match_label)
		# Now load history (may trigger signals) after match_label exists
		self._init_search_history_dropdown()
		self.case_sensitive.setCheckable(True)
		self.case_sensitive.setChecked(False)
		self.case_sensitive.triggered.connect(self._reset_search)

		self.whole_words = QAction('Whole words', self)
		self.whole_words.setCheckable(True)
		self.whole_words.setChecked(False)
		self.whole_words.triggered.connect(self._reset_search)

		self.regex_search = QAction('Regex search', self)
		self.regex_search.setCheckable(True)
		self.regex_search.setChecked(False)
		self.regex_search.triggered.connect(self._reset_search)

		options_button = QPushButton()
		options_button.setIcon(QIcon.ic('config.png'))
		options_button.setMaximumWidth(30)
		# Attach menu with search options for better UX
		try:
			menu = QMenu(self)
			menu.addAction(self.case_sensitive)
			menu.addAction(self.whole_words)
			menu.addAction(self.regex_search)
			options_button.setMenu(menu)
			options_button.setToolTip('Search options')
		except Exception:
			pass
		try:
			self._update_search_mode_indicator()
		except Exception:
			pass

		# Add options button to the search layout so it is visible
		try:
			search_layout.addWidget(options_button)
		except Exception:
			pass

		self.prev_button = QPushButton()
		self.prev_button.setIcon(QIcon.ic('arrow-up.png'))
		self.prev_button.setMaximumWidth(30)
		self.prev_button.setToolTip('Find previous match')
		self.prev_button.clicked.connect(self.find_previous)
		self.prev_button.setEnabled(False)
		search_layout.addWidget(self.prev_button)

		self.next_button = QPushButton()
		self.next_button.setIcon(QIcon.ic('arrow-down.png'))
		self.next_button.setMaximumWidth(30)
		self.next_button.setToolTip('Find next match')
		self.next_button.clicked.connect(self.find_next)
		self.next_button.setEnabled(False)
		search_layout.addWidget(self.next_button)

		layout.addLayout(search_layout)

		# Apply syntax highlighting
		try:
			# Use XMLHighlighter for standard comparison tool
			self._highlighter = XMLHighlighter(self.text.document())
		except Exception:
			self._highlighter = None

		# Button row
		btns = QHBoxLayout()
		layout.addLayout(btns)


		# Set metadata button (from edited OPF -> Calibre library metadata)
		self.set_meta_btn = QPushButton(_('Set metadata from OPF'), self)
		self.set_meta_btn.setToolTip(_('Update this book\'s Calibre library metadata from the edited OPF (does not rebuild the format).'))
		self.set_meta_btn.clicked.connect(self._set_metadata_from_current_opf)
		btns.addWidget(self.set_meta_btn)

		btns.addStretch(1)

		# Defensive: always define apply_btn early
		self.apply_btn = QPushButton(_('Rebuild && Save'), self)
		self.apply_btn.setToolTip(_('Rebuild the book and save changes to your library'))
		self.apply_btn.clicked.connect(self._do_rebuild)
		self.apply_btn.setEnabled(False)
		btns.addWidget(self.apply_btn)

		# Restore original OPF button (undo manual edits back to the loaded copy)
		try:
			self.restore_btn = QPushButton(_('Restore Original'), self)
			self.restore_btn.setToolTip(_('Restore the editor contents to the original OPF loaded at dialog open'))
			self.restore_btn.clicked.connect(self._restore_original_opf)
			btns.addWidget(self.restore_btn)
		except Exception:
			self.restore_btn = None

		# Accessibility: focus toggle for keyboard-only users
		try:
			self.toggle_focus_btn = QPushButton(_('Toggle Focus'), self)
			self.toggle_focus_btn.setToolTip(_('Toggle focus between the OPF editor and footer buttons (Alt+F)'))
			try:
				self.toggle_focus_btn.setShortcut('Alt+F')
			except Exception:
				pass
			self.toggle_focus_btn.clicked.connect(self._toggle_focus_between_editor_and_footer)
			btns.addWidget(self.toggle_focus_btn)
		except Exception:
			self.toggle_focus_btn = None

		# Cancel button (must be created before setTabOrder references it)
		self.cancel_btn = QPushButton(_('Cancel'), self)
		self.cancel_btn.clicked.connect(self.reject)
		btns.addWidget(self.cancel_btn)

		# Disable rebuild for MOBI/KFX
		if self.fmt.upper() in ('MOBI', 'KFX'):
			self.apply_btn.setEnabled(False)
			self.apply_btn.setToolTip(_('Rebuild is not supported for MOBI/KFX formats'))

		# Set tab order to match visual order (skip missing optional widgets)
		try:
			tab_chain = [
				self.text,
				self.set_meta_btn,
				getattr(self, 'restore_btn', None),
				getattr(self, 'toggle_focus_btn', None),
				getattr(self, 'apply_btn', None),
				getattr(self, 'cancel_btn', None),
			]
			tab_chain = [w for w in tab_chain if w is not None]
			for a, b in zip(tab_chain, tab_chain[1:]):
				self.setTabOrder(a, b)
		except Exception:
			pass

		# Hook focus tracking (used by Toggle Focus)
		try:
			app = QApplication.instance() if QApplication is not None else None
			if app is not None and hasattr(app, 'focusChanged'):
				app.focusChanged.connect(self._on_app_focus_changed)
		except Exception:
			pass

	def _on_app_focus_changed(self, old, now):
		try:
			self._prev_focus_widget = old
			self._last_focus_widget = now
		except Exception:
			pass

	def _refresh_book_details_ui(self):
		"""Best-effort refresh of Calibre UI after metadata changes."""
		try:
			model = self.gui.library_view.model()
			try:
				row_count = model.rowCount()
			except TypeError:
				try:
					from PyQt5.QtCore import QModelIndex
					row_count = model.rowCount(QModelIndex())
				except Exception:
					row_count = 0
			row_index = -1
			try:
				for r in range(row_count):
					try:
						if model.id(r) == self.book_id:
							row_index = r
							break
					except Exception:
						continue
			except Exception:
				row_index = -1
			try:
				if row_index >= 0:
					model.refresh_ids([self.book_id], current_row=row_index)
				else:
					model.refresh_ids([self.book_id])
			except Exception:
				try:
					model.refresh_ids([self.book_id])
				except Exception:
					pass
		except Exception:
			pass

	def _do_explode(self):
		"""Explode the book using Calibre's tweak tools (like Unpack Book)."""
		from calibre.ebooks.tweak import Error, WorkerError, get_tools

		debug_print(f"[Edit OPF] Starting explode for book_id={self.book_id}, fmt={self.fmt}")

		try:
			tdir = PersistentTemporaryDirectory('_opf_helper_edit')
			debug_print(f"[Edit OPF] Created temp dir via PersistentTemporaryDirectory: {tdir}")
		except Exception as e:
			debug_print(f"[Edit OPF] PersistentTemporaryDirectory failed: {e}, using stdlib tempfile")
			import tempfile
			tdir = tempfile.mkdtemp(prefix='_opf_helper_edit_')
			debug_print(f"[Edit OPF] Created temp dir via stdlib: {tdir}")
		self._cleanup_dirs.append(tdir)

		det_msg = None
		try:
			# Get a temp copy of the format file (same as Unpack Book).
			# Be defensive about older/newer DB signatures (especially on Portable).
			src = None
			debug_print(f"[Edit OPF] Attempting db.format() with index_is_id=True, as_path=True")
			try:
				src = self.db.format(self.book_id, self.fmt, index_is_id=True, as_path=True)
				debug_print(f"[Edit OPF] Got format path via index_is_id=True: {src}")
			except TypeError as e:
				debug_print(f"[Edit OPF] index_is_id=True failed: {e}")
				try:
					src = self.db.format(self.book_id, self.fmt, as_path=True)
					debug_print(f"[Edit OPF] Got format path without index_is_id: {src}")
				except TypeError as e2:
					debug_print(f"[Edit OPF] as_path=True also failed: {e2}")
					src = None

			if not src:
				debug_print(f"[Edit OPF] db.format() returned None, trying format_abspath fallback")
				# Fallback: copy the library file to a temp location.
				try:
					try:
						abspath = self.db.format_abspath(self.book_id, self.fmt, index_is_id=True)
						debug_print(f"[Edit OPF] format_abspath with index_is_id=True: {abspath}")
					except TypeError as e:
						debug_print(f"[Edit OPF] format_abspath index_is_id=True failed: {e}")
						abspath = self.db.format_abspath(self.book_id, self.fmt)
						debug_print(f"[Edit OPF] format_abspath without index_is_id: {abspath}")
				except Exception as e:
					debug_print(f"[Edit OPF] format_abspath failed entirely: {e}")
					abspath = None
				if not abspath or not os.path.exists(abspath):
					debug_print(f"[Edit OPF] No format file found. abspath={abspath}")
					error_dialog(self, _('Format not found'),
						_('Could not retrieve {} format for this book.').format(self.fmt), show=True)
					self.reject()
					return
				debug_print(f"[Edit OPF] Creating temp copy of {abspath}")
				try:
					of = PersistentTemporaryFile('_opf_helper_src.' + self.fmt.lower())
					of.close()
					src = of.name
					debug_print(f"[Edit OPF] Temp file via PersistentTemporaryFile: {src}")
				except Exception as e:
					debug_print(f"[Edit OPF] PersistentTemporaryFile failed: {e}, using stdlib")
					import tempfile
					fd, src = tempfile.mkstemp(prefix='_opf_helper_src_', suffix='.' + self.fmt.lower())
					try:
						os.close(fd)
					except Exception:
						pass
					debug_print(f"[Edit OPF] Temp file via stdlib: {src}")
				self._cleanup_files.append(src)
				shutil.copy2(abspath, src)
				debug_print(f"[Edit OPF] Copied format file to temp: {src}")
			else:
				self._cleanup_files.append(src)
				debug_print(f"[Edit OPF] Using format file from db: {src}")

			# Get the exploder for this format
			debug_print(f"[Edit OPF] Getting exploder for format {self.fmt}")
			exploder = get_tools(self.fmt)[0]
			debug_print(f"[Edit OPF] Exploder: {exploder}")
			debug_print(f"[Edit OPF] Exploding {src} to {tdir}")
			opf_path = exploder(src, tdir, question=self._ask_question)
			debug_print(f"[Edit OPF] Exploded, OPF path: {opf_path}")

		except WorkerError as e:
			det_msg = e.orig_tb
			debug_print(f"[Edit OPF] WorkerError during explode: {det_msg}")
		except Error as e:
			debug_print(f"[Edit OPF] Error during explode: {e}")
			error_dialog(self, _('Failed to explode'),
				(_('Could not explode the %s file.') % self.fmt) + ' ' + as_unicode(e), show=True)
			self.reject()
			return
		except Exception as e:
			import traceback
			det_msg = traceback.format_exc()
			debug_print(f"[Edit OPF] Exception during explode: {e}")
			debug_print(det_msg)

		if det_msg is not None:
			debug_print(f"[Edit OPF] Explode failed with detailed error: {det_msg}")
			error_dialog(self, _('Failed to explode'),
				_('Could not explode the %s file. Click "Show details" for more information.') % self.fmt,
				det_msg=det_msg, show=True)
			self.reject()
			return

		if opf_path is None:
			debug_print(f"[Edit OPF] opf_path is None, user may have cancelled")
			# User cancelled a question
			self.reject()
			return

		self._exploded = tdir
		# If multiple OPFs exist, prompt the user to choose which one to edit.
		try:
			opfs = []
			for root, _dirs, files in os.walk(tdir):
				for fn in files:
					if fn.lower().endswith('.opf'):
						opfs.append(os.path.join(root, fn))
			opfs = sorted(set(opfs))
			if len(opfs) > 1:
				display = []
				for p in opfs:
					try:
						display.append(os.path.relpath(p, tdir))
					except Exception:
						display.append(p)
				chooser = OPFChooserDialog(self, display)
				chooser.setWindowTitle(_('Choose OPF File'))
				try:
					chooser.setWindowIcon(self.windowIcon())
				except Exception:
					pass
				if chooser.exec() == QDialog.Accepted:
					sel = chooser.get_selected_opf()
					if sel:
						if not os.path.isabs(sel):
							sel = os.path.join(tdir, sel)
						opf_path = sel
				else:
					# Treat cancel as cancelling the whole editor.
					self.reject()
					return
		except Exception:
			pass
		self._opf_path = opf_path
		debug_print(f"[Edit OPF] Explode successful. OPF at: {opf_path}")
		# Guard: apply_btn may not have been created if _setup_ui failed
		if hasattr(self, 'apply_btn') and self.apply_btn is not None:
			self.apply_btn.setEnabled(True)
		self._load_opf_content()

	def _ask_question(self, msg):
		from calibre.gui2 import question_dialog
		return question_dialog(self, _('Are you sure?'), msg)

	def _load_opf_content(self):
		"""Load OPF content into the editor."""
		if not self._opf_path or not os.path.exists(self._opf_path):
			return
		try:
			with open(self._opf_path, 'rb') as f:
				raw = f.read()
			try:
				txt = raw.decode('utf-8')
			except Exception:
				txt = raw.decode('utf-8', 'replace')
			# Store the originally loaded OPF snapshot (only once) so Restore Original
			# always reverts to the content that was present when the dialog opened.
			if getattr(self, '_initial_content', None) is None:
				self._initial_content = txt
			self._original_content = txt
			self.text.setPlainText(txt)
		except Exception as e:
			error_dialog(self, _('Failed to load OPF'), as_unicode(e), show=True)

	def _do_rebuild(self):
		"""Rebuild the book and save to library using Calibre's tweak tools."""
		from calibre.ebooks.tweak import WorkerError, get_tools

		debug_print(f"[Edit OPF] Starting rebuild for book_id={self.book_id}")

		if not self._exploded:
			debug_print(f"[Edit OPF] Rebuild failed: book was not exploded")
			error_dialog(self, _('Not exploded'), _('Book was not exploded properly.'), show=True)
			return

		current_text = self.text.toPlainText()
		debug_print(f"[Edit OPF] OPF text length: {len(current_text)} chars")

		# Write OPF to the exploded directory
		debug_print(f"[Edit OPF] Writing OPF to {self._opf_path}")
		try:
			with open(self._opf_path, 'wb') as f:
				f.write(current_text.encode('utf-8'))
			debug_print(f"[Edit OPF] OPF written successfully")
		except Exception as e:
			debug_print(f"[Edit OPF] Failed to write OPF: {e}")
			error_dialog(self, _('Failed to write OPF'), as_unicode(e), show=True)
			return

		# Before rebuilding, create a user-accessible backup of the original format
		try:
			import tempfile as _temp
			from datetime import datetime as _dt
			backup_src = None
			try:
				# Try to get absolute path of the original format in library
				try:
					backup_src = self.db.format_abspath(self.book_id, self.fmt, index_is_id=True)
				except TypeError:
					backup_src = self.db.format_abspath(self.book_id, self.fmt)
			except Exception:
				backup_src = None
			if backup_src and os.path.exists(backup_src):
				try:
					ts = _dt.utcnow().strftime('%Y%m%dT%H%M%SZ')
					ext = os.path.splitext(backup_src)[1] or ('.' + self.fmt.lower())
					backup_name = f'opf_helper_backup_{self.book_id}_{self.fmt}_{ts}{ext}'
					backup_dir = _temp.gettempdir()
					backup_path = os.path.join(backup_dir, backup_name)
					shutil.copy2(backup_src, backup_path)
					# Keep user-visible backup (do not auto-cleanup) and inform user
					try:
						info_dialog(self, _('Backup saved'), _('A backup of the original format was saved to:') + f'\n{backup_path}', show=True)
					except Exception:
						debug_print(f"[Edit OPF] Backup saved to {backup_path}")
				except Exception as e:
					debug_print(f"[Edit OPF] Failed to create pre-rebuild backup: {e}")
		except Exception:
			# Non-fatal: continue with rebuild
				pass

		# Create output file using Calibre's temp file infrastructure, with a
		# standard-library fallback for constrained/portable environments.
		out_path = None
		try:
			of = PersistentTemporaryFile('_opf_helper_rebuild.' + self.fmt.lower())
			of.close()
			out_path = of.name
			debug_print(f"[Edit OPF] Rebuild output via PersistentTemporaryFile: {out_path}")
		except Exception as e:
			debug_print(f"[Edit OPF] PersistentTemporaryFile failed: {e}, using stdlib")
			import tempfile
			fd, out_path = tempfile.mkstemp(prefix='_opf_helper_rebuild_', suffix='.' + self.fmt.lower())
			try:
				os.close(fd)
			except Exception:
				pass
			debug_print(f"[Edit OPF] Rebuild output via stdlib: {out_path}")
		self._cleanup_files.append(out_path)

		det_msg = None
		try:
			# Get the rebuilder for this format (same as Unpack Book)
			debug_print(f"[Edit OPF] Getting rebuilder for format {self.fmt}")
			rebuilder = get_tools(self.fmt)[1]
			debug_print(f"[Edit OPF] Rebuilder: {rebuilder}")
			debug_print(f"[Edit OPF] Rebuilding from {self._exploded} to {out_path}")
			rebuilder(self._exploded, out_path)
			debug_print(f"[Edit OPF] Rebuild complete")
		except WorkerError as e:
			det_msg = e.orig_tb
			debug_print(f"[Edit OPF] WorkerError during rebuild: {det_msg}")
		except Exception as e:
			import traceback
			det_msg = traceback.format_exc()
			debug_print(f"[Edit OPF] Exception during rebuild: {e}")
			debug_print(det_msg)

		if det_msg is not None:
			debug_print(f"[Edit OPF] Rebuild failed with detailed error: {det_msg}")
			error_dialog(self, _('Failed to rebuild'),
				_('Failed to rebuild %s. For more information, click "Show details".') % self.fmt,
				det_msg=det_msg, show=True)
			return

		# Add rebuilt format back to library (exactly like Unpack Book does)
		fmt_for_add = os.path.splitext(out_path)[1][1:].upper()
		debug_print(f"[Edit OPF] Adding rebuilt format {fmt_for_add} back to library")
		debug_print(f"[Edit OPF] Rebuilt file size: {os.path.getsize(out_path)} bytes")
		try:
			with open(out_path, 'rb') as f:
				debug_print(f"[Edit OPF] Calling db.add_format with index_is_id=True")
				try:
					self.db.add_format(self.book_id, fmt_for_add, f, index_is_id=True)
					debug_print(f"[Edit OPF] db.add_format succeeded with index_is_id=True")
				except TypeError as e:
					debug_print(f"[Edit OPF] index_is_id=True failed: {e}, trying without")
					f.seek(0)
					self.db.add_format(self.book_id, fmt_for_add, f)
					debug_print(f"[Edit OPF] db.add_format succeeded without index_is_id")
		except Exception as e:
			debug_print(f"[Edit OPF] Failed to update library: {e}")
			import traceback
			debug_print(traceback.format_exc())
			error_dialog(self, _('Failed to update library'), as_unicode(e), show=True)
			return

		debug_print(f"[Edit OPF] OPF edit completed successfully for book_id={self.book_id}")
		try:
			fmt_label = (fmt_for_add or self.fmt or '').upper()
		except Exception:
			fmt_label = (self.fmt or '').upper()
		info_dialog(
			self,
			_('OPF updated'),
			_('Updated the existing {fmt} format file for this book in your Calibre library.\n\n'
			  'No new book was added to the library.\n'
			  'Library metadata (title/authors/tags/etc) is unchanged unless you click “Set metadata from OPF”.')
			.format(fmt=fmt_label or 'book'),
			show=True,
		)
		# Keep the editor open after rebuild so the user can continue to set metadata
		try:
			# Mark apply as no longer needed until further edits
			if hasattr(self, 'apply_btn') and self.apply_btn is not None:
				self.apply_btn.setEnabled(False)
		except Exception:
			pass
		# Update original content snapshot to current text so the dialog reflects saved state
		try:
			self._original_content = self.text.toPlainText()
		except Exception:
			pass

	def reject(self):
		self._cleanup()
		return QDialog.reject(self)

	def _restore_original_opf(self):
		"""Restore the editor contents to the OPF loaded when the dialog opened."""
		try:
			# Prefer the initially loaded snapshot (when dialog opened). Fall back
			# to the last saved/original snapshot if initial isn't available.
			orig = None
			if getattr(self, '_initial_content', None) is not None:
				orig = self._initial_content
			elif getattr(self, '_original_content', None) is not None:
				orig = self._original_content
			if orig is not None:
				# Replace editor contents with the original snapshot
				self.text.setPlainText(orig)
				# Reset document modified state and clear undo/redo if available
				try:
					doc = self.text.document()
					doc.setModified(False)
					# Some Qt versions expose a clearUndoRedoStacks API
					if hasattr(doc, 'clearUndoRedoStacks'):
						doc.clearUndoRedoStacks()
				except Exception:
					pass
				# Disable apply since content now matches original (no further rebuild needed)
				try:
					if hasattr(self, 'apply_btn') and self.apply_btn is not None:
						self.apply_btn.setEnabled(False)
				except Exception:
					pass
		except Exception as e:
			debug_print(f"[Edit OPF] Failed to restore original OPF: {e}")

	def _toggle_focus_between_editor_and_footer(self):
		"""Toggle keyboard focus between the main editor and the footer buttons."""
		try:
			# Determine whether the editor (or its viewport) currently has focus
			fw = self.focusWidget()
			basis = fw
			# Clicking the toggle button changes focus first; use previous focus as basis
			try:
				if getattr(self, 'toggle_focus_btn', None) is not None and fw is self.toggle_focus_btn:
					basis = getattr(self, '_prev_focus_widget', fw)
			except Exception:
				basis = fw
			editor_has_focus = False
			if getattr(self, 'text', None) is not None:
				try:
					if basis is self.text or self.text.hasFocus():
						editor_has_focus = True
					elif hasattr(self.text, 'viewport') and self.text.viewport() is not None and (basis is self.text.viewport() or self.text.viewport().hasFocus()):
						editor_has_focus = True
					else:
						# If basis is a child widget inside the editor, treat as editor focus
						try:
							if basis is not None and hasattr(self.text, 'isAncestorOf') and self.text.isAncestorOf(basis):
								editor_has_focus = True
						except Exception:
							pass
				except Exception:
					editor_has_focus = False

			if editor_has_focus:
				# Focus the first available footer button (preserve visual order)
				for btn_attr in ('set_meta_btn', 'restore_btn', 'toggle_focus_btn', 'apply_btn', 'cancel_btn'):
					btn = getattr(self, btn_attr, None)
					if btn is not None:
							try:
								btn.setFocus()
							except Exception:
								pass
							return
				# If no footer button found, clear focus from editor
				try:
					self.text.clearFocus()
				except Exception:
					pass
				return

			# Otherwise focus the editor
			if getattr(self, 'text', None) is not None:
				self.text.setFocus()
		except Exception as e:
			debug_print(f"[Edit OPF] Focus toggle failed: {e}")

	def accept(self):
		self._cleanup()
		return QDialog.accept(self)

	def closeEvent(self, ev):
		# Ensure temp files are cleaned up for modeless close paths.
		try:
			self._cleanup()
		except Exception:
			pass
		return QDialog.closeEvent(self, ev)

	def _cleanup(self):
		"""Clean up temporary files and directories."""
		if getattr(self, '_cleaned_up', False):
			return
		self._cleaned_up = True
		for f in self._cleanup_files:
			try:
				os.remove(f)
			except Exception:
				pass
		for d in self._cleanup_dirs:
			try:
				shutil.rmtree(str(d))
			except Exception:
				pass

	# ===== Search functionality (from main dialog XML tab) =====

	def _read_search_history(self):
		try:
			from calibre_plugins.opf_helper.config import prefs
			return list(prefs.get('search_history', []) or [])
		except Exception:
			return []

	def _write_search_history(self, hist):
		try:
			from calibre_plugins.opf_helper.config import prefs
			max_n = int(prefs.get('search_history_max', 12) or 12)
			if len(hist) > max_n:
				hist = hist[:max_n]
			prefs['search_history'] = list(hist or [])[:max_n]
		except Exception:
			pass

	def _init_search_history_dropdown(self):
		self._reload_search_history_dropdown()

	def _reload_search_history_dropdown(self):
		try:
			current = self.search_box.currentText()
		except Exception:
			current = ''
		try:
			self.search_box.blockSignals(True)
		except Exception:
			pass
		try:
			self.search_box.clear()
			hist = self._read_search_history()
			for entry in hist:
				self.search_box.addItem(entry)
			# Add special action item at the bottom
			if self.search_box.count() > 0:
				try:
					self.search_box.insertSeparator(self.search_box.count())
				except Exception:
					pass
			# Do NOT add 'Clear history' as a combobox item here
			try:
				self.search_box.setEditText(current)
			except Exception:
				pass
		finally:
			try:
				self.search_box.blockSignals(False)
			except Exception:
				pass

	def _clear_search_history(self, *_args):
		try:
			self._write_search_history([])
			self._reload_search_history_dropdown()
			self.search_box.setEditText('')
			self.on_search_text_changed('')
		except Exception:
			pass

	def _commit_search_history(self):
		try:
			term = (self.search_box.currentText() or '').strip()
			if not term:
				return
			hist = self._read_search_history()
			# Deduplicate (case-sensitive, preserve exact input)
			hist = [h for h in hist if h != term]
			hist.insert(0, term)
			self._write_search_history(hist)
			self._reload_search_history_dropdown()
		except Exception:
			pass

	def _on_search_history_activated(self, index):
		try:
			ud = self.search_box.itemData(index)
			if ud == getattr(self, '_clear_search_history_user_data', None):
				self._write_search_history([])
				self._reload_search_history_dropdown()
				try:
					self.search_box.setEditText('')
				except Exception:
					pass
				self.on_search_text_changed('')
				return
		except Exception:
			pass
		# Normal selection triggers a search refresh
		try:
			self.on_search_text_changed(self.search_box.currentText())
		except Exception:
			pass

	def on_search_text_changed(self, *_args):
		try:
			search_text = self.search_box.currentText()
		except Exception:
			search_text = ''
		if not search_text:
			# Clear search when text is empty
			self.current_search = ""
			self.search_positions = []
			self.current_match = -1
			try:
				if hasattr(self, 'prev_button') and self.prev_button is not None:
					self.prev_button.setEnabled(False)
				if hasattr(self, 'next_button') and self.next_button is not None:
					self.next_button.setEnabled(False)
				if hasattr(self, 'match_label') and self.match_label is not None:
					self.match_label.setText('')
				if hasattr(self, 'text') and self.text is not None:
					self.text.setExtraSelections([])
			except Exception:
				pass
			return
		self.current_search = search_text
		self.search_positions = self.find_matches(search_text)
		self.current_match = -1
		if not self.search_positions:
			try:
				if hasattr(self, 'match_label') and self.match_label is not None:
					self.match_label.setText('No matches')
			except Exception:
				pass
			try:
				if hasattr(self, 'prev_button') and self.prev_button is not None:
					self.prev_button.setEnabled(False)
			except Exception:
				pass
			try:
				if hasattr(self, 'next_button') and self.next_button is not None:
					self.next_button.setEnabled(False)
			except Exception:
				pass
			try:
				if hasattr(self, 'text') and self.text is not None:
					self.text.setExtraSelections([])
			except Exception:
				pass
			return
		self.current_match = 0
		self.jump_to_match()

	def _reset_search(self, *_args):
		try:
			self._update_search_mode_indicator()
		except Exception:
			pass
		try:
			# Recompute matches using current search term and updated options
			self.on_search_text_changed()
		except Exception:
			pass

	def _update_search_mode_indicator(self):
		try:
			lbl = getattr(self, 'search_mode_label', None)
			if lbl is None:
				return
			try:
				is_rx = bool(self.regex_search.isChecked())
			except Exception:
				is_rx = False
			try:
				is_case = bool(self.case_sensitive.isChecked())
			except Exception:
				is_case = False
			try:
				is_whole = bool(self.whole_words.isChecked())
			except Exception:
				is_whole = False
			mode = 'Regex' if is_rx else 'Text'
			flags = []
			if is_case:
				flags.append('Aa')
			if is_whole:
				flags.append('W')
			text = mode if not flags else (mode + ' ' + ' '.join(flags))
			lbl.setText(text)
			lbl.setToolTip('Mode: ' + mode + '\n' +
				('Case sensitive: yes' if is_case else 'Case sensitive: no') + '\n' +
				('Whole words: yes' if is_whole else 'Whole words: no'))
		except Exception:
			pass

	def find_matches(self, search_text):
		"""Find all occurrences of search text using QTextDocument.find for accuracy."""
		if not search_text:
			return []

		doc = self.text.document()

		# Determine option states safely (guards against early signal firing)
		try:
			use_regex = bool(self.regex_search.isChecked())
		except Exception:
			use_regex = False
		try:
			case_sensitive = bool(self.case_sensitive.isChecked())
		except Exception:
			case_sensitive = False
		try:
			whole_words = bool(self.whole_words.isChecked())
		except Exception:
			whole_words = False

		# Handle regex separately with QRegularExpression to ensure position alignment
		if use_regex:
			try:
				try:
					from qt.core import QRegularExpression
				except ImportError:
					try:
						from calibre.gui2.qt.core import QRegularExpression
					except ImportError:
						from PyQt5.QtCore import QRegularExpression
				rx = QRegularExpression(search_text)
				# Pattern options handling differs slightly by binding; use attributes if present
				flags = getattr(QRegularExpression, 'NoPatternOption', 0)
				if not case_sensitive:
					# Some bindings expose PatternOption namespace
					try:
						flags |= QRegularExpression.PatternOption.CaseInsensitiveOption
					except Exception:
						# Fallback: try direct attribute
						try:
							flags |= QRegularExpression.CaseInsensitiveOption
						except Exception:
							pass
				try:
					rx.setPatternOptions(flags)
				except Exception:
					# Older bindings may use setPatternSyntax or similar; ignore if unavailable
					pass
				# Validate pattern if possible
				try:
					if hasattr(rx, 'isValid') and not rx.isValid():
						debug_print(f"OPFHelper: Invalid regex pattern: {search_text} ({getattr(rx, 'errorString', lambda: '')()})")
						return []
				except Exception:
					pass
				positions = []
				# Ensure QTextCursor is available
				try:
					from qt.core import QTextCursor
				except ImportError:
					try:
						from calibre.gui2.qt.core import QTextCursor
					except ImportError:
						from PyQt5.QtGui import QTextCursor
				cursor = QTextCursor(doc)
				cursor.movePosition(QTextCursor.MoveOperation.Start)
				cursor = doc.find(rx, cursor)
				while cursor and not cursor.isNull():
					start = cursor.selectionStart()
					length = cursor.selectionEnd() - start
					if length > 0:
						positions.append((start, length))
					cursor = doc.find(rx, cursor)
				return positions
			except Exception as e:
				debug_print(f"OPFHelper: Regex search failed: {e}")
				return []

		# Plain text search using document.find to keep cursor positions correct
		positions = []
		# Import QTextDocument and QTextCursor robustly
		try:
			from calibre.gui2.qt.core import QTextDocument, QTextCursor
		except Exception:
			try:
				from PyQt5.QtGui import QTextDocument, QTextCursor
			except Exception:
				# If we cannot import QTextCursor, return no positions
				debug_print('OPFHelper: QTextCursor not available for search')
				return []
		# Build flags for QTextDocument.find
		try:
			flags = QTextDocument.FindFlag(0)
			if case_sensitive:
				flags |= QTextDocument.FindFlag.FindCaseSensitively
			if whole_words:
				flags |= QTextDocument.FindFlag.FindWholeWords
		except Exception:
			# Some bindings may expose flags differently; fallback to 0
			flags = 0
		cursor = QTextCursor(doc)
		cursor.movePosition(QTextCursor.MoveOperation.Start)
		try:
			cursor = doc.find(search_text, cursor, flags)
		except Exception:
			# Some overloads expect different arg order; try without flags
			cursor = doc.find(search_text, cursor)
		while cursor and not cursor.isNull():
			positions.append((cursor.selectionStart(), cursor.selectionEnd() - cursor.selectionStart()))
			try:
				cursor = doc.find(search_text, cursor, flags)
			except Exception:
				cursor = doc.find(search_text, cursor)
		return positions

	def find_next(self, *_args):
		if not self.search_positions:
			return
		self.current_match = (self.current_match + 1) % len(self.search_positions)
		self.jump_to_match()

	def find_previous(self, *_args):
		if not self.search_positions:
			return
		self.current_match = (self.current_match - 1) % len(self.search_positions)
		self.jump_to_match()

	def jump_to_match(self):
		if not self.search_positions or self.current_match < 0:
			return
		pos, length = self.search_positions[self.current_match]
		cursor = self.text.textCursor()
		cursor.setPosition(pos)
		cursor.movePosition(QTextCursor.MoveOperation.Right, QTextCursor.MoveMode.KeepAnchor, length)
		self.text.setTextCursor(cursor)
		self.text.ensureCursorVisible()
		self.match_label.setText(f'{self.current_match + 1} / {len(self.search_positions)}')
		try:
			if hasattr(self, 'prev_button') and self.prev_button is not None:
				self.prev_button.setEnabled(True)
			if hasattr(self, 'next_button') and self.next_button is not None:
				self.next_button.setEnabled(True)
		except Exception:
			pass
		# Highlight all matches
		selections = []
		for i, (match_pos, match_len) in enumerate(self.search_positions):
			sel = QTextEdit.ExtraSelection()
			sel.cursor = self.text.textCursor()
			sel.cursor.setPosition(match_pos)
			sel.cursor.movePosition(QTextCursor.MoveOperation.Right, QTextCursor.MoveMode.KeepAnchor, match_len)
			# Use theme-aware colors: highlight for current, lighter for others
			try:
				from PyQt5.QtWidgets import QApplication
				from PyQt5.QtGui import QPalette
				hl = QApplication.palette().color(QPalette.Highlight)
				other = hl.lighter(170)
				if i == self.current_match:
					sel.format.setBackground(hl)
				else:
					sel.format.setBackground(other)
			except Exception:
				if i == self.current_match:
					sel.format.setBackground(QColor(255, 150, 50))
				else:
					sel.format.setBackground(QColor(255, 255, 100))
			selections.append(sel)
		self.text.setExtraSelections(selections)

	# ===== Metadata: Set from edited OPF -> Calibre library =====

	def _metadata_from_opf_text(self, opf_text):
		"""Parse OPF XML text into a Calibre Metadata object (mi)."""
		try:
			from io import BytesIO
			data = opf_text.encode('utf-8', 'replace')
			bio = BytesIO(data)
			try:
				from calibre.ebooks.metadata.opf2 import metadata_from_opf
				mi = metadata_from_opf(bio)
			except Exception:
				# Fallback path for older Calibre builds
				from calibre.ebooks.metadata.opf2 import OPF
				opf = OPF(bio)
				mi = getattr(opf, 'to_book_metadata', None)
				mi = mi() if callable(mi) else getattr(opf, 'mi', None)
			return mi
		except Exception:
			return None

	def _merge_metadata(self, base_mi, opf_mi):
		"""Overlay common fields from opf_mi onto base_mi without wiping unrelated data."""
		if base_mi is None:
			return opf_mi
		if opf_mi is None:
			return base_mi
		mi = base_mi
		try:
			if getattr(opf_mi, 'title', None):
				mi.title = opf_mi.title
			if getattr(opf_mi, 'authors', None):
				mi.authors = list(opf_mi.authors)
			if getattr(opf_mi, 'author_sort', None):
				mi.author_sort = opf_mi.author_sort
			if getattr(opf_mi, 'series', None):
				mi.series = opf_mi.series
			if getattr(opf_mi, 'series_index', None) not in (None, 0):
				mi.series_index = opf_mi.series_index
			if getattr(opf_mi, 'publisher', None):
				mi.publisher = opf_mi.publisher
			if getattr(opf_mi, 'tags', None):
				mi.tags = list(opf_mi.tags)
			if getattr(opf_mi, 'languages', None):
				mi.languages = list(opf_mi.languages)
			if getattr(opf_mi, 'pubdate', None):
				mi.pubdate = opf_mi.pubdate
			if getattr(opf_mi, 'comments', None):
				mi.comments = opf_mi.comments
			# Merge identifiers instead of overwriting
			try:
				ids = getattr(opf_mi, 'identifiers', None) or {}
				if ids:
					base_ids = getattr(mi, 'identifiers', None)
					if base_ids is None:
						mi.identifiers = dict(ids)
					else:
						base_ids.update(ids)
			except Exception:
				pass
		except Exception:
			pass
		return mi

	def _set_metadata_from_current_opf(self, *_args):
		"""Set Calibre library metadata for this book from the edited OPF text."""
		debug_print(f"[Edit OPF] Setting library metadata from edited OPF for book_id={self.book_id}")
		try:
			opf_text = self.text.toPlainText() or ''
			if not opf_text.strip():
				error_dialog(self, _('No OPF'), _('There is no OPF content to read metadata from.'), show=True)
				return

			opf_mi = self._metadata_from_opf_text(opf_text)
			if opf_mi is None:
				error_dialog(self, _('Parse failed'), _('Could not parse metadata from the OPF text.'), show=True)
				return

			# Prefer GUI db for metadata operations
			meta_db = None
			try:
				meta_db = getattr(self.gui, 'current_db', None)
			except Exception:
				meta_db = None
			if meta_db is None:
				meta_db = self.db

			base_mi = None
			try:
				base_mi = meta_db.get_metadata(self.book_id, index_is_id=True)
			except TypeError:
				base_mi = meta_db.get_metadata(self.book_id)
			except Exception:
				base_mi = None

			merged = self._merge_metadata(base_mi, opf_mi)

			# Try common set_metadata signatures across Calibre versions
			set_ok = False
			try:
				meta_db.set_metadata(self.book_id, merged, index_is_id=True)
				set_ok = True
			except TypeError:
				try:
					meta_db.set_metadata(self.book_id, merged)
					set_ok = True
				except Exception:
					set_ok = False
			except Exception:
				set_ok = False

			if not set_ok:
				# Fallback: new_api
				try:
					new_api = getattr(meta_db, 'new_api', None)
					if new_api is not None:
						new_api.set_metadata(self.book_id, merged)
						set_ok = True
				except Exception:
					set_ok = False

			if not set_ok:
				error_dialog(self, _('Failed'), _('Could not set Calibre library metadata for this book.'), show=True)
				return

			# Refresh UI if possible. Pass the current row so Calibre emits
			# `new_bookdisplay_data` and updates the details panel immediately.
			self._refresh_book_details_ui()

			info_dialog(self, _('Metadata set'), _('Calibre library metadata was updated from the edited OPF.'), show=True)
		except Exception as e:
			import traceback
			debug_print(f"[Edit OPF] Failed to set metadata from OPF: {e}")
			debug_print(traceback.format_exc())
			error_dialog(self, _('Failed'), _('Failed to set metadata from OPF.\n\n{}').format(as_unicode(e)), show=True)


def edit_opf_for_book(gui, db, book_id):
	"""Launch the OPF editor for a book using Calibre's tweak infrastructure."""
	from calibre.gui2.dialogs.drm_error import DRMErrorMessage

	debug_print(f"[Edit OPF] edit_opf_for_book called for book_id={book_id}")

	# DRM check
	try:
		if hasattr(db, 'has_drm') and db.has_drm(book_id):
			debug_print(f"[Edit OPF] Book has DRM, aborting")
			DRMErrorMessage(gui).exec()
			return
	except Exception as e:
		debug_print(f"[Edit OPF] DRM check failed: {e}")
		pass

	# Use Calibre's legacy DB API (like Unpack Book does)
	try:
		legacy_db = gui.library_view.model().db
		debug_print(f"[Edit OPF] Using legacy_db from gui.library_view.model().db")
	except Exception as e:
		debug_print(f"[Edit OPF] Failed to get legacy_db: {e}, using provided db")
		legacy_db = db

	# Find tweakable formats
	fmts = _opfhelper_db_formats(legacy_db, book_id)
	debug_print(f"[Edit OPF] Available formats for book_id={book_id}: {fmts}")
	tweakable_fmts = set(fmts).intersection({'epub', 'kepub', 'htmlz', 'azw3'})
	debug_print(f"[Edit OPF] Tweakable formats: {tweakable_fmts}")

	if not tweakable_fmts:
		debug_print(f"[Edit OPF] No tweakable formats found")
		return error_dialog(gui, _('Cannot edit OPF'),
			_('The book must be in EPUB, KEPUB, AZW3, or HTMLZ format to edit.'),
			show=True)

	# Prefer EPUB/KEPUB
	fmt = _opfhelper_preferred_fmt(list(tweakable_fmts))
	debug_print(f"[Edit OPF] Preferred format: {fmt}")

	# Enforce single-instance: if an editor is already open, raise it.
	try:
		open_list = getattr(gui, '_opf_helper_open_edit_dialogs', None)
		existing = None
		if open_list:
			for d in list(open_list):
				try:
					if d is None:
						continue
					if hasattr(d, 'isVisible') and d.isVisible():
						existing = d
						break
				except Exception:
					continue
		if existing is not None:
			try:
				existing.show()
				existing.raise_()
				existing.activateWindow()
			except Exception:
				pass
			try:
				info_dialog(gui, _('Edit OPF already open'),
					_('An Edit OPF window is already open. Close it before opening another.'),
					show=True)
			except Exception:
				pass
			return
	except Exception:
		pass

	edit_dlg = OPFSimpleEditorDialog(gui, legacy_db, book_id, fmt)
	# Modeless: allow switching back to the main Calibre window while editing.
	try:
		edit_dlg.setModal(False)
	except Exception:
		pass
	# Keep a strong reference so the dialog isn't garbage-collected.
	try:
		open_list = getattr(gui, '_opf_helper_open_edit_dialogs', None)
		if open_list is None:
			open_list = []
			setattr(gui, '_opf_helper_open_edit_dialogs', open_list)
		open_list.append(edit_dlg)
		def _forget(_obj=None):
			try:
				if edit_dlg in open_list:
					open_list.remove(edit_dlg)
			except Exception:
				pass
		try:
			edit_dlg.destroyed.connect(_forget)
		except Exception:
			pass
	except Exception:
		pass
	try:
		edit_dlg.show()
		edit_dlg.raise_()
		edit_dlg.activateWindow()
	except Exception:
		# Fallback to modal if show path fails
		edit_dlg.exec()

__license__   = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from io import StringIO
import os
import re
import traceback
import zipfile

from calibre import prints
from calibre.constants import DEBUG
from xml.etree import ElementTree as ET
from lxml import etree
from calibre.constants import numeric_version, DEBUG
from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import error_dialog, info_dialog, gprefs
from calibre.constants import cache_dir

# Import I with fallback for Calibre 5.44 compatibility
try:
    from calibre.gui2 import I
except Exception:
    try:
        from calibre.utils.resources import get_image_path as I
    except Exception:
        I = None
try:
	from calibre_plugins.opf_helper import DEBUG_OPF_HELPER
	def debug_print(*args, **kwargs):
		env_flag = os.environ.get('DEBUG_OPF_HELPER', '').strip().lower()
		env_on = env_flag in ('1', 'true', 'yes', 'on')
		if DEBUG_OPF_HELPER or env_on:
			from calibre.utils.logging import default_log
			default_log(*args, **kwargs)
except Exception:
	def debug_print(*args, **kwargs):
		pass
try:
	from calibre.utils.localization import _
except Exception:
	try:
		_ = __builtins__['_']
	except Exception:
		_ = lambda x: x

from calibre_plugins.opf_helper.common_icons import set_plugin_icon_resources, get_icon, get_pixmap


# Direct actions are created as QActions (not InterfaceAction subclasses) and
# injected into gui.iactions via deferred QTimer in genesis(). This is the
# pattern that works for Calibre's Toolbars & menus preferences.
from calibre_plugins.opf_helper.ValidationPanel import ValidationPanel
from calibre_plugins.opf_helper.cover_display import CoverPanel
from calibre_plugins.opf_helper.widgets import ElidedLabel
from calibre_plugins.opf_helper.schema_utils import (verify_schemas, install_schemas,
												   load_schema, basic_opf_validation,
												   get_schema_parser, SchemaResolver)
from calibre_plugins.opf_helper.config import prefs
from calibre_plugins.opf_helper.opf_comparison_dialog import OPFComparisonDialog

try:
	from calibre.gui2 import open_url
	from calibre.gui2.actions import menu_action_unique_name
	from functools import partial
except ImportError:
	open_url = None

try:
	# Calibre 8+ uses qt.core namespace
	from qt.core import (Qt, QTimer, QUrl, QT_VERSION_STR, QRegularExpression,
						 QDialog, QVBoxLayout, QTextEdit, QPushButton,
						 QApplication, QLabel, QMessageBox, QTreeWidget,
						 QTreeWidgetItem, QSplitter, QLineEdit, QHBoxLayout,
						 QListWidget, QDialogButtonBox, QSpacerItem, QSizePolicy,
						 QWidget, QFrame, QTabWidget, QGroupBox, QGridLayout,
						 QMenu, QAction, QTabBar, QComboBox, QProgressDialog,
						 QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
						 QTextCursor, QTextDocument, QPainter, QPen, QPixmap,
						 QRect, QSize, QIcon, QDesktopServices)
except ImportError:
	try:
		# Calibre 6 uses calibre.gui2.qt namespace
		from calibre.gui2.qt.core import (Qt, QTimer, QUrl, QT_VERSION_STR, QRegularExpression)
		from calibre.gui2.qt.widgets import (QDialog, QVBoxLayout, QTextEdit, QPushButton,
										  QApplication, QLabel, QMessageBox, QTreeWidget,
										  QTreeWidgetItem, QSplitter, QLineEdit, QHBoxLayout,
										  QListWidget, QDialogButtonBox, QSpacerItem, QSizePolicy,
										  QWidget, QFrame, QTabWidget, QGroupBox, QGridLayout,
										  QMenu, QAction, QTabBar, QComboBox, QProgressDialog)
		from calibre.gui2.qt.gui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
									   QTextCursor, QTextDocument, QPainter, QPen, QPixmap,
									   QRect, QSize, QIcon, QDesktopServices)
	except ImportError:
		# Fall back to PyQt5 if running outside Calibre
		from PyQt5.QtWidgets import (QDialog, QVBoxLayout, QTextEdit, QPushButton,
								   QApplication, QLabel, QMessageBox, QTreeWidget,
								   QTreeWidgetItem, QSplitter, QLineEdit, QHBoxLayout,
								   QListWidget, QDialogButtonBox, QSpacerItem, QSizePolicy,
								   QWidget, QFrame, QTabWidget, QGroupBox, QGridLayout,
								   QMenu, QAction, QTabBar, QComboBox, QProgressDialog)
		from PyQt5.QtGui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
							   QTextCursor, QTextDocument, QPainter, QPen, QPixmap,
							   QIcon, QDesktopServices)
		from PyQt5.QtCore import Qt, QT_VERSION_STR, QTimer, QUrl, QRect, QSize, QRegularExpression

# All plugin-bundled icons that are referenced by this plugin. These are loaded
# via InterfaceAction.load_resources so common_icons can always resolve them
# (even when Calibre's get_icons() API changes across versions).
PLUGIN_ICONS = [
	'images/alt_icon-for-light-theme.png',
	'images/arrow-double-left.png',
	'images/arrow-double-right.png',
	'images/arrow-next.png',
	'images/arrow-previous.png',
	'images/bulk_report-validator.png',
	'images/diff_icon.png',
	'images/export_icon.png',
	'images/icon-for-dark-theme.png',
	'images/icon-for-light-theme.png',
	'images/multiple.png',
	'images/qrcode.png',
	'images/search_icon.png',
	'images/xml_error.png',
	'images/zoom_in.png',
	'images/zoom_out.png',
	'images/edit_icon.png',
]


def _common_scrollbar_stylesheet():
	"""Shared scrollbar stylesheet used across plugin dialogs."""
	return (
		"QScrollBar::handle {"
		"   border: 1px solid #5B6985;"
		"}"
		"QScrollBar:vertical {"
		"   background: transparent;"
		"   width: 12px;"
		"   margin: 0px 0px 0px 0px;"
		"   padding: 6px 0px 6px 0px;"
		"}"
		"QScrollBar::handle:vertical {"
		"   background: rgba(140, 172, 204, 0.25);"
		"   min-height: 22px;"
		"   border-radius: 4px;"
		"   margin: 4px 0px;"
		"}"
		"QScrollBar::handle:vertical:hover {"
		"   background: rgba(140, 172, 204, 0.45);"
		"}"
		"QScrollBar:horizontal {"
		"   background: transparent;"
		"   height: 12px;"
		"   margin: 0px 0px 0px 0px;"
		"   padding: 0px 6px 0px 6px;"
		"}"
		"QScrollBar::handle:horizontal {"
		"   background: rgba(140, 172, 204, 0.25);"
		"   min-width: 22px;"
		"   border-radius: 4px;"
		"   margin: 0px 4px;"
		"}"
		"QScrollBar::handle:horizontal:hover {"
		"   background: rgba(140, 172, 204, 0.45);"
		"}"
		"QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,"
		"QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {"
		"   background: none;"
		"}"
	)


def create_menu_action_unique(ia, parent_menu, menu_text, image=None, tooltip=None,
						   shortcut=None, triggered=None, is_checked=None, unique_name=None, shortcut_name=None):
		'''
		Create a menu action with the specified criteria and action, using Calibre's InterfaceAction.create_menu_action
		to ensure shortcut hints are displayed in the menu and in Preferences->Keyboard.
		'''
		from calibre.constants import numeric_version as calibre_version
		from calibre.gui2.actions import menu_action_unique_name
		orig_shortcut = shortcut
		kb = ia.gui.keyboard
		if unique_name is None:
			unique_name = menu_text
		if not shortcut == False:
			full_unique_name = menu_action_unique_name(ia, unique_name)
			if full_unique_name in kb.shortcuts:
				shortcut = False
			else:
				if shortcut is not None and not shortcut == False:
					if isinstance(shortcut, (tuple, list)) and len(shortcut) == 0:
						shortcut = None
					elif isinstance(shortcut, str) and not shortcut:
						shortcut = None
		if shortcut_name is None:
			shortcut_name = menu_text.replace('&','')
		# Ensure shortcut_name is not empty or whitespace
		if not shortcut_name or not shortcut_name.strip():
			shortcut_name = None
			try:
				debug_print(f"[Shortcuts] Skipping empty shortcut_name for menu_text='{menu_text}'")
			except Exception:
				pass
		if calibre_version >= (5,4,0):
			ac = ia.create_menu_action(parent_menu, unique_name, menu_text, icon=None,
									   shortcut=shortcut, description=tooltip,
									   triggered=triggered, shortcut_name=shortcut_name,
									   persist_shortcut=True)
		else:
			ac = ia.create_menu_action(parent_menu, unique_name, menu_text, icon=None,
									   shortcut=shortcut, description=tooltip,
									   triggered=triggered, shortcut_name=shortcut_name)
		if shortcut == False and not orig_shortcut == False:
			if hasattr(ac, 'calibre_shortcut_unique_name') and ac.calibre_shortcut_unique_name in kb.shortcuts:
				kb.replace_action(ac.calibre_shortcut_unique_name, ac)
		if image:
			ac.setIcon(get_icon(image))
		if is_checked is not None:
			ac.setCheckable(True)
			if is_checked:
				ac.setChecked(True)
		parent_menu.addAction(ac)
		return ac

def create_open_cover_with_menu(parent_menu):
	m = QMenu(_('Open cover with...'))
	ac = QAction(_('Choose program...'), parent_menu)
	ac.triggered.connect(parent_menu.choose_open_with)
	m.addAction(ac)
	return m

def copy_to_clipboard(pixmap):
	if not pixmap.isNull():
		QApplication.clipboard().setPixmap(pixmap)

def contextMenuEvent(self, ev):
	cm = QMenu(self)
	copy = cm.addAction(QIcon(get_icon('edit-copy.png')), _('Copy cover'))
	copy.triggered.connect(lambda: copy_to_clipboard(self.pixmap))
	cm.addMenu(create_open_cover_with_menu(self))
	cm.exec(ev.globalPos())

def choose_open_with(self):
	from calibre.gui2.open_with import choose_program
	entry = choose_program('cover_image', self)
	if entry is not None:
		self.open_with(entry)

def open_with(self, entry):
	book_id = self.data.get('id', None)
	if book_id is not None:
		self.open_cover_with.emit(book_id, entry)

if False:
	# This is here to keep my python error checker from complaining about
	# the builtin functions that will be defined by the plugin loading system
	get_icons = get_resources = None

# The minimum Calibre version required (5.44.0)
# Keep this in sync with OPF_Helper.__init__.py
MINIMUM_CALIBRE_VERSION = (5, 44, 0)

class XMLHighlighter(QSyntaxHighlighter):
	def __init__(self, parent=None):
		super().__init__(parent)

		# Use the carefully-tested v1.0.4 color palette, while still being
		# theme-aware (dark vs light).
		is_dark = False
		try:
			from calibre.gui2 import is_dark_theme
			is_dark = bool(is_dark_theme())
		except Exception:
			# Fallback heuristic: treat light Text on dark backgrounds as "dark".
			try:
				palette = QApplication.palette()
				is_dark = palette.color(QPalette.Text).lightness() > 160
			except Exception:
				is_dark = False

		if is_dark:
			tag_color = "#88CCFF"      # Light blue
			attr_color = "#FFB366"     # Light orange
			value_color = "#90EE90"    # Light green
			comment_color = "#999999"  # Gray
		else:
			tag_color = "#000080"      # Navy blue
			attr_color = "#A0522D"     # Brown
			value_color = "#006400"    # Dark green
			comment_color = "#808080"  # Gray

		# XML element format
		tag_format = QTextCharFormat()
		tag_format.setForeground(QColor(tag_color))
		try:
			tag_format.setFontWeight(QFont.Weight.Bold)
		except Exception:
			tag_format.setFontWeight(QFont.Bold)
		self.highlighting_rules = [(r'<[!?]?[a-zA-Z0-9_:-]+|/?>', tag_format)]

		# XML attribute format
		attribute_format = QTextCharFormat()
		attribute_format.setForeground(QColor(attr_color))
		self.highlighting_rules.append((r'\s[a-zA-Z0-9_:-]+(?=\s*=)', attribute_format))

		# XML value format
		value_format = QTextCharFormat()
		value_format.setForeground(QColor(value_color))
		self.highlighting_rules.append((r'"[^"]*"', value_format))

		# Comment format
		comment_format = QTextCharFormat()
		comment_format.setForeground(QColor(comment_color))
		self.highlighting_rules.append((r'<!--[\s\S]*?-->', comment_format))

		# CDATA blocks (styled like comments)
		self.highlighting_rules.append((r'<!\[CDATA\[[\s\S]*?\]\]>', comment_format))

		# Compile regex patterns for better performance
		import re
		self.rules = [(re.compile(pattern), fmt) for pattern, fmt in self.highlighting_rules]

	def highlightBlock(self, text):
		"""Apply syntax highlighting to the given block of text."""
		for pattern, format in self.rules:
			for match in pattern.finditer(text):
				start, length = match.start(), match.end() - match.start()
				self.setFormat(start, length, format)


# Back-compat: the OPF editor dialog expects this name.
class _OPFEditorHighlighter(XMLHighlighter):
	pass

class OPFChooserDialog(QDialog):
	def __init__(self, parent, opf_files):
		QDialog.__init__(self, parent)
		self.setWindowTitle('Choose OPF File')
		self.setMinimumWidth(400)

		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add description
		desc = QLabel("Multiple OPF files found. Please choose one to edit:")
		layout.addWidget(desc)

		# Create list widget
		self.list_widget = QListWidget()
		self.list_widget.addItems(opf_files)
		self.list_widget.setCurrentRow(0)
		layout.addWidget(self.list_widget)

		# Add standard buttons
		button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
		button_box.accepted.connect(self.accept)
		button_box.rejected.connect(self.reject)
		layout.addWidget(button_box)

	def get_selected_opf(self):
		"""Return the selected OPF file path"""
		current_item = self.list_widget.currentItem()
		if current_item:
			return current_item.text()
		return None

class StatsPanel(QWidget):
	def __init__(self, parent=None):
		super().__init__(parent)

		# Check if we're in dark mode
		try:
			from calibre.gui2 import is_dark_theme
			self.is_dark = is_dark_theme()
		except:
			self.is_dark = False

		# Define metadata field colors based on theme
		# Use a less vivid, theme-adaptive orange for 'language' (EPUB 3.0)
		from calibre.gui2 import QApplication
		palette = QApplication.palette()
		if self.is_dark:
			self.metadata_colors = {
				'title': "#fbff17",
				'creator': "#50E3C2",
				'publisher': "#FF9D88",
				'subject': "#90EE90",
				'language': palette.color(palette.Highlight).name(),  # theme-adaptive
				'identifier': "#FF99CC",
				'date': "#B19CD9",
				'description': "#88CCFF"
			}
		else:
			self.metadata_colors = {
				'title': "#8B4513",
				'creator': "#1A3333",
				'publisher': "#8B0000",
				'subject': "#004225",
				'language': palette.color(palette.Highlight).name(),  # theme-adaptive
				'identifier': "#2A0066",
				'date': "#483D8B",
				'description': "#00008B"
			}

		self.setup_ui()

	def setup_ui(self):
		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add stats sections
		general_group = QGroupBox("General Statistics")
		general_layout = QGridLayout()
		general_group.setLayout(general_layout)
		layout.addWidget(general_group)

		self.total_elements = QLabel("Total Elements: 0")
		self.max_depth = QLabel("Maximum Depth: 0")
		self.namespaces = QLabel("Namespaces: 0")

		general_layout.addWidget(self.total_elements, 0, 0)
		general_layout.addWidget(self.max_depth, 1, 0)
		general_layout.addWidget(self.namespaces, 2, 0)

		# Metadata section
		metadata_group = QGroupBox("Metadata Fields")
		metadata_layout = QVBoxLayout()
		metadata_group.setLayout(metadata_layout)
		layout.addWidget(metadata_group)

		self.metadata_tree = QTreeWidget()
		self.metadata_tree.setHeaderLabels(["Field", "Value"])
		self.metadata_tree.setAlternatingRowColors(True)

		# Set up header style
		header = self.metadata_tree.header()
		header.setDefaultAlignment(Qt.AlignLeft)
		header.setStretchLastSection(True)

		metadata_layout.addWidget(self.metadata_tree)

		# Referenced files section
		files_group = QGroupBox("Referenced Files")
		files_layout = QVBoxLayout()
		files_group.setLayout(files_layout)
		layout.addWidget(files_group)

		self.files_list = QTreeWidget()
		self.files_list.setHeaderLabels(["Type", "Path"])
		self.files_list.setAlternatingRowColors(True)
		files_layout.addWidget(self.files_list)

		# Add stretcher at bottom
		layout.addStretch()

	def update_stats(self, root):
		"""Update statistics display for given XML root element"""
		# Reset all stats
		self.metadata_tree.clear()
		self.files_list.clear()

		if root is None:
			return

		# Calculate general stats
		total_elements = 0
		max_depth = 0
		namespaces = set()

		def count_stats(element, depth=0):
			nonlocal total_elements, max_depth
			total_elements += 1
			max_depth = max(max_depth, depth)

			# Track namespaces
			if element.tag and isinstance(element.tag, str) and element.tag.startswith("{"):
				try:
					ns = element.tag[1:].split("}")[0]
					namespaces.add(ns)
				except (IndexError, AttributeError):
					# Skip elements with malformed tags
					pass

			# Recurse through children
			for child in element:
				count_stats(child, depth + 1)

		count_stats(root)

		# Update labels
		self.total_elements.setText(f"Total Elements: {total_elements}")
		self.max_depth.setText(f"Maximum Depth: {max_depth}")
		self.namespaces.setText(f"Namespaces: {len(namespaces)}")

		# Extract and display metadata
		metadata = root.find(".//{http://www.idpf.org/2007/opf}metadata") or root.find(".//metadata")
		if metadata is not None:
			dc_ns = "{http://purl.org/dc/elements/1.1/}"
			# Common DC metadata fields to look for
			dc_fields = ["title", "creator", "publisher", "date", "identifier", "language", "subject"]

			for field in dc_fields:
				# Try with DC namespace first
				elements = metadata.findall(f".//{dc_ns}{field}") or metadata.findall(f".//{field}")
				if elements:
					for elem in elements:
						item = QTreeWidgetItem(self.metadata_tree)
						# Remove namespace from tag if present
						tag = elem.tag
						if isinstance(tag, str) and "}" in tag:
							tag = tag.split("}")[1]
						item.setText(0, tag)
						item.setText(1, elem.text if elem.text else "")

						# Apply color only to the value column if it's a known metadata field
						if tag.lower() in self.metadata_colors:
							color = QColor(self.metadata_colors[tag.lower()])
							item.setForeground(1, color)  # Only color the value column

						# Add any attributes as child items
						if elem.attrib:
							for key, value in elem.attrib.items():
								if isinstance(key, str) and key.startswith("{"):
									key = key.split("}")[1]
								attr_item = QTreeWidgetItem(item)
								attr_item.setText(0, f"@{key}")
								attr_item.setText(1, value)

						# Expand metadata items by default
						item.setExpanded(True)

		# Find referenced files
		manifest = root.find(".//{http://www.idpf.org/2007/opf}manifest") or root.find(".//manifest")
		if manifest is not None:
			media_map = {}  # Group files by media-type
			for item in manifest.findall(".//item"):
				media_type = item.get("media-type", "unknown")
				href = item.get("href", "")
				if media_type not in media_map:
					media_map[media_type] = []
				media_map[media_type].append(href)

			# Add to tree grouped by type
			for media_type, hrefs in media_map.items():
				type_item = QTreeWidgetItem(self.files_list)
				type_item.setText(0, media_type)
				type_item.setText(1, f"{len(hrefs)} file(s)")

				for href in sorted(hrefs):
					file_item = QTreeWidgetItem(type_item)
					file_item.setText(1, href)

		# Expand both trees
		self.metadata_tree.expandAll()
		self.files_list.expandAll()

class OPFContentDialog(QDialog):
	def _ensure_opf_source_labels(self):
		# Add a label below the XML text box in XML tab
		if not hasattr(self, 'opf_source_label'):
			from PyQt5.QtWidgets import QLabel
			self.opf_source_label = QLabel()
			self.opf_source_label.setStyleSheet('font-size: 9pt; color: #888; margin-top: 2px;')
			# Find XML tab widget and add label below text_edit
			try:
				xml_tab_idx = self.tab_widget.indexOf(self.text_edit)
				if xml_tab_idx != -1:
					self.tab_widget.widget(xml_tab_idx).layout().addWidget(self.opf_source_label)
			except Exception:
				pass
		# Add a label below the tree in OPF Tree tab
		if not hasattr(self, 'tree_opf_source_label'):
			from PyQt5.QtWidgets import QLabel
			self.tree_opf_source_label = QLabel()
			self.tree_opf_source_label.setStyleSheet('font-size: 9pt; color: #888; margin-top: 2px;')
			try:
				tree_tab_idx = self.tab_widget.indexOf(self.tree_widget)
				if tree_tab_idx != -1:
					self.tab_widget.widget(tree_tab_idx).layout().addWidget(self.tree_opf_source_label)
			except Exception:
				pass
	def __init__(self, gui, book_ids, db, icon=None):
		# Create the dialog as a top-level window (no parent) so it can be
		# shown independently and appear on the taskbar when minimized.
		QDialog.__init__(self, None)


		self.gui = gui

		# Search state (must exist before any search-related signals fire)
		self.current_search = ''
		self.search_positions = []
		self.current_match = -1

		# Dictionary to store selected OPF file for each book ID
		self.book_opf_selections = {}

		# Store all available OPFs for current book
		self.current_book_opfs = []
		self.current_opf_path = None

		# Performance: Cache parsed OPF content to avoid re-parsing on navigation
		# Key: (book_id, opf_path), Value: parsed XML content
		self.opf_content_cache = {}
		self.max_cache_size = 10  # Keep most recent 10 OPFs in memory

		# Initialize xml_content to prevent attribute errors
		self.xml_content = ""

		# Counter for XML parsing issues
		self.xml_parsing_error_count = 0

		# Load preferences (use the plugin's canonical JSONConfig key)
		from calibre_plugins.opf_helper.config import prefs as _opf_helper_prefs
		self.prefs = _opf_helper_prefs
		self.current_font_size = self.prefs.get('font_size', 10)
		self.font_size_increment = self.prefs.get('font_size_increment', 2)

		# Load configuration settings
		self.show_cover = self.prefs.get('show_cover', True)
		self.show_book_id = self.prefs.get('show_book_id', True)

		# Default tab order configuration - ensure Validation tab is included
		self.tab_order = ['OPF Tree', 'Statistics', 'Validation', 'XML', 'Resources', 'About']

		# Load saved tab order from preferences if available
		self.prefs.defaults['tab_order'] = self.tab_order
		self.tab_order = self.prefs.get('tab_order', self.tab_order)

		# Make sure Validation tab is in the order if missing
		if 'Validation' not in self.tab_order:
			self.tab_order.insert(-1, 'Validation')  # Insert before XML tab

		# Ensure About tab is always included
		if 'About' not in self.tab_order:
			self.tab_order.append('About')

		# Set dialog icon
		opf_source = None
		if icon and not icon.isNull():
			self.setWindowIcon(icon)

		# Ensure normal window controls (minimize/maximize/restore) are available.
		# QDialog defaults can omit these buttons on Windows.
		try:
			flags = self.windowFlags()
			# Prefer the combined hint if present, else set individual hints.
			mm = getattr(Qt, 'WindowMinMaxButtonsHint', None)
			mn = getattr(Qt, 'WindowMinimizeButtonHint', None)
			mx = getattr(Qt, 'WindowMaximizeButtonHint', None)
			sy = getattr(Qt, 'WindowSystemMenuHint', None)
			if mm is not None:
				flags |= mm
			else:
				if mn is not None:
					flags |= mn
				if mx is not None:
					flags |= mx
			if sy is not None:
				flags |= sy
			# Ensure this dialog is a window (top-level) so it can appear on the
			# Windows taskbar when minimized.
			try:
				win_flag = getattr(Qt, 'Window', None)
				if win_flag is not None:
					flags |= win_flag
			except Exception:
				pass
			self.setWindowFlags(flags)
		except Exception:
			pass

		# Store other attributes before registering shortcuts
		self.db = db
		# Use all visible book ids in the current library view for navigation (Calibre 8.x+ compatible)
		# rowCount() in Calibre 8.x requires a parent argument (QModelIndex())
		model = self.gui.library_view.model()
		try:
			row_count = model.rowCount()
		except TypeError:
			# For Calibre 8.x, must pass parent argument
			from PyQt5.QtCore import QModelIndex
			row_count = model.rowCount(QModelIndex())
		self.book_ids = [model.id(row) for row in range(row_count)]
		debug_print(f'OPFHelper: Constructor - populated book_ids with {len(self.book_ids)} books from library view')
		debug_print(f'OPFHelper: Constructor - first 5 book_ids: {self.book_ids[:5] if len(self.book_ids) > 0 else []}')

		# Find the current book index in the full library list
		if book_ids and len(book_ids) > 0:
			current_book_id = book_ids[0]  # The selected book
			try:
				self.current_book_index = self.book_ids.index(current_book_id)
				debug_print(f'OPFHelper: Constructor - set current_book_index to {self.current_book_index} for book_id {current_book_id}')
			except ValueError:
				# Selected book not in library view, start at beginning
				self.current_book_index = 0
				debug_print(f'OPFHelper: Constructor - selected book {current_book_id} not found in library view, starting at index 0')
		else:
			self.current_book_index = 0

		# Set up keyboard shortcuts - these will be overridden by any user customizations
		self.register_default_shortcuts()


		# Create all widgets first (cover, tree, stats, validation, education)
		self.cover_panel = CoverPanel(self)

		self.text_edit = QTextEdit()
		self.text_edit.setReadOnly(True)
		font = QFont("Courier New", self.current_font_size)
		self.text_edit.setFont(font)

		self.tree_widget = QTreeWidget()
		self.tree_widget.setHeaderLabel("XML Structure")
		self.tree_widget.setMinimumWidth(200)
		self.tree_widget.itemClicked.connect(self.on_tree_item_clicked)

		self.stats_panel = StatsPanel()
		self.validation_panel = ValidationPanel(self)
		self.validation_panel.set_validate_callback(self.validate_opf)
		from .education_panel import EducationPanel
		self.education_panel = EducationPanel(self)

		# Create About tab
		self.about_tab = self.setup_about_tab()

		# Now store tab references with more descriptive names
		self.tabs = {
			'OPF Tree': (self.tree_widget, "OPF Tree"),
			'Statistics': (self.stats_panel, "Statistics"),
			'Validation': (self.validation_panel, "Validation"),
			'Resources': (self.education_panel, "Resources"),
			'XML': (self.text_edit, "XML"),
			'About': (self.about_tab, "About")
		}

		# Create custom tab bar subclass for bold+italic selected tabs
		class BoldSelectedTabBar(QTabBar):
			def tabTextColor(self, index):
				if self.currentIndex() == index:
					font = self.font()
					font.setWeight(QFont.Weight.Bold)
					font.setItalic(True)
					old_text = self.tabText(index)
					self.setTabText(index, old_text)  # This triggers font update
					self.setFont(font)
				else:
					font = self.font()
					font.setWeight(QFont.Weight.Normal)
					font.setItalic(False)
					old_text = self.tabText(index)
					self.setTabText(index, old_text)  # This triggers font update
					self.setFont(font)
				return super().tabTextColor(index)

		# Check if we're in dark mode
		try:
			from calibre.gui2 import is_dark_theme
			self.is_dark = is_dark_theme()
		except:
			self.is_dark = False

		# Define element colors based on theme
		if self.is_dark:
			# Bright colors for dark mode
			self.element_colors = {
				'title': "#fbff17",      # Bright yellow
				'creator': "#50E3C2",    # Bright teal
				'publisher': "#FF9D88",   # Bright salmon
				'subject': "#90EE90",    # Light green
				'language': "#FFB366",    # Bright orange
				'identifier': "#FF99CC",  # Bright pink
				'date': "#B19CD9",       # Light purple
				'description': "#88CCFF"  # Bright light blue
			}
		else:
			# Adjusted darker colors for light mode
			self.element_colors = {
				'title': "#8B4513",      # Saddle Brown (darker)
				'creator': "#1A3333",    # Darker slate gray
				'publisher': "#8B0000",   # Dark Red (unchanged)
				'subject': "#004225",    # Darker forest green
				'language': "#A0522D",   # Sienna (unchanged)
				'identifier': "#2A0066",  # Darker indigo
				'date': "#483D8B",       # Dark Slate Blue (unchanged)
				'description': "#00008B"  # Dark Blue (unchanged)
			}

		# Add references dictionary
		self.references = {
			'EPUB 3.0': 'https://www.w3.org/TR/epub-33/',
			'EPUB 2.0': 'http://idpf.org/epub/20/spec/OPF_2.0.1_draft.htm',
			'OPF 3.0 Specification': 'https://www.w3.org/TR/epub-33/#sec-package-doc',
			'Dublin Core Metadata': 'https://www.dublincore.org/specifications/dublin-core/dces/',
			'MobileRead OPF Wiki': 'https://wiki.mobileread.com/wiki/OPF',
		}

		# --- OPF Source Label ---
		self._ensure_opf_source_labels()
		if hasattr(self, 'opf_source_label'):
			self.opf_source_label.setText(f"Source: {opf_source}")
		if hasattr(self, 'tree_opf_source_label'):
			self.tree_opf_source_label.setText(f"Source: {opf_source}")
		debug_print(f'OPFHelper: OPF source: {opf_source}')

		import re
		opf_ver = None
		# Use the initialized xml_content (may be empty) rather than opf_data
		m = re.search(r'<package[^>]*\bversion\s*=\s*"([^\"]+)"', self.xml_content)
		if m:
			opf_ver = m.group(1)
		# Add help text dictionary
		self.opf_help = {
			'package': 'Root element of the OPF file. Contains metadata, manifest, and spine sections.',
			'metadata': 'Contains all book metadata like title, author, etc. Uses Dublin Core elements.',
			'manifest': 'Lists all files (content documents, images, etc.) that are part of the publication.',
			'spine': 'Defines the reading order of content documents.',
			'guide': 'Optional element providing links to key structural components (EPUB 2.0).',
			'title': 'The title of the publication.',
			'creator': 'The primary author or creator of the publication.',
			'contributor': 'Other contributors to the content (e.g., illustrator, editor).',
			'publisher': 'The publisher of the publication.',
			'date': 'Publication date, typically in YYYY[-MM[-DD]] format.',
			'language': 'The language of the content (RFC 3066 language codes).',
			'identifier': 'Unique identifier for the publication (e.g., ISBN, UUID).',
			'subject': 'Subject categories or keywords describing the content.',
			'description': 'A description or summary of the publication.',
			'rights': 'Copyright and licensing information.',
			'item': 'Manifest entry representing a publication resource (content, images, etc).',
			'itemref': 'Spine entry referencing a manifest item, defining reading order.',
			'reference': 'Guide entry pointing to a key structural component.',
			# Attributes
			'id': 'Unique identifier within the OPF file',
			'href': 'Path to the referenced file',
			'media-type': 'MIME type of the referenced file',
			'unique-identifier': 'References the primary identifier in metadata',
			'version': 'EPUB specification version (2.0 or 3.0)',
			'toc': 'References the navigation control file (NCX/Nav)',
			'properties': 'Special features of the referenced item (EPUB 3.0)',
			'fallback': 'Alternative version if primary cannot be rendered',
			'linear': 'Whether item is part of primary reading order (yes/no)'
		}

		# Check Calibre version
		if numeric_version < MINIMUM_CALIBRE_VERSION:
			error_dialog(gui, 'Version Error',
					f'This plugin requires Calibre {MINIMUM_CALIBRE_VERSION[0]}.{MINIMUM_CALIBRE_VERSION[1]}.0 or later.',
					show=True)
			self.close()
			return

		self.gui = gui
		self.db = db
		# book_ids and current_book_index are already set above

		# Initialize schema parsers
		self.schema_parsers = {}
		self.initialize_schemas()

		# Initialize UI first
		layout = QVBoxLayout()
		# Slightly reduced spacing and margins to make better use of vertical space
		layout.setSpacing(4)
		layout.setContentsMargins(4, 4, 4, 4)
		self.setLayout(layout)

		# Top toolbar with book navigation
		nav_layout = QHBoxLayout()
		nav_layout.setSpacing(2)  # Reduce spacing
		nav_layout.setContentsMargins(0, 0, 0, 0)  # Remove margins

		# Create navigation buttons with compact design
		self.first_book_button = QPushButton(get_icon('images/arrow-double-left.png'), '', self)
		self.first_book_button.setFixedWidth(32)
		self.first_book_button.setToolTip("First Book")
		self.first_book_button.clicked.connect(self.show_first_book)
		nav_layout.addWidget(self.first_book_button)

		self.prev_book_button = QPushButton(get_icon('images/arrow-previous.png'), '', self)
		self.prev_book_button.setFixedWidth(32)
		self.prev_book_button.setToolTip("Previous Book")
		self.prev_book_button.clicked.connect(lambda: (debug_print('OPFHelper: Previous button clicked'), self.show_previous_book()))
		nav_layout.addWidget(self.prev_book_button)

		# Add title label that elides text in the middle
		self.book_label = ElidedLabel()
		self.book_label.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
		self.book_label.setAlignment(Qt.AlignCenter)
		font = self.book_label.font()
		font.setPointSize(9)
		self.book_label.setFont(font)
		nav_layout.addWidget(self.book_label, 1)  # Give expanding space

		self.next_book_button = QPushButton(get_icon('images/arrow-next.png'), '', self)
		self.next_book_button.setFixedWidth(32)
		self.next_book_button.setToolTip("Next Book")
		self.next_book_button.clicked.connect(lambda: (debug_print('OPFHelper: Next button clicked'), self.show_next_book()))
		nav_layout.addWidget(self.next_book_button)

		self.last_book_button = QPushButton(get_icon('images/arrow-double-right.png'), '', self)
		self.last_book_button.setFixedWidth(32)
		self.last_book_button.setToolTip("Last Book")
		self.last_book_button.clicked.connect(self.show_last_book)
		nav_layout.addWidget(self.last_book_button)

		# Add navigation layout to main layout
		layout.addLayout(nav_layout)

		# Add OPF selector layout - will be shown only when multiple OPFs exist
		opf_selector_layout = QHBoxLayout()
		opf_selector_layout.setSpacing(5)

		# Add label for OPF selector
		opf_label = QLabel("OPF File:")
		opf_selector_layout.addWidget(opf_label)

		# Create OPF selector combo box
		self.opf_selector = QComboBox()
		self.opf_selector.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
		self.opf_selector.setToolTip("Select which OPF file to view")
		self.opf_selector.currentIndexChanged.connect(self.on_opf_selection_changed)
		opf_selector_layout.addWidget(self.opf_selector)

		# Add count label
		self.opf_count_label = QLabel("")
		opf_selector_layout.addWidget(self.opf_count_label)

		# Create container for OPF selector that can be shown/hidden
		self.opf_selector_container = QWidget()
		self.opf_selector_container.setLayout(opf_selector_layout)
		self.opf_selector_container.setVisible(False)  # Initially hidden
		layout.addWidget(self.opf_selector_container)

		# Add book info label right after navbar
		self.book_info_label = QLabel()
		self.book_info_label.setWordWrap(True)
		self.book_info_label.setTextFormat(Qt.RichText)
		self.book_info_label.setAlignment(Qt.AlignCenter)
		# Prevent the book info header from expanding vertically too much on maximize
		try:
			self.book_info_label.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
			self.book_info_label.setMaximumHeight(56)
		except Exception:
			pass
		self.book_info_label.setStyleSheet("""
			QLabel {
				padding: 3px;
				background-color: palette(window);
				border: 1px solid palette(mid);
				border-radius: 3px;
				margin-top: 2px;
				margin-bottom: 2px;
			}
		""")
		layout.addWidget(self.book_info_label)

		# Create toolbar layout
		toolbar_layout = QHBoxLayout()
		toolbar_layout.setSpacing(5)  # Consistent spacing between all buttons

		# Add version label with version-specific colors
		self.version_label = QLabel()
		self.version_label.setStyleSheet("""
			QLabel {
				padding: 2px 8px;
				border-radius: 4px;
				color: white;
				font-weight: bold;
			}
		""")
		toolbar_layout.addWidget(self.version_label)

		# Add all action buttons
		self.add_action_buttons(toolbar_layout)

		layout.addLayout(toolbar_layout)

		# Search section
		search_layout = QHBoxLayout()
		search_layout.setSpacing(5)

		# Add a label for instructions
		info_label = QLabel("Selected book:")
		search_layout.addWidget(info_label)

		# Add search box with history dropdown (CCR-style)
		self.search_box = QComboBox()
		self.search_box.setEditable(True)
		self.search_box.setInsertPolicy(QComboBox.InsertAtTop)
		self.search_box.setMaxVisibleItems(12)
		try:
			self.search_box.setToolTip('Find in OPF (XML tab **only**). Use Options for case/whole-word/regex.')
		except Exception:
			pass
		try:
			le = self.search_box.lineEdit()
			if le is not None:
				le.setPlaceholderText('Find in OPF (XML tab **only**)...')
				le.setClearButtonEnabled(True)
				le.textChanged.connect(self.on_search_text_changed)
				le.returnPressed.connect(self.find_next)
				le.editingFinished.connect(self._commit_search_history)
		except Exception:
			pass
		# When user selects from dropdown, handle special items + search
		try:
			self.search_box.activated.connect(self._on_search_history_activated)
		except Exception:
			pass
		self._init_search_history_dropdown()
		# Let the search box take available horizontal space
		search_layout.addWidget(self.search_box, 1)

		# Add search options
		self.case_sensitive = QAction("Case sensitive", self)
		self.case_sensitive.setCheckable(True)
		self.case_sensitive.setChecked(prefs.get('search_case_sensitive', False))
		self.case_sensitive.triggered.connect(self.reset_search)
		self.case_sensitive.triggered.connect(self.save_search_options)

		self.whole_words = QAction("Whole words", self)
		self.whole_words.setCheckable(True)
		self.whole_words.setChecked(prefs.get('search_whole_words', False))
		self.whole_words.triggered.connect(self.reset_search)
		self.whole_words.triggered.connect(self.save_search_options)

		self.regex_search = QAction("Regex search", self)
		self.regex_search.setCheckable(True)
		self.regex_search.setChecked(prefs.get('search_regex', False))
		self.regex_search.triggered.connect(self.reset_search)
		self.regex_search.triggered.connect(self.save_search_options)

		# Add search options button
		options_button = QPushButton()
		options_button.setIcon(QIcon.ic('config.png'))
		options_button.setMaximumWidth(30)
		options_button.setToolTip("Search options")
		options_menu = QMenu(self)
		options_menu.addAction(self.case_sensitive)
		options_menu.addAction(self.whole_words)
		options_menu.addAction(self.regex_search)
		options_button.setMenu(options_menu)
		search_layout.addWidget(options_button)

		# Visible mode indicator so the user doesn't need to open the options menu
		self.search_mode_label = QLabel('')
		self.search_mode_label.setAlignment(Qt.AlignCenter)
		self.search_mode_label.setMinimumWidth(70)
		search_layout.addWidget(self.search_mode_label)
		try:
			self._update_search_mode_indicator()
		except Exception:
			pass

		# Add counter label to show match info
		self.match_label = QLabel("")
		self.match_label.setAlignment(Qt.AlignCenter)
		self.match_label.setMinimumWidth(80)
		search_layout.addWidget(self.match_label)

		# Add search navigation buttons with proper theme-aware icons
		self.prev_button = QPushButton()
		# Use theme-aware icon directly
		self.prev_button.setIcon(QIcon.ic('arrow-up.png'))
		self.prev_button.setMaximumWidth(30)
		self.prev_button.setToolTip("Find previous match")
		self.prev_button.clicked.connect(self.find_previous)
		self.prev_button.setEnabled(False)
		search_layout.addWidget(self.prev_button)

		self.next_button = QPushButton()
		# Use theme-aware icon directly
		self.next_button.setIcon(QIcon.ic('arrow-down.png'))
		self.next_button.setMaximumWidth(30)
		self.next_button.setToolTip("Find next match")
		self.next_button.clicked.connect(self.find_next)
		self.next_button.setEnabled(False)
		search_layout.addWidget(self.next_button)

		layout.addLayout(search_layout)

		# Create main horizontal splitter
		splitter = QSplitter(Qt.Horizontal)
		self.main_splitter = splitter  # store for future roll-up behavior
		layout.addWidget(splitter)

		# Create tab widget with custom tab bar
		tab_widget = QTabWidget()
		tab_widget.setTabBar(BoldSelectedTabBar())
		tab_widget.setMovable(True)  # Enable manual tab dragging
		tab_widget.tabBar().tabMoved.connect(self.update_tab_order)  # Handle tab moves

		# Apply CCR-style tab styling
		tab_widget.setStyleSheet("""
			QTabWidget::pane {
				border-top: none;
				   margin-left: 8px;
				   margin-right: 8px;
			}
			QTabBar::tab {
				min-width: 124px;
				width: 3px;
				padding-top: 6px;
				padding-bottom: 6px;
				font-size: 9pt;
				background: transparent;
				border-top: 1px solid palette(mid);
				border-left: 1px solid palette(mid);
				border-right: 1px solid palette(mid);
				border-top-left-radius: 8px;
				border-top-right-radius: 8px;
				/*margin-right: 2px;*/
			}
			QTabBar::tab:selected {
				font-weight: bold;
				font-style: normal;
				border-top: 2px solid palette(link);
				border-left: 2px solid palette(link);
				border-right: 2px solid palette(link);
				color: palette(link);
			}
			QTabBar::tab:!selected {
				color: palette(text);
				margin-top: 2px;
			}
			QScrollBar::handle {
			   border: 1px solid #5B6985; /* subtle border */
			}
			QScrollBar:vertical {
				background: transparent;
				width: 12px;
				margin: 0px 0px 0px 0px;
				/* Reduce reserved top/bottom space so the handle can reach closer to the edges */
				padding: 6px 0px 6px 0px;
			}
			QScrollBar::handle:vertical {
				background: rgba(140, 172, 204, 0.25); /* subtle blue, low opacity */
				min-height: 22px; /* smaller handle height per checkpoint */
				border-radius: 4px;
				/* inset the handle more so it doesn't touch the add/sub slot border */
				margin: 4px 0px;
			}
			QScrollBar::handle:vertical:hover {
				background: rgba(140, 172, 204, 0.45);
			}
			QScrollBar:horizontal {
				background: transparent;
				height: 12px;
				margin: 0px 0px 0px 0px;
				/* Reduce reserved left/right space so the handle can reach closer to the edges */
				padding: 0px 6px 0px 6px;
			}
			QScrollBar::handle:horizontal {
				background: rgba(140, 172, 204, 0.25);
				min-width: 22px; /* smaller handle width per checkpoint */
				border-radius: 4px;
				margin: 0px 4px;
			}
			QScrollBar::handle:horizontal:hover {
				background: rgba(140, 172, 204, 0.45);
			}
			QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,
			QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {
				background: none;
			}
		""")



		# Add tab widget and cover panel to splitter
		splitter.addWidget(tab_widget)
		if self.show_cover:
			splitter.addWidget(self.cover_panel)
			# Set initial splitter sizes - favor main content
			splitter.setSizes([750, 250])  # 750px for main content, 250px for cover
		else:
			# Hide cover panel completely
			self.cover_panel.hide()

		# Ensure Resources tab exists in order
		if 'Resources' not in self.tab_order:
			xml_index = self.tab_order.index('XML') if 'XML' in self.tab_order else len(self.tab_order)
			self.tab_order.insert(xml_index, 'Resources')

		# Add tabs in configured order, but ensure About tab is always last
		# Remove About from tab_order if present (it will be added last)
		if 'About' in self.tab_order:
			self.tab_order.remove('About')

		# Add all other tabs in configured order
		for tab_name in self.tab_order:
			if tab_name in self.tabs:
				widget, label = self.tabs[tab_name]
				tab_widget.addTab(widget, label)

		# Always add About tab last
		if 'About' in self.tabs:
			widget, label = self.tabs['About']
			tab_widget.addTab(widget, label)

		# Connect validation button after widget creation
		try:
			self.validation_panel.validate_button.clicked.connect(self.validate_opf)
		except Exception:
			pass

		# Try to wire up toolbar validate button if present
		toolbar_validate_button = None
		for item in toolbar_layout.children():
			if isinstance(item, QPushButton) and item.text() == "Validate OPF":
				toolbar_validate_button = item
				break
		if toolbar_validate_button:
			toolbar_validate_button.clicked.connect(self.validate_opf)

		# Store reference to tab widget for context menu
		self.tab_widget = tab_widget

		# Restore last-opened tab (persisted) or fall back to configured default
		try:
			last_tab = self.prefs.get('last_tab', None)
		except Exception:
			last_tab = None
		default_tab = last_tab or 'OPF Tree'
		for i in range(self.tab_widget.count()):
			if self.tab_widget.tabText(i) == default_tab:
				self.tab_widget.setCurrentIndex(i)
				break

		# Persist tab changes so the dialog reopens on the last-used tab
		try:
			self.tab_widget.currentChanged.connect(lambda idx: self.prefs.__setitem__('last_tab', self.tab_widget.tabText(idx)))
		except Exception:
			pass

		# Enable tab bar context menu
		self.tab_widget.tabBar().setContextMenuPolicy(Qt.CustomContextMenu)
		self.tab_widget.tabBar().customContextMenuRequested.connect(self.show_tab_context_menu)

		# Apply syntax highlighting
		self.highlighter = XMLHighlighter(self.text_edit.document())

		# Initialize search variables
		self.current_search = ""
		self.search_positions = []
		self.current_match = -1

		# Create references section with theme-aware link colors
		ref_group = QGroupBox("Specifications && References")
		ref_layout = QHBoxLayout()
		ref_group.setLayout(ref_layout)
		# Keep the references group compact but allow slightly more room so links aren't clipped
		try:
			ref_group.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
			ref_group.setMaximumHeight(64)
		except Exception:
			pass

		# Get theme-aware link color from palette
		palette = self.palette()
		link_color = palette.color(QPalette.Link)
		link_hex = link_color.name()  # Get as hex string (#RRGGBB)

		for name, url in self.references.items():
			link = QPushButton(name)
			link.setStyleSheet(f"""
			QPushButton {{
				border: none;
				color: {link_hex};
				text-decoration: underline;
				background: transparent;
				text-align: left;
				padding: 0px;
			}}
			QPushButton:hover {{
				color: {link_hex};
				opacity: 0.8;
			}}
		""")
			link.setCursor(Qt.PointingHandCursor)
			def _open_link(checked, u=url):
				try:
					__import__('calibre').gui2.open_url(u)
				except Exception:
					try:
						QDesktopServices.openUrl(QUrl(u))
					except Exception:
						pass
			link.clicked.connect(_open_link)
			ref_layout.addWidget(link)

		ref_layout.addStretch()
		layout.addWidget(ref_group)

		# Add close button at the bottom using QDialogButtonBox like CCR
		button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Close)
		button_box.rejected.connect(self.reject)

		# CCR-styled version label at bottom-left (uses plugin version)
		try:
			from calibre_plugins.opf_helper import PLUGIN_VERSION, PLUGIN_NAME
			ver = '.'.join(str(x) for x in PLUGIN_VERSION)
			ver_text = f"{PLUGIN_NAME} v{ver}"
		except Exception:
			ver_text = 'OPF Helper'

		version_footer = QLabel(ver_text)
		version_footer.setObjectName('opf_helper_version_footer')
		# Apply CCR-like compact styling: small, muted, subtle border
		version_footer.setStyleSheet("""
			QLabel#opf_helper_version_footer {
				font-size: 9pt;
				color: palette(mid);
				background: transparent;
				padding: 2px 6px;
				border: 1px solid rgba(0,0,0,0.06);
				border-radius: 4px;
			}
		""")

		bottom_layout = QHBoxLayout()
		bottom_layout.addWidget(version_footer, 0, Qt.AlignLeft)
		bottom_layout.addStretch()
		bottom_layout.addWidget(button_box, 0, Qt.AlignRight)
		layout.addLayout(bottom_layout)

		def _focus_primary_widget_for_tab(_idx=None):
			"""Keep focus out of the filter/search bar.

			Single-key shortcuts won't work reliably if a text input has focus.
			"""
			try:
				idx = self.tab_widget.currentIndex() if hasattr(self, 'tab_widget') else -1
			except Exception:
				idx = -1

			try:
				name = self.tab_widget.tabText(idx) if idx >= 0 else ''
			except Exception:
				name = ''

			# Prefer focusing a non-text-input widget in the active tab.
			candidates = []
			try:
				if name == 'XML':
					candidates = [getattr(self, 'text_edit', None)]
				elif name == 'OPF Tree':
					candidates = [getattr(self, 'metadata_tree', None), getattr(self, 'files_list', None)]
				elif name == 'Statistics':
					candidates = [getattr(self, 'stats_table', None), getattr(self, 'stats_text', None)]
				elif name == 'Validation':
					candidates = [getattr(self, 'validation_results', None), getattr(self, 'validation_text', None)]
				elif name == 'Resources':
					candidates = [getattr(self, 'resources_tree', None), getattr(self, 'resources_list', None)]
				else:
					candidates = [getattr(self, 'tab_widget', None)]
			except Exception:
				candidates = [getattr(self, 'tab_widget', None)]

			for w in candidates:
				try:
					if w is not None:
						w.setFocus()
						return
				except Exception:
					continue

			# Final fallback: focus the dialog itself
			try:
				self.setFocus()
			except Exception:
				pass

		# Ensure focus is moved after the dialog is shown.
		try:
			QTimer.singleShot(0, _focus_primary_widget_for_tab)
		except Exception:
			pass

		# Also move focus when switching tabs (prevents returning to filter/search bar).
		try:
			self.tab_widget.currentChanged.connect(_focus_primary_widget_for_tab)
		except Exception:
			pass

		# Setup context menu for tree items
		self.setup_tree_context_menu()

		# Restore/save geometry using gprefs (keep minimum constraints)
		try:
			geom = gprefs.get('opf_helper_dialog_geometry', None)
			if geom:
				self.restoreGeometry(geom)
			else:
				self.resize(1200, 650)
		except Exception:
			# Fallback to defaults
			self.resize(1200, 650)

		self.setMinimumWidth(900)
		self.setMinimumHeight(500)

		# Now load the first book
		self.load_current_book()

		# Apply optional "stay on top" behavior from preferences
		try:
			stay_on_top = bool(prefs.get('opf_helper_stay_on_top', False))
			if stay_on_top:
				# Set the window to stay above the Calibre main window
				self.setWindowFlag(Qt.WindowStaysOnTopHint, True)
		except Exception:
			pass

	def bring_to_front(self):
		"""Restore if minimized, then raise and focus the dialog."""
		try:
			# Restore if minimized
			if int(self.windowState()) & int(Qt.WindowMinimized):
				self.showNormal()
			self.raise_()
			self.activateWindow()
		except Exception:
			pass

	def toggle_stay_on_top(self):
		"""Toggle the WindowStaysOnTopHint and persist preference."""
		try:
			flags = self.windowFlags()
			is_on_top = bool(int(flags) & int(Qt.WindowStaysOnTopHint))
			self.setWindowFlag(Qt.WindowStaysOnTopHint, not is_on_top)
			# Re-show to apply flag change on all platforms
			self.show()
			self.raise_()
			self.activateWindow()
			try:
				prefs.set('opf_helper_stay_on_top', not is_on_top)
			except Exception:
				pass
		except Exception:
			pass

	def _ask_question(self, msg):
		from calibre.gui2 import question_dialog
		return question_dialog(self, _('Are you sure?'), msg)

	def _get_current_book_id(self):
		try:
			if getattr(self, 'book_ids', None) and getattr(self, 'current_book_index', None) is not None:
				idx = int(self.current_book_index)
				if 0 <= idx < len(self.book_ids):
					return self.book_ids[idx]
		except Exception:
			pass
		return None

	def _get_book_title(self, book_id):
		if book_id is None:
			return None
		try:
			# Typical Calibre DB API
			if hasattr(self.db, 'get_metadata'):
				mi = self.db.get_metadata(book_id, index_is_id=True)
				title = getattr(mi, 'title', None)
				return title
		except Exception:
			pass
		try:
			# Some contexts expose the new API under db.new_api
			na = getattr(self.db, 'new_api', None)
			if na is not None and hasattr(na, 'get_metadata'):
				mi = na.get_metadata(book_id)
				title = getattr(mi, 'title', None)
				return title
		except Exception:
			pass
		return None

	def _format_book_ref(self, book_id=None):
		if book_id is None:
			book_id = self._get_current_book_id()
		if book_id is None:
			return 'Book: (unknown)'
		title = self._get_book_title(book_id)
		if title:
			return f'Book id: {book_id}\nTitle: {title}'
		return f'Book id: {book_id}'

	def _show_question_message(self, window_title, main_text, details=None):
		"""Show a less-obtrusive message with a Question icon.

		Used for recoverable/diagnostic issues (e.g. parse errors) where we want
		to show context without an alarming error icon.
		"""
		try:
			mb = QMessageBox(self)
			mb.setIcon(QMessageBox.Icon.Question)
			mb.setWindowTitle(window_title)
			try:
				mb.setTextFormat(Qt.TextFormat.PlainText)
			except Exception:
				pass
			mb.setText(main_text)
			if details:
				try:
					mb.setDetailedText(details)
				except Exception:
					pass
			mb.setStandardButtons(QMessageBox.StandardButton.Ok)
			mb.exec()
			return
		except Exception:
			# Last-resort fallback
			return error_dialog(self, window_title, main_text, show=True)

	def set_books(self, book_ids, db=None):
		"""Update this dialog to point at a new selection.

		This allows the OPF Helper window to remain modeless while still
		handling repeated invocations (e.g. user selects a different book and
		re-runs the action).
		"""
		try:
			if db is not None:
				self.db = db

			# Recompute visible ids from the current library view for navigation
			model = self.gui.library_view.model()
			try:
				row_count = model.rowCount()
			except TypeError:
				from PyQt5.QtCore import QModelIndex
				row_count = model.rowCount(QModelIndex())
			self.book_ids = [model.id(row) for row in range(row_count)]

			# Move focus to the first selected book if possible
			if book_ids and len(book_ids) > 0:
				current_book_id = book_ids[0]
				try:
					self.current_book_index = self.book_ids.index(current_book_id)
				except ValueError:
					self.current_book_index = 0
			else:
				self.current_book_index = 0

			self.load_current_book()
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to update dialog selection: {str(e)}')

	def setup_about_tab(self):
		"""Create the About tab with plugin information"""
		tab = QWidget()
		layout = QHBoxLayout(tab)  # Horizontal layout for columns

		# ===== COLUMN 1: Plugin Description =====
		desc_column = QWidget()
		desc_layout = QVBoxLayout(desc_column)

		# Header
		desc_title = QLabel(_('<b>About OPF Helper</b>'))
		desc_title.setAlignment(Qt.AlignHCenter)
		desc_title.setStyleSheet('font-size: 13pt;')
		desc_layout.addWidget(desc_title)

		# Plugin description
		desc_text = _(
			"<p>OPF Helper is a toolkit for inspecting, scanning, and managing OPF (Open Packaging Format) data inside your book formats.</p>"
			"<p>It provides a nifty OPF viewer with a tree view, statistics, validation tools, as well as EPUB, MOBI and AZW3 version scanners for finding and marking books across your library or current selection.</p>"
		)
		desc_label = QLabel(desc_text)
		desc_label.setWordWrap(True)
		desc_label.setStyleSheet('font-size: 12pt;')
		desc_layout.addWidget(desc_label)

		# Features list
		features_title = QLabel(_('<b>Key Features:</b>'))
		features_title.setStyleSheet('font-size: 12pt; margin-top: 8px;')
		desc_layout.addWidget(features_title)

		features_text = _("""
• OPF content viewer (tree, XML, resources, cover)<br>
• Search, highlight, and navigation helpers for large OPFs<br>
• OPF validation against bundled schemas<br>
• Find books with multiple OPF files or XML parsing issues<br>
• Export selected books’ OPFs to a folder<br>
• Version scanners (library or selection scope)<br>
• (Experimental) Bulk validator: Validate your library or selected books, mark all books with issues, and save a report.
""")
		features_label = QLabel(features_text)
		features_label.setWordWrap(True)
		features_label.setStyleSheet('font-size: 11pt;')
		desc_layout.addWidget(features_label)

		# Modeless note removed as requested

		desc_layout.addStretch()
		layout.addWidget(desc_column)

		# ===== COLUMN 2: Links and Support =====
		links_column = QWidget()
		links_layout = QVBoxLayout(links_column)

		# Header
		links_title = QLabel(_('<b>Support</b>'))
		links_title.setAlignment(Qt.AlignHCenter)
		links_title.setStyleSheet('font-size: 13.5pt;')
		links_layout.addWidget(links_title)

		# --- COLUMN SIZING will be applied after links are added ---
		# This defers adding the column to the main layout until children are populated

		# Helper to wire QLabel links to calibre.open_url
		def _wire_label(label):
			try:
				label.setOpenExternalLinks(False)
			except Exception:
				pass
			try:
				label.linkActivated.connect(lambda u: __import__('calibre').gui2.open_url(u))
			except Exception:
				try:
					label.linkActivated.connect(lambda u: QDesktopServices.openUrl(QUrl(u)))
				except Exception:
					pass

		# MobileRead forum link
		mr_container = QWidget()
		mr_layout = QHBoxLayout(mr_container)
		mr_img = QLabel()
		mr_icon = get_icon('images/mobileread')  # Try to get icon from common_icons
		if mr_icon and not mr_icon.isNull():
			mr_img.setPixmap(mr_icon.pixmap(24, 24))
		else:
			mr_img.setText("🔗")
		mr_layout.addWidget(mr_img)
		mr_link = QLabel('<a href="https://www.mobileread.com/forums/showthread.php?t=371086">MobileRead Forum Thread</a>')
		_wire_label(mr_link)
		mr_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
		mr_link.setStyleSheet('font-size: 11pt;')
		mr_layout.addWidget(mr_link)
		mr_layout.addStretch()

		# Donate link
		donate_container = QWidget()
		donate_layout = QHBoxLayout(donate_container)
		donate_img = QLabel()
		donate_icon = get_icon('images/donate')  # Try to get icon from common_icons
		donate_layout.addWidget(donate_img)
		donate_link = QLabel('❤️<a href="https://ko-fi.com/comfy_n"> Tip the author</a>')
		_wire_label(donate_link)
		donate_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
		donate_link.setStyleSheet('font-size: 12pt;')
		donate_layout.addWidget(donate_link)
		donate_layout.addStretch()

		# Instead of specification links, show a QR code for quick access/resources.
		# Load qrcode.png via the plugin resource helper so it works packaged or in dev mode.
		qp = None
		try:
			qp = get_pixmap('images/qrcode.png')  # Load QR code image
		except Exception:
			qp = None

		qr_label = QLabel()
		if qp is not None and not qp.isNull():
			qr_label.setPixmap(qp)
			qr_label.setAlignment(Qt.AlignHCenter)
			qr_label.setToolTip("Scan to visit Ko-fi and tip the developer: make my day and support OPF Helper development!")
		else:
			qr_label.setText(_('Resources QR'))
		# Place QR code near the top of the links column and keep remaining space below
		qr_label.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)
		try:
			qr_label.setContentsMargins(48, 14, 0, 0)
		except Exception:
			pass
		# Reorder: donation first, then QR code, then forum link (tip, qr, forum)
		links_layout.addWidget(donate_container)
		links_layout.addWidget(qr_label, alignment=Qt.AlignLeft | Qt.AlignTop)
		links_layout.addWidget(mr_container)
		links_layout.addStretch()

		# Now add the links column to the main layout and apply sizing
		layout.addWidget(links_column)
		layout.setStretch(0, 2)  # First column (desc) is 2x wider
		layout.setStretch(1, 1)  # Second column (links) is 1x
		desc_column.setMinimumWidth(420)  # Optional: ensure minimum width for first column
		links_column.setMaximumWidth(320)  # Optional: prevent links column from growing too wide

		return tab

	def update_tab_order(self, from_index, to_index):
		"""Update the tab order preference when tabs are moved. Debug output included."""
		# Prevent the About tab from being moved away from the last position.
		# Use a suppression flag to avoid handling the programmatic move's signal.
		if getattr(self, '_suppress_tab_move_handler', False):
			return
		try:
			debug_print(f"update_tab_order called: from {from_index} to {to_index}")
		except Exception:
			pass
		# If user moved tabs, ensure About stays last without changing other ordering
		if hasattr(self, 'tab_widget'):
			try:
				texts = [self.tab_widget.tabText(i) for i in range(self.tab_widget.count())]
				if 'About' in texts:
					about_idx = texts.index('About')
					last_idx = self.tab_widget.count() - 1
					if about_idx != last_idx:
						try:
							self._suppress_tab_move_handler = True
							# moveTab exists on QTabBar in Qt5+
							self.tab_widget.tabBar().moveTab(about_idx, last_idx)
						finally:
							self._suppress_tab_move_handler = False
			except Exception:
				pass
		# Optionally persist order here (excluding About) if desired

	def show_tab_context_menu(self, pos):
		"""Show a simple context menu for tabs (select tab)."""
		try:
			tab_bar = self.tab_widget.tabBar()
			global_pos = tab_bar.mapToGlobal(pos)
			menu = QMenu(self)
			for i in range(self.tab_widget.count()):
				name = self.tab_widget.tabText(i)
				a = QAction(name, self)
				a.setData(i)
				# mark current tab
				a.setCheckable(True)
				a.setChecked(i == self.tab_widget.currentIndex())
				# capture index in default arg
				a.triggered.connect(lambda checked, idx=i: self.tab_widget.setCurrentIndex(idx))
				menu.addAction(a)
			menu.exec_(global_pos)
		except Exception as e:
			debug_print(f"show_tab_context_menu error: {e}")

	def next_tab(self):
		"""Switch to the next tab"""
		try:
			idx = self.tab_widget.currentIndex()
			count = self.tab_widget.count()
			self.tab_widget.setCurrentIndex((idx + 1) % count)
		except Exception:
			pass

	def previous_tab(self):
		"""Switch to the previous tab"""
		try:
			idx = self.tab_widget.currentIndex()
			count = self.tab_widget.count()
			self.tab_widget.setCurrentIndex((idx - 1) % count)
		except Exception:
			pass

	def register_default_shortcuts(self):
		"""Register dialog actions with Calibre's keyboard manager.

		All shortcuts are *unassigned by default* to avoid conflicts.
		Users can assign keys in Preferences > Keyboard > OPF Helper.
		"""
		try:
			kb = getattr(self.gui, 'keyboard', None)
			if kb is None:
				return
		except Exception:
			return

		def add(uid, label, description, handler):
			try:
				a = QAction(label, self)
			except Exception:
				return
			try:
				a.setShortcutContext(Qt.ApplicationShortcut)
			except Exception:
				pass
			try:
				a.triggered.connect(handler)
			except Exception:
				try:
					a.triggered.connect(lambda _checked=False, h=handler: h())
				except Exception:
					pass
			try:
				self.addAction(a)
			except Exception:
				pass
			try:
				# Ensure a persistent, assignable entry exists in the keyboard manager.
				# If the UID is already present, bind our QAction; otherwise register
				# an empty entry (action=None) so it appears in Preferences, then
				# replace it with our dialog action so it becomes active.
				if uid in getattr(kb, 'shortcuts', {}):
					kb.replace_action(uid, a)
				else:
					kb.register_shortcut(
						uid,
						label,
						default_keys=(),
						description=description,
						action=None,
						group=_('OPF Helper'),
						persist_shortcut=True
					)
					# Bind the QAction so the shortcut is active when the dialog exists
					try:
						kb.replace_action(uid, a)
					except Exception:
						# Replacement may fail on older calibre versions; ignore
						pass
			except Exception:
				pass

		# Window controls
		add('opf_helper.dialog.stay_on_top', _('OPF Helper: Stay on Top'), _('Toggle always-on-top for the OPF Helper window'), self.toggle_stay_on_top)
		add('opf_helper.dialog.bring_to_front', _('OPF Helper: Bring to Front'), _('Bring OPF Helper window to the front'), self.bring_to_front)

		# Book navigation
		add('opf_helper.dialog.prev_book', _('OPF Helper: Previous Book'), _('Navigate to previous book in library view'), self.show_previous_book)
		add('opf_helper.dialog.next_book', _('OPF Helper: Next Book'), _('Navigate to next book in library view'), self.show_next_book)
		add('opf_helper.dialog.first_book', _('OPF Helper: First Book'), _('Navigate to first book in library view'), self.show_first_book)
		add('opf_helper.dialog.last_book', _('OPF Helper: Last Book'), _('Navigate to last book in library view'), self.show_last_book)

		# Tabs
		add('opf_helper.dialog.next_tab', _('OPF Helper: Next Tab'), _('Switch to next tab'), self.next_tab)
		add('opf_helper.dialog.prev_tab', _('OPF Helper: Previous Tab'), _('Switch to previous tab'), self.previous_tab)

		# Actions
		add('opf_helper.dialog.refresh', _('OPF Helper: Refresh'), _('Reload OPF content for current book'), self.refresh_opf_content)
		add('opf_helper.dialog.export_opf', _('OPF Helper: Export OPF'), _('Export OPF file for current book'), self.export_opf)
		add('opf_helper.dialog.copy_xml', _('OPF Helper: Copy OPF XML'), _('Copy current OPF XML to clipboard'), lambda: self.copy_to_clipboard())
		add('opf_helper.dialog.validate_opf', _('OPF Helper: Validate OPF'), _('Validate OPF (basic checks)'), self.validate_opf)
		add('opf_helper.dialog.edit_metadata', _('OPF Helper: Edit Metadata'), _('Open Edit Metadata dialog'), self.edit_metadata)
		# Bind Edit Book shortcut to the OPF Helper's internal OPF editor (not Calibre's external editor)
		add('opf_helper.dialog.edit_book', _('OPF Helper: Edit Book'), _('Open OPF Helper editor for the current book'), self.open_edit_opf)
		add('opf_helper.dialog.view_book', _('OPF Helper: View Book'), _('Open book in the viewer'), self.view_book)
		add('opf_helper.dialog.edit_toc', _('OPF Helper: Edit ToC'), _('Open Table of Contents editor'), self.edit_toc)
		add('opf_helper.dialog.unpack_book', _('OPF Helper: Unpack Book'), _('Unpack the book to disk'), self.unpack_book)

		# Zoom
		add('opf_helper.dialog.zoom_in', _('OPF Helper: Zoom In'), _('Increase XML font size'), self.zoom_in)
		add('opf_helper.dialog.zoom_out', _('OPF Helper: Zoom Out'), _('Decrease XML font size'), self.zoom_out)

		try:
			kb.finalize()
		except Exception:
			pass

	def _read_search_history(self):
		try:
			return list(prefs.get('search_history', []) or [])
		except Exception:
			return []

	def _write_search_history(self, hist):
		try:
			max_n = int(prefs.get('search_history_max', 12) or 12)
		except Exception:
			max_n = 12
		try:
			prefs['search_history'] = list(hist or [])[:max_n]
		except Exception:
			pass

	def _init_search_history_dropdown(self):
		# Sentinel for the special dropdown action
		try:
			self._clear_search_history_user_data = '__opf_helper_clear_search_history__'
			self._clear_search_history_label = _('Clear history')
		except Exception:
			self._clear_search_history_user_data = '__opf_helper_clear_search_history__'
			self._clear_search_history_label = 'Clear history'
		self._reload_search_history_dropdown()

	def _reload_search_history_dropdown(self):
		try:
			current = self.search_box.currentText()
		except Exception:
			current = ''
		try:
			self.search_box.blockSignals(True)
		except Exception:
			pass
		try:
			self.search_box.clear()
			hist = self._read_search_history()
			for entry in hist:
				self.search_box.addItem(entry)
			# Add special action item at the bottom
			if self.search_box.count() > 0:
				try:
					self.search_box.insertSeparator(self.search_box.count())
				except Exception:
					pass
			self.search_box.addItem(self._clear_search_history_label, self._clear_search_history_user_data)
			# Italicize the special 'Clear history' item
			try:
				m = self.search_box.model()
				last_idx = self.search_box.count() - 1
				itm = None
				try:
					itm = m.item(last_idx)
				except Exception:
					itm = None
				if itm is not None:
					f = itm.font()
					f.setItalic(True)
					itm.setFont(f)
			except Exception:
				pass
			# Restore user's typed text
			try:
				self.search_box.setEditText(current)
			except Exception:
				pass
		finally:
			try:
				self.search_box.blockSignals(False)
			except Exception:
				pass

	def _commit_search_history(self):
		try:
			term = (self.search_box.currentText() or '').strip()
			if not term:
				return
			hist = self._read_search_history()
			# Deduplicate (case-sensitive, preserve exact input)
			hist = [h for h in hist if h != term]
			hist.insert(0, term)
			self._write_search_history(hist)
			self._reload_search_history_dropdown()
		except Exception:
			pass

	def _on_search_history_activated(self, index):
		try:
			ud = self.search_box.itemData(index)
			if ud == getattr(self, '_clear_search_history_user_data', None):
				self._write_search_history([])
				self._reload_search_history_dropdown()
				try:
					self.search_box.setEditText('')
				except Exception:
					pass
				self.on_search_text_changed('')
				return
		except Exception:
			pass
		# Normal selection triggers a search refresh
		try:
			self.on_search_text_changed(self.search_box.currentText())
		except Exception:
			pass

	def find_matches(self, search_text):
		"""Find all occurrences of search text using QTextDocument.find for accuracy."""
		if not search_text:
			return []

		doc = self.text_edit.document()

		# Handle regex separately with QRegularExpression to ensure position alignment
		if self.regex_search.isChecked():
			try:
				rx = QRegularExpression(search_text)
				flags = QRegularExpression.PatternOption.NoPatternOption
				if not self.case_sensitive.isChecked():
					flags |= QRegularExpression.PatternOption.CaseInsensitiveOption
				rx.setPatternOptions(flags)
				if not rx.isValid():
					debug_print(f"OPFHelper: Invalid regex pattern: {search_text} ({rx.errorString()})")
					return []
				positions = []
				cursor = QTextCursor(doc)
				cursor.movePosition(QTextCursor.MoveOperation.Start)
				cursor = doc.find(rx, cursor)
				while cursor and not cursor.isNull():
					start = cursor.selectionStart()
					length = cursor.selectionEnd() - start
					if length > 0:
						positions.append((start, length))
					cursor = doc.find(rx, cursor)
				return positions
			except Exception as e:
				debug_print(f"OPFHelper: Regex search failed: {e}")
				return []

		# Plain text search using document.find to keep cursor positions correct
		positions = []
		flags = QTextDocument.FindFlag(0)
		if self.case_sensitive.isChecked():
			flags |= QTextDocument.FindFlag.FindCaseSensitively
		if self.whole_words.isChecked():
			flags |= QTextDocument.FindFlag.FindWholeWords
		cursor = QTextCursor(doc)
		cursor.movePosition(QTextCursor.MoveOperation.Start)
		cursor = doc.find(search_text, cursor, flags)
		while cursor and not cursor.isNull():
			positions.append((cursor.selectionStart(), cursor.selectionEnd() - cursor.selectionStart()))
			cursor = doc.find(search_text, cursor, flags)

		return positions

	def on_search_text_changed(self, *_args):
		"""Handle search text changes"""
		try:
			search_text = self.search_box.currentText()
		except Exception:
			search_text = ''
		if not search_text:
			# Clear search when text is empty
			self.current_search = ""
			self.search_positions = []
			self.current_match = -1
			try:
				if hasattr(self, 'prev_button') and self.prev_button is not None:
					self.prev_button.setEnabled(False)
			except Exception:
				pass
			try:
				if hasattr(self, 'next_button') and self.next_button is not None:
					self.next_button.setEnabled(False)
			except Exception:
				pass
			try:
				if hasattr(self, 'match_label') and self.match_label is not None:
					self.match_label.setText("")
			except Exception:
				pass
			return

		try:
			# Process search if text has changed or options changed
			if search_text != self.current_search:
				self.current_search = search_text
				self.search_positions = self.find_matches(search_text)
				self.current_match = -1

				# Enable/disable navigation buttons
				has_matches = len(self.search_positions) > 0
				try:
					if hasattr(self, 'prev_button') and self.prev_button is not None:
						self.prev_button.setEnabled(has_matches)
				except Exception:
					pass
				try:
					if hasattr(self, 'next_button') and self.next_button is not None:
						self.next_button.setEnabled(has_matches)
				except Exception:
					pass

				# Update match counter
				if has_matches:
					try:
						if hasattr(self, 'match_label') and self.match_label is not None:
							self.match_label.setText(f"1 of {len(self.search_positions)}")
					except Exception:
						pass
					self.find_next()
				else:
					try:
						if hasattr(self, 'match_label') and self.match_label is not None:
							self.match_label.setText("No matches")
					except Exception:
						pass
		except Exception as e:
			debug_print(f"OPFHelper ERROR: Search error: {e}")
			try:
				if hasattr(self, 'match_label') and self.match_label is not None:
					self.match_label.setText("Error")
			except Exception:
				pass

	def find_next(self):
		"""Find and highlight next match"""
		if not self.search_positions:
			return

		self.current_match = (self.current_match + 1) % len(self.search_positions)
		self.highlight_match(self.search_positions[self.current_match])
		# Update match counter
		try:
			if hasattr(self, 'match_label') and self.match_label is not None:
				self.match_label.setText(f"{self.current_match + 1} of {len(self.search_positions)}")
		except Exception:
			pass

	def find_previous(self):
		"""Find and highlight previous match"""
		if not self.search_positions:
			return

		self.current_match = (self.current_match - 1) % len(self.search_positions)
		self.highlight_match(self.search_positions[self.current_match])
		# Update match counter
		try:
			if hasattr(self, 'match_label') and self.match_label is not None:
				self.match_label.setText(f"{self.current_match + 1} of {len(self.search_positions)}")
		except Exception:
			pass

	def reset_search(self):
		"""Reset search when options change"""
		try:
			self._update_search_mode_indicator()
		except Exception:
			pass
		try:
			current = self.search_box.currentText()
		except Exception:
			current = ''
		if current:
			self.on_search_text_changed()

	def save_search_options(self):
		"""Save the current search option states to preferences"""
		prefs['search_case_sensitive'] = self.case_sensitive.isChecked()
		prefs['search_whole_words'] = self.whole_words.isChecked()
		prefs['search_regex'] = self.regex_search.isChecked()
		try:
			self._update_search_mode_indicator()
		except Exception:
			pass

	def _update_search_mode_indicator(self):
		try:
			lbl = getattr(self, 'search_mode_label', None)
			if lbl is None:
				return
			is_rx = bool(self.regex_search.isChecked())
			is_case = bool(self.case_sensitive.isChecked())
			is_whole = bool(self.whole_words.isChecked())
			mode = 'Regex' if is_rx else 'Text'
			flags = []
			if is_case:
				flags.append('Aa')
			if is_whole:
				flags.append('W')
			text = mode if not flags else (mode + ' ' + ' '.join(flags))
			lbl.setText(text)
			lbl.setToolTip('Mode: ' + mode + '\n' +
				('Case sensitive: yes' if is_case else 'Case sensitive: no') + '\n' +
				('Whole words: yes' if is_whole else 'Whole words: no'))
		except Exception:
			pass

	def _refresh_search_after_content_change(self):
		"""Recompute matches after the document content changes.

		When we load a new book or switch OPF files, the search term may stay the
		same, but stored match offsets become invalid.
		"""
		try:
			search_text = (self.search_box.currentText() or '').strip()
		except Exception:
			search_text = ''

		if not search_text:
			self.current_search = ''
			self.search_positions = []
			self.current_match = -1
			try:
				self.text_edit.setExtraSelections([])
			except Exception:
				pass
			try:
				if hasattr(self, 'prev_button') and self.prev_button is not None:
					self.prev_button.setEnabled(False)
			except Exception:
				pass
			try:
				if hasattr(self, 'next_button') and self.next_button is not None:
					self.next_button.setEnabled(False)
			except Exception:
				pass
			try:
				if hasattr(self, 'match_label') and self.match_label is not None:
					self.match_label.setText('')
			except Exception:
				pass
			return

		self.current_search = search_text
		self.search_positions = self.find_matches(search_text)
		self.current_match = -1

		has_matches = len(self.search_positions) > 0
		try:
			if hasattr(self, 'prev_button') and self.prev_button is not None:
				self.prev_button.setEnabled(has_matches)
		except Exception:
			pass
		try:
			if hasattr(self, 'next_button') and self.next_button is not None:
				self.next_button.setEnabled(has_matches)
		except Exception:
			pass

		if not has_matches:
			try:
				self.text_edit.setExtraSelections([])
			except Exception:
				pass
			try:
				if hasattr(self, 'match_label') and self.match_label is not None:
					self.match_label.setText('No matches')
			except Exception:
				pass
			return

		# Jump to the first match in the new content
		try:
			if hasattr(self, 'match_label') and self.match_label is not None:
				self.match_label.setText(f"1 of {len(self.search_positions)}")
		except Exception:
			pass
		self.find_next()

	def _cache_opf_content(self, book_id, opf_path, content):
		"""Cache OPF content to avoid re-reading from disk on navigation.

		Uses LRU-like eviction when cache exceeds max_cache_size.
		"""
		cache_key = (book_id, opf_path)
		self.opf_content_cache[cache_key] = content

		# Simple LRU eviction: remove oldest entries if cache is too large
		if len(self.opf_content_cache) > self.max_cache_size:
			# Remove the first (oldest) entry
			oldest_key = next(iter(self.opf_content_cache))
			del self.opf_content_cache[oldest_key]

	def _get_cached_opf_content(self, book_id, opf_path):
		"""Retrieve cached OPF content if available.

		Returns None if not cached.
		"""
		cache_key = (book_id, opf_path)
		return self.opf_content_cache.get(cache_key)

	def _clear_opf_cache(self):
		"""Clear the OPF content cache (e.g., after library changes)."""
		self.opf_content_cache.clear()

	def populate_tree(self):
		"""Parse XML and populate the tree widget and stats"""
		self.tree_widget.clear()
		try:
			# Use StringIO for parsing XML string but handle encoding declaration properly
			from io import BytesIO

			# Convert to bytes if it's not already
			xml_bytes = self.xml_content.encode('utf-8') if isinstance(self.xml_content, str) else self.xml_content

			# Parse using BytesIO to avoid encoding declaration issues
			root = ET.parse(BytesIO(xml_bytes)).getroot()

			if root is not None:
				root_item = self.add_element_to_tree(root, self.tree_widget)
				self.stats_panel.update_stats(root)

				# Find and expand only the metadata section
				root_item.setExpanded(True)  # Expand root first
				for i in range(root_item.childCount()):
					child = root_item.child(i)
					if child.text(0).startswith('metadata'):
						child.setExpanded(True)
						# Also expand its immediate children to show all metadata fields
						for j in range(child.childCount()):
							child.child(j).setExpanded(True)
		except ET.ParseError as e:
			# Handle XML parsing errors specifically
			self.xml_parsing_error_count += 1
			error_msg = f'XML parsing error: {str(e)}'
			debug_print(f'OPFHelper ERROR: {error_msg} (Total XML parsing errors: {self.xml_parsing_error_count})')

			# Try to show the problematic area around the error
			try:
				lines = self.xml_content.split('\n')
				if hasattr(e, 'position') and e.position:
					line_num, col_num = e.position
					if line_num <= len(lines):
						line = lines[line_num - 1]
						# Show context around the error
						start = max(0, col_num - 20)
						end = min(len(line), col_num + 20)
						context = line[start:end]
						error_msg += f'\n\nContext around error (line {line_num}, column {col_num}):\n{context}'
						if col_num <= len(line):
							error_msg += '\n' + ' ' * (col_num - start - 1) + '^'
			except Exception as context_error:
				debug_print(f'OPFHelper ERROR: Failed to extract error context: {str(context_error)}')

			# Show less-obtrusive dialog (Question icon) but include book id/title
			book_ref = self._format_book_ref()
			main_text = (
				f'{book_ref}\n\n'
				'Failed to parse OPF XML content.\n\n'
				f'{str(e)}\n\n'
				'Tip: The XML tab may still show the raw OPF.'
			)
			self._show_question_message('XML Parsing Issue', main_text, details=error_msg)

			# Still try to show the XML content in the text view even if tree fails
			self.text_edit.setPlainText(self.xml_content)
			self._refresh_search_after_content_change()

		except Exception as e:
			# Handle other parsing errors
			error_msg = f'Error populating tree: {str(e)}'
			debug_print(f'OPFHelper ERROR: {error_msg}')
			error_dialog(self, 'Tree Population Error',
						error_msg,
						show=True)

	def add_element_to_tree(self, element, parent):
		"""Recursively add XML elements to the tree"""
		namespace = ""
		tag = element.tag
		if tag is None:  # Handle comments and processing instructions
			return None

		if isinstance(tag, str) and tag.startswith("{"):
			try:
				namespace = tag[1:tag.index("}")]
				tag = tag[tag.index("}")+1:]
			except (ValueError, AttributeError):
				# If } not found, keep original tag
				pass

		# Determine element type
		element_type = "element"
		if len(element) > 0:
			element_type = "parent"
		elif element.text and element.text.strip():
			element_type = "text"
		elif not element.attrib and not element.text:
			element_type = "empty"

		display_text = tag
		if namespace:
			display_text = f"{tag} [{namespace}]"

		# Add attributes to display text if present
		if element.attrib:
			attr_list = []
			for key, value in element.attrib.items():
				# Handle namespaced attributes
				attr_name = key
				if isinstance(key, str) and key.startswith("{"):
					try:
						attr_name = key[key.index("}")+1:]
					except (ValueError, AttributeError):
						# If } not found, keep original key
						pass
				attr_list.append(f"{attr_name}=\"{value}\"")
			if attr_list:
				display_text += " (" + ", ".join(attr_list) + ")"

		# Create tree item
		if isinstance(parent, QTreeWidget):
			item = QTreeWidgetItem(parent)
		else:
			item = QTreeWidgetItem(parent)

		item.setText(0, display_text)

		# Add tooltips based on element tag and attributes
		base_tag = tag
		if isinstance(base_tag, str) and base_tag in self.opf_help:
			item.setToolTip(0, self.opf_help[base_tag])

		# Store element data and type
		item.setData(0, Qt.UserRole, element)
		item.setData(0, Qt.UserRole + 1, element_type)

		# Apply custom colors based on element tag
		if isinstance(base_tag, str):
			tag_lower = base_tag.lower()
			if tag_lower in self.element_colors:
				item.setForeground(0, QColor(self.element_colors[tag_lower]))

		# Add child elements
		for child in element:
			child_item = self.add_element_to_tree(child, item)
			if child_item:
				item.addChild(child_item)

		# Add text content if present
		if element.text and element.text.strip():
			text_item = QTreeWidgetItem(item)
			text_item.setText(0, element.text.strip())
			text_item.setData(0, Qt.UserRole + 1, "text")
			text_item.setData(0, Qt.UserRole + 2, element.text.strip())

			# Color the text value if parent is a metadata field
			if tag_lower in self.element_colors:
				color = QColor(self.element_colors[tag_lower])
				text_item.setForeground(0, color)  # Apply color to the text value

		return item

	def on_tree_item_clicked(self, item, column):
		"""When a tree item is clicked, find and highlight the corresponding XML"""
		element = item.data(0, Qt.UserRole)
		if element is not None:
			try:
				content = self.text_edit.toPlainText()

				# Extract element tag name
				element_tag = element.tag
				if element_tag.startswith("{"):
					_, element_tag = element_tag.split("}")

				# Build search pattern with attributes
				tag_pattern = f"<{element_tag}"
				if element.attrib:
					# Add each attribute to search pattern
					for key, value in element.attrib.items():
						if key.startswith("{"):
							_, key = key.split("}")
						tag_pattern += f' {key}="{value}"'

				# Try to find the exact element first
				pos = content.find(tag_pattern)
				if pos >= 0:
					# Find the end of this element
					tag_end = content.find(">", pos)
					if tag_end >= 0:
						if content[tag_end-1] == "/":
							# Self-closing tag
							end_pos = tag_end + 1
						else:
							# Find closing tag
							end_tag = f"</{element_tag}>"
							temp_pos = content.find(end_tag, tag_end)
							if temp_pos >= 0:
								end_pos = temp_pos + len(end_tag)
							else:
								end_pos = tag_end + 1

						# Create cursor and select text
						cursor = self.text_edit.textCursor()
						cursor.setPosition(pos)
						cursor.setPosition(end_pos, QTextCursor.KeepAnchor)

						# Update text editor
						self.text_edit.setTextCursor(cursor)
						self.text_edit.ensureCursorVisible()

						# Switch to XML tab and focus
						for i in range(self.tab_widget.count()):
							if self.tab_widget.tabText(i) == "XML":
								self.tab_widget.setCurrentIndex(i)
								self.text_edit.setFocus()
								break

			except Exception as e:
				debug_print(f"OPFHelper ERROR: Failed to highlight element: {str(e)}")
				debug_print(traceback.format_exc())

	def closeEvent(self, e):
		# Save dialog geometry to gprefs like CCR
		try:
			gprefs['opf_helper_dialog_geometry'] = bytearray(self.saveGeometry())
		except Exception:
			pass
		super().closeEvent(e)

	# New methods for context menu
	def setup_tree_context_menu(self):
		"""Set up context menu for tree items"""
		self.tree_widget.setContextMenuPolicy(Qt.CustomContextMenu)
		self.tree_widget.customContextMenuRequested.connect(self.show_tree_context_menu)

	def show_tree_context_menu(self, position):
		"""Display context menu for tree items"""
		item = self.tree_widget.itemAt(position)
		if not item:
			return

		element = item.data(0, Qt.UserRole)
		element_type = item.data(0, Qt.UserRole + 1)
		text_content = item.data(0, Qt.UserRole + 2)  # For text nodes

		from PyQt5.QtWidgets import QMenu, QAction

		menu = QMenu()

		# Text value copying (if this is a text node or has text content)
		if text_content:
			copy_value = QAction("Copy Value", self)
			copy_value.triggered.connect(lambda: self.copy_to_clipboard(text_content))
			menu.addAction(copy_value)
			menu.addSeparator()

		if element is not None:
			# Element actions
			copy_element = QAction("Copy Element XML", self)
			copy_element.triggered.connect(lambda: self.copy_element_xml(element))
			menu.addAction(copy_element)

			copy_tag = QAction("Copy Tag Name", self)
			copy_tag.triggered.connect(lambda: self.copy_element_tag(element))
			menu.addAction(copy_tag)

			# For elements with text content
			if element.text and element.text.strip():
				copy_text = QAction("Copy Text Content", self)
				copy_text.triggered.connect(lambda: self.copy_element_text(element))
				menu.addAction(copy_text)

			# For elements with attributes
			if element.attrib:
				menu.addSeparator()
				copy_attrs = QAction("Copy All Attributes", self)
				copy_attrs.triggered.connect(lambda: self.copy_element_attributes(element))
				menu.addAction(copy_attrs)

				# Add individual attribute copy actions if there aren't too many
				if len(element.attrib) < 8:
					for key, value in element.attrib.items():
						attr_name = key
						if key.startswith("{"):
							# Handle namespaced attributes
							_, attr_name = key[1:].split("}")
						copy_attr = QAction(f"Copy Attribute: {attr_name}", self)
						copy_attr.triggered.connect(lambda checked, k=key, v=value:
												 self.copy_to_clipboard(f'{k}="{v}"'))
						menu.addAction(copy_attr)

			# Element type info
			menu.addSeparator()
			type_label = QAction(f"Type: {element_type}", self)
			type_label.setEnabled(False)
			menu.addAction(type_label)

			# Namespace info if present
			if element.tag.startswith("{"):
				ns, _ = element.tag[1:].split("}")
				ns_label = QAction(f"Namespace: {ns}", self)
				ns_label.setEnabled(False)
				menu.addAction(ns_label)

			# Navigation actions for container elements
			if element_type == "container":
				menu.addSeparator()
				expand_all = QAction("Expand All Children", self)
				expand_all.triggered.connect(lambda: self.expand_children(item))
				menu.addAction(expand_all)

				collapse_all = QAction("Collapse All Children", self)
				collapse_all.triggered.connect(lambda: self.collapse_children(item))
				menu.addAction(collapse_all)

		# Execute the menu
		menu.exec_(self.tree_widget.viewport().mapToGlobal(position))

	def copy_element_xml(self, element):
		"""Copy the XML representation of an element to clipboard"""
		try:
			# Deep copy to ensure we only serialize the selected subtree
			import copy as _copy
			el = element
			try:
				# If somehow an ElementTree was stored, get its root
				if hasattr(el, 'getroot') and callable(getattr(el, 'getroot', None)) and not hasattr(el, 'tag'):
					el = el.getroot()
			except Exception:
				pass
			try:
				el_copy = _copy.deepcopy(el)
			except Exception:
				el_copy = el
			xml_str = ET.tostring(el_copy, encoding='unicode')
			self.copy_to_clipboard(xml_str)
		except Exception as e:
			debug_print(f"OPFHelper: Error copying element XML: {e}")

	def copy_element_tag(self, element):
		"""Copy the tag name of an element to clipboard"""
		tag = element.tag
		if tag.startswith("{"):
			# Handle namespaced tags
			_, tag = tag[1:].split("}")
		self.copy_to_clipboard(tag)

	def copy_element_attributes(self, element):
		"""Copy all attributes of an element to clipboard"""
		if not element.attrib:
			return

		attr_strings = []
		for key, value in element.attrib.items():
			if key.startswith("{"):
				# Handle namespaced attributes
				_, attr = key[1:].split("}")
				attr_strings.append(f'{attr}="{value}"')
			else:
				attr_strings.append(f'{key}="{value}"')
		self.copy_to_clipboard(", ".join(attr_strings))

	def copy_element_text(self, element):
		"""Copy text content of an element to clipboard"""
		if element.text and element.text.strip():
			self.copy_to_clipboard(element.text.strip())

	def expand_children(self, item):
		"""Recursively expand an item and all its children"""
		item.setExpanded(True)
		for i in range(item.childCount()):
			self.expand_children(item.child(i))

	def collapse_children(self, item):
		"""Recursively collapse all children of an item"""
		for i in range(item.childCount()):
			child = item.child(i)
			child.setExpanded(False)
			self.collapse_children(child)

	def copy_to_clipboard(self, text=None):
		"""Copy text to clipboard with passive notification

		If text is None, copies the entire content of the text edit.
		"""
		clipboard = QApplication.clipboard()
		if text is None:
			text = self.text_edit.toPlainText()
		clipboard.setText(text)

		# Create passive notification
		notification = QFrame(self)
		notification.setFrameStyle(QFrame.Panel | QFrame.Raised)
		notification.setLineWidth(2)

		layout = QHBoxLayout(notification)
		layout.setContentsMargins(6, 6, 6, 6)
		icon = QLabel()
		# Use get_icon for Calibre version compatibility
		try:
			if I is not None:
				icon.setPixmap(QIcon(I('ok.png')).pixmap(16, 16))
			else:
				icon.setPixmap(get_icon('ok.png').pixmap(16, 16))
		except Exception:
			icon.setPixmap(get_icon('ok.png').pixmap(16, 16))
		layout.addWidget(icon)

		label = QLabel("Content copied to clipboard")
		label.setStyleSheet("color: rgb(0, 140, 0);")
		layout.addWidget(label)

		# Position the notification
		pos = self.mapToGlobal(self.rect().topRight())
		notification.move(pos.x() - notification.sizeHint().width() - 10, pos.y() + 10)

		# Show notification
		notification.show()

		# Auto-hide after 2 seconds
		QTimer.singleShot(2000, notification.deleteLater)

	# Add methods for zoom functionality that are referenced by the buttons
	def zoom_in(self):
		"""Increase the font size of trees and XML view"""
		self.current_font_size = min(24, self.current_font_size + self.font_size_increment)

		# Update XML view font
		xml_font = self.text_edit.font()
		xml_font.setPointSize(self.current_font_size)
		self.text_edit.setFont(xml_font)

		# Update tree view fonts
		tree_font = self.tree_widget.font()
		tree_font.setPointSize(self.current_font_size)
		self.tree_widget.setFont(tree_font)

		# Update stats panel fonts
		stats_font = self.stats_panel.font()
		stats_font.setPointSize(self.current_font_size)
		self.stats_panel.setFont(stats_font)
		self.stats_panel.metadata_tree.setFont(stats_font)
		self.stats_panel.files_list.setFont(stats_font)

		# Update validation panel fonts - using correct attribute name 'results'
		validation_font = self.validation_panel.font()
		validation_font.setPointSize(self.current_font_size)
		self.validation_panel.setFont(validation_font)
		self.validation_panel.results.setFont(validation_font)

		# Save the new size
		self.prefs['font_size'] = self.current_font_size

	def zoom_out(self):
		"""Decrease the font size of trees and XML view"""
		self.current_font_size = max(6, self.current_font_size - self.font_size_increment)

		# Update XML view font
		xml_font = self.text_edit.font()
		xml_font.setPointSize(self.current_font_size)
		self.text_edit.setFont(xml_font)

		# Update tree view fonts
		tree_font = self.tree_widget.font()
		tree_font.setPointSize(self.current_font_size)
		self.tree_widget.setFont(tree_font)

		# Update stats panel fonts
		stats_font = self.stats_panel.font()
		stats_font.setPointSize(self.current_font_size)
		self.stats_panel.setFont(stats_font)
		self.stats_panel.metadata_tree.setFont(stats_font)
		self.stats_panel.files_list.setFont(stats_font)

		# Update validation panel fonts - using correct attribute name 'results'
		validation_font = self.validation_panel.font()
		validation_font.setPointSize(self.current_font_size)
		self.validation_panel.setFont(validation_font)
		self.validation_panel.results.setFont(validation_font)

		# Save the new size
		self.prefs['font_size'] = self.current_font_size

	def export_opf(self):
		"""Export OPF content to a file"""
		try:
			from PyQt5.QtWidgets import QFileDialog

			# Get current book info
			book_id = self.book_ids[self.current_book_index]
			title = self.db.field_for("title", book_id)

			# Create default filename from book title
			safe_title = "".join(x for x in title if x.isalnum() or x in (" ._-"))
			default_filename = f"{safe_title}_opf.xml"

			# Show save file dialog
			filename, _ = QFileDialog.getSaveFileName(
				self,
				"Save OPF Content",
				default_filename,
				"XML Files (*.xml);;OPF Files (*.opf);;All Files (*.*)"
			)

			if filename:  # If user didn't cancel
				with open(filename, 'w', encoding='utf-8') as f:
					f.write(self.xml_content)

				# Show success notification
				notification = QFrame(self)
				notification.setFrameStyle(QFrame.Panel | QFrame.Raised)
				notification.setLineWidth(2)

				layout = QHBoxLayout(notification)
				layout.setContentsMargins(6, 6, 6, 6)
				icon = QLabel()
				# Use get_icon for Calibre version compatibility
				try:
					if I is not None:
						icon.setPixmap(QIcon(I('ok.png')).pixmap(16, 16))
					else:
						icon.setPixmap(get_icon('ok.png').pixmap(16, 16))
				except Exception:
					icon.setPixmap(get_icon('ok.png').pixmap(16, 16))
				layout.addWidget(icon)

				label = QLabel(f"OPF content saved to: {os.path.basename(filename)}")
				label.setStyleSheet("color: rgb(0, 140, 0);")
				layout.addWidget(label)

				# Position notification
				pos = self.mapToGlobal(self.rect().topRight())
				notification.move(pos.x() - notification.sizeHint().width() - 10, pos.y() + 10)

				# Show and auto-hide notification
				notification.show()
				QTimer.singleShot(2000, notification.deleteLater)

		except Exception as e:
			error_dialog(self.gui, 'Export Error',
					   f'Failed to export OPF content: {str(e)}',
					   show=True)


	def edit_metadata(self):
		"""Open the metadata editor for the current book"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Opening metadata editor for book ID {book_id}')

			# Get library view and ensure we're showing it
			self.gui.show_library_view()
			library_view = self.gui.library_view

			# Store current search and selection state
			current_search = str(self.gui.search.current_text)
			current_selection = library_view.get_state()

			# First clear any search restrictions/virtual libraries that might hide our book
			if library_view.model().db.data.get_base_restriction_name():
				self.gui.apply_virtual_library()
			if library_view.model().db.data.get_search_restriction_name():
				self.gui.apply_named_search_restriction()

			# Select our book
			library_view.select_rows([book_id], using_ids=True)

			# Ensure Calibre window is active
			self.gui.activateWindow()
			self.gui.raise_()

			# Edit metadata for current book
			debug_print('OPFHelper: Opening metadata editor')
			self.gui.iactions['Edit Metadata'].edit_metadata(False)

			# Restore previous state
			debug_print('OPFHelper: Restoring previous search and selection')
			self.gui.search.set_search_string(current_search)
			if current_selection:
				library_view.state = current_selection  # Use the state property setter instead
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to edit metadata: {str(e)}')
			error_dialog(self, 'Error',
					  f'Failed to edit metadata: {str(e)}',
					  show=True)

	def load_current_book(self):
		"""Load the current book's OPF content and cover.

		Supports KEPUB, EPUB, HTMLZ, AZW3, and MOBI formats.
		For AZW3: uses Calibre's built-in exploder (same as Edit OPF action).
		For MOBI: uses KindleUnpack via mobi_opf_extract.extract_mobi_opf.
		"""
		try:
			book_id = self.book_ids[self.current_book_index]

			db = self.gui.current_db.new_api
			title = db.field_for('title', book_id)

			# Always populate basic UI metadata early so the dialog is not blank
			try:
				author = db.field_for('authors', book_id)
				if author:
					if isinstance(author, (list, tuple)) and len(author) > 0:
						author = author[0]
					elif not isinstance(author, str):
						author = str(author)
				else:
					author = 'Unknown'
			except Exception:
				author = 'Unknown'
			try:
				self.setWindowTitle(f"OPF Helper: {title}")
			except Exception:
				pass
			try:
				self.book_label.setText(f"{title} by {author}")
			except Exception:
				pass
			try:
				self.update_book_info_label()
			except Exception:
				pass
			try:
				self.prev_book_button.setEnabled(True)
				self.next_book_button.setEnabled(True)
			except Exception:
				pass

			# Try formats in preferred order: KEPUB, EPUB, HTMLZ, AZW3, MOBI
			epub_path = None
			found_fmt = None
			for fmt in ('KEPUB', 'EPUB', 'HTMLZ', 'AZW3', 'MOBI'):
				try:
					p = db.format_abspath(book_id, fmt)
				except Exception:
					p = None
				if p and os.path.isfile(p):
					epub_path = p
					found_fmt = fmt
					break

			# Persist current format/path for validators and other actions
			try:
				self.current_format = found_fmt
			except Exception:
				pass
			try:
				self.current_archive_path = epub_path
			except Exception:
				pass

			if not epub_path:
				msg = 'No EPUB/KEPUB/HTMLZ/AZW3/MOBI format available for this book'
				self.gui.status_bar.showMessage(msg, 4000)
				try:
					self.xml_content = msg
					self.text_edit.setPlainText(msg)
				except Exception:
					pass
				return

			opf_data = None
			tdir = None

			# Handle AZW3 specially (not a ZIP) — use Calibre's built-in exploder
			if found_fmt == 'AZW3':
				azw3_error = None
				try:
					# DRM check
					has_drm = False
					try:
						has_drm = db.has_drm(book_id)
					except Exception:
						try:
							has_drm = self.db.has_drm(book_id)
						except Exception:
							has_drm = False
					if has_drm:
						msg = 'Book has DRM; cannot view AZW3 OPF'
						self.gui.status_bar.showMessage(msg, 5000)
						try:
							self.xml_content = msg
							self.text_edit.setPlainText(msg)
						except Exception:
							pass
						return

					from calibre.ebooks.tweak import get_tools
					try:
						tdir = PersistentTemporaryDirectory('_opf_helper_view')
					except Exception:
						import tempfile
						tdir = tempfile.mkdtemp(prefix='_opf_helper_view_')
					explode_dir = str(tdir)

					exploder = get_tools('AZW3')[0]
					opf_path = exploder(epub_path, explode_dir, question=self._ask_question)

					if opf_path and os.path.exists(opf_path):
						with open(opf_path, 'rb') as f:
							opf_data = f.read().decode('utf-8', 'replace')
					else:
						# Scan exploded dir for any .opf
						opfs = _opfhelper_find_opf_in_exploded(explode_dir)
						if opfs:
							with open(opfs[0], 'rb') as f:
								opf_data = f.read().decode('utf-8', 'replace')

					if not opf_data:
						azw3_error = 'Could not extract OPF from AZW3 file'

				except Exception as e:
					azw3_error = f'AZW3 extraction failed: {e}'
					debug_print(f'OPFHelper: {azw3_error}')
				finally:
					# Best-effort cleanup
					try:
						if tdir and os.path.isdir(str(tdir)):
							shutil.rmtree(str(tdir))
					except Exception:
						pass

				if azw3_error:
					self.gui.status_bar.showMessage(azw3_error, 6000)
					try:
						self.xml_content = azw3_error
						self.text_edit.setPlainText(azw3_error)
					except Exception:
						pass
					return

			elif found_fmt == 'MOBI':
				# MOBI: use KindleUnpack flow to generate an OPF
				mobi_error = None
				try:
					# DRM check (similar to AZW3 path)
					has_drm = False
					try:
						has_drm = db.has_drm(book_id)
					except Exception:
						try:
							has_drm = self.db.has_drm(book_id)
						except Exception:
							has_drm = False
					if has_drm:
						msg = 'Book has DRM; cannot view MOBI OPF'
						self.gui.status_bar.showMessage(msg, 5000)
						try:
							self.xml_content = msg
							self.text_edit.setPlainText(msg)
						except Exception:
							pass
						return

					from calibre_plugins.opf_helper.mobi_opf_extract import extract_mobi_opf
					opf_xml, err = extract_mobi_opf(epub_path)
					if not opf_xml:
						mobi_error = err or 'Could not extract OPF from MOBI file'
					else:
						opf_data = opf_xml
				except Exception as e:
					mobi_error = f'MOBI extraction failed: {e}'
					debug_print(f'OPFHelper: {mobi_error}')

				if mobi_error:
					self.gui.status_bar.showMessage(mobi_error, 6000)
					try:
						self.xml_content = mobi_error
						self.text_edit.setPlainText(mobi_error)
					except Exception:
						pass
					return

			else:
				# EPUB/KEPUB/HTMLZ: treat as ZIP
				try:
					with zipfile.ZipFile(epub_path, 'r') as zf:
						opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
						if not opf_files:
							error_dialog(self.gui, 'Error', 'No OPF file found in this archive.', show=True)
							return

						self.current_book_opfs = opf_files
						if book_id in self.book_opf_selections:
							sel = self.book_opf_selections[book_id]
							self.current_opf_path = sel if sel in opf_files else opf_files[0]
						else:
							self.current_opf_path = opf_files[0]

						if len(opf_files) > 1:
							self.opf_selector_container.setVisible(True)
							self.opf_selector.clear()
							self.opf_selector.addItems(opf_files)
							self.opf_selector.setCurrentText(self.current_opf_path)
							self.opf_count_label.setText(f"({len(opf_files)} files)")
						else:
							self.opf_selector_container.setVisible(False)

						# Check cache first before reading from ZIP
						opf_data = self._get_cached_opf_content(book_id, self.current_opf_path)
						if opf_data is None:
							# Not cached, read from ZIP and cache it
							opf_data = zf.read(self.current_opf_path).decode('utf-8', errors='replace')
							self._cache_opf_content(book_id, self.current_opf_path, opf_data)
						else:
							debug_print(f'OPFHelper: Using cached OPF for book_id {book_id}')

				except zipfile.BadZipFile:
					book_ref = self._format_book_ref()
					self._show_question_message('File Problem',
						f'{book_ref}\n\nCould not open file. It may be corrupted.',
						details='zipfile.BadZipFile')
					return
				except Exception as e:
					error_dialog(self.gui, 'Error', f'Failed to read OPF content.\nError: {str(e)}', show=True)
					return

			# At this point we have opf_data — update UI
			self.xml_content = opf_data
			self.text_edit.setPlainText(opf_data)
			self._refresh_search_after_content_change()
			self.populate_tree()
			self.setWindowTitle(f"OPF Helper: {title}")

			# Book label already populated above (keep it in sync just in case)
			try:
				self.book_label.setText(f"{title} by {author}")
			except Exception:
				pass

			# Badge: show EPUB version for EPUB/KEPUB, otherwise show the actual format
			# with the OPF version when detectable (helps with AZW3/HTMLZ).
			opf_ver = None
			if 'version="3.0"' in opf_data:
				opf_ver = '3.0'
			elif 'version="2.0"' in opf_data:
				opf_ver = '2.0'

			fmt_badge = (found_fmt or '').upper().strip()
			is_epub_like = fmt_badge in ('EPUB', 'KEPUB')

			if is_epub_like:
				if opf_ver == '3.0':
					version_color = "#FFA500" if self.is_dark else "#FF8C00"
					self.version_label.setText('EPUB 3.0')
					self.version_label.setStyleSheet(f"""
						QLabel {{
							padding: 2px 8px;
							border-radius: 4px;
							color: {version_color};
							font-weight: bold;
							background-color: transparent;
							border: 1px solid {version_color};
						}}
					""")
				elif opf_ver == '2.0':
					self.version_label.setText('EPUB 2.0')
					self.version_label.setStyleSheet("""
						QLabel {
							padding: 2px 8px;
							border-radius: 4px;
							color: palette(link);
							font-weight: bold;
							background-color: transparent;
							border: 1px solid palette(link);
						}
					""")
				else:
					self.version_label.setText(fmt_badge or 'EPUB ?')
					self.version_label.setStyleSheet("""
						QLabel {
							padding: 2px 8px;
							border-radius: 4px;
							color: palette(link);
							font-weight: bold;
							background-color: transparent;
							border: 1px solid palette(link);
						}
					""")
			else:
				# Non-EPUB formats: show format badge + OPF version so users know
				# they're looking at extracted/packaged metadata.
				label = fmt_badge or 'BOOK'
				label = f'{label} \u2022 OPF {opf_ver or "?"}'
				# Theme-aware but distinctive accent colors per format.
				if fmt_badge == 'AZW3':
					accent = '#BB86FC' if self.is_dark else '#8E44AD'
				elif fmt_badge == 'HTMLZ':
					accent = '#1ABC9C' if self.is_dark else '#0E7C86'
				else:
					accent = 'palette(link)'
				self.version_label.setText(label)
				self.version_label.setStyleSheet(f"""
					QLabel {{
						padding: 2px 8px;
						border-radius: 4px;
						color: {accent};
						font-weight: bold;
						background-color: transparent;
						border: 1px solid {accent};
					}}
				""")

			# Load cover
			try:
				self.load_cover(self.current_opf_path if hasattr(self, 'current_opf_path') else None)
			except Exception:
				pass

			# Auto-validate if enabled
			if prefs.get('auto_validate', True):
				QTimer.singleShot(100, self.validate_opf)

		except Exception as e:
			error_dialog(self.gui, 'Error', f'Failed to load book information.\nError: {str(e)}', show=True)

	def show_next_book(self):
		"""Show the next book in the library view, skipping books without supported formats."""
		if not self.book_ids:
			return

		start_index = self.current_book_index
		books_checked = 0

		while books_checked < len(self.book_ids):
			old_index = self.current_book_index
			self.current_book_index = (self.current_book_index + 1) % len(self.book_ids)
			books_checked += 1

			book_id = self.book_ids[self.current_book_index]

			# Check if book has a supported format
			supported_formats = ('KEPUB', 'EPUB', 'HTMLZ', 'AZW3', 'MOBI')
			try:
				has_supported = any(self.db.has_format(book_id, fmt) for fmt in supported_formats)
			except Exception:
				try:
					na = self.gui.current_db.new_api
					has_supported = any(na.has_format(book_id, fmt) for fmt in supported_formats)
				except Exception:
					has_supported = False

			if has_supported:
				self.gui.library_view.select_rows([book_id], using_ids=True)
				self.load_current_book()
				return

		# If we get here, no compatible books found
		self.gui.status_bar.showMessage("No books with EPUB/KEPUB/HTMLZ/AZW3/MOBI format found in library", 5000)

	def show_previous_book(self):
		"""Show the previous book in the library view, skipping books without supported formats."""
		if not self.book_ids:
			return

		start_index = self.current_book_index
		books_checked = 0

		while books_checked < len(self.book_ids):
			old_index = self.current_book_index
			self.current_book_index = (self.current_book_index - 1) % len(self.book_ids)
			books_checked += 1

			book_id = self.book_ids[self.current_book_index]

			# Check if book has a supported format
			supported_formats = ('KEPUB', 'EPUB', 'HTMLZ', 'AZW3', 'MOBI')
			try:
				has_supported = any(self.db.has_format(book_id, fmt) for fmt in supported_formats)
			except Exception:
				try:
					na = self.gui.current_db.new_api
					has_supported = any(na.has_format(book_id, fmt) for fmt in supported_formats)
				except Exception:
					has_supported = False

			if has_supported:
				self.gui.library_view.select_rows([book_id], using_ids=True)
				self.load_current_book()
				return

		# If we get here, no compatible books found
		self.gui.status_bar.showMessage("No books with EPUB/KEPUB/HTMLZ/AZW3 format found in library", 5000)

	def show_first_book(self):
		"""Show the first compatible book in the library view."""
		if not self.book_ids:
			return

		# Find the first book with a supported format
		supported_formats = ('KEPUB', 'EPUB', 'HTMLZ', 'AZW3')
		for i, book_id in enumerate(self.book_ids):
			try:
				has_supported = any(self.db.has_format(book_id, fmt) for fmt in supported_formats)
			except Exception:
				try:
					na = self.gui.current_db.new_api
					has_supported = any(na.has_format(book_id, fmt) for fmt in supported_formats)
				except Exception:
					has_supported = False
			if has_supported:
				self.current_book_index = i
				self.gui.library_view.select_rows([book_id], using_ids=True)
				self.load_current_book()
				return

		# If no compatible books found
		self.gui.status_bar.showMessage("No books with EPUB/KEPUB/HTMLZ/AZW3 format found in library", 5000)

	def show_last_book(self):
		"""Show the last compatible book in the library view."""
		if not self.book_ids:
			return

		# Find the last book with a supported format
		supported_formats = ('KEPUB', 'EPUB', 'HTMLZ', 'AZW3')
		for i in range(len(self.book_ids) - 1, -1, -1):
			book_id = self.book_ids[i]
			try:
				has_supported = any(self.db.has_format(book_id, fmt) for fmt in supported_formats)
			except Exception:
				try:
					na = self.gui.current_db.new_api
					has_supported = any(na.has_format(book_id, fmt) for fmt in supported_formats)
				except Exception:
					has_supported = False
			if has_supported:
				self.current_book_index = i
				self.gui.library_view.select_rows([book_id], using_ids=True)
				self.load_current_book()
				return

		# If no compatible books found
		self.gui.status_bar.showMessage("No books with EPUB/KEPUB/HTMLZ/AZW3 format found in library", 5000)

	class SchemaResolver(etree.Resolver):
		def __init__(self, schema_dir):
			self.schema_dir = schema_dir
			debug_print(f"OPFHelper: Schema resolver initialized with dir: {schema_dir}")

		def resolve(self, system_url, public_id, context):
			"""Resolve schema imports by looking in the local schema directory"""
			debug_print(f"OPFHelper: Resolving schema: {system_url}, {public_id}")
			try:
				if (system_url.endswith('.xsd')):
					schema_file = os.path.basename(system_url)
					schema_path = os.path.join(self.schema_dir, schema_file)
					if (os.path.exists(schema_path)):
						debug_print(f"OPFHelper: Found schema at {schema_path}")
						return self.resolve_filename(schema_path, context)
					else:
						debug_print(f"OPFHelper: Schema not found at {schema_path}")
				return None
			except Exception as e:
				debug_print(f"OPFHelper ERROR: Schema resolution failed: {str(e)}")
				debug_print(traceback.format_exc())
				return None

	def initialize_schemas(self):
		"""Initialize XML Schema parsers with proper namespace handling"""
		try:
			from .schema_utils import verify_schemas, install_schemas, load_schema

			# Initialize schema parsers dictionary
			self.schema_parsers = {}

			# First verify schemas are valid
			if not verify_schemas():
				debug_print("OPFHelper ERROR: Schema verification failed")
				return

			# Load schemas for each version
			for version in ['2.0', '3.0']:
				schema = load_schema(version)
				if schema:
					self.schema_parsers[version] = schema
					debug_print(f"OPFHelper: Loaded schema for version {version}")
				else:
					debug_print(f"OPFHelper ERROR: Failed to load schema for version {version}")

		except Exception as e:
			debug_print(f"OPFHelper ERROR: Failed to initialize schemas: {str(e)}")
			debug_print(traceback.format_exc())
			self.schema_parsers = {}

	def validate_opf(self):
		"""Validate the current OPF content against appropriate schema"""
		if not self.xml_content:
			return error_dialog(self, "No OPF Content", "There is no OPF content to validate.")

		try:
			# Display version (badge) may include format label; derive a clean OPF version for schema
			display_version = self.version_label.text()

			# Parse XML content with custom parser
			parser = get_schema_parser()
			doc = etree.fromstring(self.xml_content.encode('utf-8'), parser)

			# Determine schema version from the actual OPF <package version="...">
			# Default to 2.0 if not found, but prefer explicit value
			import re as _re
			schema_version = None
			try:
				root_el = doc
				pkg_ver = root_el.get('version')
				if not pkg_ver:
					# Try finding any element named 'package' regardless of namespace
					try:
						pkg_nodes = root_el.xpath('//*[local-name()="package"]')
						if pkg_nodes:
							pkg_ver = pkg_nodes[0].get('version')
					except Exception:
						pkg_ver = None
				if pkg_ver:
					m = _re.match(r"\s*([0-9]+(?:\.[0-9]+)?)", str(pkg_ver))
					if m:
						schema_version = m.group(1)
			except Exception:
				pass
			if not schema_version:
				if 'version="3.0"' in self.xml_content:
					schema_version = '3.0'
				elif 'version="2.0"' in self.xml_content:
					schema_version = '2.0'
				else:
					schema_version = '2.0'

			results = []
			validation_errors = []

			# Always be explicit: this validator is helpful but not EPUBCheck.
			results.append('OPF Helper validation is not EPUBCheck.')
			results.append('It can find many common issues, but results may differ from official epubcheck.')
			results.append('')

			# Basic XML validation first
			results.append("Basic XML Validation:")
			parser_errors = doc.getroottree().parser.error_log
			if parser_errors:
				results.append("❌ XML parsing failed:")
				for error in parser_errors:
					error_text = str(error.message)  # Convert message to string
					results.append(f"❌ {error_text}")
					validation_errors.append(error)
			else:
				results.append("✓ XML is well-formed")
			results.append("")

			# Schema validation
			results.append(f"OPF Schema Validation (Version {display_version}):")
			schema = load_schema(schema_version)

			if schema is not None:
				try:
					schema.assertValid(doc)
					results.append("✓ Valid against schema")
				except etree.DocumentInvalid as e:
					results.append("❌ Schema validation failed:")
					suppressed = 0
					for error in e.error_log:
						error_text = str(error.message)  # Convert message to string
						# Suppress a known false-positive vs EPUBCheck: dc:language with xsi:type dcterms:RFC4646.
						# Our lightweight XSD model for dc elements cannot accurately express this constraint.
						if ('RFC4646' in error_text and 'xsi:type' in error_text and 'language' in error_text):
							suppressed += 1
							continue
						results.append(f"❌ {error_text}")
						validation_errors.append(error)
					if suppressed:
						results.append('⚠️ Suppressed %d known schema false-positive(s) for dc:language xsi:type dcterms:RFC4646 (use EPUBCheck for authoritative results).' % suppressed)
			else:
				results.append("⚠️ No schema available for version")

			# Run additional OPF-specific validation
			is_valid, basic_results = basic_opf_validation(doc.getroottree(), schema_version)
			results.append("")
			results.extend(basic_results)

			# EPUB archive-level checks (manifest/guide/zip existence + basic XHTML sanity)
			try:
				epub_path = getattr(self, 'current_archive_path', None)
				opf_name_in_zip = getattr(self, 'current_opf_path', None)
				if epub_path and os.path.exists(epub_path) and opf_name_in_zip and (opf_name_in_zip in epub_path or True):
					import zipfile
					import posixpath
					from lxml import html as _html

					results.append('')
					results.append('EPUB Archive Checks (heuristic):')

					opf_dir = posixpath.dirname(opf_name_in_zip) or ''
					def _norm_href(h):
						h = (h or '').strip().replace('\\', '/')
						return h.split('#', 1)[0] if h else ''
					def _resolve_entry(h):
						h = _norm_href(h)
						if not h or '://' in h:
							return ''
						return posixpath.normpath(posixpath.join(opf_dir, h)) if opf_dir else posixpath.normpath(h)

					# Build manifest maps
					manifest_by_id = {}
					manifest_by_entry = {}
					for item in doc.findall('.//{http://www.idpf.org/2007/opf}manifest/{http://www.idpf.org/2007/opf}item'):
						iid = (item.get('id') or '').strip()
						href = (item.get('href') or '').strip()
						entry = _resolve_entry(href)
						if iid:
							manifest_by_id[iid] = item
						if entry:
							manifest_by_entry[entry] = item

					with zipfile.ZipFile(epub_path, 'r') as zf:
						# Guide: declared in manifest and exists in zip
						for ref in doc.findall('.//{http://www.idpf.org/2007/opf}guide/{http://www.idpf.org/2007/opf}reference'):
							href = (ref.get('href') or '').strip()
							entry = _resolve_entry(href)
							if not entry:
								continue
							if entry not in manifest_by_entry:
								results.append(f'❌ File listed in reference element in guide was not declared in OPF manifest: {entry}')
								results.append(f'❌ Referenced resource "{entry}" is not declared in the OPF manifest.')
							try:
								zf.getinfo(entry)
							except Exception:
								results.append(f'❌ Referenced resource "{entry}" is missing from the EPUB archive.')

						# Spine XHTML sanity: <head><title>
						for ir in doc.findall('.//{http://www.idpf.org/2007/opf}spine/{http://www.idpf.org/2007/opf}itemref'):
							idref = (ir.get('idref') or '').strip()
							if not idref:
								continue
							item = manifest_by_id.get(idref)
							if item is None:
								continue
							entry = _resolve_entry(item.get('href'))
							if not entry:
								continue
							mtype = (item.get('media-type') or '').lower()
							if ('application/xhtml+xml' not in mtype) and (not entry.lower().endswith(('.htm', '.html', '.xhtml'))):
								continue
							try:
								data = zf.read(entry)
							except Exception:
								continue
							try:
								xdoc = _html.fromstring(data)
								titles = xdoc.xpath('//head/title')
								if not titles or not ((titles[0].text or '').strip()):
									results.append('❌ Error while parsing file: element "head" incomplete; missing required element "title" (file: %s)' % entry)
							except Exception:
								results.append('⚠️ Could not parse XHTML for sanity checks (file: %s)' % entry)
			except Exception:
				# Keep archive checks best-effort; validation remains truthful but non-fatal.
				pass

			# Show results in validation panel
			self.validation_panel.results.setPlainText('\n'.join(results))

		except Exception as e:
			# Show error in validation panel
			error_text = f"❌ Validation failed: {str(e)}"
			self.validation_panel.results.setPlainText(error_text)

	def edit_book(self):
		"""Open the current book in Calibre's Editor, focusing on the OPF file."""
		try:
			book_id = self.book_ids[self.current_book_index]
			title = self.db.field_for("title", book_id)
			# Try KEPUB first, then EPUB
			epub_path = self.db.format_abspath(book_id, 'KEPUB')
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = self.db.format_abspath(book_id, 'EPUB')
			if not epub_path or not os.path.isfile(epub_path):
				error_dialog(self.gui, 'Invalid Book', f'Could not access EPUB or KEPUB file for "{title}"', show=True)
				return

			# Debug what actions are available
			debug_print(f'OPFHelper: Available actions: {list(self.gui.iactions.keys())}')

			# Try to launch ebook-edit directly with the OPF file focused
			try:
				from calibre.gui2.tweak_book import tprefs
				tprefs.refresh()  # In case they were changed

				# Prepare notify data for the editor
				notify = f'{book_id}:EPUB:{self.db.library_id}:{self.db.library_path}'

				# Launch ebook-edit with the OPF file as the file to open
				kwargs = dict(path=epub_path, notify=notify)

				# If we have a current OPF path, pass it as an additional argument
				if hasattr(self, 'current_opf_path') and self.current_opf_path:
					debug_print(f'OPFHelper: Launching ebook-edit with OPF focus: {self.current_opf_path}')

				# Use "Tweak ePub" action which is the built-in editor in Calibre
				editor_action = self.gui.iactions.get('Tweak ePub')

				if editor_action:
					debug_print('OPFHelper: Found "Tweak ePub" action')
					# First select the book in the library view
					self.gui.library_view.select_rows([book_id], using_ids=True)

					# Call the appropriate method for Tweak ePub action
					if hasattr(editor_action, 'tweak_epub'):
						debug_print('OPFHelper: Using tweak_epub method')
						editor_action.tweak_epub(book_id)
					elif hasattr(editor_action, 'tweak_book'):
						debug_print('OPFHelper: Using tweak_book method')
						editor_action.tweak_book()
					else:
						debug_print('OPFHelper: Using menuless_qaction')
						editor_action.menuless_qaction.trigger()
				else:
					debug_print('OPFHelper: "Tweak ePub" action not found, trying alternative methods')
					for action_name in ['Edit Book', 'edit_book', 'editor']:
						action = self.gui.iactions.get(action_name)
						if action:
							debug_print(f'OPFHelper: Found editor action with name: {action_name}')
							if hasattr(action, 'edit_book_files'):
								debug_print('OPFHelper: Using edit_book_files')
								action.edit_book_files([book_id])
								return
							elif hasattr(action, 'edit_book'):
								debug_print('OPFHelper: Using edit_book')
								action.edit_book(book_id)
								return
							else:
								debug_print('OPFHelper: Using menuless_qaction for fallback editor')
								self.gui.library_view.select_rows([book_id], using_ids=True)
								action.menuless_qaction.trigger()
								return

					debug_print('OPFHelper: No suitable edit action found, opening metadata editor')
					self.edit_metadata()
					error_dialog(self.gui, 'Editor Not Available',
								'Could not find book editor action. Opening metadata editor instead.',
								show=True)

			except Exception as e:
				debug_print(f'OPFHelper ERROR: Failed to launch ebook-edit directly: {str(e)}')
				editor_action = self.gui.iactions.get('Tweak ePub')
				if editor_action:
					self.gui.library_view.select_rows([book_id], using_ids=True)
					if hasattr(editor_action, 'tweak_epub'):
						editor_action.tweak_epub(book_id)
					else:
						editor_action.menuless_qaction.trigger()
				else:
					error_dialog(self, 'Error', f'Failed to open book in editor: {str(e)}', show=True)

		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to open book in editor: {str(e)}')
			error_dialog(self, 'Error', f'Failed to open book in editor: {str(e)}', show=True)

	def view_book(self):
		"""Open the current book in Calibre's E-book Viewer"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Opening book ID {book_id} in viewer')

			# Get the book's title and path
			title = self.db.field_for("title", book_id)
			# Try KEPUB first, then EPUB
			epub_path = self.db.format_abspath(book_id, 'KEPUB')
			fmt = 'KEPUB'
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = self.db.format_abspath(book_id, 'EPUB')
				fmt = 'EPUB'
			if not epub_path or not os.path.isfile(epub_path):
				error_dialog(self.gui, 'Invalid Book', f'Could not access EPUB or KEPUB file for "{title}"', show=True)
				return

			# Launch the viewer - using correct action name lookup
			viewer_action = self.gui.iactions.get('View')
			if viewer_action:
				viewer_action.view_format_by_id(book_id, fmt)
			else:
				error_dialog(self, 'Error', 'E-book viewer action not available in this Calibre installation', show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to open book in viewer: {str(e)}')
			error_dialog(self, 'Error',
					  f'Failed to open book in viewer: {str(e)}',
					  show=True)

	def edit_toc(self):
		"""Open the ToC editor for the current book"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Opening ToC editor for book ID {book_id}')

			# Get the book's title and path
			title = self.db.field_for("title", book_id)
			# Try KEPUB first, then EPUB
			epub_path = self.db.format_abspath(book_id, 'KEPUB')
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = self.db.format_abspath(book_id, 'EPUB')
			if not epub_path or not os.path.isfile(epub_path):
				error_dialog(self.gui, 'Invalid Book', f'Could not access EPUB or KEPUB file for "{title}"', show=True)
				return

			# Try to find the ToC Editor in available actions
			toc_editor = None

			# Look for the Edit ToC action under different possible names
			for action_name in ['Edit ToC', 'edit_toc', 'toc']:
				action = self.gui.iactions.get(action_name)
				if action:
					debug_print(f'OPFHelper: Found ToC editor with name: {action_name}')
					toc_editor = action
					break

			if toc_editor:
				# First select the book in the library view
				self.gui.library_view.select_rows([book_id], using_ids=True)

				# Ensure Calibre window is active
				self.gui.activateWindow()
				self.gui.raise_()

				# Try different possible methods to invoke the ToC editor
				if hasattr(toc_editor, 'edit_toc'):
					debug_print('OPFHelper: Using edit_toc method')
					toc_editor.edit_toc(book_id)
				elif hasattr(toc_editor, 'edit_book_toc'):
					debug_print('OPFHelper: Using edit_book_toc method')
					toc_editor.edit_book_toc(book_id)
				else:
					debug_print('OPFHelper: Using menuless_qaction')
					# Use the menu action directly which works on the selected book
					toc_editor.menuless_qaction.trigger()
			else:
				# If we can't find a dedicated ToC editor action, try using the Edit Book functionality
				# which includes a ToC editor
				debug_print('OPFHelper: Dedicated ToC editor not found, trying to use Edit Book')
				editor_action = self.gui.iactions.get('Tweak ePub')

				if editor_action:
					debug_print('OPFHelper: Opening book in editor, where ToC can be edited')
					self.gui.library_view.select_rows([book_id], using_ids=True)

					# Call the appropriate method
					if hasattr(editor_action, 'tweak_epub'):
						editor_action.tweak_epub(book_id)
					elif hasattr(editor_action, 'tweak_book'):
						editor_action.tweak_book()
					else:
						editor_action.menuless_qaction.trigger()

					# Inform user to look for ToC editing tools in the editor
					info_dialog(self.gui, 'ToC Editor',
							  'The book has been opened in the editor. To edit the Table of Contents, '
							  'use the "Tools > Table of Contents > Edit Table of Contents" menu option '
							  'in the editor window.',
							  show=True)
				else:
					error_dialog(self.gui, 'ToC Editor Not Available',
							   'Could not find Table of Contents editor in this Calibre installation.',
							   show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to open ToC editor: {str(e)}')
			error_dialog(self, 'Error', f'Failed to open ToC editor: {str(e)}', show=True)

	def open_edit_opf(self):
		"""Open the OPF Helper's internal OPF editor for the current book."""
		try:
			book_id = self.book_ids[self.current_book_index]
			# Use stored db reference
			db = getattr(self, 'db', None)
			# Fallback to gui current db if missing
			if db is None:
				db = self.gui.current_db.new_api
			# Call module-level helper which launches the OPF editor dialog
			edit_opf_for_book(self.gui, db, book_id)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to open OPF Helper editor: {e}')
			error_dialog(self.gui, 'Error', f'Failed to open OPF editor: {str(e)}', show=True)

	def unpack_book(self):
		"""Unpack the current book into individual components for editing"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Unpacking book ID {book_id}')

			# Use Calibre's built-in Unpack Book action
			unpack_action = self.gui.iactions.get('Unpack Book')
			if unpack_action:
				unpack_action.do_tweak(book_id)
			else:
				error_dialog(self, 'Error', 'Unpack Book action not available in this Calibre installation', show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to unpack book: {str(e)}')
			error_dialog(self, 'Error',
					  f'Failed to unpack book: {str(e)}',
					  show=True)

	def refresh_opf_content(self):
		"""Refresh the OPF content from the current book"""
		try:
			debug_print('OPFHelper: Refreshing OPF content')
			self.load_current_book()
			# Show a brief notification that refresh was successful
			info_dialog(self, 'Refreshed', 'OPF content has been refreshed from the current book.', show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to refresh OPF content: {str(e)}')
			error_dialog(self, 'Error', f'Failed to refresh OPF content: {str(e)}', show=True)

	def update_book_info_label(self):
		"""Update the book info label with current book's details"""
		try:
			book_id = self.book_ids[self.current_book_index]

			# Get book details
			title = self.db.field_for("title", book_id)
			authors = self.db.field_for("authors", book_id)
			authors_text = " & ".join(authors) if authors else "Unknown"
			pubdate = self.db.field_for("pubdate", book_id)
			publisher = self.db.field_for("publisher", book_id)

			# Build info text with HTML formatting - conditionally show ID
			if self.show_book_id:
				info_text = f'<span style="font-weight:500">  🆔 {book_id}  📖 {title} by {authors_text}</span>'
			else:
				info_text = f'<span style="font-weight:500">  📖 {title} by {authors_text}</span>'

			if pubdate:
				pub_year = pubdate.year if hasattr(pubdate, 'year') else ''
				if pub_year:
					info_text += f' 🏛️ Published in {pub_year}'

			if publisher:
				info_text += f' by {publisher}'

			# Import human_readable function from calibre
			from calibre.ebooks.metadata.book.base import human_readable

			formats = self.db.formats(book_id)

			if formats:
				format_parts = []
				for f in formats:
					f_path = self.db.format_abspath(book_id, f)
					if os.path.isfile(f_path):
						size_bytes = os.path.getsize(f_path)
						# Use custom formatting for very small files
						if size_bytes < 102400:  # Less than 100KB
							size_str = f"{size_bytes / 1024:.1f} KB"  # Show in KB instead
						else:
							# Use the calibre function for larger files
							size_str = human_readable(size_bytes)
						format_parts.append(f'{f}: {size_str}')
				if format_parts:
					info_text += '   💾  ' + ' | '.join(format_parts)
			self.book_info_label.setText(info_text)
		except Exception as e:
			debug_print(f"Error updating book info: {e}")
			self.book_info_label.setText("")

	def add_action_buttons(self, toolbar_layout):
		"""Add all action buttons to the toolbar with consistent spacing"""
		# Add zoom buttons to toolbar
		zoom_in_button = QPushButton(get_icon('images/zoom_in.png'), '', self)
		zoom_in_button.setFixedWidth(32)
		zoom_in_button.setToolTip("Increase font size in XML view")
		zoom_in_button.clicked.connect(self.zoom_in)
		toolbar_layout.addWidget(zoom_in_button)

		zoom_out_button = QPushButton(get_icon('images/zoom_out.png'), '', self)
		zoom_out_button.setFixedWidth(32)
		zoom_out_button.setToolTip("Decrease font size in XML view")
		zoom_out_button.clicked.connect(self.zoom_out)
		toolbar_layout.addWidget(zoom_out_button)

		# Add edit book button to toolbar
		edit_book_button = QPushButton("Edit Book")
		edit_book_button.setIcon(QIcon.ic('edit_book.png'))
		edit_book_button.setToolTip("Open book in Calibre's Editor")
		edit_book_button.clicked.connect(self.edit_book)
		toolbar_layout.addWidget(edit_book_button)

		# Add view book button to toolbar
		view_book_button = QPushButton("View Book")
		view_book_button.setIcon(QIcon.ic('view.png'))
		view_book_button.setToolTip("Open book in Calibre's E-book Viewer")
		view_book_button.clicked.connect(self.view_book)
		toolbar_layout.addWidget(view_book_button)

		# Add edit ToC button to toolbar
		edit_toc_button = QPushButton("Edit ToC")
		edit_toc_button.setIcon(QIcon.ic('toc.png'))
		edit_toc_button.setToolTip("Edit Table of Contents of the book")
		edit_toc_button.clicked.connect(self.edit_toc)
		toolbar_layout.addWidget(edit_toc_button)

		# Add unpack book button to toolbar
		unpack_book_button = QPushButton("Unpack Book")
		unpack_book_button.setIcon(QIcon.ic('unpack-book.png'))
		unpack_book_button.setToolTip("Unpack book into individual components for editing")
		unpack_book_button.clicked.connect(self.unpack_book)
		toolbar_layout.addWidget(unpack_book_button)

		# Add refresh button to toolbar
		refresh_button = QPushButton("Refresh")
		refresh_button.setIcon(QIcon.ic('view-refresh.png'))
		refresh_button.setToolTip("Refresh OPF content from the current book")
		refresh_button.clicked.connect(self.refresh_opf_content)
		toolbar_layout.addWidget(refresh_button)

		export_button = QPushButton(get_icon('images/export_icon.png'), "Export OPF", self)
		export_button.setToolTip("Save OPF content to file")
		export_button.clicked.connect(self.export_opf)
		toolbar_layout.addWidget(export_button)



		edit_metadata_button = QPushButton("Open MDE")
		edit_metadata_button.setIcon(QIcon.ic('edit_input.png'))
		edit_metadata_button.setToolTip("Open Edit Metadata dialog for the current book")
		edit_metadata_button.clicked.connect(self.edit_metadata)
		toolbar_layout.addWidget(edit_metadata_button)

		copy_button = QPushButton("Copy XML")
		copy_button.setIcon(QIcon.ic('edit-copy.png'))
		copy_button.setToolTip("Copy XML content to clipboard (from XML tab)")
		copy_button.clicked.connect(lambda: self.copy_to_clipboard())
		toolbar_layout.addWidget(copy_button)

	def on_opf_selection_changed(self, index):
		"""Handle when the user selects a different OPF file from the dropdown"""
		if index < 0 or index >= len(self.current_book_opfs):
			return

		# Get the selected OPF file path
		selected_opf = self.current_book_opfs[index]
		book_id = self.book_ids[self.current_book_index]

		# Save the selection for this book
		self.book_opf_selections[book_id] = selected_opf

		# Load the selected OPF file
		self.load_opf_content(selected_opf)

	def load_opf_content(self, opf_path):
		"""Load OPF content from specified path within the current EPUB"""
		try:
			book_id = self.book_ids[self.current_book_index]
			title = self.db.field_for("title", book_id)
			epub_path = getattr(self, 'current_archive_path', None)
			if not epub_path:
				epub_path = self.db.format_abspath(book_id, 'EPUB')

			with zipfile.ZipFile(epub_path, 'r') as zf:
				# Read the OPF file
				content = zf.read(opf_path).decode('utf-8')

				# Extract EPUB version
				version = "Unknown"
				version_match = re.search(r'<package[^>]*?\s+version\s*=\s*["\']([^"\']+)["\']', content, re.IGNORECASE)
				if not version_match:
					version_match = re.search(r'package[^>]*?\s+version\s*=\s*["\']([^"\']+)["\']', content, re.IGNORECASE)
				if not version_match:
					version_match = re.search(r'version\s*=\s*["\']([^"\']+)["\']', content)

				if version_match:
					version = f"EPUB {version_match.group(1)}"
					# Set color based on version
					if "3.0" in version:
						version_color = "#FFA500" if self.is_dark else "#FF8C00"
						self.version_label.setStyleSheet(f"""
							QLabel {{
								padding: 2px 8px;
								border-radius: 4px;
								color: {version_color};
								font-weight: bold;
								background-color: transparent;
								border: 1px solid {version_color};
							}}
						""")
					else:
						self.version_label.setStyleSheet("""
							QLabel {
								padding: 2px 8px;
								border-radius: 4px;
								color: palette(link);
								font-weight: bold;
								background-color: transparent;
								border: 1px solid palette(link);
							}
						""")
				self.version_label.setText(version)

				# Update UI
				self.setWindowTitle(f'OPF Helper - {title} ({opf_path})')
				self.text_edit.setPlainText(content)
				self._refresh_search_after_content_change()
				self.xml_content = content
				self.current_opf_path = opf_path
				self.populate_tree()

				# Also update the cover based on this OPF
				self.load_cover(opf_path)

		except Exception as e:
			debug_print(f"OPFHelper ERROR: Failed to load OPF content: {str(e)}")
			error_dialog(self.gui, 'Error',
					   f'Failed to read OPF content from:\n{opf_path}\n\nError: {str(e)}',
					   show=True)

	def load_cover(self, opf_path):
		"""Load cover image for the current book based on the selected OPF"""
		try:
			book_id = self.book_ids[self.current_book_index]

			def _get_db_cover_bytes():
				# Try several Calibre DB API shapes (new_api vs old API)
				for getter in (
					lambda: self.db.cover(book_id, index_is_id=True),
					lambda: self.db.cover(book_id),
					lambda: self.gui.current_db.cover(book_id, index_is_id=True),
					lambda: self.gui.current_db.cover(book_id),
					lambda: self.gui.current_db.new_api.cover(book_id),
				):
					try:
						cover = getter()
						if cover:
							return cover
					except Exception:
						continue
				return None

			# Prefer ZIP-based formats for in-book cover lookup; if none, fall back to DB cover.
			zip_path = None
			for fmt in ('KEPUB', 'EPUB', 'HTMLZ'):
				try:
					p = self.db.format_abspath(book_id, fmt)
				except Exception:
					try:
						p = self.gui.current_db.new_api.format_abspath(book_id, fmt)
					except Exception:
						p = None
				if p and os.path.isfile(p):
					zip_path = p
					break

			# AZW3-only (or missing ZIP): use Calibre DB cover and return
			if not zip_path or not opf_path or not isinstance(opf_path, str):
				cover = _get_db_cover_bytes()
				if cover:
					debug_print("OPFHelper: Using cover from Calibre database")
					self.cover_panel.show_cover(cover)
				else:
					debug_print("OPFHelper: No cover found for book")
					self.cover_panel.show_cover(None)
				return

			cover_found = False

			with zipfile.ZipFile(zip_path, 'r') as zf:
				# Method 1: Try parsing OPF for cover image reference
				try:
					content = zf.read(opf_path).decode('utf-8')
					root = ET.fromstring(content)

					# First look for meta cover tag (EPUB3 standard)
					meta_cover = root.find(".//{http://www.idpf.org/2007/opf}meta[@name='cover']")
					if meta_cover is not None:
						cover_id = meta_cover.get('content')
						if cover_id:
							# Find the item with this ID
							cover_item = root.find(f".//{http://www.idpf.org/2007/opf}item[@id='{cover_id}']")
							if cover_item is not None:
								cover_href = cover_item.get('href')
								if cover_href:
									# Resolve relative path to OPF directory
									opf_dir = os.path.dirname(opf_path)
									cover_path = os.path.normpath(os.path.join(opf_dir, cover_href)).replace('\\', '/')
									try:
										cover_data = zf.read(cover_path)
										self.cover_panel.show_cover(cover_data)
										cover_found = True
										debug_print(f"OPFHelper: Found cover via meta content: {cover_path}")
									except Exception as e:
										debug_print(f"OPFHelper: Error reading cover from meta reference: {str(e)}")

					# If not found, look for cover in manifest with cover properties or id containing 'cover'
					if not cover_found:
						for item in root.findall(".//{http://www.idpf.org/2007/opf}item"):
							item_id = item.get('id', '').lower()
							item_props = item.get('properties', '').lower()
							item_href = item.get('href', '')

							if ('cover' in item_props) or ('cover' in item_id):
								# Found cover reference, try to load it
								try:
									# Resolve relative path to OPF directory
									opf_dir = os.path.dirname(opf_path)
									cover_path = os.path.normpath(os.path.join(opf_dir, item_href)).replace('\\', '/')
									debug_print(f"OPFHelper: Trying cover path: {cover_path}")
									cover_data = zf.read(cover_path)
									self.cover_panel.show_cover(cover_data)
									cover_found = True
									debug_print(f"OPFHelper: Found cover from item: {item_href}")
									break
								except Exception as e:
									debug_print(f"OPFHelper: Failed to read cover from item reference: {str(e)}")
				except Exception as e:
					debug_print(f"OPFHelper: Failed to parse OPF for cover: {str(e)}")

				# Method 2: Try common cover filenames relative to current OPF if not found yet
				if not cover_found:
					debug_print(f"OPFHelper: Trying common cover filenames relative to: {opf_path}")
					opf_dir = os.path.dirname(opf_path)

					# Check common cover filenames relative to the OPF directory
					cover_relative_paths = [
						'cover.jpg', 'cover.jpeg', 'cover.png',
						'images/cover.jpg', 'images/cover.jpeg', 'images/cover.png',
						'../images/cover.jpg', '../images/cover.jpeg', '../images/cover.png',
					]

					for rel_path in cover_relative_paths:
						try:
							if opf_dir:
								cover_path = os.path.normpath(os.path.join(opf_dir, rel_path)).replace('\\', '/')
							else:
								cover_path = rel_path

							if cover_path in zf.namelist():
								debug_print(f"OPFHelper: Found cover at: {cover_path}")
								cover_data = zf.read(cover_path)
								self.cover_panel.show_cover(cover_data)
								cover_found = True
								break
						except Exception as e:
							debug_print(f"OPFHelper: Error with cover path {rel_path}: {str(e)}")

				# Method 3: Try common absolute cover paths if not found yet
				if not cover_found:
					cover_files = [
						'cover.jpg', 'cover.jpeg', 'cover.png',
						'OEBPS/cover.jpg', 'OEBPS/cover.jpeg', 'OEBPS/cover.png',
						'OEBPS/images/cover.jpg', 'OEBPS/images/cover.jpeg', 'OEBPS/images/cover.png',
						'META-INF/cover.jpg', 'META-INF/cover.jpeg', 'META-INF/cover.png'
					]

					for cover_file in cover_files:
						try:
							if cover_file in zf.namelist():
								debug_print(f"OPFHelper: Found cover at absolute path: {cover_file}")
								cover_data = zf.read(cover_file)
								self.cover_panel.show_cover(cover_data)
								cover_found = True
								break
						except:
							continue

			# Method 4: Try getting cover from Calibre's metadata if still not found
			if not cover_found:
				try:
					from calibre.ebooks.metadata.meta import get_metadata
					with open(zip_path, 'rb') as f:
						mi = get_metadata(f, 'epub')
						if mi.cover_data and mi.cover_data[1]:
							debug_print("OPFHelper: Found cover in EPUB metadata")
							self.cover_panel.show_cover(mi.cover_data[1])
							cover_found = True
				except Exception as e:
					debug_print(f"OPFHelper: Failed to extract cover from EPUB metadata: {str(e)}")

			# Method 5: Final fallback - try Calibre's database cover
			if not cover_found:
				try:
					cover = _get_db_cover_bytes()
					if cover:
						debug_print("OPFHelper: Using cover from Calibre database")
						self.cover_panel.show_cover(cover)
						cover_found = True
				except Exception as e:
					debug_print(f"OPFHelper: Failed to get cover from Calibre database: {str(e)}")

			if not cover_found:
				debug_print("OPFHelper: No cover found for book")
				self.cover_panel.show_cover(None)  # Clear cover display

		except Exception as e:
			debug_print(f"OPFHelper ERROR: Failed to load cover: {str(e)}")
			from calibre_plugins.opf_helper import DEBUG_OPF_HELPER
			if DEBUG_OPF_HELPER:
				traceback.print_exc()
			self.cover_panel.show_cover(None)

	def highlight_match(self, position_info):
		"""Highlight a specific match in the text"""
		try:
			position, length = position_info

			# Clear previous extra selections
			self.text_edit.setExtraSelections([])

			# Create cursor and set selection using positions directly
			cursor = QTextCursor(self.text_edit.document())
			cursor.setPosition(position)
			cursor.movePosition(QTextCursor.MoveOperation.NextCharacter, QTextCursor.MoveMode.KeepAnchor, length)

			# Apply cursor and ensure visible
			self.text_edit.setTextCursor(cursor)
			self.text_edit.ensureCursorVisible()

			# Add extra highlighting with background color so it's visible even with syntax highlighter
			try:
				from calibre.gui2 import is_dark_theme
				is_dark = is_dark_theme()
			except:
				is_dark = False

			highlight_color = QColor('#FFFF00' if not is_dark else '#FFD700')  # Yellow for light, gold for dark
			selection = self.text_edit.ExtraSelection()
			selection.cursor = cursor
			selection.format.setBackground(highlight_color)
			self.text_edit.setExtraSelections([selection])

			return True
		except Exception as e:
			debug_print(f"OPFHelper ERROR: Highlight error: {e}")
			traceback.print_exc()
			return False

# Simple function to check if we're in dark mode
def is_dark_theme():
	return QApplication.instance().is_dark_theme

try:
	from calibre.utils.localization import _
except Exception:
	def _(x):
		return x

class ShowOPFAction(InterfaceAction):
	# Main plugin action registration: name and action_spec ensure Calibre
	# creates the toolbar/menu action and exposes the entry in Preferences > Shortcuts.
	name = _('OPF Helper')
	action_spec = (_('OPF Helper'), None, _('Inspect and analyze OPF content of EPUB books'), '')


	def _opfhelper_extract_opf_path(self, db, book_id, fmt):
		import zipfile, shutil
		fmt = fmt.upper()
		path = db.format_abspath(book_id, fmt)
		if not path or not os.path.exists(path):
			return None
		if fmt in ("EPUB", "KEPUB", "AZW3", "HTMLZ"):
			try:
				with zipfile.ZipFile(path, 'r') as zf:
					opf_name = None
					# Prefer META-INF/container.xml rootfile full-path if present
					try:
						from lxml import etree
						with zf.open('META-INF/container.xml') as cf:
							cdata = cf.read()
						cdoc = etree.fromstring(cdata)
						rootfiles = cdoc.findall('.//{urn:oasis:names:tc:opendocument:xmlns:container}rootfile')
						for rf in rootfiles:
							fp = (rf.get('full-path') or '').strip()
							if fp.lower().endswith('.opf'):
								opf_name = fp
								break
					except Exception:
						opf_name = None

					if not opf_name:
						opf_candidates = [f for f in zf.namelist() if f.lower().endswith('.opf')]
						if not opf_candidates:
							return None
						opf_name = opf_candidates[0]
					tmp = tempfile.NamedTemporaryFile(delete=False, suffix='.opf')
					with zf.open(opf_name) as src, open(tmp.name, 'wb') as dst:
						shutil.copyfileobj(src, dst)
					return (tmp.name, opf_name)
			except Exception:
				return None
		return None

	def _opfhelper_extract_opf_bytes_from_zipfile(self, zf):
		"""Return (opf_bytes, opf_name_in_zip) from an already-open ZipFile.

		This avoids writing temp OPF files, which can be very slow on Windows due
		to extra disk I/O and AV scanning in %LOCALAPPDATA%\\Temp.
		"""
		opf_name = None
		# Prefer META-INF/container.xml rootfile full-path if present
		try:
			from lxml import etree
			with zf.open('META-INF/container.xml') as cf:
				cdata = cf.read()
			cdoc = etree.fromstring(cdata)
			rootfiles = cdoc.findall('.//{urn:oasis:names:tc:opendocument:xmlns:container}rootfile')
			for rf in rootfiles:
				fp = (rf.get('full-path') or '').strip()
				if fp.lower().endswith('.opf'):
					opf_name = fp
					break
		except Exception:
			opf_name = None

		if not opf_name:
			try:
				opf_candidates = [f for f in zf.namelist() if (f or '').lower().endswith('.opf')]
			except Exception:
				opf_candidates = []
			if not opf_candidates:
				return (None, None)
			opf_name = opf_candidates[0]

		try:
			opf_bytes = zf.read(opf_name)
		except Exception:
			return (None, None)
		return (opf_bytes, opf_name)

	def _opfhelper_guess_opf_version_from_bytes(self, opf_bytes):
		from lxml import etree
		try:
			parser = etree.XMLParser(recover=True)
			root = etree.fromstring(opf_bytes or b'', parser)
			return root.get('version', '2.0')
		except Exception:
			return '2.0'

	def _opfhelper_validate_opf_bytes(self, opf_bytes, version):
		from calibre_plugins.opf_helper.schema_utils import load_schema
		from lxml import etree
		try:
			parser = etree.XMLParser(recover=True)
			root = etree.fromstring(opf_bytes or b'', parser)
			doc = etree.ElementTree(root)
			schema = load_schema(version)
			errors = []
			if schema is not None:
				try:
					schema.assertValid(doc)
				except etree.DocumentInvalid as e:
					for error in e.error_log:
						errors.append(str(error.message))
			return errors
		except Exception as e:
			return [f"Parse error: {e}"]

	def _opfhelper_validate_opf_file(self, opf_path, version):
		from calibre_plugins.opf_helper.schema_utils import load_schema
		from lxml import etree
		try:
			parser = etree.XMLParser(recover=True)
			doc = etree.parse(opf_path, parser)
			schema = load_schema(version)
			errors = []
			if schema is not None:
				try:
					schema.assertValid(doc)
				except etree.DocumentInvalid as e:
					for error in e.error_log:
						errors.append(str(error.message))
			return errors
		except Exception as e:
			return [f"Parse error: {e}"]

	def _opfhelper_guess_opf_version(self, opf_path):
		from lxml import etree
		try:
			parser = etree.XMLParser(recover=True)
			doc = etree.parse(opf_path, parser)
			root = doc.getroot()
			version = root.get('version', '2.0')
			return version
		except Exception:
			return '2.0'

	def _opfhelper_deep_validate_zf(self, zf, opf_doc, opf_name_in_zip=None):
		"""Perform additional checks that require the EPUB archive itself.
		Checks performed:
		- guide/reference hrefs are declared in the manifest
		- referenced resources exist in the EPUB archive
		- for spine XHTML items (e.g., cover.htm), basic sanity checks like presence of <head><title>
		Returns list of error strings.
		"""
		errors = []
		try:
			import posixpath
			try:
				zip_entries = set(zf.namelist())
			except Exception:
				zip_entries = None
			# Determine the OPF directory inside the archive so relative hrefs resolve correctly
			opf_dir = ''
			try:
				if opf_name_in_zip:
					opf_dir = posixpath.dirname(opf_name_in_zip) or ''
			except Exception:
				opf_dir = ''

			def _norm_href(h):
				# Remove fragment and normalize separators for zip entries
				try:
					h = (h or '').strip().replace('\\', '/')
				except Exception:
					return ''
				if not h:
					return ''
				return h.split('#', 1)[0]

			def _resolve_zip_entry(h):
				h = _norm_href(h)
				if not h:
					return ''
				# Do not attempt to resolve absolute URLs into zip paths
				if '://' in h:
					return ''
				try:
					return posixpath.normpath(posixpath.join(opf_dir, h)) if opf_dir else posixpath.normpath(h)
				except Exception:
					return h

			# Build manifest maps (id -> item, resolved_zip_entry -> item)
			manifest_by_id = {}
			manifest_by_entry = {}
			for item in opf_doc.findall('.//{http://www.idpf.org/2007/opf}manifest/{http://www.idpf.org/2007/opf}item'):
				iid = (item.get('id') or '').strip()
				href = (item.get('href') or '').strip()
				entry = _resolve_zip_entry(href)
				if iid:
					manifest_by_id[iid] = item
				if entry:
					manifest_by_entry[entry] = item

			# Check guide references are in the manifest
			for ref in opf_doc.findall('.//{http://www.idpf.org/2007/opf}guide/{http://www.idpf.org/2007/opf}reference'):
				href = (ref.get('href') or '').strip()
				if not href:
					continue
				entry = _resolve_zip_entry(href)
				if entry and entry not in manifest_by_entry:
					errors.append(f"File listed in reference element in guide was not declared in OPF manifest: {entry}")
					errors.append(f"Referenced resource \"{entry}\" is not declared in the OPF manifest.")

			# If epub_path is available, open archive and inspect referenced resources
			# Guide resources should exist in the archive
			for ref in opf_doc.findall('.//{http://www.idpf.org/2007/opf}guide/{http://www.idpf.org/2007/opf}reference'):
				href = (ref.get('href') or '').strip()
				entry = _resolve_zip_entry(href)
				if not entry:
					continue
				try:
					if zip_entries is not None:
						if entry not in zip_entries:
							errors.append(f"Referenced resource \"{entry}\" is missing from the EPUB archive.")
					else:
						zf.getinfo(entry)
				except Exception:
					errors.append(f"Referenced resource \"{entry}\" is missing from the EPUB archive.")

			# Sanity check spine XHTML files for <head><title>
			spine_entries = []
			try:
				for ir in opf_doc.findall('.//{http://www.idpf.org/2007/opf}spine/{http://www.idpf.org/2007/opf}itemref'):
					idref = (ir.get('idref') or '').strip()
					if not idref:
						continue
					item = manifest_by_id.get(idref)
					if item is None:
						continue
					href = (item.get('href') or '').strip()
					entry = _resolve_zip_entry(href)
					if entry:
						spine_entries.append((entry, item))
			except Exception:
				spine_entries = []

			for entry, item in spine_entries:
				mtype = (item.get('media-type') or '').lower()
				if ('application/xhtml+xml' not in mtype) and (not entry.lower().endswith(('.htm', '.html', '.xhtml'))):
					continue
				try:
					with zf.open(entry) as f:
						data = f.read()
				except Exception:
					continue
				try:
					from lxml import html
					doc = html.fromstring(data)
					titles = doc.xpath('//head/title')
					if not titles or not ((titles[0].text or '').strip()):
						errors.append('Error while parsing file: element "head" incomplete; missing required element "title" (file: %s)' % entry)
				except Exception:
					# If parsing fails, report a generic parse error (truthful but not claiming epubcheck equivalence)
					errors.append('Error while parsing file: could not parse XHTML (file: %s)' % entry)

		except Exception as e:
			try:
				debug_print(f"OPFHelper: Deep validation failed: {e}")
			except Exception:
				pass
		return errors

	def _opfhelper_deep_validate(self, epub_path, opf_doc, opf_name_in_zip=None):
		"""Backward-compatible wrapper that opens the archive and calls _opfhelper_deep_validate_zf."""
		errors = []
		try:
			if epub_path and os.path.exists(epub_path):
				import zipfile
				with zipfile.ZipFile(epub_path, 'r') as zf:
					return self._opfhelper_deep_validate_zf(zf, opf_doc, opf_name_in_zip=opf_name_in_zip)
		except Exception as e:
			try:
				debug_print(f"OPFHelper: Deep validation wrapper failed: {e}")
			except Exception:
				pass
		return errors

	def _opfhelper_get_title(self, db, book_id):
		try:
			return db.field_for('title', book_id)
		except Exception:
			return str(book_id)

	def _opfhelper_get_formats(self, db, book_id):
		try:
			return db.formats(book_id) or []
		except Exception:
			return []

	def _validate_all_opfs_log_errors(self):
		from calibre.gui2 import info_dialog, error_dialog
		try:
			from calibre.gui2.threaded_jobs import ThreadedJob
		except Exception:
			from calibre.gui2.jobs import ThreadedJob

		db = self.gui.current_db.new_api
		model = self.gui.library_view.model()
		scope = prefs.get('bulk_validate_opfs_scope', 'Library')
		book_ids = []

		# Clear any existing marks BEFORE starting bulk validation
		try:
			act = None
			try:
				act = self.gui.iactions.get('Mark Books')
			except Exception:
				act = None
			if act is not None and hasattr(act, 'clear_all_marked'):
				act.clear_all_marked()
			else:
				try:
					self.gui.current_db.data.set_marked_ids(())
				except Exception:
					pass
		except Exception:
			pass

		if scope == 'Selection':
			book_ids = self._get_selected_book_ids()
			if not book_ids:
				info_dialog(self.gui, _('Validation Complete'), _('No selected books.'), show=True)
				return
		else:
			# Clear active queries/restrictions BEFORE starting library-scoped validation
			try:
				self.gui.show_library_view()
				library_view = self.gui.library_view
				# Clear virtual libraries/base restrictions
				try:
					if library_view.model().db.data.get_base_restriction_name():
						self.gui.apply_virtual_library()
				except Exception:
					pass
				# Clear named search restrictions
				try:
					if library_view.model().db.data.get_search_restriction_name():
						self.gui.apply_named_search_restriction()
				except Exception:
					pass
				# Clear active search query
				try:
					self.gui.search.set_search_string('')
				except Exception:
					pass
				# Clear current selection if supported
				try:
					if hasattr(library_view, 'clear_selection'):
						library_view.clear_selection()
				except Exception:
					pass
			except Exception:
				pass

			# Library scope: validate books in the current view (respects current search/filter view).
			try:
				row_count = model.rowCount()
			except TypeError:
				from PyQt5.QtCore import QModelIndex
				row_count = model.rowCount(QModelIndex())

			# Collect book IDs with a progress dialog to avoid freezing the UI on large libraries
			from PyQt5.QtWidgets import QProgressDialog
			progress = QProgressDialog(_('Preparing OPF validation...'), _('Cancel'), 0, row_count, self.gui)
			progress.setWindowTitle(_('Preparing Validation'))
			progress.setMinimumDuration(0)
			progress.setValue(0)
			progress.setWindowModality(Qt.WindowModal)
			# Ensure the dialog is shown and painted before starting the potentially slow ID collection
			try:
				progress.show()
				QApplication.processEvents()
			except Exception:
				pass

			for i in range(row_count):
				if progress.wasCanceled():
					progress.close()
					info_dialog(self.gui, _('Validation Cancelled'), _('Validation cancelled by user.'), show=True)
					return
				# Update progress and allow the event loop to process to stay responsive
				progress.setValue(i+1)
				progress.setLabelText(_('Preparing book %d of %d...') % (i+1, row_count))
				try:
					book_ids.append(model.id(i))
				except Exception:
					# If fetching an id fails, skip it but continue
					pass
				QApplication.processEvents()

			progress.close()
		if not book_ids:
			info_dialog(self.gui, _('Validation Complete'), _('No books found in the current view.'), show=True)
			return

		try:
			desc = _('Bulk validate OPF for %d books') % len(book_ids)
		except Exception:
			desc = 'Bulk validate OPF'

		try:
			from calibre_plugins.opf_helper import debug_print
			from calibre.gui2 import Dispatcher
			debug_print(f"Creating ThreadedJob for bulk validation: desc={desc}")
			job = ThreadedJob(
				'opf_helper_validate_all',
				desc,
				self._opfhelper_validate_all_opfs_job,
				(book_ids, db),
				{},
				Dispatcher(self._opfhelper_validate_all_opfs_done)  # Wrap callback with Dispatcher for thread safety
			)
			debug_print(f"ThreadedJob created: {job}")
			debug_print(f"job.callback set to: {job.callback}")
			self.gui.job_manager.run_threaded_job(job)
			debug_print(f"ThreadedJob submitted to job_manager: {job}")
			try:
				self.gui.status_bar.show_message(_('OPF validation started'), 3000)
			except Exception:
				pass
			self._opfhelper_validate_all_job = job
		except Exception as e:
			error_dialog(self.gui, _('Validation Error'), _('Failed to start validation job: %s') % (e,), show=True)


	def _opfhelper_validate_all_opfs_job(self, book_ids, db, log=None, abort=None, notifications=None):
		import os
		import datetime
		import time

		book_ids = list(book_ids or [])
		total_books = len(book_ids) or 1
		errors = []
		scanned_tasks = 0
		error_book_ids = set()

		# Precompute validation tasks in this thread (avoid DB access from worker threads)
		zip_like_formats = {"EPUB", "KEPUB", "HTMLZ"}
		sequential_fallback_tasks = []
		parallel_tasks = []
		for book_id in book_ids:
			if abort is not None and abort.is_set():
				break
			title = self._opfhelper_get_title(db, book_id)
			fmts = self._opfhelper_get_formats(db, book_id)
			for fmt in (fmts or []):
				fmt_u = (fmt or '').upper()
				try:
					path = db.format_abspath(book_id, fmt_u)
				except Exception:
					path = None
				if not path or not os.path.exists(path):
					continue
				task = {'book_id': book_id, 'title': title, 'fmt': fmt_u, 'path': path}
				if fmt_u in zip_like_formats:
					parallel_tasks.append(task)
				else:
					sequential_fallback_tasks.append(task)

		total_tasks = (len(parallel_tasks) + len(sequential_fallback_tasks)) or 1

		def _notify_progress(done, total, extra=''):
			if notifications is None:
				return
			try:
				msg = _('Validated %d of %d') % (done, total)
				if extra:
					msg = msg + ' - ' + extra
				notifications.put((float(done) / float(total or 1), msg))
			except Exception:
				pass


		# Move imports outside thread functions for efficiency
		import zipfile
		from lxml import etree

		def _validate_zip_task(t):
			# Returns (error_block_or_None, book_id_or_None)
			book_id = t.get('book_id')
			title = t.get('title')
			fmt = t.get('fmt')
			epup = t.get('path')
			try:
				with zipfile.ZipFile(epup, 'r') as zf:
					opf_bytes, opf_name_in_zip = self._opfhelper_extract_opf_bytes_from_zipfile(zf)
					if not opf_bytes:
						return (None, None)
					version = self._opfhelper_guess_opf_version_from_bytes(opf_bytes)
					val_errors = self._opfhelper_validate_opf_bytes(opf_bytes, version)
					# Deep checks, reusing the open zipfile
					try:
						parser = etree.XMLParser(recover=True)
						root = etree.fromstring(opf_bytes, parser)
						opf_doc = etree.ElementTree(root)
						deep_errors = self._opfhelper_deep_validate_zf(zf, opf_doc, opf_name_in_zip=opf_name_in_zip)
						if deep_errors:
							val_errors = (val_errors or []) + deep_errors
					except Exception:
						# Ignore deep check failures but keep base validation
						pass

					if not val_errors:
						return (None, None)
					block = (
						f"Book: {title} (ID: {book_id}, Format: {fmt}, OPF v{version})\n" +
						"\n".join(val_errors) +
						"\n---\n"
					)
					return (block, book_id)
			except zipfile.BadZipFile:
				return (f"Book: {title} (ID: {book_id}, Format: {fmt})\nBadZipFile: could not open as zip\n---\n", book_id)
			except Exception as e:
				return (f"Book: {title} (ID: {book_id}, Format: {fmt})\nError: {e}\n---\n", book_id)

		def _validate_seq_task(t):
			# Returns (error_block_or_None, book_id_or_None)
			book_id = t.get('book_id')
			title = t.get('title')
			fmt = t.get('fmt')
			opf_extracted = self._opfhelper_extract_opf_path(db, book_id, fmt)
			if not opf_extracted:
				return (None, None)
			opf_path, opf_name_in_zip = opf_extracted
			epup = t.get('path')
			try:
				version = self._opfhelper_guess_opf_version(opf_path)
				val_errors = self._opfhelper_validate_opf_file(opf_path, version)
				try:
					parser = etree.XMLParser(recover=True)
					opf_doc = etree.parse(opf_path, parser)
					deep_errors = self._opfhelper_deep_validate(epup, opf_doc, opf_name_in_zip=opf_name_in_zip)
					if deep_errors:
						val_errors = (val_errors or []) + deep_errors
				except Exception:
					pass
				if val_errors:
					block = (
						f"Book: {title} (ID: {book_id}, Format: {fmt}, OPF v{version})\n" +
						"\n".join(val_errors) +
						"\n---\n"
					)
					return (block, book_id)
			finally:
				try:
					os.remove(opf_path)
				except Exception:
					pass
			return (None, None)


		# Use ThreadPoolExecutor for both zip-like and non-zip tasks
		import concurrent.futures
		all_tasks = [("zip", t) for t in parallel_tasks] + [("seq", t) for t in sequential_fallback_tasks]
		if all_tasks:
			try:
				try:
					max_workers = int(os.environ.get('OPFHELPER_BULK_MAXWORKERS', prefs.get('bulk_max_workers', 7)))
					max_workers = min(max_workers, len(all_tasks))
				except Exception:
					max_workers = min(5, len(all_tasks))
				last_notify = 0.0
				from calibre_plugins.opf_helper import schema_utils
				with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as ex:
					futures = []
					for kind, t in all_tasks:
						if kind == "zip":
							futures.append(ex.submit(_validate_zip_task, t))
						else:
							futures.append(ex.submit(_validate_seq_task, t))
					for fut in concurrent.futures.as_completed(futures):
						if abort is not None and abort.is_set():
							for f in futures:
								try:
									f.cancel()
								except Exception:
									pass
							if log is not None:
								try:
									log.error('Aborting...')
								except Exception:
									pass
							break
						try:
							block, bid = fut.result()
						except Exception:
							block, bid = (None, None)
						scanned_tasks += 1
						if block:
							errors.append(block)
							if bid is not None:
								try:
									error_book_ids.add(int(bid))
								except Exception:
									error_book_ids.add(bid)
						now = time.time()
						if (now - last_notify) > 0.25 or scanned_tasks == total_tasks:
							last_notify = now
							_notify_progress(scanned_tasks, total_tasks)
			except Exception:
				# If the executor fails for any reason, fall back to sequential processing
				for kind, t in all_tasks:
					if abort is not None and abort.is_set():
						break
					if kind == "zip":
						block, bid = _validate_zip_task(t)
					else:
						block, bid = _validate_seq_task(t)
					scanned_tasks += 1
					if block:
						errors.append(block)
						if bid is not None:
							try:
								error_book_ids.add(int(bid))
							except Exception:
								error_book_ids.add(bid)
					_notify_progress(scanned_tasks, total_tasks)

		out_path = None
		if errors:
			ts = datetime.datetime.now().strftime('%Y-%m-%d_%H-%M-%S')
			out_path = os.path.join(os.path.expanduser('~'), f'opf_helper_validation_errors_{ts}.txt')
			with open(out_path, 'w', encoding='utf-8') as f:
				f.write("\n".join(errors))

		return {
			'out_path': out_path,
			'scanned': scanned_tasks,
			'total': total_books,
			'error_blocks': len(errors),
			'error_book_ids': sorted(error_book_ids),
		}


	def _opfhelper_validate_all_opfs_done(self, job):
		from calibre.gui2 import info_dialog, error_dialog

		if getattr(job, 'failed', False):
			details = getattr(job, 'details', '') or ''
			error_dialog(self.gui, _('Validation Error'), _('Bulk validation failed.'), details, show=True)
			return

		res = getattr(job, 'result', None) or {}
		error_book_ids = res.get('error_book_ids') or []
		out_path = res.get('out_path')

		# Scanner-like behavior: clear existing marks, mark failing books, then show only marked.
		try:
			act = None
			try:
				act = self.gui.iactions.get('Mark Books')
			except Exception:
				act = None

			# Clear marks
			if act is not None and hasattr(act, 'clear_all_marked'):
				act.clear_all_marked()
			else:
				try:
					self.gui.current_db.data.set_marked_ids(())
				except Exception:
					pass

			# Mark failing
			if error_book_ids:
				if act is not None and hasattr(act, 'add_ids'):
					act.add_ids(error_book_ids)
				else:
					try:
						self.gui.current_db.data.add_marked_ids(error_book_ids)
					except Exception:
						pass
				try:
					self.gui.search.set_search_string('marked:true')
				except Exception:
					pass
		except Exception:
			pass

		# Completion message
		if out_path:
			info_dialog(
				self.gui,
				_('Validation Complete'),
				_('Validation errors written to:\n%s\n\nBooks with issues have been marked and are now displayed in the library view.') % out_path,
				show=True
			)
		else:
			if error_book_ids:
				info_dialog(
					self.gui,
					_('Validation Complete'),
					_('Issues found and failing books marked and displayed (marked:true).\n\nNote: OPF Helper is not EPUBCheck; results may differ from official epubcheck.'),
					show=True
				)
			else:
				info_dialog(
					self.gui,
					_('Validation Complete'),
					_('No issues were found by OPF Helper.\n\nNote: This is not EPUBCheck; results may differ from official epubcheck.'),
					show=True
				)

	def _normalize_formats(self, fmts):
		# Return a set of upper-case format strings from db.formats().
		if not fmts:
			return set()
		try:
			if isinstance(fmts, str):
				items = [x.strip() for x in fmts.split(',')]
			else:
				items = list(fmts)
		except Exception:
			items = []
		out = set()
		for x in items:
			try:
				s = (x or '').strip()
				if s:
					out.add(s.upper())
			except Exception:
				pass
		return out

	def _opf_version_from_xml(self, opf_xml):
		# Extract OPF package version from OPF XML string.
		# Returns e.g. '3.0', '2.0', or None.
		if not opf_xml:
			return None
		try:
			m = re.search(r'<\s*package\b[^>]*\bversion\s*=\s*[\"\']([^\"\']+)[\"\']', opf_xml, re.IGNORECASE)
			if m:
				v = (m.group(1) or '').strip()
				return v or None
		except Exception:
			pass
		return None

	def _opf_version_matches_filter(self, opf_xml, version_filter):
		v = self._opf_version_from_xml(opf_xml)
		if version_filter == '3.0':
			return v == '3.0'
		if version_filter == '2.0':
			return v == '2.0'
		if version_filter == 'not_3.0':
			# Preserve historical behaviour: if we cannot detect a version,
			# treat it as "not 3.0".
			return v != '3.0'
		return False

	def _sync_scope_action_checks(self):
		# Keep all Scope submenus in sync and mutually exclusive.
		scope = prefs.get('epub_version_scope', 'Library')
		is_library = scope == 'Library'
		pairs = [
			('scope_library_menu', 'scope_selection_menu'),
			('mobi_scope_library_menu', 'mobi_scope_selection_menu'),
			('all_scope_library_menu', 'all_scope_selection_menu'),
		]
		for lib_attr, sel_attr in pairs:
			try:
				lib_ac = getattr(self, lib_attr, None)
				sel_ac = getattr(self, sel_attr, None)
				if lib_ac is not None:
					lib_ac.setChecked(is_library)
				if sel_ac is not None:
					sel_ac.setChecked(not is_library)
			except Exception:
				pass

		# Bulk validator scope (independent from other scanners)
		bulk_scope = prefs.get('bulk_validate_opfs_scope', 'Library')
		bulk_is_library = bulk_scope == 'Library'
		bulk_pairs = [
			('bulk_validate_scope_library_menu', 'bulk_validate_scope_selection_menu'),
		]
		for lib_attr, sel_attr in bulk_pairs:
			try:
				lib_ac = getattr(self, lib_attr, None)
				sel_ac = getattr(self, sel_attr, None)
				if lib_ac is not None:
					lib_ac.setChecked(bulk_is_library)
				if sel_ac is not None:
					sel_ac.setChecked(not bulk_is_library)
			except Exception:
				pass

	def _get_selected_book_ids(self):
		rows = self.gui.library_view.selectionModel().selectedRows()
		if not rows:
			return []
		book_ids = []
		model = self.gui.library_view.model()
		for row in rows:
			try:
				bid = model.id(row.row())
				if bid is not None:
					book_ids.append(bid)
			except Exception:
				pass
		return book_ids

	def _library_search_ids(self, query):
		# Return matching book ids for a calibre search query.
		# Uses current_db.data.search_getting_ids() when available so it respects current search restrictions.
		try:
			view_db = getattr(self.gui, 'current_db', None)
			data = getattr(view_db, 'data', None)
			if data is not None and hasattr(data, 'search_getting_ids'):
				sr = getattr(data, 'search_restriction', None)
				try:
					ids = data.search_getting_ids(query, sr)
				except TypeError:
					ids = data.search_getting_ids(query)
				return list(ids or [])
		except Exception:
			pass

		# Fallback: try new_api search.
		try:
			db = self.gui.current_db.new_api
			ids = db.search(query)
			return list(ids or [])
		except Exception:
			return []

	def _get_scoped_book_ids(self, scope, required_any_formats=None):
		# Get book ids for the scan scope, optionally pre-filtered by formats.
		db = self.gui.current_db.new_api
		if scope == 'Selection':
			book_ids = self._get_selected_book_ids()
			if required_any_formats:
				out = []
				for bid in book_ids:
					try:
						fmts = self._normalize_formats(db.formats(bid))
						if fmts.intersection(required_any_formats):
							out.append(bid)
					except Exception:
						pass
				book_ids = out
			return book_ids

		# Library scope
		if required_any_formats:
			parts = []
			for fmt in sorted(required_any_formats):
				try:
					parts.append('formats:' + (fmt or '').lower())
				except Exception:
					pass
			query = ' or '.join(parts)
			ids = self._library_search_ids(query) if query else []
			if ids:
				return ids
		return db.all_book_ids()



	def genesis(self):
		debug_print('OPFHelper plugin: starting genesis()')

		# Register plugin icon resources so common_icons can find bundled images
		try:
			res = {}
			loader = getattr(self, 'load_resources', None)
			if callable(loader):
				try:
					# load_resources usually returns a dict of {name: bytes}
					res = dict(loader(PLUGIN_ICONS) or {})
				except Exception:
					res = {}
			# Fallback: load from local images/ folder when running from source
			if not res:
				base = os.path.join(os.path.dirname(__file__), 'images')
				for name in PLUGIN_ICONS:
					fn = os.path.join(base, os.path.basename(name))
					try:
						with open(fn, 'rb') as f:
							res[name] = f.read()
					except Exception:
						pass
			try:
				set_plugin_icon_resources(self.name, res)
			except Exception:
				pass
		except Exception:
			pass

		# --- Theme-aware icon logic (minimal, non-intrusive) ---
		def _opfhelper_update_toolbar_icon():
			try:
				from calibre.gui2 import is_dark_theme
				icon_path = 'images/icon-for-dark-theme.png' if is_dark_theme() else 'images/icon-for-light-theme.png'
				if hasattr(self, 'qaction') and self.qaction is not None:
					self.qaction.setIcon(get_icon(icon_path))
				try:
					ac = getattr(self, 'opf_show_direct', None)
					if ac is not None:
						ac.setIcon(get_icon(icon_path))
				except Exception:
					pass
				try:
					ac = getattr(self, 'opf_show_content', None)
					if ac is not None:
						ac.setIcon(get_icon(icon_path))
				except Exception:
					pass
			except Exception:
				pass

		def _opfhelper_theme_setup():
			# Try to connect to Calibre's theme_changed signal, fallback to timer
			try:
				from calibre.gui2 import gui
				g = gui()
				if hasattr(g, 'theme_changed'):
					g.theme_changed.connect(_opfhelper_update_toolbar_icon)
					return
			except Exception:
				pass
			# Fallback: poll for theme changes every second
			try:
				from calibre.gui2 import is_dark_theme
				self._opfhelper_last_theme = is_dark_theme()
				self._opfhelper_theme_timer = QTimer(self.gui if hasattr(self, 'gui') else None)
				def _opfhelper_check_theme():
					try:
						current = is_dark_theme()
						if current != self._opfhelper_last_theme:
							self._opfhelper_last_theme = current
							_opfhelper_update_toolbar_icon()
					except Exception:
						pass
				self._opfhelper_theme_timer.timeout.connect(_opfhelper_check_theme)
				self._opfhelper_theme_timer.start(1000)
			except Exception:
				pass

		_opfhelper_update_toolbar_icon()
		_opfhelper_theme_setup()


		# Create menu
		self.menu = QMenu()
		self.qaction.setMenu(self.menu)

		# Connect to menu first to prevent conflict with menu appearance
		self.menu.aboutToShow.connect(self.about_to_show_menu)

		# Register OPF Helper actions with Calibre's keyboard manager (unassigned by default)
		# and then build menus using those same QActions so shortcut hints render in menus.
		self._setup_opf_actions()
		self.rebuild_menus()

		# Add direct top-level actions for 'Show OPF Content (direct)' and 'Edit OPF (direct)'
		self._setup_direct_actions()

		# Connect the action trigger to main action last
		self.qaction.triggered.connect(self.show_opf)
		debug_print('OPFHelper plugin: genesis() complete')

	def _setup_opf_actions(self):
		"""Create QActions and register them with Calibre's keyboard manager. Users can assign their own keys in Preferences > Keyboard."""
		try:
			kb = getattr(self.gui, 'keyboard', None)
			if kb is None:
				return

			# Show OPF Content
			self.opf_show_content = QAction(_('Show OPF content'), self.gui)
			self.opf_show_content.triggered.connect(self.show_opf)
			self.opf_show_content.calibre_shortcut_unique_name = 'opf_helper.show_opf'
			self.gui.addAction(self.opf_show_content)
			# Register or replace existing shortcut entry
			if 'opf_helper.show_opf' in kb.shortcuts:
				kb.replace_action('opf_helper.show_opf', self.opf_show_content)
			else:
				kb.register_shortcut(
					'opf_helper.show_opf',
					_('OPF Helper: Show OPF content'),
					default_keys=(), description=_('Show OPF content for selected books'), action=self.opf_show_content,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find Books with Multiple OPF Files
			self.opf_find_multiple = QAction(_('Find books with multiple OPF files'), self.gui)
			self.opf_find_multiple.triggered.connect(self.check_for_multiple_opf_files)
			self.opf_find_multiple.calibre_shortcut_unique_name = 'opf_helper.find_multiple'
			self.gui.addAction(self.opf_find_multiple)
			if 'opf_helper.find_multiple' in kb.shortcuts:
				kb.replace_action('opf_helper.find_multiple', self.opf_find_multiple)
			else:
				kb.register_shortcut(
					'opf_helper.find_multiple',
					_('OPF Helper: Find books with multiple OPF files'),
					default_keys=(), description=_('Scan library to find books with multiple OPF files'), action=self.opf_find_multiple,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find Books with XML Parsing Issues
			self.opf_find_xml_issues = QAction(_('Find books with XML parsing issues'), self.gui)
			self.opf_find_xml_issues.triggered.connect(self.check_for_xml_parsing_issues)
			self.opf_find_xml_issues.calibre_shortcut_unique_name = 'opf_helper.find_xml_issues'
			self.gui.addAction(self.opf_find_xml_issues)
			if 'opf_helper.find_xml_issues' in kb.shortcuts:
				kb.replace_action('opf_helper.find_xml_issues', self.opf_find_xml_issues)
			else:
				kb.register_shortcut(
					'opf_helper.find_xml_issues',
					_('OPF Helper: Find books with XML parsing issues'),
					default_keys=(), description=_('Scan library to find books with XML parsing issues in OPF files'), action=self.opf_find_xml_issues,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Export Selected Books OPFs
			self.opf_export_selected = QAction(_('Export selected books OPFs'), self.gui)
			self.opf_export_selected.triggered.connect(self.export_selected_opfs)
			self.opf_export_selected.calibre_shortcut_unique_name = 'opf_helper.export_selected'
			self.gui.addAction(self.opf_export_selected)
			if 'opf_helper.export_selected' in kb.shortcuts:
				kb.replace_action('opf_helper.export_selected', self.opf_export_selected)
			else:
				kb.register_shortcut(
					'opf_helper.export_selected',
					_('OPF Helper: Export selected books OPFs'),
					default_keys=(), description=_('Export OPF files from selected books to a directory'), action=self.opf_export_selected,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find EPUB 3.0 Books
			self.opf_find_epub3 = QAction(_('Find EPUB 3.0 books'), self.gui)
			self.opf_find_epub3.triggered.connect(self.find_epub_3_books)
			self.opf_find_epub3.calibre_shortcut_unique_name = 'opf_helper.find_epub_3'
			self.gui.addAction(self.opf_find_epub3)
			if 'opf_helper.find_epub_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_epub_3', self.opf_find_epub3)
			else:
				kb.register_shortcut(
					'opf_helper.find_epub_3',
					_('OPF Helper: Find EPUB 3.0 books'),
					default_keys=(), description=_('Find all EPUB 3.0 books'), action=self.opf_find_epub3,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find EPUB 2.0 Books
			self.opf_find_epub2 = QAction(_('Find EPUB 2.0 books'), self.gui)
			self.opf_find_epub2.triggered.connect(self.find_epub_2_books)
			self.opf_find_epub2.calibre_shortcut_unique_name = 'opf_helper.find_epub_2'
			self.gui.addAction(self.opf_find_epub2)
			if 'opf_helper.find_epub_2' in kb.shortcuts:
				kb.replace_action('opf_helper.find_epub_2', self.opf_find_epub2)
			else:
				kb.register_shortcut(
					'opf_helper.find_epub_2',
					_('OPF Helper: Find EPUB 2.0 books'),
					default_keys=(), description=_('Find all EPUB 2.0 books'), action=self.opf_find_epub2,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find Non-3.0 EPUB Books (renamed)
			self.opf_find_not_epub3 = QAction(_('Find non-3.0 EPUB books'), self.gui)
			self.opf_find_not_epub3.triggered.connect(self.find_epub_not_3_books)
			self.opf_find_not_epub3.calibre_shortcut_unique_name = 'opf_helper.find_not_epub_3'
			self.gui.addAction(self.opf_find_not_epub3)
			if 'opf_helper.find_not_epub_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_not_epub_3', self.opf_find_not_epub3)
			else:
				kb.register_shortcut(
					'opf_helper.find_not_epub_3',
					_('OPF Helper: Find non-3.0 EPUB books'),
					default_keys=(), description=_('Find EPUB/KEPUB books that are not EPUB 3.0'), action=self.opf_find_not_epub3,
					group=_('OPF Helper'), persist_shortcut=True
				)
			# Edit OPF (unassigned shortcut)
			self.opf_edit_action = QAction(_('Edit OPF'), self.gui)
			self.opf_edit_action.triggered.connect(self._edit_opf_direct_trigger)
			self.opf_edit_action.calibre_shortcut_unique_name = 'opf_helper.edit_opf'
			self.gui.addAction(self.opf_edit_action)
			if 'opf_helper.edit_opf' in kb.shortcuts:
				kb.replace_action('opf_helper.edit_opf', self.opf_edit_action)
			else:
				kb.register_shortcut(
					'opf_helper.edit_opf',
					_('OPF Helper: Edit OPF'),
					default_keys=(), description=_('Edit the OPF file of the selected book (opens in editor)'), action=self.opf_edit_action,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# OPF Standards Comparison (unassigned shortcut)
			self.opf_compare_action = QAction(_('OPF standards comparison'), self.gui)
			self.opf_compare_action.triggered.connect(self.show_opf_comparison)
			self.opf_compare_action.calibre_shortcut_unique_name = 'opf_helper.opf_standards_comparison'
			self.gui.addAction(self.opf_compare_action)
			if 'opf_helper.opf_standards_comparison' in kb.shortcuts:
				kb.replace_action('opf_helper.opf_standards_comparison', self.opf_compare_action)
			else:
				kb.register_shortcut(
					'opf_helper.opf_standards_comparison',
					_('OPF Helper: OPF standards comparison'),
					default_keys=(), description=_('Compare current OPF with standards-corrected version'), action=self.opf_compare_action,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Bulk OPF Validator (unassigned shortcut)
			self.bulk_opf_validator_action = QAction(_('Bulk OPF validator'), self.gui)
			self.bulk_opf_validator_action.triggered.connect(self._validate_all_opfs_log_errors)
			self.bulk_opf_validator_action.calibre_shortcut_unique_name = 'opf_helper.bulk_opf_validator'
			self.gui.addAction(self.bulk_opf_validator_action)
			if 'opf_helper.bulk_opf_validator' in kb.shortcuts:
				kb.replace_action('opf_helper.bulk_opf_validator', self.bulk_opf_validator_action)
			else:
				kb.register_shortcut(
					'opf_helper.bulk_opf_validator',
					_('OPF Helper: Bulk OPF validator'),
					default_keys=(), description=_('Bulk validate OPF files for selected/library books'), action=self.bulk_opf_validator_action,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# MOBI OPF version scanners
			self.mobi_find_opf3 = QAction(_('Find MOBI OPF 3.0 books'), self.gui)
			self.mobi_find_opf3.triggered.connect(self.find_mobi_3_books)
			self.mobi_find_opf3.calibre_shortcut_unique_name = 'opf_helper.find_mobi_3'
			self.gui.addAction(self.mobi_find_opf3)
			if 'opf_helper.find_mobi_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_mobi_3', self.mobi_find_opf3)
			else:
				kb.register_shortcut(
					'opf_helper.find_mobi_3',
					_('OPF Helper: Find MOBI OPF 3.0 books'),
					default_keys=(), description=_('Find all MOBI books whose extracted OPF is 3.0'), action=self.mobi_find_opf3,
					group=_('OPF Helper'), persist_shortcut=True
				)

			self.mobi_find_opf2 = QAction(_('Find MOBI OPF 2.0 books'), self.gui)
			self.mobi_find_opf2.triggered.connect(self.find_mobi_2_books)
			self.mobi_find_opf2.calibre_shortcut_unique_name = 'opf_helper.find_mobi_2'
			self.gui.addAction(self.mobi_find_opf2)
			if 'opf_helper.find_mobi_2' in kb.shortcuts:
				kb.replace_action('opf_helper.find_mobi_2', self.mobi_find_opf2)
			else:
				kb.register_shortcut(
					'opf_helper.find_mobi_2',
					_('OPF Helper: Find MOBI OPF 2.0 books'),
					default_keys=(), description=_('Find all MOBI books whose extracted OPF is 2.0'), action=self.mobi_find_opf2,
					group=_('OPF Helper'), persist_shortcut=True
				)

			self.mobi_find_not_opf3 = QAction(_('Find MOBI non-3.0 OPF books'), self.gui)
			self.mobi_find_not_opf3.triggered.connect(self.find_mobi_not_3_books)
			self.mobi_find_not_opf3.calibre_shortcut_unique_name = 'opf_helper.find_mobi_not_3'
			self.gui.addAction(self.mobi_find_not_opf3)
			if 'opf_helper.find_mobi_not_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_mobi_not_3', self.mobi_find_not_opf3)
			else:
				kb.register_shortcut(
					'opf_helper.find_mobi_not_3',
					_('OPF Helper: Find MOBI non-3.0 OPF books'),
					default_keys=(), description=_('Find MOBI books whose extracted OPF is not 3.0'), action=self.mobi_find_not_opf3,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# All-formats OPF scanners (EPUB/KEPUB/AZW3/MOBI/HTMLZ)
			self.all_find_opf3 = QAction(_('Find OPF 3.0 (all formats)'), self.gui)
			self.all_find_opf3.triggered.connect(lambda: self.find_all_opf_version('3.0'))
			self.all_find_opf3.calibre_shortcut_unique_name = 'opf_helper.find_all_opf_3'
			self.gui.addAction(self.all_find_opf3)
			if 'opf_helper.find_all_opf_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_all_opf_3', self.all_find_opf3)
			else:
				kb.register_shortcut(
					'opf_helper.find_all_opf_3',
					_('OPF Helper: Find OPF 3.0 (all formats)'),
					default_keys=(), description=_('Find all books with OPF 3.0 (EPUB/KEPUB/AZW3/MOBI/HTMLZ)'), action=self.all_find_opf3,
					group=_('OPF Helper'), persist_shortcut=True
				)

			self.all_find_opf2 = QAction(_('Find OPF 2.0 (all formats)'), self.gui)
			self.all_find_opf2.triggered.connect(lambda: self.find_all_opf_version('2.0'))
			self.all_find_opf2.calibre_shortcut_unique_name = 'opf_helper.find_all_opf_2'
			self.gui.addAction(self.all_find_opf2)
			if 'opf_helper.find_all_opf_2' in kb.shortcuts:
				kb.replace_action('opf_helper.find_all_opf_2', self.all_find_opf2)
			else:
				kb.register_shortcut(
					'opf_helper.find_all_opf_2',
					_('OPF Helper: Find OPF 2.0 (all formats)'),
					default_keys=(), description=_('Find all books with OPF 2.0 (EPUB/KEPUB/AZW3/MOBI/HTMLZ)'), action=self.all_find_opf2,
					group=_('OPF Helper'), persist_shortcut=True
				)

			self.all_find_not_3 = QAction(_('Find non‑3.0 OPF (all formats)'), self.gui)
			self.all_find_not_3.triggered.connect(lambda: self.find_all_opf_version('not_3.0'))
			self.all_find_not_3.calibre_shortcut_unique_name = 'opf_helper.find_all_not_3'
			self.gui.addAction(self.all_find_not_3)
			if 'opf_helper.find_all_not_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_all_not_3', self.all_find_not_3)
			else:
				kb.register_shortcut(
					'opf_helper.find_all_not_3',
					_('OPF Helper: Find non-3.0 OPF (all formats)'),
					default_keys=(), description=_('Find books without OPF 3.0 (EPUB/KEPUB/AZW3/MOBI/HTMLZ)'), action=self.all_find_not_3,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Apply any existing user-defined shortcuts to these actions
			kb.finalize()
		except Exception:
			# Non-fatal if keyboard manager not ready
			pass

	def _setup_direct_actions(self):
		"""Add direct top-level actions for toolbar/menus preferences.

		Calibre's 'Preferences -> Toolbars & menus' only lists actions from gui.iactions.
		We create QActions, add them to the gui, and then inject fake iaction objects
		into gui.iactions via a deferred QTimer.singleShot to avoid RuntimeError during
		OrderedDict iteration.
		"""
		# Direct toolbar/menu injection is intentionally disabled.
		# Adding QActions directly into Calibre's global gui.iactions at genesis()
		# proved unreliable and caused mutation-of-OrderedDict issues. All
		# functionality that was previously offered as "direct" toolbar/menu
		# actions is now available from the plugin's own menu (see rebuild_menus),
		# where it can be shown alongside the EPUB scanner and other operations.
		return

	def _edit_opf_direct_trigger(self):
		"""Handler for Edit OPF (direct) action."""
		try:
			selected = self.gui.library_view.get_selected_ids()
			if not selected:
				return error_dialog(self.gui, _('No book selected'), _('Please select a book to edit its OPF.'), show=True)
			book_id = selected[0]
			db = self.gui.current_db.new_api
			edit_opf_for_book(self.gui, db, book_id)
		except Exception as e:
			debug_print('OPFHelper: direct edit failed:', str(e))

	def show_help(self):
		"""Open OPF help page in browser"""
		QDesktopServices.openUrl(QUrl('https://wiki.mobileread.com/wiki/OPF'))

	def show_configuration(self):
		self.interface_action_base_plugin.do_user_config(self.gui)

	def check_for_multiple_opf_files(self):
		"""Check library for books with multiple OPF files, mark them, and filter to show marked books"""
		db = self.gui.current_db.new_api

		# Show progress dialog
		progress = QProgressDialog("Checking for multiple OPF files...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle("Multiple OPF Checker")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)

		# Only EPUB/KEPUB can contain OPF files inside the zip container.
		book_ids = self._library_search_ids('formats:epub or formats:kepub')
		if not book_ids:
			progress.close()
			self.gui.status_bar.showMessage("No EPUB/KEPUB books found in library", 4000)
			return
		progress.setMaximum(len(book_ids))

		matching_books = []
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				break

			progress.setValue(i)
			progress.setLabelText(f"Checking book {i+1} of {len(book_ids)}...")

			# Check for EPUB or KEPUB format
			formats = db.formats(book_id)
			fmt = None
			if formats:
				if 'EPUB' in formats:
					fmt = 'EPUB'
				elif 'KEPUB' in formats:
					fmt = 'KEPUB'
			if fmt:
				epub_path = db.format_abspath(book_id, fmt)
				if epub_path:
					try:
						with zipfile.ZipFile(epub_path, 'r') as zf:
							opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
							if len(opf_files) > 1:
								matching_books.append(book_id)
					except Exception as e:
						debug_print(f"Error checking {epub_path}: {str(e)}")

		progress.close()

		# Mark matching books and filter view
		if matching_books:
			if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
				try:
					act = self.gui.iactions['Mark Books']
					# Clear previous marks first
					try:
						if hasattr(act, 'clear'):
							act.clear()
						elif hasattr(self.gui.current_db, 'data'):
							self.gui.current_db.data.set_marked_ids({})
					except Exception:
						pass  # Continue even if clearing fails
					if hasattr(act, 'add_ids'):
						act.add_ids(matching_books)
					else:
						# Fallback for older Calibre (e.g. 5.44): manually set marked ids
						try:
							db_view = self.gui.current_db
							mids = db_view.data.marked_ids.copy()
							for bid in matching_books:
								mids[bid] = 'true'
							db_view.data.set_marked_ids(mids)
						except Exception as e:
							debug_print(f'OPFHelper: failed to mark books via fallback: {e}')
				except Exception:
					pass
				self.gui.search.set_search_string('marked:true')
				count = len(matching_books)
				info_dialog(self.gui, 'Books Found and Marked',
						   f'{count} book{"s" if count != 1 else ""} with multiple OPF files found and marked. '
						   'Library filtered to show marked books only.',
						   show=True)
			else:
				error_dialog(self.gui, 'Error', 'Could not access the Mark Books action.', show=True)
		else:
			self.gui.status_bar.showMessage("No books with multiple OPF files were found.", 4000)

	def check_for_xml_parsing_issues(self):
		"""Check library for books with XML parsing issues in OPF files"""
		from io import BytesIO
		import xml.etree.ElementTree as ET

		db = self.gui.current_db.new_api

		# Show progress dialog
		progress = QProgressDialog("Checking for XML parsing issues...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle("XML Parsing Issues Checker")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)

		# Only EPUB/KEPUB can contain OPF files inside the zip container.
		book_ids = self._library_search_ids('formats:epub or formats:kepub')
		if not book_ids:
			progress.close()
			info_dialog(self.gui, "XML Parsing Issues Checker", "No EPUB/KEPUB books found in library.", show=True)
			return
		progress.setMaximum(len(book_ids))

		results = []
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				break

			progress.setValue(i)
			progress.setLabelText(f"Checking book {i+1} of {len(book_ids)}...")

			# Check for EPUB or KEPUB format
			formats = db.formats(book_id)
			fmt = None
			if formats:
				if 'EPUB' in formats:
					fmt = 'EPUB'
				elif 'KEPUB' in formats:
					fmt = 'KEPUB'
			if fmt:
				# Get book path
				epub_path = db.format_abspath(book_id, fmt)
				if epub_path:
					try:
						with zipfile.ZipFile(epub_path, 'r') as zf:
							opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
							if opf_files:
								# Check the first OPF file for parsing issues
								opf_path = opf_files[0]
								try:
									opf_data = zf.read(opf_path).decode('utf-8', errors='replace')
									xml_bytes = opf_data.encode('utf-8')
									ET.parse(BytesIO(xml_bytes))
									# If we get here, parsing was successful
								except ET.ParseError as e:
									title = db.field_for('title', book_id)
									# Extract error details
									error_details = str(e)
									results.append((book_id, title, opf_path, error_details))
								except Exception as e:
									debug_print(f"Error checking OPF parsing for {epub_path}: {str(e)}")
					except Exception as e:
						debug_print(f"Error checking {epub_path}: {str(e)}")

		progress.close()

		# Show results
		if results:
			d = XMLParsingIssuesDialog(self.gui, results)
			d.exec_()
		else:
			info_dialog(self.gui, "XML Parsing Issues Checker",
						"No books with XML parsing issues were found.", show=True)

	def change_epub_scope(self, new_scope):
		"""Change the EPUB version finder scope setting"""
		prefs['epub_version_scope'] = new_scope
		# Update menu checkmarks across all scanners
		self._sync_scope_action_checks()

	def change_bulk_validate_opfs_scope(self, new_scope):
		"""Change the Bulk validate OPFs scope setting (independent from other scanners)."""
		prefs['bulk_validate_opfs_scope'] = new_scope
		self._sync_scope_action_checks()

	def find_epub_3_books(self):
		"""Find and select all books with EPUB 3.0 format"""
		self._find_books_by_epub_version(version_filter='3.0', description="EPUB 3.0")

	def find_epub_2_books(self):
		"""Find and select all books with EPUB 2.0 format"""
		self._find_books_by_epub_version(version_filter='2.0', description="EPUB 2.0")

	def find_epub_not_3_books(self):
		"""Find and select all EPUB/KEPUB books that are not EPUB 3.0"""
		self._find_books_by_epub_version(version_filter='not_3.0', description="non-EPUB 3.0")

	def _find_books_by_epub_version(self, version_filter, description):
		"""Helper method to find books by EPUB version, mark them, and filter to show marked books"""
		# Get scope setting
		scope = prefs.get('epub_version_scope', 'Library')

		db = self.gui.current_db.new_api

		# Pre-filter candidates by format (avoids scanning the entire library).
		book_ids = self._get_scoped_book_ids(scope, required_any_formats={'EPUB', 'KEPUB'})
		if not book_ids:
			if scope == 'Selection':
				self.gui.status_bar.showMessage("No selected books with EPUB/KEPUB format", 4000)
			else:
				self.gui.status_bar.showMessage("No EPUB/KEPUB books found in library", 4000)
			return

		# Show progress dialog
		progress = QProgressDialog(f"Finding {description} books...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle(f"EPUB Version Scanner - {description}")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)
		progress.setMaximum(len(book_ids))

		matching_books = []
		canceled = False
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				canceled = True
				break

			progress.setValue(i)
			progress.setLabelText(f"Scanning book {i+1} of {len(book_ids)}...")

			# Check for EPUB or KEPUB format
			formats = db.formats(book_id)
			fmt = None
			if formats:
				if 'EPUB' in formats:
					fmt = 'EPUB'
				elif 'KEPUB' in formats:
					fmt = 'KEPUB'
			if fmt:
				epub_path = db.format_abspath(book_id, fmt)
				if epub_path:
					try:
						with zipfile.ZipFile(epub_path, 'r') as zf:
							opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
							if opf_files:
								# Read the first OPF file to check version
								opf_path = opf_files[0]
								opf_data = zf.read(opf_path).decode('utf-8', errors='replace')

								# Check version based on filter
								is_match = False
								if version_filter == '3.0':
									is_match = 'version="3.0"' in opf_data
								elif version_filter == '2.0':
									is_match = 'version="2.0"' in opf_data
								elif version_filter == 'not_3.0':
									is_match = 'version="3.0"' not in opf_data and ('version="2.0"' in opf_data or fmt == 'KEPUB')

								if is_match:
									matching_books.append(book_id)

					except Exception as e:
						debug_print(f"Error checking version for {epub_path}: {str(e)}")

		progress.close()

		if canceled and not matching_books:
			info_dialog(self.gui, f"EPUB Version Scanner - {description}", "Scan canceled. No matches found in the scanned subset.", show=True)
			return

		# Mark matching books and show them
		if matching_books:
			# Mark the books using Calibre's mark functionality
			if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
				act = self.gui.iactions['Mark Books']
				# Clear previous marks first
				try:
					if hasattr(act, 'clear'):
						act.clear()
					elif hasattr(self.gui.current_db, 'data'):
						self.gui.current_db.data.set_marked_ids({})
				except Exception:
					pass  # Continue even if clearing fails
				act.add_ids(matching_books)
				# Apply "marked:true" search to show only marked books
				self.gui.search.set_search_string('marked:true')
				count = len(matching_books)
				msg = f'{count} {description} book{"s" if count != 1 else ""} found and marked. Library filtered to show marked books only.'
				if canceled:
					msg = 'Scan canceled early (partial results).\n\n' + msg
				info_dialog(self.gui, 'Books Found and Marked', msg, show=True)
			else:
				error_dialog(self.gui, 'Error', 'Could not access the Mark Books action.', show=True)
		else:
			if canceled:
				info_dialog(self.gui, f"EPUB Version Scanner - {description}", "Scan canceled. No matches found in the scanned subset.", show=True)
			else:
				self.gui.status_bar.showMessage(f"No {description} books found in {scope.lower()}", 5000)

	def find_azw3_3_books(self):
		"""AZW3 OPF version scanning is intentionally disabled."""
		info_dialog(
			self.gui,
			'AZW3 OPF Scanner',
			'AZW3 OPF version scanning is disabled. In practice, the extracted OPF from AZW3/KF8 almost always reports OPF 2.0, and scanning requires expensive exploding that can extract many resources (images/fonts).\n\nUse the MOBI or EPUB scanners instead.',
			show=True,
		)
		return

	def find_azw3_2_books(self):
		return self.find_azw3_3_books()

	def find_azw3_not_3_books(self):
		return self.find_azw3_3_books()

	def _find_books_by_azw3_version(self, version_filter, description):
		"""Scan AZW3 books, extract OPF via Calibre exploder, filter by OPF version, and mark results"""
		# Reuse EPUB scanner scope setting for simplicity
		scope = prefs.get('epub_version_scope', 'Library')
		db = self.gui.current_db.new_api

		# Get scoped book ids
		if scope == 'Selection':
			rows = self.gui.library_view.selectionModel().selectedRows()
			if not rows:
				self.gui.status_bar.showMessage("No books selected", 3000)
				return
			book_ids = []
			model = self.gui.library_view.model()
			for row in rows:
				bid = model.id(row.row())
				if bid is not None:
					book_ids.append(bid)
		else:
			book_ids = db.all_book_ids()

		# Progress dialog
		progress = QProgressDialog(f"Finding {description}...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle("AZW3 OPF Version Scanner")
		progress.setMinimumDuration(0)
		progress.setWindowModality(Qt.WindowModal)
		progress.setMaximum(len(book_ids))

		matching = []
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				break
			progress.setValue(i)
			progress.setLabelText(f"Scanning book {i+1} of {len(book_ids)}...")

			try:
				fmts = self._normalize_formats(db.formats(book_id))
			except Exception:
				fmts = set()
			if 'AZW3' not in fmts:
				debug_print(f"OPFHelper: AZW3 scanner - book {book_id} has no AZW3 format, skipping")
				continue

			azw3_path = None
			try:
				azw3_path = db.format_abspath(book_id, 'AZW3')
			except Exception:
				azw3_path = None
			if not azw3_path:
				debug_print(f"OPFHelper: AZW3 scanner - book {book_id} AZW3 path not found, skipping")
				continue

			debug_print(f"OPFHelper: AZW3 scanner - exploding AZW3 for book {book_id}: {azw3_path}")

			# Skip DRM books
			try:
				has_drm = False
				try:
					has_drm = db.has_drm(book_id)
				except Exception:
					try:
						has_drm = self.gui.current_db.has_drm(book_id)
					except Exception:
						has_drm = False
				if has_drm:
					continue
			except Exception:
				pass

			opf_xml = None
			tdir = None
			try:
				from calibre.ebooks.tweak import get_tools
				try:
					tdir = PersistentTemporaryDirectory('_opf_helper_scan_azw3')
				except Exception:
					import tempfile
					tdir = tempfile.mkdtemp(prefix='_opf_helper_scan_azw3_')
				explode_dir = str(tdir)
				exploder = get_tools('AZW3')[0]
				opf_path = exploder(azw3_path, explode_dir, question=self._ask_question)
				if opf_path and os.path.exists(opf_path):
					with open(opf_path, 'rb') as f:
						opf_xml = f.read().decode('utf-8', 'replace')
					debug_print(f"OPFHelper: AZW3 scanner - extracted OPF at {opf_path} for book {book_id}")
				else:
					# fallback: scan exploded dir
					for root, _dirs, files in os.walk(explode_dir):
						for fn in files:
							if fn.lower().endswith('.opf'):
								p = os.path.join(root, fn)
								with open(p, 'rb') as f:
									opf_xml = f.read().decode('utf-8', 'replace')
								debug_print(f"OPFHelper: AZW3 scanner - fallback OPF at {p} for book {book_id}")
								break
						if opf_xml:
							break
			except Exception:
				opf_xml = None
			finally:
				try:
					if tdir and os.path.isdir(str(tdir)):
						shutil.rmtree(str(tdir))
				except Exception:
							pass

			if not opf_xml:
				debug_print(f"OPFHelper: AZW3 scanner - no OPF extracted for book {book_id}")
				continue


			# Robust version check (supports single quotes/whitespace)
			is_match = self._opf_version_matches_filter(opf_xml, version_filter)

			debug_print(f"OPFHelper: AZW3 scanner - book {book_id} version match={is_match} for filter {version_filter}")

			if is_match:
				matching.append(book_id)

		progress.close()

		if matching:
			try:
				if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
					act = self.gui.iactions['Mark Books']
					# Clear previous marks first
					try:
						if hasattr(act, 'clear'):
							act.clear()
						elif hasattr(self.gui.current_db, 'data'):
							self.gui.current_db.data.set_marked_ids({})
					except Exception:
						pass  # Continue even if clearing fails
					act.add_ids(matching)
					self.gui.search.set_search_string('marked:true')
					count = len(matching)
					info_dialog(self.gui, 'Books Found and Marked',
							   f'{count} {description} book{"s" if count != 1 else ""} found and marked. Library filtered to show marked books only.',
							   show=True)
				else:
					raise RuntimeError('Mark Books action not available')
			except Exception:
				error_dialog(self.gui, 'Error', 'Could not access the Mark Books action.', show=True)
		else:
			self.gui.status_bar.showMessage(f"No {description} books found in {scope.lower()}", 5000)

	def find_mobi_3_books(self):
		"""Find and select all MOBI books whose extracted OPF is 3.0"""
		self._find_books_by_mobi_version(version_filter='3.0', description="MOBI OPF 3.0")

	def find_mobi_2_books(self):
		"""Find and select all MOBI books whose extracted OPF is 2.0"""
		self._find_books_by_mobi_version(version_filter='2.0', description="MOBI OPF 2.0")

	def find_mobi_not_3_books(self):
		"""Find and select all MOBI books whose extracted OPF is not 3.0"""
		self._find_books_by_mobi_version(version_filter='not_3.0', description="MOBI non-3.0 OPF")

	def _find_books_by_mobi_version(self, version_filter, description):
		"""Scan MOBI books, extract OPF via KindleUnpack, filter by OPF version, and mark results"""
		scope = prefs.get('epub_version_scope', 'Library')
		db = self.gui.current_db.new_api

		book_ids = self._get_scoped_book_ids(scope, required_any_formats={'MOBI'})
		if not book_ids:
			if scope == 'Selection':
				self.gui.status_bar.showMessage("No selected books with MOBI format", 4000)
			else:
				self.gui.status_bar.showMessage("No MOBI books found in library", 4000)
			return

		progress = QProgressDialog(f"Finding {description}...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle("MOBI OPF Version Scanner")
		progress.setMinimumDuration(0)
		progress.setWindowModality(Qt.WindowModal)
		progress.setMaximum(len(book_ids))

		matching = []
		canceled = False
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				canceled = True
				break
			progress.setValue(i)
			progress.setLabelText(f"Scanning book {i+1} of {len(book_ids)}...")

			# Skip non-MOBI
			try:
				fmts = self._normalize_formats(db.formats(book_id))
			except Exception:
				fmts = set()
			if 'MOBI' not in fmts:
				continue

			# Skip DRM
			try:
				has_drm = False
				try:
					has_drm = db.has_drm(book_id)
				except Exception:
					try:
						has_drm = self.gui.current_db.has_drm(book_id)
					except Exception:
						has_drm = False
				if has_drm:
					continue
			except Exception:
				pass

			mobi_path = None
			try:
				mobi_path = db.format_abspath(book_id, 'MOBI')
			except Exception:
				mobi_path = None
			if not mobi_path:
				continue

			opf_xml = None
			try:
				from calibre_plugins.opf_helper.mobi_opf_extract import extract_mobi_opf
				opf_xml, err = extract_mobi_opf(mobi_path)
				if not opf_xml:
					debug_print(f"OPFHelper: MOBI scanner - failed to extract OPF for book {book_id}: {err}")
					continue
			except Exception as e:
				debug_print(f"OPFHelper: MOBI scanner - exception extracting OPF for book {book_id}: {e}")
				continue


			is_match = self._opf_version_matches_filter(opf_xml, version_filter)

			if is_match:
				matching.append(book_id)

		progress.close()

		if canceled and not matching:
			info_dialog(self.gui, "MOBI OPF Version Scanner", "Scan canceled. No matches found in the scanned subset.", show=True)
			return

		if matching:
			try:
				if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
					act = self.gui.iactions['Mark Books']
					# Clear previous marks first
					try:
						if hasattr(act, 'clear'):
							act.clear()
						elif hasattr(self.gui.current_db, 'data'):
							self.gui.current_db.data.set_marked_ids({})
					except Exception:
						pass  # Continue even if clearing fails
					act.add_ids(matching)
					self.gui.search.set_search_string('marked:true')
					count = len(matching)
					msg = f'{count} {description} book{"s" if count != 1 else ""} found and marked. Library filtered to show marked books only.'
					if canceled:
						msg = 'Scan canceled early (partial results).\n\n' + msg
					info_dialog(self.gui, 'Books Found and Marked', msg, show=True)
				else:
					raise RuntimeError('Mark Books action not available')
			except Exception:
				error_dialog(self.gui, 'Error', 'Could not access the Mark Books action.', show=True)
		else:
			if canceled:
				info_dialog(self.gui, "MOBI OPF Version Scanner", "Scan canceled. No matches found in the scanned subset.", show=True)
			else:
				self.gui.status_bar.showMessage(f"No {description} books found in {scope.lower()}", 5000)

	def find_all_opf_version(self, version_filter):
		"""Broad OPF version scanner across EPUB/KEPUB/MOBI/HTMLZ"""
		desc_map = {
			'3.0': 'OPF 3.0 (all formats)',
			'2.0': 'OPF 2.0 (all formats)',
			'not_3.0': 'non‑3.0 OPF (all formats)'
		}
		self._find_books_by_any_opf_version(version_filter=version_filter, description=desc_map.get(version_filter, 'OPF version scan'))

	def _find_books_by_any_opf_version(self, version_filter, description):
		"""Scan all supported formats and mark books whose OPF version matches the filter"""
		scope = prefs.get('epub_version_scope', 'Library')
		db = self.gui.current_db.new_api

		book_ids = self._get_scoped_book_ids(scope, required_any_formats={'EPUB', 'KEPUB', 'MOBI', 'HTMLZ'})
		if not book_ids:
			if scope == 'Selection':
				self.gui.status_bar.showMessage("No selected books with supported formats (EPUB/KEPUB/MOBI/HTMLZ)", 5000)
			else:
				self.gui.status_bar.showMessage("No books with supported formats (EPUB/KEPUB/MOBI/HTMLZ) found in library", 5000)
			return

		progress = QProgressDialog(f"Finding {description}...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle("All Formats OPF Version Scanner")
		progress.setMinimumDuration(0)
		progress.setWindowModality(Qt.WindowModal)
		progress.setMaximum(len(book_ids))

		matching = []
		canceled = False
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				canceled = True
				break
			progress.setValue(i)
			progress.setLabelText(f"Scanning book {i+1} of {len(book_ids)}...")

			fmts = self._normalize_formats(db.formats(book_id))
			opf_xml = None
			# Priority: EPUB/KEPUB, then MOBI, then HTMLZ
			try:
				fmt = None
				if 'EPUB' in fmts or 'KEPUB' in fmts:
					fmt = 'EPUB' if 'EPUB' in fmts else 'KEPUB'
					p = db.format_abspath(book_id, fmt)
					with zipfile.ZipFile(p, 'r') as zf:
						opf_files = [f for f in zf.namelist() if (f or '').lower().endswith('.opf')]
						if opf_files:
							opf_xml = zf.read(opf_files[0]).decode('utf-8', 'replace')
				elif 'MOBI' in fmts:
					p = db.format_abspath(book_id, 'MOBI')
					from calibre_plugins.opf_helper.mobi_opf_extract import extract_mobi_opf
					opf_xml, _err = extract_mobi_opf(p)
				elif 'HTMLZ' in fmts:
					p = db.format_abspath(book_id, 'HTMLZ')
					with zipfile.ZipFile(p, 'r') as zf:
						opf_files = [f for f in zf.namelist() if (f or '').lower().endswith('.opf')]
						if opf_files:
							opf_xml = zf.read(opf_files[0]).decode('utf-8', 'replace')
			except Exception as e:
				debug_print(f"OPFHelper: All-formats scanner - error extracting OPF for book {book_id}: {e}")

			if not opf_xml:
				continue


			is_match = self._opf_version_matches_filter(opf_xml, version_filter)

			if is_match:
				matching.append(book_id)

		progress.close()

		if canceled and not matching:
			info_dialog(self.gui, "All Formats OPF Version Scanner", "Scan canceled. No matches found in the scanned subset.", show=True)
			return

		if matching:
			try:
				if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
					act = self.gui.iactions['Mark Books']
					# Clear previous marks first
					try:
						if hasattr(act, 'clear'):
							act.clear()
						elif hasattr(self.gui.current_db, 'data'):
							self.gui.current_db.data.set_marked_ids({})
					except Exception:
						pass  # Continue even if clearing fails
					act.add_ids(matching)
					self.gui.search.set_search_string('marked:true')
					count = len(matching)
					msg = f'{count} {description} book{"s" if count != 1 else ""} found and marked. Library filtered to show marked books only.'
					if canceled:
						msg = 'Scan canceled early (partial results).\n\n' + msg
					info_dialog(self.gui, 'Books Found and Marked', msg, show=True)
				else:
					raise RuntimeError('Mark Books action not available')
			except Exception:
				error_dialog(self.gui, 'Error', 'Could not access the Mark Books action.', show=True)
		else:
			if canceled:
				info_dialog(self.gui, "All Formats OPF Version Scanner", "Scan canceled. No matches found in the scanned subset.", show=True)
			else:
				self.gui.status_bar.showMessage(f"No {description} books found in {scope.lower()}", 5000)

	def export_selected_opfs(self):
		"""Export OPF files from selected books to a directory.

		Supports EPUB, KEPUB, HTMLZ (as ZIP), MOBI (via KindleUnpack), and AZW3 (via Calibre exploder).
		"""
		# Get selected books
		rows = self.gui.library_view.selectionModel().selectedRows()
		if not rows or len(rows) == 0:
			self.gui.status_bar.showMessage("No books selected", 3000)
			return

		db = self.gui.current_db.new_api

		# Get selected book IDs
		book_ids = []
		model = self.gui.library_view.model()
		for row in rows:
			book_id = model.id(row.row())
			if book_id is not None:
				book_ids.append(book_id)

		if not book_ids:
			error_dialog(self.gui, 'Error', 'Could not get book IDs', show=True)
			return

		# Filter to books with supported formats
		# Priority order: EPUB > KEPUB > HTMLZ > AZW3 > MOBI
		SUPPORTED_FMTS = ('EPUB', 'KEPUB', 'HTMLZ', 'AZW3', 'MOBI')
		valid_books = []
		for book_id in book_ids:
			formats = db.formats(book_id) or []
			chosen_fmt = None
			for fmt in SUPPORTED_FMTS:
				if fmt in formats:
					chosen_fmt = fmt
					break
			if chosen_fmt:
				title = db.field_for('title', book_id)
				authors = db.field_for('authors', book_id)
				author_text = authors[0] if authors else "Unknown"
				valid_books.append((book_id, title, author_text, chosen_fmt))

		if not valid_books:
			self.gui.status_bar.showMessage("None of the selected books have a supported format (EPUB/KEPUB/HTMLZ/AZW3/MOBI)", 5000)
			return

		# Ask user for export directory
		from PyQt5.QtWidgets import QFileDialog
		export_dir = QFileDialog.getExistingDirectory(
			self.gui,
			"Select Export Directory",
			""
		)

		if not export_dir:
			return  # User cancelled

		# Show progress dialog
		from PyQt5.QtWidgets import QProgressDialog
		progress = QProgressDialog("Exporting OPF files...", "Cancel", 0, len(valid_books), self.gui)
		progress.setWindowTitle("Exporting OPF Files")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)

		exported_count = 0
		errors = []

		for i, (book_id, title, author, fmt) in enumerate(valid_books):
			if progress.wasCanceled():
				break

			progress.setValue(i)
			progress.setLabelText(f"Exporting: {title} ({fmt})")

			opf_content = None
			tdir = None

			try:
				file_path = db.format_abspath(book_id, fmt)
				if not file_path or not os.path.isfile(file_path):
					errors.append(f"{title}: File not found")
					continue

				# Extract OPF based on format
				if fmt in ('EPUB', 'KEPUB', 'HTMLZ'):
					# ZIP-based formats
					with zipfile.ZipFile(file_path, 'r') as zf:
						opf_files = [f for f in zf.namelist() if f.lower().endswith('.opf')]
						if opf_files:
							opf_content = zf.read(opf_files[0]).decode('utf-8', errors='replace')

				elif fmt == 'AZW3':
					# Use Calibre exploder
					try:
						has_drm = False
						try:
							has_drm = db.has_drm(book_id)
						except Exception:
							pass
						if has_drm:
							errors.append(f"{title}: AZW3 has DRM")
							continue

						from calibre.ebooks.tweak import get_tools
						try:
							tdir = PersistentTemporaryDirectory('_opf_export_')
						except Exception:
							import tempfile
							tdir = tempfile.mkdtemp(prefix='_opf_export_')

						exploder = get_tools('AZW3')[0]
						opf_path = exploder(file_path, str(tdir), question=self._ask_question)

						if opf_path and os.path.exists(opf_path):
							with open(opf_path, 'rb') as f:
								opf_content = f.read().decode('utf-8', 'replace')
						else:
							# Scan exploded dir for any .opf
							opfs = _opfhelper_find_opf_in_exploded(str(tdir))
							if opfs:
								with open(opfs[0], 'rb') as f:
									opf_content = f.read().decode('utf-8', 'replace')
					except Exception as e:
						errors.append(f"{title}: AZW3 extraction failed - {e}")
						continue
					finally:
						try:
							if tdir and os.path.isdir(str(tdir)):
								shutil.rmtree(str(tdir))
						except Exception:
							pass

				elif fmt == 'MOBI':
					# Use KindleUnpack
					try:
						has_drm = False
						try:
							has_drm = db.has_drm(book_id)
						except Exception:
							pass
						if has_drm:
							errors.append(f"{title}: MOBI has DRM")
							continue

						from calibre_plugins.opf_helper.mobi_opf_extract import extract_mobi_opf
						opf_xml, err = extract_mobi_opf(file_path)
						if opf_xml:
							opf_content = opf_xml
						else:
							errors.append(f"{title}: {err or 'MOBI extraction failed'}")
							continue
					except Exception as e:
						errors.append(f"{title}: MOBI extraction failed - {e}")
						continue

				if opf_content:
					# Create safe filename
					safe_title = "".join(x for x in title if x.isalnum() or x in " ._-").strip()
					safe_author = "".join(x for x in author if x.isalnum() or x in " ._-").strip()
					filename = f"{safe_title} - {safe_author}.opf"
					filepath = os.path.join(export_dir, filename)

					# Handle duplicate filenames
					counter = 1
					base_filepath = filepath
					while os.path.exists(filepath):
						name, ext = os.path.splitext(base_filepath)
						filepath = f"{name}_{counter}{ext}"
						counter += 1

					# Write OPF file
					with open(filepath, 'w', encoding='utf-8') as f:
						f.write(opf_content)

					exported_count += 1
				else:
					errors.append(f"{title}: No OPF found in {fmt}")

			except Exception as e:
				errors.append(f"{title}: {str(e)}")

		progress.close()

		# Show results
		if exported_count > 0:
			message = f"Successfully exported {exported_count} OPF file{'s' if exported_count != 1 else ''} to {export_dir}"
			if errors:
				message += f"\n\nErrors occurred for {len(errors)} book{'s' if len(errors) != 1 else ''}."
			info_dialog(self.gui, 'Export Complete', message, show=True)
		else:
			error_dialog(self.gui, 'Export Failed', 'No OPF files were exported.', show=True)

		if errors:
			# Show errors in a separate dialog if any
			error_text = "The following books had errors during export:\n\n" + "\n".join(errors)
			error_dialog(self.gui, 'Export Errors', error_text, show=True)

	def about_to_show_menu(self):
		# Keep Show OPF icon theme-correct in menus
		try:
			from calibre.gui2 import is_dark_theme
			icon_path = 'images/icon-for-dark-theme.png' if is_dark_theme() else 'images/icon-for-light-theme.png'
			ac = getattr(self, 'opf_show_content', None)
			if ac is not None:
				ac.setIcon(get_icon(icon_path))
		except Exception:
			pass

		# Update menu items state based on current context
		rows = self.gui.library_view.selectionModel().selectedRows()
		has_selection = bool(rows and len(rows) > 0)

		# Enable/disable menu items based on selection
		try:
			ac = getattr(self, '_menu_show_opf_action', None)
			if ac is not None:
				ac.setEnabled(has_selection)
		except Exception:
			pass
		try:
			ac = getattr(self, '_menu_export_opfs_action', None)
			if ac is not None:
				ac.setEnabled(has_selection)
		except Exception:
			pass

	def show_opf_comparison(self):
		"""Show OPF comparison dialog for selected books"""
		debug_print('OPFHelper: show_opf_comparison() called')
		gui = self.gui
		rows = gui.library_view.selectionModel().selectedRows()
		if not rows or len(rows) == 0:
			gui.status_bar.showMessage("No books selected", 3000)
			return

		try:
			# Get the current database API handle
			db = gui.current_db.new_api

			# Get selected book IDs using the model
			book_ids = []
			model = gui.library_view.model()
			for row in rows:
				book_id = model.id(row.row())
				if book_id is not None:
					book_ids.append(book_id)

			if not book_ids:
				error_dialog(gui, 'Error', 'Could not get book IDs', show=True)
				return

			# Use the first selected book
			book_id = book_ids[0]

			# Filter to only books with EPUB or KEPUB format
			if not (db.has_format(book_id, 'EPUB') or db.has_format(book_id, 'KEPUB')):
				gui.status_bar.showMessage("Selected book does not have EPUB or KEPUB format", 5000)
				return

			# Get book title
			title = db.field_for('title', book_id)

			# Get OPF content
			fmt = 'KEPUB' if db.has_format(book_id, 'KEPUB') else 'EPUB'
			epub_path = db.format_abspath(book_id, fmt)

			if not epub_path or not os.path.isfile(epub_path):
				gui.status_bar.showMessage("EPUB or KEPUB format not available for this book", 3000)
				return

			try:
				with zipfile.ZipFile(epub_path, 'r') as zf:
					opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
					if opf_files:
						opf_path = opf_files[0]  # Use first OPF file
						opf_content = zf.read(opf_path).decode('utf-8', errors='replace')

						# Create comparison dialog as a top-level window (no parent)
						# so it behaves independently and shows on the taskbar when minimized.
						d = OPFComparisonDialog(None, opf_content, title)
						try:
							d.setModal(False)
						except Exception:
							pass
						try:
							d.setAttribute(Qt.WA_DeleteOnClose, True)
						except Exception:
							pass

						# Keep a reference so the dialog isn't garbage-collected
						try:
							lst = getattr(self, '_opfhelper_modeless_dialogs', None)
							if lst is None:
								lst = []
								setattr(self, '_opfhelper_modeless_dialogs', lst)
							lst.append(d)
							def _drop_ref(_obj=None, dlg=d, lst_ref=lst):
								try:
									lst_ref.remove(dlg)
								except Exception:
									pass
							d.destroyed.connect(_drop_ref)
						except Exception:
							pass

						d.show()
						try:
							d.raise_()
							d.activateWindow()
						except Exception:
							pass
					else:
						gui.status_bar.showMessage("No OPF file found in the selected book", 3000)
			except Exception as e:
				error_dialog(gui, 'Error', f'Failed to read OPF content: {str(e)}', show=True)

		except Exception as e:
			error_dialog(gui, 'Error', str(e), show=True)

	def show_opf(self):
		"""Show OPF content for selected books"""
		debug_print('OPFHelper: show_opf() called')
		gui = self.gui
		rows = gui.library_view.selectionModel().selectedRows()
		if not rows or len(rows) == 0:
			gui.status_bar.showMessage("No books selected", 3000)
			return

		try:
			# Get the current database API handle
			db = gui.current_db.new_api

			# Get selected book IDs using the model
			book_ids = []
			model = gui.library_view.model()
			for row in rows:
				book_id = model.id(row.row())
				if book_id is not None:
					book_ids.append(book_id)

			if not book_ids:
				error_dialog(gui, 'Error', 'Could not get book IDs', show=True)
				return

			# Filter to only books with a supported format (now includes MOBI)
			supported_formats = ('KEPUB', 'EPUB', 'HTMLZ', 'AZW3', 'MOBI')
			valid_book_ids = [
				bid for bid in book_ids
				if any(db.has_format(bid, fmt) for fmt in supported_formats)
			]

			# Special handling for MOBI: use KindleUnpack extraction
			if valid_book_ids and all(db.has_format(bid, 'MOBI') and not any(db.has_format(bid, fmt) for fmt in ('KEPUB','EPUB','HTMLZ','AZW3')) for bid in valid_book_ids):
				try:
					from .mobi_opf_extract import extract_mobi_opf
					book_id = valid_book_ids[0]
					mobi_path = db.format_abspath(book_id, 'MOBI')
					if not mobi_path or not os.path.isfile(mobi_path):
						gui.status_bar.showMessage("MOBI format not available", 5000)
						return
					opf_xml, err = extract_mobi_opf(mobi_path)
					if opf_xml:
						dlg = OPFContentDialog(gui, valid_book_ids, db, self.qaction.icon())
						dlg.xml_content = opf_xml
						try:
							dlg.setModal(False)
						except Exception:
							pass
						dlg.show()
						try:
							if hasattr(dlg, 'bring_to_front'):
								dlg.bring_to_front()
							else:
								dlg.raise_()
								dlg.activateWindow()
						except Exception:
							pass
						return
					else:
						error_dialog(gui, 'MOBI OPF Extraction Failed', err or "Failed to extract OPF/XML", show=True)
						return
				except Exception as e:
					error_dialog(gui, 'MOBI OPF Extraction Error', str(e), show=True)
					return

			if not valid_book_ids:
				gui.status_bar.showMessage(
					"None of the selected books have EPUB/KEPUB/HTMLZ/AZW3 format",
					5000,
				)
				return

			# Open OPF Content dialog modeless (do not block Calibre UI)
			reuse_single = True
			try:
				reuse_single = bool(prefs.get('reuse_single_opf_helper_window', True))
			except Exception:
				pass

			if reuse_single:
				existing = getattr(self, '_opfhelper_opf_content_dialog', None)
				try:
					if existing is not None and existing.isVisible():
						try:
							existing.set_books(valid_book_ids, db=db)
						except Exception:
							pass
						try:
							# Restore if minimized and bring to front
							try:
								if hasattr(existing, 'bring_to_front'):
									existing.bring_to_front()
								else:
									if int(existing.windowState()) & int(Qt.WindowMinimized):
										existing.showNormal()
									existing.raise_()
									existing.activateWindow()
							except Exception:
								pass
						except Exception:
							pass
						return
				except Exception:
					pass

			d = OPFContentDialog(gui, valid_book_ids, db, self.qaction.icon())
			try:
				d.setModal(False)
			except Exception:
				pass
			try:
				d.setAttribute(Qt.WA_DeleteOnClose, True)
			except Exception:
				pass
			# Keep a strong reference while open so the dialog isn't GC'ed
			if reuse_single:
				setattr(self, '_opfhelper_opf_content_dialog', d)
				try:
					def _drop_ref(_obj=None, dlg=d):
						try:
							if getattr(self, '_opfhelper_opf_content_dialog', None) is dlg:
								setattr(self, '_opfhelper_opf_content_dialog', None)
						except Exception:
							pass
					d.destroyed.connect(_drop_ref)
				except Exception:
					pass
			else:
				try:
					lst = getattr(self, '_opfhelper_opf_content_dialogs', None)
					if lst is None:
						lst = []
						setattr(self, '_opfhelper_opf_content_dialogs', lst)
					lst.append(d)
					def _drop_ref(_obj=None, dlg=d, lst_ref=lst):
						try:
							lst_ref.remove(dlg)
						except Exception:
							pass
					d.destroyed.connect(_drop_ref)
				except Exception:
					pass
			d.show()
			try:
				if hasattr(d, 'bring_to_front'):
					d.bring_to_front()
				else:
					d.raise_()
					d.activateWindow()
			except Exception:
				pass
		except Exception as e:
			error_dialog(gui, 'Error', str(e), show=True)

	def rebuild_menus(self):
		"""Build and populate the plugin's menu"""
		debug_print('OPFHelper: rebuilding menus')
		try:
			self.menu.clear()

			# Theme-aware icon for the main "Show OPF" action
			try:
				from calibre.gui2 import is_dark_theme
				show_opf_icon = 'images/icon-for-dark-theme.png' if is_dark_theme() else 'images/icon-for-light-theme.png'
			except Exception:
				show_opf_icon = 'images/icon-for-light-theme.png'

			def _set_action_meta(ac, image=None, tooltip=None, text=None):
				if ac is None:
					return
				if text is not None:
					try:
						ac.setText(text)
					except Exception:
						pass
				if image:
					try:
						ac.setIcon(get_icon(image))
					except Exception:
						pass
				if tooltip:
					try:
						ac.setToolTip(tooltip)
					except Exception:
						pass

			def _make_check_action(menu, text, tooltip=None, triggered=None, checked=False):
				ac = QAction(text, menu)
				ac.setCheckable(True)
				try:
					ac.setChecked(bool(checked))
				except Exception:
					pass
				if tooltip:
					try:
						ac.setToolTip(tooltip)
					except Exception:
						pass
				if triggered is not None:
					try:
						ac.triggered.connect(triggered)
					except Exception:
						pass
				menu.addAction(ac)
				return ac

			# EPUB version scanner submenu (added to main menu later, sorted)
			epub_menu = QMenu(_('EPUB version scanner'), self.menu)
			epub_menu.setIcon(get_icon('images/search_icon.png'))

			# Add scope submenu (must be mutually exclusive: Library OR Selection)
			scope_menu = epub_menu.addMenu(_('Scope'))
			self.scope_library_menu = _make_check_action(
				scope_menu, _('Library'),
				tooltip=_('Search entire library'),
				triggered=lambda: self.change_epub_scope('Library'),
				checked=bool(prefs.get('epub_version_scope', 'Library') == 'Library'),
			)
			self.scope_selection_menu = _make_check_action(
				scope_menu, _('Selected book(s)'),
				tooltip=_('Search only selected books'),
				triggered=lambda: self.change_epub_scope('Selection'),
				checked=bool(prefs.get('epub_version_scope', 'Library') == 'Selection'),
			)
			epub_menu.addSeparator()

			# Use keyboard-registered actions so shortcut hints display in menus
			_set_action_meta(getattr(self, 'opf_find_epub3', None), image='images/search_icon.png',
						 tooltip=_('Find and mark all EPUB 3.0 books'))
			epub_menu.addAction(self.opf_find_epub3)
			_set_action_meta(getattr(self, 'opf_find_epub2', None), image='images/search_icon.png',
						 tooltip=_('Find and mark all EPUB 2.0 books'))
			epub_menu.addAction(self.opf_find_epub2)
			_set_action_meta(getattr(self, 'opf_find_not_epub3', None), image='images/search_icon.png',
						 tooltip=_('Find and mark all EPUB/KEPUB books that are not EPUB 3.0'))
			epub_menu.addAction(self.opf_find_not_epub3)

			# MOBI OPF version scanner submenu
			mobi_menu = QMenu(_('MOBI OPF scanner'), self.menu)
			mobi_menu.setIcon(get_icon('images/search_icon.png'))
			mobi_scope_menu = mobi_menu.addMenu(_('Scope'))
			self.mobi_scope_library_menu = _make_check_action(
				mobi_scope_menu, _('Library'),
				tooltip=_('Search entire library'),
				triggered=lambda: self.change_epub_scope('Library'),
				checked=bool(prefs.get('epub_version_scope', 'Library') == 'Library'),
			)
			self.mobi_scope_selection_menu = _make_check_action(
				mobi_scope_menu, _('Selected book(s)'),
				tooltip=_('Search only selected books'),
				triggered=lambda: self.change_epub_scope('Selection'),
				checked=bool(prefs.get('epub_version_scope', 'Library') == 'Selection'),
			)
			mobi_menu.addSeparator()
			_set_action_meta(getattr(self, 'mobi_find_opf3', None), image='images/search_icon.png',
						 tooltip=_('Find and mark all MOBI books whose extracted OPF is 3.0'))
			mobi_menu.addAction(self.mobi_find_opf3)
			_set_action_meta(getattr(self, 'mobi_find_opf2', None), image='images/search_icon.png',
						 tooltip=_('Find and mark all MOBI books whose extracted OPF is 2.0'))
			mobi_menu.addAction(self.mobi_find_opf2)
			_set_action_meta(getattr(self, 'mobi_find_not_opf3', None), image='images/search_icon.png',
						 tooltip=_('Find and mark all MOBI books whose extracted OPF is not 3.0'))
			mobi_menu.addAction(self.mobi_find_not_opf3)

			# All-formats OPF version scanner submenu
			all_menu = QMenu(_('All Formats OPF scanner'), self.menu)
			all_menu.setIcon(get_icon('images/search_icon.png'))
			all_scope_menu = all_menu.addMenu(_('Scope'))
			self.all_scope_library_menu = _make_check_action(
				all_scope_menu, _('Library'),
				tooltip=_('Search entire library'),
				triggered=lambda: self.change_epub_scope('Library'),
				checked=bool(prefs.get('epub_version_scope', 'Library') == 'Library'),
			)
			self.all_scope_selection_menu = _make_check_action(
				all_scope_menu, _('Selected book(s)'),
				tooltip=_('Search only selected books'),
				triggered=lambda: self.change_epub_scope('Selection'),
				checked=bool(prefs.get('epub_version_scope', 'Library') == 'Selection'),
			)
			all_menu.addSeparator()
			_set_action_meta(getattr(self, 'all_find_opf3', None), image='images/search_icon.png',
						 tooltip=_('Find and mark books whose OPF version is 3.0 across EPUB/KEPUB/MOBI/HTMLZ'))
			all_menu.addAction(self.all_find_opf3)
			_set_action_meta(getattr(self, 'all_find_opf2', None), image='images/search_icon.png',
						 tooltip=_('Find and mark books whose OPF version is 2.0 across EPUB/KEPUB/MOBI/HTMLZ'))
			all_menu.addAction(self.all_find_opf2)
			_set_action_meta(getattr(self, 'all_find_not_3', None), image='images/search_icon.png',
						 tooltip=_('Find and mark books whose OPF version is not 3.0 across EPUB/KEPUB/MOBI/HTMLZ'))
			all_menu.addAction(self.all_find_not_3)

			# Main actions: use the keyboard-registered actions so shortcut hints display.
			_set_action_meta(getattr(self, 'opf_show_content', None),
						 image=show_opf_icon,
						 tooltip=_('Show OPF content of selected books'))
			_set_action_meta(getattr(self, 'opf_edit_action', None),
						 image='images/edit_icon.png',
						 tooltip=_('Edit the OPF file of the selected book (opens in editor)'))
			_set_action_meta(getattr(self, 'opf_export_selected', None),
						 image='images/export_icon.png',
						 tooltip=_('Export OPF files from selected books to a directory'))
			_set_action_meta(getattr(self, 'opf_find_multiple', None),
						 image='images/multiple.png',
						 tooltip=_('Scan library to find books with multiple OPF files'))
			_set_action_meta(getattr(self, 'opf_find_xml_issues', None),
						 image='images/xml_error.png',
						 tooltip=_('Scan library to find books with XML parsing issues in OPF files'))
			_set_action_meta(getattr(self, 'opf_compare_action', None),
						 image='images/diff_icon.png',
						 tooltip=_('Compare current OPF with standards-corrected version'),
						 text=_('OPF standards comparison (experimental)'))

			for obj in [
				self.opf_show_content,
				self.opf_edit_action,
				self.opf_export_selected,
				self.opf_find_multiple,
				self.opf_find_xml_issues,
				self.opf_compare_action,
			]:
				self.menu.addAction(obj)

			# Bulk validator scope submenu (independent)
			bulk_scope_menu = self.menu.addMenu(_('Bulk validation scope'))
			self.bulk_validate_scope_library_menu = _make_check_action(
				bulk_scope_menu, _('Library'),
				tooltip=_('Validate OPFs for the entire current view'),
				triggered=lambda: self.change_bulk_validate_opfs_scope('Library'),
				checked=bool(prefs.get('bulk_validate_opfs_scope', 'Library') == 'Library'),
			)
			self.bulk_validate_scope_selection_menu = _make_check_action(
				bulk_scope_menu, _('Selected book(s)'),
				tooltip=_('Validate OPFs only for selected books'),
				triggered=lambda: self.change_bulk_validate_opfs_scope('Selection'),
				checked=bool(prefs.get('bulk_validate_opfs_scope', 'Library') == 'Selection'),
			)
			# Note: Use checkbox-style menu actions; mutual exclusivity is handled
			# by change_bulk_validate_opfs_scope() calling _sync_scope_action_checks().

			# Bulk validate OPFs action (keyboard-registered)
			_set_action_meta(getattr(self, 'bulk_opf_validator_action', None),
						 image='images/bulk_report-validator.png',
						 tooltip=_('Validate OPF files and log errors'),
						 text=_('Bulk validate OPFs'))
			self.menu.addAction(self.bulk_opf_validator_action)

			self.menu.addSeparator()
			scanners = [
				(all_menu.title(), all_menu),
				(epub_menu.title(), epub_menu),
				(mobi_menu.title(), mobi_menu),
			]
			scanners.sort(key=lambda x: (x[0] or '').casefold())
			for _text, m in scanners:
				self.menu.addMenu(m)

			# Keep references for enable/disable in about_to_show_menu
			self._menu_show_opf_action = getattr(self, 'opf_show_content', None)
			self._menu_export_opfs_action = getattr(self, 'opf_export_selected', None)

			# Ensure all Scope checkmarks are consistent
			try:
				self._sync_scope_action_checks()
			except Exception:
				pass

			# Configuration action if plugin has configuration
			if hasattr(self.interface_action_base_plugin, 'config_widget'):
				self.menu.addSeparator()
				ac = QAction(_('Customize plugin...'), self.menu)
				try:
					ac.setIcon(get_icon('config.png'))
				except Exception:
					pass
				try:
					ac.setToolTip(_('Customize plugin behavior'))
				except Exception:
					pass
				ac.triggered.connect(self.show_configuration)
				self.menu.addAction(ac)

		except Exception as e:
			debug_print(f'OPFHelper ERROR rebuilding menus: {str(e)}')
			traceback.print_exc()

class ElidedLabel(QLabel):
	"""Label that elides text in the middle when too long"""
	def __init__(self, parent=None):
		super().__init__(parent)
		self.full_text = ""

	def setText(self, text):
		self.full_text = text
		self.setToolTip(text)  # Show full text on hover
		# Elide in the middle
		metrics = self.fontMetrics()
		if metrics.horizontalAdvance(text) > self.width():
			available_width = self.width() - metrics.horizontalAdvance("...")
			left_width = available_width // 2
			right_width = available_width - left_width
			left_text = metrics.elidedText(text, Qt.ElideRight, left_width)
			right_text = metrics.elidedText(text[::-1], Qt.ElideLeft, right_width)[::-1]
			display_text = left_text[:-3] + "..." + right_text[3:]
			super().setText(display_text)
		else:
			super().setText(text)

	def resizeEvent(self, event):
		super().resizeEvent(event)
		if self.full_text:
			self.setText(self.full_text)  # Recalculate eliding on resize

class CoverPanel(QWidget):
	def __init__(self, parent=None):
		super().__init__(parent)
		self.pixmap = QPixmap()
		self.current_pixmap_size = QSize()
		self.setMinimumSize(200, 200)  # Keep minimum size reasonable
		self.setMaximumWidth(400)      # Limit maximum width
		self.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Expanding)  # Changed from Expanding to Minimum
		self.data = {}
		self.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
		self.customContextMenuRequested.connect(self.show_context_menu)

	def show_cover(self, cover_data):
		if cover_data:
			self.pixmap.loadFromData(cover_data)
			# Scale pixmap while maintaining aspect ratio
			scaled_size = self.pixmap.size()
			scaled_size.scale(400, 800, Qt.AspectRatioMode.KeepAspectRatio)  # Max dimensions
			self.current_pixmap_size = scaled_size
		else:
			self.pixmap = QPixmap()
			self.current_pixmap_size = QSize()
		self.update()

	def show_context_menu(self, pos):
		menu = QMenu(self)
		if not self.pixmap.isNull():
			copy_action = menu.addAction(QIcon(get_icon('edit-copy.png')), _('Copy cover'))
			copy_action.triggered.connect(self.copy_to_clipboard)
		menu.exec(self.mapToGlobal(pos))

	def copy_to_clipboard(self):
		if not self.pixmap.isNull():
			QApplication.clipboard().setPixmap(self.pixmap)

	def sizeHint(self):
		if self.pixmap.isNull():
			return QSize(200, 300)
		return self.current_pixmap_size

	def paintEvent(self, event):
		if self.pixmap.isNull():
			return

		canvas_size = self.rect()
		target = self.calculate_target_rect(canvas_size)

		p = QPainter(self)
		p.setRenderHints(QPainter.RenderHint.Antialiasing | QPainter.RenderHint.SmoothPixmapTransform)

		try:
			dpr = self.devicePixelRatioF()
		except AttributeError:
			dpr = self.devicePixelRatio()

		spmap = self.pixmap.scaled(target.size() * dpr, Qt.AspectRatioMode.KeepAspectRatio, Qt.TransformationMode.SmoothTransformation)
		spmap.setDevicePixelRatio(dpr)
		p.drawPixmap(target, spmap)

		# Add size overlay with dimensions
		sztgt = target.adjusted(0, 0, 0, -4)
		f = p.font()
		f.setBold(True)
		p.setFont(f)
		sz = f'\xa0{self.pixmap.width()} x {self.pixmap.height()}\xa0'
		flags = Qt.AlignmentFlag.AlignBottom|Qt.AlignmentFlag.AlignRight|Qt.TextFlag.TextSingleLine
		szrect = p.boundingRect(sztgt, flags, sz)
		p.fillRect(szrect.adjusted(0, 0, 0, 4), QColor(0, 0, 0, 200))
		p.setPen(QPen(QColor(255,255,255)))
		p.drawText(sztgt, flags, sz)
		p.end()

	def calculate_target_rect(self, canvas_size):
		"""Calculate the target rectangle for drawing the scaled pixmap"""
		if self.pixmap.isNull():
			return QRect()

		# Get available space
		available_width = min(canvas_size.width(), 400)  # Limit width
		available_height = canvas_size.height()

		# Calculate scaled dimensions maintaining aspect ratio
		scaled_size = self.pixmap.size()
		scaled_size.scale(available_width, available_height, Qt.AspectRatioMode.KeepAspectRatio)

		# Center in available space
		x = (canvas_size.width() - scaled_size.width()) // 2
		y = (canvas_size.height() - scaled_size.height()) // 2

		return QRect(x, y, scaled_size.width(), scaled_size.height())

# Add the MultipleOPFDialog class
class MultipleOPFDialog(QDialog):
	def __init__(self, gui, results):
		QDialog.__init__(self, gui)
		self.gui = gui
		self.setWindowTitle('Multiple OPF Files Report')
		self.setMinimumWidth(700)
		self.setMinimumHeight(400)

		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add description
		desc_text = f"Found {len(results)} books with multiple OPF files:"
		desc = QLabel(desc_text)
		desc.setStyleSheet("font-weight: bold;")
		layout.addWidget(desc)

		# Create text display
		self.text_edit = QTextEdit()
		self.text_edit.setReadOnly(True)
		layout.addWidget(self.text_edit)

		# Format and set results
		result_text = ""
		self.book_ids = []
		for book_id, title, opf_files in results:
			self.book_ids.append(book_id)
			result_text += f"Book: {title} (ID: {book_id})\n"
			result_text += "OPF Files:\n"
			for opf_file in opf_files:
				result_text += f"  • {opf_file}\n"
			result_text += "\n"

		self.text_edit.setPlainText(result_text)

		# Add buttons
		button_layout = QHBoxLayout()

		copy_button = QPushButton("Copy to Clipboard")
		copy_button.setIcon(QIcon.ic('edit-copy.png'))
		copy_button.clicked.connect(self.copy_to_clipboard)
		button_layout.addWidget(copy_button)

		# Add mark button for marking books in calibre with proper icon
		self.mark_button = QPushButton("Mark Affected Books")
		self.mark_button.setIcon(QIcon.ic('marked.png'))
		self.mark_button.setToolTip("Mark these books in Calibre's book list")
		self.mark_button.clicked.connect(self.mark_books)
		button_layout.addWidget(self.mark_button)

		# Add standard dialog buttons
		button_box = QDialogButtonBox(QDialogButtonBox.Ok)
		button_box.accepted.connect(self.accept)
		button_layout.addWidget(button_box)

		layout.addLayout(button_layout)

	def copy_to_clipboard(self):
		text = self.text_edit.toPlainText()
		QApplication.clipboard().setText(text)
		info_dialog(self, 'Copied', 'Report copied to clipboard', show=True)

	def mark_books(self):
		"""Mark the affected books in Calibre's book list"""
		if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
			try:
				act = self.gui.iactions['Mark Books']
				if hasattr(act, 'add_ids'):
					act.add_ids(self.book_ids)
				else:
					try:
						db_view = self.gui.current_db
						mids = db_view.data.marked_ids.copy()
						for bid in self.book_ids:
							mids[bid] = 'true'
						db_view.data.set_marked_ids(mids)
					except Exception as e:
						debug_print(f'OPFHelper: failed to mark books via fallback: {e}')
			except Exception:
				pass
			count = len(self.book_ids)
			info_dialog(self, 'Books Marked',
						f'{count} book{"s" if count != 1 else ""} with multiple OPF files '
						f'{"have" if count != 1 else "has"} been marked in your library.',
						show=True)
		else:
			error_dialog(self, 'Error', 'Could not access the Mark Books action.', show=True)

# Add the XMLParsingIssuesDialog class
class XMLParsingIssuesDialog(QDialog):
	def __init__(self, gui, results):
		QDialog.__init__(self, gui)
		self.gui = gui
		self.setWindowTitle('XML Parsing Issues Report')
		self.setMinimumWidth(700)
		self.setMinimumHeight(400)

		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add description
		desc_text = f"Found {len(results)} books with XML parsing issues in OPF files:"
		desc = QLabel(desc_text)
		desc.setStyleSheet("font-weight: bold;")
		layout.addWidget(desc)

		# Create text display
		self.text_edit = QTextEdit()
		self.text_edit.setReadOnly(True)
		layout.addWidget(self.text_edit)

		# Format and set results
		result_text = ""
		self.book_ids = []
		for book_id, title, opf_path, error_details in results:
			self.book_ids.append(book_id)
			result_text += f"Book: {title} (ID: {book_id})\n"
			result_text += f"OPF File: {opf_path}\n"
			result_text += f"Error: {error_details}\n\n"

		self.text_edit.setPlainText(result_text)

		# Add buttons
		button_layout = QHBoxLayout()

		copy_button = QPushButton("Copy to Clipboard")
		copy_button.setIcon(QIcon.ic('edit-copy.png'))
		copy_button.clicked.connect(self.copy_to_clipboard)
		button_layout.addWidget(copy_button)

		# Add mark button for marking books in calibre with proper icon
		self.mark_button = QPushButton("Mark Affected Books")
		self.mark_button.setIcon(QIcon.ic('marked.png'))
		self.mark_button.setToolTip("Mark these books in Calibre's book list")
		self.mark_button.clicked.connect(self.mark_books)
		button_layout.addWidget(self.mark_button)

		# Add standard dialog buttons
		button_box = QDialogButtonBox(QDialogButtonBox.Ok)
		button_box.accepted.connect(self.accept)
		button_layout.addWidget(button_box)

		layout.addLayout(button_layout)

	def copy_to_clipboard(self):
		text = self.text_edit.toPlainText()
		QApplication.clipboard().setText(text)
		info_dialog(self, 'Copied', 'Report copied to clipboard', show=True)

	def mark_books(self):
		"""Mark the affected books in Calibre's book list"""
		if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
			try:
				act = self.gui.iactions['Mark Books']
				# Clear previous marks first
				try:
					if hasattr(act, 'clear'):
						act.clear()
					elif hasattr(self.gui.current_db, 'data'):
						# Replace marked ids with only these book ids
						mids = {bid: 'true' for bid in self.book_ids}
						self.gui.current_db.data.set_marked_ids(mids)
				except Exception:
					pass  # Continue even if clearing fails
				if hasattr(act, 'add_ids'):
					act.add_ids(self.book_ids)
				else:
					try:
						db_view = self.gui.current_db
						# Ensure only these books are marked
						mids = {bid: 'true' for bid in self.book_ids}
						db_view.data.set_marked_ids(mids)
					except Exception as e:
						debug_print(f'OPFHelper: failed to mark books via fallback: {e}')
			except Exception:
				pass
			count = len(self.book_ids)
			info_dialog(self, 'Books Marked',
						f'{count} book{"s" if count != 1 else ""} with XML parsing issues '
						f'{"have" if count != 1 else "has"} been marked in your library.',
						show=True)
		else:
			error_dialog(self, 'Error', 'Could not access the Mark Books action.', show=True)
