# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from PyQt5.QtCore import Qt, pyqtSignal
from PyQt5.QtWidgets import QWidget, QLabel, QVBoxLayout, QSizePolicy, QMenu, QAction
from PyQt5.QtGui import QPixmap, QIcon
from calibre.gui2 import get_icon
try:
    from calibre.utils.localization import _
except Exception:
    try:
        _ = __builtins__['_']
    except Exception:
        _ = lambda x: x

class CoverPanel(QWidget):
    """A widget for displaying book covers with proper scaling and 'No Cover' fallback"""

    open_cover_with = pyqtSignal(object, object)  # Signals need to be class attributes

    def __init__(self, parent=None):
        QWidget.__init__(self, parent)
        self.data = {}  # Store book data
        self.pixmap = None  # Store current pixmap

        self._layout = QVBoxLayout(self)
        self._layout.setContentsMargins(5, 5, 5, 5)

        # Create cover label with proper styling
        self.cover_label = QLabel()
        self.cover_label.setAlignment(Qt.AlignCenter)
        self.cover_label.setMinimumSize(200, 300)
        self.cover_label.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        self.cover_label.setStyleSheet("""
            QLabel {
                background-color: palette(base);
                border: 1px solid palette(mid);
                border-radius: 5px;
                padding: 5px;
            }
        """)
        self._layout.addWidget(self.cover_label)

        # Add a "No Cover" message that's only visible when there's no cover
        self.no_cover_label = QLabel("No Cover Available")
        self.no_cover_label.setAlignment(Qt.AlignCenter)
        self.no_cover_label.setStyleSheet("color: gray; font-style: italic;")
        self.no_cover_label.hide()  # Hidden by default
        self._layout.addWidget(self.no_cover_label)

        # Set fixed width for cover panel
        self.setFixedWidth(250)

        # Enable context menu
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.contextMenuEvent)

    def show_cover(self, cover_data=None, book_id=None):
        """Display a cover image or 'No Cover' message if none available"""
        self.data['id'] = book_id  # Store book ID for context menu actions

        if not cover_data:
            self.cover_label.clear()
            self.no_cover_label.show()
            self.pixmap = None
            return

        try:
            pixmap = QPixmap()
            pixmap.loadFromData(cover_data)
            if not pixmap.isNull():
                # Scale pixmap while maintaining aspect ratio
                scaled_pixmap = pixmap.scaled(
                    self.cover_label.width() - 10,  # Subtract padding
                    self.cover_label.height() - 10,
                    Qt.KeepAspectRatio,
                    Qt.SmoothTransformation
                )
                self.pixmap = pixmap  # Store original pixmap
                self.cover_label.setPixmap(scaled_pixmap)
                self.cover_label.show()
                self.no_cover_label.hide()
            else:
                self.cover_label.clear()
                self.no_cover_label.show()
                self.pixmap = None
        except Exception as e:
            print(f"Error displaying cover: {e}")
            self.cover_label.clear()
            self.no_cover_label.show()
            self.pixmap = None

    def resizeEvent(self, event):
        """Handle resize events to maintain cover scaling"""
        super().resizeEvent(event)
        if self.pixmap and not self.pixmap.isNull():
            # Re-scale the current pixmap to fit the new size
            scaled_pixmap = self.pixmap.scaled(
                self.cover_label.width() - 10,
                self.cover_label.height() - 10,
                Qt.KeepAspectRatio,
                Qt.SmoothTransformation
            )
            self.cover_label.setPixmap(scaled_pixmap)

    def contextMenuEvent(self, ev):
        cm = QMenu(self)
        copy = cm.addAction(QIcon(get_icon('edit-copy.png')), _('Copy cover'))
        copy.triggered.connect(lambda: self.copy_to_clipboard())
        cm.addMenu(self.create_open_with_menu())
        cm.exec_(self.mapToGlobal(ev))

    def create_open_with_menu(self):
        m = QMenu(_('Open cover with...'))
        ac = QAction(_('Choose program...'), self)
        ac.triggered.connect(self.choose_open_with)
        m.addAction(ac)
        return m

    def copy_to_clipboard(self):
        if self.pixmap and not self.pixmap.isNull():
            QApplication.clipboard().setPixmap(self.pixmap)

    def choose_open_with(self):
        from calibre.gui2.open_with import choose_program
        entry = choose_program('cover_image', self)
        if entry is not None:
            self.open_with(entry)

    def open_with(self, entry):
        book_id = self.data.get('id', None)
        if book_id is not None:
            self.open_cover_with.emit(book_id, entry)