#!/usr/bin/env python
__license__   = 'GPL v3'
__copyright__ = '2025, Comfy.n'

import os

from qt.core import (
    QIcon,
    QColor,
    QFont,
    QMenu,
    QPainter,
    QPainterPath,
    QPixmap,
    QRectF,
    QToolButton,
    QFileSystemWatcher,
    QTimer,
    Qt,
)

from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import Application, info_dialog
from calibre.utils.config import prefs, config_dir

from calibre_plugins.output_format_cycler.config import KEY_SHOW_BADGE, KEY_TOGGLE_FORMATS, KEY_SHOW_CONFIRMATION, plugin_prefs


class ToggleOutputFormatAction(InterfaceAction):
    name = 'Output Format Cycler'
    action_spec = ('Output Format Cycler', None, 'Cycle the preferred output format (for news and other output).', None)
    popup_type = QToolButton.ToolButtonPopupMode.MenuButtonPopup

    def genesis(self):
        self._create_menu()
        self.rebuild_icon()
        self.qaction.triggered.connect(self.toggle_output_format)

        # Register this action in Preferences -> Keyboard shortcuts.
        try:
            self.gui.keyboard.register_shortcut(
                'plugin:output_format_cycler:cycle',
                'Output Format Cycler: cycle',
                description='Cycle the preferred output format (for news and other output).',
                action=self.qaction,
                group='Output Format Cycler',
                persist_shortcut=True,
            )
        except Exception:
            pass

        # Refresh toolbar icon when the calibre palette/icon theme changes.
        app = Application.instance()
        if app:
            app.palette_changed.connect(self.rebuild_icon)

        # Listen for external changes (Preferences -> Behavior -> Preferred output format)
        self._prefs_path = os.path.join(config_dir, 'global.py.json')
        self._last_prefs_mtime = None

        self._prefs_watcher = QFileSystemWatcher(self.gui)
        self._prefs_watcher.fileChanged.connect(self._on_prefs_file_changed, type=Qt.ConnectionType.QueuedConnection)
        if os.path.exists(self._prefs_path):
            self._prefs_watcher.addPath(self._prefs_path)
            try:
                self._last_prefs_mtime = os.path.getmtime(self._prefs_path)
            except Exception:
                self._last_prefs_mtime = None

        self._prefs_timer = QTimer(self.gui)
        self._prefs_timer.setInterval(1500)
        self._prefs_timer.timeout.connect(self._poll_prefs_file, type=Qt.ConnectionType.QueuedConnection)
        self._prefs_timer.start()

    def _create_menu(self):
        m = QMenu(self.gui)
        m.addAction('Cycle now').triggered.connect(self.toggle_output_format)
        m.addSeparator()
        m.addAction('Customize…').triggered.connect(self.customize_plugin)
        m.addAction('Open Preferences -> Behavior').triggered.connect(self.open_behavior_prefs)
        self.qaction.setMenu(m)

    def customize_plugin(self):
        # Standard way to open the plugin customization dialog.
        try:
            self.interface_action_base_plugin.do_user_config(self.gui)
        except Exception:
            try:
                self.interface_action_base_plugin.do_user_config(self.gui)
            except Exception:
                pass

    def open_behavior_prefs(self):
        try:
            self.gui.iactions['Preferences'].do_config(
                initial_plugin=('Interface', 'Behavior'),
                close_after_initial=True,
            )
        except Exception:
            pass

    def _with_badge(self, icon, text):
        try:
            base = icon.pixmap(32, 32)
            if base.isNull():
                return icon
            pm = QPixmap(base.size())
            pm.fill(Qt.GlobalColor.transparent)
            p = QPainter(pm)
            p.setRenderHint(QPainter.RenderHint.Antialiasing, True)
            p.drawPixmap(0, 0, base)

            rect = QRectF(pm.width() - 22, pm.height() - 14, 20, 12)
            path = QPainterPath()
            path.addRoundedRect(rect, 3, 3)
            p.fillPath(path, QColor(0, 0, 0, 170))
            p.setPen(QColor(255, 255, 255))
            f = QFont(p.font())
            f.setBold(True)
            f.setPointSize(max(7, f.pointSize() - 2))
            p.setFont(f)
            p.drawText(rect, Qt.AlignmentFlag.AlignCenter, text)
            p.end()
            return QIcon(pm)
        except Exception:
            return icon

    def _current_output_format(self):
        try:
            return str(prefs['output_format'] or '').upper()
        except Exception:
            return ''

    def _icon_for_output_format(self, fmt_upper):
        if not fmt_upper:
            return QIcon.ic('mimetypes/unknown.png')
        candidate = f'mimetypes/{fmt_upper.lower()}.png'
        icon = QIcon.ic(candidate)
        if icon is None or icon.isNull():
            return QIcon.ic('mimetypes/unknown.png')
        return icon

    def rebuild_icon(self):
        fmt = self._current_output_format()
        icon = self._icon_for_output_format(fmt)

        try:
            show_badge = bool(plugin_prefs.get(KEY_SHOW_BADGE, True))
        except Exception:
            show_badge = True

        self.qaction.setIcon(self._with_badge(icon, 'OUT') if show_badge else icon)

    def toggle_output_format(self):
        current = self._current_output_format()

        toggle_formats = [str(x).upper() for x in (plugin_prefs.get(KEY_TOGGLE_FORMATS, None) or []) if x]
        # Safe defaults
        if len(toggle_formats) < 2:
            toggle_formats = ['EPUB', 'PDF']

        if current in toggle_formats:
            idx = toggle_formats.index(current)
            new_fmt = toggle_formats[(idx + 1) % len(toggle_formats)]
        else:
            new_fmt = toggle_formats[0]

        # calibre.utils.config.prefs is a ConfigProxy; it commits on assignment.
        prefs['output_format'] = new_fmt

        # Refresh cache + icon
        try:
            prefs.refresh()
        except Exception:
            pass
        self.rebuild_icon()

        try:
            show_confirmation = bool(plugin_prefs.get(KEY_SHOW_CONFIRMATION, True))
        except Exception:
            show_confirmation = True

        if show_confirmation:
            info_dialog(
                self.gui,
                'Output Format Cycler',
                f'Default output format set to: {new_fmt}',
                show=True,
            )

    def _on_prefs_file_changed(self, path):
        # QFileSystemWatcher can drop the path if the file is atomically replaced.
        if path and os.path.exists(path) and path not in self._prefs_watcher.files():
            try:
                self._prefs_watcher.addPath(path)
            except Exception:
                pass
        self._refresh_prefs_and_icon()

    def _poll_prefs_file(self):
        try:
            mtime = os.path.getmtime(self._prefs_path)
        except Exception:
            return
        if self._last_prefs_mtime is None or mtime != self._last_prefs_mtime:
            self._last_prefs_mtime = mtime
            self._refresh_prefs_and_icon()

    def _refresh_prefs_and_icon(self):
        try:
            prefs.refresh()
        except Exception:
            pass
        self.rebuild_icon()
