#!/usr/bin/env python
__license__   = 'GPL v3'
__copyright__ = '2025, Comfy.n'

from qt.core import (
    QAbstractItemView,
    QCheckBox,
    QComboBox,
    QHBoxLayout,
    QIcon,
    QLabel,
    QListWidget,
    QListWidgetItem,
    QMessageBox,
    QPushButton,
    Qt,
    QVBoxLayout,
    QWidget,
)

from calibre.customize.ui import available_output_formats
from calibre.utils.config import JSONConfig


KEY_TOGGLE_FORMATS = 'toggle_formats'
KEY_SHOW_BADGE = 'show_badge'
KEY_SHOW_CONFIRMATION = 'show_confirmation'

# Stored under calibre config_dir/plugins/
plugin_prefs = JSONConfig('plugins/Output Format Cycler')
plugin_prefs.defaults[KEY_TOGGLE_FORMATS] = ['EPUB', 'PDF']
plugin_prefs.defaults[KEY_SHOW_BADGE] = True
plugin_prefs.defaults[KEY_SHOW_CONFIRMATION] = True

# Migrate from older config namespaces if present
_old_plugin_prefs_v1 = JSONConfig('plugins/Toggle Output Format')
_old_plugin_prefs_v0 = JSONConfig('plugins/Toogle Output Format')
try:
    old = _old_plugin_prefs_v1.get(KEY_TOGGLE_FORMATS, None)
    if old and not plugin_prefs.get(KEY_TOGGLE_FORMATS, None):
        plugin_prefs[KEY_TOGGLE_FORMATS] = old
except Exception:
    pass

try:
    old = _old_plugin_prefs_v0.get(KEY_TOGGLE_FORMATS, None)
    if old and not plugin_prefs.get(KEY_TOGGLE_FORMATS, None):
        plugin_prefs[KEY_TOGGLE_FORMATS] = old
except Exception:
    pass


def normalize_formats(formats):
    if not formats:
        return []
    seen = set()
    out = []
    for f in formats:
        if not f:
            continue
        fu = str(f).upper()
        if fu in seen:
            continue
        seen.add(fu)
        out.append(fu)
    return out


def icon_for_format(fmt_upper):
    if not fmt_upper:
        return QIcon.ic('mimetypes/unknown.png')
    icon = QIcon.ic(f'mimetypes/{fmt_upper.lower()}.png')
    return icon if icon and not icon.isNull() else QIcon.ic('mimetypes/unknown.png')


class ConfigWidget(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)

        # calibre's customization dialog can end up very short; enforce a minimum
        # height so the options and format list are visible without awkward scrolling.
        self.setMinimumHeight(180)

        self.l = QVBoxLayout(self)

        self.show_confirmation = QCheckBox('Show confirmation dialog after cycling output format', self)
        self.show_confirmation.setChecked(bool(plugin_prefs.get(KEY_SHOW_CONFIRMATION, True)))
        self.l.addWidget(self.show_confirmation)

        self.show_badge = QCheckBox('Show OUT badge overlay on the toolbar icon', self)
        self.show_badge.setChecked(bool(plugin_prefs.get(KEY_SHOW_BADGE, True)))
        self.l.addWidget(self.show_badge)

        self.l.addWidget(QLabel('Choose the output formats to cycle through (in order):'))

        self.formats_list = QListWidget(self)
        self.formats_list.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)
        self.formats_list.setAlternatingRowColors(True)
        self.l.addWidget(self.formats_list)

        controls = QHBoxLayout()
        self.add_combo = QComboBox(self)
        self.add_combo.setMinimumContentsLength(8)
        controls.addWidget(self.add_combo, stretch=1)

        self.add_btn = QPushButton('Add', self)
        self.remove_btn = QPushButton('Remove', self)
        self.up_btn = QPushButton('Up', self)
        self.down_btn = QPushButton('Down', self)

        for b in (self.add_btn, self.remove_btn, self.up_btn, self.down_btn):
            controls.addWidget(b)

        self.l.addLayout(controls)

        self.add_btn.clicked.connect(self.add_selected)
        self.remove_btn.clicked.connect(self.remove_selected)
        self.up_btn.clicked.connect(self.move_up)
        self.down_btn.clicked.connect(self.move_down)

        self._populate_available_formats()
        self._load_current()

    def _populate_available_formats(self):
        fmts = sorted({str(x).upper() for x in available_output_formats() if x})
        self.add_combo.clear()
        for f in fmts:
            self.add_combo.addItem(icon_for_format(f), f)

    def _load_current(self):
        current = normalize_formats(plugin_prefs.get(KEY_TOGGLE_FORMATS, None))
        self.formats_list.clear()
        for f in current:
            self._append_format(f)

    def _append_format(self, fmt_upper):
        item = QListWidgetItem(fmt_upper, self.formats_list)
        item.setIcon(icon_for_format(fmt_upper))
        item.setData(Qt.ItemDataRole.UserRole, fmt_upper)
        self.formats_list.addItem(item)

    def current_formats(self):
        out = []
        for i in range(self.formats_list.count()):
            item = self.formats_list.item(i)
            out.append(str(item.data(Qt.ItemDataRole.UserRole) or item.text()).upper())
        return normalize_formats(out)

    def add_selected(self):
        fmt = str(self.add_combo.currentText() or '').upper()
        if not fmt:
            return
        current = self.current_formats()
        if fmt in current:
            return
        self._append_format(fmt)

    def remove_selected(self):
        row = self.formats_list.currentRow()
        if row >= 0:
            self.formats_list.takeItem(row)

    def move_up(self):
        row = self.formats_list.currentRow()
        if row <= 0:
            return
        item = self.formats_list.takeItem(row)
        self.formats_list.insertItem(row - 1, item)
        self.formats_list.setCurrentRow(row - 1)

    def move_down(self):
        row = self.formats_list.currentRow()
        if row < 0 or row >= self.formats_list.count() - 1:
            return
        item = self.formats_list.takeItem(row)
        self.formats_list.insertItem(row + 1, item)
        self.formats_list.setCurrentRow(row + 1)

    def validate(self):
        fmts = self.current_formats()
        if len(fmts) < 2:
            QMessageBox.warning(self, 'Output Format Cycler', 'Select at least two formats to cycle through.')
            return False
        return True

    def save_settings(self):
        fmts = self.current_formats()
        plugin_prefs[KEY_TOGGLE_FORMATS] = fmts
        plugin_prefs[KEY_SHOW_BADGE] = bool(self.show_badge.isChecked())
        plugin_prefs[KEY_SHOW_CONFIRMATION] = bool(self.show_confirmation.isChecked())
